/*
 * linux/drivers/ide/arm/mxc_ide.c
 *
 * Based on Simtec BAST IDE driver
 * Copyright (c) 2003-2004 Simtec Electronics
 *  Ben Dooks <ben@simtec.co.uk>
 *
 * Copyright 2004-2007 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*!
 * @file mxc_ide.c
 *
 * @brief ATA driver
 *
 * @ingroup ATA
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/ide.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <linux/clk.h>
#include <linux/hdreg.h>
#include <linux/platform_device.h>

#include <asm/dma.h>
#include <asm/mach-types.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/delay.h>

#include <mach/dma.h>
#include <mach/hardware.h>

#include "mxc_ide.h"

extern void gpio_ata_active(void);
static void mxc_ide_dma_callback(void *arg, int error, unsigned int count);

static struct clk *ata_clk;

/*
 * This structure contains the timing parameters for
 * ATA bus timing in the 5 PIO modes.  The timings
 * are in nanoseconds, and are converted to clock
 * cycles before being stored in the ATA controller
 * timing registers.
 */
static struct {
	short t0, t1, t2_8, t2_16, t2i, t4, t9, tA;
} pio_specs[] = {
	[0] = {
	.t0 = 600,.t1 = 70,.t2_8 = 290,.t2_16 = 165,.t2i = 0,.t4 =
		    30,.t9 = 20,.tA = 50},[1] = {
	.t0 = 383,.t1 = 50,.t2_8 = 290,.t2_16 = 125,.t2i = 0,.t4 =
		    20,.t9 = 15,.tA = 50},[2] = {
	.t0 = 240,.t1 = 30,.t2_8 = 290,.t2_16 = 100,.t2i = 0,.t4 =
		    15,.t9 = 10,.tA = 50},[3] = {
	.t0 = 180,.t1 = 30,.t2_8 = 80,.t2_16 = 80,.t2i = 0,.t4 =
		    10,.t9 = 10,.tA = 50},[4] = {
	.t0 = 120,.t1 = 25,.t2_8 = 70,.t2_16 = 70,.t2i = 0,.t4 =
		    10,.t9 = 10,.tA = 50}
};

#define NR_PIO_SPECS (sizeof pio_specs / sizeof pio_specs[0])

/*
 * This structure contains the timing parameters for
 * ATA bus timing in the 3 MDMA modes.  The timings
 * are in nanoseconds, and are converted to clock
 * cycles before being stored in the ATA controller
 * timing registers.
 */
static struct {
	short t0M, tD, tH, tJ, tKW, tM, tN, tJNH;
} mdma_specs[] = {
	[0] = {
	.t0M = 480,.tD = 215,.tH = 20,.tJ = 20,.tKW = 215,.tM = 50,.tN =
		    15,.tJNH = 20},[1] = {
	.t0M = 150,.tD = 80,.tH = 15,.tJ = 5,.tKW = 50,.tM = 30,.tN =
		    10,.tJNH = 15},[2] = {
	.t0M = 120,.tD = 70,.tH = 10,.tJ = 5,.tKW = 25,.tM = 25,.tN =
		    10,.tJNH = 10}
};

#define NR_MDMA_SPECS (sizeof mdma_specs / sizeof mdma_specs[0])

/*
 * This structure contains the timing parameters for
 * ATA bus timing in the 6 UDMA modes.  The timings
 * are in nanoseconds, and are converted to clock
 * cycles before being stored in the ATA controller
 * timing registers.
 */
static struct {
	short t2CYC, tCYC, tDS, tDH, tDVS, tDVH, tCVS, tCVH, tFS_min, tLI_max,
	    tMLI, tAZ, tZAH, tENV_min, tSR, tRFS, tRP, tACK, tSS, tDZFS;
} udma_specs[] = {
	[0] = {
	.t2CYC = 235,.tCYC = 114,.tDS = 15,.tDH = 5,.tDVS = 70,.tDVH =
		    6,.tCVS = 70,.tCVH = 6,.tFS_min = 0,.tLI_max =
		    100,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 50,.tRFS = 75,.tRP = 160,.tACK = 20,.tSS =
		    50,.tDZFS = 80},[1] = {
	.t2CYC = 156,.tCYC = 75,.tDS = 10,.tDH = 5,.tDVS = 48,.tDVH =
		    6,.tCVS = 48,.tCVH = 6,.tFS_min = 0,.tLI_max =
		    100,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 30,.tRFS = 70,.tRP = 125,.tACK = 20,.tSS =
		    50,.tDZFS = 63},[2] = {
	.t2CYC = 117,.tCYC = 55,.tDS = 7,.tDH = 5,.tDVS = 34,.tDVH =
		    6,.tCVS = 34,.tCVH = 6,.tFS_min = 0,.tLI_max =
		    100,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 20,.tRFS = 60,.tRP = 100,.tACK = 20,.tSS =
		    50,.tDZFS = 47},[3] = {
	.t2CYC = 86,.tCYC = 39,.tDS = 7,.tDH = 5,.tDVS = 20,.tDVH =
		    6,.tCVS = 20,.tCVH = 6,.tFS_min = 0,.tLI_max =
		    100,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 20,.tRFS = 60,.tRP = 100,.tACK = 20,.tSS =
		    50,.tDZFS = 35},[4] = {
	.t2CYC = 57,.tCYC = 25,.tDS = 5,.tDH = 5,.tDVS = 7,.tDVH =
		    6,.tCVS = 7,.tCVH = 6,.tFS_min = 0,.tLI_max =
		    100,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 50,.tRFS = 60,.tRP = 100,.tACK = 20,.tSS =
		    50,.tDZFS = 25},[5] = {
	.t2CYC = 38,.tCYC = 17,.tDS = 4,.tDH = 5,.tDVS = 5,.tDVH =
		    6,.tCVS = 10,.tCVH = 10,.tFS_min = 0,.tLI_max =
		    75,.tMLI = 20,.tAZ = 10,.tZAH = 20,.tENV_min =
		    20,.tSR = 20,.tRFS = 50,.tRP = 85,.tACK = 20,.tSS =
		    50,.tDZFS = 40}
};

#define NR_UDMA_SPECS (sizeof udma_specs / sizeof udma_specs[0])

/*!
 * Calculate values for the ATA bus timing registers and store
 * them into the hardware.
 *
 * @param       mode        Selects PIO, MDMA or UDMA modes
 *
 * @param       speed       Specifies the sub-mode number
 *
 * @return      EINVAL      speed out of range, or illegal mode
 */
static int set_ata_bus_timing(int speed, enum ata_mode mode)
{
	/* get the bus clock cycle time, in ns */
	int T = 1 * 1000 * 1000 * 1000 / clk_get_rate(ata_clk);
	mxc_ide_time_cfg_t cfg0, cfg1, cfg2, cfg3, cfg4, cfg5;
	/* every mode gets the same t_off and t_on */

	GET_TIME_CFG(&cfg0, MXC_IDE_TIME_OFF);
	cfg0.bytes.field1 = 3;
	cfg0.bytes.field2 = 3;
	SET_TIME_CFG(&cfg0, 3, MXC_IDE_TIME_OFF);

	switch (mode) {
	case PIO:
		if (speed < 0 || speed >= NR_PIO_SPECS) {
			return -EINVAL;
		}
		cfg0.bytes.field3 = (pio_specs[speed].t1 + T) / T;
		cfg0.bytes.field4 = (pio_specs[speed].t2_8 + T) / T;

		cfg1.bytes.field1 = (pio_specs[speed].t2_8 + T) / T;
		cfg1.bytes.field2 = (pio_specs[speed].tA + T) / T + 2;
		cfg1.bytes.field3 = 1;
		cfg1.bytes.field4 = (pio_specs[speed].t4 + T) / T;

		GET_TIME_CFG(&cfg2, MXC_IDE_TIME_9);
		cfg2.bytes.field1 = (pio_specs[speed].t9 + T) / T;

		SET_TIME_CFG(&cfg0, 0x0C, MXC_IDE_TIME_OFF);
		SET_TIME_CFG(&cfg1, 0x0F, MXC_IDE_TIME_2r);
		SET_TIME_CFG(&cfg2, 0x01, MXC_IDE_TIME_9);
		break;
	case MDMA:
		if (speed < 0 || speed >= NR_MDMA_SPECS) {
			return -EINVAL;
		}
		GET_TIME_CFG(&cfg2, MXC_IDE_TIME_9);
		GET_TIME_CFG(&cfg3, MXC_IDE_TIME_K);

		cfg2.bytes.field2 = (mdma_specs[speed].tM + T) / T;
		cfg2.bytes.field3 = (mdma_specs[speed].tJNH + T) / T;
		cfg2.bytes.field4 = (mdma_specs[speed].tD + T) / T;

		cfg3.bytes.field1 = (mdma_specs[speed].tKW + T) / T;

		SET_TIME_CFG(&cfg2, 0x0E, MXC_IDE_TIME_9);
		SET_TIME_CFG(&cfg3, 0x01, MXC_IDE_TIME_K);
		break;
	case UDMA:
		if (speed < 0 || speed >= NR_UDMA_SPECS) {
			return -EINVAL;
		}

		GET_TIME_CFG(&cfg3, MXC_IDE_TIME_K);

		cfg3.bytes.field2 = (udma_specs[speed].tACK + T) / T;
		cfg3.bytes.field3 = (udma_specs[speed].tENV_min + T) / T;
		cfg3.bytes.field4 = (udma_specs[speed].tRP + T) / T + 2;

		cfg4.bytes.field1 = (udma_specs[speed].tZAH + T) / T;
		cfg4.bytes.field2 = (udma_specs[speed].tMLI + T) / T;
		cfg4.bytes.field3 = (udma_specs[speed].tDVH + T) / T + 1;
		cfg4.bytes.field4 = (udma_specs[speed].tDZFS + T) / T;

		cfg5.bytes.field1 = (udma_specs[speed].tDVS + T) / T;
		cfg5.bytes.field2 = (udma_specs[speed].tCVH + T) / T;
		cfg5.bytes.field3 = (udma_specs[speed].tSS + T) / T;
		cfg5.bytes.field4 = (udma_specs[speed].tCYC + T) / T;

		SET_TIME_CFG(&cfg3, 0x0E, MXC_IDE_TIME_K);
		SET_TIME_CFG(&cfg4, 0x0F, MXC_IDE_TIME_ZAH);
		SET_TIME_CFG(&cfg5, 0x0F, MXC_IDE_TIME_DVS);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/*!
 * Placeholder for code to make any hardware tweaks
 * necessary to select a drive.  Currently we are
 * not aware of any.
 */
static void mxc_ide_selectproc(ide_drive_t * drive)
{
}

/*!
 * Called to set the PIO mode.
 *
 * @param   drive       Specifies the drive
 * @param   pio    Specifies the PIO mode number desired
 */
static void mxc_ide_set_pio_mode(ide_drive_t * drive, u8 pio)
{
	set_ata_bus_timing(pio, PIO);
}

/*!
 * Hardware-specific interrupt service routine for the ATA driver,
 * called mainly just to dismiss the interrupt at the hardware, and
 * to indicate whether there actually was an interrupt pending.
 *
 * The generic IDE related interrupt handling is done by the IDE layer.
 */
static int mxc_ide_ack_intr(struct hwif_s *hw)
{
	unsigned char status = ATA_RAW_READ(MXC_IDE_INTR_PENDING);
	unsigned char enable = ATA_RAW_READ(MXC_IDE_INTR_ENABLE);

	/*
	 * The only interrupts we can actually dismiss are the FIFO conditions.
	 * INTRQ comes from the bus, and must be dismissed according to IDE
	 * protocol, which will be done by the IDE layer, even when DMA
	 * is invovled (DMA can see it, but can't dismiss it).
	 */
	ATA_RAW_WRITE(status, MXC_IDE_INTR_CLEAR);

	if (status & enable & ~MXC_IDE_INTR_ATA_INTRQ2) {
		printk(KERN_ERR "mxc_ide_ack_intr: unexpected interrupt, "
		       "status=0x%02X\n", status);
	}

	return status ? 1 : 0;
}

/*!
 * Decodes the specified transfer mode and sets both timing and ultra modes
 *
 * @param       drive       Specifies the drive
 *
 * @param       xfer_mode   Specifies the desired transfer mode
 *
 * @return      EINVAL      Illegal mode specified
 */
static void mxc_ide_set_dma_mode(ide_drive_t * drive, u8 xfer_mode)
{
	ide_hwif_t *hwif	= drive->hwif;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;

	switch (xfer_mode) {
	case XFER_UDMA_7:
	case XFER_UDMA_6:
	case XFER_UDMA_5:
	case XFER_UDMA_4:
	case XFER_UDMA_3:
	case XFER_UDMA_2:
	case XFER_UDMA_1:
	case XFER_UDMA_0:
		priv->ultra = 1;
		set_ata_bus_timing(xfer_mode - XFER_UDMA_0, UDMA);
		break;
	case XFER_MW_DMA_2:
	case XFER_MW_DMA_1:
	case XFER_MW_DMA_0:
		priv->ultra = 0;
		set_ata_bus_timing(xfer_mode - XFER_MW_DMA_0, MDMA);
		break;
	case XFER_PIO_4:
	case XFER_PIO_3:
	case XFER_PIO_2:
	case XFER_PIO_1:
	case XFER_PIO_0:
		set_ata_bus_timing(xfer_mode - XFER_PIO_0, PIO);
		break;
	}

	return;
}

/*!
 * Called by the IDE layer when something goes wrong
 *
 * @param       drive       Specifies the drive
 *
 */
static void mxc_ide_resetproc(ide_drive_t * drive)
{
	printk(KERN_INFO "%s: resetting ATA controller\n", __func__);

	ATA_RAW_WRITE(0x00, MXC_IDE_ATA_CONTROL);
	udelay(100);
	ATA_RAW_WRITE(MXC_IDE_CTRL_ATA_RST_B, MXC_IDE_ATA_CONTROL);
	udelay(100);
}

/*!
 * Set DMA hardware parameters
 *
 * @param       drive       Specifies the drive
 *
 * @return      0 if successful
 */
static int mxc_ide_set_dma_paras(ide_drive_t * drive)
{
	/* consult the list of known "bad" drives */
	if (__ide_dma_bad_drive(drive))
		return 1;

	blk_queue_max_hw_segments(drive->queue, MXC_IDE_DMA_BD_NR);
	blk_queue_max_hw_segments(drive->queue, MXC_IDE_DMA_BD_NR);
	blk_queue_max_segment_size(drive->queue, MXC_IDE_DMA_BD_SIZE_MAX);

	return 0;
}

/*!
 * Turn on DMA if the drive can handle it.
 *
 * @param       drive       Specifies the drive
 *
 * @return      0 if successful
 */
static int mxc_ide_dma_check(ide_drive_t * drive)
{
	struct hd_driveid *id = (struct hd_driveid *)drive->id;

	if (id && (id->capability & 1)) {
		/*
		 * Enable DMA on any drive that has
		 * UltraDMA (mode 0/1/2/3/4/5/6) enabled
		 */
		if ((id->field_valid & 4) && ((id->dma_ultra >> 8) & 0x7f)) {
			mxc_ide_set_dma_paras(drive);
			return 0;
		}
		/*
		 * Enable DMA on any drive that has mode2 DMA
		 * (multi or single) enabled
		 */
		if (id->field_valid & 2) {	/* regular DMA */
			if ((id->dma_mword & 0x404) == 0x404 ||
			    (id->dma_1word & 0x404) == 0x404) {
				mxc_ide_set_dma_paras(drive);
				return 0;
			}
		}
		/* Consult the list of known "good" drives */
		if (ide_dma_good_drive(drive)) {
			mxc_ide_set_dma_paras(drive);
			return 0;
		}
	}

	return 0;
}

/*!
 * The DMA is done, and the drive is done.  We'll check the BD array for
 * errors, and unmap the scatter-gather list.
 *
 * @param       drive       The drive we're servicing
 *
 * @return      0 means all is well, others means DMA signalled an error ,
 */
static int mxc_ide_dma_end(ide_drive_t * drive)
{
	ide_hwif_t *hwif = drive->hwif;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;
	int dma_stat = priv->dma_stat;

	BUG_ON(drive->waiting_for_dma == 0);
	drive->waiting_for_dma = 0;

	/*
	 * We'll unmap the sg table regardless of status.
	 */
	dma_unmap_sg(priv->dev, hwif->sg_table, hwif->sg_nents,
		     hwif->sg_dma_direction);

	return dma_stat;
}

/*!
 * The end-of-DMA interrupt handler
 *
 * @param       drive       Specifies the drive
 *
 * @return      ide_stopped or ide_started
 */
static ide_startstop_t mxc_ide_dma_intr(ide_drive_t * drive)
{
	u8 stat, dma_stat;
	ide_hwif_t *hwif = drive->hwif;
	struct request *rq = hwif->rq;
	u8 fifo_fill;

	if (!rq)
		return ide_stopped;

	fifo_fill = ATA_RAW_READ(MXC_IDE_FIFO_FILL);
	BUG_ON(fifo_fill);

	dma_stat = hwif->dma_ops->dma_end(drive);
	stat = ide_read_status(hwif);
	if (OK_STAT(stat, DRIVE_READY, drive->bad_wstat | DRQ_STAT)) {
		if (dma_stat == MXC_DMA_DONE) {
			ide_end_request(drive, 1, rq->nr_sectors);
			return ide_stopped;
		}
		printk(KERN_ERR "%s: mxc_ide_dma_intr: bad DMA status (0x%x)\n",
		       drive->name, dma_stat);
	}

	return ide_error(drive, "mxc_ide_dma_intr", stat);
	return 0;
}

/*!
 * Directs the IDE INTRQ signal to DMA or to the CPU
 *
 * @param       hwif       Specifies the IDE controller
 *
 * @param       which       \b INTRQ_DMA or \b INTRQ_MCU
 *
 */
static void mxc_ide_set_intrq(ide_hwif_t * hwif, intrq_which_t which)
{
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;
	unsigned char enable = ATA_RAW_READ(MXC_IDE_INTR_ENABLE);

	switch (which) {
	case INTRQ_DMA:
		enable &= ~MXC_IDE_INTR_ATA_INTRQ2;
		enable |= MXC_IDE_INTR_ATA_INTRQ1;
		break;
	case INTRQ_MCU:
		enable &= ~MXC_IDE_INTR_ATA_INTRQ1;
		enable |= MXC_IDE_INTR_ATA_INTRQ2;
		break;
	}
	priv->enable = enable;

	ATA_RAW_WRITE(enable, MXC_IDE_INTR_ENABLE);
}

/*!
 * Masks drive interrupts temporarily at the hardware level
 *
 * @param       drive       Specifies the drive
 *
 * @param       mask        1 = disable interrupts, 0 = re-enable
 *
 */
static void mxc_ide_maskproc(ide_drive_t * drive, int mask)
{
	ide_hwif_t *hwif	= drive->hwif;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;

	BUG_ON(!priv);

	if (mask) {
		ATA_RAW_WRITE(0, MXC_IDE_INTR_ENABLE);
	} else {
		ATA_RAW_WRITE(priv->enable, MXC_IDE_INTR_ENABLE);
	}
}

/*!
 * DMA completion callback routine.  This gets called after the DMA request
 * has completed or aborted.All we need to do here is return ownership of
 * the drive's INTRQ signal back to the CPU, which will immediately raise
 * the interrupt to the IDE layer.
 *
 * @param       arg         The drive we're servicing
 * @param	error	    The error number of DMA transfer.
 * @param	count	    The transfered length.
 */
static void mxc_ide_dma_callback(void *arg, int error, unsigned int count)
{
	ide_hwif_t *hwif = ((ide_drive_t *)arg)->hwif;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) (hwif->hwif_data);
	unsigned long fifo_fill;

	/*
	 * clean the fifo if the fill register is non-zero.
	 * If the fill register is non-zero, it is incorrect state.
	 */
	fifo_fill = ATA_RAW_READ(MXC_IDE_FIFO_FILL);
	while (fifo_fill) {
		ATA_RAW_READ(MXC_IDE_FIFO_DATA_32);
		fifo_fill = ATA_RAW_READ(MXC_IDE_FIFO_FILL);
	}

	priv->dma_stat = error;
	/*
	 * Redirect ata_intrq back to us instead of the DMA.
	 */
	mxc_ide_set_intrq(hwif, INTRQ_MCU);
}

/*!
 * DMA set up.  This is called once per DMA request to the drive. It delivers
 * the scatter-gather list into the DMA channel and prepares both the ATA
 * controller and the DMA engine for the DMA
 * transfer.
 *
 * @param       drive     The drive we're servicing
 *
 * @return      0 on success, non-zero otherwise
 */
static int mxc_ide_dma_setup(ide_drive_t * drive)
{
	ide_hwif_t *hwif = drive->hwif;
	struct request *rq = hwif->rq;
	struct scatterlist *sg = hwif->sg_table;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;
	int dma_ultra = priv->ultra ? MXC_IDE_CTRL_DMA_ULTRA : 0;
	int dma_mode = 0;
	int chan;
	u8 ata_control;
	u8 fifo_fill;

	BUG_ON(!rq);
	BUG_ON(!priv);
	BUG_ON(drive->waiting_for_dma);

	/*Initialize the dma state */
	priv->dma_stat = MXC_DMA_TRANSFER_ERROR;

	/*
	 * Prepare the ATA controller for the DMA
	 */
	if (rq_data_dir(rq)) {
		chan = priv->dma_write_chan;
		ata_control = MXC_IDE_CTRL_FIFO_RST_B |
		    MXC_IDE_CTRL_ATA_RST_B |
		    MXC_IDE_CTRL_FIFO_TX_EN |
		    MXC_IDE_CTRL_DMA_PENDING |
		    dma_ultra | MXC_IDE_CTRL_DMA_WRITE;

		dma_mode = MXC_DMA_MODE_WRITE;
	} else {
		chan = priv->dma_read_chan;
		ata_control = MXC_IDE_CTRL_FIFO_RST_B |
		    MXC_IDE_CTRL_ATA_RST_B |
		    MXC_IDE_CTRL_FIFO_RCV_EN |
		    MXC_IDE_CTRL_DMA_PENDING | dma_ultra;

		dma_mode = MXC_DMA_MODE_READ;
	}

	/*
	 * Set up the DMA interrupt callback
	 */
	mxc_dma_callback_set(chan, mxc_ide_dma_callback, (void *)drive);

	/*
	 * If the ATA FIFO isn't empty, we shouldn't even be here
	 */
	fifo_fill = ATA_RAW_READ(MXC_IDE_FIFO_FILL);
	BUG_ON(fifo_fill);	// $$$ TODO: need better recovery here

	ide_map_sg(drive, rq);

	hwif->sg_dma_direction = rq_data_dir(rq) ? DMA_TO_DEVICE :
	    DMA_FROM_DEVICE;

	hwif->sg_nents = dma_map_sg(priv->dev, sg, hwif->sg_nents,
				    hwif->sg_dma_direction);
	BUG_ON(!hwif->sg_nents);
	BUG_ON(hwif->sg_nents > MXC_IDE_DMA_BD_NR);

	mxc_dma_sg_config(chan, sg, hwif->sg_nents, 0, dma_mode);

	ATA_RAW_WRITE(ata_control, MXC_IDE_ATA_CONTROL);
	ATA_RAW_WRITE(MXC_IDE_DMA_WATERMARK / 2, MXC_IDE_FIFO_ALARM);

	/*
	 * Route ata_intrq to the DMA engine, and not to us.
	 */
	mxc_ide_set_intrq(hwif, INTRQ_DMA);

	/*
	 * The DMA and ATA controller are ready to go.
	 * mxc_ide_dma_start() will start the DMA transfer,
	 * and mxc_ide_dma_exec_cmd() will tickle the drive, which
	 * actually initiates the DMA transfer on the ATA bus.
	 * The ATA controller is DMA slave for both read and write.
	 */
	BUG_ON(drive->waiting_for_dma);
	drive->waiting_for_dma = 1;

	return 0;
}

/*!
 * DMA timeout notification.  This gets called when the IDE layer above
 * us times out waiting for a request.
 *
 * @param       drive       The drive we're servicing
 *
 * @return      0 to attempt recovery, otherwise, an additional tick count
 *              to keep waiting
 */
static int mxc_ide_dma_timer_expiry(ide_drive_t * drive)
{
	printk(KERN_ERR "%s: fifo_fill=%d\n", drive->name,
	       readb(MXC_IDE_FIFO_FILL));

	mxc_ide_resetproc(NULL);

	if (drive->waiting_for_dma) {
		drive->hwif->dma_ops->dma_end(drive);
	}

	return 0;
}

/*!
 * Called by the IDE layer to start a DMA request on the specified drive.
 * The request has already been prepared by \b mxc_ide_dma_setup().  All
 * we need to do is pass the command through while specifying our timeout
 * handler.
 *
 * @param       drive       The drive we're servicing
 *
 * @param       cmd         The IDE command for the drive
 *
 */
static void mxc_ide_dma_exec_cmd(ide_drive_t * drive, u8 cmd)
{
	ide_execute_command(drive, cmd, mxc_ide_dma_intr, 2 * WAIT_CMD,
			    mxc_ide_dma_timer_expiry);
}

/*!
 * Called by the IDE layer to start the DMA controller.  The request has
 * already been prepared by \b mxc_ide_dma_setup().  All we do here
 * is tickle the DMA channel.
 *
 * @param       drive       The drive we're servicing
 *
 */
static void mxc_ide_dma_start(ide_drive_t * drive)
{
	ide_hwif_t *hwif = drive->hwif;
	struct request *rq = hwif->rq;
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;
	int chan = rq_data_dir(rq) ? priv->dma_write_chan : priv->dma_read_chan;

	BUG_ON(chan < 0);

	/*
	 * Tickle the DMA channel.  This starts the channel, but it is likely
	 * that DMA will yield and go idle before the DMA request arrives from
	 * the drive.  Nonetheless, at least the context will be hot.
	 */
	mxc_dma_enable(chan);
}

/*!
 * There is a race between the DMA interrupt and the timeout interrupt.  This
 * gets called during the IDE layer's timeout interrupt to see if the DMA
 * interrupt has also occured or is pending.
 *
 * @param       drive       The drive we're servicing
 *
 * @return      1 means there is a DMA interrupt pending, 0 otherwise
 */
static int mxc_ide_dma_test_irq(ide_drive_t * drive)
{
	unsigned char status = ATA_RAW_READ(MXC_IDE_INTR_PENDING);

	/*
	 * We need to test the interrupt status without dismissing any.
	 */
	return status & (MXC_IDE_INTR_ATA_INTRQ1
			 | MXC_IDE_INTR_ATA_INTRQ2) ? 1 : 0;
}

static void mxc_ide_dma_host_set(ide_drive_t * drive, int on)
{
}

/*!
 * Called for special handling on timeouts.
 *
 * @param       drive       The drive we're servicing
 */
static void mxc_ide_dma_timeout(ide_drive_t * drive)
{
}
/*!
 * Called for special handling on lost irq's.
 *
 * @param       drive       The drive we're servicing
 */
static void mxc_ide_dma_lost_irq(ide_drive_t * drive)
{
}

/*!
 * Called once per controller to set up DMA
 *
 * @param       hwif       Specifies the IDE controller
 *
 */
static int mxc_ide_dma_init(ide_hwif_t * hwif, const struct ide_port_info *d)
{
	mxc_ide_private_t *priv = (mxc_ide_private_t *) hwif->hwif_data;

	hwif->dmatable_cpu = NULL;
	hwif->dmatable_dma = 0;

#if defined(CONFIG_MACH_MX3KZ)
	{
		extern int gpio_get_pinconfig(void);
		int st;
		st = gpio_get_pinconfig();
		if(st!=1 && st!=3){
			printk("MXC: IDE driver can't install (Select [RSW1]\n");
			return -ENODEV;
		}
	}
#endif

	/*
	 * Allocate and setup the DMA channels
	 */
	priv->dma_read_chan = mxc_dma_request(MXC_DMA_ATA_RX, "MXC ATA RX");
	if (priv->dma_read_chan < 0) {
		printk(KERN_ERR "%s: couldn't get RX DMA channel\n",
		       hwif->name);
		goto err_out;
	}

	priv->dma_write_chan = mxc_dma_request(MXC_DMA_ATA_TX, "MXC ATA TX");
	if (priv->dma_write_chan < 0) {
		printk(KERN_ERR "%s: couldn't get TX DMA channel\n",
		       hwif->name);
		goto err_out;
	}

	printk(KERN_INFO "%s: read chan=%d , write chan=%d \n",
	       hwif->name, priv->dma_read_chan, priv->dma_write_chan);

	set_ata_bus_timing(0, UDMA);

	/*
	 * All ready now
	 */
	hwif->cbl = ATA_CBL_PATA80;

	return 0;

err_out:
	if (priv->dma_read_chan >= 0)
		mxc_dma_free(priv->dma_read_chan);
	if (priv->dma_write_chan >= 0)
		mxc_dma_free(priv->dma_write_chan);
	return 0;
}


static const struct ide_port_ops mxc_ide_port_ops = {
	.maskproc		= mxc_ide_maskproc,
	.resetproc              = mxc_ide_resetproc,
	.selectproc             = mxc_ide_selectproc,
	.set_pio_mode           = mxc_ide_set_pio_mode,
	.set_dma_mode           = mxc_ide_set_dma_mode,
};

static struct ide_dma_ops mxc_ide_dma_ops = {
	.dma_host_set           = mxc_ide_dma_host_set,
	.dma_timeout            = mxc_ide_dma_timeout,
	.dma_test_irq           = mxc_ide_dma_test_irq,
	.dma_lost_irq           = mxc_ide_dma_lost_irq,
	.dma_start              = mxc_ide_dma_start,
	.dma_end                = mxc_ide_dma_end,
	.dma_exec_cmd           = mxc_ide_dma_exec_cmd,
	.dma_setup              = mxc_ide_dma_setup,
};

static const struct ide_port_info mxc_ide_port_info = {
	.name                   = "mxc-ide",
	.init_dma               = mxc_ide_dma_init,
	.port_ops               = &mxc_ide_port_ops,
	.dma_ops                = &mxc_ide_dma_ops,
	.host_flags             = IDE_HFLAG_MMIO,
	.pio_mask               = ATA_PIO4,
	.udma_mask              = 0x7f,
	.mwdma_mask             = 0x07,
	.swdma_mask             = 0x07,
};

/*!
 * MXC-specific IDE registration helper routine.  Among other things,
 * we tell the IDE layer where our standard IDE drive registers are,
 * through the \b io_ports array.  The IDE layer sends commands to and
 * reads status directly from attached IDE drives through these drive
 * registers.
 *
 * @param   pdev   Pointer to IDE platform device structure
 *
 * @param   priv   Pointer to private structure
 *
 * @return  ENODEV if no drives are present, 0 otherwise
 */
static int __init
mxc_ide_register(struct platform_device *pdev, mxc_ide_private_t * priv)
{
	int i = 0, ret;
	struct ide_host *host;
	ide_hwif_t *hwif;
	const int regsize = 4;
	hw_regs_t hw;
	hw_regs_t *hws[] = {&hw, NULL, NULL, NULL};
	struct ide_port_info d = mxc_ide_port_info;
	struct resource *res;
	unsigned long base;

	memset(&hw, 0, sizeof(hw));

	/* ATA controller register address */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if(!res)
		return -ENODEV;

	base = (unsigned long)(IO_ADDRESS(res->start));

	for (i = 0; i < 8; ++i)
		hw.io_ports_array[i] = base + 0xA0 + i * regsize;

	hw.io_ports.ctl_addr = base + 0xD8;

	/* IRQ resource */
	hw.irq = platform_get_irq(pdev, 0);
	if(!hw.irq)
		return -ENODEV;

	hw.ack_intr = &mxc_ide_ack_intr;

	host = ide_host_alloc(&d, hws);
	if (host == NULL)
		return -ENODEV;

	hwif = host->ports[0];
	hwif->hwif_data = (void *)priv;
	platform_set_drvdata(pdev, host);

	ret = ide_host_register(host, &d, hws);
	if (ret)
		goto err_free;

	/*
	 * The IDE layer will set hwif->present if we have devices attached,
	 * if we don't, discard the interface reset the ports.
	 */
	if (!hwif->present) {
		printk(KERN_INFO "ide%d: Bus empty, interface released.\n",
		       hwif->index);
		ret = -ENODEV;
		goto err_free;
	}

	for (i = 0; i < MAX_DRIVES; i++) {
		mxc_ide_dma_check(hwif->devices[i]);
	}

	return 0;

err_free:
	ide_host_free(host);
	return ret;
}

/*!
 * Driver initialization routine.  Prepares the hardware for ATA activity.
 */
static int __init mxc_ide_probe(struct platform_device *pdev)
{
	int index = 0;
	mxc_ide_private_t *priv;

	printk(KERN_INFO
	       "MXC: IDE driver, (c) 2004-2006 Freescale Semiconductor\n");

	/* Configure the pads */
	gpio_ata_active();

	/* Enable the clock before disabling ATA_RESET to prevent hang */
	ata_clk = clk_get(NULL, "ata_clk.0");
	clk_enable(ata_clk);

	/* Deassert the reset bit to enable the interface */
	ATA_RAW_WRITE(MXC_IDE_CTRL_ATA_RST_B, MXC_IDE_ATA_CONTROL);

	/* Set initial timing and mode */
	set_ata_bus_timing(4, PIO);

	/* Reset the interface */
	mxc_ide_resetproc(NULL);

	/*
	 * Enable hardware interrupts.
	 * INTRQ2 goes to us, so we enable it here, but we'll need to ignore
	 * it when DMA is doing the transfer.
	 */
	ATA_RAW_WRITE(MXC_IDE_INTR_ATA_INTRQ2, MXC_IDE_INTR_ENABLE);

	/*
	 * Allocate a private structure
	 */
	priv = (mxc_ide_private_t *) kmalloc(sizeof *priv, GFP_KERNEL);
	if (priv == NULL) {
		ATA_RAW_WRITE(0, MXC_IDE_INTR_ENABLE);
		return ENOMEM;
	}

	memset(priv, 0, sizeof *priv);
	priv->dev = NULL;	// dma_map_sg() ignores it anyway
	priv->dma_read_chan = -1;
	priv->dma_write_chan = -1;
	priv->enable = MXC_IDE_INTR_ATA_INTRQ2;

	/*
	 * Now register
	 */

	index = mxc_ide_register(pdev, priv);
	if (index == -1) {
		printk(KERN_ERR "Unable to register the MXC IDE driver\n");
		ATA_RAW_WRITE(0, MXC_IDE_INTR_ENABLE);
		kfree(priv);
		return ENODEV;
	}
#ifdef ATA_USE_IORDY
	/* turn on IORDY protocol */

	udelay(25);
	ATA_RAW_WRITE(MXC_IDE_CTRL_ATA_RST_B | MXC_IDE_CTRL_IORDY_EN,
		      MXC_IDE_ATA_CONTROL);
#endif

	return 0;
}

/*!
 * Driver exit routine.  Clean up.
 */
static void __exit mxc_ide_remove(struct platform_device *pdev)
{
	struct ide_host *host = platform_get_drvdata(pdev);
	ide_hwif_t *hwif = host->ports[0];
	mxc_ide_private_t *priv;

	BUG_ON(!hwif);
	priv = (mxc_ide_private_t *) hwif->hwif_data;
	BUG_ON(!priv);

	/*
	 * Unregister the interface at the IDE layer.  This should shut
	 * down any drives and pending I/O.
	 */
	ide_host_remove(host);

	/*
	 * Disable hardware interrupts.
	 */
	ATA_RAW_WRITE(0, MXC_IDE_INTR_ENABLE);

	/*
	 * Deallocate DMA channels.  Free's BDs and everything.
	 */
	if (priv->dma_read_chan >= 0)
		mxc_dma_free(priv->dma_read_chan);
	if (priv->dma_write_chan >= 0)
		mxc_dma_free(priv->dma_write_chan);

	/*
	 * Turn off the clock
	 */
	clk_disable(ata_clk);
	clk_put(ata_clk);

	/*
	 * Free the private structure.
	 */
	kfree(priv);
	hwif->hwif_data = NULL;
}

static struct platform_driver mxc_ide_driver = {
	.probe  = mxc_ide_probe,
	.remove = __exit_p(mxc_ide_remove),
	.driver = {
		.name   = "mxc-ide",
		.owner  = THIS_MODULE,
	},
};

static int __init mxc_ide_init(void)
{
	return platform_driver_register(&mxc_ide_driver);
}

static void __exit mxc_ide_exit(void)
{
	platform_driver_unregister(&mxc_ide_driver);
}

module_init(mxc_ide_init);
module_exit(mxc_ide_exit);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION
    ("Freescale MXC IDE (based on Simtec BAST IDE driver by Ben Dooks <ben@simtec.co.uk>)");
