/*
 * smac_ethtool.c
 * ethtool support for smac
 *
 * Copyright 2008,2009 Sony Corporation
 *
 * This code is based on drivers/net/e1000/e1000_ethool.c
 */
/*******************************************************************************

  Intel PRO/1000 Linux driver
  Copyright(c) 1999 - 2006 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  Linux NICS <linux.nics@intel.com>
  e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "smac_local.h"

#define DRV_NAME_LENGTH	32

extern char smac_driver_name[];
extern char smac_driver_version[];

/* unit test */
#ifdef SMAC_UNIT_TEST
extern smac_latency_range_type smac_update_itnr(smac_latency_range_type current_itnr, int packets, int bytes);
static int test_smac_update_itnr(struct smac_adapter *adapter, uint64_t *data);
extern void smac_set_itnr(struct smac_adapter *adapter);
static int test_smac_set_itnr(struct smac_adapter *adapter, uint64_t *data);
#endif

/* for statistics */
struct smac_stats {
	char stat_string[ETH_GSTRING_LEN];
	int sizeof_stat;
	int stat_offset;
};

#define SMAC_STAT(m) sizeof(((struct smac_adapter *)0)->m), \
		      offsetof(struct smac_adapter, m)
static const struct smac_stats smac_gstrings_stats[] = {
#ifdef CONFIG_SMAC_DBG	/* SMAC H/W register for debug */
	{ "intr_stat", SMAC_STAT(stats.intr_stat) },
	{ "intr_enable", SMAC_STAT(stats.intr_enable) },
	{ "operation", SMAC_STAT(stats.operation) },
	{ "tx_db_base", SMAC_STAT(stats.tx_db_base) },
	{ "tx_db_length_minus1", SMAC_STAT(stats.tx_db_length_minus1) },
	{ "tx_db_head", SMAC_STAT(stats.tx_db_head) },
	{ "tx_db_tail", SMAC_STAT(stats.tx_db_tail) },
	{ "rx_db_base", SMAC_STAT(stats.rx_db_base) },
	{ "rx_db_length_minus1", SMAC_STAT(stats.rx_db_length_minus1) },
	{ "rx_db_head", SMAC_STAT(stats.rx_db_head) },
	{ "rx_db_tail", SMAC_STAT(stats.rx_db_tail) },
	{ "bus_mode", SMAC_STAT(stats.bus_mode) },
	{ "rx_max_frm_length", SMAC_STAT(stats.rx_max_frm_length) },
	{ "rx_frame_byte_limit", SMAC_STAT(stats.rx_frame_byte_limit) },
	{ "rx_frame_count_limit", SMAC_STAT(stats.rx_frame_count_limit) },
	{ "rx_minsize_limit", SMAC_STAT(stats.rx_minsize_limit) },
	{ "rx_delay_timer", SMAC_STAT(stats.rx_delay_timer) },
	{ "rx_absolute_timer", SMAC_STAT(stats.rx_absolute_timer) },
	{ "rx_desc_limit", SMAC_STAT(stats.rx_desc_limit) },
	{ "rxfifo_frm_cnt", SMAC_STAT(stats.rxfifo_frm_cnt) },
	{ "phy_polling", SMAC_STAT(stats.phy_polling) },
	{ "phy_polled_data", SMAC_STAT(stats.phy_polled_data) },
	{ "tx_desc_addr", SMAC_STAT(stats.tx_desc_addr) },
	{ "rx_desc_addr", SMAC_STAT(stats.rx_desc_addr) },
	{ "tx_desc0", SMAC_STAT(stats.tx_desc0) },
	{ "tx_desc1", SMAC_STAT(stats.tx_desc1) },
	{ "tx_desc2", SMAC_STAT(stats.tx_desc2) },
	{ "tx_desc3", SMAC_STAT(stats.tx_desc3) },
	{ "rx_desc0", SMAC_STAT(stats.rx_desc0) },
	{ "rx_desc1", SMAC_STAT(stats.rx_desc1) },
	{ "rx_desc2", SMAC_STAT(stats.rx_desc2) },
	{ "rx_desc3", SMAC_STAT(stats.rx_desc3) },
	{ "state_tx_rd", SMAC_STAT(stats.state_tx_rd) },
	{ "state_tx_gd", SMAC_STAT(stats.state_tx_gd) },
	{ "state_tx_gb", SMAC_STAT(stats.state_tx_gb) },
	{ "state_tx_wr", SMAC_STAT(stats.state_tx_wr) },
	{ "state_tx_wb", SMAC_STAT(stats.state_tx_wb) },
	{ "state_rx_rd", SMAC_STAT(stats.state_rx_rd) },
	{ "state_rx_gd", SMAC_STAT(stats.state_rx_gd) },
	{ "state_rx_wr", SMAC_STAT(stats.state_rx_wr) },
	{ "state_rx_wb", SMAC_STAT(stats.state_rx_wb) },
	{ "state_swr", SMAC_STAT(stats.state_swr) },
	{ "state_ra", SMAC_STAT(stats.state_ra) },
	{ "state_ra_pl", SMAC_STAT(stats.state_ra_pl) },
	{ "hw_version", SMAC_STAT(stats.hw_version) },
	{ "mac_config", SMAC_STAT(stats.mac_config) },
	{ "mac_frame_filter", SMAC_STAT(stats.mac_frame_filter) },
	{ "hash_table_high", SMAC_STAT(stats.hash_table_high) },
	{ "hash_table_low", SMAC_STAT(stats.hash_table_low) },
	{ "phy_addr", SMAC_STAT(stats.phy_addr) },
	{ "phy_data", SMAC_STAT(stats.phy_data) },
	{ "control_frm", SMAC_STAT(stats.control_frm) },
	{ "vlan_tag", SMAC_STAT(stats.vlan_tag) },
	{ "mac_version", SMAC_STAT(stats.mac_version) },
	{ "wake_up_filter", SMAC_STAT(stats.wake_up_filter) },
	{ "power_management", SMAC_STAT(stats.power_management) },
	{ "mac_intr_stat", SMAC_STAT(stats.mac_intr_stat) },
	{ "mac_intr_mask", SMAC_STAT(stats.mac_intr_mask) },
	{ "mac_addr_0_high", SMAC_STAT(stats.mac_addr_0_high) },
	{ "mac_addr_0_low", SMAC_STAT(stats.mac_addr_0_low) },
	{ "rgmii_stat", SMAC_STAT(stats.rgmii_stat) },
	{ "mmc_control", SMAC_STAT(stats.mmc_control) },
	{ "mmc_intr_rx", SMAC_STAT(stats.mmc_intr_rx) },
	{ "mmc_intr_tx", SMAC_STAT(stats.mmc_intr_tx) },
	{ "mmc_intr_mask_rx", SMAC_STAT(stats.mmc_intr_mask_rx) },
	{ "mmc_intr_mask_tx", SMAC_STAT(stats.mmc_intr_mask_tx) },
#endif
	/* SMAC H/W statistics */
	{ "txoctetcount_gb", SMAC_STAT(stats.txoctetcount_gb) },
	{ "txframecount_gb", SMAC_STAT(stats.txframecount_gb) },
	{ "txbroadcastframes_g", SMAC_STAT(stats.txbroadcastframes_g) },
	{ "txmulticastframes_g", SMAC_STAT(stats.txmulticastframes_g) },
	{ "tx64octets_gb", SMAC_STAT(stats.tx64octets_gb) },
	{ "tx65to127octets_gb", SMAC_STAT(stats.tx65to127octets_gb) },
	{ "tx128to255octets_gb", SMAC_STAT(stats.tx128to255octets_gb) },
	{ "tx256to511octets_gb", SMAC_STAT(stats.tx256to511octets_gb) },
	{ "tx512to1023octets_gb", SMAC_STAT(stats.tx512to1023octets_gb) },
	{ "tx1024tomaxoctets_gb", SMAC_STAT(stats.tx1024tomaxoctets_gb) },
	{ "txunicastframes_gb", SMAC_STAT(stats.txunicastframes_gb) },
	{ "txmulticastframes_gb", SMAC_STAT(stats.txmulticastframes_gb) },
	{ "txbroadcastframes_gb", SMAC_STAT(stats.txbroadcastframes_gb) },
	{ "txunderflowerror", SMAC_STAT(stats.txunderflowerror) },
	{ "txsinglecol_g", SMAC_STAT(stats.txsinglecol_g) },
	{ "txmulticol_g", SMAC_STAT(stats.txmulticol_g) },
	{ "txdeferred", SMAC_STAT(stats.txdeferred) },
	{ "txlatecol", SMAC_STAT(stats.txlatecol) },
	{ "txexesscol", SMAC_STAT(stats.txexesscol) },
	{ "txcarriererror", SMAC_STAT(stats.txcarriererror) },
	{ "txoctetcount_g", SMAC_STAT(stats.txoctetcount_g) },
	{ "txframecount_g", SMAC_STAT(stats.txframecount_g) },
	{ "txexcessdef", SMAC_STAT(stats.txexcessdef) },
	{ "txpauseframes", SMAC_STAT(stats.txpauseframes) },
	{ "txvlanframes_g", SMAC_STAT(stats.txvlanframes_g) },
	{ "rxframecount_gb", SMAC_STAT(stats.rxframecount_gb) },
	{ "rxoctetcount_gb", SMAC_STAT(stats.rxoctetcount_gb) },
	{ "rxoctetcount_g", SMAC_STAT(stats.rxoctetcount_g) },
	{ "rxbroadcastframes_g", SMAC_STAT(stats.rxbroadcastframes_g) },
	{ "rxmulticastframes_g", SMAC_STAT(stats.rxmulticastframes_g) },
	{ "rxcrcerror", SMAC_STAT(stats.rxcrcerror) },
	{ "rxalignmenterror", SMAC_STAT(stats.rxalignmenterror) },
	{ "rxrunterror", SMAC_STAT(stats.rxrunterror) },
	{ "rxjabbererror", SMAC_STAT(stats.rxjabbererror) },
	{ "rxundersize_g", SMAC_STAT(stats.rxundersize_g) },
	{ "rxoversize_g", SMAC_STAT(stats.rxoversize_g) },
	{ "rx64octets_gb", SMAC_STAT(stats.rx64octets_gb) },
	{ "rx65to127octets_gb", SMAC_STAT(stats.rx65to127octets_gb) },
	{ "rx128to255octets_gb", SMAC_STAT(stats.rx128to255octets_gb) },
	{ "rx256to511octets_gb", SMAC_STAT(stats.rx256to511octets_gb) },
	{ "rx512to1023octets_gb", SMAC_STAT(stats.rx512to1023octets_gb) },
	{ "rx1024tomaxoctets_gb", SMAC_STAT(stats.rx1024tomaxoctets_gb) },
	{ "rxunicastframes_g", SMAC_STAT(stats.rxunicastframes_g) },
	{ "rxlengtherror", SMAC_STAT(stats.rxlengtherror) },
	{ "rxoutofrangetype", SMAC_STAT(stats.rxoutofrangetype) },
	{ "rxpauseframes", SMAC_STAT(stats.rxpauseframes) },
	{ "rxfifooverflow", SMAC_STAT(stats.rxfifooverflow) },
	{ "rxvlanframes_gb", SMAC_STAT(stats.rxvlanframes_gb) },
	{ "rxwatchdogerror", SMAC_STAT(stats.rxwatchdogerror) },
	{ "rxipv4_gd_frms", SMAC_STAT(stats.rxipv4_gd_frms) },
	{ "rxipv4_hdrerr_frms", SMAC_STAT(stats.rxipv4_hdrerr_frms) },
	{ "rxipv4_nopay_frms", SMAC_STAT(stats.rxipv4_nopay_frms) },
	{ "rxipv4_frag_frms", SMAC_STAT(stats.rxipv4_frag_frms) },
	{ "rxipv4_udsbl_frms", SMAC_STAT(stats.rxipv4_udsbl_frms) },
	{ "rxipv6_gd_frms", SMAC_STAT(stats.rxipv6_gd_frms) },
	{ "rxipv6_hdrerr_frms", SMAC_STAT(stats.rxipv6_hdrerr_frms) },
	{ "rxipv6_nopay_frms", SMAC_STAT(stats.rxipv6_nopay_frms) },
	{ "rxudp_gd_frms", SMAC_STAT(stats.rxudp_gd_frms) },
	{ "rxudp_err_frms", SMAC_STAT(stats.rxudp_err_frms) },
	{ "rxtcp_gd_frms", SMAC_STAT(stats.rxtcp_gd_frms) },
	{ "rxtcp_err_frms", SMAC_STAT(stats.rxtcp_err_frms) },
	{ "rxicmp_gd_frms", SMAC_STAT(stats.rxicmp_gd_frms) },
	{ "rxicmp_err_frms", SMAC_STAT(stats.rxicmp_err_frms) },
	{ "rxipv4_gd_octets", SMAC_STAT(stats.rxipv4_gd_octets) },
	{ "rxipv4_hdrerr_octets", SMAC_STAT(stats.rxipv4_hdrerr_octets) },
	{ "rxipv4_nopay_octets", SMAC_STAT(stats.rxipv4_nopay_octets) },
	{ "rxipv4_frag_octets", SMAC_STAT(stats.rxipv4_frag_octets) },
	{ "rxipv4_udsel_octets", SMAC_STAT(stats.rxipv4_udsel_octets) },
	{ "rxipv6_gd_octets", SMAC_STAT(stats.rxipv6_gd_octets) },
	{ "rxipv6_hdrerr_octets", SMAC_STAT(stats.rxipv6_hdrerr_octets) },
	{ "rxipv6_nopay_octets", SMAC_STAT(stats.rxipv6_nopay_octets) },
	{ "rxudp_gd_octets", SMAC_STAT(stats.rxudp_gd_octets) },
	{ "rxudp_err_octets", SMAC_STAT(stats.rxudp_err_octets) },
	{ "rxtcp_gd_octets", SMAC_STAT(stats.rxtcp_gd_octets) },
	{ "rxtcp_err_octets", SMAC_STAT(stats.rxtcp_err_octets) },
	{ "rxicmp_gd_octets", SMAC_STAT(stats.rxicmp_gd_octets) },
	{ "rxicmp_err_octets", SMAC_STAT(stats.rxicmp_err_octets) },
};

#define SMAC_QUEUE_STATS_LEN 0
#define SMAC_GLOBAL_STATS_LEN sizeof(smac_gstrings_stats) / sizeof(struct smac_stats)
#define SMAC_STATS_LEN (SMAC_GLOBAL_STATS_LEN + SMAC_QUEUE_STATS_LEN)

/* for ethtool -t */
static const char smac_gstrings_test[][ETH_GSTRING_LEN] = {
#ifdef SMAC_UNIT_TEST
	"smac_update_itnr (UnitTest)",
	"smac_set_itnr (UnitTest)",
#else
	/* normal test */
#endif
};
#define SMAC_TEST_LEN sizeof(smac_gstrings_test) / ETH_GSTRING_LEN

#ifdef SMAC_UNIT_TEST
#define TEST_SMAC_UPDATE_ITNR	0
#define TEST_SMAC_SET_ITNR	1
#endif


/*----------------------------------------------------------------------------*/
/* set struct ethtool_cmd member */
static int
smac_get_settings(struct net_device *netdev, struct ethtool_cmd *ecmd)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	spin_lock_irq(&adapter->phy_lock);
	mii_ethtool_gset(&adapter->mii, ecmd);
	spin_unlock_irq(&adapter->phy_lock);

	return 0;
}

static int
smac_set_settings(struct net_device *netdev, struct ethtool_cmd *ecmd)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	int res;

	spin_lock_irq(&adapter->phy_lock);
	res = mii_ethtool_sset(&adapter->mii, ecmd);
	spin_unlock_irq(&adapter->phy_lock);

	return res;
}


/*----------------------------------------------------------------------------*/
/* get message level */
static uint32_t
smac_get_msglevel(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	return adapter->msg_enable;
}


/*----------------------------------------------------------------------------*/
/* set message level */
static void
smac_set_msglevel(struct net_device *netdev, uint32_t data)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	adapter->msg_enable = data;
}


/*----------------------------------------------------------------------------*/
/* get link status */
static u32
smac_get_link(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	return mii_link_ok(&adapter->mii);
}


/*----------------------------------------------------------------------------*/
/* get flow control parameter */
static void
smac_get_pauseparam(struct net_device *netdev, struct ethtool_pauseparam *pause)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	pause->autoneg =
		(adapter->fc_autoneg ? AUTONEG_ENABLE : AUTONEG_DISABLE);

	if (adapter->fc == SMAC_FC_RX_PAUSE) {
		pause->rx_pause = 1;
	} else if (adapter->fc == SMAC_FC_TX_PAUSE) {
		pause->tx_pause = 1;
	} else if (adapter->fc == SMAC_FC_FULL) {
		pause->rx_pause = 1;
		pause->tx_pause = 1;
	}
}


/*----------------------------------------------------------------------------*/
/* set flow control parameter */
static int
smac_set_pauseparam(struct net_device *netdev, struct ethtool_pauseparam *pause)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	adapter->fc_autoneg = pause->autoneg;

	while (test_and_set_bit(__SMAC_RESETTING, (void *)&adapter->flags))
		msleep(1);

	if (pause->rx_pause && pause->tx_pause)
		adapter->fc = SMAC_FC_FULL;
	else if (pause->rx_pause && !pause->tx_pause)
		adapter->fc = SMAC_FC_RX_PAUSE;
	else if (!pause->rx_pause && pause->tx_pause)
		adapter->fc = SMAC_FC_TX_PAUSE;
	else if (!pause->rx_pause && !pause->tx_pause)
		adapter->fc = SMAC_FC_NONE;

	if (adapter->fc_autoneg == AUTONEG_ENABLE) {
		if (netif_running(adapter->netdev)) {
			smac_down(adapter);
			smac_up(adapter);
		}
	} else {
		smac_force_mac_fc(adapter);
	}

	clear_bit(__SMAC_RESETTING, (void *)&adapter->flags);

	return 0;
}


/*----------------------------------------------------------------------------*/
/* get RX cksum capability */
static u32
smac_get_rx_csum(struct net_device *dev)
{
	/* always enabled */
	return 1;
}


/*----------------------------------------------------------------------------*/
/* get register values */
#define SMAC_REGS_LEN	0x2000

static int
smac_get_regs_len(struct net_device *netdev)
{
	return SMAC_REGS_LEN;
}


static void
smac_get_regs(struct net_device *netdev, struct ethtool_regs *regs, void *p)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	uint32_t *regs_buff = p;
	int i;

	regs->version = 0xF;

	spin_lock_irq(&adapter->phy_lock);
	for (i = 0; i < SMAC_REGS_LEN / sizeof(uint32_t); i++) {
		regs_buff[i] = smac_readreg(adapter, i * sizeof(uint32_t));
	}
	spin_unlock_irq(&adapter->phy_lock);
}


/*----------------------------------------------------------------------------*/
/* return driver infomation */
static void smac_get_drvinfo(struct net_device *netdev,
			     struct ethtool_drvinfo *drvinfo)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	char firmware_version[DRV_NAME_LENGTH];
	uint32_t hw_version;
	uint8_t *h = (uint8_t *)&hw_version;

	/* driver name */
	strncpy(drvinfo->driver, smac_driver_name, DRV_NAME_LENGTH);

	/* driver version */
	strncpy(drvinfo->version, smac_driver_version, DRV_NAME_LENGTH);

	/* h/w version */
	hw_version = smac_readreg(adapter, SMAC_HW_VERSION);
	snprintf(firmware_version, DRV_NAME_LENGTH, "%d.%d.%d.%d",
		 h[3], h[2], h[1], h[0]);
	strncpy(drvinfo->fw_version, firmware_version, DRV_NAME_LENGTH);

	drvinfo->n_stats = SMAC_STATS_LEN;
	drvinfo->testinfo_len = SMAC_TEST_LEN;
}


/*----------------------------------------------------------------------------*/
static int
smac_diag_test_count(struct net_device *netdev)
{
	return SMAC_TEST_LEN;
}


/*----------------------------------------------------------------------------*/
static void
smac_diag_test(struct net_device *netdev,
		   struct ethtool_test *eth_test, uint64_t *data)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	set_bit(__SMAC_TESTING, (void *)&adapter->flags);

	if (eth_test->flags == ETH_TEST_FL_OFFLINE) {
		boolean_t if_running = netif_running(netdev);

		/* Offline tests */
		SMAC_DPRINTK(HW, INFO, "offline testing starting\n");

		if (if_running)
			/* indicate we're in test mode */
			dev_close(netdev);
		else
			smac_reset(adapter);

#ifdef SMAC_UNIT_TEST
		if (!test_smac_update_itnr(adapter, &data[TEST_SMAC_UPDATE_ITNR]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

		if (!test_smac_set_itnr(adapter, &data[TEST_SMAC_SET_ITNR]))
			eth_test->flags |= ETH_TEST_FL_FAILED;
#endif
		smac_reset(adapter);
		clear_bit(__SMAC_TESTING, (void *)&adapter->flags);
		if (if_running)
			dev_open(netdev);
	} else {
		SMAC_DPRINTK(HW, INFO, "online testing starting\n");

		clear_bit(__SMAC_TESTING, (void *)&adapter->flags);
	}
	msleep_interruptible(4 * 1000);
}



/*----------------------------------------------------------------------------*/
/* restart auto-negtiation */
static int smac_nway_reset(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	return mii_nway_restart(&adapter->mii);
}


/*----------------------------------------------------------------------------*/
/* return # of statistics */
static int
smac_get_stats_count(struct net_device *netdev)
{
	return SMAC_STATS_LEN;
}


/*----------------------------------------------------------------------------*/
/* set statistics value */
static void
smac_get_ethtool_stats(struct net_device *netdev,
		struct ethtool_stats *stats, uint64_t *data)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	int i;

	smac_update_stats(adapter);
	for (i = 0; i < SMAC_GLOBAL_STATS_LEN; i++) {
		char *p = (char *)adapter + smac_gstrings_stats[i].stat_offset;

		data[i] = (smac_gstrings_stats[i].sizeof_stat == sizeof(uint64_t))
		? *(uint64_t *)p : *(uint32_t *)p;
	}

	SMAC_ASSERT(i == SMAC_STATS_LEN);
}


/*----------------------------------------------------------------------------*/
/* set statistics string */
static void
smac_get_strings(struct net_device *netdev, uint32_t stringset, uint8_t *data)
{
	uint8_t *p = data;
	int i;

	switch (stringset) {
	case ETH_SS_TEST:
		memcpy(data, *smac_gstrings_test,
			SMAC_TEST_LEN*ETH_GSTRING_LEN);
		break;

	case ETH_SS_STATS:	/* for statistics: option -S */
		for (i = 0; i < SMAC_GLOBAL_STATS_LEN; i++) {
			memcpy(p, smac_gstrings_stats[i].stat_string,
			       ETH_GSTRING_LEN);
			p += ETH_GSTRING_LEN;
		}

		SMAC_ASSERT(p - data == SMAC_STATS_LEN * ETH_GSTRING_LEN);
		break;
	}
}


/*----------------------------------------------------------------------------*/
/* ethtool operations */
static struct ethtool_ops smac_ethtool_ops = {
	.get_settings		= smac_get_settings,
	.set_settings		= smac_set_settings,
	.get_drvinfo		= smac_get_drvinfo,
	.get_regs_len		= smac_get_regs_len,
	.get_regs		= smac_get_regs,
	.get_msglevel		= smac_get_msglevel,
	.set_msglevel		= smac_set_msglevel,
	.nway_reset		= smac_nway_reset,
	.get_link		= smac_get_link,
	.get_pauseparam         = smac_get_pauseparam,
	.set_pauseparam         = smac_set_pauseparam,
	.get_rx_csum		= smac_get_rx_csum,
/*	.set_rx_csum		= smac_set_rx_csum,		*/
	.get_tx_csum		= ethtool_op_get_tx_csum,
	.set_tx_csum		= ethtool_op_set_tx_hw_csum,
	.get_sg			= ethtool_op_get_sg,
/*	.set_sg			= ethtool_op_set_sg,		*/
	.get_tso		= ethtool_op_get_tso,
/*	.set_tso		= ethtool_op_set_tso,		*/
	.self_test_count	= smac_diag_test_count,
	.self_test		= smac_diag_test,
	.get_strings		= smac_get_strings,
	.get_stats_count	= smac_get_stats_count,
	.get_ethtool_stats	= smac_get_ethtool_stats,
};


/*----------------------------------------------------------------------------*/
void smac_set_ethtool_ops(struct net_device *netdev)
{
	SET_ETHTOOL_OPS(netdev, &smac_ethtool_ops);
	return;
}



#ifdef SMAC_UNIT_TEST
/*----------------------------------------------------------------------------*/
/* test smac_update_itnr() */
static int
test_smac_update_itnr(struct smac_adapter *adapter, uint64_t *data)
{
	int ret = 1;

	/* lowest_latency_range */
	if (smac_update_itnr(lowest_latency_range, 1, 8001) != bulk_latency_range) {
		ret = -lowest_latency_range;
		goto eof;
	}
	if (smac_update_itnr(lowest_latency_range, 4, 513) != low_latency_range) {
		ret = -lowest_latency_range;
		goto eof;
	}
	if (smac_update_itnr(lowest_latency_range, 5, 513) != lowest_latency_range) {
		ret = -lowest_latency_range;
		goto eof;
	}

	/* low_latency_range */
	if (smac_update_itnr(low_latency_range, 1, 10001) != bulk_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 2, 10001) != bulk_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 10, 12010) != bulk_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 36, 10001) != lowest_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 35, 10001) != low_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 2, 4002) != bulk_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 2, 511) != lowest_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}
	if (smac_update_itnr(low_latency_range, 2, 512) != low_latency_range) {
		ret = -low_latency_range;
		goto eof;
	}

	/* bulk_latency_range */
	if (smac_update_itnr(bulk_latency_range, 36, 25001) != low_latency_range) {
		ret = -bulk_latency_range;
		goto eof;
	}
	if (smac_update_itnr(bulk_latency_range, 35, 25001) != bulk_latency_range) {
		ret = -bulk_latency_range;
		goto eof;
	}
	if (smac_update_itnr(bulk_latency_range, 1, 5999) != low_latency_range) {
		ret = -bulk_latency_range;
		goto eof;
	}
	if (smac_update_itnr(bulk_latency_range, 1, 6000) != bulk_latency_range) {
		ret = -bulk_latency_range;
		goto eof;
	}

eof:
	switch (ret) {
	case -lowest_latency_range:
		*data = 1;
		SMAC_ERR_OUT("lowest_latency_range\n");
	case -low_latency_range:
		*data = 2;
		SMAC_ERR_OUT("low_latency_range\n");
	case -bulk_latency_range:
		*data = 3;
		SMAC_ERR_OUT("bulk_latency_range\n");
	default:
		/* success */
		*data = 0;
	}

	return (*data) ? FALSE : TRUE;
}


/*----------------------------------------------------------------------------*/
/* test smac_set_itnr() */
static int
test_smac_set_itnr(struct smac_adapter *adapter, uint64_t *data)
{
	*data = 0; /* success */

	/* new_itnr == adapter->itnr */
	adapter->link_speed = SPEED_1000;
	adapter->itnr_setting = SMAC_ITNR_DYNAMIC;
	adapter->current_itnr = bulk_latency_range;
	adapter->itnr = SMAC_BULK_LATENCY;
	adapter->total_rx_packets = 35;
	adapter->total_rx_bytes = 25001;
	smac_set_itnr(adapter);
	if (adapter->itnr != SMAC_BULK_LATENCY) {
		*data = 1;
		SMAC_ERR_OUT("new_itnr == adapter->itnr\n");
		return FALSE;
	}

	/* new_itnr > adapter->itnr */
	adapter->link_speed = SPEED_1000;
	adapter->itnr_setting = SMAC_ITNR_DYNAMIC;
	adapter->current_itnr = lowest_latency_range;
	adapter->itnr = SMAC_LOWEST_LATENCY;
	adapter->total_rx_packets = 1;
	adapter->total_rx_bytes = 8001;
	smac_set_itnr(adapter);
	if (adapter->itnr != SMAC_BULK_LATENCY) {
		*data = 2;
		SMAC_ERR_OUT("new_itnr > adapter->itnr\n");
		return FALSE;
	}

	/* new_itnr < adapter->itnr */
	adapter->link_speed = SPEED_1000;
	adapter->itnr_setting = SMAC_ITNR_DYNAMIC;
	adapter->current_itnr = low_latency_range;
	adapter->itnr = SMAC_LOW_LATENCY;
	adapter->total_rx_packets = 36;
	adapter->total_rx_bytes = 10001;
	smac_set_itnr(adapter);
	if (adapter->itnr != (SMAC_LOW_LATENCY - SMAC_LOWEST_LATENCY * 2)) {
		*data = 3;
		SMAC_ERR_OUT("new_itnr < adapter->itnr\n");
		return FALSE;
	}

	/* test dynamic conservative */
	adapter->link_speed = SPEED_1000;
	adapter->itnr_setting = SMAC_ITNR_DYNAMIC_CONSERVATIVE;
	adapter->current_itnr = low_latency_range;
	adapter->itnr = SMAC_LOW_LATENCY;
	adapter->total_rx_packets = 36;
	adapter->total_rx_bytes = 10001;
	smac_set_itnr(adapter);
	if (adapter->itnr != SMAC_LOW_LATENCY) {
		*data = 4;
		SMAC_ERR_OUT("dynamic conservative\n");
		return FALSE;
	}

	/* test 100base */
	adapter->link_speed = SPEED_100;
	adapter->itnr_setting = SMAC_ITNR_DYNAMIC;
	adapter->current_itnr = lowest_latency_range;
	adapter->itnr = SMAC_LOWEST_LATENCY;
	adapter->total_rx_packets = 1;
	adapter->total_rx_bytes = 8001;
	smac_set_itnr(adapter);
	if (adapter->itnr != SMAC_LOW_LATENCY) {
		*data = 5;
		SMAC_ERR_OUT("100Base\n");
		return FALSE;
	}

	return TRUE;
}

#endif /* SMAC_UNIT_TEST */
