/*
 * smac_io.c
 * hardware (smac and PHY) accessor
 *
 * Copyright 2008,2009 Sony Corporation
 *
 * This code is based on drivers/net/e1000/e1000_hw.c
 */
/*******************************************************************************

  Intel PRO/1000 Linux driver
  Copyright(c) 1999 - 2006 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  Linux NICS <linux.nics@intel.com>
  e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "smac_local.h"

static void smac_link_autoneg(struct smac_adapter *adapter);
static void smac_phy_force_speed_duplex(struct smac_adapter *adapter);

/*----------------------------------------------------------------------------*/
/* mapping hardware base address */
int
smac_map_base_address(struct smac_adapter *adapter)
{
	int err = -ENOMEM;

	SMAC_DEBUG_FUNC();

	/* SMAC register base */
	adapter->hw_addr = ioremap_nocache(adapter->hw_addr_phys, SMAC_REG_SIZE);
	if (!adapter->hw_addr) {
		goto err_smacmap;
	}

	/* Tx/Rx descriptor base */
	adapter->tx_desc_addr = dma_alloc_coherent(NULL,
						   SMAC_TX_DESC_SIZE,
						   &adapter->tx_desc_addr_phys,
						   GFP_KERNEL);
	if (!adapter->tx_desc_addr) {
		goto err_txdescmap;
	}

	adapter->rx_desc_addr = dma_alloc_coherent(NULL,
						   SMAC_RX_DESC_SIZE,
						   &adapter->rx_desc_addr_phys,
						   GFP_KERNEL);
	if (!adapter->rx_desc_addr) {
		goto err_rxdescmap;
	}

	return 0;	/* success */

/* error handling */
err_rxdescmap:
	dma_free_coherent(NULL, SMAC_TX_DESC_SIZE,
				adapter->tx_desc_addr, adapter->tx_desc_addr_phys);

err_txdescmap:
	iounmap((unsigned char __iomem *)adapter->hw_addr);

err_smacmap:
	return err;
}


/*----------------------------------------------------------------------------*/
/* unmapping hardware base address */
void
smac_unmap_base_address(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	iounmap((unsigned char __iomem *)adapter->hw_addr);

	dma_free_coherent(NULL, SMAC_TX_DESC_SIZE,
				adapter->tx_desc_addr, adapter->tx_desc_addr_phys);
	dma_free_coherent(NULL, SMAC_RX_DESC_SIZE,
				adapter->rx_desc_addr, adapter->rx_desc_addr_phys);
}


/*----------------------------------------------------------------------------*/
/* read MAC address */
void
smac_read_mac_addr(struct smac_adapter *adapter)
{
	uint32_t low, high;
	int i;

	SMAC_DEBUG_FUNC();

	low = smac_readreg(adapter, SMAC_MAC_ADDR_LOW(0));
	high = smac_readreg(adapter, SMAC_MAC_ADDR_HIGH(0));
	for ( i = 0; i < 6; i++ ) {
		if ( i < 4 ) {
			adapter->mac_addr[i] = low & 0xFF;
			low >>= 8;
		} else {
			adapter->mac_addr[i] = high & 0xFF;
			high >>= 8;
		}
	}
}


/*----------------------------------------------------------------------------*/
/* write MAC address */
void
smac_write_mac_addr(struct smac_adapter *adapter)
{
	uint32_t addr_low, addr_high;

	SMAC_DEBUG_FUNC();

	/* write MAC address in register */
	addr_low = (((uint32_t) adapter->mac_addr[0])
		  | ((uint32_t) adapter->mac_addr[1] << 8)
		  | ((uint32_t) adapter->mac_addr[2] << 16)
		  | ((uint32_t) adapter->mac_addr[3] << 24));
	addr_high = ((uint32_t) adapter->mac_addr[4]
		  | ((uint32_t) adapter->mac_addr[5] << 8));

	smac_writereg(adapter, SMAC_MAC_ADDR_HIGH(0), addr_high | SMAC_MAC_AE);
	smac_writereg(adapter, SMAC_MAC_ADDR_LOW(0), addr_low);
}


/*----------------------------------------------------------------------------*/
/* Setup auto-negotiation advertisements */
/* configure flow control settings to PHY */
static void
smac_link_autoneg(struct smac_adapter *adapter)
{
	uint16_t mii_autoneg_adv_reg;
	uint16_t mii_1000t_ctrl_reg;

	SMAC_DEBUG_FUNC();

	/* configure PHY auto-neg and flow control advertisement settings */
	mii_autoneg_adv_reg = smac_readphy(adapter, MII_ADVERTISE);
	mii_1000t_ctrl_reg = smac_readphy(adapter, MII_CTRL1000);

	/* clear all the 10/100 mb speed bits */
	mii_autoneg_adv_reg &= ~ADVERTISE_ALL;

	/* clear 1000 mb speed bits */
	mii_1000t_ctrl_reg &= ~(ADVERTISE_1000HALF |
				ADVERTISE_1000FULL);

	/* set speed and dulpex */
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_10_HALF) {
		SMAC_DEBUG_OUT("Advertise 10mb Half duplex\n");
		mii_autoneg_adv_reg |= ADVERTISE_10HALF;
	}
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_10_FULL) {
		SMAC_DEBUG_OUT("Advertise 10mb Full duplex\n");
		mii_autoneg_adv_reg |= ADVERTISE_10FULL;
	}
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_100_HALF) {
		SMAC_DEBUG_OUT("Advertise 100mb Half duplex\n");
		mii_autoneg_adv_reg |= ADVERTISE_100HALF;
	}
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_100_FULL) {
		SMAC_DEBUG_OUT("Advertise 100mb Full duplex\n");
		mii_autoneg_adv_reg |= ADVERTISE_100FULL;
	}
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_1000_HALF) {
		/* We do not allow the Phy to advertise 1000 Mb Half Duplex */
		SMAC_DEBUG_OUT("Advertise 1000mb Half duplex requested, request denied!\n");
	}
	if (adapter->autoneg_advertised & SMAC_ADVERTISE_1000_FULL) {
		SMAC_DEBUG_OUT("Advertise 1000mb Full duplex\n");
		mii_1000t_ctrl_reg |= ADVERTISE_1000FULL;
	}

	/* set flow control to PHY */
	/* The possible values of the "fc" parameter are:
	 *  SMAC_FC_NONE     :  Flow control is completely disabled
	 *  SMAC_FC_RX_PAUSE :  Rx flow control is enabled
	 *  SMAC_FC_TX_PAUSE :  Tx flow control is enabled
	 *  SMAC_FC_FULL     :  Both Rx and TX flow control are enabled
	 *  other            :  Use SMAC_FC_FULL setting
	 */
	switch (adapter->fc) {
	case SMAC_FC_NONE:
		mii_autoneg_adv_reg &= ~(ADVERTISE_PAUSE_ASYM |
					 ADVERTISE_PAUSE_CAP);
		break;
	case SMAC_FC_TX_PAUSE:
		mii_autoneg_adv_reg |= ADVERTISE_PAUSE_ASYM;
		mii_autoneg_adv_reg &= ~ADVERTISE_PAUSE_CAP;
		break;
	case SMAC_FC_RX_PAUSE:	/* fall through */
	case SMAC_FC_FULL:	/* fall through */
	default:
		mii_autoneg_adv_reg |= (ADVERTISE_PAUSE_ASYM |
					ADVERTISE_PAUSE_CAP);
		break;
	}

	/* update auto-nego setttings */
	smac_writephy(adapter, MII_ADVERTISE, mii_autoneg_adv_reg);
	smac_writephy(adapter, MII_CTRL1000, mii_1000t_ctrl_reg);

	/* restart auto-nego with new settings */
	smac_writephy(adapter, MII_BMCR, (BMCR_ANENABLE | BMCR_ANRESTART));

	/* set done fc to PHY */
	adapter->fc_set_done = FALSE;
}


/*----------------------------------------------------------------------------*/
/* Force PHY speed and duplex settings to adapter->forced_speed_duplex */
static void
smac_phy_force_speed_duplex(struct smac_adapter *adapter)
{
	uint16_t control;

	SMAC_DEBUG_FUNC();

	/* set force speed and duplex */
	control = smac_readphy(adapter, MII_BMCR);
	control &= ~BMCR_ANENABLE;
	control |= BMCR_RESET;
	switch (adapter->forced_speed_duplex) {
	case smac_100_half:
		SMAC_DEBUG_OUT("Advertise 100mb Half duplex\n");
		control |= BMCR_SPEED100;
		control &= ~(BMCR_SPEED1000 | BMCR_FULLDPLX);
		break;
	case smac_10_half:
		SMAC_DEBUG_OUT("Advertise 10mb Half duplex\n");
		control &= ~(BMCR_SPEED100 | BMCR_SPEED1000 | BMCR_FULLDPLX);
		break;
	case smac_10_full:
		SMAC_DEBUG_OUT("Advertise 10mb Full duplex\n");
		control |= BMCR_FULLDPLX;
		control &= ~(BMCR_SPEED100 | BMCR_SPEED1000);
		break;
	case smac_100_full:
	default:	/* fall through */
		SMAC_DEBUG_OUT("Advertise 100mb Full duplex\n");
		control |= (BMCR_SPEED100 | BMCR_FULLDPLX);
		control &= ~BMCR_SPEED1000;
		break;
	}
	smac_writephy(adapter, MII_BMCR, control);

	/* turn off 1000BASE-T */
	control = smac_readphy(adapter, MII_CTRL1000);
	control &= ~(ADVERTISE_1000FULL | ADVERTISE_1000HALF);
	smac_writephy(adapter, MII_CTRL1000, control);
}


/*----------------------------------------------------------------------------*/
/* Forces the MAC's flow control settings to SMAC register */
void
smac_force_mac_fc(struct smac_adapter *adapter)
{
	uint32_t control_frm = smac_readreg(adapter, SMAC_CONTROL_FRM);
	uint32_t operation = smac_readreg(adapter, SMAC_OPERATION);

	switch (adapter->fc) {
	case SMAC_FC_NONE:
		control_frm &= ~(SMAC_CF_TFE | SMAC_CF_RFE);
		SMAC_DEBUG_OUT("Flow Control = NONE\n");
		break;
	case SMAC_FC_RX_PAUSE:
		control_frm &= ~SMAC_CF_TFE;
		control_frm |= SMAC_CF_RFE;
		SMAC_DEBUG_OUT("Flow Control = RX PAUSE frames only\n");
		break;
	case SMAC_FC_TX_PAUSE:
		control_frm &= ~SMAC_CF_RFE;
		control_frm |= (SMAC_CF_TFE | SMAC_CF_PLT | SMAC_CF_PT);
		operation |= (SMAC_OP_EFC | SMAC_OP_RFA | SMAC_OP_RFD);
		SMAC_DEBUG_OUT("Flow Control = TX PAUSE frames only\n");
		break;
	case SMAC_FC_FULL:
	default:
		control_frm |= (SMAC_CF_TFE | SMAC_CF_RFE | SMAC_CF_PLT | SMAC_CF_PT);
		operation |= (SMAC_OP_EFC | SMAC_OP_RFA | SMAC_OP_RFD);
		SMAC_DEBUG_OUT("Flow Control = FULL\n");
		break;
	}

	smac_writereg(adapter, SMAC_CONTROL_FRM, control_frm);
	smac_writereg(adapter, SMAC_OPERATION, operation);

	/* set done fc to SMAC register */
	adapter->fc_set_done = TRUE;
}


/*----------------------------------------------------------------------------*/
/* Configures flow control settings to SMAC H/W after link is established */
boolean_t
smac_config_fc_after_link_up(struct smac_adapter *adapter)
{
	struct ethtool_cmd ecmd;
	uint16_t mii_status_reg;
	uint16_t mii_nway_adv_reg;
	uint16_t mii_nway_lp_ability_reg;

	SMAC_DEBUG_FUNC();

	if (adapter->fc_set_done) {
		return FALSE;
	}
	if (!mii_link_ok(&adapter->mii)) {
		return FALSE;
	}

	/* turn off Flow control if we are forcing speed and duplex */
	if (!adapter->autoneg) {
		adapter->fc = SMAC_FC_NONE;
		SMAC_DEBUG_OUT("force_mac_fc to SMAC_FC_NONE because not auto-nego\n");
		goto force_mac_fc;
	}

	/* If we auto-negotiated to HALF DUPLEX,
	 * flow control should not be enabled per IEEE 802.3 spec */
	mii_ethtool_gset(&adapter->mii, &ecmd);
	if (ecmd.duplex == DUPLEX_HALF) {
		adapter->fc = SMAC_FC_NONE;
		SMAC_DEBUG_OUT("force_mac_fc to SMAC_FC_NONE because half dulpex\n");
		goto force_mac_fc;
	}

	/* if Auto-Neg has completed, and if so,
	 * how the PHY and link partner has flow control configured */
	mii_status_reg = smac_readphy(adapter, MII_BMSR);
	if (!(mii_status_reg & BMSR_ANEGCOMPLETE)) {
		SMAC_DEBUG_OUT("Copper PHY and Auto Neg has not completed\n");
		return FALSE;
	}

	/* The AutoNeg process has completed, so we now need to
	 * read both the Auto Negotiation Advertisement Register
	 * (Address 4) and the Auto_Negotiation Base Page Ability
	 * Register (Address 5) to determine how flow control was
	 * negotiated. */
	mii_nway_adv_reg = smac_readphy(adapter, MII_ADVERTISE);
	mii_nway_lp_ability_reg = smac_readphy(adapter, MII_LPA);

	/* Two bits in the Auto Negotiation Advertisement Register
	 * (Address 4) and two bits in the Auto Negotiation Base
	 * Page Ability Register (Address 5) determine flow control
	 * for both the PHY and the link partner.  The following
	 * table, taken out of the IEEE 802.3ab/D6.0 dated March 25,
	 * 1999, describes these PAUSE resolution bits and how flow
	 * control is determined based upon these settings.
	 * NOTE:  DC = Don't Care
	 *
	 *     |   LOCAL DEVICE  |   LINK PARTNER
	 * case| PAUSE | ASM_DIR | PAUSE | ASM_DIR | NIC Resolution
	 * ----|-------|---------|-------|---------|----------------
	 *  0  |   0   |    0    |  DC   |   DC    | SMAC_FC_NONE
	 *  1  |   0   |    1    |   0   |   DC    | SMAC_FC_NONE
	 *  2  |   0   |    1    |   1   |    0    | SMAC_FC_NONE
	 *  3  |   0   |    1    |   1   |    1    | SMAC_FC_TX_PAUSE
	 *  4  |   1   |    0    |   0   |   DC    | SMAC_FC_NONE
	 *  5  |   1   |   DC    |   1   |   DC    | SMAC_FC_FULL
	 *  6  |   1   |    1    |   0   |    0    | SMAC_FC_NONE
	 *  7  |   1   |    1    |   0   |    1    | SMAC_FC_RX_PAUSE
	 */

	/* case 5: SMAC_FC_FULL */
	if ((mii_nway_adv_reg & ADVERTISE_PAUSE_CAP) &&
			(mii_nway_lp_ability_reg & LPA_PAUSE_CAP)) {
		adapter->fc = SMAC_FC_FULL;
	}
	/* case 3: SMAC_FC_TX_PAUSE */
	else if (!(mii_nway_adv_reg & ADVERTISE_PAUSE_CAP) &&
			(mii_nway_adv_reg & ADVERTISE_PAUSE_ASYM) &&
			(mii_nway_lp_ability_reg & LPA_PAUSE_CAP) &&
			(mii_nway_lp_ability_reg & LPA_PAUSE_ASYM)) {
		adapter->fc = SMAC_FC_TX_PAUSE;
	}
	/* case 7: SMAC_FC_RX_PAUSE */
	else if ((mii_nway_adv_reg & ADVERTISE_PAUSE_CAP) &&
			(mii_nway_adv_reg & ADVERTISE_PAUSE_ASYM) &&
			!(mii_nway_lp_ability_reg & LPA_PAUSE_CAP) &&
			(mii_nway_lp_ability_reg & LPA_PAUSE_ASYM)) {
		adapter->fc = SMAC_FC_RX_PAUSE;
	}
	/* the others */
	else {
		adapter->fc = SMAC_FC_NONE;
	}

force_mac_fc:
	/* set fc to SMAC register */
	smac_force_mac_fc(adapter);

	return TRUE;
}


/*----------------------------------------------------------------------------*/
/* Detects which PHY is present and setup the speed and duplex */
boolean_t
smac_setup_link(struct smac_adapter *adapter)
{
	int i;

	SMAC_DEBUG_FUNC();

	if (adapter->autoneg)
	{
		/* Setup autoneg and flow control advertisement
		 * and perform autonegotiation */
		smac_link_autoneg(adapter);
	} else {
		/* PHY will be set to 10H, 10F, 100H,or 100F
		 * depending on value from forced_speed_duplex. */
		smac_phy_force_speed_duplex(adapter);
	}

	/* Check link status. Wait up to 100 microseconds for link to become
	 * valid.
	 */
	for (i = 0; i < 10; i++) {
		if (mii_link_ok(&adapter->mii)) {
			/* config the MAC and PHY after link is up */
			if (!smac_config_fc_after_link_up(adapter)) {
				goto unable_to_establish_link;
			}
			SMAC_DEBUG_OUT("Valid link established!!!\n");
			return TRUE;
		}
		udelay(10);
	}

unable_to_establish_link:
	SMAC_DEBUG_OUT("Unable to establish link!!!\n");
	return FALSE;
}


/*----------------------------------------------------------------------------*/
/* update value from SMAC register */
void
smac_update_hw_stats(struct smac_adapter *adapter)
{
#ifdef CONFIG_SMAC_DBG
	adapter->stats.intr_stat		= smac_readreg(adapter, SMAC_INTR_STAT);
	adapter->stats.intr_enable		= smac_readreg(adapter, SMAC_INTR_ENABLE);
	adapter->stats.operation		= smac_readreg(adapter, SMAC_OPERATION);
	adapter->stats.tx_db_base		= smac_readreg(adapter, SMAC_TX_DB_BASE);
	adapter->stats.tx_db_length_minus1	= smac_readreg(adapter, SMAC_TX_DB_LENGTH_MINUS1);
	adapter->stats.tx_db_head		= smac_readreg(adapter, SMAC_TX_DB_HEAD);
	adapter->stats.tx_db_tail		= smac_readreg(adapter, SMAC_TX_DB_TAIL);
	adapter->stats.rx_db_base		= smac_readreg(adapter, SMAC_RX_DB_BASE);
	adapter->stats.rx_db_length_minus1	= smac_readreg(adapter, SMAC_RX_DB_LENGTH_MINUS1);
	adapter->stats.rx_db_head		= smac_readreg(adapter, SMAC_RX_DB_HEAD);
	adapter->stats.rx_db_tail		= smac_readreg(adapter, SMAC_RX_DB_TAIL);
	adapter->stats.bus_mode			= smac_readreg(adapter, SMAC_BUS_MODE);
	adapter->stats.rx_max_frm_length	= smac_readreg(adapter, SMAC_RX_MAX_FRM_LENGTH);
	adapter->stats.rx_frame_byte_limit	= smac_readreg(adapter, SMAC_RX_FRAME_BYTE_LIMIT);
	adapter->stats.rx_frame_count_limit	= smac_readreg(adapter, SMAC_RX_FRAME_COUNT_LIMIT);
	adapter->stats.rx_minsize_limit		= smac_readreg(adapter, SMAC_RX_MINSIZE_LIMIT);
	adapter->stats.rx_delay_timer		= smac_readreg(adapter, SMAC_RX_DELAY_TIMER);
	adapter->stats.rx_absolute_timer	= smac_readreg(adapter, SMAC_RX_ABSOLUTE_TIMER);
	adapter->stats.rx_desc_limit		= smac_readreg(adapter, SMAC_RX_DESC_LIMIT);
	adapter->stats.rxfifo_frm_cnt		= smac_readreg(adapter, SMAC_RXFIFO_FRM_CNT);
	adapter->stats.phy_polling		= smac_readreg(adapter, SMAC_PHY_POLLING);
	adapter->stats.phy_polled_data		= smac_readreg(adapter, SMAC_PHY_POLLED_DATA);
	adapter->stats.tx_desc_addr		= smac_readreg(adapter, SMAC_TX_DESC_ADDR);
	adapter->stats.rx_desc_addr		= smac_readreg(adapter, SMAC_RX_DESC_ADDR);
	adapter->stats.tx_desc0			= smac_readreg(adapter, SMAC_TX_DESC0);
	adapter->stats.tx_desc1			= smac_readreg(adapter, SMAC_TX_DESC1);
	adapter->stats.tx_desc2			= smac_readreg(adapter, SMAC_TX_DESC2);
	adapter->stats.tx_desc3			= smac_readreg(adapter, SMAC_TX_DESC3);
	adapter->stats.rx_desc0			= smac_readreg(adapter, SMAC_RX_DESC0);
	adapter->stats.rx_desc1			= smac_readreg(adapter, SMAC_RX_DESC1);
	adapter->stats.rx_desc2			= smac_readreg(adapter, SMAC_RX_DESC2);
	adapter->stats.rx_desc3			= smac_readreg(adapter, SMAC_RX_DESC3);
	adapter->stats.state_tx_rd		= smac_readreg(adapter, SMAC_STATE_TX_RD);
	adapter->stats.state_tx_gd		= smac_readreg(adapter, SMAC_STATE_TX_GD);
	adapter->stats.state_tx_gb		= smac_readreg(adapter, SMAC_STATE_TX_GB);
	adapter->stats.state_tx_wr		= smac_readreg(adapter, SMAC_STATE_TX_WR);
	adapter->stats.state_tx_wb		= smac_readreg(adapter, SMAC_STATE_TX_WB);
	adapter->stats.state_rx_rd		= smac_readreg(adapter, SMAC_STATE_RX_RD);
	adapter->stats.state_rx_gd		= smac_readreg(adapter, SMAC_STATE_RX_GD);
	adapter->stats.state_rx_wr		= smac_readreg(adapter, SMAC_STATE_RX_WR);
	adapter->stats.state_rx_wb		= smac_readreg(adapter, SMAC_STATE_RX_WB);
	adapter->stats.state_swr		= smac_readreg(adapter, SMAC_STATE_SWR);
	adapter->stats.state_ra			= smac_readreg(adapter, SMAC_STATE_RA);
	adapter->stats.state_ra_pl		= smac_readreg(adapter, SMAC_STATE_RA_PL);
	adapter->stats.hw_version		= smac_readreg(adapter, SMAC_HW_VERSION);
	adapter->stats.mac_config		= smac_readreg(adapter, SMAC_MAC_CONFIG);
	adapter->stats.mac_frame_filter		= smac_readreg(adapter, SMAC_MAC_FRAME_FILTER);
	adapter->stats.hash_table_high		= smac_readreg(adapter, SMAC_HASH_TABLE_HIGH);
	adapter->stats.hash_table_low		= smac_readreg(adapter, SMAC_HASH_TABLE_LOW);
	adapter->stats.phy_addr			= smac_readreg(adapter, SMAC_PHY_ADDR);
	adapter->stats.phy_data			= smac_readreg(adapter, SMAC_PHY_DATA);
	adapter->stats.control_frm		= smac_readreg(adapter, SMAC_CONTROL_FRM);
	adapter->stats.vlan_tag			= smac_readreg(adapter, SMAC_VLAN_TAG);
	adapter->stats.mac_version		= smac_readreg(adapter, SMAC_MAC_VERSION);
	adapter->stats.wake_up_filter		= smac_readreg(adapter, SMAC_WAKE_UP_FILTER);
	adapter->stats.power_management		= smac_readreg(adapter, SMAC_POWER_MANAGEMENT);
	adapter->stats.mac_intr_stat		= smac_readreg(adapter, SMAC_MAC_INTR_STAT);
	adapter->stats.mac_intr_mask		= smac_readreg(adapter, SMAC_MAC_INTR_MASK);
	adapter->stats.mac_addr_0_high		= smac_readreg(adapter, SMAC_MAC_ADDR_0_HIGH);
	adapter->stats.mac_addr_0_low		= smac_readreg(adapter, SMAC_MAC_ADDR_0_LOW);
	adapter->stats.rgmii_stat		= smac_readreg(adapter, SMAC_RGMII_STAT);
	adapter->stats.mmc_control		= smac_readreg(adapter, SMAC_MMC_CONTROL);
	adapter->stats.mmc_intr_rx		= smac_readreg(adapter, SMAC_MMC_INTR_RX);
	adapter->stats.mmc_intr_tx		= smac_readreg(adapter, SMAC_MMC_INTR_TX);
	adapter->stats.mmc_intr_mask_rx		= smac_readreg(adapter, SMAC_MMC_INTR_MASK_RX);
	adapter->stats.mmc_intr_mask_tx		= smac_readreg(adapter, SMAC_MMC_INTR_MASK_TX);
#endif
	adapter->stats.txoctetcount_gb		+= smac_readreg(adapter, SMAC_TXOCTETCOUNT_GB);
	adapter->stats.txframecount_gb		+= smac_readreg(adapter, SMAC_TXFRAMECOUNT_GB);
	adapter->stats.txbroadcastframes_g	+= smac_readreg(adapter, SMAC_TXBROADCASTFRAMES_G);
	adapter->stats.txmulticastframes_g	+= smac_readreg(adapter, SMAC_TXMULTICASTFRAMES_G);
	adapter->stats.tx64octets_gb		+= smac_readreg(adapter, SMAC_TX64OCTETS_GB);
	adapter->stats.tx65to127octets_gb	+= smac_readreg(adapter, SMAC_TX65TO127OCTETS_GB);
	adapter->stats.tx128to255octets_gb	+= smac_readreg(adapter, SMAC_TX128TO255OCTETS_GB);
	adapter->stats.tx256to511octets_gb	+= smac_readreg(adapter, SMAC_TX256TO511OCTETS_GB);
	adapter->stats.tx512to1023octets_gb	+= smac_readreg(adapter, SMAC_TX512TO1023OCTETS_GB);
	adapter->stats.tx1024tomaxoctets_gb	+= smac_readreg(adapter, SMAC_TX1024TOMAXOCTETS_GB);
	adapter->stats.txunicastframes_gb	+= smac_readreg(adapter, SMAC_TXUNICASTFRAMES_GB);
	adapter->stats.txmulticastframes_gb	+= smac_readreg(adapter, SMAC_TXMULTICASTFRAMES_GB);
	adapter->stats.txbroadcastframes_gb	+= smac_readreg(adapter, SMAC_TXBROADCASTFRAMES_GB);
	adapter->stats.txunderflowerror		+= smac_readreg(adapter, SMAC_TXUNDERFLOWERROR);
	adapter->stats.txsinglecol_g		+= smac_readreg(adapter, SMAC_TXSINGLECOL_G);
	adapter->stats.txmulticol_g		+= smac_readreg(adapter, SMAC_TXMULTICOL_G);
	adapter->stats.txdeferred		+= smac_readreg(adapter, SMAC_TXDEFERRED);
	adapter->stats.txlatecol		+= smac_readreg(adapter, SMAC_TXLATECOL);
	adapter->stats.txexesscol		+= smac_readreg(adapter, SMAC_TXEXESSCOL);
	adapter->stats.txcarriererror		+= smac_readreg(adapter, SMAC_TXCARRIERERROR);
	adapter->stats.txoctetcount_g		+= smac_readreg(adapter, SMAC_TXOCTETCOUNT_G);
	adapter->stats.txframecount_g		+= smac_readreg(adapter, SMAC_TXFRAMECOUNT_G);
	adapter->stats.txexcessdef		+= smac_readreg(adapter, SMAC_TXEXCESSDEF);
	adapter->stats.txpauseframes		+= smac_readreg(adapter, SMAC_TXPAUSEFRAMES);
	adapter->stats.txvlanframes_g		+= smac_readreg(adapter, SMAC_TXVLANFRAMES_G);
	adapter->stats.rxframecount_gb		+= smac_readreg(adapter, SMAC_RXFRAMECOUNT_GB);
	adapter->stats.rxoctetcount_gb		+= smac_readreg(adapter, SMAC_RXOCTETCOUNT_GB);
	adapter->stats.rxoctetcount_g		+= smac_readreg(adapter, SMAC_RXOCTETCOUNT_G);
	adapter->stats.rxbroadcastframes_g	+= smac_readreg(adapter, SMAC_RXBROADCASTFRAMES_G);
	adapter->stats.rxmulticastframes_g	+= smac_readreg(adapter, SMAC_RXMULTICASTFRAMES_G);
	adapter->stats.rxcrcerror		+= smac_readreg(adapter, SMAC_RXCRCERROR);
	adapter->stats.rxalignmenterror		+= smac_readreg(adapter, SMAC_RXALIGNMENTERROR);
	adapter->stats.rxrunterror		+= smac_readreg(adapter, SMAC_RXRUNTERROR);
	adapter->stats.rxjabbererror		+= smac_readreg(adapter, SMAC_RXJABBERERROR);
	adapter->stats.rxundersize_g		+= smac_readreg(adapter, SMAC_RXUNDERSIZE_G);
	adapter->stats.rxoversize_g		+= smac_readreg(adapter, SMAC_RXOVERSIZE_G);
	adapter->stats.rx64octets_gb		+= smac_readreg(adapter, SMAC_RX64OCTETS_GB);
	adapter->stats.rx65to127octets_gb	+= smac_readreg(adapter, SMAC_RX65TO127OCTETS_GB);
	adapter->stats.rx128to255octets_gb	+= smac_readreg(adapter, SMAC_RX128TO255OCTETS_GB);
	adapter->stats.rx256to511octets_gb	+= smac_readreg(adapter, SMAC_RX256TO511OCTETS_GB);
	adapter->stats.rx512to1023octets_gb	+= smac_readreg(adapter, SMAC_RX512TO1023OCTETS_GB);
	adapter->stats.rx1024tomaxoctets_gb	+= smac_readreg(adapter, SMAC_RX1024TOMAXOCTETS_GB);
	adapter->stats.rxunicastframes_g	+= smac_readreg(adapter, SMAC_RXUNICASTFRAMES_G);
	adapter->stats.rxlengtherror		+= smac_readreg(adapter, SMAC_RXLENGTHERROR);
	adapter->stats.rxoutofrangetype		+= smac_readreg(adapter, SMAC_RXOUTOFRANGETYPE);
	adapter->stats.rxpauseframes		+= smac_readreg(adapter, SMAC_RXPAUSEFRAMES);
	adapter->stats.rxfifooverflow		+= smac_readreg(adapter, SMAC_RXFIFOOVERFLOW);
	adapter->stats.rxvlanframes_gb		+= smac_readreg(adapter, SMAC_RXVLANFRAMES_GB);
	adapter->stats.rxwatchdogerror		+= smac_readreg(adapter, SMAC_RXWATCHDOGERROR);
	adapter->stats.rxipv4_gd_frms		+= smac_readreg(adapter, SMAC_RXIPV4_GD_FRMS);
	adapter->stats.rxipv4_hdrerr_frms	+= smac_readreg(adapter, SMAC_RXIPV4_HDRERR_FRMS);
	adapter->stats.rxipv4_nopay_frms	+= smac_readreg(adapter, SMAC_RXIPV4_NOPAY_FRMS);
	adapter->stats.rxipv4_frag_frms		+= smac_readreg(adapter, SMAC_RXIPV4_FRAG_FRMS);
	adapter->stats.rxipv4_udsbl_frms	+= smac_readreg(adapter, SMAC_RXIPV4_UDSBL_FRMS);
	adapter->stats.rxipv6_gd_frms		+= smac_readreg(adapter, SMAC_RXIPV6_GD_FRMS);
	adapter->stats.rxipv6_hdrerr_frms	+= smac_readreg(adapter, SMAC_RXIPV6_HDRERR_FRMS);
	adapter->stats.rxipv6_nopay_frms	+= smac_readreg(adapter, SMAC_RXIPV6_NOPAY_FRMS);
	adapter->stats.rxudp_gd_frms		+= smac_readreg(adapter, SMAC_RXUDP_GD_FRMS);
	adapter->stats.rxudp_err_frms		+= smac_readreg(adapter, SMAC_RXUDP_ERR_FRMS);
	adapter->stats.rxtcp_gd_frms		+= smac_readreg(adapter, SMAC_RXTCP_GD_FRMS);
	adapter->stats.rxtcp_err_frms		+= smac_readreg(adapter, SMAC_RXTCP_ERR_FRMS);
	adapter->stats.rxicmp_gd_frms		+= smac_readreg(adapter, SMAC_RXICMP_GD_FRMS);
	adapter->stats.rxicmp_err_frms		+= smac_readreg(adapter, SMAC_RXICMP_ERR_FRMS);
	adapter->stats.rxipv4_gd_octets		=  smac_readreg(adapter, SMAC_RXIPV4_GD_OCTETS);
	adapter->stats.rxipv4_hdrerr_octets	=  smac_readreg(adapter, SMAC_RXIPV4_HDRERR_OCTETS);
	adapter->stats.rxipv4_nopay_octets	=  smac_readreg(adapter, SMAC_RXIPV4_NOPAY_OCTETS);
	adapter->stats.rxipv4_frag_octets	=  smac_readreg(adapter, SMAC_RXIPV4_FRAG_OCTETS);
	adapter->stats.rxipv4_udsel_octets	=  smac_readreg(adapter, SMAC_RXIPV4_UDSEL_OCTETS);
	adapter->stats.rxipv6_gd_octets		=  smac_readreg(adapter, SMAC_RXIPV6_GD_OCTETS);
	adapter->stats.rxipv6_hdrerr_octets	=  smac_readreg(adapter, SMAC_RXIPV6_HDRERR_OCTETS);
	adapter->stats.rxipv6_nopay_octets	=  smac_readreg(adapter, SMAC_RXIPV6_NOPAY_OCTETS);
	adapter->stats.rxudp_gd_octets		=  smac_readreg(adapter, SMAC_RXUDP_GD_OCTETS);
	adapter->stats.rxudp_err_octets		=  smac_readreg(adapter, SMAC_RXUDP_ERR_OCTETS);
	adapter->stats.rxtcp_gd_octets		=  smac_readreg(adapter, SMAC_RXTCP_GD_OCTETS);
	adapter->stats.rxtcp_err_octets		=  smac_readreg(adapter, SMAC_RXTCP_ERR_OCTETS);
	adapter->stats.rxicmp_gd_octets		=  smac_readreg(adapter, SMAC_RXICMP_GD_OCTETS);
	adapter->stats.rxicmp_err_octets	=  smac_readreg(adapter, SMAC_RXICMP_ERR_OCTETS);
}


/***********
 * hardware accessor
 ***********/
/*----------------------------------------------------------------------------*/
/* read value from SMAC register */
uint32_t
smac_readreg(struct smac_adapter *adapter, unsigned long adr)
{
	adr += (unsigned long)adapter->hw_addr;
	return smac_in32(adr);
}


/*----------------------------------------------------------------------------*/
/* write value to SMAC register */
void
smac_writereg(struct smac_adapter *adapter, unsigned long adr, uint32_t data)
{
	adr += (unsigned long)adapter->hw_addr;
	smac_out32(adr, data);
}


/*----------------------------------------------------------------------------*/
/* wait phy work done */
#define MDIO_MAX_LOOP   2000
#define MDIO_LOOP_WAIT  100

static void
smac_wait_phy(struct smac_adapter *adapter)
{
	int i;

	/* wait phy work done */
	for (i = MDIO_MAX_LOOP; i > 0; i--) {
		if (!(smac_readreg(adapter, SMAC_PHY_ADDR) & SMAC_PA_M_GB))
			break;
		udelay(MDIO_LOOP_WAIT);
	}
	if (i == 0) {
		SMAC_ERR_OUT("timeout - something is wrong...\n");
	}
}


/*----------------------------------------------------------------------------*/
/* mdio operations for net_device */
int
smac_mdio_read(struct net_device *netdev, int phy_id, int location)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	uint8_t mcr = adapter->platform_data->phy_mcr;
	unsigned long flags;
	int val;

	spin_lock_irqsave(&adapter->phy_lock, flags);

	smac_wait_phy(adapter);
	smac_writereg(adapter, SMAC_PHY_ADDR,
		      ((phy_id   << SMAC_PA_S_PA)  & SMAC_PA_M_PA)  |
		      ((location << SMAC_PA_S_GR)  & SMAC_PA_M_GR)  |
		      ((mcr      << SMAC_PA_S_MCR) & SMAC_PA_M_MCR) |
		      SMAC_PA_READ | SMAC_PA_KICK);
	smac_wait_phy(adapter);
	val = smac_readreg(adapter, SMAC_PHY_DATA);

	spin_unlock_irqrestore(&adapter->phy_lock, flags);

	return val;
}

void
smac_mdio_write(struct net_device *netdev, int phy_id, int location, int val)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	uint8_t mcr = adapter->platform_data->phy_mcr;
	unsigned long flags;

	spin_lock_irqsave(&adapter->phy_lock, flags);

	smac_writereg(adapter, SMAC_PHY_DATA, val);

	smac_wait_phy(adapter);
	smac_writereg(adapter, SMAC_PHY_ADDR,
		      ((phy_id   << SMAC_PA_S_PA)  & SMAC_PA_M_PA)  |
		      ((location << SMAC_PA_S_GR)  & SMAC_PA_M_GR)  |
		      ((mcr      << SMAC_PA_S_MCR) & SMAC_PA_M_MCR) |
		      SMAC_PA_WRITE | SMAC_PA_KICK);
	smac_wait_phy(adapter);

	spin_unlock_irqrestore(&adapter->phy_lock, flags);

	return;
}


/*----------------------------------------------------------------------------*/
/* read value from PHY (SMAC_PHY_DATA) */
uint16_t
smac_readphy(struct smac_adapter *adapter, uint16_t reg)
{
	return smac_mdio_read(adapter->netdev, adapter->mii.phy_id, reg);
}


/*----------------------------------------------------------------------------*/
/* write value to PHY (SMAC_PHY_DATA) */
void
smac_writephy(struct smac_adapter *adapter, uint16_t reg, uint16_t data)
{
	smac_mdio_write(adapter->netdev, adapter->mii.phy_id, reg, data);
}


/*----------------------------------------------------------------------------*/
/* read data from descriptor */
uint32_t
smac_read_desc(void* addr)
{
	return smac_desc_to_cpu(*(uint32_t*)addr);
}


/*----------------------------------------------------------------------------*/
/* write data to descriptor */
void
smac_write_desc(uint32_t val, void* addr)
{
	*(uint32_t*)addr = smac_cpu_to_desc(val);
}


