/*
 * smac_main.c
 * smac (gigabit ethernet) driver
 *
 * Copyright 2008,2009 Sony Corporation
 *
 * This code is based on drivers/net/e1000/e1000_main.c
 */
/*******************************************************************************

  Intel PRO/1000 Linux driver
  Copyright(c) 1999 - 2006 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  Linux NICS <linux.nics@intel.com>
  e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "smac_local.h"

/*------------------------------------------------------------------------------
  Global Variables
  ------------------------------------------------------------------------------*/
#define DRV_NAME "smac"

#ifdef CONFIG_SMAC_ENABLE_NAPI
#define DRIVER_NAPI "-NAPI"
#else
#define DRIVER_NAPI
#endif

#ifdef CONFIG_SMAC_ENABLE_JUMBOFRAME
#define DRIVER_JUMBO "-JUMBO"
#else
#define DRIVER_JUMBO
#endif

#ifdef CONFIG_SMAC_DBG
#define DRIVER_DBG "-DEBUG"
#else
#define DRIVER_DBG
#endif

#define DRV_VERSION "1.02" DRIVER_NAPI DRIVER_JUMBO DRIVER_DBG

char smac_driver_name[] = DRV_NAME;
char smac_driver_version[] = DRV_VERSION;

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("SMAC Gigabit Ethernet Driver");
MODULE_LICENSE("GPL v2");
MODULE_VERSION(DRV_VERSION);

/* module parameter */
/* Network interface message level settings (include/linux/netdevice.h)
 * DRV       <= 1
 * PROBE     <= 2
 * LINK      <= 3
 * RX_ERR    <= 7
 * TX_ERR    <= 8
 * TX_QUEUED <= 9
 * HW        <= 14
 */
static int smac_debug = NETIF_MSG_DRV | NETIF_MSG_PROBE;
module_param(smac_debug, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(smac_debug, "Debug Level (0=none,...,16=all)");


/*------------------------------------------------------------------------------
  Prototype (Driver Method)
  ------------------------------------------------------------------------------*/
static int smac_init_module(void);
static void smac_exit_module(void);
static int smac_open(struct net_device *netdev);
static int smac_close(struct net_device *netdev);
static int smac_xmit_frame(struct sk_buff *skb, struct net_device *netdev);
static struct net_device_stats *smac_get_stats(struct net_device *netdev);
static void smac_set_multi(struct net_device *netdev);
static int smac_set_mac(struct net_device *netdev, void *p);
static int smac_change_mtu(struct net_device *netdev, int new_mtu);
static void smac_tx_timeout(struct net_device *dev);
void smac_set_ethtool_ops(struct net_device *netdev);
#ifdef CONFIG_NET_POLL_CONTROLLER
/* for netdump / net console */
static void smac_netpoll (struct net_device *netdev);
#endif


/*------------------------------------------------------------------------------
  Prototype (Internal Function)
  ------------------------------------------------------------------------------*/
/* descriptor */
static void smac_unmap_and_free_tx_resource(struct smac_adapter *adapter, struct smac_buffer *buffer_info);
static int smac_setup_tx_resources(struct smac_adapter *adapter, struct smac_tx_ring *txdr);
static int smac_setup_rx_resources(struct smac_adapter *adapter, struct smac_rx_ring *rxdr);
static void smac_free_tx_resources(struct smac_adapter *adapter, struct smac_tx_ring *tx_ring);
static void smac_free_rx_resources(struct smac_adapter *adapter, struct smac_rx_ring *rx_ring);
static void smac_configure_tx(struct smac_adapter *adapter);
static void smac_configure_rx(struct smac_adapter *adapter);
static void smac_clean_tx_ring(struct smac_adapter *adapter, struct smac_tx_ring *tx_ring);
static void smac_clean_rx_ring(struct smac_adapter *adapter, struct smac_rx_ring *rx_ring);
static int smac_alloc_queues(struct smac_adapter *adapter);
static void smac_free_queues(struct smac_adapter *adapter);
static int __smac_maybe_stop_tx(struct net_device *netdev, int size);
static int smac_maybe_stop_tx(struct net_device *netdev, struct smac_tx_ring *tx_ring, int size);
static int smac_tx_map(struct smac_adapter *adapter, struct smac_tx_ring *tx_ring,
			struct sk_buff *skb, unsigned int first,
			unsigned int max_per_txd, unsigned int nr_frags, uint32_t tx_flags);
static void smac_tx_queue(struct smac_adapter *adapter, struct smac_tx_ring *tx_ring, uint32_t tx_flags, int count);
static boolean_t smac_clean_tx(struct smac_adapter *adapter, struct smac_tx_ring *tx_ring);
#ifdef CONFIG_SMAC_ENABLE_NAPI
static int smac_clean(struct napi_struct *napi, int budget);
static boolean_t smac_clean_rx(struct smac_adapter *adapter, struct smac_rx_ring *rx_ring,
				int *work_done, int work_to_do);
#else
static boolean_t smac_clean_rx(struct smac_adapter *adapter, struct smac_rx_ring *rx_ring);
#endif

/* checksum offload */
static uint32_t smac_tx_csum(struct sk_buff *skb);

/* socket buffer */
static void smac_alloc_rx_buffers(struct smac_adapter *adapter, struct smac_rx_ring *rx_ring, int cleaned_count);
static void smac_rx_checksum(struct smac_adapter *adapter, uint32_t status_err, struct sk_buff *skb);

/* watchdog - check for link status */
static void smac_watchdog(unsigned long data);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
static void smac_reset_task(struct net_device *dev);
#else
static void smac_reset_task(struct work_struct *work);
#endif

/* interrupt */
static int smac_request_irq(struct smac_adapter *adapter);
static void smac_free_irq(struct smac_adapter *adapter);
static void smac_irq_disable(struct smac_adapter *adapter);
static void smac_irq_enable(struct smac_adapter *adapter);
static uint32_t smac_enable_irq_bit(struct smac_adapter *adapter);

/* update the dynamic ITNR value based on statistics */
#ifndef SMAC_UNIT_TEST
static
#endif
smac_latency_range_type smac_update_itnr(smac_latency_range_type itnr_setting, int packets, int bytes);
#ifndef SMAC_UNIT_TEST
static
#endif
void smac_set_itnr(struct smac_adapter *adapter);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
static irqreturn_t smac_intr(int irq, void *data, struct pt_regs *);
#else
static irqreturn_t smac_intr(int irq, void *data);
#endif

/* SMAC hardware */
static int smac_configure_mac(struct smac_adapter *adapter);
static int smac_init_busmode(struct smac_adapter *adapte);
static int smac_sw_init(struct smac_adapter *adapter);
static void smac_set_mac_addr(struct smac_adapter *adapter);


/*------------------------------------------------------------------------------
  Inline Functions
  ------------------------------------------------------------------------------*/
/* debug print */
#ifdef SMAC_PRINT_RX_SOCKET
static void smac_print_rx_skb(struct sk_buff *skb, uint32_t length);
#else
#define smac_print_rx_skb(a,b)
#endif

#ifdef SMAC_PRINT_TX_SOCKET
static void smac_print_tx_skb(struct sk_buff *skb, uint32_t length, uint32_t offset);
#else
#define smac_print_tx_skb(a,b,c)
#endif

#ifdef SMAC_PRINT_TX_FLAGMENT
static void smac_print_tx_flagment(struct skb_frag_struct *frag, uint32_t length, uint32_t offset);
#else
#define smac_print_tx_flagment(a,b,c)
#endif

static void smac_print_mac_address(struct smac_adapter *adapter, int i);
#ifdef SMAC_PRINT_MAC_ADDRES
static void smac_print_mac_addresses(struct smac_adapter *adapter);
#else
#define smac_print_mac_addresses()
#endif


/*------------------------------------------------------------------------------
  Functions
  ------------------------------------------------------------------------------*/
static const struct net_device_ops smac_netdev_ops = {
	.ndo_open		= smac_open,
	.ndo_stop		= smac_close,
	.ndo_start_xmit		= smac_xmit_frame,
	.ndo_get_stats		= smac_get_stats,
	.ndo_set_multicast_list	= smac_set_multi,
	.ndo_set_mac_address	= smac_set_mac,
	.ndo_tx_timeout 	= smac_tx_timeout,
	.ndo_change_mtu		= smac_change_mtu,
	.ndo_validate_addr	= eth_validate_addr,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller	= smac_netpoll,
#endif
};

/*
 * probe/remove
 */
static int smac_probe(struct platform_device *pdev)
{
	struct net_device *netdev;
	struct smac_adapter *adapter;
	struct resource *res;
	int err;

	SMAC_DEBUG_FUNC();

	/* allocate netdev */
	netdev = alloc_etherdev(sizeof(struct smac_adapter));
	if (!netdev) {
		err = -ENOMEM;
		goto err_alloc_etherdev;
	}

	/* set driver data */
	platform_set_drvdata(pdev, netdev);

	/* set private data */
	adapter = netdev_priv(netdev);
	adapter->netdev = netdev;
	adapter->pdev = pdev;
	adapter->platform_data = pdev->dev.platform_data;

	/* get resources */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "smac_regs");
	if (!res) {
		err = -ENODEV;
		goto err_ioremap;
	}
	res = request_mem_region(res->start, res->end - res->start + 1,
				 "SMAC");
	if (!res) {
		err = -EBUSY;
		goto err_ioremap;
	}
	adapter->hw_addr_phys = res->start;
 	adapter->irq = platform_get_irq_byname(pdev, "smac_irq");

	/* ioremap smac register base and descriptor */
	err = smac_map_base_address(adapter);
	if (err < 0) {
		goto err_ioremap;
	}

	/* set netdev members */
	netdev->netdev_ops = &smac_netdev_ops;
	smac_set_ethtool_ops(netdev);
	netdev->watchdog_timeo		= 10 * HZ;
#ifdef CONFIG_SMAC_ENABLE_NAPI
	netif_napi_add(netdev, &adapter->napi, smac_clean, SMAC_DESC_WEIGHT);
#endif
	netdev->features		= NETIF_F_SG | NETIF_F_HW_CSUM;

	/* setup the private structure */
	err = smac_sw_init(adapter);
	if (err < 0) {
		goto err_sw_init;
	}

	/* initialize H/W */
	smac_reset(adapter);

	/* set mac address */
	smac_configure_mac(adapter);

	/* copy the MAC address */
	memcpy(netdev->dev_addr, adapter->mac_addr, netdev->addr_len);
	if (!is_valid_ether_addr(netdev->dev_addr)) {
		SMAC_DPRINTK(PROBE, ERR, "Invalid MAC Address\n");
		smac_print_mac_address(adapter, 0);
		err = -EADDRNOTAVAIL;
		goto err_mac_address;
	}

	/* watchdog timer */
	init_timer(&adapter->watchdog_timer);
	adapter->watchdog_timer.function = smac_watchdog;
	adapter->watchdog_timer.data = (unsigned long) adapter;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
	INIT_WORK(&adapter->reset_task, (void (*)(void *))smac_reset_task, netdev);
#else
	INIT_WORK(&adapter->reset_task, smac_reset_task);
#endif

	/* set module param */
	smac_check_options(adapter);
	smac_print_options(adapter);

	/* register netdev */
	strncpy(netdev->name, "eth%d", IFNAMSIZ);
	if ((err = register_netdev(netdev))) {
		goto err_register;
	}

	/* tell the stack to leave us alone until smac_open() is called */
	netif_carrier_off(netdev);
	netif_stop_queue(netdev);

	/* show information */
	SMAC_DPRINTK(PROBE, INFO, "SMAC Network Connection (%s)\n", netdev->name);
	SMAC_DPRINTK(PROBE, INFO, "MAC Address: %02X:%02X:%02X:%02X:%02X:%02X\n",
		     adapter->mac_addr[0], adapter->mac_addr[1],
		     adapter->mac_addr[2], adapter->mac_addr[3],
		     adapter->mac_addr[4], adapter->mac_addr[5]);

	return 0;	/* success */

	/* error handling */
err_register:
err_mac_address:
	smac_free_queues(adapter);

err_sw_init:
	smac_unmap_base_address(adapter);

err_ioremap:
	free_netdev(netdev);

err_alloc_etherdev:
	SMAC_ERR_OUT("cannot initialize module\n");

	return err;	/* error */
}

static int smac_remove(struct platform_device *pdev)
{
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct smac_adapter *adapter = netdev_priv(netdev);
	struct resource *res;

	SMAC_DEBUG_FUNC();

	flush_scheduled_work();
	unregister_netdev(netdev);
	smac_free_queues(adapter);
	smac_unmap_base_address(adapter);
	free_netdev(netdev);

	/* release resources */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "smac_regs");
	release_mem_region(res->start, res->end - res->start + 1);

	return 0;
}

#ifdef CONFIG_PM
static int smac_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct smac_adapter *adapter = netdev_priv(netdev);

	if (!netif_running(netdev)) {
		return 0;
	}
	netif_stop_queue(netdev);
	netif_device_detach(netdev);
	smac_down(adapter);

	return 0;
}

static int smac_resume(struct platform_device *pdev)
{
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct smac_adapter *adapter = netdev_priv(netdev);

	smac_reset(adapter);
	if (!netif_running(netdev)) {
		return 0;
	}
	smac_up(adapter);
	netif_device_attach(netdev);
	netif_wake_queue(netdev);

	return 0;
}
#endif /* CONFIG_PM */

/*
 * init/exit
 */
static struct platform_driver smac_driver = {
	.probe		= smac_probe,
	.remove		= smac_remove,
#ifdef CONFIG_PM
	.suspend	= smac_suspend,
	.resume		= smac_resume,
#endif
	.driver		= {
		.name	= "smac",
	}
};

static int __init smac_init_module(void)
{
	return platform_driver_register(&smac_driver);
}

static void __exit smac_exit_module(void)
{
	platform_driver_unregister(&smac_driver);
}

module_init(smac_init_module);
module_exit(smac_exit_module);


/*
 * Driver Methods
 */
/*----------------------------------------------------------------------------*/
/* net_device.open() */
/* allocate descriptor and start interface */
static int smac_open(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	int err;

	SMAC_DEBUG_FUNC();

	/* disallow open during test */
	if (test_bit(__SMAC_TESTING, (void*)&adapter->flags))
		return -EBUSY;

	/* print version */
	SMAC_DPRINTK(PROBE, DEBUG, "%s version: %s\n", smac_driver_name, smac_driver_version);
	{
		uint32_t hw_version = smac_readreg(adapter, SMAC_HW_VERSION);
		uint8_t *h = (uint8_t *) &hw_version;
		SMAC_DPRINTK(PROBE, DEBUG,
				"hw version: %d.%d.%d.%d\n",
				h[3], h[2], h[1], h[0]);
	}

	/* allocate transmit descriptors */
	err = smac_setup_tx_resources(adapter, adapter->tx_ring);
	if (err < 0) {
		goto err_setup_tx;
	}

	/* allocate receive descriptors */
	err = smac_setup_rx_resources(adapter, adapter->rx_ring);
	if (err < 0) {
		goto err_setup_rx;
	}

	/* request irq */
	err = smac_request_irq(adapter);
	if (err) {
		goto err_req_irq;
	}

	err = smac_up(adapter);
	if (err < 0) {
		goto err_up;
	}

	return 0;

err_up:
	smac_free_irq(adapter);
err_req_irq:
	smac_free_rx_resources(adapter, adapter->rx_ring);
err_setup_rx:
	smac_free_tx_resources(adapter, adapter->tx_ring);
err_setup_tx:
	smac_reset(adapter);

	return err;
}

/*----------------------------------------------------------------------------*/
/* net_device.close() */
/* stop interface and free descriptor, irq-handler */
static int smac_close(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	SMAC_DEBUG_FUNC();

	WARN_ON(test_bit(__SMAC_RESETTING, (void *) &adapter->flags));
	smac_down(adapter);
	smac_free_irq(adapter);

	smac_free_tx_resources(adapter, adapter->tx_ring);
	smac_free_rx_resources(adapter, adapter->rx_ring);

	return 0;
}

/*----------------------------------------------------------------------------*/
/* net_device.hard_start_xmit() */
/* transmit packet */
#define SMAC_MAX_TXD_PWR	13 /* TxFIFO size = 8KB = 2^13 */
#define SMAC_MAX_DATA_PER_TXD	(1<<SMAC_MAX_TXD_PWR) - 1
#define TXD_USE_COUNT(S, X) (((S) >> (X)) + 1 )

static int smac_xmit_frame(struct sk_buff *skb, struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	struct smac_tx_ring *tx_ring;
	unsigned int first, max_per_txd = SMAC_MAX_DATA_PER_TXD;
	unsigned int max_txd_pwr = SMAC_MAX_TXD_PWR;
	uint32_t tx_flags = 0;
	unsigned int len = skb->len;
	unsigned long lock_flag;
	unsigned int nr_frags = 0;
	int count = 0;
	unsigned int f;
	int i;

	SMAC_DEBUG_FUNC();

	len -= skb->data_len;
	tx_ring = adapter->tx_ring;

	/* free skb */
	if (unlikely(skb->len <= 0)) {
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}

	/* count descriptors for queueing */
	count += TXD_USE_COUNT(len, max_txd_pwr);

	/* count skb fragment */
	nr_frags = skb_shinfo(skb)->nr_frags;
	for (f = 0; f < nr_frags; f++) {
		count += TXD_USE_COUNT(skb_shinfo(skb)->frags[f].size, max_txd_pwr);
	}

	if (!spin_trylock_irqsave(&tx_ring->tx_lock, lock_flag))
		/* Collision - tell upper layer to requeue */
		return NETDEV_TX_LOCKED;

	/* need: count + 2 desc gap to keep tail from touching
	 * head, otherwise try next time */
	if (unlikely(smac_maybe_stop_tx(netdev, tx_ring, count + 2))) {
		spin_unlock_irqrestore(&tx_ring->tx_lock, lock_flag);
		return NETDEV_TX_BUSY;
	}

	/* first descriptor to map */
	first = tx_ring->next_to_use;

	/* set tx checksum offload */
	tx_flags |= smac_tx_csum(skb);

	/* make descriptor */
	smac_tx_queue(adapter, tx_ring, tx_flags,
		      smac_tx_map(adapter, tx_ring, skb, first,
				  max_per_txd, nr_frags, tx_flags));

	/* set jiffies for timeout */
	netdev->trans_start = jiffies;

	/* clean tx descriptors */
	for (i = 0; i < SMAC_MAX_INTR; i++)
		if (unlikely(!smac_clean_tx(adapter, adapter->tx_ring)))
			break;

	spin_unlock_irqrestore(&tx_ring->tx_lock, lock_flag);

	return NETDEV_TX_OK;
}


/*----------------------------------------------------------------------------*/
/* net_device.get_stats() */
/* get statistics */
static struct net_device_stats *smac_get_stats(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	SMAC_DEBUG_FUNC();

	/* only return the current stats */
	return &adapter->net_stats;
}


/*----------------------------------------------------------------------------*/
/* net_device.set_mac_address() */
/* set MAC address */
static int smac_set_mac(struct net_device *netdev, void *p)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	struct sockaddr *addr = p;

	SMAC_DEBUG_FUNC();

	/* validate MAC address */
	if (!is_valid_ether_addr((uint8_t *) addr->sa_data)) {
		return -EADDRNOTAVAIL;
	}

	/* copy MAC address to struct member */
	memcpy(netdev->dev_addr, addr->sa_data, netdev->addr_len);
	SMAC_ASSERT(netdev->addr_len == sizeof(adapter->mac_addr));
	memcpy(adapter->mac_addr, addr->sa_data, netdev->addr_len);

	/* write MAC address */
	smac_write_mac_addr(adapter);

	return 0;
}

/*----------------------------------------------------------------------------*/
/* net_device.change_mtu() */
/* change MTU */
static int smac_change_mtu(struct net_device *netdev, int new_mtu)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	/* don't include ethernet FCS because hardware appends/strips */
	int max_frame = new_mtu + ENET_HEADER_SIZE + VLAN_HLEN;

	SMAC_DEBUG_FUNC();

	/* check MTU size (upper and lower limit) */
	if ((max_frame < MINIMUM_ETHERNET_FRAME_SIZE) ||
	    (max_frame > MAX_JUMBO_FRAME_SIZE)) {
		SMAC_DPRINTK(PROBE, ERR, "Invalid MTU setting\n");
		return -EINVAL;
	}

	/* set rx_buffer_len */
	if (max_frame <= SMAC_RXBUFFER_256) {
		adapter->rx_buffer_len = SMAC_RXBUFFER_256;
	} else if (max_frame <= SMAC_RXBUFFER_512) {
		adapter->rx_buffer_len = SMAC_RXBUFFER_512;
	} else if (max_frame <= SMAC_RXBUFFER_1024) {
		adapter->rx_buffer_len = SMAC_RXBUFFER_1024;
	} else if (max_frame <= SMAC_RXBUFFER_2048) {
		adapter->rx_buffer_len = SMAC_RXBUFFER_2048;
	} else {
		/* SMAC HW RxFIFO: max 4096byte (JumboFrame) */
		adapter->rx_buffer_len = SMAC_RXBUFFER_4096;
	}

	/* change mtu */
	netdev->mtu = new_mtu;

	/* re-init */
	if (netif_running(netdev)) {
		smac_reinit_locked(adapter);
	}

	return 0;
}

/*----------------------------------------------------------------------------*/
/* net_device.tx_timeout() */
/* xmit timeout */
static void smac_tx_timeout(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	SMAC_DEBUG_FUNC();

	adapter->tx_timeout_count++;
	schedule_work(&adapter->reset_task);

	return;
}

#ifdef CONFIG_SMAC_ENABLE_NAPI
/*----------------------------------------------------------------------------*/
/* net_device.poll() */
/* clean up rx/tx descriptor */
static int smac_clean(struct napi_struct *napi, int budget)
{
	struct smac_adapter *adapter = container_of(napi, struct smac_adapter, napi);
	int tx_cleaned = 0;	/* true if # of processed tx desc < SMAC_TX_WEIGHT */
	int work_done = 0;	/* number of processed rx packets */

	SMAC_DEBUG_FUNC();

	/* processed tx packets */
	tx_cleaned = smac_clean_tx(adapter, adapter->tx_ring);

	/* processed rx packets */
	smac_clean_rx(adapter, adapter->rx_ring, &work_done, budget);

	/* If no Tx and not enough Rx work done, exit the polling mode */
	if ((!tx_cleaned && (work_done == 0))) {
		if (likely(adapter->itnr_setting & 3))
			smac_set_itnr(adapter);
		netif_rx_complete(napi);
		smac_irq_enable(adapter);
	}

	return work_done;
}
#endif


/******************************************************************************/
/* Internal Func							      */
/******************************************************************************/
/*----------------------------------------------------------------------------*/
/* set SMAC_MAC_CONFIG and MAC address */
static int smac_configure_mac(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	/*
	 * MAC config
	 * set MII/RMII, Full Duplex as default
	 */
	smac_writereg(adapter, SMAC_MAC_CONFIG,
		  SMAC_MC_WD		/* disable rx watchdog */
		| SMAC_MC_JD 		/* disable tx jaber timer */
#ifdef CONFIG_SMAC_ENABLE_JUMBOFRAME
		| SMAC_MC_JE		/* enable jumbo frame */
#endif
		| SMAC_MC_PS_MII	/* MII/RMII */
		| SMAC_MC_FES_100	/* 100Mbps */
		| SMAC_MC_DM_FULL	/* full duplex */
		| SMAC_MC_TE 		/* enable tx */
		| SMAC_MC_RE  		/* enable rx */
		| SMAC_MC_DFS_ENA	/* remove FCS */
		| SMAC_MC_RCS		/* enable rx checksum offload */
		);

	/* set smac_adapter.mac_addr */
	smac_set_mac_addr(adapter);

	/* write MAC address */
	smac_write_mac_addr(adapter);

	return 0;
}


/*----------------------------------------------------------------------------*/
/* run SMAC Driver */
int smac_up(struct smac_adapter *adapter)
{
	struct net_device *netdev = adapter->netdev;
	struct smac_rx_ring *ring = adapter->rx_ring;

	SMAC_DEBUG_FUNC();

	/* set mac address */
	smac_configure_mac(adapter);

	/* set multicast address */
	smac_set_multi(netdev);

	/* configure tx descriptor */
	smac_configure_tx(adapter);
	/* configure rx descriptor */
	smac_configure_rx(adapter);

#ifdef SMAC_USE_PHY_POLLING
	/* polling PHY SMAC_PHY_SPEC_STATUS */
	smac_writereg(adapter, SMAC_PHY_POLLING,
		      (adapter->mii.phy_id << SMAC_PP_S_PA) |
		      (MII_BMSR            << SMAC_PP_S_GR) |
		      SMAC_PP_PE);
#endif

	/* set socket buffer
	 * SMAC_DESC_UNUSED is not-used-descriptor.
	 * max of descriptors here.
	 */
	smac_alloc_rx_buffers(adapter, ring, SMAC_DESC_UNUSED(ring));

	/* save tx queue length */
	adapter->tx_queue_len = netdev->tx_queue_len;

#ifdef CONFIG_SMAC_ENABLE_NAPI
	napi_enable(&adapter->napi);
#endif
	smac_irq_enable(adapter);

#ifdef SMAC_USE_PHY_POLLING
	clear_bit(__SMAC_DOWN, (void *) &adapter->flags);
#endif

	/* set up auto-nego, speed, duplex, flow control */
	smac_setup_link(adapter);

	/* start watchdog */
	smac_watchdog((unsigned long) adapter);

	return 0;
}


/*----------------------------------------------------------------------------*/
/* stop SMAC Driver */
void smac_down(struct smac_adapter *adapter)
{
	struct net_device *netdev = adapter->netdev;

	SMAC_DEBUG_FUNC();

#ifdef SMAC_USE_PHY_POLLING
	/* signal that we're down so the interrupt handler does not
	 * reschedule our watchdog timer */
	set_bit(__SMAC_DOWN, (void *) &adapter->flags);
#endif

	/* disable IRQ */
	smac_irq_disable(adapter);

	/* delete timer */
	del_timer_sync(&adapter->watchdog_timer);

#ifdef CONFIG_SMAC_ENABLE_NAPI
	/* stop poll */
	napi_disable(&adapter->napi);
#endif

	/* save queue length */
	netdev->tx_queue_len = adapter->tx_queue_len;

	/* update stat info */
	adapter->link_speed = 0;
	adapter->link_duplex = 0;

	/* stop */
	netif_carrier_off(netdev);
	netif_stop_queue(netdev);

	/* SW reset */
	smac_reset(adapter);

	/* clear tx/rx buffer */
	smac_clean_tx_ring(adapter, adapter->tx_ring);
	smac_clean_rx_ring(adapter, adapter->rx_ring);

	return;
}

/*----------------------------------------------------------------------------*/
#define RESET_MAX_LOOP		2000
#define RESET_LOOP_WAIT		100

void smac_reset(struct smac_adapter *adapter)
{
	volatile uint32_t busmode = 0;
	int i;

	SMAC_DEBUG_FUNC();

	/* SW reset */
	busmode = smac_readreg(adapter, SMAC_BUS_MODE);
	smac_writereg(adapter, SMAC_BUS_MODE, busmode | SMAC_BM_SWR);

	/* wait reset is done */
	for (i = RESET_MAX_LOOP; i > 0; i--) {
		busmode = smac_readreg(adapter, SMAC_BUS_MODE);
		SMAC_DPRINTK(HW, DEBUG, "SMAC_BUS_MODE=0x%08X\n", busmode);
		SMAC_DPRINTK(HW, DEBUG, "SMAC_STATE_SWR=0x%08X\n",
			     smac_readreg(adapter, SMAC_STATE_SWR));
		if (!(busmode & SMAC_BM_SWR))
			break;
		udelay(RESET_LOOP_WAIT);
	}
	if (i == 0) {
		SMAC_ERR_OUT("timeout - something is wrong...\n");
	}

	/* clear interrupt */
	smac_writereg(adapter, SMAC_INTR_CLEAR, ~0);

	/* initialize SMAC_BUS_MODE */
	smac_init_busmode(adapter);

	/* initialize SMAC_OPERATION */
	smac_writereg(adapter, SMAC_OPERATION, SMAC_OP_TSF | SMAC_OP_RSF);

	/* initialize SMAC_REVISE */
	smac_writereg(adapter, SMAC_REVISE, SMAC_REVISE_SRV);

	/* clear stat counter */
	smac_writereg(adapter, SMAC_MMC_CONTROL, SMAC_MMC_CR);

	return;
}


#ifdef CONFIG_NET_POLL_CONTROLLER
/*
 * Polling 'interrupt' - used by things like netconsole to send skbs
 * without having to re-enable interrupts. It's not called while
 * the interrupt routine is executing.
 */
static void
smac_netpoll(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	smac_irq_disable(adapter);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18))
	smac_intr(adapter->irq, netdev, NULL);
#else
	smac_intr(adapter->irq, netdev);
#endif

#if 0	/* unnecessary */
	smac_clean_tx(adapter, adapter->tx_ring);
#ifndef CONFIG_SMAC_ENABLE_NAPI
	smac_clean_rx(adapter, adapter->rx_ring);
#endif
#endif
	smac_irq_enable(adapter);
}
#endif


/*----------------------------------------------------------------------------*/
static void smac_unmap_and_free_tx_resource(struct smac_adapter *adapter,
					    struct smac_buffer *buffer_info)
{
	struct platform_device *pdev = adapter->pdev;

	SMAC_DEBUG_FUNC();

	if (buffer_info->dma) {
		dma_unmap_single(&pdev->dev, buffer_info->dma,
				 buffer_info->length, DMA_TO_DEVICE);
		buffer_info->dma = 0;
	}
	if (buffer_info->skb) {
		dev_kfree_skb_any(buffer_info->skb);
		buffer_info->skb = NULL;
	}
	/* buffer_info must be completely set up in the transmit path */
}

/*----------------------------------------------------------------------------*/
/* setup tx descriptor */
static int smac_setup_tx_resources(struct smac_adapter *adapter,
				   struct smac_tx_ring *txdr)
{
	size_t size;

	SMAC_DEBUG_FUNC();

	/* allocate struct smac_buffer for tx */
	size = sizeof(struct smac_buffer) * txdr->count;
	txdr->buffer_info = kmalloc(size, GFP_KERNEL);
	if (!txdr->buffer_info) {
		SMAC_DPRINTK(TX_ERR, ERR,
			"Unable to allocate memory for the transmit descriptor ring\n");
		return -ENOMEM;
	}
	memset(txdr->buffer_info, 0, size);

	/* txd ring max size (max: 256 * 16byte = 4096byte) */
	txdr->size = txdr->count * sizeof(struct smac_desc);

	/* set tx descriptor base address */
	txdr->desc = adapter->tx_desc_addr;	/* virtual */
	txdr->dma = adapter->tx_desc_addr_phys;	/* physical */
	if (!txdr->desc || !txdr->dma) {
		kfree(txdr->buffer_info);
		SMAC_DPRINTK(TX_ERR, ERR,
			"Unable to allocate memory for the transmit descriptor ring\n");
		return -ENOMEM;
	}
	memset(txdr->desc, 0, txdr->size);

	/* set next_to_use and next_to_clean to head */
	txdr->next_to_use = 0;
	txdr->next_to_clean = 0;

	spin_lock_init(&txdr->tx_lock);

	return 0;
}


/*----------------------------------------------------------------------------*/
/* setup rx descriptor */
static int smac_setup_rx_resources(struct smac_adapter *adapter,
				   struct smac_rx_ring *rxdr)
{
	size_t size;

 	SMAC_DEBUG_FUNC();

	/* allocate struct smac_buffer for rx */
	size = sizeof(struct smac_buffer) * rxdr->count;
	rxdr->buffer_info = kmalloc(size, GFP_KERNEL);
	if (!rxdr->buffer_info) {
		SMAC_DPRINTK(RX_ERR, ERR,
			"Unable to allocate memory for the receive descriptor ring\n");
		return -ENOMEM;
	}
	memset(rxdr->buffer_info, 0, size);

	/* rxd ring size (max: 256 * 16byte = 4096byte) */
	rxdr->size = rxdr->count * sizeof(struct smac_desc);

	/* set rx descriptor base address */
	rxdr->desc = adapter->rx_desc_addr;	/* virtual */
	rxdr->dma = adapter->rx_desc_addr_phys;	/* physical */
	if (!rxdr->desc || !rxdr->dma) {
		SMAC_DPRINTK(RX_ERR, ERR,
			"Unable to allocate memory for the receive descriptor ring\n");
		kfree(rxdr->buffer_info);
		return -ENOMEM;
	}
	memset(rxdr->desc, 0, rxdr->size);

	/* set next_to_clean and next_to_use to head */
	rxdr->next_to_clean = 0;
	rxdr->next_to_use = 0;

	return 0;
}

/*----------------------------------------------------------------------------*/
/* free tx struct smac_buffer */
static void smac_free_tx_resources(struct smac_adapter *adapter,
				   struct smac_tx_ring *tx_ring)
{
	SMAC_DEBUG_FUNC();

	smac_clean_tx_ring(adapter, tx_ring);
	kfree(tx_ring->buffer_info);
	tx_ring->buffer_info = NULL;
	tx_ring->desc = NULL;

	return;
}

/*----------------------------------------------------------------------------*/
/* free rx struct smac_buffer */
static void smac_free_rx_resources(struct smac_adapter *adapter,
				   struct smac_rx_ring *rx_ring)
{
	SMAC_DEBUG_FUNC();

	smac_clean_rx_ring(adapter, rx_ring);
	kfree(rx_ring->buffer_info);
	rx_ring->buffer_info = NULL;
	rx_ring->desc = NULL;

	return;
}


/*----------------------------------------------------------------------------*/
/* update struct net_device_stats */
void smac_update_stats(struct smac_adapter *adapter)
{
	unsigned long flags;

	SMAC_DEBUG_FUNC();

	/*
	 * Prevent stats update while adapter is being reset
	 */
	if (adapter->link_speed == 0) {
		return;
	}

	spin_lock_irqsave(&adapter->stats_lock, flags);

	/* update struct smac_hw_stats */
	smac_update_hw_stats(adapter);

	/* read frame dropp error */
#if 0	/* not implemented */
	adapter->total_tx_dropped	+=
	adapter->total_rx_dropped	+=
#endif

	/* Fill out the OS statistics structure */
	adapter->net_stats.multicast	= smac_readreg(adapter, SMAC_TXMULTICASTFRAMES_G);
#if 0	/* not implemented */
	adapter->net_stats.collisions	= smac_readreg();
#endif
	/* Rx Errors */
	adapter->net_stats.rx_dropped		= adapter->total_rx_dropped;
	adapter->net_stats.rx_length_errors	= smac_readreg(adapter, SMAC_RXLENGTHERROR);
	adapter->net_stats.rx_crc_errors	= smac_readreg(adapter, SMAC_RXCRCERROR);
#if 0	/* not implemented */
	adapter->net_stats.rx_errors		=
	adapter->net_stats.rx_frame_errors	=
	adapter->net_stats.rx_missed_error	=
#endif

	/* Tx Errors */
	adapter->net_stats.tx_dropped		= adapter->total_tx_dropped;
#if 0	/* not implemented */
	adapter->net_stats.tx_errors		=
	adapter->net_stats.tx_aborted_errors	=
	adapter->net_stats.tx_window_errors	=
#endif

	spin_unlock_irqrestore(&adapter->stats_lock, flags);

	return;
}


/*----------------------------------------------------------------------------*/
/* allocate struct smac_{tx,rx}_ring */
static int smac_alloc_queues(struct smac_adapter *adapter)
{
	size_t size;

	SMAC_DEBUG_FUNC();

	/* allocate tx descriptor ring */
	size = sizeof(struct smac_tx_ring);
	adapter->tx_ring = kmalloc(size, GFP_KERNEL);
	if (!adapter->tx_ring) {
		return -ENOMEM;
	}
	memset(adapter->tx_ring, 0, size);

	/* allocate rx descriptor ring */
	size = sizeof(struct smac_rx_ring);
	adapter->rx_ring = kmalloc(size, GFP_KERNEL);
	if (!adapter->rx_ring) {
		kfree(adapter->tx_ring);
		return -ENOMEM;
	}
	memset(adapter->rx_ring, 0, size);

	return 0;
}


/*----------------------------------------------------------------------------*/
/* free struct smac_{tx,rx}_ring */
static void smac_free_queues(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	/* free tx/rx descriptor */
	kfree(adapter->tx_ring);
	kfree(adapter->rx_ring);
}


/*----------------------------------------------------------------------------*/
/* configure tx descriptor */
/* Setup the SMAC_TX_DB_BASE, SMAC_TX_DB_LENGTH_MINUS1 register */
static void smac_configure_tx(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	/* tx descriptor base address */
	smac_writereg(adapter, SMAC_TX_DB_BASE, adapter->tx_desc_addr_phys);

	/* # of tx descriptor */
	smac_writereg(adapter, SMAC_TX_DB_LENGTH_MINUS1, SMAC_MAX_TXD - 1);

	return;
}

/*----------------------------------------------------------------------------*/
/* configure rx descriptor and receive interrupt */
/* Setup the SMAC_RX_DB_BASE, SMAC_RX_DB_LENGTH_MINUS1 register */
static void smac_configure_rx(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	/* rx descriptor base address */
	smac_writereg(adapter, SMAC_RX_DB_BASE, adapter->rx_desc_addr_phys);

	/* # of rx descriptor */
	smac_writereg(adapter, SMAC_RX_DB_LENGTH_MINUS1, SMAC_MAX_RXD - 1);

	/* max size of received data */
	smac_writereg(adapter, SMAC_RX_MAX_FRM_LENGTH, adapter->rx_buffer_len);

	/* set the receive delay timer */
	smac_writereg(adapter, SMAC_RX_DELAY_TIMER, adapter->rx_int_delay);

	/* set receive absolute interrupt delay */
	smac_writereg(adapter, SMAC_RX_ABSOLUTE_TIMER, adapter->rx_abs_int_delay);

	/* set frame count limit */
	if (adapter->itnr_setting != SMAC_ITNR_OFF) {
		smac_writereg(adapter, SMAC_RX_FRAME_COUNT_LIMIT, adapter->itnr);
	}

	return;
}

/*----------------------------------------------------------------------------*/
/* clean tx buffer */
static void smac_clean_tx_ring(struct smac_adapter *adapter,
			       struct smac_tx_ring *tx_ring)
{
	struct smac_buffer *buffer_info;
	unsigned long size;
	unsigned int i;

	SMAC_DEBUG_FUNC();

	/* Free all the Tx ring sk_buffs */
	for (i = 0; i < tx_ring->count; i++) {
		buffer_info = &tx_ring->buffer_info[i];
		smac_unmap_and_free_tx_resource(adapter, buffer_info);
	}

	size = sizeof(struct smac_buffer) * tx_ring->count;
	memset(tx_ring->buffer_info, 0, size);

	/* Zero out the descriptor ring */
	memset(tx_ring->desc, 0, tx_ring->size);

	tx_ring->next_to_use = 0;
	tx_ring->next_to_clean = 0;

	return;
}

/*----------------------------------------------------------------------------*/
/* clean rx buffer */
static void smac_clean_rx_ring(struct smac_adapter *adapter,
			       struct smac_rx_ring *rx_ring)
{
	struct platform_device *pdev = adapter->pdev;
	struct smac_buffer *buffer_info;
	unsigned long size;
	unsigned int i;

	SMAC_DEBUG_FUNC();

	/* Free all the Rx ring sk_buffs */
	for (i = 0; i < rx_ring->count; i++) {
		buffer_info = &rx_ring->buffer_info[i];
		if (buffer_info->dma) {
			dma_unmap_single(&pdev->dev, buffer_info->dma,
					 buffer_info->length, DMA_FROM_DEVICE);
			buffer_info->dma = 0;
		}
		if (buffer_info->skb) {
			dev_kfree_skb(buffer_info->skb);
			buffer_info->skb = NULL;
		}
	}

	size = sizeof(struct smac_buffer) * rx_ring->count;
	memset(rx_ring->buffer_info, 0, size);

	/* Zero out the descriptor ring */
	memset(rx_ring->desc, 0, rx_ring->size);

	rx_ring->next_to_clean = 0;
	rx_ring->next_to_use = 0;

	return;
}

/*----------------------------------------------------------------------------*/
/* set promiscuos mode, multicast address */
static void smac_set_multi(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev);

	SMAC_DEBUG_FUNC();

	/* set SMAC_MAC_FRAME_FILTER */
	{
		uint32_t framefilter = smac_readreg(adapter, SMAC_MAC_FRAME_FILTER);

		if (netdev->flags & IFF_PROMISC) {
			framefilter |= SMAC_MFF_PR;
		} else if (netdev->flags & IFF_ALLMULTI) {
			framefilter |= SMAC_MFF_PM;
			framefilter &= ~SMAC_MFF_PR;
		} else {
			framefilter &= ~(SMAC_MFF_PR | SMAC_MFF_PM);
		}

		smac_writereg(adapter, SMAC_MAC_FRAME_FILTER, framefilter);
	}

	/**
	 * read 16 multicast address, and copy.
	 * not use #0 (#0 is myself)
	 */
	{
		struct dev_mc_list *mc_ptr = netdev->mc_list;
		int i;

		for (i = 1; i < SMAC_MULTICAST_ADDR_NUM; i++) {
			if (mc_ptr) {
				const uint32_t addr_low = (((uint32_t) mc_ptr->dmi_addr[0])
					| ((uint32_t) mc_ptr->dmi_addr[1] << 8)
					| ((uint32_t) mc_ptr->dmi_addr[2] << 16)
					| ((uint32_t) mc_ptr->dmi_addr[3] << 24));
				const uint32_t addr_high = ((uint32_t) mc_ptr->dmi_addr[4]
					| ((uint32_t) mc_ptr->dmi_addr[5] << 8));

				smac_writereg(adapter, SMAC_MAC_ADDR_HIGH(i), addr_high | SMAC_MAC_AE);
				smac_writereg(adapter, SMAC_MAC_ADDR_LOW(i), addr_low);

				mc_ptr = mc_ptr->next;
			} else {
				smac_writereg(adapter, SMAC_MAC_ADDR_HIGH(i), 0xffff);
				smac_writereg(adapter, SMAC_MAC_ADDR_LOW(i), ~0);
			}
		}
	}

	smac_print_mac_addresses();	/* if use, define SMAC_PRINT_MAC_ADDR */

	return;
}


/*----------------------------------------------------------------------------*/
/* start timer for monitoring link status  */
static void smac_watchdog(unsigned long data)
{
	struct smac_adapter *adapter = (struct smac_adapter *) data;
	struct net_device *netdev = adapter->netdev;
	struct smac_tx_ring *txdr = adapter->tx_ring;
	struct ethtool_cmd ecmd;

	SMAC_DEBUG_FUNC();

	/* configure flow control after link up */
	smac_config_fc_after_link_up(adapter);

	/* modify mismatch link status between PHY and Kernel */
	if (mii_link_ok(&adapter->mii)) {
		if (!netif_carrier_ok(netdev)) {	/* Kernel link down */
			uint32_t config;

			/* get link up status */
			mii_ethtool_gset(&adapter->mii, &ecmd);
			adapter->link_speed = ecmd.speed;
			if (ecmd.duplex == DUPLEX_FULL) {
				adapter->link_duplex = SMAC_FULL_DUPLEX;
			} else {
				adapter->link_duplex = SMAC_HALF_DUPLEX;
			}

			/* print link up status */
			SMAC_DPRINTK(LINK, INFO,
				"NIC Link is Up %d Mbps %s, "
				"Flow Control: %s\n",
				adapter->link_speed,
				(adapter->link_duplex == SMAC_FULL_DUPLEX) ?
				"Full Duplex" : "Half Duplex",
				(adapter->fc == SMAC_FC_FULL)     ? "RX/TX" :
				(adapter->fc == SMAC_FC_RX_PAUSE) ? "RX"    :
				(adapter->fc == SMAC_FC_TX_PAUSE) ? "TX"    :
				"None");

			/* set MAC config */
			config = smac_readreg(adapter, SMAC_MAC_CONFIG);
			/* Port Select */
			if (adapter->link_speed == SPEED_1000) {
				config &= ~SMAC_MC_PS_MII;
			} else {
				config |= SMAC_MC_PS_MII;
			}
			/* Fast Ethernet */
			if (adapter->link_speed == SPEED_10) {
				config &= ~SMAC_MC_FES_100;
			} else {
				config |= SMAC_MC_FES_100;
			}
			/* Duplex Mode */
			if (adapter->link_duplex == SMAC_FULL_DUPLEX) {
				config |= SMAC_MC_DM_FULL;
			} else {
				config &= ~SMAC_MC_DM_FULL;
				config &= ~SMAC_MC_M_IFG;
				config |= SMAC_MC_IFG_64BIT;
			}
			smac_writereg(adapter, SMAC_MAC_CONFIG, config);

			/* set netdev.tx_queue_len */
			netdev->tx_queue_len = adapter->tx_queue_len;
			switch (adapter->link_speed) {
			case SPEED_10:
				netdev->tx_queue_len = 10;
				break;
			case SPEED_100:
				netdev->tx_queue_len = 100;
				/* maybe add some timeout factor ? */
				break;
			}

			netif_carrier_on(netdev);
			netif_wake_queue(netdev);
		}
	} else {	/* PHY link down */
		if (netif_carrier_ok(netdev)) {	/* Kernel link up */
			adapter->link_speed = 0;
			adapter->link_duplex = 0;
			adapter->fc_set_done = FALSE;
			SMAC_DPRINTK(LINK, INFO, "NIC Link is Down\n");

			netif_carrier_off(netdev);
			netif_stop_queue(netdev);
		}
	}
	smac_update_stats(adapter);

	if (!netif_carrier_ok(netdev)) {
		if (SMAC_DESC_UNUSED(txdr) + 1 < txdr->count) {
			/* We've lost link, so the controller stops DMA,
			 * but we've got queued Tx work that's never going
			 * to get done, so reset controller to flush Tx.
			 * (Do the reset outside of interrupt context). */
			adapter->tx_timeout_count++;
			schedule_work(&adapter->reset_task);
		}
	}

	/* reset timer */
	mod_timer(&adapter->watchdog_timer, jiffies + 2 * HZ);

	return;
}


/*----------------------------------------------------------------------------*/
/* update the dynamic ITNR value based on statistics */
#ifndef SMAC_UNIT_TEST
static
#endif
smac_latency_range_type
smac_update_itnr(smac_latency_range_type current_itnr, int packets, int bytes)
{
	smac_latency_range_type retval = current_itnr;

	if (packets == 0)
		goto update_itnr_done;

	switch (current_itnr) {
	case lowest_latency_range:
		/* jumbo frames get bulk treatment*/
		if (bytes/packets > 8000)
			retval = bulk_latency_range;
		else if ((packets < 5) && (bytes > 512))
			retval = low_latency_range;
		break;
	case low_latency_range:
		if (bytes > 10000) {
			/* jumbo frames need bulk latency setting */
			if (bytes/packets > 8000)
				retval = bulk_latency_range;
			else if ((packets < 10) || ((bytes/packets) > 1200))
				retval = bulk_latency_range;
			else if ((packets > 35))
				retval = lowest_latency_range;
		} else if (bytes/packets > 2000)
			retval = bulk_latency_range;
		else if (packets <= 2 && bytes < 512)
			retval = lowest_latency_range;
		break;
	case bulk_latency_range:
		if (bytes > 25000) {
			if (packets > 35)
				retval = low_latency_range;
		} else if (bytes < 6000) {
			retval = low_latency_range;
		}
		break;
	}

update_itnr_done:
	return retval;
}


/*----------------------------------------------------------------------------*/
/* set the dynamic ITNR value based on statistics */
#ifndef SMAC_UNIT_TEST
static
#endif
void smac_set_itnr(struct smac_adapter *adapter)
{
	uint32_t new_itnr;

	/* for non-gigabit speeds, just fix low_latency setting */
	if (unlikely(adapter->link_speed != SPEED_1000)) {
		new_itnr = SMAC_LOW_LATENCY;
		goto set_itnr_now;
	}

	adapter->current_itnr = smac_update_itnr(adapter->current_itnr,
						adapter->total_rx_packets,
						adapter->total_rx_bytes);

	/* conservative mode eliminates the lowest_latency setting */
	if (adapter->itnr_setting == SMAC_ITNR_DYNAMIC_CONSERVATIVE
		&& adapter->current_itnr == lowest_latency_range)
		adapter->current_itnr = low_latency_range;

	switch (adapter->current_itnr) {
	/* counts and packets in update_itnr are dependent on these numbers */
	case lowest_latency_range:
		new_itnr = SMAC_LOWEST_LATENCY;
		break;
	case low_latency_range:
		new_itnr = SMAC_LOW_LATENCY;
		break;
	case bulk_latency_range:
		new_itnr = SMAC_BULK_LATENCY;
		break;
	default:
		new_itnr = adapter->itnr;
		break;
	}

set_itnr_now:
	if (new_itnr != adapter->itnr) {
		if (new_itnr < adapter->itnr) {
			/* this attempts to bias the interrupt number towards Bulk
			 * by adding intermediate steps when interrupt number is
			 * increasing */
			new_itnr = max(adapter->itnr - (new_itnr << 1), new_itnr);
		}
		adapter->itnr = new_itnr;
		smac_writereg(adapter, SMAC_RX_FRAME_COUNT_LIMIT, new_itnr);
	}

	return;
}


/*----------------------------------------------------------------------------*/
/* print interrupt factor (for debug) */
#ifdef SMAC_PRINT_INTERRUPT_FACTOR
void smac_print_interrupt_factor(void)
{
	uint32_t icr = smac_readreg(adapter, SMAC_INTR_STAT);
	uint32_t icre = smac_readreg(adapter, SMAC_INTR_ENABLE);

	printk("[%lu] ", jiffies);
	printk("interrupt_factor: ");

	if ((icr & SMAC_INTR_RFC) && (icre & SMAC_INTR_RFC))
		printk("SMAC_INTR_RFC ");
	if ((icr & SMAC_INTR_RDTA) && (icre & SMAC_INTR_RDTA))
		printk("SMAC_INTR_RDTA ");
	if ((icr & SMAC_INTR_RDT) && (icre & SMAC_INTR_RDT))
		printk("SMAC_INTR_RDT ");
	if ((icr & SMAC_INTR_RDL) && (icre & SMAC_INTR_RDL))
		printk("SMAC_INTR_RDL ");
	if ((icr & SMAC_INTR_RU) && (icre & SMAC_INTR_RU))
		printk("SMAC_INTR_RU ");
	if ((icr & SMAC_INTR_RI) && (icre & SMAC_INTR_RI))
		printk("SMAC_INTR_RI ");
	if ((icr & SMAC_INTR_PSUD) && (icre & SMAC_INTR_PSUD))
		printk("SMAC_INTR_PSUD ");

	printk("\n");
}
#else
#define smac_print_interrupt_factor()
#endif


/*----------------------------------------------------------------------------*/
/* interrupt handler */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18))
static irqreturn_t smac_intr(int irq, void *data, struct pt_regs *unused)
#else
static irqreturn_t smac_intr(int irq, void *data)
#endif

{
	struct net_device *netdev = data;
	struct smac_adapter *adapter = netdev_priv(netdev);
	uint32_t icr;

	SMAC_DEBUG_FUNC();

	/* read interrupt status bit */
	icr = smac_readreg(adapter, SMAC_INTR_STAT);
	if (unlikely(!icr)) {
		/* Not our interrupt */
		return IRQ_NONE;
	}

	smac_print_interrupt_factor();	/* for debug */

#ifdef SMAC_USE_PHY_POLLING
	/* reset watchdog if update PHY status */
	if (unlikely(icr & SMAC_INTR_PSUD)) {
		if (!test_bit(__SMAC_DOWN, (void *) &adapter->flags)) {
			mod_timer(&adapter->watchdog_timer, jiffies + 1);
		}
	}
#endif

	/* interrupt clear */
	smac_writereg(adapter, SMAC_INTR_CLEAR, ~0);

	/* clean rx and tx descriptor */
#ifdef CONFIG_SMAC_ENABLE_NAPI
	if(likely(netif_rx_schedule_prep(&adapter->napi))) {
		smac_irq_disable(adapter);
		adapter->total_tx_bytes = 0;
		adapter->total_rx_bytes = 0;
		adapter->total_tx_packets = 0;
		adapter->total_rx_packets = 0;

		__netif_rx_schedule(&adapter->napi);
	}
#else
	{
		int i;

		adapter->total_tx_bytes = 0;
		adapter->total_rx_bytes = 0;
		adapter->total_tx_packets = 0;
		adapter->total_rx_packets = 0;

		for (i = 0; i < SMAC_MAX_INTR; i++) {
			if (unlikely(!smac_clean_rx(adapter, adapter->rx_ring) &
				     !smac_clean_tx(adapter, adapter->tx_ring))) {
				/* rx desc cleaned at least one && tx desc clean max */
				break;
			}
		}

		if (likely(adapter->itnr_setting & 3))
			smac_set_itnr(adapter);
	}
#endif

	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------------*/
static int smac_request_irq(struct smac_adapter *adapter)
{
	int err = 0;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18))
	int flags = SA_SHIRQ;
#else
	int flags = IRQF_SHARED;
#endif
	struct net_device *netdev = adapter->netdev;

	SMAC_DEBUG_FUNC();

	SMAC_DPRINTK(PROBE, DEBUG, "irq:%d\n", adapter->irq);
	err = request_irq(adapter->irq, &smac_intr, flags, netdev->name, netdev);
	if (err < 0) {
		SMAC_DPRINTK(PROBE, ERR, "Unable to allocate interrupt Error: %d\n", err);
	}

	return err;
}


/*----------------------------------------------------------------------------*/
static void smac_free_irq(struct smac_adapter *adapter)
{
	struct net_device *netdev = adapter->netdev;

	SMAC_DEBUG_FUNC();

	free_irq(adapter->irq, netdev);
}


/*----------------------------------------------------------------------------*/
/* disable IRQ */
static void smac_irq_disable(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	/* disable IRQ */
	smac_writereg(adapter, SMAC_INTR_DISABLE, ~0);
}


/*----------------------------------------------------------------------------*/
/* get IRQ enable bit */
static uint32_t smac_enable_irq_bit(struct smac_adapter *adapter)
{
	uint32_t intr = 0;

	/* enable IRQ */
	SMAC_ASSERT(adapter->itnr == smac_readreg(adapter, SMAC_RX_FRAME_COUNT_LIMIT));
	if (adapter->itnr)
		intr |= SMAC_INTR_RFC;

	SMAC_ASSERT(adapter->rx_abs_int_delay == smac_readreg(adapter, SMAC_RX_ABSOLUTE_TIMER));
	if (adapter->rx_abs_int_delay)
		intr |= SMAC_INTR_RDTA;

	SMAC_ASSERT(adapter->rx_int_delay == smac_readreg(adapter, SMAC_RX_DELAY_TIMER));
	if (adapter->rx_int_delay)
		intr |= SMAC_INTR_RDT;

	if (!intr)
		intr |= SMAC_INTR_RI;

	return intr;
}


/*----------------------------------------------------------------------------*/
/* enable IRQ */
static void smac_irq_enable(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

#ifndef SMAC_USE_PHY_POLLING
	smac_writereg(adapter, SMAC_INTR_ENABLE, smac_enable_irq_bit(adapter) | SMAC_INTR_RU);
#else
	smac_writereg(adapter, SMAC_INTR_ENABLE, smac_enable_irq_bit(adapter)
					| SMAC_INTR_RU
					| SMAC_INTR_PSUD);
#endif
}


/*----------------------------------------------------------------------------*/
/* clean transmitted descriptor */
static boolean_t smac_clean_tx(struct smac_adapter *adapter,
			  struct smac_tx_ring *tx_ring)
{
	struct net_device *netdev = adapter->netdev;
	struct smac_desc *tx_desc, *eop_desc;
	struct smac_buffer *buffer_info;
	unsigned int i, eop;
	boolean_t cleaned = FALSE;
	unsigned int total_tx_bytes = 0;
	unsigned int total_tx_packets = 0;
#ifdef CONFIG_SMAC_ENABLE_NAPI
	unsigned int count = 0;
#endif

	SMAC_DEBUG_FUNC();

	i = tx_ring->next_to_clean;	/* start of clean */
	eop = tx_ring->buffer_info[i].next_to_watch;	/* end of packet */
	eop_desc = SMAC_TX_DESC(*tx_ring, eop);

	rmb();
	while (smac_read_desc((void *) &eop_desc->desc0) & SMAC_STDD_M_WB) {
		/* to ensure that we don't read descriptor before WB check */
		rmb();

		/* clean one packet */
		for (cleaned = FALSE; !cleaned;) {
			tx_desc = SMAC_TX_DESC(*tx_ring, i);
			buffer_info = &tx_ring->buffer_info[i];
			cleaned = (i == eop);

			if (cleaned) {	/*  cleand one packet */
				struct sk_buff *skb = buffer_info->skb;
				total_tx_packets++;
				total_tx_bytes += skb->len;
			}

			/* free skb */
			smac_unmap_and_free_tx_resource(adapter, buffer_info);
			smac_write_desc(smac_read_desc((void *) &tx_desc->desc0) &
					~SMAC_STDD_M_WB, (void *) &tx_desc->desc0);

			/* set next_to_watch to next_to_clean */
			buffer_info->next_to_watch = i;
			if (unlikely(++i == tx_ring->count)) {
				i = 0;
			}
		}

		eop = tx_ring->buffer_info[i].next_to_watch;	/* end of next packet */
		eop_desc = SMAC_TX_DESC(*tx_ring, eop);

#ifdef CONFIG_SMAC_ENABLE_NAPI
		/* weight of a sort for tx, to avoid endless transmit cleanup */
		if (count++ == SMAC_TX_WEIGHT) {
			break;
		}
#endif
	}
	wmb();

	tx_ring->next_to_clean = i;	/* update next_to_clean */

	/*
	 * restart tx queue
	 * if "tx queue is stop" && "not-used descriptor >= threthold"
	 */
	if (unlikely(cleaned && netif_carrier_ok(netdev) &&
		     SMAC_DESC_UNUSED(tx_ring) >= SMAC_TX_WAKE_THRESHOLD)) {
		/* Make sure that anybody stopping the queue after this
		 * sees the new next_to_clean.
		 */
		smp_mb();
		if (netif_queue_stopped(netdev)) {
			netif_wake_queue(netdev);
			++adapter->restart_queue;
		}
	}

	adapter->total_tx_bytes += total_tx_bytes;
	adapter->total_tx_packets += total_tx_packets;
	adapter->net_stats.tx_bytes += total_tx_bytes;
	adapter->net_stats.tx_packets += total_tx_packets;

	return cleaned;
}

/*----------------------------------------------------------------------------*/
/* clean received descriptor */
#ifdef CONFIG_SMAC_ENABLE_NAPI
static boolean_t smac_clean_rx(struct smac_adapter *adapter,
			  struct smac_rx_ring *rx_ring,
			  int *work_done,	/* # of processed packets */
			  int work_to_do)	/* max of processing packets */
#else
static boolean_t smac_clean_rx(struct smac_adapter *adapter,
			  struct smac_rx_ring *rx_ring)
#endif
{
	struct platform_device *pdev = adapter->pdev;
	struct net_device *netdev = adapter->netdev;
	struct smac_desc *rx_desc, *next_rxd;
	struct smac_buffer *buffer_info, *next_buffer;
	uint32_t length;
	unsigned int i;
	int cleaned_count = 0;
	boolean_t cleaned = FALSE;
	unsigned int total_rx_bytes = 0, total_rx_packets = 0;

	SMAC_DEBUG_FUNC();

	/* get rx descriptor to clean */
	i = rx_ring->next_to_clean;	/* start of clean */
	rx_desc = SMAC_RX_DESC(*rx_ring, i);
	buffer_info = &rx_ring->buffer_info[i];

	/* check writebacked rx descriptors */
	rmb();
	while (smac_read_desc((void *) &rx_desc->desc0) & SMAC_SRD_M_WB) {
		struct sk_buff *skb;

		/* to ensure that we don't read descriptor before WB check */
		rmb();

#ifdef CONFIG_SMAC_ENABLE_NAPI
		if (*work_done >= work_to_do) {
			break;
		}
		(*work_done)++;
#endif

		/* unmap rx buffer */
		dma_unmap_single(&pdev->dev, buffer_info->dma,
				 buffer_info->length, DMA_FROM_DEVICE);
		buffer_info->dma = 0;

		skb = buffer_info->skb;
		buffer_info->skb = NULL;

		prefetch(skb->data - NET_IP_ALIGN);

		/* wrap around */
		if (++i == rx_ring->count) {
			i = 0;
		}
		/* get next descriptor */
		next_rxd = SMAC_RX_DESC(*rx_ring, i);

		prefetch(next_rxd);
		next_buffer = &rx_ring->buffer_info[i];

		cleaned = TRUE;
		cleaned_count++;

		/* rx data length */
		length = smac_read_desc((void *) &rx_desc->desc1) >> 16;
		length &= 0x3FFF;
		/* check rx packet's error */
		if (unlikely(smac_read_desc((void *) &rx_desc->desc0) & SMAC_SRD_M_ES)) {
			SMAC_DPRINTK(RX_ERR, ERR, "rx error!! (rx_desc=0x%08X)\n", rx_desc->desc0);
			/* recycle */
			buffer_info->skb = skb;
			goto next_desc;
		}

		/* probably a little skewed due to removing CRC */
		total_rx_bytes += length;
		total_rx_packets++;

		skb_put(skb, length);
		smac_print_rx_skb(skb, length);	/* for debug */

		/* receive checksum offload */
		smac_rx_checksum(adapter, smac_read_desc((void *) &rx_desc->desc0), skb);

		skb->dev = netdev;
		skb->protocol = eth_type_trans(skb, netdev);

#ifdef CONFIG_SMAC_ENABLE_NAPI
		/* feed packets to the kernel */
		netif_receive_skb(skb);
#else
		netif_rx(skb);
#endif
		netdev->last_rx = jiffies;

next_desc:
		/* clear descriptor WB */
		smac_write_desc(smac_read_desc((void *) &rx_desc->desc0) & ~SMAC_SRD_M_WB,
				(void *) &rx_desc->desc0);

		/* return some buffers to hardware, one at a time is too slow */
		/* re-allocate in groups if cleaned_count >= SMAC_RX_BUFFER_WRITE */
		if (unlikely(cleaned_count >= SMAC_RX_BUFFER_WRITE)) {
			smac_alloc_rx_buffers(adapter, rx_ring, cleaned_count);
			cleaned_count = 0;
		}

		/* use prefetched values */
		rx_desc = next_rxd;
		buffer_info = next_buffer;

	}
	wmb();

	rx_ring->next_to_clean = i;	/* update next_to_clean */

	/* allocate rx buffer if there is unused desc */
	cleaned_count = SMAC_DESC_UNUSED(rx_ring);
	if (cleaned_count) {
		smac_alloc_rx_buffers(adapter, rx_ring, cleaned_count);
	}

	adapter->total_rx_bytes += total_rx_bytes;
	adapter->total_rx_packets += total_rx_packets;
	adapter->net_stats.rx_bytes += total_rx_bytes;
	adapter->net_stats.rx_packets += total_rx_packets;

	return cleaned;
}


/*----------------------------------------------------------------------------*/
/* checksum received packet */
#define CHECKSUM_IS_NOT_SET		(!(smac_readreg(adapter, SMAC_MAC_CONFIG) & SMAC_MC_RCS))
#define IP_CHECKSUM_IS_ERROR(s)		( ((s) & SMAC_SRD_M_ICSE) &&  ((s) & SMAC_SRD_M_ICS))
#define TCP_UDP_CHECKSUM_IS_ERROR(s)	( ((s) & SMAC_SRD_M_TCSE) &&  ((s) & SMAC_SRD_M_TCS))
#define CHECKSUM_NOT_RUN(s)		( ((s) & SMAC_SRD_M_TCSE) && !((s) & SMAC_SRD_M_TCS))
#define IP_CHECKSUM_IS_GOOD(s)		(!((s) & SMAC_SRD_M_ICSE) &&  ((s) & SMAC_SRD_M_ICS))
#define TCP_UDP_CHECKSUM_IS_GOOD(s)	(!((s) & SMAC_SRD_M_TCSE) &&  ((s) & SMAC_SRD_M_TCS))
#define CHECKSUM_IS_GOOD(s)		(IP_CHECKSUM_IS_GOOD(s) || TCP_UDP_CHECKSUM_IS_GOOD(s))

static void smac_rx_checksum(struct smac_adapter *adapter,
			     uint32_t status_err, struct sk_buff *skb)
{
	SMAC_DEBUG_FUNC();

	skb->ip_summed = CHECKSUM_NONE;

	/* turn off rx checksum offload */
	if (unlikely(CHECKSUM_IS_NOT_SET)) {
		return;
	}

	/* not checksumming in hardware */
	if (unlikely(CHECKSUM_NOT_RUN(status_err))) {
		return;
	}

	/* IP checksum error */
	if (unlikely(IP_CHECKSUM_IS_ERROR(status_err))) {
		/* let the stack verify checksum errors */
		adapter->hw_csum_err++;
		SMAC_DPRINTK(RX_ERR, ERR, "IP checksum offload error\n");
		return;
	}

	/* TCP/UDP/ICMP checksum error */
	if (unlikely(TCP_UDP_CHECKSUM_IS_ERROR(status_err))) {
		adapter->hw_csum_err++;
		SMAC_DPRINTK(RX_ERR, ERR, "TCP/UDP/ICMP checksum offload error\n");
		return;
	}

	/* done checksum in hardware */
	if (CHECKSUM_IS_GOOD(status_err)) {
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		adapter->hw_csum_good++;
		return;
	}

	return;
}


/*----------------------------------------------------------------------------*/
/* allocate rx socket buffer */
static void smac_alloc_rx_buffers(struct smac_adapter *adapter,
				  struct smac_rx_ring *rx_ring,
				  int cleaned_count)
{
	struct platform_device *pdev = adapter->pdev;
	struct net_device *netdev = adapter->netdev;
	struct smac_desc *rx_desc;
	struct smac_buffer *buffer_info;
	struct sk_buff *skb;
	unsigned int i;
	unsigned int bufsz = adapter->rx_buffer_len + NET_IP_ALIGN;

	SMAC_DEBUG_FUNC();

	/* get next buffer_info's index */
	i = rx_ring->next_to_use;
	buffer_info = &rx_ring->buffer_info[i];

	while (cleaned_count--) {
		skb = buffer_info->skb;
		if (skb) {		/* re-use socket buffer */
			skb_trim(skb, 0);   /* initialize socket buffer */
			goto map_skb;
		}

		/* allocate socket buffer */
		skb = netdev_alloc_skb(netdev, bufsz);
		if (unlikely(!skb)) {
			/* Better luck next round */
			adapter->alloc_rx_buff_failed++;
			SMAC_DPRINTK(RX_ERR, ERR, "dev_alloc_skb() failed\n");
			break;
		}
		/* Make buffer alignment 2 beyond a 16 byte boundary
		 * this will result in a 16 byte aligned IP header after
		 * the 14 byte MAC header is removed
		 */
		skb_reserve(skb, NET_IP_ALIGN);

		buffer_info->skb = skb;
		buffer_info->length = adapter->rx_buffer_len;

map_skb:
		/* map rx buffer */
		buffer_info->dma = dma_map_single(&pdev->dev, skb->data,
					buffer_info->length, DMA_FROM_DEVICE);

		/* set RDES */
		rx_desc = SMAC_RX_DESC(*rx_ring, i);
		smac_write_desc(0, (void *) &rx_desc->desc0);
		{
			/* set RDES1 */
			uint32_t rd1 = 0;
			uint32_t fcl = smac_readreg(adapter, SMAC_RX_FRAME_COUNT_LIMIT);

			SMAC_ASSERT(adapter->rx_int_delay == smac_readreg(adapter, SMAC_RX_DELAY_TIMER));
			if (adapter->rx_int_delay)
				rd1 |= SMAC_SRD1_M_DT;
			SMAC_ASSERT(adapter->rx_abs_int_delay == smac_readreg(adapter, SMAC_RX_ABSOLUTE_TIMER));
			if (adapter->rx_abs_int_delay)
				rd1 |= SMAC_SRD1_M_ADT;
			if (fcl && (i % fcl == 0))
				rd1 |= SMAC_SRD1_M_FCL;
			smac_write_desc(rd1, (void *) &rx_desc->desc1);
		}
		smac_write_desc(0, (void *) &rx_desc->desc2);
		smac_write_desc(buffer_info->dma, (void *) &rx_desc->desc3);

		/* wrap around */
		if (unlikely(++i == rx_ring->count)) {
			i = 0;
		}
		buffer_info = &rx_ring->buffer_info[i];
	}
	wmb();

	/* move descriptor's TAIL */
	if (likely(rx_ring->next_to_use != i)) {
		rx_ring->next_to_use = i;

		/* set next descriptor to TAIL */
		smac_writereg(adapter, SMAC_RX_DB_TAIL,
			      adapter->rx_desc_addr_phys + (sizeof(struct smac_desc) * i));
	}


	return;
}

/*----------------------------------------------------------------------------*/
/* reset interface */
void smac_reinit_locked(struct smac_adapter *adapter)
{
	SMAC_DEBUG_FUNC();

	WARN_ON(in_interrupt());
	while (test_and_set_bit(__SMAC_RESETTING, (void *) &adapter->flags)) {
		msleep(1);
	}
	smac_down(adapter);
	smac_up(adapter);
	clear_bit(__SMAC_RESETTING, (void *) &adapter->flags);
}


/*----------------------------------------------------------------------------*/
/* reset interface */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
static void smac_reset_task(struct net_device *netdev)
{
	struct smac_adapter *adapter = netdev_priv(netdev)

	SMAC_DEBUG_FUNC();
	smac_reinit_locked(adapter);
	return;
}
#else
static void smac_reset_task(struct work_struct *work)
{
	struct smac_adapter *adapter =
		container_of(work, struct smac_adapter, reset_task);

	SMAC_DEBUG_FUNC();
	smac_reinit_locked(adapter);
	return;
}
#endif

/*----------------------------------------------------------------------------*/
/* set Tx ckecksum offload */
static uint32_t
smac_tx_csum(struct sk_buff *skb)
{
	uint32_t tx_flags = 0;

	SMAC_DEBUG_FUNC();

	if (likely(skb->ip_summed != CHECKSUM_PARTIAL)) {
		/* checksum is not in hardware */
		return 0;
	}

	if (skb->protocol == htons(ETH_P_IP)) {
		/* IP header checksum offload */
		tx_flags |= SMAC_TX_FLAGS_CSUM;

		switch (ip_hdr(skb)->protocol) {
		case IPPROTO_TCP:
		case IPPROTO_UDP:
		case IPPROTO_ICMP:
			/* TCP/UDP/ICMP header checksum offload */
			tx_flags |= SMAC_TX_FLAGS_TCSUM;
			break;
		case IPPROTO_ESP:
		case IPPROTO_AH:
			/* nothing */
			break;
		default:
			break;
		}
	} else if (skb->protocol == htons(ETH_P_IPV6)) {
		switch (ipv6_hdr(skb)->nexthdr) {
		case NEXTHDR_TCP:
		case NEXTHDR_UDP:
		case NEXTHDR_ICMP:
			/* SMAC H/W need this setting */
			tx_flags |= SMAC_TX_FLAGS_CSUM;
			/* TCP/UDP/ICMP header checksum offload */
			tx_flags |= SMAC_TX_FLAGS_TCSUM;
			break;

		/* IPv6 extension header */
		case NEXTHDR_HOP:
		case NEXTHDR_DEST:
		case NEXTHDR_ROUTING:
		case NEXTHDR_FRAGMENT:
		case NEXTHDR_ESP:
		case NEXTHDR_AUTH:
			/* nothing */
			break;
		default:
			break;
		}
	}

	return tx_flags;
}

#if 0
/*----------------------------------------------------------------------------*/
/* clear checksum to zero */
static void smac_clear_tx_checksum(struct sk_buff *skb, uint32_t tx_flags)
{
	if (likely(skb->ip_summed != CHECKSUM_PARTIAL)) {
		/* is not checksum in hardware */
		return;
	}

	/* TCP/UDP header offload */
	if (tx_flags & SMAC_TX_FLAGS_TCSUM) {
		if (ip_hdr(skb)->protocol == IPPROTO_TCP
			|| ipv6_hdr(skb)->nexthdr == NEXTHDR_TCP)
			tcp_hdr(skb)->check = 0;
		if (ip_hdr(skb)->protocol == IPPROTO_UDP
			|| ipv6_hdr(skb)->nexthdr == NEXTHDR_UDP)
			udp_hdr(skb)->check = 0;
	}

	/* IPv4 header offload */
	if (tx_flags & SMAC_TX_FLAGS_CSUM) {
		if (skb->protocol == htons(ETH_P_IP)) {
			ip_hdr(skb)->check = 0;
		}
	}
}
#else
#define smac_clear_tx_checksum(a, b)
#endif


/*----------------------------------------------------------------------------*/
/* set struct smac_buffer members */
static int smac_tx_map(struct smac_adapter *adapter,
		       struct smac_tx_ring *tx_ring, struct sk_buff *skb,
		       unsigned int first, unsigned int max_per_txd,
		       unsigned int nr_frags, uint32_t tx_flags)
{
	struct platform_device *pdev = adapter->pdev;
	struct smac_buffer *buffer_info;
	unsigned int len = skb->len;
	unsigned int offset = 0, size, count = 0, i;
	unsigned int f;

	SMAC_DEBUG_FUNC();

	len -= skb->data_len;
	i = tx_ring->next_to_use;

	while (len) {
		buffer_info = &tx_ring->buffer_info[i];
		size = min(len, max_per_txd);
		buffer_info->length = size & 0x3FFF;	/* TDES1 Buffer Length #2 (14bit) */
		buffer_info->dma = dma_map_single(&pdev->dev, skb->data + offset,
					buffer_info->length, DMA_TO_DEVICE);

		/* clear checksum to zero */
		smac_clear_tx_checksum(skb, tx_flags);

		smac_print_tx_skb(skb, size, offset);	/* for debug */

		buffer_info->time_stamp = jiffies;
		buffer_info->next_to_watch = i;

		len -= size;
		offset += size;
		count++;
		/* wrap around */
		if (unlikely(++i == tx_ring->count)) {
			i = 0;
		}
	}

	/* IP fragment */
	for (f = 0; f < nr_frags; f++) {
		struct skb_frag_struct *frag;

		SMAC_DEBUG_OUT("tx fragment:%d/%d\n", f+1, nr_frags);

		frag = &skb_shinfo(skb)->frags[f];
		len = frag->size;
		offset = frag->page_offset;

		while (len) {
			buffer_info = &tx_ring->buffer_info[i];
			size = min(len, max_per_txd);

			buffer_info->length = size;
			buffer_info->dma = dma_map_single(&pdev->dev,
					page_address(frag->page) + offset,
					buffer_info->length, DMA_TO_DEVICE);

			buffer_info->time_stamp = jiffies;
			buffer_info->next_to_watch = i;

			smac_print_tx_flagment(frag, size, offset);	/* for debug */

			len -= size;
			offset += size;
			count++;
			/* wrap around */
			if (unlikely(++i == tx_ring->count)) {
				i = 0;
			}
		}
	}

	i = (i == 0) ? tx_ring->count - 1 : i - 1;
	tx_ring->buffer_info[i].skb = skb;
	tx_ring->buffer_info[first].next_to_watch = i;

	return count;
}


/*----------------------------------------------------------------------------*/
/* set tx descriptor */
static void smac_tx_queue(struct smac_adapter *adapter,
			  struct smac_tx_ring *tx_ring,
			  uint32_t tx_flags, int count)
{
	struct smac_desc *tx_desc = NULL;
	struct smac_buffer *buffer_info;
	uint32_t tdes0 = 0;
	unsigned int i;

	SMAC_DEBUG_FUNC();

	/* set checksum offload to TDES0's mask */
	{
		tdes0 |= (SMAC_STDD_M_DFCS_DIS 		/* insert FCS */
			| SMAC_STDD_M_DPAD_DIS);	/* enable padding FCS */
		/* IPv4 header offload */
		if (tx_flags & SMAC_TX_FLAGS_CSUM) {
			tdes0 |= SMAC_STDD_M_ICS;	/* IP header checksum */
		}
		/* TCP/UDP header offload */
		if (likely(tx_flags & SMAC_TX_FLAGS_TCSUM)) {
			tdes0 |= (SMAC_STDD_M_TCS 	/* TCP/UDP header checksum */
				| SMAC_STDD_M_PHCS); 	/* TCP/UDP pseudo IP header checksum */
		}
		/* first frame */
		tdes0 |= SMAC_STDD_M_FS;
	}

	i = tx_ring->next_to_use;

	while (count--) {
		if (likely(count == 0)) {
			/* last frame */
			tdes0 |= SMAC_STDD_M_LS;
		}
		buffer_info = &tx_ring->buffer_info[i];

		/* set skb physical address to tx descriptor */
		tx_desc = SMAC_TX_DESC(*tx_ring, i);
		smac_write_desc(0, (void *) &tx_desc->desc2);
		smac_write_desc(buffer_info->dma, (void *) &tx_desc->desc3);
		smac_write_desc(buffer_info->length << 16, (void *) &tx_desc->desc1);
		smac_write_desc(tdes0, (void *) &tx_desc->desc0);

		/* wrap around */
		if (unlikely(++i == tx_ring->count)) {
			i = 0;
		}

		SMAC_DPRINTK(TX_QUEUED, DEBUG, "tx length: 0x%08X\n", buffer_info->length);
		SMAC_DPRINTK(TX_QUEUED, DEBUG, "tx_desc: 0x%08lx\n", *(unsigned long *) (&tx_desc->desc0));

		/* for next loop */
		tdes0 &= ~SMAC_STDD_M_FS;
		tdes0 &= ~SMAC_STDD_M_LS;
	}
	wmb();

	tx_ring->next_to_use = i;

	/* set next descriptor to TAIL */
	smac_writereg(adapter, SMAC_TX_DB_TAIL,
		      adapter->tx_desc_addr_phys + (sizeof(struct smac_desc) * i));
}


/*----------------------------------------------------------------------------*/
static int __smac_maybe_stop_tx(struct net_device *netdev, int size)
{
	struct smac_adapter *adapter = netdev_priv(netdev);
	struct smac_tx_ring *tx_ring = adapter->tx_ring;

	SMAC_DEBUG_FUNC();

	/* stop tx queue */
	netif_stop_queue(netdev);

	smp_mb();

	/* is there not-used descriptor */
	if (likely(SMAC_DESC_UNUSED(tx_ring) < size)) {
		return -EBUSY;
	}

	/* restart tx queue */
	netif_start_queue(netdev);
	++adapter->restart_queue;

	return 0;
}


/*----------------------------------------------------------------------------*/
static int smac_maybe_stop_tx(struct net_device *netdev,
			      struct smac_tx_ring *tx_ring, int size)
{
	SMAC_DEBUG_FUNC();

	if (likely(SMAC_DESC_UNUSED(tx_ring) >= size)) {
		return 0;
	}

	return __smac_maybe_stop_tx(netdev, size);
}


/*----------------------------------------------------------------------------*/
static int smac_init_busmode(struct smac_adapter *adapter)
{
	uint32_t mode;

	SMAC_DEBUG_FUNC();

	mode = SMAC_BM_PBL;
#if defined(SMAC_CPU_DESC_BE)
	mode |= SMAC_BM_BLE_BIG;
#elif defined(SMAC_CPU_DESC_LE)
	mode |= SMAC_BM_BLE_LITTLE;
#endif
	smac_writereg(adapter, SMAC_BUS_MODE, mode);

	return 0;
}


/*----------------------------------------------------------------------------*/
static int smac_sw_init(struct smac_adapter *adapter)
{
	struct smac_platform_data *pdata = adapter->platform_data;

	SMAC_DEBUG_FUNC();

	/* initialize struct smac_adapter */
 	adapter->rx_buffer_len = MAXIMUM_ETHERNET_VLAN_SIZE;
	adapter->msg_enable = (1 << smac_debug) - 1;

	/* initialize mii_if_info */
	if (pdata->phy_addr >= PHY_MAX_ADDR) {
		return -EIO;
	}
	adapter->mii.dev          = adapter->netdev;
	adapter->mii.mdio_read    = smac_mdio_read;
	adapter->mii.mdio_write   = smac_mdio_write;
	adapter->mii.phy_id       = pdata->phy_addr;
	adapter->mii.phy_id_mask  = 0x1f;
	adapter->mii.reg_num_mask = 0x1f;

	/* initialize flow control setting status flag */
	adapter->fc_set_done = FALSE;

	/* allocate descriptors */
	if (smac_alloc_queues(adapter)) {
		SMAC_ERR_OUT("Unable to allocate memory for queues\n");
		return -ENOMEM;
	}

	/* number of tx_ring and rx_ring */
	adapter->tx_ring->count = SMAC_MAX_TXD;
	adapter->rx_ring->count = SMAC_MAX_RXD;

	/* statistics */
	adapter->total_tx_bytes = 0;
	adapter->total_tx_packets = 0;
	adapter->total_rx_bytes = 0;
	adapter->total_rx_packets = 0;

	spin_lock_init(&adapter->stats_lock);
	spin_lock_init(&adapter->phy_lock);

#ifdef SMAC_USE_PHY_POLLING
	set_bit(__SMAC_DOWN, (void *) &adapter->flags);
#endif

	return 0;
}


/*----------------------------------------------------------------------------*/
/* set smac_adapter.mac_addr */
static void smac_set_mac_addr(struct smac_adapter *adapter)
{
	struct net_device *netdev = adapter->netdev;
	struct smac_platform_data *pdata = adapter->platform_data;

	SMAC_DEBUG_FUNC();

	/* read MAC Address from register or flash */
	smac_read_mac_addr(adapter);

	if (!is_valid_ether_addr(adapter->mac_addr)) {
		/* set default MAC address */
		memcpy(adapter->mac_addr, pdata->mac_addr, netdev->addr_len);
	}
}


#ifdef SMAC_PRINT_RX_SOCKET
/*----------------------------------------------------------------------------*/
/* print rx socket */
static void smac_print_rx_skb(struct sk_buff *skb, uint32_t length)
{
	int i;

	printk("----------\n");
	printk("rx socket len:%d\n", length);
	for (i = 0; i < length; i++) {
		printk("%02X ", skb->data[i]);
		if ((i % 16) == 15) printk("\n");
	}
	printk("\n");
}
#endif


#ifdef SMAC_PRINT_TX_SOCKET
/*----------------------------------------------------------------------------*/
/* print tx socket */
static void smac_print_tx_skb(struct sk_buff *skb, uint32_t length, uint32_t offset)
{
	int i;

	printk("----------\n");
	printk("tx socket len: %d[byte]\n", length);
	for (i = 0; i < length; i++) {
		printk("%02X ", skb->data[i + offset]);
		if ((i % 16) == 15) printk("\n");
	}
	printk("\n");
	if (skb->nh.iph) printk(" IP checksum: %04x\n", skb->nh.iph->check);
	if (skb->h.th) printk("TCP checksum: %04x\n", skb->h.th->check);
}
#endif


#ifdef SMAC_PRINT_TX_FLAGMENT
/*----------------------------------------------------------------------------*/
/* print tx flagment */
static void smac_print_tx_flagment(struct skb_frag_struct *frag, uint32_t length, uint32_t offset)
{
	int i;
	volatile unsigned char *ptr = (unsigned char *) page_address(frag->page);

	printk("tx socket len:%d\n", length);
	printk("offset:%d\n", offset);
	printk("page address:0x%p\n", ptr);

	for (i = 0; i < length; i++) {
		printk("%02X ", *(volatile unsigned char *) (ptr + i + offset));
		if ((i % 16) == 15) printk("\n");
	}
	printk("\n");
}
#endif


/*----------------------------------------------------------------------------*/
/* print SMAC_MAC_ADDR_i_{LOW,HIGH} */
static void smac_print_mac_address(struct smac_adapter *adapter, int i)
{
	uint32_t low = smac_readreg(adapter, SMAC_MAC_ADDR_LOW(i));
	uint32_t high = smac_readreg(adapter, SMAC_MAC_ADDR_HIGH(i));
	unsigned char mac_addr[6];
	int j;

	for ( j = 0; j < 6; j++ ) {
		if ( j < 4 ) {
			mac_addr[j] = low & 0xFF;
			low >>= 8;
		} else {
			mac_addr[j] = high & 0xFF;
			high >>= 8;
		}
	}

	printk(KERN_DEBUG "MAC(%02d): %02X:%02X:%02X:%02X:%02X:%02X\n",
		i,
		mac_addr[0], mac_addr[1], mac_addr[2],
		mac_addr[3], mac_addr[4], mac_addr[5]);
}


#ifdef SMAC_PRINT_MAC_ADDRES
/*----------------------------------------------------------------------------*/
/* print SMAC_MAC_ADDR_[0-SMAC_MULTICAST_ADDR_NUM] */
static void smac_print_mac_addresses(struct smac_adapter *adapter)
{
	int i;

	for (i = 0; i < SMAC_MULTICAST_ADDR_NUM; i++) {
		smac_print_mac_address(adapter, i);
	}
}
#endif

/*----------------------------------------------------------------------------*/

