/*
 * smac_param.c
 * module parameter for smac
 *
 * Copyright 2008,2009 Sony Corporation
 *
 * This code is based on drivers/net/e1000/e1000_param.c
 */
/*******************************************************************************

  Intel PRO/1000 Linux driver
  Copyright(c) 1999 - 2006 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  Linux NICS <linux.nics@intel.com>
  e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "smac_local.h"

/* This is the only thing that needs to be changed to adjust the
 * maximum number of ports that the driver can manage.
 */

#define SMAC_MAX_NIC	(0)

#define OPTION_UNSET	(-1)
#define OPTION_DISABLED	(0)
#define OPTION_ENABLED	(1)

/* All parameters are treated the same, as an integer array of values.
 * This macro just reduces the need to repeat the same declaration code
 * over and over (plus this helps to avoid typo bugs).
 */

#define SMAC_PARAM_INIT { [0 ... SMAC_MAX_NIC] = OPTION_UNSET }
#define SMAC_PARAM(X, desc) \
	static int __devinitdata X[SMAC_MAX_NIC+1] = SMAC_PARAM_INIT; \
	static int num_##X = 0; \
	module_param_array_named(X, X, int, &num_##X, 0); \
	MODULE_PARM_DESC(X, desc);

#if 0 /* not implemented */
/* Transmit Descriptor Count
 *
 * Valid Range: 80-256 for 82542 and 82543 gigabit ethernet controllers
 * Valid Range: 80-4096 for 82544 and newer
 *
 * Default Value: 256
 */
SMAC_PARAM(TxDescriptors, "Number of transmit descriptors");
#endif

#if 0 /* not implemented */
/* Receive Descriptor Count
 *
 * Valid Range: 80-256 for 82542 and 82543 gigabit ethernet controllers
 * Valid Range: 80-4096 for 82544 and newer
 *
 * Default Value: 256
 */
SMAC_PARAM(RxDescriptors, "Number of receive descriptors");
#endif

/* User Specified Speed Override
 *
 * Valid Range: 0, 10, 100, 1000
 *  - 0    - auto-negotiate at all supported speeds
 *  - 10   - only link at 10 Mbps
 *  - 100  - only link at 100 Mbps
 *  - 1000 - only link at 1000 Mbps
 *
 * Default Value: 0
 */
SMAC_PARAM(Speed, "Speed setting");

/* User Specified Duplex Override
 *
 * Valid Range: 0-2
 *  - 0 - auto-negotiate for duplex
 *  - 1 - only link at half duplex
 *  - 2 - only link at full duplex
 *
 * Default Value: 0
 */
SMAC_PARAM(Duplex, "Duplex setting");

/* Auto-negotiation Advertisement Override
 *
 * Valid Range: 0x01-0x0F, 0x20-0x2F (copper)
 *
 * The AutoNeg value is a bit mask describing which speed and duplex
 * combinations should be advertised during auto-negotiation.
 * The supported speed and duplex modes are listed below
 *
 * Bit           7     6     5      4      3     2     1      0
 * Speed (Mbps)  N/A   N/A   1000   N/A    100   100   10     10
 * Duplex                    Full          Full  Half  Full   Half
 *
 * Default Value: 0x2F (copper)
 */
SMAC_PARAM(AutoNeg, "Advertised auto-negotiation setting");
#define AUTONEG_ADV_DEFAULT  0x2F
#define AUTONEG_ADV_MASK     0x2F

/* User Specified Flow Control Override
 *
 * Valid Range: 0-3
 *  - 0 - No Flow Control
 *  - 1 - Rx only, respond to PAUSE frames but do not generate them
 *  - 2 - Tx only, generate PAUSE frames but ignore them on receive
 *  - 3 - Full Flow Control Support
 *
 * Default Value: Read flow control settings from the EEPROM
 */
SMAC_PARAM(FlowControl, "Flow Control setting");
#define FLOW_CONTROL_DEFAULT FLOW_CONTROL_FULL

#if 0 /* not implemented */
/* XsumRX - Receive Checksum Offload Enable/Disable
 *
 * Valid Range: 0, 1
 *  - 0 - disables all checksum offload
 *  - 1 - enables receive IP/TCP/UDP checksum offload
 *        on 82543 and newer -based NICs
 *
 * Default Value: 1
 */
SMAC_PARAM(XsumRX, "Disable or enable Receive Checksum offload");
#endif

/* Receive Interrupt Delay (bit clock time)
 *
 * Valid Range: 0-65535
 */
SMAC_PARAM(RxIntDelay, "Receive Interrupt Delay");
#define DEFAULT_RDTR                   0
#define MAX_RXDELAY               0xFFFF
#define MIN_RXDELAY                    0

/* Receive Absolute Interrupt Delay (bit clock time)
 *
 * Valid Range: 0-65535
 */
SMAC_PARAM(RxAbsIntDelay, "Receive Absolute Interrupt Delay");
#define DEFAULT_RADV                   0
#define MAX_RXABSDELAY            0xFFFF
#define MIN_RXABSDELAY                 0

/* Interrupt Throttle Number (# of frames)
 *
 * Valid Range: 0-65535 (0,2=off, 1=dynamic, 3=dynamic conservative, 4-65535=fixed)
 */
SMAC_PARAM(InterruptThrottleNumber, "Interrupt Throttling Number");
#define DEFAULT_ITNR                    0
#define MAX_ITNR                   0xFFFF
#define MIN_ITNR                        0

/* module parameter type */
struct smac_option {
	enum { enable_option, range_option, list_option } type;
	char *name;
	char *err;
	int  def;
	union {
		struct { /* range_option info */
			int min;
			int max;
		} r;
		struct { /* list_option info */
			int nr;
			struct smac_opt_list { int i; char *str; } *p;
		} l;
	} arg;
};


/**
 * smac_validate_option - validate option
 * @value: parameter value
 * @opt: option type
 * @adapter: board private structure
 **/
static int __devinit
smac_validate_option(int *value, struct smac_option *opt,
		struct smac_adapter *adapter)
{
	if (*value == OPTION_UNSET) {
		*value = opt->def;
		return 0;
	}

	switch (opt->type) {
	case enable_option:
		switch (*value) {
		case OPTION_ENABLED:
			SMAC_DPRINTK(PROBE, INFO, "%s Enabled\n", opt->name);
			return 0;
		case OPTION_DISABLED:
			SMAC_DPRINTK(PROBE, INFO, "%s Disabled\n", opt->name);
			return 0;
		}
		break;
	case range_option:
		if (*value >= opt->arg.r.min && *value <= opt->arg.r.max) {
			SMAC_DPRINTK(PROBE, INFO,
					"%s set to %i\n", opt->name, *value);
			return 0;
		}
		break;
	case list_option: {
		int i;
		struct smac_opt_list *ent;

		for (i = 0; i < opt->arg.l.nr; i++) {
			ent = &opt->arg.l.p[i];
			if (*value == ent->i) {
				if (ent->str[0] != '\0')
					SMAC_DPRINTK(PROBE, INFO, "%s\n", ent->str);
				return 0;
			}
		}
	}
		break;
	default:
		BUG();
	}

	SMAC_DPRINTK(PROBE, INFO, "Invalid %s value specified (%i) %s\n",
	       opt->name, *value, opt->err);
	*value = opt->def;
	return -1;
}

static void smac_check_copper_options(struct smac_adapter *adapter);

/**
 * smac_check_options - Range Checking for Command Line Parameters
 * @adapter: board private structure
 *
 * This routine checks all command line parameters for valid user
 * input.  If an invalid value is given, or if no user specified
 * value exists, a default value is used.  The final value is stored
 * in a variable in the adapter structure.
 **/
void __devinit
smac_check_options(struct smac_adapter *adapter)
{
#if 0 /* not implemented */
	{ /* Transmit Descriptor Count */
		struct smac_opt_list txd_list[] = {{ 256, "" },
						{ 128, "" },
						{ 64, "" }};
		struct smac_option opt = {
			.type = list_option,
			.name = "Transmit Descriptors",
			.err  = "using default of "
				__MODULE_STRING(SMAC_DEFAULT_TXD),
			.def  = SMAC_DEFAULT_TXD,
			.arg  = { .l = { .nr = ARRAY_SIZE(txd_list),
					 .p = txd_list }}
		};
		struct smac_tx_ring *tx_ring = adapter->tx_ring;

		if (num_TxDescriptors > 0) {
			tx_ring->count = TxDescriptors[0];
			smac_validate_option(&tx_ring->count, &opt, adapter);
		} else {
			tx_ring->count = opt.def;
		}
	}
#endif

#if 0 /* not implemented */
	{ /* Receive Descriptor Count */
		struct smac_opt_list rxd_list[] = {{ 256, "" },
						{ 128, "" },
						{ 64, "" }};
		struct smac_option opt = {
			.type = list_option,
			.name = "Receive Descriptors",
			.err  = "using default of "
				__MODULE_STRING(SMAC_DEFAULT_RXD),
			.def  = SMAC_DEFAULT_RXD,
			.arg  = { .l = { .nr = ARRAY_SIZE(rxd_list),
					 .p = rxd_list }}
		};
		struct smac_rx_ring *rx_ring = adapter->rx_ring;

		if (num_RxDescriptors > 0) {
			rx_ring->count = RxDescriptors[0];
			smac_validate_option(&rx_ring->count, &opt, adapter);
		} else {
			rx_ring->count = opt.def;
		}
	}
#endif

#if 0 /* not implemented */
	{ /* Checksum Offload Enable/Disable */
		struct smac_option opt = {
			.type = enable_option,
			.name = "Checksum Offload",
			.err  = "defaulting to Enabled",
			.def  = OPTION_ENABLED
		};

		if (num_XsumRX > 0) {
			int rx_csum = XsumRX[0];
			smac_validate_option(&rx_csum, &opt, adapter);
			adapter->rx_csum = rx_csum;
		} else {
			adapter->rx_csum = opt.def;
		}
	}
#endif

	{ /* Flow Control */
		struct smac_opt_list fc_list[] =
			{{ SMAC_FC_NONE,    "Flow Control Disabled" },
			 { SMAC_FC_RX_PAUSE,"Flow Control Receive Only" },
			 { SMAC_FC_TX_PAUSE,"Flow Control Transmit Only" },
			 { SMAC_FC_FULL,    "Flow Control Enabled" }};

		struct smac_option opt = {
			.type = list_option,
			.name = "Flow Control",
			.err  = "using default of " __MODULE_STRING(SMAC_FC_FULL),
			.def  = SMAC_FC_FULL,
			.arg  = { .l = { .nr = ARRAY_SIZE(fc_list),
					 .p = fc_list }}
		};

		if (num_FlowControl > 0) {
			int fc = FlowControl[0];
			smac_validate_option(&fc, &opt, adapter);
			adapter->fc = fc;
		} else {
			adapter->fc = opt.def;
		}
	}

	{ /* Receive Interrupt Delay */
		struct smac_option opt = {
			.type = range_option,
			.name = "Receive Interrupt Delay",
			.err  = "using default of " __MODULE_STRING(DEFAULT_RDTR),
			.def  = DEFAULT_RDTR,
			.arg  = { .r = { .min = MIN_RXDELAY,
					 .max = MAX_RXDELAY }}
		};

		if (num_RxIntDelay > 0) {
			adapter->rx_int_delay = RxIntDelay[0];
			smac_validate_option(&adapter->rx_int_delay, &opt,
			                      adapter);
		} else {
			adapter->rx_int_delay = opt.def;
		}
	}

	{ /* Receive Absolute Interrupt Delay */
		struct smac_option opt = {
			.type = range_option,
			.name = "Receive Absolute Interrupt Delay",
			.err  = "using default of " __MODULE_STRING(DEFAULT_RADV),
			.def  = DEFAULT_RADV,
			.arg  = { .r = { .min = MIN_RXABSDELAY,
					 .max = MAX_RXABSDELAY }}
		};

		if (num_RxAbsIntDelay > 0) {
			adapter->rx_abs_int_delay = RxAbsIntDelay[0];
			smac_validate_option(&adapter->rx_abs_int_delay, &opt,
			                      adapter);
		} else {
			adapter->rx_abs_int_delay = opt.def;
		}
	}

	{ /* Interrupt Throttling Number */
		struct smac_option opt = {
			.type = range_option,
			.name = "Interrupt Throttling Number",
			.err  = "using default of " __MODULE_STRING(DEFAULT_ITNR),
			.def  = DEFAULT_ITNR,
			.arg  = { .r = { .min = MIN_ITNR,
					 .max = MAX_ITNR }}
		};

		if (num_InterruptThrottleNumber > 0) {
			adapter->itnr = InterruptThrottleNumber[0];
			switch (adapter->itnr) {
			case SMAC_ITNR_OFF:
			case 2:
				SMAC_DPRINTK(PROBE, INFO, "%s turned off\n",
				        opt.name);
				adapter->itnr_setting = SMAC_ITNR_OFF;
				break;
			case SMAC_ITNR_DYNAMIC:
				SMAC_DPRINTK(PROBE, INFO, "%s set to dynamic mode\n",
					opt.name);
				adapter->itnr_setting = SMAC_ITNR_DYNAMIC;
				adapter->itnr = SMAC_LOW_LATENCY;
				adapter->current_itnr = lowest_latency_range;
				break;
			case SMAC_ITNR_DYNAMIC_CONSERVATIVE:
				SMAC_DPRINTK(PROBE, INFO,
				        "%s set to dynamic conservative mode\n",
					opt.name);
				adapter->itnr_setting = SMAC_ITNR_DYNAMIC_CONSERVATIVE;
				adapter->itnr = SMAC_LOW_LATENCY;
				adapter->current_itnr = lowest_latency_range;
				break;
			default: /* fixed value */
				smac_validate_option(&adapter->itnr, &opt, adapter);
				adapter->itnr_setting = adapter->itnr & ~3;
				break;
			}
		} else {
			adapter->itnr_setting = opt.def;
			adapter->itnr = 0;
		}
	}

	smac_check_copper_options(adapter);
}


/**
 * smac_check_copper_options - Range Checking for Link Options, Copper Version
 * @adapter: board private structure
 *
 * Handles speed and duplex options on copper adapters
 **/

static void __devinit
smac_check_copper_options(struct smac_adapter *adapter)
{
	int speed, dplx, an;

	{ /* Speed */
		struct smac_opt_list speed_list[] = {{          0, "" },
						      {   SPEED_10, "" },
						      {  SPEED_100, "" },
						      { SPEED_1000, "" }};
		struct smac_option opt = {
			.type = list_option,
			.name = "Speed",
			.err  = "parameter ignored",
			.def  = 0,
			.arg  = { .l = { .nr = ARRAY_SIZE(speed_list),
					 .p = speed_list }}
		};

		if (num_Speed > 0) {
			speed = Speed[0];
			smac_validate_option(&speed, &opt, adapter);
		} else {
			speed = opt.def;
		}
	}

	{ /* Duplex */
		struct smac_opt_list dplx_list[] = {{           0, "" },
						     { SMAC_HALF_DUPLEX, "" },
						     { SMAC_FULL_DUPLEX, "" }};
		struct smac_option opt = {
			.type = list_option,
			.name = "Duplex",
			.err  = "parameter ignored",
			.def  = 0,
			.arg  = { .l = { .nr = ARRAY_SIZE(dplx_list),
					 .p = dplx_list }}
		};

		if (num_Duplex > 0) {
			dplx = Duplex[0];
			smac_validate_option(&dplx, &opt, adapter);
		} else {
			dplx = opt.def;
		}
	}

	if ((num_AutoNeg > 0) && (speed != 0 || dplx != 0)) {
		SMAC_DPRINTK(PROBE, INFO,
		       "AutoNeg specified along with Speed or Duplex, "
		       "parameter ignored\n");
		adapter->autoneg_advertised = AUTONEG_ADV_DEFAULT;
	} else { /* Autoneg */
		struct smac_opt_list an_list[] =
			#define AA "AutoNeg advertising "
			{{ 0x01, AA "10/HD" },
			 { 0x02, AA "10/FD" },
			 { 0x03, AA "10/FD, 10/HD" },
			 { 0x04, AA "100/HD" },
			 { 0x05, AA "100/HD, 10/HD" },
			 { 0x06, AA "100/HD, 10/FD" },
			 { 0x07, AA "100/HD, 10/FD, 10/HD" },
			 { 0x08, AA "100/FD" },
			 { 0x09, AA "100/FD, 10/HD" },
			 { 0x0a, AA "100/FD, 10/FD" },
			 { 0x0b, AA "100/FD, 10/FD, 10/HD" },
			 { 0x0c, AA "100/FD, 100/HD" },
			 { 0x0d, AA "100/FD, 100/HD, 10/HD" },
			 { 0x0e, AA "100/FD, 100/HD, 10/FD" },
			 { 0x0f, AA "100/FD, 100/HD, 10/FD, 10/HD" },
			 { 0x20, AA "1000/FD" },
			 { 0x21, AA "1000/FD, 10/HD" },
			 { 0x22, AA "1000/FD, 10/FD" },
			 { 0x23, AA "1000/FD, 10/FD, 10/HD" },
			 { 0x24, AA "1000/FD, 100/HD" },
			 { 0x25, AA "1000/FD, 100/HD, 10/HD" },
			 { 0x26, AA "1000/FD, 100/HD, 10/FD" },
			 { 0x27, AA "1000/FD, 100/HD, 10/FD, 10/HD" },
			 { 0x28, AA "1000/FD, 100/FD" },
			 { 0x29, AA "1000/FD, 100/FD, 10/HD" },
			 { 0x2a, AA "1000/FD, 100/FD, 10/FD" },
			 { 0x2b, AA "1000/FD, 100/FD, 10/FD, 10/HD" },
			 { 0x2c, AA "1000/FD, 100/FD, 100/HD" },
			 { 0x2d, AA "1000/FD, 100/FD, 100/HD, 10/HD" },
			 { 0x2e, AA "1000/FD, 100/FD, 100/HD, 10/FD" },
			 { 0x2f, AA "1000/FD, 100/FD, 100/HD, 10/FD, 10/HD" }};

		struct smac_option opt = {
			.type = list_option,
			.name = "AutoNeg",
			.err  = "parameter ignored",
			.def  = AUTONEG_ADV_DEFAULT,
			.arg  = { .l = { .nr = ARRAY_SIZE(an_list),
					 .p = an_list }}
		};

		if (num_AutoNeg > 0) {
			an = AutoNeg[0];
			smac_validate_option(&an, &opt, adapter);
		} else {
			an = opt.def;
		}
		adapter->autoneg_advertised = an;
	}

	switch (speed + dplx) {
	case 0:
		adapter->autoneg = adapter->fc_autoneg = true;
		if (((num_Speed > 0) && (speed != 0 || dplx != 0))
			|| (num_Duplex > 0))
			SMAC_DPRINTK(PROBE, INFO,
			       "Speed and duplex autonegotiation enabled\n");
		break;
	case SMAC_HALF_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Half Duplex specified without Speed\n");
		SMAC_DPRINTK(PROBE, INFO, "Using Autonegotiation at "
			"Half Duplex only\n");
		adapter->autoneg = adapter->fc_autoneg = true;
		adapter->autoneg_advertised = SMAC_ADVERTISE_10_HALF |
					      SMAC_ADVERTISE_100_HALF;
		break;
	case SMAC_FULL_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Full Duplex specified without Speed\n");
		SMAC_DPRINTK(PROBE, INFO, "Using Autonegotiation at "
			"Full Duplex only\n");
		adapter->autoneg = adapter->fc_autoneg = true;
		adapter->autoneg_advertised = SMAC_ADVERTISE_10_FULL |
					      SMAC_ADVERTISE_100_FULL |
					      SMAC_ADVERTISE_1000_FULL;
		break;
	case SPEED_10:
		SMAC_DPRINTK(PROBE, INFO, "10 Mbps Speed specified "
			"without Duplex\n");
		SMAC_DPRINTK(PROBE, INFO, "Using Autonegotiation at 10 Mbps only\n");
		adapter->autoneg = adapter->fc_autoneg = true;
		adapter->autoneg_advertised = SMAC_ADVERTISE_10_HALF |
					      SMAC_ADVERTISE_10_FULL;
		break;
	case SPEED_10 + SMAC_HALF_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Forcing to 10 Mbps Half Duplex\n");
		adapter->autoneg = adapter->fc_autoneg = false;
		adapter->forced_speed_duplex = smac_10_half;
		adapter->autoneg_advertised = 0;
		break;
	case SPEED_10 + SMAC_FULL_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Forcing to 10 Mbps Full Duplex\n");
		adapter->autoneg = adapter->fc_autoneg = false;
		adapter->forced_speed_duplex = smac_10_full;
		adapter->autoneg_advertised = 0;
		break;
	case SPEED_100:
		SMAC_DPRINTK(PROBE, INFO, "100 Mbps Speed specified "
			"without Duplex\n");
		SMAC_DPRINTK(PROBE, INFO, "Using Autonegotiation at "
			"100 Mbps only\n");
		adapter->autoneg = adapter->fc_autoneg = true;
		adapter->autoneg_advertised = SMAC_ADVERTISE_100_HALF |
					      SMAC_ADVERTISE_100_FULL;
		break;
	case SPEED_100 + SMAC_HALF_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Forcing to 100 Mbps Half Duplex\n");
		adapter->autoneg = adapter->fc_autoneg = false;
		adapter->forced_speed_duplex = smac_100_half;
		adapter->autoneg_advertised = 0;
		break;
	case SPEED_100 + SMAC_FULL_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO, "Forcing to 100 Mbps Full Duplex\n");
		adapter->autoneg = adapter->fc_autoneg = false;
		adapter->forced_speed_duplex = smac_100_full;
		adapter->autoneg_advertised = 0;
		break;
	case SPEED_1000:
		SMAC_DPRINTK(PROBE, INFO, "1000 Mbps Speed specified without "
			"Duplex\n");
		goto full_duplex_only;
	case SPEED_1000 + SMAC_HALF_DUPLEX:
		SMAC_DPRINTK(PROBE, INFO,
			"Half Duplex is not supported at 1000 Mbps\n");
		/* fall through */
	case SPEED_1000 + SMAC_FULL_DUPLEX:
full_duplex_only:
		SMAC_DPRINTK(PROBE, INFO,
		       "Using Autonegotiation at 1000 Mbps Full Duplex only\n");
		adapter->autoneg = adapter->fc_autoneg = true;
		adapter->autoneg_advertised = SMAC_ADVERTISE_1000_FULL;
		break;
	default:
		BUG();
	}
}


/**
 * smac_print_options - print options.
 **/
#ifdef CONFIG_SMAC_DBG
void __devinit
smac_print_options(struct smac_adapter *adapter)
{
	if (adapter->autoneg) {
		SMAC_DEBUG_OUT("autonegotiaion=on\n");
	} else {
		switch (adapter->forced_speed_duplex) {
		case 0:
		case 1:
			SMAC_DEBUG_OUT("speed=10Mbps\n");
			break;
		case 2:
		case 3:
			SMAC_DEBUG_OUT("speed=100Mbps\n");
			break;
		default:
			BUG();
		}
		switch (adapter->forced_speed_duplex) {
		case 0:
		case 2:
			SMAC_DEBUG_OUT("duplex=half\n");
			break;
		case 1:
		case 3:
			SMAC_DEBUG_OUT("duplex=full\n");
			break;
		default:
			BUG();
		}
	}

	SMAC_DEBUG_OUT("autoneg_advertised=0x%08x\n", adapter->autoneg_advertised);
}
#endif


