/*
 *  drivers/net/snsc_macaddr.c
 *
 *  Copyright 2008 Sony Corporation
 *
 *  This program is free software; you can redistribute	 it and/or modify it
 *  under  the terms of	 the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the	License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED	  ``AS	IS'' AND   ANY	EXPRESS OR IMPLIED
 *  WARRANTIES,	  INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO	EVENT  SHALL   THE AUTHOR  BE	 LIABLE FOR ANY	  DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED	  TO, PROCUREMENT OF  SUBSTITUTE GOODS	OR SERVICES; LOSS OF
 *  USE, DATA,	OR PROFITS; OR	BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN	 CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/if.h>
#include <linux/snsc_macaddr.h>

#ifdef CONFIG_SNSC_MACADDR_MULTI
#define MACADDR_NR CONFIG_SNSC_NUM_MACADDRS

struct bi_macaddr {
	u8 macaddr[6];
	u8 type;
	u8 reserved;
	char drvname[IFNAMSIZ];
};

struct snsc_macaddr {
	char drvname[IFNAMSIZ];
	u8 macaddr[6];
};

static struct snsc_macaddr priv[MACADDR_NR];
#else
typedef struct snsc_macaddr {
	unsigned char enetaddr[6];     /* MAC addresses */
	int           enetaddr_valid;  /* enetaddr is valid or not */
} snsc_macaddr_t;

static snsc_macaddr_t macaddr;
#endif

/*
 * Get MAC address from macaddr
 */
int
snsc_macaddr_get(u8 data[], const char *name)
{
	int   i;

#ifdef CONFIG_SNSC_MACADDR_MULTI
	for (i = 0; i < MACADDR_NR; i++) {
		if (strncmp(name, priv[i].drvname, IFNAMSIZ) == 0) {
			memcpy(data, priv[i].macaddr, 6);
			return 0;
		}
	}
	for (i = 0; i < MACADDR_NR; i++) {
		if (strncmp("default", priv[i].drvname, IFNAMSIZ) == 0) {
			memcpy(data, priv[i].macaddr, 6);
			return 0;
		}
	}
#else
	if (macaddr.enetaddr_valid) {
		for (i = 0; i < 6; i++) {
			data[i] = macaddr.enetaddr[i];
		}
		return 0;
	}
#endif
	printk("!!! Ethernet MAC address retrieval failed. !!!\n");
	printk("!!! Please use NBL to set MAC address.     !!!\n");
	printk("\n");
	do {} while (1);

	return -1;
}
EXPORT_SYMBOL(snsc_macaddr_get);

/*
 * Copy MAC address from bootinfo to macaddr
 *   This function is called from parse_bootinfo() in arch/mips/kernel/setup.c
 */
void __init
snsc_macaddr_init(u8 *data, int size)
{
#ifdef CONFIG_SNSC_MACADDR_MULTI
	int i = 0;
	u8 *end = data + size;

	memset(priv, 0, sizeof(struct bi_macaddr) * MACADDR_NR);

	while (i < MACADDR_NR && data < end) {
		struct bi_macaddr *entry = (struct bi_macaddr *)data;

		if (entry->type != 1 && entry->type != 2) { /* invalid type */
			data += 8;
			continue;
		}
		memcpy(priv[i].macaddr, entry->macaddr, 6);
		if (entry->type == 2) {
			memcpy(priv[i].drvname, entry->drvname, IFNAMSIZ);
			data += sizeof(struct bi_macaddr);
		}
		else {	/* entry->type == 1 */
			strncpy(priv[i].drvname, "default", IFNAMSIZ);
			data += 8;
		}
		i++;
	}
#else
	/* sanity check */
	if (size < 8) {
		macaddr.enetaddr_valid = 0;
		return;
	}

	memcpy(macaddr.enetaddr, data, 6);
	macaddr.enetaddr_valid = *(data + 6);
#endif
}
