/*
 *  File Name       : drivers/power/emxx_battery.c
 *  Function        : EMMA Mobile series dummy Battery
 *  Release Version : Ver 1.00
 *  Release Date    : 2010/02/19
 *
 *  Copyright (C) NEC Electronics Corporation 2010
 *
 *  This program is free software;you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by Free
 *  Softwere Foundation; either version 2 of License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; If not, write to the Free Software Foundation, Inc., 59
 *  Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/types.h>
#ifdef CONFIG_EMXX_ANDROID
#include <linux/wakelock.h>
#endif
#include <linux/io.h>

#include <mach/charger.h>

typedef enum {
	CHARGER_BATTERY = 0,
	CHARGER_USB,
	CHARGER_AC
} charger_type_t;

struct emxx_battery_data {
	struct power_supply battery;
	struct power_supply usb;
	struct power_supply ac;

	charger_type_t charger;

	unsigned int battery_present;
	unsigned int level;
	int usb_state;
	spinlock_t lock;
	int lock_status;

#ifdef CONFIG_EMXX_ANDROID
	struct wake_lock vbus_suspend_lock;
#endif
};

static struct emxx_battery_data *battery_data;

static enum power_supply_property emxx_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_CAPACITY,
};

static enum power_supply_property emxx_power_props[] = {
	POWER_SUPPLY_PROP_ONLINE,
};

static unsigned int emxx_charger_get_status(void)
{
	unsigned int status = 0;

	battery_data->charger = CHARGER_AC;
	battery_data->battery_present = 1;

	return status;
}

static int emxx_battery_get_status(void)
{
	int ret;

	switch (battery_data->charger) {
	case CHARGER_BATTERY:
		ret = POWER_SUPPLY_STATUS_NOT_CHARGING;
		break;
	case CHARGER_USB:
	case CHARGER_AC:
		if (battery_data->level == 100)
			ret = POWER_SUPPLY_STATUS_FULL;
		else
			ret = POWER_SUPPLY_STATUS_CHARGING;
		break;
	default:
		ret = POWER_SUPPLY_STATUS_UNKNOWN;
	}
	return ret;
}

static int emxx_battery_get_property(struct power_supply *psy,
				 enum power_supply_property psp,
				 union power_supply_propval *val)
{
	int ret = 0;

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:		/* 0 */
		val->intval = emxx_battery_get_status();
		break;
	case POWER_SUPPLY_PROP_HEALTH:		/* 1 */
		val->intval = POWER_SUPPLY_HEALTH_GOOD;
		break;
	case POWER_SUPPLY_PROP_PRESENT:		/* 2 */
		val->intval = battery_data->battery_present;
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:	/* 4 */
		val->intval = POWER_SUPPLY_TECHNOLOGY_LION;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:	/* 26 */
		val->intval = battery_data->level;
		break;
	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}

static int emxx_power_get_property(struct power_supply *psy,
			enum power_supply_property psp,
			union power_supply_propval *val)
{
	charger_type_t charger;

	charger = battery_data->charger;

	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:	/* 3 */
		if (psy->type == POWER_SUPPLY_TYPE_MAINS)
			val->intval = (charger ==  CHARGER_AC ? 1 : 0);
		else if (psy->type == POWER_SUPPLY_TYPE_USB) {
			if (battery_data->usb_state == EMXX_USB_OFF)
				val->intval = 0;
			else
				val->intval = 1;
		} else
			val->intval = 0;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

int emxx_battery_usb_state(int state)
{
	unsigned long flags;
	int ret = 0;

	spin_lock_irqsave(&battery_data->lock, flags);

	if (battery_data->usb_state != state) {
		switch (state) {
		case EMXX_USB_DEVICE:
			if (battery_data->lock_status == 0) {
				battery_data->lock_status = 1;
#ifdef CONFIG_EMXX_ANDROID
				wake_lock(&battery_data->vbus_suspend_lock);
#endif
			}
			break;
		case EMXX_USB_CHARGER:
			if (battery_data->lock_status == 0) {
				battery_data->lock_status = 1;
#ifdef CONFIG_EMXX_ANDROID
				wake_lock(&battery_data->vbus_suspend_lock);
#endif
			}
			break;
		case EMXX_USB_OFF:
			if (battery_data->lock_status == 1) {
				battery_data->lock_status = 0;
#ifdef CONFIG_EMXX_ANDROID
				wake_unlock(&battery_data->vbus_suspend_lock);
#endif
			}
			break;
		default:
			printk(KERN_ERR "%s: error state. (%d)\n",
				__func__, state);
			ret = -EINVAL;
			goto err;
		}
		battery_data->usb_state = state;
	}
err:
	spin_unlock_irqrestore(&battery_data->lock, flags);
	return ret;
}
EXPORT_SYMBOL(emxx_battery_usb_state);

static int emxx_battery_probe(struct platform_device *pdev)
{
	int ret;
	struct emxx_battery_data *data;
	unsigned int status;

	data = kzalloc(sizeof(*data), GFP_KERNEL);
	if (data == NULL) {
		ret = -ENOMEM;
		goto err_data_alloc_failed;
	}
	spin_lock_init(&data->lock);

	/* Battey */
	data->battery.name = "battery";
	data->battery.type = POWER_SUPPLY_TYPE_BATTERY;
	data->battery.properties = emxx_battery_props;
	data->battery.num_properties = ARRAY_SIZE(emxx_battery_props);
	data->battery.get_property = emxx_battery_get_property;

	/* USB */
	data->usb.name = "usb";
	data->usb.type = POWER_SUPPLY_TYPE_USB;
	data->usb.properties = emxx_power_props;
	data->usb.num_properties = ARRAY_SIZE(emxx_power_props);
	data->usb.get_property = emxx_power_get_property;

	/* AC */
	data->ac.name = "ac";
	data->ac.type = POWER_SUPPLY_TYPE_MAINS;
	data->ac.properties = emxx_power_props;
	data->ac.num_properties = ARRAY_SIZE(emxx_power_props);
	data->ac.get_property = emxx_power_get_property;

	battery_data = data;

	/* Dummy battery setting (always 100%) */
	battery_data->level = 100;
	battery_data->usb_state = EMXX_USB_OFF;
	battery_data->lock_status = 0;

#ifdef CONFIG_EMXX_ANDROID
	wake_lock_init(&data->vbus_suspend_lock,
			WAKE_LOCK_SUSPEND, "vbus_suspend");
#endif

	status = emxx_charger_get_status();

	ret = power_supply_register(&pdev->dev, &data->battery);
	if (ret)
		goto err_battery_failed;

	ret = power_supply_register(&pdev->dev, &data->usb);
	if (ret)
		goto err_usb_failed;

	ret = power_supply_register(&pdev->dev, &data->ac);
	if (ret)
		goto err_ac_failed;

	platform_set_drvdata(pdev, data);

	return 0;

err_ac_failed:
	power_supply_unregister(&data->usb);
err_usb_failed:
	power_supply_unregister(&data->battery);
err_battery_failed:
	kfree(data);
err_data_alloc_failed:
	return ret;
}

static int emxx_battery_remove(struct platform_device *pdev)
{
	struct emxx_battery_data *data = platform_get_drvdata(pdev);

#ifdef CONFIG_EMXX_ANDROID
	wake_lock_destroy(&data->vbus_suspend_lock);
#endif

	power_supply_unregister(&data->battery);
	power_supply_unregister(&data->usb);
	power_supply_unregister(&data->ac);

	kfree(data);
	battery_data = NULL;
	return 0;
}

static struct platform_driver emxx_battery_driver = {
	.probe		= emxx_battery_probe,
	.remove		= emxx_battery_remove,
	.driver = {
		.name = "emxx-battery"
	}
};

static int __init emxx_battery_init(void)
{
	return platform_driver_register(&emxx_battery_driver);
}

static void __exit emxx_battery_exit(void)
{
	platform_driver_unregister(&emxx_battery_driver);
}

module_init(emxx_battery_init);
module_exit(emxx_battery_exit);

MODULE_AUTHOR("NECEL");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Battery driver for the EMMA Mobile series");
