/*
 *  File Name       : drivers/rtc/rtc-emxx.c
 *  Function        : EMXX Real Time Clock interface
 *  Release Version : Ver 1.01
 *  Release Date    : 2010/07/15
 *
 *  Copyright (C) Renesas Electronics Corporation 2010
 *
 *
 * This program is free software;you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by Free Softwere
 * Foundation; either version 2 of License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; If not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/rtc.h>
#include <linux/bcd.h>
#include <linux/log2.h>
#include <linux/delay.h>
#include <linux/uaccess.h>
#include <linux/i2c.h>

#include <mach/hardware.h>
#include <mach/pwc.h>

#ifndef BCD2BIN
#define BCD2BIN(val)	bcd2bin(val)
#endif
#ifndef BIN2BCD
#define BIN2BCD(val)	bin2bcd(val)
#endif

static DEFINE_SPINLOCK(emxx_rtc_lock);

#define RTC_SEC			DA9052_COUNTS_REG
#define RTC_SEC_RD		DA9052_COUNTS_REG
#define RTC_MIN			DA9052_COUNTMI_REG
#define RTC_MIN_RD		DA9052_COUNTMI_REG
#define RTC_HOUR		DA9052_COUNTH_REG
#define RTC_HOUR_RD		DA9052_COUNTH_REG
#define RTC_WEEK		0
#define RTC_WEEK_RD		0
#define RTC_DAY			DA9052_COUNTD_REG
#define RTC_DAY_RD		DA9052_COUNTD_REG
#define RTC_MONTH		DA9052_COUNTMO_REG
#define RTC_MONTH_RD		DA9052_COUNTMO_REG
#define RTC_YEAR		DA9052_COUNTY_REG	/* 2000-2063 */
#define RTC_YEAR_RD		DA9052_COUNTY_REG
#define RTC_ALM_MIN		DA9052_ALARMMI_REG
#define RTC_ALM_MIN_RD		DA9052_ALARMMI_REG
#define RTC_ALM_HOUR		DA9052_ALARMH_REG
#define RTC_ALM_HOUR_RD		DA9052_ALARMH_REG
#define RTC_ALM_WEEK		0
#define RTC_ALM_WEEK_RD		0
#define RTC_ALM_DAY		DA9052_ALARMD_REG
#define RTC_ALM_DAY_RD		DA9052_ALARMD_REG
#define RTC_ALM_MONTH		DA9052_ALARMMO_REG
#define RTC_ALM_MONTH_RD	DA9052_ALARMMO_REG
#define RTC_ALM_YEAR		DA9052_ALARMY_REG
#define RTC_ALM_YEAR_RD		DA9052_ALARMY_REG

#define RTC_ALARM_TYPE_ALM	0x40
#define RTC_TICK_TYPE_MIN	0x80
#define RTC_ALARMY_ALARM_ON	0x40
#define RTC_ALARMY_TICK_ON	0x80

/* RTC register access macros: */
#define RTC_READ(addr, val) pwc_read((addr), &(val));
#define RTC_WRITE(val, addr) pwc_reg_write((addr), (val)&0xff);

static unsigned long epoch = 1900;	/* year corresponding to 0x00 */

static const unsigned char days_in_mo[] =
    { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

static int rtc_irq;
static int p_freq = 60;
static int write_jiff;
static struct rtc_device *rtc_dev;
static struct i2c_client *rtc_client;

static int emxx_backup_rtc_read(struct rtc_time *rtc_tm)
{
	unsigned char val[7];
	int ret;

	ret = i2c_smbus_read_i2c_block_data(rtc_client, 2, 7, val);
	if (ret < 0)
		return ret;

	if (val[0] & 0x80)
		return -EIO;

	val[0] &= 0x7f;
	val[1] &= 0x7f;
	val[2] &= 0x3f;
	val[3] &= 0x3f;
	val[5] &= 0x1f;
	rtc_tm->tm_sec  = bcd2bin(val[0]);
	rtc_tm->tm_min  = bcd2bin(val[1]);
	rtc_tm->tm_hour = bcd2bin(val[2]);
	rtc_tm->tm_mday = bcd2bin(val[3]);
	rtc_tm->tm_mon  = bcd2bin(val[5]);
	rtc_tm->tm_year = bcd2bin(val[6]);

	rtc_tm->tm_year += 100;
	rtc_tm->tm_mon--;

	return 0;
}

static int emxx_backup_rtc_set(struct rtc_time *rtc_tm)
{
	unsigned char val[7];
	int ret;

	ret = i2c_smbus_write_byte_data(rtc_client, 0, 0x20);
	if (ret < 0)
		return ret;

	val[0] = bin2bcd(rtc_tm->tm_sec);
	val[1] = bin2bcd(rtc_tm->tm_min);
	val[2] = bin2bcd(rtc_tm->tm_hour);
	val[3] = bin2bcd(rtc_tm->tm_mday);
	val[4] = 0;
	val[5] = bin2bcd(rtc_tm->tm_mon + 1);
	val[6] = bin2bcd(rtc_tm->tm_year - 100);

	ret = i2c_smbus_write_i2c_block_data(rtc_client, 2, 7, val);
	if (ret < 0)
		return ret;

	return i2c_smbus_write_byte_data(rtc_client, 0, 0);
}

static int emxx_backup_rtc_init(void)
{
	unsigned char val[2] = {0, 0};

	return i2c_smbus_write_i2c_block_data(rtc_client, 0, 2, val);
}

static int emxx_read_time(struct device *dev, struct rtc_time *rtc_tm)
{
	unsigned long flags;

	if (write_jiff) {
		if ((jiffies - write_jiff) < HZ/2) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(HZ - (jiffies - write_jiff));
		}
		write_jiff = 0;
	}

	spin_lock_irqsave(&emxx_rtc_lock, flags);
	RTC_READ(RTC_SEC_RD, rtc_tm->tm_sec);
	RTC_READ(RTC_MIN_RD, rtc_tm->tm_min);
	RTC_READ(RTC_HOUR_RD, rtc_tm->tm_hour);
	RTC_READ(RTC_DAY_RD, rtc_tm->tm_mday);
	RTC_READ(RTC_MONTH_RD, rtc_tm->tm_mon);
	RTC_READ(RTC_YEAR_RD, rtc_tm->tm_year);
	spin_unlock_irqrestore(&emxx_rtc_lock, flags);

	rtc_tm->tm_sec &= 0x3f;	/* Mask of bit0-bit5 */

	if ((rtc_tm->tm_year + (epoch - 1900)) <= 69)
		rtc_tm->tm_year += 100;
	rtc_tm->tm_mon--;

	return 0;
}

static int emxx_set_time(struct device *dev, struct rtc_time *rtc_tm)
{
	unsigned char mon, day, hrs, min, sec, leap_yr;
	unsigned int yrs;
	unsigned long flags;

	yrs = rtc_tm->tm_year + 1900;
	mon = rtc_tm->tm_mon + 1;	/* tm_mon starts at zero */
	day = rtc_tm->tm_mday;
	hrs = rtc_tm->tm_hour;
	min = rtc_tm->tm_min;
	sec = rtc_tm->tm_sec;

	if ((yrs < 2000) || (yrs > 2063))
		return -EINVAL;

	if ((mon > 12) || (day == 0))
		return -EINVAL;

	leap_yr = ((!(yrs % 4) && (yrs % 100)) || !(yrs % 400));
	if (day > (days_in_mo[mon] + ((mon == 2) && leap_yr)))
		return -EINVAL;
	if ((hrs >= 24) || (min >= 60) || (sec >= 60))
		return -EINVAL;
	yrs -= 2000;

	spin_lock_irqsave(&emxx_rtc_lock, flags);
	RTC_WRITE(mon,  RTC_MONTH);
	RTC_WRITE(day,  RTC_DAY);
	RTC_WRITE(hrs,  RTC_HOUR);
	RTC_WRITE(min,  RTC_MIN);
	RTC_WRITE(sec,  RTC_SEC);
	RTC_WRITE(yrs,  RTC_YEAR);
	spin_unlock_irqrestore(&emxx_rtc_lock, flags);

	write_jiff = jiffies;

	emxx_backup_rtc_set(rtc_tm);

	return 0;
}

static inline void emxx_rtc_setaie(unsigned int enable)
{
	unsigned int rtc_val;

	RTC_READ(RTC_ALM_YEAR_RD, rtc_val);
	if (enable)
		rtc_val |= RTC_ALARMY_ALARM_ON;
	else
		rtc_val &= ~RTC_ALARMY_ALARM_ON;
	RTC_WRITE(rtc_val, RTC_ALM_YEAR);
}

static inline void emxx_rtc_setpie(unsigned int enable)
{
	unsigned int rtc1_val, rtc2_val;

	disable_irq(rtc_irq);

	RTC_READ(RTC_ALM_MIN_RD,  rtc1_val);
	RTC_READ(RTC_ALM_YEAR_RD, rtc2_val);
	if (enable) {
		rtc2_val |= RTC_ALARMY_TICK_ON;
		if (p_freq == 60) {
			/* one minute */
			rtc1_val |= RTC_TICK_TYPE_MIN;
		} else {
			/* one second */
			rtc1_val &= ~RTC_TICK_TYPE_MIN;
		}
	} else {
		rtc2_val &= ~RTC_ALARMY_TICK_ON;
	}
	RTC_WRITE(rtc1_val, RTC_ALM_MIN);
	RTC_WRITE(rtc2_val, RTC_ALM_YEAR);

	udelay(61);
	enable_irq(rtc_irq);
}

static inline int emxx_rtc_setfreq(int freq)
{
	switch (freq) {
	case 1:
	case 60:
		break;
	default:
		return -EINVAL;
	}

	p_freq = freq;

	return 0;
}

static int emxx_rtc_getalarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct rtc_time *alm_tm = &alrm->time;
	unsigned long flags;

	spin_lock_irqsave(&emxx_rtc_lock, flags);
	RTC_READ(RTC_ALM_MIN_RD,  alm_tm->tm_min);
	RTC_READ(RTC_ALM_HOUR_RD, alm_tm->tm_hour);
	RTC_READ(RTC_ALM_DAY_RD,   alm_tm->tm_mday);
	RTC_READ(RTC_ALM_MONTH_RD, alm_tm->tm_mon);
	RTC_READ(RTC_ALM_YEAR_RD,  alm_tm->tm_year);
	spin_unlock_irqrestore(&emxx_rtc_lock, flags);

	alm_tm->tm_min  &= 0x3f;		/* Mask of bit0-bit5 */
	alm_tm->tm_year &= 0x3f;		/* Mask of bit0-bit5 */
	alm_tm->tm_mon--;
	alm_tm->tm_year += 100;

	return 0;
}

static int emxx_rtc_setalarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct rtc_time *alm_tm = &alrm->time;
	unsigned int hrs, min, day, mon, yrs, leap_yr;
	unsigned long flags;

	hrs = alm_tm->tm_hour;
	min = alm_tm->tm_min;
	yrs = alm_tm->tm_year + 1900;
	mon  = alm_tm->tm_mon + 1;
	day  = alm_tm->tm_mday;

	if ((yrs < 2000) || (yrs > 2063))
		return -EINVAL;
	if (hrs >= 24)
		return -EINVAL;
	if (min >= 60)
		return -EINVAL;

	leap_yr = ((!(yrs % 4) && (yrs % 100)) || !(yrs % 400));
	if (day > (days_in_mo[mon] + ((mon == 2) && leap_yr)))
		return -EINVAL;
	if (mon >= 60)
		return -EINVAL;
	yrs -= 2000;

	spin_lock_irqsave(&emxx_rtc_lock, flags);
	RTC_WRITE(hrs,  RTC_ALM_HOUR);
	pwc_write(RTC_ALM_MIN, min, 0x3f);
	RTC_WRITE(day,  RTC_ALM_DAY);
	RTC_WRITE(mon,  RTC_ALM_MONTH);
	pwc_write(RTC_ALM_YEAR, yrs, 0x3f);
	spin_unlock_irqrestore(&emxx_rtc_lock, flags);

	if (alrm->enabled)
		emxx_rtc_setaie(1);
	else
		emxx_rtc_setaie(0);

	return 0;
}

int emxx_set_reboot_alarm(void)
{
	p_freq = 1;	/* second */
	emxx_rtc_setpie(1);	/* enable */

	return 0;
}

static irqreturn_t emxx_rtc_interrupt(int irq, void *dev_id)
{
	struct rtc_device *rtc = dev_id;
	unsigned long events = RTC_IRQF;
	unsigned int int_val;

	RTC_READ(RTC_ALM_MIN_RD,  int_val);

	if (int_val & RTC_ALARM_TYPE_ALM) {
		/* caused by timer alarm */
		events |= RTC_AF;
		RTC_READ(RTC_ALM_YEAR_RD,  int_val);
		if (int_val & RTC_ALARMY_TICK_ON)
			events |= RTC_PF;
	} else {
		/* caused by TICK */
		events |= RTC_PF;
	}

	rtc_update_irq(rtc, 1, events);

	return IRQ_HANDLED;
}


static int emxx_rtc_ioctl(struct device *dev, unsigned int cmd,
 unsigned long arg)
{
	switch (cmd) {
	case RTC_AIE_ON:
	case RTC_AIE_OFF:
		emxx_rtc_setaie((cmd == RTC_AIE_ON) ? 1 : 0);
		break;

	case RTC_PIE_ON:
	case RTC_PIE_OFF:
		emxx_rtc_setpie((cmd == RTC_PIE_ON) ? 1 : 0);
		break;

	case RTC_IRQP_SET:
		return emxx_rtc_setfreq(arg);

	case RTC_IRQP_READ:
		return  put_user(p_freq, (unsigned long __user *)arg);

	default:
		return -ENOIOCTLCMD;
	}

	return 0;
}


static int emxx_rtc_open(struct device *dev)
{
	return 0;
}

static void emxx_rtc_release(struct device *dev)
{
	emxx_rtc_setpie(0);
}

static struct rtc_class_ops emxx_rtcops = {
	.open       = emxx_rtc_open,
	.release    = emxx_rtc_release,
	.ioctl      = emxx_rtc_ioctl,
	.read_time  = emxx_read_time,
	.set_time   = emxx_set_time,
	.read_alarm = emxx_rtc_getalarm,
	.set_alarm  = emxx_rtc_setalarm,
};

static int
emxx_rtc_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct rtc_time rtc_tm;
	unsigned int rtc2_val;
	int ret;

	if (!i2c_check_functionality(client->adapter,
			I2C_FUNC_SMBUS_I2C_BLOCK)) {
		return -EIO;
	}

	RTC_READ(RTC_ALM_YEAR_RD, rtc2_val);
	rtc2_val &= ~RTC_ALARMY_TICK_ON;
	RTC_WRITE(rtc2_val, RTC_ALM_YEAR);

	rtc_client = client;

	emxx_backup_rtc_init();
	if (emxx_backup_rtc_read(&rtc_tm) == 0)
		emxx_set_time(NULL, &rtc_tm);

	rtc_irq = INT_RTCINT;
	rtc_dev = rtc_device_register(client->name, &client->dev,
			&emxx_rtcops, THIS_MODULE);
	if (IS_ERR(rtc_dev)) {
		pr_debug("%s: can't register RTC device, err %ld\n",
			client->name, PTR_ERR(rtc_dev));
		return -EIO;
	}

	ret = request_irq(rtc_irq, emxx_rtc_interrupt,
			IRQF_DISABLED, "rtc", rtc_dev);
	if (ret < 0) {
		printk(KERN_ERR "rtc: request_irq error! (%d)\n", rtc_irq);
		goto err_irq;
	}

	return 0;

err_irq:
	rtc_device_unregister(rtc_dev);
	return ret;
}

static int emxx_rtc_remove(struct i2c_client *client)
{
	rtc_device_unregister(rtc_dev);
	free_irq(rtc_irq, rtc_dev);

	return 0;
}

static struct i2c_device_id emxx_rtc_idtable[] = {
	{I2C_SLAVE_RTC_NAME, 0},
	{ }
};

static struct i2c_driver emxx_rtcdrv = {
	.driver = {
		.name	= "emxx-rtc",
		.owner	= THIS_MODULE,
	},
	.id_table	= emxx_rtc_idtable,
	.probe		= emxx_rtc_probe,
	.remove		= __devexit_p(emxx_rtc_remove),
};

static int __init rtc_init(void)
{
	return i2c_add_driver(&emxx_rtcdrv);

}
module_init(rtc_init);

static void __exit rtc_exit(void)
{
	i2c_del_driver(&emxx_rtcdrv);
}
module_exit(rtc_exit);

MODULE_DESCRIPTION("EMXX series RTC Driver");
MODULE_LICENSE("GPL");
