/*
 *  Copyright 2010 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <linux/hdreg.h>
#include <linux/blkdev.h>
#include <linux/dma-mapping.h>
#include <linux/buffer_head.h>
#include <linux/loop.h>
#include <linux/udif/bdev.h>
#include <linux/udif/boottime.h>
#include <linux/udif/macros.h>
#include <linux/udif/pm.h>
#include <asm/uaccess.h>

#define DISK_DATA(x)	((x)->private_data)
#define BDEV_DATA(x)	DISK_DATA((x)->bd_disk)
#define INODE_DATA(x)	BDEV_DATA((x)->i_bdev)
#define DISK_DEVNO(x)	MKDEV((x)->major, (x)->first_minor)

#ifdef CONFIG_BLK_DEV_DECOMPRESS
# define UDIF_DECLARE_REQUEST(name, q, req, s) \
UDIF_REQUEST name = { \
	.write		= rq_data_dir(req), \
	.sector		= (req)->sector, \
	.nr_sectors	= (req)->nr_sectors, \
	.buffer		= (req)->buffer, \
	.sg		= s, \
	.rq		= req, \
	.que		= q, \
	.dec		= 0, \
}

static unsigned int udif_read_decompress(struct block_device *bdev,
					 sector_t sector, unsigned long nr_sectors,
					 struct disk_decompress *dec)
{
	UDIF_BDEV *dev = DISK_DATA(bdev->bd_disk);
	UDIF_DECLARE_FILE(fl, DISK_DEVNO(bdev->bd_disk), &dev->data);
	unsigned int xferred = 0;
	int ret;
	UDIF_REQUEST rq = {
		.write		= 0,
		.sector		= sector,
		.nr_sectors	= nr_sectors,
		.buffer		= 0,
		.sg		= 0,
		.rq		= 0,
		.dec		= dec,
	};

	ret = UDIF_CALL_FN(dev, request, int, -ENOSYS, &fl, &rq, &xferred);

	if (unlikely(ret))
		return 0;

	return dec->dstlen;
}

# else
#define UDIF_DECLARE_REQUEST(name, q, req, s) \
UDIF_REQUEST name = { \
	.write		= rq_data_dir(req), \
	.sector		= (req)->sector, \
	.nr_sectors	= (req)->nr_sectors, \
	.buffer		= (req)->buffer, \
	.sg		= s, \
	.rq		= req, \
	.que		= q, \
}

#endif /* CONFIG_BLD_DEV_COMPRESS */

void udif_bdev_map_rq(UDIF_REQUEST *urq)
{
	urq->sg_len = blk_rq_map_sg(urq->que, urq->rq, urq->sg);
	urq->sg_len = dma_map_sg(NULL, urq->sg, urq->sg_len, urq->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
}

void udif_bdev_unmap_rq(UDIF_REQUEST *urq)
{
	if (!urq->write)
		dma_map_sg(NULL, urq->sg, urq->sg_len, DMA_FROM_DEVICE);

	dma_unmap_sg(NULL, urq->sg, urq->sg_len, urq->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
}

static void udif_request(struct request_queue *q)
{
	struct request *rq;

	while ((rq = elv_next_request(q))) {
		UDIF_BDEV *dev = DISK_DATA(rq->rq_disk);
		unsigned int xferred = 0;
		int ret;
		UDIF_DECLARE_FILE(fl, DISK_DEVNO(rq->rq_disk), &dev->data);
		UDIF_DECLARE_REQUEST(urq, q, rq, dev->sg);

		if (!blk_fs_request(rq)) {
			end_request(rq, 0);
			continue;
		}

		if (urq.sg && !(dev->flags & UDIF_BDEV_FLAGS_XMAPSG)) {
			udif_bdev_map_rq(&urq);
		}

		spin_unlock_irq(q->queue_lock);

		ret = UDIF_CALL_FN(dev, request, int, -ENOSYS, &fl, &urq, &xferred);

		spin_lock_irq(q->queue_lock);

		if (urq.sg && !(dev->flags & UDIF_BDEV_FLAGS_XMAPSG))
			udif_bdev_unmap_rq(&urq);

		if (unlikely(ret)) {
			rq->hard_cur_sectors = rq->hard_nr_sectors - xferred;
			if (ret > 0)
				ret = -EIO;
		} else {
			rq->hard_cur_sectors = xferred;
			ret = 1;
		}

		end_request(rq, ret);
	}
}

static int udif_open(struct block_device *bdev, fmode_t mode)
{
	UDIF_BDEV *dev = BDEV_DATA(bdev);
	UDIF_DECLARE_FILE(fl, bdev->bd_dev, &dev->data);
	UDIF_ERR err;

	/* avoid to exec rmmod while openning driver */
	if (unlikely(!try_module_get(dev->owner))) {
		UDIF_PERR("Unable get module %s\n", dev->node->name);
		return UDIF_ERR_IO;
	}

	err = UDIF_CALL_FN(dev, open, UDIF_ERR, UDIF_ERR_OK, &fl);

	if (err != UDIF_ERR_OK) {
		module_put(dev->owner);
	}

	return err;
}

static int udif_close(struct gendisk *disk, fmode_t mode)
{
	UDIF_BDEV *dev = DISK_DATA(disk);
	UDIF_DECLARE_FILE(fl, DISK_DEVNO(disk), &dev->data);
	UDIF_ERR err;

	err = UDIF_CALL_FN(dev, close, UDIF_ERR, UDIF_ERR_OK, &fl);

	module_put(dev->owner);

	return err;
}

static int udif_getgeo(struct block_device *bdev, struct hd_geometry *hd_geo)
{
	UDIF_BDEV *dev = BDEV_DATA(bdev);
	UDIF_GEOMETRY geo;
	UDIF_DECLARE_FILE(fl, bdev->bd_dev, &dev->data);
	int ret;

	ret = UDIF_CALL_FN(dev, getgeo, UDIF_INT, -ENOSYS, &fl, &geo);
	if (!ret) {
		hd_geo->heads	= geo.heads;
		hd_geo->sectors	= geo.sectors;
		hd_geo->cylinders = geo.cylinders;
		if (geo.start == UDIF_GEOMETRY_START_DEFAULT)
			hd_geo->start = get_start_sect(bdev);
		else
			hd_geo->start = geo.start;
	}

	return ret;
}

static int udif_revalidate(struct gendisk *disk)
{
	UDIF_BDEV *dev = DISK_DATA(disk);
	UDIF_DECLARE_FILE(fl, DISK_DEVNO(disk), &dev->data);
	sector_t sectors = 0;
	int ret;

	ret = UDIF_CALL_FN(dev, revalidate, int, -ENOSYS, &fl, &sectors);
	if (!ret)
		set_capacity(disk, sectors);

	return ret;
}

static int udif_media_changed(struct gendisk *disk)
{
	UDIF_BDEV *dev = DISK_DATA(disk);
	UDIF_DECLARE_FILE(fl, DISK_DEVNO(disk), &dev->data);

	return UDIF_CALL_FN(dev, media_changed, int, -ENOSYS, &fl);
}

static int udif_ioctl(struct block_device *bdev, fmode_t mode,
		      unsigned int cmd, unsigned long arg)
{
	UDIF_BDEV *dev = BDEV_DATA(bdev);
	UDIF_DECLARE_FILE(fl, bdev->bd_dev, &dev->data);
	UDIF_DECLARE_IOCTL(ictl, cmd, arg);
	struct hd_geometry geo;
	int ret;

	if (cmd == LOOP_CLR_FD) {
		/* ignore */
		ret = 0;
	} else if (cmd == BLKFLSBUF) {
		lock_kernel();
		fsync_bdev(bdev);
		invalidate_bdev(bdev);
		unlock_kernel();
		ret = 0;
	} else if (cmd == HDIO_GETGEO) {
		ret = udif_getgeo(bdev, &geo);
		if (!ret) {
			if (access_ok(VERIFY_WRITE, arg, sizeof(geo)) &&
			    copy_to_user((long *)arg, &geo, sizeof(geo))) {
				UDIF_PERR("%s: HDIO_GETGEO access error\n", dev->node->name);
				ret = -EFAULT;
			}
		}
	} else {
		ret = UDIF_CALL_FN(dev, ioctl, int, -ENOSYS, &fl, &ictl);
		if (!ret && ictl.disk_change) {
			check_disk_change(bdev);
		}
	}

	return ret;
}

static struct block_device_operations udif_fops = {
	.open	= udif_open,
	.release = udif_close,
	.ioctl	= udif_ioctl,
	.getgeo	= udif_getgeo,
	.revalidate_disk = udif_revalidate,
	.media_changed	= udif_media_changed,
};

#include <linux/udif/list.h>
#include <linux/udif/mutex.h>
#include <linux/udif/driver.h>
#include <linux/suspend.h>

static UDIF_LIST_HEAD(bdev_driver_list);
static UDIF_DECLARE_MUTEX(bdev_driver_mtx);

struct bdev_driver {
	UDIF_LIST list;
	UDIF_BDEV *dev;
	UDIF_DRIVER drv;
};

static UDIF_ERR udif_bdev_suspend(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	struct bdev_driver *bdrv = data;
	struct gendisk *disk = bdrv->dev->disk;
	int i;

	for (i=0;i<30;i++) {
		struct block_device *bdev = bdget_disk(disk, i);
		struct super_block *sb;

		if (!bdev)
			continue;

		sb = get_super(bdev);
		if (!sb)
			continue;

		if (sb->s_flags & MS_RDONLY) {
			drop_super(sb);
			continue;
		}

		printk("sync %s partition %d\n", bdrv->drv.name, i);
		fsync_super(sb);
		drop_super(sb);
	}

	return UDIF_ERR_OK;
}

static UDIF_ERR udif_bdev_resume(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	struct bdev_driver *bdrv = data;
	struct gendisk *disk = bdrv->dev->disk;
	int i;

	for (i=0;i<30;i++) {
		struct block_device *bdev = bdget_disk(disk, i);
		struct super_block *sb;

		if (!bdev)
			continue;

		sb = get_super(bdev);
		if (!sb)
			continue;

		if (sb->s_flags & MS_RDONLY) {
			drop_super(sb);
			continue;
		}

		drop_super(sb);
		printk("invalidate %s partition %d\n", bdrv->drv.name, i);
		__invalidate_device(bdev);
	}

	return UDIF_ERR_OK;
}

static UDIF_DRIVER_OPS udif_bdev_ops = {
	.suspend = udif_bdev_suspend,
	.resume = udif_bdev_resume,
};

void udif_driver_add(UDIF_DRIVER *drv);
void udif_driver_del(UDIF_DRIVER *drv);
static void add_to_driver_list(UDIF_BDEV *dev)
{
	struct bdev_driver *bdrv = kzalloc(sizeof(*bdrv), GFP_KERNEL);
	struct gendisk *disk = dev->disk;

	bdrv->dev = dev;
	snprintf((void *)bdrv->drv.name, sizeof(bdrv->drv.name), "b-%d", disk->major);
	bdrv->drv.ops = &udif_bdev_ops;
	*(void **)(&bdrv->drv.data) = bdrv;

	udif_mutex_lock(&bdev_driver_mtx);
	udif_list_add_tail(&bdrv->list, &bdev_driver_list);
	udif_mutex_unlock(&bdev_driver_mtx);

	udif_driver_add(&bdrv->drv);
}

static void del_from_driver_list(UDIF_BDEV *dev)
{
	struct bdev_driver *bdrv;

	udif_mutex_lock(&bdev_driver_mtx);
	udif_list_for_each_entry(bdrv, &bdev_driver_list, list) {
		if (bdrv->dev == dev) {
			udif_list_del(&bdrv->list);
			udif_driver_del(&bdrv->drv);
			kfree(bdrv);
			break;
		}
	}
	udif_mutex_unlock(&bdev_driver_mtx);
}

#define BLK_SET_PARAM(disk, fn, param) \
({ \
	if (param) \
		fn((disk)->queue, param); \
})

UDIF_ERR udif_bdev_register(const UDIF_DEVNODE *node)
{
	UDIF_ERR ret;

	UDIF_PARM_CHK(!node, "invalid node", UDIF_ERR_PAR);
	UDIF_PARM_CHK(node->type != UDIF_TYPE_BDEV, "type is not block device", UDIF_ERR_PAR);

	ret = register_blkdev(node->major, node->name);
	if (unlikely(ret)) {
		UDIF_PERR("failed register_blkdev(): %s, major = %d\n", node->name, node->major);
		return UDIF_ERR_PAR;
	}
	return UDIF_ERR_OK;
}

void udif_bdev_unregister(const UDIF_DEVNODE *node)
{
	if (unlikely(!node)) {
		UDIF_PERR("invalid node\n");
		return;
	}

	if (unlikely(node->type != UDIF_TYPE_BDEV)) {
		UDIF_PERR("type is not block device\n");
		return;
	}

	unregister_blkdev(node->major, node->name);
}

UDIF_ERR __udif_bdev_add_disk(UDIF_BDEV *dev)
{
	struct gendisk *disk;

	UDIF_PARM_CHK(!dev, "invalid dev", UDIF_ERR_PAR);
	UDIF_PARM_CHK(!dev->ops, "invalid ops", UDIF_ERR_PAR);
	UDIF_PARM_CHK(!dev->sg, "invalid sg", UDIF_ERR_PAR);
	UDIF_PARM_CHK(!dev->node, "invalid node", UDIF_ERR_PAR);
	UDIF_PARM_CHK(dev->node->type != UDIF_TYPE_BDEV, "type is not block device", UDIF_ERR_PAR);
	UDIF_PARM_CHK(dev->disk, "already added disk", UDIF_ERR_PAR);

	dev->disk = disk = alloc_disk(dev->node->nr_minor);
	if (unlikely(!disk)) {
		UDIF_PERR("failed alloc_disk(): %s, nr_minor = %d\n", dev->node->name, dev->node->nr_minor);
		return UDIF_ERR_NOMEM;
	}

	spin_lock_init(&dev->lock);
	disk->queue = blk_init_queue(udif_request, &dev->lock);
	if (unlikely(!disk->queue)) {
		UDIF_PERR("failed blk_init_queue(): %s\n", dev->node->name);
		del_gendisk(disk);
		return UDIF_ERR_NOMEM;
	}

	disk->major = dev->node->major;
	disk->first_minor = dev->node->first_minor;
	disk->minors = dev->node->nr_minor;
	disk->fops = &udif_fops;
	disk->flags = dev->flags;
	disk->private_data = dev;

#ifdef CONFIG_BLK_DEV_DECOMPRESS
	disk->algo = dev->algo;
	if (disk->algo)
		disk->blk_read_decompress = udif_read_decompress;
#endif

	snprintf(disk->disk_name, sizeof(disk->disk_name), \
		 "%s%d", dev->node->name, disk->first_minor);

	BLK_SET_PARAM(disk, blk_queue_hardsect_size, dev->sector_size);
	BLK_SET_PARAM(disk, blk_queue_max_sectors, dev->max_sectors);
	BLK_SET_PARAM(disk, blk_queue_max_segment_size, dev->max_segment_size);
	BLK_SET_PARAM(disk, blk_queue_max_hw_segments, dev->max_hw_segments);
	BLK_SET_PARAM(disk, blk_queue_max_phys_segments, dev->phys_segments);
	BLK_SET_PARAM(disk, blk_queue_dma_alignment, dev->dma_alignment);

	set_capacity(disk, dev->capacity);
	add_disk(disk);

	add_to_driver_list(dev);

	return UDIF_ERR_OK;
}

UDIF_ERR udif_bdev_del_disk(UDIF_BDEV *dev)
{
	struct gendisk *disk;
	struct request_queue *q;

	UDIF_PARM_CHK(!dev, "invalid dev", UDIF_ERR_PAR);
	UDIF_PARM_CHK(!dev->disk, "invalid disk", UDIF_ERR_PAR);

	del_from_driver_list(dev);

	disk = dev->disk;
	q = disk->queue;

	del_gendisk(disk);
	if (q)
		blk_cleanup_queue(q);
	put_disk(disk);

	dev->disk = NULL;

	return UDIF_ERR_OK;
}

UDIF_ERR udif_bdev_fill_buffer(UDIF_REQUEST *req, UDIF_U8 data, UDIF_UINT nsect, UDIF_UINT *xferred)
{
	return UDIF_ERR_OK;
}

EXPORT_SYMBOL(udif_bdev_register);
EXPORT_SYMBOL(udif_bdev_unregister);
EXPORT_SYMBOL(__udif_bdev_add_disk);
EXPORT_SYMBOL(udif_bdev_del_disk);
EXPORT_SYMBOL(udif_bdev_fill_buffer);
EXPORT_SYMBOL(udif_bdev_map_rq);
EXPORT_SYMBOL(udif_bdev_unmap_rq);
