/*
 * drivers/usb/gadget/emxx_udc.c
 *     EMXX FCD (Function Controller Driver) for USB.
 *
 * Copyright (C) 2010 Renesas Electronics Corporation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2
 *  as published by the Free Softwere Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warrnty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Frankling Street, Fifth Floor, Boston, MA 02111-1301, USA.
*/


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/clk.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/dma-mapping.h>
#include <linux/workqueue.h>

#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>

#include <asm/irq.h>
#include <asm/system.h>
#include <asm/uaccess.h>

#include <mach/hardware.h>
#include <mach/smu.h>
#include <mach/pmu.h>
#include <mach/gpio.h>
#include <asm/dma.h>

#include <asm/mach/irq.h>

#include "emxx_udc.h"

#ifdef CONFIG_BATTERY_EMXX
#include <mach/charger.h>
#endif

#ifdef CONFIG_SNSC_EMXX_PLATFORM
#include <mach/udc.h>
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

#define	DRIVER_DESC	"EMXX UDC driver"
#define	DRIVER_VERSION		__DATE__
#define	DMA_ADDR_INVALID	(~(dma_addr_t)0)


static const char	driver_name[] = "emxx_udc";
static const char	driver_desc[] = DRIVER_DESC;

static int g_is_ether = 0;
static int g_vbus_ow = 0;
static struct proc_dir_entry *proc_entry;
static const char vbus_proc_node_name[] = "vbus_overwrite";

#ifdef CONFIG_USB_GADGET_DEBUG_FILES

static const char proc_node_name[] = "driver/udc";

/*-------------------------------------------------------------------------*/
static int udc_proc_read(
	char *page,
	char **start,
	off_t off,
	int count,
	int *eof,
	void *_dev
)
{
	return 0;
}

#define create_proc_files() \
	create_proc_read_entry(proc_node_name, 0, NULL, udc_proc_read, pdev)
#define remove_proc_files() \
	remove_proc_entry(proc_node_name, NULL)

#else	/* !CONFIG_USB_GADGET_DEBUG_FILES */

#define create_proc_files() do { } while (0)
#define remove_proc_files() do { } while (0)

#endif	/* CONFIG_USB_GADGET_DEBUG_FILES */

/*===========================================================================*/
/* Prototype */
static void _nbu2ss_ep_dma_abort(struct nbu2ss_udc *, struct nbu2ss_ep *);
static void _nbu2ss_ep0_enable(struct nbu2ss_udc *);
/*static void _nbu2ss_ep0_disable(struct nbu2ss_udc *);*/
static void _nbu2ss_ep_done(struct nbu2ss_ep *, struct nbu2ss_req *, int);
static void _nbu2ss_set_test_mode(struct nbu2ss_udc *, u32 mode);
static void _nbu2ss_endpoint_toggle_reset(struct nbu2ss_udc *udc, u8 ep_adrs);

static int _nbu2ss_pullup(struct nbu2ss_udc *, int);
static void _nbu2ss_fifo_flush(struct nbu2ss_udc *, struct nbu2ss_ep *);

#ifdef CONFIG_BATTERY_EMXX
static void _nbu2ss_battery_state(struct nbu2ss_udc *, u32);
#endif	/* CONFIG_BATTERY_EMXX */

/*===========================================================================*/
/* Macro */
#define	_nbu2ss_zero_len_pkt(udc, epnum)	\
	_nbu2ss_ep_in_end(udc, epnum, 0, 0)


/*===========================================================================*/
/* Global */
struct nbu2ss_udc udc_controller;
#ifdef CONFIG_SNSC_EMXX_PLATFORM
static struct emxx_udc_platform_data udc_config;
#endif /* CONFIG_SNSC_EMXX_PLATFORM */


/*-------------------------------------------------------------------------*/
/* Read */
static inline u32 _nbu2ss_readl(void *address)
{
	return __raw_readl(address) ;
}

/*-------------------------------------------------------------------------*/
/* Write */
static inline void _nbu2ss_writel(void *address, u32 udata)
{
	__raw_writel(udata, address) ;
}

/*-------------------------------------------------------------------------*/
/* Set Bit */
static inline void _nbu2ss_bitset(void *address, u32 udata)
{
	u32	reg_dt = __raw_readl(address) | (udata);
	__raw_writel(reg_dt, address) ;
}

/*-------------------------------------------------------------------------*/
/* Clear Bit */
static inline void _nbu2ss_bitclr(void *address, u32 udata)
{
	u32	reg_dt = __raw_readl(address) & ~(udata);
	__raw_writel(reg_dt, address) ;
}

#ifdef UDC_DEBUG_DUMP
/*-------------------------------------------------------------------------*/
static void _nbu2ss_dump_register(struct nbu2ss_udc *udc)
{
	int		i;
	u32 reg_data;

	INFO("=== %s()\n", __func__);

	if (udc == NULL) {
		ERR("%s udc == NULL\n", __func__);
		return;
	}

	spin_unlock(&udc->lock);

	printk(KERN_DEBUG "\n-USB REG-\n");
	for (i = 0x0 ; i < USB_BASE_SIZE ; i += 16) {
		reg_data =   _nbu2ss_readl(
			(u32 *)IO_ADDRESS(USB_BASE_ADDRESS + i));
		printk(KERN_DEBUG "USB%04x =%08x", i, (int)reg_data);

		reg_data =  _nbu2ss_readl(
			(u32 *)IO_ADDRESS(USB_BASE_ADDRESS + i + 4));
		printk(KERN_DEBUG " %08x", (int)reg_data);

		reg_data =  _nbu2ss_readl(
			(u32 *)IO_ADDRESS(USB_BASE_ADDRESS + i + 8));
		printk(KERN_DEBUG " %08x", (int)reg_data);

		reg_data =  _nbu2ss_readl(
			(u32 *)IO_ADDRESS(USB_BASE_ADDRESS + i + 12));
		printk(KERN_DEBUG " %08x\n", (int)reg_data);

	}

	spin_lock(&udc->lock);
}
#endif /* UDC_DEBUG_DUMP */

/*-------------------------------------------------------------------------*/
/* Endpoint 0 Callback (Complete) */
static void _nbu2ss_ep0_complete(struct usb_ep *_ep, struct usb_request *_req)
{
	u8		recipient;
	u16		selector;
	u32		test_mode;
	struct usb_ctrlrequest	*p_ctrl;
	struct nbu2ss_udc *udc;

	if ((_ep == NULL) || (_req == NULL))
		return;

	udc = (struct nbu2ss_udc *)_req->context;
	p_ctrl = &udc->ctrl;
	if ((p_ctrl->bRequestType & USB_TYPE_MASK) == USB_TYPE_STANDARD) {

		if (p_ctrl->bRequest == USB_REQ_SET_FEATURE) {
			/*-------------------------------------------------*/
			/* SET_FEATURE */
			recipient = (u8)(p_ctrl->bRequestType & USB_RECIP_MASK);
			selector  = p_ctrl->wValue;
			if ((recipient == USB_RECIP_DEVICE) &&
				(selector == USB_DEVICE_TEST_MODE)) {
				test_mode = (u32)(p_ctrl->wIndex >> 8);
				_nbu2ss_set_test_mode(udc, test_mode);
			}
		}
	}
}

/*-------------------------------------------------------------------------*/
/* Initialization usb_request */
static void _nbu2ss_create_ep0_packet(
	struct nbu2ss_udc *udc,
	void *p_buf,
	unsigned length
)
{
	udc->ep0_req.req.buf 		= p_buf;
	udc->ep0_req.req.length		= length;
	udc->ep0_req.req.dma 		= 0;
	udc->ep0_req.req.zero		= TRUE;
	udc->ep0_req.req.complete	= _nbu2ss_ep0_complete;
	udc->ep0_req.req.status		= -EINPROGRESS;
	udc->ep0_req.req.context	= udc;
	udc->ep0_req.req.actual		= 0;
}

/*-------------------------------------------------------------------------*/
/* Acquisition of the first address of RAM(FIFO) */
static u32 _nbu2ss_get_begin_ram_address(struct nbu2ss_udc *udc)
{
	u32		num, buf_type;
	u32		data, last_ram_adr, use_ram_size;

	PT_EP_REGS	p_ep_regs;

	last_ram_adr = (D_RAM_SIZE_CTRL / sizeof(u32)) * 2;
	use_ram_size = 0;

	for (num = 0; num < NUM_ENDPOINTS - 1; num++) {
		p_ep_regs = &udc->p_regs->EP_REGS[num];
		data = _nbu2ss_readl(&p_ep_regs->EP_PCKT_ADRS);
		buf_type = _nbu2ss_readl(&p_ep_regs->EP_CONTROL) & EPn_BUF_TYPE;
		if (buf_type == 0) {
			/* Single Buffer */
			use_ram_size += (data & EPn_MPKT) / sizeof(u32);
		} else {
			/* Double Buffer */
			use_ram_size += ((data & EPn_MPKT) / sizeof(u32)) * 2;
		}

		if ((data >> 16) > last_ram_adr)
			last_ram_adr = data>>16;
	}

	return last_ram_adr + use_ram_size;
}

/*-------------------------------------------------------------------------*/
/* Construction of Endpoint */
static int _nbu2ss_ep_init(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	u32		num;
	u32		data;
	u32		begin_adrs;

	if (ep->epnum == 0)
		return	-EINVAL;

	num = ep->epnum - 1;

	/*-------------------------------------------------------------*/
	/* RAM Transfer Address */
	begin_adrs = _nbu2ss_get_begin_ram_address(udc);
	data = (begin_adrs << 16) | ep->ep.maxpacket;
	_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_PCKT_ADRS, data);

	/*-------------------------------------------------------------*/
	/* Interrupt Enable */
	data = 1 << (ep->epnum + 8);
	_nbu2ss_bitset(&udc->p_regs->USB_INT_ENA, data);

	/*-------------------------------------------------------------*/
	/* Endpoint Type(Mode) */
	/*   Bulk, Interrupt, ISO */
	switch (ep->ep_type) {
	case USB_ENDPOINT_XFER_BULK:
		data = EPn_BULK;
		break;

	case USB_ENDPOINT_XFER_INT:
		data = EPn_BUF_SINGLE | EPn_INTERRUPT;
		break;

	case USB_ENDPOINT_XFER_ISOC:
		data = EPn_ISO;
		break;

	default:
		data = 0;
		break;
	}

	_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);
	_nbu2ss_endpoint_toggle_reset(udc, (ep->epnum|ep->direct));

	if (ep->direct == USB_DIR_OUT) {
		/*---------------------------------------------------------*/
		/* OUT */
		data = EPn_EN | EPn_BCLR | EPn_DIR0;
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = (EPn_ONAK | EPn_OSTL_EN | EPn_OSTL);
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = (EPn_OUT_EN | EPn_OUT_END_EN);
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_INT_ENA, data);
	} else {
		/*---------------------------------------------------------*/
		/* IN */
		data = (EPn_EN | EPn_BCLR | EPn_AUTO);
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = (EPn_ISTL);
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = (EPn_IN_EN | EPn_IN_END_EN);
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_INT_ENA, data);
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
/* Release of Endpoint */
static int _nbu2ss_epn_exit(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	u32		num;
	u32		data;

	if ((ep->epnum == 0) || (udc->vbus_active == 0))
		return	-EINVAL;

	num = ep->epnum - 1;

	/*-------------------------------------------------------------*/
	/* RAM Transfer Address */
	_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_PCKT_ADRS, 0);

	/*-------------------------------------------------------------*/
	/* Interrupt Disable */
	data = 1 << (ep->epnum + 8);
	_nbu2ss_bitclr(&udc->p_regs->USB_INT_ENA, data);

	if (ep->direct == USB_DIR_OUT) {
		/*---------------------------------------------------------*/
		/* OUT */
		data = EPn_ONAK | EPn_BCLR;
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = EPn_EN | EPn_DIR0;
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = EPn_OUT_EN | EPn_OUT_END_EN;
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_INT_ENA, data);
	} else {
		/*---------------------------------------------------------*/
		/* IN */
		data = EPn_BCLR;
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = EPn_EN | EPn_AUTO;
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);

		data = EPn_IN_EN | EPn_IN_END_EN;
		_nbu2ss_bitclr(&udc->p_regs->EP_REGS[num].EP_INT_ENA, data);
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
/* DMA setting (without Endpoint 0) */
static void _nbu2ss_ep_dma_init(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	u32		num;
	u32		data;

	data = _nbu2ss_readl(&udc->p_regs->USBSSCONF);
	if (((ep->epnum == 0) || (data & (1 << ep->epnum)) == 0))
		return;		/* Not Support DMA */

	num = ep->epnum - 1;

	if (ep->direct == USB_DIR_OUT) {
		/*---------------------------------------------------------*/
		/* OUT */
		data = ep->ep.maxpacket;
		_nbu2ss_writel(&udc->p_regs->EP_DCR[num].EP_DCR2, data);

		/*---------------------------------------------------------*/
		/* Transfer Direct */
		data = DCR1_EPn_DIR0;
		_nbu2ss_bitset(&udc->p_regs->EP_DCR[num].EP_DCR1, data);

		/*---------------------------------------------------------*/
		/* DMA Mode etc. */
		data = EPn_STOP_MODE | EPn_STOP_SET  | EPn_DMAMODE0;
		_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_DMA_CTRL, data);
	} else {
		/*---------------------------------------------------------*/
		/* IN */
		_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, EPn_AUTO);

		/*---------------------------------------------------------*/
		/* DMA Mode etc. */
		data = EPn_BURST_SET | EPn_DMAMODE0;
		_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_DMA_CTRL, data);
	}
}

/*-------------------------------------------------------------------------*/
/* DMA setting release */
static void _nbu2ss_ep_dma_exit(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	u32		num;
	u32		data;
	PT_FC_REGS	preg = udc->p_regs;

	if (udc->vbus_active == 0)
		return;		/* VBUS OFF */

	data = _nbu2ss_readl(&preg->USBSSCONF);
	if ((ep->epnum == 0) || ((data & (1 << ep->epnum)) == 0))
		return;		/* Not Support DMA */

	num = ep->epnum - 1;

	_nbu2ss_ep_dma_abort(udc, ep);

	if (ep->direct == USB_DIR_OUT) {
		/*---------------------------------------------------------*/
		/* OUT */
		_nbu2ss_writel(&preg->EP_DCR[num].EP_DCR2, 0);
		_nbu2ss_bitclr(&preg->EP_DCR[num].EP_DCR1, DCR1_EPn_DIR0);
		_nbu2ss_writel(&preg->EP_REGS[num].EP_DMA_CTRL, 0);
	} else {
		/*---------------------------------------------------------*/
		/* IN */
		_nbu2ss_bitclr(&preg->EP_REGS[num].EP_CONTROL, EPn_AUTO);
		_nbu2ss_writel(&preg->EP_REGS[num].EP_DMA_CTRL, 0);
	}
}

/*-------------------------------------------------------------------------*/
/* Abort DMA */
static void _nbu2ss_ep_dma_abort(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	PT_FC_REGS	preg = udc->p_regs;

	_nbu2ss_bitclr(&preg->EP_DCR[ep->epnum-1].EP_DCR1, DCR1_EPn_REQEN);
#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_wait_dma_abort)
		udc->config->udc_wait_dma_abort();
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	mdelay(DMA_DISABLE_TIME);	/* DCR1_EPn_REQEN Clear */
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	_nbu2ss_bitclr(&preg->EP_REGS[ep->epnum-1].EP_DMA_CTRL, EPn_DMA_EN);
}

/*-------------------------------------------------------------------------*/
/* Start IN Transfer */
static void _nbu2ss_ep_in_end(
	struct nbu2ss_udc *udc,
	u32 epnum,
	u32 data32,
	u32 length
)
{
	u32		data;
	u32		num;
	PT_FC_REGS	preg = udc->p_regs;

	if (length >= sizeof(u32))
		return;

	if (epnum == 0) {
		_nbu2ss_bitclr(&preg->EP0_CONTROL, EP0_AUTO);

		/* Writing of 1-4 bytes */
		if (length)
			_nbu2ss_writel(&preg->EP0_WRITE, data32);

		data = ((length << 5) & EP0_DW) | EP0_DEND;
		_nbu2ss_writel(&preg->EP0_CONTROL, data);

		_nbu2ss_bitset(&preg->EP0_CONTROL, EP0_AUTO);
	} else {
		num = epnum - 1;

		_nbu2ss_bitclr(&preg->EP_REGS[num].EP_CONTROL, EPn_AUTO);

		/* Writing of 1-4 bytes */
		if (length)
			_nbu2ss_writel(&preg->EP_REGS[num].EP_WRITE, data32);

		data = (((((u32)length) << 5) & EPn_DW) | EPn_DEND);
		_nbu2ss_bitset(&preg->EP_REGS[num].EP_CONTROL, data);

		_nbu2ss_bitset(&preg->EP_REGS[num].EP_CONTROL, EPn_AUTO);
	}

	return;
}

#ifdef USE_DMA
/*-------------------------------------------------------------------------*/
static void _nbu2ss_dma_map_single(
	struct nbu2ss_udc *udc,
	struct nbu2ss_req *req,
	u8		direct
)
{
	if (req->req.dma == DMA_ADDR_INVALID) {
		req->req.dma = dma_map_single(
				udc->gadget.dev.parent,
				req->req.buf,
				req->req.length,
				(direct == USB_DIR_IN)
				? DMA_TO_DEVICE : DMA_FROM_DEVICE);

		req->mapped = 1;
	} else {
		dma_sync_single_for_device(
				udc->gadget.dev.parent,
				req->req.dma,
				req->req.length,
				(direct == USB_DIR_IN)
				? DMA_TO_DEVICE : DMA_FROM_DEVICE);

		req->mapped = 0;
	}
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_dma_unmap_single(
	struct nbu2ss_udc *udc,
	struct nbu2ss_req *req,
	u8		direct
)
{
	u8		data[4];
	u8		*p;
	u32		count = 0;

	if (direct == USB_DIR_OUT) {
		count = req->req.actual % 4;
		if (count) {
			p = req->req.buf;
			p += (req->req.actual - count);
			memcpy(data, p, count);
		}
	}

	if (req->mapped) {
		dma_unmap_single(udc->gadget.dev.parent,
			req->req.dma, req->req.length,
			(direct == USB_DIR_IN)
				? DMA_TO_DEVICE
				: DMA_FROM_DEVICE);
		req->req.dma = DMA_ADDR_INVALID;
		req->mapped = 0;
	} else {
		dma_sync_single_for_cpu(udc->gadget.dev.parent,
			req->req.dma, req->req.length,
			(direct == USB_DIR_IN)
				? DMA_TO_DEVICE
				: DMA_FROM_DEVICE);
	}

	if (count) {
		p = req->req.buf;
		p += (req->req.actual - count);
		memcpy(p, data, count);
	}
}
#endif

/*-------------------------------------------------------------------------*/
/* Endpoint 0 OUT Transfer (PIO) */
static int EP0_out_PIO(struct nbu2ss_udc *udc, u8* pBuf, u32 length)
{
	u32		i;
	int		nret   = 0;
	u32		iWordLength = 0;
	USB_REG_ACCESS* pBuf32 = (USB_REG_ACCESS *)pBuf;

	/*------------------------------------------------------------*/
	/* Read Length */
	iWordLength = length / sizeof(u32);

	/*------------------------------------------------------------*/
	/* PIO Read */
	if (iWordLength) {
		for (i = 0; i < iWordLength; i++) {
			pBuf32->dw = _nbu2ss_readl(&udc->p_regs->EP0_READ);
			pBuf32++;
		}
		nret = iWordLength * sizeof(u32);
	}

	return nret;
}

/*-------------------------------------------------------------------------*/
/* Endpoint 0 OUT Transfer (PIO, OverBytes) */
static int EP0_out_OverBytes(struct nbu2ss_udc *udc, u8* pBuf, u32 length)
{
	u32		i;
	u32		iReadSize = 0;
	USB_REG_ACCESS  Temp32;
	USB_REG_ACCESS* pBuf32 = (USB_REG_ACCESS *)pBuf;

	if ((0 < length) && (length < sizeof(u32))) {
		Temp32.dw = _nbu2ss_readl(&udc->p_regs->EP0_READ);
		for (i = 0 ; i < length ; i++)
			pBuf32->byte.DATA[i] = Temp32.byte.DATA[i];
		iReadSize += length;
	}

	return iReadSize;
}

/*-------------------------------------------------------------------------*/
/* Endpoint 0 IN Transfer (PIO) */
static int EP0_in_PIO(struct nbu2ss_udc *udc, u8 *pBuf, u32 length)
{
	u32		i;
	u32		iMaxLength   = EP0_PACKETSIZE;
	u32		iWordLength  = 0;
	u32		iWriteLength = 0;
	USB_REG_ACCESS*	pBuf32 = (USB_REG_ACCESS *)pBuf;

	/*------------------------------------------------------------*/
	/* Transfer Length */
	if (iMaxLength < length)
		iWordLength = iMaxLength / sizeof(u32);
	else
		iWordLength = length / sizeof(u32);

	/*------------------------------------------------------------*/
	/* PIO */
	for (i = 0; i < iWordLength; i++) {
		_nbu2ss_writel(&udc->p_regs->EP0_WRITE, pBuf32->dw);
		pBuf32++;
		iWriteLength += sizeof(u32);
	}

	return iWriteLength;
}

/*-------------------------------------------------------------------------*/
/* Endpoint 0 IN Transfer (PIO, OverBytes) */
static int EP0_in_OverBytes(struct nbu2ss_udc *udc, u8 *pBuf, u32 iRemainSize)
{
	u32		i;
	USB_REG_ACCESS Temp32;
	USB_REG_ACCESS* pBuf32 = (USB_REG_ACCESS *)pBuf;

	if ((0 < iRemainSize) && (iRemainSize < sizeof(u32))) {
		for (i = 0 ; i < iRemainSize ; i++)
			Temp32.byte.DATA[i] = pBuf32->byte.DATA[i];
		_nbu2ss_ep_in_end(udc, 0, Temp32.dw, iRemainSize);

		return iRemainSize;
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
/* Transfer NULL Packet (Epndoint 0) */
static int EP0_send_NULL(struct nbu2ss_udc *udc, bool pid_flag)
{
	u32		data;

	data = _nbu2ss_readl(&udc->p_regs->EP0_CONTROL);
	data &= ~(u32)EP0_INAK;

	if (pid_flag)
		data |= (EP0_INAK_EN | EP0_PIDCLR | EP0_DEND);
	else
		data |= (EP0_INAK_EN | EP0_DEND);

	_nbu2ss_writel(&udc->p_regs->EP0_CONTROL, data);

	return 0;
}

/*-------------------------------------------------------------------------*/
/* Receive NULL Packet (Endpoint 0) */
static int EP0_receive_NULL(struct nbu2ss_udc *udc, bool pid_flag)
{
	u32		data;

	data = _nbu2ss_readl(&udc->p_regs->EP0_CONTROL);
	data &= ~(u32)EP0_ONAK;

	if (pid_flag)
		data |= EP0_PIDCLR;

	_nbu2ss_writel(&udc->p_regs->EP0_CONTROL, data);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_ep0_in_transfer(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req
)
{
	u8		*pBuffer;			/* IN Data Buffer */
	u32		data;
	u32		iRemainSize = 0;
	int		result = 0;

	/*-------------------------------------------------------------*/
	/* End confirmation */
	if (req->req.actual == req->req.length) {
		if ((req->req.actual % EP0_PACKETSIZE) == 0) {
			if (req->zero) {
				req->zero = 0;
				EP0_send_NULL(udc, FALSE);
				return 1;
			}
		}

		return 0;		/* Transfer End */
	}

	/*-------------------------------------------------------------*/
	/* NAK release */
	data = _nbu2ss_readl(&udc->p_regs->EP0_CONTROL);
	data |= EP0_INAK_EN;
	data &= ~(u32)EP0_INAK;
	_nbu2ss_writel(&udc->p_regs->EP0_CONTROL, data);

	iRemainSize = req->req.length - req->req.actual;
	pBuffer = (u8 *)req->req.buf;
	pBuffer += req->req.actual;

	/*-------------------------------------------------------------*/
	/* Data transfer */
	result = EP0_in_PIO(udc, pBuffer, iRemainSize);

	req->div_len = result;
	iRemainSize -= result;

	if (iRemainSize == 0) {
		EP0_send_NULL(udc, FALSE);
		return result;
	}

	if ((iRemainSize < sizeof(u32)) && (result != EP0_PACKETSIZE)) {
		pBuffer += result;
		result += EP0_in_OverBytes(udc, pBuffer, iRemainSize);
		req->div_len = result;
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_ep0_out_transfer(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req
)
{
	u8		*pBuffer;
	u32		iRemainSize;
	u32		iRecvLength;
	int		result = 0;
	int		fRcvZero;

	/*-------------------------------------------------------------*/
	/* Receive data confirmation */
	iRecvLength = _nbu2ss_readl(&udc->p_regs->EP0_LENGTH) & EP0_LDATA;
	if (iRecvLength != 0) {

		fRcvZero = 0;

		iRemainSize = req->req.length - req->req.actual;
		pBuffer = (u8 *)req->req.buf;
		pBuffer += req->req.actual;

		result = EP0_out_PIO(udc, pBuffer
					, min(iRemainSize, iRecvLength));
		if (result < 0)
			return result;

		req->req.actual += result;
		iRecvLength -= result;

		if ((0 < iRecvLength) && (iRecvLength < sizeof(u32))) {
			pBuffer += result;
			iRemainSize -= result;

			result = EP0_out_OverBytes(udc, pBuffer
					, min(iRemainSize, iRecvLength));
			req->req.actual += result;
		}
	} else {
		fRcvZero = 1;
	}

	/*-------------------------------------------------------------*/
	/* End confirmation */
	if (req->req.actual == req->req.length) {
		if ((req->req.actual % EP0_PACKETSIZE) == 0) {
			if (req->zero) {
				req->zero = 0;
				EP0_receive_NULL(udc, FALSE);
				return 1;
			}
		}

		return 0;		/* Transfer End */
	}

	if ((req->req.actual % EP0_PACKETSIZE) != 0)
		return 0;		/* Short Packet Transfer End */

	if (req->req.actual > req->req.length) {
		ERR(" *** Overrun Error\n");
		return -EOVERFLOW;
	}

	if (fRcvZero != 0) {
		iRemainSize = _nbu2ss_readl(&udc->p_regs->EP0_CONTROL);
		if (iRemainSize & EP0_ONAK) {
			/*---------------------------------------------------*/
			/* NACK release */
			_nbu2ss_bitclr(&udc->p_regs->EP0_CONTROL, EP0_ONAK);
		}
		result = 1;
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_out_dma(
	struct nbu2ss_udc *udc,
	struct nbu2ss_req *req,
	u32		num,
	u32		length
)
{
	u8		*pBuffer;
	u32		mpkt;
	u32		lmpkt;
	u32		dmacnt;
	u32		burst = 1;
	u32		data;
	int		result = -EINVAL;
	PT_FC_REGS	preg = udc->p_regs;

	if (req->dma_flag)
		return 1;		/* DMA is forwarded */

	req->dma_flag = TRUE;
	pBuffer = (u8 *)req->req.dma;
	pBuffer += req->req.actual;

	/* DMA Address */
	_nbu2ss_writel(&preg->EP_DCR[num].EP_TADR, (u32)pBuffer);

	/* Number of transfer packets */
	mpkt = _nbu2ss_readl(&preg->EP_REGS[num].EP_PCKT_ADRS) & EPn_MPKT;
	dmacnt = (length / mpkt);
	lmpkt = (length % mpkt) & ~(u32)0x03;

	if (DMA_MAX_COUNT < dmacnt) {
		dmacnt = DMA_MAX_COUNT;
		lmpkt = 0;
	} else if (0 != lmpkt) {
		if (0 == dmacnt)
			burst = 0;	/* Burst OFF */
		dmacnt++;
	}

	data = mpkt | (lmpkt << 16);
	_nbu2ss_writel(&preg->EP_DCR[num].EP_DCR2, data);

	data = ((dmacnt & 0xff) << 16) | DCR1_EPn_DIR0 | DCR1_EPn_REQEN;
	_nbu2ss_writel(&preg->EP_DCR[num].EP_DCR1, data);

	if (0 == burst) {
		_nbu2ss_writel(&preg->EP_REGS[num].EP_LEN_DCNT, 0);
		_nbu2ss_bitclr(&preg->EP_REGS[num].EP_DMA_CTRL, EPn_BURST_SET);
	} else {
		_nbu2ss_writel(&preg->EP_REGS[num].EP_LEN_DCNT
				, (dmacnt << 16));
		_nbu2ss_bitset(&preg->EP_REGS[num].EP_DMA_CTRL, EPn_BURST_SET);
	}
	_nbu2ss_bitset(&preg->EP_REGS[num].EP_DMA_CTRL, EPn_DMA_EN);

	result = length & ~(u32)0x03;
	req->div_len = result;

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_out_pio(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	u32		length
)
{
	u8		*pBuffer;
	u32		i;
	u32		data;
	u32		iWordLength;
	USB_REG_ACCESS	Temp32;
	USB_REG_ACCESS	*pBuf32;
	int		result = 0;
	PT_FC_REGS	preg = udc->p_regs;

	if (req->dma_flag)
		return 1;		/* DMA is forwarded */

	if (length == 0)
		return 0;

	pBuffer = (u8 *)req->req.buf;
	pBuf32 = (USB_REG_ACCESS *)(pBuffer + req->req.actual);

	iWordLength = length / sizeof(u32);
	if (iWordLength > 0) {
		/*---------------------------------------------------------*/
		/* Copy of every four bytes */
		for (i = 0; i < iWordLength; i++) {
			pBuf32->dw =
			_nbu2ss_readl(&preg->EP_REGS[ep->epnum-1].EP_READ);
			pBuf32++;
		}
		result = iWordLength * sizeof(u32);
	}

	data = length - result;
	if (data > 0) {
		/*---------------------------------------------------------*/
		/* Copy of fraction byte */
		Temp32.dw = _nbu2ss_readl(&preg->EP_REGS[ep->epnum-1].EP_READ);
		for (i = 0 ; i < data ; i++)
			pBuf32->byte.DATA[i] = Temp32.byte.DATA[i];
		result += data;
	}

	req->req.actual += result;

	if ((req->req.actual == req->req.length)
			|| ((req->req.actual % ep->ep.maxpacket) != 0)) {

		result = 0;
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_out_data(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	u32		data_size
)
{
	u32		num;
	u32		iBufSize;
	int		nret = 1;

	if (ep->epnum == 0)
		return -EINVAL;

	num = ep->epnum - 1;

	iBufSize = min((req->req.length - req->req.actual), data_size);

	if ((ep->ep_type != USB_ENDPOINT_XFER_INT)
		&& (req->req.dma != 0)
		&& (iBufSize  >= sizeof(u32))) {
		nret = _nbu2ss_out_dma(udc, req, num, iBufSize);
	} else {
		iBufSize = min(iBufSize, (u32)ep->ep.maxpacket);
		nret = _nbu2ss_epn_out_pio(udc, ep, req, iBufSize);
	}

	return nret;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_out_transfer(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req
)
{
	u32		num;
	u32		iRecvLength;
	int		result = 1;
	PT_FC_REGS	preg = udc->p_regs;

	if (ep->epnum == 0)
		return -EINVAL;

	num = ep->epnum - 1;

	/*-------------------------------------------------------------*/
	/* Receive Length */
	iRecvLength
		= _nbu2ss_readl(&preg->EP_REGS[num].EP_LEN_DCNT) & EPn_LDATA;

	if (iRecvLength != 0) {
		result = _nbu2ss_epn_out_data(udc, ep, req, iRecvLength);
		if (iRecvLength < ep->ep.maxpacket) {
			if (iRecvLength == result) {
				req->req.actual += result;
				result = 0;
			}
		}
	} else {
		if ((req->req.actual == req->req.length)
			|| ((req->req.actual % ep->ep.maxpacket) != 0)) {

			result = 0;
		}
	}

	if (result == 0) {
		if ((req->req.actual % ep->ep.maxpacket) == 0) {
			if (req->zero) {
				req->zero = 0;
				return 1;
			}
		}
	}

	if (req->req.actual > req->req.length) {
		ERR(" *** Overrun Error\n");
		result = -EOVERFLOW;
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_in_dma(
	struct nbu2ss_udc *udc,
	struct nbu2ss_req *req,
	u32		num,
	u32		length
)
{
	u8		*pBuffer;
	u32		mpkt;		/* MaxPacketSize */
	u32		lmpkt;		/* Last Packet Data Size */
	u32		dmacnt;		/* IN Data Size */
	u32		iWriteLength;
	u32		data;
	int		result = -EINVAL;
	PT_FC_REGS	preg = udc->p_regs;

	if (req->dma_flag)
		return 1;		/* DMA is forwarded */

#ifdef USE_DMA
	if(g_is_ether == 0){
		if (req->req.actual == 0)
			_nbu2ss_dma_map_single(udc, req, USB_DIR_IN);
	}
#endif
	req->dma_flag = TRUE;

	/* MAX Packet Size */
	mpkt = _nbu2ss_readl(&preg->EP_REGS[num].EP_PCKT_ADRS) & EPn_MPKT;

	if ((DMA_MAX_COUNT * mpkt) < length)
		iWriteLength = DMA_MAX_COUNT * mpkt;
	else
		iWriteLength = length;

	/*------------------------------------------------------------*/
	/* Number of transmission packets */
	if (mpkt < iWriteLength) {
		dmacnt = iWriteLength / mpkt;
		lmpkt  = (iWriteLength % mpkt) & ~(u32)0x3;
		if (lmpkt != 0)
			dmacnt++;
		else
			lmpkt = mpkt & ~(u32)0x3;

	} else {
		dmacnt = 1;
		lmpkt  = iWriteLength & ~(u32)0x3;
	}

	/* Packet setting */
	data = mpkt | (lmpkt << 16);
	_nbu2ss_writel(&preg->EP_DCR[num].EP_DCR2, data);

	/* Address setting */
	pBuffer = (u8 *)req->req.dma;
	pBuffer += req->req.actual;
	_nbu2ss_writel(&preg->EP_DCR[num].EP_TADR, (u32)pBuffer);

	/* Packet and DMA setting */
	data = ((dmacnt & 0xff) << 16) | DCR1_EPn_REQEN;
	_nbu2ss_writel(&preg->EP_DCR[num].EP_DCR1, data);

	/* Packet setting of EPC */
	data = dmacnt << 16;
	_nbu2ss_writel(&preg->EP_REGS[num].EP_LEN_DCNT, data);

	/*DMA setting of EPC */
	_nbu2ss_bitset(&preg->EP_REGS[num].EP_DMA_CTRL, EPn_DMA_EN);

	result = iWriteLength & ~(u32)0x3;
	req->div_len = result;

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_in_pio(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	u32		length
)
{
	u8		*pBuffer;
	u32		i;
	u32		data;
	u32		iWordLength;
	USB_REG_ACCESS	Temp32;
	USB_REG_ACCESS	*pBuf32;
	int		result = 0;
	PT_FC_REGS	preg = udc->p_regs;

	if (req->dma_flag)
		return 1;		/* DMA is forwarded */

	if (length > 0) {
		pBuffer = (u8 *)req->req.buf;
		pBuf32 = (USB_REG_ACCESS *)(pBuffer + req->req.actual);

		iWordLength = length / sizeof(u32);
		if (iWordLength > 0) {
			for (i = 0; i < iWordLength; i++) {
				_nbu2ss_writel(
					&preg->EP_REGS[ep->epnum-1].EP_WRITE
					, pBuf32->dw
				);

				pBuf32++;
			}
			result = iWordLength * sizeof(u32);
		}
	}

	if (result != ep->ep.maxpacket) {
		data = length - result;
		Temp32.dw = 0;
		for (i = 0 ; i < data ; i++)
			Temp32.byte.DATA[i] = pBuf32->byte.DATA[i];

		_nbu2ss_ep_in_end(udc, ep->epnum, Temp32.dw, data);
		result += data;
	}

	req->div_len = result;

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_in_data(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	u32		data_size
)
{
	u32		num;
	int		nret = 1;

	if (ep->epnum == 0)
		return -EINVAL;

	num = ep->epnum - 1;

	if ((ep->ep_type != USB_ENDPOINT_XFER_INT)
		&& (req->req.dma != 0)
		&& (data_size >= sizeof(u32))) {
		nret = _nbu2ss_in_dma(udc, req, num, data_size);
	} else {
		data_size = min(data_size, (u32)ep->ep.maxpacket);
		nret = _nbu2ss_epn_in_pio(udc, ep, req, data_size);
	}

	return nret;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_epn_in_transfer(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req
)
{
	u32		num;
	u32		iBufSize;
	int		result = 0;
	u32		status;

	if (ep->epnum == 0)
		return -EINVAL;

	num = ep->epnum - 1;

	status = _nbu2ss_readl(&udc->p_regs->EP_REGS[num].EP_STATUS);

	/*-------------------------------------------------------------*/
	/* State confirmation of FIFO */
	if (req->req.actual == 0) {
		if ((status & EPn_IN_EMPTY) == 0)
			return 1;	/* Not Empty */

	} else {
		if ((status & EPn_IN_FULL) != 0)
			return 1;	/* Not Empty */
	}

	/*-------------------------------------------------------------*/
	/* Start tranfer */
	iBufSize = req->req.length - req->req.actual;
	if (iBufSize > 0)
		result = _nbu2ss_epn_in_data(udc, ep, req, iBufSize);
	else if (req->req.length == 0)
		_nbu2ss_zero_len_pkt(udc, ep->epnum);

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_start_transfer(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	bool	bflag)
{
	int		nret = -EINVAL;

	req->dma_flag = FALSE;
	req->div_len = 0;

	if (req->req.length == 0)
		req->zero = 0;
	else
		req->zero = req->req.zero;

	if (ep->epnum == 0) {
		/* EP0 */
		switch (udc->ep0state) {
		case EP0_IN_DATA_PHASE:
			nret = _nbu2ss_ep0_in_transfer(udc, ep, req);
			break;

		case EP0_OUT_DATA_PHASE:
			nret = _nbu2ss_ep0_out_transfer(udc, ep, req);
			break;

		case EP0_IN_STATUS_PHASE:
			nret = EP0_send_NULL(udc, TRUE);
			break;

		default:
			break;
		}

	} else {
		/* EPn */
		if (ep->direct == USB_DIR_OUT) {
			/* OUT */
			if (bflag == FALSE)
				nret = _nbu2ss_epn_out_transfer(udc, ep, req);
		} else {
			/* IN */
			nret = _nbu2ss_epn_in_transfer(udc, ep, req);
		}
	}

	return nret;
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_restert_transfer(struct nbu2ss_ep *ep)
{
	u32		length;
	bool	bflag = FALSE;
	struct nbu2ss_req *req;

	if (list_empty(&ep->queue))
		req = NULL;
	else
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);

	if (req == NULL)
		return;

	if (ep->epnum > 0) {
		length = _nbu2ss_readl(
			&ep->udc->p_regs->EP_REGS[ep->epnum-1].EP_LEN_DCNT);

		length &= EPn_LDATA;
		if (length < ep->ep.maxpacket)
			bflag = TRUE;
	}

	_nbu2ss_start_transfer(ep->udc, ep, req, bflag);
}

/*-------------------------------------------------------------------------*/
/*	Endpoint Toggle Reset */
static void _nbu2ss_endpoint_toggle_reset(
	struct nbu2ss_udc *udc,
	u8 ep_adrs)
{
	u8		num;
	u32		data;

	if ((ep_adrs == 0) || (ep_adrs == 0x80))
		return;

	num = (ep_adrs & 0x7F) - 1;

	if (ep_adrs & USB_DIR_IN)
		data = EPn_IPIDCLR;
	else
		data = EPn_BCLR | EPn_OPIDCLR;

	_nbu2ss_bitset(&udc->p_regs->EP_REGS[num].EP_CONTROL, data);
}

/*-------------------------------------------------------------------------*/
/*	Endpoint STALL set */
static void _nbu2ss_set_endpoint_stall(
	struct nbu2ss_udc *udc,
	u8 ep_adrs,
	bool bstall)
{
	u8		num, epnum;
	u32		data;
	struct nbu2ss_ep *ep;
	PT_FC_REGS	preg = udc->p_regs;

	if ((ep_adrs == 0) || (ep_adrs == 0x80)) {
		if (bstall) {
			/* Set STALL */
			_nbu2ss_bitset(&preg->EP0_CONTROL, EP0_STL);
		} else {
			/* Clear STALL */
			_nbu2ss_bitclr(&preg->EP0_CONTROL, EP0_STL);
		}
	} else {
		epnum = ep_adrs & USB_ENDPOINT_NUMBER_MASK;
		num = epnum - 1;
		ep = &udc->ep[epnum];

		if (bstall) {
			/* Set STALL */
			ep->halted = TRUE;

			if (ep_adrs & USB_DIR_IN)
				data = EPn_BCLR | EPn_ISTL;
			else
				data = EPn_OSTL_EN | EPn_OSTL;

			_nbu2ss_bitset(&preg->EP_REGS[num].EP_CONTROL, data);
		} else {
			/* Clear STALL */
			ep->stalled = FALSE;
			if (ep_adrs & USB_DIR_IN) {
				_nbu2ss_bitclr(&preg->EP_REGS[num].EP_CONTROL
						, EPn_ISTL);
			} else {
				data =
				_nbu2ss_readl(&preg->EP_REGS[num].EP_CONTROL);

				data &= ~EPn_OSTL;
				data |= EPn_OSTL_EN;

				_nbu2ss_writel(&preg->EP_REGS[num].EP_CONTROL
						, data);
			}

			ep->stalled = FALSE;
			if (ep->halted) {
				ep->halted = FALSE;
				_nbu2ss_restert_transfer(ep);
			}
		}
	}

	return;
}


/*-------------------------------------------------------------------------*/
/* Device Descriptor */
static struct usb_device_descriptor device_desc = {
	.bLength              = sizeof(device_desc),
	.bDescriptorType      = USB_DT_DEVICE,
	.bcdUSB               = __constant_cpu_to_le16(0x0200),
	.bDeviceClass         = USB_CLASS_VENDOR_SPEC,
	.bDeviceSubClass      = 0x00,
	.bDeviceProtocol      = 0x00,
	.bMaxPacketSize0      = 64,
	.idVendor             = __constant_cpu_to_le16 (0x0409),
	.idProduct            = __constant_cpu_to_le16 (0xfff0),
	.bcdDevice            = 0xffff,
	.iManufacturer        = 0x00,
	.iProduct             = 0x00,
	.iSerialNumber        = 0x00,
	.bNumConfigurations   = 0x01,
};

/*-------------------------------------------------------------------------*/
static void _nbu2ss_set_test_mode(struct nbu2ss_udc *udc, u32 mode)
{
	u32		data;

	if (mode > MAX_TEST_MODE_NUM)
		return;

	INFO("SET FEATURE : test mode = %d\n", mode);

	data = _nbu2ss_readl(&udc->p_regs->USB_CONTROL);
	data &= ~TEST_FORCE_ENABLE;
	data |= mode << TEST_MODE_SHIFT;

	_nbu2ss_writel(&udc->p_regs->USB_CONTROL, data);
	_nbu2ss_bitset(&udc->p_regs->TEST_CONTROL, CS_TESTMODEEN);
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_set_feature_device(
	struct nbu2ss_udc *udc,
	u16 selector,
	u16 wIndex
)
{
	int	result = -EOPNOTSUPP;

	switch (selector) {
	case USB_DEVICE_REMOTE_WAKEUP:
		if (0x0000 == wIndex) {
			udc->remote_wakeup = U2F_ENABLE;
			result = 0;
		}
		break;

	case USB_DEVICE_TEST_MODE:
		wIndex = wIndex >> 8;
		if (wIndex <= MAX_TEST_MODE_NUM)
			result = 0;
		break;

	default:
		break;
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_get_ep_stall(struct nbu2ss_udc *udc, u8 ep_adrs)
{
	u8		epnum;
	u32		data = 0, bit_data;
	PT_FC_REGS	preg = udc->p_regs;

	epnum = ep_adrs & ~USB_ENDPOINT_DIR_MASK;
	if (epnum == 0) {
		data = _nbu2ss_readl(&preg->EP0_CONTROL);
		bit_data = EP0_STL;

	} else {
		data = _nbu2ss_readl(&preg->EP_REGS[epnum-1].EP_CONTROL);
		if ((data & EPn_EN) == 0)
			return -1;

		if (ep_adrs & USB_ENDPOINT_DIR_MASK)
			bit_data = EPn_ISTL;
		else
			bit_data = EPn_OSTL;
	}

	if ((data & bit_data) == 0)
		return 0;
	else
		return 1;
}

/*-------------------------------------------------------------------------*/
static inline int _nbu2ss_req_feature(struct nbu2ss_udc *udc, bool bset)
{
	u8	recipient = (u8)(udc->ctrl.bRequestType & USB_RECIP_MASK);
	u8	direction = (u8)(udc->ctrl.bRequestType & USB_DIR_IN);
	u16	selector  = udc->ctrl.wValue;
	u16	wIndex    = udc->ctrl.wIndex;
	u8	ep_adrs;
	int	result = -EOPNOTSUPP;

	if ((0x0000 != udc->ctrl.wLength) ||
			(USB_DIR_OUT != direction)) {
		return -EINVAL;
	}

	switch (recipient) {
	case USB_RECIP_DEVICE:
		if (bset)
			result =
			_nbu2ss_set_feature_device(udc, selector, wIndex);
		break;

	case USB_RECIP_ENDPOINT:
		if (0x0000 == (wIndex & 0xFF70)) {
			if (USB_ENDPOINT_HALT == selector) {
				ep_adrs = wIndex & 0xFF;
				if (bset == FALSE) {
					_nbu2ss_endpoint_toggle_reset(
						udc, ep_adrs);
				}

				_nbu2ss_set_endpoint_stall(
					udc, ep_adrs, bset);

				result = 0;
			}
		}
		break;

	default:
		break;
	}

	if (result >= 0)
		_nbu2ss_create_ep0_packet(udc, udc->ep0_buf, 0);

	return result;
}

/*-------------------------------------------------------------------------*/
static inline enum usb_device_speed _nbu2ss_get_speed(struct nbu2ss_udc *udc)
{
	u32		data;
	enum usb_device_speed speed = USB_SPEED_FULL;

#ifdef CONFIG_BATTERY_EMXX
	_nbu2ss_battery_state(udc, EMXX_USB_DEVICE);
#endif	/* CONFIG_BATTERY_EMXX */

	data = _nbu2ss_readl(&udc->p_regs->USB_STATUS);
	if (data & HIGH_SPEED)
		speed = USB_SPEED_HIGH;

	return speed;
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_epn_set_stall(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep
)
{
	u8	ep_adrs;
	u32	regdata;
	int	limit_cnt = 0;

	PT_FC_REGS	preg = udc->p_regs;

	if (ep->direct == USB_DIR_IN) {
		for (limit_cnt = 0
			; limit_cnt < IN_DATA_EMPTY_COUNT
			; limit_cnt++) {

			regdata = _nbu2ss_readl(
				&preg->EP_REGS[ep->epnum-1].EP_STATUS);

			if ((regdata & EPn_IN_DATA) == 0)
				break;

			mdelay(1);
		}
	}

	ep_adrs = ep->epnum | ep->direct;
	_nbu2ss_set_endpoint_stall(udc, ep_adrs, 1);
}

/*-------------------------------------------------------------------------*/
static int std_req_get_status(struct nbu2ss_udc *udc)
{
	u32	length;
	u16	status_data = 0;
	u8	recipient = (u8)(udc->ctrl.bRequestType & USB_RECIP_MASK);
	u8	direction = (u8)(udc->ctrl.bRequestType & USB_DIR_IN);
	u8	ep_adrs;
	int	result = -EINVAL;

	if ((0x0000 != udc->ctrl.wValue)
		|| (USB_DIR_IN != direction)) {

		return result;
	}

	length = min(udc->ctrl.wLength, (u16)sizeof(status_data));

	switch (recipient) {
	case USB_RECIP_DEVICE:
		if (udc->ctrl.wIndex == 0x0000) {
			if (udc->self_powered)
				status_data |= (1 << USB_DEVICE_SELF_POWERED);

			if (udc->remote_wakeup)
				status_data |= (1 << USB_DEVICE_REMOTE_WAKEUP);

			result = 0;
		}
		break;

	case USB_RECIP_ENDPOINT:
		if (0x0000 == (udc->ctrl.wIndex & 0xFF70)) {
			ep_adrs = (u8)(udc->ctrl.wIndex & 0xFF);
			result = _nbu2ss_get_ep_stall(udc, ep_adrs);

			if (result > 0)
				status_data |= (1 << USB_ENDPOINT_HALT);
		}
		break;

	default:
		break;
	}

	if (result >= 0) {
		memcpy(udc->ep0_buf, &status_data, length);
		_nbu2ss_create_ep0_packet(udc, udc->ep0_buf, length);
		_nbu2ss_ep0_in_transfer(udc, &udc->ep[0], &udc->ep0_req);

	} else {
		ERR("*** Error GET_STATUS\n");
	}

	return result;
}

/*-------------------------------------------------------------------------*/
static int std_req_clear_feature(struct nbu2ss_udc *udc)
{
	return _nbu2ss_req_feature(udc, FALSE);
}

/*-------------------------------------------------------------------------*/
static int std_req_set_feature(struct nbu2ss_udc *udc)
{
	return _nbu2ss_req_feature(udc, TRUE);
}

/*-------------------------------------------------------------------------*/
static int std_req_set_address(struct nbu2ss_udc *udc)
{
	int		result = 0;
	u32		wValue = udc->ctrl.wValue;

	if ((0x00 != udc->ctrl.bRequestType)	||
		(0x0000 != udc->ctrl.wIndex)	||
		(0x0000 != udc->ctrl.wLength)) {
		return -EINVAL;
	}

	if (wValue != (wValue & 0x007F))
		return -EINVAL;

	wValue = wValue << USB_ADRS_SHIFT;

	_nbu2ss_writel(&udc->p_regs->USB_ADDRESS, wValue);
	_nbu2ss_create_ep0_packet(udc, udc->ep0_buf, 0);

	return result;
}

/*-------------------------------------------------------------------------*/
static int std_req_set_configuration(struct nbu2ss_udc *udc)
{
	u32 ConfigValue = (u32)(udc->ctrl.wValue & 0x00ff);

	if ((0x0000 != udc->ctrl.wIndex)	||
		(0x0000 != udc->ctrl.wLength)	||
		(0x00 != udc->ctrl.bRequestType)) {
		return -EINVAL;
	}

	udc->curr_config = ConfigValue;

	if (ConfigValue > 0) {
		_nbu2ss_bitset(&udc->p_regs->USB_CONTROL, CONF);
		udc->devstate = USB_STATE_CONFIGURED;

	} else {
		_nbu2ss_bitclr(&udc->p_regs->USB_CONTROL, CONF);
		udc->devstate = USB_STATE_ADDRESS;
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_read_request_data(struct nbu2ss_udc *udc, u32 *pdata)
{
	if ((udc == NULL) && (pdata == NULL))
		return;

	*pdata = _nbu2ss_readl(&udc->p_regs->SETUP_DATA0);
	pdata++;
	*pdata = _nbu2ss_readl(&udc->p_regs->SETUP_DATA1);
}

/*-------------------------------------------------------------------------*/
static inline int _nbu2ss_decode_request(struct nbu2ss_udc *udc)
{
	bool			bcall_back = TRUE;
	int			nret = -EINVAL;
	struct usb_ctrlrequest	*p_ctrl;

	p_ctrl = &udc->ctrl;
	_nbu2ss_read_request_data(udc, (u32 *)p_ctrl);

	/* ep0 state control */
	if (p_ctrl->wLength == 0) {
		udc->ep0state = EP0_IN_STATUS_PHASE;

	} else {
		if (p_ctrl->bRequestType & USB_DIR_IN)
			udc->ep0state = EP0_IN_DATA_PHASE;
		else
			udc->ep0state = EP0_OUT_DATA_PHASE;
	}

	if ((p_ctrl->bRequestType & USB_TYPE_MASK) == USB_TYPE_STANDARD) {
		switch (p_ctrl->bRequest) {
		case USB_REQ_GET_STATUS:
			nret = std_req_get_status(udc);
			bcall_back = FALSE;
			break;

		case USB_REQ_CLEAR_FEATURE:
			nret = std_req_clear_feature(udc);
			bcall_back = FALSE;
			break;

		case USB_REQ_SET_FEATURE:
			nret = std_req_set_feature(udc);
			bcall_back = FALSE;
			break;

		case USB_REQ_SET_ADDRESS:
			nret = std_req_set_address(udc);
			bcall_back = FALSE;
			break;

		case USB_REQ_SET_CONFIGURATION:
			nret = std_req_set_configuration(udc);
			break;

		default:
			break;
		}
	}

	if (bcall_back == FALSE) {
		if (udc->ep0state == EP0_IN_STATUS_PHASE) {
			if (nret >= 0) {
				/*--------------------------------------*/
				/* Status Stage */
				nret = EP0_send_NULL(udc, TRUE);
			}
		}

	} else {
		spin_unlock(&udc->lock);
		nret = udc->driver->setup(&udc->gadget, &udc->ctrl);
		spin_lock(&udc->lock);
	}

	if (nret < 0)
		udc->ep0state = EP0_IDLE;

	return nret;
}

/*-------------------------------------------------------------------------*/
static inline int _nbu2ss_ep0_in_data_stage(struct nbu2ss_udc *udc)
{
	int			nret;
	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep = &udc->ep[0];

	if (list_empty(&ep->queue))
		req = NULL;
	else
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);

	if (req == NULL)
		req = &udc->ep0_req;

	req->req.actual += req->div_len;
	req->div_len = 0;

	nret = _nbu2ss_ep0_in_transfer(udc, ep, req);
	if (nret == 0) {
		udc->ep0state = EP0_OUT_STATUS_PAHSE;
		EP0_receive_NULL(udc, TRUE);
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
static inline int _nbu2ss_ep0_out_data_stage(struct nbu2ss_udc *udc)
{
	int			nret;
	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep = &udc->ep[0];

	if (list_empty(&ep->queue))
		req = NULL;
	else
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);

	if (req == NULL)
		req = &udc->ep0_req;

	nret = _nbu2ss_ep0_out_transfer(udc, ep, req);
	if (nret == 0) {
		udc->ep0state = EP0_IN_STATUS_PHASE;
		EP0_send_NULL(udc, TRUE);

	} else if (nret < 0) {
		_nbu2ss_bitset(&udc->p_regs->EP0_CONTROL, EP0_BCLR);
		req->req.status = nret;
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
static inline int _nbu2ss_ep0_status_stage(struct nbu2ss_udc *udc)
{
	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep = &udc->ep[0];

	if (list_empty(&ep->queue))
		req = NULL;
	else
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);

	if (req == NULL) {
		req = &udc->ep0_req;
		if (req->req.complete)
			req->req.complete(&ep->ep, &req->req);

	} else {
		if (req->req.complete)
			_nbu2ss_ep_done(ep, req, 0);
	}

	udc->ep0state = EP0_IDLE;

	return 0;
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_ep0_int(struct nbu2ss_udc *udc)
{
	int		i;
	u32		status;
	u32		intr;
	int		nret = -1;

	status = _nbu2ss_readl(&udc->p_regs->EP0_STATUS);
	intr = status & EP0_STATUS_RW_BIT;
	_nbu2ss_writel(&udc->p_regs->EP0_STATUS, ~(u32)intr);

	status &= (SETUP_INT | EP0_IN_INT | EP0_OUT_INT
			| STG_END_INT | EP0_OUT_NULL_INT);

	if (status == 0) {
		INFO("--- %s Not Decode Interrupt\n", __func__);
		INFO("--- EP0_STATUS = 0x%08x\n", intr);
		return;
	}

	if (udc->gadget.speed == USB_SPEED_UNKNOWN)
		udc->gadget.speed = _nbu2ss_get_speed(udc);

	for (i = 0; i < EP0_END_XFER; i++) {
		switch (udc->ep0state) {
		case EP0_IDLE:
			if (status & SETUP_INT) {
				status = 0;
				nret = _nbu2ss_decode_request(udc);
			}
			break;

		case EP0_IN_DATA_PHASE:
			if (status & EP0_IN_INT) {
				status &= ~EP0_IN_INT;
				nret = _nbu2ss_ep0_in_data_stage(udc);
			}
			break;

		case EP0_OUT_DATA_PHASE:
			if (status & EP0_OUT_INT) {
				status &= ~EP0_OUT_INT;
				nret = _nbu2ss_ep0_out_data_stage(udc);
			}
			break;

		case EP0_IN_STATUS_PHASE:
			if ((status & STG_END_INT) || (status & SETUP_INT)) {
				status &= ~(STG_END_INT | EP0_IN_INT);
				nret = _nbu2ss_ep0_status_stage(udc);
			}
			break;

		case EP0_OUT_STATUS_PAHSE:
			if ((status & STG_END_INT)
			|| (status & SETUP_INT)
			|| (status & EP0_OUT_NULL_INT)) {
				status &= ~(STG_END_INT
						| EP0_OUT_INT
						| EP0_OUT_NULL_INT);

				nret = _nbu2ss_ep0_status_stage(udc);
			}

			break;

		default:
			status = 0;
			break;
		}

		if (status == 0)
			break;
	}

	if (nret < 0) {
		/* Send Stall */
		_nbu2ss_set_endpoint_stall(udc, 0, TRUE);
	}
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_ep_done(
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req,
	int status)
{
	struct nbu2ss_udc *udc = ep->udc;

	list_del_init(&req->queue);

	if (status == -ECONNRESET)
		_nbu2ss_fifo_flush(udc, ep);

	if (likely(req->req.status == -EINPROGRESS))
		req->req.status = status;

	if (ep->stalled)
		_nbu2ss_epn_set_stall(udc, ep);
	else {
		if (!list_empty(&ep->queue))
			_nbu2ss_restert_transfer(ep);
	}

#ifdef USE_DMA
	if(g_is_ether == 0){
		if ((ep->direct == USB_DIR_OUT) && (ep->epnum > 0))
			_nbu2ss_dma_unmap_single(udc, req, USB_DIR_OUT);
	}
#endif

	spin_unlock(&udc->lock);
	req->req.complete(&ep->ep, &req->req);
	spin_lock(&udc->lock);
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_epn_in_int(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req)
{
	int	result = 0;
	u32	status;

	PT_FC_REGS	preg = udc->p_regs;

	if (req->dma_flag)
		return;		/* DMA is forwarded */

	req->req.actual += req->div_len;
	req->div_len = 0;

	if (req->req.actual != req->req.length) {
		/*---------------------------------------------------------*/
		/* remainder of data */
		result = _nbu2ss_epn_in_transfer(udc, ep, req);

	} else {
		if ((req->zero != 0)
		&& ((req->req.actual % ep->ep.maxpacket) == 0)) {

			status =
			_nbu2ss_readl(&preg->EP_REGS[ep->epnum-1].EP_STATUS);

			if ((status & EPn_IN_FULL) == 0) {
				/*-----------------------------------------*/
				/* 0 Length Packet */
				req->zero = 0;
				_nbu2ss_zero_len_pkt(udc, ep->epnum);
			}
			return;
		}
	}

	if (result <= 0) {
		/*---------------------------------------------------------*/
		/* Complete */
		_nbu2ss_ep_done(ep, req, result);
	}
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_epn_out_int(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req)
{
	int	result = 0;

	result = _nbu2ss_epn_out_transfer(udc, ep, req);
	if (result <= 0)
		_nbu2ss_ep_done(ep, req, result);

	return;
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_epn_in_dma_int(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req)
{
	u32		mpkt;
	u32		size;
	struct usb_request *preq;

	preq = &req->req;

	if (req->dma_flag == FALSE)
		return;

	preq->actual += req->div_len;
	req->div_len = 0;
	req->dma_flag = FALSE;

#ifdef USE_DMA
	if(g_is_ether == 0){
		_nbu2ss_dma_unmap_single(udc, req, USB_DIR_IN);
	}
#endif

	if (preq->actual != preq->length) {
		_nbu2ss_epn_in_transfer(udc, ep, req);
	} else {
		mpkt = ep->ep.maxpacket;
		size = preq->actual % mpkt;
		if (size > 0) {
			if (((preq->actual & 0x03) == 0) && (size < mpkt))
				_nbu2ss_ep_in_end(udc, ep->epnum, 0, 0);
		} else {
			_nbu2ss_epn_in_int(udc, ep, req);
		}
	}

	return;
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_epn_out_dma_int(
	struct nbu2ss_udc *udc,
	struct nbu2ss_ep *ep,
	struct nbu2ss_req *req)
{
	int		i;
	u32		num;
	u32		dmacnt, ep_dmacnt;
	u32		mpkt;
	PT_FC_REGS	preg = udc->p_regs;

	num = ep->epnum - 1;

	if (req->req.actual == req->req.length) {
		if ( (req->req.length % ep->ep.maxpacket)
				&& (req->zero == 0)) {
			req->div_len = 0;
			req->dma_flag = FALSE;
			_nbu2ss_ep_done(ep, req, 0);
			return;
		}
	}

	ep_dmacnt = _nbu2ss_readl(&preg->EP_REGS[num].EP_LEN_DCNT)
		 & EPn_DMACNT;
	ep_dmacnt >>= 16;

	for (i = 0; i < EPC_PLL_LOCK_COUNT; i++) {
		dmacnt = _nbu2ss_readl(&preg->EP_DCR[num].EP_DCR1)
			 & DCR1_EPn_DMACNT;
		dmacnt >>= 16;
		if (ep_dmacnt == dmacnt)
			break;
	}

	_nbu2ss_bitclr(&preg->EP_DCR[num].EP_DCR1, DCR1_EPn_REQEN);

	if (dmacnt != 0) {
		mpkt = ep->ep.maxpacket;
		if ((req->div_len % mpkt) == 0)
			req->div_len -= mpkt * dmacnt;
	}

	req->req.actual += req->div_len;
	req->div_len = 0;
	req->dma_flag = FALSE;

	_nbu2ss_epn_out_int(udc, ep, req);
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_epn_int(struct nbu2ss_udc *udc, u32 epnum)
{
	u32	num;
	u32	status;

	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep = &udc->ep[epnum];

	num = epnum - 1;

	/* Interrupt Status */
	status = _nbu2ss_readl(&udc->p_regs->EP_REGS[num].EP_STATUS);

	/* Interrupt Clear */
	if ((status & EPn_OUT_END_INT) && !(status & EPn_OUT_EMPTY) && (g_is_ether == 0))
		_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_STATUS,
			(~(u32)status) | EPn_OUT_INT);
	else
		_nbu2ss_writel(&udc->p_regs->EP_REGS[num].EP_STATUS,
			~(u32)status);

	if (list_empty(&ep->queue))
		req = NULL;
	else
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);

	if (req == NULL) {
		/* WARNING("=== %s(%d) req == NULL\n", __func__, epnum); */
		return;
	}

	if (status & EPn_OUT_END_INT) {
		status &= ~EPn_OUT_INT;
		_nbu2ss_epn_out_dma_int(udc, ep, req);
	}

	if (status & EPn_OUT_INT)
		_nbu2ss_epn_out_int(udc, ep, req);

	if (status & EPn_IN_END_INT) {
		status &= ~EPn_IN_INT;
		_nbu2ss_epn_in_dma_int(udc, ep, req);
	}

	if (status & EPn_IN_INT)
		_nbu2ss_epn_in_int(udc, ep, req);
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_ep_int(struct nbu2ss_udc *udc, u32 epnum)
{
	if (epnum == 0)
		_nbu2ss_ep0_int(udc);
	else
		_nbu2ss_epn_int(udc, epnum);
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_ep0_enable(struct nbu2ss_udc *udc)
{
	_nbu2ss_bitset(&udc->p_regs->EP0_CONTROL, (EP0_AUTO | EP0_BCLR));
	_nbu2ss_writel(&udc->p_regs->EP0_INT_ENA, EP0_INT_EN_BIT);

	return;
}

#if 0
/*-------------------------------------------------------------------------*/
static void _nbu2ss_ep0_disable(struct nbu2ss_udc *udc)
{
	_nbu2ss_bitclr(&udc->p_regs->EP0_INT_ENA, EP0_INT_EN_BIT);

	_nbu2ss_bitset(&udc->p_regs->EP0_CONTROL
			, (EP0_BCLR | EP0_INAK | EP0_ONAK | EP0_BCLR));

	_nbu2ss_bitclr(&udc->p_regs->EP0_CONTROL, EP0_AUTO);

	return;
}
#endif

/*-------------------------------------------------------------------------*/
static int _nbu2ss_nuke(struct nbu2ss_udc *udc,
			struct nbu2ss_ep *ep,
			int status)
{
	struct nbu2ss_req *req;

	/* Endpoint Disable */
	_nbu2ss_epn_exit(udc, ep);

	/* DMA Disable */
	_nbu2ss_ep_dma_exit(udc, ep);

	if (list_empty(&ep->queue))
		return 0;

	/* called with irqs blocked */
	while (!list_empty(&ep->queue)) {
		req = list_entry(ep->queue.next, struct nbu2ss_req, queue);
		_nbu2ss_ep_done(ep, req, status);
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_quiesce(struct nbu2ss_udc *udc)
{
	struct nbu2ss_ep	*ep;

	udc->gadget.speed = USB_SPEED_UNKNOWN;

	_nbu2ss_nuke(udc, &udc->ep[0], -ESHUTDOWN);

	/* Endpoint n */
	list_for_each_entry(ep, &udc->gadget.ep_list, ep.ep_list) {
		_nbu2ss_nuke(udc, ep, -ESHUTDOWN);
	}
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_pullup(struct nbu2ss_udc *udc, int is_on)
{
	u32	reg_dt;

	if (!udc) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	if (udc->vbus_active == 0)
		return -ESHUTDOWN;

	if (is_on) {
		/* D+ Pullup */
/*		INFO(" --- D+ Pullup\n"); */

		if (udc->driver) {
			reg_dt = (_nbu2ss_readl(&udc->p_regs->USB_CONTROL)
				| PUE2) & ~(u32)CONNECTB;

			_nbu2ss_writel(&udc->p_regs->USB_CONTROL, reg_dt);
		}

#ifdef CONFIG_BATTERY_EMXX
		mod_timer(&udc->timer, jiffies +
				msecs_to_jiffies(CHATGER_TIME));
#endif	/* CONFIG_BATTERY_EMXX */
	} else {
		/* D+ Pulldown */
/*		INFO(" --- D+ Pulldown\n"); */

		reg_dt = (_nbu2ss_readl(&udc->p_regs->USB_CONTROL) | CONNECTB)
			& ~(u32)PUE2;

		_nbu2ss_writel(&udc->p_regs->USB_CONTROL, reg_dt);
		udc->gadget.speed = USB_SPEED_UNKNOWN;
	}

	return 0;
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_fifo_flush(struct nbu2ss_udc *udc, struct nbu2ss_ep *ep)
{
	PT_FC_REGS	p = udc->p_regs;

	if (udc->vbus_active == 0)
		return;

	if (ep->epnum == 0) {
		/* EP0 */
		_nbu2ss_bitset(&p->EP0_CONTROL, EP0_BCLR);

	} else {
		/* EPn */
		_nbu2ss_ep_dma_abort(udc, ep);
		_nbu2ss_bitset(&p->EP_REGS[ep->epnum - 1].EP_CONTROL, EPn_BCLR);
	}
}

/*-------------------------------------------------------------------------*/
static int _nbu2ss_enable_controller(struct nbu2ss_udc *udc)
{
	int	waitcnt = 0;

	if (udc->udc_enabled)
		return 0;

	emxx_open_clockgate(EMXX_CLK_USB1);
	/* emxx_clkctrl_off(EMXX_CLKCTRL_USB1); */
	/* emxx_clkctrl_on(EMXX_CLKCTRL_USB1); */
	emxx_unreset_device(EMXX_RST_USB1);

	/*
		Reset
	*/
	_nbu2ss_bitset(&udc->p_regs->EPCTR, (DIRPD | EPC_RST));
	udelay(EPC_RST_DISABLE_TIME);	/* 1us wait */

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_post_set_dpd)
		udc->config->udc_post_set_dpd(udc);

	if (udc->config->udc_pre_clr_dpd)
		udc->config->udc_pre_clr_dpd(udc);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

	_nbu2ss_bitclr(&udc->p_regs->EPCTR, DIRPD);
	mdelay(EPC_DIRPD_DISABLE_TIME);	/* 1ms wait */

	_nbu2ss_bitclr(&udc->p_regs->EPCTR, EPC_RST);

	_nbu2ss_writel(&udc->p_regs->AHBSCTR, WAIT_MODE);

	/* DMA Mode Setting */
	if ((system_rev & EMXX_REV_MASK) == EMXX_REV_ES1) {
		_nbu2ss_bitset(&udc->p_regs->AHBMCTR, BURST_TYPE);
		_nbu2ss_bitclr(&udc->p_regs->AHBMCTR, HTRANS_MODE);
	} else {
		_nbu2ss_writel(&udc->p_regs->AHBMCTR,
			HBUSREQ_MODE | HTRANS_MODE | WBURST_TYPE);
	}

	while (!(_nbu2ss_readl(&udc->p_regs->EPCTR) & PLL_LOCK)) {
		waitcnt++;
		udelay(1);	/* 1us wait */
		if (waitcnt == EPC_PLL_LOCK_COUNT) {
			ERR("*** Reset Cancel failed\n");
			return -EINVAL;
		}
	};

	if ((system_rev & EMXX_REV_MASK) < EMXX_REV_ES3)
		_nbu2ss_bitset(&udc->p_regs->UTMI_CHARACTER_1, USB_SQUSET);

	_nbu2ss_bitset(&udc->p_regs->USB_CONTROL, (INT_SEL | SOF_RCV));

	/* EP0 */
	_nbu2ss_ep0_enable(udc);

	/* USB Interrupt Enable */
	_nbu2ss_bitset(&udc->p_regs->USB_INT_ENA, USB_INT_EN_BIT);

	udc->udc_enabled = TRUE;

	return 0;
}


/*-------------------------------------------------------------------------*/
static void _nbu2ss_reset_controller(struct nbu2ss_udc *udc)
{
	_nbu2ss_bitset(&udc->p_regs->EPCTR, EPC_RST);
	_nbu2ss_bitclr(&udc->p_regs->EPCTR, EPC_RST);
}

/*-------------------------------------------------------------------------*/
static void _nbu2ss_disable_controller(struct nbu2ss_udc *udc)
{
	if (udc->udc_enabled) {
		udc->udc_enabled = FALSE;
		_nbu2ss_reset_controller(udc);
		_nbu2ss_bitset(&udc->p_regs->EPCTR, (DIRPD | EPC_RST));
#ifdef CONFIG_SNSC_EMXX_PLATFORM
		if (udc->config->udc_post_set_dpd)
			udc->config->udc_post_set_dpd(udc);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	}

	emxx_reset_device(EMXX_RST_USB1);
	/* emxx_clkctrl_on(EMXX_CLKCTRL_USB1); */
	emxx_close_clockgate(EMXX_CLK_USB1);
}

#ifdef CONFIG_BATTERY_EMXX
/*-------------------------------------------------------------------------*/
static void _nbu2ss_battery_state(struct nbu2ss_udc *udc, u32 new_state)
{
	if (udc->charger != new_state) {
		udc->charger = new_state;
		emxx_battery_usb_state(udc->charger);
	}
}
#endif	/* CONFIG_BATTERY_EMXX */

/*-------------------------------------------------------------------------*/
#ifdef UDC_DETECT_UEVENT
static inline void ___nbu2ss_check_vbus(struct nbu2ss_udc *udc);
static struct workqueue_struct *udc_detect_workqueue = NULL;
static struct delayed_work	udc_detect_work;

static void _nbu2ss_vbus_detect(struct work_struct *work)
{
	struct nbu2ss_udc	*udc = &udc_controller;
	___nbu2ss_check_vbus(udc);
	return;
}

static void _nbu2ss_check_vbus(struct nbu2ss_udc *udc)
{
	queue_delayed_work(udc_detect_workqueue,&udc_detect_work,0);
}

static inline void ___nbu2ss_check_vbus(struct nbu2ss_udc *udc)
#else
static inline void _nbu2ss_check_vbus(struct nbu2ss_udc *udc)
#endif
{
	int	nret;
	u32	reg_dt;

	/* chattering */
#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_wait_vbus_chattering)
		udc->config->udc_wait_vbus_chattering();
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	mdelay(VBUS_CHATTERING_MDELAY);		/* wait (ms) */
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

	/* VBUS ON Check*/
#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		reg_dt = udc->config->udc_check_vbus();
	else
		reg_dt = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	reg_dt = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) reg_dt = 1;
	if (g_vbus_ow == 2) reg_dt = 0;

	if (reg_dt == 0) {

#ifdef USE_SUSPEND_WAIT
		udc->usb_suspend_flag = 0;
#endif	/* USE_SUSPEND_WAIT */
		udc->linux_suspended = 0;

		_nbu2ss_reset_controller(udc);
		INFO(" ----- VBUS OFF\n");
		
#ifdef UDC_DETECT_UEVENT
		kobject_uevent(&udc->dev->kobj,KOBJ_OFFLINE);
#endif	/* UDC_DETECT_UEVENT */

		if (udc->vbus_active == 1) {
			/* VBUS OFF */
			udc->vbus_active = 0;
			if (udc->usb_suspended) {
				udc->usb_suspended = 0;
				/* _nbu2ss_reset_controller(udc); */
			}
			udc->devstate = USB_STATE_NOTATTACHED;

			_nbu2ss_quiesce(udc);

			if (udc->driver) {
				spin_unlock(&udc->lock);
				udc->driver->disconnect(&udc->gadget);
				mdelay(10);
				spin_lock(&udc->lock);
			}

			_nbu2ss_disable_controller(udc);
		}
#ifdef CONFIG_BATTERY_EMXX
		_nbu2ss_battery_state(udc, EMXX_USB_OFF);
#endif
	} else {
		mdelay(5);		/* wait (5ms) */
#ifdef CONFIG_SNSC_EMXX_PLATFORM
		if (udc->config->udc_check_vbus)
			reg_dt = udc->config->udc_check_vbus();
#else /* CONFIG_SNSC_EMXX_PLATFORM */
		reg_dt = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
		if (g_vbus_ow == 1) reg_dt = 1;
		if (g_vbus_ow == 2) reg_dt = 0;

		if (reg_dt == 0)
			return;

		INFO(" ----- VBUS ON\n");
		
#ifdef UDC_DETECT_UEVENT
		kobject_uevent(&udc->dev->kobj,KOBJ_ONLINE);
#endif	/* UDC_DETECT_UEVENT */

#ifdef CONFIG_BATTERY_EMXX
		_nbu2ss_battery_state(udc, EMXX_USB_CHARGER);
#endif	/* CONFIG_BATTERY_EMXX */

		if (udc->linux_suspended)
			return;

		if (udc->vbus_active == 0) {
			/* VBUS ON */
			udc->vbus_active = 1;
			udc->devstate = USB_STATE_POWERED;

			nret = _nbu2ss_enable_controller(udc);
			if (nret < 0) {
				_nbu2ss_disable_controller(udc);
				udc->vbus_active = 0;
				return;
			}

			_nbu2ss_pullup(udc, 1);

#ifdef UDC_DEBUG_DUMP
			_nbu2ss_dump_register(udc);
#endif /* UDC_DEBUG_DUMP */

		} else {
			if (udc->devstate == USB_STATE_POWERED)
				_nbu2ss_pullup(udc, 1);
		}
	}

	return;
}

#ifdef USE_SUSPEND_WAIT
/*-------------------------------------------------------------------------*/
static void _nbu2ss_usb_suspend(struct work_struct *work)
{
	u32	reg_dt;
	struct nbu2ss_udc	*udc = &udc_controller;

	if (udc->usb_suspend_flag && (udc->usb_suspended == 0)) {
		udc->usb_suspend_flag = 0;

#ifdef CONFIG_SNSC_EMXX_PLATFORM
		if (udc->config->udc_check_vbus)
			reg_dt = udc->config->udc_check_vbus();
		else
			reg_dt = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
		reg_dt = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
		if (g_vbus_ow == 1) reg_dt = 1;
		if (g_vbus_ow == 2) reg_dt = 0;

		if (reg_dt == 0)
			return;

		udc->usb_suspended = 1;
		if (udc->driver && udc->driver->suspend)
			udc->driver->suspend(&udc->gadget);

		INFO(" ----- Set SUSPEND bit\n");
		_nbu2ss_bitset(&udc->p_regs->USB_CONTROL, SUSPEND);
	}
}
#endif	/* USE_SUSPEND_WAIT */

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_int_bus_reset(struct nbu2ss_udc *udc)
{
	udc->devstate		= USB_STATE_DEFAULT;
	udc->remote_wakeup	= 0;

	_nbu2ss_quiesce(udc);

	udc->ep0state = EP0_IDLE;
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_int_usb_resume(struct nbu2ss_udc *udc)
{
#ifdef CONFIG_BATTERY_EMXX
	_nbu2ss_battery_state(udc, EMXX_USB_DEVICE);
#endif	/* CONFIG_BATTERY_EMXX */

	if (udc->usb_suspended == 1) {
		udc->usb_suspended = 0;
		if (udc->driver && udc->driver->resume)
			udc->driver->resume(&udc->gadget);
	}
}

/*-------------------------------------------------------------------------*/
static inline void _nbu2ss_int_usb_suspend(struct nbu2ss_udc *udc)
{
#ifdef USE_SUSPEND_WAIT
	udc->usb_suspend_flag = 1;
	queue_delayed_work(
		udc->usb_suspend_workqueue,
		&udc->usb_suspend_work, msecs_to_jiffies(USB_SUSPEND_TIME));
#else	/* !USE_SUSPEND_WAIT */
	u32	reg_dt;

	if (udc->usb_suspended == 0) {
#ifdef CONFIG_SNSC_EMXX_PLATFORM
		if (udc->config->udc_check_vbus)
			reg_dt = udc->config->udc_check_vbus();
		else
			reg_dt = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
		reg_dt = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

		if (g_vbus_ow == 1) reg_dt = 1;
		if (g_vbus_ow == 2) reg_dt = 0;
		
		if (reg_dt == 0)
			return;

		udc->usb_suspended = 1;
		if (udc->driver && udc->driver->suspend)
			udc->driver->suspend(&udc->gadget);

		_nbu2ss_bitset(&udc->p_regs->USB_CONTROL, SUSPEND);
	}
#endif	/* USE_SUSPEND_WAIT */
}

#ifdef CONFIG_BATTERY_EMXX
/*-------------------------------------------------------------------------*/
static void _nbu2ss_timer(unsigned long _udc)
{
	u32		data;
	struct nbu2ss_udc	*udc = (struct nbu2ss_udc *)_udc;

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (data == 0)
		return;

	if (udc->charger == EMXX_USB_OFF) {
		data = _nbu2ss_readl(&udc->p_regs->USB_INT_STA);
		if (data & SOF_INT)
			_nbu2ss_battery_state(udc, EMXX_USB_DEVICE);
		else
			_nbu2ss_battery_state(udc, EMXX_USB_CHARGER);
	}
}
#endif	/* CONFIG_BATTERY_EMXX */

#ifndef CONFIG_SNSC_EMXX_PLATFORM
/*-------------------------------------------------------------------------*/
/* VBUS (GPIO153) Interrupt */
static irqreturn_t _nbu2ss_vbus_irq(int irq, void *_udc)
{
	struct nbu2ss_udc	*udc = (struct nbu2ss_udc *)_udc;

	spin_lock(&udc->lock);
	_nbu2ss_check_vbus(udc);
	spin_unlock(&udc->lock);

	return IRQ_HANDLED;
}
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

/*-------------------------------------------------------------------------*/
/* Interrupt (udc) */
static irqreturn_t _nbu2ss_udc_irq(int irq, void *_udc)
{
	u8	suspend_flag = 0;
	u32	status;
	u32	epnum, int_bit;

	struct nbu2ss_udc	*udc = (struct nbu2ss_udc *)_udc;
	PT_FC_REGS		preg = udc->p_regs;

#ifdef USE_SUSPEND_WAIT
	udc->usb_suspend_flag = 0;
#endif	/* USE_SUSPEND_WAIT */

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if ((udc->config->udc_check_vbus && udc->config->udc_check_vbus() == 0 && g_vbus_ow == 0) || g_vbus_ow == 2){
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	if (gpio_get_value(VBUS_VALUE) == 0) {
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
		_nbu2ss_writel(&preg->USB_INT_STA, ~USB_INT_STA_RW);
		_nbu2ss_writel(&preg->USB_INT_ENA, 0);
		return IRQ_HANDLED;
	}

	spin_lock(&udc->lock);

#ifdef USE_INT_COUNT_OVER
	udc->int_count = 0;
#endif

	for (;;) {
#ifdef CONFIG_SNSC_EMXX_PLATFORM
		if ((udc->config->udc_check_vbus && udc->config->udc_check_vbus() == 0 && g_vbus_ow == 0) || g_vbus_ow == 2) {
#else /* CONFIG_SNSC_EMXX_PLATFORM */
		if (gpio_get_value(VBUS_VALUE) == 0) {
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
			_nbu2ss_writel(&preg->USB_INT_STA, ~USB_INT_STA_RW);
			_nbu2ss_writel(&preg->USB_INT_ENA, 0);
			status = 0;
		}
		else
			status = _nbu2ss_readl(&preg->USB_INT_STA);

#ifdef USE_INT_COUNT_OVER
		udc->int_count++;
		if (udc->int_count == 20 && g_is_ether == 0) {
			_nbu2ss_disable_controller(udc);
			ERR(" *** %s, int_count Over !!\n", __func__);
			status = 0;
		}
#endif

		if (status == 0)
			break;

		_nbu2ss_writel(&preg->USB_INT_STA, ~(status & USB_INT_STA_RW));

		if (status & USB_RST_INT) {
			/* USB Reset */
			_nbu2ss_int_bus_reset(udc);
		}

		if (status & RSUM_INT) {
			/* Resume */
			_nbu2ss_int_usb_resume(udc);
		}

		if (status & SPND_INT) {
			/* Suspend */
			suspend_flag = 1;
		}

		if (status & EPn_INT) {
			/* EP INT */
			int_bit = status >> 8;

			for (epnum = 0; epnum < NUM_ENDPOINTS; epnum++) {

				if (0x01 & int_bit)
					_nbu2ss_ep_int(udc, epnum);

				int_bit >>= 1;

				if (int_bit == 0)
					break;
			}
		}
	}

	if (suspend_flag)
		_nbu2ss_int_usb_suspend(udc);

	spin_unlock(&udc->lock);

	return IRQ_HANDLED;
}

/*-------------------------------------------------------------------------*/
/* usb_ep_ops */
static int nbu2ss_ep_enable(
	struct usb_ep *_ep,
	const struct usb_endpoint_descriptor *desc)
{
	u8		ep_type;
	unsigned long	flags;

	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;

	if ((_ep == NULL) || (desc == NULL)) {
		ERR(" *** %s, bad param\n", __func__);
		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if ((ep == NULL) || (ep->udc == NULL)) {
		ERR(" *** %s, ep == NULL !!\n", __func__);
		return -EINVAL;
	}

	ep_type = desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK;
	if ((ep_type == USB_ENDPOINT_XFER_CONTROL)
		|| (ep_type == USB_ENDPOINT_XFER_ISOC)) {

		ERR(" *** %s, bat bmAttributes\n", __func__);
		return -EINVAL;
	}

	udc = ep->udc;
	if (udc->vbus_active == 0)
		return -ESHUTDOWN;

	if ((udc->driver == NULL)
		|| (udc->gadget.speed == USB_SPEED_UNKNOWN)) {

		ERR(" *** %s, udc !!\n", __func__);
		return -ESHUTDOWN;
	}

	spin_lock_irqsave(&udc->lock, flags);

	ep->desc = desc;
	ep->epnum = desc->bEndpointAddress & USB_ENDPOINT_NUMBER_MASK;
	ep->direct = desc->bEndpointAddress & USB_ENDPOINT_DIR_MASK;
	ep->ep_type = ep_type;
	ep->wedged = 0;
	ep->halted = FALSE;
	ep->stalled = FALSE;

	ep->ep.maxpacket = le16_to_cpu(desc->wMaxPacketSize);

	/* DMA setting */
	_nbu2ss_ep_dma_init(udc, ep);

	/* Endpoint setting */
	_nbu2ss_ep_init(udc, ep);

	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_ep_disable(struct usb_ep *_ep)
{
	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

	if (_ep == NULL) {
		ERR(" *** %s, bad param\n", __func__);
		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if ((ep == NULL) || (ep->udc == NULL)) {
		ERR(" *** %s, ep == NULL !!\n", __func__);
		return -EINVAL;
	}

	udc = ep->udc;
	if (udc->vbus_active == 0)
		return -ESHUTDOWN;

	spin_lock_irqsave(&udc->lock, flags);
	_nbu2ss_nuke(udc, ep, -EINPROGRESS);		/* dequeue request */
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static struct usb_request *nbu2ss_ep_alloc_request(
	struct usb_ep *ep,
	gfp_t gfp_flags)
{
	struct nbu2ss_req *req;

	req = kzalloc(sizeof(*req), gfp_flags);
	if (!req)
		return 0;

#ifdef USE_DMA
	if(g_is_ether == 0){
		req->req.dma = DMA_ADDR_INVALID;
	}
#endif
	INIT_LIST_HEAD(&req->queue);

	return &req->req;
}

/*-------------------------------------------------------------------------*/
static void nbu2ss_ep_free_request(
	struct usb_ep *_ep,
	struct usb_request *_req)
{
	struct nbu2ss_req *req;

	if (_req != NULL) {
		req = container_of(_req, struct nbu2ss_req, req);

		if (req != NULL)
			kfree(req);
	}
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_ep_queue(
	struct usb_ep *_ep,
	struct usb_request *_req,
	gfp_t gfp_flags)
{
	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;
	unsigned long		flags;
	bool			bflag;
	int			result = -EINVAL;

	/* catch various bogus parameters */
	if ((_ep == NULL) || (_req == NULL)) {
		if (_ep == NULL)
			ERR("*** %s --- _ep == NULL\n", __func__);

		if (_req == NULL)
			ERR("*** %s --- _req == NULL\n", __func__);

		return -EINVAL;
	}

	req = container_of(_req, struct nbu2ss_req, req);
	if (unlikely
	    (!_req->complete || !_req->buf
	     || !list_empty(&req->queue))) {

		if (!_req->complete)
			ERR("*** %s --- !_req->complete\n", __func__);

		if (!_req->buf)
			ERR("*** %s --- !_req->buf\n", __func__);

		if (!list_empty(&req->queue))
			ERR("*** %s --- !list_empty(&req->queue)\n", __func__);

		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	udc = ep->udc;

/*	INFO("=== %s(ep%d), zero=%d\n", __func__, ep->epnum, _req->zero); */

	if (udc->vbus_active == 0) {
		INFO("Can't ep_queue (VBUS OFF)\n");
		return -ESHUTDOWN;
	}

	if (unlikely(!udc->driver)) {
		ERR("%s, bogus device state %p\n", __func__, udc->driver);
		return -ESHUTDOWN;
	}

	spin_lock_irqsave(&udc->lock, flags);

#ifdef USE_DMA
	if(g_is_ether == 0){
		if ((ep->epnum > 0) && (ep->direct == USB_DIR_OUT))
			_nbu2ss_dma_map_single(udc, req, USB_DIR_OUT);
	}
#endif

	_req->status = -EINPROGRESS;
	_req->actual = 0;

	bflag = list_empty(&ep->queue);
	list_add_tail(&req->queue, &ep->queue);

	if ((bflag != FALSE) && (ep->stalled == FALSE)) {

		result = _nbu2ss_start_transfer(udc, ep, req, FALSE);
		if (result < 0) {
			ERR(" *** %s, result = %d\n", __func__, result);
			list_del(&req->queue);
		}
	}

	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_ep_dequeue(
	struct usb_ep *_ep,
	struct usb_request *_req)
{
	struct nbu2ss_req	*req;
	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;
	unsigned long flags;

/*	INFO("=== %s()\n", __func__); */

	/* catch various bogus parameters */
	if ((_ep == NULL) || (_req == NULL)) {
		/* ERR("%s, bad param(1)\n", __func__); */
		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if (!ep) {
		ERR("%s, ep == NULL !!\n", __func__);
		return -EINVAL;
	}

	udc = ep->udc;

	spin_lock_irqsave(&udc->lock, flags);

	/* make sure it's actually queued on this endpoint */
	list_for_each_entry(req, &ep->queue, queue) {
		if (&req->req == _req)
			break;
	}
	if (&req->req != _req) {
		spin_unlock_irqrestore(&udc->lock, flags);
		DBG("%s no queue(EINVAL)\n", __func__);
		return -EINVAL;
	}

	_nbu2ss_ep_done(ep, req, -ECONNRESET);

	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_ep_set_halt(struct usb_ep *_ep, int value)
{
	u8		ep_adrs;
	unsigned long	flags;

	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;

/*	INFO("=== %s()\n", __func__); */

	if (!_ep) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if (!ep) {
		ERR("%s, bad ep\n", __func__);
		return -EINVAL;
	}

	udc = ep->udc;
	if (!udc) {
		ERR(" *** %s, bad udc\n", __func__);
		return -EINVAL;
	}

	spin_lock_irqsave(&udc->lock, flags);

	ep_adrs = ep->epnum | ep->direct;
	if (value == 0) {
		_nbu2ss_set_endpoint_stall(udc, ep_adrs, value);
		ep->stalled = FALSE;
	} else {
		if (list_empty(&ep->queue))
			_nbu2ss_epn_set_stall(udc, ep);
		else
			ep->stalled = TRUE;
	}

	if (value == 0)
		ep->wedged = 0;

	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

static int nbu2ss_ep_set_wedge(struct usb_ep *_ep)
{
	return nbu2ss_ep_set_halt(_ep, 1);
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_ep_fifo_status(struct usb_ep *_ep)
{
	u32		data;
	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;
	unsigned long		flags;
	PT_FC_REGS		preg;

/*	INFO("=== %s()\n", __func__); */

	if (!_ep) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if (!ep) {
		ERR("%s, bad ep\n", __func__);
		return -EINVAL;
	}

	udc = ep->udc;
	if (!udc) {
		ERR("%s, bad udc\n", __func__);
		return -EINVAL;
	}

	preg = udc->p_regs;

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) data = 1;
	if (g_vbus_ow == 2) data = 0;
	
	if (data == 0)
		return -EINVAL;

	spin_lock_irqsave(&udc->lock, flags);

	if (ep->epnum == 0) {
		data = _nbu2ss_readl(&preg->EP0_LENGTH) & EP0_LDATA;

	} else {
		data = _nbu2ss_readl(&preg->EP_REGS[ep->epnum-1].EP_LEN_DCNT)
			& EPn_LDATA;
	}

	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static void  nbu2ss_ep_fifo_flush(struct usb_ep *_ep)
{
	u32			data;
	struct nbu2ss_ep	*ep;
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

/*	INFO("=== %s()\n", __func__); */

	if (!_ep) {
		ERR("%s, bad param\n", __func__);
		return;
	}

	ep = container_of(_ep, struct nbu2ss_ep, ep);
	if (!_ep) {
		ERR("%s, bad ep\n", __func__);
		return;
	}

	udc = ep->udc;
	if (!udc) {
		ERR("%s, bad udc\n", __func__);
		return;
	}

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) data = 1;
	if (g_vbus_ow == 2) data = 0;
	
	if (data == 0)
		return;

	spin_lock_irqsave(&udc->lock, flags);
	_nbu2ss_fifo_flush(udc, ep);
	spin_unlock_irqrestore(&udc->lock, flags);
}

/*-------------------------------------------------------------------------*/
static struct usb_ep_ops nbu2ss_ep_ops = {
	.enable		= nbu2ss_ep_enable,
	.disable	= nbu2ss_ep_disable,

	.alloc_request	= nbu2ss_ep_alloc_request,
	.free_request	= nbu2ss_ep_free_request,

	.queue		= nbu2ss_ep_queue,
	.dequeue	= nbu2ss_ep_dequeue,

	.set_halt	= nbu2ss_ep_set_halt,
	.set_wedge	= nbu2ss_ep_set_wedge,

	.fifo_status	= nbu2ss_ep_fifo_status,
	.fifo_flush	= nbu2ss_ep_fifo_flush,
};


/*-------------------------------------------------------------------------*/
/* usb_gadget_ops */

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_get_frame(struct usb_gadget *pgadget)
{
	u32			data;
	struct nbu2ss_udc	*udc;

/*	INFO("=== %s()\n", __func__); */

	if (pgadget == NULL) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	udc = container_of(pgadget, struct nbu2ss_udc, gadget);
	if (udc == NULL) {
		ERR("%s, udc == NULL\n", __func__);
		return -EINVAL;
	}

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) data = 1;
	if (g_vbus_ow == 2) data = 0;
	
	if (data == 0)
		return -EINVAL;

	data = _nbu2ss_readl(&udc->p_regs->USB_ADDRESS) & FRAME;

	return data;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_wakeup(struct usb_gadget *pgadget)
{
	int	i;
	u32	data;

	struct nbu2ss_udc	*udc;

/*	INFO("=== %s()\n", __func__); */

	if (pgadget == NULL) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	udc = container_of(pgadget, struct nbu2ss_udc, gadget);
	if (udc == NULL) {
		ERR("%s, udc == NULL\n", __func__);
		return -EINVAL;
	}

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) data = 1;
	if (g_vbus_ow == 2) data = 0;
	
	if (data == 0) {
		WARNING("VBUS LEVEL = %d\n", data);
		return -EINVAL;
	}

	_nbu2ss_bitset(&udc->p_regs->EPCTR, PLL_RESUME);

	for (i = 0; i < EPC_PLL_LOCK_COUNT; i++) {
		data = _nbu2ss_readl(&udc->p_regs->EPCTR);

		if (data & PLL_LOCK)
			break;
	}

	_nbu2ss_bitclr(&udc->p_regs->EPCTR, PLL_RESUME);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_set_selfpowered(struct usb_gadget *pgadget,
					int is_selfpowered)
{
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

/*	INFO("=== %s()\n", __func__); */

	if (pgadget == NULL) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	udc = container_of(pgadget, struct nbu2ss_udc, gadget);

	spin_lock_irqsave(&udc->lock, flags);
	udc->self_powered = (is_selfpowered != 0);
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_vbus_session(struct usb_gadget *pgadget, int is_active)
{
/*	INFO("=== %s()\n", __func__); */
	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_vbus_draw(struct usb_gadget *pgadget, unsigned mA)
{
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

/*	INFO("=== %s()\n", __func__); */

	if (pgadget == NULL) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	udc = container_of(pgadget, struct nbu2ss_udc, gadget);

	spin_lock_irqsave(&udc->lock, flags);
	udc->mA = mA;
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_pullup(struct usb_gadget *pgadget, int is_on)
{
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

/*	INFO("=== %s()\n", __func__); */

	if (pgadget == NULL) {
		ERR("%s, bad param\n", __func__);
		return -EINVAL;
	}

	udc = container_of(pgadget, struct nbu2ss_udc, gadget);

	if (udc->driver == NULL) {
		WARNING("%s, Not Regist Driver\n", __func__);
		return -EINVAL;
	}

	if (udc->vbus_active == 0)
		return -ESHUTDOWN;

	spin_lock_irqsave(&udc->lock, flags);
	_nbu2ss_pullup(udc, is_on);
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_gad_ioctl(
	struct usb_gadget *pgadget,
	unsigned code,
	unsigned long param)
{
/*	INFO("=== %s()\n", __func__); */
	return 0;
}


static const struct usb_gadget_ops nbu2ss_gadget_ops = {
	.get_frame		= nbu2ss_gad_get_frame,
	.wakeup			= nbu2ss_gad_wakeup,
	.set_selfpowered	= nbu2ss_gad_set_selfpowered,
	.vbus_session		= nbu2ss_gad_vbus_session,
	.vbus_draw		= nbu2ss_gad_vbus_draw,
	.pullup			= nbu2ss_gad_pullup,
	.ioctl			= nbu2ss_gad_ioctl,
};


/*-------------------------------------------------------------------------*/
/*
  Register entry point for the peripheral controller driver.
*/
int usb_gadget_register_driver(struct usb_gadget_driver *driver)
{
	int	retval = 0;

	struct nbu2ss_udc	*udc;

	udc = &udc_controller;

	INFO("driver->function: [%s]\n", driver->function);
	
	if(strcmp(driver->function, "g_ether") == 0){
		g_is_ether = 1;
	}else{
		g_is_ether = 0;
	}
	
	if (!udc)
		return -EINVAL;
	if (udc->driver)
		return -EBUSY;

	/* bind */
	if (!driver
	    || !driver->bind
	    || !driver->disconnect
	    || !driver->setup) {

		if (!driver) {
			DBG("--- %s --- driver == NULL\n", __func__);

		} else {
			if (!driver->bind)
				DBG("--- %s --- bind == NULL\n", __func__);

			if (!driver->disconnect)
				DBG("--- %s --- disconnect == NULL\n"
					, __func__);

			if (!driver->setup)
				DBG("--- %s --- setup == NULL\n", __func__);
		}

		return -EINVAL;
	}

	udc->driver = driver;
	udc->gadget.dev.driver = &driver->driver;

	retval = device_add(&udc->gadget.dev);
	if (retval) {
		ERR("device_add error (%d)\n", retval);
		return -EINVAL;
	}

	retval = driver->bind(&udc->gadget);
	if (retval) {
		DBG("--- %s --- driver->bind() returned %d\n"
			, __func__, retval);

		udc->driver = NULL;
		ERR(" *** %s --- bind failed\n", __func__);
		return retval;
	}

	_nbu2ss_check_vbus(udc);

	return 0;
}
EXPORT_SYMBOL(usb_gadget_register_driver);


/*
  Unregister entry point for the peripheral controller driver.
*/
int usb_gadget_unregister_driver(struct usb_gadget_driver *driver)
{
	struct nbu2ss_udc	*udc;
	unsigned long		flags;

	udc = &udc_controller;

	spin_lock_irqsave(&udc->lock, flags);
	_nbu2ss_quiesce(udc);
	spin_unlock_irqrestore(&udc->lock, flags);

	/* disconnect */
	if (driver->disconnect) {
		driver->disconnect(&udc->gadget);
		mdelay(10);
	}

	/* unbind */
	if (driver->unbind)
		driver->unbind(&udc->gadget);

	_nbu2ss_pullup(udc, FALSE);

	udc->driver = NULL;
	udc->gadget.dev.driver = NULL;
	udc->devstate = USB_STATE_POWERED;

	device_del(&udc->gadget.dev);

#ifdef CONFIG_BATTERY_EMXX
	_nbu2ss_battery_state(udc, EMXX_USB_CHARGER);
#endif	/* CONFIG_BATTERY_EMXX */

	g_is_ether = 0;

	return 0;
}
EXPORT_SYMBOL(usb_gadget_unregister_driver);

static char g_ep0_name[] = "ep0";
static char g_ep1_name[] = "ep1-bulk";
static char g_ep2_name[] = "ep2-bulk";
static char g_ep3_name[] = "ep3in-int";
static char g_ep4_name[] = "ep4-iso";
static char g_ep5_name[] = "ep5-iso";
static char g_ep6_name[] = "ep6-bulk";
static char g_ep7_name[] = "ep7-bulk";
static char g_ep8_name[] = "ep8in-int";
static char g_ep9_name[] = "ep9-iso";
static char g_epa_name[] = "epa-iso";
static char g_epb_name[] = "epb-bulk";
static char g_epc_name[] = "epc-bulk";
static char g_epd_name[] = "epdin-int";

static char *gp_ep_name[NUM_ENDPOINTS] = {
	g_ep0_name,
	g_ep1_name,
	g_ep2_name,
	g_ep3_name,
	g_ep4_name,
	g_ep5_name,
	g_ep6_name,
	g_ep7_name,
	g_ep8_name,
	g_ep9_name,
	g_epa_name,
	g_epb_name,
	g_epc_name,
	g_epd_name,
};

/*-------------------------------------------------------------------------*/
static void __init nbu2ss_drv_set_ep_info(
	struct nbu2ss_udc	*udc,
	struct nbu2ss_ep	*ep,
	u8 *name)
{
	ep->udc = udc;
	ep->desc = NULL;

	ep->ep.driver_data = NULL;
	ep->ep.name = name;
	ep->ep.ops = &nbu2ss_ep_ops;

	if (isdigit(name[2])) {

		long	num;
		int	res;
		char	tempbuf[2];

		tempbuf[0] = name[2];
		tempbuf[1] = '\0';
		res = strict_strtol(tempbuf, 16, &num);

		if (num == 0)
			ep->ep.maxpacket = EP0_PACKETSIZE;
		else
			ep->ep.maxpacket = EP_PACKETSIZE;

	} else {
		ep->ep.maxpacket = EP_PACKETSIZE;
	}

	list_add_tail(&ep->ep.ep_list, &udc->gadget.ep_list);
	INIT_LIST_HEAD(&ep->queue);
}

/*-------------------------------------------------------------------------*/
static void __init nbu2ss_drv_ep_init(struct nbu2ss_udc *udc)
{
	int	i;

	INIT_LIST_HEAD(&udc->gadget.ep_list);
	udc->gadget.ep0 = &udc->ep[0].ep;


	for (i = 0; i < NUM_ENDPOINTS; i++)
		nbu2ss_drv_set_ep_info(udc, &udc->ep[i], gp_ep_name[i]);

	list_del_init(&udc->ep[0].ep.ep_list);
}

/*-------------------------------------------------------------------------*/
/* platform_driver */
static int __init nbu2ss_drv_contest_init(
	struct platform_device *pdev,
	struct nbu2ss_udc *udc)
{
	spin_lock_init(&udc->lock);
	udc->dev = &pdev->dev;

	udc->self_powered = 1;
	udc->devstate = USB_STATE_NOTATTACHED;
	udc->pdev = pdev;
	udc->mA = 0;

#ifdef CONFIG_BATTERY_EMXX
	udc->charger = EMXX_USB_OFF;
	init_timer(&udc->timer);
	udc->timer.function = _nbu2ss_timer;
	udc->timer.data = (unsigned long)udc;
#endif	/* CONFIG_BATTERY_EMXX */

	udc->pdev->dev.coherent_dma_mask = 0xFFFFFFFF;

	/* init Endpoint */
	nbu2ss_drv_ep_init(udc);

	/* init Gadget */
	udc->gadget.ops = &nbu2ss_gadget_ops;
	udc->gadget.ep0 = &udc->ep[0].ep;
	udc->gadget.speed = USB_SPEED_UNKNOWN;
	udc->gadget.name = driver_name;
	udc->gadget.is_dualspeed = 1;

	device_initialize(&udc->gadget.dev);

	strcpy(udc->gadget.dev.bus_id, "gadget");
	udc->gadget.dev.parent = &pdev->dev;
	udc->gadget.dev.dma_mask = pdev->dev.dma_mask;

	return 0;
}

/*
 * 	probe - binds to the platform device
 */
static int __init nbu2ss_drv_probe(struct platform_device *pdev)
{
	int	status = -ENODEV;
	struct nbu2ss_udc	*udc;

	udc = &udc_controller;
	memset(udc, 0, sizeof(struct nbu2ss_udc));

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	memcpy(&udc_config, pdev->dev.platform_data, sizeof(udc_config));
	udc->config = &udc_config;
	if (udc->config->udc_vbus_work == NULL)
		udc->config->udc_vbus_work = &_nbu2ss_check_vbus;
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

	platform_set_drvdata(pdev, udc);

	/* IO Memory Region */
	if (!request_mem_region(USB_BASE_ADDRESS, USB_BASE_SIZE
				, driver_name)) {

		ERR("request_mem_region failed\n");
		return -EBUSY;
	}

	/* IO Memory */
	udc->p_regs = (PT_FC_REGS)ioremap(USB_BASE_ADDRESS, USB_BASE_SIZE);
	if (!udc->p_regs) {
		ERR("request_io_mem failed\n");
		goto cleanup3;
	}

	/* USB Function Controller Interrupt */
	status = request_irq(USB_UDC_IRQ_1,
				_nbu2ss_udc_irq,
				IRQF_DISABLED,
				driver_name,
				udc);

	if (status != 0) {
		ERR("request_irq(USB_UDC_IRQ_1) failed\n");
		goto cleanup2;
	}

	/* Driver Initialization */
	status = nbu2ss_drv_contest_init(pdev, udc);
	if (status < 0) {
		/* Error */
		goto cleanup1;
	}

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_probe) {
		status = udc->config->udc_probe(udc);
		if (status != 0) {
			ERR("udc_probe failed\n");
			goto cleanup1;
		}
	}
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	/* VBUS Interrupt */
	set_irq_type(INT_VBUS, IRQ_TYPE_EDGE_BOTH);
	status = request_irq(INT_VBUS,
				_nbu2ss_vbus_irq,
				IRQF_SHARED,
				driver_name,
				udc);

	if (status != 0) {
		ERR("request_irq(INT_VBUS) failed\n");
		goto cleanup1;
	}
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

#ifdef UDC_DETECT_UEVENT
	udc_detect_workqueue = create_singlethread_workqueue("udc_detect");
	INIT_DELAYED_WORK(&udc_detect_work, _nbu2ss_vbus_detect);
#endif

#ifdef USE_SUSPEND_WAIT
	udc->usb_suspend_workqueue = create_singlethread_workqueue("usb_suspend");
	INIT_DELAYED_WORK(&udc->usb_suspend_work, _nbu2ss_usb_suspend);
#endif	/* USE_SUSPEND_WAIT */

	create_proc_files();

	return status;

cleanup1:
	/* Interrupt Handler - Release */
	free_irq(USB_UDC_IRQ_1, udc);

cleanup2:
	/* IO Memory - Release */
	if (udc->p_regs)
		iounmap(udc->p_regs);

cleanup3:
	/* IO Memory Region - Release */
	release_mem_region(USB_BASE_ADDRESS, USB_BASE_SIZE);

	return status;
}

/*-------------------------------------------------------------------------*/
static void nbu2ss_drv_shutdown(struct platform_device *pdev)
{
	struct nbu2ss_udc	*udc;

	udc = platform_get_drvdata(pdev);
	if (udc == NULL)
		return;

	_nbu2ss_disable_controller(udc);
}

/*-------------------------------------------------------------------------*/
static int __exit nbu2ss_drv_remove(struct platform_device *pdev)
{
	struct nbu2ss_udc	*udc;

	udc = &udc_controller;

	remove_proc_files();
#ifdef UDC_DETECT_UEVENT
	destroy_workqueue(udc_detect_workqueue);
#endif

#ifdef USE_SUSPEND_WAIT
	destroy_workqueue(udc->usb_suspend_workqueue);
#endif	/* USE_SUSPEND_WAIT */

	/* Interrupt Handler - Release */
	free_irq(USB_UDC_IRQ_1, udc);

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_remove)
		udc->config->udc_remove(udc);
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	/* Interrupt Handler - Release */
	free_irq(INT_VBUS, udc);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */

	/* IO Memory - Release */
	if (udc->p_regs)
		iounmap(udc->p_regs);

	/* IO Memory Region - Release */
	release_mem_region(USB_BASE_ADDRESS, USB_BASE_SIZE);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_drv_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct nbu2ss_udc	*udc;

	udc = platform_get_drvdata(pdev);
	if (udc == NULL)
		return 0;

	if (udc->vbus_active) {
		udc->vbus_active = 0;
		udc->devstate = USB_STATE_NOTATTACHED;
		udc->linux_suspended = 1;

		if (udc->usb_suspended) {
			udc->usb_suspended = 0;
			_nbu2ss_reset_controller(udc);
		}

		_nbu2ss_quiesce(udc);

		if (udc->driver) {
			udc->driver->disconnect(&udc->gadget);
			mdelay(10);
		}
	}
	_nbu2ss_disable_controller(udc);

	return 0;
}

/*-------------------------------------------------------------------------*/
static int nbu2ss_drv_resume(struct platform_device *pdev)
{
	u32	data;
	struct nbu2ss_udc	*udc;

	udc = platform_get_drvdata(pdev);
	if (udc == NULL)
		return 0;

#ifdef CONFIG_SNSC_EMXX_PLATFORM
	if (udc->config->udc_check_vbus)
		data = udc->config->udc_check_vbus();
	else
		data = 1;
#else /* CONFIG_SNSC_EMXX_PLATFORM */
	data = gpio_get_value(VBUS_VALUE);
#endif /* CONFIG_SNSC_EMXX_PLATFORM */
	if (g_vbus_ow == 1) data = 1;
	if (g_vbus_ow == 2) data = 0;

	if (data) {
#ifdef UDC_DETECT_UEVENT
		kobject_uevent(&udc->dev->kobj,KOBJ_ONLINE);
#endif	/* UDC_DETECT_UEVENT */
		udc->vbus_active = 1;
		udc->devstate = USB_STATE_POWERED;
		_nbu2ss_enable_controller(udc);
		_nbu2ss_pullup(udc, 1);
	} else {
#ifdef UDC_DETECT_UEVENT
		kobject_uevent(&udc->dev->kobj,KOBJ_OFFLINE);
#endif	/* UDC_DETECT_UEVENT */
	}

	udc->linux_suspended = 0;

	return 0;
}


static struct platform_driver udc_driver = {
	.probe		= nbu2ss_drv_probe,
	.shutdown	= nbu2ss_drv_shutdown,
	.remove		= __exit_p(nbu2ss_drv_remove),
	.suspend	= nbu2ss_drv_suspend,
	.resume		= nbu2ss_drv_resume,
	.driver		= {
		.owner	= THIS_MODULE,
		.name	= driver_name,
	},
};

/*-------------------------------------------------------------------------*/
static ssize_t vbus_write( struct file *filp, const char __user *buff, unsigned long len, void *data )
{
	char str[2];
	struct nbu2ss_udc	*udc;
	
	if (len != 1) {
		ERR("len: %ld\n", len);
		return -ENOSPC;
	}
	
	if(get_user(str[0], buff)) return -EFAULT;
	
	if (str[0] == '0') {
		ERR("VBUS OverWrite: Disable\n");
		g_vbus_ow = 0;
	}else if (str[0] == '1') {
		ERR("VBUS OverWrite: On\n");
		g_vbus_ow = 1;
	}else if (str[0] == '2') {
		ERR("VBUS OverWrite: Off\n");
		g_vbus_ow = 2;
	}else{
		ERR("Param Error\n");
		return len;
	}
	
	udc = &udc_controller;
	if(udc != NULL && udc->config != NULL){
		_nbu2ss_check_vbus(udc);
	}
	
	return len;
}

/*-------------------------------------------------------------------------*/
static int vbus_read( char *page, char **start, off_t off, int count, int *eof, void *data )
{
	int len;
	
	if (off > 0) {
		*eof = 1;
		return 0;
	}
	
	len = snprintf(page, count, "%d\n", g_vbus_ow);
	
	return len;
}

/*-------------------------------------------------------------------------*/
/* module */

/*-------------------------------------------------------------------------*/
static int __init udc_init(void)
{
	INFO("version %s\n", DRIVER_VERSION);
	
	proc_entry = create_proc_entry( vbus_proc_node_name, 0644, NULL );
	if(proc_entry != NULL){
		proc_entry->read_proc	= vbus_read;
		proc_entry->write_proc	= vbus_write;
		proc_entry->owner		= THIS_MODULE;
	}else{
		ERR("Error: create_proc_entry\n");
	}
	
	return platform_driver_register(&udc_driver);
}
module_init(udc_init);

/*-------------------------------------------------------------------------*/
static void __exit udc_exit(void)
{
	platform_driver_unregister(&udc_driver);
	if(proc_entry != NULL){
		remove_proc_entry( vbus_proc_node_name, NULL );
	}
}
module_exit(udc_exit);

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(" ");
MODULE_LICENSE("GPL");


