/*
 * EHCI HCD (Host Controller Driver) for USB.
 *
 * (C) Copyright 2008 Toshiba Corporation
 * (C) Copyright 2000-2004 David Brownell <dbrownell@users.sourceforge.net>
 *
 * Bus Glue for AYU
 *
 * Based on "ehci-au1xxx.c"
 *
 * This file is licenced under the GPL.
 */

#if defined(CONFIG_AYU)
#include <asm/ayu/ayu_reg.h>
#endif
#include <linux/platform_device.h>

extern int usb_disabled(void);

#if defined(CONFIG_ARCH_BST)
#if defined(CONFIG_USB_AYU_HCD_BANDLIMITING_WORKAROUND)
static spinlock_t ehci_ayu_lock[BST_USBHCB_NUM];
#endif
#endif

/*-------------------------------------------------------------------------*/

/* configure so an HC device and id are always provided */
/* always called with process context; sleeping is OK */

/**
 * usb_ehci_ayu_probe - initialize AYU-based HCDs
 * Context: !in_interrupt()
 *
 * Allocates basic resources for this USB host controller, and
 * then invokes the start() method for the HCD associated with it
 * through the hotplug entry's driver_data.
 *
 */
static int usb_ehci_ayu_probe(const struct hc_driver *driver,
			  struct usb_hcd **hcd_out, struct platform_device *dev)
{
	int retval;
	struct usb_hcd *hcd;
	struct ehci_hcd *ehci;

#if defined(CONFIG_ARCH_BST)
#if defined(CONFIG_USB_AYU_HCD_BANDLIMITING_WORKAROUND)
	if ((dev->id < 0) || (dev->id >= BST_USBHCB_NUM)) {
		pr_debug("invalid id=%d", dev->id);
		return -ENODEV;
	}
	spin_lock_init(&ehci_ayu_lock[dev->id]);
#endif
#endif

	if (dev->resource[1].flags != IORESOURCE_IRQ) {
		pr_debug("resource[1] is not IORESOURCE_IRQ");
		retval = -ENOMEM;
	}
	hcd = usb_create_hcd(driver, &dev->dev, "ayu");
	if (!hcd)
		return -ENOMEM;
	hcd->rsrc_start = dev->resource[0].start;
	hcd->rsrc_len = dev->resource[0].end - dev->resource[0].start + 1;

	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len, hcd_name)) {
		pr_debug("request_mem_region failed");
		retval = -EBUSY;
		goto err1;
	}

	hcd->regs = ioremap(hcd->rsrc_start, hcd->rsrc_len);
	if (!hcd->regs) {
		pr_debug("ioremap failed");
		retval = -ENOMEM;
		goto err2;
	}

	ehci = hcd_to_ehci(hcd);
	ehci->caps = hcd->regs;
	ehci->regs = hcd->regs +
		HC_LENGTH(ehci_readl(ehci, &ehci->caps->hc_capbase));
	/* cache this readonly data; minimize chip reads */
	ehci->hcs_params = ehci_readl(ehci, &ehci->caps->hcs_params);

	/* ehci_hcd_init(hcd_to_ehci(hcd)); */

	ehci->sbrn = 0x20;

	retval =
	    usb_add_hcd(hcd, dev->resource[1].start, IRQF_SHARED);
	if (retval == 0)
		return retval;

	iounmap(hcd->regs);
err2:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
err1:
	usb_put_hcd(hcd);
	return retval;
}

/* may be called without controller electrically present */
/* may be called with controller, bus, and devices active */

/**
 * usb_ehci_hcd_ayu_remove - shutdown processing for AYU-based HCDs
 * @dev: USB Host Controller being removed
 * Context: !in_interrupt()
 *
 * Reverses the effect of usb_ehci_hcd_ayu_probe(), first invoking
 * the HCD's stop() method.  It is always called from a thread
 * context, normally "rmmod", "apmd", or something similar.
 *
 */
static void usb_ehci_ayu_remove(struct usb_hcd *hcd, struct platform_device *dev)
{
	usb_remove_hcd(hcd);
	iounmap(hcd->regs);
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
}

/*-------------------------------------------------------------------------*/

#if defined(CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION)
#if defined(CONFIG_AYU)
static int ehci_ayu_match_companion(struct usb_hcd *hcd, struct usb_hcd *chcd)
{
	return (strcmp(chcd->product_desc, "AYU OHCI"))? 0: 1;
}
#elif defined(CONFIG_ARCH_BST)
static int ehci_ayu_match_companion(struct usb_hcd *hcd, struct usb_hcd *chcd)
{
	struct usb_bus *bus, *cbus;
	struct platform_device *pdev, *cpdev;

	bus = hcd_to_bus(hcd);
	cbus = hcd_to_bus(chcd);
	pdev = to_platform_device(bus->controller);
	cpdev = to_platform_device(cbus->controller);

	if ((pdev->id != cpdev->id) || strcmp(chcd->product_desc, "AYU OHCI"))
		return 0;
	else
		return 1;
}
#endif
#endif /* CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION */

static const struct hc_driver ehci_ayu_hc_driver = {
	.description = hcd_name,
	.product_desc = "AYU EHCI",
	.hcd_priv_size = sizeof(struct ehci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq = ehci_irq,
	.flags = HCD_MEMORY | HCD_USB2,

	/*
	 * basic lifecycle operations
	 */
	.reset = ehci_init,
	.start = ehci_run,
	.stop = ehci_stop,
	.shutdown = ehci_shutdown,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue = ehci_urb_enqueue,
	.urb_dequeue = ehci_urb_dequeue,
	.endpoint_disable = ehci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number = ehci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data = ehci_hub_status_data,
	.hub_control = ehci_hub_control,
#if defined(CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION)
	.match_companion	= ehci_ayu_match_companion,
	.hub_control_pp_by_companion = ehci_hub_control_pp_by_companion,
#endif /* CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION */
#ifdef	CONFIG_PM
	.bus_suspend = ehci_bus_suspend,
	.bus_resume = ehci_bus_resume,
#endif
};

/*-------------------------------------------------------------------------*/

static int ehci_hcd_ayu_drv_probe(struct platform_device *pdev)
{
	struct usb_hcd *hcd = NULL;
	int ret;

	pr_debug("In ehci_hcd_ayu_drv_probe\n");

	if (usb_disabled())
		return -ENODEV;

	ret = usb_ehci_ayu_probe(&ehci_ayu_hc_driver, &hcd, pdev);
	return ret;
}

static int ehci_hcd_ayu_drv_remove(struct platform_device *pdev)
{
	struct usb_hcd *hcd = platform_get_drvdata(pdev);

	usb_ehci_ayu_remove(hcd, pdev);
	return 0;
}

#ifdef CONFIG_PM
static int ehci_hcd_ayu_drv_suspend(struct platform_device *dev, pm_message_t state)
{
	struct usb_hcd *hcd = platform_get_drvdata(dev);
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	unsigned long		flags;
	int			rc = 0;

	if (time_before(jiffies, ehci->next_statechange))
		msleep(10);

	spin_lock_irqsave (&ehci->lock, flags);
	if (hcd->state != HC_STATE_SUSPENDED) {
		rc = -EINVAL;
		goto bail;
	}

	dev->dev.power.power_state = PMSG_SUSPEND;

	ehci_writel(ehci, 0, &ehci->regs->intr_enable);
	(void)ehci_readl(ehci, &ehci->regs->intr_enable);

	/* make sure snapshot being resumed re-enumerates everything */
	if (state.event == PM_EVENT_PRETHAW) {
		ehci_halt(ehci);
		ehci_reset(ehci);
	}

	clear_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
bail:
	spin_unlock_irqrestore (&ehci->lock, flags);

	return rc;
}
static int ehci_hcd_ayu_drv_resume(struct platform_device *dev)
{
	struct usb_hcd *hcd = platform_get_drvdata(dev);
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);

	if (time_before(jiffies, ehci->next_statechange))
		msleep(100);

	/* Mark hardware accessible again as we are out of D3 state by now */
	set_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);

	dev->dev.power.power_state = PMSG_ON;

	/* If CF is still set, we maintained PCI Vaux power.
	 * Just undo the effect of ehci_hcd_ayu_drv_suspend().
	 */
	if (ehci_readl(ehci, &ehci->regs->configured_flag) == FLAG_CF) {
		int	mask = INTR_MASK;

		if (!device_may_wakeup(&hcd->self.root_hub->dev))
			mask &= ~STS_PCD;
		ehci_writel(ehci, mask, &ehci->regs->intr_enable);
		ehci_readl(ehci, &ehci->regs->intr_enable);
		return 0;
	}

	ehci_dbg(ehci, "lost power, restarting\n");
	usb_root_hub_lost_power(hcd->self.root_hub);

	ehci_writel(ehci, FLAG_CF, &ehci->regs->configured_flag);

	/* here we "know" root ports should always stay powered */
	ehci_port_power(ehci, 1);

	return 0;
}
#endif

MODULE_ALIAS("ayu-ehci");
static struct platform_driver ehci_hcd_ayu_driver = {
	.probe = ehci_hcd_ayu_drv_probe,
	.remove = ehci_hcd_ayu_drv_remove,
	.shutdown = usb_hcd_platform_shutdown,
#ifdef CONFIG_PM
	.suspend      = ehci_hcd_ayu_drv_suspend,
	.resume       = ehci_hcd_ayu_drv_resume,
#endif
	.driver = {
		.name = "ayu-ehci",
		.bus = &platform_bus_type
	}
};

#if defined(CONFIG_SNSC_USB_CUSTOM_REG_ACCESS)
#if defined(CONFIG_AYU)
static inline unsigned int ehci_readl(const struct ehci_hcd *ehci,
				      __u32 __iomem * regs)
{
#if defined(CONFIG_USB_AYU_HCD_BANDLIMITING_WORKAROUND)
	unsigned long flags;
	unsigned int ret;
	u32 enabled;

	if (regs == &ehci->regs->status) {
		local_irq_save(flags);
		enabled = ayu_in32(AYU_USBHCB_LMTCTRL1);
		enabled &= AYU_USBHCB_LMTCTRL1_ENABLE;
		if (enabled) {
			ayu_out32(AYU_USBHCB_LMTCTRL1, 0);
		}
		ret = swab32(readl(regs));
		if (enabled) {
			ayu_out32(AYU_USBHCB_LMTCTRL1,
				  AYU_USBHCB_LMTCTRL1_ENABLE);
		}
		local_irq_restore(flags);
		return ret;
	}
#endif
	return swab32(readl(regs));
}

static inline void ehci_writel(const struct ehci_hcd *ehci,
			       const unsigned int val, __u32 __iomem *regs)
{
	writel(swab32(val), regs);
	ayu_iosync_cbus();
}
#elif defined(CONFIG_ARCH_BST)
static inline unsigned int ehci_readl(const struct ehci_hcd *ehci,
				      __u32 __iomem * regs)
{
#if defined(CONFIG_USB_AYU_HCD_BANDLIMITING_WORKAROUND)
	struct usb_hcd *hcd;
	struct usb_bus *bus;
	struct platform_device *pdev;
	unsigned long flags;
	unsigned int ret;
	u32 enabled;
	__u32 __iomem * addr;

	if (regs == &ehci->regs->status) {
		hcd = ehci_to_hcd((struct ehci_hcd *)ehci);
		bus = hcd_to_bus(hcd);
		pdev = to_platform_device(bus->controller);

		BUG_ON((pdev->id < 0) || (pdev->id >= BST_USBHCB_NUM));

		addr = IO_ADDRESS(BST_USB_BASE(pdev->id)) + BST_USBHCB_LMTCTRL1;

		spin_lock_irqsave(&ehci_ayu_lock[pdev->id], flags);
		enabled = bst_in32(addr) & BST_USBHCB_LMTCTRL1_ENABLE;
		if (enabled) {
			bst_out32(addr, 0);
		}
		ret = bst_raw_in32le(regs);
		if (enabled) {
			bst_out32(addr, BST_USBHCB_LMTCTRL1_ENABLE);
		}
		spin_unlock_irqrestore(&ehci_ayu_lock[pdev->id], flags);
		return ret;
	}
#endif
	return bst_raw_in32le(regs);
}

static inline void ehci_writel(const struct ehci_hcd *ehci,
			       const unsigned int val, __u32 __iomem *regs)
{
	bst_raw_out32le(regs, val);
	bst_iosync_cbus();
}
#endif
#endif
