/*
 * OHCI HCD (Host Controller Driver) for USB.
 *
 * (C) Copyright 1999 Roman Weissgaerber <weissg@vienna.at>
 * (C) Copyright 2000-2002 David Brownell <dbrownell@users.sourceforge.net>
 * (C) Copyright 2002 Hewlett-Packard Company
 * (C) Copyright 2008 Toshiba Corporation
 *
 * Bus Glue for AYU
 *
 * Based on "ohci-sa1111.c" and "ohci-au1xxx.c"
 *
 * This file is licenced under the GPL.
 */

#if defined(CONFIG_AYU)
#include <asm/ayu/ayu_reg.h>
#endif
#include <linux/platform_device.h>

extern int usb_disabled(void);

/*-------------------------------------------------------------------------*/

/* configure so an HC device and id are always provided */
/* always called with process context; sleeping is OK */


/**
 * usb_hcd_ayu_probe - initialize AYU based HCDs
 * Context: !in_interrupt()
 *
 * Allocates basic resources for this USB host controller, and
 * then invokes the start() method for the HCD associated with it
 * through the hotplug entry's driver_data.
 *
 * Store this function in the HCD's struct pci_driver as probe().
 */
static int usb_hcd_ayu_probe (const struct hc_driver *driver,
			  struct platform_device *dev)
{
	struct usb_hcd *hcd;
	int retval;

	hcd = usb_create_hcd (driver, &dev->dev, "ayu");
	if (!hcd)
		return -ENOMEM;
	hcd->rsrc_start = dev->resource[0].start;
	hcd->rsrc_len = dev->resource[0].end - dev->resource[0].start + 1;

	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len, hcd_name)) {
		dbg("request_mem_region failed");
		retval = -EBUSY;
		goto err1;
	}

	hcd->regs = ioremap(hcd->rsrc_start, hcd->rsrc_len);
	if (!hcd->regs) {
		pr_debug("ioremap failed\n");
		retval = -ENOMEM;
		goto err2;
	}

	ohci_hcd_init(hcd_to_ohci(hcd));

	retval = usb_add_hcd(hcd, dev->resource[1].start, IRQF_SHARED);
	if (retval == 0)
		return retval;

	iounmap(hcd->regs);
 err2:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
 err1:
	usb_put_hcd(hcd);
	return retval;
}


/* may be called without controller electrically present */
/* may be called with controller, bus, and devices active */

/**
 * usb_hcd_ayu_remove - shutdown processing for AYU HCDs
 * @dev: USB Host Controller being removed
 * Context: !in_interrupt()
 *
 * Reverses the effect of usb_hcd_ayu_probe(), first invoking
 * the HCD's stop() method.  It is always called from a thread
 * context, normally "rmmod", "apmd", or something similar.
 *
 */
static void usb_hcd_ayu_remove (struct usb_hcd *hcd, struct platform_device *dev)
{
	usb_remove_hcd(hcd);
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
}

/*-------------------------------------------------------------------------*/

static int __devinit
ohci_ayu_start (struct usb_hcd *hcd)
{
	struct ohci_hcd	*ohci = hcd_to_ohci (hcd);
	int		ret;
#if defined(CONFIG_USB_AYU_HCD_OHCI_PP_CONTROL)
	u32		temp;
#endif

	if ((ret = ohci_init(ohci)) < 0)
		return ret;

#if defined(CONFIG_USB_AYU_HCD_OHCI_PP_CONTROL)
	temp = roothub_a(ohci);
	temp &= ~RH_A_NPS;  /* Ports are power switched */
	temp |= RH_A_PSM;   /* each port is powered individually */
	temp |= RH_A_OCPM;  /* oc status is reported on a per-port basis */
	temp &= ~RH_A_NOCP; /* over current protection supported */
	ohci_writel (ohci, temp, &ohci->regs->roothub.a);
#endif

	if ((ret = ohci_run (ohci)) < 0) {
		err ("can't start %s", hcd->self.bus_name);
		ohci_stop (hcd);
		return ret;
	}
	return 0;
}

/*-------------------------------------------------------------------------*/

#if defined(CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION)
#if defined(CONFIG_AYU)
static int ohci_ayu_match_companion(struct usb_hcd *hcd, struct usb_hcd *chcd)
{
	return (strcmp(chcd->product_desc, "AYU EHCI"))? 0: 1;
}
#elif defined(CONFIG_ARCH_BST)
static int ohci_ayu_match_companion(struct usb_hcd *hcd, struct usb_hcd *chcd)
{
	struct usb_bus *bus, *cbus;
	struct platform_device *pdev, *cpdev;

	bus = hcd_to_bus(hcd);
	cbus = hcd_to_bus(chcd);
	pdev = to_platform_device(bus->controller);
	cpdev = to_platform_device(cbus->controller);

	if ((pdev->id != cpdev->id) || strcmp(chcd->product_desc, "AYU EHCI"))
		return 0;
	else
		return 1;
}
#endif
#endif /* CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION */

static const struct hc_driver ohci_ayu_hc_driver = {
	.description =		hcd_name,
	.product_desc =		"AYU OHCI",
	.hcd_priv_size =	sizeof(struct ohci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq =			ohci_irq,
	.flags =		HCD_USB11 | HCD_MEMORY,

	/*
	 * basic lifecycle operations
	 */
	.start =		ohci_ayu_start,
	.stop =			ohci_stop,
	.shutdown =		ohci_shutdown,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue =		ohci_urb_enqueue,
	.urb_dequeue =		ohci_urb_dequeue,
	.endpoint_disable =	ohci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number =	ohci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data =	ohci_hub_status_data,
	.hub_control =		ohci_hub_control,
#if defined(CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION)
	.match_companion	= ohci_ayu_match_companion,
	.hub_control_pp_by_companion =	ohci_hub_control_pp_by_companion,
#endif /* CONFIG_SNSC_USB_RH_PORT_POWER_CONTROL_WITH_COMPANION */
#ifdef	CONFIG_PM
	.bus_suspend =		ohci_bus_suspend,
	.bus_resume =		ohci_bus_resume,
#endif
	.start_port_reset =	ohci_start_port_reset,
};

/*-------------------------------------------------------------------------*/

static int ohci_hcd_ayu_drv_probe(struct platform_device *pdev)
{
	int ret;

	if (usb_disabled())
		return -ENODEV;

	ret = usb_hcd_ayu_probe(&ohci_ayu_hc_driver, pdev);
	return ret;
}

static int ohci_hcd_ayu_drv_remove(struct platform_device *pdev)
{
	struct usb_hcd *hcd = platform_get_drvdata(pdev);

	usb_hcd_ayu_remove(hcd, pdev);
	return 0;
}

#ifdef CONFIG_PM
static int ohci_hcd_ayu_drv_suspend(struct platform_device *dev, pm_message_t state)
{
	struct usb_hcd *hcd = platform_get_drvdata(dev);
	struct ohci_hcd *ohci = hcd_to_ohci(hcd);
	unsigned long	flags;
	int		rc = 0;

	spin_lock_irqsave (&ohci->lock, flags);
	if (hcd->state != HC_STATE_SUSPENDED) {
		rc = -EINVAL;
		goto bail;
	}

	dev->dev.power.power_state = PMSG_SUSPEND;

	ohci_writel(ohci, OHCI_INTR_MIE, &ohci->regs->intrdisable);
	(void)ohci_readl(ohci, &ohci->regs->intrdisable);

	/* make sure snapshot being resumed re-enumerates everything */
	if (state.event == PM_EVENT_PRETHAW)
		ohci_usb_reset(ohci);

	clear_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
bail:
	spin_unlock_irqrestore (&ohci->lock, flags);

	return rc;
}
static int ohci_hcd_ayu_drv_resume(struct platform_device *dev)
{
	struct usb_hcd *hcd = platform_get_drvdata(dev);
	struct ohci_hcd *ohci = hcd_to_ohci(hcd);
	int		port;

	set_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
	dev->dev.power.power_state = PMSG_ON;

	/* Here we "know" root ports should always stay powered */
	ohci_dbg(ohci, "powerup ports\n");
	for (port = 0; port < ohci->num_ports; port++)
		ohci_writel(ohci, RH_PS_PPS,
			    &ohci->regs->roothub.portstatus[port]);

	/* Flush those writes */
	ohci_readl(ohci, &ohci->regs->control);
	msleep(20);

	return 0;
}
#endif

static struct platform_driver ohci_hcd_ayu_driver = {
	.probe		= ohci_hcd_ayu_drv_probe,
	.remove		= ohci_hcd_ayu_drv_remove,
	.shutdown	= usb_hcd_platform_shutdown,
#ifdef CONFIG_PM
	.suspend	= ohci_hcd_ayu_drv_suspend,
	.resume		= ohci_hcd_ayu_drv_resume,
#endif
	.driver = {
		.name	= "ayu-ohci",
		.owner	= THIS_MODULE,
	},
};

#if defined(CONFIG_SNSC_USB_CUSTOM_REG_ACCESS)
#if defined(CONFIG_AYU)
static inline unsigned int _ohci_readl (const struct ohci_hcd *ohci,
					__hc32 __iomem * regs)
{
	return swab32(readl(regs));
}

static inline void _ohci_writel (const struct ohci_hcd *ohci,
				 const unsigned int val, __hc32 __iomem *regs)
{
	writel(swab32(val), regs);
	ayu_iosync_cbus();
}
#elif defined(CONFIG_ARCH_BST)
static inline unsigned int _ohci_readl (const struct ohci_hcd *ohci,
					__hc32 __iomem * regs)
{
	return bst_raw_in32le(regs);
}

static inline void _ohci_writel (const struct ohci_hcd *ohci,
				 const unsigned int val, __hc32 __iomem *regs)
{
	bst_raw_out32le(regs, val);
	bst_iosync_cbus();
}
#endif
#endif
