/*
 *  Driver for USB Mass Storage unusual devices
 *
 *  Copyright 2006,2007 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */


static void usb_stor_blocking_completion(struct urb *urb);
static int interpret_urb_result(struct us_data *us, unsigned int pipe,
			unsigned int length, int result, unsigned int partial);



/*
 * MSGC-US10 will be choked up if transfer length
 * is 255 bytes long for REQUEST_SENSE. 18 bytes
 * seem to be safer.
 */
static void usb_stor_fix_request_sense_size18(struct scsi_cmnd *srb,
				       struct us_data *us)
{
	if (!(us->flags_exp & US_FL_SNSC_REQUEST_SENSE_SIZE_18))
		goto end;

	if (srb->cmnd[0] != REQUEST_SENSE)
		goto end;

	if (18 < srb->cmnd[4]) {
		US_DEBUGP("Fixed REQUEST_SENSE size %d -> 18\n", srb->cmnd[4]);
		srb->cmnd[4] = 18;
	}

 end:
	return ;
}


/*
 * DSC-U40/U50 requires pause after TUR
 */
static void usb_stor_fix_wait_after_tur(struct scsi_cmnd *srb,
					struct us_data *us)
{
	signed long timeout;

	if (!(us->flags_exp & US_FL_SNSC_WAIT_AFTER_TUR))
		goto end;

	if (srb->cmnd[0] != TEST_UNIT_READY)
		goto end;

	timeout = msecs_to_jiffies(10);
	set_current_state(TASK_UNINTERRUPTIBLE);
	while ((timeout = schedule_timeout(timeout)) != 0);
	US_DEBUGP("Wait after TEST_UNIT_READY\n");

 end:
	return ;
}

/*
 * ignore sense info.  Some devices return
 * error sense info even if the previous command
 * was done successfully.
 */
static void usb_stor_fix_ignore_inquiry_sense(struct scsi_cmnd *srb,
					struct us_data *us)
{
	if (!(us->flags_exp & US_FL_SNSC_IGNORE_INQUIRY_SENSE))
		goto end;

	if (srb->cmnd[0] == INQUIRY) {
		US_DEBUGP("Ignore INQUIRY sense\n");
		srb->sense_buffer[0] = 0;
		srb->result = SAM_STAT_GOOD;
	}

 end:
	return ;
}

/*
 * some devices could not report media write protect info
 * in MODE_SENSE.
 * Simulate it by issuing WRITE command and the command
 * successed.
 * return non zero, if some error occured
 */
#define US_FIX_WP_REQ_BUF_MAX	4
int usb_stor_fix_write_protect(struct scsi_cmnd * srb, struct us_data *us)
{
	void* org_request_buffer;
	int org_sg;
	int org_request_bufflen;
	int org_underflow;
	unsigned char org_cmnd[MAX_COMMAND_SIZE];
	unsigned char org_sc_data_direction, org_cmd_len;
	int writeprotected=0;
	const int req_buf_max = US_FIX_WP_REQ_BUF_MAX;
	unsigned char* req_buf_ptr[US_FIX_WP_REQ_BUF_MAX];

	if (!(us->flags_exp & US_FL_SNSC_FIXWRITEPROTECT))
		goto end;

	if (srb->cmnd[0] != MODE_SENSE && srb->cmnd[0] != MODE_SENSE_10)
		goto end;

	US_DEBUGP("Issuing READ_10 & WRITE_10 on sector 14 for detecting "
		"WriteProtect\n");
	/* save the original command */
	memcpy(org_cmnd, srb->cmnd, MAX_COMMAND_SIZE);
	org_request_buffer = scsi_sglist(srb);
	org_request_bufflen = scsi_bufflen(srb);
	org_underflow = srb->underflow;
	org_sg = scsi_sg_count(srb);
	org_sc_data_direction = srb->sc_data_direction;
	org_cmd_len = srb->cmd_len;

	if (!(srb->sdb.table.sgl = kmalloc(512, GFP_KERNEL))) {
 		srb->result = DID_ERROR << 16;
		srb->sdb.table.sgl = org_request_buffer;
		return 1;
	}

	/* do command READ_10, sector14, count1 */
	memset(srb->cmnd, 0, MAX_COMMAND_SIZE);
	srb->cmnd[0] = READ_10;
	srb->cmnd[5] = 14;
	srb->cmnd[8] = 1;
	srb->sdb.length = 512;
	srb->underflow = 1 << 9; /* 512 = 1 sector */
	srb->sdb.table.nents = 0;

	srb->sc_data_direction = DMA_FROM_DEVICE;
	if ((us->subclass != US_SC_SCSI)) {
		srb->cmd_len = 12;
	} else {
		srb->cmd_len = 10;
	}

	US_DEBUGP("Issuing READ_10 ...\n");
	usb_stor_invoke_transport(srb, us);

	/* check result of READ */
	if (srb->result != GOOD << 1) {
		US_DEBUGP("READ_10 sector15 count1 failed.\n");
		goto DONE;
	}

	/* do command WRITE_10, sector14, count1 */
	srb->cmnd[0] = WRITE_10;
	srb->sc_data_direction = DMA_TO_DEVICE;
	US_DEBUGP("Issuing WRITE_10 ...\n");
	usb_stor_invoke_transport(srb, us);

	US_DEBUGP("-- WRITE_10 Result %x code: 0x%x, "
		  "key: 0x%x, ASC: 0x%x, ASCQ: 0x%x\n", srb->result,
		  srb->sense_buffer[0], srb->sense_buffer[2] & 0xf,
		  srb->sense_buffer[12], srb->sense_buffer[13]);

	if ((srb->result != GOOD << 1) &&
	    (srb->sense_buffer[2] & 0xf) == 0x07) {
		writeprotected = 1;
		srb->result = GOOD << 1;
	}

	US_DEBUGP("-- Write Protect is %s (by WRITE_10)\n",
		  writeprotected ? "On " : "Off");

	/* we're done here, let's clean up */
 DONE:	kfree(srb->sdb.table.sgl);
	srb->sdb.table.sgl = org_request_buffer;
	srb->sdb.length = org_request_bufflen;
	srb->underflow = org_underflow;
	srb->sdb.table.nents = org_sg;
	srb->sc_data_direction = org_sc_data_direction;
	srb->cmd_len = org_cmd_len;
	memcpy(srb->cmnd, org_cmnd, MAX_COMMAND_SIZE);

	/* check result of READ_10 & WRITE_10 */
	if (srb->result != GOOD << 1) {
		US_DEBUGP("Detecting WriteProtect failed.\n");
		return 1;
	}

	/*
	 * Zeroing srb->result is not sufficient to pretend
	 * 'MODE_SENSE' itself is OK
	 */
	memset(srb->sense_buffer, 0, sizeof(srb->sense_buffer));

	if (srb->sdb.table.nents) {
		struct scatterlist *sg = scsi_sglist(srb);
		int j, c;
		for (j = 0, c =0; c < req_buf_max; c++) {
			req_buf_ptr[c] =
				sg_virt(sg) + j;
			++j;
			if (j >= sg->length) {
				++sg;
				j = 0;
			}
		}

	} else {
		int i;
		for (i = 0; i < req_buf_max; i++)
			req_buf_ptr[i] = (unsigned char*)(srb->sdb.table.sgl + i);
	}

	if (writeprotected) {
		if (srb->cmnd[0] == MODE_SENSE)
			*(req_buf_ptr[2]) |= 0x80;
		else
			*(req_buf_ptr[3]) |= 0x80;
	} else {
		if (srb->cmnd[0] == MODE_SENSE)
			*(req_buf_ptr[2]) &= ~0x80;
		else
			*(req_buf_ptr[3]) &= ~0x80;
	}


 end:
	return 0;
} /* usb_stor_fix_write_protect */

/*
 * some devices work fine if TEST_UNIT_READY was issued
 * before every command !?
 * return non zero, if some error occured
 */
static int usb_stor_fix_tur_lover(struct scsi_cmnd *srb, struct us_data *us)
{
	void* org_request_buffer;
	int org_sg;
	int org_request_bufflen;
	int org_underflow;
	unsigned char org_cmnd[MAX_COMMAND_SIZE];
	unsigned char org_cmd_len;


 	if (!(us->flags_exp & US_FL_SNSC_TUR_LOVER))
		goto end;

	if (srb->cmnd[0] == REQUEST_SENSE ||
	    srb->cmnd[0] == TEST_UNIT_READY)
		goto end;

	/* save the original command */
	memcpy(org_cmnd, srb->cmnd, MAX_COMMAND_SIZE);
	org_request_buffer = scsi_sglist(srb);
	org_request_bufflen = scsi_bufflen(srb);
	org_underflow = srb->underflow;
	org_sg = scsi_sg_count(srb);
	org_cmd_len = srb->cmd_len;

	srb->sdb.table.sgl = NULL;
	srb->sdb.length = 0;
	srb->underflow = 0;
	srb->sdb.table.nents = 0;
	if ((us->subclass != US_SC_SCSI)) {
		srb->cmd_len = 12;
	} else {
		srb->cmd_len = 6;
	}

	/* do TEST_UNIT_READY */
	memset(srb->cmnd, 0, MAX_COMMAND_SIZE);
	usb_stor_invoke_transport(srb, us);

	/* we're done here, let's clean up */
	srb->sdb.table.sgl = org_request_buffer;
	srb->sdb.length = org_request_bufflen;
	srb->underflow = org_underflow;
	srb->sdb.table.nents = org_sg;
	srb->cmd_len = org_cmd_len;
	memcpy(srb->cmnd, org_cmnd, MAX_COMMAND_SIZE);

	/* check result of TEST_UNIT_READY */
	if (srb->result != SAM_STAT_GOOD) {
		US_DEBUGP("TEST_UNIT_READY failed. result=%x "
			  "Command=[%02x]\n", srb->result, org_cmnd[0]);
		return 1;
	}

 end:
	return 0;
} /* usb_stor_fix_tur_lover */

/*
 * some devices need special sector alignments
 * return non zero, if some error occured
 */
static int usb_stor_fix_sector_alignment(struct scsi_cmnd *srb,
					struct us_data *us)
{
	void* org_request_buffer;
	int org_sg;
	int org_request_bufflen;
	int org_underflow;
	unsigned char org_cmnd[MAX_COMMAND_SIZE];
	unsigned int block, count, left, transferred;
	struct scatterlist* sg = 0;
	unsigned sector_size = srb->device->sector_size;

	if (!(us->flags_exp & US_FL_SNSC_SECTORLIMIT))
		goto end;

	if (!(srb->cmnd[0] == READ_10 || srb->cmnd[0] == WRITE_10))
		goto end;

	/* save the original command */
	memcpy(org_cmnd, srb->cmnd, MAX_COMMAND_SIZE);
	org_request_buffer = scsi_sglist(srb);
	org_request_bufflen = scsi_bufflen(srb);
	org_underflow = srb->underflow;
	org_sg = scsi_sg_count(srb);

	/* malloc scattterlist memory */
	if (org_sg) {
		sg = kmalloc(sizeof(struct scatterlist) * org_sg, GFP_KERNEL);
		if (!sg) {
			srb->result = DID_ERROR << 16;
			return 1;
		}
	}

	block = (srb->cmnd[2] << 24) | (srb->cmnd[3] << 16) |
		(srb->cmnd[4] <<  8) |  srb->cmnd[5];
	left = (srb->cmnd[7] << 8) | srb->cmnd[8];
	transferred = 0;

	US_DEBUGP("Request: block %x total %x, %d bytes\n",
		  block, left, left * sector_size);
	US_DEBUGP("sector align=%d, max=%d\n",
		  us->sectoralign, us->sectormax);

	do {
		/* calculate number of sectors to write/read */
		count = left;
		if (us->sectoralign) {
			if (block % us->sectoralign == 0) {
				if (left > us->sectoralign)
					count = left - left % us->sectoralign;
				else
					count = left;
			} else {
				count = us->sectoralign -
				    block % us->sectoralign;
				if (count > left)
					count = left;
			}
		}
		if (us->sectormax && count > us->sectormax)
			count = us->sectormax;

		srb->cmnd[2] = (unsigned char) (block >> 24) & 0xff;
		srb->cmnd[3] = (unsigned char) (block >> 16) & 0xff;
		srb->cmnd[4] = (unsigned char) (block >>  8) & 0xff;
		srb->cmnd[5] = (unsigned char) (block      ) & 0xff;
		srb->cmnd[7] = (unsigned char) (count >> 8) & 0xff;
		srb->cmnd[8] = (unsigned char) (count     ) & 0xff;

		/* set scatter-gather list */
		if (sg) {
			int i,c,bsum;
			int len, reqlen;
			struct scatterlist* sg_org = (struct scatterlist*)
				org_request_buffer;

			for (i=0,bsum=0; bsum < transferred; i++)
				bsum += sg_org[i].length;

			if ( transferred < bsum ) {
				int q = bsum - transferred;
				sg[0].page_link = sg_org[i-1].page_link;
				sg[0].length  = q;
				sg[0].offset  = sg_org[i-1].offset
				    + sg_org[i-1].length - q;
			} else {	/* transferred == bsum */
				sg[0].page_link = sg_org[i].page_link;
				sg[0].length  = sg_org[i].length;
				sg[0].offset  = sg_org[i].offset;
				i++;
			}
			len = sg[0].length;
			reqlen = count * sector_size;
			c = 1;

			for ( ; len < reqlen; c++,i++ ) {
				sg[c].page_link = sg_org[i].page_link;
				sg[c].length  = sg_org[i].length;
				sg[c].offset  = sg_org[i].offset;
				len += sg[c].length;
			}

			if ( len > reqlen )
				sg[c-1].length -= len - reqlen;
			srb->sdb.table.nents = c;
			srb->sdb.table.sgl = sg;
			srb->sdb.length = reqlen;

			US_DEBUGP("++ split to: block %x count %x, %d bytes\n",
				  block, count, reqlen);
		} else {
			srb->sdb.table.sgl = org_request_buffer + transferred;
			srb->sdb.length = count * sector_size;
		}
		/* As sd_init_command(), "underflow" is the minimum
		 * transfered data size that is calcurated with the
		 * minimum sector_size(=512) in linux. */
		srb->underflow = count << 9;

		/* do transport */
		usb_stor_invoke_transport(srb, us);

		/* if error occurred, break */
		if (srb->result) break;

		block += count;
		left -= count;
		transferred += count * sector_size;
	} while( left > 0 );

	/* we're done here, let's clean up */
	if (sg) kfree(sg);
	srb->sdb.table.sgl = org_request_buffer;
	srb->sdb.length = org_request_bufflen;
	srb->underflow = org_underflow;
	srb->sdb.table.nents = org_sg;
	memcpy(srb->cmnd, org_cmnd, MAX_COMMAND_SIZE);

 end:
	return 0;
} /* usb_stor_fix_sector_alignment() */

/* US_FL_SNSC_COMBINE_SG_STAGECD */
static inline void usb_stor_bulk_transfer_buf_set(struct us_data *us,
			unsigned int pipe, void *buf, unsigned int length,
			unsigned int *act_len, struct urb *urb);
static int usb_stor_2msg_transfer(struct us_data *us);

/*
 * To submit data phase transfer in one urb, combine all scatter/gather
 * buffer in one, then pass it trasfer routine.
 * US30 would return unexpected short data if there is some time (over 1ms?)
 * between data phase transfers (IN/OUT).  One urb for whole data phase
 * results keeping clear it from the issue.
 */
int usb_stor_combine_sg_transfer(struct scsi_cmnd *srb,
					struct us_data* us)
{
	int i;
	void *org_request_buffer;
	int org_use_sg;
	int org_request_bufflen;
	struct scatterlist *sg;
	unsigned int transfer_amount;
	unsigned char *buffer;

	unsigned int pipe;
	int result;

	/* calculate how much we want to transfer */
	transfer_amount = scsi_bufflen(srb);

	/* save the original command */
	org_request_buffer = scsi_sglist(srb);
	org_request_bufflen = scsi_bufflen(srb);
	/* underflow is unchanged. */
	org_use_sg = scsi_sg_count(srb);

	/* allocate continuous memory */
	srb->sdb.table.sgl = kmalloc(transfer_amount, GFP_KERNEL);
	if (!srb->sdb.table.sgl) {
		srb->result = DID_ERROR << 16;
		srb->sdb.table.sgl = org_request_buffer;
		return USB_STOR_XFER_ERROR;
	}

	/* copy data to write */
	if (srb->cmnd[0] == WRITE_10) {
		sg = (struct scatterlist *) org_request_buffer;
		buffer = (unsigned char*)scsi_sglist(srb);
		for_each_sg (sg, sg, org_use_sg, i) {
			memcpy(buffer,
				sg_virt(sg),
				sg->length);
			buffer += sg->length;
		}
	}
	srb->sdb.length = transfer_amount;
	srb->sdb.table.nents = 0;

	/* do command */
	/* no scatter-gather, just make the request */
	pipe = srb->sc_data_direction == DMA_FROM_DEVICE ?
	    us->recv_bulk_pipe : us->send_bulk_pipe;

	if(us->flags_exp & US_FL_SNSC_COMBINE_SG_STAGECD){
		/* now, US_FL_SNSC_COMBINE_SG_STAGECD supports
		 * Bulk Only Transport devices only. (not support CBI,CB) */
		if (us->protocol == US_PR_BULK) {
			/* Set DATA STAGE urb */
			usb_stor_bulk_transfer_buf_set(us,
						       pipe,
						       scsi_sglist(srb),
						       transfer_amount,
						       NULL,
						       us->current_urb);
			/* Submit COMMAND and DATA STAGE urbs */
			result = usb_stor_2msg_transfer(us);
		} else {
			US_DEBUGP("US_FL_SNSC_COMBINE_SG_STAGECD:"
				  " not support PR=%02x."
				  " combined sg buffer only.", us->protocol);
			result = usb_stor_bulk_transfer_buf(us, pipe,
						    scsi_sglist(srb),
						    transfer_amount, NULL);
		}
	} else {
		result = usb_stor_bulk_transfer_buf(us, pipe,
						    scsi_sglist(srb),
						    transfer_amount, NULL);
	}

	/* copy data if read */
	if (srb->cmnd[0] == READ_10) {
		sg = (struct scatterlist *) org_request_buffer;
		buffer = (unsigned char*)scsi_sglist(srb);
		for_each_sg (sg, sg, org_use_sg, i) {
			memcpy(sg_virt(sg),
				buffer,
				sg->length);
			buffer += sg->length;
		}
	}

	/* we're done here, let's clean up */
	kfree(srb->sdb.table.sgl);

	srb->sdb.table.sgl = org_request_buffer;
	srb->sdb.length = org_request_bufflen;
	srb->sdb.table.nents = org_use_sg;

	return result;
}


/*
 * First READ_10 command to read MBR may return stall, so we retry
 * this command in order to mount this device.
 */
static void usb_stor_fix_retry_mbr_read(struct scsi_cmnd *srb,
					struct us_data *us)
{
	u32 lba;

 	if (!(us->flags_exp & US_FL_SNSC_RETRY_MBR_READ))
 		goto end;

	if (srb->cmnd[0] != READ_10)
		goto end;

	lba = ((srb->cmnd[2] << 24) | (srb->cmnd[3] << 16)
	       | (srb->cmnd[4] << 8) | srb->cmnd[5]);
	if (lba == 0) { /* MBR */
		msleep(1);
		US_DEBUGP("Retry READ_10 MBR %xh\n", lba);
		srb->result = DID_ERROR << 16;
	}
 end:
	return ;
}


/*
 * With some Bulk Only Transport devices (MSAC-US30),
 * they may need to combine COMMAND stage urb and DATA stage urb,
 * if CPU power is less than AVPPC.
 * DATA stage urb must be combined one buffer from scatter/gather buffers.
 * So US_FL_SNSC_COMBINE_SG_STAGECD includes US_FL_SNSC_COMBINE_SG.
 *
 * US_FL_SNSC_COMBINE_SG_STAGECD: the prep URB is COMMAND STAGE URB and
 * the current URB is DATA STAGE URB.
 */
/*
 * usb_stor_bulk_transfer_buf_set() only prepares the urb.
 * you need to submit urbs with usb_stor_2msg_transfer().
 *
 * Based on usb_stor_bulk_transfer_buf(), usb_stor_msg_common().
 */
static inline void usb_stor_bulk_transfer_buf_set(struct us_data *us,
			unsigned int pipe, void *buf, unsigned int length,
			unsigned int *act_len, struct urb *urb)
{
	US_DEBUGP("%s: xfer %u bytes\n", __FUNCTION__, length);

	/* fill the URB */
	usb_fill_bulk_urb(urb, us->pusb_dev, pipe, buf, length,
			  usb_stor_blocking_completion, NULL);

	/* fill the common fields in the URB */
	urb->actual_length = 0;
	urb->error_count = 0;
	urb->status = 0;

	/* we assume that if transfer_buffer isn't us->iobuf then it
	 * hasn't been mapped for DMA.  Yes, this is clunky, but it's
	 * easier than always having the caller tell us whether the
	 * transfer buffer has already been mapped. */
	urb->transfer_flags = URB_NO_SETUP_DMA_MAP;
	if (urb->transfer_buffer == us->iobuf)
		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
	urb->transfer_dma = us->iobuf_dma;
	urb->setup_dma = us->cr_dma;
}

static int usb_stor_2msg_transfer(struct us_data *us)
{
	struct completion urb_done_prep, urb_done_current;
	int status;
	int result;
	int timeout = 0;
	int timeleft_prep, timeleft_cur;

	US_DEBUGP("%s\n", __FUNCTION__);

#if defined(CONFIG_SNSC_USB_STORAGE_TIMEOUT_BULK_MSG_OPTION)
	/* timeout */
	timeout = (max((unsigned int)(us->prep_urb->transfer_buffer_length
			+ us->current_urb->transfer_buffer_length)
		       * HZ / BYTES_PER_SEC,
		       (unsigned int)MIN_TIMEOUT_BULK_TRANSFER) +
		   CONFIG_SNSC_USB_STORAGE_TIMEOUT_BULK_MSG * HZ);
#endif

	/* set up data structures for the wakeup system */
	init_completion(&urb_done_prep);
	us->prep_urb->context = &urb_done_prep;
	init_completion(&urb_done_current);
	us->current_urb->context = &urb_done_current;

	/* submit the prep and current URBs at once. */
	status = usb_submit_urb(us->prep_urb, GFP_NOIO);
	if (status) {
		/* something went wrong */
		result = status;
		goto end;
	}
	status = usb_submit_urb(us->current_urb, GFP_NOIO);
	if (status) {
		/* something went wrong */
		result = status;
		usb_unlink_urb(us->prep_urb);
		goto end;
	}

	/* since the URB has been submitted successfully, it's now okay
	 * to cancel it */
	set_bit(US_FLIDX_URB_ACTIVE, &us->dflags);

	/* did an abort/disconnect occur during the submission? */
	if (test_bit(US_FLIDX_ABORTING ,&us->dflags) ||
		test_bit(US_FLIDX_DISCONNECTING, &us->dflags)) {

		/* cancel the URB, if it hasn't been cancelled already */
		if (test_and_clear_bit(US_FLIDX_URB_ACTIVE, &us->dflags)) {
			US_DEBUGP("-- cancelling URB\n");
			usb_unlink_urb(us->prep_urb);
			usb_unlink_urb(us->current_urb);
		}
	}

	/* wait for the completion of the URB */
	timeleft_prep = wait_for_completion_interruptible_timeout(
			&urb_done_prep, timeout ? : MAX_SCHEDULE_TIMEOUT);
	timeleft_cur = wait_for_completion_interruptible_timeout(
			&urb_done_current, timeout ? : MAX_SCHEDULE_TIMEOUT);

	clear_bit(US_FLIDX_URB_ACTIVE, &us->dflags);

	if (timeleft_prep <= 0) {
		US_DEBUGP("%s -- cancelling URB(prep)\n",
			  timeleft_prep == 0 ? "Timeout" : "Signal");
		usb_kill_urb(us->current_urb);
	}
	if (timeleft_cur <= 0) {
		US_DEBUGP("%s -- cancelling URB(cur)\n",
			  timeleft_cur == 0 ? "Timeout" : "Signal");
		usb_kill_urb(us->prep_urb);
	}

	/* return the URB status */
	result = us->prep_urb->status;
	result = interpret_urb_result(us,
				      us->prep_urb->pipe,
				      us->prep_urb->transfer_buffer_length,
				      result,
				      us->prep_urb->actual_length);
	if (result != USB_STOR_XFER_GOOD) {
		return result;
	}
	result = us->current_urb->status;
end:
	return interpret_urb_result(us,
				    us->current_urb->pipe,
				    us->current_urb->transfer_buffer_length,
				    result,
				    us->current_urb->actual_length);
}



/*
 * a virtuad CD-ROM device looks like not to support READ_TOC command
 * with StartingTrack/SessionNumber.
 * use LBA always instead of MSF.
 */
#include <linux/cdrom.h>
static void usb_stor_read_toc_lba(struct scsi_cmnd *srb,
				  struct us_data* us)
{
 	if (!(us->flags_exp & US_FL_SNSC_READ_TOC_LBA))
 		goto end;

	if (srb->cmnd[0] != READ_TOC)
		goto end;

	/* MSF -> LBA */
	if (srb->cmnd[1] & CDROM_MSF) {
		srb->cmnd[1] &= ~02;
	}
end:
	return ;
}


/*
 * a virtuad CD-ROM device looks like to disallow to access the medium
 * information before the medium is ready.
 * repeat to issue TEST_UNIT_READY command until the medium is ready.
 */
static void usb_stor_wait_medium_ready(struct scsi_cmnd *srb,
				       struct us_data* us)
{
	int retries;

 	if (!(us->flags_exp & US_FL_SNSC_WAIT_MEDIUM_READY))
 		goto end;

	if (srb->cmnd[0] != TEST_UNIT_READY)
		goto end;

	for (retries = 0;
	     retries < CONFIG_SNSC_USB_STOR_WAIT_MEDIUM_READY_TIMES;
	     retries++) {
		if (srb->result != SAM_STAT_GOOD) {
			usb_stor_invoke_transport(srb, us);
		} else {
			break;
		}
	}
end:
	return ;
}
