/*
 * File Name       : drivers/video/emxx/emxx_mod_pmwlcd.c
 * Function        : PMWLCD Driver (Module Control)
 * Release Version : Ver 1.17
 * Release Date    : 2012.11.14
 *
 * Copyright (C) 2012 Sony Corporation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */


/********************************************************
 *  Definitions                                         *
 *******************************************************/
#define _DEBUG_PMWLCD 0x00 /* 00008421(bit) */
			   /* 0x01: debug function in
			    * 0x02: debug function out
			    * 0x04: debug polling
			    */


#define DEV_NAME "sublcd"

#define PMWLCD_ENABLE_SYSCOLOR 1
#define PMWLCD_ENABLE_SYSEDGE  1
#define PMWLCD_ENABLE_SYSCOEF  1
#define PMWLCD_ENABLE_SYSBUSSEL 1
#define PMWLCD_ENABLE_SYSEVENT 1

//#define DEBUG_SEND_ERROR

/********************************************************
 *  Macros                                              *
 *******************************************************/
#define printk_err(fmt, arg...) \
	do {                     \
		printk(KERN_ERR DEV_NAME "(%d): %s: " fmt, __LINE__, __func__, ## arg); \
	} while (0)

#define printk_wrn(fmt, arg...) \
	do {                     \
		printk(KERN_WARNING DEV_NAME "(%d): %s: " fmt, __LINE__, __func__, ## arg); \
	} while (0)

#define printk_info(fmt, arg...) \
	do {                      \
		printk(KERN_INFO DEV_NAME "(%d): %s: " fmt, __LINE__, __func__, ## arg); \
	} while (0)

#if _DEBUG_PMWLCD
#define printk_dbg(level, fmt, arg...) \
	do {                            \
		if (level > 0) \
			printk(KERN_DEBUG DEV_NAME "(%d): %s: " fmt, \
			 __LINE__,__func__, ## arg); \
	} while (0)
#else
#define printk_dbg(level, fmt, arg...) \
	;
#endif

#define flush_delayed_work(p)	flush_work(&((p)->work))

/********************************************************
 *  Include Files                                       *
 *******************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/err.h>

#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/kobject.h>
#include <linux/sysfs.h>
#include <linux/device.h>
#include <linux/mutex.h>

#include <linux/fb.h>

#include "lcdc.h"
#include "emxx_common.h"
#include "emxx_lcdhw.h"
#include "emxx_lcd.h"

#include "emxx_mod_pmwlcd.h"

/**
 * イベント待機用構造体
 */
struct lcdm_event_wait_t {
	atomic_t		on_event;
	struct list_head	list;
}lcdm_event_wait_t;

/**
 * 電源操作タイプ
 */
enum PWRCTRL_TYPE {
	PWRCTRL_TYPE_NOP,
	PWRCTRL_TYPE_ON_INIT,
	PWRCTRL_TYPE_ON,
	PWRCTRL_TYPE_OFF,
};

/**
 * Module-ID保持
 */
#if 0
static lcdm_param_module_t s_lcdm_param_module = {
	.moduleid	= LCDM_PARAM_MODULEID_QVGA,
	.devinfo	= LCDM_PARAM_DEVINFO_0,
};
#else
static lcdm_param_module_t s_lcdm_param_module = {
	.moduleid	= LCDM_PARAM_MODULEID_VGA,
	.devinfo	= LCDM_PARAM_DEVINFO_0,
};
#endif

/**
 * ユーザ調整機能パラメタ保持
 */
static lcdm_param_userfunc_t s_lcdm_param_userfunc = {
	.framerate	= LCDM_PARAM_FRAMERATE_60HZ,
	.flip_h		= LCDM_PARAM_NORMAL,
	.flip_v		= LCDM_PARAM_NORMAL,
};

/**
 * Device別制御機能パラメタ保持
 */
static lcdm_param_devfunc_t s_lcdm_param_devfunc = {
	.sidebyside	= LCDM_PARAM_OFF,
	.dimension	= LCDM_PARAM_OFF,
};

/**
 * バックライト輝度パラメタ保持
 */
static unsigned int lcdm_param_brightness = LCDM_PARAM_BRIGHTNESS_MIN;

/**
 * 電源ON/FF状態保持
 */
static unsigned int lcdm_status_power = LCDM_PARAM_POWER_OFF;

/**
 * HSYNC / VSYNC 入力タイミング定義(60Hz)
 */
static const lcdm_input_timing_t sc_lcdm_input_timing_60HZ[] = {
	/* harea  hpulse  hfrontp  hbackp  varea  vpulse  vfrontp  vbackp  voff-odd  voff-even */
	{     0,      0,      0,      0,      0,      0,      0,      0,      0,       0  },	//!< dummy
	{   320,    128,   1152,    244,    240,      3,      4,     18,      0,     858  },	//!< Module-ID=0x01 : 60Hz QVGA
//	{   640,    100,     96,    122,    480,      3,      7,     38,      0,       0  },	//!< Module-ID=0x02 : 60Hz VGA
	{   640,      3,    137,     81,    480,      1,     24,     21,      0,       0  },	//!< Module-ID=0x02 : 60Hz VGA
	{   852,      8,     28,     30,    480,      6,      6,      9,      0,       0  },	//!< Module-ID=0x03 : 60Hz WVGA
	{   960,    100,    128,     37,    540,      3,      8,      2,      0,       0  },	//!< Module-ID=0x04 : 60Hz QHD
	{  1280,     40,    110,    260,    720,      5,      5,     25,      0,       0  },	//!< Module-ID=0x05 : 60Hz WXGA
	{  1920,     44,     88,    192,   1080,      5,      4,     41,      0,       0  },	//!< Module-ID=0x06 : 60Hz FHD
};

/**
 * HSYNC / VSYNC 入力タイミング定義(50Hz)
 */
static const lcdm_input_timing_t sc_lcdm_input_timing_50HZ[] = {
	/* harea  hpulse  hfrontp  hbackp  varea  vpulse  vfrontp  vbackp  voff-odd  voff-even */
	{     0,      0,      0,      0,      0,      0,      0,      0,      0,       0 },	//!< dummy
	{   320,    128,   1164,    244,    240,      3,     54,     18,      0,     864 },	//!< Module-ID=0x01 : 50Hz QVGA
//	{   640,    100,    102,    122,    480,      3,    107,     38,      0,       0 },	//!< Module-ID=0x02 : 50Hz VGA
	{   640,      3,    102,    122,    480,      3,    107,     38,      0,       0 },	//!< Module-ID=0x02 : 50Hz VGA
	{   852,      8,     18,     30,    480,      6,     50,     70,      0,       0 },	//!< Module-ID=0x03 : 50Hz WVGA
	{   960,    100,    353,     37,    540,      3,      8,      2,      0,       0 },	//!< Module-ID=0x04 : 50Hz QHD
	{  1280,     40,    440,    260,    720,      5,      5,     25,      0,       0 },	//!< Module-ID=0x05 : 50Hz WXGA
	{  1920,     44,    528,    192,   1080,      5,      4,     41,      0,       0 },	//!< Module-ID=0x06 : 50Hz FHD
};

/**
 * SPI通信ポーリング処理用関数
 */
static void lcdm_do_polling_work(struct work_struct *work);
static void lcdm_do_chauge_output_work(struct work_struct *work);

/**
 * SPI通信ポーリング処理用変数
 */
static DECLARE_DELAYED_WORK(lcdm_polling_work, lcdm_do_polling_work);
static DECLARE_WORK(lcdm_chauge_output_work, lcdm_do_chauge_output_work);
static struct workqueue_struct *lcdm_workqueue;
static const unsigned int lcdm_polling_cycle = HZ;

/**
 * sysfs処理用関数(今回はクラスを直接使用)
 */
static int     lcdm_register_class(void);
static int     lcdm_unregister_class(void);
static ssize_t lcdm_class_show_moduleid(struct class *cls, char *buf);
static ssize_t lcdm_class_store_moduleid(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_flip(struct class *cls, char *buf);
static ssize_t lcdm_class_store_flip(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_framerate(struct class *cls, char *buf);
static ssize_t lcdm_class_store_framerate(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_brightness(struct class *cls, char *buf);
static ssize_t lcdm_class_store_brightness(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_power(struct class *cls, char *buf);
static ssize_t lcdm_class_store_power(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_refresh(struct class *cls, char *buf);
static ssize_t lcdm_class_store_refresh(struct class *cls, const char *buf, size_t count);
#ifdef PMWLCD_ENABLE_SYSCOLOR
static ssize_t lcdm_class_show_color(struct class *cls, char *buf);
static ssize_t lcdm_class_store_color(struct class *cls, const char *buf, size_t count);
#endif
#ifdef PMWLCD_ENABLE_SYSEDGE
static ssize_t lcdm_class_show_vedge(struct class *cls, char *buf);
static ssize_t lcdm_class_store_vedge(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_hedge(struct class *cls, char *buf);
static ssize_t lcdm_class_store_hedge(struct class *cls, const char *buf, size_t count);
#endif
#ifdef PMWLCD_ENABLE_SYSCOEF
static ssize_t lcdm_class_show_coef_y(struct class *cls, char *buf);
static ssize_t lcdm_class_store_coef_y(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_coef_u(struct class *cls, char *buf);
static ssize_t lcdm_class_store_coef_u(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_coef_v(struct class *cls, char *buf);
static ssize_t lcdm_class_store_coef_v(struct class *cls, const char *buf, size_t count);
#endif
#ifdef PMWLCD_ENABLE_SYSBUSSEL
static ssize_t lcdm_class_show_bussel(struct class *cls, char *buf);
static ssize_t lcdm_class_store_bussel(struct class *cls, const char *buf, size_t count);
#endif
#ifdef PMWLCD_ENABLE_SYSEVENT
static ssize_t lcdm_class_show_event(struct class *cls, char *buf);
static ssize_t lcdm_class_store_event(struct class *cls, const char *buf, size_t count);
static ssize_t lcdm_class_show_waitevent(struct class *cls, char *buf);
static ssize_t lcdm_class_store_waitevent(struct class *cls, const char *buf, size_t count);
#endif

/**
 * sysfs処理用クラス
 */
static struct class *lcdm_class;

/**
 * sysfs処理用クラス属性
 */
static struct class_attribute lcdm_class_attributes[] = {
	__ATTR(moduleid,   0444, lcdm_class_show_moduleid,   lcdm_class_store_moduleid),
	__ATTR(flip,       0666, lcdm_class_show_flip,       lcdm_class_store_flip),
	__ATTR(framerate,  0666, lcdm_class_show_framerate,  lcdm_class_store_framerate),
	__ATTR(brightness, 0666, lcdm_class_show_brightness, lcdm_class_store_brightness),
	__ATTR(power,      0666, lcdm_class_show_power,      lcdm_class_store_power),
	__ATTR(refresh,    0666, lcdm_class_show_refresh,    lcdm_class_store_refresh),
#ifdef PMWLCD_ENABLE_SYSCOLOR
	__ATTR(color,      0666, lcdm_class_show_color,      lcdm_class_store_color),
#endif
#ifdef PMWLCD_ENABLE_SYSEDGE
	__ATTR(vedge,      0666, lcdm_class_show_vedge,      lcdm_class_store_vedge),
	__ATTR(hedge,      0666, lcdm_class_show_hedge,      lcdm_class_store_hedge),
#endif
#ifdef PMWLCD_ENABLE_SYSCOEF
	__ATTR(coef_y,     0666, lcdm_class_show_coef_y,     lcdm_class_store_coef_y),
	__ATTR(coef_u,     0666, lcdm_class_show_coef_u,     lcdm_class_store_coef_u),
	__ATTR(coef_v,     0666, lcdm_class_show_coef_v,     lcdm_class_store_coef_v),
#endif
#ifdef PMWLCD_ENABLE_SYSBUSSEL
	__ATTR(bussel,     0666, lcdm_class_show_bussel,     lcdm_class_store_bussel),
#endif
#ifdef PMWLCD_ENABLE_SYSEVENT
	__ATTR(event,      0444, lcdm_class_show_event,      lcdm_class_store_event),
	__ATTR(waitevent,  0444, lcdm_class_show_waitevent,  lcdm_class_store_waitevent),
#endif
	__ATTR_NULL
};

/**
 * SubLCDパラメータ用read/writeロック
 */
static DEFINE_RWLOCK(lcdm_param_rwlock);

/**
 * 周期監視ログ表示用jiffies
 */
static unsigned long s_jiffies_work;

/**
 * イベント通知用変数
 */
static unsigned int s_lcdm_event_value = 0;
static DEFINE_RWLOCK(s_lcdm_event_rwlock);
static LIST_HEAD(s_lcdm_event_wait_list);
static wait_queue_head_t s_lcdm_event_wait_queue;

/* SPI レジスタ制御 */
static unsigned char	s_lcdm_read_type;
static unsigned char	s_lcdm_read_page;
static unsigned char	s_lcdm_read_addr;

#ifdef DEBUG_SEND_ERROR
static bool s_lcdm_debug_spi_force_error = false;
#endif


/*******************************************************************************
 * @brief	HSYNC / VSYNC デフォルト入力タイミング取得
 *
 * @param[out]	input_timing	入力タイミング格納用領域
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_timing_fix(lcdm_input_timing_t *input_timing)
{
	if (input_timing) {
#if 0
		// フレームレート=60Hz、Module-ID=1(QVGA)の値を固定で返す
		memcpy(input_timing, 
			&(sc_lcdm_input_timing_60HZ[LCDM_PARAM_MODULEID_QVGA]), sizeof(*input_timing));
#else
		memcpy(input_timing, &(sc_lcdm_input_timing_60HZ[LCDM_PARAM_MODULEID_VGA]), sizeof(*input_timing));
#endif
	}
}

/*******************************************************************************
 * @brief	HSYNC / VSYNC 入力タイミング取得
 *
 * @param[out]	input_timing	入力タイミング格納用領域
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_timing(lcdm_input_timing_t *input_timing)
{
	lcdm_param_module_t   wk_module;
	lcdm_param_userfunc_t wk_userfunc;
	
	// フレームレートおよびModule-ID取得
	lcdm_param_get_module(&wk_module);
	lcdm_param_get_userfunc(&wk_userfunc);
	
	if (input_timing) {
		if (LCDM_PARAM_FRAMERATE_60HZ == wk_userfunc.framerate) {
			// 60Hz
			memcpy(input_timing, 
				&(sc_lcdm_input_timing_60HZ[wk_module.moduleid]), sizeof(*input_timing));
		} else if (LCDM_PARAM_FRAMERATE_50HZ == wk_userfunc.framerate) {
			// 50Hz
			memcpy(input_timing, 
				&(sc_lcdm_input_timing_50HZ[wk_module.moduleid]), sizeof(*input_timing));
		} else {
			// 不明 - 60Hzとする
			memcpy(input_timing, 
				&(sc_lcdm_input_timing_60HZ[wk_module.moduleid]), sizeof(*input_timing));
			printk_wrn("unkonwn timing! - default 60Hz set.\n");
		}
	}
}

/*******************************************************************************
 * @brief	module取得
 *
 * @param[out]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_module(lcdm_param_module_t *param)
{
	unsigned long flags;
	read_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(param, &s_lcdm_param_module, sizeof(lcdm_param_module_t));
	read_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	module設定
 *
 * @param[in]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_set_module(lcdm_param_module_t *param)
{
	unsigned long flags;
	write_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(&s_lcdm_param_module, param, sizeof(lcdm_param_module_t));
	write_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	userfunc取得
 *
 * @param[out]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_userfunc(lcdm_param_userfunc_t *param)
{
	unsigned long flags;
	read_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(param, &s_lcdm_param_userfunc, sizeof(lcdm_param_userfunc_t));
	read_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	userfunc設定
 *
 * @param[in]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_set_userfunc(lcdm_param_userfunc_t *param)
{
	unsigned long flags;
	write_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(&s_lcdm_param_userfunc, param, sizeof(lcdm_param_userfunc_t));
	write_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	devfunc取得
 *
 * @param[out]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_devfunc(lcdm_param_devfunc_t *param)
{
	unsigned long flags;
	read_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(param, &s_lcdm_param_devfunc, sizeof(lcdm_param_devfunc_t));
	read_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	devfunc設定
 *
 * @param[in]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_set_devfunc(lcdm_param_devfunc_t *param)
{
	unsigned long flags;
	write_lock_irqsave(&lcdm_param_rwlock, flags);
	memcpy(&s_lcdm_param_devfunc, param, sizeof(lcdm_param_devfunc_t));
	write_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	brightness取得
 *
 * @param[out]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_get_brightness(unsigned int *param)
{
	unsigned long flags;
	read_lock_irqsave(&lcdm_param_rwlock, flags);
	*param = lcdm_param_brightness;
	read_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	brightness設定
 *
 * @param[in]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_param_set_brightness(unsigned int *param)
{
	unsigned long flags;
	write_lock_irqsave(&lcdm_param_rwlock, flags);
	lcdm_param_brightness = *param;
	write_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

/*******************************************************************************
 * @brief	power状態取得
 *
 * @param	なし
 *
 * @retval	LCDHW_STATUS_NOT_INITIALIZED	電源OFF
 * @retval	LCDHW_STATUS_INITIALIZED	電源ON
 *******************************************************************************/
unsigned int lcdm_status_get_power(void)
{
	unsigned long flags;
	unsigned int status = 0;
	
	read_lock_irqsave(&lcdm_param_rwlock, flags);
	status = lcdm_status_power;
	read_unlock_irqrestore(&lcdm_param_rwlock, flags);
	
	return status;
}

/*******************************************************************************
 * @brief	power状態設定
 *
 * @param[in]	*param	ユーザへ返すデータへのポインタ
 *
 * @return	なし
 *******************************************************************************/
void lcdm_status_set_power(unsigned int status)
{
	unsigned long flags;
	write_lock_irqsave(&lcdm_param_rwlock, flags);
	lcdm_status_power = status;
	write_unlock_irqrestore(&lcdm_param_rwlock, flags);
}

#ifdef PMWLCD_ENABLE_SYSEVENT
/*******************************************************************************
 * @brief	event値取得
 *
 * @param	なし
 *
 * @retval	event値
 *******************************************************************************/
static unsigned int lcdm_event_get_value(void)
{
	unsigned long flags;
	unsigned int event = 0;
	
	read_lock_irqsave(&s_lcdm_event_rwlock, flags);
	event = s_lcdm_event_value;
	read_unlock_irqrestore(&s_lcdm_event_rwlock, flags);
	
	return event;
}
#endif	/* PMWLCD_ENABLE_SYSEVENT */

/*******************************************************************************
 * @brief	event値設定
 *
 * @param[in]	event	イベントの種類
 * @param[in]	ctrl	イベントに対する設定値
 *
 * @return	なし
 *******************************************************************************/
void lcdm_event_set_value(unsigned int event, unsigned int ctrl)
{
	unsigned long flags;
	unsigned int old_event;
	struct lcdm_event_wait_t *p_wait;
	bool on_event = false;
	
	write_lock_irqsave(&s_lcdm_event_rwlock, flags);
	
	// 現在のevent値退避
	old_event = s_lcdm_event_value;
	
	// event値更新
	s_lcdm_event_value = (s_lcdm_event_value & ~event) | (ctrl * event);
	
	// event値に変化があった場合、イベントを通知する
	if (old_event != s_lcdm_event_value) {
		list_for_each_entry(p_wait, &s_lcdm_event_wait_list, list) {
			atomic_set(&p_wait->on_event, LCDM_ON);
		}
		on_event = true;
	}
	
	write_unlock_irqrestore(&s_lcdm_event_rwlock, flags);
	
	// イベント通知
	if (true == on_event) {
		wake_up_interruptible(&s_lcdm_event_wait_queue);
	}
}

#ifdef PMWLCD_ENABLE_SYSEVENT
/*******************************************************************************
 * @brief	event値更新待ち
 *
 * @param	なし
 *
 * @return	0:成功  0以外:失敗
 *******************************************************************************/
static int lcdm_event_wait_update(void)
{
	unsigned long flags;
	struct lcdm_event_wait_t *p_wait;
	int result = 0;
	
	// イベント待機用情報作成
	p_wait = kmalloc(sizeof(struct lcdm_event_wait_t), GFP_KERNEL);
	if (!p_wait) {
		result = -ENOMEM;
		return result;
	}
	INIT_LIST_HEAD(&p_wait->list);
	atomic_set(&p_wait->on_event, LCDM_OFF);
	
	// イベント待機用情報を待ちリストに追加
	write_lock_irqsave(&s_lcdm_event_rwlock, flags);
	list_add(&p_wait->list, &s_lcdm_event_wait_list);
	write_unlock_irqrestore(&s_lcdm_event_rwlock, flags);
	
	// イベント待機
	if (wait_event_interruptible(s_lcdm_event_wait_queue, atomic_read(&p_wait->on_event))) {
		result = -ERESTARTSYS;
	}
	
	// イベント待機用情報を待ちリストから削除
	write_lock_irqsave(&s_lcdm_event_rwlock, flags);
	list_del(&p_wait->list);
	write_unlock_irqrestore(&s_lcdm_event_rwlock, flags);
	
	// イベント待機用情報削除
	kfree(p_wait);
	
	return result;
}
#endif	/* PMWLCD_ENABLE_SYSEVENT */

/*******************************************************************************
 * @brief	get send error for debug
 * @param	none
 * @return	true: error
 *******************************************************************************/
bool lcdm_debug_get_spi_force_error(void)
{
#ifdef DEBUG_SEND_ERROR
	return s_lcdm_debug_spi_force_error;
#else
	return false;
#endif
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：moduleid読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_moduleid(struct class *cls, char *buf)
{
	lcdm_param_module_t param;
	
	lcdm_param_get_module(&param);
	
	return sprintf(buf, "%u\n", param.moduleid);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：moduleid書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_moduleid(struct class *cls, const char *buf, size_t count)
{
	return -EINVAL;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：flip読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_flip(struct class *cls, char *buf)
{
	lcdm_param_userfunc_t param;
	
	lcdm_param_get_userfunc(&param);
	
	return sprintf(buf, "%u,%u\n", param.flip_v, param.flip_h);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：flip書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_flip(struct class *cls, const char *buf, size_t count)
{
	unsigned long flip_h = 0, flip_v = 0;
	lcdm_param_userfunc_t param;
	int rc = 0;
	
	// パラメタチェック(0 or 1のみ、カンマで区切られている)
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "01"))) )
	{
		switch(type)
		{
		case 0:
			flip_v = simple_strtoul(ptr, &ptr, 0);
			break;
		case 1:
			
			flip_h = simple_strtoul(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	
	// 設定値作成
	lcdm_param_get_userfunc(&param);
	param.flip_v = flip_v;
	param.flip_h = flip_h;
	
	// SubLCDへ反映
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();
	if (LCDM_PARAM_POWER_ON == lcdm_status_get_power()) {
		// H/Wに反映する
		rc = lcd_module_hw_set_userfunc(&param);
	} else {
		// 値のみ保持
		rc = 0;
	}
	if (0 == rc) {
		// パラメタ更新
		lcdm_param_set_userfunc(&param);
	}
	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();
	
	return ((0 == rc) ? count : rc);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：framerate読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_framerate(struct class *cls, char *buf)
{
	lcdm_param_userfunc_t param;
	
	lcdm_param_get_userfunc(&param);
	
	return sprintf(buf, "%u\n", param.framerate);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：framerate書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_framerate(struct class *cls, const char *buf, size_t count)
{
	unsigned long framerate = 0, old_framerate = 0;
	unsigned bussel = 0;
	lcdm_param_userfunc_t param;
	int rc = 0;
	
	rc = strict_strtoul(buf, 0, &framerate);
	// フォーマットチェック
	if (0 > rc) {
		return rc;
	}
	// 範囲チェック
	if ((LCDM_PARAM_FRAMERATE_60HZ != framerate) && (LCDM_PARAM_FRAMERATE_50HZ != framerate)) {
		return  -EINVAL;
	}
	
	// 設定値作成
	lcdm_param_get_userfunc(&param);
	old_framerate = param.framerate;
	param.framerate = framerate;
	
	// SubLCDへ反映
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();
	bussel = readl(LCDCMmioV + LCD_BUSSEL);
	writel(LCD_BUSSEL_BLACK, LCDCMmioV + LCD_BUSSEL);
	mdelay(33);
	if (LCDM_PARAM_POWER_ON == lcdm_status_get_power()) {
		// H/Wに反映する
		rc = lcd_module_hw_set_userfunc(&param);
	} else {
		// 値のみ保持
		rc = 0;
	}
	if (0 == rc) {
		// パラメタ更新
		lcdm_param_set_userfunc(&param);
		// フレームレートに差分がある時はタイミング情報更新
		if (framerate != old_framerate) {
			lcd_hw_update_timing();
		}
	}
	writel(bussel, LCDCMmioV + LCD_BUSSEL);
	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();
	return ((0 == rc) ? count : rc);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：brightness読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_brightness(struct class *cls, char *buf)
{
	unsigned int param = 0;
	
	lcdm_param_get_brightness(&param);
	
	return sprintf(buf, "%u\n", param);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：brightness書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_brightness(struct class *cls, const char *buf, size_t count)
{
	unsigned long brightness = 0;
	unsigned int param = 0;
	int err = 0;
	int ret = 0;
	int retry_count = 0;

	err = strict_strtoul(buf, 0, &brightness);

	// フォーマットチェック
	if (err != 0) {
		ret = err;
		printk_err("error: ret=%d, err=%d, buf=%.16s\n", ret, err, buf);
		return ret;
	}

	// 範囲チェック
	if (brightness > LCDM_PARAM_BRIGHTNESS_MAX) {
		ret = -EINVAL;
		printk_err("error: ret=%d, brightness=%d, buf=%.16s\n", ret, (int)brightness, buf);
		return ret;
	}

	// 設定値作成
	param = brightness;

	// SubLCDへ反映
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();

	if (lcdm_status_get_power() == LCDM_PARAM_POWER_ON) {
		// H/Wに反映する
		for (retry_count = 0 ; retry_count < 5 ; retry_count++) {
			err = lcd_hw_backlight_adjust(param);
			if (err == 0) {
				break;
			} else {
				printk_err("error: retry_count=%d, err=%d, param=%d\n", retry_count, err, param);
			}
		}
	}

	if (err == 0) {
		// パラメタ更新
		lcdm_param_set_brightness(&param);
		ret = count;
	} else {
		lcdm_param_get_brightness(&param);
		err = lcd_hw_backlight_adjust(param);
		if (err != 0) {
			printk_err("error: err=%d, param=%d\n", err, param);
		}
		ret = -EIO;
		printk_err("error: ret=%d, err=%d\n", ret, err);
	}

	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();

	return ret;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：refresh値読み出し
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_refresh(struct class *cls, char *buf)
{
	int err = 0;
	unsigned char read_type = 0;
	unsigned char read_page = 0;
	unsigned char read_addr = 0;
	unsigned char read_data = 0;

	read_type = s_lcdm_read_type;
	read_page = s_lcdm_read_page;
	read_addr = s_lcdm_read_addr;
	printk_dbg(_DEBUG_PMWLCD, "read_type=%02x, read_page=%02x, read_addr=%02x\n", read_type, read_page, read_addr);

	if (read_type == 2) {
		err = lcd_hw_debug_spi_reg_get(read_page, read_addr, &read_data);

		if (err != 0) {
			printk_err("error: err=%d, type=0x%x, page=0x%x, addr=0x%x\n", err, read_type, read_page, read_addr);
			read_type = 0xFF;
		}
	} else {
		printk_err("error: err=%d, type=0x%x, page=0x%x, addr=0x%x\n", err, read_type, read_page, read_addr);
		read_type = 0xFF;
	}

	return snprintf(buf, 13, "%02x %02x %02x %02x\n", read_type, read_page, read_addr, read_data);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：refresh値書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_refresh(struct class *cls, const char *buf, size_t count)
{
	int ret = count;
	int err;
	char *ptr = (char *)buf;
	unsigned char write_type = 0;
	unsigned char write_page = 0;
	unsigned char write_addr = 0;
	unsigned char write_data = 0;

	write_type = simple_strtoul(&buf[0], &ptr, 16);
	write_page = simple_strtoul(&buf[3], &ptr, 16);
	write_addr = simple_strtoul(&buf[6], &ptr, 16);
	write_data = simple_strtoul(&buf[9], &ptr, 16);

	switch (write_type) {
	/* "00 ** ** **" */
	case 0:
		if (lcdm_status_get_power() == LCDM_PARAM_POWER_ON) {
			err = lcd_hw_spi_refresh();

			if (err != 0) {
				ret = -EIO;
				printk_err("error: ret=%d, err=%d\n", ret, err);
			}
		} else {
			ret = -ENODEV;
			printk_err("error: ret=%d, buf=%.16s\n", ret, buf);
		}
		break;

	/* "01 <page> <addr> <data>" */
	case 1:
		if (lcdm_status_get_power() == LCDM_PARAM_POWER_ON) {
			err = lcd_hw_debug_spi_reg_set(write_page, write_addr, write_data);

			if (err != 0) {
				ret = err;
				printk_err("error: ret=%d, err=%d\n", ret, err);
			}
		} else {
			ret = -ENODEV;
			printk_err("error: ret=%d, buf=%.16s\n", ret, buf);
		}
		break;

	/* "02 <page> <addr> **" */
	case 2:
		s_lcdm_read_type = write_type;
		s_lcdm_read_page = write_page;
		s_lcdm_read_addr = write_addr;
		break;

#ifdef DEBUG_SEND_ERROR
	/* "EE 00 00 <offon>" */
	case 0xEE:
		if ((write_page == 0x00) && (write_addr == 0x00)) {
			if (write_data == 0x00) {
				s_lcdm_debug_spi_force_error = false;
			} else {
				s_lcdm_debug_spi_force_error = true;
			}
		} else {
			printk_err("error: type=0x%x, page=0x%x, addr=0x%x, data=0x%x\n", write_type, write_page, write_addr, write_data);
		}
		break;
#endif

	default:
		ret = -EINVAL;
		printk_err("error: ret=%d, write_type=0x%x\n", ret, write_type);
		break;
	}

	return ret;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：電源状態読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_power(struct class *cls, char *buf)
{
	unsigned int status = 0;
	
	status = lcdm_status_get_power();
	
	return sprintf(buf, "%u\n", status);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：電源状態書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_power(struct class *cls, const char *buf, size_t count)
{
	unsigned long power = 0, old_power = 0;
	int rc = 0;
	enum LCDHW_STATUS lcdhw_stat = LCDHW_STATUS_NOT_INITIALIZED;
	enum PWRCTRL_TYPE ctrl_type = PWRCTRL_TYPE_NOP;
	
	rc = strict_strtoul(buf, 0, &power);
	// フォーマットチェック
	if (0 > rc) {
		printk_err("error: rc=%d, power=0x%x, buf=%.16s\n", rc, (unsigned int)power, buf);
		return rc;
	}
	// 範囲チェック
	if ((LCDM_PARAM_POWER_OFF != power) && (LCDM_PARAM_POWER_ON != power)) {
		rc = -EINVAL;
		printk_err("error: rc=%d, power=0x%x, buf=%.16s\n", rc, (unsigned int)power, buf);
		return rc;
	}

	// 制御内容判定
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();
	old_power = lcdm_status_get_power();
	lcdhw_stat = get_lcdhw_status();
	if (old_power == power) {
		// 現在値と設定値が同じなら何もしない
		ctrl_type = PWRCTRL_TYPE_NOP;
	} else if (LCDM_PARAM_POWER_OFF == power){
		// 電源OFF処理
		ctrl_type = PWRCTRL_TYPE_OFF;
	} else if (LCDHW_STATUS_INITIALIZED == lcdhw_stat) {
		// 電源ON処理：LCDHWが一度初期化されている
		ctrl_type = PWRCTRL_TYPE_ON;
	} else {
		// 電源ON処理：LCDHWが一度も初期化されていない
		ctrl_type = PWRCTRL_TYPE_ON_INIT;
	}

	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();
	
	// SubLCDへ反映
	switch (ctrl_type) {
	case PWRCTRL_TYPE_ON_INIT:
		// FrameBufferドライバのopen契機で実行する処理を利用する
		// 排他および値の更新は関数内で行っている
		rc = init_lcdhw();
		break;
	case PWRCTRL_TYPE_ON:
		// FrameBufferドライバのioctl契機で実行する処理を利用する
		// 排他および値の更新は関数内で行っている
		rc = emxx_lcd_blank(FB_BLANK_UNBLANK);
		break;
	case PWRCTRL_TYPE_OFF:
		// FrameBufferドライバのioctl契機で実行する処理を利用する
		// 排他および値の更新は関数内で行っている
//		rc = emxx_lcd_blank(FB_BLANK_NORMAL);
		rc = emxx_lcd_blank(FB_BLANK_POWERDOWN);
		break;
	case PWRCTRL_TYPE_NOP:
	default:
		// 何もしない
		rc = 0;
		break;
	}
	
	return ((0 == rc) ? count : rc);
}

#ifdef PMWLCD_ENABLE_SYSCOLOR
/*******************************************************************************
 * @brief	sysfsクラスインタフェース：color読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_color(struct class *cls, char *buf)
{
	unsigned color = 0;
	
	color = readl(LCDCMmioV + LCD_BACKCOLOR);
	
	return sprintf(buf, "%08x\n", color);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：color書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_color(struct class *cls, const char *buf, size_t count)
{
	unsigned long color = 0;
	int rc = 0;
	enum LCDHW_STATUS lcdhw_stat = LCDHW_STATUS_NOT_INITIALIZED;
	
	rc = strict_strtoul(buf, 16, &color);
	// フォーマットチェック
	if (0 > rc) {
		return rc;
	}
	// 範囲チェック
	if (0x00FFFFFF < color) {
		return  -EINVAL;
	}
	
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();
	
	lcdhw_stat = get_lcdhw_status();
	
	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();
	
	// 電源状態操作
	if (LCDHW_STATUS_INITIALIZED == lcdhw_stat) {
		// 電源ON処理：LCDHWが一度初期化されている
		rc = 0;
	} else {
		// 電源ON処理：LCDHWが一度も初期化されていない
		rc = init_lcdhw();
	}
	
	// SubLCDへ反映
	if (0 == rc) {
		writel(color, LCDCMmioV + LCD_BACKCOLOR);
		writel(LCD_BUSSEL_BACKCOLOR, LCDCMmioV + LCD_BUSSEL);
	}
	
	return ((0 == rc) ? count : rc);
}
#endif	/* PMWLCD_ENABLE_SYSCOLOR */

#ifdef PMWLCD_ENABLE_SYSEDGE
/*******************************************************************************
 * @brief	sysfsクラスインタフェース：vedge読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_vedge(struct class *cls, char *buf)
{
	unsigned total, total_even, area, edge1, edge2, offset_odd, offset_even;
	
	total = readl(LCDCMmioV + LCD_VTOTAL);
	total_even  = readl(LCDCMmioV + LCD_VAREA_EVEN);
	area  = readl(LCDCMmioV + LCD_VAREA);
	edge1 = readl(LCDCMmioV + LCD_VEDGE1);
	edge2 = readl(LCDCMmioV + LCD_VEDGE2);
	offset_odd = readl(LCDCMmioV + LCD_VOFFSET_ODD);
	offset_even = readl(LCDCMmioV + LCD_VOFFSET_EVEN);
	
	return sprintf(buf, "%u %u %u %u %u %u %u\n", total, total_even, area, edge1, edge2, offset_odd, offset_even);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：vedge書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_vedge(struct class *cls, const char *buf, size_t count)
{
	unsigned long total=0, total_even=0, area=0, edge1=0, edge2=0, offset_odd=0, offset_even=0;
	int rc = 0;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	offset_odd = readl(LCDCMmioV + LCD_VOFFSET_ODD);
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "0123456789"))) )
	{
		switch(type)
		{
		case 0:
			total = simple_strtoul(ptr, &ptr, 0);
			break;
		case 1:
			total_even = simple_strtoul(ptr, &ptr, 0);
			break;
		case 2:
			area = simple_strtoul(ptr, &ptr, 0);
			break;
		case 3:
			edge1 = simple_strtoul(ptr, &ptr, 0);
			break;
		case 4:
			edge2 = simple_strtoul(ptr, &ptr, 0);
			break;
		case 5:
			offset_odd = simple_strtoul(ptr, &ptr, 0);
			break;
		case 6:
			offset_even = simple_strtoul(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	
	lcd_hw_stop();
	writel(total, LCDCMmioV + LCD_VTOTAL);
	writel(total_even, LCDCMmioV + LCD_VAREA_EVEN);
	writel(area,  LCDCMmioV + LCD_VAREA);
	writel(edge1, LCDCMmioV + LCD_VEDGE1);
	writel(edge2, LCDCMmioV + LCD_VEDGE2);
	writel(offset_odd, LCDCMmioV + LCD_VOFFSET_ODD);
	writel(offset_even, LCDCMmioV + LCD_VOFFSET_EVEN);
	lcd_hw_start();
	
	return count;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：hedge読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_hedge(struct class *cls, char *buf)
{
	unsigned total, area, edge1, edge2;
	
	total = readl(LCDCMmioV + LCD_HTOTAL);
	area  = readl(LCDCMmioV + LCD_HAREA);
	edge1 = readl(LCDCMmioV + LCD_HEDGE1);
	edge2 = readl(LCDCMmioV + LCD_HEDGE2);
	
	return sprintf(buf, "%u %u %u %u\n", total, area, edge1, edge2);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：hedge書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_hedge(struct class *cls, const char *buf, size_t count)
{
	unsigned long total=0, area=0, edge1=0, edge2=0;
	int rc = 0;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "0123456789"))) )
	{
		switch(type)
		{
		case 0:
			total = simple_strtoul(ptr, &ptr, 0);
			break;
		case 1:
			area = simple_strtoul(ptr, &ptr, 0);
			break;
		case 2:
			edge1 = simple_strtoul(ptr, &ptr, 0);
			break;
		case 3:
			edge2 = simple_strtoul(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	
	lcd_hw_stop();
	writel(total, LCDCMmioV + LCD_HTOTAL);
	writel(area,  LCDCMmioV + LCD_HAREA);
	writel(edge1, LCDCMmioV + LCD_HEDGE1);
	writel(edge2, LCDCMmioV + LCD_HEDGE2);
	lcd_hw_start();
	
	return count;
}
#endif	/* PMWLCD_ENABLE_SYSEDGE */

#ifdef PMWLCD_ENABLE_SYSCOEF
/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_y読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_coef_y(struct class *cls, char *buf)
{
	unsigned coef0, coef1, coef2, coef3;
	int icoef0, icoef1, icoef2, icoef3;
	
	coef0 = readl(LCDCMmioV + LCD_COEF_Y0);
	coef1 = readl(LCDCMmioV + LCD_COEF_Y1);
	coef2 = readl(LCDCMmioV + LCD_COEF_Y2);
	coef3 = readl(LCDCMmioV + LCD_COEF_Y3);
	
	icoef0 = (1023 >= coef0) ? ((int)coef0) : (-2048 + (int)coef0);
	icoef1 = (1023 >= coef1) ? ((int)coef1) : (-2048 + (int)coef1);
	icoef2 = (1023 >= coef2) ? ((int)coef2) : (-2048 + (int)coef2);
	icoef3 = (1023 >= coef3) ? ((int)coef3) : (-2048 + (int)coef3);
	
	return sprintf(buf, "%d %d %d %d\n", icoef0, icoef1, icoef2, icoef3);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_y書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_coef_y(struct class *cls, const char *buf, size_t count)
{
	unsigned coef0=0, coef1=0, coef2=0, coef3=0;
	int icoef0=0, icoef1=0, icoef2=0, icoef3=0;
	int rc = 0;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "-0123456789"))) )
	{
		switch(type)
		{
		case 0:
			icoef0 = simple_strtol(ptr, &ptr, 0);
			break;
		case 1:
			icoef1 = simple_strtol(ptr, &ptr, 0);
			break;
		case 2:
			icoef2 = simple_strtol(ptr, &ptr, 0);
			break;
		case 3:
			icoef3 = simple_strtol(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	if ((-1024 > icoef0) || (1023 < icoef0) || (-1024 > icoef1) || (1023 < icoef1)
	  ||(-1024 > icoef2) || (1023 < icoef2) || (-1024 > icoef3) || (1023 < icoef3)) {
		return -EINVAL;
	}
	
	coef0 = (0 <= icoef0) ? ((unsigned)icoef0) : (unsigned)((int)2048 + icoef0);
	coef1 = (0 <= icoef1) ? ((unsigned)icoef1) : (unsigned)((int)2048 + icoef1);
	coef2 = (0 <= icoef2) ? ((unsigned)icoef2) : (unsigned)((int)2048 + icoef2);
	coef3 = (0 <= icoef3) ? ((unsigned)icoef3) : (unsigned)((int)2048 + icoef3);
	
	lcd_hw_stop();
	writel(coef0, LCDCMmioV + LCD_COEF_Y0);
	writel(coef1, LCDCMmioV + LCD_COEF_Y1);
	writel(coef2, LCDCMmioV + LCD_COEF_Y2);
	writel(coef3, LCDCMmioV + LCD_COEF_Y3);
	lcd_hw_start();
	
	return count;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_u読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_coef_u(struct class *cls, char *buf)
{
	unsigned coef0, coef1, coef2, coef3;
	int icoef0, icoef1, icoef2, icoef3;
	
	coef0 = readl(LCDCMmioV + LCD_COEF_U0);
	coef1 = readl(LCDCMmioV + LCD_COEF_U1);
	coef2 = readl(LCDCMmioV + LCD_COEF_U2);
	coef3 = readl(LCDCMmioV + LCD_COEF_U3);
	
	icoef0 = (1023 >= coef0) ? ((int)coef0) : (-2048 + (int)coef0);
	icoef1 = (1023 >= coef1) ? ((int)coef1) : (-2048 + (int)coef1);
	icoef2 = (1023 >= coef2) ? ((int)coef2) : (-2048 + (int)coef2);
	icoef3 = (1023 >= coef3) ? ((int)coef3) : (-2048 + (int)coef3);
	
	return sprintf(buf, "%d %d %d %d\n", icoef0, icoef1, icoef2, icoef3);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_u書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_coef_u(struct class *cls, const char *buf, size_t count)
{
	unsigned coef0=0, coef1=0, coef2=0, coef3=0;
	int icoef0=0, icoef1=0, icoef2=0, icoef3=0;
	int rc = 0;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "-0123456789"))) )
	{
		switch(type)
		{
		case 0:
			icoef0 = simple_strtol(ptr, &ptr, 0);
			break;
		case 1:
			icoef1 = simple_strtol(ptr, &ptr, 0);
			break;
		case 2:
			icoef2 = simple_strtol(ptr, &ptr, 0);
			break;
		case 3:
			icoef3 = simple_strtol(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	if ((-1024 > icoef0) || (1023 < icoef0) || (-1024 > icoef1) || (1023 < icoef1)
	  ||(-1024 > icoef2) || (1023 < icoef2) || (-1024 > icoef3) || (1023 < icoef3)) {
		return -EINVAL;
	}
	
	coef0 = (0 <= icoef0) ? ((unsigned)icoef0) : (unsigned)((int)2048 + icoef0);
	coef1 = (0 <= icoef1) ? ((unsigned)icoef1) : (unsigned)((int)2048 + icoef1);
	coef2 = (0 <= icoef2) ? ((unsigned)icoef2) : (unsigned)((int)2048 + icoef2);
	coef3 = (0 <= icoef3) ? ((unsigned)icoef3) : (unsigned)((int)2048 + icoef3);
	
	lcd_hw_stop();
	writel(coef0, LCDCMmioV + LCD_COEF_U0);
	writel(coef1, LCDCMmioV + LCD_COEF_U1);
	writel(coef2, LCDCMmioV + LCD_COEF_U2);
	writel(coef3, LCDCMmioV + LCD_COEF_U3);
	lcd_hw_start();
	
	return count;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_v読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_coef_v(struct class *cls, char *buf)
{
	unsigned coef0, coef1, coef2, coef3;
	int icoef0, icoef1, icoef2, icoef3;
	
	coef0 = readl(LCDCMmioV + LCD_COEF_V0);
	coef1 = readl(LCDCMmioV + LCD_COEF_V1);
	coef2 = readl(LCDCMmioV + LCD_COEF_V2);
	coef3 = readl(LCDCMmioV + LCD_COEF_V3);
	
	icoef0 = (1023 >= coef0) ? ((int)coef0) : (-2048 + (int)coef0);
	icoef1 = (1023 >= coef1) ? ((int)coef1) : (-2048 + (int)coef1);
	icoef2 = (1023 >= coef2) ? ((int)coef2) : (-2048 + (int)coef2);
	icoef3 = (1023 >= coef3) ? ((int)coef3) : (-2048 + (int)coef3);
	
	return sprintf(buf, "%d %d %d %d\n", icoef0, icoef1, icoef2, icoef3);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：coef_v書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_coef_v(struct class *cls, const char *buf, size_t count)
{
	unsigned coef0=0, coef1=0, coef2=0, coef3=0;
	int icoef0=0, icoef1=0, icoef2=0, icoef3=0;
	int rc = 0;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "-0123456789"))) )
	{
		switch(type)
		{
		case 0:
			icoef0 = simple_strtol(ptr, &ptr, 0);
			break;
		case 1:
			icoef1 = simple_strtol(ptr, &ptr, 0);
			break;
		case 2:
			icoef2 = simple_strtol(ptr, &ptr, 0);
			break;
		case 3:
			icoef3 = simple_strtol(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	if ((-1024 > icoef0) || (1023 < icoef0) || (-1024 > icoef1) || (1023 < icoef1)
	  ||(-1024 > icoef2) || (1023 < icoef2) || (-1024 > icoef3) || (1023 < icoef3)) {
		return -EINVAL;
	}
	
	coef0 = (0 <= icoef0) ? ((unsigned)icoef0) : (unsigned)((int)2048 + icoef0);
	coef1 = (0 <= icoef1) ? ((unsigned)icoef1) : (unsigned)((int)2048 + icoef1);
	coef2 = (0 <= icoef2) ? ((unsigned)icoef2) : (unsigned)((int)2048 + icoef2);
	coef3 = (0 <= icoef3) ? ((unsigned)icoef3) : (unsigned)((int)2048 + icoef3);
	
	lcd_hw_stop();
	writel(coef0, LCDCMmioV + LCD_COEF_V0);
	writel(coef1, LCDCMmioV + LCD_COEF_V1);
	writel(coef2, LCDCMmioV + LCD_COEF_V2);
	writel(coef3, LCDCMmioV + LCD_COEF_V3);
	lcd_hw_start();
	
	return count;
}
#endif	/* PMWLCD_ENABLE_SYSCOEF */

#ifdef PMWLCD_ENABLE_SYSBUSSEL
/*******************************************************************************
 * @brief	sysfsクラスインタフェース：bussel読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_bussel(struct class *cls, char *buf)
{
	unsigned bussel;
	
	bussel = readl(LCDCMmioV + LCD_BUSSEL) & LCD_BUSSEL_BIT;
	
	return sprintf(buf, "%u\n", bussel);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：bussel書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_bussel(struct class *cls, const char *buf, size_t count)
{
	unsigned bussel = LCD_BUSSEL_BLACK;
	int rc;
	
	int type = 0;
	char *ptr = (char *)buf;
	rc = -EINVAL;
	while( (NULL != ptr) && (NULL != (ptr = strpbrk(ptr, "012345"))) )
	{
		switch(type)
		{
		case 0:
			bussel = simple_strtoul(ptr, &ptr, 0);
			rc = 0;
			break;
		default:
			ptr = NULL;
			rc = -EINVAL;
			break;
		}
		type++;
	}
	if (0 > rc) {
		return rc;
	}
	if ((LCD_BUSSEL_LOCAL    != bussel) && (LCD_BUSSEL_DIRECT    != bussel)
	 && (LCD_BUSSEL_WB_LOCAL != bussel) && (LCD_BUSSEL_WB_DIRECT != bussel)
	 && (LCD_BUSSEL_BLACK    != bussel) && (LCD_BUSSEL_BACKCOLOR != bussel))
	{
		return -EINVAL;
	}
	
	writel(bussel, LCDCMmioV + LCD_BUSSEL);
	
	return count;
}
#endif	/* PMWLCD_ENABLE_SYSBUSSEL */

#ifdef PMWLCD_ENABLE_SYSEVENT
/*******************************************************************************
 * @brief	sysfsクラスインタフェース：event読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_event(struct class *cls, char *buf)
{
	unsigned event;
	
	event = lcdm_event_get_value();
	
	return sprintf(buf, "%08x\n", event);
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：event書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_event(struct class *cls, const char *buf, size_t count)
{
	return -EINVAL;
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：waitevent読み込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[out]	*buf	ユーザへ返すデータへのポインタ
 *
 * @return	読み込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_show_waitevent(struct class *cls, char *buf)
{
	int result;
	unsigned event_1st, event_2nd;
	
	event_1st = lcdm_event_get_value();
	result = lcdm_event_wait_update();
	event_2nd = lcdm_event_get_value();
	
	if (0 > result) {
		return result;
	}
	return sprintf(buf, "%08x\n", (event_1st ^ event_2nd));
}

/*******************************************************************************
 * @brief	sysfsクラスインタフェース：waitevent書き込み
 *
 * @param[in]	*cls	クラスへのポインタ
 * @param[in]	*buf	ユーザが書き込んだデータへのポインタ
 * @param[in]	*count	ユーザが書き込んだデータサイズ
 *
 * @return	書き込みサイズ
 *******************************************************************************/
static ssize_t lcdm_class_store_waitevent(struct class *cls, const char *buf, size_t count)
{
	return -EINVAL;
}
#endif	/* PMWLCD_ENABLE_SYSEVENT */

/*******************************************************************************
 * @brief	Sysfs用クラス登録
 *
 * @param	なし
 *
 * @retval	0	成功
 * @retval	0以外	失敗
 *******************************************************************************/
static int lcdm_register_class(void)
{
	int result = 0, index = 0;
	
	printk_dbg((_DEBUG_PMWLCD & 0x01), "\n");
	
	// クラス登録
	lcdm_class = class_create(THIS_MODULE, DEV_NAME);
	if (IS_ERR(lcdm_class)) {
		result = PTR_ERR(lcdm_class);
		goto out_exit;
	}
	// クラス属性登録
	for (index=0; (NULL != lcdm_class_attributes[index].attr.name); index++) {
		result = class_create_file(lcdm_class, &lcdm_class_attributes[index]);
		if (0 != result) {
			goto out_destroy;
		}
	}
	goto out_exit;

out_destroy:
	lcdm_unregister_class();
out_exit:
	printk_dbg((_DEBUG_PMWLCD & 0x02), "result = %d, index = %d\n", result, index);
	return result;
}

/*******************************************************************************
 * @brief	Sysfs用クラス削除
 *
 * @param	なし
 *
 * @retval	0	成功
 * @retval	0以外	失敗
 *******************************************************************************/
static int lcdm_unregister_class(void)
{
	int result = 0, index = 0;
	
	printk_dbg((_DEBUG_PMWLCD & 0x01), "\n");
	
	for (index=0; (NULL != lcdm_class_attributes[index].attr.name); index++) {
		class_remove_file(lcdm_class, &lcdm_class_attributes[index]);
	}
	class_destroy(lcdm_class);
	
	printk_dbg((_DEBUG_PMWLCD & 0x02), "result = %d, index = %d\n", result, index);
	return result;
}

/*******************************************************************************
 * @brief	LCDモジュールとの通信ポーリング開始
 *
 * @param	なし
 *
 * @return	なし
 *******************************************************************************/
void lcdm_start_polling(void)
{
#if 0
	// ワークキュー初期化
	PREPARE_DELAYED_WORK(&lcdm_polling_work, lcdm_do_polling_work);
	
	// ポーリング処理スケジューリング
	s_jiffies_work = jiffies;
	queue_delayed_work(lcdm_workqueue, &lcdm_polling_work, lcdm_polling_cycle);
	printk_dbg((_DEBUG_PMWLCD & 0x04), "Scheduled after %u jiffies.\n", lcdm_polling_cycle);
#else
	return;
#endif
}

/*******************************************************************************
 * @brief	LCDモジュールとの通信ポーリング停止
 *
 * @param	なし
 *
 * @return	なし
 *******************************************************************************/
void lcdm_stop_polling(void)
{
#if 0
	int result = 0;
	
	// 処理がキューに溜まっていた場合、後段のflushで実行されてしまうので
	// まず最初にキャンセルを行う
	result = cancel_delayed_work(&lcdm_polling_work);
	printk_dbg((_DEBUG_PMWLCD & 0x04), "1st cancel_delayed_work() = %d\n", result);
	
	// flushを実行し、もし現在実行中の処理があれば完了させる
	flush_delayed_work(&lcdm_polling_work);
	
	// タスクの処理の中で自分自身を再登録しているので、改めてキャンセル実行
	// 次のタスク実行までの時間が十分に長い事が前提
	result = cancel_delayed_work(&lcdm_polling_work);
	printk_dbg((_DEBUG_PMWLCD & 0x04), "2nd cancel_delayed_work() = %d\n", result);
#else
	return;
#endif
}

/*******************************************************************************
 * @brief	LCDモジュールとの静電対策用通信処理
 *
 * @param[in]	work	work構造体へのポインタ
 *
 * @return	なし
 *******************************************************************************/
static void lcdm_do_polling_work(struct work_struct *work)
{
	lcdm_param_userfunc_t param_local;
	lcdm_param_userfunc_t param_now;
	int result = 0, restart = 0, nextwork = 0;
	unsigned long diffmsec = 0;
	unsigned long status = 0;
	
	diffmsec = ((jiffies - s_jiffies_work) * 1000 / HZ);
	printk_dbg((_DEBUG_PMWLCD & 0x04), "wakeup diff = %lu msec.\n", diffmsec);
	
	result = 0;
	restart = 0;
	nextwork = 0;
	
	//########## LCDHW排他開始 ##########
	lcdhw_power_mutex_lock();
	do {
		// 電源OFFなのに本処理が来るのは異常
		// ポーリング処理を停止するが、再起動対象とはしない
		if (LCDM_PARAM_POWER_OFF ==lcdm_status_get_power()) {
			printk_wrn("SubLCD Polling - Power Status Unmatch\n");
			break;
		}
		
		// ビデオクロックが供給されているか確認する
		// SPI通信はクロックが供給されている状態で行うシーケンスの為
		// ポーリング処理は継続するが、再起動対象とはしない
		status = lcd_hw_chk_status();
		if (!(status & LCD_STATUS_ON)) {
			nextwork = 1;
			break;
		}
		
		// ドライバ内で保持している値を取得
		lcdm_param_get_userfunc(&param_local);
		
#if 0
		// SubLCDから実際の設定値を読み出し
		result = lcd_module_hw_get_userfunc(&param_now);
#else
		param_now = param_local;
#endif
		
		// SPI通信に失敗した場合は異常
		// SubLCD再起動対象とする
		if (0 > result) {
			printk_wrn("SubLCD Polling - SPI NG\n");
			restart = 2;
			break;
		}
		
		// SubLCDで取得した値と保持している値が違う
		// SubLCD再起動対象とする
		if (0 != memcmp(&param_local, &param_now, sizeof(param_local))) {
			printk_wrn("SubLCD Polling - SPI READ!=WRITE\n");
			printk_wrn("SPI(0xFE) EXPECTED VALUE: framerate=%d, flip_h=%d, flip_v=%d\n",
				param_local.framerate, param_local.flip_h, param_local.flip_v);
			printk_wrn("SPI(0xFE) READ VALUE: framerate=%d, flip_h=%d, flip_v=%d\n",
				param_now.framerate, param_now.flip_h, param_now.flip_v);
			restart = 3;
			break;
		}
		
		// ポーリング処理を継続する
		nextwork = 1;
		
	}while(0);
	//########## LCDHW排他終了 ##########
	lcdhw_power_mutex_unlock();
	
	// SubLCD再起動が必要か？
	if (restart) {
		// SubLCDの電源OFF時に、ポーリング処理の終了待ちをしている
		// よって、ここで直接SubLCDの電源OFFを呼び出す事はできない
		// 遅延タスクとして実行する
		printk_err("Scheduled Restart SubLCD: %d\n", restart);
		queue_work(lcdm_workqueue, &lcdm_chauge_output_work);
		
		/* Set Restart Error */
		lcdm_event_set_value(LCDM_EVENT_RESTART, LCDM_ON);
	}
	
	// ポーリング処理スケジューリング
	if (nextwork) {
		s_jiffies_work = jiffies;
		queue_delayed_work(lcdm_workqueue, &lcdm_polling_work, lcdm_polling_cycle);
		printk_dbg((_DEBUG_PMWLCD & 0x04), "Scheduled after %u jiffies.\n", lcdm_polling_cycle);
	}
}

/*******************************************************************************
 * @brief	SubLCD電源OFF->ON処理
 *
 * @param[in]	work	work構造体へのポインタ
 *
 * @return	なし
 *******************************************************************************/
static void lcdm_do_chauge_output_work(struct work_struct *work)
{
	const int wait_sleep_msec = 200;	//!< 電源OFF→ON時に少し待つ時間(msec)
	unsigned color, bussel;
	
	printk_err("Do Restart SubLCD\n");
	
	// 必要なレジスタ情報退避
	color = readl(LCDCMmioV + LCD_BACKCOLOR);
	bussel = readl(LCDCMmioV + LCD_BUSSEL) & LCD_BUSSEL_BIT;
	
	// FrameBufferドライバのioctl契機で実行する処理を利用する
	// 排他および値の更新は関数内で行っている
	emxx_lcd_blank(FB_BLANK_NORMAL);
	
	// パネル仕様書より
	mdelay(wait_sleep_msec);
	
	// FrameBufferドライバのioctl契機で実行する処理を利用する
	// 排他および値の更新は関数内で行っている
	emxx_lcd_blank(FB_BLANK_UNBLANK);
	
	// レジスタ情報書き戻し
	writel(color, LCDCMmioV + LCD_BACKCOLOR);
	writel(bussel, LCDCMmioV + LCD_BUSSEL);
	
	/* Clear Restart Error */
	lcdm_event_set_value(LCDM_EVENT_RESTART, LCDM_OFF);
}

/*******************************************************************************
 * @brief	PMWLCDモジュール初期化
 *
 * @param[in]	work	work構造体へのポインタ
 *
 * @retval	0	成功
 * @retval	0以外	失敗
 *******************************************************************************/
int init_mod_pmwlcd(void)
{
	// ワークキュー作成
	lcdm_workqueue = create_singlethread_workqueue(DEV_NAME);

	// イベント待ちキュー初期化
	init_waitqueue_head(&s_lcdm_event_wait_queue);

	// クラス登録
	lcdm_register_class();

	return 0;
}

/*******************************************************************************
 * @brief	PMWLCDモジュール終了化
 *
 * @param[in]	work	work構造体へのポインタ
 *
 * @retval	0	成功
 * @retval	0以外	失敗
 *******************************************************************************/
int exit_mod_pmwlcd(void)
{
	// クラス削除
	lcdm_unregister_class();

	// ポーリング処理停止
	lcdm_stop_polling();

	// 念のため再起動処理停止
	cancel_work_sync(&lcdm_chauge_output_work);
	flush_work(&lcdm_chauge_output_work);

	// ワークキュー削除
	destroy_workqueue(lcdm_workqueue);

	return 0;
}
