/*
 * linux/fs/nls/nls_euc-jp.c
 *
 * Added `OSF/JVC Recommended Code Set Conversion Specification
 * between Japanese EUC and Shift-JIS' support: <hirofumi@mail.parknet.co.jp>
 * (http://www.opengroup.or.jp/jvc/cde/sjis-euc-e.html)
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/nls.h>
#include <linux/errno.h>

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
static struct nls_table *p_nls;
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
#define IS_SJIS_LOW_BYTE(l)	((0x40 <= (l)) && ((l) <= 0xFC) && ((l) != 0x7F))
/* JIS X 0208 (include NEC spesial characters) */
#define IS_SJIS_JISX0208(h, l)	((((0x81 <= (h)) && ((h) <= 0x9F))	\
				 || ((0xE0 <= (h)) && ((h) <= 0xEA)))	\
				 && IS_SJIS_LOW_BYTE(l))
#define IS_SJIS_JISX0201KANA(c)	((0xA1 <= (c)) && ((c) <= 0xDF))
#define IS_SJIS_UDC_LOW(h, l)	(((0xF0 <= (h)) && ((h) <= 0xF4))	\
				 && IS_SJIS_LOW_BYTE(l))
#define IS_SJIS_UDC_HI(h, l)	(((0xF5 <= (h)) && ((h) <= 0xF9))	\
				 && IS_SJIS_LOW_BYTE(l))
#define IS_SJIS_IBM(h, l)	(((0xFA <= (h)) && ((h) <= 0xFC))	\
				 && IS_SJIS_LOW_BYTE(l))
#define IS_SJIS_NECIBM(h, l)	(((0xED <= (h)) && ((h) <= 0xEE))	\
				 && IS_SJIS_LOW_BYTE(l))
#define MAP_SJIS2EUC(sjis_hi, sjis_lo, sjis_p, euc_hi, euc_lo, euc_p) {		\
	if ((sjis_lo) >= 0x9F) {						\
		(euc_hi) = (sjis_hi) * 2 - (((sjis_p) * 2 - (euc_p)) - 1);	\
		(euc_lo) = (sjis_lo) + 2;					\
	} else {								\
		(euc_hi) = (sjis_hi) * 2 - ((sjis_p) * 2 - (euc_p));		\
		(euc_lo) = (sjis_lo) + ((sjis_lo) >= 0x7F ? 0x60 : 0x61);	\
	}									\
} while(0)
#endif

#define SS2		(0x8E)		/* Single Shift 2 */
#define SS3		(0x8F)		/* Single Shift 3 */
#define IS_EUC_BYTE(c)		((0xA1 <= (c)) && ((c) <= 0xFE))
#define IS_EUC_JISX0208(h, l)	(IS_EUC_BYTE(h) && IS_EUC_BYTE(l))
#define IS_EUC_JISX0201KANA(h, l)	(((h) == SS2) && (0xA1 <= (l) && (l) <= 0xDF))
#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
#define IS_EUC_UDC_LOW(h, l)	(((0xF5 <= (h)) && ((h) <= 0xFE))	\
				 && IS_EUC_BYTE(l))
#define IS_EUC_UDC_HI(h, l)	IS_EUC_UDC_LOW(h, l) /* G3 block */
#define MAP_EUC2SJIS(euc_hi, euc_lo, euc_p, sjis_hi, sjis_lo, sjis_p) {		\
	if ((euc_hi) & 1) {							\
		(sjis_hi) = (euc_hi) / 2 + ((sjis_p) - (euc_p) / 2);		\
		(sjis_lo) = (euc_lo) - ((euc_lo) >= 0xE0 ? 0x60 : 0x61);	\
	} else {								\
		(sjis_hi) = (euc_hi) / 2 + (((sjis_p) - (euc_p) / 2) - 1);	\
		(sjis_lo) = (euc_lo) - 2;					\
	}									\
} while(0)
#else /* defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932) */
#define IS_GL(c)		(((unsigned char)(c) & 0x80) == 0)
#define IS_C1(c)		((((unsigned char)(c) & 0xe0) == 0x80)	\
				 && ((c) != SS2 && (c) != SS3))
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* SJIS IBM extended characters to EUC map */
static const unsigned char sjisibm2euc_map[][2] = {
	{0xF3, 0xF3}, {0xF3, 0xF4}, {0xF3, 0xF5}, {0xF3, 0xF6}, {0xF3, 0xF7},
	{0xF3, 0xF8}, {0xF3, 0xF9}, {0xF3, 0xFA}, {0xF3, 0xFB}, {0xF3, 0xFC},
	{0xF3, 0xFD}, {0xF3, 0xFE}, {0xF4, 0xA1}, {0xF4, 0xA2}, {0xF4, 0xA3},
	{0xF4, 0xA4}, {0xF4, 0xA5}, {0xF4, 0xA6}, {0xF4, 0xA7}, {0xF4, 0xA8},
	{0xA2, 0xCC}, {0xA2, 0xC3}, {0xF4, 0xA9}, {0xF4, 0xAA}, {0xF4, 0xAB},
	{0xF4, 0xAC}, {0xF4, 0xAD}, {0xA2, 0xE8}, {0xD4, 0xE3}, {0xDC, 0xDF},
	{0xE4, 0xE9}, {0xE3, 0xF8}, {0xD9, 0xA1}, {0xB1, 0xBB}, {0xF4, 0xAE},
	{0xC2, 0xAD}, {0xC3, 0xFC}, {0xE4, 0xD0}, {0xC2, 0xBF}, {0xBC, 0xF4},
	{0xB0, 0xA9}, {0xB0, 0xC8}, {0xF4, 0xAF}, {0xB0, 0xD2}, {0xB0, 0xD4},
	{0xB0, 0xE3}, {0xB0, 0xEE}, {0xB1, 0xA7}, {0xB1, 0xA3}, {0xB1, 0xAC},
	{0xB1, 0xA9}, {0xB1, 0xBE}, {0xB1, 0xDF}, {0xB1, 0xD8}, {0xB1, 0xC8},
	{0xB1, 0xD7}, {0xB1, 0xE3}, {0xB1, 0xF4}, {0xB1, 0xE1}, {0xB2, 0xA3},
	{0xF4, 0xB0}, {0xB2, 0xBB}, {0xB2, 0xE6}, {0x00, 0x00}, {0xB2, 0xED},
	{0xB2, 0xF5}, {0xB2, 0xFC}, {0xF4, 0xB1}, {0xB3, 0xB5}, {0xB3, 0xD8},
	{0xB3, 0xDB}, {0xB3, 0xE5}, {0xB3, 0xEE}, {0xB3, 0xFB}, {0xF4, 0xB2},
	{0xF4, 0xB3}, {0xB4, 0xC0}, {0xB4, 0xC7}, {0xB4, 0xD0}, {0xB4, 0xDE},
	{0xF4, 0xB4}, {0xB5, 0xAA}, {0xF4, 0xB5}, {0xB5, 0xAF}, {0xB5, 0xC4},
	{0xB5, 0xE8}, {0xF4, 0xB6}, {0xB7, 0xC2}, {0xB7, 0xE4}, {0xB7, 0xE8},
	{0xB7, 0xE7}, {0xF4, 0xB7}, {0xF4, 0xB8}, {0xF4, 0xB9}, {0xB8, 0xCE},
	{0xB8, 0xE1}, {0xB8, 0xF5}, {0xB8, 0xF7}, {0xB8, 0xF8}, {0xB8, 0xFC},
	{0xB9, 0xAF}, {0xB9, 0xB7}, {0xBA, 0xBE}, {0xBA, 0xDB}, {0xCD, 0xAA},
	{0xBA, 0xE1}, {0xF4, 0xBA}, {0xBA, 0xEB}, {0xBB, 0xB3}, {0xBB, 0xB8},
	{0xF4, 0xBB}, {0xBB, 0xCA}, {0xF4, 0xBC}, {0xF4, 0xBD}, {0xBB, 0xD0},
	{0xBB, 0xDE}, {0xBB, 0xF4}, {0xBB, 0xF5}, {0xBB, 0xF9}, {0xBC, 0xE4},
	{0xBC, 0xED}, {0xBC, 0xFE}, {0xF4, 0xBE}, {0xBD, 0xC2}, {0xBD, 0xE7},
	{0xF4, 0xBF}, {0xBD, 0xF0}, {0xBE, 0xB0}, {0xBE, 0xAC}, {0xF4, 0xC0},
	{0xBE, 0xB3}, {0xBE, 0xBD}, {0xBE, 0xCD}, {0xBE, 0xC9}, {0xBE, 0xE4},
	{0xBF, 0xA8}, {0xBF, 0xC9}, {0xC0, 0xC4}, {0xC0, 0xE4}, {0xC0, 0xF4},
	{0xC1, 0xA6}, {0xF4, 0xC1}, {0xC1, 0xF5}, {0xC1, 0xFC}, {0xF4, 0xC2},
	{0xC1, 0xF8}, {0xC2, 0xAB}, {0xC2, 0xA1}, {0xC2, 0xA5}, {0xF4, 0xC3},
	{0xC2, 0xB8}, {0xC2, 0xBA}, {0xF4, 0xC4}, {0xC2, 0xC4}, {0xC2, 0xD2},
	{0xC2, 0xD7}, {0xC2, 0xDB}, {0xC2, 0xDE}, {0xC2, 0xED}, {0xC2, 0xF0},
	{0xF4, 0xC5}, {0xC3, 0xA1}, {0xC3, 0xB5}, {0xC3, 0xC9}, {0xC3, 0xB9},
	{0xF4, 0xC6}, {0xC3, 0xD8}, {0xC3, 0xFE}, {0xF4, 0xC7}, {0xC4, 0xCC},
	{0xF4, 0xC8}, {0xC4, 0xD9}, {0xC4, 0xEA}, {0xC4, 0xFD}, {0xF4, 0xC9},
	{0xC5, 0xA7}, {0xC5, 0xB5}, {0xC5, 0xB6}, {0xF4, 0xCA}, {0xC5, 0xD5},
	{0xC6, 0xB8}, {0xC6, 0xD7}, {0xC6, 0xE0}, {0xC6, 0xEA}, {0xC6, 0xE3},
	{0xC7, 0xA1}, {0xC7, 0xAB}, {0xC7, 0xC7}, {0xC7, 0xC3}, {0xC7, 0xCB},
	{0xC7, 0xCF}, {0xC7, 0xD9}, {0xF4, 0xCB}, {0xF4, 0xCC}, {0xC7, 0xE6},
	{0xC7, 0xEE}, {0xC7, 0xFC}, {0xC7, 0xEB}, {0xC7, 0xF0}, {0xC8, 0xB1},
	{0xC8, 0xE5}, {0xC8, 0xF8}, {0xC9, 0xA6}, {0xC9, 0xAB}, {0xC9, 0xAD},
	{0xF4, 0xCD}, {0xC9, 0xCA}, {0xC9, 0xD3}, {0xC9, 0xE9}, {0xC9, 0xE3},
	{0xC9, 0xFC}, {0xC9, 0xF4}, {0xC9, 0xF5}, {0xF4, 0xCE}, {0xCA, 0xB3},
	{0xCA, 0xBD}, {0xCA, 0xEF}, {0xCA, 0xF1}, {0xCB, 0xAE}, {0xF4, 0xCF},
	{0xCB, 0xCA}, {0xCB, 0xE6}, {0xCB, 0xEA}, {0xCB, 0xF0}, {0xCB, 0xF4},
	{0xCB, 0xEE}, {0xCC, 0xA5}, {0xCB, 0xF9}, {0xCC, 0xAB}, {0xCC, 0xAE},
	{0xCC, 0xAD}, {0xCC, 0xB2}, {0xCC, 0xC2}, {0xCC, 0xD0}, {0xCC, 0xD9},
	{0xF4, 0xD0}, {0xCD, 0xBB}, {0xF4, 0xD1}, {0xCE, 0xBB}, {0xF4, 0xD2},
	{0xCE, 0xBA}, {0xCE, 0xC3}, {0xF4, 0xD3}, {0xCE, 0xF2}, {0xB3, 0xDD},
	{0xCF, 0xD5}, {0xCF, 0xE2}, {0xCF, 0xE9}, {0xCF, 0xED}, {0xF4, 0xD4},
	{0xF4, 0xD5}, {0xF4, 0xD6}, {0x00, 0x00}, {0xF4, 0xD7}, {0xD0, 0xE5},
	{0xF4, 0xD8}, {0xD0, 0xE9}, {0xD1, 0xE8}, {0xF4, 0xD9}, {0xF4, 0xDA},
	{0xD1, 0xEC}, {0xD2, 0xBB}, {0xF4, 0xDB}, {0xD3, 0xE1}, {0xD3, 0xE8},
	{0xD4, 0xA7}, {0xF4, 0xDC}, {0xF4, 0xDD}, {0xD4, 0xD4}, {0xD4, 0xF2},
	{0xD5, 0xAE}, {0xF4, 0xDE}, {0xD7, 0xDE}, {0xF4, 0xDF}, {0xD8, 0xA2},
	{0xD8, 0xB7}, {0xD8, 0xC1}, {0xD8, 0xD1}, {0xD8, 0xF4}, {0xD9, 0xC6},
	{0xD9, 0xC8}, {0xD9, 0xD1}, {0xF4, 0xE0}, {0xF4, 0xE1}, {0xF4, 0xE2},
	{0xF4, 0xE3}, {0xF4, 0xE4}, {0xDC, 0xD3}, {0xDD, 0xC8}, {0xDD, 0xD4},
	{0xDD, 0xEA}, {0xDD, 0xFA}, {0xDE, 0xA4}, {0xDE, 0xB0}, {0xF4, 0xE5},
	{0xDE, 0xB5}, {0xDE, 0xCB}, {0xF4, 0xE6}, {0xDF, 0xB9}, {0xF4, 0xE7},
	{0xDF, 0xC3}, {0xF4, 0xE8}, {0xF4, 0xE9}, {0xE0, 0xD9}, {0xF4, 0xEA},
	{0xF4, 0xEB}, {0xE1, 0xE2}, {0xF4, 0xEC}, {0xF4, 0xED}, {0xF4, 0xEE},
	{0xE2, 0xC7}, {0xE3, 0xA8}, {0xE3, 0xA6}, {0xE3, 0xA9}, {0xE3, 0xAF},
	{0xE3, 0xB0}, {0xE3, 0xAA}, {0xE3, 0xAB}, {0xE3, 0xBC}, {0xE3, 0xC1},
	{0xE3, 0xBF}, {0xE3, 0xD5}, {0xE3, 0xD8}, {0xE3, 0xD6}, {0xE3, 0xDF},
	{0xE3, 0xE3}, {0xE3, 0xE1}, {0xE3, 0xD4}, {0xE3, 0xE9}, {0xE4, 0xA6},
	{0xE3, 0xF1}, {0xE3, 0xF2}, {0xE4, 0xCB}, {0xE4, 0xC1}, {0xE4, 0xC3},
	{0xE4, 0xBE}, {0xF4, 0xEF}, {0xE4, 0xC0}, {0xE4, 0xC7}, {0xE4, 0xBF},
	{0xE4, 0xE0}, {0xE4, 0xDE}, {0xE4, 0xD1}, {0xF4, 0xF0}, {0xE4, 0xDC},
	{0xE4, 0xD2}, {0xE4, 0xDB}, {0xE4, 0xD4}, {0xE4, 0xFA}, {0xE4, 0xEF},
	{0xE5, 0xB3}, {0xE5, 0xBF}, {0xE5, 0xC9}, {0xE5, 0xD0}, {0xE5, 0xE2},
	{0xE5, 0xEA}, {0xE5, 0xEB}, {0xF4, 0xF1}, {0xF4, 0xF2}, {0xF4, 0xF3},
	{0xE6, 0xE8}, {0xE6, 0xEF}, {0xE7, 0xAC}, {0xF4, 0xF4}, {0xE7, 0xAE},
	{0xF4, 0xF5}, {0xE7, 0xB1}, {0xF4, 0xF6}, {0xE7, 0xB2}, {0xE8, 0xB1},
	{0xE8, 0xB6}, {0xF4, 0xF7}, {0xF4, 0xF8}, {0xE8, 0xDD}, {0xF4, 0xF9},
	{0xF4, 0xFA}, {0xE9, 0xD1}, {0xF4, 0xFB}, {0xE9, 0xED}, {0xEA, 0xCD},
	{0xF4, 0xFC}, {0xEA, 0xDB}, {0xEA, 0xE6}, {0xEA, 0xEA}, {0xEB, 0xA5},
	{0xEB, 0xFB}, {0xEB, 0xFA}, {0xF4, 0xFD}, {0xEC, 0xD6}, {0xF4, 0xFE},
};
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
#define IS_EUC_IBM2JISX0208(h, l) \
		(((h) == 0xA2 && (l) == 0xCC) || ((h) == 0xA2 && (l) == 0xE8))
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* EUC to SJIS IBM extended characters map (G3 JIS X 0212 block) */
static struct {
	unsigned short euc;
	unsigned char sjis[2];
} euc2sjisibm_jisx0212_map[] = {
	{0xA2C3, {0xFA, 0x55}}, {0xB0A9, {0xFA, 0x68}}, {0xB0C8, {0xFA, 0x69}},
	{0xB0D2, {0xFA, 0x6B}}, {0xB0D4, {0xFA, 0x6C}}, {0xB0E3, {0xFA, 0x6D}},
	{0xB0EE, {0xFA, 0x6E}}, {0xB1A3, {0xFA, 0x70}}, {0xB1A7, {0xFA, 0x6F}},
	{0xB1A9, {0xFA, 0x72}}, {0xB1AC, {0xFA, 0x71}}, {0xB1BB, {0xFA, 0x61}},
	{0xB1BE, {0xFA, 0x73}}, {0xB1C8, {0xFA, 0x76}}, {0xB1D7, {0xFA, 0x77}},
	{0xB1D8, {0xFA, 0x75}}, {0xB1DF, {0xFA, 0x74}}, {0xB1E1, {0xFA, 0x7A}},
	{0xB1E3, {0xFA, 0x78}}, {0xB1F4, {0xFA, 0x79}}, {0xB2A3, {0xFA, 0x7B}},
	{0xB2BB, {0xFA, 0x7D}}, {0xB2E6, {0xFA, 0x7E}}, {0xB2ED, {0xFA, 0x80}},
	{0xB2F5, {0xFA, 0x81}}, {0xB2FC, {0xFA, 0x82}}, {0xB3B5, {0xFA, 0x84}},
	{0xB3D8, {0xFA, 0x85}}, {0xB3DB, {0xFA, 0x86}}, {0xB3DD, {0xFB, 0x77}},
	{0xB3E5, {0xFA, 0x87}}, {0xB3EE, {0xFA, 0x88}}, {0xB3FB, {0xFA, 0x89}},
	{0xB4C0, {0xFA, 0x8C}}, {0xB4C7, {0xFA, 0x8D}}, {0xB4D0, {0xFA, 0x8E}},
	{0xB4DE, {0xFA, 0x8F}}, {0xB5AA, {0xFA, 0x91}}, {0xB5AF, {0xFA, 0x93}},
	{0xB5C4, {0xFA, 0x94}}, {0xB5E8, {0xFA, 0x95}}, {0xB7C2, {0xFA, 0x97}},
	{0xB7E4, {0xFA, 0x98}}, {0xB7E7, {0xFA, 0x9A}}, {0xB7E8, {0xFA, 0x99}},
	{0xB8CE, {0xFA, 0x9E}}, {0xB8E1, {0xFA, 0x9F}}, {0xB8F5, {0xFA, 0xA0}},
	{0xB8F7, {0xFA, 0xA1}}, {0xB8F8, {0xFA, 0xA2}}, {0xB8FC, {0xFA, 0xA3}},
	{0xB9AF, {0xFA, 0xA4}}, {0xB9B7, {0xFA, 0xA5}}, {0xBABE, {0xFA, 0xA6}},
	{0xBADB, {0xFA, 0xA7}}, {0xBAE1, {0xFA, 0xA9}}, {0xBAEB, {0xFA, 0xAB}},
	{0xBBB3, {0xFA, 0xAC}}, {0xBBB8, {0xFA, 0xAD}}, {0xBBCA, {0xFA, 0xAF}},
	{0xBBD0, {0xFA, 0xB2}}, {0xBBDE, {0xFA, 0xB3}}, {0xBBF4, {0xFA, 0xB4}},
	{0xBBF5, {0xFA, 0xB5}}, {0xBBF9, {0xFA, 0xB6}}, {0xBCE4, {0xFA, 0xB7}},
	{0xBCED, {0xFA, 0xB8}}, {0xBCF4, {0xFA, 0x67}}, {0xBCFE, {0xFA, 0xB9}},
	{0xBDC2, {0xFA, 0xBB}}, {0xBDE7, {0xFA, 0xBC}}, {0xBDF0, {0xFA, 0xBE}},
	{0xBEAC, {0xFA, 0xC0}}, {0xBEB0, {0xFA, 0xBF}}, {0xBEB3, {0xFA, 0xC2}},
	{0xBEBD, {0xFA, 0xC3}}, {0xBEC9, {0xFA, 0xC5}}, {0xBECD, {0xFA, 0xC4}},
	{0xBEE4, {0xFA, 0xC6}}, {0xBFA8, {0xFA, 0xC7}}, {0xBFC9, {0xFA, 0xC8}},
	{0xC0C4, {0xFA, 0xC9}}, {0xC0E4, {0xFA, 0xCA}}, {0xC0F4, {0xFA, 0xCB}},
	{0xC1A6, {0xFA, 0xCC}}, {0xC1F5, {0xFA, 0xCE}}, {0xC1F8, {0xFA, 0xD1}},
	{0xC1FC, {0xFA, 0xCF}}, {0xC2A1, {0xFA, 0xD3}}, {0xC2A5, {0xFA, 0xD4}},
	{0xC2AB, {0xFA, 0xD2}}, {0xC2AD, {0xFA, 0x63}}, {0xC2B8, {0xFA, 0xD6}},
	{0xC2BA, {0xFA, 0xD7}}, {0xC2BF, {0xFA, 0x66}}, {0xC2C4, {0xFA, 0xD9}},
	{0xC2D2, {0xFA, 0xDA}}, {0xC2D7, {0xFA, 0xDB}}, {0xC2DB, {0xFA, 0xDC}},
	{0xC2DE, {0xFA, 0xDD}}, {0xC2ED, {0xFA, 0xDE}}, {0xC2F0, {0xFA, 0xDF}},
	{0xC3A1, {0xFA, 0xE1}}, {0xC3B5, {0xFA, 0xE2}}, {0xC3B9, {0xFA, 0xE4}},
	{0xC3C9, {0xFA, 0xE3}}, {0xC3D8, {0xFA, 0xE6}}, {0xC3FC, {0xFA, 0x64}},
	{0xC3FE, {0xFA, 0xE7}}, {0xC4CC, {0xFA, 0xE9}}, {0xC4D9, {0xFA, 0xEB}},
	{0xC4EA, {0xFA, 0xEC}}, {0xC4FD, {0xFA, 0xED}}, {0xC5A7, {0xFA, 0xEF}},
	{0xC5B5, {0xFA, 0xF0}}, {0xC5B6, {0xFA, 0xF1}}, {0xC5D5, {0xFA, 0xF3}},
	{0xC6B8, {0xFA, 0xF4}}, {0xC6D7, {0xFA, 0xF5}}, {0xC6E0, {0xFA, 0xF6}},
	{0xC6E3, {0xFA, 0xF8}}, {0xC6EA, {0xFA, 0xF7}}, {0xC7A1, {0xFA, 0xF9}},
	{0xC7AB, {0xFA, 0xFA}}, {0xC7C3, {0xFA, 0xFC}}, {0xC7C7, {0xFA, 0xFB}},
	{0xC7CB, {0xFB, 0x40}}, {0xC7CF, {0xFB, 0x41}}, {0xC7D9, {0xFB, 0x42}},
	{0xC7E6, {0xFB, 0x45}}, {0xC7EB, {0xFB, 0x48}}, {0xC7EE, {0xFB, 0x46}},
	{0xC7F0, {0xFB, 0x49}}, {0xC7FC, {0xFB, 0x47}}, {0xC8B1, {0xFB, 0x4A}},
	{0xC8E5, {0xFB, 0x4B}}, {0xC8F8, {0xFB, 0x4C}}, {0xC9A6, {0xFB, 0x4D}},
	{0xC9AB, {0xFB, 0x4E}}, {0xC9AD, {0xFB, 0x4F}}, {0xC9CA, {0xFB, 0x51}},
	{0xC9D3, {0xFB, 0x52}}, {0xC9E3, {0xFB, 0x54}}, {0xC9E9, {0xFB, 0x53}},
	{0xC9F4, {0xFB, 0x56}}, {0xC9F5, {0xFB, 0x57}}, {0xC9FC, {0xFB, 0x55}},
	{0xCAB3, {0xFB, 0x59}}, {0xCABD, {0xFB, 0x5A}}, {0xCAEF, {0xFB, 0x5B}},
	{0xCAF1, {0xFB, 0x5C}}, {0xCBAE, {0xFB, 0x5D}}, {0xCBCA, {0xFB, 0x5F}},
	{0xCBE6, {0xFB, 0x60}}, {0xCBEA, {0xFB, 0x61}}, {0xCBEE, {0xFB, 0x64}},
	{0xCBF0, {0xFB, 0x62}}, {0xCBF4, {0xFB, 0x63}}, {0xCBF9, {0xFB, 0x66}},
	{0xCCA5, {0xFB, 0x65}}, {0xCCAB, {0xFB, 0x67}}, {0xCCAD, {0xFB, 0x69}},
	{0xCCAE, {0xFB, 0x68}}, {0xCCB2, {0xFB, 0x6A}}, {0xCCC2, {0xFB, 0x6B}},
	{0xCCD0, {0xFB, 0x6C}}, {0xCCD9, {0xFB, 0x6D}}, {0xCDAA, {0xFA, 0xA8}},
	{0xCDBB, {0xFB, 0x6F}}, {0xCEBA, {0xFB, 0x73}}, {0xCEBB, {0xFB, 0x71}},
	{0xCEC3, {0xFB, 0x74}}, {0xCEF2, {0xFB, 0x76}}, {0xCFD5, {0xFB, 0x78}},
	{0xCFE2, {0xFB, 0x79}}, {0xCFE9, {0xFB, 0x7A}}, {0xCFED, {0xFB, 0x7B}},
	{0xD0E5, {0xFB, 0x81}}, {0xD0E9, {0xFB, 0x83}}, {0xD1E8, {0xFB, 0x84}},
	{0xD1EC, {0xFB, 0x87}}, {0xD2BB, {0xFB, 0x88}}, {0xD3E1, {0xFB, 0x8A}},
	{0xD3E8, {0xFB, 0x8B}}, {0xD4A7, {0xFB, 0x8C}}, {0xD4D4, {0xFB, 0x8F}},
	{0xD4E3, {0xFA, 0x5C}}, {0xD4F2, {0xFB, 0x90}}, {0xD5AE, {0xFB, 0x91}},
	{0xD7DE, {0xFB, 0x93}}, {0xD8A2, {0xFB, 0x95}}, {0xD8B7, {0xFB, 0x96}},
	{0xD8C1, {0xFB, 0x97}}, {0xD8D1, {0xFB, 0x98}}, {0xD8F4, {0xFB, 0x99}},
	{0xD9A1, {0xFA, 0x60}}, {0xD9C6, {0xFB, 0x9A}}, {0xD9C8, {0xFB, 0x9B}},
	{0xD9D1, {0xFB, 0x9C}}, {0xDCD3, {0xFB, 0xA2}}, {0xDCDF, {0xFA, 0x5D}},
	{0xDDC8, {0xFB, 0xA3}}, {0xDDD4, {0xFB, 0xA4}}, {0xDDEA, {0xFB, 0xA5}},
	{0xDDFA, {0xFB, 0xA6}}, {0xDEA4, {0xFB, 0xA7}}, {0xDEB0, {0xFB, 0xA8}},
	{0xDEB5, {0xFB, 0xAA}}, {0xDECB, {0xFB, 0xAB}}, {0xDFB9, {0xFB, 0xAD}},
	{0xDFC3, {0xFB, 0xAF}}, {0xE0D9, {0xFB, 0xB2}}, {0xE1E2, {0xFB, 0xB5}},
	{0xE2C7, {0xFB, 0xB9}}, {0xE3A6, {0xFB, 0xBB}}, {0xE3A8, {0xFB, 0xBA}},
	{0xE3A9, {0xFB, 0xBC}}, {0xE3AA, {0xFB, 0xBF}}, {0xE3AB, {0xFB, 0xC0}},
	{0xE3AF, {0xFB, 0xBD}}, {0xE3B0, {0xFB, 0xBE}}, {0xE3BC, {0xFB, 0xC1}},
	{0xE3BF, {0xFB, 0xC3}}, {0xE3C1, {0xFB, 0xC2}}, {0xE3D4, {0xFB, 0xCA}},
	{0xE3D5, {0xFB, 0xC4}}, {0xE3D6, {0xFB, 0xC6}}, {0xE3D8, {0xFB, 0xC5}},
	{0xE3DF, {0xFB, 0xC7}}, {0xE3E1, {0xFB, 0xC9}}, {0xE3E3, {0xFB, 0xC8}},
	{0xE3E9, {0xFB, 0xCB}}, {0xE3F1, {0xFB, 0xCD}}, {0xE3F2, {0xFB, 0xCE}},
	{0xE3F8, {0xFA, 0x5F}}, {0xE4A6, {0xFB, 0xCC}}, {0xE4BE, {0xFB, 0xD2}},
	{0xE4BF, {0xFB, 0xD6}}, {0xE4C0, {0xFB, 0xD4}}, {0xE4C1, {0xFB, 0xD0}},
	{0xE4C3, {0xFB, 0xD1}}, {0xE4C7, {0xFB, 0xD5}}, {0xE4CB, {0xFB, 0xCF}},
	{0xE4D0, {0xFA, 0x65}}, {0xE4D1, {0xFB, 0xD9}}, {0xE4D2, {0xFB, 0xDC}},
	{0xE4D4, {0xFB, 0xDE}}, {0xE4DB, {0xFB, 0xDD}}, {0xE4DC, {0xFB, 0xDB}},
	{0xE4DE, {0xFB, 0xD8}}, {0xE4E0, {0xFB, 0xD7}}, {0xE4E9, {0xFA, 0x5E}},
	{0xE4EF, {0xFB, 0xE0}}, {0xE4FA, {0xFB, 0xDF}}, {0xE5B3, {0xFB, 0xE1}},
	{0xE5BF, {0xFB, 0xE2}}, {0xE5C9, {0xFB, 0xE3}}, {0xE5D0, {0xFB, 0xE4}},
	{0xE5E2, {0xFB, 0xE5}}, {0xE5EA, {0xFB, 0xE6}}, {0xE5EB, {0xFB, 0xE7}},
	{0xE6E8, {0xFB, 0xEB}}, {0xE6EF, {0xFB, 0xEC}}, {0xE7AC, {0xFB, 0xED}},
	{0xE7AE, {0xFB, 0xEF}}, {0xE7B1, {0xFB, 0xF1}}, {0xE7B2, {0xFB, 0xF3}},
	{0xE8B1, {0xFB, 0xF4}}, {0xE8B6, {0xFB, 0xF5}}, {0xE8DD, {0xFB, 0xF8}},
	{0xE9D1, {0xFB, 0xFB}}, {0xE9ED, {0xFC, 0x40}}, {0xEACD, {0xFC, 0x41}},
	{0xEADB, {0xFC, 0x43}}, {0xEAE6, {0xFC, 0x44}}, {0xEAEA, {0xFC, 0x45}},
	{0xEBA5, {0xFC, 0x46}}, {0xEBFA, {0xFC, 0x48}}, {0xEBFB, {0xFC, 0x47}},
	{0xECD6, {0xFC, 0x4A}},
};
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* EUC to SJIS IBM extended characters map (G3 Upper block) */
static const unsigned char euc2sjisibm_g3upper_map[][2] = {
	{0xFA, 0x40}, {0xFA, 0x41}, {0xFA, 0x42}, {0xFA, 0x43}, {0xFA, 0x44},
	{0xFA, 0x45}, {0xFA, 0x46}, {0xFA, 0x47}, {0xFA, 0x48}, {0xFA, 0x49},
	{0xFA, 0x4A}, {0xFA, 0x4B}, {0xFA, 0x4C}, {0xFA, 0x4D}, {0xFA, 0x4E},
	{0xFA, 0x4F}, {0xFA, 0x50}, {0xFA, 0x51}, {0xFA, 0x52}, {0xFA, 0x53},
	{0xFA, 0x56}, {0xFA, 0x57}, {0xFA, 0x58}, {0xFA, 0x59}, {0xFA, 0x5A},
	{0xFA, 0x62}, {0xFA, 0x6A}, {0xFA, 0x7C}, {0xFA, 0x83}, {0xFA, 0x8A},
	{0xFA, 0x8B}, {0xFA, 0x90}, {0xFA, 0x92}, {0xFA, 0x96}, {0xFA, 0x9B},
	{0xFA, 0x9C}, {0xFA, 0x9D}, {0xFA, 0xAA}, {0xFA, 0xAE}, {0xFA, 0xB0},
	{0xFA, 0xB1}, {0xFA, 0xBA}, {0xFA, 0xBD}, {0xFA, 0xC1}, {0xFA, 0xCD},
	{0xFA, 0xD0}, {0xFA, 0xD5}, {0xFA, 0xD8}, {0xFA, 0xE0}, {0xFA, 0xE5},
	{0xFA, 0xE8}, {0xFA, 0xEA}, {0xFA, 0xEE}, {0xFA, 0xF2}, {0xFB, 0x43},
	{0xFB, 0x44}, {0xFB, 0x50}, {0xFB, 0x58}, {0xFB, 0x5E}, {0xFB, 0x6E},
	{0xFB, 0x70}, {0xFB, 0x72}, {0xFB, 0x75}, {0xFB, 0x7C}, {0xFB, 0x7D},
	{0xFB, 0x7E}, {0xFB, 0x80}, {0xFB, 0x82}, {0xFB, 0x85}, {0xFB, 0x86},
	{0xFB, 0x89}, {0xFB, 0x8D}, {0xFB, 0x8E}, {0xFB, 0x92}, {0xFB, 0x94},
	{0xFB, 0x9D}, {0xFB, 0x9E}, {0xFB, 0x9F}, {0xFB, 0xA0}, {0xFB, 0xA1},
	{0xFB, 0xA9}, {0xFB, 0xAC}, {0xFB, 0xAE}, {0xFB, 0xB0}, {0xFB, 0xB1},
	{0xFB, 0xB3}, {0xFB, 0xB4}, {0xFB, 0xB6}, {0xFB, 0xB7}, {0xFB, 0xB8},
	{0xFB, 0xD3}, {0xFB, 0xDA}, {0xFB, 0xE8}, {0xFB, 0xE9}, {0xFB, 0xEA},
	{0xFB, 0xEE}, {0xFB, 0xF0}, {0xFB, 0xF2}, {0xFB, 0xF6}, {0xFB, 0xF7},
	{0xFB, 0xF9}, {0xFB, 0xFA}, {0xFB, 0xFC}, {0xFC, 0x42}, {0xFC, 0x49},
	{0xFC, 0x4B},
};
#endif

#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
static wchar_t c2u_jisx0208[] = {
/* 0xA1A1 */	 0x3000, 0x3001, 0x3002, 0xFF0C, 0xFF0E, 0x30FB, 0xFF1A,
	 0xFF1B, 0xFF1F, 0xFF01, 0x309B, 0x309C, 0x00B4, 0xFF40, 0x00A8,
	 0xFF3E, 0xFFE3, 0xFF3F, 0x30FD, 0x30FE, 0x309D, 0x309E, 0x3003,
	 0x4EDD, 0x3005, 0x3006, 0x3007, 0x30FC, 0x2015, 0x2010, 0xFF0F,
	 0xFF3C, 0xFF5E, 0x2225, 0xFF5C, 0x2026, 0x2025, 0x2018, 0x2019,
	 0x201C, 0x201D, 0xFF08, 0xFF09, 0x3014, 0x3015, 0xFF3B, 0xFF3D,
	 0xFF5B, 0xFF5D, 0x3008, 0x3009, 0x300A, 0x300B, 0x300C, 0x300D,
	 0x300E, 0x300F, 0x3010, 0x3011, 0xFF0B, 0xFF0D, 0x00B1, 0x00D7,
	 0x00F7, 0xFF1D, 0x2260, 0xFF1C, 0xFF1E, 0x2266, 0x2267, 0x221E,
	 0x2234, 0x2642, 0x2640, 0x00B0, 0x2032, 0x2033, 0x2103, 0xFFE5,
	 0xFF04, 0xFFE0, 0xFFE1, 0xFF05, 0xFF03, 0xFF06, 0xFF0A, 0xFF20,
	 0x00A7, 0x2606, 0x2605, 0x25CB, 0x25CF, 0x25CE, 0x25C7,
/* 0xA2A1 */	 0x25C6, 0x25A1, 0x25A0, 0x25B3, 0x25B2, 0x25BD, 0x25BC,
	 0x203B, 0x3012, 0x2192, 0x2190, 0x2191, 0x2193, 0x3013, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x2208, 0x220B, 0x2286, 0x2287, 0x2282, 0x2283,
	 0x222A, 0x2229, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x2227, 0x2228, 0xFFE2, 0x21D2, 0x21D4, 0x2200,
	 0x2203, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x2220, 0x22A5, 0x2312, 0x2202,
	 0x2207, 0x2261, 0x2252, 0x226A, 0x226B, 0x221A, 0x223D, 0x221D,
	 0x2235, 0x222B, 0x222C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x212B, 0x2030, 0x266F, 0x266D, 0x266A, 0x2020,
	 0x2021, 0x00B6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x25EF,
/* 0xA3A1 */	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFF10, 0xFF11, 0xFF12, 0xFF13, 0xFF14, 0xFF15, 0xFF16, 0xFF17,
	 0xFF18, 0xFF19, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFF21, 0xFF22, 0xFF23, 0xFF24, 0xFF25, 0xFF26, 0xFF27,
	 0xFF28, 0xFF29, 0xFF2A, 0xFF2B, 0xFF2C, 0xFF2D, 0xFF2E, 0xFF2F,
	 0xFF30, 0xFF31, 0xFF32, 0xFF33, 0xFF34, 0xFF35, 0xFF36, 0xFF37,
	 0xFF38, 0xFF39, 0xFF3A, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFF41, 0xFF42, 0xFF43, 0xFF44, 0xFF45, 0xFF46, 0xFF47,
	 0xFF48, 0xFF49, 0xFF4A, 0xFF4B, 0xFF4C, 0xFF4D, 0xFF4E, 0xFF4F,
	 0xFF50, 0xFF51, 0xFF52, 0xFF53, 0xFF54, 0xFF55, 0xFF56, 0xFF57,
	 0xFF58, 0xFF59, 0xFF5A, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA4A1 */	 0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047,
	 0x3048, 0x3049, 0x304A, 0x304B, 0x304C, 0x304D, 0x304E, 0x304F,
	 0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057,
	 0x3058, 0x3059, 0x305A, 0x305B, 0x305C, 0x305D, 0x305E, 0x305F,
	 0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067,
	 0x3068, 0x3069, 0x306A, 0x306B, 0x306C, 0x306D, 0x306E, 0x306F,
	 0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077,
	 0x3078, 0x3079, 0x307A, 0x307B, 0x307C, 0x307D, 0x307E, 0x307F,
	 0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087,
	 0x3088, 0x3089, 0x308A, 0x308B, 0x308C, 0x308D, 0x308E, 0x308F,
	 0x3090, 0x3091, 0x3092, 0x3093, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA5A1 */	 0x30A1, 0x30A2, 0x30A3, 0x30A4, 0x30A5, 0x30A6, 0x30A7,
	 0x30A8, 0x30A9, 0x30AA, 0x30AB, 0x30AC, 0x30AD, 0x30AE, 0x30AF,
	 0x30B0, 0x30B1, 0x30B2, 0x30B3, 0x30B4, 0x30B5, 0x30B6, 0x30B7,
	 0x30B8, 0x30B9, 0x30BA, 0x30BB, 0x30BC, 0x30BD, 0x30BE, 0x30BF,
	 0x30C0, 0x30C1, 0x30C2, 0x30C3, 0x30C4, 0x30C5, 0x30C6, 0x30C7,
	 0x30C8, 0x30C9, 0x30CA, 0x30CB, 0x30CC, 0x30CD, 0x30CE, 0x30CF,
	 0x30D0, 0x30D1, 0x30D2, 0x30D3, 0x30D4, 0x30D5, 0x30D6, 0x30D7,
	 0x30D8, 0x30D9, 0x30DA, 0x30DB, 0x30DC, 0x30DD, 0x30DE, 0x30DF,
	 0x30E0, 0x30E1, 0x30E2, 0x30E3, 0x30E4, 0x30E5, 0x30E6, 0x30E7,
	 0x30E8, 0x30E9, 0x30EA, 0x30EB, 0x30EC, 0x30ED, 0x30EE, 0x30EF,
	 0x30F0, 0x30F1, 0x30F2, 0x30F3, 0x30F4, 0x30F5, 0x30F6, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA6A1 */	 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
	 0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F,
	 0x03A0, 0x03A1, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8,
	 0x03A9, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7,
	 0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
	 0x03C0, 0x03C1, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 0x03C8,
	 0x03C9, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA7A1 */	 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401,
	 0x0416, 0x0417, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D,
	 0x041E, 0x041F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425,
	 0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D,
	 0x042E, 0x042F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451,
	 0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D,
	 0x043E, 0x043F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445,
	 0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D,
	 0x044E, 0x044F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA8A1 */	 0x2500, 0x2502, 0x250C, 0x2510, 0x2518, 0x2514, 0x251C,
	 0x252C, 0x2524, 0x2534, 0x253C, 0x2501, 0x2503, 0x250F, 0x2513,
	 0x251B, 0x2517, 0x2523, 0x2533, 0x252B, 0x253B, 0x254B, 0x2520,
	 0x252F, 0x2528, 0x2537, 0x253F, 0x251D, 0x2530, 0x2525, 0x2538,
	 0x2542, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xB0A1 */	 0x4E9C, 0x5516, 0x5A03, 0x963F, 0x54C0, 0x611B, 0x6328,
	 0x59F6, 0x9022, 0x8475, 0x831C, 0x7A50, 0x60AA, 0x63E1, 0x6E25,
	 0x65ED, 0x8466, 0x82A6, 0x9BF5, 0x6893, 0x5727, 0x65A1, 0x6271,
	 0x5B9B, 0x59D0, 0x867B, 0x98F4, 0x7D62, 0x7DBE, 0x9B8E, 0x6216,
	 0x7C9F, 0x88B7, 0x5B89, 0x5EB5, 0x6309, 0x6697, 0x6848, 0x95C7,
	 0x978D, 0x674F, 0x4EE5, 0x4F0A, 0x4F4D, 0x4F9D, 0x5049, 0x56F2,
	 0x5937, 0x59D4, 0x5A01, 0x5C09, 0x60DF, 0x610F, 0x6170, 0x6613,
	 0x6905, 0x70BA, 0x754F, 0x7570, 0x79FB, 0x7DAD, 0x7DEF, 0x80C3,
	 0x840E, 0x8863, 0x8B02, 0x9055, 0x907A, 0x533B, 0x4E95, 0x4EA5,
	 0x57DF, 0x80B2, 0x90C1, 0x78EF, 0x4E00, 0x58F1, 0x6EA2, 0x9038,
	 0x7A32, 0x8328, 0x828B, 0x9C2F, 0x5141, 0x5370, 0x54BD, 0x54E1,
	 0x56E0, 0x59FB, 0x5F15, 0x98F2, 0x6DEB, 0x80E4, 0x852D,
/* 0xB1A1 */	 0x9662, 0x9670, 0x96A0, 0x97FB, 0x540B, 0x53F3, 0x5B87,
	 0x70CF, 0x7FBD, 0x8FC2, 0x96E8, 0x536F, 0x9D5C, 0x7ABA, 0x4E11,
	 0x7893, 0x81FC, 0x6E26, 0x5618, 0x5504, 0x6B1D, 0x851A, 0x9C3B,
	 0x59E5, 0x53A9, 0x6D66, 0x74DC, 0x958F, 0x5642, 0x4E91, 0x904B,
	 0x96F2, 0x834F, 0x990C, 0x53E1, 0x55B6, 0x5B30, 0x5F71, 0x6620,
	 0x66F3, 0x6804, 0x6C38, 0x6CF3, 0x6D29, 0x745B, 0x76C8, 0x7A4E,
	 0x9834, 0x82F1, 0x885B, 0x8A60, 0x92ED, 0x6DB2, 0x75AB, 0x76CA,
	 0x99C5, 0x60A6, 0x8B01, 0x8D8A, 0x95B2, 0x698E, 0x53AD, 0x5186,
	 0x5712, 0x5830, 0x5944, 0x5BB4, 0x5EF6, 0x6028, 0x63A9, 0x63F4,
	 0x6CBF, 0x6F14, 0x708E, 0x7114, 0x7159, 0x71D5, 0x733F, 0x7E01,
	 0x8276, 0x82D1, 0x8597, 0x9060, 0x925B, 0x9D1B, 0x5869, 0x65BC,
	 0x6C5A, 0x7525, 0x51F9, 0x592E, 0x5965, 0x5F80, 0x5FDC,
/* 0xB2A1 */	 0x62BC, 0x65FA, 0x6A2A, 0x6B27, 0x6BB4, 0x738B, 0x7FC1,
	 0x8956, 0x9D2C, 0x9D0E, 0x9EC4, 0x5CA1, 0x6C96, 0x837B, 0x5104,
	 0x5C4B, 0x61B6, 0x81C6, 0x6876, 0x7261, 0x4E59, 0x4FFA, 0x5378,
	 0x6069, 0x6E29, 0x7A4F, 0x97F3, 0x4E0B, 0x5316, 0x4EEE, 0x4F55,
	 0x4F3D, 0x4FA1, 0x4F73, 0x52A0, 0x53EF, 0x5609, 0x590F, 0x5AC1,
	 0x5BB6, 0x5BE1, 0x79D1, 0x6687, 0x679C, 0x67B6, 0x6B4C, 0x6CB3,
	 0x706B, 0x73C2, 0x798D, 0x79BE, 0x7A3C, 0x7B87, 0x82B1, 0x82DB,
	 0x8304, 0x8377, 0x83EF, 0x83D3, 0x8766, 0x8AB2, 0x5629, 0x8CA8,
	 0x8FE6, 0x904E, 0x971E, 0x868A, 0x4FC4, 0x5CE8, 0x6211, 0x7259,
	 0x753B, 0x81E5, 0x82BD, 0x86FE, 0x8CC0, 0x96C5, 0x9913, 0x99D5,
	 0x4ECB, 0x4F1A, 0x89E3, 0x56DE, 0x584A, 0x58CA, 0x5EFB, 0x5FEB,
	 0x602A, 0x6094, 0x6062, 0x61D0, 0x6212, 0x62D0, 0x6539,
/* 0xB3A1 */	 0x9B41, 0x6666, 0x68B0, 0x6D77, 0x7070, 0x754C, 0x7686,
	 0x7D75, 0x82A5, 0x87F9, 0x958B, 0x968E, 0x8C9D, 0x51F1, 0x52BE,
	 0x5916, 0x54B3, 0x5BB3, 0x5D16, 0x6168, 0x6982, 0x6DAF, 0x788D,
	 0x84CB, 0x8857, 0x8A72, 0x93A7, 0x9AB8, 0x6D6C, 0x99A8, 0x86D9,
	 0x57A3, 0x67FF, 0x86CE, 0x920E, 0x5283, 0x5687, 0x5404, 0x5ED3,
	 0x62E1, 0x64B9, 0x683C, 0x6838, 0x6BBB, 0x7372, 0x78BA, 0x7A6B,
	 0x899A, 0x89D2, 0x8D6B, 0x8F03, 0x90ED, 0x95A3, 0x9694, 0x9769,
	 0x5B66, 0x5CB3, 0x697D, 0x984D, 0x984E, 0x639B, 0x7B20, 0x6A2B,
	 0x6A7F, 0x68B6, 0x9C0D, 0x6F5F, 0x5272, 0x559D, 0x6070, 0x62EC,
	 0x6D3B, 0x6E07, 0x6ED1, 0x845B, 0x8910, 0x8F44, 0x4E14, 0x9C39,
	 0x53F6, 0x691B, 0x6A3A, 0x9784, 0x682A, 0x515C, 0x7AC3, 0x84B2,
	 0x91DC, 0x938C, 0x565B, 0x9D28, 0x6822, 0x8305, 0x8431,
/* 0xB4A1 */	 0x7CA5, 0x5208, 0x82C5, 0x74E6, 0x4E7E, 0x4F83, 0x51A0,
	 0x5BD2, 0x520A, 0x52D8, 0x52E7, 0x5DFB, 0x559A, 0x582A, 0x59E6,
	 0x5B8C, 0x5B98, 0x5BDB, 0x5E72, 0x5E79, 0x60A3, 0x611F, 0x6163,
	 0x61BE, 0x63DB, 0x6562, 0x67D1, 0x6853, 0x68FA, 0x6B3E, 0x6B53,
	 0x6C57, 0x6F22, 0x6F97, 0x6F45, 0x74B0, 0x7518, 0x76E3, 0x770B,
	 0x7AFF, 0x7BA1, 0x7C21, 0x7DE9, 0x7F36, 0x7FF0, 0x809D, 0x8266,
	 0x839E, 0x89B3, 0x8ACC, 0x8CAB, 0x9084, 0x9451, 0x9593, 0x9591,
	 0x95A2, 0x9665, 0x97D3, 0x9928, 0x8218, 0x4E38, 0x542B, 0x5CB8,
	 0x5DCC, 0x73A9, 0x764C, 0x773C, 0x5CA9, 0x7FEB, 0x8D0B, 0x96C1,
	 0x9811, 0x9854, 0x9858, 0x4F01, 0x4F0E, 0x5371, 0x559C, 0x5668,
	 0x57FA, 0x5947, 0x5B09, 0x5BC4, 0x5C90, 0x5E0C, 0x5E7E, 0x5FCC,
	 0x63EE, 0x673A, 0x65D7, 0x65E2, 0x671F, 0x68CB, 0x68C4,
/* 0xB5A1 */	 0x6A5F, 0x5E30, 0x6BC5, 0x6C17, 0x6C7D, 0x757F, 0x7948,
	 0x5B63, 0x7A00, 0x7D00, 0x5FBD, 0x898F, 0x8A18, 0x8CB4, 0x8D77,
	 0x8ECC, 0x8F1D, 0x98E2, 0x9A0E, 0x9B3C, 0x4E80, 0x507D, 0x5100,
	 0x5993, 0x5B9C, 0x622F, 0x6280, 0x64EC, 0x6B3A, 0x72A0, 0x7591,
	 0x7947, 0x7FA9, 0x87FB, 0x8ABC, 0x8B70, 0x63AC, 0x83CA, 0x97A0,
	 0x5409, 0x5403, 0x55AB, 0x6854, 0x6A58, 0x8A70, 0x7827, 0x6775,
	 0x9ECD, 0x5374, 0x5BA2, 0x811A, 0x8650, 0x9006, 0x4E18, 0x4E45,
	 0x4EC7, 0x4F11, 0x53CA, 0x5438, 0x5BAE, 0x5F13, 0x6025, 0x6551,
	 0x673D, 0x6C42, 0x6C72, 0x6CE3, 0x7078, 0x7403, 0x7A76, 0x7AAE,
	 0x7B08, 0x7D1A, 0x7CFE, 0x7D66, 0x65E7, 0x725B, 0x53BB, 0x5C45,
	 0x5DE8, 0x62D2, 0x62E0, 0x6319, 0x6E20, 0x865A, 0x8A31, 0x8DDD,
	 0x92F8, 0x6F01, 0x79A6, 0x9B5A, 0x4EA8, 0x4EAB, 0x4EAC,
/* 0xB6A1 */	 0x4F9B, 0x4FA0, 0x50D1, 0x5147, 0x7AF6, 0x5171, 0x51F6,
	 0x5354, 0x5321, 0x537F, 0x53EB, 0x55AC, 0x5883, 0x5CE1, 0x5F37,
	 0x5F4A, 0x602F, 0x6050, 0x606D, 0x631F, 0x6559, 0x6A4B, 0x6CC1,
	 0x72C2, 0x72ED, 0x77EF, 0x80F8, 0x8105, 0x8208, 0x854E, 0x90F7,
	 0x93E1, 0x97FF, 0x9957, 0x9A5A, 0x4EF0, 0x51DD, 0x5C2D, 0x6681,
	 0x696D, 0x5C40, 0x66F2, 0x6975, 0x7389, 0x6850, 0x7C81, 0x50C5,
	 0x52E4, 0x5747, 0x5DFE, 0x9326, 0x65A4, 0x6B23, 0x6B3D, 0x7434,
	 0x7981, 0x79BD, 0x7B4B, 0x7DCA, 0x82B9, 0x83CC, 0x887F, 0x895F,
	 0x8B39, 0x8FD1, 0x91D1, 0x541F, 0x9280, 0x4E5D, 0x5036, 0x53E5,
	 0x533A, 0x72D7, 0x7396, 0x77E9, 0x82E6, 0x8EAF, 0x99C6, 0x99C8,
	 0x99D2, 0x5177, 0x611A, 0x865E, 0x55B0, 0x7A7A, 0x5076, 0x5BD3,
	 0x9047, 0x9685, 0x4E32, 0x6ADB, 0x91E7, 0x5C51, 0x5C48,
/* 0xB7A1 */	 0x6398, 0x7A9F, 0x6C93, 0x9774, 0x8F61, 0x7AAA, 0x718A,
	 0x9688, 0x7C82, 0x6817, 0x7E70, 0x6851, 0x936C, 0x52F2, 0x541B,
	 0x85AB, 0x8A13, 0x7FA4, 0x8ECD, 0x90E1, 0x5366, 0x8888, 0x7941,
	 0x4FC2, 0x50BE, 0x5211, 0x5144, 0x5553, 0x572D, 0x73EA, 0x578B,
	 0x5951, 0x5F62, 0x5F84, 0x6075, 0x6176, 0x6167, 0x61A9, 0x63B2,
	 0x643A, 0x656C, 0x666F, 0x6842, 0x6E13, 0x7566, 0x7A3D, 0x7CFB,
	 0x7D4C, 0x7D99, 0x7E4B, 0x7F6B, 0x830E, 0x834A, 0x86CD, 0x8A08,
	 0x8A63, 0x8B66, 0x8EFD, 0x981A, 0x9D8F, 0x82B8, 0x8FCE, 0x9BE8,
	 0x5287, 0x621F, 0x6483, 0x6FC0, 0x9699, 0x6841, 0x5091, 0x6B20,
	 0x6C7A, 0x6F54, 0x7A74, 0x7D50, 0x8840, 0x8A23, 0x6708, 0x4EF6,
	 0x5039, 0x5026, 0x5065, 0x517C, 0x5238, 0x5263, 0x55A7, 0x570F,
	 0x5805, 0x5ACC, 0x5EFA, 0x61B2, 0x61F8, 0x62F3, 0x6372,
/* 0xB8A1 */	 0x691C, 0x6A29, 0x727D, 0x72AC, 0x732E, 0x7814, 0x786F,
	 0x7D79, 0x770C, 0x80A9, 0x898B, 0x8B19, 0x8CE2, 0x8ED2, 0x9063,
	 0x9375, 0x967A, 0x9855, 0x9A13, 0x9E78, 0x5143, 0x539F, 0x53B3,
	 0x5E7B, 0x5F26, 0x6E1B, 0x6E90, 0x7384, 0x73FE, 0x7D43, 0x8237,
	 0x8A00, 0x8AFA, 0x9650, 0x4E4E, 0x500B, 0x53E4, 0x547C, 0x56FA,
	 0x59D1, 0x5B64, 0x5DF1, 0x5EAB, 0x5F27, 0x6238, 0x6545, 0x67AF,
	 0x6E56, 0x72D0, 0x7CCA, 0x88B4, 0x80A1, 0x80E1, 0x83F0, 0x864E,
	 0x8A87, 0x8DE8, 0x9237, 0x96C7, 0x9867, 0x9F13, 0x4E94, 0x4E92,
	 0x4F0D, 0x5348, 0x5449, 0x543E, 0x5A2F, 0x5F8C, 0x5FA1, 0x609F,
	 0x68A7, 0x6A8E, 0x745A, 0x7881, 0x8A9E, 0x8AA4, 0x8B77, 0x9190,
	 0x4E5E, 0x9BC9, 0x4EA4, 0x4F7C, 0x4FAF, 0x5019, 0x5016, 0x5149,
	 0x516C, 0x529F, 0x52B9, 0x52FE, 0x539A, 0x53E3, 0x5411,
/* 0xB9A1 */	 0x540E, 0x5589, 0x5751, 0x57A2, 0x597D, 0x5B54, 0x5B5D,
	 0x5B8F, 0x5DE5, 0x5DE7, 0x5DF7, 0x5E78, 0x5E83, 0x5E9A, 0x5EB7,
	 0x5F18, 0x6052, 0x614C, 0x6297, 0x62D8, 0x63A7, 0x653B, 0x6602,
	 0x6643, 0x66F4, 0x676D, 0x6821, 0x6897, 0x69CB, 0x6C5F, 0x6D2A,
	 0x6D69, 0x6E2F, 0x6E9D, 0x7532, 0x7687, 0x786C, 0x7A3F, 0x7CE0,
	 0x7D05, 0x7D18, 0x7D5E, 0x7DB1, 0x8015, 0x8003, 0x80AF, 0x80B1,
	 0x8154, 0x818F, 0x822A, 0x8352, 0x884C, 0x8861, 0x8B1B, 0x8CA2,
	 0x8CFC, 0x90CA, 0x9175, 0x9271, 0x783F, 0x92FC, 0x95A4, 0x964D,
	 0x9805, 0x9999, 0x9AD8, 0x9D3B, 0x525B, 0x52AB, 0x53F7, 0x5408,
	 0x58D5, 0x62F7, 0x6FE0, 0x8C6A, 0x8F5F, 0x9EB9, 0x514B, 0x523B,
	 0x544A, 0x56FD, 0x7A40, 0x9177, 0x9D60, 0x9ED2, 0x7344, 0x6F09,
	 0x8170, 0x7511, 0x5FFD, 0x60DA, 0x9AA8, 0x72DB, 0x8FBC,
/* 0xBAA1 */	 0x6B64, 0x9803, 0x4ECA, 0x56F0, 0x5764, 0x58BE, 0x5A5A,
	 0x6068, 0x61C7, 0x660F, 0x6606, 0x6839, 0x68B1, 0x6DF7, 0x75D5,
	 0x7D3A, 0x826E, 0x9B42, 0x4E9B, 0x4F50, 0x53C9, 0x5506, 0x5D6F,
	 0x5DE6, 0x5DEE, 0x67FB, 0x6C99, 0x7473, 0x7802, 0x8A50, 0x9396,
	 0x88DF, 0x5750, 0x5EA7, 0x632B, 0x50B5, 0x50AC, 0x518D, 0x6700,
	 0x54C9, 0x585E, 0x59BB, 0x5BB0, 0x5F69, 0x624D, 0x63A1, 0x683D,
	 0x6B73, 0x6E08, 0x707D, 0x91C7, 0x7280, 0x7815, 0x7826, 0x796D,
	 0x658E, 0x7D30, 0x83DC, 0x88C1, 0x8F09, 0x969B, 0x5264, 0x5728,
	 0x6750, 0x7F6A, 0x8CA1, 0x51B4, 0x5742, 0x962A, 0x583A, 0x698A,
	 0x80B4, 0x54B2, 0x5D0E, 0x57FC, 0x7895, 0x9DFA, 0x4F5C, 0x524A,
	 0x548B, 0x643E, 0x6628, 0x6714, 0x67F5, 0x7A84, 0x7B56, 0x7D22,
	 0x932F, 0x685C, 0x9BAD, 0x7B39, 0x5319, 0x518A, 0x5237,
/* 0xBBA1 */	 0x5BDF, 0x62F6, 0x64AE, 0x64E6, 0x672D, 0x6BBA, 0x85A9,
	 0x96D1, 0x7690, 0x9BD6, 0x634C, 0x9306, 0x9BAB, 0x76BF, 0x6652,
	 0x4E09, 0x5098, 0x53C2, 0x5C71, 0x60E8, 0x6492, 0x6563, 0x685F,
	 0x71E6, 0x73CA, 0x7523, 0x7B97, 0x7E82, 0x8695, 0x8B83, 0x8CDB,
	 0x9178, 0x9910, 0x65AC, 0x66AB, 0x6B8B, 0x4ED5, 0x4ED4, 0x4F3A,
	 0x4F7F, 0x523A, 0x53F8, 0x53F2, 0x55E3, 0x56DB, 0x58EB, 0x59CB,
	 0x59C9, 0x59FF, 0x5B50, 0x5C4D, 0x5E02, 0x5E2B, 0x5FD7, 0x601D,
	 0x6307, 0x652F, 0x5B5C, 0x65AF, 0x65BD, 0x65E8, 0x679D, 0x6B62,
	 0x6B7B, 0x6C0F, 0x7345, 0x7949, 0x79C1, 0x7CF8, 0x7D19, 0x7D2B,
	 0x80A2, 0x8102, 0x81F3, 0x8996, 0x8A5E, 0x8A69, 0x8A66, 0x8A8C,
	 0x8AEE, 0x8CC7, 0x8CDC, 0x96CC, 0x98FC, 0x6B6F, 0x4E8B, 0x4F3C,
	 0x4F8D, 0x5150, 0x5B57, 0x5BFA, 0x6148, 0x6301, 0x6642,
/* 0xBCA1 */	 0x6B21, 0x6ECB, 0x6CBB, 0x723E, 0x74BD, 0x75D4, 0x78C1,
	 0x793A, 0x800C, 0x8033, 0x81EA, 0x8494, 0x8F9E, 0x6C50, 0x9E7F,
	 0x5F0F, 0x8B58, 0x9D2B, 0x7AFA, 0x8EF8, 0x5B8D, 0x96EB, 0x4E03,
	 0x53F1, 0x57F7, 0x5931, 0x5AC9, 0x5BA4, 0x6089, 0x6E7F, 0x6F06,
	 0x75BE, 0x8CEA, 0x5B9F, 0x8500, 0x7BE0, 0x5072, 0x67F4, 0x829D,
	 0x5C61, 0x854A, 0x7E1E, 0x820E, 0x5199, 0x5C04, 0x6368, 0x8D66,
	 0x659C, 0x716E, 0x793E, 0x7D17, 0x8005, 0x8B1D, 0x8ECA, 0x906E,
	 0x86C7, 0x90AA, 0x501F, 0x52FA, 0x5C3A, 0x6753, 0x707C, 0x7235,
	 0x914C, 0x91C8, 0x932B, 0x82E5, 0x5BC2, 0x5F31, 0x60F9, 0x4E3B,
	 0x53D6, 0x5B88, 0x624B, 0x6731, 0x6B8A, 0x72E9, 0x73E0, 0x7A2E,
	 0x816B, 0x8DA3, 0x9152, 0x9996, 0x5112, 0x53D7, 0x546A, 0x5BFF,
	 0x6388, 0x6A39, 0x7DAC, 0x9700, 0x56DA, 0x53CE, 0x5468,
/* 0xBDA1 */	 0x5B97, 0x5C31, 0x5DDE, 0x4FEE, 0x6101, 0x62FE, 0x6D32,
	 0x79C0, 0x79CB, 0x7D42, 0x7E4D, 0x7FD2, 0x81ED, 0x821F, 0x8490,
	 0x8846, 0x8972, 0x8B90, 0x8E74, 0x8F2F, 0x9031, 0x914B, 0x916C,
	 0x96C6, 0x919C, 0x4EC0, 0x4F4F, 0x5145, 0x5341, 0x5F93, 0x620E,
	 0x67D4, 0x6C41, 0x6E0B, 0x7363, 0x7E26, 0x91CD, 0x9283, 0x53D4,
	 0x5919, 0x5BBF, 0x6DD1, 0x795D, 0x7E2E, 0x7C9B, 0x587E, 0x719F,
	 0x51FA, 0x8853, 0x8FF0, 0x4FCA, 0x5CFB, 0x6625, 0x77AC, 0x7AE3,
	 0x821C, 0x99FF, 0x51C6, 0x5FAA, 0x65EC, 0x696F, 0x6B89, 0x6DF3,
	 0x6E96, 0x6F64, 0x76FE, 0x7D14, 0x5DE1, 0x9075, 0x9187, 0x9806,
	 0x51E6, 0x521D, 0x6240, 0x6691, 0x66D9, 0x6E1A, 0x5EB6, 0x7DD2,
	 0x7F72, 0x66F8, 0x85AF, 0x85F7, 0x8AF8, 0x52A9, 0x53D9, 0x5973,
	 0x5E8F, 0x5F90, 0x6055, 0x92E4, 0x9664, 0x50B7, 0x511F,
/* 0xBEA1 */	 0x52DD, 0x5320, 0x5347, 0x53EC, 0x54E8, 0x5546, 0x5531,
	 0x5617, 0x5968, 0x59BE, 0x5A3C, 0x5BB5, 0x5C06, 0x5C0F, 0x5C11,
	 0x5C1A, 0x5E84, 0x5E8A, 0x5EE0, 0x5F70, 0x627F, 0x6284, 0x62DB,
	 0x638C, 0x6377, 0x6607, 0x660C, 0x662D, 0x6676, 0x677E, 0x68A2,
	 0x6A1F, 0x6A35, 0x6CBC, 0x6D88, 0x6E09, 0x6E58, 0x713C, 0x7126,
	 0x7167, 0x75C7, 0x7701, 0x785D, 0x7901, 0x7965, 0x79F0, 0x7AE0,
	 0x7B11, 0x7CA7, 0x7D39, 0x8096, 0x83D6, 0x848B, 0x8549, 0x885D,
	 0x88F3, 0x8A1F, 0x8A3C, 0x8A54, 0x8A73, 0x8C61, 0x8CDE, 0x91A4,
	 0x9266, 0x937E, 0x9418, 0x969C, 0x9798, 0x4E0A, 0x4E08, 0x4E1E,
	 0x4E57, 0x5197, 0x5270, 0x57CE, 0x5834, 0x58CC, 0x5B22, 0x5E38,
	 0x60C5, 0x64FE, 0x6761, 0x6756, 0x6D44, 0x72B6, 0x7573, 0x7A63,
	 0x84B8, 0x8B72, 0x91B8, 0x9320, 0x5631, 0x57F4, 0x98FE,
/* 0xBFA1 */	 0x62ED, 0x690D, 0x6B96, 0x71ED, 0x7E54, 0x8077, 0x8272,
	 0x89E6, 0x98DF, 0x8755, 0x8FB1, 0x5C3B, 0x4F38, 0x4FE1, 0x4FB5,
	 0x5507, 0x5A20, 0x5BDD, 0x5BE9, 0x5FC3, 0x614E, 0x632F, 0x65B0,
	 0x664B, 0x68EE, 0x699B, 0x6D78, 0x6DF1, 0x7533, 0x75B9, 0x771F,
	 0x795E, 0x79E6, 0x7D33, 0x81E3, 0x82AF, 0x85AA, 0x89AA, 0x8A3A,
	 0x8EAB, 0x8F9B, 0x9032, 0x91DD, 0x9707, 0x4EBA, 0x4EC1, 0x5203,
	 0x5875, 0x58EC, 0x5C0B, 0x751A, 0x5C3D, 0x814E, 0x8A0A, 0x8FC5,
	 0x9663, 0x976D, 0x7B25, 0x8ACF, 0x9808, 0x9162, 0x56F3, 0x53A8,
	 0x9017, 0x5439, 0x5782, 0x5E25, 0x63A8, 0x6C34, 0x708A, 0x7761,
	 0x7C8B, 0x7FE0, 0x8870, 0x9042, 0x9154, 0x9310, 0x9318, 0x968F,
	 0x745E, 0x9AC4, 0x5D07, 0x5D69, 0x6570, 0x67A2, 0x8DA8, 0x96DB,
	 0x636E, 0x6749, 0x6919, 0x83C5, 0x9817, 0x96C0, 0x88FE,
/* 0xC0A1 */	 0x6F84, 0x647A, 0x5BF8, 0x4E16, 0x702C, 0x755D, 0x662F,
	 0x51C4, 0x5236, 0x52E2, 0x59D3, 0x5F81, 0x6027, 0x6210, 0x653F,
	 0x6574, 0x661F, 0x6674, 0x68F2, 0x6816, 0x6B63, 0x6E05, 0x7272,
	 0x751F, 0x76DB, 0x7CBE, 0x8056, 0x58F0, 0x88FD, 0x897F, 0x8AA0,
	 0x8A93, 0x8ACB, 0x901D, 0x9192, 0x9752, 0x9759, 0x6589, 0x7A0E,
	 0x8106, 0x96BB, 0x5E2D, 0x60DC, 0x621A, 0x65A5, 0x6614, 0x6790,
	 0x77F3, 0x7A4D, 0x7C4D, 0x7E3E, 0x810A, 0x8CAC, 0x8D64, 0x8DE1,
	 0x8E5F, 0x78A9, 0x5207, 0x62D9, 0x63A5, 0x6442, 0x6298, 0x8A2D,
	 0x7A83, 0x7BC0, 0x8AAC, 0x96EA, 0x7D76, 0x820C, 0x8749, 0x4ED9,
	 0x5148, 0x5343, 0x5360, 0x5BA3, 0x5C02, 0x5C16, 0x5DDD, 0x6226,
	 0x6247, 0x64B0, 0x6813, 0x6834, 0x6CC9, 0x6D45, 0x6D17, 0x67D3,
	 0x6F5C, 0x714E, 0x717D, 0x65CB, 0x7A7F, 0x7BAD, 0x7DDA,
/* 0xC1A1 */	 0x7E4A, 0x7FA8, 0x817A, 0x821B, 0x8239, 0x85A6, 0x8A6E,
	 0x8CCE, 0x8DF5, 0x9078, 0x9077, 0x92AD, 0x9291, 0x9583, 0x9BAE,
	 0x524D, 0x5584, 0x6F38, 0x7136, 0x5168, 0x7985, 0x7E55, 0x81B3,
	 0x7CCE, 0x564C, 0x5851, 0x5CA8, 0x63AA, 0x66FE, 0x66FD, 0x695A,
	 0x72D9, 0x758F, 0x758E, 0x790E, 0x7956, 0x79DF, 0x7C97, 0x7D20,
	 0x7D44, 0x8607, 0x8A34, 0x963B, 0x9061, 0x9F20, 0x50E7, 0x5275,
	 0x53CC, 0x53E2, 0x5009, 0x55AA, 0x58EE, 0x594F, 0x723D, 0x5B8B,
	 0x5C64, 0x531D, 0x60E3, 0x60F3, 0x635C, 0x6383, 0x633F, 0x63BB,
	 0x64CD, 0x65E9, 0x66F9, 0x5DE3, 0x69CD, 0x69FD, 0x6F15, 0x71E5,
	 0x4E89, 0x75E9, 0x76F8, 0x7A93, 0x7CDF, 0x7DCF, 0x7D9C, 0x8061,
	 0x8349, 0x8358, 0x846C, 0x84BC, 0x85FB, 0x88C5, 0x8D70, 0x9001,
	 0x906D, 0x9397, 0x971C, 0x9A12, 0x50CF, 0x5897, 0x618E,
/* 0xC2A1 */	 0x81D3, 0x8535, 0x8D08, 0x9020, 0x4FC3, 0x5074, 0x5247,
	 0x5373, 0x606F, 0x6349, 0x675F, 0x6E2C, 0x8DB3, 0x901F, 0x4FD7,
	 0x5C5E, 0x8CCA, 0x65CF, 0x7D9A, 0x5352, 0x8896, 0x5176, 0x63C3,
	 0x5B58, 0x5B6B, 0x5C0A, 0x640D, 0x6751, 0x905C, 0x4ED6, 0x591A,
	 0x592A, 0x6C70, 0x8A51, 0x553E, 0x5815, 0x59A5, 0x60F0, 0x6253,
	 0x67C1, 0x8235, 0x6955, 0x9640, 0x99C4, 0x9A28, 0x4F53, 0x5806,
	 0x5BFE, 0x8010, 0x5CB1, 0x5E2F, 0x5F85, 0x6020, 0x614B, 0x6234,
	 0x66FF, 0x6CF0, 0x6EDE, 0x80CE, 0x817F, 0x82D4, 0x888B, 0x8CB8,
	 0x9000, 0x902E, 0x968A, 0x9EDB, 0x9BDB, 0x4EE3, 0x53F0, 0x5927,
	 0x7B2C, 0x918D, 0x984C, 0x9DF9, 0x6EDD, 0x7027, 0x5353, 0x5544,
	 0x5B85, 0x6258, 0x629E, 0x62D3, 0x6CA2, 0x6FEF, 0x7422, 0x8A17,
	 0x9438, 0x6FC1, 0x8AFE, 0x8338, 0x51E7, 0x86F8, 0x53EA,
/* 0xC3A1 */	 0x53E9, 0x4F46, 0x9054, 0x8FB0, 0x596A, 0x8131, 0x5DFD,
	 0x7AEA, 0x8FBF, 0x68DA, 0x8C37, 0x72F8, 0x9C48, 0x6A3D, 0x8AB0,
	 0x4E39, 0x5358, 0x5606, 0x5766, 0x62C5, 0x63A2, 0x65E6, 0x6B4E,
	 0x6DE1, 0x6E5B, 0x70AD, 0x77ED, 0x7AEF, 0x7BAA, 0x7DBB, 0x803D,
	 0x80C6, 0x86CB, 0x8A95, 0x935B, 0x56E3, 0x58C7, 0x5F3E, 0x65AD,
	 0x6696, 0x6A80, 0x6BB5, 0x7537, 0x8AC7, 0x5024, 0x77E5, 0x5730,
	 0x5F1B, 0x6065, 0x667A, 0x6C60, 0x75F4, 0x7A1A, 0x7F6E, 0x81F4,
	 0x8718, 0x9045, 0x99B3, 0x7BC9, 0x755C, 0x7AF9, 0x7B51, 0x84C4,
	 0x9010, 0x79E9, 0x7A92, 0x8336, 0x5AE1, 0x7740, 0x4E2D, 0x4EF2,
	 0x5B99, 0x5FE0, 0x62BD, 0x663C, 0x67F1, 0x6CE8, 0x866B, 0x8877,
	 0x8A3B, 0x914E, 0x92F3, 0x99D0, 0x6A17, 0x7026, 0x732A, 0x82E7,
	 0x8457, 0x8CAF, 0x4E01, 0x5146, 0x51CB, 0x558B, 0x5BF5,
/* 0xC4A1 */	 0x5E16, 0x5E33, 0x5E81, 0x5F14, 0x5F35, 0x5F6B, 0x5FB4,
	 0x61F2, 0x6311, 0x66A2, 0x671D, 0x6F6E, 0x7252, 0x753A, 0x773A,
	 0x8074, 0x8139, 0x8178, 0x8776, 0x8ABF, 0x8ADC, 0x8D85, 0x8DF3,
	 0x929A, 0x9577, 0x9802, 0x9CE5, 0x52C5, 0x6357, 0x76F4, 0x6715,
	 0x6C88, 0x73CD, 0x8CC3, 0x93AE, 0x9673, 0x6D25, 0x589C, 0x690E,
	 0x69CC, 0x8FFD, 0x939A, 0x75DB, 0x901A, 0x585A, 0x6802, 0x63B4,
	 0x69FB, 0x4F43, 0x6F2C, 0x67D8, 0x8FBB, 0x8526, 0x7DB4, 0x9354,
	 0x693F, 0x6F70, 0x576A, 0x58F7, 0x5B2C, 0x7D2C, 0x722A, 0x540A,
	 0x91E3, 0x9DB4, 0x4EAD, 0x4F4E, 0x505C, 0x5075, 0x5243, 0x8C9E,
	 0x5448, 0x5824, 0x5B9A, 0x5E1D, 0x5E95, 0x5EAD, 0x5EF7, 0x5F1F,
	 0x608C, 0x62B5, 0x633A, 0x63D0, 0x68AF, 0x6C40, 0x7887, 0x798E,
	 0x7A0B, 0x7DE0, 0x8247, 0x8A02, 0x8AE6, 0x8E44, 0x9013,
/* 0xC5A1 */	 0x90B8, 0x912D, 0x91D8, 0x9F0E, 0x6CE5, 0x6458, 0x64E2,
	 0x6575, 0x6EF4, 0x7684, 0x7B1B, 0x9069, 0x93D1, 0x6EBA, 0x54F2,
	 0x5FB9, 0x64A4, 0x8F4D, 0x8FED, 0x9244, 0x5178, 0x586B, 0x5929,
	 0x5C55, 0x5E97, 0x6DFB, 0x7E8F, 0x751C, 0x8CBC, 0x8EE2, 0x985B,
	 0x70B9, 0x4F1D, 0x6BBF, 0x6FB1, 0x7530, 0x96FB, 0x514E, 0x5410,
	 0x5835, 0x5857, 0x59AC, 0x5C60, 0x5F92, 0x6597, 0x675C, 0x6E21,
	 0x767B, 0x83DF, 0x8CED, 0x9014, 0x90FD, 0x934D, 0x7825, 0x783A,
	 0x52AA, 0x5EA6, 0x571F, 0x5974, 0x6012, 0x5012, 0x515A, 0x51AC,
	 0x51CD, 0x5200, 0x5510, 0x5854, 0x5858, 0x5957, 0x5B95, 0x5CF6,
	 0x5D8B, 0x60BC, 0x6295, 0x642D, 0x6771, 0x6843, 0x68BC, 0x68DF,
	 0x76D7, 0x6DD8, 0x6E6F, 0x6D9B, 0x706F, 0x71C8, 0x5F53, 0x75D8,
	 0x7977, 0x7B49, 0x7B54, 0x7B52, 0x7CD6, 0x7D71, 0x5230,
/* 0xC6A1 */	 0x8463, 0x8569, 0x85E4, 0x8A0E, 0x8B04, 0x8C46, 0x8E0F,
	 0x9003, 0x900F, 0x9419, 0x9676, 0x982D, 0x9A30, 0x95D8, 0x50CD,
	 0x52D5, 0x540C, 0x5802, 0x5C0E, 0x61A7, 0x649E, 0x6D1E, 0x77B3,
	 0x7AE5, 0x80F4, 0x8404, 0x9053, 0x9285, 0x5CE0, 0x9D07, 0x533F,
	 0x5F97, 0x5FB3, 0x6D9C, 0x7279, 0x7763, 0x79BF, 0x7BE4, 0x6BD2,
	 0x72EC, 0x8AAD, 0x6803, 0x6A61, 0x51F8, 0x7A81, 0x6934, 0x5C4A,
	 0x9CF6, 0x82EB, 0x5BC5, 0x9149, 0x701E, 0x5678, 0x5C6F, 0x60C7,
	 0x6566, 0x6C8C, 0x8C5A, 0x9041, 0x9813, 0x5451, 0x66C7, 0x920D,
	 0x5948, 0x90A3, 0x5185, 0x4E4D, 0x51EA, 0x8599, 0x8B0E, 0x7058,
	 0x637A, 0x934B, 0x6962, 0x99B4, 0x7E04, 0x7577, 0x5357, 0x6960,
	 0x8EDF, 0x96E3, 0x6C5D, 0x4E8C, 0x5C3C, 0x5F10, 0x8FE9, 0x5302,
	 0x8CD1, 0x8089, 0x8679, 0x5EFF, 0x65E5, 0x4E73, 0x5165,
/* 0xC7A1 */	 0x5982, 0x5C3F, 0x97EE, 0x4EFB, 0x598A, 0x5FCD, 0x8A8D,
	 0x6FE1, 0x79B0, 0x7962, 0x5BE7, 0x8471, 0x732B, 0x71B1, 0x5E74,
	 0x5FF5, 0x637B, 0x649A, 0x71C3, 0x7C98, 0x4E43, 0x5EFC, 0x4E4B,
	 0x57DC, 0x56A2, 0x60A9, 0x6FC3, 0x7D0D, 0x80FD, 0x8133, 0x81BF,
	 0x8FB2, 0x8997, 0x86A4, 0x5DF4, 0x628A, 0x64AD, 0x8987, 0x6777,
	 0x6CE2, 0x6D3E, 0x7436, 0x7834, 0x5A46, 0x7F75, 0x82AD, 0x99AC,
	 0x4FF3, 0x5EC3, 0x62DD, 0x6392, 0x6557, 0x676F, 0x76C3, 0x724C,
	 0x80CC, 0x80BA, 0x8F29, 0x914D, 0x500D, 0x57F9, 0x5A92, 0x6885,
	 0x6973, 0x7164, 0x72FD, 0x8CB7, 0x58F2, 0x8CE0, 0x966A, 0x9019,
	 0x877F, 0x79E4, 0x77E7, 0x8429, 0x4F2F, 0x5265, 0x535A, 0x62CD,
	 0x67CF, 0x6CCA, 0x767D, 0x7B94, 0x7C95, 0x8236, 0x8584, 0x8FEB,
	 0x66DD, 0x6F20, 0x7206, 0x7E1B, 0x83AB, 0x99C1, 0x9EA6,
/* 0xC8A1 */	 0x51FD, 0x7BB1, 0x7872, 0x7BB8, 0x8087, 0x7B48, 0x6AE8,
	 0x5E61, 0x808C, 0x7551, 0x7560, 0x516B, 0x9262, 0x6E8C, 0x767A,
	 0x9197, 0x9AEA, 0x4F10, 0x7F70, 0x629C, 0x7B4F, 0x95A5, 0x9CE9,
	 0x567A, 0x5859, 0x86E4, 0x96BC, 0x4F34, 0x5224, 0x534A, 0x53CD,
	 0x53DB, 0x5E06, 0x642C, 0x6591, 0x677F, 0x6C3E, 0x6C4E, 0x7248,
	 0x72AF, 0x73ED, 0x7554, 0x7E41, 0x822C, 0x85E9, 0x8CA9, 0x7BC4,
	 0x91C6, 0x7169, 0x9812, 0x98EF, 0x633D, 0x6669, 0x756A, 0x76E4,
	 0x78D0, 0x8543, 0x86EE, 0x532A, 0x5351, 0x5426, 0x5983, 0x5E87,
	 0x5F7C, 0x60B2, 0x6249, 0x6279, 0x62AB, 0x6590, 0x6BD4, 0x6CCC,
	 0x75B2, 0x76AE, 0x7891, 0x79D8, 0x7DCB, 0x7F77, 0x80A5, 0x88AB,
	 0x8AB9, 0x8CBB, 0x907F, 0x975E, 0x98DB, 0x6A0B, 0x7C38, 0x5099,
	 0x5C3E, 0x5FAE, 0x6787, 0x6BD8, 0x7435, 0x7709, 0x7F8E,
/* 0xC9A1 */	 0x9F3B, 0x67CA, 0x7A17, 0x5339, 0x758B, 0x9AED, 0x5F66,
	 0x819D, 0x83F1, 0x8098, 0x5F3C, 0x5FC5, 0x7562, 0x7B46, 0x903C,
	 0x6867, 0x59EB, 0x5A9B, 0x7D10, 0x767E, 0x8B2C, 0x4FF5, 0x5F6A,
	 0x6A19, 0x6C37, 0x6F02, 0x74E2, 0x7968, 0x8868, 0x8A55, 0x8C79,
	 0x5EDF, 0x63CF, 0x75C5, 0x79D2, 0x82D7, 0x9328, 0x92F2, 0x849C,
	 0x86ED, 0x9C2D, 0x54C1, 0x5F6C, 0x658C, 0x6D5C, 0x7015, 0x8CA7,
	 0x8CD3, 0x983B, 0x654F, 0x74F6, 0x4E0D, 0x4ED8, 0x57E0, 0x592B,
	 0x5A66, 0x5BCC, 0x51A8, 0x5E03, 0x5E9C, 0x6016, 0x6276, 0x6577,
	 0x65A7, 0x666E, 0x6D6E, 0x7236, 0x7B26, 0x8150, 0x819A, 0x8299,
	 0x8B5C, 0x8CA0, 0x8CE6, 0x8D74, 0x961C, 0x9644, 0x4FAE, 0x64AB,
	 0x6B66, 0x821E, 0x8461, 0x856A, 0x90E8, 0x5C01, 0x6953, 0x98A8,
	 0x847A, 0x8557, 0x4F0F, 0x526F, 0x5FA9, 0x5E45, 0x670D,
/* 0xCAA1 */	 0x798F, 0x8179, 0x8907, 0x8986, 0x6DF5, 0x5F17, 0x6255,
	 0x6CB8, 0x4ECF, 0x7269, 0x9B92, 0x5206, 0x543B, 0x5674, 0x58B3,
	 0x61A4, 0x626E, 0x711A, 0x596E, 0x7C89, 0x7CDE, 0x7D1B, 0x96F0,
	 0x6587, 0x805E, 0x4E19, 0x4F75, 0x5175, 0x5840, 0x5E63, 0x5E73,
	 0x5F0A, 0x67C4, 0x4E26, 0x853D, 0x9589, 0x965B, 0x7C73, 0x9801,
	 0x50FB, 0x58C1, 0x7656, 0x78A7, 0x5225, 0x77A5, 0x8511, 0x7B86,
	 0x504F, 0x5909, 0x7247, 0x7BC7, 0x7DE8, 0x8FBA, 0x8FD4, 0x904D,
	 0x4FBF, 0x52C9, 0x5A29, 0x5F01, 0x97AD, 0x4FDD, 0x8217, 0x92EA,
	 0x5703, 0x6355, 0x6B69, 0x752B, 0x88DC, 0x8F14, 0x7A42, 0x52DF,
	 0x5893, 0x6155, 0x620A, 0x66AE, 0x6BCD, 0x7C3F, 0x83E9, 0x5023,
	 0x4FF8, 0x5305, 0x5446, 0x5831, 0x5949, 0x5B9D, 0x5CF0, 0x5CEF,
	 0x5D29, 0x5E96, 0x62B1, 0x6367, 0x653E, 0x65B9, 0x670B,
/* 0xCBA1 */	 0x6CD5, 0x6CE1, 0x70F9, 0x7832, 0x7E2B, 0x80DE, 0x82B3,
	 0x840C, 0x84EC, 0x8702, 0x8912, 0x8A2A, 0x8C4A, 0x90A6, 0x92D2,
	 0x98FD, 0x9CF3, 0x9D6C, 0x4E4F, 0x4EA1, 0x508D, 0x5256, 0x574A,
	 0x59A8, 0x5E3D, 0x5FD8, 0x5FD9, 0x623F, 0x66B4, 0x671B, 0x67D0,
	 0x68D2, 0x5192, 0x7D21, 0x80AA, 0x81A8, 0x8B00, 0x8C8C, 0x8CBF,
	 0x927E, 0x9632, 0x5420, 0x982C, 0x5317, 0x50D5, 0x535C, 0x58A8,
	 0x64B2, 0x6734, 0x7267, 0x7766, 0x7A46, 0x91E6, 0x52C3, 0x6CA1,
	 0x6B86, 0x5800, 0x5E4C, 0x5954, 0x672C, 0x7FFB, 0x51E1, 0x76C6,
	 0x6469, 0x78E8, 0x9B54, 0x9EBB, 0x57CB, 0x59B9, 0x6627, 0x679A,
	 0x6BCE, 0x54E9, 0x69D9, 0x5E55, 0x819C, 0x6795, 0x9BAA, 0x67FE,
	 0x9C52, 0x685D, 0x4EA6, 0x4FE3, 0x53C8, 0x62B9, 0x672B, 0x6CAB,
	 0x8FC4, 0x4FAD, 0x7E6D, 0x9EBF, 0x4E07, 0x6162, 0x6E80,
/* 0xCCA1 */	 0x6F2B, 0x8513, 0x5473, 0x672A, 0x9B45, 0x5DF3, 0x7B95,
	 0x5CAC, 0x5BC6, 0x871C, 0x6E4A, 0x84D1, 0x7A14, 0x8108, 0x5999,
	 0x7C8D, 0x6C11, 0x7720, 0x52D9, 0x5922, 0x7121, 0x725F, 0x77DB,
	 0x9727, 0x9D61, 0x690B, 0x5A7F, 0x5A18, 0x51A5, 0x540D, 0x547D,
	 0x660E, 0x76DF, 0x8FF7, 0x9298, 0x9CF4, 0x59EA, 0x725D, 0x6EC5,
	 0x514D, 0x68C9, 0x7DBF, 0x7DEC, 0x9762, 0x9EBA, 0x6478, 0x6A21,
	 0x8302, 0x5984, 0x5B5F, 0x6BDB, 0x731B, 0x76F2, 0x7DB2, 0x8017,
	 0x8499, 0x5132, 0x6728, 0x9ED9, 0x76EE, 0x6762, 0x52FF, 0x9905,
	 0x5C24, 0x623B, 0x7C7E, 0x8CB0, 0x554F, 0x60B6, 0x7D0B, 0x9580,
	 0x5301, 0x4E5F, 0x51B6, 0x591C, 0x723A, 0x8036, 0x91CE, 0x5F25,
	 0x77E2, 0x5384, 0x5F79, 0x7D04, 0x85AC, 0x8A33, 0x8E8D, 0x9756,
	 0x67F3, 0x85AE, 0x9453, 0x6109, 0x6108, 0x6CB9, 0x7652,
/* 0xCDA1 */	 0x8AED, 0x8F38, 0x552F, 0x4F51, 0x512A, 0x52C7, 0x53CB,
	 0x5BA5, 0x5E7D, 0x60A0, 0x6182, 0x63D6, 0x6709, 0x67DA, 0x6E67,
	 0x6D8C, 0x7336, 0x7337, 0x7531, 0x7950, 0x88D5, 0x8A98, 0x904A,
	 0x9091, 0x90F5, 0x96C4, 0x878D, 0x5915, 0x4E88, 0x4F59, 0x4E0E,
	 0x8A89, 0x8F3F, 0x9810, 0x50AD, 0x5E7C, 0x5996, 0x5BB9, 0x5EB8,
	 0x63DA, 0x63FA, 0x64C1, 0x66DC, 0x694A, 0x69D8, 0x6D0B, 0x6EB6,
	 0x7194, 0x7528, 0x7AAF, 0x7F8A, 0x8000, 0x8449, 0x84C9, 0x8981,
	 0x8B21, 0x8E0A, 0x9065, 0x967D, 0x990A, 0x617E, 0x6291, 0x6B32,
	 0x6C83, 0x6D74, 0x7FCC, 0x7FFC, 0x6DC0, 0x7F85, 0x87BA, 0x88F8,
	 0x6765, 0x83B1, 0x983C, 0x96F7, 0x6D1B, 0x7D61, 0x843D, 0x916A,
	 0x4E71, 0x5375, 0x5D50, 0x6B04, 0x6FEB, 0x85CD, 0x862D, 0x89A7,
	 0x5229, 0x540F, 0x5C65, 0x674E, 0x68A8, 0x7406, 0x7483,
/* 0xCEA1 */	 0x75E2, 0x88CF, 0x88E1, 0x91CC, 0x96E2, 0x9678, 0x5F8B,
	 0x7387, 0x7ACB, 0x844E, 0x63A0, 0x7565, 0x5289, 0x6D41, 0x6E9C,
	 0x7409, 0x7559, 0x786B, 0x7C92, 0x9686, 0x7ADC, 0x9F8D, 0x4FB6,
	 0x616E, 0x65C5, 0x865C, 0x4E86, 0x4EAE, 0x50DA, 0x4E21, 0x51CC,
	 0x5BEE, 0x6599, 0x6881, 0x6DBC, 0x731F, 0x7642, 0x77AD, 0x7A1C,
	 0x7CE7, 0x826F, 0x8AD2, 0x907C, 0x91CF, 0x9675, 0x9818, 0x529B,
	 0x7DD1, 0x502B, 0x5398, 0x6797, 0x6DCB, 0x71D0, 0x7433, 0x81E8,
	 0x8F2A, 0x96A3, 0x9C57, 0x9E9F, 0x7460, 0x5841, 0x6D99, 0x7D2F,
	 0x985E, 0x4EE4, 0x4F36, 0x4F8B, 0x51B7, 0x52B1, 0x5DBA, 0x601C,
	 0x73B2, 0x793C, 0x82D3, 0x9234, 0x96B7, 0x96F6, 0x970A, 0x9E97,
	 0x9F62, 0x66A6, 0x6B74, 0x5217, 0x52A3, 0x70C8, 0x88C2, 0x5EC9,
	 0x604B, 0x6190, 0x6F23, 0x7149, 0x7C3E, 0x7DF4, 0x806F,
/* 0xCFA1 */	 0x84EE, 0x9023, 0x932C, 0x5442, 0x9B6F, 0x6AD3, 0x7089,
	 0x8CC2, 0x8DEF, 0x9732, 0x52B4, 0x5A41, 0x5ECA, 0x5F04, 0x6717,
	 0x697C, 0x6994, 0x6D6A, 0x6F0F, 0x7262, 0x72FC, 0x7BED, 0x8001,
	 0x807E, 0x874B, 0x90CE, 0x516D, 0x9E93, 0x7984, 0x808B, 0x9332,
	 0x8AD6, 0x502D, 0x548C, 0x8A71, 0x6B6A, 0x8CC4, 0x8107, 0x60D1,
	 0x67A0, 0x9DF2, 0x4E99, 0x4E98, 0x9C10, 0x8A6B, 0x85C1, 0x8568,
	 0x6900, 0x6E7E, 0x7897, 0x8155, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xD0A1 */	 0x5F0C, 0x4E10, 0x4E15, 0x4E2A, 0x4E31, 0x4E36, 0x4E3C,
	 0x4E3F, 0x4E42, 0x4E56, 0x4E58, 0x4E82, 0x4E85, 0x8C6B, 0x4E8A,
	 0x8212, 0x5F0D, 0x4E8E, 0x4E9E, 0x4E9F, 0x4EA0, 0x4EA2, 0x4EB0,
	 0x4EB3, 0x4EB6, 0x4ECE, 0x4ECD, 0x4EC4, 0x4EC6, 0x4EC2, 0x4ED7,
	 0x4EDE, 0x4EED, 0x4EDF, 0x4EF7, 0x4F09, 0x4F5A, 0x4F30, 0x4F5B,
	 0x4F5D, 0x4F57, 0x4F47, 0x4F76, 0x4F88, 0x4F8F, 0x4F98, 0x4F7B,
	 0x4F69, 0x4F70, 0x4F91, 0x4F6F, 0x4F86, 0x4F96, 0x5118, 0x4FD4,
	 0x4FDF, 0x4FCE, 0x4FD8, 0x4FDB, 0x4FD1, 0x4FDA, 0x4FD0, 0x4FE4,
	 0x4FE5, 0x501A, 0x5028, 0x5014, 0x502A, 0x5025, 0x5005, 0x4F1C,
	 0x4FF6, 0x5021, 0x5029, 0x502C, 0x4FFE, 0x4FEF, 0x5011, 0x5006,
	 0x5043, 0x5047, 0x6703, 0x5055, 0x5050, 0x5048, 0x505A, 0x5056,
	 0x506C, 0x5078, 0x5080, 0x509A, 0x5085, 0x50B4, 0x50B2,
/* 0xD1A1 */	 0x50C9, 0x50CA, 0x50B3, 0x50C2, 0x50D6, 0x50DE, 0x50E5,
	 0x50ED, 0x50E3, 0x50EE, 0x50F9, 0x50F5, 0x5109, 0x5101, 0x5102,
	 0x5116, 0x5115, 0x5114, 0x511A, 0x5121, 0x513A, 0x5137, 0x513C,
	 0x513B, 0x513F, 0x5140, 0x5152, 0x514C, 0x5154, 0x5162, 0x7AF8,
	 0x5169, 0x516A, 0x516E, 0x5180, 0x5182, 0x56D8, 0x518C, 0x5189,
	 0x518F, 0x5191, 0x5193, 0x5195, 0x5196, 0x51A4, 0x51A6, 0x51A2,
	 0x51A9, 0x51AA, 0x51AB, 0x51B3, 0x51B1, 0x51B2, 0x51B0, 0x51B5,
	 0x51BD, 0x51C5, 0x51C9, 0x51DB, 0x51E0, 0x8655, 0x51E9, 0x51ED,
	 0x51F0, 0x51F5, 0x51FE, 0x5204, 0x520B, 0x5214, 0x520E, 0x5227,
	 0x522A, 0x522E, 0x5233, 0x5239, 0x524F, 0x5244, 0x524B, 0x524C,
	 0x525E, 0x5254, 0x526A, 0x5274, 0x5269, 0x5273, 0x527F, 0x527D,
	 0x528D, 0x5294, 0x5292, 0x5271, 0x5288, 0x5291, 0x8FA8,
/* 0xD2A1 */	 0x8FA7, 0x52AC, 0x52AD, 0x52BC, 0x52B5, 0x52C1, 0x52CD,
	 0x52D7, 0x52DE, 0x52E3, 0x52E6, 0x98ED, 0x52E0, 0x52F3, 0x52F5,
	 0x52F8, 0x52F9, 0x5306, 0x5308, 0x7538, 0x530D, 0x5310, 0x530F,
	 0x5315, 0x531A, 0x5323, 0x532F, 0x5331, 0x5333, 0x5338, 0x5340,
	 0x5346, 0x5345, 0x4E17, 0x5349, 0x534D, 0x51D6, 0x535E, 0x5369,
	 0x536E, 0x5918, 0x537B, 0x5377, 0x5382, 0x5396, 0x53A0, 0x53A6,
	 0x53A5, 0x53AE, 0x53B0, 0x53B6, 0x53C3, 0x7C12, 0x96D9, 0x53DF,
	 0x66FC, 0x71EE, 0x53EE, 0x53E8, 0x53ED, 0x53FA, 0x5401, 0x543D,
	 0x5440, 0x542C, 0x542D, 0x543C, 0x542E, 0x5436, 0x5429, 0x541D,
	 0x544E, 0x548F, 0x5475, 0x548E, 0x545F, 0x5471, 0x5477, 0x5470,
	 0x5492, 0x547B, 0x5480, 0x5476, 0x5484, 0x5490, 0x5486, 0x54C7,
	 0x54A2, 0x54B8, 0x54A5, 0x54AC, 0x54C4, 0x54C8, 0x54A8,
/* 0xD3A1 */	 0x54AB, 0x54C2, 0x54A4, 0x54BE, 0x54BC, 0x54D8, 0x54E5,
	 0x54E6, 0x550F, 0x5514, 0x54FD, 0x54EE, 0x54ED, 0x54FA, 0x54E2,
	 0x5539, 0x5540, 0x5563, 0x554C, 0x552E, 0x555C, 0x5545, 0x5556,
	 0x5557, 0x5538, 0x5533, 0x555D, 0x5599, 0x5580, 0x54AF, 0x558A,
	 0x559F, 0x557B, 0x557E, 0x5598, 0x559E, 0x55AE, 0x557C, 0x5583,
	 0x55A9, 0x5587, 0x55A8, 0x55DA, 0x55C5, 0x55DF, 0x55C4, 0x55DC,
	 0x55E4, 0x55D4, 0x5614, 0x55F7, 0x5616, 0x55FE, 0x55FD, 0x561B,
	 0x55F9, 0x564E, 0x5650, 0x71DF, 0x5634, 0x5636, 0x5632, 0x5638,
	 0x566B, 0x5664, 0x562F, 0x566C, 0x566A, 0x5686, 0x5680, 0x568A,
	 0x56A0, 0x5694, 0x568F, 0x56A5, 0x56AE, 0x56B6, 0x56B4, 0x56C2,
	 0x56BC, 0x56C1, 0x56C3, 0x56C0, 0x56C8, 0x56CE, 0x56D1, 0x56D3,
	 0x56D7, 0x56EE, 0x56F9, 0x5700, 0x56FF, 0x5704, 0x5709,
/* 0xD4A1 */	 0x5708, 0x570B, 0x570D, 0x5713, 0x5718, 0x5716, 0x55C7,
	 0x571C, 0x5726, 0x5737, 0x5738, 0x574E, 0x573B, 0x5740, 0x574F,
	 0x5769, 0x57C0, 0x5788, 0x5761, 0x577F, 0x5789, 0x5793, 0x57A0,
	 0x57B3, 0x57A4, 0x57AA, 0x57B0, 0x57C3, 0x57C6, 0x57D4, 0x57D2,
	 0x57D3, 0x580A, 0x57D6, 0x57E3, 0x580B, 0x5819, 0x581D, 0x5872,
	 0x5821, 0x5862, 0x584B, 0x5870, 0x6BC0, 0x5852, 0x583D, 0x5879,
	 0x5885, 0x58B9, 0x589F, 0x58AB, 0x58BA, 0x58DE, 0x58BB, 0x58B8,
	 0x58AE, 0x58C5, 0x58D3, 0x58D1, 0x58D7, 0x58D9, 0x58D8, 0x58E5,
	 0x58DC, 0x58E4, 0x58DF, 0x58EF, 0x58FA, 0x58F9, 0x58FB, 0x58FC,
	 0x58FD, 0x5902, 0x590A, 0x5910, 0x591B, 0x68A6, 0x5925, 0x592C,
	 0x592D, 0x5932, 0x5938, 0x593E, 0x7AD2, 0x5955, 0x5950, 0x594E,
	 0x595A, 0x5958, 0x5962, 0x5960, 0x5967, 0x596C, 0x5969,
/* 0xD5A1 */	 0x5978, 0x5981, 0x599D, 0x4F5E, 0x4FAB, 0x59A3, 0x59B2,
	 0x59C6, 0x59E8, 0x59DC, 0x598D, 0x59D9, 0x59DA, 0x5A25, 0x5A1F,
	 0x5A11, 0x5A1C, 0x5A09, 0x5A1A, 0x5A40, 0x5A6C, 0x5A49, 0x5A35,
	 0x5A36, 0x5A62, 0x5A6A, 0x5A9A, 0x5ABC, 0x5ABE, 0x5ACB, 0x5AC2,
	 0x5ABD, 0x5AE3, 0x5AD7, 0x5AE6, 0x5AE9, 0x5AD6, 0x5AFA, 0x5AFB,
	 0x5B0C, 0x5B0B, 0x5B16, 0x5B32, 0x5AD0, 0x5B2A, 0x5B36, 0x5B3E,
	 0x5B43, 0x5B45, 0x5B40, 0x5B51, 0x5B55, 0x5B5A, 0x5B5B, 0x5B65,
	 0x5B69, 0x5B70, 0x5B73, 0x5B75, 0x5B78, 0x6588, 0x5B7A, 0x5B80,
	 0x5B83, 0x5BA6, 0x5BB8, 0x5BC3, 0x5BC7, 0x5BC9, 0x5BD4, 0x5BD0,
	 0x5BE4, 0x5BE6, 0x5BE2, 0x5BDE, 0x5BE5, 0x5BEB, 0x5BF0, 0x5BF6,
	 0x5BF3, 0x5C05, 0x5C07, 0x5C08, 0x5C0D, 0x5C13, 0x5C20, 0x5C22,
	 0x5C28, 0x5C38, 0x5C39, 0x5C41, 0x5C46, 0x5C4E, 0x5C53,
/* 0xD6A1 */	 0x5C50, 0x5C4F, 0x5B71, 0x5C6C, 0x5C6E, 0x4E62, 0x5C76,
	 0x5C79, 0x5C8C, 0x5C91, 0x5C94, 0x599B, 0x5CAB, 0x5CBB, 0x5CB6,
	 0x5CBC, 0x5CB7, 0x5CC5, 0x5CBE, 0x5CC7, 0x5CD9, 0x5CE9, 0x5CFD,
	 0x5CFA, 0x5CED, 0x5D8C, 0x5CEA, 0x5D0B, 0x5D15, 0x5D17, 0x5D5C,
	 0x5D1F, 0x5D1B, 0x5D11, 0x5D14, 0x5D22, 0x5D1A, 0x5D19, 0x5D18,
	 0x5D4C, 0x5D52, 0x5D4E, 0x5D4B, 0x5D6C, 0x5D73, 0x5D76, 0x5D87,
	 0x5D84, 0x5D82, 0x5DA2, 0x5D9D, 0x5DAC, 0x5DAE, 0x5DBD, 0x5D90,
	 0x5DB7, 0x5DBC, 0x5DC9, 0x5DCD, 0x5DD3, 0x5DD2, 0x5DD6, 0x5DDB,
	 0x5DEB, 0x5DF2, 0x5DF5, 0x5E0B, 0x5E1A, 0x5E19, 0x5E11, 0x5E1B,
	 0x5E36, 0x5E37, 0x5E44, 0x5E43, 0x5E40, 0x5E4E, 0x5E57, 0x5E54,
	 0x5E5F, 0x5E62, 0x5E64, 0x5E47, 0x5E75, 0x5E76, 0x5E7A, 0x9EBC,
	 0x5E7F, 0x5EA0, 0x5EC1, 0x5EC2, 0x5EC8, 0x5ED0, 0x5ECF,
/* 0xD7A1 */	 0x5ED6, 0x5EE3, 0x5EDD, 0x5EDA, 0x5EDB, 0x5EE2, 0x5EE1,
	 0x5EE8, 0x5EE9, 0x5EEC, 0x5EF1, 0x5EF3, 0x5EF0, 0x5EF4, 0x5EF8,
	 0x5EFE, 0x5F03, 0x5F09, 0x5F5D, 0x5F5C, 0x5F0B, 0x5F11, 0x5F16,
	 0x5F29, 0x5F2D, 0x5F38, 0x5F41, 0x5F48, 0x5F4C, 0x5F4E, 0x5F2F,
	 0x5F51, 0x5F56, 0x5F57, 0x5F59, 0x5F61, 0x5F6D, 0x5F73, 0x5F77,
	 0x5F83, 0x5F82, 0x5F7F, 0x5F8A, 0x5F88, 0x5F91, 0x5F87, 0x5F9E,
	 0x5F99, 0x5F98, 0x5FA0, 0x5FA8, 0x5FAD, 0x5FBC, 0x5FD6, 0x5FFB,
	 0x5FE4, 0x5FF8, 0x5FF1, 0x5FDD, 0x60B3, 0x5FFF, 0x6021, 0x6060,
	 0x6019, 0x6010, 0x6029, 0x600E, 0x6031, 0x601B, 0x6015, 0x602B,
	 0x6026, 0x600F, 0x603A, 0x605A, 0x6041, 0x606A, 0x6077, 0x605F,
	 0x604A, 0x6046, 0x604D, 0x6063, 0x6043, 0x6064, 0x6042, 0x606C,
	 0x606B, 0x6059, 0x6081, 0x608D, 0x60E7, 0x6083, 0x609A,
/* 0xD8A1 */	 0x6084, 0x609B, 0x6096, 0x6097, 0x6092, 0x60A7, 0x608B,
	 0x60E1, 0x60B8, 0x60E0, 0x60D3, 0x60B4, 0x5FF0, 0x60BD, 0x60C6,
	 0x60B5, 0x60D8, 0x614D, 0x6115, 0x6106, 0x60F6, 0x60F7, 0x6100,
	 0x60F4, 0x60FA, 0x6103, 0x6121, 0x60FB, 0x60F1, 0x610D, 0x610E,
	 0x6147, 0x613E, 0x6128, 0x6127, 0x614A, 0x613F, 0x613C, 0x612C,
	 0x6134, 0x613D, 0x6142, 0x6144, 0x6173, 0x6177, 0x6158, 0x6159,
	 0x615A, 0x616B, 0x6174, 0x616F, 0x6165, 0x6171, 0x615F, 0x615D,
	 0x6153, 0x6175, 0x6199, 0x6196, 0x6187, 0x61AC, 0x6194, 0x619A,
	 0x618A, 0x6191, 0x61AB, 0x61AE, 0x61CC, 0x61CA, 0x61C9, 0x61F7,
	 0x61C8, 0x61C3, 0x61C6, 0x61BA, 0x61CB, 0x7F79, 0x61CD, 0x61E6,
	 0x61E3, 0x61F6, 0x61FA, 0x61F4, 0x61FF, 0x61FD, 0x61FC, 0x61FE,
	 0x6200, 0x6208, 0x6209, 0x620D, 0x620C, 0x6214, 0x621B,
/* 0xD9A1 */	 0x621E, 0x6221, 0x622A, 0x622E, 0x6230, 0x6232, 0x6233,
	 0x6241, 0x624E, 0x625E, 0x6263, 0x625B, 0x6260, 0x6268, 0x627C,
	 0x6282, 0x6289, 0x627E, 0x6292, 0x6293, 0x6296, 0x62D4, 0x6283,
	 0x6294, 0x62D7, 0x62D1, 0x62BB, 0x62CF, 0x62FF, 0x62C6, 0x64D4,
	 0x62C8, 0x62DC, 0x62CC, 0x62CA, 0x62C2, 0x62C7, 0x629B, 0x62C9,
	 0x630C, 0x62EE, 0x62F1, 0x6327, 0x6302, 0x6308, 0x62EF, 0x62F5,
	 0x6350, 0x633E, 0x634D, 0x641C, 0x634F, 0x6396, 0x638E, 0x6380,
	 0x63AB, 0x6376, 0x63A3, 0x638F, 0x6389, 0x639F, 0x63B5, 0x636B,
	 0x6369, 0x63BE, 0x63E9, 0x63C0, 0x63C6, 0x63E3, 0x63C9, 0x63D2,
	 0x63F6, 0x63C4, 0x6416, 0x6434, 0x6406, 0x6413, 0x6426, 0x6436,
	 0x651D, 0x6417, 0x6428, 0x640F, 0x6467, 0x646F, 0x6476, 0x644E,
	 0x652A, 0x6495, 0x6493, 0x64A5, 0x64A9, 0x6488, 0x64BC,
/* 0xDAA1 */	 0x64DA, 0x64D2, 0x64C5, 0x64C7, 0x64BB, 0x64D8, 0x64C2,
	 0x64F1, 0x64E7, 0x8209, 0x64E0, 0x64E1, 0x62AC, 0x64E3, 0x64EF,
	 0x652C, 0x64F6, 0x64F4, 0x64F2, 0x64FA, 0x6500, 0x64FD, 0x6518,
	 0x651C, 0x6505, 0x6524, 0x6523, 0x652B, 0x6534, 0x6535, 0x6537,
	 0x6536, 0x6538, 0x754B, 0x6548, 0x6556, 0x6555, 0x654D, 0x6558,
	 0x655E, 0x655D, 0x6572, 0x6578, 0x6582, 0x6583, 0x8B8A, 0x659B,
	 0x659F, 0x65AB, 0x65B7, 0x65C3, 0x65C6, 0x65C1, 0x65C4, 0x65CC,
	 0x65D2, 0x65DB, 0x65D9, 0x65E0, 0x65E1, 0x65F1, 0x6772, 0x660A,
	 0x6603, 0x65FB, 0x6773, 0x6635, 0x6636, 0x6634, 0x661C, 0x664F,
	 0x6644, 0x6649, 0x6641, 0x665E, 0x665D, 0x6664, 0x6667, 0x6668,
	 0x665F, 0x6662, 0x6670, 0x6683, 0x6688, 0x668E, 0x6689, 0x6684,
	 0x6698, 0x669D, 0x66C1, 0x66B9, 0x66C9, 0x66BE, 0x66BC,
/* 0xDBA1 */	 0x66C4, 0x66B8, 0x66D6, 0x66DA, 0x66E0, 0x663F, 0x66E6,
	 0x66E9, 0x66F0, 0x66F5, 0x66F7, 0x670F, 0x6716, 0x671E, 0x6726,
	 0x6727, 0x9738, 0x672E, 0x673F, 0x6736, 0x6741, 0x6738, 0x6737,
	 0x6746, 0x675E, 0x6760, 0x6759, 0x6763, 0x6764, 0x6789, 0x6770,
	 0x67A9, 0x677C, 0x676A, 0x678C, 0x678B, 0x67A6, 0x67A1, 0x6785,
	 0x67B7, 0x67EF, 0x67B4, 0x67EC, 0x67B3, 0x67E9, 0x67B8, 0x67E4,
	 0x67DE, 0x67DD, 0x67E2, 0x67EE, 0x67B9, 0x67CE, 0x67C6, 0x67E7,
	 0x6A9C, 0x681E, 0x6846, 0x6829, 0x6840, 0x684D, 0x6832, 0x684E,
	 0x68B3, 0x682B, 0x6859, 0x6863, 0x6877, 0x687F, 0x689F, 0x688F,
	 0x68AD, 0x6894, 0x689D, 0x689B, 0x6883, 0x6AAE, 0x68B9, 0x6874,
	 0x68B5, 0x68A0, 0x68BA, 0x690F, 0x688D, 0x687E, 0x6901, 0x68CA,
	 0x6908, 0x68D8, 0x6922, 0x6926, 0x68E1, 0x690C, 0x68CD,
/* 0xDCA1 */	 0x68D4, 0x68E7, 0x68D5, 0x6936, 0x6912, 0x6904, 0x68D7,
	 0x68E3, 0x6925, 0x68F9, 0x68E0, 0x68EF, 0x6928, 0x692A, 0x691A,
	 0x6923, 0x6921, 0x68C6, 0x6979, 0x6977, 0x695C, 0x6978, 0x696B,
	 0x6954, 0x697E, 0x696E, 0x6939, 0x6974, 0x693D, 0x6959, 0x6930,
	 0x6961, 0x695E, 0x695D, 0x6981, 0x696A, 0x69B2, 0x69AE, 0x69D0,
	 0x69BF, 0x69C1, 0x69D3, 0x69BE, 0x69CE, 0x5BE8, 0x69CA, 0x69DD,
	 0x69BB, 0x69C3, 0x69A7, 0x6A2E, 0x6991, 0x69A0, 0x699C, 0x6995,
	 0x69B4, 0x69DE, 0x69E8, 0x6A02, 0x6A1B, 0x69FF, 0x6B0A, 0x69F9,
	 0x69F2, 0x69E7, 0x6A05, 0x69B1, 0x6A1E, 0x69ED, 0x6A14, 0x69EB,
	 0x6A0A, 0x6A12, 0x6AC1, 0x6A23, 0x6A13, 0x6A44, 0x6A0C, 0x6A72,
	 0x6A36, 0x6A78, 0x6A47, 0x6A62, 0x6A59, 0x6A66, 0x6A48, 0x6A38,
	 0x6A22, 0x6A90, 0x6A8D, 0x6AA0, 0x6A84, 0x6AA2, 0x6AA3,
/* 0xDDA1 */	 0x6A97, 0x8617, 0x6ABB, 0x6AC3, 0x6AC2, 0x6AB8, 0x6AB3,
	 0x6AAC, 0x6ADE, 0x6AD1, 0x6ADF, 0x6AAA, 0x6ADA, 0x6AEA, 0x6AFB,
	 0x6B05, 0x8616, 0x6AFA, 0x6B12, 0x6B16, 0x9B31, 0x6B1F, 0x6B38,
	 0x6B37, 0x76DC, 0x6B39, 0x98EE, 0x6B47, 0x6B43, 0x6B49, 0x6B50,
	 0x6B59, 0x6B54, 0x6B5B, 0x6B5F, 0x6B61, 0x6B78, 0x6B79, 0x6B7F,
	 0x6B80, 0x6B84, 0x6B83, 0x6B8D, 0x6B98, 0x6B95, 0x6B9E, 0x6BA4,
	 0x6BAA, 0x6BAB, 0x6BAF, 0x6BB2, 0x6BB1, 0x6BB3, 0x6BB7, 0x6BBC,
	 0x6BC6, 0x6BCB, 0x6BD3, 0x6BDF, 0x6BEC, 0x6BEB, 0x6BF3, 0x6BEF,
	 0x9EBE, 0x6C08, 0x6C13, 0x6C14, 0x6C1B, 0x6C24, 0x6C23, 0x6C5E,
	 0x6C55, 0x6C62, 0x6C6A, 0x6C82, 0x6C8D, 0x6C9A, 0x6C81, 0x6C9B,
	 0x6C7E, 0x6C68, 0x6C73, 0x6C92, 0x6C90, 0x6CC4, 0x6CF1, 0x6CD3,
	 0x6CBD, 0x6CD7, 0x6CC5, 0x6CDD, 0x6CAE, 0x6CB1, 0x6CBE,
/* 0xDEA1 */	 0x6CBA, 0x6CDB, 0x6CEF, 0x6CD9, 0x6CEA, 0x6D1F, 0x884D,
	 0x6D36, 0x6D2B, 0x6D3D, 0x6D38, 0x6D19, 0x6D35, 0x6D33, 0x6D12,
	 0x6D0C, 0x6D63, 0x6D93, 0x6D64, 0x6D5A, 0x6D79, 0x6D59, 0x6D8E,
	 0x6D95, 0x6FE4, 0x6D85, 0x6DF9, 0x6E15, 0x6E0A, 0x6DB5, 0x6DC7,
	 0x6DE6, 0x6DB8, 0x6DC6, 0x6DEC, 0x6DDE, 0x6DCC, 0x6DE8, 0x6DD2,
	 0x6DC5, 0x6DFA, 0x6DD9, 0x6DE4, 0x6DD5, 0x6DEA, 0x6DEE, 0x6E2D,
	 0x6E6E, 0x6E2E, 0x6E19, 0x6E72, 0x6E5F, 0x6E3E, 0x6E23, 0x6E6B,
	 0x6E2B, 0x6E76, 0x6E4D, 0x6E1F, 0x6E43, 0x6E3A, 0x6E4E, 0x6E24,
	 0x6EFF, 0x6E1D, 0x6E38, 0x6E82, 0x6EAA, 0x6E98, 0x6EC9, 0x6EB7,
	 0x6ED3, 0x6EBD, 0x6EAF, 0x6EC4, 0x6EB2, 0x6ED4, 0x6ED5, 0x6E8F,
	 0x6EA5, 0x6EC2, 0x6E9F, 0x6F41, 0x6F11, 0x704C, 0x6EEC, 0x6EF8,
	 0x6EFE, 0x6F3F, 0x6EF2, 0x6F31, 0x6EEF, 0x6F32, 0x6ECC,
/* 0xDFA1 */	 0x6F3E, 0x6F13, 0x6EF7, 0x6F86, 0x6F7A, 0x6F78, 0x6F81,
	 0x6F80, 0x6F6F, 0x6F5B, 0x6FF3, 0x6F6D, 0x6F82, 0x6F7C, 0x6F58,
	 0x6F8E, 0x6F91, 0x6FC2, 0x6F66, 0x6FB3, 0x6FA3, 0x6FA1, 0x6FA4,
	 0x6FB9, 0x6FC6, 0x6FAA, 0x6FDF, 0x6FD5, 0x6FEC, 0x6FD4, 0x6FD8,
	 0x6FF1, 0x6FEE, 0x6FDB, 0x7009, 0x700B, 0x6FFA, 0x7011, 0x7001,
	 0x700F, 0x6FFE, 0x701B, 0x701A, 0x6F74, 0x701D, 0x7018, 0x701F,
	 0x7030, 0x703E, 0x7032, 0x7051, 0x7063, 0x7099, 0x7092, 0x70AF,
	 0x70F1, 0x70AC, 0x70B8, 0x70B3, 0x70AE, 0x70DF, 0x70CB, 0x70DD,
	 0x70D9, 0x7109, 0x70FD, 0x711C, 0x7119, 0x7165, 0x7155, 0x7188,
	 0x7166, 0x7162, 0x714C, 0x7156, 0x716C, 0x718F, 0x71FB, 0x7184,
	 0x7195, 0x71A8, 0x71AC, 0x71D7, 0x71B9, 0x71BE, 0x71D2, 0x71C9,
	 0x71D4, 0x71CE, 0x71E0, 0x71EC, 0x71E7, 0x71F5, 0x71FC,
/* 0xE0A1 */	 0x71F9, 0x71FF, 0x720D, 0x7210, 0x721B, 0x7228, 0x722D,
	 0x722C, 0x7230, 0x7232, 0x723B, 0x723C, 0x723F, 0x7240, 0x7246,
	 0x724B, 0x7258, 0x7274, 0x727E, 0x7282, 0x7281, 0x7287, 0x7292,
	 0x7296, 0x72A2, 0x72A7, 0x72B9, 0x72B2, 0x72C3, 0x72C6, 0x72C4,
	 0x72CE, 0x72D2, 0x72E2, 0x72E0, 0x72E1, 0x72F9, 0x72F7, 0x500F,
	 0x7317, 0x730A, 0x731C, 0x7316, 0x731D, 0x7334, 0x732F, 0x7329,
	 0x7325, 0x733E, 0x734E, 0x734F, 0x9ED8, 0x7357, 0x736A, 0x7368,
	 0x7370, 0x7378, 0x7375, 0x737B, 0x737A, 0x73C8, 0x73B3, 0x73CE,
	 0x73BB, 0x73C0, 0x73E5, 0x73EE, 0x73DE, 0x74A2, 0x7405, 0x746F,
	 0x7425, 0x73F8, 0x7432, 0x743A, 0x7455, 0x743F, 0x745F, 0x7459,
	 0x7441, 0x745C, 0x7469, 0x7470, 0x7463, 0x746A, 0x7476, 0x747E,
	 0x748B, 0x749E, 0x74A7, 0x74CA, 0x74CF, 0x74D4, 0x73F1,
/* 0xE1A1 */	 0x74E0, 0x74E3, 0x74E7, 0x74E9, 0x74EE, 0x74F2, 0x74F0,
	 0x74F1, 0x74F8, 0x74F7, 0x7504, 0x7503, 0x7505, 0x750C, 0x750E,
	 0x750D, 0x7515, 0x7513, 0x751E, 0x7526, 0x752C, 0x753C, 0x7544,
	 0x754D, 0x754A, 0x7549, 0x755B, 0x7546, 0x755A, 0x7569, 0x7564,
	 0x7567, 0x756B, 0x756D, 0x7578, 0x7576, 0x7586, 0x7587, 0x7574,
	 0x758A, 0x7589, 0x7582, 0x7594, 0x759A, 0x759D, 0x75A5, 0x75A3,
	 0x75C2, 0x75B3, 0x75C3, 0x75B5, 0x75BD, 0x75B8, 0x75BC, 0x75B1,
	 0x75CD, 0x75CA, 0x75D2, 0x75D9, 0x75E3, 0x75DE, 0x75FE, 0x75FF,
	 0x75FC, 0x7601, 0x75F0, 0x75FA, 0x75F2, 0x75F3, 0x760B, 0x760D,
	 0x7609, 0x761F, 0x7627, 0x7620, 0x7621, 0x7622, 0x7624, 0x7634,
	 0x7630, 0x763B, 0x7647, 0x7648, 0x7646, 0x765C, 0x7658, 0x7661,
	 0x7662, 0x7668, 0x7669, 0x766A, 0x7667, 0x766C, 0x7670,
/* 0xE2A1 */	 0x7672, 0x7676, 0x7678, 0x767C, 0x7680, 0x7683, 0x7688,
	 0x768B, 0x768E, 0x7696, 0x7693, 0x7699, 0x769A, 0x76B0, 0x76B4,
	 0x76B8, 0x76B9, 0x76BA, 0x76C2, 0x76CD, 0x76D6, 0x76D2, 0x76DE,
	 0x76E1, 0x76E5, 0x76E7, 0x76EA, 0x862F, 0x76FB, 0x7708, 0x7707,
	 0x7704, 0x7729, 0x7724, 0x771E, 0x7725, 0x7726, 0x771B, 0x7737,
	 0x7738, 0x7747, 0x775A, 0x7768, 0x776B, 0x775B, 0x7765, 0x777F,
	 0x777E, 0x7779, 0x778E, 0x778B, 0x7791, 0x77A0, 0x779E, 0x77B0,
	 0x77B6, 0x77B9, 0x77BF, 0x77BC, 0x77BD, 0x77BB, 0x77C7, 0x77CD,
	 0x77D7, 0x77DA, 0x77DC, 0x77E3, 0x77EE, 0x77FC, 0x780C, 0x7812,
	 0x7926, 0x7820, 0x792A, 0x7845, 0x788E, 0x7874, 0x7886, 0x787C,
	 0x789A, 0x788C, 0x78A3, 0x78B5, 0x78AA, 0x78AF, 0x78D1, 0x78C6,
	 0x78CB, 0x78D4, 0x78BE, 0x78BC, 0x78C5, 0x78CA, 0x78EC,
/* 0xE3A1 */	 0x78E7, 0x78DA, 0x78FD, 0x78F4, 0x7907, 0x7912, 0x7911,
	 0x7919, 0x792C, 0x792B, 0x7940, 0x7960, 0x7957, 0x795F, 0x795A,
	 0x7955, 0x7953, 0x797A, 0x797F, 0x798A, 0x799D, 0x79A7, 0x9F4B,
	 0x79AA, 0x79AE, 0x79B3, 0x79B9, 0x79BA, 0x79C9, 0x79D5, 0x79E7,
	 0x79EC, 0x79E1, 0x79E3, 0x7A08, 0x7A0D, 0x7A18, 0x7A19, 0x7A20,
	 0x7A1F, 0x7980, 0x7A31, 0x7A3B, 0x7A3E, 0x7A37, 0x7A43, 0x7A57,
	 0x7A49, 0x7A61, 0x7A62, 0x7A69, 0x9F9D, 0x7A70, 0x7A79, 0x7A7D,
	 0x7A88, 0x7A97, 0x7A95, 0x7A98, 0x7A96, 0x7AA9, 0x7AC8, 0x7AB0,
	 0x7AB6, 0x7AC5, 0x7AC4, 0x7ABF, 0x9083, 0x7AC7, 0x7ACA, 0x7ACD,
	 0x7ACF, 0x7AD5, 0x7AD3, 0x7AD9, 0x7ADA, 0x7ADD, 0x7AE1, 0x7AE2,
	 0x7AE6, 0x7AED, 0x7AF0, 0x7B02, 0x7B0F, 0x7B0A, 0x7B06, 0x7B33,
	 0x7B18, 0x7B19, 0x7B1E, 0x7B35, 0x7B28, 0x7B36, 0x7B50,
/* 0xE4A1 */	 0x7B7A, 0x7B04, 0x7B4D, 0x7B0B, 0x7B4C, 0x7B45, 0x7B75,
	 0x7B65, 0x7B74, 0x7B67, 0x7B70, 0x7B71, 0x7B6C, 0x7B6E, 0x7B9D,
	 0x7B98, 0x7B9F, 0x7B8D, 0x7B9C, 0x7B9A, 0x7B8B, 0x7B92, 0x7B8F,
	 0x7B5D, 0x7B99, 0x7BCB, 0x7BC1, 0x7BCC, 0x7BCF, 0x7BB4, 0x7BC6,
	 0x7BDD, 0x7BE9, 0x7C11, 0x7C14, 0x7BE6, 0x7BE5, 0x7C60, 0x7C00,
	 0x7C07, 0x7C13, 0x7BF3, 0x7BF7, 0x7C17, 0x7C0D, 0x7BF6, 0x7C23,
	 0x7C27, 0x7C2A, 0x7C1F, 0x7C37, 0x7C2B, 0x7C3D, 0x7C4C, 0x7C43,
	 0x7C54, 0x7C4F, 0x7C40, 0x7C50, 0x7C58, 0x7C5F, 0x7C64, 0x7C56,
	 0x7C65, 0x7C6C, 0x7C75, 0x7C83, 0x7C90, 0x7CA4, 0x7CAD, 0x7CA2,
	 0x7CAB, 0x7CA1, 0x7CA8, 0x7CB3, 0x7CB2, 0x7CB1, 0x7CAE, 0x7CB9,
	 0x7CBD, 0x7CC0, 0x7CC5, 0x7CC2, 0x7CD8, 0x7CD2, 0x7CDC, 0x7CE2,
	 0x9B3B, 0x7CEF, 0x7CF2, 0x7CF4, 0x7CF6, 0x7CFA, 0x7D06,
/* 0xE5A1 */	 0x7D02, 0x7D1C, 0x7D15, 0x7D0A, 0x7D45, 0x7D4B, 0x7D2E,
	 0x7D32, 0x7D3F, 0x7D35, 0x7D46, 0x7D73, 0x7D56, 0x7D4E, 0x7D72,
	 0x7D68, 0x7D6E, 0x7D4F, 0x7D63, 0x7D93, 0x7D89, 0x7D5B, 0x7D8F,
	 0x7D7D, 0x7D9B, 0x7DBA, 0x7DAE, 0x7DA3, 0x7DB5, 0x7DC7, 0x7DBD,
	 0x7DAB, 0x7E3D, 0x7DA2, 0x7DAF, 0x7DDC, 0x7DB8, 0x7D9F, 0x7DB0,
	 0x7DD8, 0x7DDD, 0x7DE4, 0x7DDE, 0x7DFB, 0x7DF2, 0x7DE1, 0x7E05,
	 0x7E0A, 0x7E23, 0x7E21, 0x7E12, 0x7E31, 0x7E1F, 0x7E09, 0x7E0B,
	 0x7E22, 0x7E46, 0x7E66, 0x7E3B, 0x7E35, 0x7E39, 0x7E43, 0x7E37,
	 0x7E32, 0x7E3A, 0x7E67, 0x7E5D, 0x7E56, 0x7E5E, 0x7E59, 0x7E5A,
	 0x7E79, 0x7E6A, 0x7E69, 0x7E7C, 0x7E7B, 0x7E83, 0x7DD5, 0x7E7D,
	 0x8FAE, 0x7E7F, 0x7E88, 0x7E89, 0x7E8C, 0x7E92, 0x7E90, 0x7E93,
	 0x7E94, 0x7E96, 0x7E8E, 0x7E9B, 0x7E9C, 0x7F38, 0x7F3A,
/* 0xE6A1 */	 0x7F45, 0x7F4C, 0x7F4D, 0x7F4E, 0x7F50, 0x7F51, 0x7F55,
	 0x7F54, 0x7F58, 0x7F5F, 0x7F60, 0x7F68, 0x7F69, 0x7F67, 0x7F78,
	 0x7F82, 0x7F86, 0x7F83, 0x7F88, 0x7F87, 0x7F8C, 0x7F94, 0x7F9E,
	 0x7F9D, 0x7F9A, 0x7FA3, 0x7FAF, 0x7FB2, 0x7FB9, 0x7FAE, 0x7FB6,
	 0x7FB8, 0x8B71, 0x7FC5, 0x7FC6, 0x7FCA, 0x7FD5, 0x7FD4, 0x7FE1,
	 0x7FE6, 0x7FE9, 0x7FF3, 0x7FF9, 0x98DC, 0x8006, 0x8004, 0x800B,
	 0x8012, 0x8018, 0x8019, 0x801C, 0x8021, 0x8028, 0x803F, 0x803B,
	 0x804A, 0x8046, 0x8052, 0x8058, 0x805A, 0x805F, 0x8062, 0x8068,
	 0x8073, 0x8072, 0x8070, 0x8076, 0x8079, 0x807D, 0x807F, 0x8084,
	 0x8086, 0x8085, 0x809B, 0x8093, 0x809A, 0x80AD, 0x5190, 0x80AC,
	 0x80DB, 0x80E5, 0x80D9, 0x80DD, 0x80C4, 0x80DA, 0x80D6, 0x8109,
	 0x80EF, 0x80F1, 0x811B, 0x8129, 0x8123, 0x812F, 0x814B,
/* 0xE7A1 */	 0x968B, 0x8146, 0x813E, 0x8153, 0x8151, 0x80FC, 0x8171,
	 0x816E, 0x8165, 0x8166, 0x8174, 0x8183, 0x8188, 0x818A, 0x8180,
	 0x8182, 0x81A0, 0x8195, 0x81A4, 0x81A3, 0x815F, 0x8193, 0x81A9,
	 0x81B0, 0x81B5, 0x81BE, 0x81B8, 0x81BD, 0x81C0, 0x81C2, 0x81BA,
	 0x81C9, 0x81CD, 0x81D1, 0x81D9, 0x81D8, 0x81C8, 0x81DA, 0x81DF,
	 0x81E0, 0x81E7, 0x81FA, 0x81FB, 0x81FE, 0x8201, 0x8202, 0x8205,
	 0x8207, 0x820A, 0x820D, 0x8210, 0x8216, 0x8229, 0x822B, 0x8238,
	 0x8233, 0x8240, 0x8259, 0x8258, 0x825D, 0x825A, 0x825F, 0x8264,
	 0x8262, 0x8268, 0x826A, 0x826B, 0x822E, 0x8271, 0x8277, 0x8278,
	 0x827E, 0x828D, 0x8292, 0x82AB, 0x829F, 0x82BB, 0x82AC, 0x82E1,
	 0x82E3, 0x82DF, 0x82D2, 0x82F4, 0x82F3, 0x82FA, 0x8393, 0x8303,
	 0x82FB, 0x82F9, 0x82DE, 0x8306, 0x82DC, 0x8309, 0x82D9,
/* 0xE8A1 */	 0x8335, 0x8334, 0x8316, 0x8332, 0x8331, 0x8340, 0x8339,
	 0x8350, 0x8345, 0x832F, 0x832B, 0x8317, 0x8318, 0x8385, 0x839A,
	 0x83AA, 0x839F, 0x83A2, 0x8396, 0x8323, 0x838E, 0x8387, 0x838A,
	 0x837C, 0x83B5, 0x8373, 0x8375, 0x83A0, 0x8389, 0x83A8, 0x83F4,
	 0x8413, 0x83EB, 0x83CE, 0x83FD, 0x8403, 0x83D8, 0x840B, 0x83C1,
	 0x83F7, 0x8407, 0x83E0, 0x83F2, 0x840D, 0x8422, 0x8420, 0x83BD,
	 0x8438, 0x8506, 0x83FB, 0x846D, 0x842A, 0x843C, 0x855A, 0x8484,
	 0x8477, 0x846B, 0x84AD, 0x846E, 0x8482, 0x8469, 0x8446, 0x842C,
	 0x846F, 0x8479, 0x8435, 0x84CA, 0x8462, 0x84B9, 0x84BF, 0x849F,
	 0x84D9, 0x84CD, 0x84BB, 0x84DA, 0x84D0, 0x84C1, 0x84C6, 0x84D6,
	 0x84A1, 0x8521, 0x84FF, 0x84F4, 0x8517, 0x8518, 0x852C, 0x851F,
	 0x8515, 0x8514, 0x84FC, 0x8540, 0x8563, 0x8558, 0x8548,
/* 0xE9A1 */	 0x8541, 0x8602, 0x854B, 0x8555, 0x8580, 0x85A4, 0x8588,
	 0x8591, 0x858A, 0x85A8, 0x856D, 0x8594, 0x859B, 0x85EA, 0x8587,
	 0x859C, 0x8577, 0x857E, 0x8590, 0x85C9, 0x85BA, 0x85CF, 0x85B9,
	 0x85D0, 0x85D5, 0x85DD, 0x85E5, 0x85DC, 0x85F9, 0x860A, 0x8613,
	 0x860B, 0x85FE, 0x85FA, 0x8606, 0x8622, 0x861A, 0x8630, 0x863F,
	 0x864D, 0x4E55, 0x8654, 0x865F, 0x8667, 0x8671, 0x8693, 0x86A3,
	 0x86A9, 0x86AA, 0x868B, 0x868C, 0x86B6, 0x86AF, 0x86C4, 0x86C6,
	 0x86B0, 0x86C9, 0x8823, 0x86AB, 0x86D4, 0x86DE, 0x86E9, 0x86EC,
	 0x86DF, 0x86DB, 0x86EF, 0x8712, 0x8706, 0x8708, 0x8700, 0x8703,
	 0x86FB, 0x8711, 0x8709, 0x870D, 0x86F9, 0x870A, 0x8734, 0x873F,
	 0x8737, 0x873B, 0x8725, 0x8729, 0x871A, 0x8760, 0x875F, 0x8778,
	 0x874C, 0x874E, 0x8774, 0x8757, 0x8768, 0x876E, 0x8759,
/* 0xEAA1 */	 0x8753, 0x8763, 0x876A, 0x8805, 0x87A2, 0x879F, 0x8782,
	 0x87AF, 0x87CB, 0x87BD, 0x87C0, 0x87D0, 0x96D6, 0x87AB, 0x87C4,
	 0x87B3, 0x87C7, 0x87C6, 0x87BB, 0x87EF, 0x87F2, 0x87E0, 0x880F,
	 0x880D, 0x87FE, 0x87F6, 0x87F7, 0x880E, 0x87D2, 0x8811, 0x8816,
	 0x8815, 0x8822, 0x8821, 0x8831, 0x8836, 0x8839, 0x8827, 0x883B,
	 0x8844, 0x8842, 0x8852, 0x8859, 0x885E, 0x8862, 0x886B, 0x8881,
	 0x887E, 0x889E, 0x8875, 0x887D, 0x88B5, 0x8872, 0x8882, 0x8897,
	 0x8892, 0x88AE, 0x8899, 0x88A2, 0x888D, 0x88A4, 0x88B0, 0x88BF,
	 0x88B1, 0x88C3, 0x88C4, 0x88D4, 0x88D8, 0x88D9, 0x88DD, 0x88F9,
	 0x8902, 0x88FC, 0x88F4, 0x88E8, 0x88F2, 0x8904, 0x890C, 0x890A,
	 0x8913, 0x8943, 0x891E, 0x8925, 0x892A, 0x892B, 0x8941, 0x8944,
	 0x893B, 0x8936, 0x8938, 0x894C, 0x891D, 0x8960, 0x895E,
/* 0xEBA1 */	 0x8966, 0x8964, 0x896D, 0x896A, 0x896F, 0x8974, 0x8977,
	 0x897E, 0x8983, 0x8988, 0x898A, 0x8993, 0x8998, 0x89A1, 0x89A9,
	 0x89A6, 0x89AC, 0x89AF, 0x89B2, 0x89BA, 0x89BD, 0x89BF, 0x89C0,
	 0x89DA, 0x89DC, 0x89DD, 0x89E7, 0x89F4, 0x89F8, 0x8A03, 0x8A16,
	 0x8A10, 0x8A0C, 0x8A1B, 0x8A1D, 0x8A25, 0x8A36, 0x8A41, 0x8A5B,
	 0x8A52, 0x8A46, 0x8A48, 0x8A7C, 0x8A6D, 0x8A6C, 0x8A62, 0x8A85,
	 0x8A82, 0x8A84, 0x8AA8, 0x8AA1, 0x8A91, 0x8AA5, 0x8AA6, 0x8A9A,
	 0x8AA3, 0x8AC4, 0x8ACD, 0x8AC2, 0x8ADA, 0x8AEB, 0x8AF3, 0x8AE7,
	 0x8AE4, 0x8AF1, 0x8B14, 0x8AE0, 0x8AE2, 0x8AF7, 0x8ADE, 0x8ADB,
	 0x8B0C, 0x8B07, 0x8B1A, 0x8AE1, 0x8B16, 0x8B10, 0x8B17, 0x8B20,
	 0x8B33, 0x97AB, 0x8B26, 0x8B2B, 0x8B3E, 0x8B28, 0x8B41, 0x8B4C,
	 0x8B4F, 0x8B4E, 0x8B49, 0x8B56, 0x8B5B, 0x8B5A, 0x8B6B,
/* 0xECA1 */	 0x8B5F, 0x8B6C, 0x8B6F, 0x8B74, 0x8B7D, 0x8B80, 0x8B8C,
	 0x8B8E, 0x8B92, 0x8B93, 0x8B96, 0x8B99, 0x8B9A, 0x8C3A, 0x8C41,
	 0x8C3F, 0x8C48, 0x8C4C, 0x8C4E, 0x8C50, 0x8C55, 0x8C62, 0x8C6C,
	 0x8C78, 0x8C7A, 0x8C82, 0x8C89, 0x8C85, 0x8C8A, 0x8C8D, 0x8C8E,
	 0x8C94, 0x8C7C, 0x8C98, 0x621D, 0x8CAD, 0x8CAA, 0x8CBD, 0x8CB2,
	 0x8CB3, 0x8CAE, 0x8CB6, 0x8CC8, 0x8CC1, 0x8CE4, 0x8CE3, 0x8CDA,
	 0x8CFD, 0x8CFA, 0x8CFB, 0x8D04, 0x8D05, 0x8D0A, 0x8D07, 0x8D0F,
	 0x8D0D, 0x8D10, 0x9F4E, 0x8D13, 0x8CCD, 0x8D14, 0x8D16, 0x8D67,
	 0x8D6D, 0x8D71, 0x8D73, 0x8D81, 0x8D99, 0x8DC2, 0x8DBE, 0x8DBA,
	 0x8DCF, 0x8DDA, 0x8DD6, 0x8DCC, 0x8DDB, 0x8DCB, 0x8DEA, 0x8DEB,
	 0x8DDF, 0x8DE3, 0x8DFC, 0x8E08, 0x8E09, 0x8DFF, 0x8E1D, 0x8E1E,
	 0x8E10, 0x8E1F, 0x8E42, 0x8E35, 0x8E30, 0x8E34, 0x8E4A,
/* 0xEDA1 */	 0x8E47, 0x8E49, 0x8E4C, 0x8E50, 0x8E48, 0x8E59, 0x8E64,
	 0x8E60, 0x8E2A, 0x8E63, 0x8E55, 0x8E76, 0x8E72, 0x8E7C, 0x8E81,
	 0x8E87, 0x8E85, 0x8E84, 0x8E8B, 0x8E8A, 0x8E93, 0x8E91, 0x8E94,
	 0x8E99, 0x8EAA, 0x8EA1, 0x8EAC, 0x8EB0, 0x8EC6, 0x8EB1, 0x8EBE,
	 0x8EC5, 0x8EC8, 0x8ECB, 0x8EDB, 0x8EE3, 0x8EFC, 0x8EFB, 0x8EEB,
	 0x8EFE, 0x8F0A, 0x8F05, 0x8F15, 0x8F12, 0x8F19, 0x8F13, 0x8F1C,
	 0x8F1F, 0x8F1B, 0x8F0C, 0x8F26, 0x8F33, 0x8F3B, 0x8F39, 0x8F45,
	 0x8F42, 0x8F3E, 0x8F4C, 0x8F49, 0x8F46, 0x8F4E, 0x8F57, 0x8F5C,
	 0x8F62, 0x8F63, 0x8F64, 0x8F9C, 0x8F9F, 0x8FA3, 0x8FAD, 0x8FAF,
	 0x8FB7, 0x8FDA, 0x8FE5, 0x8FE2, 0x8FEA, 0x8FEF, 0x9087, 0x8FF4,
	 0x9005, 0x8FF9, 0x8FFA, 0x9011, 0x9015, 0x9021, 0x900D, 0x901E,
	 0x9016, 0x900B, 0x9027, 0x9036, 0x9035, 0x9039, 0x8FF8,
/* 0xEEA1 */	 0x904F, 0x9050, 0x9051, 0x9052, 0x900E, 0x9049, 0x903E,
	 0x9056, 0x9058, 0x905E, 0x9068, 0x906F, 0x9076, 0x96A8, 0x9072,
	 0x9082, 0x907D, 0x9081, 0x9080, 0x908A, 0x9089, 0x908F, 0x90A8,
	 0x90AF, 0x90B1, 0x90B5, 0x90E2, 0x90E4, 0x6248, 0x90DB, 0x9102,
	 0x9112, 0x9119, 0x9132, 0x9130, 0x914A, 0x9156, 0x9158, 0x9163,
	 0x9165, 0x9169, 0x9173, 0x9172, 0x918B, 0x9189, 0x9182, 0x91A2,
	 0x91AB, 0x91AF, 0x91AA, 0x91B5, 0x91B4, 0x91BA, 0x91C0, 0x91C1,
	 0x91C9, 0x91CB, 0x91D0, 0x91D6, 0x91DF, 0x91E1, 0x91DB, 0x91FC,
	 0x91F5, 0x91F6, 0x921E, 0x91FF, 0x9214, 0x922C, 0x9215, 0x9211,
	 0x925E, 0x9257, 0x9245, 0x9249, 0x9264, 0x9248, 0x9295, 0x923F,
	 0x924B, 0x9250, 0x929C, 0x9296, 0x9293, 0x929B, 0x925A, 0x92CF,
	 0x92B9, 0x92B7, 0x92E9, 0x930F, 0x92FA, 0x9344, 0x932E,
/* 0xEFA1 */	 0x9319, 0x9322, 0x931A, 0x9323, 0x933A, 0x9335, 0x933B,
	 0x935C, 0x9360, 0x937C, 0x936E, 0x9356, 0x93B0, 0x93AC, 0x93AD,
	 0x9394, 0x93B9, 0x93D6, 0x93D7, 0x93E8, 0x93E5, 0x93D8, 0x93C3,
	 0x93DD, 0x93D0, 0x93C8, 0x93E4, 0x941A, 0x9414, 0x9413, 0x9403,
	 0x9407, 0x9410, 0x9436, 0x942B, 0x9435, 0x9421, 0x943A, 0x9441,
	 0x9452, 0x9444, 0x945B, 0x9460, 0x9462, 0x945E, 0x946A, 0x9229,
	 0x9470, 0x9475, 0x9477, 0x947D, 0x945A, 0x947C, 0x947E, 0x9481,
	 0x947F, 0x9582, 0x9587, 0x958A, 0x9594, 0x9596, 0x9598, 0x9599,
	 0x95A0, 0x95A8, 0x95A7, 0x95AD, 0x95BC, 0x95BB, 0x95B9, 0x95BE,
	 0x95CA, 0x6FF6, 0x95C3, 0x95CD, 0x95CC, 0x95D5, 0x95D4, 0x95D6,
	 0x95DC, 0x95E1, 0x95E5, 0x95E2, 0x9621, 0x9628, 0x962E, 0x962F,
	 0x9642, 0x964C, 0x964F, 0x964B, 0x9677, 0x965C, 0x965E,
/* 0xF0A1 */	 0x965D, 0x965F, 0x9666, 0x9672, 0x966C, 0x968D, 0x9698,
	 0x9695, 0x9697, 0x96AA, 0x96A7, 0x96B1, 0x96B2, 0x96B0, 0x96B4,
	 0x96B6, 0x96B8, 0x96B9, 0x96CE, 0x96CB, 0x96C9, 0x96CD, 0x894D,
	 0x96DC, 0x970D, 0x96D5, 0x96F9, 0x9704, 0x9706, 0x9708, 0x9713,
	 0x970E, 0x9711, 0x970F, 0x9716, 0x9719, 0x9724, 0x972A, 0x9730,
	 0x9739, 0x973D, 0x973E, 0x9744, 0x9746, 0x9748, 0x9742, 0x9749,
	 0x975C, 0x9760, 0x9764, 0x9766, 0x9768, 0x52D2, 0x976B, 0x9771,
	 0x9779, 0x9785, 0x977C, 0x9781, 0x977A, 0x9786, 0x978B, 0x978F,
	 0x9790, 0x979C, 0x97A8, 0x97A6, 0x97A3, 0x97B3, 0x97B4, 0x97C3,
	 0x97C6, 0x97C8, 0x97CB, 0x97DC, 0x97ED, 0x9F4F, 0x97F2, 0x7ADF,
	 0x97F6, 0x97F5, 0x980F, 0x980C, 0x9838, 0x9824, 0x9821, 0x9837,
	 0x983D, 0x9846, 0x984F, 0x984B, 0x986B, 0x986F, 0x9870,
/* 0xF1A1 */	 0x9871, 0x9874, 0x9873, 0x98AA, 0x98AF, 0x98B1, 0x98B6,
	 0x98C4, 0x98C3, 0x98C6, 0x98E9, 0x98EB, 0x9903, 0x9909, 0x9912,
	 0x9914, 0x9918, 0x9921, 0x991D, 0x991E, 0x9924, 0x9920, 0x992C,
	 0x992E, 0x993D, 0x993E, 0x9942, 0x9949, 0x9945, 0x9950, 0x994B,
	 0x9951, 0x9952, 0x994C, 0x9955, 0x9997, 0x9998, 0x99A5, 0x99AD,
	 0x99AE, 0x99BC, 0x99DF, 0x99DB, 0x99DD, 0x99D8, 0x99D1, 0x99ED,
	 0x99EE, 0x99F1, 0x99F2, 0x99FB, 0x99F8, 0x9A01, 0x9A0F, 0x9A05,
	 0x99E2, 0x9A19, 0x9A2B, 0x9A37, 0x9A45, 0x9A42, 0x9A40, 0x9A43,
	 0x9A3E, 0x9A55, 0x9A4D, 0x9A5B, 0x9A57, 0x9A5F, 0x9A62, 0x9A65,
	 0x9A64, 0x9A69, 0x9A6B, 0x9A6A, 0x9AAD, 0x9AB0, 0x9ABC, 0x9AC0,
	 0x9ACF, 0x9AD1, 0x9AD3, 0x9AD4, 0x9ADE, 0x9ADF, 0x9AE2, 0x9AE3,
	 0x9AE6, 0x9AEF, 0x9AEB, 0x9AEE, 0x9AF4, 0x9AF1, 0x9AF7,
/* 0xF2A1 */	 0x9AFB, 0x9B06, 0x9B18, 0x9B1A, 0x9B1F, 0x9B22, 0x9B23,
	 0x9B25, 0x9B27, 0x9B28, 0x9B29, 0x9B2A, 0x9B2E, 0x9B2F, 0x9B32,
	 0x9B44, 0x9B43, 0x9B4F, 0x9B4D, 0x9B4E, 0x9B51, 0x9B58, 0x9B74,
	 0x9B93, 0x9B83, 0x9B91, 0x9B96, 0x9B97, 0x9B9F, 0x9BA0, 0x9BA8,
	 0x9BB4, 0x9BC0, 0x9BCA, 0x9BB9, 0x9BC6, 0x9BCF, 0x9BD1, 0x9BD2,
	 0x9BE3, 0x9BE2, 0x9BE4, 0x9BD4, 0x9BE1, 0x9C3A, 0x9BF2, 0x9BF1,
	 0x9BF0, 0x9C15, 0x9C14, 0x9C09, 0x9C13, 0x9C0C, 0x9C06, 0x9C08,
	 0x9C12, 0x9C0A, 0x9C04, 0x9C2E, 0x9C1B, 0x9C25, 0x9C24, 0x9C21,
	 0x9C30, 0x9C47, 0x9C32, 0x9C46, 0x9C3E, 0x9C5A, 0x9C60, 0x9C67,
	 0x9C76, 0x9C78, 0x9CE7, 0x9CEC, 0x9CF0, 0x9D09, 0x9D08, 0x9CEB,
	 0x9D03, 0x9D06, 0x9D2A, 0x9D26, 0x9DAF, 0x9D23, 0x9D1F, 0x9D44,
	 0x9D15, 0x9D12, 0x9D41, 0x9D3F, 0x9D3E, 0x9D46, 0x9D48,
/* 0xF3A1 */	 0x9D5D, 0x9D5E, 0x9D64, 0x9D51, 0x9D50, 0x9D59, 0x9D72,
	 0x9D89, 0x9D87, 0x9DAB, 0x9D6F, 0x9D7A, 0x9D9A, 0x9DA4, 0x9DA9,
	 0x9DB2, 0x9DC4, 0x9DC1, 0x9DBB, 0x9DB8, 0x9DBA, 0x9DC6, 0x9DCF,
	 0x9DC2, 0x9DD9, 0x9DD3, 0x9DF8, 0x9DE6, 0x9DED, 0x9DEF, 0x9DFD,
	 0x9E1A, 0x9E1B, 0x9E1E, 0x9E75, 0x9E79, 0x9E7D, 0x9E81, 0x9E88,
	 0x9E8B, 0x9E8C, 0x9E92, 0x9E95, 0x9E91, 0x9E9D, 0x9EA5, 0x9EA9,
	 0x9EB8, 0x9EAA, 0x9EAD, 0x9761, 0x9ECC, 0x9ECE, 0x9ECF, 0x9ED0,
	 0x9ED4, 0x9EDC, 0x9EDE, 0x9EDD, 0x9EE0, 0x9EE5, 0x9EE8, 0x9EEF,
	 0x9EF4, 0x9EF6, 0x9EF7, 0x9EF9, 0x9EFB, 0x9EFC, 0x9EFD, 0x9F07,
	 0x9F08, 0x76B7, 0x9F15, 0x9F21, 0x9F2C, 0x9F3E, 0x9F4A, 0x9F52,
	 0x9F54, 0x9F63, 0x9F5F, 0x9F60, 0x9F61, 0x9F66, 0x9F67, 0x9F6C,
	 0x9F6A, 0x9F77, 0x9F72, 0x9F76, 0x9F95, 0x9F9C, 0x9FA0,
/* 0xF4A1 */	 0x582F, 0x69C7, 0x9059, 0x7464, 0x51DC, 0x7199,
};

static wchar_t c2u_nec[] = {
/*  13 */	 0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466,
	 0x2467, 0x2468, 0x2469, 0x246A, 0x246B, 0x246C, 0x246D, 0x246E,
	 0x246F, 0x2470, 0x2471, 0x2472, 0x2473, 0x2160, 0x2161, 0x2162,
	 0x2163, 0x2164, 0x2165, 0x2166, 0x2167, 0x2168, 0x2169, 0xFFFF,
	 0x3349, 0x3314, 0x3322, 0x334D, 0x3318, 0x3327, 0x3303, 0x3336,
	 0x3351, 0x3357, 0x330D, 0x3326, 0x3323, 0x332B, 0x334A, 0x333B,
	 0x339C, 0x339D, 0x339E, 0x338E, 0x338F, 0x33C4, 0x33A1, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x337B,
	 0x301D, 0x301F, 0x2116, 0x33CD, 0x2121, 0x32A4, 0x32A5, 0x32A6,
	 0x32A7, 0x32A8, 0x3231, 0x3232, 0x3239, 0x337E, 0x337D, 0x337C,
	 0x2252, 0x2261, 0x222B, 0x222E, 0x2211, 0x221A, 0x22A5, 0x2220,
	 0x221F, 0x22BF, 0x2235, 0x2229, 0x222A, 0xFFFF, 0xFFFF,
};

static wchar_t c2u_nec2[] = {
/*  89 */	 0x7E8A, 0x891C, 0x9348, 0x9288, 0x84DC, 0x4FC9, 0x70BB,
	 0x6631, 0x68C8, 0x92F9, 0x66FB, 0x5F45, 0x4E28, 0x4EE1, 0x4EFC,
	 0x4F00, 0x4F03, 0x4F39, 0x4F56, 0x4F92, 0x4F8A, 0x4F9A, 0x4F94,
	 0x4FCD, 0x5040, 0x5022, 0x4FFF, 0x501E, 0x5046, 0x5070, 0x5042,
	 0x5094, 0x50F4, 0x50D8, 0x514A, 0x5164, 0x519D, 0x51BE, 0x51EC,
	 0x5215, 0x529C, 0x52A6, 0x52C0, 0x52DB, 0x5300, 0x5307, 0x5324,
	 0x5372, 0x5393, 0x53B2, 0x53DD, 0xFA0E, 0x549C, 0x548A, 0x54A9,
	 0x54FF, 0x5586, 0x5759, 0x5765, 0x57AC, 0x57C8, 0x57C7, 0xFA0F,
	 0xFA10, 0x589E, 0x58B2, 0x590B, 0x5953, 0x595B, 0x595D, 0x5963,
	 0x59A4, 0x59BA, 0x5B56, 0x5BC0, 0x752F, 0x5BD8, 0x5BEC, 0x5C1E,
	 0x5CA6, 0x5CBA, 0x5CF5, 0x5D27, 0x5D53, 0xFA11, 0x5D42, 0x5D6D,
	 0x5DB8, 0x5DB9, 0x5DD0, 0x5F21, 0x5F34, 0x5F67, 0x5FB7,
/*  90 */	 0x5FDE, 0x605D, 0x6085, 0x608A, 0x60DE, 0x60D5, 0x6120,
	 0x60F2, 0x6111, 0x6137, 0x6130, 0x6198, 0x6213, 0x62A6, 0x63F5,
	 0x6460, 0x649D, 0x64CE, 0x654E, 0x6600, 0x6615, 0x663B, 0x6609,
	 0x662E, 0x661E, 0x6624, 0x6665, 0x6657, 0x6659, 0xFA12, 0x6673,
	 0x6699, 0x66A0, 0x66B2, 0x66BF, 0x66FA, 0x670E, 0xF929, 0x6766,
	 0x67BB, 0x6852, 0x67C0, 0x6801, 0x6844, 0x68CF, 0xFA13, 0x6968,
	 0xFA14, 0x6998, 0x69E2, 0x6A30, 0x6A6B, 0x6A46, 0x6A73, 0x6A7E,
	 0x6AE2, 0x6AE4, 0x6BD6, 0x6C3F, 0x6C5C, 0x6C86, 0x6C6F, 0x6CDA,
	 0x6D04, 0x6D87, 0x6D6F, 0x6D96, 0x6DAC, 0x6DCF, 0x6DF8, 0x6DF2,
	 0x6DFC, 0x6E39, 0x6E5C, 0x6E27, 0x6E3C, 0x6EBF, 0x6F88, 0x6FB5,
	 0x6FF5, 0x7005, 0x7007, 0x7028, 0x7085, 0x70AB, 0x710F, 0x7104,
	 0x715C, 0x7146, 0x7147, 0xFA15, 0x71C1, 0x71FE, 0x72B1,
/*  91 */	 0x72BE, 0x7324, 0xFA16, 0x7377, 0x73BD, 0x73C9, 0x73D6,
	 0x73E3, 0x73D2, 0x7407, 0x73F5, 0x7426, 0x742A, 0x7429, 0x742E,
	 0x7462, 0x7489, 0x749F, 0x7501, 0x756F, 0x7682, 0x769C, 0x769E,
	 0x769B, 0x76A6, 0xFA17, 0x7746, 0x52AF, 0x7821, 0x784E, 0x7864,
	 0x787A, 0x7930, 0xFA18, 0xFA19, 0xFA1A, 0x7994, 0xFA1B, 0x799B,
	 0x7AD1, 0x7AE7, 0xFA1C, 0x7AEB, 0x7B9E, 0xFA1D, 0x7D48, 0x7D5C,
	 0x7DB7, 0x7DA0, 0x7DD6, 0x7E52, 0x7F47, 0x7FA1, 0xFA1E, 0x8301,
	 0x8362, 0x837F, 0x83C7, 0x83F6, 0x8448, 0x84B4, 0x8553, 0x8559,
	 0x856B, 0xFA1F, 0x85B0, 0xFA20, 0xFA21, 0x8807, 0x88F5, 0x8A12,
	 0x8A37, 0x8A79, 0x8AA7, 0x8ABE, 0x8ADF, 0xFA22, 0x8AF6, 0x8B53,
	 0x8B7F, 0x8CF0, 0x8CF4, 0x8D12, 0x8D76, 0xFA23, 0x8ECF, 0xFA24,
	 0xFA25, 0x9067, 0x90DE, 0xFA26, 0x9115, 0x9127, 0x91DA,
/*  92 */	 0x91D7, 0x91DE, 0x91ED, 0x91EE, 0x91E4, 0x91E5, 0x9206,
	 0x9210, 0x920A, 0x923A, 0x9240, 0x923C, 0x924E, 0x9259, 0x9251,
	 0x9239, 0x9267, 0x92A7, 0x9277, 0x9278, 0x92E7, 0x92D7, 0x92D9,
	 0x92D0, 0xFA27, 0x92D5, 0x92E0, 0x92D3, 0x9325, 0x9321, 0x92FB,
	 0xFA28, 0x931E, 0x92FF, 0x931D, 0x9302, 0x9370, 0x9357, 0x93A4,
	 0x93C6, 0x93DE, 0x93F8, 0x9431, 0x9445, 0x9448, 0x9592, 0xF9DC,
	 0xFA29, 0x969D, 0x96AF, 0x9733, 0x973B, 0x9743, 0x974D, 0x974F,
	 0x9751, 0x9755, 0x9857, 0x9865, 0xFA2A, 0xFA2B, 0x9927, 0xFA2C,
	 0x999E, 0x9A4E, 0x9AD9, 0x9ADC, 0x9B75, 0x9B72, 0x9B8F, 0x9BB1,
	 0x9BBB, 0x9C00, 0x9D70, 0x9D6B, 0xFA2D, 0x9E19, 0x9ED1, 0xFFFF,
	 0xFFFF, 0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176,
	 0x2177, 0x2178, 0x2179, 0xFFE2, 0xFFE4, 0xFF07, 0xFF02,
};

static wchar_t c2u_jisx0212[] = {
/* 0xA2A1 */	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x02D8,
	 0x02C7, 0x00B8, 0x02D9, 0x02DD, 0x00AF, 0x02DB, 0x02DA, 0xFF5E,
	 0x0384, 0x0385, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x00A1, 0xFFE4, 0x00BF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0x00BA, 0x00AA, 0x00A9, 0x00AE, 0x2122,
	 0x00A4, 0x2116, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xA6A1 */	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0x0386, 0x0388, 0x0389, 0x038A, 0x03AA, 0xFFFF, 0x038C,
	 0xFFFF, 0x038E, 0x03AB, 0xFFFF, 0x038F, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0x03AC, 0x03AD, 0x03AE, 0x03AF, 0x03CA, 0x0390, 0x03CC,
	 0x03C2, 0x03CD, 0x03CB, 0x03B0, 0x03CE, 0xFFFF, 0xFFFF,
/* 0xA7A1 */	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407,
	 0x0408, 0x0409, 0x040A, 0x040B, 0x040C, 0x040E, 0x040F, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457,
	 0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x045E, 0x045F,
/* 0xA9A1 */	 0x00C6, 0x0110, 0xFFFF, 0x0126, 0xFFFF, 0x0132, 0xFFFF,
	 0x0141, 0x013F, 0xFFFF, 0x014A, 0x00D8, 0x0152, 0xFFFF, 0x0166,
	 0x00DE, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0x00E6, 0x0111, 0x00F0, 0x0127, 0x0131, 0x0133, 0x0138,
	 0x0142, 0x0140, 0x0149, 0x014B, 0x00F8, 0x0153, 0x00DF, 0x0167,
	 0x00FE, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xAAA1 */	 0x00C1, 0x00C0, 0x00C4, 0x00C2, 0x0102, 0x01CD, 0x0100,
	 0x0104, 0x00C5, 0x00C3, 0x0106, 0x0108, 0x010C, 0x00C7, 0x010A,
	 0x010E, 0x00C9, 0x00C8, 0x00CB, 0x00CA, 0x011A, 0x0116, 0x0112,
	 0x0118, 0xFFFF, 0x011C, 0x011E, 0x0122, 0x0120, 0x0124, 0x00CD,
	 0x00CC, 0x00CF, 0x00CE, 0x01CF, 0x0130, 0x012A, 0x012E, 0x0128,
	 0x0134, 0x0136, 0x0139, 0x013D, 0x013B, 0x0143, 0x0147, 0x0145,
	 0x00D1, 0x00D3, 0x00D2, 0x00D6, 0x00D4, 0x01D1, 0x0150, 0x014C,
	 0x00D5, 0x0154, 0x0158, 0x0156, 0x015A, 0x015C, 0x0160, 0x015E,
	 0x0164, 0x0162, 0x00DA, 0x00D9, 0x00DC, 0x00DB, 0x016C, 0x01D3,
	 0x0170, 0x016A, 0x0172, 0x016E, 0x0168, 0x01D7, 0x01DB, 0x01D9,
	 0x01D5, 0x0174, 0x00DD, 0x0178, 0x0176, 0x0179, 0x017D, 0x017B,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xABA1 */	 0x00E1, 0x00E0, 0x00E4, 0x00E2, 0x0103, 0x01CE, 0x0101,
	 0x0105, 0x00E5, 0x00E3, 0x0107, 0x0109, 0x010D, 0x00E7, 0x010B,
	 0x010F, 0x00E9, 0x00E8, 0x00EB, 0x00EA, 0x011B, 0x0117, 0x0113,
	 0x0119, 0x01F5, 0x011D, 0x011F, 0xFFFF, 0x0121, 0x0125, 0x00ED,
	 0x00EC, 0x00EF, 0x00EE, 0x01D0, 0xFFFF, 0x012B, 0x012F, 0x0129,
	 0x0135, 0x0137, 0x013A, 0x013E, 0x013C, 0x0144, 0x0148, 0x0146,
	 0x00F1, 0x00F3, 0x00F2, 0x00F6, 0x00F4, 0x01D2, 0x0151, 0x014D,
	 0x00F5, 0x0155, 0x0159, 0x0157, 0x015B, 0x015D, 0x0161, 0x015F,
	 0x0165, 0x0163, 0x00FA, 0x00F9, 0x00FC, 0x00FB, 0x016D, 0x01D4,
	 0x0171, 0x016B, 0x0173, 0x016F, 0x0169, 0x01D8, 0x01DC, 0x01DA,
	 0x01D6, 0x0175, 0x00FD, 0x00FF, 0x0177, 0x017A, 0x017E, 0x017C,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
/* 0xB0A1 */	 0x4E02, 0x4E04, 0x4E05, 0x4E0C, 0x4E12, 0x4E1F, 0x4E23,
	 0x4E24, 0x4E28, 0x4E2B, 0x4E2E, 0x4E2F, 0x4E30, 0x4E35, 0x4E40,
	 0x4E41, 0x4E44, 0x4E47, 0x4E51, 0x4E5A, 0x4E5C, 0x4E63, 0x4E68,
	 0x4E69, 0x4E74, 0x4E75, 0x4E79, 0x4E7F, 0x4E8D, 0x4E96, 0x4E97,
	 0x4E9D, 0x4EAF, 0x4EB9, 0x4EC3, 0x4ED0, 0x4EDA, 0x4EDB, 0x4EE0,
	 0x4EE1, 0x4EE2, 0x4EE8, 0x4EEF, 0x4EF1, 0x4EF3, 0x4EF5, 0x4EFD,
	 0x4EFE, 0x4EFF, 0x4F00, 0x4F02, 0x4F03, 0x4F08, 0x4F0B, 0x4F0C,
	 0x4F12, 0x4F15, 0x4F16, 0x4F17, 0x4F19, 0x4F2E, 0x4F31, 0x4F60,
	 0x4F33, 0x4F35, 0x4F37, 0x4F39, 0x4F3B, 0x4F3E, 0x4F40, 0x4F42,
	 0x4F48, 0x4F49, 0x4F4B, 0x4F4C, 0x4F52, 0x4F54, 0x4F56, 0x4F58,
	 0x4F5F, 0x4F63, 0x4F6A, 0x4F6C, 0x4F6E, 0x4F71, 0x4F77, 0x4F78,
	 0x4F79, 0x4F7A, 0x4F7D, 0x4F7E, 0x4F81, 0x4F82, 0x4F84,
/* 0xB1A1 */	 0x4F85, 0x4F89, 0x4F8A, 0x4F8C, 0x4F8E, 0x4F90, 0x4F92,
	 0x4F93, 0x4F94, 0x4F97, 0x4F99, 0x4F9A, 0x4F9E, 0x4F9F, 0x4FB2,
	 0x4FB7, 0x4FB9, 0x4FBB, 0x4FBC, 0x4FBD, 0x4FBE, 0x4FC0, 0x4FC1,
	 0x4FC5, 0x4FC6, 0x4FC8, 0x4FC9, 0x4FCB, 0x4FCC, 0x4FCD, 0x4FCF,
	 0x4FD2, 0x4FDC, 0x4FE0, 0x4FE2, 0x4FF0, 0x4FF2, 0x4FFC, 0x4FFD,
	 0x4FFF, 0x5000, 0x5001, 0x5004, 0x5007, 0x500A, 0x500C, 0x500E,
	 0x5010, 0x5013, 0x5017, 0x5018, 0x501B, 0x501C, 0x501D, 0x501E,
	 0x5022, 0x5027, 0x502E, 0x5030, 0x5032, 0x5033, 0x5035, 0x5040,
	 0x5041, 0x5042, 0x5045, 0x5046, 0x504A, 0x504C, 0x504E, 0x5051,
	 0x5052, 0x5053, 0x5057, 0x5059, 0x505F, 0x5060, 0x5062, 0x5063,
	 0x5066, 0x5067, 0x506A, 0x506D, 0x5070, 0x5071, 0x503B, 0x5081,
	 0x5083, 0x5084, 0x5086, 0x508A, 0x508E, 0x508F, 0x5090,
/* 0xB2A1 */	 0x5092, 0x5093, 0x5094, 0x5096, 0x509B, 0x509C, 0x509E,
	 0x509F, 0x50A0, 0x50A1, 0x50A2, 0x50AA, 0x50AF, 0x50B0, 0x50B9,
	 0x50BA, 0x50BD, 0x50C0, 0x50C3, 0x50C4, 0x50C7, 0x50CC, 0x50CE,
	 0x50D0, 0x50D3, 0x50D4, 0x50D8, 0x50DC, 0x50DD, 0x50DF, 0x50E2,
	 0x50E4, 0x50E6, 0x50E8, 0x50E9, 0x50EF, 0x50F1, 0x50F6, 0x50FA,
	 0x50FE, 0x5103, 0x5106, 0x5107, 0x5108, 0x510B, 0x510C, 0x510D,
	 0x510E, 0x50F2, 0x5110, 0x5117, 0x5119, 0x511B, 0x511C, 0x511D,
	 0x511E, 0x5123, 0x5127, 0x5128, 0x512C, 0x512D, 0x512F, 0x5131,
	 0x5133, 0x5134, 0x5135, 0x5138, 0x5139, 0x5142, 0x514A, 0x514F,
	 0x5153, 0x5155, 0x5157, 0x5158, 0x515F, 0x5164, 0x5166, 0x517E,
	 0x5183, 0x5184, 0x518B, 0x518E, 0x5198, 0x519D, 0x51A1, 0x51A3,
	 0x51AD, 0x51B8, 0x51BA, 0x51BC, 0x51BE, 0x51BF, 0x51C2,
/* 0xB3A1 */	 0x51C8, 0x51CF, 0x51D1, 0x51D2, 0x51D3, 0x51D5, 0x51D8,
	 0x51DE, 0x51E2, 0x51E5, 0x51EE, 0x51F2, 0x51F3, 0x51F4, 0x51F7,
	 0x5201, 0x5202, 0x5205, 0x5212, 0x5213, 0x5215, 0x5216, 0x5218,
	 0x5222, 0x5228, 0x5231, 0x5232, 0x5235, 0x523C, 0x5245, 0x5249,
	 0x5255, 0x5257, 0x5258, 0x525A, 0x525C, 0x525F, 0x5260, 0x5261,
	 0x5266, 0x526E, 0x5277, 0x5278, 0x5279, 0x5280, 0x5282, 0x5285,
	 0x528A, 0x528C, 0x5293, 0x5295, 0x5296, 0x5297, 0x5298, 0x529A,
	 0x529C, 0x52A4, 0x52A5, 0x52A6, 0x52A7, 0x52AF, 0x52B0, 0x52B6,
	 0x52B7, 0x52B8, 0x52BA, 0x52BB, 0x52BD, 0x52C0, 0x52C4, 0x52C6,
	 0x52C8, 0x52CC, 0x52CF, 0x52D1, 0x52D4, 0x52D6, 0x52DB, 0x52DC,
	 0x52E1, 0x52E5, 0x52E8, 0x52E9, 0x52EA, 0x52EC, 0x52F0, 0x52F1,
	 0x52F4, 0x52F6, 0x52F7, 0x5300, 0x5303, 0x530A, 0x530B,
/* 0xB4A1 */	 0x530C, 0x5311, 0x5313, 0x5318, 0x531B, 0x531C, 0x531E,
	 0x531F, 0x5325, 0x5327, 0x5328, 0x5329, 0x532B, 0x532C, 0x532D,
	 0x5330, 0x5332, 0x5335, 0x533C, 0x533D, 0x533E, 0x5342, 0x534C,
	 0x534B, 0x5359, 0x535B, 0x5361, 0x5363, 0x5365, 0x536C, 0x536D,
	 0x5372, 0x5379, 0x537E, 0x5383, 0x5387, 0x5388, 0x538E, 0x5393,
	 0x5394, 0x5399, 0x539D, 0x53A1, 0x53A4, 0x53AA, 0x53AB, 0x53AF,
	 0x53B2, 0x53B4, 0x53B5, 0x53B7, 0x53B8, 0x53BA, 0x53BD, 0x53C0,
	 0x53C5, 0x53CF, 0x53D2, 0x53D3, 0x53D5, 0x53DA, 0x53DD, 0x53DE,
	 0x53E0, 0x53E6, 0x53E7, 0x53F5, 0x5402, 0x5413, 0x541A, 0x5421,
	 0x5427, 0x5428, 0x542A, 0x542F, 0x5431, 0x5434, 0x5435, 0x5443,
	 0x5444, 0x5447, 0x544D, 0x544F, 0x545E, 0x5462, 0x5464, 0x5466,
	 0x5467, 0x5469, 0x546B, 0x546D, 0x546E, 0x5474, 0x547F,
/* 0xB5A1 */	 0x5481, 0x5483, 0x5485, 0x5488, 0x5489, 0x548D, 0x5491,
	 0x5495, 0x5496, 0x549C, 0x549F, 0x54A1, 0x54A6, 0x54A7, 0x54A9,
	 0x54AA, 0x54AD, 0x54AE, 0x54B1, 0x54B7, 0x54B9, 0x54BA, 0x54BB,
	 0x54BF, 0x54C6, 0x54CA, 0x54CD, 0x54CE, 0x54E0, 0x54EA, 0x54EC,
	 0x54EF, 0x54F6, 0x54FC, 0x54FE, 0x54FF, 0x5500, 0x5501, 0x5505,
	 0x5508, 0x5509, 0x550C, 0x550D, 0x550E, 0x5515, 0x552A, 0x552B,
	 0x5532, 0x5535, 0x5536, 0x553B, 0x553C, 0x553D, 0x5541, 0x5547,
	 0x5549, 0x554A, 0x554D, 0x5550, 0x5551, 0x5558, 0x555A, 0x555B,
	 0x555E, 0x5560, 0x5561, 0x5564, 0x5566, 0x557F, 0x5581, 0x5582,
	 0x5586, 0x5588, 0x558E, 0x558F, 0x5591, 0x5592, 0x5593, 0x5594,
	 0x5597, 0x55A3, 0x55A4, 0x55AD, 0x55B2, 0x55BF, 0x55C1, 0x55C3,
	 0x55C6, 0x55C9, 0x55CB, 0x55CC, 0x55CE, 0x55D1, 0x55D2,
/* 0xB6A1 */	 0x55D3, 0x55D7, 0x55D8, 0x55DB, 0x55DE, 0x55E2, 0x55E9,
	 0x55F6, 0x55FF, 0x5605, 0x5608, 0x560A, 0x560D, 0x560E, 0x560F,
	 0x5610, 0x5611, 0x5612, 0x5619, 0x562C, 0x5630, 0x5633, 0x5635,
	 0x5637, 0x5639, 0x563B, 0x563C, 0x563D, 0x563F, 0x5640, 0x5641,
	 0x5643, 0x5644, 0x5646, 0x5649, 0x564B, 0x564D, 0x564F, 0x5654,
	 0x565E, 0x5660, 0x5661, 0x5662, 0x5663, 0x5666, 0x5669, 0x566D,
	 0x566F, 0x5671, 0x5672, 0x5675, 0x5684, 0x5685, 0x5688, 0x568B,
	 0x568C, 0x5695, 0x5699, 0x569A, 0x569D, 0x569E, 0x569F, 0x56A6,
	 0x56A7, 0x56A8, 0x56A9, 0x56AB, 0x56AC, 0x56AD, 0x56B1, 0x56B3,
	 0x56B7, 0x56BE, 0x56C5, 0x56C9, 0x56CA, 0x56CB, 0x56CF, 0x56D0,
	 0x56CC, 0x56CD, 0x56D9, 0x56DC, 0x56DD, 0x56DF, 0x56E1, 0x56E4,
	 0x56E5, 0x56E6, 0x56E7, 0x56E8, 0x56F1, 0x56EB, 0x56ED,
/* 0xB7A1 */	 0x56F6, 0x56F7, 0x5701, 0x5702, 0x5707, 0x570A, 0x570C,
	 0x5711, 0x5715, 0x571A, 0x571B, 0x571D, 0x5720, 0x5722, 0x5723,
	 0x5724, 0x5725, 0x5729, 0x572A, 0x572C, 0x572E, 0x572F, 0x5733,
	 0x5734, 0x573D, 0x573E, 0x573F, 0x5745, 0x5746, 0x574C, 0x574D,
	 0x5752, 0x5762, 0x5765, 0x5767, 0x5768, 0x576B, 0x576D, 0x576E,
	 0x576F, 0x5770, 0x5771, 0x5773, 0x5774, 0x5775, 0x5777, 0x5779,
	 0x577A, 0x577B, 0x577C, 0x577E, 0x5781, 0x5783, 0x578C, 0x5794,
	 0x5797, 0x5799, 0x579A, 0x579C, 0x579D, 0x579E, 0x579F, 0x57A1,
	 0x5795, 0x57A7, 0x57A8, 0x57A9, 0x57AC, 0x57B8, 0x57BD, 0x57C7,
	 0x57C8, 0x57CC, 0x57CF, 0x57D5, 0x57DD, 0x57DE, 0x57E4, 0x57E6,
	 0x57E7, 0x57E9, 0x57ED, 0x57F0, 0x57F5, 0x57F6, 0x57F8, 0x57FD,
	 0x57FE, 0x57FF, 0x5803, 0x5804, 0x5808, 0x5809, 0x57E1,
/* 0xB8A1 */	 0x580C, 0x580D, 0x581B, 0x581E, 0x581F, 0x5820, 0x5826,
	 0x5827, 0x582D, 0x5832, 0x5839, 0x583F, 0x5849, 0x584C, 0x584D,
	 0x584F, 0x5850, 0x5855, 0x585F, 0x5861, 0x5864, 0x5867, 0x5868,
	 0x5878, 0x587C, 0x587F, 0x5880, 0x5881, 0x5887, 0x5888, 0x5889,
	 0x588A, 0x588C, 0x588D, 0x588F, 0x5890, 0x5894, 0x5896, 0x589D,
	 0x58A0, 0x58A1, 0x58A2, 0x58A6, 0x58A9, 0x58B1, 0x58B2, 0x58C4,
	 0x58BC, 0x58C2, 0x58C8, 0x58CD, 0x58CE, 0x58D0, 0x58D2, 0x58D4,
	 0x58D6, 0x58DA, 0x58DD, 0x58E1, 0x58E2, 0x58E9, 0x58F3, 0x5905,
	 0x5906, 0x590B, 0x590C, 0x5912, 0x5913, 0x5914, 0x8641, 0x591D,
	 0x5921, 0x5923, 0x5924, 0x5928, 0x592F, 0x5930, 0x5933, 0x5935,
	 0x5936, 0x593F, 0x5943, 0x5946, 0x5952, 0x5953, 0x5959, 0x595B,
	 0x595D, 0x595E, 0x595F, 0x5961, 0x5963, 0x596B, 0x596D,
/* 0xB9A1 */	 0x596F, 0x5972, 0x5975, 0x5976, 0x5979, 0x597B, 0x597C,
	 0x598B, 0x598C, 0x598E, 0x5992, 0x5995, 0x5997, 0x599F, 0x59A4,
	 0x59A7, 0x59AD, 0x59AE, 0x59AF, 0x59B0, 0x59B3, 0x59B7, 0x59BA,
	 0x59BC, 0x59C1, 0x59C3, 0x59C4, 0x59C8, 0x59CA, 0x59CD, 0x59D2,
	 0x59DD, 0x59DE, 0x59DF, 0x59E3, 0x59E4, 0x59E7, 0x59EE, 0x59EF,
	 0x59F1, 0x59F2, 0x59F4, 0x59F7, 0x5A00, 0x5A04, 0x5A0C, 0x5A0D,
	 0x5A0E, 0x5A12, 0x5A13, 0x5A1E, 0x5A23, 0x5A24, 0x5A27, 0x5A28,
	 0x5A2A, 0x5A2D, 0x5A30, 0x5A44, 0x5A45, 0x5A47, 0x5A48, 0x5A4C,
	 0x5A50, 0x5A55, 0x5A5E, 0x5A63, 0x5A65, 0x5A67, 0x5A6D, 0x5A77,
	 0x5A7A, 0x5A7B, 0x5A7E, 0x5A8B, 0x5A90, 0x5A93, 0x5A96, 0x5A99,
	 0x5A9C, 0x5A9E, 0x5A9F, 0x5AA0, 0x5AA2, 0x5AA7, 0x5AAC, 0x5AB1,
	 0x5AB2, 0x5AB3, 0x5AB5, 0x5AB8, 0x5ABA, 0x5ABB, 0x5ABF,
/* 0xBAA1 */	 0x5AC4, 0x5AC6, 0x5AC8, 0x5ACF, 0x5ADA, 0x5ADC, 0x5AE0,
	 0x5AE5, 0x5AEA, 0x5AEE, 0x5AF5, 0x5AF6, 0x5AFD, 0x5B00, 0x5B01,
	 0x5B08, 0x5B17, 0x5B34, 0x5B19, 0x5B1B, 0x5B1D, 0x5B21, 0x5B25,
	 0x5B2D, 0x5B38, 0x5B41, 0x5B4B, 0x5B4C, 0x5B52, 0x5B56, 0x5B5E,
	 0x5B68, 0x5B6E, 0x5B6F, 0x5B7C, 0x5B7D, 0x5B7E, 0x5B7F, 0x5B81,
	 0x5B84, 0x5B86, 0x5B8A, 0x5B8E, 0x5B90, 0x5B91, 0x5B93, 0x5B94,
	 0x5B96, 0x5BA8, 0x5BA9, 0x5BAC, 0x5BAD, 0x5BAF, 0x5BB1, 0x5BB2,
	 0x5BB7, 0x5BBA, 0x5BBC, 0x5BC0, 0x5BC1, 0x5BCD, 0x5BCF, 0x5BD6,
	 0x5BD7, 0x5BD8, 0x5BD9, 0x5BDA, 0x5BE0, 0x5BEF, 0x5BF1, 0x5BF4,
	 0x5BFD, 0x5C0C, 0x5C17, 0x5C1E, 0x5C1F, 0x5C23, 0x5C26, 0x5C29,
	 0x5C2B, 0x5C2C, 0x5C2E, 0x5C30, 0x5C32, 0x5C35, 0x5C36, 0x5C59,
	 0x5C5A, 0x5C5C, 0x5C62, 0x5C63, 0x5C67, 0x5C68, 0x5C69,
/* 0xBBA1 */	 0x5C6D, 0x5C70, 0x5C74, 0x5C75, 0x5C7A, 0x5C7B, 0x5C7C,
	 0x5C7D, 0x5C87, 0x5C88, 0x5C8A, 0x5C8F, 0x5C92, 0x5C9D, 0x5C9F,
	 0x5CA0, 0x5CA2, 0x5CA3, 0x5CA6, 0x5CAA, 0x5CB2, 0x5CB4, 0x5CB5,
	 0x5CBA, 0x5CC9, 0x5CCB, 0x5CD2, 0x5CDD, 0x5CD7, 0x5CEE, 0x5CF1,
	 0x5CF2, 0x5CF4, 0x5D01, 0x5D06, 0x5D0D, 0x5D12, 0x5D2B, 0x5D23,
	 0x5D24, 0x5D26, 0x5D27, 0x5D31, 0x5D34, 0x5D39, 0x5D3D, 0x5D3F,
	 0x5D42, 0x5D43, 0x5D46, 0x5D48, 0x5D55, 0x5D51, 0x5D59, 0x5D4A,
	 0x5D5F, 0x5D60, 0x5D61, 0x5D62, 0x5D64, 0x5D6A, 0x5D6D, 0x5D70,
	 0x5D79, 0x5D7A, 0x5D7E, 0x5D7F, 0x5D81, 0x5D83, 0x5D88, 0x5D8A,
	 0x5D92, 0x5D93, 0x5D94, 0x5D95, 0x5D99, 0x5D9B, 0x5D9F, 0x5DA0,
	 0x5DA7, 0x5DAB, 0x5DB0, 0x5DB4, 0x5DB8, 0x5DB9, 0x5DC3, 0x5DC7,
	 0x5DCB, 0x5DD0, 0x5DCE, 0x5DD8, 0x5DD9, 0x5DE0, 0x5DE4,
/* 0xBCA1 */	 0x5DE9, 0x5DF8, 0x5DF9, 0x5E00, 0x5E07, 0x5E0D, 0x5E12,
	 0x5E14, 0x5E15, 0x5E18, 0x5E1F, 0x5E20, 0x5E2E, 0x5E28, 0x5E32,
	 0x5E35, 0x5E3E, 0x5E4B, 0x5E50, 0x5E49, 0x5E51, 0x5E56, 0x5E58,
	 0x5E5B, 0x5E5C, 0x5E5E, 0x5E68, 0x5E6A, 0x5E6B, 0x5E6C, 0x5E6D,
	 0x5E6E, 0x5E70, 0x5E80, 0x5E8B, 0x5E8E, 0x5EA2, 0x5EA4, 0x5EA5,
	 0x5EA8, 0x5EAA, 0x5EAC, 0x5EB1, 0x5EB3, 0x5EBD, 0x5EBE, 0x5EBF,
	 0x5EC6, 0x5ECC, 0x5ECB, 0x5ECE, 0x5ED1, 0x5ED2, 0x5ED4, 0x5ED5,
	 0x5EDC, 0x5EDE, 0x5EE5, 0x5EEB, 0x5F02, 0x5F06, 0x5F07, 0x5F08,
	 0x5F0E, 0x5F19, 0x5F1C, 0x5F1D, 0x5F21, 0x5F22, 0x5F23, 0x5F24,
	 0x5F28, 0x5F2B, 0x5F2C, 0x5F2E, 0x5F30, 0x5F34, 0x5F36, 0x5F3B,
	 0x5F3D, 0x5F3F, 0x5F40, 0x5F44, 0x5F45, 0x5F47, 0x5F4D, 0x5F50,
	 0x5F54, 0x5F58, 0x5F5B, 0x5F60, 0x5F63, 0x5F64, 0x5F67,
/* 0xBDA1 */	 0x5F6F, 0x5F72, 0x5F74, 0x5F75, 0x5F78, 0x5F7A, 0x5F7D,
	 0x5F7E, 0x5F89, 0x5F8D, 0x5F8F, 0x5F96, 0x5F9C, 0x5F9D, 0x5FA2,
	 0x5FA7, 0x5FAB, 0x5FA4, 0x5FAC, 0x5FAF, 0x5FB0, 0x5FB1, 0x5FB8,
	 0x5FC4, 0x5FC7, 0x5FC8, 0x5FC9, 0x5FCB, 0x5FD0, 0x5FD1, 0x5FD2,
	 0x5FD3, 0x5FD4, 0x5FDE, 0x5FE1, 0x5FE2, 0x5FE8, 0x5FE9, 0x5FEA,
	 0x5FEC, 0x5FED, 0x5FEE, 0x5FEF, 0x5FF2, 0x5FF3, 0x5FF6, 0x5FFA,
	 0x5FFC, 0x6007, 0x600A, 0x600D, 0x6013, 0x6014, 0x6017, 0x6018,
	 0x601A, 0x601F, 0x6024, 0x602D, 0x6033, 0x6035, 0x6040, 0x6047,
	 0x6048, 0x6049, 0x604C, 0x6051, 0x6054, 0x6056, 0x6057, 0x605D,
	 0x6061, 0x6067, 0x6071, 0x607E, 0x607F, 0x6082, 0x6086, 0x6088,
	 0x608A, 0x608E, 0x6091, 0x6093, 0x6095, 0x6098, 0x609D, 0x609E,
	 0x60A2, 0x60A4, 0x60A5, 0x60A8, 0x60B0, 0x60B1, 0x60B7,
/* 0xBEA1 */	 0x60BB, 0x60BE, 0x60C2, 0x60C4, 0x60C8, 0x60C9, 0x60CA,
	 0x60CB, 0x60CE, 0x60CF, 0x60D4, 0x60D5, 0x60D9, 0x60DB, 0x60DD,
	 0x60DE, 0x60E2, 0x60E5, 0x60F2, 0x60F5, 0x60F8, 0x60FC, 0x60FD,
	 0x6102, 0x6107, 0x610A, 0x610C, 0x6110, 0x6111, 0x6112, 0x6113,
	 0x6114, 0x6116, 0x6117, 0x6119, 0x611C, 0x611E, 0x6122, 0x612A,
	 0x612B, 0x6130, 0x6131, 0x6135, 0x6136, 0x6137, 0x6139, 0x6141,
	 0x6145, 0x6146, 0x6149, 0x615E, 0x6160, 0x616C, 0x6172, 0x6178,
	 0x617B, 0x617C, 0x617F, 0x6180, 0x6181, 0x6183, 0x6184, 0x618B,
	 0x618D, 0x6192, 0x6193, 0x6197, 0x6198, 0x619C, 0x619D, 0x619F,
	 0x61A0, 0x61A5, 0x61A8, 0x61AA, 0x61AD, 0x61B8, 0x61B9, 0x61BC,
	 0x61C0, 0x61C1, 0x61C2, 0x61CE, 0x61CF, 0x61D5, 0x61DC, 0x61DD,
	 0x61DE, 0x61DF, 0x61E1, 0x61E2, 0x61E7, 0x61E9, 0x61E5,
/* 0xBFA1 */	 0x61EC, 0x61ED, 0x61EF, 0x6201, 0x6203, 0x6204, 0x6207,
	 0x6213, 0x6215, 0x621C, 0x6220, 0x6222, 0x6223, 0x6227, 0x6229,
	 0x622B, 0x6239, 0x623D, 0x6242, 0x6243, 0x6244, 0x6246, 0x624C,
	 0x6250, 0x6251, 0x6252, 0x6254, 0x6256, 0x625A, 0x625C, 0x6264,
	 0x626D, 0x626F, 0x6273, 0x627A, 0x627D, 0x628D, 0x628E, 0x628F,
	 0x6290, 0x62A6, 0x62A8, 0x62B3, 0x62B6, 0x62B7, 0x62BA, 0x62BE,
	 0x62BF, 0x62C4, 0x62CE, 0x62D5, 0x62D6, 0x62DA, 0x62EA, 0x62F2,
	 0x62F4, 0x62FC, 0x62FD, 0x6303, 0x6304, 0x630A, 0x630B, 0x630D,
	 0x6310, 0x6313, 0x6316, 0x6318, 0x6329, 0x632A, 0x632D, 0x6335,
	 0x6336, 0x6339, 0x633C, 0x6341, 0x6342, 0x6343, 0x6344, 0x6346,
	 0x634A, 0x634B, 0x634E, 0x6352, 0x6353, 0x6354, 0x6358, 0x635B,
	 0x6365, 0x6366, 0x636C, 0x636D, 0x6371, 0x6374, 0x6375,
/* 0xC0A1 */	 0x6378, 0x637C, 0x637D, 0x637F, 0x6382, 0x6384, 0x6387,
	 0x638A, 0x6390, 0x6394, 0x6395, 0x6399, 0x639A, 0x639E, 0x63A4,
	 0x63A6, 0x63AD, 0x63AE, 0x63AF, 0x63BD, 0x63C1, 0x63C5, 0x63C8,
	 0x63CE, 0x63D1, 0x63D3, 0x63D4, 0x63D5, 0x63DC, 0x63E0, 0x63E5,
	 0x63EA, 0x63EC, 0x63F2, 0x63F3, 0x63F5, 0x63F8, 0x63F9, 0x6409,
	 0x640A, 0x6410, 0x6412, 0x6414, 0x6418, 0x641E, 0x6420, 0x6422,
	 0x6424, 0x6425, 0x6429, 0x642A, 0x642F, 0x6430, 0x6435, 0x643D,
	 0x643F, 0x644B, 0x644F, 0x6451, 0x6452, 0x6453, 0x6454, 0x645A,
	 0x645B, 0x645C, 0x645D, 0x645F, 0x6460, 0x6461, 0x6463, 0x646D,
	 0x6473, 0x6474, 0x647B, 0x647D, 0x6485, 0x6487, 0x648F, 0x6490,
	 0x6491, 0x6498, 0x6499, 0x649B, 0x649D, 0x649F, 0x64A1, 0x64A3,
	 0x64A6, 0x64A8, 0x64AC, 0x64B3, 0x64BD, 0x64BE, 0x64BF,
/* 0xC1A1 */	 0x64C4, 0x64C9, 0x64CA, 0x64CB, 0x64CC, 0x64CE, 0x64D0,
	 0x64D1, 0x64D5, 0x64D7, 0x64E4, 0x64E5, 0x64E9, 0x64EA, 0x64ED,
	 0x64F0, 0x64F5, 0x64F7, 0x64FB, 0x64FF, 0x6501, 0x6504, 0x6508,
	 0x6509, 0x650A, 0x650F, 0x6513, 0x6514, 0x6516, 0x6519, 0x651B,
	 0x651E, 0x651F, 0x6522, 0x6526, 0x6529, 0x652E, 0x6531, 0x653A,
	 0x653C, 0x653D, 0x6543, 0x6547, 0x6549, 0x6550, 0x6552, 0x6554,
	 0x655F, 0x6560, 0x6567, 0x656B, 0x657A, 0x657D, 0x6581, 0x6585,
	 0x658A, 0x6592, 0x6595, 0x6598, 0x659D, 0x65A0, 0x65A3, 0x65A6,
	 0x65AE, 0x65B2, 0x65B3, 0x65B4, 0x65BF, 0x65C2, 0x65C8, 0x65C9,
	 0x65CE, 0x65D0, 0x65D4, 0x65D6, 0x65D8, 0x65DF, 0x65F0, 0x65F2,
	 0x65F4, 0x65F5, 0x65F9, 0x65FE, 0x65FF, 0x6600, 0x6604, 0x6608,
	 0x6609, 0x660D, 0x6611, 0x6612, 0x6615, 0x6616, 0x661D,
/* 0xC2A1 */	 0x661E, 0x6621, 0x6622, 0x6623, 0x6624, 0x6626, 0x6629,
	 0x662A, 0x662B, 0x662C, 0x662E, 0x6630, 0x6631, 0x6633, 0x6639,
	 0x6637, 0x6640, 0x6645, 0x6646, 0x664A, 0x664C, 0x6651, 0x664E,
	 0x6657, 0x6658, 0x6659, 0x665B, 0x665C, 0x6660, 0x6661, 0x66FB,
	 0x666A, 0x666B, 0x666C, 0x667E, 0x6673, 0x6675, 0x667F, 0x6677,
	 0x6678, 0x6679, 0x667B, 0x6680, 0x667C, 0x668B, 0x668C, 0x668D,
	 0x6690, 0x6692, 0x6699, 0x669A, 0x669B, 0x669C, 0x669F, 0x66A0,
	 0x66A4, 0x66AD, 0x66B1, 0x66B2, 0x66B5, 0x66BB, 0x66BF, 0x66C0,
	 0x66C2, 0x66C3, 0x66C8, 0x66CC, 0x66CE, 0x66CF, 0x66D4, 0x66DB,
	 0x66DF, 0x66E8, 0x66EB, 0x66EC, 0x66EE, 0x66FA, 0x6705, 0x6707,
	 0x670E, 0x6713, 0x6719, 0x671C, 0x6720, 0x6722, 0x6733, 0x673E,
	 0x6745, 0x6747, 0x6748, 0x674C, 0x6754, 0x6755, 0x675D,
/* 0xC3A1 */	 0x6766, 0x676C, 0x676E, 0x6774, 0x6776, 0x677B, 0x6781,
	 0x6784, 0x678E, 0x678F, 0x6791, 0x6793, 0x6796, 0x6798, 0x6799,
	 0x679B, 0x67B0, 0x67B1, 0x67B2, 0x67B5, 0x67BB, 0x67BC, 0x67BD,
	 0x67F9, 0x67C0, 0x67C2, 0x67C3, 0x67C5, 0x67C8, 0x67C9, 0x67D2,
	 0x67D7, 0x67D9, 0x67DC, 0x67E1, 0x67E6, 0x67F0, 0x67F2, 0x67F6,
	 0x67F7, 0x6852, 0x6814, 0x6819, 0x681D, 0x681F, 0x6828, 0x6827,
	 0x682C, 0x682D, 0x682F, 0x6830, 0x6831, 0x6833, 0x683B, 0x683F,
	 0x6844, 0x6845, 0x684A, 0x684C, 0x6855, 0x6857, 0x6858, 0x685B,
	 0x686B, 0x686E, 0x686F, 0x6870, 0x6871, 0x6872, 0x6875, 0x6879,
	 0x687A, 0x687B, 0x687C, 0x6882, 0x6884, 0x6886, 0x6888, 0x6896,
	 0x6898, 0x689A, 0x689C, 0x68A1, 0x68A3, 0x68A5, 0x68A9, 0x68AA,
	 0x68AE, 0x68B2, 0x68BB, 0x68C5, 0x68C8, 0x68CC, 0x68CF,
/* 0xC4A1 */	 0x68D0, 0x68D1, 0x68D3, 0x68D6, 0x68D9, 0x68DC, 0x68DD,
	 0x68E5, 0x68E8, 0x68EA, 0x68EB, 0x68EC, 0x68ED, 0x68F0, 0x68F1,
	 0x68F5, 0x68F6, 0x68FB, 0x68FC, 0x68FD, 0x6906, 0x6909, 0x690A,
	 0x6910, 0x6911, 0x6913, 0x6916, 0x6917, 0x6931, 0x6933, 0x6935,
	 0x6938, 0x693B, 0x6942, 0x6945, 0x6949, 0x694E, 0x6957, 0x695B,
	 0x6963, 0x6964, 0x6965, 0x6966, 0x6968, 0x6969, 0x696C, 0x6970,
	 0x6971, 0x6972, 0x697A, 0x697B, 0x697F, 0x6980, 0x698D, 0x6992,
	 0x6996, 0x6998, 0x69A1, 0x69A5, 0x69A6, 0x69A8, 0x69AB, 0x69AD,
	 0x69AF, 0x69B7, 0x69B8, 0x69BA, 0x69BC, 0x69C5, 0x69C8, 0x69D1,
	 0x69D6, 0x69D7, 0x69E2, 0x69E5, 0x69EE, 0x69EF, 0x69F1, 0x69F3,
	 0x69F5, 0x69FE, 0x6A00, 0x6A01, 0x6A03, 0x6A0F, 0x6A11, 0x6A15,
	 0x6A1A, 0x6A1D, 0x6A20, 0x6A24, 0x6A28, 0x6A30, 0x6A32,
/* 0xC5A1 */	 0x6A34, 0x6A37, 0x6A3B, 0x6A3E, 0x6A3F, 0x6A45, 0x6A46,
	 0x6A49, 0x6A4A, 0x6A4E, 0x6A50, 0x6A51, 0x6A52, 0x6A55, 0x6A56,
	 0x6A5B, 0x6A64, 0x6A67, 0x6A6A, 0x6A71, 0x6A73, 0x6A7E, 0x6A81,
	 0x6A83, 0x6A86, 0x6A87, 0x6A89, 0x6A8B, 0x6A91, 0x6A9B, 0x6A9D,
	 0x6A9E, 0x6A9F, 0x6AA5, 0x6AAB, 0x6AAF, 0x6AB0, 0x6AB1, 0x6AB4,
	 0x6ABD, 0x6ABE, 0x6ABF, 0x6AC6, 0x6AC9, 0x6AC8, 0x6ACC, 0x6AD0,
	 0x6AD4, 0x6AD5, 0x6AD6, 0x6ADC, 0x6ADD, 0x6AE4, 0x6AE7, 0x6AEC,
	 0x6AF0, 0x6AF1, 0x6AF2, 0x6AFC, 0x6AFD, 0x6B02, 0x6B03, 0x6B06,
	 0x6B07, 0x6B09, 0x6B0F, 0x6B10, 0x6B11, 0x6B17, 0x6B1B, 0x6B1E,
	 0x6B24, 0x6B28, 0x6B2B, 0x6B2C, 0x6B2F, 0x6B35, 0x6B36, 0x6B3B,
	 0x6B3F, 0x6B46, 0x6B4A, 0x6B4D, 0x6B52, 0x6B56, 0x6B58, 0x6B5D,
	 0x6B60, 0x6B67, 0x6B6B, 0x6B6E, 0x6B70, 0x6B75, 0x6B7D,
/* 0xC6A1 */	 0x6B7E, 0x6B82, 0x6B85, 0x6B97, 0x6B9B, 0x6B9F, 0x6BA0,
	 0x6BA2, 0x6BA3, 0x6BA8, 0x6BA9, 0x6BAC, 0x6BAD, 0x6BAE, 0x6BB0,
	 0x6BB8, 0x6BB9, 0x6BBD, 0x6BBE, 0x6BC3, 0x6BC4, 0x6BC9, 0x6BCC,
	 0x6BD6, 0x6BDA, 0x6BE1, 0x6BE3, 0x6BE6, 0x6BE7, 0x6BEE, 0x6BF1,
	 0x6BF7, 0x6BF9, 0x6BFF, 0x6C02, 0x6C04, 0x6C05, 0x6C09, 0x6C0D,
	 0x6C0E, 0x6C10, 0x6C12, 0x6C19, 0x6C1F, 0x6C26, 0x6C27, 0x6C28,
	 0x6C2C, 0x6C2E, 0x6C33, 0x6C35, 0x6C36, 0x6C3A, 0x6C3B, 0x6C3F,
	 0x6C4A, 0x6C4B, 0x6C4D, 0x6C4F, 0x6C52, 0x6C54, 0x6C59, 0x6C5B,
	 0x6C5C, 0x6C6B, 0x6C6D, 0x6C6F, 0x6C74, 0x6C76, 0x6C78, 0x6C79,
	 0x6C7B, 0x6C85, 0x6C86, 0x6C87, 0x6C89, 0x6C94, 0x6C95, 0x6C97,
	 0x6C98, 0x6C9C, 0x6C9F, 0x6CB0, 0x6CB2, 0x6CB4, 0x6CC2, 0x6CC6,
	 0x6CCD, 0x6CCF, 0x6CD0, 0x6CD1, 0x6CD2, 0x6CD4, 0x6CD6,
/* 0xC7A1 */	 0x6CDA, 0x6CDC, 0x6CE0, 0x6CE7, 0x6CE9, 0x6CEB, 0x6CEC,
	 0x6CEE, 0x6CF2, 0x6CF4, 0x6D04, 0x6D07, 0x6D0A, 0x6D0E, 0x6D0F,
	 0x6D11, 0x6D13, 0x6D1A, 0x6D26, 0x6D27, 0x6D28, 0x6C67, 0x6D2E,
	 0x6D2F, 0x6D31, 0x6D39, 0x6D3C, 0x6D3F, 0x6D57, 0x6D5E, 0x6D5F,
	 0x6D61, 0x6D65, 0x6D67, 0x6D6F, 0x6D70, 0x6D7C, 0x6D82, 0x6D87,
	 0x6D91, 0x6D92, 0x6D94, 0x6D96, 0x6D97, 0x6D98, 0x6DAA, 0x6DAC,
	 0x6DB4, 0x6DB7, 0x6DB9, 0x6DBD, 0x6DBF, 0x6DC4, 0x6DC8, 0x6DCA,
	 0x6DCE, 0x6DCF, 0x6DD6, 0x6DDB, 0x6DDD, 0x6DDF, 0x6DE0, 0x6DE2,
	 0x6DE5, 0x6DE9, 0x6DEF, 0x6DF0, 0x6DF4, 0x6DF6, 0x6DFC, 0x6E00,
	 0x6E04, 0x6E1E, 0x6E22, 0x6E27, 0x6E32, 0x6E36, 0x6E39, 0x6E3B,
	 0x6E3C, 0x6E44, 0x6E45, 0x6E48, 0x6E49, 0x6E4B, 0x6E4F, 0x6E51,
	 0x6E52, 0x6E53, 0x6E54, 0x6E57, 0x6E5C, 0x6E5D, 0x6E5E,
/* 0xC8A1 */	 0x6E62, 0x6E63, 0x6E68, 0x6E73, 0x6E7B, 0x6E7D, 0x6E8D,
	 0x6E93, 0x6E99, 0x6EA0, 0x6EA7, 0x6EAD, 0x6EAE, 0x6EB1, 0x6EB3,
	 0x6EBB, 0x6EBF, 0x6EC0, 0x6EC1, 0x6EC3, 0x6EC7, 0x6EC8, 0x6ECA,
	 0x6ECD, 0x6ECE, 0x6ECF, 0x6EEB, 0x6EED, 0x6EEE, 0x6EF9, 0x6EFB,
	 0x6EFD, 0x6F04, 0x6F08, 0x6F0A, 0x6F0C, 0x6F0D, 0x6F16, 0x6F18,
	 0x6F1A, 0x6F1B, 0x6F26, 0x6F29, 0x6F2A, 0x6F2F, 0x6F30, 0x6F33,
	 0x6F36, 0x6F3B, 0x6F3C, 0x6F2D, 0x6F4F, 0x6F51, 0x6F52, 0x6F53,
	 0x6F57, 0x6F59, 0x6F5A, 0x6F5D, 0x6F5E, 0x6F61, 0x6F62, 0x6F68,
	 0x6F6C, 0x6F7D, 0x6F7E, 0x6F83, 0x6F87, 0x6F88, 0x6F8B, 0x6F8C,
	 0x6F8D, 0x6F90, 0x6F92, 0x6F93, 0x6F94, 0x6F96, 0x6F9A, 0x6F9F,
	 0x6FA0, 0x6FA5, 0x6FA6, 0x6FA7, 0x6FA8, 0x6FAE, 0x6FAF, 0x6FB0,
	 0x6FB5, 0x6FB6, 0x6FBC, 0x6FC5, 0x6FC7, 0x6FC8, 0x6FCA,
/* 0xC9A1 */	 0x6FDA, 0x6FDE, 0x6FE8, 0x6FE9, 0x6FF0, 0x6FF5, 0x6FF9,
	 0x6FFC, 0x6FFD, 0x7000, 0x7005, 0x7006, 0x7007, 0x700D, 0x7017,
	 0x7020, 0x7023, 0x702F, 0x7034, 0x7037, 0x7039, 0x703C, 0x7043,
	 0x7044, 0x7048, 0x7049, 0x704A, 0x704B, 0x7054, 0x7055, 0x705D,
	 0x705E, 0x704E, 0x7064, 0x7065, 0x706C, 0x706E, 0x7075, 0x7076,
	 0x707E, 0x7081, 0x7085, 0x7086, 0x7094, 0x7095, 0x7096, 0x7097,
	 0x7098, 0x709B, 0x70A4, 0x70AB, 0x70B0, 0x70B1, 0x70B4, 0x70B7,
	 0x70CA, 0x70D1, 0x70D3, 0x70D4, 0x70D5, 0x70D6, 0x70D8, 0x70DC,
	 0x70E4, 0x70FA, 0x7103, 0x7104, 0x7105, 0x7106, 0x7107, 0x710B,
	 0x710C, 0x710F, 0x711E, 0x7120, 0x712B, 0x712D, 0x712F, 0x7130,
	 0x7131, 0x7138, 0x7141, 0x7145, 0x7146, 0x7147, 0x714A, 0x714B,
	 0x7150, 0x7152, 0x7157, 0x715A, 0x715C, 0x715E, 0x7160,
/* 0xCAA1 */	 0x7168, 0x7179, 0x7180, 0x7185, 0x7187, 0x718C, 0x7192,
	 0x719A, 0x719B, 0x71A0, 0x71A2, 0x71AF, 0x71B0, 0x71B2, 0x71B3,
	 0x71BA, 0x71BF, 0x71C0, 0x71C1, 0x71C4, 0x71CB, 0x71CC, 0x71D3,
	 0x71D6, 0x71D9, 0x71DA, 0x71DC, 0x71F8, 0x71FE, 0x7200, 0x7207,
	 0x7208, 0x7209, 0x7213, 0x7217, 0x721A, 0x721D, 0x721F, 0x7224,
	 0x722B, 0x722F, 0x7234, 0x7238, 0x7239, 0x7241, 0x7242, 0x7243,
	 0x7245, 0x724E, 0x724F, 0x7250, 0x7253, 0x7255, 0x7256, 0x725A,
	 0x725C, 0x725E, 0x7260, 0x7263, 0x7268, 0x726B, 0x726E, 0x726F,
	 0x7271, 0x7277, 0x7278, 0x727B, 0x727C, 0x727F, 0x7284, 0x7289,
	 0x728D, 0x728E, 0x7293, 0x729B, 0x72A8, 0x72AD, 0x72AE, 0x72B1,
	 0x72B4, 0x72BE, 0x72C1, 0x72C7, 0x72C9, 0x72CC, 0x72D5, 0x72D6,
	 0x72D8, 0x72DF, 0x72E5, 0x72F3, 0x72F4, 0x72FA, 0x72FB,
/* 0xCBA1 */	 0x72FE, 0x7302, 0x7304, 0x7305, 0x7307, 0x730B, 0x730D,
	 0x7312, 0x7313, 0x7318, 0x7319, 0x731E, 0x7322, 0x7324, 0x7327,
	 0x7328, 0x732C, 0x7331, 0x7332, 0x7335, 0x733A, 0x733B, 0x733D,
	 0x7343, 0x734D, 0x7350, 0x7352, 0x7356, 0x7358, 0x735D, 0x735E,
	 0x735F, 0x7360, 0x7366, 0x7367, 0x7369, 0x736B, 0x736C, 0x736E,
	 0x736F, 0x7371, 0x7377, 0x7379, 0x737C, 0x7380, 0x7381, 0x7383,
	 0x7385, 0x7386, 0x738E, 0x7390, 0x7393, 0x7395, 0x7397, 0x7398,
	 0x739C, 0x739E, 0x739F, 0x73A0, 0x73A2, 0x73A5, 0x73A6, 0x73AA,
	 0x73AB, 0x73AD, 0x73B5, 0x73B7, 0x73B9, 0x73BC, 0x73BD, 0x73BF,
	 0x73C5, 0x73C6, 0x73C9, 0x73CB, 0x73CC, 0x73CF, 0x73D2, 0x73D3,
	 0x73D6, 0x73D9, 0x73DD, 0x73E1, 0x73E3, 0x73E6, 0x73E7, 0x73E9,
	 0x73F4, 0x73F5, 0x73F7, 0x73F9, 0x73FA, 0x73FB, 0x73FD,
/* 0xCCA1 */	 0x73FF, 0x7400, 0x7401, 0x7404, 0x7407, 0x740A, 0x7411,
	 0x741A, 0x741B, 0x7424, 0x7426, 0x7428, 0x7429, 0x742A, 0x742B,
	 0x742C, 0x742D, 0x742E, 0x742F, 0x7430, 0x7431, 0x7439, 0x7440,
	 0x7443, 0x7444, 0x7446, 0x7447, 0x744B, 0x744D, 0x7451, 0x7452,
	 0x7457, 0x745D, 0x7462, 0x7466, 0x7467, 0x7468, 0x746B, 0x746D,
	 0x746E, 0x7471, 0x7472, 0x7480, 0x7481, 0x7485, 0x7486, 0x7487,
	 0x7489, 0x748F, 0x7490, 0x7491, 0x7492, 0x7498, 0x7499, 0x749A,
	 0x749C, 0x749F, 0x74A0, 0x74A1, 0x74A3, 0x74A6, 0x74A8, 0x74A9,
	 0x74AA, 0x74AB, 0x74AE, 0x74AF, 0x74B1, 0x74B2, 0x74B5, 0x74B9,
	 0x74BB, 0x74BF, 0x74C8, 0x74C9, 0x74CC, 0x74D0, 0x74D3, 0x74D8,
	 0x74DA, 0x74DB, 0x74DE, 0x74DF, 0x74E4, 0x74E8, 0x74EA, 0x74EB,
	 0x74EF, 0x74F4, 0x74FA, 0x74FB, 0x74FC, 0x74FF, 0x7506,
/* 0xCDA1 */	 0x7512, 0x7516, 0x7517, 0x7520, 0x7521, 0x7524, 0x7527,
	 0x7529, 0x752A, 0x752F, 0x7536, 0x7539, 0x753D, 0x753E, 0x753F,
	 0x7540, 0x7543, 0x7547, 0x7548, 0x754E, 0x7550, 0x7552, 0x7557,
	 0x755E, 0x755F, 0x7561, 0x756F, 0x7571, 0x7579, 0x757A, 0x757B,
	 0x757C, 0x757D, 0x757E, 0x7581, 0x7585, 0x7590, 0x7592, 0x7593,
	 0x7595, 0x7599, 0x759C, 0x75A2, 0x75A4, 0x75B4, 0x75BA, 0x75BF,
	 0x75C0, 0x75C1, 0x75C4, 0x75C6, 0x75CC, 0x75CE, 0x75CF, 0x75D7,
	 0x75DC, 0x75DF, 0x75E0, 0x75E1, 0x75E4, 0x75E7, 0x75EC, 0x75EE,
	 0x75EF, 0x75F1, 0x75F9, 0x7600, 0x7602, 0x7603, 0x7604, 0x7607,
	 0x7608, 0x760A, 0x760C, 0x760F, 0x7612, 0x7613, 0x7615, 0x7616,
	 0x7619, 0x761B, 0x761C, 0x761D, 0x761E, 0x7623, 0x7625, 0x7626,
	 0x7629, 0x762D, 0x7632, 0x7633, 0x7635, 0x7638, 0x7639,
/* 0xCEA1 */	 0x763A, 0x763C, 0x764A, 0x7640, 0x7641, 0x7643, 0x7644,
	 0x7645, 0x7649, 0x764B, 0x7655, 0x7659, 0x765F, 0x7664, 0x7665,
	 0x766D, 0x766E, 0x766F, 0x7671, 0x7674, 0x7681, 0x7685, 0x768C,
	 0x768D, 0x7695, 0x769B, 0x769C, 0x769D, 0x769F, 0x76A0, 0x76A2,
	 0x76A3, 0x76A4, 0x76A5, 0x76A6, 0x76A7, 0x76A8, 0x76AA, 0x76AD,
	 0x76BD, 0x76C1, 0x76C5, 0x76C9, 0x76CB, 0x76CC, 0x76CE, 0x76D4,
	 0x76D9, 0x76E0, 0x76E6, 0x76E8, 0x76EC, 0x76F0, 0x76F1, 0x76F6,
	 0x76F9, 0x76FC, 0x7700, 0x7706, 0x770A, 0x770E, 0x7712, 0x7714,
	 0x7715, 0x7717, 0x7719, 0x771A, 0x771C, 0x7722, 0x7728, 0x772D,
	 0x772E, 0x772F, 0x7734, 0x7735, 0x7736, 0x7739, 0x773D, 0x773E,
	 0x7742, 0x7745, 0x7746, 0x774A, 0x774D, 0x774E, 0x774F, 0x7752,
	 0x7756, 0x7757, 0x775C, 0x775E, 0x775F, 0x7760, 0x7762,
/* 0xCFA1 */	 0x7764, 0x7767, 0x776A, 0x776C, 0x7770, 0x7772, 0x7773,
	 0x7774, 0x777A, 0x777D, 0x7780, 0x7784, 0x778C, 0x778D, 0x7794,
	 0x7795, 0x7796, 0x779A, 0x779F, 0x77A2, 0x77A7, 0x77AA, 0x77AE,
	 0x77AF, 0x77B1, 0x77B5, 0x77BE, 0x77C3, 0x77C9, 0x77D1, 0x77D2,
	 0x77D5, 0x77D9, 0x77DE, 0x77DF, 0x77E0, 0x77E4, 0x77E6, 0x77EA,
	 0x77EC, 0x77F0, 0x77F1, 0x77F4, 0x77F8, 0x77FB, 0x7805, 0x7806,
	 0x7809, 0x780D, 0x780E, 0x7811, 0x781D, 0x7821, 0x7822, 0x7823,
	 0x782D, 0x782E, 0x7830, 0x7835, 0x7837, 0x7843, 0x7844, 0x7847,
	 0x7848, 0x784C, 0x784E, 0x7852, 0x785C, 0x785E, 0x7860, 0x7861,
	 0x7863, 0x7864, 0x7868, 0x786A, 0x786E, 0x787A, 0x787E, 0x788A,
	 0x788F, 0x7894, 0x7898, 0x78A1, 0x789D, 0x789E, 0x789F, 0x78A4,
	 0x78A8, 0x78AC, 0x78AD, 0x78B0, 0x78B1, 0x78B2, 0x78B3,
/* 0xD0A1 */	 0x78BB, 0x78BD, 0x78BF, 0x78C7, 0x78C8, 0x78C9, 0x78CC,
	 0x78CE, 0x78D2, 0x78D3, 0x78D5, 0x78D6, 0x78E4, 0x78DB, 0x78DF,
	 0x78E0, 0x78E1, 0x78E6, 0x78EA, 0x78F2, 0x78F3, 0x7900, 0x78F6,
	 0x78F7, 0x78FA, 0x78FB, 0x78FF, 0x7906, 0x790C, 0x7910, 0x791A,
	 0x791C, 0x791E, 0x791F, 0x7920, 0x7925, 0x7927, 0x7929, 0x792D,
	 0x7931, 0x7934, 0x7935, 0x793B, 0x793D, 0x793F, 0x7944, 0x7945,
	 0x7946, 0x794A, 0x794B, 0x794F, 0x7951, 0x7954, 0x7958, 0x795B,
	 0x795C, 0x7967, 0x7969, 0x796B, 0x7972, 0x7979, 0x797B, 0x797C,
	 0x797E, 0x798B, 0x798C, 0x7991, 0x7993, 0x7994, 0x7995, 0x7996,
	 0x7998, 0x799B, 0x799C, 0x79A1, 0x79A8, 0x79A9, 0x79AB, 0x79AF,
	 0x79B1, 0x79B4, 0x79B8, 0x79BB, 0x79C2, 0x79C4, 0x79C7, 0x79C8,
	 0x79CA, 0x79CF, 0x79D4, 0x79D6, 0x79DA, 0x79DD, 0x79DE,
/* 0xD1A1 */	 0x79E0, 0x79E2, 0x79E5, 0x79EA, 0x79EB, 0x79ED, 0x79F1,
	 0x79F8, 0x79FC, 0x7A02, 0x7A03, 0x7A07, 0x7A09, 0x7A0A, 0x7A0C,
	 0x7A11, 0x7A15, 0x7A1B, 0x7A1E, 0x7A21, 0x7A27, 0x7A2B, 0x7A2D,
	 0x7A2F, 0x7A30, 0x7A34, 0x7A35, 0x7A38, 0x7A39, 0x7A3A, 0x7A44,
	 0x7A45, 0x7A47, 0x7A48, 0x7A4C, 0x7A55, 0x7A56, 0x7A59, 0x7A5C,
	 0x7A5D, 0x7A5F, 0x7A60, 0x7A65, 0x7A67, 0x7A6A, 0x7A6D, 0x7A75,
	 0x7A78, 0x7A7E, 0x7A80, 0x7A82, 0x7A85, 0x7A86, 0x7A8A, 0x7A8B,
	 0x7A90, 0x7A91, 0x7A94, 0x7A9E, 0x7AA0, 0x7AA3, 0x7AAC, 0x7AB3,
	 0x7AB5, 0x7AB9, 0x7ABB, 0x7ABC, 0x7AC6, 0x7AC9, 0x7ACC, 0x7ACE,
	 0x7AD1, 0x7ADB, 0x7AE8, 0x7AE9, 0x7AEB, 0x7AEC, 0x7AF1, 0x7AF4,
	 0x7AFB, 0x7AFD, 0x7AFE, 0x7B07, 0x7B14, 0x7B1F, 0x7B23, 0x7B27,
	 0x7B29, 0x7B2A, 0x7B2B, 0x7B2D, 0x7B2E, 0x7B2F, 0x7B30,
/* 0xD2A1 */	 0x7B31, 0x7B34, 0x7B3D, 0x7B3F, 0x7B40, 0x7B41, 0x7B47,
	 0x7B4E, 0x7B55, 0x7B60, 0x7B64, 0x7B66, 0x7B69, 0x7B6A, 0x7B6D,
	 0x7B6F, 0x7B72, 0x7B73, 0x7B77, 0x7B84, 0x7B89, 0x7B8E, 0x7B90,
	 0x7B91, 0x7B96, 0x7B9B, 0x7B9E, 0x7BA0, 0x7BA5, 0x7BAC, 0x7BAF,
	 0x7BB0, 0x7BB2, 0x7BB5, 0x7BB6, 0x7BBA, 0x7BBB, 0x7BBC, 0x7BBD,
	 0x7BC2, 0x7BC5, 0x7BC8, 0x7BCA, 0x7BD4, 0x7BD6, 0x7BD7, 0x7BD9,
	 0x7BDA, 0x7BDB, 0x7BE8, 0x7BEA, 0x7BF2, 0x7BF4, 0x7BF5, 0x7BF8,
	 0x7BF9, 0x7BFA, 0x7BFC, 0x7BFE, 0x7C01, 0x7C02, 0x7C03, 0x7C04,
	 0x7C06, 0x7C09, 0x7C0B, 0x7C0C, 0x7C0E, 0x7C0F, 0x7C19, 0x7C1B,
	 0x7C20, 0x7C25, 0x7C26, 0x7C28, 0x7C2C, 0x7C31, 0x7C33, 0x7C34,
	 0x7C36, 0x7C39, 0x7C3A, 0x7C46, 0x7C4A, 0x7C55, 0x7C51, 0x7C52,
	 0x7C53, 0x7C59, 0x7C5A, 0x7C5B, 0x7C5C, 0x7C5D, 0x7C5E,
/* 0xD3A1 */	 0x7C61, 0x7C63, 0x7C67, 0x7C69, 0x7C6D, 0x7C6E, 0x7C70,
	 0x7C72, 0x7C79, 0x7C7C, 0x7C7D, 0x7C86, 0x7C87, 0x7C8F, 0x7C94,
	 0x7C9E, 0x7CA0, 0x7CA6, 0x7CB0, 0x7CB6, 0x7CB7, 0x7CBA, 0x7CBB,
	 0x7CBC, 0x7CBF, 0x7CC4, 0x7CC7, 0x7CC8, 0x7CC9, 0x7CCD, 0x7CCF,
	 0x7CD3, 0x7CD4, 0x7CD5, 0x7CD7, 0x7CD9, 0x7CDA, 0x7CDD, 0x7CE6,
	 0x7CE9, 0x7CEB, 0x7CF5, 0x7D03, 0x7D07, 0x7D08, 0x7D09, 0x7D0F,
	 0x7D11, 0x7D12, 0x7D13, 0x7D16, 0x7D1D, 0x7D1E, 0x7D23, 0x7D26,
	 0x7D2A, 0x7D2D, 0x7D31, 0x7D3C, 0x7D3D, 0x7D3E, 0x7D40, 0x7D41,
	 0x7D47, 0x7D48, 0x7D4D, 0x7D51, 0x7D53, 0x7D57, 0x7D59, 0x7D5A,
	 0x7D5C, 0x7D5D, 0x7D65, 0x7D67, 0x7D6A, 0x7D70, 0x7D78, 0x7D7A,
	 0x7D7B, 0x7D7F, 0x7D81, 0x7D82, 0x7D83, 0x7D85, 0x7D86, 0x7D88,
	 0x7D8B, 0x7D8C, 0x7D8D, 0x7D91, 0x7D96, 0x7D97, 0x7D9D,
/* 0xD4A1 */	 0x7D9E, 0x7DA6, 0x7DA7, 0x7DAA, 0x7DB3, 0x7DB6, 0x7DB7,
	 0x7DB9, 0x7DC2, 0x7DC3, 0x7DC4, 0x7DC5, 0x7DC6, 0x7DCC, 0x7DCD,
	 0x7DCE, 0x7DD7, 0x7DD9, 0x7E00, 0x7DE2, 0x7DE5, 0x7DE6, 0x7DEA,
	 0x7DEB, 0x7DED, 0x7DF1, 0x7DF5, 0x7DF6, 0x7DF9, 0x7DFA, 0x7E08,
	 0x7E10, 0x7E11, 0x7E15, 0x7E17, 0x7E1C, 0x7E1D, 0x7E20, 0x7E27,
	 0x7E28, 0x7E2C, 0x7E2D, 0x7E2F, 0x7E33, 0x7E36, 0x7E3F, 0x7E44,
	 0x7E45, 0x7E47, 0x7E4E, 0x7E50, 0x7E52, 0x7E58, 0x7E5F, 0x7E61,
	 0x7E62, 0x7E65, 0x7E6B, 0x7E6E, 0x7E6F, 0x7E73, 0x7E78, 0x7E7E,
	 0x7E81, 0x7E86, 0x7E87, 0x7E8A, 0x7E8D, 0x7E91, 0x7E95, 0x7E98,
	 0x7E9A, 0x7E9D, 0x7E9E, 0x7F3C, 0x7F3B, 0x7F3D, 0x7F3E, 0x7F3F,
	 0x7F43, 0x7F44, 0x7F47, 0x7F4F, 0x7F52, 0x7F53, 0x7F5B, 0x7F5C,
	 0x7F5D, 0x7F61, 0x7F63, 0x7F64, 0x7F65, 0x7F66, 0x7F6D,
/* 0xD5A1 */	 0x7F71, 0x7F7D, 0x7F7E, 0x7F7F, 0x7F80, 0x7F8B, 0x7F8D,
	 0x7F8F, 0x7F90, 0x7F91, 0x7F96, 0x7F97, 0x7F9C, 0x7FA1, 0x7FA2,
	 0x7FA6, 0x7FAA, 0x7FAD, 0x7FB4, 0x7FBC, 0x7FBF, 0x7FC0, 0x7FC3,
	 0x7FC8, 0x7FCE, 0x7FCF, 0x7FDB, 0x7FDF, 0x7FE3, 0x7FE5, 0x7FE8,
	 0x7FEC, 0x7FEE, 0x7FEF, 0x7FF2, 0x7FFA, 0x7FFD, 0x7FFE, 0x7FFF,
	 0x8007, 0x8008, 0x800A, 0x800D, 0x800E, 0x800F, 0x8011, 0x8013,
	 0x8014, 0x8016, 0x801D, 0x801E, 0x801F, 0x8020, 0x8024, 0x8026,
	 0x802C, 0x802E, 0x8030, 0x8034, 0x8035, 0x8037, 0x8039, 0x803A,
	 0x803C, 0x803E, 0x8040, 0x8044, 0x8060, 0x8064, 0x8066, 0x806D,
	 0x8071, 0x8075, 0x8081, 0x8088, 0x808E, 0x809C, 0x809E, 0x80A6,
	 0x80A7, 0x80AB, 0x80B8, 0x80B9, 0x80C8, 0x80CD, 0x80CF, 0x80D2,
	 0x80D4, 0x80D5, 0x80D7, 0x80D8, 0x80E0, 0x80ED, 0x80EE,
/* 0xD6A1 */	 0x80F0, 0x80F2, 0x80F3, 0x80F6, 0x80F9, 0x80FA, 0x80FE,
	 0x8103, 0x810B, 0x8116, 0x8117, 0x8118, 0x811C, 0x811E, 0x8120,
	 0x8124, 0x8127, 0x812C, 0x8130, 0x8135, 0x813A, 0x813C, 0x8145,
	 0x8147, 0x814A, 0x814C, 0x8152, 0x8157, 0x8160, 0x8161, 0x8167,
	 0x8168, 0x8169, 0x816D, 0x816F, 0x8177, 0x8181, 0x8190, 0x8184,
	 0x8185, 0x8186, 0x818B, 0x818E, 0x8196, 0x8198, 0x819B, 0x819E,
	 0x81A2, 0x81AE, 0x81B2, 0x81B4, 0x81BB, 0x81CB, 0x81C3, 0x81C5,
	 0x81CA, 0x81CE, 0x81CF, 0x81D5, 0x81D7, 0x81DB, 0x81DD, 0x81DE,
	 0x81E1, 0x81E4, 0x81EB, 0x81EC, 0x81F0, 0x81F1, 0x81F2, 0x81F5,
	 0x81F6, 0x81F8, 0x81F9, 0x81FD, 0x81FF, 0x8200, 0x8203, 0x820F,
	 0x8213, 0x8214, 0x8219, 0x821A, 0x821D, 0x8221, 0x8222, 0x8228,
	 0x8232, 0x8234, 0x823A, 0x8243, 0x8244, 0x8245, 0x8246,
/* 0xD7A1 */	 0x824B, 0x824E, 0x824F, 0x8251, 0x8256, 0x825C, 0x8260,
	 0x8263, 0x8267, 0x826D, 0x8274, 0x827B, 0x827D, 0x827F, 0x8280,
	 0x8281, 0x8283, 0x8284, 0x8287, 0x8289, 0x828A, 0x828E, 0x8291,
	 0x8294, 0x8296, 0x8298, 0x829A, 0x829B, 0x82A0, 0x82A1, 0x82A3,
	 0x82A4, 0x82A7, 0x82A8, 0x82A9, 0x82AA, 0x82AE, 0x82B0, 0x82B2,
	 0x82B4, 0x82B7, 0x82BA, 0x82BC, 0x82BE, 0x82BF, 0x82C6, 0x82D0,
	 0x82D5, 0x82DA, 0x82E0, 0x82E2, 0x82E4, 0x82E8, 0x82EA, 0x82ED,
	 0x82EF, 0x82F6, 0x82F7, 0x82FD, 0x82FE, 0x8300, 0x8301, 0x8307,
	 0x8308, 0x830A, 0x830B, 0x8354, 0x831B, 0x831D, 0x831E, 0x831F,
	 0x8321, 0x8322, 0x832C, 0x832D, 0x832E, 0x8330, 0x8333, 0x8337,
	 0x833A, 0x833C, 0x833D, 0x8342, 0x8343, 0x8344, 0x8347, 0x834D,
	 0x834E, 0x8351, 0x8355, 0x8356, 0x8357, 0x8370, 0x8378,
/* 0xD8A1 */	 0x837D, 0x837F, 0x8380, 0x8382, 0x8384, 0x8386, 0x838D,
	 0x8392, 0x8394, 0x8395, 0x8398, 0x8399, 0x839B, 0x839C, 0x839D,
	 0x83A6, 0x83A7, 0x83A9, 0x83AC, 0x83BE, 0x83BF, 0x83C0, 0x83C7,
	 0x83C9, 0x83CF, 0x83D0, 0x83D1, 0x83D4, 0x83DD, 0x8353, 0x83E8,
	 0x83EA, 0x83F6, 0x83F8, 0x83F9, 0x83FC, 0x8401, 0x8406, 0x840A,
	 0x840F, 0x8411, 0x8415, 0x8419, 0x83AD, 0x842F, 0x8439, 0x8445,
	 0x8447, 0x8448, 0x844A, 0x844D, 0x844F, 0x8451, 0x8452, 0x8456,
	 0x8458, 0x8459, 0x845A, 0x845C, 0x8460, 0x8464, 0x8465, 0x8467,
	 0x846A, 0x8470, 0x8473, 0x8474, 0x8476, 0x8478, 0x847C, 0x847D,
	 0x8481, 0x8485, 0x8492, 0x8493, 0x8495, 0x849E, 0x84A6, 0x84A8,
	 0x84A9, 0x84AA, 0x84AF, 0x84B1, 0x84B4, 0x84BA, 0x84BD, 0x84BE,
	 0x84C0, 0x84C2, 0x84C7, 0x84C8, 0x84CC, 0x84CF, 0x84D3,
/* 0xD9A1 */	 0x84DC, 0x84E7, 0x84EA, 0x84EF, 0x84F0, 0x84F1, 0x84F2,
	 0x84F7, 0x8532, 0x84FA, 0x84FB, 0x84FD, 0x8502, 0x8503, 0x8507,
	 0x850C, 0x850E, 0x8510, 0x851C, 0x851E, 0x8522, 0x8523, 0x8524,
	 0x8525, 0x8527, 0x852A, 0x852B, 0x852F, 0x8533, 0x8534, 0x8536,
	 0x853F, 0x8546, 0x854F, 0x8550, 0x8551, 0x8552, 0x8553, 0x8556,
	 0x8559, 0x855C, 0x855D, 0x855E, 0x855F, 0x8560, 0x8561, 0x8562,
	 0x8564, 0x856B, 0x856F, 0x8579, 0x857A, 0x857B, 0x857D, 0x857F,
	 0x8581, 0x8585, 0x8586, 0x8589, 0x858B, 0x858C, 0x858F, 0x8593,
	 0x8598, 0x859D, 0x859F, 0x85A0, 0x85A2, 0x85A5, 0x85A7, 0x85B4,
	 0x85B6, 0x85B7, 0x85B8, 0x85BC, 0x85BD, 0x85BE, 0x85BF, 0x85C2,
	 0x85C7, 0x85CA, 0x85CB, 0x85CE, 0x85AD, 0x85D8, 0x85DA, 0x85DF,
	 0x85E0, 0x85E6, 0x85E8, 0x85ED, 0x85F3, 0x85F6, 0x85FC,
/* 0xDAA1 */	 0x85FF, 0x8600, 0x8604, 0x8605, 0x860D, 0x860E, 0x8610,
	 0x8611, 0x8612, 0x8618, 0x8619, 0x861B, 0x861E, 0x8621, 0x8627,
	 0x8629, 0x8636, 0x8638, 0x863A, 0x863C, 0x863D, 0x8640, 0x8642,
	 0x8646, 0x8652, 0x8653, 0x8656, 0x8657, 0x8658, 0x8659, 0x865D,
	 0x8660, 0x8661, 0x8662, 0x8663, 0x8664, 0x8669, 0x866C, 0x866F,
	 0x8675, 0x8676, 0x8677, 0x867A, 0x868D, 0x8691, 0x8696, 0x8698,
	 0x869A, 0x869C, 0x86A1, 0x86A6, 0x86A7, 0x86A8, 0x86AD, 0x86B1,
	 0x86B3, 0x86B4, 0x86B5, 0x86B7, 0x86B8, 0x86B9, 0x86BF, 0x86C0,
	 0x86C1, 0x86C3, 0x86C5, 0x86D1, 0x86D2, 0x86D5, 0x86D7, 0x86DA,
	 0x86DC, 0x86E0, 0x86E3, 0x86E5, 0x86E7, 0x8688, 0x86FA, 0x86FC,
	 0x86FD, 0x8704, 0x8705, 0x8707, 0x870B, 0x870E, 0x870F, 0x8710,
	 0x8713, 0x8714, 0x8719, 0x871E, 0x871F, 0x8721, 0x8723,
/* 0xDBA1 */	 0x8728, 0x872E, 0x872F, 0x8731, 0x8732, 0x8739, 0x873A,
	 0x873C, 0x873D, 0x873E, 0x8740, 0x8743, 0x8745, 0x874D, 0x8758,
	 0x875D, 0x8761, 0x8764, 0x8765, 0x876F, 0x8771, 0x8772, 0x877B,
	 0x8783, 0x8784, 0x8785, 0x8786, 0x8787, 0x8788, 0x8789, 0x878B,
	 0x878C, 0x8790, 0x8793, 0x8795, 0x8797, 0x8798, 0x8799, 0x879E,
	 0x87A0, 0x87A3, 0x87A7, 0x87AC, 0x87AD, 0x87AE, 0x87B1, 0x87B5,
	 0x87BE, 0x87BF, 0x87C1, 0x87C8, 0x87C9, 0x87CA, 0x87CE, 0x87D5,
	 0x87D6, 0x87D9, 0x87DA, 0x87DC, 0x87DF, 0x87E2, 0x87E3, 0x87E4,
	 0x87EA, 0x87EB, 0x87ED, 0x87F1, 0x87F3, 0x87F8, 0x87FA, 0x87FF,
	 0x8801, 0x8803, 0x8806, 0x8809, 0x880A, 0x880B, 0x8810, 0x8819,
	 0x8812, 0x8813, 0x8814, 0x8818, 0x881A, 0x881B, 0x881C, 0x881E,
	 0x881F, 0x8828, 0x882D, 0x882E, 0x8830, 0x8832, 0x8835,
/* 0xDCA1 */	 0x883A, 0x883C, 0x8841, 0x8843, 0x8845, 0x8848, 0x8849,
	 0x884A, 0x884B, 0x884E, 0x8851, 0x8855, 0x8856, 0x8858, 0x885A,
	 0x885C, 0x885F, 0x8860, 0x8864, 0x8869, 0x8871, 0x8879, 0x887B,
	 0x8880, 0x8898, 0x889A, 0x889B, 0x889C, 0x889F, 0x88A0, 0x88A8,
	 0x88AA, 0x88BA, 0x88BD, 0x88BE, 0x88C0, 0x88CA, 0x88CB, 0x88CC,
	 0x88CD, 0x88CE, 0x88D1, 0x88D2, 0x88D3, 0x88DB, 0x88DE, 0x88E7,
	 0x88EF, 0x88F0, 0x88F1, 0x88F5, 0x88F7, 0x8901, 0x8906, 0x890D,
	 0x890E, 0x890F, 0x8915, 0x8916, 0x8918, 0x8919, 0x891A, 0x891C,
	 0x8920, 0x8926, 0x8927, 0x8928, 0x8930, 0x8931, 0x8932, 0x8935,
	 0x8939, 0x893A, 0x893E, 0x8940, 0x8942, 0x8945, 0x8946, 0x8949,
	 0x894F, 0x8952, 0x8957, 0x895A, 0x895B, 0x895C, 0x8961, 0x8962,
	 0x8963, 0x896B, 0x896E, 0x8970, 0x8973, 0x8975, 0x897A,
/* 0xDDA1 */	 0x897B, 0x897C, 0x897D, 0x8989, 0x898D, 0x8990, 0x8994,
	 0x8995, 0x899B, 0x899C, 0x899F, 0x89A0, 0x89A5, 0x89B0, 0x89B4,
	 0x89B5, 0x89B6, 0x89B7, 0x89BC, 0x89D4, 0x89D5, 0x89D6, 0x89D7,
	 0x89D8, 0x89E5, 0x89E9, 0x89EB, 0x89ED, 0x89F1, 0x89F3, 0x89F6,
	 0x89F9, 0x89FD, 0x89FF, 0x8A04, 0x8A05, 0x8A07, 0x8A0F, 0x8A11,
	 0x8A12, 0x8A14, 0x8A15, 0x8A1E, 0x8A20, 0x8A22, 0x8A24, 0x8A26,
	 0x8A2B, 0x8A2C, 0x8A2F, 0x8A35, 0x8A37, 0x8A3D, 0x8A3E, 0x8A40,
	 0x8A43, 0x8A45, 0x8A47, 0x8A49, 0x8A4D, 0x8A4E, 0x8A53, 0x8A56,
	 0x8A57, 0x8A58, 0x8A5C, 0x8A5D, 0x8A61, 0x8A65, 0x8A67, 0x8A75,
	 0x8A76, 0x8A77, 0x8A79, 0x8A7A, 0x8A7B, 0x8A7E, 0x8A7F, 0x8A80,
	 0x8A83, 0x8A86, 0x8A8B, 0x8A8F, 0x8A90, 0x8A92, 0x8A96, 0x8A97,
	 0x8A99, 0x8A9F, 0x8AA7, 0x8AA9, 0x8AAE, 0x8AAF, 0x8AB3,
/* 0xDEA1 */	 0x8AB6, 0x8AB7, 0x8ABB, 0x8ABE, 0x8AC3, 0x8AC6, 0x8AC8,
	 0x8AC9, 0x8ACA, 0x8AD1, 0x8AD3, 0x8AD4, 0x8AD5, 0x8AD7, 0x8ADD,
	 0x8ADF, 0x8AEC, 0x8AF0, 0x8AF4, 0x8AF5, 0x8AF6, 0x8AFC, 0x8AFF,
	 0x8B05, 0x8B06, 0x8B0B, 0x8B11, 0x8B1C, 0x8B1E, 0x8B1F, 0x8B0A,
	 0x8B2D, 0x8B30, 0x8B37, 0x8B3C, 0x8B42, 0x8B43, 0x8B44, 0x8B45,
	 0x8B46, 0x8B48, 0x8B52, 0x8B53, 0x8B54, 0x8B59, 0x8B4D, 0x8B5E,
	 0x8B63, 0x8B6D, 0x8B76, 0x8B78, 0x8B79, 0x8B7C, 0x8B7E, 0x8B81,
	 0x8B84, 0x8B85, 0x8B8B, 0x8B8D, 0x8B8F, 0x8B94, 0x8B95, 0x8B9C,
	 0x8B9E, 0x8B9F, 0x8C38, 0x8C39, 0x8C3D, 0x8C3E, 0x8C45, 0x8C47,
	 0x8C49, 0x8C4B, 0x8C4F, 0x8C51, 0x8C53, 0x8C54, 0x8C57, 0x8C58,
	 0x8C5B, 0x8C5D, 0x8C59, 0x8C63, 0x8C64, 0x8C66, 0x8C68, 0x8C69,
	 0x8C6D, 0x8C73, 0x8C75, 0x8C76, 0x8C7B, 0x8C7E, 0x8C86,
/* 0xDFA1 */	 0x8C87, 0x8C8B, 0x8C90, 0x8C92, 0x8C93, 0x8C99, 0x8C9B,
	 0x8C9C, 0x8CA4, 0x8CB9, 0x8CBA, 0x8CC5, 0x8CC6, 0x8CC9, 0x8CCB,
	 0x8CCF, 0x8CD6, 0x8CD5, 0x8CD9, 0x8CDD, 0x8CE1, 0x8CE8, 0x8CEC,
	 0x8CEF, 0x8CF0, 0x8CF2, 0x8CF5, 0x8CF7, 0x8CF8, 0x8CFE, 0x8CFF,
	 0x8D01, 0x8D03, 0x8D09, 0x8D12, 0x8D17, 0x8D1B, 0x8D65, 0x8D69,
	 0x8D6C, 0x8D6E, 0x8D7F, 0x8D82, 0x8D84, 0x8D88, 0x8D8D, 0x8D90,
	 0x8D91, 0x8D95, 0x8D9E, 0x8D9F, 0x8DA0, 0x8DA6, 0x8DAB, 0x8DAC,
	 0x8DAF, 0x8DB2, 0x8DB5, 0x8DB7, 0x8DB9, 0x8DBB, 0x8DC0, 0x8DC5,
	 0x8DC6, 0x8DC7, 0x8DC8, 0x8DCA, 0x8DCE, 0x8DD1, 0x8DD4, 0x8DD5,
	 0x8DD7, 0x8DD9, 0x8DE4, 0x8DE5, 0x8DE7, 0x8DEC, 0x8DF0, 0x8DBC,
	 0x8DF1, 0x8DF2, 0x8DF4, 0x8DFD, 0x8E01, 0x8E04, 0x8E05, 0x8E06,
	 0x8E0B, 0x8E11, 0x8E14, 0x8E16, 0x8E20, 0x8E21, 0x8E22,
/* 0xE0A1 */	 0x8E23, 0x8E26, 0x8E27, 0x8E31, 0x8E33, 0x8E36, 0x8E37,
	 0x8E38, 0x8E39, 0x8E3D, 0x8E40, 0x8E41, 0x8E4B, 0x8E4D, 0x8E4E,
	 0x8E4F, 0x8E54, 0x8E5B, 0x8E5C, 0x8E5D, 0x8E5E, 0x8E61, 0x8E62,
	 0x8E69, 0x8E6C, 0x8E6D, 0x8E6F, 0x8E70, 0x8E71, 0x8E79, 0x8E7A,
	 0x8E7B, 0x8E82, 0x8E83, 0x8E89, 0x8E90, 0x8E92, 0x8E95, 0x8E9A,
	 0x8E9B, 0x8E9D, 0x8E9E, 0x8EA2, 0x8EA7, 0x8EA9, 0x8EAD, 0x8EAE,
	 0x8EB3, 0x8EB5, 0x8EBA, 0x8EBB, 0x8EC0, 0x8EC1, 0x8EC3, 0x8EC4,
	 0x8EC7, 0x8ECF, 0x8ED1, 0x8ED4, 0x8EDC, 0x8EE8, 0x8EEE, 0x8EF0,
	 0x8EF1, 0x8EF7, 0x8EF9, 0x8EFA, 0x8EED, 0x8F00, 0x8F02, 0x8F07,
	 0x8F08, 0x8F0F, 0x8F10, 0x8F16, 0x8F17, 0x8F18, 0x8F1E, 0x8F20,
	 0x8F21, 0x8F23, 0x8F25, 0x8F27, 0x8F28, 0x8F2C, 0x8F2D, 0x8F2E,
	 0x8F34, 0x8F35, 0x8F36, 0x8F37, 0x8F3A, 0x8F40, 0x8F41,
/* 0xE1A1 */	 0x8F43, 0x8F47, 0x8F4F, 0x8F51, 0x8F52, 0x8F53, 0x8F54,
	 0x8F55, 0x8F58, 0x8F5D, 0x8F5E, 0x8F65, 0x8F9D, 0x8FA0, 0x8FA1,
	 0x8FA4, 0x8FA5, 0x8FA6, 0x8FB5, 0x8FB6, 0x8FB8, 0x8FBE, 0x8FC0,
	 0x8FC1, 0x8FC6, 0x8FCA, 0x8FCB, 0x8FCD, 0x8FD0, 0x8FD2, 0x8FD3,
	 0x8FD5, 0x8FE0, 0x8FE3, 0x8FE4, 0x8FE8, 0x8FEE, 0x8FF1, 0x8FF5,
	 0x8FF6, 0x8FFB, 0x8FFE, 0x9002, 0x9004, 0x9008, 0x900C, 0x9018,
	 0x901B, 0x9028, 0x9029, 0x902F, 0x902A, 0x902C, 0x902D, 0x9033,
	 0x9034, 0x9037, 0x903F, 0x9043, 0x9044, 0x904C, 0x905B, 0x905D,
	 0x9062, 0x9066, 0x9067, 0x906C, 0x9070, 0x9074, 0x9079, 0x9085,
	 0x9088, 0x908B, 0x908C, 0x908E, 0x9090, 0x9095, 0x9097, 0x9098,
	 0x9099, 0x909B, 0x90A0, 0x90A1, 0x90A2, 0x90A5, 0x90B0, 0x90B2,
	 0x90B3, 0x90B4, 0x90B6, 0x90BD, 0x90CC, 0x90BE, 0x90C3,
/* 0xE2A1 */	 0x90C4, 0x90C5, 0x90C7, 0x90C8, 0x90D5, 0x90D7, 0x90D8,
	 0x90D9, 0x90DC, 0x90DD, 0x90DF, 0x90E5, 0x90D2, 0x90F6, 0x90EB,
	 0x90EF, 0x90F0, 0x90F4, 0x90FE, 0x90FF, 0x9100, 0x9104, 0x9105,
	 0x9106, 0x9108, 0x910D, 0x9110, 0x9114, 0x9116, 0x9117, 0x9118,
	 0x911A, 0x911C, 0x911E, 0x9120, 0x9125, 0x9122, 0x9123, 0x9127,
	 0x9129, 0x912E, 0x912F, 0x9131, 0x9134, 0x9136, 0x9137, 0x9139,
	 0x913A, 0x913C, 0x913D, 0x9143, 0x9147, 0x9148, 0x914F, 0x9153,
	 0x9157, 0x9159, 0x915A, 0x915B, 0x9161, 0x9164, 0x9167, 0x916D,
	 0x9174, 0x9179, 0x917A, 0x917B, 0x9181, 0x9183, 0x9185, 0x9186,
	 0x918A, 0x918E, 0x9191, 0x9193, 0x9194, 0x9195, 0x9198, 0x919E,
	 0x91A1, 0x91A6, 0x91A8, 0x91AC, 0x91AD, 0x91AE, 0x91B0, 0x91B1,
	 0x91B2, 0x91B3, 0x91B6, 0x91BB, 0x91BC, 0x91BD, 0x91BF,
/* 0xE3A1 */	 0x91C2, 0x91C3, 0x91C5, 0x91D3, 0x91D4, 0x91D7, 0x91D9,
	 0x91DA, 0x91DE, 0x91E4, 0x91E5, 0x91E9, 0x91EA, 0x91EC, 0x91ED,
	 0x91EE, 0x91EF, 0x91F0, 0x91F1, 0x91F7, 0x91F9, 0x91FB, 0x91FD,
	 0x9200, 0x9201, 0x9204, 0x9205, 0x9206, 0x9207, 0x9209, 0x920A,
	 0x920C, 0x9210, 0x9212, 0x9213, 0x9216, 0x9218, 0x921C, 0x921D,
	 0x9223, 0x9224, 0x9225, 0x9226, 0x9228, 0x922E, 0x922F, 0x9230,
	 0x9233, 0x9235, 0x9236, 0x9238, 0x9239, 0x923A, 0x923C, 0x923E,
	 0x9240, 0x9242, 0x9243, 0x9246, 0x9247, 0x924A, 0x924D, 0x924E,
	 0x924F, 0x9251, 0x9258, 0x9259, 0x925C, 0x925D, 0x9260, 0x9261,
	 0x9265, 0x9267, 0x9268, 0x9269, 0x926E, 0x926F, 0x9270, 0x9275,
	 0x9276, 0x9277, 0x9278, 0x9279, 0x927B, 0x927C, 0x927D, 0x927F,
	 0x9288, 0x9289, 0x928A, 0x928D, 0x928E, 0x9292, 0x9297,
/* 0xE4A1 */	 0x9299, 0x929F, 0x92A0, 0x92A4, 0x92A5, 0x92A7, 0x92A8,
	 0x92AB, 0x92AF, 0x92B2, 0x92B6, 0x92B8, 0x92BA, 0x92BB, 0x92BC,
	 0x92BD, 0x92BF, 0x92C0, 0x92C1, 0x92C2, 0x92C3, 0x92C5, 0x92C6,
	 0x92C7, 0x92C8, 0x92CB, 0x92CC, 0x92CD, 0x92CE, 0x92D0, 0x92D3,
	 0x92D5, 0x92D7, 0x92D8, 0x92D9, 0x92DC, 0x92DD, 0x92DF, 0x92E0,
	 0x92E1, 0x92E3, 0x92E5, 0x92E7, 0x92E8, 0x92EC, 0x92EE, 0x92F0,
	 0x92F9, 0x92FB, 0x92FF, 0x9300, 0x9302, 0x9308, 0x930D, 0x9311,
	 0x9314, 0x9315, 0x931C, 0x931D, 0x931E, 0x931F, 0x9321, 0x9324,
	 0x9325, 0x9327, 0x9329, 0x932A, 0x9333, 0x9334, 0x9336, 0x9337,
	 0x9347, 0x9348, 0x9349, 0x9350, 0x9351, 0x9352, 0x9355, 0x9357,
	 0x9358, 0x935A, 0x935E, 0x9364, 0x9365, 0x9367, 0x9369, 0x936A,
	 0x936D, 0x936F, 0x9370, 0x9371, 0x9373, 0x9374, 0x9376,
/* 0xE5A1 */	 0x937A, 0x937D, 0x937F, 0x9380, 0x9381, 0x9382, 0x9388,
	 0x938A, 0x938B, 0x938D, 0x938F, 0x9392, 0x9395, 0x9398, 0x939B,
	 0x939E, 0x93A1, 0x93A3, 0x93A4, 0x93A6, 0x93A8, 0x93AB, 0x93B4,
	 0x93B5, 0x93B6, 0x93BA, 0x93A9, 0x93C1, 0x93C4, 0x93C5, 0x93C6,
	 0x93C7, 0x93C9, 0x93CA, 0x93CB, 0x93CC, 0x93CD, 0x93D3, 0x93D9,
	 0x93DC, 0x93DE, 0x93DF, 0x93E2, 0x93E6, 0x93E7, 0x93F9, 0x93F7,
	 0x93F8, 0x93FA, 0x93FB, 0x93FD, 0x9401, 0x9402, 0x9404, 0x9408,
	 0x9409, 0x940D, 0x940E, 0x940F, 0x9415, 0x9416, 0x9417, 0x941F,
	 0x942E, 0x942F, 0x9431, 0x9432, 0x9433, 0x9434, 0x943B, 0x943F,
	 0x943D, 0x9443, 0x9445, 0x9448, 0x944A, 0x944C, 0x9455, 0x9459,
	 0x945C, 0x945F, 0x9461, 0x9463, 0x9468, 0x946B, 0x946D, 0x946E,
	 0x946F, 0x9471, 0x9472, 0x9484, 0x9483, 0x9578, 0x9579,
/* 0xE6A1 */	 0x957E, 0x9584, 0x9588, 0x958C, 0x958D, 0x958E, 0x959D,
	 0x959E, 0x959F, 0x95A1, 0x95A6, 0x95A9, 0x95AB, 0x95AC, 0x95B4,
	 0x95B6, 0x95BA, 0x95BD, 0x95BF, 0x95C6, 0x95C8, 0x95C9, 0x95CB,
	 0x95D0, 0x95D1, 0x95D2, 0x95D3, 0x95D9, 0x95DA, 0x95DD, 0x95DE,
	 0x95DF, 0x95E0, 0x95E4, 0x95E6, 0x961D, 0x961E, 0x9622, 0x9624,
	 0x9625, 0x9626, 0x962C, 0x9631, 0x9633, 0x9637, 0x9638, 0x9639,
	 0x963A, 0x963C, 0x963D, 0x9641, 0x9652, 0x9654, 0x9656, 0x9657,
	 0x9658, 0x9661, 0x966E, 0x9674, 0x967B, 0x967C, 0x967E, 0x967F,
	 0x9681, 0x9682, 0x9683, 0x9684, 0x9689, 0x9691, 0x9696, 0x969A,
	 0x969D, 0x969F, 0x96A4, 0x96A5, 0x96A6, 0x96A9, 0x96AE, 0x96AF,
	 0x96B3, 0x96BA, 0x96CA, 0x96D2, 0x5DB2, 0x96D8, 0x96DA, 0x96DD,
	 0x96DE, 0x96DF, 0x96E9, 0x96EF, 0x96F1, 0x96FA, 0x9702,
/* 0xE7A1 */	 0x9703, 0x9705, 0x9709, 0x971A, 0x971B, 0x971D, 0x9721,
	 0x9722, 0x9723, 0x9728, 0x9731, 0x9733, 0x9741, 0x9743, 0x974A,
	 0x974E, 0x974F, 0x9755, 0x9757, 0x9758, 0x975A, 0x975B, 0x9763,
	 0x9767, 0x976A, 0x976E, 0x9773, 0x9776, 0x9777, 0x9778, 0x977B,
	 0x977D, 0x977F, 0x9780, 0x9789, 0x9795, 0x9796, 0x9797, 0x9799,
	 0x979A, 0x979E, 0x979F, 0x97A2, 0x97AC, 0x97AE, 0x97B1, 0x97B2,
	 0x97B5, 0x97B6, 0x97B8, 0x97B9, 0x97BA, 0x97BC, 0x97BE, 0x97BF,
	 0x97C1, 0x97C4, 0x97C5, 0x97C7, 0x97C9, 0x97CA, 0x97CC, 0x97CD,
	 0x97CE, 0x97D0, 0x97D1, 0x97D4, 0x97D7, 0x97D8, 0x97D9, 0x97DD,
	 0x97DE, 0x97E0, 0x97DB, 0x97E1, 0x97E4, 0x97EF, 0x97F1, 0x97F4,
	 0x97F7, 0x97F8, 0x97FA, 0x9807, 0x980A, 0x9819, 0x980D, 0x980E,
	 0x9814, 0x9816, 0x981C, 0x981E, 0x9820, 0x9823, 0x9826,
/* 0xE8A1 */	 0x982B, 0x982E, 0x982F, 0x9830, 0x9832, 0x9833, 0x9835,
	 0x9825, 0x983E, 0x9844, 0x9847, 0x984A, 0x9851, 0x9852, 0x9853,
	 0x9856, 0x9857, 0x9859, 0x985A, 0x9862, 0x9863, 0x9865, 0x9866,
	 0x986A, 0x986C, 0x98AB, 0x98AD, 0x98AE, 0x98B0, 0x98B4, 0x98B7,
	 0x98B8, 0x98BA, 0x98BB, 0x98BF, 0x98C2, 0x98C5, 0x98C8, 0x98CC,
	 0x98E1, 0x98E3, 0x98E5, 0x98E6, 0x98E7, 0x98EA, 0x98F3, 0x98F6,
	 0x9902, 0x9907, 0x9908, 0x9911, 0x9915, 0x9916, 0x9917, 0x991A,
	 0x991B, 0x991C, 0x991F, 0x9922, 0x9926, 0x9927, 0x992B, 0x9931,
	 0x9932, 0x9933, 0x9934, 0x9935, 0x9939, 0x993A, 0x993B, 0x993C,
	 0x9940, 0x9941, 0x9946, 0x9947, 0x9948, 0x994D, 0x994E, 0x9954,
	 0x9958, 0x9959, 0x995B, 0x995C, 0x995E, 0x995F, 0x9960, 0x999B,
	 0x999D, 0x999F, 0x99A6, 0x99B0, 0x99B1, 0x99B2, 0x99B5,
/* 0xE9A1 */	 0x99B9, 0x99BA, 0x99BD, 0x99BF, 0x99C3, 0x99C9, 0x99D3,
	 0x99D4, 0x99D9, 0x99DA, 0x99DC, 0x99DE, 0x99E7, 0x99EA, 0x99EB,
	 0x99EC, 0x99F0, 0x99F4, 0x99F5, 0x99F9, 0x99FD, 0x99FE, 0x9A02,
	 0x9A03, 0x9A04, 0x9A0B, 0x9A0C, 0x9A10, 0x9A11, 0x9A16, 0x9A1E,
	 0x9A20, 0x9A22, 0x9A23, 0x9A24, 0x9A27, 0x9A2D, 0x9A2E, 0x9A33,
	 0x9A35, 0x9A36, 0x9A38, 0x9A47, 0x9A41, 0x9A44, 0x9A4A, 0x9A4B,
	 0x9A4C, 0x9A4E, 0x9A51, 0x9A54, 0x9A56, 0x9A5D, 0x9AAA, 0x9AAC,
	 0x9AAE, 0x9AAF, 0x9AB2, 0x9AB4, 0x9AB5, 0x9AB6, 0x9AB9, 0x9ABB,
	 0x9ABE, 0x9ABF, 0x9AC1, 0x9AC3, 0x9AC6, 0x9AC8, 0x9ACE, 0x9AD0,
	 0x9AD2, 0x9AD5, 0x9AD6, 0x9AD7, 0x9ADB, 0x9ADC, 0x9AE0, 0x9AE4,
	 0x9AE5, 0x9AE7, 0x9AE9, 0x9AEC, 0x9AF2, 0x9AF3, 0x9AF5, 0x9AF9,
	 0x9AFA, 0x9AFD, 0x9AFF, 0x9B00, 0x9B01, 0x9B02, 0x9B03,
/* 0xEAA1 */	 0x9B04, 0x9B05, 0x9B08, 0x9B09, 0x9B0B, 0x9B0C, 0x9B0D,
	 0x9B0E, 0x9B10, 0x9B12, 0x9B16, 0x9B19, 0x9B1B, 0x9B1C, 0x9B20,
	 0x9B26, 0x9B2B, 0x9B2D, 0x9B33, 0x9B34, 0x9B35, 0x9B37, 0x9B39,
	 0x9B3A, 0x9B3D, 0x9B48, 0x9B4B, 0x9B4C, 0x9B55, 0x9B56, 0x9B57,
	 0x9B5B, 0x9B5E, 0x9B61, 0x9B63, 0x9B65, 0x9B66, 0x9B68, 0x9B6A,
	 0x9B6B, 0x9B6C, 0x9B6D, 0x9B6E, 0x9B73, 0x9B75, 0x9B77, 0x9B78,
	 0x9B79, 0x9B7F, 0x9B80, 0x9B84, 0x9B85, 0x9B86, 0x9B87, 0x9B89,
	 0x9B8A, 0x9B8B, 0x9B8D, 0x9B8F, 0x9B90, 0x9B94, 0x9B9A, 0x9B9D,
	 0x9B9E, 0x9BA6, 0x9BA7, 0x9BA9, 0x9BAC, 0x9BB0, 0x9BB1, 0x9BB2,
	 0x9BB7, 0x9BB8, 0x9BBB, 0x9BBC, 0x9BBE, 0x9BBF, 0x9BC1, 0x9BC7,
	 0x9BC8, 0x9BCE, 0x9BD0, 0x9BD7, 0x9BD8, 0x9BDD, 0x9BDF, 0x9BE5,
	 0x9BE7, 0x9BEA, 0x9BEB, 0x9BEF, 0x9BF3, 0x9BF7, 0x9BF8,
/* 0xEBA1 */	 0x9BF9, 0x9BFA, 0x9BFD, 0x9BFF, 0x9C00, 0x9C02, 0x9C0B,
	 0x9C0F, 0x9C11, 0x9C16, 0x9C18, 0x9C19, 0x9C1A, 0x9C1C, 0x9C1E,
	 0x9C22, 0x9C23, 0x9C26, 0x9C27, 0x9C28, 0x9C29, 0x9C2A, 0x9C31,
	 0x9C35, 0x9C36, 0x9C37, 0x9C3D, 0x9C41, 0x9C43, 0x9C44, 0x9C45,
	 0x9C49, 0x9C4A, 0x9C4E, 0x9C4F, 0x9C50, 0x9C53, 0x9C54, 0x9C56,
	 0x9C58, 0x9C5B, 0x9C5D, 0x9C5E, 0x9C5F, 0x9C63, 0x9C69, 0x9C6A,
	 0x9C5C, 0x9C6B, 0x9C68, 0x9C6E, 0x9C70, 0x9C72, 0x9C75, 0x9C77,
	 0x9C7B, 0x9CE6, 0x9CF2, 0x9CF7, 0x9CF9, 0x9D0B, 0x9D02, 0x9D11,
	 0x9D17, 0x9D18, 0x9D1C, 0x9D1D, 0x9D1E, 0x9D2F, 0x9D30, 0x9D32,
	 0x9D33, 0x9D34, 0x9D3A, 0x9D3C, 0x9D45, 0x9D3D, 0x9D42, 0x9D43,
	 0x9D47, 0x9D4A, 0x9D53, 0x9D54, 0x9D5F, 0x9D63, 0x9D62, 0x9D65,
	 0x9D69, 0x9D6A, 0x9D6B, 0x9D70, 0x9D76, 0x9D77, 0x9D7B,
/* 0xECA1 */	 0x9D7C, 0x9D7E, 0x9D83, 0x9D84, 0x9D86, 0x9D8A, 0x9D8D,
	 0x9D8E, 0x9D92, 0x9D93, 0x9D95, 0x9D96, 0x9D97, 0x9D98, 0x9DA1,
	 0x9DAA, 0x9DAC, 0x9DAE, 0x9DB1, 0x9DB5, 0x9DB9, 0x9DBC, 0x9DBF,
	 0x9DC3, 0x9DC7, 0x9DC9, 0x9DCA, 0x9DD4, 0x9DD5, 0x9DD6, 0x9DD7,
	 0x9DDA, 0x9DDE, 0x9DDF, 0x9DE0, 0x9DE5, 0x9DE7, 0x9DE9, 0x9DEB,
	 0x9DEE, 0x9DF0, 0x9DF3, 0x9DF4, 0x9DFE, 0x9E0A, 0x9E02, 0x9E07,
	 0x9E0E, 0x9E10, 0x9E11, 0x9E12, 0x9E15, 0x9E16, 0x9E19, 0x9E1C,
	 0x9E1D, 0x9E7A, 0x9E7B, 0x9E7C, 0x9E80, 0x9E82, 0x9E83, 0x9E84,
	 0x9E85, 0x9E87, 0x9E8E, 0x9E8F, 0x9E96, 0x9E98, 0x9E9B, 0x9E9E,
	 0x9EA4, 0x9EA8, 0x9EAC, 0x9EAE, 0x9EAF, 0x9EB0, 0x9EB3, 0x9EB4,
	 0x9EB5, 0x9EC6, 0x9EC8, 0x9ECB, 0x9ED5, 0x9EDF, 0x9EE4, 0x9EE7,
	 0x9EEC, 0x9EED, 0x9EEE, 0x9EF0, 0x9EF1, 0x9EF2, 0x9EF5,
/* 0xEDA1 */	 0x9EF8, 0x9EFF, 0x9F02, 0x9F03, 0x9F09, 0x9F0F, 0x9F10,
	 0x9F11, 0x9F12, 0x9F14, 0x9F16, 0x9F17, 0x9F19, 0x9F1A, 0x9F1B,
	 0x9F1F, 0x9F22, 0x9F26, 0x9F2A, 0x9F2B, 0x9F2F, 0x9F31, 0x9F32,
	 0x9F34, 0x9F37, 0x9F39, 0x9F3A, 0x9F3C, 0x9F3D, 0x9F3F, 0x9F41,
	 0x9F43, 0x9F44, 0x9F45, 0x9F46, 0x9F47, 0x9F53, 0x9F55, 0x9F56,
	 0x9F57, 0x9F58, 0x9F5A, 0x9F5D, 0x9F5E, 0x9F68, 0x9F69, 0x9F6D,
	 0x9F6E, 0x9F6F, 0x9F70, 0x9F71, 0x9F73, 0x9F75, 0x9F7A, 0x9F7D,
	 0x9F8F, 0x9F90, 0x9F91, 0x9F92, 0x9F94, 0x9F96, 0x9F97, 0x9F9E,
	 0x9FA1, 0x9FA2, 0x9FA3, 0x9FA5, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
};
#endif

#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
static unsigned char u2c_00hi[256 - 0xA0][2] = {
	{0x00, 0x00}, {0x22, 0xC2}, {0x00, 0x00}, {0x00, 0x00},/* 0xA0-0xA3 */
	{0x22, 0xF0}, {0x00, 0x00}, {0x00, 0x00}, {0xA1, 0xF8},/* 0xA4-0xA7 */
	{0xA1, 0xAF}, {0x22, 0xED}, {0x22, 0xEC}, {0x00, 0x00},/* 0xA8-0xAB */
	{0x00, 0x00}, {0x00, 0x00}, {0x22, 0xEE}, {0x22, 0xB4},/* 0xAC-0xAF */
	{0xA1, 0xEB}, {0xA1, 0xDE}, {0x00, 0x00}, {0x00, 0x00},/* 0xB0-0xB3 */
	{0xA1, 0xAD}, {0x00, 0x00}, {0xA2, 0xF9}, {0x00, 0x00},/* 0xB4-0xB7 */
	{0x22, 0xB1}, {0x00, 0x00}, {0x22, 0xEB}, {0x00, 0x00},/* 0xB8-0xBB */
	{0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x22, 0xC4},/* 0xBC-0xBF */
	{0x2A, 0xA2}, {0x2A, 0xA1}, {0x2A, 0xA4}, {0x2A, 0xAA},/* 0xC0-0xC3 */
	{0x2A, 0xA3}, {0x2A, 0xA9}, {0x29, 0xA1}, {0x2A, 0xAE},/* 0xC4-0xC7 */
	{0x2A, 0xB2}, {0x2A, 0xB1}, {0x2A, 0xB4}, {0x2A, 0xB3},/* 0xC8-0xCB */
	{0x2A, 0xC0}, {0x2A, 0xBF}, {0x2A, 0xC2}, {0x2A, 0xC1},/* 0xCC-0xCF */
	{0x00, 0x00}, {0x2A, 0xD0}, {0x2A, 0xD2}, {0x2A, 0xD1},/* 0xD0-0xD3 */
	{0x2A, 0xD4}, {0x2A, 0xD8}, {0x2A, 0xD3}, {0xA1, 0xDF},/* 0xD4-0xD7 */
	{0x29, 0xAC}, {0x2A, 0xE3}, {0x2A, 0xE2}, {0x2A, 0xE5},/* 0xD8-0xDB */
	{0x2A, 0xE4}, {0x2A, 0xF2}, {0x29, 0xB0}, {0x29, 0xCE},/* 0xDC-0xDF */
	{0x2B, 0xA2}, {0x2B, 0xA1}, {0x2B, 0xA4}, {0x2B, 0xAA},/* 0xE0-0xE3 */
	{0x2B, 0xA3}, {0x2B, 0xA9}, {0x29, 0xC1}, {0x2B, 0xAE},/* 0xE4-0xE7 */
	{0x2B, 0xB2}, {0x2B, 0xB1}, {0x2B, 0xB4}, {0x2B, 0xB3},/* 0xE8-0xEB */
	{0x2B, 0xC0}, {0x2B, 0xBF}, {0x2B, 0xC2}, {0x2B, 0xC1},/* 0xEC-0xEF */
	{0x29, 0xC3}, {0x2B, 0xD0}, {0x2B, 0xD2}, {0x2B, 0xD1},/* 0xF0-0xF3 */
	{0x2B, 0xD4}, {0x2B, 0xD8}, {0x2B, 0xD3}, {0xA1, 0xE0},/* 0xF4-0xF7 */
	{0x29, 0xCC}, {0x2B, 0xE3}, {0x2B, 0xE2}, {0x2B, 0xE5},/* 0xF8-0xFB */
	{0x2B, 0xE4}, {0x2B, 0xF2}, {0x29, 0xD0}, {0x2B, 0xF3},/* 0xFC-0xFF */
};

static unsigned char u2c_01[512] = {
	0x2A, 0xA7, 0x2B, 0xA7, 0x2A, 0xA5, 0x2B, 0xA5, /* 0x00-0x03 */
	0x2A, 0xA8, 0x2B, 0xA8, 0x2A, 0xAB, 0x2B, 0xAB, /* 0x04-0x07 */
	0x2A, 0xAC, 0x2B, 0xAC, 0x2A, 0xAF, 0x2B, 0xAF, /* 0x08-0x0B */
	0x2A, 0xAD, 0x2B, 0xAD, 0x2A, 0xB0, 0x2B, 0xB0, /* 0x0C-0x0F */
	0x29, 0xA2, 0x29, 0xC2, 0x2A, 0xB7, 0x2B, 0xB7, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x2A, 0xB6, 0x2B, 0xB6, /* 0x14-0x17 */
	0x2A, 0xB8, 0x2B, 0xB8, 0x2A, 0xB5, 0x2B, 0xB5, /* 0x18-0x1B */
	0x2A, 0xBA, 0x2B, 0xBA, 0x2A, 0xBB, 0x2B, 0xBB, /* 0x1C-0x1F */
	0x2A, 0xBD, 0x2B, 0xBD, 0x2A, 0xBC, 0x00, 0x00, /* 0x20-0x23 */
	0x2A, 0xBE, 0x2B, 0xBE, 0x29, 0xA4, 0x29, 0xC4, /* 0x24-0x27 */
	0x2A, 0xC7, 0x2B, 0xC7, 0x2A, 0xC5, 0x2B, 0xC5, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x2A, 0xC6, 0x2B, 0xC6, /* 0x2C-0x2F */
	0x2A, 0xC4, 0x29, 0xC5, 0x29, 0xA6, 0x29, 0xC6, /* 0x30-0x33 */
	0x2A, 0xC8, 0x2B, 0xC8, 0x2A, 0xC9, 0x2B, 0xC9, /* 0x34-0x37 */
	0x29, 0xC7, 0x2A, 0xCA, 0x2B, 0xCA, 0x2A, 0xCC, /* 0x38-0x3B */
	0x2B, 0xCC, 0x2A, 0xCB, 0x2B, 0xCB, 0x29, 0xA9, /* 0x3C-0x3F */
	0x29, 0xC9, 0x29, 0xA8, 0x29, 0xC8, 0x2A, 0xCD, /* 0x40-0x43 */
	0x2B, 0xCD, 0x2A, 0xCF, 0x2B, 0xCF, 0x2A, 0xCE, /* 0x44-0x47 */
	0x2B, 0xCE, 0x29, 0xCA, 0x29, 0xAB, 0x29, 0xCB, /* 0x48-0x4B */
	0x2A, 0xD7, 0x2B, 0xD7, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x2A, 0xD6, 0x2B, 0xD6, 0x29, 0xAD, 0x29, 0xCD, /* 0x50-0x53 */
	0x2A, 0xD9, 0x2B, 0xD9, 0x2A, 0xDB, 0x2B, 0xDB, /* 0x54-0x57 */
	0x2A, 0xDA, 0x2B, 0xDA, 0x2A, 0xDC, 0x2B, 0xDC, /* 0x58-0x5B */
	0x2A, 0xDD, 0x2B, 0xDD, 0x2A, 0xDF, 0x2B, 0xDF, /* 0x5C-0x5F */
	0x2A, 0xDE, 0x2B, 0xDE, 0x2A, 0xE1, 0x2B, 0xE1, /* 0x60-0x63 */
	0x2A, 0xE0, 0x2B, 0xE0, 0x29, 0xAF, 0x29, 0xCF, /* 0x64-0x67 */
	0x2A, 0xEC, 0x2B, 0xEC, 0x2A, 0xE9, 0x2B, 0xE9, /* 0x68-0x6B */
	0x2A, 0xE6, 0x2B, 0xE6, 0x2A, 0xEB, 0x2B, 0xEB, /* 0x6C-0x6F */
	0x2A, 0xE8, 0x2B, 0xE8, 0x2A, 0xEA, 0x2B, 0xEA, /* 0x70-0x73 */
	0x2A, 0xF1, 0x2B, 0xF1, 0x2A, 0xF4, 0x2B, 0xF4, /* 0x74-0x77 */
	0x2A, 0xF3, 0x2A, 0xF5, 0x2B, 0xF5, 0x2A, 0xF7, /* 0x78-0x7B */
	0x2B, 0xF7, 0x2A, 0xF6, 0x2B, 0xF6, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x2A, 0xA6, 0x2B, 0xA6, 0x2A, 0xC3, /* 0xCC-0xCF */
	0x2B, 0xC3, 0x2A, 0xD5, 0x2B, 0xD5, 0x2A, 0xE7, /* 0xD0-0xD3 */
	0x2B, 0xE7, 0x2A, 0xF0, 0x2B, 0xF0, 0x2A, 0xED, /* 0xD4-0xD7 */
	0x2B, 0xED, 0x2A, 0xEF, 0x2B, 0xEF, 0x2A, 0xEE, /* 0xD8-0xDB */
	0x2B, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x2B, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
};

static unsigned char u2c_02[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0xB0, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x22, 0xAF, 0x22, 0xB2, 0x22, 0xB6, 0x22, 0xB5, /* 0xD8-0xDB */
	0x00, 0x00, 0x22, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
};

static unsigned char u2c_03[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x22, 0xB8, 0x22, 0xB9, 0x26, 0xE1, 0x00, 0x00, /* 0x84-0x87 */
	0x26, 0xE2, 0x26, 0xE3, 0x26, 0xE4, 0x00, 0x00, /* 0x88-0x8B */
	0x26, 0xE7, 0x00, 0x00, 0x26, 0xE9, 0x26, 0xEC, /* 0x8C-0x8F */
	0x26, 0xF6, 0xA6, 0xA1, 0xA6, 0xA2, 0xA6, 0xA3, /* 0x90-0x93 */
	0xA6, 0xA4, 0xA6, 0xA5, 0xA6, 0xA6, 0xA6, 0xA7, /* 0x94-0x97 */
	0xA6, 0xA8, 0xA6, 0xA9, 0xA6, 0xAA, 0xA6, 0xAB, /* 0x98-0x9B */
	0xA6, 0xAC, 0xA6, 0xAD, 0xA6, 0xAE, 0xA6, 0xAF, /* 0x9C-0x9F */
	0xA6, 0xB0, 0xA6, 0xB1, 0x00, 0x00, 0xA6, 0xB2, /* 0xA0-0xA3 */
	0xA6, 0xB3, 0xA6, 0xB4, 0xA6, 0xB5, 0xA6, 0xB6, /* 0xA4-0xA7 */
	0xA6, 0xB7, 0xA6, 0xB8, 0x26, 0xE5, 0x26, 0xEA, /* 0xA8-0xAB */
	0x26, 0xF1, 0x26, 0xF2, 0x26, 0xF3, 0x26, 0xF4, /* 0xAC-0xAF */
	0x26, 0xFB, 0xA6, 0xC1, 0xA6, 0xC2, 0xA6, 0xC3, /* 0xB0-0xB3 */
	0xA6, 0xC4, 0xA6, 0xC5, 0xA6, 0xC6, 0xA6, 0xC7, /* 0xB4-0xB7 */
	0xA6, 0xC8, 0xA6, 0xC9, 0xA6, 0xCA, 0xA6, 0xCB, /* 0xB8-0xBB */
	0xA6, 0xCC, 0xA6, 0xCD, 0xA6, 0xCE, 0xA6, 0xCF, /* 0xBC-0xBF */
	0xA6, 0xD0, 0xA6, 0xD1, 0x26, 0xF8, 0xA6, 0xD2, /* 0xC0-0xC3 */
	0xA6, 0xD3, 0xA6, 0xD4, 0xA6, 0xD5, 0xA6, 0xD6, /* 0xC4-0xC7 */
	0xA6, 0xD7, 0xA6, 0xD8, 0x26, 0xF5, 0x26, 0xFA, /* 0xC8-0xCB */
	0x26, 0xF7, 0x26, 0xF9, 0x26, 0xFC, 0x00, 0x00, /* 0xCC-0xCF */
};

static unsigned char u2c_04[512] = {
	0x00, 0x00, 0xA7, 0xA7, 0x27, 0xC2, 0x27, 0xC3, /* 0x00-0x03 */
	0x27, 0xC4, 0x27, 0xC5, 0x27, 0xC6, 0x27, 0xC7, /* 0x04-0x07 */
	0x27, 0xC8, 0x27, 0xC9, 0x27, 0xCA, 0x27, 0xCB, /* 0x08-0x0B */
	0x27, 0xCC, 0x00, 0x00, 0x27, 0xCD, 0x27, 0xCE, /* 0x0C-0x0F */
	0xA7, 0xA1, 0xA7, 0xA2, 0xA7, 0xA3, 0xA7, 0xA4, /* 0x10-0x13 */
	0xA7, 0xA5, 0xA7, 0xA6, 0xA7, 0xA8, 0xA7, 0xA9, /* 0x14-0x17 */
	0xA7, 0xAA, 0xA7, 0xAB, 0xA7, 0xAC, 0xA7, 0xAD, /* 0x18-0x1B */
	0xA7, 0xAE, 0xA7, 0xAF, 0xA7, 0xB0, 0xA7, 0xB1, /* 0x1C-0x1F */
	0xA7, 0xB2, 0xA7, 0xB3, 0xA7, 0xB4, 0xA7, 0xB5, /* 0x20-0x23 */
	0xA7, 0xB6, 0xA7, 0xB7, 0xA7, 0xB8, 0xA7, 0xB9, /* 0x24-0x27 */
	0xA7, 0xBA, 0xA7, 0xBB, 0xA7, 0xBC, 0xA7, 0xBD, /* 0x28-0x2B */
	0xA7, 0xBE, 0xA7, 0xBF, 0xA7, 0xC0, 0xA7, 0xC1, /* 0x2C-0x2F */
	0xA7, 0xD1, 0xA7, 0xD2, 0xA7, 0xD3, 0xA7, 0xD4, /* 0x30-0x33 */
	0xA7, 0xD5, 0xA7, 0xD6, 0xA7, 0xD8, 0xA7, 0xD9, /* 0x34-0x37 */
	0xA7, 0xDA, 0xA7, 0xDB, 0xA7, 0xDC, 0xA7, 0xDD, /* 0x38-0x3B */
	0xA7, 0xDE, 0xA7, 0xDF, 0xA7, 0xE0, 0xA7, 0xE1, /* 0x3C-0x3F */
	0xA7, 0xE2, 0xA7, 0xE3, 0xA7, 0xE4, 0xA7, 0xE5, /* 0x40-0x43 */
	0xA7, 0xE6, 0xA7, 0xE7, 0xA7, 0xE8, 0xA7, 0xE9, /* 0x44-0x47 */
	0xA7, 0xEA, 0xA7, 0xEB, 0xA7, 0xEC, 0xA7, 0xED, /* 0x48-0x4B */
	0xA7, 0xEE, 0xA7, 0xEF, 0xA7, 0xF0, 0xA7, 0xF1, /* 0x4C-0x4F */
	0x00, 0x00, 0xA7, 0xD7, 0x27, 0xF2, 0x27, 0xF3, /* 0x50-0x53 */
	0x27, 0xF4, 0x27, 0xF5, 0x27, 0xF6, 0x27, 0xF7, /* 0x54-0x57 */
	0x27, 0xF8, 0x27, 0xF9, 0x27, 0xFA, 0x27, 0xFB, /* 0x58-0x5B */
	0x27, 0xFC, 0x00, 0x00, 0x27, 0xFD, 0x27, 0xFE, /* 0x5C-0x5F */
};

static unsigned char u2c_20[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0xA1, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0xA1, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xA1, 0xC6, 0xA1, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0xA1, 0xC8, 0xA1, 0xC9, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0xA2, 0xF7, 0xA2, 0xF8, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0xA1, 0xC5, 0xA1, 0xC4, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xA2, 0xF3, 0x00, 0x00, 0xA1, 0xEC, 0xA1, 0xED, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA2, 0xA8, /* 0x38-0x3B */
};

static unsigned char u2c_21[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA1, 0xEE, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xAD, 0xE2, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0xAD, 0xE4, 0x22, 0xEF, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA2, 0xF2, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0xAD, 0xB5, 0xAD, 0xB6, 0xAD, 0xB7, 0xAD, 0xB8, /* 0x60-0x63 */
	0xAD, 0xB9, 0xAD, 0xBA, 0xAD, 0xBB, 0xAD, 0xBC, /* 0x64-0x67 */
	0xAD, 0xBD, 0xAD, 0xBE, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xFC, 0xF1, 0xFC, 0xF2, 0xFC, 0xF3, 0xFC, 0xF4, /* 0x70-0x73 */
	0xFC, 0xF5, 0xFC, 0xF6, 0xFC, 0xF7, 0xFC, 0xF8, /* 0x74-0x77 */
	0xFC, 0xF9, 0xFC, 0xFA, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0xA2, 0xAB, 0xA2, 0xAC, 0xA2, 0xAA, 0xA2, 0xAD, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xCD, 0x00, 0x00, /* 0xD0-0xD3 */
	0xA2, 0xCE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
};

static unsigned char u2c_22[512] = {
	0xA2, 0xCF, 0x00, 0x00, 0xA2, 0xDF, 0xA2, 0xD0, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA2, 0xE0, /* 0x04-0x07 */
	0xA2, 0xBA, 0x00, 0x00, 0x00, 0x00, 0xA2, 0xBB, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xAD, 0xF4, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xE5, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0xA2, 0xE7, 0xA1, 0xE7, 0xAD, 0xF8, /* 0x1C-0x1F */
	0xA2, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0xA1, 0xC2, 0x00, 0x00, 0xA2, 0xCA, /* 0x24-0x27 */
	0xA2, 0xCB, 0xA2, 0xC1, 0xA2, 0xC0, 0xA2, 0xE9, /* 0x28-0x2B */
	0xA2, 0xEA, 0x00, 0x00, 0xAD, 0xF3, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0xA1, 0xE8, 0xA2, 0xE8, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xA2, 0xE6, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xE2, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0xA1, 0xE2, 0xA2, 0xE1, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0xA1, 0xE5, 0xA1, 0xE6, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xE3, 0xA2, 0xE4, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0xA2, 0xBE, 0xA2, 0xBF, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xBC, 0xA2, 0xBD, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xA2, 0xDD, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xF9, /* 0xBC-0xBF */
};

static unsigned char u2c_23[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xDE, 0x00, 0x00, /* 0x10-0x13 */
};

static unsigned char u2c_24[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0xAD, 0xA1, 0xAD, 0xA2, 0xAD, 0xA3, 0xAD, 0xA4, /* 0x60-0x63 */
	0xAD, 0xA5, 0xAD, 0xA6, 0xAD, 0xA7, 0xAD, 0xA8, /* 0x64-0x67 */
	0xAD, 0xA9, 0xAD, 0xAA, 0xAD, 0xAB, 0xAD, 0xAC, /* 0x68-0x6B */
	0xAD, 0xAD, 0xAD, 0xAE, 0xAD, 0xAF, 0xAD, 0xB0, /* 0x6C-0x6F */
	0xAD, 0xB1, 0xAD, 0xB2, 0xAD, 0xB3, 0xAD, 0xB4, /* 0x70-0x73 */
};

static unsigned char u2c_25[512] = {
	0xA8, 0xA1, 0xA8, 0xAC, 0xA8, 0xA2, 0xA8, 0xAD, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0xA8, 0xA3, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xAE, /* 0x0C-0x0F */
	0xA8, 0xA4, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xAF, /* 0x10-0x13 */
	0xA8, 0xA6, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB1, /* 0x14-0x17 */
	0xA8, 0xA5, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB0, /* 0x18-0x1B */
	0xA8, 0xA7, 0xA8, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0xA8, 0xB7, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB2, /* 0x20-0x23 */
	0xA8, 0xA9, 0xA8, 0xBE, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0xA8, 0xB9, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB4, /* 0x28-0x2B */
	0xA8, 0xA8, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB8, /* 0x2C-0x2F */
	0xA8, 0xBD, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB3, /* 0x30-0x33 */
	0xA8, 0xAA, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xBA, /* 0x34-0x37 */
	0xA8, 0xBF, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB5, /* 0x38-0x3B */
	0xA8, 0xAB, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xBB, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0xA8, 0xC0, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xB6, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xA2, 0xA3, 0xA2, 0xA2, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xA5, 0xA2, 0xA4, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0xA2, 0xA7, 0xA2, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xA1, 0xA1, 0xFE, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA1, 0xFB, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0xA1, 0xFD, 0xA1, 0xFC, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA2, 0xFE, /* 0xEC-0xEF */
};

static unsigned char u2c_26[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0xA1, 0xFA, 0xA1, 0xF9, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xA1, 0xEA, 0x00, 0x00, 0xA1, 0xE9, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xA2, 0xF6, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0xA2, 0xF5, 0x00, 0x00, 0xA2, 0xF4, /* 0x6C-0x6F */
};

static unsigned char u2c_30[512] = {
	0xA1, 0xA1, 0xA1, 0xA2, 0xA1, 0xA3, 0xA1, 0xB7, /* 0x00-0x03 */
	0x00, 0x00, 0xA1, 0xB9, 0xA1, 0xBA, 0xA1, 0xBB, /* 0x04-0x07 */
	0xA1, 0xD2, 0xA1, 0xD3, 0xA1, 0xD4, 0xA1, 0xD5, /* 0x08-0x0B */
	0xA1, 0xD6, 0xA1, 0xD7, 0xA1, 0xD8, 0xA1, 0xD9, /* 0x0C-0x0F */
	0xA1, 0xDA, 0xA1, 0xDB, 0xA2, 0xA9, 0xA2, 0xAE, /* 0x10-0x13 */
	0xA1, 0xCC, 0xA1, 0xCD, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0xAD, 0xE0, 0x00, 0x00, 0xAD, 0xE1, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0xA4, 0xA1, 0xA4, 0xA2, 0xA4, 0xA3, /* 0x40-0x43 */
	0xA4, 0xA4, 0xA4, 0xA5, 0xA4, 0xA6, 0xA4, 0xA7, /* 0x44-0x47 */
	0xA4, 0xA8, 0xA4, 0xA9, 0xA4, 0xAA, 0xA4, 0xAB, /* 0x48-0x4B */
	0xA4, 0xAC, 0xA4, 0xAD, 0xA4, 0xAE, 0xA4, 0xAF, /* 0x4C-0x4F */
	0xA4, 0xB0, 0xA4, 0xB1, 0xA4, 0xB2, 0xA4, 0xB3, /* 0x50-0x53 */
	0xA4, 0xB4, 0xA4, 0xB5, 0xA4, 0xB6, 0xA4, 0xB7, /* 0x54-0x57 */
	0xA4, 0xB8, 0xA4, 0xB9, 0xA4, 0xBA, 0xA4, 0xBB, /* 0x58-0x5B */
	0xA4, 0xBC, 0xA4, 0xBD, 0xA4, 0xBE, 0xA4, 0xBF, /* 0x5C-0x5F */
	0xA4, 0xC0, 0xA4, 0xC1, 0xA4, 0xC2, 0xA4, 0xC3, /* 0x60-0x63 */
	0xA4, 0xC4, 0xA4, 0xC5, 0xA4, 0xC6, 0xA4, 0xC7, /* 0x64-0x67 */
	0xA4, 0xC8, 0xA4, 0xC9, 0xA4, 0xCA, 0xA4, 0xCB, /* 0x68-0x6B */
	0xA4, 0xCC, 0xA4, 0xCD, 0xA4, 0xCE, 0xA4, 0xCF, /* 0x6C-0x6F */
	0xA4, 0xD0, 0xA4, 0xD1, 0xA4, 0xD2, 0xA4, 0xD3, /* 0x70-0x73 */
	0xA4, 0xD4, 0xA4, 0xD5, 0xA4, 0xD6, 0xA4, 0xD7, /* 0x74-0x77 */
	0xA4, 0xD8, 0xA4, 0xD9, 0xA4, 0xDA, 0xA4, 0xDB, /* 0x78-0x7B */
	0xA4, 0xDC, 0xA4, 0xDD, 0xA4, 0xDE, 0xA4, 0xDF, /* 0x7C-0x7F */

	0xA4, 0xE0, 0xA4, 0xE1, 0xA4, 0xE2, 0xA4, 0xE3, /* 0x80-0x83 */
	0xA4, 0xE4, 0xA4, 0xE5, 0xA4, 0xE6, 0xA4, 0xE7, /* 0x84-0x87 */
	0xA4, 0xE8, 0xA4, 0xE9, 0xA4, 0xEA, 0xA4, 0xEB, /* 0x88-0x8B */
	0xA4, 0xEC, 0xA4, 0xED, 0xA4, 0xEE, 0xA4, 0xEF, /* 0x8C-0x8F */
	0xA4, 0xF0, 0xA4, 0xF1, 0xA4, 0xF2, 0xA4, 0xF3, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA1, 0xAB, /* 0x98-0x9B */
	0xA1, 0xAC, 0xA1, 0xB5, 0xA1, 0xB6, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0xA5, 0xA1, 0xA5, 0xA2, 0xA5, 0xA3, /* 0xA0-0xA3 */
	0xA5, 0xA4, 0xA5, 0xA5, 0xA5, 0xA6, 0xA5, 0xA7, /* 0xA4-0xA7 */
	0xA5, 0xA8, 0xA5, 0xA9, 0xA5, 0xAA, 0xA5, 0xAB, /* 0xA8-0xAB */
	0xA5, 0xAC, 0xA5, 0xAD, 0xA5, 0xAE, 0xA5, 0xAF, /* 0xAC-0xAF */
	0xA5, 0xB0, 0xA5, 0xB1, 0xA5, 0xB2, 0xA5, 0xB3, /* 0xB0-0xB3 */
	0xA5, 0xB4, 0xA5, 0xB5, 0xA5, 0xB6, 0xA5, 0xB7, /* 0xB4-0xB7 */
	0xA5, 0xB8, 0xA5, 0xB9, 0xA5, 0xBA, 0xA5, 0xBB, /* 0xB8-0xBB */
	0xA5, 0xBC, 0xA5, 0xBD, 0xA5, 0xBE, 0xA5, 0xBF, /* 0xBC-0xBF */
	0xA5, 0xC0, 0xA5, 0xC1, 0xA5, 0xC2, 0xA5, 0xC3, /* 0xC0-0xC3 */
	0xA5, 0xC4, 0xA5, 0xC5, 0xA5, 0xC6, 0xA5, 0xC7, /* 0xC4-0xC7 */
	0xA5, 0xC8, 0xA5, 0xC9, 0xA5, 0xCA, 0xA5, 0xCB, /* 0xC8-0xCB */
	0xA5, 0xCC, 0xA5, 0xCD, 0xA5, 0xCE, 0xA5, 0xCF, /* 0xCC-0xCF */
	0xA5, 0xD0, 0xA5, 0xD1, 0xA5, 0xD2, 0xA5, 0xD3, /* 0xD0-0xD3 */
	0xA5, 0xD4, 0xA5, 0xD5, 0xA5, 0xD6, 0xA5, 0xD7, /* 0xD4-0xD7 */
	0xA5, 0xD8, 0xA5, 0xD9, 0xA5, 0xDA, 0xA5, 0xDB, /* 0xD8-0xDB */
	0xA5, 0xDC, 0xA5, 0xDD, 0xA5, 0xDE, 0xA5, 0xDF, /* 0xDC-0xDF */
	0xA5, 0xE0, 0xA5, 0xE1, 0xA5, 0xE2, 0xA5, 0xE3, /* 0xE0-0xE3 */
	0xA5, 0xE4, 0xA5, 0xE5, 0xA5, 0xE6, 0xA5, 0xE7, /* 0xE4-0xE7 */
	0xA5, 0xE8, 0xA5, 0xE9, 0xA5, 0xEA, 0xA5, 0xEB, /* 0xE8-0xEB */
	0xA5, 0xEC, 0xA5, 0xED, 0xA5, 0xEE, 0xA5, 0xEF, /* 0xEC-0xEF */
	0xA5, 0xF0, 0xA5, 0xF1, 0xA5, 0xF2, 0xA5, 0xF3, /* 0xF0-0xF3 */
	0xA5, 0xF4, 0xA5, 0xF5, 0xA5, 0xF6, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA1, 0xA6, /* 0xF8-0xFB */
	0xA1, 0xBC, 0xA1, 0xB3, 0xA1, 0xB4, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_32[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0xAD, 0xEA, 0xAD, 0xEB, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xAD, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0xAD, 0xE5, 0xAD, 0xE6, 0xAD, 0xE7, 0xAD, 0xE8, /* 0xA4-0xA7 */
	0xAD, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
};

static unsigned char u2c_33[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xC6, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0xAD, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0xAD, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xAD, 0xC4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0xAD, 0xC2, 0xAD, 0xCC, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xAD, 0xCB, 0xAD, 0xC5, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xCD, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0xAD, 0xC7, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xCF, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0xAD, 0xC0, 0xAD, 0xCE, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0xAD, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0xAD, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xC9, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAD, 0xDF, /* 0x78-0x7B */
	0xAD, 0xEF, 0xAD, 0xEE, 0xAD, 0xED, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0xAD, 0xD3, 0xAD, 0xD4, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xAD, 0xD0, 0xAD, 0xD1, 0xAD, 0xD2, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0xAD, 0xD6, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0xAD, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0xAD, 0xE3, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
};

static unsigned char u2c_4E[512] = {
	0xB0, 0xEC, 0xC3, 0xFA, 0x30, 0xA1, 0xBC, 0xB7, /* 0x00-0x03 */
	0x30, 0xA2, 0x30, 0xA3, 0x00, 0x00, 0xCB, 0xFC, /* 0x04-0x07 */
	0xBE, 0xE6, 0xBB, 0xB0, 0xBE, 0xE5, 0xB2, 0xBC, /* 0x08-0x0B */
	0x30, 0xA4, 0xC9, 0xD4, 0xCD, 0xBF, 0x00, 0x00, /* 0x0C-0x0F */
	0xD0, 0xA2, 0xB1, 0xAF, 0x30, 0xA5, 0x00, 0x00, /* 0x10-0x13 */
	0xB3, 0xEE, 0xD0, 0xA3, 0xC0, 0xA4, 0xD2, 0xC2, /* 0x14-0x17 */
	0xB5, 0xD6, 0xCA, 0xBA, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0xBE, 0xE7, 0x30, 0xA6, /* 0x1C-0x1F */
	0x00, 0x00, 0xCE, 0xBE, 0x00, 0x00, 0x30, 0xA7, /* 0x20-0x23 */
	0x30, 0xA8, 0x00, 0x00, 0xCA, 0xC2, 0x00, 0x00, /* 0x24-0x27 */
	0xF9, 0xAD, 0x00, 0x00, 0xD0, 0xA4, 0x30, 0xAA, /* 0x28-0x2B */
	0x00, 0x00, 0xC3, 0xE6, 0x30, 0xAB, 0x30, 0xAC, /* 0x2C-0x2F */
	0x30, 0xAD, 0xD0, 0xA5, 0xB6, 0xFA, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x30, 0xAE, 0xD0, 0xA6, 0x00, 0x00, /* 0x34-0x37 */
	0xB4, 0xDD, 0xC3, 0xB0, 0x00, 0x00, 0xBC, 0xE7, /* 0x38-0x3B */
	0xD0, 0xA7, 0x00, 0x00, 0x00, 0x00, 0xD0, 0xA8, /* 0x3C-0x3F */
	0x30, 0xAF, 0x30, 0xB0, 0xD0, 0xA9, 0xC7, 0xB5, /* 0x40-0x43 */
	0x30, 0xB1, 0xB5, 0xD7, 0x00, 0x00, 0x30, 0xB2, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC7, 0xB7, /* 0x48-0x4B */
	0x00, 0x00, 0xC6, 0xE3, 0xB8, 0xC3, 0xCB, 0xB3, /* 0x4C-0x4F */
	0x00, 0x00, 0x30, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0xE9, 0xC9, 0xD0, 0xAA, 0xBE, 0xE8, /* 0x54-0x57 */
	0xD0, 0xAB, 0xB2, 0xB5, 0x30, 0xB4, 0x00, 0x00, /* 0x58-0x5B */
	0x30, 0xB5, 0xB6, 0xE5, 0xB8, 0xF0, 0xCC, 0xE9, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0xD6, 0xA6, 0x30, 0xB6, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x30, 0xB7, 0x30, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0xCD, 0xF0, 0x00, 0x00, 0xC6, 0xFD, /* 0x70-0x73 */
	0x30, 0xB9, 0x30, 0xBA, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x30, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0xB4, 0xA5, 0x30, 0xBC, /* 0x7C-0x7F */

	0xB5, 0xB5, 0x00, 0x00, 0xD0, 0xAC, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0xD0, 0xAD, 0xCE, 0xBB, 0x00, 0x00, /* 0x84-0x87 */
	0xCD, 0xBD, 0xC1, 0xE8, 0xD0, 0xAF, 0xBB, 0xF6, /* 0x88-0x8B */
	0xC6, 0xF3, 0x30, 0xBD, 0xD0, 0xB2, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0xB1, 0xBE, 0xB8, 0xDF, 0x00, 0x00, /* 0x90-0x93 */
	0xB8, 0xDE, 0xB0, 0xE6, 0x30, 0xBE, 0x30, 0xBF, /* 0x94-0x97 */
	0xCF, 0xCB, 0xCF, 0xCA, 0x00, 0x00, 0xBA, 0xB3, /* 0x98-0x9B */
	0xB0, 0xA1, 0x30, 0xC0, 0xD0, 0xB3, 0xD0, 0xB4, /* 0x9C-0x9F */
	0xD0, 0xB5, 0xCB, 0xB4, 0xD0, 0xB6, 0x00, 0x00, /* 0xA0-0xA3 */
	0xB8, 0xF2, 0xB0, 0xE7, 0xCB, 0xF2, 0x00, 0x00, /* 0xA4-0xA7 */
	0xB5, 0xFC, 0x00, 0x00, 0x00, 0x00, 0xB5, 0xFD, /* 0xA8-0xAB */
	0xB5, 0xFE, 0xC4, 0xE2, 0xCE, 0xBC, 0x30, 0xC1, /* 0xAC-0xAF */
	0xD0, 0xB7, 0x00, 0x00, 0x00, 0x00, 0xD0, 0xB8, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0xD0, 0xB9, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x30, 0xC2, 0xBF, 0xCD, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xBD, 0xBA, 0xBF, 0xCE, 0xD0, 0xBE, 0x30, 0xC3, /* 0xC0-0xC3 */
	0xD0, 0xBC, 0x00, 0x00, 0xD0, 0xBD, 0xB5, 0xD8, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xA3, 0xB2, 0xF0, /* 0xC8-0xCB */
	0x00, 0x00, 0xD0, 0xBB, 0xD0, 0xBA, 0xCA, 0xA9, /* 0xCC-0xCF */
	0x30, 0xC4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0xBB, 0xC6, 0xBB, 0xC5, 0xC2, 0xBE, 0xD0, 0xBF, /* 0xD4-0xD7 */
	0xC9, 0xD5, 0xC0, 0xE7, 0x30, 0xC5, 0x30, 0xC6, /* 0xD8-0xDB */
	0x00, 0x00, 0xA1, 0xB8, 0xD0, 0xC0, 0xD0, 0xC2, /* 0xDC-0xDF */
	0x30, 0xC7, 0xF9, 0xAE, 0x30, 0xC9, 0xC2, 0xE5, /* 0xE0-0xE3 */
	0xCE, 0xE1, 0xB0, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x30, 0xCA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xD0, 0xC1, 0xB2, 0xBE, 0x30, 0xCB, /* 0xEC-0xEF */
	0xB6, 0xC4, 0x30, 0xCC, 0xC3, 0xE7, 0x30, 0xCD, /* 0xF0-0xF3 */
	0x00, 0x00, 0x30, 0xCE, 0xB7, 0xEF, 0xD0, 0xC3, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC7, 0xA4, /* 0xF8-0xFB */
	0xF9, 0xAF, 0x30, 0xCF, 0x30, 0xD0, 0x30, 0xD1, /* 0xFC-0xFF */
};

static unsigned char u2c_4F[512] = {
	0xF9, 0xB0, 0xB4, 0xEB, 0x30, 0xD3, 0xF9, 0xB1, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x30, 0xD5, 0xD0, 0xC4, 0xB0, 0xCB, 0x30, 0xD6, /* 0x08-0x0B */
	0x30, 0xD7, 0xB8, 0xE0, 0xB4, 0xEC, 0xC9, 0xFA, /* 0x0C-0x0F */
	0xC8, 0xB2, 0xB5, 0xD9, 0x30, 0xD8, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x30, 0xD9, 0x30, 0xDA, 0x30, 0xDB, /* 0x14-0x17 */
	0x00, 0x00, 0x30, 0xDC, 0xB2, 0xF1, 0x00, 0x00, /* 0x18-0x1B */
	0xD0, 0xE7, 0xC5, 0xC1, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x30, 0xDD, 0xC7, 0xEC, /* 0x2C-0x2F */
	0xD0, 0xC6, 0x30, 0xDE, 0x00, 0x00, 0x30, 0xE0, /* 0x30-0x33 */
	0xC8, 0xBC, 0x30, 0xE1, 0xCE, 0xE2, 0x30, 0xE2, /* 0x34-0x37 */
	0xBF, 0xAD, 0xF9, 0xB2, 0xBB, 0xC7, 0x30, 0xE4, /* 0x38-0x3B */
	0xBB, 0xF7, 0xB2, 0xC0, 0x30, 0xE5, 0x00, 0x00, /* 0x3C-0x3F */
	0x30, 0xE6, 0x00, 0x00, 0x30, 0xE7, 0xC4, 0xD1, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0xC3, 0xA2, 0xD0, 0xCA, /* 0x44-0x47 */
	0x30, 0xE8, 0x30, 0xE9, 0x00, 0x00, 0x30, 0xEA, /* 0x48-0x4B */
	0x30, 0xEB, 0xB0, 0xCC, 0xC4, 0xE3, 0xBD, 0xBB, /* 0x4C-0x4F */
	0xBA, 0xB4, 0xCD, 0xA4, 0x30, 0xEC, 0xC2, 0xCE, /* 0x50-0x53 */
	0x30, 0xED, 0xB2, 0xBF, 0xF9, 0xB3, 0xD0, 0xC9, /* 0x54-0x57 */
	0x30, 0xEF, 0xCD, 0xBE, 0xD0, 0xC5, 0xD0, 0xC7, /* 0x58-0x5B */
	0xBA, 0xEE, 0xD0, 0xC8, 0xD5, 0xA4, 0x30, 0xF0, /* 0x5C-0x5F */
	0x30, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x30, 0xF1, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0xD0, 0xD0, 0x30, 0xF2, 0x00, 0x00, /* 0x68-0x6B */
	0x30, 0xF3, 0x00, 0x00, 0x30, 0xF4, 0xD0, 0xD3, /* 0x6C-0x6F */
	0xD0, 0xD1, 0x30, 0xF5, 0x00, 0x00, 0xB2, 0xC2, /* 0x70-0x73 */
	0x00, 0x00, 0xCA, 0xBB, 0xD0, 0xCB, 0x30, 0xF6, /* 0x74-0x77 */
	0x30, 0xF7, 0x30, 0xF8, 0x30, 0xF9, 0xD0, 0xCF, /* 0x78-0x7B */
	0xB8, 0xF3, 0x30, 0xFA, 0x30, 0xFB, 0xBB, 0xC8, /* 0x7C-0x7F */

	0x00, 0x00, 0x30, 0xFC, 0x30, 0xFD, 0xB4, 0xA6, /* 0x80-0x83 */
	0x30, 0xFE, 0x31, 0xA1, 0xD0, 0xD4, 0x00, 0x00, /* 0x84-0x87 */
	0xD0, 0xCC, 0x31, 0xA2, 0xF9, 0xB5, 0xCE, 0xE3, /* 0x88-0x8B */
	0x31, 0xA4, 0xBB, 0xF8, 0x31, 0xA5, 0xD0, 0xCD, /* 0x8C-0x8F */
	0x31, 0xA6, 0xD0, 0xD2, 0xF9, 0xB4, 0x31, 0xA8, /* 0x90-0x93 */
	0xF9, 0xB7, 0x00, 0x00, 0xD0, 0xD5, 0x31, 0xAA, /* 0x94-0x97 */
	0xD0, 0xCE, 0x31, 0xAB, 0xF9, 0xB6, 0xB6, 0xA1, /* 0x98-0x9B */
	0x00, 0x00, 0xB0, 0xCD, 0x31, 0xAD, 0x31, 0xAE, /* 0x9C-0x9F */
	0xB6, 0xA2, 0xB2, 0xC1, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD5, 0xA5, /* 0xA8-0xAB */
	0x00, 0x00, 0xCB, 0xF9, 0xC9, 0xEE, 0xB8, 0xF4, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x31, 0xAF, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0xBF, 0xAF, 0xCE, 0xB7, 0x31, 0xB0, /* 0xB4-0xB7 */
	0x00, 0x00, 0x31, 0xB1, 0x00, 0x00, 0x31, 0xB2, /* 0xB8-0xBB */
	0x31, 0xB3, 0x31, 0xB4, 0x31, 0xB5, 0xCA, 0xD8, /* 0xBC-0xBF */
	0x31, 0xB6, 0x31, 0xB7, 0xB7, 0xB8, 0xC2, 0xA5, /* 0xC0-0xC3 */
	0xB2, 0xE4, 0x31, 0xB8, 0x31, 0xB9, 0x00, 0x00, /* 0xC4-0xC7 */
	0x31, 0xBA, 0xF9, 0xA6, 0xBD, 0xD3, 0x31, 0xBC, /* 0xC8-0xCB */
	0x31, 0xBD, 0xF9, 0xB8, 0xD0, 0xD9, 0x31, 0xBF, /* 0xCC-0xCF */
	0xD0, 0xDE, 0xD0, 0xDC, 0x31, 0xC0, 0x00, 0x00, /* 0xD0-0xD3 */
	0xD0, 0xD7, 0x00, 0x00, 0x00, 0x00, 0xC2, 0xAF, /* 0xD4-0xD7 */
	0xD0, 0xDA, 0x00, 0x00, 0xD0, 0xDD, 0xD0, 0xDB, /* 0xD8-0xDB */
	0x31, 0xC1, 0xCA, 0xDD, 0x00, 0x00, 0xD0, 0xD8, /* 0xDC-0xDF */
	0x31, 0xC2, 0xBF, 0xAE, 0x31, 0xC3, 0xCB, 0xF3, /* 0xE0-0xE3 */
	0xD0, 0xDF, 0xD0, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0xBD, 0xA4, 0xD0, 0xED, /* 0xEC-0xEF */
	0x31, 0xC4, 0x00, 0x00, 0x31, 0xC5, 0xC7, 0xD0, /* 0xF0-0xF3 */
	0x00, 0x00, 0xC9, 0xB6, 0xD0, 0xE8, 0x00, 0x00, /* 0xF4-0xF7 */
	0xCA, 0xF0, 0x00, 0x00, 0xB2, 0xB6, 0x00, 0x00, /* 0xF8-0xFB */
	0x31, 0xC6, 0x31, 0xC7, 0xD0, 0xEC, 0xF9, 0xBB, /* 0xFC-0xFF */
};

static unsigned char u2c_50[512] = {
	0x31, 0xC9, 0x31, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x31, 0xCB, 0xD0, 0xE6, 0xD0, 0xEF, 0x31, 0xCC, /* 0x04-0x07 */
	0x00, 0x00, 0xC1, 0xD2, 0x31, 0xCD, 0xB8, 0xC4, /* 0x08-0x0B */
	0x31, 0xCE, 0xC7, 0xDC, 0x31, 0xCF, 0xE0, 0xC7, /* 0x0C-0x0F */
	0x31, 0xD0, 0xD0, 0xEE, 0xC5, 0xDD, 0x31, 0xD1, /* 0x10-0x13 */
	0xD0, 0xE3, 0x00, 0x00, 0xB8, 0xF6, 0x31, 0xD2, /* 0x14-0x17 */
	0x31, 0xD3, 0xB8, 0xF5, 0xD0, 0xE1, 0x31, 0xD4, /* 0x18-0x1B */
	0x31, 0xD5, 0x31, 0xD6, 0xF9, 0xBC, 0xBC, 0xDA, /* 0x1C-0x1F */
	0x00, 0x00, 0xD0, 0xE9, 0xF9, 0xBA, 0xCA, 0xEF, /* 0x20-0x23 */
	0xC3, 0xCD, 0xD0, 0xE5, 0xB7, 0xF1, 0x31, 0xD9, /* 0x24-0x27 */
	0xD0, 0xE2, 0xD0, 0xEA, 0xD0, 0xE4, 0xCE, 0xD1, /* 0x28-0x2B */
	0xD0, 0xEB, 0xCF, 0xC1, 0x31, 0xDA, 0x00, 0x00, /* 0x2C-0x2F */
	0x31, 0xDB, 0x00, 0x00, 0x31, 0xDC, 0x31, 0xDD, /* 0x30-0x33 */
	0x00, 0x00, 0x31, 0xDE, 0xB6, 0xE6, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xB7, 0xF0, 0x00, 0x00, 0x31, 0xF6, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xF9, 0xB9, 0x31, 0xE0, 0xF9, 0xBF, 0xD0, 0xF0, /* 0x40-0x43 */
	0x00, 0x00, 0x31, 0xE2, 0xF9, 0xBD, 0xD0, 0xF1, /* 0x44-0x47 */
	0xD0, 0xF5, 0xB0, 0xCE, 0x31, 0xE4, 0x00, 0x00, /* 0x48-0x4B */
	0x31, 0xE5, 0x00, 0x00, 0x31, 0xE6, 0xCA, 0xD0, /* 0x4C-0x4F */
	0xD0, 0xF4, 0x31, 0xE7, 0x31, 0xE8, 0x31, 0xE9, /* 0x50-0x53 */
	0x00, 0x00, 0xD0, 0xF3, 0xD0, 0xF7, 0x31, 0xEA, /* 0x54-0x57 */
	0x00, 0x00, 0x31, 0xEB, 0xD0, 0xF6, 0x00, 0x00, /* 0x58-0x5B */
	0xC4, 0xE4, 0x00, 0x00, 0x00, 0x00, 0x31, 0xEC, /* 0x5C-0x5F */
	0x31, 0xED, 0x00, 0x00, 0x31, 0xEE, 0x31, 0xEF, /* 0x60-0x63 */
	0x00, 0x00, 0xB7, 0xF2, 0x31, 0xF0, 0x31, 0xF1, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x31, 0xF2, 0x00, 0x00, /* 0x68-0x6B */
	0xD0, 0xF8, 0x31, 0xF3, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xF9, 0xBE, 0x31, 0xF5, 0xBC, 0xC5, 0x00, 0x00, /* 0x70-0x73 */
	0xC2, 0xA6, 0xC4, 0xE5, 0xB6, 0xF6, 0x00, 0x00, /* 0x74-0x77 */
	0xD0, 0xF9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0xB5, 0xB6, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0xD0, 0xFA, 0x31, 0xF7, 0x00, 0x00, 0x31, 0xF8, /* 0x80-0x83 */
	0x31, 0xF9, 0xD0, 0xFC, 0x31, 0xFA, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x31, 0xFB, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0xCB, 0xB5, 0x31, 0xFC, 0x31, 0xFD, /* 0x8C-0x8F */
	0x31, 0xFE, 0xB7, 0xE6, 0x32, 0xA1, 0x32, 0xA2, /* 0x90-0x93 */
	0xF9, 0xC0, 0x00, 0x00, 0x32, 0xA4, 0x00, 0x00, /* 0x94-0x97 */
	0xBB, 0xB1, 0xC8, 0xF7, 0xD0, 0xFB, 0x32, 0xA5, /* 0x98-0x9B */
	0x32, 0xA6, 0x00, 0x00, 0x32, 0xA7, 0x32, 0xA8, /* 0x9C-0x9F */
	0x32, 0xA9, 0x32, 0xAA, 0x32, 0xAB, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x32, 0xAC, 0x00, 0x00, /* 0xA8-0xAB */
	0xBA, 0xC5, 0xCD, 0xC3, 0x00, 0x00, 0x32, 0xAD, /* 0xAC-0xAF */
	0x32, 0xAE, 0x00, 0x00, 0xD0, 0xFE, 0xD1, 0xA3, /* 0xB0-0xB3 */
	0xD0, 0xFD, 0xBA, 0xC4, 0x00, 0x00, 0xBD, 0xFD, /* 0xB4-0xB7 */
	0x00, 0x00, 0x32, 0xAF, 0x32, 0xB0, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x32, 0xB1, 0xB7, 0xB9, 0x00, 0x00, /* 0xBC-0xBF */
	0x32, 0xB2, 0x00, 0x00, 0xD1, 0xA4, 0x32, 0xB3, /* 0xC0-0xC3 */
	0x32, 0xB4, 0xB6, 0xCF, 0x00, 0x00, 0x32, 0xB5, /* 0xC4-0xC7 */
	0x00, 0x00, 0xD1, 0xA1, 0xD1, 0xA2, 0x00, 0x00, /* 0xC8-0xCB */
	0x32, 0xB6, 0xC6, 0xAF, 0x32, 0xB7, 0xC1, 0xFC, /* 0xCC-0xCF */
	0x32, 0xB8, 0xB6, 0xA3, 0x00, 0x00, 0x32, 0xB9, /* 0xD0-0xD3 */
	0x32, 0xBA, 0xCB, 0xCD, 0xD1, 0xA5, 0x00, 0x00, /* 0xD4-0xD7 */
	0xF9, 0xC2, 0x00, 0x00, 0xCE, 0xBD, 0x00, 0x00, /* 0xD8-0xDB */
	0x32, 0xBC, 0x32, 0xBD, 0xD1, 0xA6, 0x32, 0xBE, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x32, 0xBF, 0xD1, 0xA9, /* 0xE0-0xE3 */
	0x32, 0xC0, 0xD1, 0xA7, 0x32, 0xC1, 0xC1, 0xCE, /* 0xE4-0xE7 */
	0x32, 0xC2, 0x32, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xD1, 0xA8, 0xD1, 0xAA, 0x32, 0xC4, /* 0xEC-0xEF */
	0x00, 0x00, 0x32, 0xC5, 0x32, 0xD1, 0x00, 0x00, /* 0xF0-0xF3 */
	0xF9, 0xC1, 0xD1, 0xAC, 0x32, 0xC6, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0xD1, 0xAB, 0x32, 0xC7, 0xCA, 0xC8, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0x32, 0xC8, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_51[512] = {
	0xB5, 0xB7, 0xD1, 0xAE, 0xD1, 0xAF, 0x32, 0xC9, /* 0x00-0x03 */
	0xB2, 0xAF, 0x00, 0x00, 0x32, 0xCA, 0x32, 0xCB, /* 0x04-0x07 */
	0x32, 0xCC, 0xD1, 0xAD, 0x00, 0x00, 0x32, 0xCD, /* 0x08-0x0B */
	0x32, 0xCE, 0x32, 0xCF, 0x32, 0xD0, 0x00, 0x00, /* 0x0C-0x0F */
	0x32, 0xD2, 0x00, 0x00, 0xBC, 0xF4, 0x00, 0x00, /* 0x10-0x13 */
	0xD1, 0xB2, 0xD1, 0xB1, 0xD1, 0xB0, 0x32, 0xD3, /* 0x14-0x17 */
	0xD0, 0xD6, 0x32, 0xD4, 0xD1, 0xB3, 0x32, 0xD5, /* 0x18-0x1B */
	0x32, 0xD6, 0x32, 0xD7, 0x32, 0xD8, 0xBD, 0xFE, /* 0x1C-0x1F */
	0x00, 0x00, 0xD1, 0xB4, 0x00, 0x00, 0x32, 0xD9, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0xDA, /* 0x24-0x27 */
	0x32, 0xDB, 0x00, 0x00, 0xCD, 0xA5, 0x00, 0x00, /* 0x28-0x2B */
	0x32, 0xDC, 0x32, 0xDD, 0x00, 0x00, 0x32, 0xDE, /* 0x2C-0x2F */
	0x00, 0x00, 0x32, 0xDF, 0xCC, 0xD9, 0x32, 0xE0, /* 0x30-0x33 */
	0x32, 0xE1, 0x32, 0xE2, 0x00, 0x00, 0xD1, 0xB6, /* 0x34-0x37 */
	0x32, 0xE3, 0x32, 0xE4, 0xD1, 0xB5, 0xD1, 0xB8, /* 0x38-0x3B */
	0xD1, 0xB7, 0x00, 0x00, 0x00, 0x00, 0xD1, 0xB9, /* 0x3C-0x3F */
	0xD1, 0xBA, 0xB0, 0xF4, 0x32, 0xE5, 0xB8, 0xB5, /* 0x40-0x43 */
	0xB7, 0xBB, 0xBD, 0xBC, 0xC3, 0xFB, 0xB6, 0xA4, /* 0x44-0x47 */
	0xC0, 0xE8, 0xB8, 0xF7, 0xF9, 0xC3, 0xB9, 0xEE, /* 0x48-0x4B */
	0xD1, 0xBC, 0xCC, 0xC8, 0xC5, 0xC6, 0x32, 0xE7, /* 0x4C-0x4F */
	0xBB, 0xF9, 0x00, 0x00, 0xD1, 0xBB, 0x32, 0xE8, /* 0x50-0x53 */
	0xD1, 0xBD, 0x32, 0xE9, 0x00, 0x00, 0x32, 0xEA, /* 0x54-0x57 */
	0x32, 0xEB, 0x00, 0x00, 0xC5, 0xDE, 0x00, 0x00, /* 0x58-0x5B */
	0xB3, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x32, 0xEC, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0xD1, 0xBE, 0x00, 0x00, /* 0x60-0x63 */
	0xF9, 0xC4, 0xC6, 0xFE, 0x32, 0xEE, 0x00, 0x00, /* 0x64-0x67 */
	0xC1, 0xB4, 0xD1, 0xC0, 0xD1, 0xC1, 0xC8, 0xAC, /* 0x68-0x6B */
	0xB8, 0xF8, 0xCF, 0xBB, 0xD1, 0xC2, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0xB6, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xCA, 0xBC, 0xC2, 0xB6, 0xB6, 0xF1, /* 0x74-0x77 */
	0xC5, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0xB7, 0xF3, 0x00, 0x00, 0x32, 0xEF, 0x00, 0x00, /* 0x7C-0x7F */

	0xD1, 0xC3, 0x00, 0x00, 0xD1, 0xC4, 0x32, 0xF0, /* 0x80-0x83 */
	0x32, 0xF1, 0xC6, 0xE2, 0xB1, 0xDF, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0xD1, 0xC7, 0xBA, 0xFD, 0x32, 0xF2, /* 0x88-0x8B */
	0xD1, 0xC6, 0xBA, 0xC6, 0x32, 0xF3, 0xD1, 0xC8, /* 0x8C-0x8F */
	0xE6, 0xEE, 0xD1, 0xC9, 0xCB, 0xC1, 0xD1, 0xCA, /* 0x90-0x93 */
	0x00, 0x00, 0xD1, 0xCB, 0xD1, 0xCC, 0xBE, 0xE9, /* 0x94-0x97 */
	0x32, 0xF4, 0xBC, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0xF9, 0xC5, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xB4, 0xA7, 0x32, 0xF6, 0xD1, 0xCF, 0x32, 0xF7, /* 0xA0-0xA3 */
	0xD1, 0xCD, 0xCC, 0xBD, 0xD1, 0xCE, 0x00, 0x00, /* 0xA4-0xA7 */
	0xC9, 0xDA, 0xD1, 0xD0, 0xD1, 0xD1, 0xD1, 0xD2, /* 0xA8-0xAB */
	0xC5, 0xDF, 0x32, 0xF8, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0xD1, 0xD6, 0xD1, 0xD4, 0xD1, 0xD5, 0xD1, 0xD3, /* 0xB0-0xB3 */
	0xBA, 0xE3, 0xD1, 0xD7, 0xCC, 0xEA, 0xCE, 0xE4, /* 0xB4-0xB7 */
	0x32, 0xF9, 0x00, 0x00, 0x32, 0xFA, 0x00, 0x00, /* 0xB8-0xBB */
	0x32, 0xFB, 0xD1, 0xD8, 0xF9, 0xC6, 0x32, 0xFD, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x32, 0xFE, 0x00, 0x00, /* 0xC0-0xC3 */
	0xC0, 0xA8, 0xD1, 0xD9, 0xBD, 0xDA, 0x00, 0x00, /* 0xC4-0xC7 */
	0x33, 0xA1, 0xD1, 0xDA, 0x00, 0x00, 0xC3, 0xFC, /* 0xC8-0xCB */
	0xCE, 0xBF, 0xC5, 0xE0, 0x00, 0x00, 0x33, 0xA2, /* 0xCC-0xCF */
	0x00, 0x00, 0x33, 0xA3, 0x33, 0xA4, 0x33, 0xA5, /* 0xD0-0xD3 */
	0x00, 0x00, 0x33, 0xA6, 0xD2, 0xC5, 0x00, 0x00, /* 0xD4-0xD7 */
	0x33, 0xA7, 0x00, 0x00, 0x00, 0x00, 0xD1, 0xDB, /* 0xD8-0xDB */
	0xF4, 0xA5, 0xB6, 0xC5, 0x33, 0xA8, 0x00, 0x00, /* 0xDC-0xDF */
	0xD1, 0xDC, 0xCB, 0xDE, 0x33, 0xA9, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x33, 0xAA, 0xBD, 0xE8, 0xC2, 0xFC, /* 0xE4-0xE7 */
	0x00, 0x00, 0xD1, 0xDE, 0xC6, 0xE4, 0x00, 0x00, /* 0xE8-0xEB */
	0xF9, 0xC7, 0xD1, 0xDF, 0x33, 0xAB, 0x00, 0x00, /* 0xEC-0xEF */
	0xD1, 0xE0, 0xB3, 0xAE, 0x33, 0xAC, 0x33, 0xAD, /* 0xF0-0xF3 */
	0x33, 0xAE, 0xD1, 0xE1, 0xB6, 0xA7, 0x33, 0xAF, /* 0xF4-0xF7 */
	0xC6, 0xCC, 0xB1, 0xFA, 0xBD, 0xD0, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xC8, 0xA1, 0xD1, 0xE2, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_52[512] = {
	0xC5, 0xE1, 0x33, 0xB0, 0x33, 0xB1, 0xBF, 0xCF, /* 0x00-0x03 */
	0xD1, 0xE3, 0x33, 0xB2, 0xCA, 0xAC, 0xC0, 0xDA, /* 0x04-0x07 */
	0xB4, 0xA2, 0x00, 0x00, 0xB4, 0xA9, 0xD1, 0xE4, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0xD1, 0xE6, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xB7, 0xBA, 0x33, 0xB3, 0x33, 0xB4, /* 0x10-0x13 */
	0xD1, 0xE5, 0xF9, 0xC8, 0x33, 0xB6, 0xCE, 0xF3, /* 0x14-0x17 */
	0x33, 0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0xBD, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x33, 0xB8, 0x00, 0x00, /* 0x20-0x23 */
	0xC8, 0xBD, 0xCA, 0xCC, 0x00, 0x00, 0xD1, 0xE7, /* 0x24-0x27 */
	0x33, 0xB9, 0xCD, 0xF8, 0xD1, 0xE8, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0xD1, 0xE9, 0x00, 0x00, /* 0x2C-0x2F */
	0xC5, 0xFE, 0x33, 0xBA, 0x33, 0xBB, 0xD1, 0xEA, /* 0x30-0x33 */
	0x00, 0x00, 0x33, 0xBC, 0xC0, 0xA9, 0xBA, 0xFE, /* 0x34-0x37 */
	0xB7, 0xF4, 0xD1, 0xEB, 0xBB, 0xC9, 0xB9, 0xEF, /* 0x38-0x3B */
	0x33, 0xBD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC4, 0xE6, /* 0x40-0x43 */
	0xD1, 0xED, 0x33, 0xBE, 0x00, 0x00, 0xC2, 0xA7, /* 0x44-0x47 */
	0x00, 0x00, 0x33, 0xBF, 0xBA, 0xEF, 0xD1, 0xEE, /* 0x48-0x4B */
	0xD1, 0xEF, 0xC1, 0xB0, 0x00, 0x00, 0xD1, 0xEC, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0xD1, 0xF1, 0x33, 0xC0, 0xCB, 0xB6, 0x33, 0xC1, /* 0x54-0x57 */
	0x33, 0xC2, 0x00, 0x00, 0x33, 0xC3, 0xB9, 0xE4, /* 0x58-0x5B */
	0x33, 0xC4, 0x00, 0x00, 0xD1, 0xF0, 0x33, 0xC5, /* 0x5C-0x5F */
	0x33, 0xC6, 0x33, 0xC7, 0x00, 0x00, 0xB7, 0xF5, /* 0x60-0x63 */
	0xBA, 0xDE, 0xC7, 0xED, 0x33, 0xC8, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0xD1, 0xF4, 0xD1, 0xF2, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x33, 0xC9, 0xC9, 0xFB, /* 0x6C-0x6F */
	0xBE, 0xEA, 0xD1, 0xFB, 0xB3, 0xE4, 0xD1, 0xF5, /* 0x70-0x73 */
	0xD1, 0xF3, 0xC1, 0xCF, 0x00, 0x00, 0x33, 0xCA, /* 0x74-0x77 */
	0x33, 0xCB, 0x33, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0xD1, 0xF7, 0x00, 0x00, 0xD1, 0xF6, /* 0x7C-0x7F */

	0x33, 0xCD, 0x00, 0x00, 0x33, 0xCE, 0xB3, 0xC4, /* 0x80-0x83 */
	0x00, 0x00, 0x33, 0xCF, 0x00, 0x00, 0xB7, 0xE0, /* 0x84-0x87 */
	0xD1, 0xFC, 0xCE, 0xAD, 0x33, 0xD0, 0x00, 0x00, /* 0x88-0x8B */
	0x33, 0xD1, 0xD1, 0xF8, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0xD1, 0xFD, 0xD1, 0xFA, 0x33, 0xD2, /* 0x90-0x93 */
	0xD1, 0xF9, 0x33, 0xD3, 0x33, 0xD4, 0x33, 0xD5, /* 0x94-0x97 */
	0x33, 0xD6, 0x00, 0x00, 0x33, 0xD7, 0xCE, 0xCF, /* 0x98-0x9B */
	0xF9, 0xC9, 0x00, 0x00, 0x00, 0x00, 0xB8, 0xF9, /* 0x9C-0x9F */
	0xB2, 0xC3, 0x00, 0x00, 0x00, 0x00, 0xCE, 0xF4, /* 0xA0-0xA3 */
	0x33, 0xD9, 0x33, 0xDA, 0xF9, 0xCA, 0x33, 0xDC, /* 0xA4-0xA7 */
	0x00, 0x00, 0xBD, 0xF5, 0xC5, 0xD8, 0xB9, 0xE5, /* 0xA8-0xAB */
	0xD2, 0xA2, 0xD2, 0xA3, 0x00, 0x00, 0xFB, 0xBC, /* 0xAC-0xAF */
	0x33, 0xDE, 0xCE, 0xE5, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0xCF, 0xAB, 0xD2, 0xA5, 0x33, 0xDF, 0x33, 0xE0, /* 0xB4-0xB7 */
	0x33, 0xE1, 0xB8, 0xFA, 0x33, 0xE2, 0x33, 0xE3, /* 0xB8-0xBB */
	0xD2, 0xA4, 0x33, 0xE4, 0xB3, 0xAF, 0x00, 0x00, /* 0xBC-0xBF */
	0xF9, 0xCB, 0xD2, 0xA6, 0x00, 0x00, 0xCB, 0xD6, /* 0xC0-0xC3 */
	0x33, 0xE6, 0xC4, 0xBC, 0x33, 0xE7, 0xCD, 0xA6, /* 0xC4-0xC7 */
	0x33, 0xE8, 0xCA, 0xD9, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x33, 0xE9, 0xD2, 0xA7, 0x00, 0x00, 0x33, 0xEA, /* 0xCC-0xCF */
	0x00, 0x00, 0x33, 0xEB, 0xF0, 0xD5, 0x00, 0x00, /* 0xD0-0xD3 */
	0x33, 0xEC, 0xC6, 0xB0, 0x33, 0xED, 0xD2, 0xA8, /* 0xD4-0xD7 */
	0xB4, 0xAA, 0xCC, 0xB3, 0x00, 0x00, 0xF9, 0xCC, /* 0xD8-0xDB */
	0x33, 0xEF, 0xBE, 0xA1, 0xD2, 0xA9, 0xCA, 0xE7, /* 0xDC-0xDF */
	0xD2, 0xAD, 0x33, 0xF0, 0xC0, 0xAA, 0xD2, 0xAA, /* 0xE0-0xE3 */
	0xB6, 0xD0, 0x33, 0xF1, 0xD2, 0xAB, 0xB4, 0xAB, /* 0xE4-0xE7 */
	0x33, 0xF2, 0x33, 0xF3, 0x33, 0xF4, 0x00, 0x00, /* 0xE8-0xEB */
	0x33, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x33, 0xF6, 0x33, 0xF7, 0xB7, 0xAE, 0xD2, 0xAE, /* 0xF0-0xF3 */
	0x33, 0xF8, 0xD2, 0xAF, 0x33, 0xF9, 0x33, 0xFA, /* 0xF4-0xF7 */
	0xD2, 0xB0, 0xD2, 0xB1, 0xBC, 0xDB, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0xB8, 0xFB, 0xCC, 0xDE, /* 0xFC-0xFF */
};

static unsigned char u2c_53[512] = {
	0xF9, 0xCD, 0xCC, 0xE8, 0xC6, 0xF7, 0x33, 0xFC, /* 0x00-0x03 */
	0x00, 0x00, 0xCA, 0xF1, 0xD2, 0xB2, 0xF9, 0xCE, /* 0x04-0x07 */
	0xD2, 0xB3, 0x00, 0x00, 0x33, 0xFD, 0x33, 0xFE, /* 0x08-0x0B */
	0x34, 0xA1, 0xD2, 0xB5, 0x00, 0x00, 0xD2, 0xB7, /* 0x0C-0x0F */
	0xD2, 0xB6, 0x34, 0xA2, 0x00, 0x00, 0x34, 0xA3, /* 0x10-0x13 */
	0x00, 0x00, 0xD2, 0xB8, 0xB2, 0xBD, 0xCB, 0xCC, /* 0x14-0x17 */
	0x34, 0xA4, 0xBA, 0xFC, 0xD2, 0xB9, 0x34, 0xA5, /* 0x18-0x1B */
	0x34, 0xA6, 0xC1, 0xD9, 0x34, 0xA7, 0x34, 0xA8, /* 0x1C-0x1F */
	0xBE, 0xA2, 0xB6, 0xA9, 0x00, 0x00, 0xD2, 0xBA, /* 0x20-0x23 */
	0xF9, 0xCF, 0x34, 0xA9, 0x00, 0x00, 0x34, 0xAA, /* 0x24-0x27 */
	0x34, 0xAB, 0x34, 0xAC, 0xC8, 0xDB, 0x34, 0xAD, /* 0x28-0x2B */
	0x34, 0xAE, 0x34, 0xAF, 0x00, 0x00, 0xD2, 0xBB, /* 0x2C-0x2F */
	0x34, 0xB0, 0xD2, 0xBC, 0x34, 0xB1, 0xD2, 0xBD, /* 0x30-0x33 */
	0x00, 0x00, 0x34, 0xB2, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0xD2, 0xBE, 0xC9, 0xA4, 0xB6, 0xE8, 0xB0, 0xE5, /* 0x38-0x3B */
	0x34, 0xB3, 0x34, 0xB4, 0x34, 0xB5, 0xC6, 0xBF, /* 0x3C-0x3F */
	0xD2, 0xBF, 0xBD, 0xBD, 0x34, 0xB6, 0xC0, 0xE9, /* 0x40-0x43 */
	0x00, 0x00, 0xD2, 0xC1, 0xD2, 0xC0, 0xBE, 0xA3, /* 0x44-0x47 */
	0xB8, 0xE1, 0xD2, 0xC3, 0xC8, 0xBE, 0x34, 0xB8, /* 0x48-0x4B */
	0x34, 0xB7, 0xD2, 0xC4, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0xC8, 0xDC, 0xC2, 0xB4, 0xC2, 0xEE, /* 0x50-0x53 */
	0xB6, 0xA8, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xEE, /* 0x54-0x57 */
	0xC3, 0xB1, 0x34, 0xB9, 0xC7, 0xEE, 0x34, 0xBA, /* 0x58-0x5B */
	0xCB, 0xCE, 0x00, 0x00, 0xD2, 0xC6, 0x00, 0x00, /* 0x5C-0x5F */
	0xC0, 0xEA, 0x34, 0xBB, 0x00, 0x00, 0x34, 0xBC, /* 0x60-0x63 */
	0x00, 0x00, 0x34, 0xBD, 0xB7, 0xB5, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0xD2, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x34, 0xBE, 0x34, 0xBF, 0xD2, 0xC8, 0xB1, 0xAC, /* 0x6C-0x6F */
	0xB0, 0xF5, 0xB4, 0xED, 0xF9, 0xD0, 0xC2, 0xA8, /* 0x70-0x73 */
	0xB5, 0xD1, 0xCD, 0xF1, 0x00, 0x00, 0xD2, 0xCB, /* 0x74-0x77 */
	0xB2, 0xB7, 0x34, 0xC1, 0x00, 0x00, 0xD2, 0xCA, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xC2, 0xB6, 0xAA, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0xD2, 0xCC, 0x34, 0xC3, /* 0x80-0x83 */
	0xCC, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x34, 0xC4, /* 0x84-0x87 */
	0x34, 0xC5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xC6, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF9, 0xD1, /* 0x90-0x93 */
	0x34, 0xC8, 0x00, 0x00, 0xD2, 0xCD, 0x00, 0x00, /* 0x94-0x97 */
	0xCE, 0xD2, 0x34, 0xC9, 0xB8, 0xFC, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x34, 0xCA, 0x00, 0x00, 0xB8, 0xB6, /* 0x9C-0x9F */
	0xD2, 0xCE, 0x34, 0xCB, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x34, 0xCC, 0xD2, 0xD0, 0xD2, 0xCF, 0x00, 0x00, /* 0xA4-0xA7 */
	0xBF, 0xDF, 0xB1, 0xB9, 0x34, 0xCD, 0x34, 0xCE, /* 0xA8-0xAB */
	0x00, 0x00, 0xB1, 0xDE, 0xD2, 0xD1, 0x34, 0xCF, /* 0xAC-0xAF */
	0xD2, 0xD2, 0x00, 0x00, 0xF9, 0xD2, 0xB8, 0xB7, /* 0xB0-0xB3 */
	0x34, 0xD1, 0x34, 0xD2, 0xD2, 0xD3, 0x34, 0xD3, /* 0xB4-0xB7 */
	0x34, 0xD4, 0x00, 0x00, 0x34, 0xD5, 0xB5, 0xEE, /* 0xB8-0xBB */
	0x00, 0x00, 0x34, 0xD6, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x34, 0xD7, 0x00, 0x00, 0xBB, 0xB2, 0xD2, 0xD4, /* 0xC0-0xC3 */
	0x00, 0x00, 0x34, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0xCB, 0xF4, 0xBA, 0xB5, 0xB5, 0xDA, 0xCD, 0xA7, /* 0xC8-0xCB */
	0xC1, 0xD0, 0xC8, 0xBF, 0xBC, 0xFD, 0x34, 0xD9, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xDA, 0x34, 0xDB, /* 0xD0-0xD3 */
	0xBD, 0xC7, 0x34, 0xDC, 0xBC, 0xE8, 0xBC, 0xF5, /* 0xD4-0xD7 */
	0x00, 0x00, 0xBD, 0xF6, 0x34, 0xDD, 0xC8, 0xC0, /* 0xD8-0xDB */
	0x00, 0x00, 0xF9, 0xD3, 0x34, 0xDF, 0xD2, 0xD7, /* 0xDC-0xDF */
	0x34, 0xE0, 0xB1, 0xC3, 0xC1, 0xD1, 0xB8, 0xFD, /* 0xE0-0xE3 */
	0xB8, 0xC5, 0xB6, 0xE7, 0x34, 0xE1, 0x34, 0xE2, /* 0xE4-0xE7 */
	0xD2, 0xDB, 0xC3, 0xA1, 0xC2, 0xFE, 0xB6, 0xAB, /* 0xE8-0xEB */
	0xBE, 0xA4, 0xD2, 0xDC, 0xD2, 0xDA, 0xB2, 0xC4, /* 0xEC-0xEF */
	0xC2, 0xE6, 0xBC, 0xB8, 0xBB, 0xCB, 0xB1, 0xA6, /* 0xF0-0xF3 */
	0x00, 0x00, 0x34, 0xE3, 0xB3, 0xF0, 0xB9, 0xE6, /* 0xF4-0xF7 */
	0xBB, 0xCA, 0x00, 0x00, 0xD2, 0xDD, 0x00, 0x00, /* 0xF8-0xFB */
};

static unsigned char u2c_54[512] = {
	0x00, 0x00, 0xD2, 0xDE, 0x34, 0xE4, 0xB5, 0xC9, /* 0x00-0x03 */
	0xB3, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0xB9, 0xE7, 0xB5, 0xC8, 0xC4, 0xDF, 0xB1, 0xA5, /* 0x08-0x0B */
	0xC6, 0xB1, 0xCC, 0xBE, 0xB9, 0xA1, 0xCD, 0xF9, /* 0x0C-0x0F */
	0xC5, 0xC7, 0xB8, 0xFE, 0x00, 0x00, 0x34, 0xE5, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xE6, 0xB7, 0xAF, /* 0x18-0x1B */
	0x00, 0x00, 0xD2, 0xE7, 0x00, 0x00, 0xB6, 0xE3, /* 0x1C-0x1F */
	0xCB, 0xCA, 0x34, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xC8, 0xDD, 0x34, 0xE8, /* 0x24-0x27 */
	0x34, 0xE9, 0xD2, 0xE6, 0x34, 0xEA, 0xB4, 0xDE, /* 0x28-0x2B */
	0xD2, 0xE1, 0xD2, 0xE2, 0xD2, 0xE4, 0x34, 0xEB, /* 0x2C-0x2F */
	0x00, 0x00, 0x34, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x34, 0xED, 0x34, 0xEE, 0xD2, 0xE5, 0x00, 0x00, /* 0x34-0x37 */
	0xB5, 0xDB, 0xBF, 0xE1, 0x00, 0x00, 0xCA, 0xAD, /* 0x38-0x3B */
	0xD2, 0xE3, 0xD2, 0xDF, 0xB8, 0xE3, 0x00, 0x00, /* 0x3C-0x3F */
	0xD2, 0xE0, 0x00, 0x00, 0xCF, 0xA4, 0x34, 0xEF, /* 0x40-0x43 */
	0x34, 0xF0, 0x00, 0x00, 0xCA, 0xF2, 0x34, 0xF1, /* 0x44-0x47 */
	0xC4, 0xE8, 0xB8, 0xE2, 0xB9, 0xF0, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x34, 0xF2, 0xD2, 0xE8, 0x34, 0xF3, /* 0x4C-0x4F */
	0x00, 0x00, 0xC6, 0xDD, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xF4, 0xD2, 0xEC, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x34, 0xF5, 0x00, 0x00, /* 0x60-0x63 */
	0x34, 0xF6, 0x00, 0x00, 0x34, 0xF7, 0x34, 0xF8, /* 0x64-0x67 */
	0xBC, 0xFE, 0x34, 0xF9, 0xBC, 0xF6, 0x34, 0xFA, /* 0x68-0x6B */
	0x00, 0x00, 0x34, 0xFB, 0x34, 0xFC, 0x00, 0x00, /* 0x6C-0x6F */
	0xD2, 0xEF, 0xD2, 0xED, 0x00, 0x00, 0xCC, 0xA3, /* 0x70-0x73 */
	0x34, 0xFD, 0xD2, 0xEA, 0xD2, 0xF3, 0xD2, 0xEE, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD2, 0xF1, /* 0x78-0x7B */
	0xB8, 0xC6, 0xCC, 0xBF, 0x00, 0x00, 0x34, 0xFE, /* 0x7C-0x7F */

	0xD2, 0xF2, 0x35, 0xA1, 0x00, 0x00, 0x35, 0xA2, /* 0x80-0x83 */
	0xD2, 0xF4, 0x35, 0xA3, 0xD2, 0xF6, 0x00, 0x00, /* 0x84-0x87 */
	0x35, 0xA4, 0x35, 0xA5, 0xF9, 0xD6, 0xBA, 0xF0, /* 0x88-0x8B */
	0xCF, 0xC2, 0x35, 0xA6, 0xD2, 0xEB, 0xD2, 0xE9, /* 0x8C-0x8F */
	0xD2, 0xF5, 0x35, 0xA7, 0xD2, 0xF0, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x35, 0xA8, 0x35, 0xA9, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xF9, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x35, 0xAB, /* 0x9C-0x9F */
	0x00, 0x00, 0x35, 0xAC, 0xD2, 0xF8, 0x00, 0x00, /* 0xA0-0xA3 */
	0xD3, 0xA3, 0xD2, 0xFA, 0x35, 0xAD, 0x35, 0xAE, /* 0xA4-0xA7 */
	0xD2, 0xFE, 0xF9, 0xD7, 0x35, 0xB0, 0xD3, 0xA1, /* 0xA8-0xAB */
	0xD2, 0xFB, 0x35, 0xB1, 0x35, 0xB2, 0xD3, 0xBE, /* 0xAC-0xAF */
	0x00, 0x00, 0x35, 0xB3, 0xBA, 0xE9, 0xB3, 0xB1, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0xB4, /* 0xB4-0xB7 */
	0xD2, 0xF9, 0x35, 0xB5, 0x35, 0xB6, 0x35, 0xB7, /* 0xB8-0xBB */
	0xD3, 0xA5, 0xB0, 0xF6, 0xD3, 0xA4, 0x35, 0xB8, /* 0xBC-0xBF */
	0xB0, 0xA5, 0xC9, 0xCA, 0xD3, 0xA2, 0x00, 0x00, /* 0xC0-0xC3 */
	0xD2, 0xFC, 0x00, 0x00, 0x35, 0xB9, 0xD2, 0xF7, /* 0xC4-0xC7 */
	0xD2, 0xFD, 0xBA, 0xC8, 0x35, 0xBA, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x35, 0xBB, 0x35, 0xBC, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xD3, 0xA6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x35, 0xBD, 0xB0, 0xF7, 0xD3, 0xAF, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0xD3, 0xA7, 0xD3, 0xA8, 0x00, 0x00, /* 0xE4-0xE7 */
	0xBE, 0xA5, 0xCB, 0xE9, 0x35, 0xBE, 0x00, 0x00, /* 0xE8-0xEB */
	0x35, 0xBF, 0xD3, 0xAD, 0xD3, 0xAC, 0x35, 0xC0, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0xC5, 0xAF, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x35, 0xC1, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0xD3, 0xAE, 0x00, 0x00, /* 0xF8-0xFB */
	0x35, 0xC2, 0xD3, 0xAB, 0x35, 0xC3, 0xF9, 0xD8, /* 0xFC-0xFF */
};

static unsigned char u2c_55[512] = {
	0x35, 0xC5, 0x35, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0xB1, 0xB4, 0x35, 0xC7, 0xBA, 0xB6, 0xBF, 0xB0, /* 0x04-0x07 */
	0x35, 0xC8, 0x35, 0xC9, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x35, 0xCA, 0x35, 0xCB, 0x35, 0xCC, 0xD3, 0xA9, /* 0x0C-0x0F */
	0xC5, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0xD3, 0xAA, 0x35, 0xCD, 0xB0, 0xA2, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x35, 0xCE, 0x35, 0xCF, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0xD3, 0xB4, 0xCD, 0xA3, /* 0x2C-0x2F */
	0x00, 0x00, 0xBE, 0xA7, 0x35, 0xD0, 0xD3, 0xBA, /* 0x30-0x33 */
	0x00, 0x00, 0x35, 0xD1, 0x35, 0xD2, 0x00, 0x00, /* 0x34-0x37 */
	0xD3, 0xB9, 0xD3, 0xB0, 0x00, 0x00, 0x35, 0xD3, /* 0x38-0x3B */
	0x35, 0xD4, 0x35, 0xD5, 0xC2, 0xC3, 0x00, 0x00, /* 0x3C-0x3F */
	0xD3, 0xB1, 0x35, 0xD6, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0xC2, 0xEF, 0xD3, 0xB6, 0xBE, 0xA6, 0x35, 0xD7, /* 0x44-0x47 */
	0x00, 0x00, 0x35, 0xD8, 0x35, 0xD9, 0x00, 0x00, /* 0x48-0x4B */
	0xD3, 0xB3, 0x35, 0xDA, 0x00, 0x00, 0xCC, 0xE4, /* 0x4C-0x4F */
	0x35, 0xDB, 0x35, 0xDC, 0x00, 0x00, 0xB7, 0xBC, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0xD3, 0xB7, 0xD3, 0xB8, /* 0x54-0x57 */
	0x35, 0xDD, 0x00, 0x00, 0x35, 0xDE, 0x35, 0xDF, /* 0x58-0x5B */
	0xD3, 0xB5, 0xD3, 0xBB, 0x35, 0xE0, 0x00, 0x00, /* 0x5C-0x5F */
	0x35, 0xE1, 0x35, 0xE2, 0x00, 0x00, 0xD3, 0xB2, /* 0x60-0x63 */
	0x35, 0xE3, 0x00, 0x00, 0x35, 0xE4, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD3, 0xC1, /* 0x78-0x7B */
	0xD3, 0xC6, 0x00, 0x00, 0xD3, 0xC2, 0x35, 0xE5, /* 0x7C-0x7F */

	0xD3, 0xBD, 0x35, 0xE6, 0x35, 0xE7, 0xD3, 0xC7, /* 0x80-0x83 */
	0xC1, 0xB1, 0x00, 0x00, 0xF9, 0xD9, 0xD3, 0xC9, /* 0x84-0x87 */
	0x35, 0xE9, 0xB9, 0xA2, 0xD3, 0xBF, 0xC3, 0xFD, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x35, 0xEA, 0x35, 0xEB, /* 0x8C-0x8F */
	0x00, 0x00, 0x35, 0xEC, 0x35, 0xED, 0x35, 0xEE, /* 0x90-0x93 */
	0x35, 0xEF, 0x00, 0x00, 0x00, 0x00, 0x35, 0xF0, /* 0x94-0x97 */
	0xD3, 0xC3, 0xD3, 0xBC, 0xB4, 0xAD, 0x00, 0x00, /* 0x98-0x9B */
	0xB4, 0xEE, 0xB3, 0xE5, 0xD3, 0xC4, 0xD3, 0xC0, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0xF1, /* 0xA0-0xA3 */
	0x35, 0xF2, 0x00, 0x00, 0x00, 0x00, 0xB7, 0xF6, /* 0xA4-0xA7 */
	0xD3, 0xCA, 0xD3, 0xC8, 0xC1, 0xD3, 0xB5, 0xCA, /* 0xA8-0xAB */
	0xB6, 0xAC, 0x35, 0xF3, 0xD3, 0xC5, 0x00, 0x00, /* 0xAC-0xAF */
	0xB6, 0xF4, 0x00, 0x00, 0x35, 0xF4, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0xB1, 0xC4, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0xF5, /* 0xBC-0xBF */
	0x00, 0x00, 0x35, 0xF6, 0x00, 0x00, 0x35, 0xF7, /* 0xC0-0xC3 */
	0xD3, 0xCE, 0xD3, 0xCC, 0x35, 0xF8, 0xD4, 0xA7, /* 0xC4-0xC7 */
	0x00, 0x00, 0x35, 0xF9, 0x00, 0x00, 0x35, 0xFA, /* 0xC8-0xCB */
	0x35, 0xFB, 0x00, 0x00, 0x35, 0xFC, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x35, 0xFD, 0x35, 0xFE, 0x36, 0xA1, /* 0xD0-0xD3 */
	0xD3, 0xD1, 0x00, 0x00, 0x00, 0x00, 0x36, 0xA2, /* 0xD4-0xD7 */
	0x36, 0xA3, 0x00, 0x00, 0xD3, 0xCB, 0x36, 0xA4, /* 0xD8-0xDB */
	0xD3, 0xCF, 0x00, 0x00, 0x36, 0xA5, 0xD3, 0xCD, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x36, 0xA6, 0xBB, 0xCC, /* 0xE0-0xE3 */
	0xD3, 0xD0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x36, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x36, 0xA8, 0xD3, 0xD3, /* 0xF4-0xF7 */
	0x00, 0x00, 0xD3, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xD3, 0xD6, 0xD3, 0xD5, 0x36, 0xA9, /* 0xFC-0xFF */
};

static unsigned char u2c_56[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x36, 0xAA, 0xC3, 0xB2, 0x00, 0x00, /* 0x04-0x07 */
	0x36, 0xAB, 0xB2, 0xC5, 0x36, 0xAC, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x36, 0xAD, 0x36, 0xAE, 0x36, 0xAF, /* 0x0C-0x0F */
	0x36, 0xB0, 0x36, 0xB1, 0x36, 0xB2, 0x00, 0x00, /* 0x10-0x13 */
	0xD3, 0xD2, 0x00, 0x00, 0xD3, 0xD4, 0xBE, 0xA8, /* 0x14-0x17 */
	0xB1, 0xB3, 0x36, 0xB3, 0x00, 0x00, 0xD3, 0xD7, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0xB2, 0xDE, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x36, 0xB4, 0x00, 0x00, 0x00, 0x00, 0xD3, 0xE2, /* 0x2C-0x2F */
	0x36, 0xB5, 0xBE, 0xFC, 0xD3, 0xDE, 0x36, 0xB6, /* 0x30-0x33 */
	0xD3, 0xDC, 0x36, 0xB7, 0xD3, 0xDD, 0x36, 0xB8, /* 0x34-0x37 */
	0xD3, 0xDF, 0x36, 0xB9, 0x00, 0x00, 0x36, 0xBA, /* 0x38-0x3B */
	0x36, 0xBB, 0x36, 0xBC, 0x00, 0x00, 0x36, 0xBD, /* 0x3C-0x3F */
	0x36, 0xBE, 0x36, 0xBF, 0xB1, 0xBD, 0x36, 0xC0, /* 0x40-0x43 */
	0x36, 0xC1, 0x00, 0x00, 0x36, 0xC2, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x36, 0xC3, 0x00, 0x00, 0x36, 0xC4, /* 0x48-0x4B */
	0xC1, 0xB9, 0x36, 0xC5, 0xD3, 0xD9, 0x36, 0xC6, /* 0x4C-0x4F */
	0xD3, 0xDA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x36, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB3, 0xFA, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x36, 0xC8, 0x00, 0x00, /* 0x5C-0x5F */
	0x36, 0xC9, 0x36, 0xCA, 0x36, 0xCB, 0x36, 0xCC, /* 0x60-0x63 */
	0xD3, 0xE1, 0x00, 0x00, 0x36, 0xCD, 0x00, 0x00, /* 0x64-0x67 */
	0xB4, 0xEF, 0x36, 0xCE, 0xD3, 0xE4, 0xD3, 0xE0, /* 0x68-0x6B */
	0xD3, 0xE3, 0x36, 0xCF, 0x00, 0x00, 0x36, 0xD0, /* 0x6C-0x6F */
	0x00, 0x00, 0x36, 0xD1, 0x36, 0xD2, 0x00, 0x00, /* 0x70-0x73 */
	0xCA, 0xAE, 0x36, 0xD3, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xC6, 0xD5, 0x00, 0x00, 0xC8, 0xB8, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0xD3, 0xE6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x36, 0xD4, 0x36, 0xD5, 0xD3, 0xE5, 0xB3, 0xC5, /* 0x84-0x87 */
	0x36, 0xD6, 0x00, 0x00, 0xD3, 0xE7, 0x36, 0xD7, /* 0x88-0x8B */
	0x36, 0xD8, 0x00, 0x00, 0x00, 0x00, 0xD3, 0xEA, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0xD3, 0xE9, 0x36, 0xD9, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x36, 0xDA, 0x36, 0xDB, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x36, 0xDC, 0x36, 0xDD, 0x36, 0xDE, /* 0x9C-0x9F */
	0xD3, 0xE8, 0x00, 0x00, 0xC7, 0xB9, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xD3, 0xEB, 0x36, 0xDF, 0x36, 0xE0, /* 0xA4-0xA7 */
	0x36, 0xE1, 0x36, 0xE2, 0x00, 0x00, 0x36, 0xE3, /* 0xA8-0xAB */
	0x36, 0xE4, 0x36, 0xE5, 0xD3, 0xEC, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x36, 0xE6, 0x00, 0x00, 0x36, 0xE7, /* 0xB0-0xB3 */
	0xD3, 0xEE, 0x00, 0x00, 0xD3, 0xED, 0x36, 0xE8, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0xD3, 0xF0, 0x00, 0x00, 0x36, 0xE9, 0x00, 0x00, /* 0xBC-0xBF */
	0xD3, 0xF3, 0xD3, 0xF1, 0xD3, 0xEF, 0xD3, 0xF2, /* 0xC0-0xC3 */
	0x00, 0x00, 0x36, 0xEA, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0xD3, 0xF4, 0x36, 0xEB, 0x36, 0xEC, 0x36, 0xED, /* 0xC8-0xCB */
	0x36, 0xF0, 0x36, 0xF1, 0xD3, 0xF5, 0x36, 0xEE, /* 0xCC-0xCF */
	0x36, 0xEF, 0xD3, 0xF6, 0x00, 0x00, 0xD3, 0xF7, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD3, 0xF8, /* 0xD4-0xD7 */
	0xD1, 0xC5, 0x36, 0xF2, 0xBC, 0xFC, 0xBB, 0xCD, /* 0xD8-0xDB */
	0x36, 0xF3, 0x36, 0xF4, 0xB2, 0xF3, 0x36, 0xF5, /* 0xDC-0xDF */
	0xB0, 0xF8, 0x36, 0xF6, 0x00, 0x00, 0xC3, 0xC4, /* 0xE0-0xE3 */
	0x36, 0xF7, 0x36, 0xF8, 0x36, 0xF9, 0x36, 0xFA, /* 0xE4-0xE7 */
	0x36, 0xFB, 0x00, 0x00, 0x00, 0x00, 0x36, 0xFD, /* 0xE8-0xEB */
	0x00, 0x00, 0x36, 0xFE, 0xD3, 0xF9, 0x00, 0x00, /* 0xEC-0xEF */
	0xBA, 0xA4, 0x36, 0xFC, 0xB0, 0xCF, 0xBF, 0xDE, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x37, 0xA1, 0x37, 0xA2, /* 0xF4-0xF7 */
	0x00, 0x00, 0xD3, 0xFA, 0xB8, 0xC7, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xB9, 0xF1, 0x00, 0x00, 0xD3, 0xFC, /* 0xFC-0xFF */
};

static unsigned char u2c_57[512] = {
	0xD3, 0xFB, 0x37, 0xA3, 0x37, 0xA4, 0xCA, 0xE0, /* 0x00-0x03 */
	0xD3, 0xFD, 0x00, 0x00, 0x00, 0x00, 0x37, 0xA5, /* 0x04-0x07 */
	0xD4, 0xA1, 0xD3, 0xFE, 0x37, 0xA6, 0xD4, 0xA2, /* 0x08-0x0B */
	0x37, 0xA7, 0xD4, 0xA3, 0x00, 0x00, 0xB7, 0xF7, /* 0x0C-0x0F */
	0x00, 0x00, 0x37, 0xA8, 0xB1, 0xE0, 0xD4, 0xA4, /* 0x10-0x13 */
	0x00, 0x00, 0x37, 0xA9, 0xD4, 0xA6, 0x00, 0x00, /* 0x14-0x17 */
	0xD4, 0xA5, 0x00, 0x00, 0x37, 0xAA, 0x37, 0xAB, /* 0x18-0x1B */
	0xD4, 0xA8, 0x37, 0xAC, 0x00, 0x00, 0xC5, 0xDA, /* 0x1C-0x1F */
	0x37, 0xAD, 0x00, 0x00, 0x37, 0xAE, 0x37, 0xAF, /* 0x20-0x23 */
	0x37, 0xB0, 0x37, 0xB1, 0xD4, 0xA9, 0xB0, 0xB5, /* 0x24-0x27 */
	0xBA, 0xDF, 0x37, 0xB2, 0x37, 0xB3, 0x00, 0x00, /* 0x28-0x2B */
	0x37, 0xB4, 0xB7, 0xBD, 0x37, 0xB5, 0x37, 0xB6, /* 0x2C-0x2F */
	0xC3, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x37, 0xB7, /* 0x30-0x33 */
	0x37, 0xB8, 0x00, 0x00, 0x00, 0x00, 0xD4, 0xAA, /* 0x34-0x37 */
	0xD4, 0xAB, 0x00, 0x00, 0x00, 0x00, 0xD4, 0xAD, /* 0x38-0x3B */
	0x00, 0x00, 0x37, 0xB9, 0x37, 0xBA, 0x37, 0xBB, /* 0x3C-0x3F */
	0xD4, 0xAE, 0x00, 0x00, 0xBA, 0xE4, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x37, 0xBC, 0x37, 0xBD, 0xB6, 0xD1, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0xCB, 0xB7, 0x00, 0x00, /* 0x48-0x4B */
	0x37, 0xBE, 0x37, 0xBF, 0xD4, 0xAC, 0xD4, 0xAF, /* 0x4C-0x4F */
	0xBA, 0xC1, 0xB9, 0xA3, 0x37, 0xC0, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0xF9, 0xDA, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0xD4, 0xB3, 0x37, 0xC1, 0x00, 0x00, /* 0x60-0x63 */
	0xBA, 0xA5, 0xF9, 0xDB, 0xC3, 0xB3, 0x37, 0xC3, /* 0x64-0x67 */
	0x37, 0xC4, 0xD4, 0xB0, 0xC4, 0xDA, 0x37, 0xC5, /* 0x68-0x6B */
	0x00, 0x00, 0x37, 0xC6, 0x37, 0xC7, 0x37, 0xC8, /* 0x6C-0x6F */
	0x37, 0xC9, 0x37, 0xCA, 0x00, 0x00, 0x37, 0xCB, /* 0x70-0x73 */
	0x37, 0xCC, 0x37, 0xCD, 0x00, 0x00, 0x37, 0xCE, /* 0x74-0x77 */
	0x00, 0x00, 0x37, 0xCF, 0x37, 0xD0, 0x37, 0xD1, /* 0x78-0x7B */
	0x37, 0xD2, 0x00, 0x00, 0x37, 0xD3, 0xD4, 0xB4, /* 0x7C-0x7F */

	0x00, 0x00, 0x37, 0xD4, 0xBF, 0xE2, 0x37, 0xD5, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0xD4, 0xB2, 0xD4, 0xB5, 0x00, 0x00, 0xB7, 0xBF, /* 0x88-0x8B */
	0x37, 0xD6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD4, 0xB6, /* 0x90-0x93 */
	0x37, 0xD7, 0x37, 0xE0, 0x00, 0x00, 0x37, 0xD8, /* 0x94-0x97 */
	0x00, 0x00, 0x37, 0xD9, 0x37, 0xDA, 0x00, 0x00, /* 0x98-0x9B */
	0x37, 0xDB, 0x37, 0xDC, 0x37, 0xDD, 0x37, 0xDE, /* 0x9C-0x9F */
	0xD4, 0xB7, 0x37, 0xDF, 0xB9, 0xA4, 0xB3, 0xC0, /* 0xA0-0xA3 */
	0xD4, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x37, 0xE1, /* 0xA4-0xA7 */
	0x37, 0xE2, 0x37, 0xE3, 0xD4, 0xBA, 0x00, 0x00, /* 0xA8-0xAB */
	0xF9, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0xD4, 0xBB, 0x00, 0x00, 0x00, 0x00, 0xD4, 0xB8, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x37, 0xE5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x37, 0xE6, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xD4, 0xB1, 0x00, 0x00, 0x00, 0x00, 0xD4, 0xBC, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xBD, 0xF9, 0xDE, /* 0xC4-0xC7 */
	0xF9, 0xDD, 0x00, 0x00, 0x00, 0x00, 0xCB, 0xE4, /* 0xC8-0xCB */
	0x37, 0xE9, 0x00, 0x00, 0xBE, 0xEB, 0x37, 0xEA, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xBF, 0xD4, 0xC0, /* 0xD0-0xD3 */
	0xD4, 0xBE, 0x37, 0xEB, 0xD4, 0xC2, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0xC7, 0xB8, 0x37, 0xEC, 0x37, 0xED, 0xB0, 0xE8, /* 0xDC-0xDF */
	0xC9, 0xD6, 0x37, 0xFE, 0x00, 0x00, 0xD4, 0xC3, /* 0xE0-0xE3 */
	0x37, 0xEE, 0x00, 0x00, 0x37, 0xEF, 0x37, 0xF0, /* 0xE4-0xE7 */
	0x00, 0x00, 0x37, 0xF1, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x37, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x37, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0xBE, 0xFD, 0x37, 0xF4, 0x37, 0xF5, 0xBC, 0xB9, /* 0xF4-0xF7 */
	0x37, 0xF6, 0xC7, 0xDD, 0xB4, 0xF0, 0x00, 0x00, /* 0xF8-0xFB */
	0xBA, 0xEB, 0x37, 0xF7, 0x37, 0xF8, 0x37, 0xF9, /* 0xFC-0xFF */
};

static unsigned char u2c_58[512] = {
	0xCB, 0xD9, 0x00, 0x00, 0xC6, 0xB2, 0x37, 0xFA, /* 0x00-0x03 */
	0x37, 0xFB, 0xB7, 0xF8, 0xC2, 0xCF, 0x00, 0x00, /* 0x04-0x07 */
	0x37, 0xFC, 0x37, 0xFD, 0xD4, 0xC1, 0xD4, 0xC4, /* 0x08-0x0B */
	0x38, 0xA1, 0x38, 0xA2, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0xC2, 0xC4, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xD4, 0xC5, 0x00, 0x00, 0x38, 0xA3, /* 0x18-0x1B */
	0x00, 0x00, 0xD4, 0xC6, 0x38, 0xA4, 0x38, 0xA5, /* 0x1C-0x1F */
	0x38, 0xA6, 0xD4, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0xC4, 0xE9, 0x00, 0x00, 0x38, 0xA7, 0x38, 0xA8, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0xB4, 0xAE, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x38, 0xA9, 0x00, 0x00, 0xF4, 0xA1, /* 0x2C-0x2F */
	0xB1, 0xE1, 0xCA, 0xF3, 0x38, 0xAA, 0x00, 0x00, /* 0x30-0x33 */
	0xBE, 0xEC, 0xC5, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x38, 0xAB, 0xBA, 0xE6, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xD4, 0xCE, 0x00, 0x00, 0x38, 0xAC, /* 0x3C-0x3F */
	0xCA, 0xBD, 0xCE, 0xDD, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x38, 0xAD, 0xB2, 0xF4, 0xD4, 0xCA, /* 0x48-0x4B */
	0x38, 0xAE, 0x38, 0xAF, 0x00, 0x00, 0x38, 0xB0, /* 0x4C-0x4F */
	0x38, 0xB1, 0xC1, 0xBA, 0xD4, 0xCD, 0x00, 0x00, /* 0x50-0x53 */
	0xC5, 0xE3, 0x38, 0xB2, 0x00, 0x00, 0xC5, 0xC9, /* 0x54-0x57 */
	0xC5, 0xE4, 0xC8, 0xB9, 0xC4, 0xCD, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xC9, 0x38, 0xB3, /* 0x5C-0x5F */
	0x00, 0x00, 0x38, 0xB4, 0xD4, 0xC9, 0x00, 0x00, /* 0x60-0x63 */
	0x38, 0xB5, 0x00, 0x00, 0x00, 0x00, 0x38, 0xB6, /* 0x64-0x67 */
	0x38, 0xB7, 0xB1, 0xF6, 0x00, 0x00, 0xC5, 0xB6, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xD4, 0xCB, 0x00, 0x00, 0xD4, 0xC7, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xBF, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x38, 0xB8, 0xD4, 0xCF, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x38, 0xB9, 0x00, 0x00, 0xBD, 0xCE, 0x38, 0xBA, /* 0x7C-0x7F */

	0x38, 0xBB, 0x38, 0xBC, 0x00, 0x00, 0xB6, 0xAD, /* 0x80-0x83 */
	0x00, 0x00, 0xD4, 0xD0, 0x00, 0x00, 0x38, 0xBD, /* 0x84-0x87 */
	0x38, 0xBE, 0x38, 0xBF, 0x38, 0xC0, 0x00, 0x00, /* 0x88-0x8B */
	0x38, 0xC1, 0x38, 0xC2, 0x00, 0x00, 0x38, 0xC3, /* 0x8C-0x8F */
	0x38, 0xC4, 0x00, 0x00, 0x00, 0x00, 0xCA, 0xE8, /* 0x90-0x93 */
	0x38, 0xC5, 0x00, 0x00, 0x38, 0xC6, 0xC1, 0xFD, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xC4, 0xC6, 0x38, 0xC7, 0xF9, 0xE1, 0xD4, 0xD2, /* 0x9C-0x9F */
	0x38, 0xC8, 0x38, 0xC9, 0x38, 0xCA, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x38, 0xCB, 0x00, 0x00, /* 0xA4-0xA7 */
	0xCB, 0xCF, 0x38, 0xCC, 0x00, 0x00, 0xD4, 0xD3, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xD8, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x38, 0xCD, 0xF9, 0xE2, 0xCA, 0xAF, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xD4, 0xD7, 0xD4, 0xD1, 0xD4, 0xD4, 0xD4, 0xD6, /* 0xB8-0xBB */
	0x38, 0xD0, 0x00, 0x00, 0xBA, 0xA6, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0xCA, 0xC9, 0x38, 0xD1, 0x00, 0x00, /* 0xC0-0xC3 */
	0x38, 0xCF, 0xD4, 0xD9, 0x00, 0x00, 0xC3, 0xC5, /* 0xC4-0xC7 */
	0x38, 0xD2, 0x00, 0x00, 0xB2, 0xF5, 0x00, 0x00, /* 0xC8-0xCB */
	0xBE, 0xED, 0x38, 0xD3, 0x38, 0xD4, 0x00, 0x00, /* 0xCC-0xCF */
	0x38, 0xD5, 0xD4, 0xDB, 0x38, 0xD6, 0xD4, 0xDA, /* 0xD0-0xD3 */
	0x38, 0xD7, 0xB9, 0xE8, 0x38, 0xD8, 0xD4, 0xDC, /* 0xD4-0xD7 */
	0xD4, 0xDE, 0xD4, 0xDD, 0x38, 0xD9, 0x00, 0x00, /* 0xD8-0xDB */
	0xD4, 0xE0, 0x38, 0xDA, 0xD4, 0xD5, 0xD4, 0xE2, /* 0xDC-0xDF */
	0x00, 0x00, 0x38, 0xDB, 0x38, 0xDC, 0x00, 0x00, /* 0xE0-0xE3 */
	0xD4, 0xE1, 0xD4, 0xDF, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x38, 0xDD, 0x00, 0x00, 0xBB, 0xCE, /* 0xE8-0xEB */
	0xBF, 0xD1, 0x00, 0x00, 0xC1, 0xD4, 0xD4, 0xE3, /* 0xEC-0xEF */
	0xC0, 0xBC, 0xB0, 0xED, 0xC7, 0xE4, 0x38, 0xDE, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC4, 0xDB, /* 0xF4-0xF7 */
	0x00, 0x00, 0xD4, 0xE5, 0xD4, 0xE4, 0xD4, 0xE6, /* 0xF8-0xFB */
	0xD4, 0xE7, 0xD4, 0xE8, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_59[512] = {
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xE9, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x38, 0xDF, 0x38, 0xE0, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0xCA, 0xD1, 0xD4, 0xEA, 0xF9, 0xE3, /* 0x08-0x0B */
	0x38, 0xE2, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xC6, /* 0x0C-0x0F */
	0xD4, 0xEB, 0x00, 0x00, 0x38, 0xE3, 0x38, 0xE4, /* 0x10-0x13 */
	0x38, 0xE5, 0xCD, 0xBC, 0xB3, 0xB0, 0x00, 0x00, /* 0x14-0x17 */
	0xD2, 0xC9, 0xBD, 0xC8, 0xC2, 0xBF, 0xD4, 0xEC, /* 0x18-0x1B */
	0xCC, 0xEB, 0x38, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x38, 0xE8, 0xCC, 0xB4, 0x38, 0xE9, /* 0x20-0x23 */
	0x38, 0xEA, 0xD4, 0xEE, 0x00, 0x00, 0xC2, 0xE7, /* 0x24-0x27 */
	0x38, 0xEB, 0xC5, 0xB7, 0xC2, 0xC0, 0xC9, 0xD7, /* 0x28-0x2B */
	0xD4, 0xEF, 0xD4, 0xF0, 0xB1, 0xFB, 0x38, 0xEC, /* 0x2C-0x2F */
	0x38, 0xED, 0xBC, 0xBA, 0xD4, 0xF1, 0x38, 0xEE, /* 0x30-0x33 */
	0x00, 0x00, 0x38, 0xEF, 0x38, 0xF0, 0xB0, 0xD0, /* 0x34-0x37 */
	0xD4, 0xF2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xF3, 0x38, 0xF1, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0xF2, /* 0x40-0x43 */
	0xB1, 0xE2, 0x00, 0x00, 0x38, 0xF3, 0xB4, 0xF1, /* 0x44-0x47 */
	0xC6, 0xE0, 0xCA, 0xF4, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0xD4, 0xF7, 0xC1, 0xD5, /* 0x4C-0x4F */
	0xD4, 0xF6, 0xB7, 0xC0, 0x38, 0xF4, 0xF9, 0xE4, /* 0x50-0x53 */
	0xCB, 0xDB, 0xD4, 0xF5, 0x00, 0x00, 0xC5, 0xE5, /* 0x54-0x57 */
	0xD4, 0xF9, 0x38, 0xF6, 0xD4, 0xF8, 0xF9, 0xE5, /* 0x58-0x5B */
	0x00, 0x00, 0xF9, 0xE6, 0x38, 0xF9, 0x38, 0xFA, /* 0x5C-0x5F */
	0xD4, 0xFB, 0x38, 0xFB, 0xD4, 0xFA, 0xF9, 0xE7, /* 0x60-0x63 */
	0x00, 0x00, 0xB1, 0xFC, 0x00, 0x00, 0xD4, 0xFC, /* 0x64-0x67 */
	0xBE, 0xA9, 0xD4, 0xFE, 0xC3, 0xA5, 0x38, 0xFD, /* 0x68-0x6B */
	0xD4, 0xFD, 0x38, 0xFE, 0xCA, 0xB3, 0x39, 0xA1, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xA2, 0xBD, 0xF7, /* 0x70-0x73 */
	0xC5, 0xDB, 0x39, 0xA3, 0x39, 0xA4, 0x00, 0x00, /* 0x74-0x77 */
	0xD5, 0xA1, 0x39, 0xA5, 0x00, 0x00, 0x39, 0xA6, /* 0x78-0x7B */
	0x39, 0xA7, 0xB9, 0xA5, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0xD5, 0xA2, 0xC7, 0xA1, 0xC8, 0xDE, /* 0x80-0x83 */
	0xCC, 0xD1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xC7, 0xA5, 0x39, 0xA8, /* 0x88-0x8B */
	0x39, 0xA9, 0xD5, 0xAB, 0x39, 0xAA, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xAB, 0xB5, 0xB8, /* 0x90-0x93 */
	0x00, 0x00, 0x39, 0xAC, 0xCD, 0xC5, 0x39, 0xAD, /* 0x94-0x97 */
	0x00, 0x00, 0xCC, 0xAF, 0x00, 0x00, 0xD6, 0xAC, /* 0x98-0x9B */
	0x00, 0x00, 0xD5, 0xA3, 0x00, 0x00, 0x39, 0xAE, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD5, 0xA6, /* 0xA0-0xA3 */
	0xF9, 0xE8, 0xC2, 0xC5, 0x00, 0x00, 0x39, 0xB0, /* 0xA4-0xA7 */
	0xCB, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0xC5, 0xCA, 0x39, 0xB1, 0x39, 0xB2, 0x39, 0xB3, /* 0xAC-0xAF */
	0x39, 0xB4, 0x00, 0x00, 0xD5, 0xA7, 0x39, 0xB5, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0xB6, /* 0xB4-0xB7 */
	0x00, 0x00, 0xCB, 0xE5, 0xF9, 0xE9, 0xBA, 0xCA, /* 0xB8-0xBB */
	0x39, 0xB8, 0x00, 0x00, 0xBE, 0xAA, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x39, 0xB9, 0x00, 0x00, 0x39, 0xBA, /* 0xC0-0xC3 */
	0x39, 0xBB, 0x00, 0x00, 0xD5, 0xA8, 0x00, 0x00, /* 0xC4-0xC7 */
	0x39, 0xBC, 0xBB, 0xD0, 0x39, 0xBD, 0xBB, 0xCF, /* 0xC8-0xCB */
	0x00, 0x00, 0x39, 0xBE, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0xB0, 0xB9, 0xB8, 0xC8, 0x39, 0xBF, 0xC0, 0xAB, /* 0xD0-0xD3 */
	0xB0, 0xD1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0xD5, 0xAC, 0xD5, 0xAD, 0x00, 0x00, /* 0xD8-0xDB */
	0xD5, 0xAA, 0x39, 0xC0, 0x39, 0xC1, 0x39, 0xC2, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0xC3, /* 0xE0-0xE3 */
	0x39, 0xC4, 0xB1, 0xB8, 0xB4, 0xAF, 0x39, 0xC5, /* 0xE4-0xE7 */
	0xD5, 0xA9, 0x00, 0x00, 0xCC, 0xC5, 0xC9, 0xB1, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xC6, 0x39, 0xC7, /* 0xEC-0xEF */
	0x00, 0x00, 0x39, 0xC8, 0x39, 0xC9, 0x00, 0x00, /* 0xF0-0xF3 */
	0x39, 0xCA, 0x00, 0x00, 0xB0, 0xA8, 0x39, 0xCB, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB0, 0xF9, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBB, 0xD1, /* 0xFC-0xFF */
};

static unsigned char u2c_5A[512] = {
	0x39, 0xCC, 0xB0, 0xD2, 0x00, 0x00, 0xB0, 0xA3, /* 0x00-0x03 */
	0x39, 0xCD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0xD5, 0xB2, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x39, 0xCE, 0x39, 0xCF, 0x39, 0xD0, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xD5, 0xB0, 0x39, 0xD1, 0x39, 0xD2, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xCC, 0xBC, 0x00, 0x00, 0xD5, 0xB3, 0x00, 0x00, /* 0x18-0x1B */
	0xD5, 0xB1, 0x00, 0x00, 0x39, 0xD3, 0xD5, 0xAF, /* 0x1C-0x1F */
	0xBF, 0xB1, 0x00, 0x00, 0x00, 0x00, 0x39, 0xD4, /* 0x20-0x23 */
	0x39, 0xD5, 0xD5, 0xAE, 0x00, 0x00, 0x39, 0xD6, /* 0x24-0x27 */
	0x39, 0xD7, 0xCA, 0xDA, 0x39, 0xD8, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x39, 0xD9, 0x00, 0x00, 0xB8, 0xE4, /* 0x2C-0x2F */
	0x39, 0xDA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0xD5, 0xB7, 0xD5, 0xB8, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0xBE, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xD5, 0xB4, 0xCF, 0xAC, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x39, 0xDB, 0x39, 0xDC, 0xC7, 0xCC, 0x39, 0xDD, /* 0x44-0x47 */
	0x39, 0xDE, 0xD5, 0xB6, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x39, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x39, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x39, 0xE1, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xA7, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xE2, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xB9, 0x39, 0xE3, /* 0x60-0x63 */
	0x00, 0x00, 0x39, 0xE4, 0xC9, 0xD8, 0x39, 0xE5, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xBA, 0x00, 0x00, /* 0x68-0x6B */
	0xD5, 0xB5, 0x39, 0xE6, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0xE7, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xE8, 0x39, 0xE9, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xEA, 0xCC, 0xBB, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0xEB, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x39, 0xEC, 0x00, 0x00, 0xC7, 0xDE, 0x39, 0xED, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x39, 0xEE, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x39, 0xEF, 0xD5, 0xBB, 0xC9, 0xB2, /* 0x98-0x9B */
	0x39, 0xF0, 0x00, 0x00, 0x39, 0xF1, 0x39, 0xF2, /* 0x9C-0x9F */
	0x39, 0xF3, 0x00, 0x00, 0x39, 0xF4, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0xF5, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x39, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x39, 0xF7, 0x39, 0xF8, 0x39, 0xF9, /* 0xB0-0xB3 */
	0x00, 0x00, 0x39, 0xFA, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x39, 0xFB, 0x00, 0x00, 0x39, 0xFC, 0x39, 0xFD, /* 0xB8-0xBB */
	0xD5, 0xBC, 0xD5, 0xC0, 0xD5, 0xBD, 0x39, 0xFE, /* 0xBC-0xBF */
	0x00, 0x00, 0xB2, 0xC7, 0xD5, 0xBF, 0x00, 0x00, /* 0xC0-0xC3 */
	0x3A, 0xA1, 0x00, 0x00, 0x3A, 0xA2, 0x00, 0x00, /* 0xC4-0xC7 */
	0x3A, 0xA3, 0xBC, 0xBB, 0x00, 0x00, 0xD5, 0xBE, /* 0xC8-0xCB */
	0xB7, 0xF9, 0x00, 0x00, 0x00, 0x00, 0x3A, 0xA4, /* 0xCC-0xCF */
	0xD5, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xC5, 0xD5, 0xC2, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x3A, 0xA5, 0x00, 0x00, /* 0xD8-0xDB */
	0x3A, 0xA6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x3A, 0xA7, 0xC3, 0xE4, 0x00, 0x00, 0xD5, 0xC1, /* 0xE0-0xE3 */
	0x00, 0x00, 0x3A, 0xA8, 0xD5, 0xC3, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0xD5, 0xC4, 0x3A, 0xA9, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x3A, 0xAA, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x3A, 0xAB, 0x3A, 0xAC, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xC6, 0xD5, 0xC7, /* 0xF8-0xFB */
	0x00, 0x00, 0x3A, 0xAD, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_5B[512] = {
	0x3A, 0xAE, 0x3A, 0xAF, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x3A, 0xB0, 0xB4, 0xF2, 0x00, 0x00, 0xD5, 0xC9, /* 0x08-0x0B */
	0xD5, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xCA, 0x3A, 0xB1, /* 0x14-0x17 */
	0x00, 0x00, 0x3A, 0xB3, 0x00, 0x00, 0x3A, 0xB4, /* 0x18-0x1B */
	0x00, 0x00, 0x3A, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x3A, 0xB6, 0xBE, 0xEE, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x3A, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xCD, 0x00, 0x00, /* 0x28-0x2B */
	0xC4, 0xDC, 0x3A, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xB1, 0xC5, 0x00, 0x00, 0xD5, 0xCB, 0x00, 0x00, /* 0x30-0x33 */
	0x3A, 0xB2, 0x00, 0x00, 0xD5, 0xCE, 0x00, 0x00, /* 0x34-0x37 */
	0x3A, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0xD5, 0xCF, 0x00, 0x00, /* 0x3C-0x3F */
	0xD5, 0xD2, 0x3A, 0xBA, 0x00, 0x00, 0xD5, 0xD0, /* 0x40-0x43 */
	0x00, 0x00, 0xD5, 0xD1, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3A, 0xBB, /* 0x48-0x4B */
	0x3A, 0xBC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0xBB, 0xD2, 0xD5, 0xD3, 0x3A, 0xBD, 0x00, 0x00, /* 0x50-0x53 */
	0xB9, 0xA6, 0xD5, 0xD4, 0xF9, 0xEA, 0xBB, 0xFA, /* 0x54-0x57 */
	0xC2, 0xB8, 0x00, 0x00, 0xD5, 0xD5, 0xD5, 0xD6, /* 0x58-0x5B */
	0xBB, 0xDA, 0xB9, 0xA7, 0x3A, 0xBF, 0xCC, 0xD2, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB5, 0xA8, /* 0x60-0x63 */
	0xB8, 0xC9, 0xD5, 0xD7, 0xB3, 0xD8, 0x00, 0x00, /* 0x64-0x67 */
	0x3A, 0xC0, 0xD5, 0xD8, 0x00, 0x00, 0xC2, 0xB9, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x3A, 0xC1, 0x3A, 0xC2, /* 0x6C-0x6F */
	0xD5, 0xD9, 0xD6, 0xA3, 0x00, 0x00, 0xD5, 0xDA, /* 0x70-0x73 */
	0x00, 0x00, 0xD5, 0xDB, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xD5, 0xDC, 0x00, 0x00, 0xD5, 0xDE, 0x00, 0x00, /* 0x78-0x7B */
	0x3A, 0xC3, 0x3A, 0xC4, 0x3A, 0xC5, 0x3A, 0xC6, /* 0x7C-0x7F */

	0xD5, 0xDF, 0x3A, 0xC7, 0x00, 0x00, 0xD5, 0xE0, /* 0x80-0x83 */
	0x3A, 0xC8, 0xC2, 0xF0, 0x3A, 0xC9, 0xB1, 0xA7, /* 0x84-0x87 */
	0xBC, 0xE9, 0xB0, 0xC2, 0x3A, 0xCA, 0xC1, 0xD7, /* 0x88-0x8B */
	0xB4, 0xB0, 0xBC, 0xB5, 0x3A, 0xCB, 0xB9, 0xA8, /* 0x8C-0x8F */
	0x3A, 0xCC, 0x3A, 0xCD, 0x00, 0x00, 0x3A, 0xCE, /* 0x90-0x93 */
	0x3A, 0xCF, 0xC5, 0xE6, 0x3A, 0xD0, 0xBD, 0xA1, /* 0x94-0x97 */
	0xB4, 0xB1, 0xC3, 0xE8, 0xC4, 0xEA, 0xB0, 0xB8, /* 0x98-0x9B */
	0xB5, 0xB9, 0xCA, 0xF5, 0x00, 0x00, 0xBC, 0xC2, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0xB5, 0xD2, 0xC0, 0xEB, /* 0xA0-0xA3 */
	0xBC, 0xBC, 0xCD, 0xA8, 0xD5, 0xE1, 0x00, 0x00, /* 0xA4-0xA7 */
	0x3A, 0xD1, 0x3A, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x3A, 0xD3, 0x3A, 0xD4, 0xB5, 0xDC, 0x3A, 0xD5, /* 0xAC-0xAF */
	0xBA, 0xCB, 0x3A, 0xD6, 0x3A, 0xD7, 0xB3, 0xB2, /* 0xB0-0xB3 */
	0xB1, 0xE3, 0xBE, 0xAC, 0xB2, 0xC8, 0x3A, 0xD8, /* 0xB4-0xB7 */
	0xD5, 0xE2, 0xCD, 0xC6, 0x3A, 0xD9, 0x00, 0x00, /* 0xB8-0xBB */
	0x3A, 0xDA, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xC9, /* 0xBC-0xBF */
	0xF9, 0xEB, 0x3A, 0xDC, 0xBC, 0xE4, 0xD5, 0xE3, /* 0xC0-0xC3 */
	0xB4, 0xF3, 0xC6, 0xD2, 0xCC, 0xA9, 0xD5, 0xE4, /* 0xC4-0xC7 */
	0x00, 0x00, 0xD5, 0xE5, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0xC9, 0xD9, 0x3A, 0xDD, 0x00, 0x00, 0x3A, 0xDE, /* 0xCC-0xCF */
	0xD5, 0xE7, 0x00, 0x00, 0xB4, 0xA8, 0xB6, 0xF7, /* 0xD0-0xD3 */
	0xD5, 0xE6, 0x00, 0x00, 0x3A, 0xDF, 0x3A, 0xE0, /* 0xD4-0xD7 */
	0xF9, 0xED, 0x3A, 0xE2, 0x3A, 0xE3, 0xB4, 0xB2, /* 0xD8-0xDB */
	0x00, 0x00, 0xBF, 0xB2, 0xD5, 0xEB, 0xBB, 0xA1, /* 0xDC-0xDF */
	0x3A, 0xE4, 0xB2, 0xC9, 0xD5, 0xEA, 0x00, 0x00, /* 0xE0-0xE3 */
	0xD5, 0xE8, 0xD5, 0xEC, 0xD5, 0xE9, 0xC7, 0xAB, /* 0xE4-0xE7 */
	0xDC, 0xCD, 0xBF, 0xB3, 0x00, 0x00, 0xD5, 0xED, /* 0xE8-0xEB */
	0xF9, 0xEE, 0x00, 0x00, 0xCE, 0xC0, 0x3A, 0xE5, /* 0xEC-0xEF */
	0xD5, 0xEE, 0x3A, 0xE6, 0x00, 0x00, 0xD5, 0xF0, /* 0xF0-0xF3 */
	0x3A, 0xE7, 0xC3, 0xFE, 0xD5, 0xEF, 0x00, 0x00, /* 0xF4-0xF7 */
	0xC0, 0xA3, 0x00, 0x00, 0xBB, 0xFB, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0x3A, 0xE8, 0xC2, 0xD0, 0xBC, 0xF7, /* 0xFC-0xFF */
};

static unsigned char u2c_5C[512] = {
	0x00, 0x00, 0xC9, 0xF5, 0xC0, 0xEC, 0x00, 0x00, /* 0x00-0x03 */
	0xBC, 0xCD, 0xD5, 0xF1, 0xBE, 0xAD, 0xD5, 0xF2, /* 0x04-0x07 */
	0xD5, 0xF3, 0xB0, 0xD3, 0xC2, 0xBA, 0xBF, 0xD2, /* 0x08-0x0B */
	0x3A, 0xE9, 0xD5, 0xF4, 0xC6, 0xB3, 0xBE, 0xAE, /* 0x0C-0x0F */
	0x00, 0x00, 0xBE, 0xAF, 0x00, 0x00, 0xD5, 0xF5, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xC0, 0xED, 0x3A, 0xEA, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0xBE, 0xB0, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0xF9, 0xEF, 0x3A, 0xEC, /* 0x1C-0x1F */
	0xD5, 0xF6, 0x00, 0x00, 0xD5, 0xF7, 0x3A, 0xED, /* 0x20-0x23 */
	0xCC, 0xE0, 0x00, 0x00, 0x3A, 0xEE, 0x00, 0x00, /* 0x24-0x27 */
	0xD5, 0xF8, 0x3A, 0xEF, 0x00, 0x00, 0x3A, 0xF0, /* 0x28-0x2B */
	0x3A, 0xF1, 0xB6, 0xC6, 0x3A, 0xF2, 0x00, 0x00, /* 0x2C-0x2F */
	0x3A, 0xF3, 0xBD, 0xA2, 0x3A, 0xF4, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x3A, 0xF5, 0x3A, 0xF6, 0x00, 0x00, /* 0x34-0x37 */
	0xD5, 0xF9, 0xD5, 0xFA, 0xBC, 0xDC, 0xBF, 0xAC, /* 0x38-0x3B */
	0xC6, 0xF4, 0xBF, 0xD4, 0xC8, 0xF8, 0xC7, 0xA2, /* 0x3C-0x3F */
	0xB6, 0xC9, 0xD5, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0xB5, 0xEF, 0xD5, 0xFC, 0x00, 0x00, /* 0x44-0x47 */
	0xB6, 0xFE, 0x00, 0x00, 0xC6, 0xCF, 0xB2, 0xB0, /* 0x48-0x4B */
	0x00, 0x00, 0xBB, 0xD3, 0xD5, 0xFD, 0xD6, 0xA2, /* 0x4C-0x4F */
	0xD6, 0xA1, 0xB6, 0xFD, 0x00, 0x00, 0xD5, 0xFE, /* 0x50-0x53 */
	0x00, 0x00, 0xC5, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x3A, 0xF7, 0x3A, 0xF8, 0x00, 0x00, /* 0x58-0x5B */
	0x3A, 0xF9, 0x00, 0x00, 0xC2, 0xB0, 0x00, 0x00, /* 0x5C-0x5F */
	0xC5, 0xCB, 0xBC, 0xC8, 0x3A, 0xFA, 0x3A, 0xFB, /* 0x60-0x63 */
	0xC1, 0xD8, 0xCD, 0xFA, 0x00, 0x00, 0x3A, 0xFC, /* 0x64-0x67 */
	0x3A, 0xFD, 0x3A, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0xD6, 0xA4, 0x3B, 0xA1, 0xD6, 0xA5, 0xC6, 0xD6, /* 0x6C-0x6F */
	0x3B, 0xA2, 0xBB, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x3B, 0xA3, 0x3B, 0xA4, 0xD6, 0xA7, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0xD6, 0xA8, 0x3B, 0xA5, 0x3B, 0xA6, /* 0x78-0x7B */
	0x3B, 0xA7, 0x3B, 0xA8, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xA9, /* 0x84-0x87 */
	0x3B, 0xAA, 0x00, 0x00, 0x3B, 0xAB, 0x00, 0x00, /* 0x88-0x8B */
	0xD6, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xAC, /* 0x8C-0x8F */
	0xB4, 0xF4, 0xD6, 0xAA, 0x3B, 0xAD, 0x00, 0x00, /* 0x90-0x93 */
	0xD6, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x3B, 0xAE, 0x00, 0x00, 0x3B, 0xAF, /* 0x9C-0x9F */
	0x3B, 0xB0, 0xB2, 0xAC, 0x3B, 0xB1, 0x3B, 0xB2, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0xF9, 0xF0, 0x00, 0x00, /* 0xA4-0xA7 */
	0xC1, 0xBB, 0xB4, 0xE4, 0x3B, 0xB4, 0xD6, 0xAD, /* 0xA8-0xAB */
	0xCC, 0xA8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0xC2, 0xD2, 0x3B, 0xB5, 0xB3, 0xD9, /* 0xB0-0xB3 */
	0x3B, 0xB6, 0x3B, 0xB7, 0xD6, 0xAF, 0xD6, 0xB1, /* 0xB4-0xB7 */
	0xB4, 0xDF, 0x00, 0x00, 0xF9, 0xF1, 0xD6, 0xAE, /* 0xB8-0xBB */
	0xD6, 0xB0, 0x00, 0x00, 0xD6, 0xB3, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0xD6, 0xB2, 0x00, 0x00, 0xD6, 0xB4, /* 0xC4-0xC7 */
	0x00, 0x00, 0x3B, 0xB9, 0x00, 0x00, 0x3B, 0xBA, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x3B, 0xBB, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xBD, /* 0xD4-0xD7 */
	0x00, 0x00, 0xD6, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x3B, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0xC6, 0xBD, 0xB6, 0xAE, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0xB2, 0xE5, 0xD6, 0xB6, 0xD6, 0xBB, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xD6, 0xB9, 0x3B, 0xBE, 0xCA, 0xF7, /* 0xEC-0xEF */
	0xCA, 0xF6, 0x3B, 0xBF, 0x3B, 0xC0, 0x00, 0x00, /* 0xF0-0xF3 */
	0x3B, 0xC1, 0xF9, 0xF2, 0xC5, 0xE7, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0xD6, 0xB8, 0xBD, 0xD4, /* 0xF8-0xFB */
	0x00, 0x00, 0xD6, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_5D[512] = {
	0x00, 0x00, 0x3B, 0xC2, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x3B, 0xC3, 0xBF, 0xF2, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xBC, /* 0x08-0x0B */
	0x00, 0x00, 0x3B, 0xC4, 0xBA, 0xEA, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xD6, 0xC2, 0x3B, 0xC5, 0x00, 0x00, /* 0x10-0x13 */
	0xD6, 0xC3, 0xD6, 0xBD, 0xB3, 0xB3, 0xD6, 0xBE, /* 0x14-0x17 */
	0xD6, 0xC7, 0xD6, 0xC6, 0xD6, 0xC5, 0xD6, 0xC1, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xC0, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0xD6, 0xC4, 0x3B, 0xC7, /* 0x20-0x23 */
	0x3B, 0xC8, 0x00, 0x00, 0x3B, 0xC9, 0xF9, 0xF3, /* 0x24-0x27 */
	0x00, 0x00, 0xCA, 0xF8, 0x00, 0x00, 0x3B, 0xC6, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x3B, 0xCB, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x3B, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x3B, 0xCD, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x3B, 0xCE, 0x00, 0x00, 0x3B, 0xCF, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0xF9, 0xF6, 0x3B, 0xD1, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x3B, 0xD2, 0x00, 0x00, /* 0x44-0x47 */
	0x3B, 0xD3, 0x00, 0x00, 0x3B, 0xD7, 0xD6, 0xCB, /* 0x48-0x4B */
	0xD6, 0xC8, 0x00, 0x00, 0xD6, 0xCA, 0x00, 0x00, /* 0x4C-0x4F */
	0xCD, 0xF2, 0x3B, 0xD5, 0xD6, 0xC9, 0xF9, 0xF4, /* 0x50-0x53 */
	0x00, 0x00, 0x3B, 0xD4, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x3B, 0xD6, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0xD6, 0xBF, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xD8, /* 0x5C-0x5F */
	0x3B, 0xD9, 0x3B, 0xDA, 0x3B, 0xDB, 0x00, 0x00, /* 0x60-0x63 */
	0x3B, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0xBF, 0xF3, 0x3B, 0xDD, 0x00, 0x00, /* 0x68-0x6B */
	0xD6, 0xCC, 0xF9, 0xF7, 0x00, 0x00, 0xBA, 0xB7, /* 0x6C-0x6F */
	0x3B, 0xDF, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xCD, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0xD6, 0xCE, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x3B, 0xE0, 0x3B, 0xE1, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x3B, 0xE2, 0x3B, 0xE3, /* 0x7C-0x7F */

	0x00, 0x00, 0x3B, 0xE4, 0xD6, 0xD1, 0x3B, 0xE5, /* 0x80-0x83 */
	0xD6, 0xD0, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xCF, /* 0x84-0x87 */
	0x3B, 0xE6, 0x00, 0x00, 0x3B, 0xE7, 0xC5, 0xE8, /* 0x88-0x8B */
	0xD6, 0xBA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0xD6, 0xD7, 0x00, 0x00, 0x3B, 0xE8, 0x3B, 0xE9, /* 0x90-0x93 */
	0x3B, 0xEA, 0x3B, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x3B, 0xEC, 0x00, 0x00, 0x3B, 0xED, /* 0x98-0x9B */
	0x00, 0x00, 0xD6, 0xD3, 0x00, 0x00, 0x3B, 0xEE, /* 0x9C-0x9F */
	0x3B, 0xEF, 0x00, 0x00, 0xD6, 0xD2, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xF0, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xF1, /* 0xA8-0xAB */
	0xD6, 0xD4, 0x00, 0x00, 0xD6, 0xD5, 0x00, 0x00, /* 0xAC-0xAF */
	0x3B, 0xF2, 0x00, 0x00, 0x66, 0xF4, 0x00, 0x00, /* 0xB0-0xB3 */
	0x3B, 0xF3, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xD8, /* 0xB4-0xB7 */
	0xF9, 0xF8, 0xF9, 0xF9, 0xCE, 0xE6, 0x00, 0x00, /* 0xB8-0xBB */
	0xD6, 0xD9, 0xD6, 0xD6, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xF6, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0xF7, /* 0xC4-0xC7 */
	0x00, 0x00, 0xD6, 0xDA, 0x00, 0x00, 0x3B, 0xF8, /* 0xC8-0xCB */
	0xB4, 0xE0, 0xD6, 0xDB, 0x3B, 0xFA, 0x00, 0x00, /* 0xCC-0xCF */
	0xF9, 0xFA, 0x00, 0x00, 0xD6, 0xDD, 0xD6, 0xDC, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0xD6, 0xDE, 0x00, 0x00, /* 0xD4-0xD7 */
	0x3B, 0xFB, 0x3B, 0xFC, 0x00, 0x00, 0xD6, 0xDF, /* 0xD8-0xDB */
	0x00, 0x00, 0xC0, 0xEE, 0xBD, 0xA3, 0x00, 0x00, /* 0xDC-0xDF */
	0x3B, 0xFD, 0xBD, 0xE4, 0x00, 0x00, 0xC1, 0xE3, /* 0xE0-0xE3 */
	0x3B, 0xFE, 0xB9, 0xA9, 0xBA, 0xB8, 0xB9, 0xAA, /* 0xE4-0xE7 */
	0xB5, 0xF0, 0x3C, 0xA1, 0x00, 0x00, 0xD6, 0xE0, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xB9, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0xB8, 0xCA, 0xD6, 0xE1, 0xCC, 0xA6, /* 0xF0-0xF3 */
	0xC7, 0xC3, 0xD6, 0xE2, 0x00, 0x00, 0xB9, 0xAB, /* 0xF4-0xF7 */
	0x3C, 0xA2, 0x3C, 0xA3, 0x00, 0x00, 0xB4, 0xAC, /* 0xF8-0xFB */
	0x00, 0x00, 0xC3, 0xA7, 0xB6, 0xD2, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_5E[512] = {
	0x3C, 0xA4, 0x00, 0x00, 0xBB, 0xD4, 0xC9, 0xDB, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xC8, 0xC1, 0x3C, 0xA5, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xE3, /* 0x08-0x0B */
	0xB4, 0xF5, 0x3C, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xD6, 0xE6, 0x3C, 0xA7, 0x00, 0x00, /* 0x10-0x13 */
	0x3C, 0xA8, 0x3C, 0xA9, 0xC4, 0xA1, 0x00, 0x00, /* 0x14-0x17 */
	0x3C, 0xAA, 0xD6, 0xE5, 0xD6, 0xE4, 0xD6, 0xE7, /* 0x18-0x1B */
	0x00, 0x00, 0xC4, 0xEB, 0x00, 0x00, 0x3C, 0xAB, /* 0x1C-0x1F */
	0x3C, 0xAC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0xBF, 0xE3, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x3C, 0xAE, 0x00, 0x00, 0x00, 0x00, 0xBB, 0xD5, /* 0x28-0x2B */
	0x00, 0x00, 0xC0, 0xCA, 0x3C, 0xAD, 0xC2, 0xD3, /* 0x2C-0x2F */
	0xB5, 0xA2, 0x00, 0x00, 0x3C, 0xAF, 0xC4, 0xA2, /* 0x30-0x33 */
	0x00, 0x00, 0x3C, 0xB0, 0xD6, 0xE8, 0xD6, 0xE9, /* 0x34-0x37 */
	0xBE, 0xEF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xCB, 0xB9, 0x3C, 0xB1, 0x00, 0x00, /* 0x3C-0x3F */
	0xD6, 0xEC, 0x00, 0x00, 0x00, 0x00, 0xD6, 0xEB, /* 0x40-0x43 */
	0xD6, 0xEA, 0xC9, 0xFD, 0x00, 0x00, 0xD6, 0xF3, /* 0x44-0x47 */
	0x00, 0x00, 0x3C, 0xB4, 0x00, 0x00, 0x3C, 0xB2, /* 0x48-0x4B */
	0xCB, 0xDA, 0x00, 0x00, 0xD6, 0xED, 0x00, 0x00, /* 0x4C-0x4F */
	0x3C, 0xB3, 0x3C, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0xD6, 0xEF, 0xCB, 0xEB, 0x3C, 0xB6, 0xD6, 0xEE, /* 0x54-0x57 */
	0x3C, 0xB7, 0x00, 0x00, 0x00, 0x00, 0x3C, 0xB8, /* 0x58-0x5B */
	0x3C, 0xB9, 0x00, 0x00, 0x3C, 0xBA, 0xD6, 0xF0, /* 0x5C-0x5F */
	0x00, 0x00, 0xC8, 0xA8, 0xD6, 0xF1, 0xCA, 0xBE, /* 0x60-0x63 */
	0xD6, 0xF2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x3C, 0xBB, 0x00, 0x00, 0x3C, 0xBC, 0x3C, 0xBD, /* 0x68-0x6B */
	0x3C, 0xBE, 0x3C, 0xBF, 0x3C, 0xC0, 0x00, 0x00, /* 0x6C-0x6F */
	0x3C, 0xC1, 0x00, 0x00, 0xB4, 0xB3, 0xCA, 0xBF, /* 0x70-0x73 */
	0xC7, 0xAF, 0xD6, 0xF4, 0xD6, 0xF5, 0x00, 0x00, /* 0x74-0x77 */
	0xB9, 0xAC, 0xB4, 0xB4, 0xD6, 0xF6, 0xB8, 0xB8, /* 0x78-0x7B */
	0xCD, 0xC4, 0xCD, 0xA9, 0xB4, 0xF6, 0xD6, 0xF8, /* 0x7C-0x7F */

	0x3C, 0xC2, 0xC4, 0xA3, 0x00, 0x00, 0xB9, 0xAD, /* 0x80-0x83 */
	0xBE, 0xB1, 0x00, 0x00, 0x00, 0x00, 0xC8, 0xDF, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xBE, 0xB2, 0x3C, 0xC3, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x3C, 0xC4, 0xBD, 0xF8, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0xC4, 0xEC, 0xCA, 0xF9, 0xC5, 0xB9, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0xB9, 0xAE, 0x00, 0x00, /* 0x98-0x9B */
	0xC9, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xD6, 0xF9, 0x00, 0x00, 0x3C, 0xC5, 0x00, 0x00, /* 0xA0-0xA3 */
	0x3C, 0xC6, 0x3C, 0xC7, 0xC5, 0xD9, 0xBA, 0xC2, /* 0xA4-0xA7 */
	0x3C, 0xC8, 0x00, 0x00, 0x3C, 0xC9, 0xB8, 0xCB, /* 0xA8-0xAB */
	0x3C, 0xCA, 0xC4, 0xED, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x3C, 0xCB, 0x00, 0x00, 0x3C, 0xCC, /* 0xB0-0xB3 */
	0x00, 0x00, 0xB0, 0xC3, 0xBD, 0xEE, 0xB9, 0xAF, /* 0xB4-0xB7 */
	0xCD, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x3C, 0xCD, 0x3C, 0xCE, 0x3C, 0xCF, /* 0xBC-0xBF */
	0x00, 0x00, 0xD6, 0xFA, 0xD6, 0xFB, 0xC7, 0xD1, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x3C, 0xD0, 0x00, 0x00, /* 0xC4-0xC7 */
	0xD6, 0xFC, 0xCE, 0xF7, 0xCF, 0xAD, 0x3C, 0xD2, /* 0xC8-0xCB */
	0x3C, 0xD1, 0x00, 0x00, 0x3C, 0xD3, 0xD6, 0xFE, /* 0xCC-0xCF */
	0xD6, 0xFD, 0x3C, 0xD4, 0x3C, 0xD5, 0xB3, 0xC7, /* 0xD0-0xD3 */
	0x3C, 0xD6, 0x3C, 0xD7, 0xD7, 0xA1, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xD7, 0xA4, 0xD7, 0xA5, /* 0xD8-0xDB */
	0x3C, 0xD8, 0xD7, 0xA3, 0x3C, 0xD9, 0xC9, 0xC0, /* 0xDC-0xDF */
	0xBE, 0xB3, 0xD7, 0xA7, 0xD7, 0xA6, 0xD7, 0xA2, /* 0xE0-0xE3 */
	0x00, 0x00, 0x3C, 0xDA, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0xD7, 0xA8, 0xD7, 0xA9, 0x00, 0x00, 0x3C, 0xDB, /* 0xE8-0xEB */
	0xD7, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0xD7, 0xAD, 0xD7, 0xAB, 0x00, 0x00, 0xD7, 0xAC, /* 0xF0-0xF3 */
	0xD7, 0xAE, 0x00, 0x00, 0xB1, 0xE4, 0xC4, 0xEE, /* 0xF4-0xF7 */
	0xD7, 0xAF, 0x00, 0x00, 0xB7, 0xFA, 0xB2, 0xF6, /* 0xF8-0xFB */
	0xC7, 0xB6, 0x00, 0x00, 0xD7, 0xB0, 0xC6, 0xFB, /* 0xFC-0xFF */
};

static unsigned char u2c_5F[512] = {
	0x00, 0x00, 0xCA, 0xDB, 0x3C, 0xDC, 0xD7, 0xB1, /* 0x00-0x03 */
	0xCF, 0xAE, 0x00, 0x00, 0x3C, 0xDD, 0x3C, 0xDE, /* 0x04-0x07 */
	0x3C, 0xDF, 0xD7, 0xB2, 0xCA, 0xC0, 0xD7, 0xB5, /* 0x08-0x0B */
	0xD0, 0xA1, 0xD0, 0xB1, 0x3C, 0xE0, 0xBC, 0xB0, /* 0x0C-0x0F */
	0xC6, 0xF5, 0xD7, 0xB6, 0x00, 0x00, 0xB5, 0xDD, /* 0x10-0x13 */
	0xC4, 0xA4, 0xB0, 0xFA, 0xD7, 0xB7, 0xCA, 0xA6, /* 0x14-0x17 */
	0xB9, 0xB0, 0x3C, 0xE1, 0x00, 0x00, 0xC3, 0xD0, /* 0x18-0x1B */
	0x3C, 0xE2, 0x3C, 0xE3, 0x00, 0x00, 0xC4, 0xEF, /* 0x1C-0x1F */
	0x00, 0x00, 0xF9, 0xFB, 0x3C, 0xE5, 0x3C, 0xE6, /* 0x20-0x23 */
	0x3C, 0xE7, 0xCC, 0xEF, 0xB8, 0xB9, 0xB8, 0xCC, /* 0x24-0x27 */
	0x3C, 0xE8, 0xD7, 0xB8, 0x00, 0x00, 0x3C, 0xE9, /* 0x28-0x2B */
	0x3C, 0xEA, 0xD7, 0xB9, 0x3C, 0xEB, 0xD7, 0xBF, /* 0x2C-0x2F */
	0x3C, 0xEC, 0xBC, 0xE5, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0xF9, 0xFC, 0xC4, 0xA5, 0x3C, 0xEE, 0xB6, 0xAF, /* 0x34-0x37 */
	0xD7, 0xBA, 0x00, 0x00, 0x00, 0x00, 0x3C, 0xEF, /* 0x38-0x3B */
	0xC9, 0xAB, 0x3C, 0xF0, 0xC3, 0xC6, 0x3C, 0xF1, /* 0x3C-0x3F */
	0x3C, 0xF2, 0xD7, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x3C, 0xF3, 0xF9, 0xAC, 0x00, 0x00, 0x3C, 0xF5, /* 0x44-0x47 */
	0xD7, 0xBC, 0x00, 0x00, 0xB6, 0xB0, 0x00, 0x00, /* 0x48-0x4B */
	0xD7, 0xBD, 0x3C, 0xF6, 0xD7, 0xBE, 0x00, 0x00, /* 0x4C-0x4F */
	0x3C, 0xF7, 0xD7, 0xC0, 0x00, 0x00, 0xC5, 0xF6, /* 0x50-0x53 */
	0x3C, 0xF8, 0x00, 0x00, 0xD7, 0xC1, 0xD7, 0xC2, /* 0x54-0x57 */
	0x3C, 0xF9, 0xD7, 0xC3, 0x00, 0x00, 0x3C, 0xFA, /* 0x58-0x5B */
	0xD7, 0xB4, 0xD7, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x3C, 0xFB, 0xD7, 0xC4, 0xB7, 0xC1, 0x3C, 0xFC, /* 0x60-0x63 */
	0x3C, 0xFD, 0x00, 0x00, 0xC9, 0xA7, 0xF9, 0xFD, /* 0x64-0x67 */
	0x00, 0x00, 0xBA, 0xCC, 0xC9, 0xB7, 0xC4, 0xA6, /* 0x68-0x6B */
	0xC9, 0xCB, 0xD7, 0xC5, 0x00, 0x00, 0x3D, 0xA1, /* 0x6C-0x6F */
	0xBE, 0xB4, 0xB1, 0xC6, 0x3D, 0xA2, 0xD7, 0xC6, /* 0x70-0x73 */
	0x3D, 0xA3, 0x3D, 0xA4, 0x00, 0x00, 0xD7, 0xC7, /* 0x74-0x77 */
	0x3D, 0xA5, 0xCC, 0xF2, 0x3D, 0xA6, 0x00, 0x00, /* 0x78-0x7B */
	0xC8, 0xE0, 0x3D, 0xA7, 0x3D, 0xA8, 0xD7, 0xCA, /* 0x7C-0x7F */

	0xB1, 0xFD, 0xC0, 0xAC, 0xD7, 0xC9, 0xD7, 0xC8, /* 0x80-0x83 */
	0xB7, 0xC2, 0xC2, 0xD4, 0x00, 0x00, 0xD7, 0xCE, /* 0x84-0x87 */
	0xD7, 0xCC, 0x3D, 0xA9, 0xD7, 0xCB, 0xCE, 0xA7, /* 0x88-0x8B */
	0xB8, 0xE5, 0x3D, 0xAA, 0x00, 0x00, 0x3D, 0xAB, /* 0x8C-0x8F */
	0xBD, 0xF9, 0xD7, 0xCD, 0xC5, 0xCC, 0xBD, 0xBE, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x3D, 0xAC, 0xC6, 0xC0, /* 0x94-0x97 */
	0xD7, 0xD1, 0xD7, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x3D, 0xAD, 0x3D, 0xAE, 0xD7, 0xCF, 0x00, 0x00, /* 0x9C-0x9F */
	0xD7, 0xD2, 0xB8, 0xE6, 0x3D, 0xAF, 0x00, 0x00, /* 0xA0-0xA3 */
	0x3D, 0xB2, 0x00, 0x00, 0x00, 0x00, 0x3D, 0xB0, /* 0xA4-0xA7 */
	0xD7, 0xD3, 0xC9, 0xFC, 0xBD, 0xDB, 0x3D, 0xB1, /* 0xA8-0xAB */
	0x3D, 0xB3, 0xD7, 0xD4, 0xC8, 0xF9, 0x3D, 0xB4, /* 0xAC-0xAF */
	0x3D, 0xB5, 0x3D, 0xB6, 0x00, 0x00, 0xC6, 0xC1, /* 0xB0-0xB3 */
	0xC4, 0xA7, 0x00, 0x00, 0x00, 0x00, 0xF9, 0xFE, /* 0xB4-0xB7 */
	0x3D, 0xB7, 0xC5, 0xB0, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0xD7, 0xD5, 0xB5, 0xAB, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBF, 0xB4, /* 0xC0-0xC3 */
	0x3D, 0xB8, 0xC9, 0xAC, 0x00, 0x00, 0x3D, 0xB9, /* 0xC4-0xC7 */
	0x3D, 0xBA, 0x3D, 0xBB, 0x00, 0x00, 0x3D, 0xBC, /* 0xC8-0xCB */
	0xB4, 0xF7, 0xC7, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x3D, 0xBD, 0x3D, 0xBE, 0x3D, 0xBF, 0x3D, 0xC0, /* 0xD0-0xD3 */
	0x3D, 0xC1, 0x00, 0x00, 0xD7, 0xD6, 0xBB, 0xD6, /* 0xD4-0xD7 */
	0xCB, 0xBA, 0xCB, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0xB1, 0xFE, 0xD7, 0xDB, 0xFA, 0xA1, 0x00, 0x00, /* 0xDC-0xDF */
	0xC3, 0xE9, 0x3D, 0xC3, 0x3D, 0xC4, 0x00, 0x00, /* 0xE0-0xE3 */
	0xD7, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x3D, 0xC5, 0x3D, 0xC6, 0x3D, 0xC7, 0xB2, 0xF7, /* 0xE8-0xEB */
	0x3D, 0xC8, 0x3D, 0xC9, 0x3D, 0xCA, 0x3D, 0xCB, /* 0xEC-0xEF */
	0xD8, 0xAD, 0xD7, 0xDA, 0x3D, 0xCC, 0x3D, 0xCD, /* 0xF0-0xF3 */
	0x00, 0x00, 0xC7, 0xB0, 0x3D, 0xCE, 0x00, 0x00, /* 0xF4-0xF7 */
	0xD7, 0xD9, 0x00, 0x00, 0x3D, 0xCF, 0xD7, 0xD7, /* 0xF8-0xFB */
	0x3D, 0xD0, 0xB9, 0xFA, 0x00, 0x00, 0xD7, 0xDD, /* 0xFC-0xFF */
};

static unsigned char u2c_60[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3D, 0xD1, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x3D, 0xD2, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x3D, 0xD3, 0xD7, 0xE3, 0xD7, 0xE9, /* 0x0C-0x0F */
	0xD7, 0xE1, 0x00, 0x00, 0xC5, 0xDC, 0x3D, 0xD4, /* 0x10-0x13 */
	0x3D, 0xD5, 0xD7, 0xE6, 0xC9, 0xDD, 0x3D, 0xD6, /* 0x14-0x17 */
	0x3D, 0xD7, 0xD7, 0xE0, 0x3D, 0xD8, 0xD7, 0xE5, /* 0x18-0x1B */
	0xCE, 0xE7, 0xBB, 0xD7, 0x00, 0x00, 0x3D, 0xD9, /* 0x1C-0x1F */
	0xC2, 0xD5, 0xD7, 0xDE, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x3D, 0xDA, 0xB5, 0xDE, 0xD7, 0xE8, 0xC0, 0xAD, /* 0x24-0x27 */
	0xB1, 0xE5, 0xD7, 0xE2, 0xB2, 0xF8, 0xD7, 0xE7, /* 0x28-0x2B */
	0x00, 0x00, 0x3D, 0xDB, 0x00, 0x00, 0xB6, 0xB1, /* 0x2C-0x2F */
	0x00, 0x00, 0xD7, 0xE4, 0x00, 0x00, 0x3D, 0xDC, /* 0x30-0x33 */
	0x00, 0x00, 0x3D, 0xDD, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xD7, 0xEA, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x3D, 0xDE, 0xD7, 0xEC, 0xD7, 0xF6, 0xD7, 0xF4, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0xD7, 0xF1, 0x3D, 0xDF, /* 0x44-0x47 */
	0x3D, 0xE0, 0x3D, 0xE1, 0xD7, 0xF0, 0xCE, 0xF8, /* 0x48-0x4B */
	0x3D, 0xE2, 0xD7, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0xB6, 0xB2, 0x3D, 0xE3, 0xB9, 0xB1, 0x00, 0x00, /* 0x50-0x53 */
	0x3D, 0xE4, 0xBD, 0xFA, 0x3D, 0xE5, 0x3D, 0xE6, /* 0x54-0x57 */
	0x00, 0x00, 0xD7, 0xF9, 0xD7, 0xEB, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0xFA, 0xA2, 0x00, 0x00, 0xD7, 0xEF, /* 0x5C-0x5F */
	0xD7, 0xDF, 0x3D, 0xE8, 0xB2, 0xFA, 0xD7, 0xF3, /* 0x60-0x63 */
	0xD7, 0xF5, 0xC3, 0xD1, 0x00, 0x00, 0x3D, 0xE9, /* 0x64-0x67 */
	0xBA, 0xA8, 0xB2, 0xB8, 0xD7, 0xED, 0xD7, 0xF8, /* 0x68-0x6B */
	0xD7, 0xF7, 0xB6, 0xB3, 0x00, 0x00, 0xC2, 0xA9, /* 0x6C-0x6F */
	0xB3, 0xE6, 0x3D, 0xEA, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xB7, 0xC3, 0x00, 0x00, 0xD7, 0xEE, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x3D, 0xEB, 0x3D, 0xEC, /* 0x7C-0x7F */

	0x00, 0x00, 0xD7, 0xFA, 0x3D, 0xED, 0xD7, 0xFD, /* 0x80-0x83 */
	0xD8, 0xA1, 0xFA, 0xA3, 0x3D, 0xEE, 0x00, 0x00, /* 0x84-0x87 */
	0x3D, 0xEF, 0xBC, 0xBD, 0xFA, 0xA4, 0xD8, 0xA7, /* 0x88-0x8B */
	0xC4, 0xF0, 0xD7, 0xFB, 0x3D, 0xF1, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x3D, 0xF2, 0xD8, 0xA5, 0x3D, 0xF3, /* 0x90-0x93 */
	0xB2, 0xF9, 0x3D, 0xF4, 0xD8, 0xA3, 0xD8, 0xA4, /* 0x94-0x97 */
	0x3D, 0xF5, 0x00, 0x00, 0xD7, 0xFE, 0xD8, 0xA2, /* 0x98-0x9B */
	0x00, 0x00, 0x3D, 0xF6, 0x3D, 0xF7, 0xB8, 0xE7, /* 0x9C-0x9F */
	0xCD, 0xAA, 0x00, 0x00, 0x3D, 0xF8, 0xB4, 0xB5, /* 0xA0-0xA3 */
	0x3D, 0xF9, 0x3D, 0xFA, 0xB1, 0xD9, 0xD8, 0xA6, /* 0xA4-0xA7 */
	0x3D, 0xFB, 0xC7, 0xBA, 0xB0, 0xAD, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x3D, 0xFC, 0x3D, 0xFD, 0xC8, 0xE1, 0xD7, 0xDC, /* 0xB0-0xB3 */
	0xD8, 0xAC, 0xD8, 0xB0, 0xCC, 0xE5, 0x3D, 0xFE, /* 0xB4-0xB7 */
	0xD8, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x3E, 0xA1, /* 0xB8-0xBB */
	0xC5, 0xE9, 0xD8, 0xAE, 0x3E, 0xA2, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x3E, 0xA3, 0x00, 0x00, /* 0xC0-0xC3 */
	0x3E, 0xA4, 0xBE, 0xF0, 0xD8, 0xAF, 0xC6, 0xD7, /* 0xC4-0xC7 */
	0x3E, 0xA5, 0x3E, 0xA6, 0x3E, 0xA7, 0x3E, 0xA8, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x3E, 0xA9, 0x3E, 0xAA, /* 0xCC-0xCF */
	0x00, 0x00, 0xCF, 0xC7, 0x00, 0x00, 0xD8, 0xAB, /* 0xD0-0xD3 */
	0x3E, 0xAB, 0xFA, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xD8, 0xB1, 0x3E, 0xAD, 0xB9, 0xFB, 0x3E, 0xAE, /* 0xD8-0xDB */
	0xC0, 0xCB, 0x3E, 0xAF, 0xFA, 0xA5, 0xB0, 0xD4, /* 0xDC-0xDF */
	0xD8, 0xAA, 0xD8, 0xA8, 0x3E, 0xB1, 0xC1, 0xDA, /* 0xE0-0xE3 */
	0x00, 0x00, 0x3E, 0xB2, 0x00, 0x00, 0xD7, 0xFC, /* 0xE4-0xE7 */
	0xBB, 0xB4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0xC2, 0xC6, 0xD8, 0xBD, 0xFA, 0xA8, 0xC1, 0xDB, /* 0xF0-0xF3 */
	0xD8, 0xB8, 0x3E, 0xB4, 0xD8, 0xB5, 0xD8, 0xB6, /* 0xF4-0xF7 */
	0x3E, 0xB5, 0xBC, 0xE6, 0xD8, 0xB9, 0xD8, 0xBC, /* 0xF8-0xFB */
	0x3E, 0xB6, 0x3E, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_61[512] = {
	0xD8, 0xB7, 0xBD, 0xA5, 0x3E, 0xB8, 0xD8, 0xBA, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xD8, 0xB4, 0x3E, 0xB9, /* 0x04-0x07 */
	0xCC, 0xFC, 0xCC, 0xFB, 0x3E, 0xBA, 0x00, 0x00, /* 0x08-0x0B */
	0x3E, 0xBB, 0xD8, 0xBE, 0xD8, 0xBF, 0xB0, 0xD5, /* 0x0C-0x0F */
	0x3E, 0xBC, 0xFA, 0xA9, 0x3E, 0xBE, 0x3E, 0xBF, /* 0x10-0x13 */
	0x3E, 0xC0, 0xD8, 0xB3, 0x3E, 0xC1, 0x3E, 0xC2, /* 0x14-0x17 */
	0x00, 0x00, 0x3E, 0xC3, 0xB6, 0xF2, 0xB0, 0xA6, /* 0x18-0x1B */
	0x3E, 0xC4, 0x00, 0x00, 0x3E, 0xC5, 0xB4, 0xB6, /* 0x1C-0x1F */
	0xFA, 0xA7, 0xD8, 0xBB, 0x3E, 0xC6, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8, 0xC3, /* 0x24-0x27 */
	0xD8, 0xC2, 0x00, 0x00, 0x3E, 0xC7, 0x3E, 0xC8, /* 0x28-0x2B */
	0xD8, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xFA, 0xAB, 0x3E, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0xD8, 0xC8, 0x3E, 0xCB, 0x3E, 0xCC, 0xFA, 0xAA, /* 0x34-0x37 */
	0x00, 0x00, 0x3E, 0xCE, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0xD8, 0xC6, 0xD8, 0xC9, 0xD8, 0xC1, 0xD8, 0xC5, /* 0x3C-0x3F */
	0x00, 0x00, 0x3E, 0xCF, 0xD8, 0xCA, 0x00, 0x00, /* 0x40-0x43 */
	0xD8, 0xCB, 0x3E, 0xD0, 0x3E, 0xD1, 0xD8, 0xC0, /* 0x44-0x47 */
	0xBB, 0xFC, 0x3E, 0xD2, 0xD8, 0xC4, 0xC2, 0xD6, /* 0x48-0x4B */
	0xB9, 0xB2, 0xD8, 0xB2, 0xBF, 0xB5, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8, 0xD8, /* 0x50-0x53 */
	0x00, 0x00, 0xCA, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0xD8, 0xCE, 0xD8, 0xCF, 0xD8, 0xD0, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0xD8, 0xD7, 0x3E, 0xD3, 0xD8, 0xD6, /* 0x5C-0x5F */
	0x3E, 0xD4, 0x00, 0x00, 0xCB, 0xFD, 0xB4, 0xB7, /* 0x60-0x63 */
	0x00, 0x00, 0xD8, 0xD4, 0x00, 0x00, 0xB7, 0xC5, /* 0x64-0x67 */
	0xB3, 0xB4, 0x00, 0x00, 0x00, 0x00, 0xD8, 0xD1, /* 0x68-0x6B */
	0x3E, 0xD5, 0x00, 0x00, 0xCE, 0xB8, 0xD8, 0xD3, /* 0x6C-0x6F */
	0xB0, 0xD6, 0xD8, 0xD5, 0x3E, 0xD6, 0xD8, 0xCC, /* 0x70-0x73 */
	0xD8, 0xD2, 0xD8, 0xD9, 0xB7, 0xC4, 0xD8, 0xCD, /* 0x74-0x77 */
	0x3E, 0xD7, 0x00, 0x00, 0x00, 0x00, 0x3E, 0xD8, /* 0x78-0x7B */
	0x3E, 0xD9, 0x00, 0x00, 0xCD, 0xDD, 0x3E, 0xDA, /* 0x7C-0x7F */

	0x3E, 0xDB, 0x3E, 0xDC, 0xCD, 0xAB, 0x3E, 0xDD, /* 0x80-0x83 */
	0x3E, 0xDE, 0x00, 0x00, 0x00, 0x00, 0xD8, 0xDC, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xD8, 0xE0, 0x3E, 0xDF, /* 0x88-0x8B */
	0x00, 0x00, 0x3E, 0xE0, 0xC1, 0xFE, 0x00, 0x00, /* 0x8C-0x8F */
	0xCE, 0xF9, 0xD8, 0xE1, 0x3E, 0xE1, 0x3E, 0xE2, /* 0x90-0x93 */
	0xD8, 0xDE, 0x00, 0x00, 0xD8, 0xDB, 0x3E, 0xE3, /* 0x94-0x97 */
	0xFA, 0xAC, 0xD8, 0xDA, 0xD8, 0xDF, 0x00, 0x00, /* 0x98-0x9B */
	0x3E, 0xE5, 0x3E, 0xE6, 0x00, 0x00, 0x3E, 0xE7, /* 0x9C-0x9F */
	0x3E, 0xE8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0xCA, 0xB0, 0x3E, 0xE9, 0x00, 0x00, 0xC6, 0xB4, /* 0xA4-0xA7 */
	0x3E, 0xEA, 0xB7, 0xC6, 0x3E, 0xEB, 0xD8, 0xE2, /* 0xA8-0xAB */
	0xD8, 0xDD, 0x3E, 0xEC, 0xD8, 0xE3, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xB7, 0xFB, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0xB2, 0xB1, 0x00, 0x00, /* 0xB4-0xB7 */
	0x3E, 0xED, 0x3E, 0xEE, 0xD8, 0xEB, 0x00, 0x00, /* 0xB8-0xBB */
	0x3E, 0xEF, 0x00, 0x00, 0xB4, 0xB8, 0x00, 0x00, /* 0xBC-0xBF */
	0x3E, 0xF0, 0x3E, 0xF1, 0x3E, 0xF2, 0xD8, 0xE9, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0xD8, 0xEA, 0xBA, 0xA9, /* 0xC4-0xC7 */
	0xD8, 0xE8, 0xD8, 0xE6, 0xD8, 0xE5, 0xD8, 0xEC, /* 0xC8-0xCB */
	0xD8, 0xE4, 0xD8, 0xEE, 0x3E, 0xF3, 0x3E, 0xF4, /* 0xCC-0xCF */
	0xB2, 0xFB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x3E, 0xF5, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x3E, 0xF6, 0x3E, 0xF7, 0x3E, 0xF8, 0x3E, 0xF9, /* 0xDC-0xDF */
	0x00, 0x00, 0x3E, 0xFA, 0x3E, 0xFB, 0xD8, 0xF0, /* 0xE0-0xE3 */
	0x00, 0x00, 0x3E, 0xFE, 0xD8, 0xEF, 0x3E, 0xFC, /* 0xE4-0xE7 */
	0x00, 0x00, 0x3E, 0xFD, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x3F, 0xA1, 0x3F, 0xA2, 0x00, 0x00, 0x3F, 0xA3, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0xC4, 0xA8, 0x00, 0x00, /* 0xF0-0xF3 */
	0xD8, 0xF3, 0x00, 0x00, 0xD8, 0xF1, 0xD8, 0xE7, /* 0xF4-0xF7 */
	0xB7, 0xFC, 0x00, 0x00, 0xD8, 0xF2, 0x00, 0x00, /* 0xF8-0xFB */
	0xD8, 0xF6, 0xD8, 0xF5, 0xD8, 0xF7, 0xD8, 0xF4, /* 0xFC-0xFF */
};

static unsigned char u2c_62[512] = {
	0xD8, 0xF8, 0x3F, 0xA4, 0x00, 0x00, 0x3F, 0xA5, /* 0x00-0x03 */
	0x3F, 0xA6, 0x00, 0x00, 0x00, 0x00, 0x3F, 0xA7, /* 0x04-0x07 */
	0xD8, 0xF9, 0xD8, 0xFA, 0xCA, 0xEA, 0x00, 0x00, /* 0x08-0x0B */
	0xD8, 0xFC, 0xD8, 0xFB, 0xBD, 0xBF, 0x00, 0x00, /* 0x0C-0x0F */
	0xC0, 0xAE, 0xB2, 0xE6, 0xB2, 0xFC, 0xFA, 0xAD, /* 0x10-0x13 */
	0xD8, 0xFD, 0x3F, 0xA9, 0xB0, 0xBF, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0xC0, 0xCC, 0xD8, 0xFE, /* 0x18-0x1B */
	0x3F, 0xAA, 0xEC, 0xC3, 0xD9, 0xA1, 0xB7, 0xE1, /* 0x1C-0x1F */
	0x3F, 0xAB, 0xD9, 0xA2, 0x3F, 0xAC, 0x3F, 0xAD, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xC0, 0xEF, 0x3F, 0xAE, /* 0x24-0x27 */
	0x00, 0x00, 0x3F, 0xAF, 0xD9, 0xA3, 0x3F, 0xB0, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0xD9, 0xA4, 0xB5, 0xBA, /* 0x2C-0x2F */
	0xD9, 0xA5, 0x00, 0x00, 0xD9, 0xA6, 0xD9, 0xA7, /* 0x30-0x33 */
	0xC2, 0xD7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0xB8, 0xCD, 0x3F, 0xB1, 0x00, 0x00, 0xCC, 0xE1, /* 0x38-0x3B */
	0x00, 0x00, 0x3F, 0xB2, 0x00, 0x00, 0xCB, 0xBC, /* 0x3C-0x3F */
	0xBD, 0xEA, 0xD9, 0xA8, 0x3F, 0xB3, 0x3F, 0xB4, /* 0x40-0x43 */
	0x3F, 0xB5, 0x00, 0x00, 0x3F, 0xB6, 0xC0, 0xF0, /* 0x44-0x47 */
	0xEE, 0xBD, 0xC8, 0xE2, 0x00, 0x00, 0xBC, 0xEA, /* 0x48-0x4B */
	0x3F, 0xB7, 0xBA, 0xCD, 0xD9, 0xA9, 0x00, 0x00, /* 0x4C-0x4F */
	0x3F, 0xB8, 0x3F, 0xB9, 0x3F, 0xBA, 0xC2, 0xC7, /* 0x50-0x53 */
	0x3F, 0xBB, 0xCA, 0xA7, 0x3F, 0xBC, 0x00, 0x00, /* 0x54-0x57 */
	0xC2, 0xF1, 0x00, 0x00, 0x3F, 0xBD, 0xD9, 0xAC, /* 0x58-0x5B */
	0x3F, 0xBE, 0x00, 0x00, 0xD9, 0xAA, 0x00, 0x00, /* 0x5C-0x5F */
	0xD9, 0xAD, 0x00, 0x00, 0x00, 0x00, 0xD9, 0xAB, /* 0x60-0x63 */
	0x3F, 0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0xD9, 0xAE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x3F, 0xC0, 0xCA, 0xB1, 0x3F, 0xC1, /* 0x6C-0x6F */
	0x00, 0x00, 0xB0, 0xB7, 0x00, 0x00, 0x3F, 0xC2, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0xC9, 0xDE, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0xC8, 0xE3, 0x3F, 0xC3, 0x00, 0x00, /* 0x78-0x7B */
	0xD9, 0xAF, 0x3F, 0xC4, 0xD9, 0xB2, 0xBE, 0xB5, /* 0x7C-0x7F */

	0xB5, 0xBB, 0x00, 0x00, 0xD9, 0xB0, 0xD9, 0xB7, /* 0x80-0x83 */
	0xBE, 0xB6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0xD9, 0xB1, 0xC7, 0xC4, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x3F, 0xC5, 0x3F, 0xC6, 0x3F, 0xC7, /* 0x8C-0x8F */
	0x3F, 0xC8, 0xCD, 0xDE, 0xD9, 0xB3, 0xD9, 0xB4, /* 0x90-0x93 */
	0xD9, 0xB8, 0xC5, 0xEA, 0xD9, 0xB5, 0xB9, 0xB3, /* 0x94-0x97 */
	0xC0, 0xDE, 0x00, 0x00, 0x00, 0x00, 0xD9, 0xC6, /* 0x98-0x9B */
	0xC8, 0xB4, 0x00, 0x00, 0xC2, 0xF2, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0xFA, 0xAE, 0x00, 0x00, /* 0xA4-0xA7 */
	0x3F, 0xCA, 0x00, 0x00, 0x00, 0x00, 0xC8, 0xE4, /* 0xA8-0xAB */
	0xDA, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0xCA, 0xFA, 0x00, 0x00, 0x3F, 0xCB, /* 0xB0-0xB3 */
	0x00, 0x00, 0xC4, 0xF1, 0x3F, 0xCC, 0x3F, 0xCD, /* 0xB4-0xB7 */
	0x00, 0x00, 0xCB, 0xF5, 0x3F, 0xCE, 0xD9, 0xBB, /* 0xB8-0xBB */
	0xB2, 0xA1, 0xC3, 0xEA, 0x3F, 0xCF, 0x3F, 0xD0, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0xD9, 0xC4, 0x00, 0x00, /* 0xC0-0xC3 */
	0x3F, 0xD1, 0xC3, 0xB4, 0xD9, 0xBE, 0xD9, 0xC5, /* 0xC4-0xC7 */
	0xD9, 0xC0, 0xD9, 0xC7, 0xD9, 0xC3, 0x00, 0x00, /* 0xC8-0xCB */
	0xD9, 0xC2, 0xC7, 0xEF, 0x3F, 0xD2, 0xD9, 0xBC, /* 0xCC-0xCF */
	0xB2, 0xFD, 0xD9, 0xBA, 0xB5, 0xF1, 0xC2, 0xF3, /* 0xD0-0xD3 */
	0xD9, 0xB6, 0x3F, 0xD3, 0x3F, 0xD4, 0xD9, 0xB9, /* 0xD4-0xD7 */
	0xB9, 0xB4, 0xC0, 0xDB, 0x3F, 0xD5, 0xBE, 0xB7, /* 0xD8-0xDB */
	0xD9, 0xC1, 0xC7, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0xB5, 0xF2, 0xB3, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x3F, 0xD6, 0x00, 0x00, /* 0xE8-0xEB */
	0xB3, 0xE7, 0xBF, 0xA1, 0xD9, 0xC9, 0xD9, 0xCE, /* 0xEC-0xEF */
	0x00, 0x00, 0xD9, 0xCA, 0x3F, 0xD7, 0xB7, 0xFD, /* 0xF0-0xF3 */
	0x3F, 0xD8, 0xD9, 0xCF, 0xBB, 0xA2, 0xB9, 0xE9, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0x3F, 0xD9, 0x3F, 0xDA, 0xBD, 0xA6, 0xD9, 0xBD, /* 0xFC-0xFF */
};

static unsigned char u2c_63[512] = {
	0x00, 0x00, 0xBB, 0xFD, 0xD9, 0xCC, 0x3F, 0xDB, /* 0x00-0x03 */
	0x3F, 0xDC, 0x00, 0x00, 0x00, 0x00, 0xBB, 0xD8, /* 0x04-0x07 */
	0xD9, 0xCD, 0xB0, 0xC4, 0x3F, 0xDD, 0x3F, 0xDE, /* 0x08-0x0B */
	0xD9, 0xC8, 0x3F, 0xDF, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x3F, 0xE0, 0xC4, 0xA9, 0x00, 0x00, 0x3F, 0xE1, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x3F, 0xE2, 0x00, 0x00, /* 0x14-0x17 */
	0x3F, 0xE3, 0xB5, 0xF3, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB6, 0xB4, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD9, 0xCB, /* 0x24-0x27 */
	0xB0, 0xA7, 0x3F, 0xE4, 0x3F, 0xE5, 0xBA, 0xC3, /* 0x28-0x2B */
	0x00, 0x00, 0x3F, 0xE6, 0x00, 0x00, 0xBF, 0xB6, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x3F, 0xE7, 0x3F, 0xE8, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x3F, 0xE9, 0xC4, 0xF2, 0x00, 0x00, /* 0x38-0x3B */
	0x3F, 0xEA, 0xC8, 0xD4, 0xD9, 0xD1, 0xC1, 0xDE, /* 0x3C-0x3F */
	0x00, 0x00, 0x3F, 0xEB, 0x3F, 0xEC, 0x3F, 0xED, /* 0x40-0x43 */
	0x3F, 0xEE, 0x00, 0x00, 0x3F, 0xEF, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0xC2, 0xAA, 0x3F, 0xF0, 0x3F, 0xF1, /* 0x48-0x4B */
	0xBB, 0xAB, 0xD9, 0xD2, 0x3F, 0xF2, 0xD9, 0xD4, /* 0x4C-0x4F */
	0xD9, 0xD0, 0x00, 0x00, 0x3F, 0xF3, 0x3F, 0xF4, /* 0x50-0x53 */
	0x3F, 0xF5, 0xCA, 0xE1, 0x00, 0x00, 0xC4, 0xBD, /* 0x54-0x57 */
	0x3F, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x3F, 0xF7, /* 0x58-0x5B */
	0xC1, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x3F, 0xF8, 0x3F, 0xF9, 0xCA, 0xFB, /* 0x64-0x67 */
	0xBC, 0xCE, 0xD9, 0xE0, 0x00, 0x00, 0xD9, 0xDF, /* 0x68-0x6B */
	0x3F, 0xFA, 0x3F, 0xFB, 0xBF, 0xF8, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x3F, 0xFC, 0xB7, 0xFE, 0x00, 0x00, /* 0x70-0x73 */
	0x3F, 0xFD, 0x3F, 0xFE, 0xD9, 0xD9, 0xBE, 0xB9, /* 0x74-0x77 */
	0x40, 0xA1, 0x00, 0x00, 0xC6, 0xE8, 0xC7, 0xB1, /* 0x78-0x7B */
	0x40, 0xA2, 0x40, 0xA3, 0x00, 0x00, 0x40, 0xA4, /* 0x7C-0x7F */

	0xD9, 0xD7, 0x00, 0x00, 0x40, 0xA5, 0xC1, 0xDD, /* 0x80-0x83 */
	0x40, 0xA6, 0x00, 0x00, 0x00, 0x00, 0x40, 0xA7, /* 0x84-0x87 */
	0xBC, 0xF8, 0xD9, 0xDC, 0x40, 0xA8, 0x00, 0x00, /* 0x88-0x8B */
	0xBE, 0xB8, 0x00, 0x00, 0xD9, 0xD6, 0xD9, 0xDB, /* 0x8C-0x8F */
	0x40, 0xA9, 0x00, 0x00, 0xC7, 0xD3, 0x00, 0x00, /* 0x90-0x93 */
	0x40, 0xAA, 0x40, 0xAB, 0xD9, 0xD5, 0x00, 0x00, /* 0x94-0x97 */
	0xB7, 0xA1, 0x40, 0xAC, 0x40, 0xAD, 0xB3, 0xDD, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x40, 0xAE, 0xD9, 0xDD, /* 0x9C-0x9F */
	0xCE, 0xAB, 0xBA, 0xCE, 0xC3, 0xB5, 0xD9, 0xDA, /* 0xA0-0xA3 */
	0x40, 0xAF, 0xC0, 0xDC, 0x40, 0xB0, 0xB9, 0xB5, /* 0xA4-0xA7 */
	0xBF, 0xE4, 0xB1, 0xE6, 0xC1, 0xBC, 0xD9, 0xD8, /* 0xA8-0xAB */
	0xB5, 0xC5, 0x40, 0xB1, 0x40, 0xB2, 0x40, 0xB3, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xB7, 0xC7, 0x00, 0x00, /* 0xB0-0xB3 */
	0xC4, 0xCF, 0xD9, 0xDE, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC1, 0xDF, /* 0xB8-0xBB */
	0x00, 0x00, 0x40, 0xB4, 0xD9, 0xE1, 0x00, 0x00, /* 0xBC-0xBF */
	0xD9, 0xE3, 0x40, 0xB5, 0x00, 0x00, 0xC2, 0xB7, /* 0xC0-0xC3 */
	0xD9, 0xE9, 0x40, 0xB6, 0xD9, 0xE4, 0x00, 0x00, /* 0xC4-0xC7 */
	0x40, 0xB7, 0xD9, 0xE6, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x40, 0xB8, 0xC9, 0xC1, /* 0xCC-0xCF */
	0xC4, 0xF3, 0x40, 0xB9, 0xD9, 0xE7, 0x40, 0xBA, /* 0xD0-0xD3 */
	0x40, 0xBB, 0x40, 0xBC, 0xCD, 0xAC, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xCD, 0xC8, 0xB4, 0xB9, /* 0xD8-0xDB */
	0x40, 0xBD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x40, 0xBE, 0xB0, 0xAE, 0x00, 0x00, 0xD9, 0xE5, /* 0xE0-0xE3 */
	0x00, 0x00, 0x40, 0xBF, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0xD9, 0xE2, 0x40, 0xC0, 0x00, 0x00, /* 0xE8-0xEB */
	0x40, 0xC1, 0x00, 0x00, 0xB4, 0xF8, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x40, 0xC2, 0x40, 0xC3, /* 0xF0-0xF3 */
	0xB1, 0xE7, 0xFA, 0xAF, 0xD9, 0xE8, 0x00, 0x00, /* 0xF4-0xF7 */
	0x40, 0xC5, 0x40, 0xC6, 0xCD, 0xC9, 0x00, 0x00, /* 0xF8-0xFB */
};

static unsigned char u2c_64[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xD9, 0xEC, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x40, 0xC7, 0x40, 0xC8, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0xC2, 0xBB, 0x00, 0x00, 0xD9, 0xF3, /* 0x0C-0x0F */
	0x40, 0xC9, 0x00, 0x00, 0x40, 0xCA, 0xD9, 0xED, /* 0x10-0x13 */
	0x40, 0xCB, 0x00, 0x00, 0xD9, 0xEA, 0xD9, 0xF1, /* 0x14-0x17 */
	0x40, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0xD9, 0xD3, 0x00, 0x00, 0x40, 0xCD, 0x00, 0x00, /* 0x1C-0x1F */
	0x40, 0xCE, 0x00, 0x00, 0x40, 0xCF, 0x00, 0x00, /* 0x20-0x23 */
	0x40, 0xD0, 0x40, 0xD1, 0xD9, 0xEE, 0x00, 0x00, /* 0x24-0x27 */
	0xD9, 0xF2, 0x40, 0xD2, 0x40, 0xD3, 0x00, 0x00, /* 0x28-0x2B */
	0xC8, 0xC2, 0xC5, 0xEB, 0x00, 0x00, 0x40, 0xD4, /* 0x2C-0x2F */
	0x40, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0xD9, 0xEB, 0x40, 0xD6, 0xD9, 0xEF, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xB7, 0xC8, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x40, 0xD7, 0xBA, 0xF1, 0x40, 0xD8, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0xC0, 0xDD, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xD9, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0xD9, 0xF7, 0x40, 0xDA, /* 0x4C-0x4F */
	0x00, 0x00, 0x40, 0xDB, 0x40, 0xDC, 0x40, 0xDD, /* 0x50-0x53 */
	0x40, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0xC5, 0xA6, 0x00, 0x00, 0x40, 0xDF, 0x40, 0xE0, /* 0x58-0x5B */
	0x40, 0xE1, 0x40, 0xE2, 0x00, 0x00, 0x40, 0xE3, /* 0x5C-0x5F */
	0xFA, 0xB0, 0x40, 0xE5, 0x00, 0x00, 0x40, 0xE6, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD9, 0xF4, /* 0x64-0x67 */
	0x00, 0x00, 0xCB, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x40, 0xE7, 0x00, 0x00, 0xD9, 0xF5, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xE8, /* 0x70-0x73 */
	0x40, 0xE9, 0x00, 0x00, 0xD9, 0xF6, 0x00, 0x00, /* 0x74-0x77 */
	0xCC, 0xCE, 0x00, 0x00, 0xC0, 0xA2, 0x40, 0xEA, /* 0x78-0x7B */
	0x00, 0x00, 0x40, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB7, 0xE2, /* 0x80-0x83 */
	0x00, 0x00, 0x40, 0xEC, 0x00, 0x00, 0x40, 0xED, /* 0x84-0x87 */
	0xD9, 0xFD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xEE, /* 0x8C-0x8F */
	0x40, 0xEF, 0x40, 0xF0, 0xBB, 0xB5, 0xD9, 0xFA, /* 0x90-0x93 */
	0x00, 0x00, 0xD9, 0xF9, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x40, 0xF1, 0x40, 0xF2, 0xC7, 0xB2, 0x40, 0xF3, /* 0x98-0x9B */
	0x00, 0x00, 0xFA, 0xB1, 0xC6, 0xB5, 0x40, 0xF5, /* 0x9C-0x9F */
	0x00, 0x00, 0x40, 0xF6, 0x00, 0x00, 0x40, 0xF7, /* 0xA0-0xA3 */
	0xC5, 0xB1, 0xD9, 0xFB, 0x40, 0xF8, 0x00, 0x00, /* 0xA4-0xA7 */
	0x40, 0xF9, 0xD9, 0xFC, 0x00, 0x00, 0xC9, 0xEF, /* 0xA8-0xAB */
	0x40, 0xFA, 0xC7, 0xC5, 0xBB, 0xA3, 0x00, 0x00, /* 0xAC-0xAF */
	0xC0, 0xF1, 0x00, 0x00, 0xCB, 0xD0, 0x40, 0xFB, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xB3, 0xC9, 0x00, 0x00, 0xDA, 0xA5, /* 0xB8-0xBB */
	0xD9, 0xFE, 0x40, 0xFC, 0x40, 0xFD, 0x40, 0xFE, /* 0xBC-0xBF */
	0x00, 0x00, 0xCD, 0xCA, 0xDA, 0xA7, 0x00, 0x00, /* 0xC0-0xC3 */
	0x41, 0xA1, 0xDA, 0xA3, 0x00, 0x00, 0xDA, 0xA4, /* 0xC4-0xC7 */
	0x00, 0x00, 0x41, 0xA2, 0x41, 0xA3, 0x41, 0xA4, /* 0xC8-0xCB */
	0x41, 0xA5, 0xC1, 0xE0, 0xFA, 0xB2, 0x00, 0x00, /* 0xCC-0xCF */
	0x41, 0xA7, 0x41, 0xA8, 0xDA, 0xA2, 0x00, 0x00, /* 0xD0-0xD3 */
	0xD9, 0xBF, 0x41, 0xA9, 0x00, 0x00, 0x41, 0xAA, /* 0xD4-0xD7 */
	0xDA, 0xA6, 0x00, 0x00, 0xDA, 0xA1, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0xDA, 0xAB, 0xDA, 0xAC, 0xC5, 0xA7, 0xDA, 0xAE, /* 0xE0-0xE3 */
	0x41, 0xAB, 0x41, 0xAC, 0xBB, 0xA4, 0xDA, 0xA9, /* 0xE4-0xE7 */
	0x00, 0x00, 0x41, 0xAD, 0x41, 0xAE, 0x00, 0x00, /* 0xE8-0xEB */
	0xB5, 0xBC, 0x41, 0xAF, 0x00, 0x00, 0xDA, 0xAF, /* 0xEC-0xEF */
	0x41, 0xB0, 0xDA, 0xA8, 0xDA, 0xB3, 0x00, 0x00, /* 0xF0-0xF3 */
	0xDA, 0xB2, 0x41, 0xB1, 0xDA, 0xB1, 0x41, 0xB2, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0xDA, 0xB4, 0x41, 0xB3, /* 0xF8-0xFB */
	0x00, 0x00, 0xDA, 0xB6, 0xBE, 0xF1, 0x41, 0xB4, /* 0xFC-0xFF */
};

static unsigned char u2c_65[512] = {
	0xDA, 0xB5, 0x41, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x41, 0xB6, 0xDA, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x41, 0xB7, 0x41, 0xB8, 0x41, 0xB9, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0xBA, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0xBB, /* 0x10-0x13 */
	0x41, 0xBC, 0x00, 0x00, 0x41, 0xBD, 0x00, 0x00, /* 0x14-0x17 */
	0xDA, 0xB7, 0x41, 0xBE, 0x00, 0x00, 0x41, 0xBF, /* 0x18-0x1B */
	0xDA, 0xB8, 0xD9, 0xF0, 0x41, 0xC0, 0x41, 0xC1, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x41, 0xC2, 0xDA, 0xBB, /* 0x20-0x23 */
	0xDA, 0xBA, 0x00, 0x00, 0x41, 0xC3, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x41, 0xC4, 0xD9, 0xF8, 0xDA, 0xBC, /* 0x28-0x2B */
	0xDA, 0xB0, 0x00, 0x00, 0x41, 0xC5, 0xBB, 0xD9, /* 0x2C-0x2F */
	0x00, 0x00, 0x41, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0xDA, 0xBD, 0xDA, 0xBE, 0xDA, 0xC0, 0xDA, 0xBF, /* 0x34-0x37 */
	0xDA, 0xC1, 0xB2, 0xFE, 0x41, 0xC7, 0xB9, 0xB6, /* 0x38-0x3B */
	0x41, 0xC8, 0x41, 0xC9, 0xCA, 0xFC, 0xC0, 0xAF, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0xCA, /* 0x40-0x43 */
	0x00, 0x00, 0xB8, 0xCE, 0x00, 0x00, 0x41, 0xCB, /* 0x44-0x47 */
	0xDA, 0xC3, 0x41, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0xDA, 0xC6, 0xFA, 0xB3, 0xC9, 0xD2, /* 0x4C-0x4F */
	0x41, 0xCD, 0xB5, 0xDF, 0x41, 0xCE, 0x00, 0x00, /* 0x50-0x53 */
	0x41, 0xCF, 0xDA, 0xC5, 0xDA, 0xC4, 0xC7, 0xD4, /* 0x54-0x57 */
	0xDA, 0xC7, 0xB6, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0xDA, 0xC9, 0xDA, 0xC8, 0x41, 0xD0, /* 0x5C-0x5F */
	0x41, 0xD1, 0x00, 0x00, 0xB4, 0xBA, 0xBB, 0xB6, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0xC6, 0xD8, 0x41, 0xD2, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0xD3, /* 0x68-0x6B */
	0xB7, 0xC9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xBF, 0xF4, 0x00, 0x00, 0xDA, 0xCA, 0x00, 0x00, /* 0x70-0x73 */
	0xC0, 0xB0, 0xC5, 0xA8, 0x00, 0x00, 0xC9, 0xDF, /* 0x74-0x77 */
	0xDA, 0xCB, 0x00, 0x00, 0x41, 0xD4, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x41, 0xD5, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x41, 0xD6, 0xDA, 0xCC, 0xDA, 0xCD, /* 0x80-0x83 */
	0x00, 0x00, 0x41, 0xD7, 0x00, 0x00, 0xCA, 0xB8, /* 0x84-0x87 */
	0xD5, 0xDD, 0xC0, 0xC6, 0x41, 0xD8, 0x00, 0x00, /* 0x88-0x8B */
	0xC9, 0xCC, 0x00, 0x00, 0xBA, 0xD8, 0x00, 0x00, /* 0x8C-0x8F */
	0xC8, 0xE5, 0xC8, 0xC3, 0x41, 0xD9, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x41, 0xDA, 0x00, 0x00, 0xC5, 0xCD, /* 0x94-0x97 */
	0x41, 0xDB, 0xCE, 0xC1, 0x00, 0x00, 0xDA, 0xCF, /* 0x98-0x9B */
	0xBC, 0xD0, 0x41, 0xDC, 0x00, 0x00, 0xDA, 0xD0, /* 0x9C-0x9F */
	0x41, 0xDD, 0xB0, 0xB6, 0x00, 0x00, 0x41, 0xDE, /* 0xA0-0xA3 */
	0xB6, 0xD4, 0xC0, 0xCD, 0x41, 0xDF, 0xC9, 0xE0, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDA, 0xD1, /* 0xA8-0xAB */
	0xBB, 0xC2, 0xC3, 0xC7, 0x41, 0xE0, 0xBB, 0xDB, /* 0xAC-0xAF */
	0xBF, 0xB7, 0x00, 0x00, 0x41, 0xE1, 0x41, 0xE2, /* 0xB0-0xB3 */
	0x41, 0xE3, 0x00, 0x00, 0x00, 0x00, 0xDA, 0xD2, /* 0xB4-0xB7 */
	0x00, 0x00, 0xCA, 0xFD, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0xB1, 0xF7, 0xBB, 0xDC, 0x00, 0x00, 0x41, 0xE4, /* 0xBC-0xBF */
	0x00, 0x00, 0xDA, 0xD5, 0x41, 0xE5, 0xDA, 0xD3, /* 0xC0-0xC3 */
	0xDA, 0xD6, 0xCE, 0xB9, 0xDA, 0xD4, 0x00, 0x00, /* 0xC4-0xC7 */
	0x41, 0xE6, 0x41, 0xE7, 0x00, 0x00, 0xC0, 0xFB, /* 0xC8-0xCB */
	0xDA, 0xD7, 0x00, 0x00, 0x41, 0xE8, 0xC2, 0xB2, /* 0xCC-0xCF */
	0x41, 0xE9, 0x00, 0x00, 0xDA, 0xD8, 0x00, 0x00, /* 0xD0-0xD3 */
	0x41, 0xEA, 0x00, 0x00, 0x41, 0xEB, 0xB4, 0xFA, /* 0xD4-0xD7 */
	0x41, 0xEC, 0xDA, 0xDA, 0x00, 0x00, 0xDA, 0xD9, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0xED, /* 0xDC-0xDF */
	0xDA, 0xDB, 0xDA, 0xDC, 0xB4, 0xFB, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0xC6, 0xFC, 0xC3, 0xB6, 0xB5, 0xEC, /* 0xE4-0xE7 */
	0xBB, 0xDD, 0xC1, 0xE1, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0xBD, 0xDC, 0xB0, 0xB0, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x41, 0xEE, 0xDA, 0xDD, 0x41, 0xEF, 0x00, 0x00, /* 0xF0-0xF3 */
	0x41, 0xF0, 0x41, 0xF1, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x41, 0xF2, 0xB2, 0xA2, 0xDA, 0xE1, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0x41, 0xF3, 0x41, 0xF4, /* 0xFC-0xFF */
};

static unsigned char u2c_66[512] = {
	0xFA, 0xB4, 0x00, 0x00, 0xB9, 0xB7, 0xDA, 0xE0, /* 0x00-0x03 */
	0x41, 0xF6, 0x00, 0x00, 0xBA, 0xAB, 0xBE, 0xBA, /* 0x04-0x07 */
	0x41, 0xF7, 0xFA, 0xB7, 0xDA, 0xDF, 0x00, 0x00, /* 0x08-0x0B */
	0xBE, 0xBB, 0x41, 0xF9, 0xCC, 0xC0, 0xBA, 0xAA, /* 0x0C-0x0F */
	0x00, 0x00, 0x41, 0xFA, 0x41, 0xFB, 0xB0, 0xD7, /* 0x10-0x13 */
	0xC0, 0xCE, 0xFA, 0xB5, 0x41, 0xFD, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0xDA, 0xE6, 0x41, 0xFE, 0xFA, 0xB9, 0xC0, 0xB1, /* 0x1C-0x1F */
	0xB1, 0xC7, 0x42, 0xA2, 0x42, 0xA3, 0x42, 0xA4, /* 0x20-0x23 */
	0xFA, 0xBA, 0xBD, 0xD5, 0x42, 0xA6, 0xCB, 0xE6, /* 0x24-0x27 */
	0xBA, 0xF2, 0x42, 0xA7, 0x42, 0xA8, 0x42, 0xA9, /* 0x28-0x2B */
	0x42, 0xAA, 0xBE, 0xBC, 0xFA, 0xB8, 0xC0, 0xA7, /* 0x2C-0x2F */
	0x42, 0xAC, 0xF9, 0xA8, 0x00, 0x00, 0x42, 0xAE, /* 0x30-0x33 */
	0xDA, 0xE5, 0xDA, 0xE3, 0xDA, 0xE4, 0x42, 0xB0, /* 0x34-0x37 */
	0x00, 0x00, 0x42, 0xAF, 0x00, 0x00, 0xFA, 0xB6, /* 0x38-0x3B */
	0xC3, 0xEB, 0x00, 0x00, 0x00, 0x00, 0xDB, 0xA6, /* 0x3C-0x3F */
	0x42, 0xB1, 0xDA, 0xEA, 0xBB, 0xFE, 0xB9, 0xB8, /* 0x40-0x43 */
	0xDA, 0xE8, 0x42, 0xB2, 0x42, 0xB3, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0xDA, 0xE9, 0x42, 0xB4, 0xBF, 0xB8, /* 0x48-0x4B */
	0x42, 0xB5, 0x00, 0x00, 0x42, 0xB7, 0xDA, 0xE7, /* 0x4C-0x4F */
	0x00, 0x00, 0x42, 0xB6, 0xBB, 0xAF, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFA, 0xBC, /* 0x54-0x57 */
	0x42, 0xB9, 0xFA, 0xBD, 0x00, 0x00, 0x42, 0xBB, /* 0x58-0x5B */
	0x42, 0xBC, 0xDA, 0xEC, 0xDA, 0xEB, 0xDA, 0xF0, /* 0x5C-0x5F */
	0x42, 0xBD, 0x42, 0xBE, 0xDA, 0xF1, 0x00, 0x00, /* 0x60-0x63 */
	0xDA, 0xED, 0xFA, 0xBB, 0xB3, 0xA2, 0xDA, 0xEE, /* 0x64-0x67 */
	0xDA, 0xEF, 0xC8, 0xD5, 0x42, 0xC0, 0x42, 0xC1, /* 0x68-0x6B */
	0x42, 0xC2, 0x00, 0x00, 0xC9, 0xE1, 0xB7, 0xCA, /* 0x6C-0x6F */
	0xDA, 0xF2, 0x00, 0x00, 0x00, 0x00, 0xFA, 0xBF, /* 0x70-0x73 */
	0xC0, 0xB2, 0x42, 0xC5, 0xBE, 0xBD, 0x42, 0xC7, /* 0x74-0x77 */
	0x42, 0xC8, 0x42, 0xC9, 0xC3, 0xD2, 0x42, 0xCA, /* 0x78-0x7B */
	0x42, 0xCC, 0x00, 0x00, 0x42, 0xC3, 0x42, 0xC6, /* 0x7C-0x7F */

	0x42, 0xCB, 0xB6, 0xC7, 0x00, 0x00, 0xDA, 0xF3, /* 0x80-0x83 */
	0xDA, 0xF7, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xCB, /* 0x84-0x87 */
	0xDA, 0xF4, 0xDA, 0xF6, 0x00, 0x00, 0x42, 0xCD, /* 0x88-0x8B */
	0x42, 0xCE, 0x42, 0xCF, 0xDA, 0xF5, 0x00, 0x00, /* 0x8C-0x8F */
	0x42, 0xD0, 0xBD, 0xEB, 0x42, 0xD1, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xC3, 0xC8, 0xB0, 0xC5, /* 0x94-0x97 */
	0xDA, 0xF8, 0xFA, 0xC0, 0x42, 0xD3, 0x42, 0xD4, /* 0x98-0x9B */
	0x42, 0xD5, 0xDA, 0xF9, 0x00, 0x00, 0x42, 0xD6, /* 0x9C-0x9F */
	0xFA, 0xC1, 0x00, 0x00, 0xC4, 0xAA, 0x00, 0x00, /* 0xA0-0xA3 */
	0x42, 0xD8, 0x00, 0x00, 0xCE, 0xF1, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBB, 0xC3, /* 0xA8-0xAB */
	0x00, 0x00, 0x42, 0xD9, 0xCA, 0xEB, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x42, 0xDA, 0xFA, 0xC2, 0x00, 0x00, /* 0xB0-0xB3 */
	0xCB, 0xBD, 0x42, 0xDC, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xDB, 0xA2, 0xDA, 0xFB, 0x00, 0x00, 0x42, 0xDD, /* 0xB8-0xBB */
	0xDA, 0xFE, 0x00, 0x00, 0xDA, 0xFD, 0xFA, 0xC3, /* 0xBC-0xBF */
	0x42, 0xDF, 0xDA, 0xFA, 0x42, 0xE0, 0x42, 0xE1, /* 0xC0-0xC3 */
	0xDB, 0xA1, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xDE, /* 0xC4-0xC7 */
	0x42, 0xE2, 0xDA, 0xFC, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x42, 0xE3, 0x00, 0x00, 0x42, 0xE4, 0x42, 0xE5, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x42, 0xE6, 0x00, 0x00, 0xDB, 0xA3, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0xBD, 0xEC, 0xDB, 0xA4, 0x42, 0xE7, /* 0xD8-0xDB */
	0xCD, 0xCB, 0xC7, 0xF8, 0x00, 0x00, 0x42, 0xE8, /* 0xDC-0xDF */
	0xDB, 0xA5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0xDB, 0xA7, 0x00, 0x00, /* 0xE4-0xE7 */
	0x42, 0xE9, 0xDB, 0xA8, 0x00, 0x00, 0x42, 0xEA, /* 0xE8-0xEB */
	0x42, 0xEB, 0x00, 0x00, 0x42, 0xEC, 0x00, 0x00, /* 0xEC-0xEF */
	0xDB, 0xA9, 0x00, 0x00, 0xB6, 0xCA, 0xB1, 0xC8, /* 0xF0-0xF3 */
	0xB9, 0xB9, 0xDB, 0xAA, 0x00, 0x00, 0xDB, 0xAB, /* 0xF4-0xF7 */
	0xBD, 0xF1, 0xC1, 0xE2, 0xFA, 0xC4, 0xF9, 0xAB, /* 0xF8-0xFB */
	0xD2, 0xD8, 0xC1, 0xBE, 0xC1, 0xBD, 0xC2, 0xD8, /* 0xFC-0xFF */
};

static unsigned char u2c_67[512] = {
	0xBA, 0xC7, 0x00, 0x00, 0x00, 0x00, 0xD0, 0xF2, /* 0x00-0x03 */
	0x00, 0x00, 0x42, 0xEE, 0x00, 0x00, 0x42, 0xEF, /* 0x04-0x07 */
	0xB7, 0xEE, 0xCD, 0xAD, 0x00, 0x00, 0xCA, 0xFE, /* 0x08-0x0B */
	0x00, 0x00, 0xC9, 0xFE, 0xFA, 0xC5, 0xDB, 0xAC, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0xF1, /* 0x10-0x13 */
	0xBA, 0xF3, 0xC4, 0xBF, 0xDB, 0xAD, 0xCF, 0xAF, /* 0x14-0x17 */
	0x00, 0x00, 0x42, 0xF2, 0x00, 0x00, 0xCB, 0xBE, /* 0x18-0x1B */
	0x42, 0xF3, 0xC4, 0xAB, 0xDB, 0xAE, 0xB4, 0xFC, /* 0x1C-0x1F */
	0x42, 0xF4, 0x00, 0x00, 0x42, 0xF5, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xDB, 0xAF, 0xDB, 0xB0, /* 0x24-0x27 */
	0xCC, 0xDA, 0x00, 0x00, 0xCC, 0xA4, 0xCB, 0xF6, /* 0x28-0x2B */
	0xCB, 0xDC, 0xBB, 0xA5, 0xDB, 0xB2, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0xBC, 0xEB, 0x00, 0x00, 0x42, 0xF6, /* 0x30-0x33 */
	0xCB, 0xD1, 0x00, 0x00, 0xDB, 0xB4, 0xDB, 0xB7, /* 0x34-0x37 */
	0xDB, 0xB6, 0x00, 0x00, 0xB4, 0xF9, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xB5, 0xE0, 0x42, 0xF7, 0xDB, 0xB3, /* 0x3C-0x3F */
	0x00, 0x00, 0xDB, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x42, 0xF8, 0xDB, 0xB8, 0x42, 0xF9, /* 0x44-0x47 */
	0x42, 0xFA, 0xBF, 0xF9, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x42, 0xFB, 0x00, 0x00, 0xCD, 0xFB, 0xB0, 0xC9, /* 0x4C-0x4F */
	0xBA, 0xE0, 0xC2, 0xBC, 0x00, 0x00, 0xBC, 0xDD, /* 0x50-0x53 */
	0x42, 0xFC, 0x42, 0xFD, 0xBE, 0xF3, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0xDB, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0xC5, 0xCE, 0x42, 0xFE, 0xDB, 0xB9, 0xC2, 0xAB, /* 0x5C-0x5F */
	0xDB, 0xBA, 0xBE, 0xF2, 0xCC, 0xDD, 0xDB, 0xBC, /* 0x60-0x63 */
	0xDB, 0xBD, 0xCD, 0xE8, 0xFA, 0xC7, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xDB, 0xC2, 0x00, 0x00, /* 0x68-0x6B */
	0x43, 0xA2, 0xB9, 0xBA, 0x43, 0xA3, 0xC7, 0xD5, /* 0x6C-0x6F */
	0xDB, 0xBF, 0xC5, 0xEC, 0xDA, 0xDE, 0xDA, 0xE2, /* 0x70-0x73 */
	0x43, 0xA4, 0xB5, 0xCF, 0x43, 0xA5, 0xC7, 0xC7, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0xA6, /* 0x78-0x7B */
	0xDB, 0xC1, 0x00, 0x00, 0xBE, 0xBE, 0xC8, 0xC4, /* 0x7C-0x7F */

	0x00, 0x00, 0x43, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x43, 0xA8, 0xDB, 0xC7, 0x00, 0x00, 0xC8, 0xFA, /* 0x84-0x87 */
	0x00, 0x00, 0xDB, 0xBE, 0x00, 0x00, 0xDB, 0xC4, /* 0x88-0x8B */
	0xDB, 0xC3, 0x00, 0x00, 0x43, 0xA9, 0x43, 0xAA, /* 0x8C-0x8F */
	0xC0, 0xCF, 0x43, 0xAB, 0x00, 0x00, 0x43, 0xAC, /* 0x90-0x93 */
	0x00, 0x00, 0xCB, 0xED, 0x43, 0xAD, 0xCE, 0xD3, /* 0x94-0x97 */
	0x43, 0xAE, 0x43, 0xAF, 0xCB, 0xE7, 0x43, 0xB0, /* 0x98-0x9B */
	0xB2, 0xCC, 0xBB, 0xDE, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xCF, 0xC8, 0xDB, 0xC6, 0xBF, 0xF5, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0xDB, 0xC5, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0xDB, 0xC0, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB8, 0xCF, /* 0xAC-0xAF */
	0x43, 0xB1, 0x43, 0xB2, 0x43, 0xB3, 0xDB, 0xCC, /* 0xB0-0xB3 */
	0xDB, 0xCA, 0x43, 0xB4, 0xB2, 0xCD, 0xDB, 0xC8, /* 0xB4-0xB7 */
	0xDB, 0xCE, 0xDB, 0xD4, 0x00, 0x00, 0xFA, 0xC8, /* 0xB8-0xBB */
	0x43, 0xB6, 0x43, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xFA, 0xCA, 0xC2, 0xC8, 0x43, 0xBA, 0x43, 0xBB, /* 0xC0-0xC3 */
	0xCA, 0xC1, 0x43, 0xBC, 0xDB, 0xD6, 0x00, 0x00, /* 0xC4-0xC7 */
	0x43, 0xBD, 0x43, 0xBE, 0xC9, 0xA2, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0xDB, 0xD5, 0xC7, 0xF0, /* 0xCC-0xCF */
	0xCB, 0xBF, 0xB4, 0xBB, 0x43, 0xBF, 0xC0, 0xF7, /* 0xD0-0xD3 */
	0xBD, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x43, 0xC0, /* 0xD4-0xD7 */
	0xC4, 0xD3, 0x43, 0xC1, 0xCD, 0xAE, 0x00, 0x00, /* 0xD8-0xDB */
	0x43, 0xC2, 0xDB, 0xD1, 0xDB, 0xD0, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x43, 0xC3, 0xDB, 0xD2, 0x00, 0x00, /* 0xE0-0xE3 */
	0xDB, 0xCF, 0x00, 0x00, 0x43, 0xC4, 0xDB, 0xD7, /* 0xE4-0xE7 */
	0x00, 0x00, 0xDB, 0xCD, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0xDB, 0xCB, 0x00, 0x00, 0xDB, 0xD3, 0xDB, 0xC9, /* 0xEC-0xEF */
	0x43, 0xC5, 0xC3, 0xEC, 0x43, 0xC6, 0xCC, 0xF8, /* 0xF0-0xF3 */
	0xBC, 0xC6, 0xBA, 0xF4, 0x43, 0xC7, 0x43, 0xC8, /* 0xF4-0xF7 */
	0x00, 0x00, 0x43, 0xB8, 0x00, 0x00, 0xBA, 0xBA, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0xCB, 0xEF, 0xB3, 0xC1, /* 0xFC-0xFF */
};

static unsigned char u2c_68[512] = {
	0x00, 0x00, 0xFA, 0xCB, 0xC4, 0xCE, 0xC6, 0xCA, /* 0x00-0x03 */
	0xB1, 0xC9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xF2, /* 0x10-0x13 */
	0x43, 0xCA, 0x00, 0x00, 0xC0, 0xB4, 0xB7, 0xAA, /* 0x14-0x17 */
	0x00, 0x00, 0x43, 0xCB, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x43, 0xCC, 0xDB, 0xD9, 0x43, 0xCD, /* 0x1C-0x1F */
	0x00, 0x00, 0xB9, 0xBB, 0xB3, 0xFC, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0xCF, /* 0x24-0x27 */
	0x43, 0xCE, 0xDB, 0xDB, 0xB3, 0xF4, 0xDB, 0xE1, /* 0x28-0x2B */
	0x43, 0xD0, 0x43, 0xD1, 0x00, 0x00, 0x43, 0xD2, /* 0x2C-0x2F */
	0x43, 0xD3, 0x43, 0xD4, 0xDB, 0xDE, 0x43, 0xD5, /* 0x30-0x33 */
	0xC0, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0xB3, 0xCB, 0xBA, 0xAC, 0x00, 0x00, 0x43, 0xD6, /* 0x38-0x3B */
	0xB3, 0xCA, 0xBA, 0xCF, 0x00, 0x00, 0x43, 0xD7, /* 0x3C-0x3F */
	0xDB, 0xDC, 0xB7, 0xE5, 0xB7, 0xCB, 0xC5, 0xED, /* 0x40-0x43 */
	0xFA, 0xCC, 0x43, 0xD9, 0xDB, 0xDA, 0x00, 0x00, /* 0x44-0x47 */
	0xB0, 0xC6, 0x00, 0x00, 0x43, 0xDA, 0x00, 0x00, /* 0x48-0x4B */
	0x43, 0xDB, 0xDB, 0xDD, 0xDB, 0xDF, 0x00, 0x00, /* 0x4C-0x4F */
	0xB6, 0xCD, 0xB7, 0xAC, 0xFA, 0xC9, 0xB4, 0xBC, /* 0x50-0x53 */
	0xB5, 0xCB, 0x43, 0xDC, 0x00, 0x00, 0x43, 0xDD, /* 0x54-0x57 */
	0x43, 0xDE, 0xDB, 0xE2, 0x00, 0x00, 0x43, 0xDF, /* 0x58-0x5B */
	0xBA, 0xF9, 0xCB, 0xF1, 0x00, 0x00, 0xBB, 0xB7, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDB, 0xE3, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC9, 0xB0, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0xE0, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x43, 0xE1, 0x43, 0xE2, /* 0x6C-0x6F */
	0x43, 0xE3, 0x43, 0xE4, 0x43, 0xE5, 0x00, 0x00, /* 0x70-0x73 */
	0xDB, 0xEF, 0x43, 0xE6, 0xB2, 0xB3, 0xDB, 0xE4, /* 0x74-0x77 */
	0x00, 0x00, 0x43, 0xE7, 0x43, 0xE8, 0x43, 0xE9, /* 0x78-0x7B */
	0x43, 0xEA, 0x00, 0x00, 0xDB, 0xF5, 0xDB, 0xE5, /* 0x7C-0x7F */

	0x00, 0x00, 0xCE, 0xC2, 0x43, 0xEB, 0xDB, 0xEC, /* 0x80-0x83 */
	0x43, 0xEC, 0xC7, 0xDF, 0x43, 0xED, 0x00, 0x00, /* 0x84-0x87 */
	0x43, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0xDB, 0xF4, 0x00, 0x00, 0xDB, 0xE7, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB0, 0xB4, /* 0x90-0x93 */
	0xDB, 0xE9, 0x00, 0x00, 0x43, 0xEF, 0xB9, 0xBC, /* 0x94-0x97 */
	0x43, 0xF0, 0x00, 0x00, 0x43, 0xF1, 0xDB, 0xEB, /* 0x98-0x9B */
	0x43, 0xF2, 0xDB, 0xEA, 0x00, 0x00, 0xDB, 0xE6, /* 0x9C-0x9F */
	0xDB, 0xF1, 0x43, 0xF3, 0xBE, 0xBF, 0x43, 0xF4, /* 0xA0-0xA3 */
	0x00, 0x00, 0x43, 0xF5, 0xD4, 0xED, 0xB8, 0xE8, /* 0xA4-0xA7 */
	0xCD, 0xFC, 0x43, 0xF6, 0x43, 0xF7, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0xDB, 0xE8, 0x43, 0xF8, 0xC4, 0xF4, /* 0xAC-0xAF */
	0xB3, 0xA3, 0xBA, 0xAD, 0x43, 0xF9, 0xDB, 0xE0, /* 0xB0-0xB3 */
	0x00, 0x00, 0xDB, 0xF0, 0xB3, 0xE1, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xDB, 0xEE, 0xDB, 0xF2, 0x43, 0xFA, /* 0xB8-0xBB */
	0xC5, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0xB4, 0xFE, 0x43, 0xFB, 0xDC, 0xB2, 0x00, 0x00, /* 0xC4-0xC7 */
	0xF9, 0xA9, 0xCC, 0xC9, 0xDB, 0xF7, 0xB4, 0xFD, /* 0xC8-0xCB */
	0x43, 0xFD, 0xDB, 0xFE, 0x00, 0x00, 0xFA, 0xCD, /* 0xCC-0xCF */
	0x44, 0xA1, 0x44, 0xA2, 0xCB, 0xC0, 0x44, 0xA3, /* 0xD0-0xD3 */
	0xDC, 0xA1, 0xDC, 0xA3, 0x44, 0xA4, 0xDC, 0xA7, /* 0xD4-0xD7 */
	0xDB, 0xF9, 0x44, 0xA5, 0xC3, 0xAA, 0x00, 0x00, /* 0xD8-0xDB */
	0x44, 0xA6, 0x44, 0xA7, 0x00, 0x00, 0xC5, 0xEF, /* 0xDC-0xDF */
	0xDC, 0xAB, 0xDB, 0xFC, 0x00, 0x00, 0xDC, 0xA8, /* 0xE0-0xE3 */
	0x00, 0x00, 0x44, 0xA8, 0x00, 0x00, 0xDC, 0xA2, /* 0xE4-0xE7 */
	0x44, 0xA9, 0x00, 0x00, 0x44, 0xAA, 0x44, 0xAB, /* 0xE8-0xEB */
	0x44, 0xAC, 0x44, 0xAD, 0xBF, 0xB9, 0xDC, 0xAC, /* 0xEC-0xEF */
	0x44, 0xAE, 0x44, 0xAF, 0xC0, 0xB3, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x44, 0xB0, 0x44, 0xB1, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0xDC, 0xAA, 0xB4, 0xBD, 0x44, 0xB2, /* 0xF8-0xFB */
	0x44, 0xB3, 0x44, 0xB4, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_69[512] = {
	0xCF, 0xD0, 0xDB, 0xF6, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0xDC, 0xA6, 0xB0, 0xD8, 0x44, 0xB5, 0x00, 0x00, /* 0x04-0x07 */
	0xDB, 0xF8, 0x44, 0xB6, 0x44, 0xB7, 0xCC, 0xBA, /* 0x08-0x0B */
	0xDB, 0xFD, 0xBF, 0xA2, 0xC4, 0xC7, 0xDB, 0xF3, /* 0x0C-0x0F */
	0x44, 0xB8, 0x44, 0xB9, 0xDC, 0xA5, 0x44, 0xBA, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x44, 0xBB, 0x44, 0xBC, /* 0x14-0x17 */
	0x00, 0x00, 0xBF, 0xFA, 0xDC, 0xAF, 0xB3, 0xF1, /* 0x18-0x1B */
	0xB8, 0xA1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0xDC, 0xB1, 0xDB, 0xFA, 0xDC, 0xB0, /* 0x20-0x23 */
	0x00, 0x00, 0xDC, 0xA9, 0xDB, 0xFB, 0x00, 0x00, /* 0x24-0x27 */
	0xDC, 0xAD, 0x00, 0x00, 0xDC, 0xAE, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xDC, 0xBF, 0x44, 0xBD, 0x00, 0x00, 0x44, 0xBE, /* 0x30-0x33 */
	0xC6, 0xCE, 0x44, 0xBF, 0xDC, 0xA4, 0x00, 0x00, /* 0x34-0x37 */
	0x44, 0xC0, 0xDC, 0xBB, 0x00, 0x00, 0x44, 0xC1, /* 0x38-0x3B */
	0x00, 0x00, 0xDC, 0xBD, 0x00, 0x00, 0xC4, 0xD8, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x44, 0xC2, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x44, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x44, 0xC4, 0xCD, 0xCC, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x44, 0xC5, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC9, 0xF6, /* 0x50-0x53 */
	0xDC, 0xB8, 0xC2, 0xCA, 0x00, 0x00, 0x44, 0xC6, /* 0x54-0x57 */
	0x00, 0x00, 0xDC, 0xBE, 0xC1, 0xBF, 0x44, 0xC7, /* 0x58-0x5B */
	0xDC, 0xB5, 0xDC, 0xC2, 0xDC, 0xC1, 0x00, 0x00, /* 0x5C-0x5F */
	0xC6, 0xEF, 0xDC, 0xC0, 0xC6, 0xEA, 0x44, 0xC8, /* 0x60-0x63 */
	0x44, 0xC9, 0x44, 0xCA, 0x44, 0xCB, 0x00, 0x00, /* 0x64-0x67 */
	0xFA, 0xCF, 0x44, 0xCD, 0xDC, 0xC4, 0xDC, 0xB7, /* 0x68-0x6B */
	0x44, 0xCE, 0xB6, 0xC8, 0xDC, 0xBA, 0xBD, 0xDD, /* 0x6C-0x6F */
	0x44, 0xCF, 0x44, 0xD0, 0x44, 0xD1, 0xC7, 0xE0, /* 0x70-0x73 */
	0xDC, 0xBC, 0xB6, 0xCB, 0x00, 0x00, 0xDC, 0xB4, /* 0x74-0x77 */
	0xDC, 0xB6, 0xDC, 0xB3, 0x44, 0xD2, 0x44, 0xD3, /* 0x78-0x7B */
	0xCF, 0xB0, 0xB3, 0xDA, 0xDC, 0xB9, 0x44, 0xD4, /* 0x7C-0x7F */

	0x44, 0xD5, 0xDC, 0xC3, 0xB3, 0xB5, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xE7, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x44, 0xD6, 0xB1, 0xDD, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0xDC, 0xD4, 0x44, 0xD7, 0x00, 0x00, /* 0x90-0x93 */
	0xCF, 0xB1, 0xDC, 0xD7, 0x44, 0xD8, 0x00, 0x00, /* 0x94-0x97 */
	0xFA, 0xD1, 0x00, 0x00, 0x00, 0x00, 0xBF, 0xBA, /* 0x98-0x9B */
	0xDC, 0xD6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xDC, 0xD5, 0x44, 0xDA, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x44, 0xDB, 0x44, 0xDC, 0xDC, 0xD2, /* 0xA4-0xA7 */
	0x44, 0xDD, 0x00, 0x00, 0x00, 0x00, 0x44, 0xDE, /* 0xA8-0xAB */
	0x00, 0x00, 0x44, 0xDF, 0xDC, 0xC6, 0x44, 0xE0, /* 0xAC-0xAF */
	0x00, 0x00, 0xDC, 0xE3, 0xDC, 0xC5, 0x00, 0x00, /* 0xB0-0xB3 */
	0xDC, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x44, 0xE1, /* 0xB4-0xB7 */
	0x44, 0xE2, 0x00, 0x00, 0x44, 0xE3, 0xDC, 0xD0, /* 0xB8-0xBB */
	0x44, 0xE4, 0x00, 0x00, 0xDC, 0xCB, 0xDC, 0xC8, /* 0xBC-0xBF */
	0x00, 0x00, 0xDC, 0xC9, 0x00, 0x00, 0xDC, 0xD1, /* 0xC0-0xC3 */
	0x00, 0x00, 0x44, 0xE5, 0x00, 0x00, 0xF4, 0xA2, /* 0xC4-0xC7 */
	0x44, 0xE6, 0x00, 0x00, 0xDC, 0xCE, 0xB9, 0xBD, /* 0xC8-0xCB */
	0xC4, 0xC8, 0xC1, 0xE4, 0xDC, 0xCC, 0x00, 0x00, /* 0xCC-0xCF */
	0xDC, 0xC7, 0x44, 0xE7, 0x00, 0x00, 0xDC, 0xCA, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x44, 0xE8, 0x44, 0xE9, /* 0xD4-0xD7 */
	0xCD, 0xCD, 0xCB, 0xEA, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0xDC, 0xCF, 0xDC, 0xD9, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0xFA, 0xD2, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x44, 0xEB, 0x00, 0x00, 0xDC, 0xE1, /* 0xE4-0xE7 */
	0xDC, 0xDA, 0x00, 0x00, 0x00, 0x00, 0xDC, 0xE7, /* 0xE8-0xEB */
	0x00, 0x00, 0xDC, 0xE5, 0x44, 0xEC, 0x44, 0xED, /* 0xEC-0xEF */
	0x00, 0x00, 0x44, 0xEE, 0xDC, 0xE0, 0x44, 0xEF, /* 0xF0-0xF3 */
	0x00, 0x00, 0x44, 0xF0, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0xDC, 0xDF, 0x00, 0x00, 0xC4, 0xD0, /* 0xF8-0xFB */
	0x00, 0x00, 0xC1, 0xE5, 0x44, 0xF1, 0xDC, 0xDD, /* 0xFC-0xFF */
};

static unsigned char u2c_6A[512] = {
	0x44, 0xF2, 0x44, 0xF3, 0xDC, 0xDB, 0x44, 0xF4, /* 0x00-0x03 */
	0x00, 0x00, 0xDC, 0xE2, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0xDC, 0xE8, 0xC8, 0xF5, /* 0x08-0x0B */
	0xDC, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x44, 0xF5, /* 0x0C-0x0F */
	0x00, 0x00, 0x44, 0xF6, 0xDC, 0xE9, 0xDC, 0xEC, /* 0x10-0x13 */
	0xDC, 0xE6, 0x44, 0xF7, 0x00, 0x00, 0xC3, 0xF4, /* 0x14-0x17 */
	0x00, 0x00, 0xC9, 0xB8, 0x44, 0xF8, 0xDC, 0xDC, /* 0x18-0x1B */
	0x00, 0x00, 0x44, 0xF9, 0xDC, 0xE4, 0xBE, 0xC0, /* 0x1C-0x1F */
	0x44, 0xFA, 0xCC, 0xCF, 0xDC, 0xF8, 0xDC, 0xEB, /* 0x20-0x23 */
	0x44, 0xFB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x44, 0xFC, 0xB8, 0xA2, 0xB2, 0xA3, 0xB3, 0xDF, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0xDC, 0xD3, 0x00, 0x00, /* 0x2C-0x2F */
	0xFA, 0xD3, 0x00, 0x00, 0x44, 0xFE, 0x00, 0x00, /* 0x30-0x33 */
	0x45, 0xA1, 0xBE, 0xC1, 0xDC, 0xF0, 0x45, 0xA2, /* 0x34-0x37 */
	0xDC, 0xF7, 0xBC, 0xF9, 0xB3, 0xF2, 0x45, 0xA3, /* 0x38-0x3B */
	0x00, 0x00, 0xC3, 0xAE, 0x45, 0xA4, 0x45, 0xA5, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0xDC, 0xED, 0x45, 0xA6, 0xFA, 0xD5, 0xDC, 0xF2, /* 0x44-0x47 */
	0xDC, 0xF6, 0x45, 0xA8, 0x45, 0xA9, 0xB6, 0xB6, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x45, 0xAA, 0x00, 0x00, /* 0x4C-0x4F */
	0x45, 0xAB, 0x45, 0xAC, 0x45, 0xAD, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x45, 0xAE, 0x45, 0xAF, 0x00, 0x00, /* 0x54-0x57 */
	0xB5, 0xCC, 0xDC, 0xF4, 0x00, 0x00, 0x45, 0xB0, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB5, 0xA1, /* 0x5C-0x5F */
	0x00, 0x00, 0xC6, 0xCB, 0xDC, 0xF3, 0x00, 0x00, /* 0x60-0x63 */
	0x45, 0xB1, 0x00, 0x00, 0xDC, 0xF5, 0x45, 0xB2, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x45, 0xB3, 0xFA, 0xD4, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x45, 0xB4, 0xDC, 0xEF, 0xFA, 0xD6, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xDC, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0xFA, 0xD7, 0xB3, 0xE0, /* 0x7C-0x7F */

	0xC3, 0xC9, 0x45, 0xB7, 0x00, 0x00, 0x45, 0xB8, /* 0x80-0x83 */
	0xDC, 0xFC, 0x00, 0x00, 0x45, 0xB9, 0x45, 0xBA, /* 0x84-0x87 */
	0x00, 0x00, 0x45, 0xBB, 0x00, 0x00, 0x45, 0xBC, /* 0x88-0x8B */
	0x00, 0x00, 0xDC, 0xFA, 0xB8, 0xE9, 0x00, 0x00, /* 0x8C-0x8F */
	0xDC, 0xF9, 0x45, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xA1, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x45, 0xBE, /* 0x98-0x9B */
	0xDB, 0xD8, 0x45, 0xBF, 0x45, 0xC0, 0x45, 0xC1, /* 0x9C-0x9F */
	0xDC, 0xFB, 0x00, 0x00, 0xDC, 0xFD, 0xDC, 0xFE, /* 0xA0-0xA3 */
	0x00, 0x00, 0x45, 0xC2, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xAC, 0x45, 0xC3, /* 0xA8-0xAB */
	0xDD, 0xA8, 0x00, 0x00, 0xDB, 0xED, 0x45, 0xC4, /* 0xAC-0xAF */
	0x45, 0xC5, 0x45, 0xC6, 0x00, 0x00, 0xDD, 0xA7, /* 0xB0-0xB3 */
	0x45, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xDD, 0xA6, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xA3, /* 0xB8-0xBB */
	0x00, 0x00, 0x45, 0xC8, 0x45, 0xC9, 0x45, 0xCA, /* 0xBC-0xBF */
	0x00, 0x00, 0xDC, 0xEA, 0xDD, 0xA5, 0xDD, 0xA4, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x45, 0xCB, 0x00, 0x00, /* 0xC4-0xC7 */
	0x45, 0xCD, 0x45, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x45, 0xCE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x45, 0xCF, 0xDD, 0xAA, 0x00, 0x00, 0xCF, 0xA6, /* 0xD0-0xD3 */
	0x45, 0xD0, 0x45, 0xD1, 0x45, 0xD2, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xAD, 0xB6, 0xFB, /* 0xD8-0xDB */
	0x45, 0xD3, 0x45, 0xD4, 0xDD, 0xA9, 0xDD, 0xAB, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0xFA, 0xD8, 0x00, 0x00, /* 0xE0-0xE3 */
	0xFA, 0xD9, 0x00, 0x00, 0x00, 0x00, 0x45, 0xD6, /* 0xE4-0xE7 */
	0xC8, 0xA7, 0x00, 0x00, 0xDD, 0xAE, 0x00, 0x00, /* 0xE8-0xEB */
	0x45, 0xD7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x45, 0xD8, 0x45, 0xD9, 0x45, 0xDA, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xB2, 0xDD, 0xAF, /* 0xF8-0xFB */
	0x45, 0xDB, 0x45, 0xDC, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_6B[512] = {
	0x00, 0x00, 0x00, 0x00, 0x45, 0xDD, 0x45, 0xDE, /* 0x00-0x03 */
	0xCD, 0xF3, 0xDD, 0xB0, 0x45, 0xDF, 0x45, 0xE0, /* 0x04-0x07 */
	0x00, 0x00, 0x45, 0xE1, 0xDC, 0xDE, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x45, 0xE2, /* 0x0C-0x0F */
	0x45, 0xE3, 0x45, 0xE4, 0xDD, 0xB3, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xB4, 0x45, 0xE5, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x45, 0xE6, /* 0x18-0x1B */
	0x00, 0x00, 0xB1, 0xB5, 0x45, 0xE7, 0xDD, 0xB6, /* 0x1C-0x1F */
	0xB7, 0xE7, 0xBC, 0xA1, 0x00, 0x00, 0xB6, 0xD5, /* 0x20-0x23 */
	0x45, 0xE8, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xA4, /* 0x24-0x27 */
	0x45, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x45, 0xEA, /* 0x28-0x2B */
	0x45, 0xEB, 0x00, 0x00, 0x00, 0x00, 0x45, 0xEC, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0xCD, 0xDF, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x45, 0xED, 0x45, 0xEE, 0xDD, 0xB8, /* 0x34-0x37 */
	0xDD, 0xB7, 0xDD, 0xBA, 0xB5, 0xBD, 0x45, 0xEF, /* 0x38-0x3B */
	0x00, 0x00, 0xB6, 0xD6, 0xB4, 0xBE, 0x45, 0xF0, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xBD, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x45, 0xF1, 0xDD, 0xBC, /* 0x44-0x47 */
	0x00, 0x00, 0xDD, 0xBE, 0x45, 0xF2, 0x00, 0x00, /* 0x48-0x4B */
	0xB2, 0xCE, 0x45, 0xF3, 0xC3, 0xB7, 0x00, 0x00, /* 0x4C-0x4F */
	0xDD, 0xBF, 0x00, 0x00, 0x45, 0xF4, 0xB4, 0xBF, /* 0x50-0x53 */
	0xDD, 0xC1, 0x00, 0x00, 0x45, 0xF5, 0x00, 0x00, /* 0x54-0x57 */
	0x45, 0xF6, 0xDD, 0xC0, 0x00, 0x00, 0xDD, 0xC2, /* 0x58-0x5B */
	0x00, 0x00, 0x45, 0xF7, 0x00, 0x00, 0xDD, 0xC3, /* 0x5C-0x5F */
	0x45, 0xF8, 0xDD, 0xC4, 0xBB, 0xDF, 0xC0, 0xB5, /* 0x60-0x63 */
	0xBA, 0xA1, 0x00, 0x00, 0xC9, 0xF0, 0x45, 0xF9, /* 0x64-0x67 */
	0x00, 0x00, 0xCA, 0xE2, 0xCF, 0xC4, 0x45, 0xFA, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x45, 0xFB, 0xBB, 0xF5, /* 0x6C-0x6F */
	0x45, 0xFC, 0x00, 0x00, 0x00, 0x00, 0xBA, 0xD0, /* 0x70-0x73 */
	0xCE, 0xF2, 0x45, 0xFD, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xDD, 0xC5, 0xDD, 0xC6, 0x00, 0x00, 0xBB, 0xE0, /* 0x78-0x7B */
	0x00, 0x00, 0x45, 0xFE, 0x46, 0xA1, 0xDD, 0xC7, /* 0x7C-0x7F */

	0xDD, 0xC8, 0x00, 0x00, 0x46, 0xA2, 0xDD, 0xCA, /* 0x80-0x83 */
	0xDD, 0xC9, 0x46, 0xA3, 0xCB, 0xD8, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0xBD, 0xDE, 0xBC, 0xEC, 0xBB, 0xC4, /* 0x88-0x8B */
	0x00, 0x00, 0xDD, 0xCB, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0xDD, 0xCD, 0xBF, 0xA3, 0x46, 0xA4, /* 0x94-0x97 */
	0xDD, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x46, 0xA5, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xCE, 0x46, 0xA6, /* 0x9C-0x9F */
	0x46, 0xA7, 0x00, 0x00, 0x46, 0xA8, 0x46, 0xA9, /* 0xA0-0xA3 */
	0xDD, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x46, 0xAA, 0x46, 0xAB, 0xDD, 0xD0, 0xDD, 0xD1, /* 0xA8-0xAB */
	0x46, 0xAC, 0x46, 0xAD, 0x46, 0xAE, 0xDD, 0xD2, /* 0xAC-0xAF */
	0x46, 0xAF, 0xDD, 0xD4, 0xDD, 0xD3, 0xDD, 0xD5, /* 0xB0-0xB3 */
	0xB2, 0xA5, 0xC3, 0xCA, 0x00, 0x00, 0xDD, 0xD6, /* 0xB4-0xB7 */
	0x46, 0xB0, 0x46, 0xB1, 0xBB, 0xA6, 0xB3, 0xCC, /* 0xB8-0xBB */
	0xDD, 0xD7, 0x46, 0xB2, 0x46, 0xB3, 0xC5, 0xC2, /* 0xBC-0xBF */
	0xD4, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x46, 0xB4, /* 0xC0-0xC3 */
	0x46, 0xB5, 0xB5, 0xA3, 0xDD, 0xD8, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x46, 0xB6, 0x00, 0x00, 0xDD, 0xD9, /* 0xC8-0xCB */
	0x46, 0xB7, 0xCA, 0xEC, 0xCB, 0xE8, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xC6, 0xC7, 0xDD, 0xDA, /* 0xD0-0xD3 */
	0xC8, 0xE6, 0x00, 0x00, 0xFA, 0xDA, 0x00, 0x00, /* 0xD4-0xD7 */
	0xC8, 0xFB, 0x00, 0x00, 0x46, 0xB9, 0xCC, 0xD3, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xDB, /* 0xDC-0xDF */
	0x00, 0x00, 0x46, 0xBA, 0x00, 0x00, 0x46, 0xBB, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x46, 0xBC, 0x46, 0xBD, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xDD, /* 0xE8-0xEB */
	0xDD, 0xDC, 0x00, 0x00, 0x46, 0xBE, 0xDD, 0xDF, /* 0xEC-0xEF */
	0x00, 0x00, 0x46, 0xBF, 0x00, 0x00, 0xDD, 0xDE, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xC0, /* 0xF4-0xF7 */
	0x00, 0x00, 0x46, 0xC1, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xC2, /* 0xFC-0xFF */
};

static unsigned char u2c_6C[512] = {
	0x00, 0x00, 0x00, 0x00, 0x46, 0xC3, 0x00, 0x00, /* 0x00-0x03 */
	0x46, 0xC4, 0x46, 0xC5, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0xDD, 0xE1, 0x46, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x46, 0xC7, 0x46, 0xC8, 0xBB, 0xE1, /* 0x0C-0x0F */
	0x46, 0xC9, 0xCC, 0xB1, 0x46, 0xCA, 0xDD, 0xE2, /* 0x10-0x13 */
	0xDD, 0xE3, 0x00, 0x00, 0x00, 0x00, 0xB5, 0xA4, /* 0x14-0x17 */
	0x00, 0x00, 0x46, 0xCB, 0x00, 0x00, 0xDD, 0xE4, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xCC, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDD, 0xE6, /* 0x20-0x23 */
	0xDD, 0xE5, 0x00, 0x00, 0x46, 0xCD, 0x46, 0xCE, /* 0x24-0x27 */
	0x46, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x46, 0xD0, 0x00, 0x00, 0x46, 0xD1, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xD2, /* 0x30-0x33 */
	0xBF, 0xE5, 0x46, 0xD3, 0x46, 0xD4, 0xC9, 0xB9, /* 0x34-0x37 */
	0xB1, 0xCA, 0x00, 0x00, 0x46, 0xD5, 0x46, 0xD6, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0xC8, 0xC5, 0xFA, 0xDB, /* 0x3C-0x3F */
	0xC4, 0xF5, 0xBD, 0xC1, 0xB5, 0xE1, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x46, 0xD8, 0x46, 0xD9, /* 0x48-0x4B */
	0x00, 0x00, 0x46, 0xDA, 0xC8, 0xC6, 0x46, 0xDB, /* 0x4C-0x4F */
	0xBC, 0xAE, 0x00, 0x00, 0x46, 0xDC, 0x00, 0x00, /* 0x50-0x53 */
	0x46, 0xDD, 0xDD, 0xE8, 0x00, 0x00, 0xB4, 0xC0, /* 0x54-0x57 */
	0x00, 0x00, 0x46, 0xDE, 0xB1, 0xF8, 0x46, 0xDF, /* 0x58-0x5B */
	0xFA, 0xDC, 0xC6, 0xF2, 0xDD, 0xE7, 0xB9, 0xBE, /* 0x5C-0x5F */
	0xC3, 0xD3, 0x00, 0x00, 0xDD, 0xE9, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0xB6, /* 0x64-0x67 */
	0xDD, 0xF1, 0x00, 0x00, 0xDD, 0xEA, 0x46, 0xE1, /* 0x68-0x6B */
	0x00, 0x00, 0x46, 0xE2, 0x00, 0x00, 0xFA, 0xDE, /* 0x6C-0x6F */
	0xC2, 0xC1, 0x00, 0x00, 0xB5, 0xE2, 0xDD, 0xF2, /* 0x70-0x73 */
	0x46, 0xE4, 0x00, 0x00, 0x46, 0xE5, 0x00, 0x00, /* 0x74-0x77 */
	0x46, 0xE6, 0x46, 0xE7, 0xB7, 0xE8, 0x46, 0xE8, /* 0x78-0x7B */
	0x00, 0x00, 0xB5, 0xA5, 0xDD, 0xF0, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0xDD, 0xEE, 0xDD, 0xEB, 0xCD, 0xE0, /* 0x80-0x83 */
	0x00, 0x00, 0x46, 0xE9, 0xFA, 0xDD, 0x46, 0xEB, /* 0x84-0x87 */
	0xC4, 0xC0, 0x46, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0xC6, 0xD9, 0xDD, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0xDD, 0xF4, 0x00, 0x00, 0xDD, 0xF3, 0xB7, 0xA3, /* 0x90-0x93 */
	0x46, 0xED, 0x46, 0xEE, 0xB2, 0xAD, 0x46, 0xEF, /* 0x94-0x97 */
	0x46, 0xF0, 0xBA, 0xBB, 0xDD, 0xED, 0xDD, 0xEF, /* 0x98-0x9B */
	0x46, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x46, 0xF2, /* 0x9C-0x9F */
	0x00, 0x00, 0xCB, 0xD7, 0xC2, 0xF4, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCB, 0xF7, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xFC, 0x00, 0x00, /* 0xAC-0xAF */
	0x46, 0xF3, 0xDD, 0xFD, 0x46, 0xF4, 0xB2, 0xCF, /* 0xB0-0xB3 */
	0x46, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xCA, 0xA8, 0xCC, 0xFD, 0xDE, 0xA1, 0xBC, 0xA3, /* 0xB8-0xBB */
	0xBE, 0xC2, 0xDD, 0xF8, 0xDD, 0xFE, 0xB1, 0xE8, /* 0xBC-0xBF */
	0x00, 0x00, 0xB6, 0xB7, 0x46, 0xF6, 0x00, 0x00, /* 0xC0-0xC3 */
	0xDD, 0xF5, 0xDD, 0xFA, 0x46, 0xF7, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0xC0, 0xF4, 0xC7, 0xF1, 0x00, 0x00, /* 0xC8-0xCB */
	0xC8, 0xE7, 0x46, 0xF8, 0x00, 0x00, 0x46, 0xF9, /* 0xCC-0xCF */
	0x46, 0xFA, 0x46, 0xFB, 0x46, 0xFC, 0xDD, 0xF7, /* 0xD0-0xD3 */
	0x46, 0xFD, 0xCB, 0xA1, 0x46, 0xFE, 0xDD, 0xF9, /* 0xD4-0xD7 */
	0x00, 0x00, 0xDE, 0xA4, 0xFA, 0xDF, 0xDE, 0xA2, /* 0xD8-0xDB */
	0x47, 0xA2, 0xDD, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x47, 0xA3, 0xCB, 0xA2, 0xC7, 0xC8, 0xB5, 0xE3, /* 0xE0-0xE3 */
	0x00, 0x00, 0xC5, 0xA5, 0x00, 0x00, 0x47, 0xA4, /* 0xE4-0xE7 */
	0xC3, 0xED, 0x47, 0xA5, 0xDE, 0xA5, 0x47, 0xA6, /* 0xE8-0xEB */
	0x47, 0xA7, 0x00, 0x00, 0x47, 0xA8, 0xDE, 0xA3, /* 0xEC-0xEF */
	0xC2, 0xD9, 0xDD, 0xF6, 0x47, 0xA9, 0xB1, 0xCB, /* 0xF0-0xF3 */
	0x47, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
};

static unsigned char u2c_6D[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0xFA, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x47, 0xAC, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x47, 0xAD, 0xCD, 0xCE, /* 0x08-0x0B */
	0xDE, 0xB0, 0x00, 0x00, 0x47, 0xAE, 0x47, 0xAF, /* 0x0C-0x0F */
	0x00, 0x00, 0x47, 0xB0, 0xDE, 0xAF, 0x47, 0xB1, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xF6, /* 0x14-0x17 */
	0x00, 0x00, 0xDE, 0xAC, 0x47, 0xB2, 0xCD, 0xEC, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0xC6, 0xB6, 0xDE, 0xA6, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0xC4, 0xC5, 0x47, 0xB3, 0x47, 0xB4, /* 0x24-0x27 */
	0x47, 0xB5, 0xB1, 0xCC, 0xB9, 0xBF, 0xDE, 0xA9, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x47, 0xB7, 0x47, 0xB8, /* 0x2C-0x2F */
	0x00, 0x00, 0x47, 0xB9, 0xBD, 0xA7, 0xDE, 0xAE, /* 0x30-0x33 */
	0x00, 0x00, 0xDE, 0xAD, 0xDE, 0xA8, 0x00, 0x00, /* 0x34-0x37 */
	0xDE, 0xAB, 0x47, 0xBA, 0x00, 0x00, 0xB3, 0xE8, /* 0x38-0x3B */
	0x47, 0xBB, 0xDE, 0xAA, 0xC7, 0xC9, 0x47, 0xBC, /* 0x3C-0x3F */
	0x00, 0x00, 0xCE, 0xAE, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0xBE, 0xF4, 0xC0, 0xF5, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0xBD, /* 0x54-0x57 */
	0x00, 0x00, 0xDE, 0xB6, 0xDE, 0xB4, 0x00, 0x00, /* 0x58-0x5B */
	0xC9, 0xCD, 0x00, 0x00, 0x47, 0xBE, 0x47, 0xBF, /* 0x5C-0x5F */
	0x00, 0x00, 0x47, 0xC0, 0x00, 0x00, 0xDE, 0xB1, /* 0x60-0x63 */
	0xDE, 0xB3, 0x47, 0xC1, 0xB1, 0xBA, 0x47, 0xC2, /* 0x64-0x67 */
	0x00, 0x00, 0xB9, 0xC0, 0xCF, 0xB2, 0x00, 0x00, /* 0x68-0x6B */
	0xB3, 0xBD, 0x00, 0x00, 0xC9, 0xE2, 0xFA, 0xE2, /* 0x6C-0x6F */
	0x47, 0xC4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0xCD, 0xE1, 0x00, 0x00, 0x00, 0x00, 0xB3, 0xA4, /* 0x74-0x77 */
	0xBF, 0xBB, 0xDE, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x47, 0xC5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x47, 0xC6, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0xDE, 0xBA, 0x00, 0x00, 0xFA, 0xE1, /* 0x84-0x87 */
	0xBE, 0xC3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0xCD, 0xB0, 0x00, 0x00, 0xDE, 0xB7, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x47, 0xC8, 0x47, 0xC9, 0xDE, 0xB2, /* 0x90-0x93 */
	0x47, 0xCA, 0xDE, 0xB8, 0xFA, 0xE3, 0x47, 0xCC, /* 0x94-0x97 */
	0x47, 0xCD, 0xCE, 0xDE, 0x00, 0x00, 0xC5, 0xF3, /* 0x98-0x9B */
	0xC6, 0xC2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x47, 0xCE, 0x00, 0x00, /* 0xA8-0xAB */
	0xFA, 0xE4, 0x00, 0x00, 0x00, 0x00, 0xB3, 0xB6, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xB1, 0xD5, 0x00, 0x00, /* 0xB0-0xB3 */
	0x47, 0xD0, 0xDE, 0xBE, 0x00, 0x00, 0x47, 0xD1, /* 0xB4-0xB7 */
	0xDE, 0xC1, 0x47, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0xCE, 0xC3, 0x47, 0xD3, 0x00, 0x00, 0x47, 0xD4, /* 0xBC-0xBF */
	0xCD, 0xE4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x47, 0xD5, 0xDE, 0xC8, 0xDE, 0xC2, 0xDE, 0xBF, /* 0xC4-0xC7 */
	0x47, 0xD6, 0x00, 0x00, 0x47, 0xD7, 0xCE, 0xD4, /* 0xC8-0xCB */
	0xDE, 0xC5, 0x00, 0x00, 0x47, 0xD8, 0xFA, 0xE5, /* 0xCC-0xCF */
	0x00, 0x00, 0xBD, 0xCA, 0xDE, 0xC7, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0xDE, 0xCC, 0x47, 0xDA, 0x00, 0x00, /* 0xD4-0xD7 */
	0xC5, 0xF1, 0xDE, 0xCA, 0x00, 0x00, 0x47, 0xDB, /* 0xD8-0xDB */
	0x00, 0x00, 0x47, 0xDC, 0xDE, 0xC4, 0x47, 0xDD, /* 0xDC-0xDF */
	0x47, 0xDE, 0xC3, 0xB8, 0x47, 0xDF, 0x00, 0x00, /* 0xE0-0xE3 */
	0xDE, 0xCB, 0x47, 0xE0, 0xDE, 0xC0, 0x00, 0x00, /* 0xE4-0xE7 */
	0xDE, 0xC6, 0x47, 0xE1, 0xDE, 0xCD, 0xB0, 0xFC, /* 0xE8-0xEB */
	0xDE, 0xC3, 0x00, 0x00, 0xDE, 0xCE, 0x47, 0xE2, /* 0xEC-0xEF */
	0x47, 0xE3, 0xBF, 0xBC, 0xFA, 0xE7, 0xBD, 0xDF, /* 0xF0-0xF3 */
	0x47, 0xE4, 0xCA, 0xA5, 0x47, 0xE5, 0xBA, 0xAE, /* 0xF4-0xF7 */
	0xFA, 0xE6, 0xDE, 0xBB, 0xDE, 0xC9, 0xC5, 0xBA, /* 0xF8-0xFB */
	0xFA, 0xE8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_6E[512] = {
	0x47, 0xE7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x47, 0xE8, 0xC0, 0xB6, 0x00, 0x00, 0xB3, 0xE9, /* 0x04-0x07 */
	0xBA, 0xD1, 0xBE, 0xC4, 0xDE, 0xBD, 0xBD, 0xC2, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB7, 0xCC, /* 0x10-0x13 */
	0x00, 0x00, 0xDE, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xDE, 0xD2, 0xBD, 0xED, 0xB8, 0xBA, /* 0x18-0x1B */
	0x00, 0x00, 0xDE, 0xE1, 0x47, 0xE9, 0xDE, 0xDB, /* 0x1C-0x1F */
	0xB5, 0xF4, 0xC5, 0xCF, 0x47, 0xEA, 0xDE, 0xD6, /* 0x20-0x23 */
	0xDE, 0xDF, 0xB0, 0xAF, 0xB1, 0xB2, 0xFA, 0xEB, /* 0x24-0x27 */
	0x00, 0x00, 0xB2, 0xB9, 0x00, 0x00, 0xDE, 0xD8, /* 0x28-0x2B */
	0xC2, 0xAC, 0xDE, 0xCF, 0xDE, 0xD1, 0xB9, 0xC1, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x47, 0xEC, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x47, 0xED, 0x00, 0x00, /* 0x34-0x37 */
	0xDE, 0xE2, 0xFA, 0xE9, 0xDE, 0xDD, 0x47, 0xEF, /* 0x38-0x3B */
	0xFA, 0xEC, 0x00, 0x00, 0xDE, 0xD5, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDE, 0xDC, /* 0x40-0x43 */
	0x47, 0xF1, 0x47, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x47, 0xF3, 0x47, 0xF4, 0xCC, 0xAB, 0x47, 0xF5, /* 0x48-0x4B */
	0x00, 0x00, 0xDE, 0xDA, 0xDE, 0xDE, 0x47, 0xF6, /* 0x4C-0x4F */
	0x00, 0x00, 0x47, 0xF7, 0x47, 0xF8, 0x47, 0xF9, /* 0x50-0x53 */
	0x47, 0xFA, 0x00, 0x00, 0xB8, 0xD0, 0x47, 0xFB, /* 0x54-0x57 */
	0xBE, 0xC5, 0x00, 0x00, 0x00, 0x00, 0xC3, 0xB9, /* 0x58-0x5B */
	0xFA, 0xEA, 0x47, 0xFD, 0x47, 0xFE, 0xDE, 0xD4, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x48, 0xA1, 0x48, 0xA2, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCD, 0xAF, /* 0x64-0x67 */
	0x48, 0xA3, 0x00, 0x00, 0x00, 0x00, 0xDE, 0xD7, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0xDE, 0xD0, 0xC5, 0xF2, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0xDE, 0xD3, 0x48, 0xA4, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0xDE, 0xD9, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0xA5, /* 0x78-0x7B */
	0x00, 0x00, 0x48, 0xA6, 0xCF, 0xD1, 0xBC, 0xBE, /* 0x7C-0x7F */

	0xCB, 0xFE, 0x00, 0x00, 0xDE, 0xE3, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0xC8, 0xAE, 0x48, 0xA7, 0x00, 0x00, 0xDE, 0xEF, /* 0x8C-0x8F */
	0xB8, 0xBB, 0x00, 0x00, 0x00, 0x00, 0x48, 0xA8, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xBD, 0xE0, 0x00, 0x00, /* 0x94-0x97 */
	0xDE, 0xE5, 0x48, 0xA9, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xCE, 0xAF, 0xB9, 0xC2, 0x00, 0x00, 0xDE, 0xF2, /* 0x9C-0x9F */
	0x48, 0xAA, 0x00, 0x00, 0xB0, 0xEE, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xDE, 0xF0, 0x00, 0x00, 0x48, 0xAB, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0xDE, 0xE4, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x48, 0xAC, 0x48, 0xAD, 0xDE, 0xEA, /* 0xAC-0xAF */
	0x00, 0x00, 0x48, 0xAE, 0xDE, 0xEC, 0x48, 0xAF, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0xCD, 0xCF, 0xDE, 0xE7, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0xC5, 0xAE, 0x48, 0xB0, /* 0xB8-0xBB */
	0x00, 0x00, 0xDE, 0xE9, 0x00, 0x00, 0xFA, 0xED, /* 0xBC-0xBF */
	0x48, 0xB2, 0x48, 0xB3, 0xDE, 0xF1, 0x48, 0xB4, /* 0xC0-0xC3 */
	0xDE, 0xEB, 0xCC, 0xC7, 0x00, 0x00, 0x48, 0xB5, /* 0xC4-0xC7 */
	0x48, 0xB6, 0xDE, 0xE6, 0x48, 0xB7, 0xBC, 0xA2, /* 0xC8-0xCB */
	0xDE, 0xFE, 0x48, 0xB8, 0x48, 0xB9, 0x48, 0xBA, /* 0xCC-0xCF */
	0x00, 0x00, 0xB3, 0xEA, 0x00, 0x00, 0xDE, 0xE8, /* 0xD0-0xD3 */
	0xDE, 0xED, 0xDE, 0xEE, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0xC2, 0xEC, 0xC2, 0xDA, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0xBB, /* 0xE8-0xEB */
	0xDE, 0xF6, 0x48, 0xBC, 0x48, 0xBD, 0xDE, 0xFC, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0xDE, 0xFA, 0x00, 0x00, /* 0xF0-0xF3 */
	0xC5, 0xA9, 0x00, 0x00, 0x00, 0x00, 0xDF, 0xA3, /* 0xF4-0xF7 */
	0xDE, 0xF7, 0x48, 0xBE, 0x00, 0x00, 0x48, 0xBF, /* 0xF8-0xFB */
	0x00, 0x00, 0x48, 0xC0, 0xDE, 0xF8, 0xDE, 0xE0, /* 0xFC-0xFF */
};

static unsigned char u2c_6F[512] = {
	0x00, 0x00, 0xB5, 0xF9, 0xC9, 0xBA, 0x00, 0x00, /* 0x00-0x03 */
	0x48, 0xC1, 0x00, 0x00, 0xBC, 0xBF, 0x00, 0x00, /* 0x04-0x07 */
	0x48, 0xC2, 0xB9, 0xF7, 0x48, 0xC3, 0x00, 0x00, /* 0x08-0x0B */
	0x48, 0xC4, 0x48, 0xC5, 0x00, 0x00, 0xCF, 0xB3, /* 0x0C-0x0F */
	0x00, 0x00, 0xDE, 0xF4, 0x00, 0x00, 0xDF, 0xA2, /* 0x10-0x13 */
	0xB1, 0xE9, 0xC1, 0xE6, 0x48, 0xC6, 0x00, 0x00, /* 0x14-0x17 */
	0x48, 0xC7, 0x00, 0x00, 0x48, 0xC8, 0x48, 0xC9, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0xC7, 0xF9, 0x00, 0x00, 0xB4, 0xC1, 0xCE, 0xFA, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x48, 0xCA, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x48, 0xCB, 0x48, 0xCC, 0xCC, 0xA1, /* 0x28-0x2B */
	0xC4, 0xD2, 0x48, 0xD3, 0x00, 0x00, 0x48, 0xCD, /* 0x2C-0x2F */
	0x48, 0xCE, 0xDE, 0xFB, 0xDE, 0xFD, 0x48, 0xCF, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x48, 0xD0, 0x00, 0x00, /* 0x34-0x37 */
	0xC1, 0xB2, 0x00, 0x00, 0x00, 0x00, 0x48, 0xD1, /* 0x38-0x3B */
	0x48, 0xD2, 0x00, 0x00, 0xDF, 0xA1, 0xDE, 0xF9, /* 0x3C-0x3F */
	0x00, 0x00, 0xDE, 0xF3, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0xB4, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0xD4, /* 0x4C-0x4F */
	0x00, 0x00, 0x48, 0xD5, 0x48, 0xD6, 0x48, 0xD7, /* 0x50-0x53 */
	0xB7, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x48, 0xD8, /* 0x54-0x57 */
	0xDF, 0xAF, 0x48, 0xD9, 0x48, 0xDA, 0xDF, 0xAA, /* 0x58-0x5B */
	0xC0, 0xF8, 0x48, 0xDB, 0x48, 0xDC, 0xB3, 0xE3, /* 0x5C-0x5F */
	0x00, 0x00, 0x48, 0xDD, 0x48, 0xDE, 0x00, 0x00, /* 0x60-0x63 */
	0xBD, 0xE1, 0x00, 0x00, 0xDF, 0xB3, 0x00, 0x00, /* 0x64-0x67 */
	0x48, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x48, 0xE0, 0xDF, 0xAC, 0xC4, 0xAC, 0xDF, 0xA9, /* 0x6C-0x6F */
	0xC4, 0xD9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0xDF, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xDF, 0xA6, 0x00, 0x00, 0xDF, 0xA5, 0x00, 0x00, /* 0x78-0x7B */
	0xDF, 0xAE, 0x48, 0xE1, 0x48, 0xE2, 0x00, 0x00, /* 0x7C-0x7F */

	0xDF, 0xA8, 0xDF, 0xA7, 0xDF, 0xAD, 0x48, 0xE3, /* 0x80-0x83 */
	0xC0, 0xA1, 0x00, 0x00, 0xDF, 0xA4, 0x48, 0xE4, /* 0x84-0x87 */
	0xFA, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x48, 0xE6, /* 0x88-0x8B */
	0x48, 0xE7, 0x48, 0xE8, 0xDF, 0xB0, 0x00, 0x00, /* 0x8C-0x8F */
	0x48, 0xE9, 0xDF, 0xB1, 0x48, 0xEA, 0x48, 0xEB, /* 0x90-0x93 */
	0x48, 0xEC, 0x00, 0x00, 0x48, 0xED, 0xB4, 0xC2, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x48, 0xEE, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0xEF, /* 0x9C-0x9F */
	0x48, 0xF0, 0xDF, 0xB6, 0x00, 0x00, 0xDF, 0xB5, /* 0xA0-0xA3 */
	0xDF, 0xB7, 0x48, 0xF1, 0x48, 0xF2, 0x48, 0xF3, /* 0xA4-0xA7 */
	0x48, 0xF4, 0x00, 0x00, 0xDF, 0xBA, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x48, 0xF5, 0x48, 0xF6, /* 0xAC-0xAF */
	0x48, 0xF7, 0xC5, 0xC3, 0x00, 0x00, 0xDF, 0xB4, /* 0xB0-0xB3 */
	0x00, 0x00, 0xFA, 0xEF, 0x48, 0xF9, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xDF, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x48, 0xFA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xB7, 0xE3, 0xC2, 0xF9, 0xDF, 0xB2, 0xC7, 0xBB, /* 0xC0-0xC3 */
	0x00, 0x00, 0x48, 0xFB, 0xDF, 0xB9, 0x48, 0xFC, /* 0xC4-0xC7 */
	0x48, 0xFD, 0x00, 0x00, 0x48, 0xFE, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0xDF, 0xBE, 0xDF, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xDF, 0xBF, 0x00, 0x00, 0x49, 0xA1, 0xDF, 0xC2, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x49, 0xA2, 0xDF, 0xBB, /* 0xDC-0xDF */
	0xB9, 0xEA, 0xC7, 0xA8, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0xDE, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x49, 0xA3, 0x49, 0xA4, 0x00, 0x00, 0xCD, 0xF4, /* 0xE8-0xEB */
	0xDF, 0xBD, 0x00, 0x00, 0xDF, 0xC1, 0xC2, 0xF5, /* 0xEC-0xEF */
	0x49, 0xA5, 0xDF, 0xC0, 0x00, 0x00, 0xDF, 0xAB, /* 0xF0-0xF3 */
	0x00, 0x00, 0xFA, 0xF0, 0xEF, 0xE9, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x49, 0xA7, 0xDF, 0xC5, 0x00, 0x00, /* 0xF8-0xFB */
	0x49, 0xA8, 0x49, 0xA9, 0xDF, 0xC9, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_70[512] = {
	0x49, 0xAA, 0xDF, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0xFA, 0xF1, 0x49, 0xAC, 0xFA, 0xF2, /* 0x04-0x07 */
	0x00, 0x00, 0xDF, 0xC3, 0x00, 0x00, 0xDF, 0xC4, /* 0x08-0x0B */
	0x00, 0x00, 0x49, 0xAE, 0x00, 0x00, 0xDF, 0xC8, /* 0x0C-0x0F */
	0x00, 0x00, 0xDF, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0xC9, 0xCE, 0x00, 0x00, 0x49, 0xAF, /* 0x14-0x17 */
	0xDF, 0xCE, 0x00, 0x00, 0xDF, 0xCB, 0xDF, 0xCA, /* 0x18-0x1B */
	0x00, 0x00, 0xDF, 0xCD, 0xC6, 0xD4, 0xDF, 0xCF, /* 0x1C-0x1F */
	0x49, 0xB0, 0x00, 0x00, 0x00, 0x00, 0x49, 0xB1, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xC3, 0xF5, 0xC2, 0xED, /* 0x24-0x27 */
	0xFA, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0xC0, 0xA5, 0x00, 0x00, 0x00, 0x00, 0x49, 0xB2, /* 0x2C-0x2F */
	0xDF, 0xD0, 0x00, 0x00, 0xDF, 0xD2, 0x00, 0x00, /* 0x30-0x33 */
	0x49, 0xB3, 0x00, 0x00, 0x00, 0x00, 0x49, 0xB4, /* 0x34-0x37 */
	0x00, 0x00, 0x49, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x49, 0xB6, 0x00, 0x00, 0xDF, 0xD1, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0xB7, /* 0x40-0x43 */
	0x49, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x49, 0xB9, 0x49, 0xBA, 0x49, 0xBB, 0x49, 0xBC, /* 0x48-0x4B */
	0xDE, 0xF5, 0x00, 0x00, 0x49, 0xC1, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0xDF, 0xD3, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x49, 0xBD, 0x49, 0xBE, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0xC6, 0xE7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x49, 0xBF, 0x49, 0xC0, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDF, 0xD4, /* 0x60-0x63 */
	0x49, 0xC2, 0x49, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xD0, /* 0x68-0x6B */
	0x49, 0xC4, 0x00, 0x00, 0x49, 0xC5, 0xC5, 0xF4, /* 0x6C-0x6F */
	0xB3, 0xA5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x49, 0xC6, 0x49, 0xC7, 0x00, 0x00, /* 0x74-0x77 */
	0xB5, 0xE4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0xBC, 0xDE, 0xBA, 0xD2, 0x49, 0xC8, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x49, 0xC9, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0xFA, 0xF4, 0x49, 0xCB, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0xCF, 0xA7, 0xBF, 0xE6, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0xB1, 0xEA, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0xDF, 0xD6, 0x00, 0x00, /* 0x90-0x93 */
	0x49, 0xCC, 0x49, 0xCD, 0x49, 0xCE, 0x49, 0xCF, /* 0x94-0x97 */
	0x49, 0xD0, 0xDF, 0xD5, 0x00, 0x00, 0x49, 0xD1, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x49, 0xD2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFA, 0xF5, /* 0xA8-0xAB */
	0xDF, 0xD9, 0xC3, 0xBA, 0xDF, 0xDC, 0xDF, 0xD7, /* 0xAC-0xAF */
	0x49, 0xD4, 0x49, 0xD5, 0x00, 0x00, 0xDF, 0xDB, /* 0xB0-0xB3 */
	0x49, 0xD6, 0x00, 0x00, 0x00, 0x00, 0x49, 0xD7, /* 0xB4-0xB7 */
	0xDF, 0xDA, 0xC5, 0xC0, 0xB0, 0xD9, 0xF9, 0xA7, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0xCE, 0xF5, 0x00, 0x00, 0x49, 0xD8, 0xDF, 0xDE, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xA8, /* 0xCC-0xCF */
	0x00, 0x00, 0x49, 0xD9, 0x00, 0x00, 0x49, 0xDA, /* 0xD0-0xD3 */
	0x49, 0xDB, 0x49, 0xDC, 0x49, 0xDD, 0x00, 0x00, /* 0xD4-0xD7 */
	0x49, 0xDE, 0xDF, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x49, 0xDF, 0xDF, 0xDF, 0x00, 0x00, 0xDF, 0xDD, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x49, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0xDF, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0xCB, 0xA3, 0x49, 0xE1, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xDF, 0xE2, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_71[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0xE2, /* 0x00-0x03 */
	0xFA, 0xF7, 0x49, 0xE4, 0x49, 0xE5, 0x49, 0xE6, /* 0x04-0x07 */
	0x00, 0x00, 0xDF, 0xE1, 0x00, 0x00, 0x49, 0xE7, /* 0x08-0x0B */
	0x49, 0xE8, 0x00, 0x00, 0x00, 0x00, 0xFA, 0xF6, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0xB1, 0xEB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xDF, 0xE4, 0xCA, 0xB2, 0x00, 0x00, /* 0x18-0x1B */
	0xDF, 0xE3, 0x00, 0x00, 0x49, 0xEA, 0x00, 0x00, /* 0x1C-0x1F */
	0x49, 0xEB, 0xCC, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xBE, 0xC7, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0xEC, /* 0x28-0x2B */
	0x00, 0x00, 0x49, 0xED, 0x00, 0x00, 0x49, 0xEE, /* 0x2C-0x2F */
	0x49, 0xEF, 0x49, 0xF0, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0xC1, 0xB3, 0x00, 0x00, /* 0x34-0x37 */
	0x49, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0xBE, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x49, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x49, 0xF3, 0xFA, 0xF9, 0xFA, 0xFA, /* 0x44-0x47 */
	0x00, 0x00, 0xCE, 0xFB, 0x49, 0xF6, 0x49, 0xF7, /* 0x48-0x4B */
	0xDF, 0xEA, 0x00, 0x00, 0xC0, 0xF9, 0x00, 0x00, /* 0x4C-0x4F */
	0x49, 0xF8, 0x00, 0x00, 0x49, 0xF9, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0xDF, 0xE6, 0xDF, 0xEB, 0x49, 0xFA, /* 0x54-0x57 */
	0x00, 0x00, 0xB1, 0xEC, 0x49, 0xFB, 0x00, 0x00, /* 0x58-0x5B */
	0xFA, 0xF8, 0x00, 0x00, 0x49, 0xFD, 0x00, 0x00, /* 0x5C-0x5F */
	0x49, 0xFE, 0x00, 0x00, 0xDF, 0xE9, 0x00, 0x00, /* 0x60-0x63 */
	0xC7, 0xE1, 0xDF, 0xE5, 0xDF, 0xE8, 0xBE, 0xC8, /* 0x64-0x67 */
	0x4A, 0xA1, 0xC8, 0xD1, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0xDF, 0xEC, 0x00, 0x00, 0xBC, 0xD1, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x4A, 0xA2, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0xC0, 0xFA, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x4A, 0xA3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0xDF, 0xEF, 0x4A, 0xA4, 0x00, 0x00, 0x4A, 0xA5, /* 0x84-0x87 */
	0xDF, 0xE7, 0x00, 0x00, 0xB7, 0xA7, 0x00, 0x00, /* 0x88-0x8B */
	0x4A, 0xA6, 0x00, 0x00, 0x00, 0x00, 0xDF, 0xED, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x4A, 0xA7, 0x00, 0x00, /* 0x90-0x93 */
	0xCD, 0xD0, 0xDF, 0xF0, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xF4, 0xA6, 0x4A, 0xA8, 0x4A, 0xA9, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xCF, /* 0x9C-0x9F */
	0x4A, 0xAA, 0x00, 0x00, 0x4A, 0xAB, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0xDF, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0xDF, 0xF2, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xAC, /* 0xAC-0xAF */
	0x4A, 0xAD, 0xC7, 0xAE, 0x4A, 0xAE, 0x4A, 0xAF, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xDF, 0xF4, 0x4A, 0xB0, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0xDF, 0xF5, 0x4A, 0xB1, /* 0xBC-0xBF */
	0x4A, 0xB2, 0xFA, 0xFC, 0x00, 0x00, 0xC7, 0xB3, /* 0xC0-0xC3 */
	0x4A, 0xB4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0xC5, 0xF5, 0xDF, 0xF7, 0x00, 0x00, 0x4A, 0xB5, /* 0xC8-0xCB */
	0x4A, 0xB6, 0x00, 0x00, 0xDF, 0xF9, 0x00, 0x00, /* 0xCC-0xCF */
	0xCE, 0xD5, 0x00, 0x00, 0xDF, 0xF6, 0x4A, 0xB7, /* 0xD0-0xD3 */
	0xDF, 0xF8, 0xB1, 0xED, 0x4A, 0xB8, 0xDF, 0xF3, /* 0xD4-0xD7 */
	0x00, 0x00, 0x4A, 0xB9, 0x4A, 0xBA, 0x00, 0x00, /* 0xD8-0xDB */
	0x4A, 0xBB, 0x00, 0x00, 0x00, 0x00, 0xD3, 0xDB, /* 0xDC-0xDF */
	0xDF, 0xFA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0xC1, 0xE7, 0xBB, 0xB8, 0xDF, 0xFC, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0xDF, 0xFB, 0xBF, 0xA4, 0xD2, 0xD9, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0xDF, 0xFD, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x4A, 0xBC, 0xE0, 0xA1, 0x00, 0x00, 0xDF, 0xEE, /* 0xF8-0xFB */
	0xDF, 0xFE, 0x00, 0x00, 0xFA, 0xFD, 0xE0, 0xA2, /* 0xFC-0xFF */
};

static unsigned char u2c_72[512] = {
	0x4A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xC7, 0xFA, 0x4A, 0xBF, /* 0x04-0x07 */
	0x4A, 0xC0, 0x4A, 0xC1, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0xE0, 0xA3, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0xE0, 0xA4, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xC2, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xC3, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x4A, 0xC4, 0xE0, 0xA5, /* 0x18-0x1B */
	0x00, 0x00, 0x4A, 0xC5, 0x00, 0x00, 0x4A, 0xC6, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x4A, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0xE0, 0xA6, 0x00, 0x00, 0xC4, 0xDE, 0x4A, 0xC8, /* 0x28-0x2B */
	0xE0, 0xA8, 0xE0, 0xA7, 0x00, 0x00, 0x4A, 0xC9, /* 0x2C-0x2F */
	0xE0, 0xA9, 0x00, 0x00, 0xE0, 0xAA, 0x00, 0x00, /* 0x30-0x33 */
	0x4A, 0xCA, 0xBC, 0xDF, 0xC9, 0xE3, 0x00, 0x00, /* 0x34-0x37 */
	0x4A, 0xCB, 0x4A, 0xCC, 0xCC, 0xEC, 0xE0, 0xAB, /* 0x38-0x3B */
	0xE0, 0xAC, 0xC1, 0xD6, 0xBC, 0xA4, 0xE0, 0xAD, /* 0x3C-0x3F */
	0xE0, 0xAE, 0x4A, 0xCD, 0x4A, 0xCE, 0x4A, 0xCF, /* 0x40-0x43 */
	0x00, 0x00, 0x4A, 0xD0, 0xE0, 0xAF, 0xCA, 0xD2, /* 0x44-0x47 */
	0xC8, 0xC7, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xB0, /* 0x48-0x4B */
	0xC7, 0xD7, 0x00, 0x00, 0x4A, 0xD1, 0x4A, 0xD2, /* 0x4C-0x4F */
	0x4A, 0xD3, 0x00, 0x00, 0xC4, 0xAD, 0x4A, 0xD4, /* 0x50-0x53 */
	0x00, 0x00, 0x4A, 0xD5, 0x4A, 0xD6, 0x00, 0x00, /* 0x54-0x57 */
	0xE0, 0xB1, 0xB2, 0xE7, 0x4A, 0xD7, 0xB5, 0xED, /* 0x58-0x5B */
	0x4A, 0xD8, 0xCC, 0xC6, 0x4A, 0xD9, 0xCC, 0xB6, /* 0x5C-0x5F */
	0x4A, 0xDA, 0xB2, 0xB4, 0xCF, 0xB4, 0x4A, 0xDB, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCB, 0xD2, /* 0x64-0x67 */
	0x4A, 0xDC, 0xCA, 0xAA, 0x00, 0x00, 0x4A, 0xDD, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x4A, 0xDE, 0x4A, 0xDF, /* 0x6C-0x6F */
	0x00, 0x00, 0x4A, 0xE0, 0xC0, 0xB7, 0x00, 0x00, /* 0x70-0x73 */
	0xE0, 0xB2, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xE1, /* 0x74-0x77 */
	0x4A, 0xE2, 0xC6, 0xC3, 0x00, 0x00, 0x4A, 0xE3, /* 0x78-0x7B */
	0x4A, 0xE4, 0xB8, 0xA3, 0xE0, 0xB3, 0x4A, 0xE5, /* 0x7C-0x7F */

	0xBA, 0xD4, 0xE0, 0xB5, 0xE0, 0xB4, 0x00, 0x00, /* 0x80-0x83 */
	0x4A, 0xE6, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xB6, /* 0x84-0x87 */
	0x00, 0x00, 0x4A, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x4A, 0xE8, 0x4A, 0xE9, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xB7, 0x4A, 0xEA, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xB8, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xEB, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0xB5, 0xBE, 0x00, 0x00, 0xE0, 0xB9, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xBA, /* 0xA4-0xA7 */
	0x4A, 0xEC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0xB8, 0xA4, 0x4A, 0xED, 0x4A, 0xEE, 0xC8, 0xC8, /* 0xAC-0xAF */
	0x00, 0x00, 0xFA, 0xFE, 0xE0, 0xBC, 0x00, 0x00, /* 0xB0-0xB3 */
	0x4A, 0xF0, 0x00, 0x00, 0xBE, 0xF5, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xE0, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0xFB, 0xA1, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x4A, 0xF2, 0xB6, 0xB8, 0xE0, 0xBD, /* 0xC0-0xC3 */
	0xE0, 0xBF, 0x00, 0x00, 0xE0, 0xBE, 0x4A, 0xF3, /* 0xC4-0xC7 */
	0x00, 0x00, 0x4A, 0xF4, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x4A, 0xF5, 0x00, 0x00, 0xE0, 0xC0, 0x00, 0x00, /* 0xCC-0xCF */
	0xB8, 0xD1, 0x00, 0x00, 0xE0, 0xC1, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x4A, 0xF6, 0x4A, 0xF7, 0xB6, 0xE9, /* 0xD4-0xD7 */
	0x4A, 0xF8, 0xC1, 0xC0, 0x00, 0x00, 0xB9, 0xFD, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xF9, /* 0xDC-0xDF */
	0xE0, 0xC3, 0xE0, 0xC4, 0xE0, 0xC2, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x4A, 0xFA, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0xBC, 0xED, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0xC6, 0xC8, 0xB6, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4A, 0xFB, /* 0xF0-0xF3 */
	0x4A, 0xFC, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xC6, /* 0xF4-0xF7 */
	0xC3, 0xAC, 0xE0, 0xC5, 0x4A, 0xFD, 0x4A, 0xFE, /* 0xF8-0xFB */
	0xCF, 0xB5, 0xC7, 0xE2, 0x4B, 0xA1, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_73[512] = {
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xA2, 0x00, 0x00, /* 0x00-0x03 */
	0x4B, 0xA3, 0x4B, 0xA4, 0x00, 0x00, 0x4B, 0xA5, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xC9, 0x4B, 0xA6, /* 0x08-0x0B */
	0x00, 0x00, 0x4B, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xA8, 0x4B, 0xA9, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xCB, 0xE0, 0xC8, /* 0x14-0x17 */
	0x4B, 0xAA, 0x4B, 0xAB, 0x00, 0x00, 0xCC, 0xD4, /* 0x18-0x1B */
	0xE0, 0xCA, 0xE0, 0xCC, 0x4B, 0xAC, 0xCE, 0xC4, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xAD, 0x00, 0x00, /* 0x20-0x23 */
	0xFB, 0xA2, 0xE0, 0xD0, 0x00, 0x00, 0x4B, 0xAF, /* 0x24-0x27 */
	0x4B, 0xB0, 0xE0, 0xCF, 0xC3, 0xF6, 0xC7, 0xAD, /* 0x28-0x2B */
	0x4B, 0xB1, 0x00, 0x00, 0xB8, 0xA5, 0xE0, 0xCE, /* 0x2C-0x2F */
	0x00, 0x00, 0x4B, 0xB2, 0x4B, 0xB3, 0x00, 0x00, /* 0x30-0x33 */
	0xE0, 0xCD, 0x4B, 0xB4, 0xCD, 0xB1, 0xCD, 0xB2, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xB5, 0x4B, 0xB6, /* 0x38-0x3B */
	0x00, 0x00, 0x4B, 0xB7, 0xE0, 0xD1, 0xB1, 0xEE, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4B, 0xB8, /* 0x40-0x43 */
	0xB9, 0xF6, 0xBB, 0xE2, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x4B, 0xB9, 0xE0, 0xD2, 0xE0, 0xD3, /* 0x4C-0x4F */
	0x4B, 0xBA, 0x00, 0x00, 0x4B, 0xBB, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xBC, 0xE0, 0xD5, /* 0x54-0x57 */
	0x4B, 0xBD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x4B, 0xBE, 0x4B, 0xBF, 0x4B, 0xC0, /* 0x5C-0x5F */
	0x4B, 0xC1, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xC3, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xC2, 0x4B, 0xC3, /* 0x64-0x67 */
	0xE0, 0xD7, 0x4B, 0xC4, 0xE0, 0xD6, 0x4B, 0xC5, /* 0x68-0x6B */
	0x4B, 0xC6, 0x00, 0x00, 0x4B, 0xC7, 0x4B, 0xC8, /* 0x6C-0x6F */
	0xE0, 0xD8, 0x4B, 0xC9, 0xB3, 0xCD, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xE0, 0xDA, 0x00, 0x00, 0xFB, 0xA4, /* 0x74-0x77 */
	0xE0, 0xD9, 0x4B, 0xCB, 0xE0, 0xDC, 0xE0, 0xDB, /* 0x78-0x7B */
	0x4B, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x4B, 0xCD, 0x4B, 0xCE, 0x00, 0x00, 0x4B, 0xCF, /* 0x80-0x83 */
	0xB8, 0xBC, 0x4B, 0xD0, 0x4B, 0xD1, 0xCE, 0xA8, /* 0x84-0x87 */
	0x00, 0x00, 0xB6, 0xCC, 0x00, 0x00, 0xB2, 0xA6, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x4B, 0xD2, 0x00, 0x00, /* 0x8C-0x8F */
	0x4B, 0xD3, 0x00, 0x00, 0x00, 0x00, 0x4B, 0xD4, /* 0x90-0x93 */
	0x00, 0x00, 0x4B, 0xD5, 0xB6, 0xEA, 0x4B, 0xD6, /* 0x94-0x97 */
	0x4B, 0xD7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x4B, 0xD8, 0x00, 0x00, 0x4B, 0xD9, 0x4B, 0xDA, /* 0x9C-0x9F */
	0x4B, 0xDB, 0x00, 0x00, 0x4B, 0xDC, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x4B, 0xDD, 0x4B, 0xDE, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0xB4, 0xE1, 0x4B, 0xDF, 0x4B, 0xE0, /* 0xA8-0xAB */
	0x00, 0x00, 0x4B, 0xE1, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xCE, 0xE8, 0xE0, 0xDE, /* 0xB0-0xB3 */
	0x00, 0x00, 0x4B, 0xE2, 0x00, 0x00, 0x4B, 0xE3, /* 0xB4-0xB7 */
	0x00, 0x00, 0x4B, 0xE4, 0x00, 0x00, 0xE0, 0xE0, /* 0xB8-0xBB */
	0x4B, 0xE5, 0xFB, 0xA5, 0x00, 0x00, 0x4B, 0xE7, /* 0xBC-0xBF */
	0xE0, 0xE1, 0x00, 0x00, 0xB2, 0xD1, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x4B, 0xE8, 0x4B, 0xE9, 0x00, 0x00, /* 0xC4-0xC7 */
	0xE0, 0xDD, 0xFB, 0xA6, 0xBB, 0xB9, 0x4B, 0xEB, /* 0xC8-0xCB */
	0x4B, 0xEC, 0xC4, 0xC1, 0xE0, 0xDF, 0x4B, 0xED, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xFB, 0xA9, 0x4B, 0xEF, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0xFB, 0xA7, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x4B, 0xF1, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x4B, 0xF2, 0xE0, 0xE4, 0x00, 0x00, /* 0xDC-0xDF */
	0xBC, 0xEE, 0x4B, 0xF3, 0x00, 0x00, 0xFB, 0xA8, /* 0xE0-0xE3 */
	0x00, 0x00, 0xE0, 0xE2, 0x4B, 0xF5, 0x4B, 0xF6, /* 0xE4-0xE7 */
	0x00, 0x00, 0x4B, 0xF7, 0xB7, 0xBE, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xC8, 0xC9, 0xE0, 0xE3, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0xE0, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x4B, 0xF8, 0xFB, 0xAB, 0x00, 0x00, 0x4B, 0xFA, /* 0xF4-0xF7 */
	0xE0, 0xE9, 0x4B, 0xFB, 0x4B, 0xFC, 0x4B, 0xFD, /* 0xF8-0xFB */
	0x00, 0x00, 0x4B, 0xFE, 0xB8, 0xBD, 0x4C, 0xA1, /* 0xFC-0xFF */
};

static unsigned char u2c_74[512] = {
	0x4C, 0xA2, 0x4C, 0xA3, 0x00, 0x00, 0xB5, 0xE5, /* 0x00-0x03 */
	0x4C, 0xA4, 0xE0, 0xE6, 0xCD, 0xFD, 0xFB, 0xAA, /* 0x04-0x07 */
	0x00, 0x00, 0xCE, 0xB0, 0x4C, 0xA6, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x4C, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x4C, 0xA8, 0x4C, 0xA9, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0xC2, 0xF6, 0x00, 0x00, /* 0x20-0x23 */
	0x4C, 0xAA, 0xE0, 0xE8, 0xFB, 0xAC, 0x00, 0x00, /* 0x24-0x27 */
	0x4C, 0xAC, 0xFB, 0xAE, 0xFB, 0xAD, 0x4C, 0xAF, /* 0x28-0x2B */
	0x4C, 0xB0, 0x4C, 0xB1, 0xFB, 0xAF, 0x4C, 0xB3, /* 0x2C-0x2F */
	0x4C, 0xB4, 0x4C, 0xB5, 0xE0, 0xEA, 0xCE, 0xD6, /* 0x30-0x33 */
	0xB6, 0xD7, 0xC8, 0xFC, 0xC7, 0xCA, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x4C, 0xB6, 0xE0, 0xEB, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xED, /* 0x3C-0x3F */
	0x4C, 0xB7, 0xE0, 0xF0, 0x00, 0x00, 0x4C, 0xB8, /* 0x40-0x43 */
	0x4C, 0xB9, 0x00, 0x00, 0x4C, 0xBA, 0x4C, 0xBB, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4C, 0xBC, /* 0x48-0x4B */
	0x00, 0x00, 0x4C, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x4C, 0xBE, 0x4C, 0xBF, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0xE0, 0xEC, 0x00, 0x00, 0x4C, 0xC0, /* 0x54-0x57 */
	0x00, 0x00, 0xE0, 0xEF, 0xB8, 0xEA, 0xB1, 0xCD, /* 0x58-0x5B */
	0xE0, 0xF1, 0x4C, 0xC1, 0xBF, 0xF0, 0xE0, 0xEE, /* 0x5C-0x5F */
	0xCE, 0xDC, 0x00, 0x00, 0xFB, 0xB0, 0xE0, 0xF4, /* 0x60-0x63 */
	0xF4, 0xA4, 0x00, 0x00, 0x4C, 0xC3, 0x4C, 0xC4, /* 0x64-0x67 */
	0x4C, 0xC5, 0xE0, 0xF2, 0xE0, 0xF5, 0x4C, 0xC6, /* 0x68-0x6B */
	0x00, 0x00, 0x4C, 0xC7, 0x4C, 0xC8, 0xE0, 0xE7, /* 0x6C-0x6F */
	0xE0, 0xF3, 0x4C, 0xC9, 0x4C, 0xCA, 0xBA, 0xBC, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xF6, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xF7, 0x00, 0x00, /* 0x7C-0x7F */

	0x4C, 0xCB, 0x4C, 0xCC, 0x00, 0x00, 0xCD, 0xFE, /* 0x80-0x83 */
	0x00, 0x00, 0x4C, 0xCD, 0x4C, 0xCE, 0x4C, 0xCF, /* 0x84-0x87 */
	0x00, 0x00, 0xFB, 0xB1, 0x00, 0x00, 0xE0, 0xF8, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4C, 0xD1, /* 0x8C-0x8F */
	0x4C, 0xD2, 0x4C, 0xD3, 0x4C, 0xD4, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x4C, 0xD5, 0x4C, 0xD6, 0x4C, 0xD7, 0x00, 0x00, /* 0x98-0x9B */
	0x4C, 0xD8, 0x00, 0x00, 0xE0, 0xF9, 0xFB, 0xB2, /* 0x9C-0x9F */
	0x4C, 0xDA, 0x4C, 0xDB, 0xE0, 0xE5, 0x4C, 0xDC, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x4C, 0xDD, 0xE0, 0xFA, /* 0xA4-0xA7 */
	0x4C, 0xDE, 0x4C, 0xDF, 0x4C, 0xE0, 0x4C, 0xE1, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x4C, 0xE2, 0x4C, 0xE3, /* 0xAC-0xAF */
	0xB4, 0xC4, 0x4C, 0xE4, 0x4C, 0xE5, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x4C, 0xE6, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x4C, 0xE7, 0x00, 0x00, 0x4C, 0xE8, /* 0xB8-0xBB */
	0x00, 0x00, 0xBC, 0xA5, 0x00, 0x00, 0x4C, 0xE9, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x4C, 0xEA, 0x4C, 0xEB, 0xE0, 0xFB, 0x00, 0x00, /* 0xC8-0xCB */
	0x4C, 0xEC, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xFC, /* 0xCC-0xCF */
	0x4C, 0xED, 0x00, 0x00, 0x00, 0x00, 0x4C, 0xEE, /* 0xD0-0xD3 */
	0xE0, 0xFD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x4C, 0xEF, 0x00, 0x00, 0x4C, 0xF0, 0x4C, 0xF1, /* 0xD8-0xDB */
	0xB1, 0xBB, 0x00, 0x00, 0x4C, 0xF2, 0x4C, 0xF3, /* 0xDC-0xDF */
	0xE1, 0xA1, 0x00, 0x00, 0xC9, 0xBB, 0xE1, 0xA2, /* 0xE0-0xE3 */
	0x4C, 0xF4, 0x00, 0x00, 0xB4, 0xA4, 0xE1, 0xA3, /* 0xE4-0xE7 */
	0x4C, 0xF5, 0xE1, 0xA4, 0x4C, 0xF6, 0x4C, 0xF7, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0xE1, 0xA5, 0x4C, 0xF8, /* 0xEC-0xEF */
	0xE1, 0xA7, 0xE1, 0xA8, 0xE1, 0xA6, 0x00, 0x00, /* 0xF0-0xF3 */
	0x4C, 0xF9, 0x00, 0x00, 0xC9, 0xD3, 0xE1, 0xAA, /* 0xF4-0xF7 */
	0xE1, 0xA9, 0x00, 0x00, 0x4C, 0xFA, 0x4C, 0xFB, /* 0xF8-0xFB */
	0x4C, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x4C, 0xFD, /* 0xFC-0xFF */
};

static unsigned char u2c_75[512] = {
	0x00, 0x00, 0xFB, 0xB3, 0x00, 0x00, 0xE1, 0xAC, /* 0x00-0x03 */
	0xE1, 0xAB, 0xE1, 0xAD, 0x4C, 0xFE, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0xE1, 0xAE, 0xE1, 0xB0, 0xE1, 0xAF, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0xB9, 0xF9, 0x4D, 0xA1, 0xE1, 0xB2, /* 0x10-0x13 */
	0x00, 0x00, 0xE1, 0xB1, 0x4D, 0xA2, 0x4D, 0xA3, /* 0x14-0x17 */
	0xB4, 0xC5, 0x00, 0x00, 0xBF, 0xD3, 0x00, 0x00, /* 0x18-0x1B */
	0xC5, 0xBC, 0x00, 0x00, 0xE1, 0xB3, 0xC0, 0xB8, /* 0x1C-0x1F */
	0x4D, 0xA4, 0x4D, 0xA5, 0x00, 0x00, 0xBB, 0xBA, /* 0x20-0x23 */
	0x4D, 0xA6, 0xB1, 0xF9, 0xE1, 0xB4, 0x4D, 0xA7, /* 0x24-0x27 */
	0xCD, 0xD1, 0x4D, 0xA8, 0x4D, 0xA9, 0xCA, 0xE3, /* 0x28-0x2B */
	0xE1, 0xB5, 0x00, 0x00, 0x00, 0x00, 0xF9, 0xEC, /* 0x2C-0x2F */
	0xC5, 0xC4, 0xCD, 0xB3, 0xB9, 0xC3, 0xBF, 0xBD, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x4D, 0xAB, 0xC3, 0xCB, /* 0x34-0x37 */
	0xD2, 0xB4, 0x4D, 0xAC, 0xC4, 0xAE, 0xB2, 0xE8, /* 0x38-0x3B */
	0xE1, 0xB6, 0x4D, 0xAD, 0x4D, 0xAE, 0x4D, 0xAF, /* 0x3C-0x3F */
	0x4D, 0xB0, 0x00, 0x00, 0x00, 0x00, 0x4D, 0xB1, /* 0x40-0x43 */
	0xE1, 0xB7, 0x00, 0x00, 0xE1, 0xBC, 0x4D, 0xB2, /* 0x44-0x47 */
	0x4D, 0xB3, 0xE1, 0xBA, 0xE1, 0xB9, 0xDA, 0xC2, /* 0x48-0x4B */
	0xB3, 0xA6, 0xE1, 0xB8, 0x4D, 0xB4, 0xB0, 0xDA, /* 0x4C-0x4F */
	0x4D, 0xB5, 0xC8, 0xAA, 0x4D, 0xB6, 0x00, 0x00, /* 0x50-0x53 */
	0xC8, 0xCA, 0x00, 0x00, 0x00, 0x00, 0x4D, 0xB7, /* 0x54-0x57 */
	0x00, 0x00, 0xCE, 0xB1, 0xE1, 0xBD, 0xE1, 0xBB, /* 0x58-0x5B */
	0xC3, 0xDC, 0xC0, 0xA6, 0x4D, 0xB8, 0x4D, 0xB9, /* 0x5C-0x5F */
	0xC8, 0xAB, 0x4D, 0xBA, 0xC9, 0xAD, 0x00, 0x00, /* 0x60-0x63 */
	0xE1, 0xBF, 0xCE, 0xAC, 0xB7, 0xCD, 0xE1, 0xC0, /* 0x64-0x67 */
	0x00, 0x00, 0xE1, 0xBE, 0xC8, 0xD6, 0xE1, 0xC1, /* 0x68-0x6B */
	0x00, 0x00, 0xE1, 0xC2, 0x00, 0x00, 0xFB, 0xB4, /* 0x6C-0x6F */
	0xB0, 0xDB, 0x4D, 0xBC, 0x00, 0x00, 0xBE, 0xF6, /* 0x70-0x73 */
	0xE1, 0xC7, 0x00, 0x00, 0xE1, 0xC4, 0xC6, 0xED, /* 0x74-0x77 */
	0xE1, 0xC3, 0x4D, 0xBD, 0x4D, 0xBE, 0x4D, 0xBF, /* 0x78-0x7B */
	0x4D, 0xC0, 0x4D, 0xC1, 0x4D, 0xC2, 0xB5, 0xA6, /* 0x7C-0x7F */

	0x00, 0x00, 0x4D, 0xC3, 0xE1, 0xCA, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x4D, 0xC4, 0xE1, 0xC5, 0xE1, 0xC6, /* 0x84-0x87 */
	0x00, 0x00, 0xE1, 0xC9, 0xE1, 0xC8, 0xC9, 0xA5, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0xC1, 0xC2, 0xC1, 0xC1, /* 0x8C-0x8F */
	0x4D, 0xC5, 0xB5, 0xBF, 0x4D, 0xC6, 0x4D, 0xC7, /* 0x90-0x93 */
	0xE1, 0xCB, 0x4D, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x4D, 0xC9, 0xE1, 0xCC, 0x00, 0x00, /* 0x98-0x9B */
	0x4D, 0xCA, 0xE1, 0xCD, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x4D, 0xCB, 0xE1, 0xCF, /* 0xA0-0xA3 */
	0x4D, 0xCC, 0xE1, 0xCE, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xD6, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0xE1, 0xD7, 0xC8, 0xE8, 0xE1, 0xD1, /* 0xB0-0xB3 */
	0x4D, 0xCD, 0xE1, 0xD3, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xE1, 0xD5, 0xBF, 0xBE, 0x4D, 0xCE, 0x00, 0x00, /* 0xB8-0xBB */
	0xE1, 0xD6, 0xE1, 0xD4, 0xBC, 0xC0, 0x4D, 0xCF, /* 0xBC-0xBF */
	0x4D, 0xD0, 0x4D, 0xD1, 0xE1, 0xD0, 0xE1, 0xD2, /* 0xC0-0xC3 */
	0x4D, 0xD2, 0xC9, 0xC2, 0x4D, 0xD3, 0xBE, 0xC9, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0xE1, 0xD9, 0x00, 0x00, /* 0xC8-0xCB */
	0x4D, 0xD4, 0xE1, 0xD8, 0x4D, 0xD5, 0x4D, 0xD6, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xE1, 0xDA, 0x00, 0x00, /* 0xD0-0xD3 */
	0xBC, 0xA6, 0xBA, 0xAF, 0x00, 0x00, 0x4D, 0xD7, /* 0xD4-0xD7 */
	0xC5, 0xF7, 0xE1, 0xDB, 0x00, 0x00, 0xC4, 0xCB, /* 0xD8-0xDB */
	0x4D, 0xD8, 0x00, 0x00, 0xE1, 0xDD, 0x4D, 0xD9, /* 0xDC-0xDF */
	0x4D, 0xDA, 0x4D, 0xDB, 0xCE, 0xA1, 0xE1, 0xDC, /* 0xE0-0xE3 */
	0x4D, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x4D, 0xDD, /* 0xE4-0xE7 */
	0x00, 0x00, 0xC1, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x4D, 0xDE, 0x00, 0x00, 0x4D, 0xDF, 0x4D, 0xE0, /* 0xEC-0xEF */
	0xE1, 0xE2, 0x4D, 0xE1, 0xE1, 0xE4, 0xE1, 0xE5, /* 0xF0-0xF3 */
	0xC3, 0xD4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x4D, 0xE2, 0xE1, 0xE3, 0x00, 0x00, /* 0xF8-0xFB */
	0xE1, 0xE0, 0x00, 0x00, 0xE1, 0xDE, 0xE1, 0xDF, /* 0xFC-0xFF */
};

static unsigned char u2c_76[512] = {
	0x4D, 0xE3, 0xE1, 0xE1, 0x4D, 0xE4, 0x4D, 0xE5, /* 0x00-0x03 */
	0x4D, 0xE6, 0x00, 0x00, 0x00, 0x00, 0x4D, 0xE7, /* 0x04-0x07 */
	0x4D, 0xE8, 0xE1, 0xE8, 0x4D, 0xE9, 0xE1, 0xE6, /* 0x08-0x0B */
	0x4D, 0xEA, 0xE1, 0xE7, 0x00, 0x00, 0x4D, 0xEB, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x4D, 0xEC, 0x4D, 0xED, /* 0x10-0x13 */
	0x00, 0x00, 0x4D, 0xEE, 0x4D, 0xEF, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x4D, 0xF0, 0x00, 0x00, 0x4D, 0xF1, /* 0x18-0x1B */
	0x4D, 0xF2, 0x4D, 0xF3, 0x4D, 0xF4, 0xE1, 0xE9, /* 0x1C-0x1F */
	0xE1, 0xEB, 0xE1, 0xEC, 0xE1, 0xED, 0x4D, 0xF5, /* 0x20-0x23 */
	0xE1, 0xEE, 0x4D, 0xF6, 0x4D, 0xF7, 0xE1, 0xEA, /* 0x24-0x27 */
	0x00, 0x00, 0x4D, 0xF8, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x4D, 0xF9, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xE1, 0xF0, 0x00, 0x00, 0x4D, 0xFA, 0x4D, 0xFB, /* 0x30-0x33 */
	0xE1, 0xEF, 0x4D, 0xFC, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x4D, 0xFD, 0x4D, 0xFE, 0x4E, 0xA1, 0xE1, 0xF1, /* 0x38-0x3B */
	0x4E, 0xA2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x4E, 0xA4, 0x4E, 0xA5, 0xCE, 0xC5, 0x4E, 0xA6, /* 0x40-0x43 */
	0x4E, 0xA7, 0x4E, 0xA8, 0xE1, 0xF4, 0xE1, 0xF2, /* 0x44-0x47 */
	0xE1, 0xF3, 0x4E, 0xA9, 0x4E, 0xA3, 0x4E, 0xAA, /* 0x48-0x4B */
	0xB4, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0xCC, 0xFE, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x4E, 0xAB, 0xCA, 0xCA, 0x00, 0x00, /* 0x54-0x57 */
	0xE1, 0xF6, 0x4E, 0xAC, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0xE1, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x4E, 0xAD, /* 0x5C-0x5F */
	0x00, 0x00, 0xE1, 0xF7, 0xE1, 0xF8, 0x00, 0x00, /* 0x60-0x63 */
	0x4E, 0xAE, 0x4E, 0xAF, 0x00, 0x00, 0xE1, 0xFC, /* 0x64-0x67 */
	0xE1, 0xF9, 0xE1, 0xFA, 0xE1, 0xFB, 0x00, 0x00, /* 0x68-0x6B */
	0xE1, 0xFD, 0x4E, 0xB0, 0x4E, 0xB1, 0x4E, 0xB2, /* 0x6C-0x6F */
	0xE1, 0xFE, 0x4E, 0xB3, 0xE2, 0xA1, 0x00, 0x00, /* 0x70-0x73 */
	0x4E, 0xB4, 0x00, 0x00, 0xE2, 0xA2, 0x00, 0x00, /* 0x74-0x77 */
	0xE2, 0xA3, 0x00, 0x00, 0xC8, 0xAF, 0xC5, 0xD0, /* 0x78-0x7B */
	0xE2, 0xA4, 0xC7, 0xF2, 0xC9, 0xB4, 0x00, 0x00, /* 0x7C-0x7F */

	0xE2, 0xA5, 0x4E, 0xB5, 0xFB, 0xB5, 0xE2, 0xA6, /* 0x80-0x83 */
	0xC5, 0xAA, 0x4E, 0xB6, 0xB3, 0xA7, 0xB9, 0xC4, /* 0x84-0x87 */
	0xE2, 0xA7, 0x00, 0x00, 0x00, 0x00, 0xE2, 0xA8, /* 0x88-0x8B */
	0x4E, 0xB7, 0x4E, 0xB8, 0xE2, 0xA9, 0x00, 0x00, /* 0x8C-0x8F */
	0xBB, 0xA9, 0x00, 0x00, 0x00, 0x00, 0xE2, 0xAB, /* 0x90-0x93 */
	0x00, 0x00, 0x4E, 0xB9, 0xE2, 0xAA, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xE2, 0xAC, 0xE2, 0xAD, 0xFB, 0xB8, /* 0x98-0x9B */
	0xFB, 0xB6, 0x4E, 0xBC, 0xFB, 0xB7, 0x4E, 0xBD, /* 0x9C-0x9F */
	0x4E, 0xBE, 0x00, 0x00, 0x4E, 0xBF, 0x4E, 0xC0, /* 0xA0-0xA3 */
	0x4E, 0xC1, 0x4E, 0xC2, 0xFB, 0xB9, 0x4E, 0xC4, /* 0xA4-0xA7 */
	0x4E, 0xC5, 0x00, 0x00, 0x4E, 0xC6, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x4E, 0xC7, 0xC8, 0xE9, 0x00, 0x00, /* 0xAC-0xAF */
	0xE2, 0xAE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0xE2, 0xAF, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xE9, /* 0xB4-0xB7 */
	0xE2, 0xB0, 0xE2, 0xB1, 0xE2, 0xB2, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x4E, 0xC8, 0x00, 0x00, 0xBB, 0xAE, /* 0xBC-0xBF */
	0x00, 0x00, 0x4E, 0xC9, 0xE2, 0xB3, 0xC7, 0xD6, /* 0xC0-0xC3 */
	0x00, 0x00, 0x4E, 0xCA, 0xCB, 0xDF, 0x00, 0x00, /* 0xC4-0xC7 */
	0xB1, 0xCE, 0x4E, 0xCB, 0xB1, 0xD7, 0x4E, 0xCC, /* 0xC8-0xCB */
	0x4E, 0xCD, 0xE2, 0xB4, 0x4E, 0xCE, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xE2, 0xB6, 0x00, 0x00, /* 0xD0-0xD3 */
	0x4E, 0xCF, 0x00, 0x00, 0xE2, 0xB5, 0xC5, 0xF0, /* 0xD4-0xD7 */
	0x00, 0x00, 0x4E, 0xD0, 0x00, 0x00, 0xC0, 0xB9, /* 0xD8-0xDB */
	0xDD, 0xB9, 0x00, 0x00, 0xE2, 0xB7, 0xCC, 0xC1, /* 0xDC-0xDF */
	0x4E, 0xD1, 0xE2, 0xB8, 0x00, 0x00, 0xB4, 0xC6, /* 0xE0-0xE3 */
	0xC8, 0xD7, 0xE2, 0xB9, 0x4E, 0xD2, 0xE2, 0xBA, /* 0xE4-0xE7 */
	0x4E, 0xD3, 0x00, 0x00, 0xE2, 0xBB, 0x00, 0x00, /* 0xE8-0xEB */
	0x4E, 0xD4, 0x00, 0x00, 0xCC, 0xDC, 0x00, 0x00, /* 0xEC-0xEF */
	0x4E, 0xD5, 0x4E, 0xD6, 0xCC, 0xD5, 0x00, 0x00, /* 0xF0-0xF3 */
	0xC4, 0xBE, 0x00, 0x00, 0x4E, 0xD7, 0x00, 0x00, /* 0xF4-0xF7 */
	0xC1, 0xEA, 0x4E, 0xD8, 0x00, 0x00, 0xE2, 0xBD, /* 0xF8-0xFB */
	0x4E, 0xD9, 0x00, 0x00, 0xBD, 0xE2, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_77[512] = {
	0x4E, 0xDA, 0xBE, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0xE2, 0xC0, 0x00, 0x00, 0x4E, 0xDB, 0xE2, 0xBF, /* 0x04-0x07 */
	0xE2, 0xBE, 0xC8, 0xFD, 0x4E, 0xDC, 0xB4, 0xC7, /* 0x08-0x0B */
	0xB8, 0xA9, 0x00, 0x00, 0x4E, 0xDD, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x4E, 0xDE, 0x00, 0x00, /* 0x10-0x13 */
	0x4E, 0xDF, 0x4E, 0xE0, 0x00, 0x00, 0x4E, 0xE1, /* 0x14-0x17 */
	0x00, 0x00, 0x4E, 0xE2, 0x4E, 0xE3, 0xE2, 0xC6, /* 0x18-0x1B */
	0x4E, 0xE4, 0x00, 0x00, 0xE2, 0xC3, 0xBF, 0xBF, /* 0x1C-0x1F */
	0xCC, 0xB2, 0x00, 0x00, 0x4E, 0xE5, 0x00, 0x00, /* 0x20-0x23 */
	0xE2, 0xC2, 0xE2, 0xC4, 0xE2, 0xC5, 0x00, 0x00, /* 0x24-0x27 */
	0x4E, 0xE6, 0xE2, 0xC1, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x4E, 0xE7, 0x4E, 0xE8, 0x4E, 0xE9, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x4E, 0xEA, 0x4E, 0xEB, 0x4E, 0xEC, 0xE2, 0xC7, /* 0x34-0x37 */
	0xE2, 0xC8, 0x4E, 0xED, 0xC4, 0xAF, 0x00, 0x00, /* 0x38-0x3B */
	0xB4, 0xE3, 0x4E, 0xEE, 0x4E, 0xEF, 0x00, 0x00, /* 0x3C-0x3F */
	0xC3, 0xE5, 0x00, 0x00, 0x4E, 0xF0, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x4E, 0xF1, 0xFB, 0xBB, 0xE2, 0xC9, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x4E, 0xF3, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x4E, 0xF4, 0x4E, 0xF5, 0x4E, 0xF6, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x4E, 0xF7, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x4E, 0xF8, 0x4E, 0xF9, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0xE2, 0xCA, 0xE2, 0xCD, /* 0x58-0x5B */
	0x4E, 0xFA, 0x00, 0x00, 0x4E, 0xFB, 0x4E, 0xFC, /* 0x5C-0x5F */
	0x4E, 0xFD, 0xBF, 0xE7, 0x4E, 0xFE, 0xC6, 0xC4, /* 0x60-0x63 */
	0x4F, 0xA1, 0xE2, 0xCE, 0xCB, 0xD3, 0x4F, 0xA2, /* 0x64-0x67 */
	0xE2, 0xCB, 0x00, 0x00, 0x4F, 0xA3, 0xE2, 0xCC, /* 0x68-0x6B */
	0x4F, 0xA4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x4F, 0xA5, 0x00, 0x00, 0x4F, 0xA6, 0x4F, 0xA7, /* 0x70-0x73 */
	0x4F, 0xA8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0xE2, 0xD1, 0x4F, 0xA9, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x4F, 0xAA, 0xE2, 0xD0, 0xE2, 0xCF, /* 0x7C-0x7F */

	0x4F, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x4F, 0xAC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE2, 0xD3, /* 0x88-0x8B */
	0x4F, 0xAD, 0x4F, 0xAE, 0xE2, 0xD2, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0xE2, 0xD4, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x4F, 0xAF, 0x4F, 0xB0, 0x4F, 0xB1, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x4F, 0xB2, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0xE2, 0xD6, 0x4F, 0xB3, /* 0x9C-0x9F */
	0xE2, 0xD5, 0x00, 0x00, 0x4F, 0xB4, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xCA, 0xCD, 0x00, 0x00, 0x4F, 0xB5, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x4F, 0xB6, 0x00, 0x00, /* 0xA8-0xAB */
	0xBD, 0xD6, 0xCE, 0xC6, 0x4F, 0xB7, 0x4F, 0xB8, /* 0xAC-0xAF */
	0xE2, 0xD7, 0x4F, 0xB9, 0x00, 0x00, 0xC6, 0xB7, /* 0xB0-0xB3 */
	0x00, 0x00, 0x4F, 0xBA, 0xE2, 0xD8, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xE2, 0xD9, 0x00, 0x00, 0xE2, 0xDD, /* 0xB8-0xBB */
	0xE2, 0xDB, 0xE2, 0xDC, 0x4F, 0xBB, 0xE2, 0xDA, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4F, 0xBC, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE2, 0xDE, /* 0xC4-0xC7 */
	0x00, 0x00, 0x4F, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0xE2, 0xDF, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x4F, 0xBE, 0x4F, 0xBF, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x4F, 0xC0, 0x00, 0x00, 0xE2, 0xE0, /* 0xD4-0xD7 */
	0x00, 0x00, 0x4F, 0xC1, 0xE2, 0xE1, 0xCC, 0xB7, /* 0xD8-0xDB */
	0xE2, 0xE2, 0x00, 0x00, 0x4F, 0xC2, 0x4F, 0xC3, /* 0xDC-0xDF */
	0x4F, 0xC4, 0x00, 0x00, 0xCC, 0xF0, 0xE2, 0xE3, /* 0xE0-0xE3 */
	0x4F, 0xC5, 0xC3, 0xCE, 0x4F, 0xC6, 0xC7, 0xEA, /* 0xE4-0xE7 */
	0x00, 0x00, 0xB6, 0xEB, 0x4F, 0xC7, 0x00, 0x00, /* 0xE8-0xEB */
	0x4F, 0xC8, 0xC3, 0xBB, 0xE2, 0xE4, 0xB6, 0xBA, /* 0xEC-0xEF */
	0x4F, 0xC9, 0x4F, 0xCA, 0x00, 0x00, 0xC0, 0xD0, /* 0xF0-0xF3 */
	0x4F, 0xCB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x4F, 0xCC, 0x00, 0x00, 0x00, 0x00, 0x4F, 0xCD, /* 0xF8-0xFB */
	0xE2, 0xE5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_78[512] = {
	0x00, 0x00, 0x00, 0x00, 0xBA, 0xBD, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x4F, 0xCE, 0x4F, 0xCF, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x4F, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0xE2, 0xE6, 0x4F, 0xD1, 0x4F, 0xD2, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x4F, 0xD3, 0xE2, 0xE7, 0x00, 0x00, /* 0x10-0x13 */
	0xB8, 0xA6, 0xBA, 0xD5, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x4F, 0xD4, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0xE2, 0xE9, 0xFB, 0xBD, 0x4F, 0xD6, 0x4F, 0xD7, /* 0x20-0x23 */
	0x00, 0x00, 0xC5, 0xD6, 0xBA, 0xD6, 0xB5, 0xCE, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x4F, 0xD8, 0x4F, 0xD9, 0x00, 0x00, /* 0x2C-0x2F */
	0x4F, 0xDA, 0x00, 0x00, 0xCB, 0xA4, 0x00, 0x00, /* 0x30-0x33 */
	0xC7, 0xCB, 0x4F, 0xDB, 0x00, 0x00, 0x4F, 0xDC, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xC5, 0xD7, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB9, 0xDC, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4F, 0xDD, /* 0x40-0x43 */
	0x4F, 0xDE, 0xE2, 0xEB, 0x00, 0x00, 0x4F, 0xDF, /* 0x44-0x47 */
	0x4F, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x4F, 0xE1, 0x00, 0x00, 0xFB, 0xBE, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x4F, 0xE3, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x4F, 0xE4, 0xBE, 0xCB, 0x4F, 0xE5, 0x00, 0x00, /* 0x5C-0x5F */
	0x4F, 0xE6, 0x4F, 0xE7, 0x00, 0x00, 0x4F, 0xE8, /* 0x60-0x63 */
	0xFB, 0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x4F, 0xEA, 0x00, 0x00, 0x4F, 0xEB, 0xCE, 0xB2, /* 0x68-0x6B */
	0xB9, 0xC5, 0x00, 0x00, 0x4F, 0xEC, 0xB8, 0xA7, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0xC8, 0xA3, 0x00, 0x00, /* 0x70-0x73 */
	0xE2, 0xED, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0xFB, 0xC0, 0x00, 0x00, /* 0x78-0x7B */
	0xE2, 0xEF, 0x00, 0x00, 0x4F, 0xEE, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0xB8, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0xE2, 0xEE, 0xC4, 0xF6, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x4F, 0xEF, 0x00, 0x00, /* 0x88-0x8B */
	0xE2, 0xF1, 0xB3, 0xB7, 0xE2, 0xEC, 0x4F, 0xF0, /* 0x8C-0x8F */
	0x00, 0x00, 0xC8, 0xEA, 0x00, 0x00, 0xB1, 0xB0, /* 0x90-0x93 */
	0x4F, 0xF1, 0xBA, 0xEC, 0x00, 0x00, 0xCF, 0xD2, /* 0x94-0x97 */
	0x4F, 0xF2, 0x00, 0x00, 0xE2, 0xF0, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x4F, 0xF4, 0x4F, 0xF5, 0x4F, 0xF6, /* 0x9C-0x9F */
	0x00, 0x00, 0x4F, 0xF3, 0x00, 0x00, 0xE2, 0xF2, /* 0xA0-0xA3 */
	0x4F, 0xF7, 0x00, 0x00, 0x00, 0x00, 0xCA, 0xCB, /* 0xA4-0xA7 */
	0x4F, 0xF8, 0xC0, 0xD9, 0xE2, 0xF4, 0x00, 0x00, /* 0xA8-0xAB */
	0x4F, 0xF9, 0x4F, 0xFA, 0x00, 0x00, 0xE2, 0xF5, /* 0xAC-0xAF */
	0x4F, 0xFB, 0x4F, 0xFC, 0x4F, 0xFD, 0x4F, 0xFE, /* 0xB0-0xB3 */
	0x00, 0x00, 0xE2, 0xF3, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0xB3, 0xCE, 0x50, 0xA1, /* 0xB8-0xBB */
	0xE2, 0xFB, 0x50, 0xA2, 0xE2, 0xFA, 0x50, 0xA3, /* 0xBC-0xBF */
	0x00, 0x00, 0xBC, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0xE2, 0xFC, 0xE2, 0xF7, 0x50, 0xA4, /* 0xC4-0xC7 */
	0x50, 0xA5, 0x50, 0xA6, 0xE2, 0xFD, 0xE2, 0xF8, /* 0xC8-0xCB */
	0x50, 0xA7, 0x00, 0x00, 0x50, 0xA8, 0x00, 0x00, /* 0xCC-0xCF */
	0xC8, 0xD8, 0xE2, 0xF6, 0x50, 0xA9, 0x50, 0xAA, /* 0xD0-0xD3 */
	0xE2, 0xF9, 0x50, 0xAB, 0x50, 0xAC, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xE3, 0xA2, 0x50, 0xAE, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0xAF, /* 0xDC-0xDF */
	0x50, 0xB0, 0x50, 0xB1, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x50, 0xAD, 0x00, 0x00, 0x50, 0xB2, 0xE3, 0xA1, /* 0xE4-0xE7 */
	0xCB, 0xE1, 0x00, 0x00, 0x50, 0xB3, 0x00, 0x00, /* 0xE8-0xEB */
	0xE2, 0xFE, 0x00, 0x00, 0x00, 0x00, 0xB0, 0xEB, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x50, 0xB4, 0x50, 0xB5, /* 0xF0-0xF3 */
	0xE3, 0xA4, 0x00, 0x00, 0x50, 0xB7, 0x50, 0xB8, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x50, 0xB9, 0x50, 0xBA, /* 0xF8-0xFB */
	0x00, 0x00, 0xE3, 0xA3, 0x00, 0x00, 0x50, 0xBB, /* 0xFC-0xFF */
};

static unsigned char u2c_79[512] = {
	0x50, 0xB6, 0xBE, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x50, 0xBC, 0xE3, 0xA5, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x50, 0xBD, 0x00, 0x00, 0xC1, 0xC3, 0x00, 0x00, /* 0x0C-0x0F */
	0x50, 0xBE, 0xE3, 0xA7, 0xE3, 0xA6, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xE3, 0xA8, 0x50, 0xBF, 0x00, 0x00, /* 0x18-0x1B */
	0x50, 0xC0, 0x00, 0x00, 0x50, 0xC1, 0x50, 0xC2, /* 0x1C-0x1F */
	0x50, 0xC3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x50, 0xC4, 0xE2, 0xE8, 0x50, 0xC5, /* 0x24-0x27 */
	0x00, 0x00, 0x50, 0xC6, 0xE2, 0xEA, 0xE3, 0xAA, /* 0x28-0x2B */
	0xE3, 0xA9, 0x50, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xFB, 0xC1, 0x50, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x50, 0xC9, 0x50, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xBC, 0xA8, 0x50, 0xCB, /* 0x38-0x3B */
	0xCE, 0xE9, 0x50, 0xCC, 0xBC, 0xD2, 0x50, 0xCD, /* 0x3C-0x3F */
	0xE3, 0xAB, 0xB7, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x50, 0xCE, 0x50, 0xCF, 0x50, 0xD0, 0xB5, 0xC0, /* 0x44-0x47 */
	0xB5, 0xA7, 0xBB, 0xE3, 0x50, 0xD1, 0x50, 0xD2, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0xD3, /* 0x4C-0x4F */
	0xCD, 0xB4, 0x50, 0xD4, 0x00, 0x00, 0xE3, 0xB1, /* 0x50-0x53 */
	0x50, 0xD5, 0xE3, 0xB0, 0xC1, 0xC4, 0xE3, 0xAD, /* 0x54-0x57 */
	0x50, 0xD6, 0x00, 0x00, 0xE3, 0xAF, 0x50, 0xD7, /* 0x58-0x5B */
	0x50, 0xD8, 0xBD, 0xCB, 0xBF, 0xC0, 0xE3, 0xAE, /* 0x5C-0x5F */
	0xE3, 0xAC, 0x00, 0x00, 0xC7, 0xAA, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0xBE, 0xCD, 0x00, 0x00, 0x50, 0xD9, /* 0x64-0x67 */
	0xC9, 0xBC, 0x50, 0xDA, 0x00, 0x00, 0x50, 0xDB, /* 0x68-0x6B */
	0x00, 0x00, 0xBA, 0xD7, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x50, 0xDC, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC5, 0xF8, /* 0x74-0x77 */
	0x00, 0x00, 0x50, 0xDD, 0xE3, 0xB2, 0x50, 0xDE, /* 0x78-0x7B */
	0x50, 0xDF, 0x00, 0x00, 0x50, 0xE0, 0xE3, 0xB3, /* 0x7C-0x7F */

	0xE3, 0xC9, 0xB6, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0xCF, 0xBD, 0xC1, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xE3, 0xB4, 0x50, 0xE1, /* 0x88-0x8B */
	0x50, 0xE2, 0xB2, 0xD2, 0xC4, 0xF7, 0xCA, 0xA1, /* 0x8C-0x8F */
	0x00, 0x00, 0x50, 0xE3, 0x00, 0x00, 0x50, 0xE4, /* 0x90-0x93 */
	0xFB, 0xC5, 0x50, 0xE6, 0x50, 0xE7, 0x00, 0x00, /* 0x94-0x97 */
	0x50, 0xE8, 0x00, 0x00, 0x00, 0x00, 0xFB, 0xC7, /* 0x98-0x9B */
	0x50, 0xEA, 0xE3, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x50, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0xB5, 0xFA, 0xE3, 0xB6, /* 0xA4-0xA7 */
	0x50, 0xEC, 0x50, 0xED, 0xE3, 0xB8, 0x50, 0xEE, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0xE3, 0xB9, 0x50, 0xEF, /* 0xAC-0xAF */
	0xC7, 0xA9, 0x50, 0xF0, 0x00, 0x00, 0xE3, 0xBA, /* 0xB0-0xB3 */
	0x50, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x50, 0xF2, 0xE3, 0xBB, 0xE3, 0xBC, 0x50, 0xF3, /* 0xB8-0xBB */
	0x00, 0x00, 0xB6, 0xD9, 0xB2, 0xD3, 0xC6, 0xC5, /* 0xBC-0xBF */
	0xBD, 0xA8, 0xBB, 0xE4, 0x50, 0xF4, 0x00, 0x00, /* 0xC0-0xC3 */
	0x50, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x50, 0xF6, /* 0xC4-0xC7 */
	0x50, 0xF7, 0xE3, 0xBD, 0x50, 0xF8, 0xBD, 0xA9, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0xF9, /* 0xCC-0xCF */
	0x00, 0x00, 0xB2, 0xCA, 0xC9, 0xC3, 0x00, 0x00, /* 0xD0-0xD3 */
	0x50, 0xFA, 0xE3, 0xBE, 0x50, 0xFB, 0x00, 0x00, /* 0xD4-0xD7 */
	0xC8, 0xEB, 0x00, 0x00, 0x50, 0xFC, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x50, 0xFD, 0x50, 0xFE, 0xC1, 0xC5, /* 0xDC-0xDF */
	0x51, 0xA1, 0xE3, 0xC1, 0x51, 0xA2, 0xE3, 0xC2, /* 0xE0-0xE3 */
	0xC7, 0xE9, 0x51, 0xA3, 0xBF, 0xC1, 0xE3, 0xBF, /* 0xE4-0xE7 */
	0x00, 0x00, 0xC3, 0xE1, 0x51, 0xA4, 0x51, 0xA5, /* 0xE8-0xEB */
	0xE3, 0xC0, 0x51, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0xBE, 0xCE, 0x51, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x51, 0xA8, 0x00, 0x00, 0x00, 0x00, 0xB0, 0xDC, /* 0xF8-0xFB */
	0x51, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_7A[512] = {
	0xB5, 0xA9, 0x00, 0x00, 0x51, 0xAA, 0x51, 0xAB, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0xAC, /* 0x04-0x07 */
	0xE3, 0xC3, 0x51, 0xAD, 0x51, 0xAE, 0xC4, 0xF8, /* 0x08-0x0B */
	0x51, 0xAF, 0xE3, 0xC4, 0xC0, 0xC7, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x51, 0xB0, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0xCC, 0xAD, 0x51, 0xB1, 0x00, 0x00, 0xC9, 0xA3, /* 0x14-0x17 */
	0xE3, 0xC5, 0xE3, 0xC6, 0xC3, 0xD5, 0x51, 0xB2, /* 0x18-0x1B */
	0xCE, 0xC7, 0x00, 0x00, 0x51, 0xB3, 0xE3, 0xC8, /* 0x1C-0x1F */
	0xE3, 0xC7, 0x51, 0xB4, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0xB5, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0xB6, /* 0x28-0x2B */
	0x00, 0x00, 0x51, 0xB7, 0xBC, 0xEF, 0x51, 0xB8, /* 0x2C-0x2F */
	0x51, 0xB9, 0xE3, 0xCA, 0xB0, 0xF0, 0x00, 0x00, /* 0x30-0x33 */
	0x51, 0xBA, 0x51, 0xBB, 0x00, 0x00, 0xE3, 0xCD, /* 0x34-0x37 */
	0x51, 0xBC, 0x51, 0xBD, 0x51, 0xBE, 0xE3, 0xCB, /* 0x38-0x3B */
	0xB2, 0xD4, 0xB7, 0xCE, 0xE3, 0xCC, 0xB9, 0xC6, /* 0x3C-0x3F */
	0xB9, 0xF2, 0x00, 0x00, 0xCA, 0xE6, 0xE3, 0xCE, /* 0x40-0x43 */
	0x51, 0xBF, 0x51, 0xC0, 0xCB, 0xD4, 0x51, 0xC1, /* 0x44-0x47 */
	0x51, 0xC2, 0xE3, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x51, 0xC3, 0xC0, 0xD1, 0xB1, 0xCF, 0xB2, 0xBA, /* 0x4C-0x4F */
	0xB0, 0xAC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x51, 0xC4, 0x51, 0xC5, 0xE3, 0xCF, /* 0x54-0x57 */
	0x00, 0x00, 0x51, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x51, 0xC7, 0x51, 0xC8, 0x00, 0x00, 0x51, 0xC9, /* 0x5C-0x5F */
	0x51, 0xCA, 0xE3, 0xD1, 0xE3, 0xD2, 0xBE, 0xF7, /* 0x60-0x63 */
	0x00, 0x00, 0x51, 0xCB, 0x00, 0x00, 0x51, 0xCC, /* 0x64-0x67 */
	0x00, 0x00, 0xE3, 0xD3, 0x51, 0xCD, 0xB3, 0xCF, /* 0x68-0x6B */
	0x00, 0x00, 0x51, 0xCE, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xE3, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0xB7, 0xEA, 0x51, 0xCF, 0xB5, 0xE6, 0x00, 0x00, /* 0x74-0x77 */
	0x51, 0xD0, 0xE3, 0xD6, 0xB6, 0xF5, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0xE3, 0xD7, 0x51, 0xD1, 0xC0, 0xFC, /* 0x7C-0x7F */

	0x51, 0xD2, 0xC6, 0xCD, 0x51, 0xD3, 0xC0, 0xE0, /* 0x80-0x83 */
	0xBA, 0xF5, 0x51, 0xD4, 0x51, 0xD5, 0x00, 0x00, /* 0x84-0x87 */
	0xE3, 0xD8, 0x00, 0x00, 0x51, 0xD6, 0x51, 0xD7, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x51, 0xD8, 0x51, 0xD9, 0xC3, 0xE2, 0xC1, 0xEB, /* 0x90-0x93 */
	0x51, 0xDA, 0xE3, 0xDA, 0xE3, 0xDC, 0xE3, 0xD9, /* 0x94-0x97 */
	0xE3, 0xDB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x51, 0xDB, 0xB7, 0xA2, /* 0x9C-0x9F */
	0x51, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x51, 0xDD, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0xE3, 0xDD, 0xB7, 0xA6, 0x00, 0x00, /* 0xA8-0xAB */
	0x51, 0xDE, 0x00, 0x00, 0xB5, 0xE7, 0xCD, 0xD2, /* 0xAC-0xAF */
	0xE3, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x51, 0xDF, /* 0xB0-0xB3 */
	0x00, 0x00, 0x51, 0xE0, 0xE3, 0xE0, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x51, 0xE1, 0xB1, 0xAE, 0x51, 0xE2, /* 0xB8-0xBB */
	0x51, 0xE3, 0x00, 0x00, 0x00, 0x00, 0xE3, 0xE3, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB3, 0xF6, /* 0xC0-0xC3 */
	0xE3, 0xE2, 0xE3, 0xE1, 0x51, 0xE4, 0xE3, 0xE5, /* 0xC4-0xC7 */
	0xE3, 0xDE, 0x51, 0xE5, 0xE3, 0xE6, 0xCE, 0xA9, /* 0xC8-0xCB */
	0x51, 0xE6, 0xE3, 0xE7, 0x51, 0xE7, 0xE3, 0xE8, /* 0xCC-0xCF */
	0x00, 0x00, 0xFB, 0xC8, 0xD4, 0xF4, 0xE3, 0xEA, /* 0xD0-0xD3 */
	0x00, 0x00, 0xE3, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0xE3, 0xEB, 0xE3, 0xEC, 0x51, 0xE9, /* 0xD8-0xDB */
	0xCE, 0xB5, 0xE3, 0xED, 0x00, 0x00, 0xF0, 0xEF, /* 0xDC-0xDF */
	0xBE, 0xCF, 0xE3, 0xEE, 0xE3, 0xEF, 0xBD, 0xD7, /* 0xE0-0xE3 */
	0x00, 0x00, 0xC6, 0xB8, 0xE3, 0xF0, 0xFB, 0xC9, /* 0xE4-0xE7 */
	0x51, 0xEA, 0x51, 0xEB, 0xC3, 0xA8, 0xFB, 0xCB, /* 0xE8-0xEB */
	0x51, 0xED, 0xE3, 0xF1, 0x00, 0x00, 0xC3, 0xBC, /* 0xEC-0xEF */
	0xE3, 0xF2, 0x51, 0xEE, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x51, 0xEF, 0x00, 0x00, 0xB6, 0xA5, 0x00, 0x00, /* 0xF4-0xF7 */
	0xD1, 0xBF, 0xC3, 0xDD, 0xBC, 0xB3, 0x51, 0xF0, /* 0xF8-0xFB */
	0x00, 0x00, 0x51, 0xF1, 0x51, 0xF2, 0xB4, 0xC8, /* 0xFC-0xFF */
};

static unsigned char u2c_7B[512] = {
	0x00, 0x00, 0x00, 0x00, 0xE3, 0xF3, 0x00, 0x00, /* 0x00-0x03 */
	0xE4, 0xA2, 0x00, 0x00, 0xE3, 0xF6, 0x51, 0xF3, /* 0x04-0x07 */
	0xB5, 0xE8, 0x00, 0x00, 0xE3, 0xF5, 0xE4, 0xA4, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE3, 0xF4, /* 0x0C-0x0F */
	0x00, 0x00, 0xBE, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x51, 0xF4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xE3, 0xF8, 0xE3, 0xF9, 0x00, 0x00, 0xC5, 0xAB, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0xE3, 0xFA, 0x51, 0xF5, /* 0x1C-0x1F */
	0xB3, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x51, 0xF6, /* 0x20-0x23 */
	0x00, 0x00, 0xBF, 0xDA, 0xC9, 0xE4, 0x51, 0xF7, /* 0x24-0x27 */
	0xE3, 0xFC, 0x51, 0xF8, 0x51, 0xF9, 0x51, 0xFA, /* 0x28-0x2B */
	0xC2, 0xE8, 0x51, 0xFB, 0x51, 0xFC, 0x51, 0xFD, /* 0x2C-0x2F */
	0x51, 0xFE, 0x52, 0xA1, 0x00, 0x00, 0xE3, 0xF7, /* 0x30-0x33 */
	0x52, 0xA2, 0xE3, 0xFB, 0xE3, 0xFD, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xBA, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x52, 0xA3, 0x00, 0x00, 0x52, 0xA4, /* 0x3C-0x3F */
	0x52, 0xA5, 0x52, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0xE4, 0xA6, 0xC9, 0xAE, 0x52, 0xA7, /* 0x44-0x47 */
	0xC8, 0xA6, 0xC5, 0xF9, 0x00, 0x00, 0xB6, 0xDA, /* 0x48-0x4B */
	0xE4, 0xA5, 0xE4, 0xA3, 0x52, 0xA8, 0xC8, 0xB5, /* 0x4C-0x4F */
	0xE3, 0xFE, 0xC3, 0xDE, 0xC5, 0xFB, 0x00, 0x00, /* 0x50-0x53 */
	0xC5, 0xFA, 0x52, 0xA9, 0xBA, 0xF6, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0xE4, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x52, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x52, 0xAB, 0xE4, 0xA8, 0x52, 0xAC, 0xE4, 0xAA, /* 0x64-0x67 */
	0x00, 0x00, 0x52, 0xAD, 0x52, 0xAE, 0x00, 0x00, /* 0x68-0x6B */
	0xE4, 0xAD, 0x52, 0xAF, 0xE4, 0xAE, 0x52, 0xB0, /* 0x6C-0x6F */
	0xE4, 0xAB, 0xE4, 0xAC, 0x52, 0xB1, 0x52, 0xB2, /* 0x70-0x73 */
	0xE4, 0xA9, 0xE4, 0xA7, 0x00, 0x00, 0x52, 0xB3, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0xE4, 0xA1, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x52, 0xB4, 0x00, 0x00, 0xCA, 0xCF, 0xB2, 0xD5, /* 0x84-0x87 */
	0x00, 0x00, 0x52, 0xB5, 0x00, 0x00, 0xE4, 0xB5, /* 0x88-0x8B */
	0x00, 0x00, 0xE4, 0xB2, 0x52, 0xB6, 0xE4, 0xB7, /* 0x8C-0x8F */
	0x52, 0xB7, 0x52, 0xB8, 0xE4, 0xB6, 0x00, 0x00, /* 0x90-0x93 */
	0xC7, 0xF3, 0xCC, 0xA7, 0x52, 0xB9, 0xBB, 0xBB, /* 0x94-0x97 */
	0xE4, 0xB0, 0xE4, 0xB9, 0xE4, 0xB4, 0x52, 0xBA, /* 0x98-0x9B */
	0xE4, 0xB3, 0xE4, 0xAF, 0xFB, 0xCC, 0xE4, 0xB1, /* 0x9C-0x9F */
	0x52, 0xBC, 0xB4, 0xC9, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x52, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0xC3, 0xBD, 0x00, 0x00, /* 0xA8-0xAB */
	0x52, 0xBE, 0xC0, 0xFD, 0x00, 0x00, 0x52, 0xBF, /* 0xAC-0xAF */
	0x52, 0xC0, 0xC8, 0xA2, 0x52, 0xC1, 0x00, 0x00, /* 0xB0-0xB3 */
	0xE4, 0xBE, 0x52, 0xC2, 0x52, 0xC3, 0x00, 0x00, /* 0xB4-0xB7 */
	0xC8, 0xA4, 0x00, 0x00, 0x52, 0xC4, 0x52, 0xC5, /* 0xB8-0xBB */
	0x52, 0xC6, 0x52, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xC0, 0xE1, 0xE4, 0xBB, 0x52, 0xC8, 0x00, 0x00, /* 0xC0-0xC3 */
	0xC8, 0xCF, 0x52, 0xC9, 0xE4, 0xBF, 0xCA, 0xD3, /* 0xC4-0xC7 */
	0x52, 0xCA, 0xC3, 0xDB, 0x52, 0xCB, 0xE4, 0xBA, /* 0xC8-0xCB */
	0xE4, 0xBC, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xBD, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x52, 0xCC, 0x00, 0x00, 0x52, 0xCD, 0x52, 0xCE, /* 0xD4-0xD7 */
	0x00, 0x00, 0x52, 0xCF, 0x52, 0xD0, 0x52, 0xD1, /* 0xD8-0xDB */
	0x00, 0x00, 0xE4, 0xC0, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0xBC, 0xC4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0xC6, 0xC6, 0xE4, 0xC5, 0xE4, 0xC4, 0x00, 0x00, /* 0xE4-0xE7 */
	0x52, 0xD2, 0xE4, 0xC1, 0x52, 0xD3, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xCF, 0xB6, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x52, 0xD4, 0xE4, 0xCA, /* 0xF0-0xF3 */
	0x52, 0xD5, 0x52, 0xD6, 0xE4, 0xCE, 0xE4, 0xCB, /* 0xF4-0xF7 */
	0x52, 0xD7, 0x52, 0xD8, 0x52, 0xD9, 0x00, 0x00, /* 0xF8-0xFB */
	0x52, 0xDA, 0x00, 0x00, 0x52, 0xDB, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_7C[512] = {
	0xE4, 0xC7, 0x52, 0xDC, 0x52, 0xDD, 0x52, 0xDE, /* 0x00-0x03 */
	0x52, 0xDF, 0x00, 0x00, 0x52, 0xE0, 0xE4, 0xC8, /* 0x04-0x07 */
	0x00, 0x00, 0x52, 0xE1, 0x00, 0x00, 0x52, 0xE2, /* 0x08-0x0B */
	0x52, 0xE3, 0xE4, 0xCD, 0x52, 0xE4, 0x52, 0xE5, /* 0x0C-0x0F */
	0x00, 0x00, 0xE4, 0xC2, 0xD2, 0xD5, 0xE4, 0xC9, /* 0x10-0x13 */
	0xE4, 0xC3, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xCC, /* 0x14-0x17 */
	0x00, 0x00, 0x52, 0xE6, 0x00, 0x00, 0x52, 0xE7, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xD2, /* 0x1C-0x1F */
	0x52, 0xE8, 0xB4, 0xCA, 0x00, 0x00, 0xE4, 0xCF, /* 0x20-0x23 */
	0x00, 0x00, 0x52, 0xE9, 0x52, 0xEA, 0xE4, 0xD0, /* 0x24-0x27 */
	0x52, 0xEB, 0x00, 0x00, 0xE4, 0xD1, 0xE4, 0xD4, /* 0x28-0x2B */
	0x52, 0xEC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x52, 0xED, 0x00, 0x00, 0x52, 0xEE, /* 0x30-0x33 */
	0x52, 0xEF, 0x00, 0x00, 0x52, 0xF0, 0xE4, 0xD3, /* 0x34-0x37 */
	0xC8, 0xF6, 0x52, 0xF1, 0x52, 0xF2, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xE4, 0xD5, 0xCE, 0xFC, 0xCA, 0xED, /* 0x3C-0x3F */
	0xE4, 0xDA, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xD7, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x52, 0xF3, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x52, 0xF4, 0x00, 0x00, /* 0x48-0x4B */
	0xE4, 0xD6, 0xC0, 0xD2, 0x00, 0x00, 0xE4, 0xD9, /* 0x4C-0x4F */
	0xE4, 0xDB, 0x52, 0xF6, 0x52, 0xF7, 0x52, 0xF8, /* 0x50-0x53 */
	0xE4, 0xD8, 0x52, 0xF5, 0xE4, 0xDF, 0x00, 0x00, /* 0x54-0x57 */
	0xE4, 0xDC, 0x52, 0xF9, 0x52, 0xFA, 0x52, 0xFB, /* 0x58-0x5B */
	0x52, 0xFC, 0x52, 0xFD, 0x52, 0xFE, 0xE4, 0xDD, /* 0x5C-0x5F */
	0xE4, 0xC6, 0x53, 0xA1, 0x00, 0x00, 0x53, 0xA2, /* 0x60-0x63 */
	0xE4, 0xDE, 0xE4, 0xE0, 0x00, 0x00, 0x53, 0xA3, /* 0x64-0x67 */
	0x00, 0x00, 0x53, 0xA4, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0xE4, 0xE1, 0x53, 0xA5, 0x53, 0xA6, 0x00, 0x00, /* 0x6C-0x6F */
	0x53, 0xA7, 0x00, 0x00, 0x53, 0xA8, 0xCA, 0xC6, /* 0x70-0x73 */
	0x00, 0x00, 0xE4, 0xE2, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x53, 0xA9, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x53, 0xAA, 0x53, 0xAB, 0xCC, 0xE2, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0xB6, 0xCE, 0xB7, 0xA9, 0xE4, 0xE3, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xAC, 0x53, 0xAD, /* 0x84-0x87 */
	0x00, 0x00, 0xCA, 0xB4, 0x00, 0x00, 0xBF, 0xE8, /* 0x88-0x8B */
	0x00, 0x00, 0xCC, 0xB0, 0x00, 0x00, 0x53, 0xAE, /* 0x8C-0x8F */
	0xE4, 0xE4, 0x00, 0x00, 0xCE, 0xB3, 0x00, 0x00, /* 0x90-0x93 */
	0x53, 0xAF, 0xC7, 0xF4, 0x00, 0x00, 0xC1, 0xC6, /* 0x94-0x97 */
	0xC7, 0xB4, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xCD, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xB0, 0xB0, 0xC0, /* 0x9C-0x9F */
	0x53, 0xB1, 0xE4, 0xE9, 0xE4, 0xE7, 0x00, 0x00, /* 0xA0-0xA3 */
	0xE4, 0xE5, 0xB4, 0xA1, 0x53, 0xB2, 0xBE, 0xD1, /* 0xA4-0xA7 */
	0xE4, 0xEA, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xE8, /* 0xA8-0xAB */
	0x00, 0x00, 0xE4, 0xE6, 0xE4, 0xEE, 0x00, 0x00, /* 0xAC-0xAF */
	0x53, 0xB3, 0xE4, 0xED, 0xE4, 0xEC, 0xE4, 0xEB, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xB4, 0x53, 0xB5, /* 0xB4-0xB7 */
	0x00, 0x00, 0xE4, 0xEF, 0x53, 0xB6, 0x53, 0xB7, /* 0xB8-0xBB */
	0x53, 0xB8, 0xE4, 0xF0, 0xC0, 0xBA, 0x53, 0xB9, /* 0xBC-0xBF */
	0xE4, 0xF1, 0x00, 0x00, 0xE4, 0xF3, 0x00, 0x00, /* 0xC0-0xC3 */
	0x53, 0xBA, 0xE4, 0xF2, 0x00, 0x00, 0x53, 0xBB, /* 0xC4-0xC7 */
	0x53, 0xBC, 0x53, 0xBD, 0xB8, 0xD2, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x53, 0xBE, 0xC1, 0xB8, 0x53, 0xBF, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xE4, 0xF5, 0x53, 0xC0, /* 0xD0-0xD3 */
	0x53, 0xC1, 0x53, 0xC2, 0xC5, 0xFC, 0x53, 0xC3, /* 0xD4-0xD7 */
	0xE4, 0xF4, 0x53, 0xC4, 0x53, 0xC5, 0x00, 0x00, /* 0xD8-0xDB */
	0xE4, 0xF6, 0x53, 0xC6, 0xCA, 0xB5, 0xC1, 0xEC, /* 0xDC-0xDF */
	0xB9, 0xC7, 0x00, 0x00, 0xE4, 0xF7, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xC7, 0xCE, 0xC8, /* 0xE4-0xE7 */
	0x00, 0x00, 0x53, 0xC8, 0x00, 0x00, 0x53, 0xC9, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE4, 0xF9, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0xE4, 0xFA, 0x00, 0x00, /* 0xF0-0xF3 */
	0xE4, 0xFB, 0x53, 0xCA, 0xE4, 0xFC, 0x00, 0x00, /* 0xF4-0xF7 */
	0xBB, 0xE5, 0x00, 0x00, 0xE4, 0xFD, 0xB7, 0xCF, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0xB5, 0xEA, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_7D[512] = {
	0xB5, 0xAA, 0x00, 0x00, 0xE5, 0xA1, 0x53, 0xCB, /* 0x00-0x03 */
	0xCC, 0xF3, 0xB9, 0xC8, 0xE4, 0xFE, 0x53, 0xCC, /* 0x04-0x07 */
	0x53, 0xCD, 0x53, 0xCE, 0xE5, 0xA4, 0xCC, 0xE6, /* 0x08-0x0B */
	0x00, 0x00, 0xC7, 0xBC, 0x00, 0x00, 0x53, 0xCF, /* 0x0C-0x0F */
	0xC9, 0xB3, 0x53, 0xD0, 0x53, 0xD1, 0x53, 0xD2, /* 0x10-0x13 */
	0xBD, 0xE3, 0xE5, 0xA3, 0x53, 0xD3, 0xBC, 0xD3, /* 0x14-0x17 */
	0xB9, 0xC9, 0xBB, 0xE6, 0xB5, 0xE9, 0xCA, 0xB6, /* 0x18-0x1B */
	0xE5, 0xA2, 0x53, 0xD4, 0x53, 0xD5, 0x00, 0x00, /* 0x1C-0x1F */
	0xC1, 0xC7, 0xCB, 0xC2, 0xBA, 0xF7, 0x53, 0xD6, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xD7, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xD8, 0xBB, 0xE7, /* 0x28-0x2B */
	0xC4, 0xDD, 0x53, 0xD9, 0xE5, 0xA7, 0xCE, 0xDF, /* 0x2C-0x2F */
	0xBA, 0xD9, 0x53, 0xDA, 0xE5, 0xA8, 0xBF, 0xC2, /* 0x30-0x33 */
	0x00, 0x00, 0xE5, 0xAA, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xBE, 0xD2, 0xBA, 0xB0, 0x00, 0x00, /* 0x38-0x3B */
	0x53, 0xDB, 0x53, 0xDC, 0x53, 0xDD, 0xE5, 0xA9, /* 0x3C-0x3F */
	0x53, 0xDE, 0x53, 0xDF, 0xBD, 0xAA, 0xB8, 0xBE, /* 0x40-0x43 */
	0xC1, 0xC8, 0xE5, 0xA5, 0xE5, 0xAB, 0x53, 0xE0, /* 0x44-0x47 */
	0xFB, 0xCE, 0x00, 0x00, 0x00, 0x00, 0xE5, 0xA6, /* 0x48-0x4B */
	0xB7, 0xD0, 0x53, 0xE2, 0xE5, 0xAE, 0xE5, 0xB2, /* 0x4C-0x4F */
	0xB7, 0xEB, 0x53, 0xE3, 0x00, 0x00, 0x53, 0xE4, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0xE5, 0xAD, 0x53, 0xE5, /* 0x54-0x57 */
	0x00, 0x00, 0x53, 0xE6, 0x53, 0xE7, 0xE5, 0xB6, /* 0x58-0x5B */
	0xFB, 0xCF, 0x53, 0xE9, 0xB9, 0xCA, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0xCD, 0xED, 0xB0, 0xBC, 0xE5, 0xB3, /* 0x60-0x63 */
	0x00, 0x00, 0x53, 0xEA, 0xB5, 0xEB, 0x53, 0xEB, /* 0x64-0x67 */
	0xE5, 0xB0, 0x00, 0x00, 0x53, 0xEC, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0xE5, 0xB1, 0x00, 0x00, /* 0x6C-0x6F */
	0x53, 0xED, 0xC5, 0xFD, 0xE5, 0xAF, 0xE5, 0xAC, /* 0x70-0x73 */
	0x00, 0x00, 0xB3, 0xA8, 0xC0, 0xE4, 0x00, 0x00, /* 0x74-0x77 */
	0x53, 0xEE, 0xB8, 0xA8, 0x53, 0xEF, 0x53, 0xF0, /* 0x78-0x7B */
	0x00, 0x00, 0xE5, 0xB8, 0x00, 0x00, 0x53, 0xF1, /* 0x7C-0x7F */

	0x00, 0x00, 0x53, 0xF2, 0x53, 0xF3, 0x53, 0xF4, /* 0x80-0x83 */
	0x00, 0x00, 0x53, 0xF5, 0x53, 0xF6, 0x00, 0x00, /* 0x84-0x87 */
	0x53, 0xF7, 0xE5, 0xB5, 0x00, 0x00, 0x53, 0xF8, /* 0x88-0x8B */
	0x53, 0xF9, 0x53, 0xFA, 0x00, 0x00, 0xE5, 0xB7, /* 0x8C-0x8F */
	0x00, 0x00, 0x53, 0xFB, 0x00, 0x00, 0xE5, 0xB4, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x53, 0xFC, 0x53, 0xFD, /* 0x94-0x97 */
	0x00, 0x00, 0xB7, 0xD1, 0xC2, 0xB3, 0xE5, 0xB9, /* 0x98-0x9B */
	0xC1, 0xEE, 0x53, 0xFE, 0x54, 0xA1, 0xE5, 0xC6, /* 0x9C-0x9F */
	0xFB, 0xD1, 0x00, 0x00, 0xE5, 0xC2, 0xE5, 0xBC, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x54, 0xA2, 0x54, 0xA3, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x54, 0xA4, 0xE5, 0xC0, /* 0xA8-0xAB */
	0xBC, 0xFA, 0xB0, 0xDD, 0xE5, 0xBB, 0xE5, 0xC3, /* 0xAC-0xAF */
	0xE5, 0xC7, 0xB9, 0xCB, 0xCC, 0xD6, 0x54, 0xA5, /* 0xB0-0xB3 */
	0xC4, 0xD6, 0xE5, 0xBD, 0x54, 0xA6, 0xFB, 0xD0, /* 0xB4-0xB7 */
	0xE5, 0xC5, 0x54, 0xA8, 0xE5, 0xBA, 0xC3, 0xBE, /* 0xB8-0xBB */
	0x00, 0x00, 0xE5, 0xBF, 0xB0, 0xBD, 0xCC, 0xCA, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x54, 0xA9, 0x54, 0xAA, /* 0xC0-0xC3 */
	0x54, 0xAB, 0x54, 0xAC, 0x54, 0xAD, 0xE5, 0xBE, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0xB6, 0xDB, 0xC8, 0xEC, /* 0xC8-0xCB */
	0x54, 0xAE, 0x54, 0xAF, 0x54, 0xB0, 0xC1, 0xED, /* 0xCC-0xCF */
	0x00, 0x00, 0xCE, 0xD0, 0xBD, 0xEF, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0xE5, 0xEE, 0xFB, 0xD2, 0x54, 0xB1, /* 0xD4-0xD7 */
	0xE5, 0xC8, 0x54, 0xB2, 0xC0, 0xFE, 0x00, 0x00, /* 0xD8-0xDB */
	0xE5, 0xC4, 0xE5, 0xC9, 0xE5, 0xCB, 0x00, 0x00, /* 0xDC-0xDF */
	0xC4, 0xF9, 0xE5, 0xCE, 0x54, 0xB4, 0x00, 0x00, /* 0xE0-0xE3 */
	0xE5, 0xCA, 0x54, 0xB5, 0x54, 0xB6, 0x00, 0x00, /* 0xE4-0xE7 */
	0xCA, 0xD4, 0xB4, 0xCB, 0x54, 0xB7, 0x54, 0xB8, /* 0xE8-0xEB */
	0xCC, 0xCB, 0x54, 0xB9, 0x00, 0x00, 0xB0, 0xDE, /* 0xEC-0xEF */
	0x00, 0x00, 0x54, 0xBA, 0xE5, 0xCD, 0x00, 0x00, /* 0xF0-0xF3 */
	0xCE, 0xFD, 0x54, 0xBB, 0x54, 0xBC, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x54, 0xBD, 0x54, 0xBE, 0xE5, 0xCC, /* 0xF8-0xFB */
};

static unsigned char u2c_7E[512] = {
	0x54, 0xB3, 0xB1, 0xEF, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0xC6, 0xEC, 0xE5, 0xCF, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x54, 0xBF, 0xE5, 0xD6, 0xE5, 0xD0, 0xE5, 0xD7, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x54, 0xC0, 0x54, 0xC1, 0xE5, 0xD3, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x54, 0xC2, 0x00, 0x00, 0x54, 0xC3, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC7, 0xFB, /* 0x18-0x1B */
	0x54, 0xC4, 0x54, 0xC5, 0xBC, 0xCA, 0xE5, 0xD5, /* 0x1C-0x1F */
	0x54, 0xC6, 0xE5, 0xD2, 0xE5, 0xD8, 0xE5, 0xD1, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xBD, 0xC4, 0x54, 0xC7, /* 0x24-0x27 */
	0x54, 0xC8, 0x00, 0x00, 0x00, 0x00, 0xCB, 0xA5, /* 0x28-0x2B */
	0x54, 0xC9, 0x54, 0xCA, 0xBD, 0xCC, 0x54, 0xCB, /* 0x2C-0x2F */
	0x00, 0x00, 0xE5, 0xD4, 0xE5, 0xE0, 0x54, 0xCC, /* 0x30-0x33 */
	0x00, 0x00, 0xE5, 0xDC, 0x54, 0xCD, 0xE5, 0xDF, /* 0x34-0x37 */
	0x00, 0x00, 0xE5, 0xDD, 0xE5, 0xE1, 0xE5, 0xDB, /* 0x38-0x3B */
	0x00, 0x00, 0xE5, 0xC1, 0xC0, 0xD3, 0x54, 0xCE, /* 0x3C-0x3F */
	0x00, 0x00, 0xC8, 0xCB, 0x00, 0x00, 0xE5, 0xDE, /* 0x40-0x43 */
	0x54, 0xCF, 0x54, 0xD0, 0xE5, 0xD9, 0x54, 0xD1, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0xC1, 0xA1, 0xB7, 0xD2, /* 0x48-0x4B */
	0x00, 0x00, 0xBD, 0xAB, 0x54, 0xD2, 0x00, 0x00, /* 0x4C-0x4F */
	0x54, 0xD3, 0x00, 0x00, 0xFB, 0xD3, 0x00, 0x00, /* 0x50-0x53 */
	0xBF, 0xA5, 0xC1, 0xB6, 0xE5, 0xE4, 0x00, 0x00, /* 0x54-0x57 */
	0x54, 0xD5, 0xE5, 0xE6, 0xE5, 0xE7, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0xE5, 0xE3, 0xE5, 0xE5, 0x54, 0xD6, /* 0x5C-0x5F */
	0x00, 0x00, 0x54, 0xD7, 0x54, 0xD8, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x54, 0xD9, 0xE5, 0xDA, 0xE5, 0xE2, /* 0x64-0x67 */
	0x00, 0x00, 0xE5, 0xEA, 0xE5, 0xE9, 0x54, 0xDA, /* 0x68-0x6B */
	0x00, 0x00, 0xCB, 0xFA, 0x54, 0xDB, 0x54, 0xDC, /* 0x6C-0x6F */
	0xB7, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x54, 0xDD, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x54, 0xDE, 0xE5, 0xE8, 0x00, 0x00, 0xE5, 0xEC, /* 0x78-0x7B */
	0xE5, 0xEB, 0xE5, 0xEF, 0x54, 0xDF, 0xE5, 0xF1, /* 0x7C-0x7F */

	0x00, 0x00, 0x54, 0xE0, 0xBB, 0xBC, 0xE5, 0xED, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x54, 0xE1, 0x54, 0xE2, /* 0x84-0x87 */
	0xE5, 0xF2, 0xE5, 0xF3, 0xF9, 0xA1, 0x00, 0x00, /* 0x88-0x8B */
	0xE5, 0xF4, 0x54, 0xE4, 0xE5, 0xFA, 0xC5, 0xBB, /* 0x8C-0x8F */
	0xE5, 0xF6, 0x54, 0xE5, 0xE5, 0xF5, 0xE5, 0xF7, /* 0x90-0x93 */
	0xE5, 0xF8, 0x54, 0xE6, 0xE5, 0xF9, 0x00, 0x00, /* 0x94-0x97 */
	0x54, 0xE7, 0x00, 0x00, 0x54, 0xE8, 0xE5, 0xFB, /* 0x98-0x9B */
	0xE5, 0xFC, 0x54, 0xE9, 0x54, 0xEA, 0x00, 0x00, /* 0x9C-0x9F */
};

static unsigned char u2c_7F[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0xB4, 0xCC, 0x00, 0x00, /* 0x34-0x37 */
	0xE5, 0xFD, 0x00, 0x00, 0xE5, 0xFE, 0x54, 0xEC, /* 0x38-0x3B */
	0x54, 0xEB, 0x54, 0xED, 0x54, 0xEE, 0x54, 0xEF, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0xF0, /* 0x40-0x43 */
	0x54, 0xF1, 0xE6, 0xA1, 0x00, 0x00, 0xFB, 0xD4, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0xE6, 0xA2, 0xE6, 0xA3, 0xE6, 0xA4, 0x54, 0xF3, /* 0x4C-0x4F */
	0xE6, 0xA5, 0xE6, 0xA6, 0x54, 0xF4, 0x54, 0xF5, /* 0x50-0x53 */
	0xE6, 0xA8, 0xE6, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0xE6, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x54, 0xF6, /* 0x58-0x5B */
	0x54, 0xF7, 0x54, 0xF8, 0x00, 0x00, 0xE6, 0xAA, /* 0x5C-0x5F */
	0xE6, 0xAB, 0x54, 0xF9, 0x00, 0x00, 0x54, 0xFA, /* 0x60-0x63 */
	0x54, 0xFB, 0x54, 0xFC, 0x54, 0xFD, 0xE6, 0xAE, /* 0x64-0x67 */
	0xE6, 0xAC, 0xE6, 0xAD, 0xBA, 0xE1, 0xB7, 0xD3, /* 0x68-0x6B */
	0x00, 0x00, 0x54, 0xFE, 0xC3, 0xD6, 0x00, 0x00, /* 0x6C-0x6F */
	0xC8, 0xB3, 0x55, 0xA1, 0xBD, 0xF0, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xC7, 0xCD, 0x00, 0x00, 0xC8, 0xED, /* 0x74-0x77 */
	0xE6, 0xAF, 0xD8, 0xED, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x55, 0xA2, 0x55, 0xA3, 0x55, 0xA4, /* 0x7C-0x7F */

	0x55, 0xA5, 0x00, 0x00, 0xE6, 0xB0, 0xE6, 0xB2, /* 0x80-0x83 */
	0x00, 0x00, 0xCD, 0xE5, 0xE6, 0xB1, 0xE6, 0xB4, /* 0x84-0x87 */
	0xE6, 0xB3, 0x00, 0x00, 0xCD, 0xD3, 0x55, 0xA6, /* 0x88-0x8B */
	0xE6, 0xB5, 0x55, 0xA7, 0xC8, 0xFE, 0x55, 0xA8, /* 0x8C-0x8F */
	0x55, 0xA9, 0x55, 0xAA, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0xE6, 0xB6, 0x00, 0x00, 0x55, 0xAB, 0x55, 0xAC, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0xE6, 0xB9, 0x00, 0x00, /* 0x98-0x9B */
	0x55, 0xAD, 0xE6, 0xB8, 0xE6, 0xB7, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0xFB, 0xD5, 0x55, 0xAF, 0xE6, 0xBA, /* 0xA0-0xA3 */
	0xB7, 0xB2, 0x00, 0x00, 0x55, 0xB0, 0x00, 0x00, /* 0xA4-0xA7 */
	0xC1, 0xA2, 0xB5, 0xC1, 0x55, 0xB1, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x55, 0xB2, 0xE6, 0xBE, 0xE6, 0xBB, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xE6, 0xBC, 0x00, 0x00, /* 0xB0-0xB3 */
	0x55, 0xB3, 0x00, 0x00, 0xE6, 0xBF, 0x00, 0x00, /* 0xB4-0xB7 */
	0xE6, 0xC0, 0xE6, 0xBD, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x55, 0xB4, 0xB1, 0xA9, 0x00, 0x00, 0x55, 0xB5, /* 0xBC-0xBF */
	0x55, 0xB6, 0xB2, 0xA7, 0x00, 0x00, 0x55, 0xB7, /* 0xC0-0xC3 */
	0x00, 0x00, 0xE6, 0xC2, 0xE6, 0xC3, 0x00, 0x00, /* 0xC4-0xC7 */
	0x55, 0xB8, 0x00, 0x00, 0xE6, 0xC4, 0x00, 0x00, /* 0xC8-0xCB */
	0xCD, 0xE2, 0x00, 0x00, 0x55, 0xB9, 0x55, 0xBA, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xBD, 0xAC, 0x00, 0x00, /* 0xD0-0xD3 */
	0xE6, 0xC6, 0xE6, 0xC5, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0xBB, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0xBC, /* 0xDC-0xDF */
	0xBF, 0xE9, 0xE6, 0xC7, 0x00, 0x00, 0x55, 0xBD, /* 0xE0-0xE3 */
	0x00, 0x00, 0x55, 0xBE, 0xE6, 0xC8, 0x00, 0x00, /* 0xE4-0xE7 */
	0x55, 0xBF, 0xE6, 0xC9, 0x00, 0x00, 0xB4, 0xE5, /* 0xE8-0xEB */
	0x55, 0xC0, 0x00, 0x00, 0x55, 0xC1, 0x55, 0xC2, /* 0xEC-0xEF */
	0xB4, 0xCD, 0x00, 0x00, 0x55, 0xC3, 0xE6, 0xCA, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0xE6, 0xCB, 0x55, 0xC4, 0xCB, 0xDD, /* 0xF8-0xFB */
	0xCD, 0xE3, 0x55, 0xC5, 0x55, 0xC6, 0x55, 0xC7, /* 0xFC-0xFF */
};

static unsigned char u2c_80[512] = {
	0xCD, 0xD4, 0xCF, 0xB7, 0x00, 0x00, 0xB9, 0xCD, /* 0x00-0x03 */
	0xE6, 0xCE, 0xBC, 0xD4, 0xE6, 0xCD, 0x55, 0xC8, /* 0x04-0x07 */
	0x55, 0xC9, 0x00, 0x00, 0x55, 0xCA, 0xE6, 0xCF, /* 0x08-0x0B */
	0xBC, 0xA9, 0x55, 0xCB, 0x55, 0xCC, 0x55, 0xCD, /* 0x0C-0x0F */
	0xC2, 0xD1, 0x55, 0xCE, 0xE6, 0xD0, 0x55, 0xCF, /* 0x10-0x13 */
	0x55, 0xD0, 0xB9, 0xCC, 0x55, 0xD1, 0xCC, 0xD7, /* 0x14-0x17 */
	0xE6, 0xD1, 0xE6, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0xE6, 0xD3, 0x55, 0xD2, 0x55, 0xD3, 0x55, 0xD4, /* 0x1C-0x1F */
	0x55, 0xD5, 0xE6, 0xD4, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x55, 0xD6, 0x00, 0x00, 0x55, 0xD7, 0x00, 0x00, /* 0x24-0x27 */
	0xE6, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x55, 0xD8, 0x00, 0x00, 0x55, 0xD9, 0x00, 0x00, /* 0x2C-0x2F */
	0x55, 0xDA, 0x00, 0x00, 0x00, 0x00, 0xBC, 0xAA, /* 0x30-0x33 */
	0x55, 0xDB, 0x55, 0xDC, 0xCC, 0xED, 0x55, 0xDD, /* 0x34-0x37 */
	0x00, 0x00, 0x55, 0xDE, 0x55, 0xDF, 0xE6, 0xD7, /* 0x38-0x3B */
	0x55, 0xE0, 0xC3, 0xBF, 0x55, 0xE1, 0xE6, 0xD6, /* 0x3C-0x3F */
	0x55, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x55, 0xE3, 0x00, 0x00, 0xE6, 0xD9, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0xE6, 0xD8, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0xE6, 0xDA, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0xC0, 0xBB, 0x00, 0x00, /* 0x54-0x57 */
	0xE6, 0xDB, 0x00, 0x00, 0xE6, 0xDC, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0xCA, 0xB9, 0xE6, 0xDD, /* 0x5C-0x5F */
	0x55, 0xE4, 0xC1, 0xEF, 0xE6, 0xDE, 0x00, 0x00, /* 0x60-0x63 */
	0x55, 0xE5, 0x00, 0x00, 0x55, 0xE6, 0x00, 0x00, /* 0x64-0x67 */
	0xE6, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x55, 0xE7, 0x00, 0x00, 0xCE, 0xFE, /* 0x6C-0x6F */
	0xE6, 0xE2, 0x55, 0xE8, 0xE6, 0xE1, 0xE6, 0xE0, /* 0x70-0x73 */
	0xC4, 0xB0, 0x55, 0xE9, 0xE6, 0xE3, 0xBF, 0xA6, /* 0x74-0x77 */
	0x00, 0x00, 0xE6, 0xE4, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0xE6, 0xE5, 0xCF, 0xB8, 0xE6, 0xE6, /* 0x7C-0x7F */

	0x00, 0x00, 0x55, 0xEA, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0xE6, 0xE7, 0xE6, 0xE9, 0xE6, 0xE8, 0xC8, 0xA5, /* 0x84-0x87 */
	0x55, 0xEB, 0xC6, 0xF9, 0x00, 0x00, 0xCF, 0xBE, /* 0x88-0x8B */
	0xC8, 0xA9, 0x00, 0x00, 0x55, 0xEC, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE6, 0xEB, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xBE, 0xD3, 0x00, 0x00, /* 0x94-0x97 */
	0xC9, 0xAA, 0x00, 0x00, 0xE6, 0xEC, 0xE6, 0xEA, /* 0x98-0x9B */
	0x55, 0xED, 0xB4, 0xCE, 0x55, 0xEE, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0xB8, 0xD4, 0xBB, 0xE8, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xC8, 0xEE, 0x55, 0xEF, 0x55, 0xF0, /* 0xA4-0xA7 */
	0x00, 0x00, 0xB8, 0xAA, 0xCB, 0xC3, 0x55, 0xF1, /* 0xA8-0xAB */
	0xE6, 0xEF, 0xE6, 0xED, 0x00, 0x00, 0xB9, 0xCE, /* 0xAC-0xAF */
	0x00, 0x00, 0xB9, 0xCF, 0xB0, 0xE9, 0x00, 0x00, /* 0xB0-0xB3 */
	0xBA, 0xE8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x55, 0xF2, 0x55, 0xF3, 0xC7, 0xD9, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB0, 0xDF, /* 0xC0-0xC3 */
	0xE6, 0xF4, 0x00, 0x00, 0xC3, 0xC0, 0x00, 0x00, /* 0xC4-0xC7 */
	0x55, 0xF4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0xC7, 0xD8, 0x55, 0xF5, 0xC2, 0xDB, 0x55, 0xF6, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x55, 0xF7, 0x00, 0x00, /* 0xD0-0xD3 */
	0x55, 0xF8, 0x55, 0xF9, 0xE6, 0xF6, 0x55, 0xFA, /* 0xD4-0xD7 */
	0x55, 0xFB, 0xE6, 0xF2, 0xE6, 0xF5, 0xE6, 0xF0, /* 0xD8-0xDB */
	0x00, 0x00, 0xE6, 0xF3, 0xCB, 0xA6, 0x00, 0x00, /* 0xDC-0xDF */
	0x55, 0xFC, 0xB8, 0xD5, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0xB0, 0xFD, 0xE6, 0xF1, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x55, 0xFD, 0x55, 0xFE, 0xE6, 0xF8, /* 0xEC-0xEF */
	0x56, 0xA1, 0xE6, 0xF9, 0x56, 0xA2, 0x56, 0xA3, /* 0xF0-0xF3 */
	0xC6, 0xB9, 0x00, 0x00, 0x56, 0xA4, 0x00, 0x00, /* 0xF4-0xF7 */
	0xB6, 0xBB, 0x56, 0xA5, 0x56, 0xA6, 0x00, 0x00, /* 0xF8-0xFB */
	0xE7, 0xA6, 0xC7, 0xBD, 0x56, 0xA7, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_81[512] = {
	0x00, 0x00, 0x00, 0x00, 0xBB, 0xE9, 0x56, 0xA8, /* 0x00-0x03 */
	0x00, 0x00, 0xB6, 0xBC, 0xC0, 0xC8, 0xCF, 0xC6, /* 0x04-0x07 */
	0xCC, 0xAE, 0xE6, 0xF7, 0xC0, 0xD4, 0x56, 0xA9, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x56, 0xAA, 0x56, 0xAB, /* 0x14-0x17 */
	0x56, 0xAC, 0x00, 0x00, 0xB5, 0xD3, 0xE6, 0xFA, /* 0x18-0x1B */
	0x56, 0xAD, 0x00, 0x00, 0x56, 0xAE, 0x00, 0x00, /* 0x1C-0x1F */
	0x56, 0xAF, 0x00, 0x00, 0x00, 0x00, 0xE6, 0xFC, /* 0x20-0x23 */
	0x56, 0xB0, 0x00, 0x00, 0x00, 0x00, 0x56, 0xB1, /* 0x24-0x27 */
	0x00, 0x00, 0xE6, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x56, 0xB2, 0x00, 0x00, 0x00, 0x00, 0xE6, 0xFD, /* 0x2C-0x2F */
	0x56, 0xB3, 0xC3, 0xA6, 0x00, 0x00, 0xC7, 0xBE, /* 0x30-0x33 */
	0x00, 0x00, 0x56, 0xB4, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xC4, 0xB1, 0x56, 0xB5, 0x00, 0x00, /* 0x38-0x3B */
	0x56, 0xB6, 0x00, 0x00, 0xE7, 0xA3, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x56, 0xB7, 0xE7, 0xA2, 0x56, 0xB8, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x56, 0xB9, 0xE6, 0xFE, /* 0x48-0x4B */
	0x56, 0xBA, 0x00, 0x00, 0xBF, 0xD5, 0x00, 0x00, /* 0x4C-0x4F */
	0xC9, 0xE5, 0xE7, 0xA5, 0x56, 0xBB, 0xE7, 0xA4, /* 0x50-0x53 */
	0xB9, 0xD0, 0xCF, 0xD3, 0x00, 0x00, 0x56, 0xBC, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE7, 0xB5, /* 0x5C-0x5F */
	0x56, 0xBD, 0x56, 0xBE, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0xE7, 0xA9, 0xE7, 0xAA, 0x56, 0xBF, /* 0x64-0x67 */
	0x56, 0xC0, 0x56, 0xC1, 0x00, 0x00, 0xBC, 0xF0, /* 0x68-0x6B */
	0x00, 0x00, 0x56, 0xC2, 0xE7, 0xA8, 0x56, 0xC3, /* 0x6C-0x6F */
	0xB9, 0xF8, 0xE7, 0xA7, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0xE7, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x56, 0xC4, /* 0x74-0x77 */
	0xC4, 0xB2, 0xCA, 0xA2, 0xC1, 0xA3, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC2, 0xDC, /* 0x7C-0x7F */

	0xE7, 0xAF, 0x56, 0xC5, 0xE7, 0xB0, 0xE7, 0xAC, /* 0x80-0x83 */
	0x56, 0xC7, 0x56, 0xC8, 0x56, 0xC9, 0x00, 0x00, /* 0x84-0x87 */
	0xE7, 0xAD, 0x00, 0x00, 0xE7, 0xAE, 0x56, 0xCA, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x56, 0xCB, 0xB9, 0xD1, /* 0x8C-0x8F */
	0x56, 0xC6, 0x00, 0x00, 0x00, 0x00, 0xE7, 0xB6, /* 0x90-0x93 */
	0x00, 0x00, 0xE7, 0xB2, 0x56, 0xCC, 0x00, 0x00, /* 0x94-0x97 */
	0x56, 0xCD, 0x00, 0x00, 0xC9, 0xE6, 0x56, 0xCE, /* 0x98-0x9B */
	0xCB, 0xEC, 0xC9, 0xA8, 0x56, 0xCF, 0x00, 0x00, /* 0x9C-0x9F */
	0xE7, 0xB1, 0x00, 0x00, 0x56, 0xD0, 0xE7, 0xB4, /* 0xA0-0xA3 */
	0xE7, 0xB3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0xCB, 0xC4, 0xE7, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x56, 0xD1, 0x00, 0x00, /* 0xAC-0xAF */
	0xE7, 0xB8, 0x00, 0x00, 0x56, 0xD2, 0xC1, 0xB7, /* 0xB0-0xB3 */
	0x56, 0xD3, 0xE7, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xE7, 0xBB, 0x00, 0x00, 0xE7, 0xBF, 0x56, 0xD4, /* 0xB8-0xBB */
	0x00, 0x00, 0xE7, 0xBC, 0xE7, 0xBA, 0xC7, 0xBF, /* 0xBC-0xBF */
	0xE7, 0xBD, 0x00, 0x00, 0xE7, 0xBE, 0x56, 0xD6, /* 0xC0-0xC3 */
	0x00, 0x00, 0x56, 0xD7, 0xB2, 0xB2, 0x00, 0x00, /* 0xC4-0xC7 */
	0xE7, 0xC5, 0xE7, 0xC0, 0x56, 0xD8, 0x56, 0xD5, /* 0xC8-0xCB */
	0x00, 0x00, 0xE7, 0xC1, 0x56, 0xD9, 0x56, 0xDA, /* 0xCC-0xCF */
	0x00, 0x00, 0xE7, 0xC2, 0x00, 0x00, 0xC2, 0xA1, /* 0xD0-0xD3 */
	0x00, 0x00, 0x56, 0xDB, 0x00, 0x00, 0x56, 0xDC, /* 0xD4-0xD7 */
	0xE7, 0xC4, 0xE7, 0xC3, 0xE7, 0xC6, 0x56, 0xDD, /* 0xD8-0xDB */
	0x00, 0x00, 0x56, 0xDE, 0x56, 0xDF, 0xE7, 0xC7, /* 0xDC-0xDF */
	0xE7, 0xC8, 0x56, 0xE0, 0x00, 0x00, 0xBF, 0xC3, /* 0xE0-0xE3 */
	0x56, 0xE1, 0xB2, 0xE9, 0x00, 0x00, 0xE7, 0xC9, /* 0xE4-0xE7 */
	0xCE, 0xD7, 0x00, 0x00, 0xBC, 0xAB, 0x56, 0xE2, /* 0xE8-0xEB */
	0x56, 0xE3, 0xBD, 0xAD, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x56, 0xE4, 0x56, 0xE5, 0x56, 0xE6, 0xBB, 0xEA, /* 0xF0-0xF3 */
	0xC3, 0xD7, 0x56, 0xE7, 0x56, 0xE8, 0x00, 0x00, /* 0xF4-0xF7 */
	0x56, 0xE9, 0x56, 0xEA, 0xE7, 0xCA, 0xE7, 0xCB, /* 0xF8-0xFB */
	0xB1, 0xB1, 0x56, 0xEB, 0xE7, 0xCC, 0x56, 0xEC, /* 0xFC-0xFF */
};

static unsigned char u2c_82[512] = {
	0x56, 0xED, 0xE7, 0xCD, 0xE7, 0xCE, 0x56, 0xEE, /* 0x00-0x03 */
	0x00, 0x00, 0xE7, 0xCF, 0x00, 0x00, 0xE7, 0xD0, /* 0x04-0x07 */
	0xB6, 0xBD, 0xDA, 0xAA, 0xE7, 0xD1, 0x00, 0x00, /* 0x08-0x0B */
	0xC0, 0xE5, 0xE7, 0xD2, 0xBC, 0xCB, 0x56, 0xEF, /* 0x0C-0x0F */
	0xE7, 0xD3, 0x00, 0x00, 0xD0, 0xB0, 0x56, 0xF0, /* 0x10-0x13 */
	0x56, 0xF1, 0x00, 0x00, 0xE7, 0xD4, 0xCA, 0xDE, /* 0x14-0x17 */
	0xB4, 0xDC, 0x56, 0xF2, 0x56, 0xF3, 0xC1, 0xA4, /* 0x18-0x1B */
	0xBD, 0xD8, 0x56, 0xF4, 0xC9, 0xF1, 0xBD, 0xAE, /* 0x1C-0x1F */
	0x00, 0x00, 0x56, 0xF5, 0x56, 0xF6, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x56, 0xF7, 0xE7, 0xD5, 0xB9, 0xD2, 0xE7, 0xD6, /* 0x28-0x2B */
	0xC8, 0xCC, 0x00, 0x00, 0xE7, 0xE4, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x56, 0xF8, 0xE7, 0xD8, /* 0x30-0x33 */
	0x56, 0xF9, 0xC2, 0xC9, 0xC7, 0xF5, 0xB8, 0xBF, /* 0x34-0x37 */
	0xE7, 0xD7, 0xC1, 0xA5, 0x56, 0xFA, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xE7, 0xD9, 0x00, 0x00, 0x00, 0x00, 0x56, 0xFB, /* 0x40-0x43 */
	0x56, 0xFC, 0x56, 0xFD, 0x56, 0xFE, 0xC4, 0xFA, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57, 0xA1, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x57, 0xA2, 0x57, 0xA3, /* 0x4C-0x4F */
	0x00, 0x00, 0x57, 0xA4, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x57, 0xA5, 0x00, 0x00, /* 0x54-0x57 */
	0xE7, 0xDB, 0xE7, 0xDA, 0xE7, 0xDD, 0x00, 0x00, /* 0x58-0x5B */
	0x57, 0xA6, 0xE7, 0xDC, 0x00, 0x00, 0xE7, 0xDE, /* 0x5C-0x5F */
	0x57, 0xA7, 0x00, 0x00, 0xE7, 0xE0, 0x57, 0xA8, /* 0x60-0x63 */
	0xE7, 0xDF, 0x00, 0x00, 0xB4, 0xCF, 0x57, 0xA9, /* 0x64-0x67 */
	0xE7, 0xE1, 0x00, 0x00, 0xE7, 0xE2, 0xE7, 0xE3, /* 0x68-0x6B */
	0x00, 0x00, 0x57, 0xAA, 0xBA, 0xB1, 0xCE, 0xC9, /* 0x6C-0x6F */
	0x00, 0x00, 0xE7, 0xE5, 0xBF, 0xA7, 0x00, 0x00, /* 0x70-0x73 */
	0x57, 0xAB, 0x00, 0x00, 0xB1, 0xF0, 0xE7, 0xE6, /* 0x74-0x77 */
	0xE7, 0xE7, 0x00, 0x00, 0x00, 0x00, 0x57, 0xAC, /* 0x78-0x7B */
	0x00, 0x00, 0x57, 0xAD, 0xE7, 0xE8, 0x57, 0xAE, /* 0x7C-0x7F */

	0x57, 0xAF, 0x57, 0xB0, 0x00, 0x00, 0x57, 0xB1, /* 0x80-0x83 */
	0x57, 0xB2, 0x00, 0x00, 0x00, 0x00, 0x57, 0xB3, /* 0x84-0x87 */
	0x00, 0x00, 0x57, 0xB4, 0x57, 0xB5, 0xB0, 0xF2, /* 0x88-0x8B */
	0x00, 0x00, 0xE7, 0xE9, 0x57, 0xB6, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x57, 0xB7, 0xE7, 0xEA, 0x00, 0x00, /* 0x90-0x93 */
	0x57, 0xB8, 0x00, 0x00, 0x57, 0xB9, 0x00, 0x00, /* 0x94-0x97 */
	0x57, 0xBA, 0xC9, 0xE7, 0x57, 0xBB, 0x57, 0xBC, /* 0x98-0x9B */
	0x00, 0x00, 0xBC, 0xC7, 0x00, 0x00, 0xE7, 0xEC, /* 0x9C-0x9F */
	0x57, 0xBD, 0x57, 0xBE, 0x00, 0x00, 0x57, 0xBF, /* 0xA0-0xA3 */
	0x57, 0xC0, 0xB3, 0xA9, 0xB0, 0xB2, 0x57, 0xC1, /* 0xA4-0xA7 */
	0x57, 0xC2, 0x57, 0xC3, 0x57, 0xC4, 0xE7, 0xEB, /* 0xA8-0xAB */
	0xE7, 0xEE, 0xC7, 0xCE, 0x57, 0xC5, 0xBF, 0xC4, /* 0xAC-0xAF */
	0x57, 0xC6, 0xB2, 0xD6, 0x57, 0xC7, 0xCB, 0xA7, /* 0xB0-0xB3 */
	0x57, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x57, 0xC9, /* 0xB4-0xB7 */
	0xB7, 0xDD, 0xB6, 0xDC, 0x57, 0xCA, 0xE7, 0xED, /* 0xB8-0xBB */
	0x57, 0xCB, 0xB2, 0xEA, 0x57, 0xCC, 0x57, 0xCD, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0xB4, 0xA3, 0x57, 0xCE, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x57, 0xCF, 0xB1, 0xF1, 0xE7, 0xF2, 0xCE, 0xEA, /* 0xD0-0xD3 */
	0xC2, 0xDD, 0x57, 0xD0, 0x00, 0x00, 0xC9, 0xC4, /* 0xD4-0xD7 */
	0x00, 0x00, 0xE7, 0xFE, 0x57, 0xD1, 0xB2, 0xD7, /* 0xD8-0xDB */
	0xE7, 0xFC, 0x00, 0x00, 0xE7, 0xFA, 0xE7, 0xF1, /* 0xDC-0xDF */
	0x57, 0xD2, 0xE7, 0xEF, 0x57, 0xD3, 0xE7, 0xF0, /* 0xE0-0xE3 */
	0x57, 0xD4, 0xBC, 0xE3, 0xB6, 0xEC, 0xC3, 0xF7, /* 0xE4-0xE7 */
	0x57, 0xD5, 0x00, 0x00, 0x57, 0xD6, 0xC6, 0xD1, /* 0xE8-0xEB */
	0x00, 0x00, 0x57, 0xD7, 0x00, 0x00, 0x57, 0xD8, /* 0xEC-0xEF */
	0x00, 0x00, 0xB1, 0xD1, 0x00, 0x00, 0xE7, 0xF4, /* 0xF0-0xF3 */
	0xE7, 0xF3, 0x00, 0x00, 0x57, 0xD9, 0x57, 0xDA, /* 0xF4-0xF7 */
	0x00, 0x00, 0xE7, 0xF9, 0xE7, 0xF5, 0xE7, 0xF8, /* 0xF8-0xFB */
	0x00, 0x00, 0x57, 0xDB, 0x57, 0xDC, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_83[512] = {
	0x57, 0xDD, 0xFB, 0xD7, 0xCC, 0xD0, 0xE7, 0xF7, /* 0x00-0x03 */
	0xB2, 0xD8, 0xB3, 0xFD, 0xE7, 0xFB, 0x57, 0xDF, /* 0x04-0x07 */
	0x57, 0xE0, 0xE7, 0xFD, 0x57, 0xE1, 0x57, 0xE2, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0xB7, 0xD4, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xE8, 0xA3, 0xE8, 0xAC, /* 0x14-0x17 */
	0xE8, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x57, 0xE4, /* 0x18-0x1B */
	0xB0, 0xAB, 0x57, 0xE5, 0x57, 0xE6, 0x57, 0xE7, /* 0x1C-0x1F */
	0x00, 0x00, 0x57, 0xE8, 0x57, 0xE9, 0xE8, 0xB4, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0xB0, 0xF1, 0x00, 0x00, 0x00, 0x00, 0xE8, 0xAB, /* 0x28-0x2B */
	0x57, 0xEA, 0x57, 0xEB, 0x57, 0xEC, 0xE8, 0xAA, /* 0x2C-0x2F */
	0x57, 0xED, 0xE8, 0xA5, 0xE8, 0xA4, 0x57, 0xEE, /* 0x30-0x33 */
	0xE8, 0xA2, 0xE8, 0xA1, 0xC3, 0xE3, 0x57, 0xEF, /* 0x34-0x37 */
	0xC2, 0xFB, 0xE8, 0xA7, 0x57, 0xF0, 0x00, 0x00, /* 0x38-0x3B */
	0x57, 0xF1, 0x57, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xE8, 0xA6, 0x00, 0x00, 0x57, 0xF3, 0x57, 0xF4, /* 0x40-0x43 */
	0x57, 0xF5, 0xE8, 0xA9, 0x00, 0x00, 0x57, 0xF6, /* 0x44-0x47 */
	0x00, 0x00, 0xC1, 0xF0, 0xB7, 0xD5, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x57, 0xF7, 0x57, 0xF8, 0xB1, 0xC1, /* 0x4C-0x4F */
	0xE8, 0xA8, 0x57, 0xF9, 0xB9, 0xD3, 0x58, 0xBE, /* 0x50-0x53 */
	0x57, 0xE3, 0x57, 0xFA, 0x57, 0xFB, 0x57, 0xFC, /* 0x54-0x57 */
	0xC1, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0xFB, 0xD8, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x57, 0xFD, 0x00, 0x00, 0x00, 0x00, 0xE8, 0xBA, /* 0x70-0x73 */
	0x00, 0x00, 0xE8, 0xBB, 0x00, 0x00, 0xB2, 0xD9, /* 0x74-0x77 */
	0x57, 0xFE, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xAE, /* 0x78-0x7B */
	0xE8, 0xB8, 0x58, 0xA1, 0x00, 0x00, 0xFB, 0xD9, /* 0x7C-0x7F */

	0x58, 0xA3, 0x00, 0x00, 0x58, 0xA4, 0x00, 0x00, /* 0x80-0x83 */
	0x58, 0xA5, 0xE8, 0xAE, 0x58, 0xA6, 0xE8, 0xB6, /* 0x84-0x87 */
	0x00, 0x00, 0xE8, 0xBD, 0xE8, 0xB7, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x58, 0xA7, 0xE8, 0xB5, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x58, 0xA8, 0xE7, 0xF6, /* 0x90-0x93 */
	0x58, 0xA9, 0x58, 0xAA, 0xE8, 0xB3, 0x00, 0x00, /* 0x94-0x97 */
	0x58, 0xAB, 0x58, 0xAC, 0xE8, 0xAF, 0x58, 0xAD, /* 0x98-0x9B */
	0x58, 0xAE, 0x58, 0xAF, 0xB4, 0xD0, 0xE8, 0xB1, /* 0x9C-0x9F */
	0xE8, 0xBC, 0x00, 0x00, 0xE8, 0xB2, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x58, 0xB0, 0x58, 0xB1, /* 0xA4-0xA7 */
	0xE8, 0xBE, 0x58, 0xB2, 0xE8, 0xB0, 0xC7, 0xFC, /* 0xA8-0xAB */
	0x58, 0xB3, 0x58, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0xCD, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0xE8, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0xE8, 0xCF, 0x58, 0xB4, 0x58, 0xB5, /* 0xBC-0xBF */
	0x58, 0xB6, 0xE8, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0xBF, 0xFB, 0x00, 0x00, 0xFB, 0xDA, /* 0xC4-0xC7 */
	0x00, 0x00, 0x58, 0xB8, 0xB5, 0xC6, 0x00, 0x00, /* 0xC8-0xCB */
	0xB6, 0xDD, 0x00, 0x00, 0xE8, 0xC2, 0x58, 0xB9, /* 0xCC-0xCF */
	0x58, 0xBA, 0x58, 0xBB, 0x00, 0x00, 0xB2, 0xDB, /* 0xD0-0xD3 */
	0x58, 0xBC, 0x00, 0x00, 0xBE, 0xD4, 0x00, 0x00, /* 0xD4-0xD7 */
	0xE8, 0xC5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0xBA, 0xDA, 0x58, 0xBD, 0x00, 0x00, 0xC5, 0xD1, /* 0xDC-0xDF */
	0xE8, 0xCA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x58, 0xBF, 0xCA, 0xEE, 0x58, 0xC0, 0xE8, 0xC1, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xDA, /* 0xEC-0xEF */
	0xB8, 0xD6, 0xC9, 0xA9, 0xE8, 0xCB, 0x00, 0x00, /* 0xF0-0xF3 */
	0xE8, 0xBF, 0x00, 0x00, 0xFB, 0xDB, 0xE8, 0xC8, /* 0xF4-0xF7 */
	0x58, 0xC2, 0x58, 0xC3, 0x00, 0x00, 0xE8, 0xD2, /* 0xF8-0xFB */
	0x58, 0xC4, 0xE8, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_84[512] = {
	0x00, 0x00, 0x58, 0xC5, 0x00, 0x00, 0xE8, 0xC4, /* 0x00-0x03 */
	0xC6, 0xBA, 0x00, 0x00, 0x58, 0xC6, 0xE8, 0xC9, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x58, 0xC7, 0xE8, 0xC6, /* 0x08-0x0B */
	0xCB, 0xA8, 0xE8, 0xCC, 0xB0, 0xE0, 0x58, 0xC8, /* 0x0C-0x0F */
	0x00, 0x00, 0x58, 0xC9, 0x00, 0x00, 0xE8, 0xC0, /* 0x10-0x13 */
	0x00, 0x00, 0x58, 0xCA, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x58, 0xCB, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0xE8, 0xCE, 0x00, 0x00, 0xE8, 0xCD, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0xC7, 0xEB, 0xE8, 0xD4, 0x00, 0x00, /* 0x28-0x2B */
	0xE8, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x58, 0xCD, /* 0x2C-0x2F */
	0x00, 0x00, 0xB3, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0xE8, 0xE2, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0xE8, 0xD0, 0x58, 0xCE, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0xE8, 0xD5, 0xCD, 0xEE, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x58, 0xCF, 0xE8, 0xDE, 0x58, 0xD0, /* 0x44-0x47 */
	0xFB, 0xDC, 0xCD, 0xD5, 0x58, 0xD2, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x58, 0xD3, 0xCE, 0xAA, 0x58, 0xD4, /* 0x4C-0x4F */
	0x00, 0x00, 0x58, 0xD5, 0x58, 0xD6, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x58, 0xD7, 0xC3, 0xF8, /* 0x54-0x57 */
	0x58, 0xD8, 0x58, 0xD9, 0x58, 0xDA, 0xB3, 0xEB, /* 0x58-0x5B */
	0x58, 0xDB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x58, 0xDC, 0xC9, 0xF2, 0xE8, 0xE4, 0xC6, 0xA1, /* 0x60-0x63 */
	0x58, 0xDD, 0x58, 0xDE, 0xB0, 0xB1, 0x58, 0xDF, /* 0x64-0x67 */
	0x00, 0x00, 0xE8, 0xDD, 0x58, 0xE0, 0xE8, 0xD9, /* 0x68-0x6B */
	0xC1, 0xF2, 0xE8, 0xD3, 0xE8, 0xDB, 0xE8, 0xE0, /* 0x6C-0x6F */
	0x58, 0xE1, 0xC7, 0xAC, 0x00, 0x00, 0x58, 0xE2, /* 0x70-0x73 */
	0x58, 0xE3, 0xB0, 0xAA, 0x58, 0xE4, 0xE8, 0xD8, /* 0x74-0x77 */
	0x58, 0xE5, 0xE8, 0xE1, 0xC9, 0xF8, 0x00, 0x00, /* 0x78-0x7B */
	0x58, 0xE6, 0x58, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x58, 0xE8, 0xE8, 0xDC, 0x00, 0x00, /* 0x80-0x83 */
	0xE8, 0xD7, 0x58, 0xE9, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBE, 0xD5, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0xBD, 0xAF, 0x00, 0x00, 0x58, 0xEA, 0x58, 0xEB, /* 0x90-0x93 */
	0xBC, 0xAC, 0x58, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xCC, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xC9, 0xC7, 0x00, 0x00, 0x58, 0xED, 0xE8, 0xE7, /* 0x9C-0x9F */
	0x00, 0x00, 0xE8, 0xF0, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x58, 0xEE, 0x00, 0x00, /* 0xA4-0xA7 */
	0x58, 0xEF, 0x58, 0xF0, 0x58, 0xF1, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0xE8, 0xDA, 0x00, 0x00, 0x58, 0xF2, /* 0xAC-0xAF */
	0x00, 0x00, 0x58, 0xF3, 0xB3, 0xF7, 0x00, 0x00, /* 0xB0-0xB3 */
	0xFB, 0xDD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xBE, 0xF8, 0xE8, 0xE5, 0x58, 0xF5, 0xE8, 0xEA, /* 0xB8-0xBB */
	0xC1, 0xF3, 0x58, 0xF6, 0x58, 0xF7, 0xE8, 0xE6, /* 0xBC-0xBF */
	0x58, 0xF8, 0xE8, 0xED, 0x58, 0xF9, 0x00, 0x00, /* 0xC0-0xC3 */
	0xC3, 0xDF, 0x00, 0x00, 0xE8, 0xEE, 0x58, 0xFA, /* 0xC4-0xC7 */
	0x58, 0xFB, 0xCD, 0xD6, 0xE8, 0xE3, 0xB3, 0xB8, /* 0xC8-0xCB */
	0x58, 0xFC, 0xE8, 0xE9, 0x00, 0x00, 0x58, 0xFD, /* 0xCC-0xCF */
	0xE8, 0xEC, 0xCC, 0xAC, 0x00, 0x00, 0x58, 0xFE, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0xE8, 0xEF, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0xE8, 0xE8, 0xE8, 0xEB, 0x00, 0x00, /* 0xD8-0xDB */
	0xF9, 0xA5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0xA2, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xA3, 0x00, 0x00, /* 0xE8-0xEB */
	0xCB, 0xA9, 0x00, 0x00, 0xCF, 0xA1, 0x59, 0xA4, /* 0xEC-0xEF */
	0x59, 0xA5, 0x59, 0xA6, 0x59, 0xA7, 0x00, 0x00, /* 0xF0-0xF3 */
	0xE8, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x59, 0xA8, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xAA, 0x59, 0xAB, /* 0xF8-0xFB */
	0xE8, 0xFA, 0x59, 0xAC, 0x00, 0x00, 0xE8, 0xF2, /* 0xFC-0xFF */
};

static unsigned char u2c_85[512] = {
	0xBC, 0xC3, 0x00, 0x00, 0x59, 0xAD, 0x59, 0xAE, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xE8, 0xD1, 0x59, 0xAF, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x59, 0xB0, 0x00, 0x00, 0x59, 0xB1, 0x00, 0x00, /* 0x0C-0x0F */
	0x59, 0xB2, 0xCA, 0xCE, 0x00, 0x00, 0xCC, 0xA2, /* 0x10-0x13 */
	0xE8, 0xF9, 0xE8, 0xF8, 0x00, 0x00, 0xE8, 0xF4, /* 0x14-0x17 */
	0xE8, 0xF5, 0x00, 0x00, 0xB1, 0xB6, 0x00, 0x00, /* 0x18-0x1B */
	0x59, 0xB3, 0x00, 0x00, 0x59, 0xB4, 0xE8, 0xF7, /* 0x1C-0x1F */
	0x00, 0x00, 0xE8, 0xF1, 0x59, 0xB5, 0x59, 0xB6, /* 0x20-0x23 */
	0x59, 0xB7, 0x59, 0xB8, 0xC4, 0xD5, 0x59, 0xB9, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xBA, 0x59, 0xBB, /* 0x28-0x2B */
	0xE8, 0xF6, 0xB0, 0xFE, 0x00, 0x00, 0x59, 0xBC, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xA9, 0x59, 0xBD, /* 0x30-0x33 */
	0x59, 0xBE, 0xC2, 0xA2, 0x59, 0xBF, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0xCA, 0xC3, 0x00, 0x00, 0x59, 0xC0, /* 0x3C-0x3F */
	0xE8, 0xFB, 0xE9, 0xA1, 0x00, 0x00, 0xC8, 0xD9, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xC1, 0x00, 0x00, /* 0x44-0x47 */
	0xE8, 0xFE, 0xBE, 0xD6, 0xBC, 0xC9, 0xE9, 0xA3, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0xB6, 0xBE, 0x59, 0xC2, /* 0x4C-0x4F */
	0x59, 0xC3, 0x59, 0xC4, 0x59, 0xC5, 0xFB, 0xDE, /* 0x50-0x53 */
	0x00, 0x00, 0xE9, 0xA4, 0x59, 0xC7, 0xC9, 0xF9, /* 0x54-0x57 */
	0xE8, 0xFD, 0xFB, 0xDF, 0xE8, 0xD6, 0x00, 0x00, /* 0x58-0x5B */
	0x59, 0xC9, 0x59, 0xCA, 0x59, 0xCB, 0x59, 0xCC, /* 0x5C-0x5F */
	0x59, 0xCD, 0x59, 0xCE, 0x59, 0xCF, 0xE8, 0xFC, /* 0x60-0x63 */
	0x59, 0xD0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0xCF, 0xCF, 0xC6, 0xA2, 0xC9, 0xF3, 0xFB, 0xE0, /* 0x68-0x6B */
	0x00, 0x00, 0xE9, 0xAB, 0x00, 0x00, 0x59, 0xD2, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE9, 0xB1, /* 0x74-0x77 */
	0x00, 0x00, 0x59, 0xD3, 0x59, 0xD4, 0x59, 0xD5, /* 0x78-0x7B */
	0x00, 0x00, 0x59, 0xD6, 0xE9, 0xB2, 0x59, 0xD7, /* 0x7C-0x7F */

	0xE9, 0xA5, 0x59, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0xC7, 0xF6, 0x59, 0xD9, 0x59, 0xDA, 0xE9, 0xAF, /* 0x84-0x87 */
	0xE9, 0xA7, 0x59, 0xDB, 0xE9, 0xA9, 0x59, 0xDC, /* 0x88-0x8B */
	0x59, 0xDD, 0x00, 0x00, 0x00, 0x00, 0x59, 0xDE, /* 0x8C-0x8F */
	0xE9, 0xB3, 0xE9, 0xA8, 0x00, 0x00, 0x59, 0xDF, /* 0x90-0x93 */
	0xE9, 0xAC, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xF2, /* 0x94-0x97 */
	0x59, 0xE0, 0xC6, 0xE5, 0x00, 0x00, 0xE9, 0xAD, /* 0x98-0x9B */
	0xE9, 0xB0, 0x59, 0xE1, 0x00, 0x00, 0x59, 0xE2, /* 0x9C-0x9F */
	0x59, 0xE3, 0x00, 0x00, 0x59, 0xE4, 0x00, 0x00, /* 0xA0-0xA3 */
	0xE9, 0xA6, 0x59, 0xE5, 0xC1, 0xA6, 0x59, 0xE6, /* 0xA4-0xA7 */
	0xE9, 0xAA, 0xBB, 0xA7, 0xBF, 0xC5, 0xB7, 0xB0, /* 0xA8-0xAB */
	0xCC, 0xF4, 0x59, 0xF4, 0xCC, 0xF9, 0xBD, 0xF2, /* 0xAC-0xAF */
	0xFB, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x59, 0xE7, 0x00, 0x00, 0x59, 0xE8, 0x59, 0xE9, /* 0xB4-0xB7 */
	0x59, 0xEA, 0xE9, 0xB7, 0xE9, 0xB5, 0x00, 0x00, /* 0xB8-0xBB */
	0x59, 0xEB, 0x59, 0xEC, 0x59, 0xED, 0x59, 0xEE, /* 0xBC-0xBF */
	0x00, 0x00, 0xCF, 0xCE, 0x59, 0xEF, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0xF0, /* 0xC4-0xC7 */
	0x00, 0x00, 0xE9, 0xB4, 0x59, 0xF1, 0x59, 0xF2, /* 0xC8-0xCB */
	0x00, 0x00, 0xCD, 0xF5, 0x59, 0xF3, 0xE9, 0xB6, /* 0xCC-0xCF */
	0xE9, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0xE9, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x59, 0xF5, 0x00, 0x00, 0x59, 0xF6, 0x00, 0x00, /* 0xD8-0xDB */
	0xE9, 0xBC, 0xE9, 0xBA, 0x00, 0x00, 0x59, 0xF7, /* 0xDC-0xDF */
	0x59, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0xC6, 0xA3, 0xE9, 0xBB, 0x59, 0xF9, 0x00, 0x00, /* 0xE4-0xE7 */
	0x59, 0xFA, 0xC8, 0xCD, 0xE9, 0xAE, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x59, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0xFC, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x59, 0xFD, 0xBD, 0xF3, /* 0xF4-0xF7 */
	0x00, 0x00, 0xE9, 0xBD, 0xE9, 0xC2, 0xC1, 0xF4, /* 0xF8-0xFB */
	0x59, 0xFE, 0x00, 0x00, 0xE9, 0xC1, 0x5A, 0xA1, /* 0xFC-0xFF */
};

static unsigned char u2c_86[512] = {
	0x5A, 0xA2, 0x00, 0x00, 0xE9, 0xA2, 0x00, 0x00, /* 0x00-0x03 */
	0x5A, 0xA3, 0x5A, 0xA4, 0xE9, 0xC3, 0xC1, 0xC9, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0xE9, 0xBE, 0xE9, 0xC0, /* 0x08-0x0B */
	0x00, 0x00, 0x5A, 0xA5, 0x5A, 0xA6, 0x00, 0x00, /* 0x0C-0x0F */
	0x5A, 0xA7, 0x5A, 0xA8, 0x5A, 0xA9, 0xE9, 0xBF, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xDD, 0xB1, 0xDD, 0xA2, /* 0x14-0x17 */
	0x5A, 0xAA, 0x5A, 0xAB, 0xE9, 0xC5, 0x5A, 0xAC, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x5A, 0xAD, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x5A, 0xAE, 0xE9, 0xC4, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5A, 0xAF, /* 0x24-0x27 */
	0x00, 0x00, 0x5A, 0xB0, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0xCD, 0xF6, 0x00, 0x00, 0xE2, 0xBC, /* 0x2C-0x2F */
	0xE9, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x5A, 0xB1, 0x00, 0x00, /* 0x34-0x37 */
	0x5A, 0xB2, 0x00, 0x00, 0x5A, 0xB3, 0x00, 0x00, /* 0x38-0x3B */
	0x5A, 0xB4, 0x5A, 0xB5, 0x00, 0x00, 0xE9, 0xC7, /* 0x3C-0x3F */
	0x5A, 0xB6, 0x38, 0xE6, 0x5A, 0xB7, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x5A, 0xB8, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0xE9, 0xC8, 0xB8, 0xD7, 0x00, 0x00, /* 0x4C-0x4F */
	0xB5, 0xD4, 0x00, 0x00, 0x5A, 0xB9, 0x5A, 0xBA, /* 0x50-0x53 */
	0xE9, 0xCA, 0xD1, 0xDD, 0x5A, 0xBB, 0x5A, 0xBC, /* 0x54-0x57 */
	0x5A, 0xBD, 0x5A, 0xBE, 0xB5, 0xF5, 0x00, 0x00, /* 0x58-0x5B */
	0xCE, 0xBA, 0x5A, 0xBF, 0xB6, 0xF3, 0xE9, 0xCB, /* 0x5C-0x5F */
	0x5A, 0xC0, 0x5A, 0xC1, 0x5A, 0xC2, 0x5A, 0xC3, /* 0x60-0x63 */
	0x5A, 0xC4, 0x00, 0x00, 0x00, 0x00, 0xE9, 0xCC, /* 0x64-0x67 */
	0x00, 0x00, 0x5A, 0xC5, 0x00, 0x00, 0xC3, 0xEE, /* 0x68-0x6B */
	0x5A, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x5A, 0xC7, /* 0x6C-0x6F */
	0x00, 0x00, 0xE9, 0xCD, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x5A, 0xC8, 0x5A, 0xC9, 0x5A, 0xCA, /* 0x74-0x77 */
	0x00, 0x00, 0xC6, 0xFA, 0x5A, 0xCB, 0xB0, 0xBA, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x5A, 0xED, 0x00, 0x00, 0xB2, 0xE3, 0xE9, 0xD2, /* 0x88-0x8B */
	0xE9, 0xD3, 0x5A, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x5A, 0xCD, 0x00, 0x00, 0xE9, 0xCE, /* 0x90-0x93 */
	0x00, 0x00, 0xBB, 0xBD, 0x5A, 0xCE, 0x00, 0x00, /* 0x94-0x97 */
	0x5A, 0xCF, 0x00, 0x00, 0x5A, 0xD0, 0x00, 0x00, /* 0x98-0x9B */
	0x5A, 0xD1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x5A, 0xD2, 0x00, 0x00, 0xE9, 0xCF, /* 0xA0-0xA3 */
	0xC7, 0xC2, 0x00, 0x00, 0x5A, 0xD3, 0x5A, 0xD4, /* 0xA4-0xA7 */
	0x5A, 0xD5, 0xE9, 0xD0, 0xE9, 0xD1, 0xE9, 0xDB, /* 0xA8-0xAB */
	0x00, 0x00, 0x5A, 0xD6, 0x00, 0x00, 0xE9, 0xD5, /* 0xAC-0xAF */
	0xE9, 0xD8, 0x5A, 0xD7, 0x00, 0x00, 0x5A, 0xD8, /* 0xB0-0xB3 */
	0x5A, 0xD9, 0x5A, 0xDA, 0xE9, 0xD4, 0x5A, 0xDB, /* 0xB4-0xB7 */
	0x5A, 0xDC, 0x5A, 0xDD, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5A, 0xDE, /* 0xBC-0xBF */
	0x5A, 0xDF, 0x5A, 0xE0, 0x00, 0x00, 0x5A, 0xE1, /* 0xC0-0xC3 */
	0xE9, 0xD6, 0x5A, 0xE2, 0xE9, 0xD7, 0xBC, 0xD8, /* 0xC4-0xC7 */
	0x00, 0x00, 0xE9, 0xD9, 0x00, 0x00, 0xC3, 0xC1, /* 0xC8-0xCB */
	0x00, 0x00, 0xB7, 0xD6, 0xB3, 0xC2, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x5A, 0xE3, 0x5A, 0xE4, 0x00, 0x00, /* 0xD0-0xD3 */
	0xE9, 0xDC, 0x5A, 0xE5, 0x00, 0x00, 0x5A, 0xE6, /* 0xD4-0xD7 */
	0x00, 0x00, 0xB3, 0xBF, 0x5A, 0xE7, 0xE9, 0xE1, /* 0xD8-0xDB */
	0x5A, 0xE8, 0x00, 0x00, 0xE9, 0xDD, 0xE9, 0xE0, /* 0xDC-0xDF */
	0x5A, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x5A, 0xEA, /* 0xE0-0xE3 */
	0xC8, 0xBA, 0x5A, 0xEB, 0x00, 0x00, 0x5A, 0xEC, /* 0xE4-0xE7 */
	0x00, 0x00, 0xE9, 0xDE, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0xE9, 0xDF, 0xC9, 0xC8, 0xC8, 0xDA, 0xE9, 0xE2, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0xC2, 0xFD, 0xE9, 0xEC, 0x5A, 0xEE, 0xE9, 0xE8, /* 0xF8-0xFB */
	0x5A, 0xEF, 0x5A, 0xF0, 0xB2, 0xEB, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_87[512] = {
	0xE9, 0xE6, 0x00, 0x00, 0xCB, 0xAA, 0xE9, 0xE7, /* 0x00-0x03 */
	0x5A, 0xF1, 0x5A, 0xF2, 0xE9, 0xE4, 0x5A, 0xF3, /* 0x04-0x07 */
	0xE9, 0xE5, 0xE9, 0xEA, 0xE9, 0xED, 0x5A, 0xF4, /* 0x08-0x0B */
	0x00, 0x00, 0xE9, 0xEB, 0x5A, 0xF5, 0x5A, 0xF6, /* 0x0C-0x0F */
	0x5A, 0xF7, 0xE9, 0xE9, 0xE9, 0xE3, 0x5A, 0xF8, /* 0x10-0x13 */
	0x5A, 0xF9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xC3, 0xD8, 0x5A, 0xFA, 0xE9, 0xF4, 0x00, 0x00, /* 0x18-0x1B */
	0xCC, 0xAA, 0x00, 0x00, 0x5A, 0xFB, 0x5A, 0xFC, /* 0x1C-0x1F */
	0x00, 0x00, 0x5A, 0xFD, 0x00, 0x00, 0x5A, 0xFE, /* 0x20-0x23 */
	0x00, 0x00, 0xE9, 0xF2, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x5B, 0xA1, 0xE9, 0xF3, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x5B, 0xA2, 0x5B, 0xA3, /* 0x2C-0x2F */
	0x00, 0x00, 0x5B, 0xA4, 0x5B, 0xA5, 0x00, 0x00, /* 0x30-0x33 */
	0xE9, 0xEE, 0x00, 0x00, 0x00, 0x00, 0xE9, 0xF0, /* 0x34-0x37 */
	0x00, 0x00, 0x5B, 0xA6, 0x5B, 0xA7, 0xE9, 0xF1, /* 0x38-0x3B */
	0x5B, 0xA8, 0x5B, 0xA9, 0x5B, 0xAA, 0xE9, 0xEF, /* 0x3C-0x3F */
	0x5B, 0xAB, 0x00, 0x00, 0x00, 0x00, 0x5B, 0xAC, /* 0x40-0x43 */
	0x00, 0x00, 0x5B, 0xAD, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0xC0, 0xE6, 0x00, 0x00, 0xCF, 0xB9, /* 0x48-0x4B */
	0xE9, 0xF8, 0x5B, 0xAE, 0xE9, 0xF9, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEA, 0xA1, /* 0x50-0x53 */
	0x00, 0x00, 0xBF, 0xAA, 0x00, 0x00, 0xE9, 0xFB, /* 0x54-0x57 */
	0x5B, 0xAF, 0xE9, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x5B, 0xB0, 0x00, 0x00, 0xE9, 0xF6, /* 0x5C-0x5F */
	0xE9, 0xF5, 0x5B, 0xB1, 0x00, 0x00, 0xEA, 0xA2, /* 0x60-0x63 */
	0x5B, 0xB2, 0x5B, 0xB3, 0xB2, 0xDC, 0x00, 0x00, /* 0x64-0x67 */
	0xE9, 0xFC, 0x00, 0x00, 0xEA, 0xA3, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0xE9, 0xFD, 0x5B, 0xB4, /* 0x6C-0x6F */
	0x00, 0x00, 0x5B, 0xB5, 0x5B, 0xB6, 0x00, 0x00, /* 0x70-0x73 */
	0xE9, 0xFA, 0x00, 0x00, 0xC4, 0xB3, 0x00, 0x00, /* 0x74-0x77 */
	0xE9, 0xF7, 0x00, 0x00, 0x00, 0x00, 0x5B, 0xB7, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC7, 0xE8, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0xEA, 0xA7, 0x5B, 0xB8, /* 0x80-0x83 */
	0x5B, 0xB9, 0x5B, 0xBA, 0x5B, 0xBB, 0x5B, 0xBC, /* 0x84-0x87 */
	0x5B, 0xBD, 0x5B, 0xBE, 0x00, 0x00, 0x5B, 0xBF, /* 0x88-0x8B */
	0x5B, 0xC0, 0xCD, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x5B, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x5B, 0xC2, /* 0x90-0x93 */
	0x00, 0x00, 0x5B, 0xC3, 0x00, 0x00, 0x5B, 0xC4, /* 0x94-0x97 */
	0x5B, 0xC5, 0x5B, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x5B, 0xC7, 0xEA, 0xA6, /* 0x9C-0x9F */
	0x5B, 0xC8, 0x00, 0x00, 0xEA, 0xA5, 0x5B, 0xC9, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5B, 0xCA, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEA, 0xAE, /* 0xA8-0xAB */
	0x5B, 0xCB, 0x5B, 0xCC, 0x5B, 0xCD, 0xEA, 0xA8, /* 0xAC-0xAF */
	0x00, 0x00, 0x5B, 0xCE, 0x00, 0x00, 0xEA, 0xB0, /* 0xB0-0xB3 */
	0x00, 0x00, 0x5B, 0xCF, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0xCD, 0xE6, 0xEA, 0xB3, /* 0xB8-0xBB */
	0x00, 0x00, 0xEA, 0xAA, 0x5B, 0xD0, 0x5B, 0xD1, /* 0xBC-0xBF */
	0xEA, 0xAB, 0x5B, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0xEA, 0xAF, 0x00, 0x00, 0xEA, 0xB2, 0xEA, 0xB1, /* 0xC4-0xC7 */
	0x5B, 0xD3, 0x5B, 0xD4, 0x5B, 0xD5, 0xEA, 0xA9, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x5B, 0xD6, 0x00, 0x00, /* 0xCC-0xCF */
	0xEA, 0xAC, 0x00, 0x00, 0xEA, 0xBD, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x5B, 0xD7, 0x5B, 0xD8, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x5B, 0xD9, 0x5B, 0xDA, 0x00, 0x00, /* 0xD8-0xDB */
	0x5B, 0xDB, 0x00, 0x00, 0x00, 0x00, 0x5B, 0xDC, /* 0xDC-0xDF */
	0xEA, 0xB6, 0x00, 0x00, 0x5B, 0xDD, 0x5B, 0xDE, /* 0xE0-0xE3 */
	0x5B, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x5B, 0xE0, 0x5B, 0xE1, /* 0xE8-0xEB */
	0x00, 0x00, 0x5B, 0xE2, 0x00, 0x00, 0xEA, 0xB4, /* 0xEC-0xEF */
	0x00, 0x00, 0x5B, 0xE3, 0xEA, 0xB5, 0x5B, 0xE4, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0xEA, 0xBA, 0xEA, 0xBB, /* 0xF4-0xF7 */
	0x5B, 0xE5, 0xB3, 0xAA, 0x5B, 0xE6, 0xB5, 0xC2, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0xEA, 0xB9, 0x5B, 0xE7, /* 0xFC-0xFF */
};

static unsigned char u2c_88[512] = {
	0x00, 0x00, 0x5B, 0xE8, 0x00, 0x00, 0x5B, 0xE9, /* 0x00-0x03 */
	0x00, 0x00, 0xEA, 0xA4, 0x5B, 0xEA, 0xFB, 0xE5, /* 0x04-0x07 */
	0x00, 0x00, 0x5B, 0xEB, 0x5B, 0xEC, 0x5B, 0xED, /* 0x08-0x0B */
	0x00, 0x00, 0xEA, 0xB8, 0xEA, 0xBC, 0xEA, 0xB7, /* 0x0C-0x0F */
	0x5B, 0xEE, 0xEA, 0xBE, 0x5B, 0xF0, 0x5B, 0xF1, /* 0x10-0x13 */
	0x5B, 0xF2, 0xEA, 0xC0, 0xEA, 0xBF, 0x00, 0x00, /* 0x14-0x17 */
	0x5B, 0xF3, 0x5B, 0xEF, 0x5B, 0xF4, 0x5B, 0xF5, /* 0x18-0x1B */
	0x5B, 0xF6, 0x00, 0x00, 0x5B, 0xF7, 0x5B, 0xF8, /* 0x1C-0x1F */
	0x00, 0x00, 0xEA, 0xC2, 0xEA, 0xC1, 0xE9, 0xDA, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEA, 0xC6, /* 0x24-0x27 */
	0x5B, 0xF9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x5B, 0xFA, 0x5B, 0xFB, 0x00, 0x00, /* 0x2C-0x2F */
	0x5B, 0xFC, 0xEA, 0xC3, 0x5B, 0xFD, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x5B, 0xFE, 0xEA, 0xC4, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0xEA, 0xC5, 0x5C, 0xA1, 0xEA, 0xC7, /* 0x38-0x3B */
	0x5C, 0xA2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0xB7, 0xEC, 0x5C, 0xA3, 0xEA, 0xC9, 0x5C, 0xA4, /* 0x40-0x43 */
	0xEA, 0xC8, 0x5C, 0xA5, 0xBD, 0xB0, 0x00, 0x00, /* 0x44-0x47 */
	0x5C, 0xA6, 0x5C, 0xA7, 0x5C, 0xA8, 0x5C, 0xA9, /* 0x48-0x4B */
	0xB9, 0xD4, 0xDE, 0xA7, 0x5C, 0xAA, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x5C, 0xAB, 0xEA, 0xCA, 0xBD, 0xD1, /* 0x50-0x53 */
	0x00, 0x00, 0x5C, 0xAC, 0x5C, 0xAD, 0xB3, 0xB9, /* 0x54-0x57 */
	0x5C, 0xAE, 0xEA, 0xCB, 0x5C, 0xAF, 0xB1, 0xD2, /* 0x58-0x5B */
	0x5C, 0xB0, 0xBE, 0xD7, 0xEA, 0xCC, 0x5C, 0xB1, /* 0x5C-0x5F */
	0x5C, 0xB2, 0xB9, 0xD5, 0xEA, 0xCD, 0xB0, 0xE1, /* 0x60-0x63 */
	0x5C, 0xB3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0xC9, 0xBD, 0x5C, 0xB4, 0x00, 0x00, 0xEA, 0xCE, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0xBF, 0xEA, 0x5C, 0xB5, 0xEA, 0xD5, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xEA, 0xD2, 0x00, 0x00, 0xC3, 0xEF, /* 0x74-0x77 */
	0x00, 0x00, 0x5C, 0xB6, 0x00, 0x00, 0x5C, 0xB7, /* 0x78-0x7B */
	0x00, 0x00, 0xEA, 0xD3, 0xEA, 0xD0, 0xB6, 0xDE, /* 0x7C-0x7F */

	0x5C, 0xB8, 0xEA, 0xCF, 0xEA, 0xD6, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0xB7, 0xB6, 0x00, 0x00, 0x00, 0x00, 0xC2, 0xDE, /* 0x88-0x8B */
	0x00, 0x00, 0xEA, 0xDC, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0xEA, 0xD8, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xC2, 0xB5, 0xEA, 0xD7, /* 0x94-0x97 */
	0x5C, 0xB9, 0xEA, 0xDA, 0x5C, 0xBA, 0x5C, 0xBB, /* 0x98-0x9B */
	0x5C, 0xBC, 0x00, 0x00, 0xEA, 0xD1, 0x5C, 0xBD, /* 0x9C-0x9F */
	0x5C, 0xBE, 0x00, 0x00, 0xEA, 0xDB, 0x00, 0x00, /* 0xA0-0xA3 */
	0xEA, 0xDD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x5C, 0xBF, 0x00, 0x00, 0x5C, 0xC0, 0xC8, 0xEF, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0xEA, 0xD9, 0x00, 0x00, /* 0xAC-0xAF */
	0xEA, 0xDE, 0xEA, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0xB8, 0xD3, 0xEA, 0xD4, 0x00, 0x00, 0xB0, 0xC1, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xC1, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x5C, 0xC2, 0x5C, 0xC3, 0xEA, 0xDF, /* 0xBC-0xBF */
	0x5C, 0xC4, 0xBA, 0xDB, 0xCE, 0xF6, 0xEA, 0xE1, /* 0xC0-0xC3 */
	0xEA, 0xE2, 0xC1, 0xF5, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xC5, 0x5C, 0xC6, /* 0xC8-0xCB */
	0x5C, 0xC7, 0x5C, 0xC8, 0x5C, 0xC9, 0xCE, 0xA2, /* 0xCC-0xCF */
	0x00, 0x00, 0x5C, 0xCA, 0x5C, 0xCB, 0x5C, 0xCC, /* 0xD0-0xD3 */
	0xEA, 0xE3, 0xCD, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xEA, 0xE4, 0xEA, 0xE5, 0x00, 0x00, 0x5C, 0xCD, /* 0xD8-0xDB */
	0xCA, 0xE4, 0xEA, 0xE6, 0x5C, 0xCE, 0xBA, 0xC0, /* 0xDC-0xDF */
	0x00, 0x00, 0xCE, 0xA3, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5C, 0xCF, /* 0xE4-0xE7 */
	0xEA, 0xEB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5C, 0xD0, /* 0xEC-0xEF */
	0x5C, 0xD1, 0x5C, 0xD2, 0xEA, 0xEC, 0xBE, 0xD8, /* 0xF0-0xF3 */
	0xEA, 0xEA, 0xFB, 0xE6, 0x00, 0x00, 0x5C, 0xD4, /* 0xF4-0xF7 */
	0xCD, 0xE7, 0xEA, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0xEA, 0xE9, 0xC0, 0xBD, 0xBF, 0xFE, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_89[512] = {
	0x00, 0x00, 0x5C, 0xD5, 0xEA, 0xE8, 0x00, 0x00, /* 0x00-0x03 */
	0xEA, 0xED, 0x00, 0x00, 0x5C, 0xD6, 0xCA, 0xA3, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0xEA, 0xEF, 0x00, 0x00, /* 0x08-0x0B */
	0xEA, 0xEE, 0x5C, 0xD7, 0x5C, 0xD8, 0x5C, 0xD9, /* 0x0C-0x0F */
	0xB3, 0xEC, 0x00, 0x00, 0xCB, 0xAB, 0xEA, 0xF0, /* 0x10-0x13 */
	0x00, 0x00, 0x5C, 0xDA, 0x5C, 0xDB, 0x00, 0x00, /* 0x14-0x17 */
	0x5C, 0xDC, 0x5C, 0xDD, 0x5C, 0xDE, 0x00, 0x00, /* 0x18-0x1B */
	0xF9, 0xA2, 0xEA, 0xFC, 0xEA, 0xF2, 0x00, 0x00, /* 0x1C-0x1F */
	0x5C, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0xEA, 0xF3, 0x5C, 0xE1, 0x5C, 0xE2, /* 0x24-0x27 */
	0x5C, 0xE3, 0x00, 0x00, 0xEA, 0xF4, 0xEA, 0xF5, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x5C, 0xE4, 0x5C, 0xE5, 0x5C, 0xE6, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x5C, 0xE7, 0xEA, 0xF9, 0x00, 0x00, /* 0x34-0x37 */
	0xEA, 0xFA, 0x5C, 0xE8, 0x5C, 0xE9, 0xEA, 0xF8, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xEA, 0x00, 0x00, /* 0x3C-0x3F */
	0x5C, 0xEB, 0xEA, 0xF6, 0x5C, 0xEC, 0xEA, 0xF1, /* 0x40-0x43 */
	0xEA, 0xF7, 0x5C, 0xED, 0x5C, 0xEE, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x5C, 0xEF, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0xEA, 0xFB, 0xF0, 0xB7, 0x00, 0x00, 0x5C, 0xF0, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xF1, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0xB2, 0xA8, 0x5C, 0xF2, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xF3, 0x5C, 0xF4, /* 0x58-0x5B */
	0x5C, 0xF5, 0x00, 0x00, 0xEA, 0xFE, 0xB6, 0xDF, /* 0x5C-0x5F */
	0xEA, 0xFD, 0x5C, 0xF6, 0x5C, 0xF7, 0x5C, 0xF8, /* 0x60-0x63 */
	0xEB, 0xA2, 0x00, 0x00, 0xEB, 0xA1, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xEB, 0xA4, 0x5C, 0xF9, /* 0x68-0x6B */
	0x00, 0x00, 0xEB, 0xA3, 0x5C, 0xFA, 0xEB, 0xA5, /* 0x6C-0x6F */
	0x5C, 0xFB, 0x00, 0x00, 0xBD, 0xB1, 0x5C, 0xFC, /* 0x70-0x73 */
	0xEB, 0xA6, 0x5C, 0xFD, 0x00, 0x00, 0xEB, 0xA7, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x5C, 0xFE, 0x5D, 0xA1, /* 0x78-0x7B */
	0x5D, 0xA2, 0x5D, 0xA3, 0xEB, 0xA8, 0xC0, 0xBE, /* 0x7C-0x7F */

	0x00, 0x00, 0xCD, 0xD7, 0x00, 0x00, 0xEB, 0xA9, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0xCA, 0xA4, 0xC7, 0xC6, /* 0x84-0x87 */
	0xEB, 0xAA, 0x5D, 0xA4, 0xEB, 0xAB, 0xB8, 0xAB, /* 0x88-0x8B */
	0x00, 0x00, 0x5D, 0xA5, 0x00, 0x00, 0xB5, 0xAC, /* 0x8C-0x8F */
	0x5D, 0xA6, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xAC, /* 0x90-0x93 */
	0x5D, 0xA7, 0x5D, 0xA8, 0xBB, 0xEB, 0xC7, 0xC1, /* 0x94-0x97 */
	0xEB, 0xAD, 0x00, 0x00, 0xB3, 0xD0, 0x5D, 0xA9, /* 0x98-0x9B */
	0x5D, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x5D, 0xAB, /* 0x9C-0x9F */
	0x5D, 0xAC, 0xEB, 0xAE, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x5D, 0xAD, 0xEB, 0xB0, 0xCD, 0xF7, /* 0xA4-0xA7 */
	0x00, 0x00, 0xEB, 0xAF, 0xBF, 0xC6, 0x00, 0x00, /* 0xA8-0xAB */
	0xEB, 0xB1, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xB2, /* 0xAC-0xAF */
	0x5D, 0xAE, 0x00, 0x00, 0xEB, 0xB3, 0xB4, 0xD1, /* 0xB0-0xB3 */
	0x5D, 0xAF, 0x5D, 0xB0, 0x5D, 0xB1, 0x5D, 0xB2, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0xEB, 0xB4, 0x00, 0x00, /* 0xB8-0xBB */
	0x5D, 0xB3, 0xEB, 0xB5, 0x00, 0x00, 0xEB, 0xB6, /* 0xBC-0xBF */
	0xEB, 0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0xB3, 0xD1, 0x00, 0x00, /* 0xD0-0xD3 */
	0x5D, 0xB4, 0x5D, 0xB5, 0x5D, 0xB6, 0x5D, 0xB7, /* 0xD4-0xD7 */
	0x5D, 0xB8, 0x00, 0x00, 0xEB, 0xB8, 0x00, 0x00, /* 0xD8-0xDB */
	0xEB, 0xB9, 0xEB, 0xBA, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB2, 0xF2, /* 0xE0-0xE3 */
	0x00, 0x00, 0x5D, 0xB9, 0xBF, 0xA8, 0xEB, 0xBB, /* 0xE4-0xE7 */
	0x00, 0x00, 0x5D, 0xBA, 0x00, 0x00, 0x5D, 0xBB, /* 0xE8-0xEB */
	0x00, 0x00, 0x5D, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x5D, 0xBD, 0x00, 0x00, 0x5D, 0xBE, /* 0xF0-0xF3 */
	0xEB, 0xBC, 0x00, 0x00, 0x5D, 0xBF, 0x00, 0x00, /* 0xF4-0xF7 */
	0xEB, 0xBD, 0x5D, 0xC0, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0x5D, 0xC1, 0x00, 0x00, 0x5D, 0xC2, /* 0xFC-0xFF */
};

static unsigned char u2c_8A[512] = {
	0xB8, 0xC0, 0x00, 0x00, 0xC4, 0xFB, 0xEB, 0xBE, /* 0x00-0x03 */
	0x5D, 0xC3, 0x5D, 0xC4, 0x00, 0x00, 0x5D, 0xC5, /* 0x04-0x07 */
	0xB7, 0xD7, 0x00, 0x00, 0xBF, 0xD6, 0x00, 0x00, /* 0x08-0x0B */
	0xEB, 0xC1, 0x00, 0x00, 0xC6, 0xA4, 0x5D, 0xC6, /* 0x0C-0x0F */
	0xEB, 0xC0, 0x5D, 0xC7, 0xFB, 0xE7, 0xB7, 0xB1, /* 0x10-0x13 */
	0x5D, 0xC9, 0x5D, 0xCA, 0xEB, 0xBF, 0xC2, 0xF7, /* 0x14-0x17 */
	0xB5, 0xAD, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xC2, /* 0x18-0x1B */
	0x00, 0x00, 0xEB, 0xC3, 0x5D, 0xCB, 0xBE, 0xD9, /* 0x1C-0x1F */
	0x5D, 0xCC, 0x00, 0x00, 0x5D, 0xCD, 0xB7, 0xED, /* 0x20-0x23 */
	0x5D, 0xCE, 0xEB, 0xC4, 0x5D, 0xCF, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0xCB, 0xAC, 0x5D, 0xD0, /* 0x28-0x2B */
	0x5D, 0xD1, 0xC0, 0xDF, 0x00, 0x00, 0x5D, 0xD2, /* 0x2C-0x2F */
	0x00, 0x00, 0xB5, 0xF6, 0x00, 0x00, 0xCC, 0xF5, /* 0x30-0x33 */
	0xC1, 0xCA, 0x5D, 0xD3, 0xEB, 0xC5, 0xFB, 0xE8, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xBF, 0xC7, 0xC3, 0xF0, /* 0x38-0x3B */
	0xBE, 0xDA, 0x5D, 0xD5, 0x5D, 0xD6, 0x00, 0x00, /* 0x3C-0x3F */
	0x5D, 0xD7, 0xEB, 0xC6, 0x00, 0x00, 0x5D, 0xD8, /* 0x40-0x43 */
	0x00, 0x00, 0x5D, 0xD9, 0xEB, 0xC9, 0x5D, 0xDA, /* 0x44-0x47 */
	0xEB, 0xCA, 0x5D, 0xDB, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x5D, 0xDC, 0x5D, 0xDD, 0x00, 0x00, /* 0x4C-0x4F */
	0xBA, 0xBE, 0xC2, 0xC2, 0xEB, 0xC8, 0x5D, 0xDE, /* 0x50-0x53 */
	0xBE, 0xDB, 0xC9, 0xBE, 0x5D, 0xDF, 0x5D, 0xE0, /* 0x54-0x57 */
	0x5D, 0xE1, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xC7, /* 0x58-0x5B */
	0x5D, 0xE2, 0x5D, 0xE3, 0xBB, 0xEC, 0x00, 0x00, /* 0x5C-0x5F */
	0xB1, 0xD3, 0x5D, 0xE4, 0xEB, 0xCE, 0xB7, 0xD8, /* 0x60-0x63 */
	0x00, 0x00, 0x5D, 0xE5, 0xBB, 0xEE, 0x5D, 0xE6, /* 0x64-0x67 */
	0x00, 0x00, 0xBB, 0xED, 0x00, 0x00, 0xCF, 0xCD, /* 0x68-0x6B */
	0xEB, 0xCD, 0xEB, 0xCC, 0xC1, 0xA7, 0x00, 0x00, /* 0x6C-0x6F */
	0xB5, 0xCD, 0xCF, 0xC3, 0xB3, 0xBA, 0xBE, 0xDC, /* 0x70-0x73 */
	0x00, 0x00, 0x5D, 0xE7, 0x5D, 0xE8, 0x5D, 0xE9, /* 0x74-0x77 */
	0x00, 0x00, 0xFB, 0xE9, 0x5D, 0xEB, 0x5D, 0xEC, /* 0x78-0x7B */
	0xEB, 0xCB, 0x00, 0x00, 0x5D, 0xED, 0x5D, 0xEE, /* 0x7C-0x7F */

	0x5D, 0xEF, 0x00, 0x00, 0xEB, 0xD0, 0x5D, 0xF0, /* 0x80-0x83 */
	0xEB, 0xD1, 0xEB, 0xCF, 0x5D, 0xF1, 0xB8, 0xD8, /* 0x84-0x87 */
	0x00, 0x00, 0xCD, 0xC0, 0x00, 0x00, 0x5D, 0xF2, /* 0x88-0x8B */
	0xBB, 0xEF, 0xC7, 0xA7, 0x00, 0x00, 0x5D, 0xF3, /* 0x8C-0x8F */
	0x5D, 0xF4, 0xEB, 0xD4, 0x5D, 0xF5, 0xC0, 0xC0, /* 0x90-0x93 */
	0x00, 0x00, 0xC3, 0xC2, 0x5D, 0xF6, 0x5D, 0xF7, /* 0x94-0x97 */
	0xCD, 0xB6, 0x5D, 0xF8, 0xEB, 0xD7, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0xB8, 0xEC, 0x5D, 0xF9, /* 0x9C-0x9F */
	0xC0, 0xBF, 0xEB, 0xD3, 0x00, 0x00, 0xEB, 0xD8, /* 0xA0-0xA3 */
	0xB8, 0xED, 0xEB, 0xD5, 0xEB, 0xD6, 0xFB, 0xEA, /* 0xA4-0xA7 */
	0xEB, 0xD2, 0x5D, 0xFB, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0xC0, 0xE2, 0xC6, 0xC9, 0x5D, 0xFC, 0x5D, 0xFD, /* 0xAC-0xAF */
	0xC3, 0xAF, 0x00, 0x00, 0xB2, 0xDD, 0x5D, 0xFE, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x5E, 0xA1, 0x5E, 0xA2, /* 0xB4-0xB7 */
	0x00, 0x00, 0xC8, 0xF0, 0x00, 0x00, 0x5E, 0xA3, /* 0xB8-0xBB */
	0xB5, 0xC3, 0x00, 0x00, 0xFB, 0xEB, 0xC4, 0xB4, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0xEB, 0xDB, 0x5E, 0xA5, /* 0xC0-0xC3 */
	0xEB, 0xD9, 0x00, 0x00, 0x5E, 0xA6, 0xC3, 0xCC, /* 0xC4-0xC7 */
	0x5E, 0xA7, 0x5E, 0xA8, 0x5E, 0xA9, 0xC0, 0xC1, /* 0xC8-0xCB */
	0xB4, 0xD2, 0xEB, 0xDA, 0x00, 0x00, 0xBF, 0xDB, /* 0xCC-0xCF */
	0x00, 0x00, 0x5E, 0xAA, 0xCE, 0xCA, 0x5E, 0xAB, /* 0xD0-0xD3 */
	0x5E, 0xAC, 0x5E, 0xAD, 0xCF, 0xC0, 0x5E, 0xAE, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xEB, 0xDC, 0xEB, 0xE7, /* 0xD8-0xDB */
	0xC4, 0xB5, 0x5E, 0xAF, 0xEB, 0xE6, 0xFB, 0xEC, /* 0xDC-0xDF */
	0xEB, 0xE3, 0xEB, 0xEB, 0xEB, 0xE4, 0x00, 0x00, /* 0xE0-0xE3 */
	0xEB, 0xE0, 0x00, 0x00, 0xC4, 0xFC, 0xEB, 0xDF, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xDD, /* 0xE8-0xEB */
	0x5E, 0xB1, 0xCD, 0xA1, 0xBB, 0xF0, 0x00, 0x00, /* 0xEC-0xEF */
	0x5E, 0xB2, 0xEB, 0xE1, 0x00, 0x00, 0xEB, 0xDE, /* 0xF0-0xF3 */
	0x5E, 0xB3, 0x5E, 0xB4, 0xFB, 0xEE, 0xEB, 0xE5, /* 0xF4-0xF7 */
	0xBD, 0xF4, 0x00, 0x00, 0xB8, 0xC1, 0x00, 0x00, /* 0xF8-0xFB */
	0x5E, 0xB6, 0x00, 0x00, 0xC2, 0xFA, 0x5E, 0xB7, /* 0xFC-0xFF */
};

static unsigned char u2c_8B[512] = {
	0xCB, 0xC5, 0xB1, 0xDA, 0xB0, 0xE2, 0x00, 0x00, /* 0x00-0x03 */
	0xC6, 0xA5, 0x5E, 0xB8, 0x5E, 0xB9, 0xEB, 0xE9, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x5E, 0xBF, 0x5E, 0xBA, /* 0x08-0x0B */
	0xEB, 0xE8, 0x00, 0x00, 0xC6, 0xE6, 0x00, 0x00, /* 0x0C-0x0F */
	0xEB, 0xED, 0x5E, 0xBB, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0xEB, 0xE2, 0x00, 0x00, 0xEB, 0xEC, 0xEB, 0xEE, /* 0x14-0x17 */
	0x00, 0x00, 0xB8, 0xAC, 0xEB, 0xEA, 0xB9, 0xD6, /* 0x18-0x1B */
	0x5E, 0xBC, 0xBC, 0xD5, 0x5E, 0xBD, 0x5E, 0xBE, /* 0x1C-0x1F */
	0xEB, 0xEF, 0xCD, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xEB, 0xF2, 0x00, 0x00, /* 0x24-0x27 */
	0xEB, 0xF5, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xF3, /* 0x28-0x2B */
	0xC9, 0xB5, 0x5E, 0xC0, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x5E, 0xC1, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xF0, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5E, 0xC2, /* 0x34-0x37 */
	0x00, 0x00, 0xB6, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x5E, 0xC3, 0x00, 0x00, 0xEB, 0xF4, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0xEB, 0xF6, 0x5E, 0xC4, 0x5E, 0xC5, /* 0x40-0x43 */
	0x5E, 0xC6, 0x5E, 0xC7, 0x5E, 0xC8, 0x00, 0x00, /* 0x44-0x47 */
	0x5E, 0xC9, 0xEB, 0xFA, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0xEB, 0xF7, 0x5E, 0xCE, 0xEB, 0xF9, 0xEB, 0xF8, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x5E, 0xCA, 0xFB, 0xEF, /* 0x50-0x53 */
	0x5E, 0xCC, 0x00, 0x00, 0xEB, 0xFB, 0x00, 0x00, /* 0x54-0x57 */
	0xBC, 0xB1, 0x5E, 0xCD, 0xEB, 0xFD, 0xEB, 0xFC, /* 0x58-0x5B */
	0xC9, 0xE8, 0x00, 0x00, 0x5E, 0xCF, 0xEC, 0xA1, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5E, 0xD0, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0xB7, 0xD9, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xFE, /* 0x68-0x6B */
	0xEC, 0xA2, 0x5E, 0xD1, 0x00, 0x00, 0xEC, 0xA3, /* 0x6C-0x6F */
	0xB5, 0xC4, 0xE6, 0xC1, 0xBE, 0xF9, 0x00, 0x00, /* 0x70-0x73 */
	0xEC, 0xA4, 0x00, 0x00, 0x5E, 0xD2, 0xB8, 0xEE, /* 0x74-0x77 */
	0x5E, 0xD3, 0x5E, 0xD4, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x5E, 0xD5, 0xEC, 0xA5, 0x5E, 0xD6, 0xFB, 0xF0, /* 0x7C-0x7F */

	0xEC, 0xA6, 0x5E, 0xD7, 0x00, 0x00, 0xBB, 0xBE, /* 0x80-0x83 */
	0x5E, 0xD8, 0x5E, 0xD9, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0xDA, 0xCE, 0x5E, 0xDA, /* 0x88-0x8B */
	0xEC, 0xA7, 0x5E, 0xDB, 0xEC, 0xA8, 0x5E, 0xDC, /* 0x8C-0x8F */
	0xBD, 0xB2, 0x00, 0x00, 0xEC, 0xA9, 0xEC, 0xAA, /* 0x90-0x93 */
	0x5E, 0xDD, 0x5E, 0xDE, 0xEC, 0xAB, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xEC, 0xAC, 0xEC, 0xAD, 0x00, 0x00, /* 0x98-0x9B */
	0x5E, 0xDF, 0x00, 0x00, 0x5E, 0xE0, 0x5E, 0xE1, /* 0x9C-0x9F */
};

static unsigned char u2c_8C[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC3, 0xAB, /* 0x34-0x37 */
	0x5E, 0xE2, 0x5E, 0xE3, 0xEC, 0xAE, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x5E, 0xE4, 0x5E, 0xE5, 0xEC, 0xB0, /* 0x3C-0x3F */
	0x00, 0x00, 0xEC, 0xAF, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x5E, 0xE6, 0xC6, 0xA6, 0x5E, 0xE7, /* 0x44-0x47 */
	0xEC, 0xB1, 0x5E, 0xE8, 0xCB, 0xAD, 0x5E, 0xE9, /* 0x48-0x4B */
	0xEC, 0xB2, 0x00, 0x00, 0xEC, 0xB3, 0x5E, 0xEA, /* 0x4C-0x4F */
	0xEC, 0xB4, 0x5E, 0xEB, 0x00, 0x00, 0x5E, 0xEC, /* 0x50-0x53 */
	0x5E, 0xED, 0xEC, 0xB5, 0x00, 0x00, 0x5E, 0xEE, /* 0x54-0x57 */
	0x5E, 0xEF, 0x5E, 0xF2, 0xC6, 0xDA, 0x5E, 0xF0, /* 0x58-0x5B */
	0x00, 0x00, 0x5E, 0xF1, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0xBE, 0xDD, 0xEC, 0xB6, 0x5E, 0xF3, /* 0x60-0x63 */
	0x5E, 0xF4, 0x00, 0x00, 0x5E, 0xF5, 0x00, 0x00, /* 0x64-0x67 */
	0x5E, 0xF6, 0x5E, 0xF7, 0xB9, 0xEB, 0xD0, 0xAE, /* 0x68-0x6B */
	0xEC, 0xB7, 0x5E, 0xF8, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5E, 0xF9, /* 0x70-0x73 */
	0x00, 0x00, 0x5E, 0xFA, 0x5E, 0xFB, 0x00, 0x00, /* 0x74-0x77 */
	0xEC, 0xB8, 0xC9, 0xBF, 0xEC, 0xB9, 0x5E, 0xFC, /* 0x78-0x7B */
	0xEC, 0xC1, 0x00, 0x00, 0x5E, 0xFD, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0xEC, 0xBA, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0xEC, 0xBC, 0x5E, 0xFE, 0x5F, 0xA1, /* 0x84-0x87 */
	0x00, 0x00, 0xEC, 0xBB, 0xEC, 0xBD, 0x5F, 0xA2, /* 0x88-0x8B */
	0xCB, 0xC6, 0xEC, 0xBE, 0xEC, 0xBF, 0x00, 0x00, /* 0x8C-0x8F */
	0x5F, 0xA3, 0x00, 0x00, 0x5F, 0xA4, 0x5F, 0xA5, /* 0x90-0x93 */
	0xEC, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0xEC, 0xC2, 0x5F, 0xA6, 0x00, 0x00, 0x5F, 0xA7, /* 0x98-0x9B */
	0x5F, 0xA8, 0xB3, 0xAD, 0xC4, 0xE7, 0x00, 0x00, /* 0x9C-0x9F */
	0xC9, 0xE9, 0xBA, 0xE2, 0xB9, 0xD7, 0x00, 0x00, /* 0xA0-0xA3 */
	0x5F, 0xA9, 0x00, 0x00, 0x00, 0x00, 0xC9, 0xCF, /* 0xA4-0xA7 */
	0xB2, 0xDF, 0xC8, 0xCE, 0xEC, 0xC5, 0xB4, 0xD3, /* 0xA8-0xAB */
	0xC0, 0xD5, 0xEC, 0xC4, 0xEC, 0xC9, 0xC3, 0xF9, /* 0xAC-0xAF */
	0xCC, 0xE3, 0x00, 0x00, 0xEC, 0xC7, 0xEC, 0xC8, /* 0xB0-0xB3 */
	0xB5, 0xAE, 0x00, 0x00, 0xEC, 0xCA, 0xC7, 0xE3, /* 0xB4-0xB7 */
	0xC2, 0xDF, 0x5F, 0xAA, 0x5F, 0xAB, 0xC8, 0xF1, /* 0xB8-0xBB */
	0xC5, 0xBD, 0xEC, 0xC6, 0x00, 0x00, 0xCB, 0xC7, /* 0xBC-0xBF */
	0xB2, 0xEC, 0xEC, 0xCC, 0xCF, 0xA8, 0xC4, 0xC2, /* 0xC0-0xC3 */
	0xCF, 0xC5, 0x5F, 0xAC, 0x5F, 0xAD, 0xBB, 0xF1, /* 0xC4-0xC7 */
	0xEC, 0xCB, 0x5F, 0xAE, 0xC2, 0xB1, 0x5F, 0xAF, /* 0xC8-0xCB */
	0x00, 0x00, 0xEC, 0xDC, 0xC1, 0xA8, 0x5F, 0xB0, /* 0xCC-0xCF */
	0x00, 0x00, 0xC6, 0xF8, 0x00, 0x00, 0xC9, 0xD0, /* 0xD0-0xD3 */
	0x00, 0x00, 0x5F, 0xB2, 0x5F, 0xB1, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x5F, 0xB3, 0xEC, 0xCF, 0xBB, 0xBF, /* 0xD8-0xDB */
	0xBB, 0xF2, 0x5F, 0xB4, 0xBE, 0xDE, 0x00, 0x00, /* 0xDC-0xDF */
	0xC7, 0xE5, 0x5F, 0xB5, 0xB8, 0xAD, 0xEC, 0xCE, /* 0xE0-0xE3 */
	0xEC, 0xCD, 0x00, 0x00, 0xC9, 0xEA, 0x00, 0x00, /* 0xE4-0xE7 */
	0x5F, 0xB6, 0x00, 0x00, 0xBC, 0xC1, 0x00, 0x00, /* 0xE8-0xEB */
	0x5F, 0xB7, 0xC5, 0xD2, 0x00, 0x00, 0x5F, 0xB8, /* 0xEC-0xEF */
	0xFB, 0xF1, 0x00, 0x00, 0x5F, 0xBA, 0x00, 0x00, /* 0xF0-0xF3 */
	0xFB, 0xF2, 0x5F, 0xBB, 0x00, 0x00, 0x5F, 0xBC, /* 0xF4-0xF7 */
	0x5F, 0xBD, 0x00, 0x00, 0xEC, 0xD1, 0xEC, 0xD2, /* 0xF8-0xFB */
	0xB9, 0xD8, 0xEC, 0xD0, 0x5F, 0xBE, 0x5F, 0xBF, /* 0xFC-0xFF */
};

static unsigned char u2c_8D[512] = {
	0x00, 0x00, 0x5F, 0xC0, 0x00, 0x00, 0x5F, 0xC1, /* 0x00-0x03 */
	0xEC, 0xD3, 0xEC, 0xD4, 0x00, 0x00, 0xEC, 0xD6, /* 0x04-0x07 */
	0xC2, 0xA3, 0x5F, 0xC2, 0xEC, 0xD5, 0xB4, 0xE6, /* 0x08-0x0B */
	0x00, 0x00, 0xEC, 0xD8, 0x00, 0x00, 0xEC, 0xD7, /* 0x0C-0x0F */
	0xEC, 0xD9, 0x00, 0x00, 0xFB, 0xF3, 0xEC, 0xDB, /* 0x10-0x13 */
	0xEC, 0xDD, 0x00, 0x00, 0xEC, 0xDE, 0x5F, 0xC4, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5F, 0xC5, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0xC0, 0xD6, 0x5F, 0xC6, 0xBC, 0xCF, 0xEC, 0xDF, /* 0x64-0x67 */
	0x00, 0x00, 0x5F, 0xC7, 0x00, 0x00, 0xB3, 0xD2, /* 0x68-0x6B */
	0x5F, 0xC8, 0xEC, 0xE0, 0x5F, 0xC9, 0x00, 0x00, /* 0x6C-0x6F */
	0xC1, 0xF6, 0xEC, 0xE1, 0x00, 0x00, 0xEC, 0xE2, /* 0x70-0x73 */
	0xC9, 0xEB, 0x00, 0x00, 0xFB, 0xF4, 0xB5, 0xAF, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5F, 0xCA, /* 0x7C-0x7F */

	0x00, 0x00, 0xEC, 0xE3, 0x5F, 0xCB, 0x00, 0x00, /* 0x80-0x83 */
	0x5F, 0xCC, 0xC4, 0xB6, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x5F, 0xCD, 0x00, 0x00, 0xB1, 0xDB, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x5F, 0xCE, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x5F, 0xCF, 0x5F, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x5F, 0xD1, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xEC, 0xE4, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x5F, 0xD2, 0x5F, 0xD3, /* 0x9C-0x9F */
	0x5F, 0xD4, 0x00, 0x00, 0x00, 0x00, 0xBC, 0xF1, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x5F, 0xD5, 0x00, 0x00, /* 0xA4-0xA7 */
	0xBF, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x5F, 0xD6, /* 0xA8-0xAB */
	0x5F, 0xD7, 0x00, 0x00, 0x00, 0x00, 0x5F, 0xD8, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x5F, 0xD9, 0xC2, 0xAD, /* 0xB0-0xB3 */
	0x00, 0x00, 0x5F, 0xDA, 0x00, 0x00, 0x5F, 0xDB, /* 0xB4-0xB7 */
	0x00, 0x00, 0x5F, 0xDC, 0xEC, 0xE7, 0x5F, 0xDD, /* 0xB8-0xBB */
	0x5F, 0xEF, 0x00, 0x00, 0xEC, 0xE6, 0x00, 0x00, /* 0xBC-0xBF */
	0x5F, 0xDE, 0x00, 0x00, 0xEC, 0xE5, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x5F, 0xDF, 0x5F, 0xE0, 0x5F, 0xE1, /* 0xC4-0xC7 */
	0x5F, 0xE2, 0x00, 0x00, 0x5F, 0xE3, 0xEC, 0xED, /* 0xC8-0xCB */
	0xEC, 0xEB, 0x00, 0x00, 0x5F, 0xE4, 0xEC, 0xE8, /* 0xCC-0xCF */
	0x00, 0x00, 0x5F, 0xE5, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x5F, 0xE6, 0x5F, 0xE7, 0xEC, 0xEA, 0x5F, 0xE8, /* 0xD4-0xD7 */
	0x00, 0x00, 0x5F, 0xE9, 0xEC, 0xE9, 0xEC, 0xEC, /* 0xD8-0xDB */
	0x00, 0x00, 0xB5, 0xF7, 0x00, 0x00, 0xEC, 0xF0, /* 0xDC-0xDF */
	0x00, 0x00, 0xC0, 0xD7, 0x00, 0x00, 0xEC, 0xF1, /* 0xE0-0xE3 */
	0x5F, 0xEA, 0x5F, 0xEB, 0x00, 0x00, 0x5F, 0xEC, /* 0xE4-0xE7 */
	0xB8, 0xD9, 0x00, 0x00, 0xEC, 0xEE, 0xEC, 0xEF, /* 0xE8-0xEB */
	0x5F, 0xED, 0x00, 0x00, 0x00, 0x00, 0xCF, 0xA9, /* 0xEC-0xEF */
	0x5F, 0xEE, 0x5F, 0xF0, 0x5F, 0xF1, 0xC4, 0xB7, /* 0xF0-0xF3 */
	0x5F, 0xF2, 0xC1, 0xA9, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0xEC, 0xF2, 0x5F, 0xF3, 0x00, 0x00, 0xEC, 0xF5, /* 0xFC-0xFF */
};

static unsigned char u2c_8E[512] = {
	0x00, 0x00, 0x5F, 0xF4, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x5F, 0xF5, 0x5F, 0xF6, 0x5F, 0xF7, 0x00, 0x00, /* 0x04-0x07 */
	0xEC, 0xF3, 0xEC, 0xF4, 0xCD, 0xD9, 0x5F, 0xF8, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xA7, /* 0x0C-0x0F */
	0xEC, 0xF8, 0x5F, 0xF9, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x5F, 0xFA, 0x00, 0x00, 0x5F, 0xFB, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0xEC, 0xF6, 0xEC, 0xF7, 0xEC, 0xF9, /* 0x1C-0x1F */
	0x5F, 0xFC, 0x5F, 0xFD, 0x5F, 0xFE, 0x60, 0xA1, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x60, 0xA2, 0x60, 0xA3, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0xED, 0xA9, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xEC, 0xFC, 0x60, 0xA4, 0x00, 0x00, 0x60, 0xA5, /* 0x30-0x33 */
	0xEC, 0xFD, 0xEC, 0xFB, 0x60, 0xA6, 0x60, 0xA7, /* 0x34-0x37 */
	0x60, 0xA8, 0x60, 0xA9, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x60, 0xAA, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x60, 0xAB, 0x60, 0xAC, 0xEC, 0xFA, 0x00, 0x00, /* 0x40-0x43 */
	0xC4, 0xFD, 0x00, 0x00, 0x00, 0x00, 0xED, 0xA1, /* 0x44-0x47 */
	0xED, 0xA5, 0xED, 0xA2, 0xEC, 0xFE, 0x60, 0xAD, /* 0x48-0x4B */
	0xED, 0xA3, 0x60, 0xAE, 0x60, 0xAF, 0x60, 0xB0, /* 0x4C-0x4F */
	0xED, 0xA4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x60, 0xB1, 0xED, 0xAB, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0xED, 0xA6, 0x00, 0x00, 0x60, 0xB2, /* 0x58-0x5B */
	0x60, 0xB3, 0x60, 0xB4, 0x60, 0xB5, 0xC0, 0xD8, /* 0x5C-0x5F */
	0xED, 0xA8, 0x60, 0xB6, 0x60, 0xB7, 0xED, 0xAA, /* 0x60-0x63 */
	0xED, 0xA7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x60, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x60, 0xB9, 0x60, 0xBA, 0x00, 0x00, 0x60, 0xBB, /* 0x6C-0x6F */
	0x60, 0xBC, 0x60, 0xBD, 0xED, 0xAD, 0x00, 0x00, /* 0x70-0x73 */
	0xBD, 0xB3, 0x00, 0x00, 0xED, 0xAC, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x60, 0xBE, 0x60, 0xBF, 0x60, 0xC0, /* 0x78-0x7B */
	0xED, 0xAE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0xED, 0xAF, 0x60, 0xC1, 0x60, 0xC2, /* 0x80-0x83 */
	0xED, 0xB2, 0xED, 0xB1, 0x00, 0x00, 0xED, 0xB0, /* 0x84-0x87 */
	0x00, 0x00, 0x60, 0xC3, 0xED, 0xB4, 0xED, 0xB3, /* 0x88-0x8B */
	0x00, 0x00, 0xCC, 0xF6, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x60, 0xC4, 0xED, 0xB6, 0x60, 0xC5, 0xED, 0xB5, /* 0x90-0x93 */
	0xED, 0xB7, 0x60, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0xED, 0xB8, 0x60, 0xC7, 0x60, 0xC8, /* 0x98-0x9B */
	0x00, 0x00, 0x60, 0xC9, 0x60, 0xCA, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0xED, 0xBA, 0x60, 0xCB, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xCC, /* 0xA4-0xA7 */
	0x00, 0x00, 0x60, 0xCD, 0xED, 0xB9, 0xBF, 0xC8, /* 0xA8-0xAB */
	0xED, 0xBB, 0x60, 0xCE, 0x60, 0xCF, 0xB6, 0xED, /* 0xAC-0xAF */
	0xED, 0xBC, 0xED, 0xBE, 0x00, 0x00, 0x60, 0xD0, /* 0xB0-0xB3 */
	0x00, 0x00, 0x60, 0xD1, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x60, 0xD2, 0x60, 0xD3, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0xED, 0xBF, 0x00, 0x00, /* 0xBC-0xBF */
	0x60, 0xD4, 0x60, 0xD5, 0x00, 0x00, 0x60, 0xD6, /* 0xC0-0xC3 */
	0x60, 0xD7, 0xED, 0xC0, 0xED, 0xBD, 0x60, 0xD8, /* 0xC4-0xC7 */
	0xED, 0xC1, 0x00, 0x00, 0xBC, 0xD6, 0xED, 0xC2, /* 0xC8-0xCB */
	0xB5, 0xB0, 0xB7, 0xB3, 0x00, 0x00, 0xFB, 0xF6, /* 0xCC-0xCF */
	0x00, 0x00, 0x60, 0xDA, 0xB8, 0xAE, 0x00, 0x00, /* 0xD0-0xD3 */
	0x60, 0xDB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xED, 0xC3, /* 0xD8-0xDB */
	0x60, 0xDC, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xF0, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0xC5, 0xBE, 0xED, 0xC4, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x60, 0xDD, 0x00, 0x00, 0x00, 0x00, 0xED, 0xC7, /* 0xE8-0xEB */
	0x00, 0x00, 0x60, 0xE4, 0x60, 0xDE, 0x00, 0x00, /* 0xEC-0xEF */
	0x60, 0xDF, 0x60, 0xE0, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xE1, /* 0xF4-0xF7 */
	0xBC, 0xB4, 0x60, 0xE2, 0x60, 0xE3, 0xED, 0xC6, /* 0xF8-0xFB */
	0xED, 0xC5, 0xB7, 0xDA, 0xED, 0xC8, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_8F[512] = {
	0x60, 0xE5, 0x00, 0x00, 0x60, 0xE6, 0xB3, 0xD3, /* 0x00-0x03 */
	0x00, 0x00, 0xED, 0xCA, 0x00, 0x00, 0x60, 0xE7, /* 0x04-0x07 */
	0x60, 0xE8, 0xBA, 0xDC, 0xED, 0xC9, 0x00, 0x00, /* 0x08-0x0B */
	0xED, 0xD2, 0x00, 0x00, 0x00, 0x00, 0x60, 0xE9, /* 0x0C-0x0F */
	0x60, 0xEA, 0x00, 0x00, 0xED, 0xCC, 0xED, 0xCE, /* 0x10-0x13 */
	0xCA, 0xE5, 0xED, 0xCB, 0x60, 0xEB, 0x60, 0xEC, /* 0x14-0x17 */
	0x60, 0xED, 0xED, 0xCD, 0x00, 0x00, 0xED, 0xD1, /* 0x18-0x1B */
	0xED, 0xCF, 0xB5, 0xB1, 0x60, 0xEE, 0xED, 0xD0, /* 0x1C-0x1F */
	0x60, 0xEF, 0x60, 0xF0, 0x00, 0x00, 0x60, 0xF1, /* 0x20-0x23 */
	0x00, 0x00, 0x60, 0xF2, 0xED, 0xD3, 0x60, 0xF3, /* 0x24-0x27 */
	0x60, 0xF4, 0xC7, 0xDA, 0xCE, 0xD8, 0x00, 0x00, /* 0x28-0x2B */
	0x60, 0xF5, 0x60, 0xF6, 0x60, 0xF7, 0xBD, 0xB4, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xED, 0xD4, /* 0x30-0x33 */
	0x60, 0xF8, 0x60, 0xF9, 0x60, 0xFA, 0x60, 0xFB, /* 0x34-0x37 */
	0xCD, 0xA2, 0xED, 0xD6, 0x60, 0xFC, 0xED, 0xD5, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0xED, 0xD9, 0xCD, 0xC1, /* 0x3C-0x3F */
	0x60, 0xFD, 0x60, 0xFE, 0xED, 0xD8, 0x61, 0xA1, /* 0x40-0x43 */
	0xB3, 0xED, 0xED, 0xD7, 0xED, 0xDC, 0x61, 0xA2, /* 0x44-0x47 */
	0x00, 0x00, 0xED, 0xDB, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0xED, 0xDA, 0xC5, 0xB2, 0xED, 0xDD, 0x61, 0xA3, /* 0x4C-0x4F */
	0x00, 0x00, 0x61, 0xA4, 0x61, 0xA5, 0x61, 0xA6, /* 0x50-0x53 */
	0x61, 0xA7, 0x61, 0xA8, 0x00, 0x00, 0xED, 0xDE, /* 0x54-0x57 */
	0x61, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0xED, 0xDF, 0x61, 0xAA, 0x61, 0xAB, 0xB9, 0xEC, /* 0x5C-0x5F */
	0x00, 0x00, 0xB7, 0xA5, 0xED, 0xE0, 0xED, 0xE1, /* 0x60-0x63 */
	0xED, 0xE2, 0x61, 0xAC, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBF, 0xC9, /* 0x98-0x9B */
	0xED, 0xE3, 0x61, 0xAD, 0xBC, 0xAD, 0xED, 0xE4, /* 0x9C-0x9F */
	0x61, 0xAE, 0x61, 0xAF, 0x00, 0x00, 0xED, 0xE5, /* 0xA0-0xA3 */
	0x61, 0xB0, 0x61, 0xB1, 0x61, 0xB2, 0xD2, 0xA1, /* 0xA4-0xA7 */
	0xD1, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0xED, 0xE6, 0xE5, 0xF0, 0xED, 0xE7, /* 0xAC-0xAF */
	0xC3, 0xA4, 0xBF, 0xAB, 0xC7, 0xC0, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x61, 0xB3, 0x61, 0xB4, 0xED, 0xE8, /* 0xB4-0xB7 */
	0x61, 0xB5, 0x00, 0x00, 0xCA, 0xD5, 0xC4, 0xD4, /* 0xB8-0xBB */
	0xB9, 0xFE, 0x00, 0x00, 0x61, 0xB6, 0xC3, 0xA9, /* 0xBC-0xBF */
	0x61, 0xB7, 0x61, 0xB8, 0xB1, 0xAA, 0x00, 0x00, /* 0xC0-0xC3 */
	0xCB, 0xF8, 0xBF, 0xD7, 0x61, 0xB9, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x61, 0xBA, 0x61, 0xBB, /* 0xC8-0xCB */
	0x00, 0x00, 0x61, 0xBC, 0xB7, 0xDE, 0x00, 0x00, /* 0xCC-0xCF */
	0x61, 0xBD, 0xB6, 0xE1, 0x61, 0xBE, 0x61, 0xBF, /* 0xD0-0xD3 */
	0xCA, 0xD6, 0x61, 0xC0, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0xED, 0xE9, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x61, 0xC1, 0x00, 0x00, 0xED, 0xEB, 0x61, 0xC2, /* 0xE0-0xE3 */
	0x61, 0xC3, 0xED, 0xEA, 0xB2, 0xE0, 0x00, 0x00, /* 0xE4-0xE7 */
	0x61, 0xC4, 0xC6, 0xF6, 0xED, 0xEC, 0xC7, 0xF7, /* 0xE8-0xEB */
	0x00, 0x00, 0xC5, 0xB3, 0x61, 0xC5, 0xED, 0xED, /* 0xEC-0xEF */
	0xBD, 0xD2, 0x61, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0xED, 0xEF, 0x61, 0xC7, 0x61, 0xC8, 0xCC, 0xC2, /* 0xF4-0xF7 */
	0xED, 0xFE, 0xED, 0xF1, 0xED, 0xF2, 0x61, 0xC9, /* 0xF8-0xFB */
	0x00, 0x00, 0xC4, 0xC9, 0x61, 0xCA, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_90[512] = {
	0xC2, 0xE0, 0xC1, 0xF7, 0x61, 0xCB, 0xC6, 0xA8, /* 0x00-0x03 */
	0x61, 0xCC, 0xED, 0xF0, 0xB5, 0xD5, 0x00, 0x00, /* 0x04-0x07 */
	0x61, 0xCD, 0x00, 0x00, 0x00, 0x00, 0xED, 0xF9, /* 0x08-0x0B */
	0x61, 0xCE, 0xED, 0xF6, 0xEE, 0xA5, 0xC6, 0xA9, /* 0x0C-0x0F */
	0xC3, 0xE0, 0xED, 0xF3, 0x00, 0x00, 0xC4, 0xFE, /* 0x10-0x13 */
	0xC5, 0xD3, 0xED, 0xF4, 0xED, 0xF8, 0xBF, 0xE0, /* 0x14-0x17 */
	0x61, 0xCF, 0xC7, 0xE7, 0xC4, 0xCC, 0x61, 0xD0, /* 0x18-0x1B */
	0x00, 0x00, 0xC0, 0xC2, 0xED, 0xF7, 0xC2, 0xAE, /* 0x1C-0x1F */
	0xC2, 0xA4, 0xED, 0xF5, 0xB0, 0xA9, 0xCF, 0xA2, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xED, 0xFA, /* 0x24-0x27 */
	0x61, 0xD1, 0x61, 0xD2, 0x61, 0xD4, 0x00, 0x00, /* 0x28-0x2B */
	0x61, 0xD5, 0x61, 0xD6, 0xC2, 0xE1, 0x61, 0xD3, /* 0x2C-0x2F */
	0x00, 0x00, 0xBD, 0xB5, 0xBF, 0xCA, 0x61, 0xD7, /* 0x30-0x33 */
	0x61, 0xD8, 0xED, 0xFC, 0xED, 0xFB, 0x61, 0xD9, /* 0x34-0x37 */
	0xB0, 0xEF, 0xED, 0xFD, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0xC9, 0xAF, 0x00, 0x00, 0xEE, 0xA7, 0x61, 0xDA, /* 0x3C-0x3F */
	0x00, 0x00, 0xC6, 0xDB, 0xBF, 0xEB, 0x61, 0xDB, /* 0x40-0x43 */
	0x61, 0xDC, 0xC3, 0xD9, 0x00, 0x00, 0xB6, 0xF8, /* 0x44-0x47 */
	0x00, 0x00, 0xEE, 0xA6, 0xCD, 0xB7, 0xB1, 0xBF, /* 0x48-0x4B */
	0x61, 0xDD, 0xCA, 0xD7, 0xB2, 0xE1, 0xEE, 0xA1, /* 0x4C-0x4F */
	0xEE, 0xA2, 0xEE, 0xA3, 0xEE, 0xA4, 0xC6, 0xBB, /* 0x50-0x53 */
	0xC3, 0xA3, 0xB0, 0xE3, 0xEE, 0xA8, 0x00, 0x00, /* 0x54-0x57 */
	0xEE, 0xA9, 0xF4, 0xA3, 0x00, 0x00, 0x61, 0xDE, /* 0x58-0x5B */
	0xC2, 0xBD, 0x61, 0xDF, 0xEE, 0xAA, 0x00, 0x00, /* 0x5C-0x5F */
	0xB1, 0xF3, 0xC1, 0xCC, 0x61, 0xE0, 0xB8, 0xAF, /* 0x60-0x63 */
	0x00, 0x00, 0xCD, 0xDA, 0x61, 0xE1, 0xFB, 0xF9, /* 0x64-0x67 */
	0xEE, 0xAB, 0xC5, 0xAC, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x61, 0xE3, 0xC1, 0xF8, 0xBC, 0xD7, 0xEE, 0xAC, /* 0x6C-0x6F */
	0x61, 0xE4, 0x00, 0x00, 0xEE, 0xAF, 0x00, 0x00, /* 0x70-0x73 */
	0x61, 0xE5, 0xBD, 0xE5, 0xEE, 0xAD, 0xC1, 0xAB, /* 0x74-0x77 */
	0xC1, 0xAA, 0x61, 0xE6, 0xB0, 0xE4, 0x00, 0x00, /* 0x78-0x7B */
	0xCE, 0xCB, 0xEE, 0xB1, 0x00, 0x00, 0xC8, 0xF2, /* 0x7C-0x7F */

	0xEE, 0xB3, 0xEE, 0xB2, 0xEE, 0xB0, 0xE3, 0xE4, /* 0x80-0x83 */
	0xB4, 0xD4, 0x61, 0xE7, 0x00, 0x00, 0xED, 0xEE, /* 0x84-0x87 */
	0x61, 0xE8, 0xEE, 0xB5, 0xEE, 0xB4, 0x61, 0xE9, /* 0x88-0x8B */
	0x61, 0xEA, 0x00, 0x00, 0x61, 0xEB, 0xEE, 0xB6, /* 0x8C-0x8F */
	0x61, 0xEC, 0xCD, 0xB8, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x61, 0xED, 0x00, 0x00, 0x61, 0xEE, /* 0x94-0x97 */
	0x61, 0xEF, 0x61, 0xF0, 0x00, 0x00, 0x61, 0xF1, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x61, 0xF2, 0x61, 0xF3, 0x61, 0xF4, 0xC6, 0xE1, /* 0xA0-0xA3 */
	0x00, 0x00, 0x61, 0xF5, 0xCB, 0xAE, 0x00, 0x00, /* 0xA4-0xA7 */
	0xEE, 0xB7, 0x00, 0x00, 0xBC, 0xD9, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xB8, /* 0xAC-0xAF */
	0x61, 0xF6, 0xEE, 0xB9, 0x61, 0xF7, 0x61, 0xF8, /* 0xB0-0xB3 */
	0x61, 0xF9, 0xEE, 0xBA, 0x61, 0xFA, 0x00, 0x00, /* 0xB4-0xB7 */
	0xC5, 0xA1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x61, 0xFB, 0x61, 0xFD, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0xB0, 0xEA, 0x00, 0x00, 0x61, 0xFE, /* 0xC0-0xC3 */
	0x62, 0xA1, 0x62, 0xA2, 0x00, 0x00, 0x62, 0xA3, /* 0xC4-0xC7 */
	0x62, 0xA4, 0x00, 0x00, 0xB9, 0xD9, 0x00, 0x00, /* 0xC8-0xCB */
	0x61, 0xFC, 0x00, 0x00, 0xCF, 0xBA, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x62, 0xAD, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x62, 0xA5, 0x00, 0x00, 0x62, 0xA6, /* 0xD4-0xD7 */
	0x62, 0xA7, 0x62, 0xA8, 0x00, 0x00, 0xEE, 0xBE, /* 0xD8-0xDB */
	0x62, 0xA9, 0x62, 0xAA, 0xFB, 0xFA, 0x62, 0xAB, /* 0xDC-0xDF */
	0x00, 0x00, 0xB7, 0xB4, 0xEE, 0xBB, 0x00, 0x00, /* 0xE0-0xE3 */
	0xEE, 0xBC, 0x62, 0xAC, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0xC9, 0xF4, 0x00, 0x00, 0x00, 0x00, 0x62, 0xAF, /* 0xE8-0xEB */
	0x00, 0x00, 0xB3, 0xD4, 0x00, 0x00, 0x62, 0xB0, /* 0xEC-0xEF */
	0x62, 0xB1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x62, 0xB2, 0xCD, 0xB9, 0x62, 0xAE, 0xB6, 0xBF, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xC5, 0xD4, 0x62, 0xB3, 0x62, 0xB4, /* 0xFC-0xFF */
};

static unsigned char u2c_91[512] = {
	0x62, 0xB5, 0x00, 0x00, 0xEE, 0xBF, 0x00, 0x00, /* 0x00-0x03 */
	0x62, 0xB6, 0x62, 0xB7, 0x62, 0xB8, 0x00, 0x00, /* 0x04-0x07 */
	0x62, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x62, 0xBA, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x62, 0xBB, 0x00, 0x00, 0xEE, 0xC0, 0x00, 0x00, /* 0x10-0x13 */
	0x62, 0xBC, 0xFB, 0xFC, 0x62, 0xBD, 0x62, 0xBE, /* 0x14-0x17 */
	0x62, 0xBF, 0xEE, 0xC1, 0x62, 0xC0, 0x00, 0x00, /* 0x18-0x1B */
	0x62, 0xC1, 0x00, 0x00, 0x62, 0xC2, 0x00, 0x00, /* 0x1C-0x1F */
	0x62, 0xC3, 0x00, 0x00, 0x62, 0xC5, 0x62, 0xC6, /* 0x20-0x23 */
	0x00, 0x00, 0x62, 0xC4, 0x00, 0x00, 0xFB, 0xFD, /* 0x24-0x27 */
	0x00, 0x00, 0x62, 0xC8, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0xC5, 0xA2, 0x62, 0xC9, 0x62, 0xCA, /* 0x2C-0x2F */
	0xEE, 0xC3, 0x62, 0xCB, 0xEE, 0xC2, 0x00, 0x00, /* 0x30-0x33 */
	0x62, 0xCC, 0x00, 0x00, 0x62, 0xCD, 0x62, 0xCE, /* 0x34-0x37 */
	0x00, 0x00, 0x62, 0xCF, 0x62, 0xD0, 0x00, 0x00, /* 0x38-0x3B */
	0x62, 0xD1, 0x62, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62, 0xD3, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62, 0xD4, /* 0x44-0x47 */
	0x62, 0xD5, 0xC6, 0xD3, 0xEE, 0xC4, 0xBD, 0xB6, /* 0x48-0x4B */
	0xBC, 0xE0, 0xC7, 0xDB, 0xC3, 0xF1, 0x62, 0xD6, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0xBC, 0xF2, 0x62, 0xD7, /* 0x50-0x53 */
	0xBF, 0xEC, 0x00, 0x00, 0xEE, 0xC5, 0x62, 0xD8, /* 0x54-0x57 */
	0xEE, 0xC6, 0x62, 0xD9, 0x62, 0xDA, 0x62, 0xDB, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x62, 0xDC, 0xBF, 0xDD, 0xEE, 0xC7, /* 0x60-0x63 */
	0x62, 0xDD, 0xEE, 0xC8, 0x00, 0x00, 0x62, 0xDE, /* 0x64-0x67 */
	0x00, 0x00, 0xEE, 0xC9, 0xCD, 0xEF, 0x00, 0x00, /* 0x68-0x6B */
	0xBD, 0xB7, 0x62, 0xDF, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0xEE, 0xCB, 0xEE, 0xCA, /* 0x70-0x73 */
	0x62, 0xE0, 0xB9, 0xDA, 0x00, 0x00, 0xB9, 0xF3, /* 0x74-0x77 */
	0xBB, 0xC0, 0x62, 0xE1, 0x62, 0xE2, 0x62, 0xE3, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x62, 0xE4, 0xEE, 0xCE, 0x62, 0xE5, /* 0x80-0x83 */
	0x00, 0x00, 0x62, 0xE6, 0x62, 0xE7, 0xBD, 0xE6, /* 0x84-0x87 */
	0x00, 0x00, 0xEE, 0xCD, 0x62, 0xE8, 0xEE, 0xCC, /* 0x88-0x8B */
	0x00, 0x00, 0xC2, 0xE9, 0x62, 0xE9, 0x00, 0x00, /* 0x8C-0x8F */
	0xB8, 0xEF, 0x62, 0xEA, 0xC0, 0xC3, 0x62, 0xEB, /* 0x90-0x93 */
	0x62, 0xEC, 0x62, 0xED, 0x00, 0x00, 0xC8, 0xB0, /* 0x94-0x97 */
	0x62, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xBD, 0xB9, 0x00, 0x00, 0x62, 0xEF, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x62, 0xF0, 0xEE, 0xCF, 0x00, 0x00, /* 0xA0-0xA3 */
	0xBE, 0xDF, 0x00, 0x00, 0x62, 0xF1, 0x00, 0x00, /* 0xA4-0xA7 */
	0x62, 0xF2, 0x00, 0x00, 0xEE, 0xD2, 0xEE, 0xD0, /* 0xA8-0xAB */
	0x62, 0xF3, 0x62, 0xF4, 0x62, 0xF5, 0xEE, 0xD1, /* 0xAC-0xAF */
	0x62, 0xF6, 0x62, 0xF7, 0x62, 0xF8, 0x62, 0xF9, /* 0xB0-0xB3 */
	0xEE, 0xD4, 0xEE, 0xD3, 0x62, 0xFA, 0x00, 0x00, /* 0xB4-0xB7 */
	0xBE, 0xFA, 0x00, 0x00, 0xEE, 0xD5, 0x62, 0xFB, /* 0xB8-0xBB */
	0x62, 0xFC, 0x62, 0xFD, 0x00, 0x00, 0x62, 0xFE, /* 0xBC-0xBF */
	0xEE, 0xD6, 0xEE, 0xD7, 0x63, 0xA1, 0x63, 0xA2, /* 0xC0-0xC3 */
	0x00, 0x00, 0x63, 0xA3, 0xC8, 0xD0, 0xBA, 0xD3, /* 0xC4-0xC7 */
	0xBC, 0xE1, 0xEE, 0xD8, 0x00, 0x00, 0xEE, 0xD9, /* 0xC8-0xCB */
	0xCE, 0xA4, 0xBD, 0xC5, 0xCC, 0xEE, 0xCE, 0xCC, /* 0xCC-0xCF */
	0xEE, 0xDA, 0xB6, 0xE2, 0x00, 0x00, 0x63, 0xA4, /* 0xD0-0xD3 */
	0x63, 0xA5, 0x00, 0x00, 0xEE, 0xDB, 0xFC, 0xA1, /* 0xD4-0xD7 */
	0xC5, 0xA3, 0x63, 0xA7, 0xFB, 0xFE, 0xEE, 0xDE, /* 0xD8-0xDB */
	0xB3, 0xF8, 0xBF, 0xCB, 0xFC, 0xA2, 0xEE, 0xDC, /* 0xDC-0xDF */
	0x00, 0x00, 0xEE, 0xDD, 0x00, 0x00, 0xC4, 0xE0, /* 0xE0-0xE3 */
	0xFC, 0xA5, 0xFC, 0xA6, 0xCB, 0xD5, 0xB6, 0xFC, /* 0xE4-0xE7 */
	0x00, 0x00, 0x63, 0xAC, 0x63, 0xAD, 0x00, 0x00, /* 0xE8-0xEB */
	0x63, 0xAE, 0xFC, 0xA3, 0xFC, 0xA4, 0x63, 0xB1, /* 0xEC-0xEF */
	0x63, 0xB2, 0x63, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0xEE, 0xE0, 0xEE, 0xE1, 0x63, 0xB4, /* 0xF4-0xF7 */
	0x00, 0x00, 0x63, 0xB5, 0x00, 0x00, 0x63, 0xB6, /* 0xF8-0xFB */
	0xEE, 0xDF, 0x63, 0xB7, 0x00, 0x00, 0xEE, 0xE3, /* 0xFC-0xFF */
};

static unsigned char u2c_92[512] = {
	0x63, 0xB8, 0x63, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x63, 0xBA, 0x63, 0xBB, 0xFC, 0xA7, 0x63, 0xBD, /* 0x04-0x07 */
	0x00, 0x00, 0x63, 0xBE, 0xFC, 0xA9, 0x00, 0x00, /* 0x08-0x0B */
	0x63, 0xC0, 0xC6, 0xDF, 0xB3, 0xC3, 0x00, 0x00, /* 0x0C-0x0F */
	0xFC, 0xA8, 0xEE, 0xE7, 0x63, 0xC2, 0x63, 0xC3, /* 0x10-0x13 */
	0xEE, 0xE4, 0xEE, 0xE6, 0x63, 0xC4, 0x00, 0x00, /* 0x14-0x17 */
	0x63, 0xC5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x63, 0xC6, 0x63, 0xC7, 0xEE, 0xE2, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0xC8, /* 0x20-0x23 */
	0x63, 0xC9, 0x63, 0xCA, 0x63, 0xCB, 0x00, 0x00, /* 0x24-0x27 */
	0x63, 0xCC, 0xEF, 0xCF, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0xEE, 0xE5, 0x00, 0x00, 0x63, 0xCD, 0x63, 0xCE, /* 0x2C-0x2F */
	0x63, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x63, 0xD0, /* 0x30-0x33 */
	0xCE, 0xEB, 0x63, 0xD1, 0x63, 0xD2, 0xB8, 0xDA, /* 0x34-0x37 */
	0x63, 0xD3, 0xFC, 0xB0, 0xFC, 0xAA, 0x00, 0x00, /* 0x38-0x3B */
	0xFC, 0xAC, 0x00, 0x00, 0x63, 0xD7, 0xEE, 0xEF, /* 0x3C-0x3F */
	0xFC, 0xAB, 0x00, 0x00, 0x63, 0xD9, 0x63, 0xDA, /* 0x40-0x43 */
	0xC5, 0xB4, 0xEE, 0xEA, 0x63, 0xDB, 0x63, 0xDC, /* 0x44-0x47 */
	0xEE, 0xED, 0xEE, 0xEB, 0x63, 0xDD, 0xEE, 0xF0, /* 0x48-0x4B */
	0x00, 0x00, 0x63, 0xDE, 0xFC, 0xAD, 0x63, 0xE0, /* 0x4C-0x4F */
	0xEE, 0xF1, 0xFC, 0xAF, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xE9, /* 0x54-0x57 */
	0x63, 0xE2, 0xFC, 0xAE, 0xEE, 0xF6, 0xB1, 0xF4, /* 0x58-0x5B */
	0x63, 0xE4, 0x63, 0xE5, 0xEE, 0xE8, 0x00, 0x00, /* 0x5C-0x5F */
	0x63, 0xE6, 0x63, 0xE7, 0xC8, 0xAD, 0x00, 0x00, /* 0x60-0x63 */
	0xEE, 0xEC, 0x63, 0xE8, 0xBE, 0xE0, 0xFC, 0xB1, /* 0x64-0x67 */
	0x63, 0xEA, 0x63, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x63, 0xEC, 0x63, 0xED, /* 0x6C-0x6F */
	0x63, 0xEE, 0xB9, 0xDB, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x63, 0xEF, 0x63, 0xF0, 0xFC, 0xB3, /* 0x74-0x77 */
	0xFC, 0xB4, 0x63, 0xF3, 0x00, 0x00, 0x63, 0xF4, /* 0x78-0x7B */
	0x63, 0xF5, 0x63, 0xF6, 0xCB, 0xC8, 0x63, 0xF7, /* 0x7C-0x7F */

	0xB6, 0xE4, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xC6, /* 0x80-0x83 */
	0x00, 0x00, 0xC6, 0xBC, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0xF9, 0xA4, 0x63, 0xF9, 0x63, 0xFA, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x63, 0xFB, 0x63, 0xFC, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0xC1, 0xAD, 0x63, 0xFD, 0xEE, 0xF4, /* 0x90-0x93 */
	0x00, 0x00, 0xEE, 0xEE, 0xEE, 0xF3, 0x63, 0xFE, /* 0x94-0x97 */
	0xCC, 0xC3, 0x64, 0xA1, 0xC4, 0xB8, 0xEE, 0xF5, /* 0x98-0x9B */
	0xEE, 0xF2, 0x00, 0x00, 0x00, 0x00, 0x64, 0xA2, /* 0x9C-0x9F */
	0x64, 0xA3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x64, 0xA4, 0x64, 0xA5, 0x00, 0x00, 0xFC, 0xB2, /* 0xA4-0xA7 */
	0x64, 0xA7, 0x00, 0x00, 0x00, 0x00, 0x64, 0xA8, /* 0xA8-0xAB */
	0x00, 0x00, 0xC1, 0xAC, 0x00, 0x00, 0x64, 0xA9, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x64, 0xAA, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x64, 0xAB, 0xEE, 0xF9, /* 0xB4-0xB7 */
	0x64, 0xAC, 0xEE, 0xF8, 0x64, 0xAD, 0x64, 0xAE, /* 0xB8-0xBB */
	0x64, 0xAF, 0x64, 0xB0, 0x00, 0x00, 0x64, 0xB1, /* 0xBC-0xBF */
	0x64, 0xB2, 0x64, 0xB3, 0x64, 0xB4, 0x64, 0xB5, /* 0xC0-0xC3 */
	0x00, 0x00, 0x64, 0xB6, 0x64, 0xB7, 0x64, 0xB8, /* 0xC4-0xC7 */
	0x64, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x64, 0xBA, /* 0xC8-0xCB */
	0x64, 0xBB, 0x64, 0xBC, 0x64, 0xBD, 0xEE, 0xF7, /* 0xCC-0xCF */
	0xFC, 0xB8, 0x00, 0x00, 0xCB, 0xAF, 0xFC, 0xBC, /* 0xD0-0xD3 */
	0x00, 0x00, 0xFC, 0xBA, 0x00, 0x00, 0xFC, 0xB6, /* 0xD4-0xD7 */
	0x64, 0xC2, 0xFC, 0xB7, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x64, 0xC4, 0x64, 0xC5, 0x00, 0x00, 0x64, 0xC6, /* 0xDC-0xDF */
	0xFC, 0xBB, 0x64, 0xC8, 0x00, 0x00, 0x64, 0xC9, /* 0xE0-0xE3 */
	0xBD, 0xFB, 0x64, 0xCA, 0x00, 0x00, 0xFC, 0xB5, /* 0xE4-0xE7 */
	0x64, 0xCC, 0xEE, 0xFA, 0xCA, 0xDF, 0x00, 0x00, /* 0xE8-0xEB */
	0x64, 0xCD, 0xB1, 0xD4, 0x64, 0xCE, 0x00, 0x00, /* 0xEC-0xEF */
	0x64, 0xCF, 0x00, 0x00, 0xC9, 0xC6, 0xC3, 0xF2, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0xB5, 0xF8, 0xF9, 0xAA, 0xEE, 0xFC, 0xFC, 0xBF, /* 0xF8-0xFB */
	0xB9, 0xDD, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xC2, /* 0xFC-0xFF */
};

static unsigned char u2c_93[512] = {
	0x64, 0xD3, 0x00, 0x00, 0xFC, 0xC4, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xBB, 0xAC, 0x00, 0x00, /* 0x04-0x07 */
	0x64, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x64, 0xD6, 0x00, 0x00, 0xEE, 0xFB, /* 0x0C-0x0F */
	0xBF, 0xED, 0x64, 0xD7, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x64, 0xD8, 0x64, 0xD9, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0xBF, 0xEE, 0xEF, 0xA1, 0xEF, 0xA3, 0x00, 0x00, /* 0x18-0x1B */
	0x64, 0xDA, 0xFC, 0xC3, 0xFC, 0xC1, 0x64, 0xDD, /* 0x1C-0x1F */
	0xBE, 0xFB, 0xFC, 0xBE, 0xEF, 0xA2, 0xEF, 0xA4, /* 0x20-0x23 */
	0x64, 0xDF, 0xFC, 0xBD, 0xB6, 0xD3, 0x64, 0xE1, /* 0x24-0x27 */
	0xC9, 0xC5, 0x64, 0xE2, 0x64, 0xE3, 0xBC, 0xE2, /* 0x28-0x2B */
	0xCF, 0xA3, 0x00, 0x00, 0xEE, 0xFE, 0xBA, 0xF8, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0xCF, 0xBF, 0x64, 0xE4, /* 0x30-0x33 */
	0x64, 0xE5, 0xEF, 0xA6, 0x64, 0xE6, 0x64, 0xE7, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0xEF, 0xA5, 0xEF, 0xA7, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0xEE, 0xFD, 0x00, 0x00, 0x00, 0x00, 0x64, 0xE8, /* 0x44-0x47 */
	0xF9, 0xA3, 0x64, 0xEA, 0x00, 0x00, 0xC6, 0xE9, /* 0x48-0x4B */
	0x00, 0x00, 0xC5, 0xD5, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x64, 0xEB, 0x64, 0xEC, 0x64, 0xED, 0x00, 0x00, /* 0x50-0x53 */
	0xC4, 0xD7, 0x64, 0xEE, 0xEF, 0xAC, 0xFC, 0xC6, /* 0x54-0x57 */
	0x64, 0xF0, 0x00, 0x00, 0x64, 0xF1, 0xC3, 0xC3, /* 0x58-0x5B */
	0xEF, 0xA8, 0x00, 0x00, 0x64, 0xF2, 0x00, 0x00, /* 0x5C-0x5F */
	0xEF, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x64, 0xF3, 0x64, 0xF4, 0x00, 0x00, 0x64, 0xF5, /* 0x64-0x67 */
	0x00, 0x00, 0x64, 0xF6, 0x64, 0xF7, 0x00, 0x00, /* 0x68-0x6B */
	0xB7, 0xAD, 0x64, 0xF8, 0xEF, 0xAB, 0x64, 0xF9, /* 0x6C-0x6F */
	0xFC, 0xC5, 0x64, 0xFB, 0x00, 0x00, 0x64, 0xFC, /* 0x70-0x73 */
	0x64, 0xFD, 0xB8, 0xB0, 0x64, 0xFE, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x65, 0xA1, 0x00, 0x00, /* 0x78-0x7B */
	0xEF, 0xAA, 0x65, 0xA2, 0xBE, 0xE1, 0x65, 0xA3, /* 0x7C-0x7F */

	0x65, 0xA4, 0x65, 0xA5, 0x65, 0xA6, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x65, 0xA7, 0x00, 0x00, 0x65, 0xA8, 0x65, 0xA9, /* 0x88-0x8B */
	0xB3, 0xF9, 0x65, 0xAA, 0x00, 0x00, 0x65, 0xAB, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x65, 0xAC, 0x00, 0x00, /* 0x90-0x93 */
	0xEF, 0xB0, 0x65, 0xAD, 0xBA, 0xBF, 0xC1, 0xF9, /* 0x94-0x97 */
	0x65, 0xAE, 0x00, 0x00, 0xC4, 0xCA, 0x65, 0xAF, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x65, 0xB0, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x65, 0xB1, 0x00, 0x00, 0x65, 0xB2, /* 0xA0-0xA3 */
	0xFC, 0xC7, 0x00, 0x00, 0x65, 0xB4, 0xB3, 0xBB, /* 0xA4-0xA7 */
	0x65, 0xB5, 0x65, 0xBB, 0x00, 0x00, 0x65, 0xB6, /* 0xA8-0xAB */
	0xEF, 0xAE, 0xEF, 0xAF, 0xC4, 0xC3, 0x00, 0x00, /* 0xAC-0xAF */
	0xEF, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x65, 0xB7, 0x65, 0xB8, 0x65, 0xB9, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xEF, 0xB1, 0x65, 0xBA, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x65, 0xBC, 0x00, 0x00, 0xEF, 0xB7, /* 0xC0-0xC3 */
	0x65, 0xBD, 0x65, 0xBE, 0xFC, 0xC8, 0x65, 0xC0, /* 0xC4-0xC7 */
	0xEF, 0xBA, 0x65, 0xC1, 0x65, 0xC2, 0x65, 0xC3, /* 0xC8-0xCB */
	0x65, 0xC4, 0x65, 0xC5, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0xEF, 0xB9, 0xC5, 0xAD, 0x00, 0x00, 0x65, 0xC6, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0xEF, 0xB2, 0xEF, 0xB3, /* 0xD4-0xD7 */
	0xEF, 0xB6, 0x65, 0xC7, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x65, 0xC8, 0xEF, 0xB8, 0xFC, 0xC9, 0x65, 0xCA, /* 0xDC-0xDF */
	0x00, 0x00, 0xB6, 0xC0, 0x65, 0xCB, 0x00, 0x00, /* 0xE0-0xE3 */
	0xEF, 0xBB, 0xEF, 0xB5, 0x65, 0xCC, 0x65, 0xCD, /* 0xE4-0xE7 */
	0xEF, 0xB4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65, 0xCF, /* 0xF4-0xF7 */
	0xFC, 0xCA, 0x65, 0xCE, 0x65, 0xD1, 0x65, 0xD2, /* 0xF8-0xFB */
	0x00, 0x00, 0x65, 0xD3, 0x00, 0x00, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_94[512] = {
	0x00, 0x00, 0x65, 0xD4, 0x65, 0xD5, 0xEF, 0xBF, /* 0x00-0x03 */
	0x65, 0xD6, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xC0, /* 0x04-0x07 */
	0x65, 0xD7, 0x65, 0xD8, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x65, 0xD9, 0x65, 0xDA, 0x65, 0xDB, /* 0x0C-0x0F */
	0xEF, 0xC1, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xBE, /* 0x10-0x13 */
	0xEF, 0xBD, 0x65, 0xDC, 0x65, 0xDD, 0x65, 0xDE, /* 0x14-0x17 */
	0xBE, 0xE2, 0xC6, 0xAA, 0xEF, 0xBC, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65, 0xDF, /* 0x1C-0x1F */
	0x00, 0x00, 0xEF, 0xC5, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xC3, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x65, 0xE0, 0x65, 0xE1, /* 0x2C-0x2F */
	0x00, 0x00, 0xFC, 0xCB, 0x65, 0xE3, 0x65, 0xE4, /* 0x30-0x33 */
	0x65, 0xE5, 0xEF, 0xC4, 0xEF, 0xC2, 0x00, 0x00, /* 0x34-0x37 */
	0xC2, 0xF8, 0x00, 0x00, 0xEF, 0xC6, 0x65, 0xE6, /* 0x38-0x3B */
	0x00, 0x00, 0x65, 0xE8, 0x00, 0x00, 0x65, 0xE7, /* 0x3C-0x3F */
	0x00, 0x00, 0xEF, 0xC7, 0x00, 0x00, 0x65, 0xE9, /* 0x40-0x43 */
	0xEF, 0xC9, 0xFC, 0xCC, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0xFC, 0xCD, 0x00, 0x00, 0x65, 0xEC, 0x00, 0x00, /* 0x48-0x4B */
	0x65, 0xED, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0xB4, 0xD5, 0xEF, 0xC8, 0xCC, 0xFA, /* 0x50-0x53 */
	0x00, 0x00, 0x65, 0xEE, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x65, 0xEF, 0xEF, 0xD4, 0xEF, 0xCA, /* 0x58-0x5B */
	0x65, 0xF0, 0x00, 0x00, 0xEF, 0xCD, 0x65, 0xF1, /* 0x5C-0x5F */
	0xEF, 0xCB, 0x65, 0xF2, 0xEF, 0xCC, 0x65, 0xF3, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x65, 0xF4, 0x00, 0x00, 0xEF, 0xCE, 0x65, 0xF5, /* 0x68-0x6B */
	0x00, 0x00, 0x65, 0xF6, 0x65, 0xF7, 0x65, 0xF8, /* 0x6C-0x6F */
	0xEF, 0xD0, 0x65, 0xF9, 0x65, 0xFA, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xEF, 0xD1, 0x00, 0x00, 0xEF, 0xD2, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0xEF, 0xD5, 0xEF, 0xD3, 0xEF, 0xD6, 0xEF, 0xD8, /* 0x7C-0x7F */

	0x00, 0x00, 0xEF, 0xD7, 0x00, 0x00, 0x65, 0xFC, /* 0x80-0x83 */
	0x65, 0xFB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
};

static unsigned char u2c_95[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC4, 0xB9, /* 0x74-0x77 */
	0x65, 0xFD, 0x65, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x66, 0xA1, 0x00, 0x00, /* 0x7C-0x7F */

	0xCC, 0xE7, 0x00, 0x00, 0xEF, 0xD9, 0xC1, 0xAE, /* 0x80-0x83 */
	0x66, 0xA2, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xDA, /* 0x84-0x87 */
	0x66, 0xA3, 0xCA, 0xC4, 0xEF, 0xDB, 0xB3, 0xAB, /* 0x88-0x8B */
	0x66, 0xA4, 0x66, 0xA5, 0x66, 0xA6, 0xB1, 0xBC, /* 0x8C-0x8F */
	0x00, 0x00, 0xB4, 0xD7, 0xFC, 0xCE, 0xB4, 0xD6, /* 0x90-0x93 */
	0xEF, 0xDC, 0x00, 0x00, 0xEF, 0xDD, 0x00, 0x00, /* 0x94-0x97 */
	0xEF, 0xDE, 0xEF, 0xDF, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x66, 0xA7, 0x66, 0xA8, 0x66, 0xA9, /* 0x9C-0x9F */
	0xEF, 0xE0, 0x66, 0xAA, 0xB4, 0xD8, 0xB3, 0xD5, /* 0xA0-0xA3 */
	0xB9, 0xDE, 0xC8, 0xB6, 0x66, 0xAB, 0xEF, 0xE2, /* 0xA4-0xA7 */
	0xEF, 0xE1, 0x66, 0xAC, 0x00, 0x00, 0x66, 0xAD, /* 0xA8-0xAB */
	0x66, 0xAE, 0xEF, 0xE3, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0xB1, 0xDC, 0x00, 0x00, /* 0xB0-0xB3 */
	0x66, 0xAF, 0x00, 0x00, 0x66, 0xB0, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0xEF, 0xE6, 0x66, 0xB1, 0xEF, 0xE5, /* 0xB8-0xBB */
	0xEF, 0xE4, 0x66, 0xB2, 0xEF, 0xE7, 0x66, 0xB3, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xEA, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x66, 0xB4, 0xB0, 0xC7, /* 0xC4-0xC7 */
	0x66, 0xB5, 0x66, 0xB6, 0xEF, 0xE8, 0x66, 0xB7, /* 0xC8-0xCB */
	0xEF, 0xEC, 0xEF, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x66, 0xB8, 0x66, 0xB9, 0x66, 0xBA, 0x66, 0xBB, /* 0xD0-0xD3 */
	0xEF, 0xEE, 0xEF, 0xED, 0xEF, 0xEF, 0x00, 0x00, /* 0xD4-0xD7 */
	0xC6, 0xAE, 0x66, 0xBC, 0x66, 0xBD, 0x00, 0x00, /* 0xD8-0xDB */
	0xEF, 0xF0, 0x66, 0xBE, 0x66, 0xBF, 0x66, 0xC0, /* 0xDC-0xDF */
	0x66, 0xC1, 0xEF, 0xF1, 0xEF, 0xF3, 0x00, 0x00, /* 0xE0-0xE3 */
	0x66, 0xC2, 0xEF, 0xF2, 0x66, 0xC3, 0x00, 0x00, /* 0xE4-0xE7 */
};

static unsigned char u2c_96[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0xC9, 0xEC, 0x66, 0xC4, 0x66, 0xC5, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0xEF, 0xF4, 0x66, 0xC6, 0x00, 0x00, /* 0x20-0x23 */
	0x66, 0xC7, 0x66, 0xC8, 0x66, 0xC9, 0x00, 0x00, /* 0x24-0x27 */
	0xEF, 0xF5, 0x00, 0x00, 0xBA, 0xE5, 0x00, 0x00, /* 0x28-0x2B */
	0x66, 0xCA, 0x00, 0x00, 0xEF, 0xF6, 0xEF, 0xF7, /* 0x2C-0x2F */
	0x00, 0x00, 0x66, 0xCB, 0xCB, 0xC9, 0x66, 0xCC, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0xCD, /* 0x34-0x37 */
	0x66, 0xCE, 0x66, 0xCF, 0x66, 0xD0, 0xC1, 0xCB, /* 0x38-0x3B */
	0x66, 0xD1, 0x66, 0xD2, 0x00, 0x00, 0xB0, 0xA4, /* 0x3C-0x3F */
	0xC2, 0xCB, 0x66, 0xD3, 0xEF, 0xF8, 0x00, 0x00, /* 0x40-0x43 */
	0xC9, 0xED, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEF, 0xFB, /* 0x48-0x4B */
	0xEF, 0xF9, 0xB9, 0xDF, 0x00, 0x00, 0xEF, 0xFA, /* 0x4C-0x4F */
	0xB8, 0xC2, 0x00, 0x00, 0x66, 0xD4, 0x00, 0x00, /* 0x50-0x53 */
	0x66, 0xD5, 0x00, 0x00, 0x66, 0xD6, 0x66, 0xD7, /* 0x54-0x57 */
	0x66, 0xD8, 0x00, 0x00, 0x00, 0x00, 0xCA, 0xC5, /* 0x58-0x5B */
	0xEF, 0xFD, 0xF0, 0xA1, 0xEF, 0xFE, 0xF0, 0xA2, /* 0x5C-0x5F */
	0x00, 0x00, 0x66, 0xD9, 0xB1, 0xA1, 0xBF, 0xD8, /* 0x60-0x63 */
	0xBD, 0xFC, 0xB4, 0xD9, 0xF0, 0xA3, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0xC7, 0xE6, 0x00, 0x00, /* 0x68-0x6B */
	0xF0, 0xA5, 0x00, 0x00, 0x66, 0xDA, 0x00, 0x00, /* 0x6C-0x6F */
	0xB1, 0xA2, 0x00, 0x00, 0xF0, 0xA4, 0xC4, 0xC4, /* 0x70-0x73 */
	0x66, 0xDB, 0xCE, 0xCD, 0xC6, 0xAB, 0xEF, 0xFC, /* 0x74-0x77 */
	0xCE, 0xA6, 0x00, 0x00, 0xB8, 0xB1, 0x66, 0xDC, /* 0x78-0x7B */
	0x66, 0xDD, 0xCD, 0xDB, 0x66, 0xDE, 0x66, 0xDF, /* 0x7C-0x7F */

	0x00, 0x00, 0x66, 0xE0, 0x66, 0xE1, 0x66, 0xE2, /* 0x80-0x83 */
	0x66, 0xE3, 0xB6, 0xF9, 0xCE, 0xB4, 0x00, 0x00, /* 0x84-0x87 */
	0xB7, 0xA8, 0x66, 0xE4, 0xC2, 0xE2, 0xE7, 0xA1, /* 0x88-0x8B */
	0x00, 0x00, 0xF0, 0xA6, 0xB3, 0xAC, 0xBF, 0xEF, /* 0x8C-0x8F */
	0x00, 0x00, 0x66, 0xE5, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0xB3, 0xD6, 0xF0, 0xA8, 0x66, 0xE6, 0xF0, 0xA9, /* 0x94-0x97 */
	0xF0, 0xA7, 0xB7, 0xE4, 0x66, 0xE7, 0xBA, 0xDD, /* 0x98-0x9B */
	0xBE, 0xE3, 0xFC, 0xD1, 0x00, 0x00, 0x66, 0xE9, /* 0x9C-0x9F */
	0xB1, 0xA3, 0x00, 0x00, 0x00, 0x00, 0xCE, 0xD9, /* 0xA0-0xA3 */
	0x66, 0xEA, 0x66, 0xEB, 0x66, 0xEC, 0xF0, 0xAB, /* 0xA4-0xA7 */
	0xEE, 0xAE, 0x66, 0xED, 0xF0, 0xAA, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x66, 0xEE, 0xFC, 0xD2, /* 0xAC-0xAF */
	0xF0, 0xAE, 0xF0, 0xAC, 0xF0, 0xAD, 0x66, 0xF0, /* 0xB0-0xB3 */
	0xF0, 0xAF, 0x00, 0x00, 0xF0, 0xB0, 0xCE, 0xEC, /* 0xB4-0xB7 */
	0xF0, 0xB1, 0xF0, 0xB2, 0x66, 0xF1, 0xC0, 0xC9, /* 0xB8-0xBB */
	0xC8, 0xBB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0xBF, 0xFD, 0xB4, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0xCD, 0xBA, 0xB2, 0xED, 0xBD, 0xB8, 0xB8, 0xDB, /* 0xC4-0xC7 */
	0x00, 0x00, 0xF0, 0xB5, 0x66, 0xF2, 0xF0, 0xB4, /* 0xC8-0xCB */
	0xBB, 0xF3, 0xF0, 0xB6, 0xF0, 0xB3, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0xBB, 0xA8, 0x66, 0xF3, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0xF0, 0xBA, 0xEA, 0xAD, 0x00, 0x00, /* 0xD4-0xD7 */
	0x66, 0xF5, 0xD2, 0xD6, 0x66, 0xF6, 0xBF, 0xF7, /* 0xD8-0xDB */
	0xF0, 0xB8, 0x66, 0xF7, 0x66, 0xF8, 0x66, 0xF9, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0xCE, 0xA5, 0xC6, 0xF1, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0xB1, 0xAB, 0x66, 0xFA, 0xC0, 0xE3, 0xBC, 0xB6, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0xFB, /* 0xEC-0xEF */
	0xCA, 0xB7, 0x66, 0xFC, 0xB1, 0xC0, 0x00, 0x00, /* 0xF0-0xF3 */
	0x00, 0x00, 0x00, 0x00, 0xCE, 0xED, 0xCD, 0xEB, /* 0xF4-0xF7 */
	0x00, 0x00, 0xF0, 0xBB, 0x66, 0xFD, 0xC5, 0xC5, /* 0xF8-0xFB */
};

static unsigned char u2c_97[512] = {
	0xBC, 0xFB, 0x00, 0x00, 0x66, 0xFE, 0x67, 0xA1, /* 0x00-0x03 */
	0xF0, 0xBC, 0x67, 0xA2, 0xF0, 0xBD, 0xBF, 0xCC, /* 0x04-0x07 */
	0xF0, 0xBE, 0x67, 0xA3, 0xCE, 0xEE, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0xF0, 0xB9, 0xF0, 0xC0, 0xF0, 0xC2, /* 0x0C-0x0F */
	0x00, 0x00, 0xF0, 0xC1, 0x00, 0x00, 0xF0, 0xBF, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xC3, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xF0, 0xC4, 0x67, 0xA4, 0x67, 0xA5, /* 0x18-0x1B */
	0xC1, 0xFA, 0x67, 0xA6, 0xB2, 0xE2, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x67, 0xA7, 0x67, 0xA8, 0x67, 0xA9, /* 0x20-0x23 */
	0xF0, 0xC5, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xB8, /* 0x24-0x27 */
	0x67, 0xAA, 0x00, 0x00, 0xF0, 0xC6, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0xF0, 0xC7, 0x67, 0xAB, 0xCF, 0xAA, 0xFC, 0xD3, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0xDB, 0xB1, 0xF0, 0xC8, 0x00, 0x00, 0xFC, 0xD4, /* 0x38-0x3B */
	0x00, 0x00, 0xF0, 0xC9, 0xF0, 0xCA, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x67, 0xAD, 0xF0, 0xCE, 0xFC, 0xD5, /* 0x40-0x43 */
	0xF0, 0xCB, 0x00, 0x00, 0xF0, 0xCC, 0x00, 0x00, /* 0x44-0x47 */
	0xF0, 0xCD, 0xF0, 0xCF, 0x67, 0xAF, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0xFC, 0xD6, 0x67, 0xB0, 0xFC, 0xD7, /* 0x4C-0x4F */
	0x00, 0x00, 0xFC, 0xD8, 0xC0, 0xC4, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0xFC, 0xD9, 0xCC, 0xF7, 0x67, 0xB3, /* 0x54-0x57 */
	0x67, 0xB4, 0xC0, 0xC5, 0x67, 0xB5, 0x67, 0xB6, /* 0x58-0x5B */
	0xF0, 0xD0, 0x00, 0x00, 0xC8, 0xF3, 0x00, 0x00, /* 0x5C-0x5F */
	0xF0, 0xD1, 0xF3, 0xD3, 0xCC, 0xCC, 0x67, 0xB7, /* 0x60-0x63 */
	0xF0, 0xD2, 0x00, 0x00, 0xF0, 0xD3, 0x67, 0xB8, /* 0x64-0x67 */
	0xF0, 0xD4, 0xB3, 0xD7, 0x67, 0xB9, 0xF0, 0xD6, /* 0x68-0x6B */
	0x00, 0x00, 0xBF, 0xD9, 0x67, 0xBA, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0xF0, 0xD7, 0x00, 0x00, 0x67, 0xBB, /* 0x70-0x73 */
	0xB7, 0xA4, 0x00, 0x00, 0x67, 0xBC, 0x67, 0xBD, /* 0x74-0x77 */
	0x67, 0xBE, 0xF0, 0xD8, 0xF0, 0xDC, 0x67, 0xBF, /* 0x78-0x7B */
	0xF0, 0xDA, 0x67, 0xC0, 0x00, 0x00, 0x67, 0xC1, /* 0x7C-0x7F */

	0x67, 0xC2, 0xF0, 0xDB, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0xB3, 0xF3, 0xF0, 0xD9, 0xF0, 0xDD, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x67, 0xC3, 0x00, 0x00, 0xF0, 0xDE, /* 0x88-0x8B */
	0x00, 0x00, 0xB0, 0xC8, 0x00, 0x00, 0xF0, 0xDF, /* 0x8C-0x8F */
	0xF0, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x67, 0xC4, 0x67, 0xC5, 0x67, 0xC6, /* 0x94-0x97 */
	0xBE, 0xE4, 0x67, 0xC7, 0x67, 0xC8, 0x00, 0x00, /* 0x98-0x9B */
	0xF0, 0xE1, 0x00, 0x00, 0x67, 0xC9, 0x67, 0xCA, /* 0x9C-0x9F */
	0xB5, 0xC7, 0x00, 0x00, 0x67, 0xCB, 0xF0, 0xE4, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0xF0, 0xE3, 0x00, 0x00, /* 0xA4-0xA7 */
	0xF0, 0xE2, 0x00, 0x00, 0x00, 0x00, 0xEB, 0xF1, /* 0xA8-0xAB */
	0x67, 0xCC, 0xCA, 0xDC, 0x67, 0xCD, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x67, 0xCE, 0x67, 0xCF, 0xF0, 0xE5, /* 0xB0-0xB3 */
	0xF0, 0xE6, 0x67, 0xD0, 0x67, 0xD1, 0x00, 0x00, /* 0xB4-0xB7 */
	0x67, 0xD2, 0x67, 0xD3, 0x67, 0xD4, 0x00, 0x00, /* 0xB8-0xBB */
	0x67, 0xD5, 0x00, 0x00, 0x67, 0xD6, 0x67, 0xD7, /* 0xBC-0xBF */
	0x00, 0x00, 0x67, 0xD8, 0x00, 0x00, 0xF0, 0xE7, /* 0xC0-0xC3 */
	0x67, 0xD9, 0x67, 0xDA, 0xF0, 0xE8, 0x67, 0xDB, /* 0xC4-0xC7 */
	0xF0, 0xE9, 0x67, 0xDC, 0x67, 0xDD, 0xF0, 0xEA, /* 0xC8-0xCB */
	0x67, 0xDE, 0x67, 0xDF, 0x67, 0xE0, 0x00, 0x00, /* 0xCC-0xCF */
	0x67, 0xE1, 0x67, 0xE2, 0x00, 0x00, 0xB4, 0xDA, /* 0xD0-0xD3 */
	0x67, 0xE3, 0x00, 0x00, 0x00, 0x00, 0x67, 0xE4, /* 0xD4-0xD7 */
	0x67, 0xE5, 0x67, 0xE6, 0x00, 0x00, 0x67, 0xEA, /* 0xD8-0xDB */
	0xF0, 0xEB, 0x67, 0xE7, 0x67, 0xE8, 0x00, 0x00, /* 0xDC-0xDF */
	0x67, 0xE9, 0x67, 0xEB, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x67, 0xEC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x00, 0x00, 0xF0, 0xEC, 0xC7, 0xA3, 0x67, 0xED, /* 0xEC-0xEF */
	0x00, 0x00, 0x67, 0xEE, 0xF0, 0xEE, 0xB2, 0xBB, /* 0xF0-0xF3 */
	0x67, 0xEF, 0xF0, 0xF1, 0xF0, 0xF0, 0x67, 0xF0, /* 0xF4-0xF7 */
	0x67, 0xF1, 0x00, 0x00, 0x67, 0xF2, 0xB1, 0xA4, /* 0xF8-0xFB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB6, 0xC1, /* 0xFC-0xFF */
};

static unsigned char u2c_98[512] = {
	0x00, 0x00, 0xCA, 0xC7, 0xC4, 0xBA, 0xBA, 0xA2, /* 0x00-0x03 */
	0x00, 0x00, 0xB9, 0xE0, 0xBD, 0xE7, 0x67, 0xF3, /* 0x04-0x07 */
	0xBF, 0xDC, 0x00, 0x00, 0x67, 0xF4, 0x00, 0x00, /* 0x08-0x0B */
	0xF0, 0xF3, 0x67, 0xF6, 0x67, 0xF7, 0xF0, 0xF2, /* 0x0C-0x0F */
	0xCD, 0xC2, 0xB4, 0xE8, 0xC8, 0xD2, 0xC6, 0xDC, /* 0x10-0x13 */
	0x67, 0xF8, 0x00, 0x00, 0x67, 0xF9, 0xBF, 0xFC, /* 0x14-0x17 */
	0xCE, 0xCE, 0x67, 0xF5, 0xB7, 0xDB, 0x00, 0x00, /* 0x18-0x1B */
	0x67, 0xFA, 0x00, 0x00, 0x67, 0xFB, 0x00, 0x00, /* 0x1C-0x1F */
	0x67, 0xFC, 0xF0, 0xF6, 0x00, 0x00, 0x67, 0xFD, /* 0x20-0x23 */
	0xF0, 0xF5, 0x68, 0xA8, 0x67, 0xFE, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0xA1, /* 0x28-0x2B */
	0xCB, 0xCB, 0xC6, 0xAC, 0x68, 0xA2, 0x68, 0xA3, /* 0x2C-0x2F */
	0x68, 0xA4, 0x00, 0x00, 0x68, 0xA5, 0x68, 0xA6, /* 0x30-0x33 */
	0xB1, 0xD0, 0x68, 0xA7, 0x00, 0x00, 0xF0, 0xF7, /* 0x34-0x37 */
	0xF0, 0xF4, 0x00, 0x00, 0x00, 0x00, 0xC9, 0xD1, /* 0x38-0x3B */
	0xCD, 0xEA, 0xF0, 0xF8, 0x68, 0xA9, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x68, 0xAA, 0x00, 0x00, 0xF0, 0xF9, 0x68, 0xAB, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x68, 0xAC, 0xF0, 0xFB, /* 0x48-0x4B */
	0xC2, 0xEA, 0xB3, 0xDB, 0xB3, 0xDC, 0xF0, 0xFA, /* 0x4C-0x4F */
	0x00, 0x00, 0x68, 0xAD, 0x68, 0xAE, 0x68, 0xAF, /* 0x50-0x53 */
	0xB4, 0xE9, 0xB8, 0xB2, 0x68, 0xB0, 0xFC, 0xDA, /* 0x54-0x57 */
	0xB4, 0xEA, 0x68, 0xB2, 0x68, 0xB3, 0xC5, 0xBF, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0xCE, 0xE0, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x68, 0xB4, 0x68, 0xB5, /* 0x60-0x63 */
	0x00, 0x00, 0xFC, 0xDB, 0x68, 0xB7, 0xB8, 0xDC, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x68, 0xB8, 0xF0, 0xFC, /* 0x68-0x6B */
	0x68, 0xB9, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFD, /* 0x6C-0x6F */
	0xF0, 0xFE, 0xF1, 0xA1, 0x00, 0x00, 0xF1, 0xA3, /* 0x70-0x73 */
	0xF1, 0xA2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0xC9, 0xF7, 0x00, 0x00, 0xF1, 0xA4, 0x68, 0xBA, /* 0xA8-0xAB */
	0x00, 0x00, 0x68, 0xBB, 0x68, 0xBC, 0xF1, 0xA5, /* 0xAC-0xAF */
	0x68, 0xBD, 0xF1, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x68, 0xBE, 0x00, 0x00, 0xF1, 0xA7, 0x68, 0xBF, /* 0xB4-0xB7 */
	0x68, 0xC0, 0x00, 0x00, 0x68, 0xC1, 0x68, 0xC2, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0xC3, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x68, 0xC4, 0xF1, 0xA9, /* 0xC0-0xC3 */
	0xF1, 0xA8, 0x68, 0xC5, 0xF1, 0xAA, 0x00, 0x00, /* 0xC4-0xC7 */
	0x68, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x68, 0xC7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC8, 0xF4, /* 0xD8-0xDB */
	0xE6, 0xCC, 0x00, 0x00, 0x00, 0x00, 0xBF, 0xA9, /* 0xDC-0xDF */
	0x00, 0x00, 0x68, 0xC8, 0xB5, 0xB2, 0x68, 0xC9, /* 0xE0-0xE3 */
	0x00, 0x00, 0x68, 0xCA, 0x68, 0xCB, 0x68, 0xCC, /* 0xE4-0xE7 */
	0x00, 0x00, 0xF1, 0xAB, 0x68, 0xCD, 0xF1, 0xAC, /* 0xE8-0xEB */
	0x00, 0x00, 0xD2, 0xAC, 0xDD, 0xBB, 0xC8, 0xD3, /* 0xEC-0xEF */
	0x00, 0x00, 0x00, 0x00, 0xB0, 0xFB, 0x68, 0xCE, /* 0xF0-0xF3 */
	0xB0, 0xBB, 0x00, 0x00, 0x68, 0xCF, 0x00, 0x00, /* 0xF4-0xF7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
	0xBB, 0xF4, 0xCB, 0xB0, 0xBE, 0xFE, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_99[512] = {
	0x00, 0x00, 0x00, 0x00, 0x68, 0xD0, 0xF1, 0xAD, /* 0x00-0x03 */
	0x00, 0x00, 0xCC, 0xDF, 0x00, 0x00, 0x68, 0xD1, /* 0x04-0x07 */
	0x68, 0xD2, 0xF1, 0xAE, 0xCD, 0xDC, 0x00, 0x00, /* 0x08-0x0B */
	0xB1, 0xC2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0xBB, 0xC1, 0x68, 0xD3, 0xF1, 0xAF, 0xB2, 0xEE, /* 0x10-0x13 */
	0xF1, 0xB0, 0x68, 0xD4, 0x68, 0xD5, 0x68, 0xD6, /* 0x14-0x17 */
	0xF1, 0xB1, 0x00, 0x00, 0x68, 0xD7, 0x68, 0xD8, /* 0x18-0x1B */
	0x68, 0xD9, 0xF1, 0xB3, 0xF1, 0xB4, 0x68, 0xDA, /* 0x1C-0x1F */
	0xF1, 0xB6, 0xF1, 0xB2, 0x68, 0xDB, 0x00, 0x00, /* 0x20-0x23 */
	0xF1, 0xB5, 0x00, 0x00, 0x68, 0xDC, 0xFC, 0xDE, /* 0x24-0x27 */
	0xB4, 0xDB, 0x00, 0x00, 0x00, 0x00, 0x68, 0xDE, /* 0x28-0x2B */
	0xF1, 0xB7, 0x00, 0x00, 0xF1, 0xB8, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x68, 0xDF, 0x68, 0xE0, 0x68, 0xE1, /* 0x30-0x33 */
	0x68, 0xE2, 0x68, 0xE3, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x68, 0xE4, 0x68, 0xE5, 0x68, 0xE6, /* 0x38-0x3B */
	0x68, 0xE7, 0xF1, 0xB9, 0xF1, 0xBA, 0x00, 0x00, /* 0x3C-0x3F */
	0x68, 0xE8, 0x68, 0xE9, 0xF1, 0xBB, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0xF1, 0xBD, 0x68, 0xEA, 0x68, 0xEB, /* 0x44-0x47 */
	0x68, 0xEC, 0xF1, 0xBC, 0x00, 0x00, 0xF1, 0xBF, /* 0x48-0x4B */
	0xF1, 0xC2, 0x68, 0xED, 0x68, 0xEE, 0x00, 0x00, /* 0x4C-0x4F */
	0xF1, 0xBE, 0xF1, 0xC0, 0xF1, 0xC1, 0x00, 0x00, /* 0x50-0x53 */
	0x68, 0xEF, 0xF1, 0xC3, 0x00, 0x00, 0xB6, 0xC2, /* 0x54-0x57 */
	0x68, 0xF0, 0x68, 0xF1, 0x00, 0x00, 0x68, 0xF2, /* 0x58-0x5B */
	0x68, 0xF3, 0x00, 0x00, 0x68, 0xF4, 0x68, 0xF5, /* 0x5C-0x5F */
	0x68, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0xBC, 0xF3, 0xF1, 0xC4, /* 0x94-0x97 */
	0xF1, 0xC5, 0xB9, 0xE1, 0x00, 0x00, 0x68, 0xF7, /* 0x98-0x9B */
	0x00, 0x00, 0x68, 0xF8, 0xFC, 0xE0, 0x68, 0xF9, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0xF1, 0xC6, 0x68, 0xFA, 0x00, 0x00, /* 0xA4-0xA7 */
	0xB3, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0xC7, 0xCF, 0xF1, 0xC7, 0xF1, 0xC8, 0x00, 0x00, /* 0xAC-0xAF */
	0x68, 0xFB, 0x68, 0xFC, 0x68, 0xFD, 0xC3, 0xDA, /* 0xB0-0xB3 */
	0xC6, 0xEB, 0x68, 0xFE, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x69, 0xA1, 0x69, 0xA2, 0x00, 0x00, /* 0xB8-0xBB */
	0xF1, 0xC9, 0x69, 0xA3, 0x00, 0x00, 0x69, 0xA4, /* 0xBC-0xBF */
	0x00, 0x00, 0xC7, 0xFD, 0x00, 0x00, 0x69, 0xA5, /* 0xC0-0xC3 */
	0xC2, 0xCC, 0xB1, 0xD8, 0xB6, 0xEE, 0x00, 0x00, /* 0xC4-0xC7 */
	0xB6, 0xEF, 0x69, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0xC3, 0xF3, 0xF1, 0xCE, 0xB6, 0xF0, 0x69, 0xA7, /* 0xD0-0xD3 */
	0x69, 0xA8, 0xB2, 0xEF, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xF1, 0xCD, 0x69, 0xA9, 0x69, 0xAA, 0xF1, 0xCB, /* 0xD8-0xDB */
	0x69, 0xAB, 0xF1, 0xCC, 0x69, 0xAC, 0xF1, 0xCA, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0xF1, 0xD8, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x69, 0xAD, /* 0xE4-0xE7 */
	0x00, 0x00, 0x00, 0x00, 0x69, 0xAE, 0x69, 0xAF, /* 0xE8-0xEB */
	0x69, 0xB0, 0xF1, 0xCF, 0xF1, 0xD0, 0x00, 0x00, /* 0xEC-0xEF */
	0x69, 0xB1, 0xF1, 0xD1, 0xF1, 0xD2, 0x00, 0x00, /* 0xF0-0xF3 */
	0x69, 0xB2, 0x69, 0xB3, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0xF1, 0xD4, 0x69, 0xB4, 0x00, 0x00, 0xF1, 0xD3, /* 0xF8-0xFB */
	0x00, 0x00, 0x69, 0xB5, 0x69, 0xB6, 0xBD, 0xD9, /* 0xFC-0xFF */
};

static unsigned char u2c_9A[512] = {
	0x00, 0x00, 0xF1, 0xD5, 0x69, 0xB7, 0x69, 0xB8, /* 0x00-0x03 */
	0x69, 0xB9, 0xF1, 0xD7, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x69, 0xBA, /* 0x08-0x0B */
	0x69, 0xBB, 0x00, 0x00, 0xB5, 0xB3, 0xF1, 0xD6, /* 0x0C-0x0F */
	0x69, 0xBC, 0x69, 0xBD, 0xC1, 0xFB, 0xB8, 0xB3, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x69, 0xBE, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xF1, 0xD9, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x69, 0xBF, 0x00, 0x00, /* 0x1C-0x1F */
	0x69, 0xC0, 0x00, 0x00, 0x69, 0xC1, 0x69, 0xC2, /* 0x20-0x23 */
	0x69, 0xC3, 0x00, 0x00, 0x00, 0x00, 0x69, 0xC4, /* 0x24-0x27 */
	0xC2, 0xCD, 0x00, 0x00, 0x00, 0x00, 0xF1, 0xDA, /* 0x28-0x2B */
	0x00, 0x00, 0x69, 0xC5, 0x69, 0xC6, 0x00, 0x00, /* 0x2C-0x2F */
	0xC6, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x69, 0xC7, /* 0x30-0x33 */
	0x00, 0x00, 0x69, 0xC8, 0x69, 0xC9, 0xF1, 0xDB, /* 0x34-0x37 */
	0x69, 0xCA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0xF1, 0xE0, 0x00, 0x00, /* 0x3C-0x3F */
	0xF1, 0xDE, 0x69, 0xCC, 0xF1, 0xDD, 0xF1, 0xDF, /* 0x40-0x43 */
	0x69, 0xCD, 0xF1, 0xDC, 0x00, 0x00, 0x69, 0xCB, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x69, 0xCE, 0x69, 0xCF, /* 0x48-0x4B */
	0x69, 0xD0, 0xF1, 0xE2, 0xFC, 0xE1, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x69, 0xD2, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x69, 0xD3, 0xF1, 0xE1, 0x69, 0xD4, 0xF1, 0xE4, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0xB6, 0xC3, 0xF1, 0xE3, /* 0x58-0x5B */
	0x00, 0x00, 0x69, 0xD5, 0x00, 0x00, 0xF1, 0xE5, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0xF1, 0xE6, 0x00, 0x00, /* 0x60-0x63 */
	0xF1, 0xE8, 0xF1, 0xE7, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0xF1, 0xE9, 0xF1, 0xEB, 0xF1, 0xEA, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0xB9, 0xFC, 0x00, 0x00, 0x69, 0xD6, 0x00, 0x00, /* 0xA8-0xAB */
	0x69, 0xD7, 0xF1, 0xEC, 0x69, 0xD8, 0x69, 0xD9, /* 0xAC-0xAF */
	0xF1, 0xED, 0x00, 0x00, 0x69, 0xDA, 0x00, 0x00, /* 0xB0-0xB3 */
	0x69, 0xDB, 0x69, 0xDC, 0x69, 0xDD, 0x00, 0x00, /* 0xB4-0xB7 */
	0xB3, 0xBC, 0x69, 0xDE, 0x00, 0x00, 0x69, 0xDF, /* 0xB8-0xBB */
	0xF1, 0xEE, 0x00, 0x00, 0x69, 0xE0, 0x69, 0xE1, /* 0xBC-0xBF */
	0xF1, 0xEF, 0x69, 0xE2, 0x00, 0x00, 0x69, 0xE3, /* 0xC0-0xC3 */
	0xBF, 0xF1, 0x00, 0x00, 0x69, 0xE4, 0x00, 0x00, /* 0xC4-0xC7 */
	0x69, 0xE5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x69, 0xE6, 0xF1, 0xF0, /* 0xCC-0xCF */
	0x69, 0xE7, 0xF1, 0xF1, 0x69, 0xE8, 0xF1, 0xF2, /* 0xD0-0xD3 */
	0xF1, 0xF3, 0x69, 0xE9, 0x69, 0xEA, 0x69, 0xEB, /* 0xD4-0xD7 */
	0xB9, 0xE2, 0xFC, 0xE2, 0x00, 0x00, 0x69, 0xEC, /* 0xD8-0xDB */
	0xFC, 0xE3, 0x00, 0x00, 0xF1, 0xF4, 0xF1, 0xF5, /* 0xDC-0xDF */
	0x69, 0xEE, 0x00, 0x00, 0xF1, 0xF6, 0xF1, 0xF7, /* 0xE0-0xE3 */
	0x69, 0xEF, 0x69, 0xF0, 0xF1, 0xF8, 0x69, 0xF1, /* 0xE4-0xE7 */
	0x00, 0x00, 0x69, 0xF2, 0xC8, 0xB1, 0xF1, 0xFA, /* 0xE8-0xEB */
	0x69, 0xF3, 0xC9, 0xA6, 0xF1, 0xFB, 0xF1, 0xF9, /* 0xEC-0xEF */
	0x00, 0x00, 0xF1, 0xFD, 0x69, 0xF4, 0x69, 0xF5, /* 0xF0-0xF3 */
	0xF1, 0xFC, 0x69, 0xF6, 0x00, 0x00, 0xF1, 0xFE, /* 0xF4-0xF7 */
	0x00, 0x00, 0x69, 0xF7, 0x69, 0xF8, 0xF2, 0xA1, /* 0xF8-0xFB */
	0x00, 0x00, 0x69, 0xF9, 0x00, 0x00, 0x69, 0xFA, /* 0xFC-0xFF */
};

static unsigned char u2c_9B[512] = {
	0x69, 0xFB, 0x69, 0xFC, 0x69, 0xFD, 0x69, 0xFE, /* 0x00-0x03 */
	0x6A, 0xA1, 0x6A, 0xA2, 0xF2, 0xA2, 0x00, 0x00, /* 0x04-0x07 */
	0x6A, 0xA3, 0x6A, 0xA4, 0x00, 0x00, 0x6A, 0xA5, /* 0x08-0x0B */
	0x6A, 0xA6, 0x6A, 0xA7, 0x6A, 0xA8, 0x00, 0x00, /* 0x0C-0x0F */
	0x6A, 0xA9, 0x00, 0x00, 0x6A, 0xAA, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x6A, 0xAB, 0x00, 0x00, /* 0x14-0x17 */
	0xF2, 0xA3, 0x6A, 0xAC, 0xF2, 0xA4, 0x6A, 0xAD, /* 0x18-0x1B */
	0x6A, 0xAE, 0x00, 0x00, 0x00, 0x00, 0xF2, 0xA5, /* 0x1C-0x1F */
	0x6A, 0xAF, 0x00, 0x00, 0xF2, 0xA6, 0xF2, 0xA7, /* 0x20-0x23 */
	0x00, 0x00, 0xF2, 0xA8, 0x6A, 0xB0, 0xF2, 0xA9, /* 0x24-0x27 */
	0xF2, 0xAA, 0xF2, 0xAB, 0xF2, 0xAC, 0x6A, 0xB1, /* 0x28-0x2B */
	0x00, 0x00, 0x6A, 0xB2, 0xF2, 0xAD, 0xF2, 0xAE, /* 0x2C-0x2F */
	0x00, 0x00, 0xDD, 0xB5, 0xF2, 0xAF, 0x6A, 0xB3, /* 0x30-0x33 */
	0x6A, 0xB4, 0x6A, 0xB5, 0x00, 0x00, 0x6A, 0xB6, /* 0x34-0x37 */
	0x00, 0x00, 0x6A, 0xB7, 0x6A, 0xB8, 0xE4, 0xF8, /* 0x38-0x3B */
	0xB5, 0xB4, 0x6A, 0xB9, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0xB3, 0xA1, 0xBA, 0xB2, 0xF2, 0xB1, /* 0x40-0x43 */
	0xF2, 0xB0, 0xCC, 0xA5, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x6A, 0xBA, 0x00, 0x00, 0x00, 0x00, 0x6A, 0xBB, /* 0x48-0x4B */
	0x6A, 0xBC, 0xF2, 0xB3, 0xF2, 0xB4, 0xF2, 0xB2, /* 0x4C-0x4F */
	0x00, 0x00, 0xF2, 0xB5, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0xCB, 0xE2, 0x6A, 0xBD, 0x6A, 0xBE, 0x6A, 0xBF, /* 0x54-0x57 */
	0xF2, 0xB6, 0x00, 0x00, 0xB5, 0xFB, 0x6A, 0xC0, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x6A, 0xC1, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x6A, 0xC2, 0x00, 0x00, 0x6A, 0xC3, /* 0x60-0x63 */
	0x00, 0x00, 0x6A, 0xC4, 0x6A, 0xC5, 0x00, 0x00, /* 0x64-0x67 */
	0x6A, 0xC6, 0x00, 0x00, 0x6A, 0xC7, 0x6A, 0xC8, /* 0x68-0x6B */
	0x6A, 0xC9, 0x6A, 0xCA, 0x6A, 0xCB, 0xCF, 0xA5, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0xFC, 0xE5, 0x6A, 0xCC, /* 0x70-0x73 */
	0xF2, 0xB7, 0xFC, 0xE4, 0x00, 0x00, 0x6A, 0xCE, /* 0x74-0x77 */
	0x6A, 0xCF, 0x6A, 0xD0, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6A, 0xD1, /* 0x7C-0x7F */

	0x6A, 0xD2, 0x00, 0x00, 0x00, 0x00, 0xF2, 0xB9, /* 0x80-0x83 */
	0x6A, 0xD3, 0x6A, 0xD4, 0x6A, 0xD5, 0x6A, 0xD6, /* 0x84-0x87 */
	0x00, 0x00, 0x6A, 0xD7, 0x6A, 0xD8, 0x6A, 0xD9, /* 0x88-0x8B */
	0x00, 0x00, 0x6A, 0xDA, 0xB0, 0xBE, 0xFC, 0xE6, /* 0x8C-0x8F */
	0x6A, 0xDC, 0xF2, 0xBA, 0xCA, 0xAB, 0xF2, 0xB8, /* 0x90-0x93 */
	0x6A, 0xDD, 0x00, 0x00, 0xF2, 0xBB, 0xF2, 0xBC, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x6A, 0xDE, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x6A, 0xDF, 0x6A, 0xE0, 0xF2, 0xBD, /* 0x9C-0x9F */
	0xF2, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x6A, 0xE1, 0x6A, 0xE2, /* 0xA4-0xA7 */
	0xF2, 0xBF, 0x6A, 0xE3, 0xCB, 0xEE, 0xBB, 0xAD, /* 0xA8-0xAB */
	0x6A, 0xE4, 0xBA, 0xFA, 0xC1, 0xAF, 0x00, 0x00, /* 0xAC-0xAF */
	0x6A, 0xE5, 0xFC, 0xE7, 0x6A, 0xE7, 0x00, 0x00, /* 0xB0-0xB3 */
	0xF2, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x6A, 0xE8, /* 0xB4-0xB7 */
	0x6A, 0xE9, 0xF2, 0xC3, 0x00, 0x00, 0xFC, 0xE8, /* 0xB8-0xBB */
	0x6A, 0xEB, 0x00, 0x00, 0x6A, 0xEC, 0x6A, 0xED, /* 0xBC-0xBF */
	0xF2, 0xC1, 0x6A, 0xEE, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0xF2, 0xC4, 0x6A, 0xEF, /* 0xC4-0xC7 */
	0x6A, 0xF0, 0xB8, 0xF1, 0xF2, 0xC2, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x6A, 0xF1, 0xF2, 0xC5, /* 0xCC-0xCF */
	0x6A, 0xF2, 0xF2, 0xC6, 0xF2, 0xC7, 0x00, 0x00, /* 0xD0-0xD3 */
	0xF2, 0xCB, 0x00, 0x00, 0xBB, 0xAA, 0x6A, 0xF3, /* 0xD4-0xD7 */
	0x6A, 0xF4, 0x00, 0x00, 0x00, 0x00, 0xC2, 0xE4, /* 0xD8-0xDB */
	0x00, 0x00, 0x6A, 0xF5, 0x00, 0x00, 0x6A, 0xF6, /* 0xDC-0xDF */
	0x00, 0x00, 0xF2, 0xCC, 0xF2, 0xC9, 0xF2, 0xC8, /* 0xE0-0xE3 */
	0xF2, 0xCA, 0x6A, 0xF7, 0x00, 0x00, 0x6A, 0xF8, /* 0xE4-0xE7 */
	0xB7, 0xDF, 0x00, 0x00, 0x6A, 0xF9, 0x6A, 0xFA, /* 0xE8-0xEB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6A, 0xFB, /* 0xEC-0xEF */
	0xF2, 0xD0, 0xF2, 0xCF, 0xF2, 0xCE, 0x6A, 0xFC, /* 0xF0-0xF3 */
	0x00, 0x00, 0xB0, 0xB3, 0x00, 0x00, 0x6A, 0xFD, /* 0xF4-0xF7 */
	0x6A, 0xFE, 0x6B, 0xA1, 0x6B, 0xA2, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0x6B, 0xA3, 0x00, 0x00, 0x6B, 0xA4, /* 0xFC-0xFF */
};

static unsigned char u2c_9C[512] = {
	0xFC, 0xE9, 0x00, 0x00, 0x6B, 0xA6, 0x00, 0x00, /* 0x00-0x03 */
	0xF2, 0xDA, 0x00, 0x00, 0xF2, 0xD6, 0x00, 0x00, /* 0x04-0x07 */
	0xF2, 0xD7, 0xF2, 0xD3, 0xF2, 0xD9, 0x6B, 0xA7, /* 0x08-0x0B */
	0xF2, 0xD5, 0xB3, 0xE2, 0x00, 0x00, 0x6B, 0xA8, /* 0x0C-0x0F */
	0xCF, 0xCC, 0x6B, 0xA9, 0xF2, 0xD8, 0xF2, 0xD4, /* 0x10-0x13 */
	0xF2, 0xD2, 0xF2, 0xD1, 0x6B, 0xAA, 0x00, 0x00, /* 0x14-0x17 */
	0x6B, 0xAB, 0x6B, 0xAC, 0x6B, 0xAD, 0xF2, 0xDC, /* 0x18-0x1B */
	0x6B, 0xAE, 0x00, 0x00, 0x6B, 0xAF, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0xF2, 0xDF, 0x6B, 0xB0, 0x6B, 0xB1, /* 0x20-0x23 */
	0xF2, 0xDE, 0xF2, 0xDD, 0x6B, 0xB2, 0x6B, 0xB3, /* 0x24-0x27 */
	0x6B, 0xB4, 0x6B, 0xB5, 0x6B, 0xB6, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0xC9, 0xC9, 0xF2, 0xDB, 0xB0, 0xF3, /* 0x2C-0x2F */
	0xF2, 0xE0, 0x6B, 0xB7, 0xF2, 0xE2, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x6B, 0xB8, 0x6B, 0xB9, 0x6B, 0xBA, /* 0x34-0x37 */
	0x00, 0x00, 0xB3, 0xEF, 0xF2, 0xCD, 0xB1, 0xB7, /* 0x38-0x3B */
	0x00, 0x00, 0x6B, 0xBB, 0xF2, 0xE4, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x6B, 0xBC, 0x00, 0x00, 0x6B, 0xBD, /* 0x40-0x43 */
	0x6B, 0xBE, 0x6B, 0xBF, 0xF2, 0xE3, 0xF2, 0xE1, /* 0x44-0x47 */
	0xC3, 0xAD, 0x6B, 0xC0, 0x6B, 0xC1, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x6B, 0xC2, 0x6B, 0xC3, /* 0x4C-0x4F */
	0x6B, 0xC4, 0x00, 0x00, 0xCB, 0xF0, 0x6B, 0xC5, /* 0x50-0x53 */
	0x6B, 0xC6, 0x00, 0x00, 0x6B, 0xC7, 0xCE, 0xDA, /* 0x54-0x57 */
	0x6B, 0xC8, 0x00, 0x00, 0xF2, 0xE5, 0x6B, 0xC9, /* 0x58-0x5B */
	0x6B, 0xD0, 0x6B, 0xCA, 0x6B, 0xCB, 0x6B, 0xCC, /* 0x5C-0x5F */
	0xF2, 0xE6, 0x00, 0x00, 0x00, 0x00, 0x6B, 0xCD, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF2, 0xE7, /* 0x64-0x67 */
	0x6B, 0xD2, 0x6B, 0xCE, 0x6B, 0xCF, 0x6B, 0xD1, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x6B, 0xD3, 0x00, 0x00, /* 0x6C-0x6F */
	0x6B, 0xD4, 0x00, 0x00, 0x6B, 0xD5, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x6B, 0xD6, 0xF2, 0xE8, 0x6B, 0xD7, /* 0x74-0x77 */
	0xF2, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x6B, 0xD8, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0xC4, 0xBB, 0x6B, 0xD9, 0xF2, 0xEA, /* 0xE4-0xE7 */
	0x00, 0x00, 0xC8, 0xB7, 0x00, 0x00, 0xF2, 0xEF, /* 0xE8-0xEB */
	0xF2, 0xEB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xEC-0xEF */
	0xF2, 0xEC, 0x00, 0x00, 0x6B, 0xDA, 0xCB, 0xB1, /* 0xF0-0xF3 */
	0xCC, 0xC4, 0x00, 0x00, 0xC6, 0xD0, 0x6B, 0xDB, /* 0xF4-0xF7 */
	0x00, 0x00, 0x6B, 0xDC, 0x00, 0x00, 0x00, 0x00, /* 0xF8-0xFB */
};

static unsigned char u2c_9D[512] = {
	0x00, 0x00, 0x00, 0x00, 0x6B, 0xDE, 0xF2, 0xF0, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0xF2, 0xF1, 0xC6, 0xBE, /* 0x04-0x07 */
	0xF2, 0xEE, 0xF2, 0xED, 0x00, 0x00, 0x6B, 0xDD, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0xB2, 0xAA, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x6B, 0xDF, 0xF2, 0xF9, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0xF2, 0xF8, 0x00, 0x00, 0x6B, 0xE0, /* 0x14-0x17 */
	0x6B, 0xE1, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xF5, /* 0x18-0x1B */
	0x6B, 0xE2, 0x6B, 0xE3, 0x6B, 0xE4, 0xF2, 0xF6, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF2, 0xF5, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0xF2, 0xF3, 0x00, 0x00, /* 0x24-0x27 */
	0xB3, 0xFB, 0x00, 0x00, 0xF2, 0xF2, 0xBC, 0xB2, /* 0x28-0x2B */
	0xB2, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x6B, 0xE5, /* 0x2C-0x2F */
	0x6B, 0xE6, 0x00, 0x00, 0x6B, 0xE7, 0x6B, 0xE8, /* 0x30-0x33 */
	0x6B, 0xE9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x6B, 0xEA, 0xB9, 0xE3, /* 0x38-0x3B */
	0x6B, 0xEB, 0x6B, 0xED, 0xF2, 0xFC, 0xF2, 0xFB, /* 0x3C-0x3F */
	0x00, 0x00, 0xF2, 0xFA, 0x6B, 0xEE, 0x6B, 0xEF, /* 0x40-0x43 */
	0xF2, 0xF7, 0x6B, 0xEC, 0xF2, 0xFD, 0x6B, 0xF0, /* 0x44-0x47 */
	0xF2, 0xFE, 0x00, 0x00, 0x6B, 0xF1, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0xF3, 0xA5, 0xF3, 0xA4, 0x00, 0x00, 0x6B, 0xF2, /* 0x50-0x53 */
	0x6B, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0xF3, 0xA6, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0xB1, 0xAD, 0xF3, 0xA1, 0xF3, 0xA2, 0x6B, 0xF4, /* 0x5C-0x5F */
	0xB9, 0xF4, 0xCC, 0xB9, 0x6B, 0xF6, 0x6B, 0xF5, /* 0x60-0x63 */
	0xF3, 0xA3, 0x6B, 0xF7, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x6B, 0xF8, 0x6B, 0xF9, 0xFC, 0xEB, /* 0x68-0x6B */
	0xCB, 0xB2, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xAB, /* 0x6C-0x6F */
	0xFC, 0xEA, 0x00, 0x00, 0xF3, 0xA7, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x6B, 0xFC, 0x6B, 0xFD, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0xF3, 0xAC, 0x6B, 0xFE, /* 0x78-0x7B */
	0x6C, 0xA1, 0x00, 0x00, 0x6C, 0xA2, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xA3, /* 0x80-0x83 */
	0x6C, 0xA4, 0x00, 0x00, 0x6C, 0xA5, 0xF3, 0xA9, /* 0x84-0x87 */
	0x00, 0x00, 0xF3, 0xA8, 0x6C, 0xA6, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x6C, 0xA7, 0x6C, 0xA8, 0xB7, 0xDC, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x6C, 0xA9, 0x6C, 0xAA, /* 0x90-0x93 */
	0x00, 0x00, 0x6C, 0xAB, 0x6C, 0xAC, 0x6C, 0xAD, /* 0x94-0x97 */
	0x6C, 0xAE, 0x00, 0x00, 0xF3, 0xAD, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x6C, 0xAF, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0xF3, 0xAE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0xF3, 0xAF, 0x6C, 0xB0, 0xF3, 0xAA, /* 0xA8-0xAB */
	0x6C, 0xB1, 0x00, 0x00, 0x6C, 0xB2, 0xF2, 0xF4, /* 0xAC-0xAF */
	0x00, 0x00, 0x6C, 0xB3, 0xF3, 0xB0, 0x00, 0x00, /* 0xB0-0xB3 */
	0xC4, 0xE1, 0x6C, 0xB4, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xF3, 0xB4, 0x6C, 0xB5, 0xF3, 0xB5, 0xF3, 0xB3, /* 0xB8-0xBB */
	0x6C, 0xB6, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xB7, /* 0xBC-0xBF */
	0x00, 0x00, 0xF3, 0xB2, 0xF3, 0xB8, 0x6C, 0xB8, /* 0xC0-0xC3 */
	0xF3, 0xB1, 0x00, 0x00, 0xF3, 0xB6, 0x6C, 0xB9, /* 0xC4-0xC7 */
	0x00, 0x00, 0x6C, 0xBA, 0x6C, 0xBB, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xB7, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xBA, /* 0xD0-0xD3 */
	0x6C, 0xBC, 0x6C, 0xBD, 0x6C, 0xBE, 0x6C, 0xBF, /* 0xD4-0xD7 */
	0x00, 0x00, 0xF3, 0xB9, 0x6C, 0xC0, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x6C, 0xC1, 0x6C, 0xC2, /* 0xDC-0xDF */
	0x6C, 0xC3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x00, 0x00, 0x6C, 0xC4, 0xF3, 0xBC, 0x6C, 0xC5, /* 0xE4-0xE7 */
	0x00, 0x00, 0x6C, 0xC6, 0x00, 0x00, 0x6C, 0xC7, /* 0xE8-0xEB */
	0x00, 0x00, 0xF3, 0xBD, 0x6C, 0xC8, 0xF3, 0xBE, /* 0xEC-0xEF */
	0x6C, 0xC9, 0x00, 0x00, 0xCF, 0xC9, 0x6C, 0xCA, /* 0xF0-0xF3 */
	0x6C, 0xCB, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF4-0xF7 */
	0xF3, 0xBB, 0xC2, 0xEB, 0xBA, 0xED, 0x00, 0x00, /* 0xF8-0xFB */
	0x00, 0x00, 0xF3, 0xBF, 0x6C, 0xCC, 0x00, 0x00, /* 0xFC-0xFF */
};

static unsigned char u2c_9E[512] = {
	0x00, 0x00, 0x00, 0x00, 0x6C, 0xCE, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xCF, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x6C, 0xCD, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x6C, 0xD0, 0x00, 0x00, /* 0x0C-0x0F */
	0x6C, 0xD1, 0x6C, 0xD2, 0x6C, 0xD3, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x6C, 0xD4, 0x6C, 0xD5, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0xFC, 0xED, 0xF3, 0xC0, 0xF3, 0xC1, /* 0x18-0x1B */
	0x6C, 0xD7, 0x6C, 0xD8, 0xF3, 0xC2, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0xF3, 0xC3, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0xB8, 0xB4, 0xF3, 0xC4, 0x6C, 0xD9, 0x6C, 0xDA, /* 0x78-0x7B */
	0x6C, 0xDB, 0xF3, 0xC5, 0x00, 0x00, 0xBC, 0xAF, /* 0x7C-0x7F */

	0x6C, 0xDC, 0xF3, 0xC6, 0x6C, 0xDD, 0x6C, 0xDE, /* 0x80-0x83 */
	0x6C, 0xDF, 0x6C, 0xE0, 0x00, 0x00, 0x6C, 0xE1, /* 0x84-0x87 */
	0xF3, 0xC7, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xC8, /* 0x88-0x8B */
	0xF3, 0xC9, 0x00, 0x00, 0x6C, 0xE2, 0x6C, 0xE3, /* 0x8C-0x8F */
	0x00, 0x00, 0xF3, 0xCC, 0xF3, 0xCA, 0xCF, 0xBC, /* 0x90-0x93 */
	0x00, 0x00, 0xF3, 0xCB, 0x6C, 0xE4, 0xCE, 0xEF, /* 0x94-0x97 */
	0x6C, 0xE5, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xE6, /* 0x98-0x9B */
	0x00, 0x00, 0xF3, 0xCD, 0x6C, 0xE7, 0xCE, 0xDB, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x6C, 0xE8, 0xF3, 0xCE, 0xC7, 0xFE, 0x00, 0x00, /* 0xA4-0xA7 */
	0x6C, 0xE9, 0xF3, 0xCF, 0xF3, 0xD1, 0x00, 0x00, /* 0xA8-0xAB */
	0x6C, 0xEA, 0xF3, 0xD2, 0x6C, 0xEB, 0x6C, 0xEC, /* 0xAC-0xAF */
	0x6C, 0xED, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xEE, /* 0xB0-0xB3 */
	0x6C, 0xEF, 0x6C, 0xF0, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0xF3, 0xD0, 0xB9, 0xED, 0xCC, 0xCD, 0xCB, 0xE3, /* 0xB8-0xBB */
	0xD6, 0xF7, 0x00, 0x00, 0xDD, 0xE0, 0xCB, 0xFB, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0xB2, 0xAB, 0x00, 0x00, 0x6C, 0xF1, 0x00, 0x00, /* 0xC4-0xC7 */
	0x6C, 0xF2, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xF3, /* 0xC8-0xCB */
	0xF3, 0xD4, 0xB5, 0xD0, 0xF3, 0xD5, 0xF3, 0xD6, /* 0xCC-0xCF */
	0xF3, 0xD7, 0xFC, 0xEE, 0xB9, 0xF5, 0x00, 0x00, /* 0xD0-0xD3 */
	0xF3, 0xD8, 0x6C, 0xF4, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0xE0, 0xD4, 0xCC, 0xDB, 0x00, 0x00, 0xC2, 0xE3, /* 0xD8-0xDB */
	0xF3, 0xD9, 0xF3, 0xDB, 0xF3, 0xDA, 0x6C, 0xF5, /* 0xDC-0xDF */
	0xF3, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE0-0xE3 */
	0x6C, 0xF6, 0xF3, 0xDD, 0x00, 0x00, 0x6C, 0xF7, /* 0xE4-0xE7 */
	0xF3, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE8-0xEB */
	0x6C, 0xF8, 0x6C, 0xF9, 0x6C, 0xFA, 0xF3, 0xDF, /* 0xEC-0xEF */
	0x6C, 0xFB, 0x6C, 0xFC, 0x6C, 0xFD, 0x00, 0x00, /* 0xF0-0xF3 */
	0xF3, 0xE0, 0x6C, 0xFE, 0xF3, 0xE1, 0xF3, 0xE2, /* 0xF4-0xF7 */
	0x6D, 0xA1, 0xF3, 0xE3, 0x00, 0x00, 0xF3, 0xE4, /* 0xF8-0xFB */
	0xF3, 0xE5, 0xF3, 0xE6, 0x00, 0x00, 0x6D, 0xA2, /* 0xFC-0xFF */
};

static unsigned char u2c_9F[512] = {
	0x00, 0x00, 0x00, 0x00, 0x6D, 0xA3, 0x6D, 0xA4, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF3, 0xE7, /* 0x04-0x07 */
	0xF3, 0xE8, 0x6D, 0xA5, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0xC5, 0xA4, 0x6D, 0xA6, /* 0x0C-0x0F */
	0x6D, 0xA7, 0x6D, 0xA8, 0x6D, 0xA9, 0xB8, 0xDD, /* 0x10-0x13 */
	0x6D, 0xAA, 0xF3, 0xEA, 0x6D, 0xAB, 0x6D, 0xAC, /* 0x14-0x17 */
	0x00, 0x00, 0x6D, 0xAD, 0x6D, 0xAE, 0x6D, 0xAF, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6D, 0xB0, /* 0x1C-0x1F */
	0xC1, 0xCD, 0xF3, 0xEB, 0x6D, 0xB1, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x6D, 0xB2, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0x00, 0x00, 0x6D, 0xB3, 0x6D, 0xB4, /* 0x28-0x2B */
	0xF3, 0xEC, 0x00, 0x00, 0x00, 0x00, 0x6D, 0xB5, /* 0x2C-0x2F */
	0x00, 0x00, 0x6D, 0xB6, 0x6D, 0xB7, 0x00, 0x00, /* 0x30-0x33 */
	0x6D, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x6D, 0xB9, /* 0x34-0x37 */
	0x00, 0x00, 0x6D, 0xBA, 0x6D, 0xBB, 0xC9, 0xA1, /* 0x38-0x3B */
	0x6D, 0xBC, 0x6D, 0xBD, 0xF3, 0xED, 0x6D, 0xBE, /* 0x3C-0x3F */
	0x00, 0x00, 0x6D, 0xBF, 0x00, 0x00, 0x6D, 0xC0, /* 0x40-0x43 */
	0x6D, 0xC1, 0x6D, 0xC2, 0x6D, 0xC3, 0x6D, 0xC4, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0xF3, 0xEE, 0xE3, 0xB7, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0xEC, 0xDA, 0xF0, 0xED, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0xF3, 0xEF, 0x6D, 0xC5, /* 0x50-0x53 */
	0xF3, 0xF0, 0x6D, 0xC6, 0x6D, 0xC7, 0x6D, 0xC8, /* 0x54-0x57 */
	0x6D, 0xC9, 0x00, 0x00, 0x6D, 0xCA, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x6D, 0xCB, 0x6D, 0xCC, 0xF3, 0xF2, /* 0x5C-0x5F */
	0xF3, 0xF3, 0xF3, 0xF4, 0xCE, 0xF0, 0xF3, 0xF1, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0xF3, 0xF5, 0xF3, 0xF6, /* 0x64-0x67 */
	0x6D, 0xCD, 0x6D, 0xCE, 0xF3, 0xF8, 0x00, 0x00, /* 0x68-0x6B */
	0xF3, 0xF7, 0x6D, 0xCF, 0x6D, 0xD0, 0x6D, 0xD1, /* 0x6C-0x6F */
	0x6D, 0xD2, 0x6D, 0xD3, 0xF3, 0xFA, 0x6D, 0xD4, /* 0x70-0x73 */
	0x00, 0x00, 0x6D, 0xD5, 0xF3, 0xFB, 0xF3, 0xF9, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x6D, 0xD6, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x6D, 0xD7, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0xCE, 0xB6, 0x00, 0x00, 0x6D, 0xD8, /* 0x8C-0x8F */
	0x6D, 0xD9, 0x6D, 0xDA, 0x6D, 0xDB, 0x00, 0x00, /* 0x90-0x93 */
	0x6D, 0xDC, 0xF3, 0xFC, 0x6D, 0xDD, 0x6D, 0xDE, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0xF3, 0xFD, 0xE3, 0xD4, 0x6D, 0xDF, 0x00, 0x00, /* 0x9C-0x9F */
	0xF3, 0xFE, 0x6D, 0xE0, 0x6D, 0xE1, 0x6D, 0xE2, /* 0xA0-0xA3 */
	0x00, 0x00, 0x6D, 0xE3, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
};

static unsigned char u2c_F9[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0C-0x0F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x10-0x13 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x14-0x17 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x18-0x1B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1C-0x1F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x20-0x23 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x24-0x27 */
	0x00, 0x00, 0xFA, 0xC6, 0x00, 0x00, 0x00, 0x00, /* 0x28-0x2B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x30-0x33 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x34-0x37 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x38-0x3B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3C-0x3F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x40-0x43 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x44-0x47 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x48-0x4B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x4C-0x4F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x50-0x53 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x54-0x57 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x58-0x5B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x60-0x63 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x64-0x67 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x68-0x6B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6C-0x6F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x70-0x73 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x74-0x77 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x78-0x7B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7C-0x7F */

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x80-0x83 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x84-0x87 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x88-0x8B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8C-0x8F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x90-0x93 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x94-0x97 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x98-0x9B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0xFC, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
};

static unsigned char u2c_FA[512] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x00-0x03 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x04-0x07 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x08-0x0B */
	0x00, 0x00, 0x00, 0x00, 0xF9, 0xD4, 0xF9, 0xDF, /* 0x0C-0x0F */
	0xF9, 0xE0, 0xF9, 0xF5, 0xFA, 0xBE, 0xFA, 0xCE, /* 0x10-0x13 */
	0xFA, 0xD0, 0xFA, 0xFB, 0xFB, 0xA3, 0xFB, 0xBA, /* 0x14-0x17 */
	0xFB, 0xC2, 0xFB, 0xC3, 0xFB, 0xC4, 0xFB, 0xC6, /* 0x18-0x1B */
	0xFB, 0xCA, 0xFB, 0xCD, 0xFB, 0xD6, 0xFB, 0xE1, /* 0x1C-0x1F */
	0xFB, 0xE3, 0xFB, 0xE4, 0xFB, 0xED, 0xFB, 0xF5, /* 0x20-0x23 */
	0xFB, 0xF7, 0xFB, 0xF8, 0xFB, 0xFB, 0xFC, 0xB9, /* 0x24-0x27 */
	0xFC, 0xC0, 0xFC, 0xD0, 0xFC, 0xDC, 0xFC, 0xDD, /* 0x28-0x2B */
	0xFC, 0xDF, 0xFC, 0xEC, 0x00, 0x00, 0x00, 0x00, /* 0x2C-0x2F */
};

static unsigned char u2c_FF[512] = {
	0x00, 0x00, 0xA1, 0xAA, 0xFC, 0xFE, 0xA1, 0xF4, /* 0x00-0x03 */
	0xA1, 0xF0, 0xA1, 0xF3, 0xA1, 0xF5, 0xFC, 0xFD, /* 0x04-0x07 */
	0xA1, 0xCA, 0xA1, 0xCB, 0xA1, 0xF6, 0xA1, 0xDC, /* 0x08-0x0B */
	0xA1, 0xA4, 0xA1, 0xDD, 0xA1, 0xA5, 0xA1, 0xBF, /* 0x0C-0x0F */
	0xA3, 0xB0, 0xA3, 0xB1, 0xA3, 0xB2, 0xA3, 0xB3, /* 0x10-0x13 */
	0xA3, 0xB4, 0xA3, 0xB5, 0xA3, 0xB6, 0xA3, 0xB7, /* 0x14-0x17 */
	0xA3, 0xB8, 0xA3, 0xB9, 0xA1, 0xA7, 0xA1, 0xA8, /* 0x18-0x1B */
	0xA1, 0xE3, 0xA1, 0xE1, 0xA1, 0xE4, 0xA1, 0xA9, /* 0x1C-0x1F */
	0xA1, 0xF7, 0xA3, 0xC1, 0xA3, 0xC2, 0xA3, 0xC3, /* 0x20-0x23 */
	0xA3, 0xC4, 0xA3, 0xC5, 0xA3, 0xC6, 0xA3, 0xC7, /* 0x24-0x27 */
	0xA3, 0xC8, 0xA3, 0xC9, 0xA3, 0xCA, 0xA3, 0xCB, /* 0x28-0x2B */
	0xA3, 0xCC, 0xA3, 0xCD, 0xA3, 0xCE, 0xA3, 0xCF, /* 0x2C-0x2F */
	0xA3, 0xD0, 0xA3, 0xD1, 0xA3, 0xD2, 0xA3, 0xD3, /* 0x30-0x33 */
	0xA3, 0xD4, 0xA3, 0xD5, 0xA3, 0xD6, 0xA3, 0xD7, /* 0x34-0x37 */
	0xA3, 0xD8, 0xA3, 0xD9, 0xA3, 0xDA, 0xA1, 0xCE, /* 0x38-0x3B */
	0xA1, 0xC0, 0xA1, 0xCF, 0xA1, 0xB0, 0xA1, 0xB2, /* 0x3C-0x3F */
	0xA1, 0xAE, 0xA3, 0xE1, 0xA3, 0xE2, 0xA3, 0xE3, /* 0x40-0x43 */
	0xA3, 0xE4, 0xA3, 0xE5, 0xA3, 0xE6, 0xA3, 0xE7, /* 0x44-0x47 */
	0xA3, 0xE8, 0xA3, 0xE9, 0xA3, 0xEA, 0xA3, 0xEB, /* 0x48-0x4B */
	0xA3, 0xEC, 0xA3, 0xED, 0xA3, 0xEE, 0xA3, 0xEF, /* 0x4C-0x4F */
	0xA3, 0xF0, 0xA3, 0xF1, 0xA3, 0xF2, 0xA3, 0xF3, /* 0x50-0x53 */
	0xA3, 0xF4, 0xA3, 0xF5, 0xA3, 0xF6, 0xA3, 0xF7, /* 0x54-0x57 */
	0xA3, 0xF8, 0xA3, 0xF9, 0xA3, 0xFA, 0xA1, 0xD0, /* 0x58-0x5B */
	0xA1, 0xC3, 0xA1, 0xD1, 0xA1, 0xC1, 0x00, 0x00, /* 0x5C-0x5F */
	0x00, 0x00, 0x8E, 0xA1, 0x8E, 0xA2, 0x8E, 0xA3, /* 0x60-0x63 */
	0x8E, 0xA4, 0x8E, 0xA5, 0x8E, 0xA6, 0x8E, 0xA7, /* 0x64-0x67 */
	0x8E, 0xA8, 0x8E, 0xA9, 0x8E, 0xAA, 0x8E, 0xAB, /* 0x68-0x6B */
	0x8E, 0xAC, 0x8E, 0xAD, 0x8E, 0xAE, 0x8E, 0xAF, /* 0x6C-0x6F */
	0x8E, 0xB0, 0x8E, 0xB1, 0x8E, 0xB2, 0x8E, 0xB3, /* 0x70-0x73 */
	0x8E, 0xB4, 0x8E, 0xB5, 0x8E, 0xB6, 0x8E, 0xB7, /* 0x74-0x77 */
	0x8E, 0xB8, 0x8E, 0xB9, 0x8E, 0xBA, 0x8E, 0xBB, /* 0x78-0x7B */
	0x8E, 0xBC, 0x8E, 0xBD, 0x8E, 0xBE, 0x8E, 0xBF, /* 0x7C-0x7F */

	0x8E, 0xC0, 0x8E, 0xC1, 0x8E, 0xC2, 0x8E, 0xC3, /* 0x80-0x83 */
	0x8E, 0xC4, 0x8E, 0xC5, 0x8E, 0xC6, 0x8E, 0xC7, /* 0x84-0x87 */
	0x8E, 0xC8, 0x8E, 0xC9, 0x8E, 0xCA, 0x8E, 0xCB, /* 0x88-0x8B */
	0x8E, 0xCC, 0x8E, 0xCD, 0x8E, 0xCE, 0x8E, 0xCF, /* 0x8C-0x8F */
	0x8E, 0xD0, 0x8E, 0xD1, 0x8E, 0xD2, 0x8E, 0xD3, /* 0x90-0x93 */
	0x8E, 0xD4, 0x8E, 0xD5, 0x8E, 0xD6, 0x8E, 0xD7, /* 0x94-0x97 */
	0x8E, 0xD8, 0x8E, 0xD9, 0x8E, 0xDA, 0x8E, 0xDB, /* 0x98-0x9B */
	0x8E, 0xDC, 0x8E, 0xDD, 0x8E, 0xDE, 0x8E, 0xDF, /* 0x9C-0x9F */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA0-0xA3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA4-0xA7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA8-0xAB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xAC-0xAF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB0-0xB3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB4-0xB7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB8-0xBB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xBC-0xBF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC0-0xC3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC4-0xC7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xC8-0xCB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xCC-0xCF */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD0-0xD3 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD4-0xD7 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD8-0xDB */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xDC-0xDF */
	0xA1, 0xF1, 0xA1, 0xF2, 0xA2, 0xCC, 0xA1, 0xB1, /* 0xE0-0xE3 */
	0xFC, 0xFC, 0xA1, 0xEF, 0x00, 0x00, 0x00, 0x00, /* 0xE4-0xE7 */
};



static unsigned char *page_uni2charset[256] = {
	NULL,   u2c_01, u2c_02, u2c_03, u2c_04, NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	u2c_20, u2c_21, u2c_22, u2c_23, u2c_24, u2c_25, u2c_26, NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	u2c_30, NULL,   u2c_32, u2c_33, NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   u2c_4E, u2c_4F,
	u2c_50, u2c_51, u2c_52, u2c_53, u2c_54, u2c_55, u2c_56, u2c_57,
	u2c_58, u2c_59, u2c_5A, u2c_5B, u2c_5C, u2c_5D, u2c_5E, u2c_5F,
	u2c_60, u2c_61, u2c_62, u2c_63, u2c_64, u2c_65, u2c_66, u2c_67,
	u2c_68, u2c_69, u2c_6A, u2c_6B, u2c_6C, u2c_6D, u2c_6E, u2c_6F,
	u2c_70, u2c_71, u2c_72, u2c_73, u2c_74, u2c_75, u2c_76, u2c_77,
	u2c_78, u2c_79, u2c_7A, u2c_7B, u2c_7C, u2c_7D, u2c_7E, u2c_7F,
	u2c_80, u2c_81, u2c_82, u2c_83, u2c_84, u2c_85, u2c_86, u2c_87,
	u2c_88, u2c_89, u2c_8A, u2c_8B, u2c_8C, u2c_8D, u2c_8E, u2c_8F,
	u2c_90, u2c_91, u2c_92, u2c_93, u2c_94, u2c_95, u2c_96, u2c_97,
	u2c_98, u2c_99, u2c_9A, u2c_9B, u2c_9C, u2c_9D, u2c_9E, u2c_9F,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,   NULL,
	NULL,   u2c_F9, u2c_FA, NULL,   NULL,   NULL,   NULL,   u2c_FF, };

static unsigned char charset2lower[256] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, /* 0x00-0x07 */
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, /* 0x08-0x0F */
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* 0x10-0x17 */
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, /* 0x18-0x1F */
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, /* 0x20-0x27 */
	0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, /* 0x28-0x2F */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 0x30-0x37 */
	0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, /* 0x38-0x3F */
	0x40, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, /* 0x40-0x47 */
	0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, /* 0x48-0x4F */
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, /* 0x50-0x57 */
	0x78, 0x79, 0x7A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, /* 0x58-0x5F */
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, /* 0x60-0x67 */
	0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, /* 0x68-0x6F */
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, /* 0x70-0x77 */
	0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, /* 0x78-0x7F */

	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, /* 0x80-0x87 */
	0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, /* 0x88-0x8F */
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, /* 0x90-0x97 */
	0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, /* 0x98-0x9F */
	0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, /* 0xA0-0xA7 */
	0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, /* 0xA8-0xAF */
	0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, /* 0xB0-0xB7 */
	0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, /* 0xB8-0xBF */
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, /* 0xC0-0xC7 */
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, /* 0xC8-0xCF */
	0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, /* 0xD0-0xD7 */
	0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, /* 0xD8-0xDF */
	0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, /* 0xE0-0xE7 */
	0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, /* 0xE8-0xEF */
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, /* 0xF0-0xF7 */
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, /* 0xF8-0xFF */
};

static unsigned char charset2upper[256] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, /* 0x00-0x07 */
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, /* 0x08-0x0F */
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* 0x10-0x17 */
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, /* 0x18-0x1F */
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, /* 0x20-0x27 */
	0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, /* 0x28-0x2F */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 0x30-0x37 */
	0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, /* 0x38-0x3F */
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, /* 0x40-0x47 */
	0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, /* 0x48-0x4F */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, /* 0x50-0x57 */
	0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, /* 0x58-0x5F */
	0x60, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, /* 0x60-0x67 */
	0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, /* 0x68-0x6F */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, /* 0x70-0x77 */
	0x58, 0x59, 0x5A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, /* 0x78-0x7F */

	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, /* 0x80-0x87 */
	0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, /* 0x88-0x8F */
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, /* 0x90-0x97 */
	0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, /* 0x98-0x9F */
	0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, /* 0xA0-0xA7 */
	0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, /* 0xA8-0xAF */
	0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, /* 0xB0-0xB7 */
	0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, /* 0xB8-0xBF */
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, /* 0xC0-0xC7 */
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, /* 0xC8-0xCF */
	0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, /* 0xD0-0xD7 */
	0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, /* 0xD8-0xDF */
	0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, /* 0xE0-0xE7 */
	0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, /* 0xE8-0xEF */
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, /* 0xF0-0xF7 */
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, /* 0xF8-0xFF */
};
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
static inline int sjisibm2euc(unsigned char *euc, const unsigned char sjis_hi,
			      const unsigned char sjis_lo);
static inline int euc2sjisibm_jisx0212(unsigned char *sjis, const unsigned char euc_hi,
				       const unsigned char euc_lo);
static inline int euc2sjisibm_g3upper(unsigned char *sjis, const unsigned char euc_hi,
				      const unsigned char euc_lo);
static inline int euc2sjisibm(unsigned char *sjis, const unsigned char euc_hi,
			      const unsigned char euc_lo);
static inline int sjisnec2sjisibm(unsigned char *sjisibm,
				  const unsigned char sjisnec_hi,
				  const unsigned char sjisnec_lo);
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* SJIS IBM extended characters to EUC */
static inline int sjisibm2euc(unsigned char *euc, const unsigned char sjis_hi,
			      const unsigned char sjis_lo)
{
	int index;

	index = ((sjis_hi - 0xFA) * (0xFD - 0x40)) + (sjis_lo - 0x40);
	if (IS_EUC_IBM2JISX0208(sjisibm2euc_map[index][0],
				sjisibm2euc_map[index][1])) {
		euc[0] = sjisibm2euc_map[index][0];
		euc[1] = sjisibm2euc_map[index][1];
		return 2;
	} else {
		euc[0] = SS3;
		euc[1] = sjisibm2euc_map[index][0];
		euc[2] = sjisibm2euc_map[index][1];
		return 3;
	}
}
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* EUC to SJIS IBM extended characters (G3 JIS X 0212 block) */
static inline int euc2sjisibm_jisx0212(unsigned char *sjis, const unsigned char euc_hi,
				       const unsigned char euc_lo)
{
	int index, min_index, max_index;
	unsigned short euc;

	min_index = 0;
	max_index = ARRAY_SIZE(euc2sjisibm_jisx0212_map) - 1;
	euc = (euc_hi << 8) | euc_lo;

	while (min_index <= max_index) {
		index = (min_index + max_index) / 2;
		if (euc < euc2sjisibm_jisx0212_map[index].euc)
			max_index = index - 1;
		else
			min_index = index + 1;
		if (euc == euc2sjisibm_jisx0212_map[index].euc) {
			sjis[0] = euc2sjisibm_jisx0212_map[index].sjis[0];
			sjis[1] = euc2sjisibm_jisx0212_map[index].sjis[1];
			return 3;
		}
	}
	return 0;
}
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* EUC to SJIS IBM extended characters (G3 Upper block) */
static inline int euc2sjisibm_g3upper(unsigned char *sjis, const unsigned char euc_hi,
				      const unsigned char euc_lo)
{
	int index;

	if (euc_hi == 0xF3)
		index = ((euc_hi << 8) | euc_lo) - 0xF3F3;
	else
		index = ((euc_hi << 8) | euc_lo) - 0xF4A1 + 12;

	if ((index < 0) || (index >= ARRAY_SIZE(euc2sjisibm_g3upper_map)))
		return 0;

	sjis[0] = euc2sjisibm_g3upper_map[index][0];
	sjis[1] = euc2sjisibm_g3upper_map[index][1];

	return 3;
}
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* EUC to SJIS IBM extended characters (G3 block) */
static inline int euc2sjisibm(unsigned char *sjis, const unsigned char euc_hi,
			      const unsigned char euc_lo)
{
	int n;

#if 0
	if ((euc_hi == 0xA2) && (euc_lo == 0xCC)) {
		sjis[0] = 0xFA;
		sjis[1] = 0x54;
		return 2;
	} else if ((euc_hi == 0xA2) && (euc_lo == 0xE8)) {
		sjis[0] = 0xFA;
		sjis[1] = 0x5B;
		return 2;
	}
#endif
	if ((n = euc2sjisibm_g3upper(sjis, euc_hi, euc_lo))) {
		return n;
	} else if ((n = euc2sjisibm_jisx0212(sjis, euc_hi, euc_lo))) {
		return n;
	}

	return 0;
}
#endif

#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
/* NEC/IBM extended characters to IBM extended characters */
static inline int sjisnec2sjisibm(unsigned char *sjisibm,
				  const unsigned char sjisnec_hi,
				  const unsigned char sjisnec_lo)
{
	int count;

	if (! IS_SJIS_NECIBM(sjisnec_hi, sjisnec_lo))
		return 0;

	if ((sjisnec_hi == 0xEE) && (sjisnec_lo == 0xF9)) {
		sjisibm[0] = 0x81;
		sjisibm[1] = 0xCA;
		return 2;
	}

	if ((sjisnec_hi == 0xEE) && (sjisnec_lo >= 0xEF)) {
		count = (sjisnec_hi << 8 | sjisnec_lo)
			- (sjisnec_lo <= 0xF9 ? 0xEEEF : (0xEEEF - 10));
	} else {
		count = (sjisnec_hi - 0xED) * (0xFC - 0x40)
			+ (sjisnec_lo - 0x40) + (0x5C - 0x40);
		if (sjisnec_lo >= 0x7F)
			count--;
	}

	sjisibm[0] = 0xFA + (count / (0xFC - 0x40));
	sjisibm[1] = 0x40 + (count % (0xFC - 0x40));
	if (sjisibm[1] >= 0x7F)
		sjisibm[1]++;

	return 2;
}
#endif

static int uni2char(const wchar_t uni,
		    unsigned char *out, int boundlen)
#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
{
	unsigned char *uni2charset;
	unsigned char cl = uni&0xFF;
	unsigned char ch = (uni>>8)&0xFF;
	unsigned char out0, out1;

	if (boundlen <= 0)
		return -ENAMETOOLONG;

	if (ch == 0xFF && 0x61 <= cl && cl <= 0x9F) {
		if (boundlen < 2)
			return -ENAMETOOLONG;
		out[0] = SS2;
		out[1] = cl + 0x40;
		return 2;
	}
	uni2charset = page_uni2charset[ch];
	if (uni2charset) {
		out0 = uni2charset[cl*2];
		out1 = uni2charset[cl*2+1];

		if (out0 == 0x00) {
			if (out1 == 0x00)
				return -EINVAL;
			out[0] = out1;
			return 1;
		}

		if (out0 & 0x80) {
			if (boundlen < 2)
				return -ENAMETOOLONG;
			out[0] = out0 | 0x80;
			out[1] = out1 | 0x80;
			return 2;
		} else {
			if (boundlen < 3)
				return -ENAMETOOLONG;
			out[0] = SS3;
			out[1] = out0;
			out[2] = out1;
			return 3;
		}
	} else if (ch == 0) {
		if (cl <= 0x9F && cl != SS2 && cl != SS3) {
			out[0] = cl;
			return 1;
		} else if (0xA0 <= cl) {
			out0 = u2c_00hi[cl - 0xA0][0];
			out1 = u2c_00hi[cl - 0xA0][1];

			if (out0 & 0x80) {
				if (boundlen < 2)
					return -ENAMETOOLONG;
				out[0] = out0;
				out[1] = out1;
				return 2;
			} else {
				if (boundlen < 3)
					return -ENAMETOOLONG;
				if (out1) {
					out[0] = SS3;
					out[1] = out0;
					out[2] = out1;
					return 3;
				}
			}
		}
		return -EINVAL;
	}
	else
		return -EINVAL;
}
#else /* !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932) */
{
	int n;

	if (!p_nls)
		return -EINVAL;
	if ((n = p_nls->uni2char(uni, out, boundlen)) < 0)
		return n;

	/* translate SJIS into EUC-JP */
	if (n == 1) {
		if (IS_SJIS_JISX0201KANA(out[0])) {
			/* JIS X 0201 KANA */
			if (boundlen < 2)
				return -ENAMETOOLONG;

			out[1] = out[0];
			out[0] = SS2;
			return 2;
		}
	} else if (n == 2) {
		/* NEC/IBM extended characters to IBM extended characters */
		sjisnec2sjisibm(out, out[0], out[1]);

		if (IS_SJIS_UDC_LOW(out[0], out[1])) {
			/* User defined characters half low */
			MAP_SJIS2EUC(out[0], out[1], 0xF0, out[0], out[1], 0xF5);
		} else if (IS_SJIS_UDC_HI(out[0], out[1])) {
			/* User defined characters half high */
			unsigned char ch, cl;

			if (boundlen < 3)
				return -ENAMETOOLONG;

			n = 3; ch = out[0]; cl = out[1];
			out[0] = SS3;
			MAP_SJIS2EUC(ch, cl, 0xF5, out[1], out[2], 0xF5);
		} else if (IS_SJIS_IBM(out[0], out[1])) {
			/* IBM extended characters */
			unsigned char euc[3], i;

			n = sjisibm2euc(euc, out[0], out[1]);
			if (boundlen < n)
				return -ENAMETOOLONG;
			for (i = 0; i < n; i++)
				out[i] = euc[i];
		} else if (IS_SJIS_JISX0208(out[0], out[1])) {
			/* JIS X 0208 (include NEC special characters) */
			out[0] = (out[0]^0xA0)*2 + 0x5F;
			if (out[1] > 0x9E)
				out[0]++;

			if (out[1] < 0x7F)
				out[1] = out[1] + 0x61;
			else if (out[1] < 0x9F)
				out[1] = out[1] + 0x60;
			else
				out[1] = out[1] + 0x02;
		} else {
			/* Invalid characters */
			return -EINVAL;
		}
	}
	else
		return -EINVAL;

	return n;
}
#endif

static int char2uni(const unsigned char *rawstring, int boundlen,
		    wchar_t *uni)
#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
{
	unsigned char raw0, raw1;
	int offset;

	if (boundlen <= 0)
		return -ENAMETOOLONG;

	raw0 = rawstring[0];
	raw1 = rawstring[1];
	if (IS_EUC_JISX0208(raw0, raw1)) {
		if (boundlen < 2)
			return -EINVAL;

		if (raw0 == 0xAD) {
			offset = raw1 - 0xA1;
			if (c2u_nec[offset] == 0xFFFF)
				return -EINVAL;
			*uni = c2u_nec[offset];
			return 2;
		}
		if (raw0 <= 0xF4 && !(raw0 >= 0xA9  &&  raw0 < 0xB0)) {
			if (raw0 == 0xF4 && 0xA6 < raw1)
				return -EINVAL;
			offset = ((raw0 - 0xA1) * 94) + (raw1 - 0xA1);
			if (raw0 >= 0xB0)
				offset -= (0xB0 - 0xA9) * 94;
			if (c2u_jisx0208[offset] == 0xFFFF)
				return -EINVAL;
			*uni = c2u_jisx0208[offset];
			return 2;
		}
		if (raw0 >= 0xF9 && raw0 <= 0xFC) {
			offset = ((raw0 - 0xF9) * 94) + (raw1 - 0xA1);
			if (c2u_nec2[offset] == 0xFFFF)
				return -EINVAL;
			*uni = c2u_nec2[offset];
			return 2;
		}
	} else if (IS_EUC_JISX0201KANA(raw0, raw1)) {
		if (boundlen < 2)
			return -EINVAL;
		*uni = raw1 - 0xA1 + 0xFF61;
		return 2;
	} else if (raw0 == SS3) {
		if (boundlen < 3)
			return -EINVAL;
		raw0 = raw1;
		raw1 = rawstring[2];
		if (IS_EUC_BYTE(raw0) && IS_EUC_BYTE(raw1)) {
			if (raw0 == 0xA1 ||
			    (raw0 >= 0xA3 && raw0 < 0xA6) ||
			    raw0 == 0xA8 ||
			    (raw0 >= 0xAC && raw0 < 0xB0) ||
			    raw0 >= 0xEE)
			    return -EINVAL;

			offset = ((raw0 - 0xA2) *94) + (raw1 - 0xA1);
			if (raw0 >= 0xA6) {
				offset -= (0xA6 - 0xA3) * 94;
				if (raw0 >= 0xA9) {
					offset -= (0xA9 - 0xA8) * 94;
					if (raw0 >= 0xB0)
						offset -= (0xB0 - 0xAC) * 94;
				}
			}
			if (c2u_jisx0212[offset] != 0xFFFF) {
				*uni = c2u_jisx0212[offset];
				return 3;
			}
		}
	} else if (IS_GL(raw0) || IS_C1(raw0)) {
		*uni = raw0;
		return 1;
	}

	return -EINVAL;
}
#else /* !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932) */
{
	unsigned char sjis_temp[2];
	int euc_offset, n;

	if ( !p_nls )
		return -EINVAL;
	if (boundlen <= 0)
		return -ENAMETOOLONG;

	/* translate EUC-JP into SJIS */
	if (rawstring[0] > 0x7F) {
		if (rawstring[0] == SS3) {
			if (boundlen < 3)
				return -EINVAL;
			euc_offset = 3;

			if (IS_EUC_UDC_HI(rawstring[1], rawstring[2])) {
				/* User defined characters half high */
				MAP_EUC2SJIS(rawstring[1], rawstring[2], 0xF5,
					     sjis_temp[0], sjis_temp[1], 0xF5);
			} else if (euc2sjisibm(sjis_temp,rawstring[1],rawstring[2])) {
				/* IBM extended characters */
			} else {
				/* JIS X 0212 and Invalid characters*/
				return -EINVAL;

				/* 'GETA' with SJIS coding */
				/* sjis_temp[0] = 0x81; */
				/* sjis_temp[1] = 0xAC; */
			}
		} else {
			if (boundlen < 2)
				return -EINVAL;
			euc_offset = 2;

			if (IS_EUC_JISX0201KANA(rawstring[0], rawstring[1])) {
				/* JIS X 0201 KANA */
				sjis_temp[0] = rawstring[1];
				sjis_temp[1] = 0x00;
			} else if (IS_EUC_UDC_LOW(rawstring[0], rawstring[1])) {
				/* User defined characters half low */
				MAP_EUC2SJIS(rawstring[0], rawstring[1], 0xF5,
					     sjis_temp[0], sjis_temp[1], 0xF0);
			} else if (IS_EUC_JISX0208(rawstring[0], rawstring[1])) {
				/* JIS X 0208 (include NEC spesial characters) */
				sjis_temp[0] = ((rawstring[0]-0x5f)/2) ^ 0xA0;
				if (!(rawstring[0] & 1))
					sjis_temp[1] = rawstring[1] - 0x02;
				else if (rawstring[1] < 0xE0)
					sjis_temp[1] = rawstring[1] - 0x61;
				else
					sjis_temp[1] = rawstring[1] - 0x60;
			} else {
				/* Invalid characters */
				return -EINVAL;
			}
		}
	} else {
		euc_offset = 1;

		/* JIS X 0201 ROMAJI */
		sjis_temp[0] = rawstring[0];
		sjis_temp[1] = 0x00;
	}

	if ( (n = p_nls->char2uni(sjis_temp, sizeof(sjis_temp), uni)) < 0)
		return n;

	return euc_offset;
}
#endif

#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
static struct nls_table table = {
	.charset	= "euc-jp",
	.uni2char	= uni2char,
	.char2uni	= char2uni,
	.charset2lower	= charset2lower,
	.charset2upper	= charset2upper,
	.owner		= THIS_MODULE,
};
#else
static struct nls_table table = {
	.charset	= "euc-jp",
	.uni2char	= uni2char,
	.char2uni	= char2uni,
	.owner		= THIS_MODULE,
};
#endif

static int __init init_nls_euc_jp(void)
{
#if defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
	return register_nls(&table);
#else
	p_nls = load_nls("cp932");

	if (p_nls) {
		table.charset2upper = p_nls->charset2upper;
		table.charset2lower = p_nls->charset2lower;
		return register_nls(&table);
	}

	return -EINVAL;
#endif
}

static void __exit exit_nls_euc_jp(void)
{
	unregister_nls(&table);
#if !defined(CONFIG_SNSC_FIX_NLS_CODEPAGE_932)
	unload_nls(p_nls);
#endif
}

module_init(init_nls_euc_jp)
module_exit(exit_nls_euc_jp)

MODULE_LICENSE("Dual BSD/GPL");
