/*
 *  fs/proc/memmap.c
 *
 *
 * Copyright 2010 Sony Corporation.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation; version 2 of the License.
 *
 * THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 * WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 * NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 * USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write  to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This is a system wide /proc entry.
 * This entry provides a global view of the free and allocated physical
 * memory pages. It provides data in a bit map format, one bit represents
 * a single page frame. A one means the page is allocated, a zero means
 * the page is free.
 */

#include <linux/mman.h>
#include <linux/proc_fs.h>
#include <linux/swap.h>
#include <linux/seq_file.h>
#include <linux/vmalloc.h>

#include "internal.h"

#define pfn_to_pte(pfn) (__pte(pfn << PAGE_SHIFT))
#define MEMMAP_OUTPUT_PER_LINE	8
#define BITS_PER_INT 32
#define ASCII_VALUE_DOT 0x2e
#define MEMORY_LARGE_THAN_1MB (1024 * 1024/PAGE_SIZE) + 1

static int lowwater = 0;
static int highwater = 0;
static int mem_acc_debug = 0;
static int memmap_total_memsize;

static void *memmap_start(struct seq_file *m, loff_t *pos)
{
	unsigned int *bitmap = m->private;
	int n;

	if(!*pos)
		return SEQ_START_TOKEN;

	n = *pos * MEMMAP_OUTPUT_PER_LINE;
	if (n < memmap_total_memsize)
		return bitmap + n;

	return 0;
}

static void memmap_stop(struct seq_file *m, void *p)
{
}

static void *memmap_next(struct seq_file *m, void *p, loff_t *pos)
{
	unsigned int *bitmap = m->private;
	int n;

	n = *pos * MEMMAP_OUTPUT_PER_LINE;
	++ *pos;

	if (n < memmap_total_memsize)
		return bitmap + n;

	return 0;
}

static int memmap_show(struct seq_file *m, void *p)
{
	int i, d, index;
	unsigned int *bitmap = p;

	if (p == SEQ_START_TOKEN) {
		seq_printf(m, "High water used pages =%d\t Low water used pages %d\n", highwater, lowwater);
	} else {
		for (i = 0; i < MEMMAP_OUTPUT_PER_LINE; i++) {
			d = *bitmap++;
			if(d ==  ASCII_VALUE_DOT){
				for(index = 0; index < MEMMAP_OUTPUT_PER_LINE; index++)
					seq_printf(m, "%c", d);
			} else
				seq_printf(m, "%08x", d);
		}
		seq_printf(m, "\n");
	}

	return 0;
}

struct seq_operations memmapinfo_op = {
	.start	= memmap_start,
	.next	= memmap_next,
	.stop	= memmap_stop,
	.show	= memmap_show,
};

/*
 * We walk all the page structs and translate them to pfns
 * and then to ptes.  We get the free pages by the page count, count == 0.
 *
 * We have an array of unsigned ints of which each bit in an int
 * represents one page in memory.
 *
 * The pfn is the index into the array, pfn 0 == index 0 (index = pfn / 32).
 * We figure out which bit of the unsigned int to set by the following:
 * bit = pfn % 32, we then set that bit to zero to denote it's free via:
 * array[index] &= ~(1<<bit);   then we pump the data to /proc
 */

static int memmap_file_open(struct inode *inode, struct file *file)
{
	struct page *p;
	int bit, pfn;
	int mark, ret;
	int free, total;
	struct seq_file *m;
	static int index;
	pte_t pte;
	unsigned long  mem_hole_start_addr = 0;
	unsigned long max_pfn, min_pfn;
	int mem_hole_size;
	int pfn_count;
	unsigned int *bitmap = NULL;

	ret = seq_open(file, &memmapinfo_op);
	if(ret)
		return ret;

	memmap_total_memsize = 0;
	min_pfn = calc_min_pfn();
	max_pfn = calc_max_pfn();
	memmap_total_memsize = (max_pfn - min_pfn) / BITS_PER_INT;
	bitmap = vmalloc(memmap_total_memsize * sizeof(int));

	if (!bitmap) {
		return -ENOMEM;
	}

	/*
	 * one means used, zero means free, set them all to 1,
	 * clear them as we find them free.
	 */
	for (index = 0; index < memmap_total_memsize; index++) {
		bitmap[index] = 0xffffffff;
	}

	total = free = 0;
	index = 0;
	mem_hole_size = 0;
	pfn_count  = 0;

	for (pfn = min_pfn; pfn < max_pfn; pfn++) {
		if (!pfn_valid(pfn)) {
			if(mem_hole_size == 0)
				mem_hole_start_addr = pfn << PAGE_SHIFT;
			mem_hole_size ++;
			if(mem_hole_size == MEMORY_LARGE_THAN_1MB)
				printk("Starting page address of the Memory hole : 0x%lx\n",mem_hole_start_addr);
			pfn_count++;
			index = pfn_count;
			index /= BITS_PER_INT;
			bitmap[index] = ASCII_VALUE_DOT;
			continue;
		} else if (!pfn_valid(pfn))
			continue;

		p = pfn_to_page(pfn);
		total++;
		if (!PageReserved(p) && page_count(p) == 0
			&& pfn_valid(pfn)) {
			index = pfn_count;
			pfn_count++;
			bit = index % BITS_PER_INT;
			index /= BITS_PER_INT;
			free++;
			bitmap[index] &= ~(1 << bit);
			mem_hole_size = 0;

		} else if (pfn_valid(pfn))
			pfn_count++;

		if (mem_acc_debug) {
			pte = pfn_to_pte(pfn);
			index = pfn_count;
			pfn_count++;
			bit = index % BITS_PER_INT;
			index /= BITS_PER_INT;
			printk("i %d bit %d pfn %d pa 0x%lx va 0x%lx "
				"c %d\n", index, bit, pfn, pte_val(pte),
				(unsigned long)__va(pte_val(pte)),
				page_count(p));
		}
	}
	m = file->private_data;
	m->private = bitmap;
	mark = total - free;

	if (mark < lowwater)
		lowwater = mark;

	if (mark > highwater)
		highwater = mark;

	return 0;
}

static int memmap_file_release (struct inode *inode, struct file *file)
{
	struct seq_file *m = file->private_data;

	if (m->private)
		vfree(m->private);

	return seq_release(inode, file);
}

static struct file_operations proc_memmap_file_operations = {
	.open		= memmap_file_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= memmap_file_release,
};

void __init proc_memmap_init(void)
{
	struct proc_dir_entry *entry;
	/*
	 * get the amount of free pages right here, as this
	 * should be the lowest amount of allocated pages
	 * the system ever sees.  After log in there
	 * should just be more allocated pages.
	 */
	lowwater = num_physpages - nr_free_pages();
	entry = create_proc_entry("memmap", 0, NULL);
	if (entry)
		entry->proc_fops = &proc_memmap_file_operations;
}

module_init(proc_memmap_init);
