/*
 * fs/proc/ppgstat.c
 *
 *
 * Copyright 2010 Sony Corporation.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation; version 2 of the License.
 *
 * THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 * WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 * NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 * USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write  to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This is a system wide /proc entry.
 * This entry shows the status of each page of a process in detail,
 * such as  active/inactive, shared/non-shared, slab/normal,
 * reserved/not.
 *
 */

#include <linux/mman.h>
#include <linux/proc_fs.h>
#include <linux/swap.h>
#include <linux/seq_file.h>
#include <linux/vmalloc.h>

#include "internal.h"

static unsigned int ppgstat_total_memsize;
#define PPGSTAT_PAGES_PER_LINE	(16)

static unsigned int base64(unsigned int x) {

	x &= 63;
	if (x < 26) return x + 'A';
	if (x < 52) return x + 'a' - 26;
	if (x < 62) return x + '0' - 52;
	if (x < 63) return '+';
	return '/';
}

static void *ppgstat_start(struct seq_file *m, loff_t *pos)
{
	unsigned int *bitmap = m->private;
	int n;

	n = *pos * PPGSTAT_PAGES_PER_LINE;
	if (n < ppgstat_total_memsize)
		return bitmap + n;

	return 0;
}

static void ppgstat_stop(struct seq_file *m, void *p)
{
}

static void *ppgstat_next(struct seq_file *m, void *p, loff_t *pos)
{
	unsigned int *bitmap = m->private;
	int n;

	++*pos;
	n = *pos * PPGSTAT_PAGES_PER_LINE;
	if (n < ppgstat_total_memsize)
		return bitmap + n;

	return 0;
}

static int ppgstat_show(struct seq_file *m, void *p)
{
	int i, d;
	unsigned int *bitmap = p;

	for (i = 0; i < PPGSTAT_PAGES_PER_LINE; i++) {
		d = *bitmap++;
		seq_putc(m, base64(d));
		seq_putc(m, base64(d >>6));
		seq_putc(m, base64(d >>12));
		seq_putc(m, base64(d >>18));
	}
	seq_putc(m, '\n');

	return 0;
}

struct seq_operations ppgstatinfo_op = {
        .start  = ppgstat_start,
        .next   = ppgstat_next,
        .stop   = ppgstat_stop,
        .show   = ppgstat_show,
};

static int ppgstat_file_open(struct inode *inode, struct file *file)
{
	int ret, index;
	struct page *p;
	unsigned int *bitmap = NULL;
	struct seq_file *m;
	int pfn, d;
	unsigned long max_pfn, min_pfn;

	ret = seq_open(file, &ppgstatinfo_op);

	if(ret) return ret;

	ppgstat_total_memsize = 0;

	min_pfn = calc_min_pfn();
	max_pfn = calc_max_pfn();
	ppgstat_total_memsize = max_pfn - min_pfn;

 	bitmap = vmalloc(ppgstat_total_memsize * sizeof(int));
	if (!bitmap)
		return -ENOMEM;

	for (index = 0; index < ppgstat_total_memsize; index++)
		bitmap[index] = 0;

	index = 0;
	for (pfn = min_pfn; pfn < max_pfn; pfn++) {
		if (!pfn_valid(pfn)) {
			d = 0;
			d |= (1 << 20);
			bitmap[index++] = d;
			continue;
		} else if (!pfn_valid(pfn))
			continue;

		p = pfn_to_page(pfn);
		d = 0;
		if (PageReserved(p))    d |= (1 << 0);
		if (page_count(p) == 0) d |= (1 << 1);
		if (PageSlab(p))        d |= (1 << 2);
		if (PageLRU(p))         d |= (1 << 3);
		if (PageReferenced(p))  d |= (1 << 4);
		if (PageActive(p))      d |= (1 << 5);
		if (page_count(p) > 1)  d |= (1 << 6);
		if (PagePrivate(p))     d |= (1 << 7);
		if (PageMappedToDisk(p)) d |= (1 << 8);
		if (PageChecked(p))     d |= (1 << 9);
		if (PageDirty(p))       d |= (1 << 10);
		if (PageUptodate(p))    d |= (1 << 11);
		if (PageLocked(p))      d |= (1 << 12);
		if (PageWriteback(p))   d |= (1 << 13);
		if (PageCompound(p))    d |= (1 << 14);
		if (PageBuddy(p))       d |= (1 << 15);
		if (PageSwapCache(p))   d |= (1 << 16);
		if (PageError(p))       d |= (1 << 17);
		if (PageReclaim(p))     d |= (1 << 18);
		if (PageHighMem(p))     d |= (1 << 19);

		bitmap[index++] = d;
	}
	m = file->private_data;
	m->private = bitmap;
	return 0;
}

static int ppgstat_file_release (struct inode *inode, struct file *file)
{
	struct seq_file *m = file->private_data;

	if (m->private)
		vfree(m->private);

	return seq_release(inode, file);
}

static struct file_operations proc_ppgstat_file_operations = {
	.open           = ppgstat_file_open,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = ppgstat_file_release,
};

void __init proc_ppgstat_init(void)
{
	struct proc_dir_entry *entry;
	entry = create_proc_entry("ppgstat", 0, NULL);
	if (entry)
		entry->proc_fops = &proc_ppgstat_file_operations;
}

module_init(proc_ppgstat_init);
