/*
 *  fs/snsc_batch_mknod.c
 *
 *  Copyright 2009 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/snsc_batch_mknod.h>
#include <linux/namei.h>
#include <linux/syscalls.h>
#include <linux/uaccess.h>

static int
batch_mknod_one(const char* pathname, int mode, dev_t dev)
{
	struct dentry* dentry;
	struct nameidata nd;
	int err = 0;
#ifdef CONFIG_SNSC_BATCH_MKNOD_FORCE_OVERWRITE
	mm_segment_t old_fs;
#endif

	if (!S_ISCHR(mode) && !S_ISBLK(mode)) {
		return -EINVAL;
	}

	err = path_lookup(pathname, LOOKUP_PARENT, &nd);
	if (err) {
		return err;
	}

#ifdef CONFIG_SNSC_BATCH_MKNOD_FORCE_OVERWRITE
	old_fs = get_fs();
	set_fs(get_ds());

	sys_unlink(pathname);

	set_fs(old_fs);
#endif

	dentry = lookup_create(&nd, 0);
	err = PTR_ERR(dentry);

	if (!IS_ERR(dentry)) {
		err = vfs_mknod(nd.path.dentry->d_inode, dentry, mode, dev);
		dput(dentry);
	}

	mutex_unlock(&nd.path.dentry->d_inode->i_mutex);
	path_put(&nd.path);

	return err;
}

int
snsc_batch_mknod(const snsc_batch_mknod_t *list, unsigned int list_num)
{
	unsigned int i;
	int ret, err = 0;

	for (i = 0; i < list_num; i++) {
		ret = batch_mknod_one(list->pathname, list->mode, list->dev);
		if (ret < 0) {
			pr_err("snsc_batch_mknod: cannot create '%s': %d\n",
			       list->pathname, ret);
			err = ret;
		}
		list++;
	}
	return err;
}

EXPORT_SYMBOL(snsc_batch_mknod);
