
#include <linux/irq.h>

void move_masked_irq(int irq)
{
	struct irq_desc *desc = irq_to_desc(irq);

	if (likely(!(desc->status & IRQ_MOVE_PENDING)))
		return;

	/*
	 * Paranoia: cpu-local interrupts shouldn't be calling in here anyway.
	 */
	if (CHECK_IRQ_PER_CPU(desc->status)) {
		WARN_ON(1);
		return;
	}

	desc->status &= ~IRQ_MOVE_PENDING;

	if (unlikely(cpumask_empty(desc->pending_mask)))
		return;

	if (!desc->chip->set_affinity)
		return;

	assert_spin_locked(&desc->lock);

	/*
	 * If there was a valid mask to work with, please
	 * do the disable, re-program, enable sequence.
	 * This is *not* particularly important for level triggered
	 * but in a edge trigger case, we might be setting rte
	 * when an active trigger is comming in. This could
	 * cause some ioapics to mal-function.
	 * Being paranoid i guess!
	 *
	 * For correct operation this depends on the caller
	 * masking the irqs.
	 */
	if (likely(cpumask_any_and(desc->pending_mask, cpu_online_mask)
		   < nr_cpu_ids)) {
		cpumask_and(desc->affinity,
			    desc->pending_mask, cpu_online_mask);
		desc->chip->set_affinity(irq, desc->affinity);
	}
	cpumask_clear(desc->pending_mask);
}

void move_native_irq(int irq)
{
	struct irq_desc *desc = irq_to_desc(irq);
	int mask = 1;

	if (likely(!(desc->status & IRQ_MOVE_PENDING)))
		return;

	if (unlikely(desc->status & IRQ_DISABLED))
		return;

	/*
	 * If the irq is already in progress, it should be masked.
	 * If we unmask it, we might cause an interrupt storm on RT.
	 */
	if (unlikely(desc->status & IRQ_INPROGRESS))
		mask = 0;

	if (mask)
		desc->chip->mask(irq);
	move_masked_irq(irq);
	if (mask)
		desc->chip->unmask(irq);
}

