/*
 * drivers/power/process.c - Functions for starting/stopping processes on 
 *                           suspend transitions.
 *
 * Originally from swsusp.
 */


#undef DEBUG

#include <linux/interrupt.h>
#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/syscalls.h>
#include <linux/freezer.h>

/* 
 * Timeout for stopping processes
 */
#define TIMEOUT	(20 * HZ)

static inline int freezeable(struct task_struct * p)
{
	if ((p == current) ||
	    (p->flags & PF_NOFREEZE) ||
	    (p->exit_state != 0))
		return 0;
	return 1;
}

static int try_to_freeze_tasks(bool sig_only)
{
	struct task_struct *g, *p;
	unsigned long end_time;
	unsigned int todo;
	struct timeval start, end;
	u64 elapsed_csecs64;
	unsigned int elapsed_csecs;

#ifdef CONFIG_SNSC_PM_SCHED_NORMAL_AT_FREEZE
	struct sched_param param = { .sched_priority = 0 };
	int policy = current->policy;
	int prio = current->prio;
	int error;

	/* change scheduling policy to normal */
	error = sched_setscheduler(current, SCHED_NORMAL, &param);
	if (error)
		return error;
#endif

	do_gettimeofday(&start);

	end_time = jiffies + TIMEOUT;
	do {
		todo = 0;
		read_lock(&tasklist_lock);
		do_each_thread(g, p) {
#ifdef CONFIG_SNSC_SAVE_AFFINITY
			p->cpus_allowed_saved = p->cpus_allowed;
			p->cpus_allowed_saved_valid = 1;
#endif
			if (frozen(p) || !freezeable(p))
				continue;

			if (!freeze_task(p, sig_only))
				continue;

			/*
			 * Now that we've done set_freeze_flag, don't
			 * perturb a task in TASK_STOPPED or TASK_TRACED.
			 * It is "frozen enough".  If the task does wake
			 * up, it will immediately call try_to_freeze.
			 */
			if (!task_is_stopped_or_traced(p) &&
			    !freezer_should_skip(p))
				todo++;
		} while_each_thread(g, p);
		read_unlock(&tasklist_lock);
		yield();			/* Yield is okay here */
		if (time_after(jiffies, end_time))
			break;
	} while (todo);

	do_gettimeofday(&end);
	elapsed_csecs64 = timeval_to_ns(&end) - timeval_to_ns(&start);
	do_div(elapsed_csecs64, NSEC_PER_SEC / 100);
	elapsed_csecs = elapsed_csecs64;

	if (todo) {
		/* This does not unfreeze processes that are already frozen
		 * (we have slightly ugly calling convention in that respect,
		 * and caller must call thaw_processes() if something fails),
		 * but it cleans up leftover PF_FREEZE requests.
		 */
		printk("\n");
		printk(KERN_ERR "Freezing of tasks failed after %d.%02d seconds "
				"(%d tasks refusing to freeze):\n",
				elapsed_csecs / 100, elapsed_csecs % 100, todo);
		show_state();
		read_lock(&tasklist_lock);
		do_each_thread(g, p) {
			task_lock(p);
#ifdef CONFIG_SNSC
			if (freezing(p) && !freezer_should_skip(p)) {
				static const char stat_nam[] = "RSDTtZX?";
				unsigned int state;
				state = p->state ? __ffs(p->state) + 1 : 0;
				state = min(state, sizeof(stat_nam) - 2);
				printk(KERN_ERR " %s (pid: %d, state: %c)\n",
				       p->comm, p->pid, stat_nam[state]);
			}
#else
			if (freezing(p) && !freezer_should_skip(p))
				printk(KERN_ERR " %s\n", p->comm);
#endif
			cancel_freezing(p);
			task_unlock(p);
		} while_each_thread(g, p);
		read_unlock(&tasklist_lock);
	} else {
		printk("(elapsed %d.%02d seconds) ", elapsed_csecs / 100,
			elapsed_csecs % 100);
	}

#ifdef CONFIG_SNSC_PM_SCHED_NORMAL_AT_FREEZE
	/* restore scheduling policy/priority */
	if (policy == SCHED_FIFO || policy == SCHED_RR) {
		param.sched_priority = prio;
	}
	error = sched_setscheduler(current, policy, &param);
	if (error) {
		return error;
	}
#endif

	return todo ? -EBUSY : 0;
}

/**
 *	freeze_processes - tell processes to enter the refrigerator
 */
int freeze_processes(void)
{
	int error;

	printk("Freezing user space processes ... ");
	error = try_to_freeze_tasks(true);
	if (error)
		goto Exit;
	printk("done.\n");

	printk("Freezing remaining freezable tasks ... ");
	error = try_to_freeze_tasks(false);
	if (error)
		goto Exit;
	printk("done.");
 Exit:
	BUG_ON(in_atomic());
	printk("\n");
	return error;
}

#ifdef CONFIG_SNSC_SSBOOT
void thaw_tasks(bool nosig_only)
#else
static void thaw_tasks(bool nosig_only)
#endif
{
	struct task_struct *g, *p;

	read_lock(&tasklist_lock);
	do_each_thread(g, p) {
#ifdef CONFIG_SNSC_SAVE_AFFINITY
		if (p->cpus_allowed_saved_valid) {
			p->cpus_allowed = p->cpus_allowed_saved;
			p->cpus_allowed_saved_valid = 0;
		}
#endif
		if (!freezeable(p))
			continue;

		if (nosig_only && should_send_signal(p))
			continue;

		if (cgroup_frozen(p))
			continue;

		thaw_process(p);
	} while_each_thread(g, p);
	read_unlock(&tasklist_lock);
}

void thaw_processes(void)
{
	printk("Restarting tasks ... ");
	thaw_tasks(true);
	thaw_tasks(false);
	schedule();
	printk("done.\n");
}

#ifdef CONFIG_SNSC_SSBOOT
int freeze_tasks(int freeze_user_space)
{
	int error;
	bool sig_only;

	printk("Stopping %s tasks ... ",
	       freeze_user_space ? "user" : "kernel");

	/* freeze tasks */
	sig_only = (freeze_user_space) ? true : false;
	error = try_to_freeze_tasks(sig_only);
	if (error)
		return error;

	/* do sync after freeze user space */
	if (freeze_user_space)
		sys_sync();

	printk("done.\n");
	BUG_ON(in_atomic());
	return 0;
}
#endif

#ifdef CONFIG_SNSC_DEFERRED_RESUME

/* Thaw processes blong to PGID process group.  */

void thaw_process_group(int pgid)
{
	struct task_struct *p;
	struct pid *pgrp;
	read_lock(&tasklist_lock);
	pgrp = find_vpid(pgid);
	do_each_pid_task(pgrp, PIDTYPE_PGID, p) {
		if (!freezeable(p))
			continue;
		if (thaw_process(p)) {
			pr_debug("thaw pgid(%d): [%d] %s\n", pgid, p->pid, p->comm);
		} else {
			printk(KERN_INFO " Thaw pgid(%d): Strange, [%d] %s not stopped\n",
				pgid, p->pid, p->comm);
		}
	} while_each_pid_task(pgrp, PIDTYPE_PGID, p);
	read_unlock(&tasklist_lock);
}

#endif /* CONFIG_SNSC_DEFERRED_RESUME */
