/*
 * ltt/probes/net-trace.c
 *
 * Net tracepoint probes.
 */

#include <linux/module.h>
#include <linux/netdevice.h>
#include <trace/net.h>
#include <trace/ipv4.h>
#include <trace/ipv6.h>
#include <trace/socket.h>

void probe_net_dev_xmit(struct sk_buff *skb)
{
	trace_mark_tp(net, dev_xmit, net_dev_xmit,
		probe_net_dev_xmit,
		"skb %p protocol #2u%hu", skb, skb->protocol);
}

void probe_net_dev_receive(struct sk_buff *skb)
{
	trace_mark_tp(net, dev_receive, net_dev_receive,
		probe_net_dev_receive, "skb %p protocol #2u%hu",
		skb, skb->protocol);
}

void probe_ipv4_addr_add(struct in_ifaddr *ifa)
{
	trace_mark_tp(netif_state, insert_ifa_ipv4, ipv4_addr_add,
		probe_ipv4_addr_add, "label %s address #4u%u",
		ifa->ifa_label, (unsigned int)ifa->ifa_address);
}

void probe_ipv4_addr_del(struct in_ifaddr *ifa)
{
	trace_mark_tp(netif_state, del_ifa_ipv4, ipv4_addr_del,
		probe_ipv4_addr_del, "label %s address #4u%u",
		ifa->ifa_label, (unsigned int)ifa->ifa_address);
}

void probe_ipv6_addr_add(struct inet6_ifaddr *ifa)
{
	__u8 *addr = ifa->addr.s6_addr;

	trace_mark_tp(netif_state, insert_ifa_ipv6, ipv6_addr_add,
		probe_ipv6_addr_add,
		"label %s "
		"a15 #1x%c a14 #1x%c a13 #1x%c a12 #1x%c "
		"a11 #1x%c a10 #1x%c a9 #1x%c a8 #1x%c "
		"a7 #1x%c a6 #1x%c a5 #1x%c a4 #1x%c "
		"a3 #1x%c a2 #1x%c a1 #1x%c a0 #1x%c",
		ifa->idev->dev->name,
		addr[15], addr[14], addr[13], addr[12],
		addr[11], addr[10], addr[9], addr[8],
		addr[7], addr[6], addr[5], addr[4],
		addr[3], addr[2], addr[1], addr[0]);
}

void probe_ipv6_addr_del(struct inet6_ifaddr *ifa)
{
	__u8 *addr = ifa->addr.s6_addr;

	trace_mark_tp(netif_state, insert_ifa_ipv6, ipv6_addr_del,
		probe_ipv6_addr_del,
		"label %s "
		"a15 #1x%c a14 #1x%c a13 #1x%c a12 #1x%c "
		"a11 #1x%c a10 #1x%c a9 #1x%c a8 #1x%c "
		"a7 #1x%c a6 #1x%c a5 #1x%c a4 #1x%c "
		"a3 #1x%c a2 #1x%c a1 #1x%c a0 #1x%c",
		ifa->idev->dev->name,
		addr[15], addr[14], addr[13], addr[12],
		addr[11], addr[10], addr[9], addr[8],
		addr[7], addr[6], addr[5], addr[4],
		addr[3], addr[2], addr[1], addr[0]);
}

void probe_socket_create(int family, int type, int protocol,
	struct socket *sock, int ret)
{
	trace_mark_tp(net, socket_create, socket_create, probe_socket_create,
		"family %d type %d protocol %d sock %p ret %d",
		family, type, protocol, sock, ret);
}

void probe_socket_bind(int fd, struct sockaddr __user *umyaddr, int addrlen,
	int ret)
{
	trace_mark_tp(net, socket_bind, socket_bind, probe_socket_bind,
		"fd %d umyaddr %p addrlen %d ret %d",
		fd, umyaddr, addrlen, ret);
}

void probe_socket_connect(int fd, struct sockaddr __user *uservaddr,
	int addrlen, int ret)
{
	trace_mark_tp(net, socket_connect, socket_connect, probe_socket_connect,
		"fd %d uservaddr %p addrlen %d ret %d",
		fd, uservaddr, addrlen, ret);
}

void probe_socket_listen(int fd, int backlog, int ret)
{
	trace_mark_tp(net, socket_listen, socket_listen, probe_socket_listen,
		"fd %d backlog %d ret %d",
		fd, backlog, ret);
}

void probe_socket_accept(int fd, struct sockaddr __user *upeer_sockaddr,
	int __user *upeer_addrlen, int flags, int ret)
{
	trace_mark_tp(net, socket_accept, socket_accept, probe_socket_accept,
		"fd %d upeer_sockaddr %p upeer_addrlen %p flags %d ret %d",
		fd, upeer_sockaddr, upeer_addrlen, flags, ret);
}

void probe_socket_getsockname(int fd, struct sockaddr __user *usockaddr,
	int __user *usockaddr_len, int ret)
{
	trace_mark_tp(net, socket_getsockname, socket_getsockname,
		probe_socket_getsockname,
		"fd %d usockaddr %p usockaddr_len %p ret %d",
		fd, usockaddr, usockaddr_len, ret);
}

void probe_socket_getpeername(int fd, struct sockaddr __user *usockaddr,
	int __user *usockaddr_len, int ret)
{
	trace_mark_tp(net, socket_getpeername, socket_getpeername,
		probe_socket_getpeername,
		"fd %d usockaddr %p usockaddr_len %p ret %d",
		fd, usockaddr, usockaddr_len, ret);
}

void probe_socket_socketpair(int family, int type, int protocol,
	int __user *usockvec, int ret)
{
	trace_mark_tp(net, socket_socketpair, socket_socketpair,
		probe_socket_socketpair,
		"family %d type %d protocol %d usockvec %p ret %d",
		family, type, protocol, usockvec, ret);
}

void probe_socket_sendmsg(struct socket *sock, struct msghdr *msg, size_t size,
	int ret)
{
	trace_mark_tp(net, socket_sendmsg, socket_sendmsg, probe_socket_sendmsg,
		"sock %p msg %p size %zu ret %d",
		sock, msg, size, ret);
}

void probe_socket_recvmsg(struct socket *sock, struct msghdr *msg, size_t size,
	int flags, int ret)
{
	trace_mark_tp(net, socket_recvmsg, socket_recvmsg, probe_socket_recvmsg,
		"sock %p msg %p size %zu flags %d ret %d",
		sock, msg, size, flags, ret);
}

void probe_socket_setsockopt(int fd, int level, int optname,
	char __user *optval, int optlen, int ret)
{
	trace_mark_tp(net, socket_setsockopt, socket_setsockopt,
		probe_socket_setsockopt,
		"fd %d level %d optname %d optval %p optlen %d ret %d",
		fd, level, optname, optval, optlen, ret);
}

void probe_socket_getsockopt(int fd, int level, int optname,
	char __user *optval, int __user *optlen, int ret)
{
	trace_mark_tp(net, socket_getsockopt, socket_getsockopt,
		probe_socket_getsockopt,
		"fd %d level %d optname %d optval %p optlen %p ret %d",
		fd, level, optname, optval, optlen, ret);
}

void probe_socket_shutdown(int fd, int how, int ret)
{
	trace_mark_tp(net, socket_shutdown, socket_shutdown,
		probe_socket_shutdown,
		"fd %d how %d ret %d",
		fd, how, ret);
}

void probe_socket_call(int call, unsigned long a0)
{
	trace_mark_tp(net, socket_call, socket_call, probe_socket_call,
		"call %d a0 %lu", call, a0);
}

#ifdef CONFIG_NETPOLL
void probe_net_napi_schedule(struct napi_struct *n)
{
	trace_mark_tp(net, napi_schedule, net_napi_schedule,
		probe_net_napi_schedule,
		"napi_struct %p name %s",
		n, n->dev->name);
}

void probe_net_napi_poll(struct napi_struct *n)
{
	trace_mark_tp(net, napi_poll, net_napi_poll, probe_net_napi_poll,
		"napi_struct %p name %s",
		n, n->dev->name);
}

void probe_net_napi_complete(struct napi_struct *n)
{
	trace_mark_tp(net, napi_complete, net_napi_complete,
		probe_net_napi_complete,
		"napi_struct %p name %s",
		n, n->dev->name);
}
#else /* !CONFIG_NETPOLL */
void probe_net_napi_schedule(struct napi_struct *n)
{
	trace_mark_tp(net, napi_schedule, net_napi_schedule,
		probe_net_napi_schedule,
		"napi_struct %p", n);
}

void probe_net_napi_poll(struct napi_struct *n)
{
	trace_mark_tp(net, napi_poll, net_napi_poll, probe_net_napi_poll,
		"napi_struct %p", n);
}

void probe_net_napi_complete(struct napi_struct *n)
{
	trace_mark_tp(net, napi_complete, net_napi_complete,
		probe_net_napi_complete,
		"napi_struct %p", n);
}
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Mathieu Desnoyers");
MODULE_DESCRIPTION("Net Tracepoint Probes");
