/*
 * arch/arm/kernel/unwind_user.c
 *
 * Copyright (C) 2011 Sony corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * Stack unwinding support for ARM user space
 * ... included in unwind.c
 *
 * An ARM EABI version of gcc is required to generate the unwind
 * tables. For information about the structure of the unwind tables,
 * see "Exception Handling ABI for the ARM Architecture" at:
 *
 * http://infocenter.arm.com/help/topic/com.arm.doc.subset.swdev.abi/index.html
 */

#include <asm/unaligned.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/snsc_backtrace.h>
#include <linux/kallsyms.h>

static LIST_HEAD(unwind_tables_user);

static const struct unwind_idx *search_index_safe(unsigned long addr,
				       const struct unwind_idx *first,
				       const struct unwind_idx *last)
{
	pr_debug("%s(%08lx, %p, %p, fa:%lx, la:%lx)\n", __func__, addr, first, last,
		em_prel31_to_addr(&first->addr_offset), em_prel31_to_addr(&last->addr_offset));

	if (addr < em_prel31_to_addr(&first->addr_offset)) {
		pr_debug("unwind: Unknown symbol address %08lx\n", addr);
		return NULL;
	} else if (addr >= em_prel31_to_addr(&last->addr_offset))
		return last;

	while (first < last - 1) {
		const struct unwind_idx *mid = first + ((last - first + 1) >> 1);

		if (addr < em_prel31_to_addr(&mid->addr_offset))
			last = mid;
		else
			first = mid;
	}

	return first;
}

static const struct unwind_idx *unwind_find_idx_user(unsigned long addr, struct bt_elf_cache *ep)
{
	const struct unwind_idx *idx = NULL;
	struct unwind_table *table;

	pr_debug("%s(%08lx)\n", __func__, addr);

	/* module unwind tables */

	list_for_each_entry(table, &unwind_tables_user, list) {
		if (addr >= table->begin_addr &&
		    addr < table->end_addr &&
			current->pid == table->pid) {
			unsigned long ofs = addr - ep->be_base + ep->be_vaddr + ep->be_adjust_exidx;
			idx = search_index_safe(ofs, table->start,
					   table->stop - 1);
			pr_debug("check uw tab:%p idx:%p\n", table, idx);
			if (idx) {
				if ((((bt_get_user32(&idx->insn) & 0x80000000) == 0)
					|| ((bt_get_user32(&idx->insn) & 0xff000000) == 0x80000000))
					&& (bt_get_user32(&idx->insn) != 1)) {
					break;
				}else{
					/* maybe unneeded to check multiple mapped area */
					pr_debug("idx %p will fail to unwind\n", table);
				}
			}else{
				pr_debug("cant find idx in table:%p\n", table);
			}
		}
	}

	pr_debug("%s: idx = %p\n", __func__, idx);
	return idx;
}

static unsigned long unwind_get_byte_user(struct unwind_ctrl_block *ctrl)
{
	unsigned long ret;

	if (ctrl->entries <= 0) {
		pr_debug("unwind: Corrupt unwind table\n");
		return 0;
	}

	ret = (bt_get_user32(ctrl->insn) >> (ctrl->byte * 8)) & 0xff;

	if (ctrl->byte == 0) {
		ctrl->insn++;
		ctrl->entries--;
		ctrl->byte = 3;
	} else
		ctrl->byte--;

	return ret;
}


/*
 * Execute the current unwind instruction.
 */
static int unwind_exec_insn_user(struct unwind_ctrl_block *ctrl)
{
	unsigned long insn = unwind_get_byte_user(ctrl);

	pr_debug("%s: insn = %08lx\n", __func__, insn);

	if ((insn & 0xc0) == 0x00)
	{
		pr_debug("insn&c0==0: SP:%lx -> SP+%lx\n", ctrl->vrs[SP], ((insn & 0x3f) << 2)+4);
		ctrl->vrs[SP] += ((insn & 0x3f) << 2) + 4;
	}
	else if ((insn & 0xc0) == 0x40)
	{
		pr_debug("insn&c0==0x40: SP:%lx -> SP-%lx\n", ctrl->vrs[SP], ((insn & 0x3f) << 2)+4);
		ctrl->vrs[SP] -= ((insn & 0x3f) << 2) + 4;
	}
	else if ((insn & 0xf0) == 0x80) {
		unsigned long mask;
		unsigned long *vsp = (unsigned long *)ctrl->vrs[SP];
		int load_sp, reg = 4;

		insn = (insn << 8) | unwind_get_byte_user(ctrl);
		mask = insn & 0x0fff;
		pr_debug("insn&f0==0x80: masked insn:%lx\n", mask);
		if (mask == 0) {
			pr_debug("unwind: 'Refuse to unwind' instruction %04lx\n",
				   insn);
			return -URC_FAILURE;
		}

		/* pop R4-R15 according to mask */
		load_sp = mask & (1 << (13 - 4));
		while (mask) {
			if (mask & 1) {
				pr_debug("vrs[%d]:%p(%lx) <= vsp:%p(%lx)\n", reg, &(ctrl->vrs[reg]), ctrl->vrs[reg], vsp, bt_get_user32(vsp));
				ctrl->vrs[reg] = bt_get_user32(vsp);
				vsp++;
			}
			mask >>= 1;
			reg++;
		}
		if (!load_sp)
		{
			pr_debug("load_sp == 0 vrs[SP](%lx) <= %p\n", ctrl->vrs[SP], vsp);
			ctrl->vrs[SP] = (unsigned long)vsp;
		}
		if (ctrl->vrs[PC] == 0)
			ctrl->vrs[PC] = ctrl->vrs[LR];
	} else if ((insn & 0xf0) == 0x90 &&
		   (insn & 0x0d) != 0x0d)
	{
		ctrl->vrs[SP] = ctrl->vrs[insn & 0x0f];
		pr_debug("insn&0xf0 == 0x90 && ((insn & 0x0d) != 0x0d) vrs[SP](%lx) <= vrs[%lx]:%p\n", ctrl->vrs[SP], insn & 0x0f, &(ctrl->vrs[insn & 0x0f]));
	}
	else if ((insn & 0xf0) == 0xa0) {
		unsigned long *vsp = (unsigned long *)ctrl->vrs[SP];
		int reg;

		/* pop R4-R[4+bbb] */
		for (reg = 4; reg <= 4 + (insn & 7); reg++)
		{
			pr_debug("vrs[%d]:%p(%lx) <= vsp:%p(%lx)\n", reg, &(ctrl->vrs[reg]), ctrl->vrs[reg], vsp, bt_get_user32(vsp));
			ctrl->vrs[reg] = bt_get_user32(vsp);
			vsp++;
		}
		if (insn & 0x8) /* pop R4-R[4+bbb] + R14 if b000 */
		{
			pr_debug("vrs[%d]:%p(%lx) <= vsp:%p(%lx)\n", 14, &(ctrl->vrs[14]), ctrl->vrs[reg], vsp, bt_get_user32(vsp));
			ctrl->vrs[LR] = bt_get_user32(vsp);
			ctrl->vrs[PC] = ctrl->vrs[LR];
			vsp++;
		}

		ctrl->vrs[SP] = (unsigned long)vsp;
	} else if (insn == 0xb0) {
		pr_debug("pop 0xb0 PC:%lx LR:%lx\n", ctrl->vrs[PC], ctrl->vrs[LR]);
		if (ctrl->vrs[PC] == 0)
			ctrl->vrs[PC] = ctrl->vrs[LR];
		/* no further processing */
		ctrl->entries = 0;
	} else if (insn == 0xb1) {
		unsigned long mask = unwind_get_byte_user(ctrl);
		unsigned long *vsp = (unsigned long *)ctrl->vrs[SP];
		int reg = 0;

		if (mask == 0 || mask & 0xf0) {
			pr_debug("unwind: Spare encoding %04lx\n",
			       (insn << 8) | mask);
			return -URC_FAILURE;
		}

		/* pop R0-R3 according to mask */
		while (mask) {
			if (mask & 1)
			{
				pr_debug("vrs[%d]:%p(%lx) <= vsp:%p(%lx)\n", reg, &(ctrl->vrs[14]), ctrl->vrs[reg], vsp, bt_get_user32(vsp));
				ctrl->vrs[reg] = bt_get_user32(vsp);
				vsp++;
			}
			mask >>= 1;
			reg++;
		}
		ctrl->vrs[SP] = (unsigned long)vsp;
	} else if (insn == 0xb2) {
		unsigned long uleb128 = unwind_conv_uleb128(unwind_get_byte_user, ctrl);

		ctrl->vrs[SP] += 0x204 + (uleb128 << 2);
	} else {
		pr_debug("unwind: Unhandled instruction %02lx\n", insn);
		return -URC_FAILURE;
	}

	pr_debug("%s: fp = %08lx sp = %08lx lr = %08lx pc = %08lx\n", __func__,
		 ctrl->vrs[FP], ctrl->vrs[SP], ctrl->vrs[LR], ctrl->vrs[PC]);

	return URC_OK;
}

/*
 * Unwind a single frame starting with *sp for the symbol at *pc. It
 * updates the *pc and *sp with the new values.
 */
int unwind_frame_user(struct stackframe *frame, struct bt_elf_cache *ep, struct unwind_ctrl_block *pctrl)
{
	unsigned long high, low;
	const struct unwind_idx *idx = NULL;
	struct unwind_ctrl_block saved_ctrl;
	int i;

	/* only go to a higher address on the stack */
	low = frame->sp;
	high = ALIGN(low, THREAD_SIZE) + THREAD_SIZE;

	pr_debug("%s(pc = %lx lr = %08lx sp = %08lx) iaddr:%p \n", __func__,
		 frame->pc, frame->lr, frame->sp, pctrl->insn);

	idx = unwind_find_idx_user(frame->pc, ep);
	if (!idx) {
		pr_debug("unwind: Index not found %08lx\n", frame->pc);
		return -URC_FAILURE;
	}

	pr_debug("unwind: found idx user:%p idx->insn:%lx\n",idx, idx->insn);
	pctrl->vrs[FP] = frame->fp;
	pctrl->vrs[SP] = frame->sp;
	pctrl->vrs[LR] = frame->lr;
	pctrl->vrs[PC] = 0;

	if (bt_get_user32(&idx->insn) == 1)
	{
		/* can't unwind */
		pr_debug("unwind: %s:%d cant unwind this function\n", __FUNCTION__, __LINE__);
		return -URC_FAILURE;
	}
	else if ((bt_get_user32(&idx->insn) & 0x80000000) == 0)
	{
		/* prel31 to the unwind table */
		unsigned long prelfix = (unsigned long)em_prel31_to_addr(&idx->insn);
		/* this is extab case extab is mapped with be_base */
		pctrl->insn = (unsigned long *)(prelfix - ep->be_adjust_exidx + ep->be_base - ep->be_vaddr);
		pr_debug("pctrl->insn idx->insn:%p prel31:%p prel31:%lx - adjust:%lx + base:%lx\n", pctrl->insn, &idx->insn, prelfix, ep->be_adjust_exidx, ep->be_base - ep->be_vaddr);
	}
	else if ((bt_get_user32(&idx->insn) & 0xff000000) == 0x80000000)
		/* only personality routine 0 supported in the index */
		pctrl->insn = &idx->insn;
	else {
		pr_debug("unwind: Unsupported personality routine %08lx in the index at %p\n",
			   idx->insn, idx);
		return -URC_FAILURE;
	}

	/* check the personality routine */
	do {
		pr_debug("personality: pctrl->insn:%p *pctrl->insn:%lx\n", pctrl->insn, *pctrl->insn);
		if ((bt_get_user32(pctrl->insn) & 0xff000000) == 0x80000000) {
			pctrl->byte = 2;
			pctrl->entries = 1;
			break;
		} else if ((bt_get_user32(pctrl->insn) & 0xff000000) == 0x81000000) {
			pctrl->byte = 1;
			pctrl->entries = 1 + ((bt_get_user32(pctrl->insn) & 0x00ff0000) >> 16);
			break;
		}else if ((bt_get_user32(&idx->insn) & 0x80000000) == 0) {
			/* prel31 to the unwind table */
			unsigned long prelfix = (unsigned long)em_prel31_to_addr(pctrl->insn);
			prelfix -= (ep->be_base - ep->be_vaddr);
			pr_debug("personality routine offset:%lx insn:%p *insn:%lx\n", prelfix, pctrl->insn, *(pctrl->insn));
			pctrl->insn++;
			pctrl->byte = 2;
			pctrl->entries = 1 + ((bt_get_user32(pctrl->insn) & 0xff000000) >> 24);
			break;
		} else {
			pr_debug("unwind: Unsupported personality routine %08lx at %p\n",
					bt_get_user32(pctrl->insn), pctrl->insn);
			return -URC_FAILURE;
		}
	}while(1);

	for (i = 0; i < 0xf; i++) {
		saved_ctrl.vrs[i] = pctrl->vrs[i];
	}
	while (pctrl->entries > 0) {
		int urc = unwind_exec_insn_user(pctrl);
		if (urc < 0) {
			pr_debug("unwind: %s:%d exec_insn failed\n", __FUNCTION__, __LINE__);
			return urc;
		}
		if (pctrl->vrs[SP] < low || pctrl->vrs[SP] >= high) {
			pr_debug("unwind: %s:%d sp overflow sp:%lx low:%lx high:%lx\n", __FUNCTION__, __LINE__, pctrl->vrs[SP], low, high);
			pctrl->vrs[PC] = 0;
			break;
		}
	}

	if (pctrl->vrs[PC] == 0) {
		pr_debug("unwind: %s:%d unwind failed... restore and use lr\n", __FUNCTION__, __LINE__);
		for (i = 0; i < 0xf; i++) {
			pctrl->vrs[i] = saved_ctrl.vrs[i];
		}
		pctrl->vrs[PC] = pctrl->vrs[LR];
	}

	/* check for infinite loop */
	if (frame->pc == pctrl->vrs[PC])
		return -URC_FAILURE;

	frame->fp = pctrl->vrs[FP];
	frame->sp = pctrl->vrs[SP];
	frame->lr = pctrl->vrs[LR];
	frame->pc = pctrl->vrs[PC];

	return URC_OK;
}


struct unwind_table *unwind_table_add_user(unsigned long start, unsigned long size,
				      unsigned long text_addr,
				      unsigned long text_size)
{
	unsigned long flags;
	struct unwind_table *tab = NULL;
	struct unwind_table *table;
	int newtab = 0;
	pr_debug("%s(%08lx, %08lx, %08lx, %08lx)\n", __func__, start, size,
		 text_addr, text_size);

	raw_spin_lock_irqsave(&unwind_lock, flags);
	list_for_each_entry(table, &unwind_tables_user, list) {
		if(table->begin_addr == text_addr
			&& table->end_addr == (text_addr + text_size)
			&& table->pid == current->pid) {
			tab = table;
			break;
		}
	}
	raw_spin_unlock_irqrestore(&unwind_lock, flags);

	if (!tab) {
		tab = kmalloc(sizeof(*tab), GFP_KERNEL);
		newtab = 1;
	}

	tab->start = (struct unwind_idx *)start;
	tab->stop = (struct unwind_idx *)(start + size);
	tab->begin_addr = text_addr;
	tab->end_addr = text_addr + text_size;
	tab->pid = current->pid;

	if(newtab) {
		raw_spin_lock_irqsave(&unwind_lock, flags);
		list_add_tail(&tab->list, &unwind_tables_user);
		raw_spin_unlock_irqrestore(&unwind_lock, flags);
	}

	return tab;
}

