/*
 * arch/x86/kernel/snsc_backtrace.c
 *
 * Copyright (C) 2013 Sony corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/sched.h>
#include <linux/export.h>
#include <linux/nmi.h>
#include <linux/mm.h>
#include <linux/file.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/snsc_backtrace.h>

#include <asm/mmu_context.h>
#include <asm/kdebug.h>
#include <asm/stacktrace.h>
#include <asm/mman.h>
#include <linux/sched/mm.h>

struct bt_callback_entry {
	bt_callback_t *cb;
	void *uarg;
};

static inline void bt_get_user(void *to, void __user *from, unsigned long len)
{
	mm_segment_t oldfs = get_fs();

	set_fs(KERNEL_DS);
	if (copy_from_user(to, from, len))
		printk("bt_get_user: illegal address %p\n", from);
	set_fs(oldfs);

	return;
}

/* get a stack top address(vm_end) from addr. */
static unsigned long bt_get_stack_top(unsigned long addr)
{
	struct vm_area_struct *vma = NULL;

	down_read(&current->mm->mmap_sem);
	vma = find_vma(current->mm, addr);
	if (!vma
	    || addr < vma->vm_start
	    || vma->vm_end < addr) {
		up_read(&current->mm->mmap_sem);
		return 0;
	}
	up_read(&current->mm->mmap_sem);
	return vma->vm_end;
}

/* get file info (struct file) from addr. also set base as vm_start */
struct file *bt_get_mapped_file(unsigned long addr, unsigned long *base)
{
	struct vm_area_struct *vma = NULL;

	down_read(&current->mm->mmap_sem);
	vma = find_vma(current->mm, addr);
	if (!vma
	    || addr < vma->vm_start
	    || (vma->vm_flags & (VM_READ | VM_EXEC)) != (VM_READ | VM_EXEC)
	    || vma->vm_file == NULL) {
		up_read(&current->mm->mmap_sem);
		return NULL;
	}
	*base = vma->vm_start;
	get_file(vma->vm_file);
	up_read(&current->mm->mmap_sem);
	return vma->vm_file;
}

const char *bt_file_name(struct file *filp)
{
	if (filp && filp->f_path.dentry)
		return filp->f_path.dentry->d_name.name;
	else
		return "<noname>";
}
EXPORT_SYMBOL(bt_file_name);

/*
 * mmap with non-page-aligned offset.  Return similarly unaligned
 * address.
 *
 * No need down_write(&current->mm->mmap_sem).
 * vm_mmap takes the mm semaphore by itself.
 */
void *bt_mmap(struct file *filp, unsigned long offset, unsigned long len)
{
	unsigned long offset_in_page = offset & ~PAGE_MASK;
	unsigned long paged_offset = offset & PAGE_MASK;
	unsigned long ret;

	/* len alignup will be done by do_mmap() */
	ret = vm_mmap(filp, 0, len + offset_in_page,
		      PROT_READ, MAP_PRIVATE, paged_offset);
	if (IS_ERR_VALUE(ret))
		return (void *)ret;
	return (void *)(ret + offset_in_page);
}

int bt_munmap(const void *addr, unsigned long len)
{
	unsigned long offset_in_page = (unsigned long)addr & ~PAGE_MASK;
	unsigned long paged_addr = (unsigned long)addr & PAGE_MASK;

	return do_munmap(current->mm, paged_addr, len + offset_in_page, NULL);
}

void bt_release_elf_cache(struct bt_elf_cache *ep)
{
	if (ep->be_shdr)
		bt_munmap(ep->be_shdr, sizeof(Elf_Shdr) * ep->be_ehdr.e_shnum);
	if (ep->be_shstrtab)
		bt_munmap(ep->be_shstrtab, ep->be_shstrtab_size);
	if (ep->be_strtab)
		bt_munmap(ep->be_strtab, ep->be_strtab_size);
	if (ep->be_symtab)
		bt_munmap(ep->be_symtab, ep->be_symtab_size);
	fput(ep->be_filp);
}

int bt_map_section(struct bt_elf_cache *ep, int idx, void **paddr,
		   unsigned long *psize)
{
	unsigned long size;
	void *addr;
	Elf_Shdr *shdr;
	unsigned long sh_offset;

	shdr = ep->be_shdr + idx;
	bt_get_user(&size, &shdr->sh_size, sizeof(size));
	bt_get_user(&sh_offset, &shdr->sh_offset, sizeof(sh_offset));
	addr = bt_mmap(ep->be_filp, sh_offset, size);
	if (IS_ERR(addr)) {
		pr_debug("backtrace: mmap fail(%ld): %s\n",
			 PTR_ERR(addr), bt_file_name(ep->be_filp));
		return PTR_ERR(addr);
	}
	*paddr = addr;
	*psize = size;

	return 0;
}

/* This function read ELF file, such as sections, to check symbols */
int bt_init_elf_cache(struct bt_session *session,
		      struct bt_elf_cache *ep, struct file *filp,
		      unsigned long base)
{
	int ret;
	struct elfhdr *ehdr = &ep->be_ehdr;
	struct elf_phdr phdr;
	int i, idx_strtab, idx_symtab;
	loff_t offset = 0;
	int progbits_found = 0;

	memset(ep, 0, sizeof(*ep));
	/* checking elf header */
	ep->be_filp = filp;
	ret = kernel_read(filp, (char *)ehdr, sizeof(*ehdr), &offset);
	if (ret != sizeof(*ehdr)) {
		pr_debug("backtrace: too small elf: %s\n",
			 bt_file_name(filp));
		return -EINVAL;
	}
	if (memcmp(ehdr->e_ident, ELFMAG, SELFMAG) != 0) {
		pr_debug("backtrace: not elf: %s\n",
			 bt_file_name(filp));
		return -EINVAL;
	}
	if (ehdr->e_shoff == 0 || ehdr->e_shstrndx == 0) {
		pr_debug("backtrace: no section info in elf: %s\n",
			 bt_file_name(filp));
		return -EINVAL;
	}
	offset = ehdr->e_phoff;
	for (i = 0; i < ehdr->e_phnum; ++i) {
		ret = kernel_read(filp, (char *)&phdr, sizeof(phdr), &offset);
		if (ret != sizeof(phdr)) {
			pr_debug("backtrace: too small elf: %s\n",
				 bt_file_name(filp));
			return -EINVAL;
		}
		if (phdr.p_type == PT_LOAD) { /* first PT_LOAD segment */
			/* This is a text segment */
			ep->be_vaddr = phdr.p_vaddr;
			break;
		}
	}
	/* set vm_start of this region to be_base */
	ep->be_base = base;

	/* reading section header */
	ep->be_shdr = (Elf_Shdr *)bt_mmap(filp, ehdr->e_shoff,
					  sizeof(Elf_Shdr) * ehdr->e_shnum);
	if (IS_ERR(ep->be_shdr)) {
		pr_debug("backtrace: mmap fail: %p\n", ep->be_shdr);
		return PTR_ERR(ep->be_shdr);
	}
	/* map section header string table too */
	ret = bt_map_section(ep, ehdr->e_shstrndx, (void **)&ep->be_shstrtab,
			     &ep->be_shstrtab_size);
	if (ret < 0)
		goto err_out;

	/* checking section names */
	idx_strtab = SHN_UNDEF;
	idx_symtab = SHN_UNDEF;
	for (i = 1; i < ehdr->e_shnum; ++i) { /* skip 0 (SHN_UNDEF) */
		Elf_Shdr *shdr = ep->be_shdr + i;
		char name[32];
		int ret;
		int sh_name;
		int sh_type;
		unsigned long sh_addr;
		bt_get_user(&sh_name, &shdr->sh_name, sizeof(sh_name));
		if (sh_name == 0)
			continue;
		ret = copy_from_user(name, ep->be_shstrtab + sh_name, 32);
		name[31] = '\0';
		if (strcmp(name, ".symtab") == 0)
			idx_symtab = i;
		else if (strcmp(name, ".strtab") == 0)
			idx_strtab = i;
		bt_get_user(&sh_type, &shdr->sh_type, sizeof(sh_type));
		bt_get_user(&sh_addr, &shdr->sh_addr, sizeof(sh_addr));
		if (sh_type == SHT_PROGBITS
		    && !progbits_found
		    && strcmp(name, ".interp")) {
			/* first non-.interp progbits section */
			progbits_found = 1;
			ep->be_progbits_vaddr = sh_addr;
		}
	}

	/* read symbols */
	if (idx_symtab != SHN_UNDEF) {
		ret = bt_map_section(ep, idx_symtab, (void **)&ep->be_symtab,
				     &ep->be_symtab_size);
		if (ret < 0)
			goto err_out;
	}

	if (idx_strtab != SHN_UNDEF) {
		ret = bt_map_section(ep, idx_strtab, (void **)&ep->be_strtab,
				     &ep->be_strtab_size);
		if (ret < 0)
			goto err_out;
	}

	return 0;
err_out:
	bt_release_elf_cache(ep);
	return ret;
}

/* search session's bs_elfs using filp as a key */
static struct bt_elf_cache *bt_find_elf_cache(struct bt_session *session,
					      struct file *filp)
{
	struct bt_elf_cache *ep;
	list_for_each_entry(ep, &session->bs_elfs, be_list) {
		if (ep->be_filp == filp)
			return ep;
	}
	return 0;
}

/*
 * If this elf is already checked, return its cache.
 * Otherwise create a new elf cache and return it.
 *
 * Return previously created elf_cache. If none, create new one.
 *
 * Upon succeeded, filp's refcount (which is assumed to be
 * incremented by the caller) is automatically decremented when
 * session is destroyed, Upon failure, caller need to decrement the
 * refcount.
 */
static struct bt_elf_cache *
bt_find_or_new_elf_cache(struct bt_session *session, struct file *filp,
			 unsigned long base)
{
	int ret;
	struct bt_elf_cache *ep;

	ep = bt_find_elf_cache(session, filp);
	if (ep) {
		/* filp's refcount is doubly incremented, one when ep
		   is registered to the session and one by me.  undo
		   mine. */
		fput(filp);
		return ep;
	}
	ep = kmalloc(sizeof(struct bt_elf_cache), session->bs_memflag);
	if (!ep)
		return ERR_PTR(-ENOMEM);
	ret = bt_init_elf_cache(session, ep, filp, base);
	if (ret < 0) {
		kfree(ep);
		return ERR_PTR(ret);
	}
	/* ret == 0 */
	list_add(&ep->be_list, &session->bs_elfs);
	return ep;
}

void bt_init_session(struct bt_session *session, int memflag)
{
	INIT_LIST_HEAD(&session->bs_elfs);
	session->bs_memflag = memflag;
}
EXPORT_SYMBOL(bt_init_session);

void bt_release_session(struct bt_session *session)
{
	struct bt_elf_cache *ep, *n;
	list_for_each_entry_safe(ep, n, &session->bs_elfs, be_list) {
		bt_release_elf_cache(ep);
		kfree(ep);
	}
}
EXPORT_SYMBOL(bt_release_session);

/*
 * 0: symbol found
 * < 0: symbol not found
 */
int bt_symtab_search(struct bt_elf_cache *ep, unsigned long addr,
		     char *buf, unsigned long buflen,
		     struct bt_arch_callback_arg *cbarg)
{
	Elf_Sym *symtab = ep->be_symtab;
	Elf_Sym *max_ent = NULL;
	unsigned long sym_num = ep->be_symtab_size / sizeof(Elf_Sym);
	unsigned long max = 0;
	unsigned long i, sym_addr;
	unsigned long st_size;

	if (!ep->be_symtab)
		return BT_SYMBOL_NOT_FOUND;

	for (i = 0; i < sym_num; ++i) {
		Elf_Sym *ent = symtab + i;
		unsigned char st_info;
		unsigned long st_value;

		bt_get_user(&st_info, &ent->st_info, sizeof(st_info));
		if (ELF_ST_TYPE(st_info) != STT_FUNC)
			continue;
		bt_get_user(&st_value, &ent->st_value, sizeof(st_value));
		sym_addr = st_value - ep->be_vaddr + ep->be_base;
		if (sym_addr <= addr && sym_addr > max) {
			max = sym_addr;
			max_ent = ent;
		}
	}
	if (!max && !max_ent)
		return -1;

	/*
	 * Now we collect symbol information, such as start address,
	 * in cbarg
	 */
	cbarg->ba_sym_start = max;
	bt_get_user(&st_size, &max_ent->st_size, sizeof(st_size));
	cbarg->ba_sym_size = st_size;
	cbarg->ba_file = ep->be_filp;
	memset(&cbarg->ba_hash, 0, sizeof(cbarg->ba_hash));
	if (ep->be_strtab && buf) {
		int ret; /* warning workaround */
		int st_name;
		bt_get_user(&st_name, &max_ent->st_name, sizeof(st_name));
		ret = copy_from_user(buf, ep->be_strtab + st_name,
				     buflen - 1);
		buf[buflen - 1] = '\0';
	}
	cbarg->ba_adjust = 0;

	return 0;
}

/*
 *   0: success, symbol name (or '\0' if unavailable) set to buf)
 * < 0: error, description (or '\0' if unavailable) set to buf)
 * It is OK to pass NULL as buf.
 */
int bt_find_symbol(unsigned long addr, struct bt_session *session,
		   char *buf, unsigned long buflen,
		   struct bt_arch_callback_arg *cbarg,
		   struct bt_elf_cache **rep)
{
	int ret;
	struct file *filp;
	struct bt_elf_cache *ep;
	unsigned long base;

	if (buf)
		buf[0] = '\0';
	filp = bt_get_mapped_file(addr, &base);
	if (!filp) {
		if (buf)
			snprintf(buf, buflen, "no file associated to addr");
		return -1;
	}
	pr_debug("%s addr:%lx\n", bt_file_name(filp), addr);
	ep = bt_find_or_new_elf_cache(session, filp, base);

	if (IS_ERR(ep)) {
		if (buf)
			snprintf(buf, buflen, "reading elf failed (%s)",
				 bt_file_name(filp));
		fput(filp);
		return PTR_ERR(ep);
	}
	*rep = ep;

	/* using .symtab section */
	ret = bt_symtab_search(ep, addr, buf, buflen, cbarg);

	return ret;
}

EXPORT_SYMBOL(bt_find_symbol);

static unsigned long bt_ustack_unwind_frame(unsigned long *bp, unsigned long *sp,
					    unsigned long stacktop,
					    int *reliable)
{
	unsigned long bytes, return_address, base, next_frame;
	unsigned long __user *fp = (unsigned long *)*bp;
	unsigned long __user *stack = (unsigned long *)*sp;
	unsigned long __user *top = (unsigned long *)stacktop;

	*reliable = 0;
	while (stack < top) {
		bytes = copy_from_user_nmi(&return_address, stack,
					   sizeof(return_address));
		if (bytes != 0)
			return 0;

		if (bt_get_mapped_file(return_address, &base)) {
			if (stack == fp + 1) {
				bytes = copy_from_user_nmi(&next_frame, fp,
							   sizeof(next_frame));
				if (bytes != 0)
					return 0;

				*bp = next_frame;
				*reliable = 1;
			}
			*sp = (unsigned long)++stack;
			return return_address;
		}

		*sp = (unsigned long)++stack;
	}

	return 0;
}

/* Following APIs are architecture specific unwind related APIS. */
unsigned long bt_ustack_unwind(struct bt_session *session, unsigned long pc,
			       unsigned long *bp, unsigned long *sp,
			       unsigned long stacktop, int *reliable,
			       bt_callback_t *cb, void *uarg)
{
	int ret, cbret;
	char buf[256];
	int buflen = 256;
	struct bt_arch_callback_arg cbarg;
	struct bt_elf_cache *ep = NULL;

	/* 1. get a symbol information from ELF file */
	buf[0] = '\0';
	cbarg.ba_extra = 0;
	cbarg.ba_str = buf;
	cbarg.ba_addr = pc;
	ret = bt_find_symbol(pc, session, buf, buflen, &cbarg, &ep);

	/* 2. call cb function and print entry information */
	if (ret == BT_SYMBOL_NOT_FOUND) {
		cbarg.ba_status = BT_STATUS_SUCCESS;
		cbarg.reliable = *reliable;
		cbarg.ba_sym_start = 0x0;
		cbarg.ba_file = ep->be_filp;
		cbarg.ba_adjust = ep->be_base;
		cb(&cbarg, uarg);
	}
	else if (ret < 0) {
		/* symbol finding was failed, but keep unwinding */
		cbarg.ba_status = BT_STATUS_SUCCESS;
		cbarg.reliable = *reliable;
		cbarg.ba_sym_start = 0x0;
		cbarg.ba_file = NULL;
		cbarg.ba_adjust = 0;
		cb(&cbarg, uarg);
	}
	else {
		cbarg.ba_status = BT_STATUS_SUCCESS;
		cbarg.reliable = *reliable;
		cbret = cb(&cbarg, uarg);
	}

	/* 3. unwind a frame */
	return bt_ustack_unwind_frame(bp, sp, stacktop, reliable);
}

int bt_null_callback(struct bt_arch_callback_arg *cbarg, void *uarg)
{
	return 0;
}

int bt_ustack_user(struct bt_session *session,
		   int ba_size, void **ba_buf, void *ba_skip_addr)
{
	unsigned long pc, bp, sp, stacktop;
	struct pt_regs *_regs = task_pt_regs(current);
	int count = 0;
	int skipping = 1;
	int reliable = 1;

	pc = _regs->ip;
	bp = _regs->bp;
	sp = _regs->sp;

	stacktop = bt_get_stack_top(sp);
	if (!stacktop)
		return count;

	if (ba_skip_addr == NULL)
		skipping = 0;

	while (count < ba_size) {
		pc = bt_ustack_unwind(session, pc, &bp, &sp,
				      stacktop, &reliable,
				      bt_null_callback, NULL);
		if (!pc)
			break;
		if (skipping)
			skipping = ((void *)pc != ba_skip_addr);
		if (!skipping) {
			put_user((void *)pc, ba_buf + count);
			count++;
		}
	}
	return count;
}
EXPORT_SYMBOL(bt_ustack_user);

void bt_ustack(const char *mode, int is_atomic,
	       struct pt_regs *_regs, bt_callback_t *cb, void *uarg)
{
	struct bt_session session;
	unsigned long pc, bp, sp, stacktop;
	int reliable = 1;

	if (is_atomic) {
		struct bt_arch_callback_arg cbarg;
		cbarg.ba_status = BT_STATUS_ERROR;
		cbarg.ba_extra = 0;
		cbarg.ba_addr = _regs->ip;
		cbarg.ba_str = "ustack disabled in atomic";
		cb(&cbarg, uarg);
		return;
	}

	pc = _regs->ip;
	bp = _regs->bp;
	sp = _regs->sp;

	stacktop = bt_get_stack_top(sp);
	if (!stacktop) {
		struct bt_arch_callback_arg cbarg;
		cbarg.ba_status = BT_STATUS_ERROR;
		cbarg.ba_extra = 0;
		cbarg.ba_addr = _regs->ip;
		cbarg.ba_str = "ustack fails stack access";
		cb(&cbarg, uarg);
		return;
	}

	/* setup session for openning ELF file */
	bt_init_session(&session, GFP_KERNEL);
	do {
		pc = bt_ustack_unwind(&session, pc, &bp, &sp,
				      stacktop, &reliable, cb, uarg);
	} while (pc);
	/* release session and close ELF file */
	bt_release_session(&session);
}
EXPORT_SYMBOL(bt_ustack);

static int backtrace_stack(void *data, const char *name)
{
	return 0;
}

static int backtrace_address(void *data, unsigned long addr, int reliable)
{
	struct bt_callback_entry *ent;
	struct bt_arch_callback_arg cbarg = {0};

	touch_nmi_watchdog();

	ent = data;
	cbarg.ba_addr = addr;
	cbarg.reliable = reliable;
	ent->cb(&cbarg, ent->uarg);
	return 0;
}

static const struct stacktrace_ops backtrace_ops = {
	.stack = backtrace_stack,
	.address = backtrace_address,
	.walk_stack = print_context_stack,
};

void bt_kstack_regs(struct task_struct *task, struct pt_regs *_regs,
		    bt_callback_t *cb, void *uarg, int funtop_possible)
{
	unsigned long *stack;
	unsigned long bp;
	struct bt_callback_entry ent;
	struct bt_arch_callback_arg cbarg = {0};
	int reliable = 1;

	bp = _regs->bp;
	stack = (unsigned long *)_regs->sp;

	ent.cb = cb;
	ent.uarg = uarg;
	cbarg.ba_addr = _regs->ip;
	cbarg.reliable = reliable;

	cb(&cbarg, uarg);
	dump_trace(task, _regs, stack, bp, &backtrace_ops, &ent);
}

EXPORT_SYMBOL(bt_kstack_regs);

void bt_kstack_current(const char *mode, bt_callback_t *cb, void *uarg)
{
	struct pt_regs _regs = {0};
	struct task_struct *task = current;

	_regs.bp = (unsigned long)get_frame_pointer(current, NULL);
	_regs.sp = (unsigned long)get_stack_pointer(current, NULL);
	_regs.ip = (unsigned long)bt_kstack_current;

	bt_kstack_regs(task, &_regs, cb, uarg, 0);
}
EXPORT_SYMBOL(bt_kstack_current);

/*
 * get a backtrace for a spcified task.
 * fill the uarg buffer with that information.
 * This is used as a generic call gate for getting bt through proc entry.
 *
 * call bt_ustack with regs of specified task.
 */
void bt_ustack_task(struct task_struct *task,
		    const char *mode, int is_atomic,
		    bt_callback_t *cb, void *uarg)
{
	struct pt_regs *regs = task_pt_regs(task);
	struct task_struct *cur = current;
	struct mm_struct *saved_mm, *saved_active_mm, *new_mm;

	if (cur == task) {
		bt_ustack(mode, is_atomic, regs, cb, uarg);
		return;
	}

	new_mm = get_task_mm(task); /* need mmput() below */
	/* sync mm's RSS info before switching mm */
	sync_mm_rss(cur->mm);
	task_lock(cur);
	saved_mm = cur->mm;
	saved_active_mm = cur->active_mm;
	cur->mm = new_mm;
	cur->active_mm = new_mm;
	switch_mm(saved_mm, new_mm, cur);
	task_unlock(cur);

	bt_ustack(mode, is_atomic, regs, cb, uarg);

	/* sync mm's RSS info before switching mm */
	sync_mm_rss(cur->mm);
	task_lock(cur);
	cur->mm = saved_mm;
	cur->active_mm = saved_active_mm;
	switch_mm(new_mm, saved_mm, cur);
	task_unlock(cur);
	mmput(new_mm);
}
EXPORT_SYMBOL(bt_ustack_task);
