// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: file handling
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/version.h>
#include <linux/pagemap.h>
#include <linux/writeback.h>

#include "ssbootfs.h"

/*
 * lock/unlock the file using the mutex in the private data
 * this is to avoid races between VFS and the sysfs or suspend/resume
 * interfaces coming in here.
 *
 * In some cases the kernel can call into a function here
 * while inside another function here. For example release() can
 * trigger writepage(). The context is used to check where the
 * lock is currently held and skip taking the lock if it is
 * safe to continue.
 */
static inline void ssbootfs_file_lock(struct file *file, unsigned context)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;

	mutex_lock(&file_priv->lock);
	atomic_set(&file_priv->lock_context, context);
}

static inline int ssbootfs_file_try_lock(struct file *file, unsigned context)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ret = mutex_trylock(&file_priv->lock);
	if(ret)
		atomic_set(&file_priv->lock_context, context);

	return ret;
}

static inline void ssbootfs_file_unlock(struct file *file)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;

	atomic_set(&file_priv->lock_context, 0);
	mutex_unlock(&file_priv->lock);
}

static inline int ssbootfs_sync_fake_file_from_real_file(struct file *fakefile, struct file *realfile)
{
	int ret = 0;

	vfs_setpos(fakefile, realfile->f_pos, realfile->f_pos);
	fsstack_copy_inode_size(fakefile->f_inode, realfile->f_inode);

	return ret;
}

/*
 * Just check if the real file is open or not.
 * You need to be holding the ssbootfs inode lock.
 */

static int ssbootfs_file_validate_fast(struct file *file)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;

	int ret = 0;

	if (unlikely(!(file_priv->flags & SSBOOTFS_FLAG_VALID)))
		ret = -ESTALE;

	return ret;
}

/*
 * Check if a file processes think is open is actually the same file
 * on disk. This is expensive so the check is only done if the needs_validation
 * flag is set.
 */
static int ssbootfs_file_validate(struct file *file)
{
	struct inode *inode = file->f_inode;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	struct file *realfile;
	int ret;
	loff_t fake_inode_size, real_inode_size;

	ret = ssbootfs_file_validate_fast(file);
	if (ret) {
		realfile = ssbootfs_backingfs_open(file);

		if (IS_ERR(realfile)) {
			ssbootfs_info("couldn't open real file for %s, doesn't exist anymore?\n",
					d_name(file->f_path.dentry));
			goto stale;
		}

		if (!ssbootfs_inode_issame(inode, realfile->f_inode)) {
			ssbootfs_info("inode number or generation has changed on disk for %s(%lu:%u)\n",
					d_name(file->f_path.dentry), inode->i_ino, inode->i_generation);
			goto stale_open;
		}

		fake_inode_size = i_size_read(inode);
		real_inode_size = i_size_read(realfile->f_inode);

		if (fake_inode_size != real_inode_size) {
			ssbootfs_info("inode size has changed on disk for %s(%lu), was %lld now %lld, affected pid %ld\n",
					f_name(file), inode->i_ino, fake_inode_size, real_inode_size, (long) ssbootfs_file_pid(file));
		}

		fsstack_copy_inode_size(inode, realfile->f_inode);

		file_priv->flags |= SSBOOTFS_FLAG_VALID;

		ret = 0;
	}

	goto out;

stale_open:
	ssbootfs_backingfs_close(file);
stale:
	ret = -ESTALE;
out:
	return ret;
}

/*
 * This is a container for the junk we need
 * to create shadow inodes on the fly while
 * doing a look up
 */
struct wrapped_dir_context {
	struct ssbootfs_priv *priv;
	struct super_block *sb;
	struct dir_context *cntx;
	struct dir_context realcntx;
};

static int ssbootfs_fill_dir(struct dir_context *cntx, const char *name, int namelen,
		loff_t offset, u64 ino, unsigned d_type)
{
	struct wrapped_dir_context *wrappedcntx = container_of(cntx,
			struct wrapped_dir_context, realcntx);
	struct dir_context *ssbootfs_cntx = wrappedcntx->cntx;

	/* update the fake state to match the new real state */
	ssbootfs_cntx->pos = cntx->pos;
	/*
	 * This is slightly confusing: dir_emit returns true if
	 * the actor from the realfs returns zero. So we need to
	 * return zero if dir_emit returns true
	 */
	if (dir_emit(ssbootfs_cntx, name, namelen, ino, d_type))
		return 0;

	return -1;
}

static int ssbootfs_inode_dir_op_iterate(struct file *file,
		struct dir_context *cntx)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	struct wrapped_dir_context wrappedcntx = {
			.priv = priv,
			.sb = ssbootfs_sb_from_file(file),
			.cntx = cntx,
			.realcntx = {
				.actor = ssbootfs_fill_dir,
				.pos = cntx->pos,
			},
	};

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_OTHER);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ssbootfs_sync_real_file_from_fake_file(file_priv->fake_file, file_priv->real_file);
	ret = iterate_dir(file_priv->real_file, &wrappedcntx.realcntx);
	ssbootfs_sync_fake_file_from_real_file(file_priv->fake_file, file_priv->real_file);

	/* update the fake state to match the new real state */
	cntx->pos = wrappedcntx.realcntx.pos;

out:
	ssbootfs_file_unlock(file);

	return ret;
}

static int ssbootfs_file_open(struct inode *inode, struct file *file)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(file->f_inode);
	struct ssbootfs_file_priv *file_priv;
	struct file *realfile;
	int ret = 0;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	file_priv = ssbootfs_file_priv_alloc();
	if (!file_priv) {
		return -ENOMEM;
	}

	file->private_data = file_priv;

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_OTHER);

	file_priv->fake_file = file;

	realfile = ssbootfs_backingfs_open(file);
	if (IS_ERR(realfile)) {
		ret = PTR_ERR(realfile);
		goto out;
	}

	SSBOOTFS_SYNC_INODE(inode, realfile->f_inode);

	ssbootfs_inode_lock(inode);
	list_add(&file_priv->i_files, &ssbootfs_inode->files);
	ssbootfs_inode_unlock(inode);

	file_priv->flags |= SSBOOTFS_FLAG_VALID;

out:
	ssbootfs_file_unlock(file);

	/*
	 * The lock for the file is inside of the priv data so
	 * the file needs to be unlocked file otherwise we get
	 * a read after free. The ordering is hard to get right
	 * with labels so check if there was an error and free
	 * before returning if there was.
	 */
	if (ret)
		ssbootfs_file_priv_free(file_priv);

	return ret;
}

/*
 * Force dirty pages to be sent to the backing filesystem.
 * This might cause pages to get written so the caller must hold the
 * file_priv lock with a context that allows the write to happen
 * without a lock or you will get a deadlock.
 */
static void ssbootfs_file_flush_pages(struct ssbootfs_file_priv *file_priv)
{
	struct file *file = file_priv->fake_file;

	if (file_priv->flags & SSBOOTFS_FLAG_MMAPPED_W){
		write_inode_now(file->f_inode, true);
	}
}

static int ssbootfs_file_release(struct inode *inode, struct file *file)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_RELEASE);

	/*
	 * Fast path, if the real file is already closed
	 * don't bother checking it's attributes etc.
	 */
	ret = ssbootfs_file_validate_fast(file);
	if (ret)
		goto out_release;

	/*
	 * mmap is on our side so we must force unwritten pages
	 * out before the real file goes away.
	 */
	ssbootfs_file_flush_pages(file_priv);

	ssbootfs_backingfs_close(file);

out_release:
	ssbootfs_inode_lock(inode);
	list_del(&file_priv->i_files);
	ssbootfs_inode_unlock(inode);

	/*
	 * The private data is free'd outside of the lock,
	 * make it invisible inside the lock. By doing so we
	 * also make it impossible to use ssbotfs_file_unlock()
	 * so clean up the lock manually.
	 */
	file->private_data = NULL;
	atomic_set(&file_priv->lock_context, 0);
	mutex_unlock(&file_priv->lock);

	ssbootfs_file_priv_free(file_priv);

	return 0;
}

static ssize_t ssbootfs_file_read(struct file *file, char __user *buf, size_t sz, loff_t *pos)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	ssize_t ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_READ);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ret = ssbootfs_backingfs_read(file, buf, sz, pos);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

ssize_t ssbootfs_file_splice_read(struct file *in, loff_t *ppos,
				 struct pipe_inode_info *pipe, size_t len,
				 unsigned int flags)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(in);
	ssize_t ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(in, SSBOOTFS_LOCK_CONTEXT_READ);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(in);
	if (ret)
		goto out;

	ret = ssbootfs_backingfs_splice_read(in, ppos, pipe, len, flags);

out:
	ssbootfs_file_unlock(in);

	return ret;
}

static ssize_t ssbootfs_file_write(struct file *file, const char __user *buf, size_t sz, loff_t *pos)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	ssize_t ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_WRITE);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ret = ssbootfs_backingfs_write(file, buf, sz, pos);

	/* the real inode might have changed so sync */
	SSBOOTFS_SYNC_INODE(file->f_inode, file_priv->real_file->f_inode);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

ssize_t ssbootfs_file_splice_write(struct pipe_inode_info *pipe, struct file *out,
		loff_t *ppos, size_t len, unsigned int flags)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(out);
	struct ssbootfs_file_priv *file_priv = out->private_data;
	ssize_t ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(out, SSBOOTFS_LOCK_CONTEXT_WRITE);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(out);
	if (ret)
		goto out;

	ret = ssbootfs_backingfs_splice_write(pipe, out, ppos, len, flags);

	/* the real inode might have changed so sync */
	SSBOOTFS_SYNC_INODE(out->f_inode, file_priv->real_file->f_inode);

out:
	ssbootfs_file_unlock(out);

	return ret;
}

static int ssbootfs_file_fsync(struct file *file, loff_t x, loff_t y, int datasync)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_FSYNC);

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ssbootfs_file_flush_pages(file_priv);

	ret = ssbootfs_backingfs_fsync_range(file, x, y, datasync);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

loff_t ssbootfs_file_llseek(struct file *file, loff_t offset, int whence)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	loff_t ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_OTHER);

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ssbootfs_backingfs_llseek(file, offset, whence);

	ret = generic_file_llseek(file, offset, whence);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

/*
 * This returns the number of files that got disengaged. Because we
 * are disengaging a file at a time this will be between 0 and 1.
 */
int ssbootfs_file_disengage(struct ssbootfs_priv *priv, struct file *file)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret = 0;
	int mmapped = (int)(file_priv->flags & SSBOOTFS_FLAG_MMAPPED ? 1 : 0);
	int mmappedw = (int)(file_priv->flags & SSBOOTFS_FLAG_MMAPPED_W ? 1 : 0);

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	int lock_context = atomic_read(&file_priv->lock_context);
	if (lock_context) {
		ssbootfs_err("Dangling lock: %d\n", lock_context);
		BUG();
	}
#endif

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_DISENGAGE);

	if (!(file_priv->flags & SSBOOTFS_FLAG_VALID))
		goto out;

	ssbootfs_debug_info("closing file %s, mmapped %d (writeable %d)\n", f_name(file),
			mmapped, mmappedw);
	/*
	 * If there are unwritten pages here we will need the inode lock
	 * to find the file to write into. So unlock, flush
	 * and relock it..
	 */
	ssbootfs_inode_unlock(file->f_inode);
	ssbootfs_file_flush_pages(file_priv);
	ssbootfs_inode_lock(file->f_inode);
	/* Probably not need */
	ssbootfs_backingfs_fsync(file, 0);
	ssbootfs_backingfs_close(file);

	file_priv->flags &= ~SSBOOTFS_FLAG_VALID;
	ret = 1;

out:
	ssbootfs_file_unlock(file);

	return ret;
}

static int ssbootfs_file_writepage(struct page *page, struct writeback_control *wbc)
{
	struct inode *inode = page->mapping->host;
	struct ssbootfs_priv *priv = ssbootfs_priv_from_inode(inode);
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
	struct ssbootfs_file_priv *file_priv = NULL;
	struct file *file = NULL;
	bool unlocked = true;
	int ret;

	/*
	 * We need a struct file to write into.
	 * This is a bit of a problem:
	 *
	 * We might have multiple mappings of the same file but we
	 * only have one private data pointer in struct page so we keep the
	 * last file right? What happens if the last file gets closed??
	 * We would end up using a struct file that is gone.
	 *
	 * We just need a struct file that points to the right inode.
	 * We have a list of files for each inode. So just get one of them
	 * and use that to write into the inode. For almost every case
	 * there is only one open file so this only applies to the odd case
	 * where two processes have the same file mmap'd and the mapping is
	 * shared.
	 *
	 * Anyhow, go through the open file list and find a file that is
	 * mmapped r/w and is either waiting for writepage() or is unlocked.
	 */

	ssbootfs_inode_lock(inode);

	WARN_ON(list_empty(&ssbootfs_inode->files));
	/*
	 * The file might disappear under us, so either this file
	 * needs to be expecting writepage() or not be locked somewhere
	 * else i.e in release()
	 */
	/* prefer an unlocked file */
	list_for_each_entry(file_priv, &ssbootfs_inode->files, i_files){
		if (file_priv->flags & SSBOOTFS_FLAG_MMAPPED_W){
			unlocked = atomic_read(&file_priv->lock_context) & SSBOOTFS_LOCK_CONTEXT_MMAPWSAFE_MASK;
			if (unlocked) {
				file = file_priv->fake_file;
				break;
			}
		}
	}
	/* last resort */
	if (!unlocked) {
		list_for_each_entry(file_priv, &ssbootfs_inode->files, i_files){
			if (file_priv->flags & SSBOOTFS_FLAG_MMAPPED_W){
				if (ssbootfs_file_try_lock(file_priv->fake_file, SSBOOTFS_LOCK_CONTEXT_OTHER)) {
					file = file_priv->fake_file;
					break;
				}
			}
		}
	}
	ssbootfs_inode_unlock(inode);
	/*
	 * These are really BUG_ON situations but the kernel will
	 * just explode if we use BUG_ON here.
	 */
	if (!file_priv || !file) {
		WARN_ON_ONCE(1);
		ret = -EINVAL;
		unlocked = true;
		goto out;
	}

	if (unlocked)
		goto file_locked;

	/*
	 * The barrier must be taken here and not before otherwise
	 * you will get a deadlock disenganging the mounts.
	 * Do not move this!
	 */
	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;
file_locked:
	ret = ssbootfs_backingfs_file_writepage(file, page);
	WARN_ONCE(ret, "writing page to backing fs failed: %d\n", ret);
	/* the real inode might have changed so sync */
	SSBOOTFS_SYNC_INODE(inode, file_priv->real_file->f_inode);

out:
	/*
	 * we must always mark the page status and unlock it so
	 * we do this here and not in backingfs
	 */
	if (ret)
		ClearPageUptodate(page);
	else
		ClearPageError(page);

	unlock_page(page);

	if (unlocked)
		goto file_unlocked;

	ssbootfs_file_unlock(file);

file_unlocked:
	return ret;
}

static int ssbootfs_file_readpage(struct file *file, struct page *page)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	bool unlocked = atomic_read(&file_priv->lock_context) & SSBOOTFS_LOCK_CONTEXT_WRITE;
	int ret;

	ssbootfs_barrier(priv);

	/*
	 * This is special case handling for concurrent vfs_write and a mmap
	 * read page fault.
	 */
	if (unlocked)
		goto readpage;

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_OTHER);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out_updatepage;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out_updatepage;

readpage:
	ret = ssbootfs_backingfs_file_readpage(file, page);
	/* the real inode might have changed so sync */
	SSBOOTFS_SYNC_INODE(file->f_inode, file_priv->real_file->f_inode);

out_updatepage:
	/*
	 * we must always mark the page status and unlock it so
	 * we do this here and not in backingfs
	 */
	if (ret)
		ClearPageUptodate(page);
	else
		SetPageUptodate(page);
	unlock_page(page);

	if(!unlocked)
		ssbootfs_file_unlock(file);

	return ret;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 17, 0)
/*
 * For kernels older than 4.17 libfs.c doesn't contain noop_direct_IO
 * so provide it here.
 */
ssize_t noop_direct_IO(struct kiocb *iocb, struct iov_iter *iter)
{
	return -EINVAL;
}
#endif

const struct address_space_operations ssbootfs_file_aops = {
	.writepage		= ssbootfs_file_writepage,
	.readpage		= ssbootfs_file_readpage,
	.set_page_dirty		= __set_page_dirty_nobuffers,
	/* For O_DIRECT dentry_open() checks f_mapping->a_ops->direct_IO */
	.direct_IO		= noop_direct_IO,
};

static int ssbootfs_file_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_OTHER);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	file_priv->flags |= SSBOOTFS_FLAG_MMAPPED;
	/* This is based on the check the kernel does in generic_file_readonly_mmap() */
	if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE))
		file_priv->flags |= SSBOOTFS_FLAG_MMAPPED_W;

	ret = generic_file_mmap(file, vma);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

static int ssbootfs_file_flush(struct file *file, fl_owner_t id)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ssbootfs_barrier(priv);

	ssbootfs_file_lock(file, SSBOOTFS_LOCK_CONTEXT_FLUSH);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		ret = -EIO;
		goto out;
	}

	ret = ssbootfs_file_validate(file);
	if (ret)
		goto out;

	ssbootfs_file_flush_pages(file_priv);

	ret = ssbootfs_backingfs_flush(file, id);

out:
	ssbootfs_file_unlock(file);

	return ret;
}

const struct file_operations ssbootfs_dir_fops = {
	.open			= ssbootfs_file_open,
	.release		= ssbootfs_file_release,
	.read			= generic_read_dir,
	.iterate		= ssbootfs_inode_dir_op_iterate,
	.llseek			= generic_file_llseek,
	.fsync			= ssbootfs_file_fsync,
};

const struct file_operations ssbootfs_file_fops = {
	.open			= ssbootfs_file_open,
	.read			= ssbootfs_file_read,
	.write			= ssbootfs_file_write,
	.mmap			= ssbootfs_file_mmap,
	.flush			= ssbootfs_file_flush,
	.release		= ssbootfs_file_release,
	.fsync			= ssbootfs_file_fsync,
	.llseek			= ssbootfs_file_llseek,
/*
 * For now the kernel emulations of these is good enough.
 * The splice operations are implemented but not well tested.
 * If a performance need arises enabling these instead of using
 * the emulations would be a good idea as long as they are
 * properly tested.
 */
#if 0
	.splice_read		= ssbootfs_file_splice_read,
	.splice_write		= ssbootfs_file_splice_write,
	.read_iter		= generic_file_read_iter,
	.write_iter		= generic_file_write_iter,
#endif
};

const struct file_operations ssbootfs_symlink_fops = {

};
