// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: filesystem registration, cleanup
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/idr.h>
#include <linux/vfs.h>
#include <linux/module.h>
#include <linux/magic.h>
#include <linux/namei.h>
#include <linux/slab.h>
#include <linux/parser.h>
#include <linux/seq_file.h>
#include <linux/version.h>

#include "ssbootfs.h"
#include "sysfs.h"

static DEFINE_IDA(ssbootfs_bdi_ida);

static int ssbootfs_statfs(struct dentry *dentry, struct kstatfs *buf)
{
	struct super_block *sb = dentry->d_sb;
	struct ssbootfs_priv *priv = sb->s_fs_info;
	int ret;

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ret = vfs_statfs(&priv->backingfs, buf);

	/* fix up the backing fs type */
	buf->f_type = SSBOOTFS_MAGIC;

	return ret;
}

/* custom inode allocation */
static struct kmem_cache *ssbootfs_inode_cachep;

static void ssbootfs_inode_init_once(void *inode)
{
	struct ssbootfs_inode *ssbootfs_inode = inode;

	inode_init_once(&ssbootfs_inode->vfs_inode);
	mutex_init(&ssbootfs_inode->lock);
	INIT_LIST_HEAD(&ssbootfs_inode->files);
}

static struct inode *ssbootfs_alloc_inode(struct super_block *sb)
{
	struct ssbootfs_inode *fi = kmem_cache_alloc(ssbootfs_inode_cachep, GFP_KERNEL);

	return &fi->vfs_inode;
}

/*
 * TODO	free is for the RCU safe part of freeing the inode...
 * maybe the destroy bit could go in here too?
 */
static void ssbootfs_free_inode(struct inode *inode)
{
	struct ssbootfs_inode *fi = SSBOOTFS_I_TO_SSBFSI(inode);

	kmem_cache_free(ssbootfs_inode_cachep, fi);
}

static void ssbootfs_destroy_inode(struct inode *inode)
{
	ssbootfs_mapping_invalidate_inode(inode);
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 1, 0)
	ssbootfs_free_inode(inode);
#endif
}

/* file private allocation */
static struct kmem_cache *ssbootfs_file_priv_cache;

static void ssbootfs_file_priv_init_once(void *fp)
{
	struct ssbootfs_file_priv *file_priv = fp;

	mutex_init(&file_priv->lock);
	INIT_LIST_HEAD(&file_priv->i_files);
	file_priv->flags = 0;
}

struct ssbootfs_file_priv *ssbootfs_file_priv_alloc(void)
{
	struct ssbootfs_file_priv *file_priv = kmem_cache_alloc(ssbootfs_file_priv_cache, GFP_KERNEL);

	return file_priv;
}

void ssbootfs_file_priv_free(struct ssbootfs_file_priv *file_priv)
{
	kmem_cache_free(ssbootfs_file_priv_cache, file_priv);
}

static void ssbootfs_put_super(struct super_block *sb)
{
	struct ssbootfs_priv *priv = sb->s_fs_info;

	if (priv) {
		ssbootfs_disengage_sb(priv);
		ssbootfs_detach(priv, true);
		ssbootfs_sysfs_destroy_mount(priv);
		kfree(priv->name);
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
		kfree(priv->backingdev);
		kfree(priv->backingtype);
#endif
		kfree(priv);
	}
}

static int ssbootfs_show_options(struct seq_file *m, struct dentry *dentry)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);

	seq_show_option(m, "name", priv->name);

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	seq_show_option(m, "backingdev", priv->backingdev);
	seq_show_option(m, "backingtype", priv->backingtype);
	if(strlen(priv->backingopts))
		seq_show_option(m, "backingopts", priv->backingopts);
#endif

	return 0;
}

static struct super_operations ssbootfs_ops = {
	.alloc_inode	= ssbootfs_alloc_inode,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 1, 0)
	.free_inode	= ssbootfs_free_inode,
#endif
	.destroy_inode	= ssbootfs_destroy_inode,
	.drop_inode	= generic_delete_inode,
	.statfs		= ssbootfs_statfs,
	.put_super	= ssbootfs_put_super,
	.show_options	= ssbootfs_show_options,
};

enum {
	OPT_NAME = 1,
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	OPT_BACKINGDEV,
	OPT_BACKINGTYPE,
#endif
	OPT_ERR,
};

static const match_table_t ssbootfs_tokens = {
	{OPT_NAME,		"name=%s"},
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	{OPT_BACKINGDEV,	"backingdev=%s"},
	{OPT_BACKINGTYPE,	"backingtype=%s"},
#endif
	{OPT_ERR,		NULL}
};

static int ssbootfs_parse_opts(struct ssbootfs_priv *priv, char *opts)
{
	int ret = 0;
	char *p;
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	char *backingopts = priv->backingopts, *lastcomma;
	size_t backingoptssz = sizeof(priv->backingopts);
#endif
	/*
	 * match_token will give us up to 3 arguments for an option,
	 * we only ever use the first
	 */
	substring_t args[MAX_OPT_ARGS] = { 0 };
	int token;

	if (!opts)
		return -EINVAL;

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	*backingopts = '\0';
#endif

	while ((p = strsep(&opts, ",")) != NULL) {
		if ((token = match_token(p, ssbootfs_tokens, args))) {
			switch (token) {
			case OPT_NAME:
				priv->name = match_strdup(&args[0]);
				break;
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
			case OPT_BACKINGDEV:
				priv->backingdev = match_strdup(&args[0]);
				break;
			case OPT_BACKINGTYPE:
				priv->backingtype = match_strdup(&args[0]);
				break;
			/*
			 * Anything we don't consume here goes through to the backing
			 * filesystem.
			 */
			case OPT_ERR:
				if ((strlcat(backingopts, p, backingoptssz) >= backingoptssz) ||
				    (strlcat(backingopts, ",", backingoptssz) >= backingoptssz)) {
					ret = -ENOMEM;
					goto out;
				}
				break;
#else
			case OPT_ERR:
				ret = -EINVAL;
				goto out;
#endif
			}
		}
	}

	if (!priv->name) {
		ssbootfs_err("name must be passed\n");
		ret = -EINVAL;
		goto out;
	}

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	if ((priv->backingdev && !priv->backingtype) ||
	    (!priv->backingdev && priv->backingtype)) {
		ssbootfs_err("Either both or neither backingdev and backingtype need to be specified\n");
		ret = -EINVAL;
		goto out;
	}

	/*
	 * Creating a mount without a backing dev is used for tests,
	 * but not for normal usage with the in kernel mount support
	 * so tell the user if that's what's going to happen.
	 */
	if (!priv->backingdev) {
		ssbootfs_warn("No backing dev supplied, automounting disabled\n");
	}

	/* Remove a dangling comma on the backing options if there is one */
	lastcomma = strrchr(backingopts, ',');
	if (lastcomma)
		*lastcomma = '\0';
#endif

out:
	return ret;
}

static int ssbootfs_fill_super(struct super_block *sb, void *data, int flags)
{
	int ret = 0;
	struct ssbootfs_priv *priv;

	priv = kzalloc(sizeof(*priv), GFP_KERNEL);

	if (!priv) {
		ret = -ENOMEM;
		goto out;
	}

	mutex_init(&priv->lock);
	init_rwsem(&priv->tree_rw_sem);
	atomic_set(&priv->token, 0);

	ret = ssbootfs_parse_opts(priv, data);
	if (ret)
		goto err_free_priv;

	sb->s_fs_info = priv;
	priv->sb = sb;

	/*
	 * Fill out the static bits this should come before making the
	 * root dentry so that it's inode is allocated in the right
	 * place
	 */
	sb->s_magic = SSBOOTFS_MAGIC;
	sb->s_op = &ssbootfs_ops;
	/* TODO extended attribute handling */
	//sb->s_xattr
	sb->s_d_op = &ssbootfs_dentry_ops;

	/* create a root dentry */
	sb->s_root = d_make_root(ssbootfs_inode_new_fake_inode(sb, 0, 0, S_IFDIR));
	if (!sb->s_root) {
		ret = -ENOMEM;
		goto err_free_priv;
	}

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	/* allow for some control of processes coming in */
	sb->s_root->d_flags |= DCACHE_MANAGE_TRANSIT;
#endif

	ret = ssbootfs_sysfs_init_mount(priv);
	if (ret)
		goto err_free_priv;

	priv->bdi_id = ida_simple_get(&ssbootfs_bdi_ida, 0, 0, GFP_KERNEL);
	if (priv->bdi_id < 0) {
		ret = priv->bdi_id;
		goto err_free_priv;
	}

	ret = super_setup_bdi_name(sb, SSBOOTFS_NAME"-%d", priv->bdi_id);
	if (ret)
		goto err_free_priv;

	return 0;

err_free_priv:
	sb->s_fs_info = NULL;
	kfree(priv->name);
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	kfree(priv->backingdev);
	kfree(priv->backingtype);
#endif
	kfree(priv);
out:
	return ret;
}

static struct dentry *ssbootfs_mount(struct file_system_type *fs_type, int flags,
				const char *dev_name, void *raw_data)
{
	return mount_nodev(fs_type, flags, raw_data, ssbootfs_fill_super);
}

static struct file_system_type ssbootfs_fs_type = {
	.owner		= THIS_MODULE,
	.name		= SSBOOTFS_NAME,
	.mount		= ssbootfs_mount,
	.kill_sb	= kill_anon_super,
};
MODULE_ALIAS_FS(SSBOOTFS_NAME);

static int __init ssbootfs_init(void)
{
	int ret;

	ssbootfs_inode_cachep = kmem_cache_create("ssbootfs_inode_cache",
					     sizeof(struct ssbootfs_inode), 0,
					     (SLAB_RECLAIM_ACCOUNT |
					      SLAB_MEM_SPREAD |
					      SLAB_ACCOUNT),
					      ssbootfs_inode_init_once);
	if (!ssbootfs_inode_cachep)
		return -ENOMEM;

	ssbootfs_file_priv_cache = kmem_cache_create("ssbootfs_file_priv_cache",
					     sizeof(struct ssbootfs_file_priv), 0,
					     (SLAB_RECLAIM_ACCOUNT |
					      SLAB_MEM_SPREAD |
					      SLAB_ACCOUNT),
					      ssbootfs_file_priv_init_once);
	if (!ssbootfs_inode_cachep){
		ret = -ENOMEM;
		goto out_free_inode;
	}

	ret = register_filesystem(&ssbootfs_fs_type);
	if (ret)
		goto out_free_file_priv;

	ret = ssbootfs_sysfs_init();
	if (ret)
		goto out_unregister;

	return 0;

out_unregister:
	unregister_filesystem(&ssbootfs_fs_type);
out_free_file_priv:
	kmem_cache_destroy(ssbootfs_file_priv_cache);
out_free_inode:
	kmem_cache_destroy(ssbootfs_inode_cachep);
	return ret;
}

static void __exit ssbootfs_exit(void)
{
	kmem_cache_destroy(ssbootfs_file_priv_cache);
	kmem_cache_destroy(ssbootfs_inode_cachep);
	ssbootfs_sysfs_exit();
	unregister_filesystem(&ssbootfs_fs_type);
}

module_init(ssbootfs_init);
module_exit(ssbootfs_exit);

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION(SSBOOTFS_NAME" filesystem");
MODULE_LICENSE("GPL v2");
