/*
 * (C) Copyright 2000-2002
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <console.h>
#include <div64.h>
#include <inttypes.h>
#include <version.h>
#include <linux/ctype.h>
#include <asm/io.h>

char *display_options_get_banner_priv(bool newlines, const char *build_tag,
				      char *buf, int size)
{
	int len;

	len = snprintf(buf, size, "%s%s", newlines ? "\n\n" : "",
		       version_string);
	if (build_tag && len < size)
		len += snprintf(buf + len, size - len, ", Build: %s",
				build_tag);
	if (len > size - 3)
		len = size - 3;
	strcpy(buf + len, "\n\n");

	return buf;
}

#ifndef BUILD_TAG
#define BUILD_TAG NULL
#endif

char *display_options_get_banner(bool newlines, char *buf, int size)
{
	return display_options_get_banner_priv(newlines, BUILD_TAG, buf, size);
}

int display_options(void)
{
#ifndef __CONFIG_ZYNQMP_ENC185_H

	char buf[DISPLAY_OPTIONS_BANNER_LENGTH];

	display_options_get_banner(true, buf, sizeof(buf));
	printf("%s", buf);
#else
#pragma message "display_options.c mode ENC185"
	// banjo
	int i;
	// PMUFW version
        VersionInfo *pmufw_ocm_ver_info;
        VersionInfo *pmufw_ver_info;

        pmufw_ocm_ver_info = (VersionInfo *)VERSION_INFO_Q_OCM_ADDR;
	pmufw_ver_info     = (VersionInfo *)SW_PMUFW_VER_INFO_ADDRESS;

	pmufw_ver_info->mMajorVer = pmufw_ocm_ver_info->mMajorVer;
	pmufw_ver_info->mMinorVer = pmufw_ocm_ver_info->mMinorVer;
	pmufw_ver_info->mUserCode = pmufw_ocm_ver_info->mUserCode;
	pmufw_ver_info->mIndex 	  = pmufw_ocm_ver_info->mIndex;

	pmufw_ver_info->mYearH  = pmufw_ocm_ver_info->mYearH;
	pmufw_ver_info->mYearL  = pmufw_ocm_ver_info->mYearL;
	pmufw_ver_info->mMonth  = pmufw_ocm_ver_info->mMonth;
	pmufw_ver_info->mDay   	= pmufw_ocm_ver_info->mDay;

	pmufw_ver_info->mHour  	= pmufw_ocm_ver_info->mHour;
	pmufw_ver_info->mMinute = pmufw_ocm_ver_info->mMinute;

	for(i=0;i<6;i++) pmufw_ver_info->mReserved[i] = 0x00;

        printf("SYS0 [pmufw ] Ver %02X.%02X.%02X%02X (SDK Ver %02X%02X.%01X) %02X%02X/%02X/%02X %02X:%02X===\n\r",
                pmufw_ver_info->mMajorVer, pmufw_ver_info->mMinorVer, pmufw_ver_info->mUserCode, pmufw_ver_info->mIndex, 
		pmufw_ocm_ver_info->mReserved[0], pmufw_ocm_ver_info->mReserved[1], pmufw_ocm_ver_info->mReserved[2],
                pmufw_ver_info->mYearH, pmufw_ver_info->mYearL, pmufw_ver_info->mMonth, pmufw_ver_info->mDay, 
		pmufw_ver_info->mHour, pmufw_ver_info->mMinute );

	// ATF version
        VersionInfo *atf_mem_ver_info;
        VersionInfo *atf_ver_info;

        atf_mem_ver_info = (VersionInfo *)VERSION_INFO_ATF_ADDR;
        atf_ver_info     = (VersionInfo *)SW_ATF_VER_INFO_ADDRESS;

        atf_ver_info->mMajorVer = atf_mem_ver_info->mMajorVer;
        atf_ver_info->mMinorVer = atf_mem_ver_info->mMinorVer;
        atf_ver_info->mUserCode = atf_mem_ver_info->mUserCode;
        atf_ver_info->mIndex    = atf_mem_ver_info->mIndex;

        atf_ver_info->mYearH  = atf_mem_ver_info->mYearH;
        atf_ver_info->mYearL  = atf_mem_ver_info->mYearL;
        atf_ver_info->mMonth  = atf_mem_ver_info->mMonth;
        atf_ver_info->mDay    = atf_mem_ver_info->mDay;

        atf_ver_info->mHour   = atf_mem_ver_info->mHour;
        atf_ver_info->mMinute = atf_mem_ver_info->mMinute;

        for(i=0;i<6;i++) atf_ver_info->mReserved[i] = 0x00;

	int j;
	VersionInfo *ver_info;

	ver_info = (VersionInfo *)SW_UBOOT_VER_INFO_ADDRESS;

	ver_info->mMajorVer = SW_VER_MAJ;
	ver_info->mMinorVer = SW_VER_MIN;
	ver_info->mUserCode = SW_VER_USR;
	ver_info->mIndex    = SW_VER_INDEX;
	ver_info->mYearH    = SW_VER_YEAR_H;
	ver_info->mYearL    = SW_VER_YEAR_L;
	ver_info->mMonth    = SW_VER_MONTH;
	ver_info->mDay      = SW_VER_DAY;
	ver_info->mHour     = SW_VER_HOUR;
	ver_info->mMinute   = SW_VER_MINUTE;

	for(j=0;j<6;j++) ver_info->mReserved[j] = 0x00;

	printf("SYS0 [u-boot] Ver %02X.%02X.%02X%02X (SDK Ver %02X%02X.%01X) %02X%02X/%02X/%02X %02X:%02X===\n\r",
		SW_VER_MAJ, SW_VER_MIN, SW_VER_USR, SW_VER_INDEX, SW_SDK_YEAR_H, SW_SDK_YEAR_L, SW_SDK_VER,
		SW_VER_YEAR_H, SW_VER_YEAR_L, SW_VER_MONTH, SW_VER_DAY, SW_VER_HOUR, SW_VER_MINUTE );
#endif
	return 0;
}

void print_freq(uint64_t freq, const char *s)
{
	unsigned long m = 0;
	uint32_t f;
	static const char names[] = {'G', 'M', 'K'};
	unsigned long d = 1e9;
	char c = 0;
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(names); i++, d /= 1000) {
		if (freq >= d) {
			c = names[i];
			break;
		}
	}

	if (!c) {
		printf("%" PRIu64 " Hz%s", freq, s);
		return;
	}

	f = do_div(freq, d);

	/* If there's a remainder, show the first few digits */
	if (f) {
		m = f;
		while (m > 1000)
			m /= 10;
		while (m && !(m % 10))
			m /= 10;
		if (m >= 100)
			m = (m / 10) + (m % 100 >= 50);
	}

	printf("%lu", (unsigned long) freq);
	if (m)
		printf(".%ld", m);
	printf(" %cHz%s", c, s);
}

void print_size(uint64_t size, const char *s)
{
	unsigned long m = 0, n;
	uint64_t f;
	static const char names[] = {'E', 'P', 'T', 'G', 'M', 'K'};
	unsigned long d = 10 * ARRAY_SIZE(names);
	char c = 0;
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(names); i++, d -= 10) {
		if (size >> d) {
			c = names[i];
			break;
		}
	}

	if (!c) {
		printf("%" PRIu64 " Bytes%s", size, s);
		return;
	}

	n = size >> d;
	f = size & ((1ULL << d) - 1);

	/* If there's a remainder, deal with it */
	if (f) {
		m = (10ULL * f + (1ULL << (d - 1))) >> d;

		if (m >= 10) {
			m -= 10;
			n += 1;
		}
	}

	printf ("%lu", n);
	if (m) {
		printf (".%ld", m);
	}
	printf (" %ciB%s", c, s);
}

#define MAX_LINE_LENGTH_BYTES (64)
#define DEFAULT_LINE_LENGTH_BYTES (16)
int print_buffer(ulong addr, const void *data, uint width, uint count,
		 uint linelen)
{
	/* linebuf as a union causes proper alignment */
	union linebuf {
#ifdef CONFIG_SYS_SUPPORT_64BIT_DATA
		uint64_t uq[MAX_LINE_LENGTH_BYTES/sizeof(uint64_t) + 1];
#endif
		uint32_t ui[MAX_LINE_LENGTH_BYTES/sizeof(uint32_t) + 1];
		uint16_t us[MAX_LINE_LENGTH_BYTES/sizeof(uint16_t) + 1];
		uint8_t  uc[MAX_LINE_LENGTH_BYTES/sizeof(uint8_t) + 1];
	} lb;
	int i;
#ifdef CONFIG_SYS_SUPPORT_64BIT_DATA
	uint64_t __maybe_unused x;
#else
	uint32_t __maybe_unused x;
#endif

	if (linelen*width > MAX_LINE_LENGTH_BYTES)
		linelen = MAX_LINE_LENGTH_BYTES / width;
	if (linelen < 1)
		linelen = DEFAULT_LINE_LENGTH_BYTES / width;

	while (count) {
		uint thislinelen = linelen;
		printf("%08lx:", addr);

		/* check for overflow condition */
		if (count < thislinelen)
			thislinelen = count;

		/* Copy from memory into linebuf and print hex values */
		for (i = 0; i < thislinelen; i++) {
			if (width == 4)
				x = lb.ui[i] = *(volatile uint32_t *)data;
#ifdef CONFIG_SYS_SUPPORT_64BIT_DATA
			else if (width == 8)
				x = lb.uq[i] = *(volatile uint64_t *)data;
#endif
			else if (width == 2)
				x = lb.us[i] = *(volatile uint16_t *)data;
			else
				x = lb.uc[i] = *(volatile uint8_t *)data;
#ifdef CONFIG_SYS_SUPPORT_64BIT_DATA
			printf(" %0*llx", width * 2, (long long)x);
#else
			printf(" %0*x", width * 2, x);
#endif
			data += width;
		}

		while (thislinelen < linelen) {
			/* fill line with whitespace for nice ASCII print */
			for (i=0; i<width*2+1; i++)
				puts(" ");
			linelen--;
		}

		/* Print data in ASCII characters */
		for (i = 0; i < thislinelen * width; i++) {
			if (!isprint(lb.uc[i]) || lb.uc[i] >= 0x80)
				lb.uc[i] = '.';
		}
		lb.uc[i] = '\0';
		printf("    %s\n", lb.uc);

		/* update references */
		addr += thislinelen * width;
		count -= thislinelen;

		if (ctrlc())
			return -1;
	}

	return 0;
}
