/*
 * drivers/misc/pmon.c
 *
 * Performance Monitor driver
 *
 * Copyright 2010,2011,2012 Sony Corporation
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 * This code is based on arch/arm/oprofile/op_model_mpcore.c
 */
/**
 * @file op_model_mpcore.c
 * MPCORE Event Monitor Driver
 * @remark Copyright 2004 ARM SMP Development Team
 * @remark Copyright 2000-2004 Deepak Saxena <dsaxena@mvista.com>
 * @remark Copyright 2000-2004 MontaVista Software Inc
 * @remark Copyright 2004 Dave Jiang <dave.jiang@intel.com>
 * @remark Copyright 2004 Intel Corporation
 * @remark Copyright 2004 Zwane Mwaikambo <zwane@arm.linux.org.uk>
 * @remark Copyright 2004 Oprofile Authors
 *
 * @remark Read the file COPYING
 *
 * @author Zwane Mwaikambo
 *
 *  Counters:
 *    0: PMN0 on CPU0, per-cpu configurable event counter
 *    1: PMN1 on CPU0, per-cpu configurable event counter
 *    2: CCNT on CPU0
 *    3: PMN0 on CPU1
 *    4: PMN1 on CPU1
 *    5: CCNT on CPU1
 *    6: PMN0 on CPU1
 *    7: PMN1 on CPU1
 *    8: CCNT on CPU1
 *    9: PMN0 on CPU1
 *   10: PMN1 on CPU1
 *   11: CCNT on CPU1
 *   12-19: configurable SCU event counters
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/interrupt.h>
#include <asm/system_misc.h>
#include <asm/sysreg.h>
#include <asm/perf_event.h>
#include <mach/platform.h>
#include <mach/irqs.h>

#define N_PMN_COUNTER 6 /* MAX 31 */

#define PMNC_E		ARMV8_PMU_PMCR_E
#define PMNC_PMNX_CLR	ARMV8_PMU_PMCR_P
#define PMNC_CCNT_CLR	ARMV8_PMU_PMCR_C
#define PMNC_LC		ARMV8_PMU_PMCR_LC
#define PMNC_CCNT	BIT(31)

#define PMON_PROCNAME "pmon"

static bool pmon_enable;
module_param_named(en, pmon_enable, bool, S_IRUSR);

static int pmon_pmn[N_PMN_COUNTER] = { -1, -1, -1, -1, -1, -1, };
module_param_named(pmn0, pmon_pmn[0], int, S_IRUSR|S_IWUSR);
module_param_named(pmn1, pmon_pmn[1], int, S_IRUSR|S_IWUSR);
module_param_named(pmn2, pmon_pmn[2], int, S_IRUSR|S_IWUSR);
module_param_named(pmn3, pmon_pmn[3], int, S_IRUSR|S_IWUSR);
module_param_named(pmn4, pmon_pmn[4], int, S_IRUSR|S_IWUSR);
module_param_named(pmn5, pmon_pmn[5], int, S_IRUSR|S_IWUSR);

static int pmon_ccnt = -1;
module_param_named(ccnt, pmon_ccnt, int, S_IRUSR|S_IWUSR);

static int n_pmn_counter;

struct counter {
	uint ovf; /* overflow count */
	u32 rest; /* last count */
};

static u32 pmon_pmnc    = 0;
static u32 pmon_pmnintr = 0;
static u32 pmon_pmnen   = 0;
struct pmon_info {
	struct counter cnt[N_PMN_COUNTER];
	u64 ccnt;
} ____cacheline_aligned_in_smp;
static struct pmon_info pmon_data[NR_CPUS] __cacheline_aligned_in_smp;

static void pmon_start_pmu(void *info)
{
	int i;

	for (i = 0; i < n_pmn_counter; i++) {
		write_sysreg(i, pmselr_el0);
		write_sysreg(pmon_pmn[i], pmxevtyper_el0);
	}

	write_sysreg(~0U, pmintenclr_el1);
	write_sysreg(pmon_pmnintr, pmintenset_el1);
	write_sysreg(~0U, pmcntenclr_el0);
	write_sysreg(pmon_pmnen, pmcntenset_el0);

	write_sysreg(~0U, pmovsclr_el0);
	write_sysreg(0, pmccfiltr_el0);
	write_sysreg(pmon_pmnc, pmcr_el0);
}

static void pmon_stop_pmu(void *info)
{
	int i;
	int cpu = smp_processor_id();
	u32 val;
	u64 ccnt;

	/* stop */
	write_sysreg(0, pmcr_el0);

	/* read counters */
	for (i = 0; i < n_pmn_counter; i++) {
		write_sysreg(i, pmselr_el0);
		val = read_sysreg(pmxevcntr_el0);
		pmon_data[cpu].cnt[i].rest = val;
	}

	ccnt = read_sysreg(pmccntr_el0);
	pmon_data[cpu].ccnt = ccnt;
}

static irqreturn_t pmon_intr(int irq, void *dev_id)
{
	int i, n;
	u32 pmnintr, pmnovf, mask;

	n = irq - IRQ_PMU_BASE;
	if (n < 0  ||  n >= NR_CPUS) {
		printk(KERN_ERR "pmon_intr: illegal irq=%d\n", irq);
		return IRQ_HANDLED;
	}

	pmnovf  = read_sysreg(pmovsclr_el0);
	write_sysreg(pmnovf, pmovsclr_el0); /* clear overflow flags */

	pmnintr = read_sysreg(pmintenset_el1);
	for (i = 0, mask = 1U; i < n_pmn_counter; i++, mask <<= 1) {
		if ((pmnovf & mask) && (pmnintr & mask)) {
			pmon_data[n].cnt[i].ovf++;
		}
	}

	return IRQ_HANDLED;
}

static void pmon_route_irq(void)
{
	int i;

	for (i = 0; i < NR_CPUS; i++) {
		irq_set_affinity(IRQ_PMU(i), cpumask_of(i));
	}
}

static void pmon_setup_pmu(void)
{
	int i;
	u32 pmnc = 0, pmnintr = 0, pmnen = 0;

	pmon_route_irq();
	memset(pmon_data, 0, sizeof pmon_data);
	for (i = 0; i < n_pmn_counter; i++) {
		if (pmon_pmn[i] >= 0) {
			pmnc    |= PMNC_E;
			pmnintr |= BIT(i);
			pmnen   |= BIT(i);
		}
	}
	if (pmon_ccnt >= 0) {
		pmnc    |= PMNC_E;
		pmnintr |= PMNC_CCNT;
		pmnen   |= PMNC_CCNT;
	}
	pmon_pmnen   = pmnen;
	pmon_pmnintr = pmnintr;
	pmon_pmnc    = pmnc | PMNC_PMNX_CLR | PMNC_CCNT_CLR | PMNC_LC;
}

#define PMON_NR_SEQ (1 + NR_CPUS)

static void pmon_show(struct seq_file *seq, int i)
{
	int cpu, n;
	u64 count;

	if (0 == i) { /* header */
		seq_printf(seq, "     ");
		for (n = 0; n < n_pmn_counter; n++) {
			seq_printf(seq, "                 ");
			if (pmon_pmn[n] >= 0)
				seq_printf(seq, "0x%02x", pmon_pmn[n]);
			else
				seq_printf(seq, "----");
		}
		seq_printf(seq, "                 CCNT");
	} else {
		cpu = i - 1;
		seq_printf(seq, "CPU%d:", cpu);
		for (n = 0; n < n_pmn_counter; n++) {
			count = (u64)pmon_data[cpu].cnt[n].ovf << 32;
			count += pmon_data[cpu].cnt[n].rest;
			seq_printf(seq, " %20llu", count);
		}
		seq_printf(seq, " %20llu", pmon_data[cpu].ccnt);
	}
}
/*-------------------------------------------------------*/

static void pmon_start(void)
{
	/* setup */
	pmon_setup_pmu();

	/* start */
	wmb();
	on_each_cpu(pmon_start_pmu, NULL, 1);
}

static void pmon_stop(void)
{
	on_each_cpu(pmon_stop_pmu, NULL, 1);
}

static int pmon_setup_irq(void)
{
	int i, ret;

	/* cp15 */
	for (i = 0; i < NR_CPUS; i++) {
		ret = request_irq(IRQ_PMU(i), pmon_intr, 0, "pmn", NULL);
		if (ret) {
			printk(KERN_ERR "pmon: request_irq %d failed: %d\n",
			       IRQ_PMU(i), ret);
			goto err;
		}
	}
	return 0;

 err:
	while (i--) {
		free_irq(IRQ_PMU(i), NULL);
	}
	return ret;
}

/*----------------------------------------------------------------*/
static ssize_t pmon_write(struct file *file, const char __user *buf,
			  size_t len, loff_t *ppos)
{
	char arg[16];
	long start;

	if (len >= sizeof arg) {
		printk(KERN_ERR "/proc/%s: too long.\n", PMON_PROCNAME);
		return -EOVERFLOW;
	}
	if (copy_from_user(arg, buf, len)) {
		return -EFAULT;
	}
	arg[len] = '\0';

	start = simple_strtol(arg, NULL, 0);
	if (start) {
		pmon_start();
	} else {
		pmon_stop();
	}

	return len;
}

static void *pmon_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos < PMON_NR_SEQ)
		return pos;
	return NULL;
}

static void *pmon_seq_next(struct seq_file *seq, void *v, loff_t *pos)
{
	(*pos)++;
	if (*pos < PMON_NR_SEQ)
		return pos;
	return NULL;
}

static void pmon_seq_stop(struct seq_file *seq, void *v)
{
}

static int pmon_seq_show(struct seq_file *seq, void *v)
{
	int i = *(loff_t *)v;

	pmon_show(seq, i);
	seq_printf(seq, "\n");
	return 0;
}

static struct seq_operations pmon_seq_ops = {
	.start	= pmon_seq_start,
	.next	= pmon_seq_next,
	.stop	= pmon_seq_stop,
	.show	= pmon_seq_show,
};

static int pmon_seq_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &pmon_seq_ops);
}

static struct file_operations proc_pmon_ops = {
	.owner		= THIS_MODULE,
	.open		= pmon_seq_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
	.write		= pmon_write,
};

static struct proc_dir_entry *proc_pmon = NULL;

static int __init pmon_init(void)
{
	u32 pmcr;

	if (!pmon_enable)
		return 0;

	pmcr = read_sysreg(pmcr_el0);
	n_pmn_counter = (pmcr >> ARMV8_PMU_PMCR_N_SHIFT) & ARMV8_PMU_PMCR_N_MASK;
	if (n_pmn_counter > N_PMN_COUNTER)
		n_pmn_counter = N_PMN_COUNTER;

	proc_pmon = proc_create_data(PMON_PROCNAME, 0, NULL, &proc_pmon_ops, NULL);
	if (!proc_pmon) {
		printk(KERN_ERR "pmon: can not create proc entry\n");
		return -1;
	}

	if (pmon_setup_irq()) {
		remove_proc_entry(PMON_PROCNAME, NULL);
		return -1;
	}

	return 0;
}

module_init(pmon_init);
