/*
 * driver/misc/wdt.c
 *
 * Watchdog timer driver
 *
 * Copyright 2018 Sony Corporation
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/udif/module.h>
#include <linux/wdt.h>
#include <linux/syscore_ops.h>
#include <linux/proc_fs.h>
#include <linux/snsc_boot_time.h>
#include <mach/errind.h>
#include <mach/gic_export.h>

#define WDT_CLK		4000 /* KHz */
#define WDT_CFG		(WDT_RPL256)

/* WDT mode */
#define WDT_OFF  0
#define WDT_ON   1
static unsigned int wdt_mode = WDT_OFF;
module_param_named(mode, wdt_mode, uint, S_IRUSR);

static int wdt_debug = 0;
module_param_named(debug, wdt_debug, int, S_IRUSR|S_IWUSR);

/*-------------------- WDT access ---------------------*/
struct wdt_info {
	UDIF_VA  va[UDIF_NR_WDT];
	const UDIF_DEVICE *dev;
	UDIF_CH ch;
	UDIF_IRQ irq;
	int state;
	unsigned int tmo;
};
static struct wdt_info wdt;

static inline void wdt_state_init(struct wdt_info *w)
{
	w->state = UDIF_NR_WDT;
	smp_wmb();
}

static inline int wdt_ready(void)
{
	smp_rmb();
	return !wdt.state;
}

static void wdt_enable_irq(struct wdt_info *w)
{
	if (!w->irq)
		return;
	udif_maskirq(w->dev, w->ch, 1);
}

static void wdt_disable_irq(struct wdt_info *w)
{
	if (!w->irq)
		return;
	udif_maskirq(w->dev, w->ch, 0);
}

static inline u32 wdt_read(int reg)
{
	return readl_relaxed(wdt.va[UDIF_WDT_REG] + reg);
}

static inline void wdt_write(int reg, u32 val)
{
	writel_relaxed(val, wdt.va[UDIF_WDT_REG] + reg);
}

static inline void wdt_misc_write(u32 val)
{
	writel_relaxed(val, wdt.va[UDIF_WDT_MISC]);
}

/*------------- XRESET_REQ output control -------------*/
#define CONNECT     1
#define DISCONNECT  0
static void xrstreq(int conn)
{
	wdt_misc_write((conn) ? WDT_RST_OUT : 0);
}

/*---------------------- timeout ----------------------*/
#define WDT_MAX_MSEC		((1U << WDT_TO_MAX) / WDT_CLK)

static unsigned int wdt_msec_to_tmo(unsigned int msec)
{
	unsigned int count;
	int i;

	if (msec > WDT_MAX_MSEC) {
		msec = WDT_MAX_MSEC;
	}
	count = WDT_CLK * msec; /* 0..2^31 */
	i = fls(count); /* 0..32 */
	if (i > 0) {
		if ((1U << (i - 1)) == count) /* 2^n ? */
			i--;

		i -= WDT_TO_MIN;
		if (i < 0)
			i = 0;
	}
	return i;
}

static unsigned int wdt_tmo_to_msec(unsigned int tmo)
{
	return (1U << (WDT_TO_MIN + tmo)) / WDT_CLK;
}

static unsigned int wdt_timeout = 0; /* msec */
static int wdt_timeout_set(const char *val, const struct kernel_param *kp)
{
	unsigned int msec;
	int ret;

	ret = kstrtouint(val, 0, &msec);
	if (ret)
		return ret;

	wdt.tmo = wdt_msec_to_tmo(msec);
	*(unsigned int *)kp->arg = wdt_tmo_to_msec(wdt.tmo);
	return 0;
}

static const struct kernel_param_ops wdt_timeout_ops = {
        .set    = wdt_timeout_set,
        .get    = param_get_uint,
};
module_param_cb(timeout, &wdt_timeout_ops, &wdt_timeout, S_IRUSR);

/* wdt time extend to dump in msec */
#define WDT_EXTEDED_TIME_EMDUMP         (2000)
static unsigned int wdt_timeout_mp_excpt = WDT_EXTEDED_TIME_EMDUMP;
module_param_named(mp_excpt, wdt_timeout_mp_excpt, uint, S_IRUSR);

/*-------------------- WDT control --------------------*/
static inline void wdt_setup(void)
{
	wdt_write(WDT_TORR, wdt.tmo);
}

static inline void wdt_restart(void)
{
	/* counter restart and clear wdt_sys_rst */
	wdt_write(WDT_CRR, WDT_RESTART);
}

static void wdt_start(int intr)
{
	wdt_restart();
	wdt_write(WDT_CR, WDT_CFG|WDT_EN|((intr)?WDT_RMOD_INT:0));
	wmb();
}

/*
 * Kernel API
 */
void watchdog_start(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	wdt_setup();
	wdt_start(0);
	xrstreq(CONNECT);
}
EXPORT_SYMBOL(watchdog_start);

void watchdog_stop(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	xrstreq(DISCONNECT);
}
EXPORT_SYMBOL(watchdog_stop);

void watchdog_touch(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	if (wdt_debug) {
		u32 count;
		char buf[64];

		count = wdt_read(WDT_CCVR);
		scnprintf(buf, sizeof buf, "WDT:%u", count);
		BOOT_TIME_ADD1(buf);
	}

	wdt_restart();
}
EXPORT_SYMBOL(watchdog_touch);

/*
 * Touches watchdog from exception.c
 */
static DEFINE_RAW_SPINLOCK(wdte_lock);
static volatile int s_except = 0;

int watchdog_touch_exception(void)
{
	unsigned long flags;
	char msg[64];
	int cpu;

	if (WDT_OFF == wdt_mode)
		return 0;

	if (!wdt_ready())
		return 0;

	if (s_except)
		return 1;

	raw_spin_lock_irqsave(&wdte_lock, flags);

	/* touch before calculation */
	watchdog_touch();

	/* modify wdt_timeout */
	if (wdt_timeout_mp_excpt > wdt_timeout) {
		wdt.tmo = wdt_msec_to_tmo(wdt_timeout_mp_excpt);
	}
	wdt_setup();
	wdt_start(0);
	xrstreq(CONNECT);
	/* log some info */
	printk(KERN_ERR "WDT:new_timeout=%u\n", wdt_tmo_to_msec(wdt.tmo));
	cpu = smp_processor_id();
	scnprintf(msg, sizeof(msg), "watchdog extended on cpu%d", cpu);
	boot_time_add(msg);
	/* we'd like to prevent making it chain */
	s_except = 1;

	raw_spin_unlock_irqrestore(&wdte_lock, flags);

	return 1;
}


/*-------------------- WDT Driver -------------------- */
#define LOCK_ENABLE     1
#define LOCK_DISABLE    0

static DEFINE_SPINLOCK(wdt_lock);
static unsigned char wdt_lock_status = LOCK_DISABLE;

static int wdt_proc_open(struct inode *node, struct file *fp)
{
	if (!spin_trylock(&wdt_lock)) {
		printk(KERN_ERR "[WDT][OPEN1] WDT is Busy.\n");
		return -EBUSY;
	}

	if (wdt_lock_status & LOCK_ENABLE) {
		spin_unlock(&wdt_lock);
		printk(KERN_ERR "[WDT][OPEN2] WDT is Busy.\n");
		return -EBUSY;
	}

	fp->private_data = PDE_DATA(node);
	wdt_lock_status = LOCK_ENABLE;
	spin_unlock(&wdt_lock);

	return 0;
}

static int wdt_proc_release(struct inode *node, struct file *fp)
{
	wdt_lock_status = LOCK_DISABLE;
	smp_wmb();

	return 0;
}

static long wdt_proc_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
	long ret = 0;
	unsigned int usec;
	struct wdt_info *w = (struct wdt_info *)fp->private_data;

	printk(KERN_INFO "[INFO][%s] ioctl: cmd=%08X, arg=%u\n", __func__, cmd, (unsigned int)arg);
	switch(cmd) {
	case WDT_SET_STOP:
		/* WDT Reset Request disable. */
		wdt_disable_irq(w);
		xrstreq(DISCONNECT);
		break;
	case WDT_SET_START:
		/* WDT Reset Request enable. */
		usec = arg;
		w->tmo = wdt_msec_to_tmo(usec/USEC_PER_MSEC);
		if (wdt_debug) {
			printk(KERN_INFO "[INFO][%s] tmout=%u[ms]\n", __func__, wdt_tmo_to_msec(w->tmo));
		}
		wdt_setup();
		wdt_start(0);
		wdt_enable_irq(w);
		xrstreq(CONNECT);
		break;
	case WDT_SET_DEBUG:
		usec = arg;
		w->tmo = wdt_msec_to_tmo(usec/USEC_PER_MSEC);
		if (wdt_debug) {
			printk(KERN_INFO "[INFO][%s] tmout=%u[ms]\n", __func__, wdt_tmo_to_msec(w->tmo));
		}
		xrstreq(DISCONNECT);
		wdt_setup();
		wdt_start(1);
		wdt_enable_irq(w);
		break;
	default:
		printk(KERN_ERR "[ERR][%s] cmd error. cmd=%08X\n", __func__, cmd);
		ret = -1;
		break;
	}

	return ret;
}

static UDIF_ERR emg_intr(UDIF_INTR *hndl)
{
	struct wdt_info *w = udif_intr_data(hndl);
	u32 stat;

	stat = wdt_read(WDT_STAT);
	if (!(stat & WDT_INTSTAT))
		return UDIF_ERR_OK;

	stat = wdt_read(WDT_EOI); /* clear intr */
	udif_disable_irq_nosync(w->irq); /* single shot */

	printk(KERN_ERR "START Emergency!\n");
	boot_time_add("Emergency interrupt!!");
	errind_start(ERRIND_EMG);

	return UDIF_ERR_OK;
}

#ifdef CONFIG_PM
static void wdt_resume(void)
{
	watchdog_start();
}

static struct syscore_ops wdt_syscore_ops = {
	.resume		= wdt_resume,
};
#endif /* CONFIG_PM */

static struct file_operations wdt_proc_ops = {
	.open               = wdt_proc_open,
	.unlocked_ioctl     = wdt_proc_ioctl,
	.release            = wdt_proc_release,
};

static UDIF_ERR wdt_init(UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	wdt_state_init(w);

	if (!proc_create_data(WDT_NAME, S_IRUSR|S_IWUSR, NULL, &wdt_proc_ops, w)) {
		printk(KERN_ERR "WDT:ERROR:%s:proc_create_data\n", __func__);
	}

#ifdef CONFIG_PM
	register_syscore_ops(&wdt_syscore_ops);
#endif

	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_exit(UDIF_VP data)
{
	remove_proc_entry(WDT_NAME, NULL);
	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_probe(const UDIF_DEVICE *dev, UDIF_CH i, UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	w->va[i] = udif_devio_virt(dev, i);
	w->state--;

	if (WDT_OFF != wdt_mode)
		return UDIF_ERR_OK;

	/* setup interrupt handler for EMG */
	if (!i) {
		w->dev = dev;
		w->ch = i;
		w->irq = udif_devint_irq(dev, i);
		if (udif_request_irq(dev, i, emg_intr, w)) {
			w->irq = 0;
			printk(KERN_ERR "WDT:ERROR:%s:request_irq %d failed.\n", __func__, w->irq);
		}
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_remove(const UDIF_DEVICE *dev, UDIF_CH i, UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	wdt_state_init(w);

	if (WDT_OFF != wdt_mode)
		return UDIF_ERR_OK;

	if (!i && w->irq) {
		udif_free_irq(dev, i);
		w->irq = 0;
	}

	return UDIF_ERR_OK;
}

static UDIF_DRIVER_OPS wdt_ops = {
	.init		= wdt_init,
	.probe		= wdt_probe,
	.remove		= wdt_remove,
	.exit		= wdt_exit,
};

UDIF_IDS(wdt_devs) = {
	UDIF_ID(UDIF_ID_WDT, UDIF_CH_MASK_DEFAULT),
};
UDIF_DEPS(wdt_deps) = {};

static UDIF_MODULE(wdt, "wdt", "2.0", wdt_ops, wdt_devs, wdt_deps, &wdt);
