/**
 *	@file	usbg_still_pvt.h
 *	@brief	USB SICD(Still Image Capture Device class) - private header
 *	
 *		Copyright 2005,2006,2008 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 */

#ifndef __USBG_STILL_PVT_H__
#define __USBG_STILL_PVT_H__

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <asm/page.h>
//#include <asm/spinlock.h>

#ifdef CONFIG_OSAL_UDIF
 #include <linux/udif/mutex.h>
 #include <linux/udif/cdev.h>
#endif

#include "usbg_still_conf.h"

#ifndef FNAME
#define FNAME "usbg_still"
#endif

#ifdef  SICD_DEBUG_ERR
#define SICD_ERR(fmt, args...) { printk("%s(%d): [E]" fmt, FNAME, __LINE__, ##args); }
#else
#define SICD_ERR(fmt, args...) { }
#endif

#ifdef  SICD_DEBUG_INF
#define SICD_INF(fmt, args...) { printk("%s(%d): [I]" fmt, FNAME, __LINE__, ##args);   }
#else
#define SICD_INF(fmt, args...) { }
#endif

#ifdef  SICD_DEBUG_API
#define SICD_API()             { printk("%s(%d): [A]%s()\n", FNAME, __LINE__, __FUNCTION__); }
#else
#define SICD_API()             { }
#endif

#ifdef  SICD_DEBUG_FNC
#define SICD_FNC()             { printk("%s(%d): [F]%s()\n", FNAME, __LINE__, __FUNCTION__); }
#else
#define SICD_FNC()             { }
#endif

#ifdef  SICD_DEBUG_DMP
#define SICD_DMP(addr, size)   { sicd_dump(addr, size, FNAME, __LINE__); }
#else
#define SICD_DMP(addr, size)   { }
#endif

#ifdef   SICD_SEMAPHORE
 #ifdef   CONFIG_OSAL_UDIF
  #define SICD_LOCK_INI()             { udif_mutex_init(   &(sicd_get_this_ex()->ep_lock)           ); \
                                        udif_mutex_init(   &(sicd_get_this_ex()->reset_req_lock)    ); \
                                        udif_mutex_init(   &(sicd_get_this_ex()->requesting_lock)   ); \
                                        udif_mutex_init(   &(sicd_get_this_ex()->evt_que_lock)      ); \
                                        udif_mutex_init(   &(sicd_get_this_ex()->status_lock)       ); }
                                        
  #define SICD_EP_LOCK_ON()           { udif_mutex_lock(   &(sicd_get_this_ex()->ep_lock) ); \
                                        SICD_INF("### EP LOCK ON  ---> \n"); }
  #define SICD_EP_LOCK_OFF()          { udif_mutex_unlock( &(sicd_get_this_ex()->ep_lock) ); \
                                        SICD_INF("### EP LOCK OFF <--- \n"); }
                                        
  #define SICD_RST_REQ_LOCK_ON()      { udif_mutex_lock(   &(sicd_get_this_ex()->reset_req_lock) ); \
                                        SICD_INF("### ResetReq LOCK ON  ---> \n"); }
  #define SICD_RST_REQ_LOCK_OFF()     { udif_mutex_unlock( &(sicd_get_this_ex()->reset_req_lock) ); \
                                        SICD_INF("### ResetReq LOCK OFF <--- \n"); }
                                        
  #define SICD_STATUS_LOCK_ON()       { udif_mutex_lock(   &(sicd_get_this_ex()->status_lock) ); \
                                        SICD_INF("### STATUS LOCK ON  ---> \n"); }
  #define SICD_STATUS_LOCK_OFF()      { udif_mutex_unlock( &(sicd_get_this_ex()->status_lock) ); \
                                        SICD_INF("### STATUS LOCK OFF <--- \n"); }
                                        
  #define SICD_REQUESTING_LOCK_ON()   { udif_mutex_lock(   &(sicd_get_this_ex()->requesting_lock) ); \
                                        SICD_INF("### REQUESTING LOCK ON  ---> \n"); }
  #define SICD_REQUESTING_LOCK_OFF()  { udif_mutex_unlock( &(sicd_get_this_ex()->requesting_lock) ); \
                                        SICD_INF("### REQUESTING LOCK OFF <--- \n"); }
                                        
  #define SICD_EVT_QUE_LOCK_ON()      { udif_mutex_lock(   &(sicd_get_this_ex()->evt_que_lock) ); \
                                        SICD_INF("### EVT QUEUE LOCK ON  ---> \n"); }
  #define SICD_EVT_QUE_LOCK_OFF()     { udif_mutex_unlock( &(sicd_get_this_ex()->evt_que_lock) ); \
                                        SICD_INF("### EVT QUEUE LOCK OFF <--- \n"); }
                                        
 #else
  #error
 #endif
#else
 #ifdef   SICD_SPINLOCK
  #include <linux/spinlock.h>
  #define SICD_LOCK_INI()  { spin_lock_init(&(sicd_get_this_ex()->lock)); }
  #define SICD_EP_LOCK_ON()   { spin_lock_irqsave(&(sicd_get_this_ex()->lock), sicd_get_this_ex()->lockflag); }
  #define SICD_EP_LOCK_OFF()  { spin_unlock_irqrestore(&(sicd_get_this_ex()->lock), sicd_get_this_ex()->lockflag); }
 #else
  #define SICD_LOCK_INI()  { }
  #define SICD_EP_LOCK_ON()   { }
  #define SICD_EP_LOCK_OFF()  { }
 #endif
#endif

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#define SICD_FUNC_NAME			"Still Image Capture Device"
#define SICD_EP_NAME_BULK_IN 	"SICD data send (Bulk-in) "
#define SICD_EP_NAME_BULK_OUT	"SICD data recv (Bulk-out)"
#define SICD_EP_NAME_INT_IN		"SICD event send (Int.-in)"

#define SICD_STATUS_OK						0x2001	///< SICD status OK
#define SICD_STATUS_BUSY					0x2019	///< SICD status BUSY

#define SICD_HEADER_LEN						12		///< SICD header len
#define SICD_EP0_BUFSIZE					512		///< Buffer size (EP0)  /* SICD gadgetbMaxPacketSize0̍ől(SXs[h[hԂ) */
#define SICD_DAT_BUFSIZE	(USBG_STILL_BUFFER_SIZE/USBG_STILL_BUFFER_NUM)	///< Buffer size (Data)
#define SICD_EVT_BUFSIZE					32		///< Buffer size (Event)
#define SICD_CANCEL_REQ_DATA_LEN			4		///< CancelRequest data size
#define SICD_GET_DEVICE_STATUS_MIN_LEN		4		///< len(2)+code(2)

#define SICD_B_REQUEST_CANCEL				0x64	///< Cancel_Request
#define SICD_B_REQUEST_GET_EX_EVENT_DATA	0x65	///< Get_Extended_Event_Data
#define SICD_B_REQUEST_DEVICE_RESET			0x66	///< Device_Reset_Request
#define SICD_B_REQUEST_GET_DEVICE_STATUS	0x67	///< Get_Device_Status

struct sicd_data_req {
    u8 requesting;
    struct usb_request *req;
};

struct sicd_evt_req {
    u8 index;
    u8 requesting;
    u8 *evt_buffer;
    struct usb_request *req;
};

#define SICD_MAX_EVT_REQ_NUM 100

/*
 *	@brief	SICD ep information
 */
struct sicd_ep_info {
	u8 ep_no;								///< EP no. (eg.0x01, 0x82, 0x83)
	u8 stalled;								///< stall? or not?
	union {
		struct sicd_data_req data;			///< usb request(for BULK_IN, BULK_OUT ep)
		struct sicd_evt_req  *evt;			///< usb request(for INT_IN ep)
	} u;
};

/*
 *	@brief	Callback ID (to userspace)
 */
enum {
    SICD_CBID_START = 0,					///< started
    SICD_CBID_STOP,							///< stopped
    SICD_CBID_EP_SET_HALT,					///< EP halt set
    SICD_CBID_EP_CLR_HALT,					///< EP halt cleared
    SICD_CBID_SUSPEND,						///< USB bus suspened
    SICD_CBID_RESUME,						///< USB bus resumed
    SICD_CBID_CLASS,						///< class request
    SICD_CBID_VENDOR,						///< vendor request
    SICD_CBID_NBROF,						///< callback id MAX
};

/*
 *	@brief	SICD ep ID
 */
enum {
	SICD_EP_IDX_BULK_IN = 0,					///< ID: Bulk-in  EP
	SICD_EP_IDX_BULK_OUT,				///< ID: Bulk-out EP
	SICD_EP_IDX_INT_IN,						///< ID: Int.-in EP
	SICD_EP_IDX_MAX,						///< ID: MAX
};

/*
 *	@brief	SICD driver information (extension)
 */
struct func_data_ex {
	void*			buffer;					///< kernel-virtual address (for SICD data)
	dma_addr_t		buffer_dma;				///< dma address
	uintptr_t		buffer_phys;			///< physical address
	void*			buffer_user;			///< user-virtual address
	int				sicd_status;			///< SICD protocol status
	struct sicd_ep_info ep_info[SICD_EP_IDX_MAX];	///< EP information
	u8				ep0_buffer[SICD_EP0_BUFSIZE];	///< EP0 buffer
	struct sicd_evt_req sicd_evt_req_list[SICD_MAX_EVT_REQ_NUM];
	int				reset_request_flag;				///< Flag for DeviceResetRequest
	uint32_t		UnitHandle;				///UnitHandle for Memory Manager
	uint32_t		UnitSize;				///UnitSize for Memory Manager
#ifdef SICD_SEMAPHORE
 #ifdef CONFIG_OSAL_UDIF
	UDIF_MUTEX		ep_lock;							///< lock for endpoint
	UDIF_MUTEX		reset_req_lock;						///< lock for reset request flag
	UDIF_MUTEX		status_lock;						///< lock for sicd status
	UDIF_MUTEX		requesting_lock;					///< lock for requesting flag
	UDIF_MUTEX		evt_que_lock;						///< lock for evt queue flag
 #endif
#else
 #ifdef  SICD_SPINLOCK
 	spinlock_t		ep_lock;							///< lock struct
 	unsigned long	lockflag;						///< lock flag
 #endif
#endif
	bool req_comp_waiting;						///< ep request comp wait flag
	struct completion req_wait_completion;		///< ep request comp completion
};

//----------------------------------------------------------------------
// sicd_core file operations
UDIF_ERR sicd_core_open(UDIF_FILE *filp);
UDIF_ERR sicd_core_release(UDIF_FILE *filp);
UDIF_ERR sicd_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);

//----------------------------------------------------------------------
// sicd_data file operations
UDIF_ERR sicd_data_open(UDIF_FILE *filp);
UDIF_ERR sicd_data_release(UDIF_FILE *filp);
UDIF_SSIZE sicd_data_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param);
UDIF_SSIZE sicd_data_read(UDIF_FILE *filp, UDIF_CDEV_READ *param);
UDIF_ERR sicd_data_mmap(UDIF_FILE *filp, UDIF_VMAREA *vm);
UDIF_ERR sicd_data_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);

//----------------------------------------------------------------------
// sicd_event file operations
UDIF_ERR sicd_event_open(UDIF_FILE *filp);
UDIF_ERR sicd_event_release(UDIF_FILE *filp);
UDIF_SSIZE sicd_event_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param);


//----------------------------------------------------------------------
// sicd local functions
struct func_data*		sicd_get_this(void);
struct func_data_ex*	sicd_get_this_ex(void);
struct usb_ep*			sicd_get_ep(int ep_id);
struct sicd_ep_info*	sicd_get_ep_info(struct usb_ep *ep);
dma_addr_t				sicd_user_to_dma(void *uaddr);
void*					sicd_user_to_kvirt(void *uaddr);
void*					sicd_get_ep0_buffer(void);

int  sicd_create_this(void);
void sicd_delete_this(void);
void sicd_link_this_to_file(struct file *file);
void sicd_put_u8(u8 **p, u8 val);
void sicd_put_u16(u8 **p, u16 val);
void sicd_put_u32(u8 **p, u32 val);
void sicd_call_user_cb(int cb_id, int evt_id, void *arg, int size);
void sicd_request_start(int ep_id);
void sicd_request_stop(int ep_id);
int  sicd_is_requesting(int ep_id);
void sicd_set_requesting_flag( int ep_id, int flag );
void sicd_ep_dequeue(struct usb_ep *ep);
void sicd_ep_dequeue_all(void);
void sicd_ep_dequeue_bulk(void);
void sicd_dump(u8 *p, int size, char *fname, int line);
int  sicd_get_status(void);
void sicd_set_status(int status);
int  sicd_bulk_out_halt(void);
int  sicd_bulk_in_halt(void);
int  sicd_bulk_out_halt_clear(void);
int  sicd_bulk_in_halt_clear(void);
int  sicd_is_valid_buffer(u8 *p, size_t size);
struct usb_request *sicd_get_request(struct usb_ep *ep);
struct sicd_evt_req *sicd_alloc_evt_req_info(struct usb_ep *ep);
void sicd_free_evt_req_info(struct usb_ep *ep, struct sicd_evt_req *evt_req);
bool sicd_check_req_comp_waiting(void);
void sicd_recheck_req_comp_waiting(void);

#endif
