/*
 * usbg_storage_cmn.c
 *
 * USB Mass Storage Gadget Function Driver
 *
 * Copyright 2005,2006,2008,2009,2010,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * << MassStorage Gadget Common Function >>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define __USBG_STORAGE_CMN_C__
/*

 󥯥롼ɥե                                                 
*/
#include <asm/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>

#include <generated/autoconf.h>

#include <linux/usb/ch9.h>
#include <linux/device.h>

#include <linux/usb/gadget.h>

#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#else
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_ioctl.h>
#include <linux/usb/specific_gadget/usb_ioctl_storage.h>

#include "usbg_storage_conf.h"
#include "usbg_storage_cmn.h"
#include "usbg_storage_maincnt.h"
#include "usbg_storage_drvcnt.h"
#include "usbg_bdio.h"

#include <linux/udif/cache.h>


#include <linux/udif/cdev.h>


/*

 饤                                                           
*/
MODULE_AUTHOR( "Sony Corporation" );
MODULE_DESCRIPTION( MY_DESC
                    " driver ver " MY_VER );
MODULE_VERSION(MY_VER);
MODULE_LICENSE("GPL");

/*

 ؿץȥ                                                 
*/
/* [Svc] ****************************************************************/
/* open ----------------------------------------------------------------*/
static UDIF_ERR storage_open(UDIF_FILE *filp);
/* release -------------------------------------------------------------*/
static UDIF_ERR storage_release(UDIF_FILE *filp);
/* ioctl ---------------------------------------------------------------*/
static UDIF_ERR storage_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);
/* ioctl common --------------------------------------------------------*/
static int  storage_ioctl_common( struct inode *, struct file *, unsigned int, unsigned long );
/* ioctl specific ------------------------------------------------------*/
static int  storage_ioctl_specific( struct inode *, struct file *, unsigned int, unsigned long );
/* probe ---------------------------------------------------------------*/
static int  storage_ioctl_probe( struct inode *, struct file *, void * );
/* remove --------------------------------------------------------------*/
static int  storage_ioctl_remove( struct inode *, struct file *, void * );
/* set init info -------------------------------------------------------*/
static int  storage_ioctl_setInitInfo( struct inode *, struct file *, void * );
/* set media status ----------------------------------------------------*/
static int  storage_ioctl_setMediaStatus( struct inode *, struct file *, void * );
/* check media access --------------------------------------------------*/
static int  storage_ioctl_checkMediaAccess( struct inode *, struct file *, void * );

/* [Core] ***************************************************************/
/* start ---------------------------------------------------------------*/
static void storage_start( struct usb_gadget_func_driver *, unsigned char, struct usb_gadget_ep_list );
/* stop ----------------------------------------------------------------*/
static void storage_stop( struct usb_gadget_func_driver * );
/* set halt ------------------------------------------------------------*/
static int  storage_ep_set_halt( struct usb_gadget_func_driver *, struct usb_ep * );
/* clear halt ----------------------------------------------------------*/
static int  storage_ep_clear_halt( struct usb_gadget_func_driver *, struct usb_ep * );
/* class request -------------------------------------------------------*/
static int  storage_class( struct usb_gadget_func_driver *, const struct usb_ctrlrequest *, struct usb_ep * );
/* sub function --------------------------------------------------------*/
static void clear_ep_data( struct usb_ep * );
static void ep0_complete( struct usb_ep *, struct usb_request * );
/* usb_ep_queue() return error [control in] */
static void error_ep_request_control_in( int );
/* usb_ep_queue() callback error [control in] */
static void error_ep_request_cb_control_in( int );
/* block I/O access state read operation -------------------------------*/
static int     gusb_storage_acsr_open(struct inode *inode, struct file *filp);
static ssize_t gusb_storage_acsr_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos);


/*

 ѿ                                                             
*/
/* File Operations ******************************************************/
static struct UDIF_CDEV_OPS storage_fops = {
    .open  = storage_open,
    .close = storage_release,
    .read  = NULL,            /* nothing */
    .write = NULL,            /* nothing */
    .ioctl = storage_ioctl,
};

#ifdef CONFIG_OSAL_UDIF
struct usbg_cdev sony_storage_dev = {
    .node = udif_device_node(UDIF_NODE_USB_MASS),
    .fops = &storage_fops,
};
#else
/* sonymisc struct*******************************************************/
struct sonymisc_device sony_storage_dev = {
    .class = DEV_CLASS_CHR,
    .minor = USB_MINOR_MASSSTORAGE,
    .name  = GSTORAGE__NAME,
    .fops  = &storage_fops,             /* if use character device use  */
    .gd    = NULL,                      /* if use block device use      */
};
#endif

static GSTORAGE__Cmn *pCmn = NULL;

static unsigned char  f_waitReset;      /* Reset Ԥ                   */
static unsigned char  f_waitClearIn;    /* Clear Feature(IN) Ԥ       */
static unsigned char  f_waitClearOut;   /* Clear Feature(OUT) Ԥ      */
static unsigned char  f_calledStorageIN; /* MassStorageINƤФ줿ɤ*/

#if GS_OUT__MBF_QUEUE_NUM
/* Max Message Queue Num */
extern int	maxMsgQueueNum;
#endif

static bool cmn_probe_done = false;
static struct file_operations gusb_acsr_proc_fops = {
    .owner = THIS_MODULE,
    .open  = gusb_storage_acsr_open,
    .read  = gusb_storage_acsr_read,
};

/*

 ؿ                                                             
*/
/**
 * storage_execFunc()
 *
 * סGadget MassStorage Main Loop Function
 **/
GSTORAGE__State storage_execFunc( GSTORAGE__Msg          *pMsg       ,
                                  GSTORAGE__State        currentState,
                                  const GSTORAGE__MsgTbl *MsgTbl[]    )
{
    GSTORAGE__State nextState = currentState;
    const GSTORAGE__MsgTbl *pMsgTbl = MsgTbl[ currentState ];

    /* åơ֥ˤϼ¹Ԥ٤ؿİʾ¸  do-while */
    do {
        if ( pMsg->id == pMsgTbl->id ){
            nextState = pMsgTbl->func( pMsg );
            break;
        }
        pMsgTbl++;
    } while( pMsgTbl->id != GSTORAGE__MSG_TBL_EOT );

    if ( nextState == GSTORAGE__STATE_NO_CHANGE ){
        /* ܤʤ(ΥơȤϸߤΥơȤƱ) */
        nextState = currentState;
    }

    return nextState;
}


/************************************************************************/
/* [Svc]                                                                */
/************************************************************************/
/**
 * storage_open()
 *
 * סǥХΥץ(ƥॳ)
 **/
UDIF_ERR storage_open(UDIF_FILE *filp)
{
    struct file file;
    struct file *fd = &file;
    UDIF_ERR err = UDIF_ERR_OK;
    struct func_data *p_this;

    GS_DET( GS_OUT__CMN, "func Req" );

    /* <framework> */
    /* 󥹥󥹤 */
    if ( usbg_cmn_new( NULL, fd ) ){
        err = UDIF_ERR_DEV;
        GS_ERR( GS_OUT__CMN, "instance create Error" );
        goto SUB_RET;
    }
    p_this = fd->private_data;
    udif_file_data(filp) = fd->private_data;
    /* </framework> */

    /* <user> */
    p_this->vp_user = kmalloc( sizeof( struct GSTORAGE__UserData ), GFP_ATOMIC );
    if ( p_this->vp_user == NULL ){
        usbg_cmn_delete( p_this );
        err = (UDIF_ERR)-ENOBUFS;
        GS_ERR( GS_OUT__CMN, "kmalloc(vp_user)" );
        goto SUB_RET;
    }

    pCmn = kmalloc( sizeof( GSTORAGE__Cmn ), GFP_ATOMIC );
    if ( pCmn == NULL ){
        usbg_cmn_delete( p_this->vp_user );
        usbg_cmn_delete( p_this );
        err = (UDIF_ERR)-ENOBUFS;
        GS_ERR( GS_OUT__CMN, "kmalloc(pCmn)" );
        goto SUB_RET;
    }

    pCmn->maxLunNum = 0;
    pCmn->cond      = 0;
    init_waitqueue_head( &pCmn->wqh );
    pCmn->pThis     = NULL;
    /* </user> */

SUB_RET:
    return err;
}

/**
 * storage_release()
 *
 * סǥХΥ
 * ƽС쥯饹
 **/
UDIF_ERR storage_release(UDIF_FILE *filp)
{
    struct func_data *p_this = ( struct func_data * )udif_file_data(filp);
    struct usbg_std_user *p_user_data = ( struct usbg_std_user * )p_this->vp_user;

    GS_DET( GS_OUT__CMN, "func Req" );

    /* ǡγ */
    if ( p_user_data ){
        kfree( p_user_data );
    }

    /* 󥹥󥹤γ */
    usbg_cmn_delete( p_this );
    udif_file_data(filp) = NULL;

    return 0;
}

/**
 * storage_ioctl()
 *
 * סǥХͭΥޥ
 * ƽСMassStorage Service
 **/
UDIF_ERR storage_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
    struct file file;
    struct file *fd = &file;
    unsigned int cmd = param->cmd;
    unsigned long arg = param->arg;
	int err = 0;

    GS_DET( GS_OUT__CMN, "func Req" );

    fd->private_data = udif_file_data(filp);
    if ( _IOC_TYPE( cmd ) == USB_IOC_CMN ){
        err = storage_ioctl_common( NULL, fd, cmd, arg );
    }
    else if( _IOC_TYPE( cmd ) == USB_IOC_MASSSTORAGE ) {
        err = storage_ioctl_specific( NULL, fd, cmd, arg );
    }
    else {
        GS_ERR( GS_OUT__CMN, "invalid ioctl magic number" );
        err = UDIF_ERR_NOTTY;
    }
    udif_file_data(filp) = fd->private_data;
    
    return (UDIF_ERR)err;
}

/**
 *  IOCTL
 **/
static int storage_ioctl_common( struct inode  *inode, struct file   *fd,
                                 unsigned int  cmd   , unsigned long arg )
{
    int err = 0;

    GS_DET( GS_OUT__CMN, "func Req" );

    if ( _IOC_NR( cmd ) > USB_IOC_CMN_NBROF ){
        GS_ERR( GS_OUT__CMN, "invalid ioctl seq. number" );
        return -ENOTTY;
    }

    switch( cmd ){
      case USB_IOC_CMN_PROBE:
        err = storage_ioctl_probe( inode, fd, ( void * ) arg );
        break;
      case USB_IOC_CMN_REMOVE:
        err = storage_ioctl_remove( inode, fd, ( void * ) arg );
        break;
      default:
        err = -ENOTTY;
        break;
    }

    if ( err ){
        GS_ERR( GS_OUT__CMN, "err=%d", err );
    }

    return err;
}

/**
 *  IOCTL
 **/
static int storage_ioctl_specific( struct inode  *inode, struct file   *fd,
                                   unsigned int  cmd   , unsigned long arg )
{
    int err = -ENOSYS;

    GS_DET( GS_OUT__CMN, "func Req" );

    if ( _IOC_NR( cmd ) > USBG_IOC_STORAGE_NBROF ){
        GS_ERR( GS_OUT__CMN, "invalid ioctl seq. number" );
        return -ENOTTY;
    }

    switch( cmd ){
      case USBG_IOC_STORAGE_SET_INIT_INFO :
        err = storage_ioctl_setInitInfo( inode, fd, ( void * ) arg );
        break;
      case USBG_IOC_STORAGE_SET_MEDIA_STATUS :
        err = storage_ioctl_setMediaStatus( inode, fd, ( void * ) arg );
        break;
      case USBG_IOC_STORAGE_CHECK_MEDIA_ACCESS :
        err = storage_ioctl_checkMediaAccess( inode, fd, ( void * ) arg );
        break;
      default:
        err = -ENOTTY;
        break;
    }

    if ( err ){
        GS_ERR( GS_OUT__CMN, "err=%d", err );
    }

    return err;
}

/**
 * storage_ioctl_probe()
 **/
static int storage_ioctl_probe( struct inode *inode, struct file *fd, void *p_arg )
{
    int err = -ENOSYS;
    struct func_data *p_this = ( struct func_data * )( fd->private_data );
    struct probe_info *p_info = p_arg;

    GS_DET( GS_OUT__CMN, "func Req" );

    pCmn->pThis = p_this;

    /* <framework> */
    /* probe ѥ᡼򥳥ԡ */
    if ( usbg_cmn_copy_probe_info( p_this, p_info ) ){
        GS_INF( GS_OUT__CMN, "copy probe info error" );
        err = -ENOBUFS;
        goto SUB_RET;
    }

    p_this->core_reg_info.function      = "USB Gadget Mass Storage";
    p_this->core_reg_info.context       = p_this;
    p_this->core_reg_info.config        = p_this->info.tbl_interface[0].uc_configuration;
    p_this->core_reg_info.interface     = p_this->info.tbl_interface[0].uc_interface;
    p_this->core_reg_info.start         = storage_start;
    p_this->core_reg_info.stop          = storage_stop;
    p_this->core_reg_info.ep_set_halt   = storage_ep_set_halt;
    p_this->core_reg_info.ep_clear_halt = storage_ep_clear_halt;
    p_this->core_reg_info.suspend       = NULL;
    p_this->core_reg_info.resume        = NULL;
    p_this->core_reg_info.class         = storage_class;
    p_this->core_reg_info.vendor        = NULL;

    err = usb_gadgetcore_register_driver( &p_this->core_reg_info );
    if ( err ){
        GS_ERR( GS_OUT__CMN, "usb_gadgetcore_register_driver()" );
        goto SUB_RET;
    }
    
    p_this->info.HasExtendedTblIf = false;
    
    if( p_this->info.extended_tbl_interface ){
        
        GS_INF( GS_OUT__CMN, "Start registering extended_tbl_interface.\n");
        
        p_this->info.HasExtendedTblIf = true;
        
        memcpy( &p_this->extended_core_reg_info, &p_this->core_reg_info, sizeof( struct usb_gadget_func_driver ) );
        
        p_this->extended_core_reg_info.config = p_this->info.extended_tbl_interface[0].uc_configuration;
        p_this->extended_core_reg_info.interface = p_this->info.extended_tbl_interface[0].uc_interface;
        
        GS_INF( GS_OUT__CMN, "register %s, cfg=%d, if=%d\n",
            p_this->extended_core_reg_info.function,
            p_this->extended_core_reg_info.config,
            p_this->extended_core_reg_info.interface);
        
        err = usb_gadgetcore_register_driver( &p_this->extended_core_reg_info );
        
        if ( err ){
            
            GS_ERR( GS_OUT__CMN, "usb_gadgetcore_register_driver()" );
            goto SUB_RET;
            
        }
        
    }
    
    /* </framework> */

    /* Main Controller, Drive Controller γ */
    err = storage_maincnt_allocThread();
    if ( err ){
        GS_ERR( GS_OUT__CMN, "storage_maincnt_allocThread()" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    err = storage_drvcnt_allocThread();
    if ( err ){
        GS_ERR( GS_OUT__CMN, "storage_drvcnt_allocThread()" );
        err = -ENOBUFS;
        goto SUB_RET;
    }

    cmn_probe_done = true;

    storage_probeMassStorage();

SUB_RET:
    return err;
}

/**
 * storage_ioctl_remove()
 **/
static int storage_ioctl_remove( struct inode *inode, struct file *fd, void *p_arg )
{
    struct func_data *p_this = pCmn->pThis;

    GS_DET( GS_OUT__CMN, "func Req" );

    /* Remove Request to MassStorage Main Controller */
    storage_reqRemoveMassStorage();

    /* Remove ǽ֤ܤޤ Wait Event                 */
    /* wakeup Τ MainCnt  maincnt_threadFunc() end  */
    wait_event_interruptible( pCmn->wqh, pCmn->cond );
    pCmn->cond = 0;

    /* unregister driver */
    if( p_this->info.HasExtendedTblIf == true ){
        usb_gadgetcore_unregister_driver( &p_this->extended_core_reg_info );
    }
    usb_gadgetcore_unregister_driver( &p_this->core_reg_info );

    /* clear probe info */
    usbg_cmn_clear_probe_info( p_this );

    /* MainCnt, DrvCnt Υꥢ */
    storage_maincnt_clearInitInfo();
    storage_drvcnt_clearInitInfo();

    /* Thread ׵ȯ */
    storage_maincnt_freeThread();
    storage_drvcnt_freeThread();

    cmn_probe_done = false;

    /* pCmn γ */
    kfree( pCmn );

    return 0;
}

/**
 * storage_exec_remove()
 **/
void storage_exec_remove( void )
{
    struct GSTORAGE__Event *pEvent;
    struct func_data       *p_this = pCmn->pThis;

    GS_DET( GS_OUT__CMN, "func Req" );

    /* MassStorage Svc  Remove λ Event  */
    pEvent = p_this->info.vp_user;
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
                         pEvent->CompRemove,
                         (usb_hndl_t)(p_this->info.hndl),
                         GSTORAGE__KEVENT_ID_COMP_REMOVE,
                         0,
                         (void*)NULL);

    /* Remove Ƴ */
    pCmn->cond = 1;
    wake_up_interruptible( &pCmn->wqh );

#if GS_OUT__MBF_QUEUE_NUM
    GS_INF( GS_OUT__MBF_QUEUE_NUM, "maxMsgQueueNum = %d", maxMsgQueueNum );
#endif

    return;
}

/**
 * storage_ioctl_setInitInfo()
 **/
static int storage_ioctl_setInitInfo( struct inode *inode, struct file *fd, void *p_arg )
{
    int                     err = 0, num= 0, maxLun = 0, nodeSize = 0, ret = 0, i = 0;
    char                    *pDevNode = NULL, *pInquiryData = NULL;
    GSTORAGE__InfoInit      *pInfo = NULL;
    GSTORAGE__InfoInitDrive *pInfoDrv = NULL, *pInfoDrvOrg = NULL;
    GSTORAGE__InfoInitDrive **ppInfoDrv = NULL, **ppInfoDrvOrg = NULL;
    GSTORAGE__InfoInitCmn   *pInfoCmn = NULL;
    GSTORAGE__InfoInitMain  *pInfoMainCnt = NULL;
    GSTORAGE__InfoInitLun   *pInfoLun = NULL, *pInfoLunOrg = NULL;
    GSTORAGE__InfoInitLun   **ppInfoLun = NULL;
    GSTORAGE__InfoInitExtCmd    *pInfoExtCmd = NULL;
    GSTORAGE__InfoInitExtCmd    **ppInfoExtCmd = NULL;
    char                        cmdNum;

#ifdef GSTORAGE_USE_BOOTMEM
    uint32_t bmem = GSTORAGE_BOOTMEM_BASE;
    if ( bmem & 0x00000FFF ){
        GS_ERR( "MassStorage 4KB alignment" );
    }
#endif

    GS_DET( GS_OUT__INIT_INFO, "func Req" );

    /* (1) USER ֤Υɥ쥹 KERNEL ֤˥ԡ */
    pInfo = kmalloc( sizeof( GSTORAGE__InfoInit ), GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfo)                       =%p", pInfo );
    if ( !pInfo ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfo)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    err = copy_from_user( pInfo, p_arg, sizeof( GSTORAGE__InfoInit ) );
    if ( err ){
        GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(pInfo)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    num    = pInfo->num;
    maxLun = pCmn->maxLunNum = num - 1;

    /* (2) USER ֤Υǡ KERNEL ֤˥ԡ */
    ppInfoDrv = ppInfoDrvOrg = kmalloc( sizeof( GSTORAGE__InfoInitDrive * ) * num, GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(ppInfoDrvOrg)                =%p", ppInfoDrvOrg );
    if ( !ppInfoDrv ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(ppInfoDrv)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    err = copy_from_user( ppInfoDrv, pInfo->ppInfoDrives, sizeof( GSTORAGE__InfoInitDrive * ) * num );
    if ( err ){
        GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(ppInfoDrv)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    pInfoDrv = pInfoDrvOrg = kmalloc( sizeof( GSTORAGE__InfoInitDrive ) * num, GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoDrvOrg)                 =%p", pInfoDrvOrg );
    if ( !pInfoDrv ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfoDrv)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    for ( i = 0; i < num; i++ ){
        err = copy_from_user( pInfoDrv, *ppInfoDrv, sizeof( GSTORAGE__InfoInitDrive ) );
        if ( err ){
            pInfoDrv->devNode      = NULL;
            pInfoDrv->pInquiryData = NULL;
            GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(pInfoDrv)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }

        /* node ̾ԡ */
        nodeSize = GSTORAGE_MAX_DEV_NODE_LEN + 1;
        pDevNode = kmalloc( nodeSize, GFP_ATOMIC );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoDrv[%d]->devNode)        =%p", i, pDevNode );
        if ( !pDevNode ){
            pInfoDrv->devNode      = NULL;
            pInfoDrv->pInquiryData = NULL;
            GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pDevNode)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }
        ret = strncpy_from_user( pDevNode, pInfoDrv->devNode, nodeSize );
        if ( ( ret < 0 ) || ( ret >= nodeSize ) ){
            GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->devNode)        =%p", i, pDevNode );
            kfree( pDevNode );
            pInfoDrv->devNode      = NULL;
            pInfoDrv->pInquiryData = NULL;
            GS_ERR( GS_OUT__INIT_INFO, "strncpy_from_user(pDevNode)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }
        pInfoDrv->devNode = pDevNode;
        /* Inquiry 󥳥ԡ */
        if ( pInfoDrv->inquiryDataLen ){
            pInquiryData = kmalloc( pInfoDrv->inquiryDataLen, GFP_ATOMIC );
            GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoDrv[%d]->pInquiryData)   =%p", i, pInquiryData );
            if ( !pInquiryData ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->devNode)        =%p", i, pDevNode );
                kfree( pDevNode );
                pInfoDrv->devNode      = NULL;
                pInfoDrv->pInquiryData = NULL;
                GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInquiryData)" );
                err = -ENOBUFS;
                goto SUB_RET;
            }
            err = copy_from_user( pInquiryData, pInfoDrv->pInquiryData, pInfoDrv->inquiryDataLen );
            if ( err ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->devNode)        =%p", i, pDevNode );
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->pInquiryData)   =%p", i, pInquiryData );
                kfree( pDevNode );
                kfree( pInquiryData );
                pInfoDrv->devNode      = NULL;
                pInfoDrv->pInquiryData = NULL;
                GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(pInquiryData)" );
                err = -ENOBUFS;
                goto SUB_RET;
            }
            pInfoDrv->pInquiryData = pInquiryData;
        }
        else {
            pInfoDrv->pInquiryData = NULL;
        }
        /* pointer ++ */
        ppInfoDrv++;
        pInfoDrv++;
    }

    /* (3) Cnt ̤Υǡ */
    pInfoCmn = kmalloc( sizeof( GSTORAGE__InfoInitCmn ), GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoCmn)                    =%p", pInfoCmn );
    if ( !pInfoCmn ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfoCmn)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
#ifdef GSTORAGE_USE_BOOTMEM
    pInfoCmn->pSendBuf    = phys_to_virt(bmem); bmem += GSTORAGE__COMMON_BUF_SIZE;
    pInfoCmn->pReceiveBuf = phys_to_virt(bmem); bmem += GSTORAGE__COMMON_BUF_SIZE;
#else
    pInfoCmn->pSendBuf    = phys_to_virt(pInfo->PhySendBuf);
    pInfoCmn->pReceiveBuf = phys_to_virt(pInfo->PhyReceiveBuf);

	udif_cache_ctrl(pInfoCmn->pSendBuf,    (64*1024), UDIF_CACHE_INVALIDATE);
	udif_cache_ctrl(pInfoCmn->pReceiveBuf, (64*1024), UDIF_CACHE_INVALIDATE);

    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoCmn->pSendBuf)          =%p", pInfoCmn->pSendBuf );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoCmn->pReceiveBuf)       =%p", pInfoCmn->pReceiveBuf );
#endif
    if ( ( !pInfoCmn->pSendBuf ) || ( !pInfoCmn->pReceiveBuf ) ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pSendBuf,pReceiveBuf)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    pInfoCmn->extCmdNum = pInfo->extCmdNum;     /* ĥޥϿ */
    cmdNum = pInfoCmn->extCmdNum;
    if( 0 < cmdNum ) {
        ppInfoExtCmd = kmalloc( sizeof( GSTORAGE__InfoInitExtCmd * ) * cmdNum, GFP_ATOMIC );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(ppInfoExtCmd)                =%p", ppInfoExtCmd );
        if( !ppInfoExtCmd ) {
            GS_ERR( GS_OUT__INIT_INFO, "kmalloc(ppInfoExtCmd)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }
        err = copy_from_user( ppInfoExtCmd, pInfo->ppInfoExtCmd, sizeof( GSTORAGE__InfoInitExtCmd * ) * cmdNum );
        if( err ){
            GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(ppInfoExtCmd)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }

        pInfoExtCmd = kmalloc( sizeof( GSTORAGE__InfoInitExtCmd ) * cmdNum, GFP_ATOMIC );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoExtCmd)                 =%p", pInfoExtCmd );
        if( !pInfoExtCmd ) {
            GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfoExtCmd)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }
        for( i = 0; i < cmdNum; i++ ) {
            err = copy_from_user( &pInfoExtCmd[ i ], ppInfoExtCmd[ i ], sizeof( GSTORAGE__InfoInitExtCmd ) );
            if( err ){
                GS_ERR( GS_OUT__INIT_INFO, "copy_from_user(pInfoExtCmd)" );
                err = -ENOBUFS;
                goto SUB_RET;
            }
        }
        pInfoCmn->pExtCmd = pInfoExtCmd;    /* ĥޥɾ   */
    }
    else {
        pInfoCmn->pExtCmd = NULL;
    }

    /* (4) MainCnt ѤΥǡ */
    pInfoMainCnt = kmalloc( sizeof( GSTORAGE__InfoInitMain ), GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoMainCnt)                =%p", pInfoMainCnt );
    if ( !pInfoMainCnt ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfoMainCnt)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    pInfoMainCnt->maxLun  = maxLun;

    /* (5) DrvCnt ѤΥǡ */
    pInfoDrv = pInfoDrvOrg;
    pInfoLun = pInfoLunOrg = kmalloc( sizeof( GSTORAGE__InfoInitLun ) * num, GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoLunOrg)                 =%p", pInfoLunOrg );
    if ( !pInfoLun ){
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pInfoLun)" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    ppInfoLun = &pInfoLunOrg;
    for ( i = 0; i < num; i++ ){
#ifdef GSTORAGE_USE_BOOTMEM
        pInfoLun->pSendReadBuf[ 0 ] = phys_to_virt(bmem); bmem += GSTORAGE__COMMON_BUF_SIZE;
        pInfoLun->pSendReadBuf[ 1 ] = phys_to_virt(bmem); bmem += GSTORAGE__COMMON_BUF_SIZE;
#else
        pInfoLun->pSendReadBuf[ 0 ] = phys_to_virt(pInfo->PhySendReadBuf0[i]);
        pInfoLun->pSendReadBuf[ 1 ] = phys_to_virt(pInfo->PhySendReadBuf1[i]);

		udif_cache_ctrl(pInfoLun->pSendReadBuf[ 0 ], (64*1024), UDIF_CACHE_INVALIDATE);
		udif_cache_ctrl(pInfoLun->pSendReadBuf[ 1 ], (64*1024), UDIF_CACHE_INVALIDATE);

        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoLun[%d]->pSendReadBuf[0])=%p", i, pInfoLun->pSendReadBuf[0] );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pInfoLun[%d]->pSendReadBuf[1])=%p", i, pInfoLun->pSendReadBuf[1] );
#endif
        if ( ( !pInfoLun->pSendReadBuf[ 0 ] ) || ( !pInfoLun->pSendReadBuf[ 1 ] ) ){
            GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pSendReadBuf)" );
            err = -ENOBUFS;
            goto SUB_RET;
        }
        pInfoLun->pDrive = pInfoDrv;
        /* pointer ++ */
        pInfoDrv++;
        pInfoLun++;
    }

    /* (6) set MainCnt, DrvCnt */
    err = storage_maincnt_setInitInfo( pInfoMainCnt );
    if ( err ){
        GS_ERR( GS_OUT__INIT_INFO, "storage_maincnt_setInitInfo() error" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    err = storage_drvcnt_setInitInfo( maxLun, pInfoCmn, ppInfoLun );
    if ( err ){
        storage_maincnt_clearInitInfo();
        GS_ERR( GS_OUT__INIT_INFO, "storage_drvcnt_setInitInfo() error" );
        err = -ENOBUFS;
        goto SUB_RET;
    }

SUB_RET:
    /* DEBUG PRINT */
    if ( 0 ){
        pInfoDrv = pInfoDrvOrg;
        for ( i = 0; i < num; i++ ){
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]                  = %p", i, pInfoDrv );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->devNode         = %s"   , i, (char *)pInfoDrv->devNode );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->drvType         = %d"   , i, pInfoDrv->drvType );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->pInquiryData    = %p", i, pInfoDrv->pInquiryData );
            pInquiryData = pInfoDrv->pInquiryData;
            for ( ret = 0; ret < pInfoDrv->inquiryDataLen; ret++ ){
                GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->pInquiryData[%d] = %2X"   , i, ret, (int)*pInquiryData );
                pInquiryData++;
            }
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->inquiryDataLen  = %d"  , i, pInfoDrv->inquiryDataLen );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->blockNum        = %d"  , i, pInfoDrv->blockNum );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->blockLength     = %d"  , i, pInfoDrv->blockLength );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->forceWP         = %d"   , i, pInfoDrv->forceWP );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->driveRemovable  = %d"   , i, pInfoDrv->driveRemovable );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->driveLock       = %d"   , i, pInfoDrv->driveLock );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->flushEnable     = %d"   , i, pInfoDrv->flushEnable );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoDrv[%d]->preventEnable   = %d"   , i, pInfoDrv->preventEnable );
            pInfoDrv++;
        }
        GS_DBG( GS_OUT__INIT_INFO, "pInfoCmn                      = %p", pInfoCmn );
        GS_DBG( GS_OUT__INIT_INFO, "pInfoCmn->pSendBuf            = %p", pInfoCmn->pSendBuf );
        GS_DBG( GS_OUT__INIT_INFO, "pInfoCmn->pReceiveBuf         = %p", pInfoCmn->pReceiveBuf );

        GS_DBG( GS_OUT__INIT_INFO, "pInfoMainCnt                  = %p", pInfoMainCnt );
        GS_DBG( GS_OUT__INIT_INFO, "pInfoMainCnt->maxLun          = %d"   , (int)pInfoMainCnt->maxLun );

        pInfoLun = pInfoLunOrg;
        for ( i = 0; i < num; i++ ){
            GS_DBG( GS_OUT__INIT_INFO, "pInfoLun[%d]                  = %p", i, pInfoLun );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoLun[%d]->pSendReadBuf[0] = %p", i, pInfoLun->pSendReadBuf[0] );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoLun[%d]->pSendReadBuf[1] = %p", i, pInfoLun->pSendReadBuf[1] );
            GS_DBG( GS_OUT__INIT_INFO, "pInfoLun[%d]->pDrive          = %p", i, pInfoLun->pDrive );
            pInfoLun++;
        }
    }

    /* (1) pInfo  ==========================================*/
    if ( pInfo ){
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfo)                       =%p", pInfo );
        kfree( pInfo );
    }

    /* (3) ppInfoDrvOrg  ===================================*/
    if ( ppInfoDrvOrg ){
        GS_DBG( GS_OUT__INIT_INFO, "kfree(ppInfoDrvOrg)                =%p", ppInfoDrvOrg );
        kfree( ppInfoDrvOrg );
    }

    /* (3) pInfoDrvOrg  ====================================*/
    if ( pInfoDrvOrg ){
        pInfoDrv = pInfoDrvOrg;
        for ( i = 0; i < num; i++ ){
            if ( pInfoDrv->devNode ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->devNode)        =%p", i, pInfoDrv->devNode );
                kfree( pInfoDrv->devNode );
            } else {
                break;
            }
            if ( pInfoDrv->pInquiryData ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrv[%d]->pInquiryData)   =%p", i, pInfoDrv->pInquiryData );
                kfree( pInfoDrv->pInquiryData );
            } else {
                break;
            }
            pInfoDrv++;
        }
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoDrvOrg)                 =%p", pInfoDrvOrg );
        kfree( pInfoDrvOrg );
    }

    /* (4) pInfoCmn  =======================================*/
    if ( pInfoCmn ){
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoCmn)                    =%p", pInfoCmn );
        kfree( pInfoCmn );
    }

    /* (5) pInfoMainCnt  ===========================*/
    if ( pInfoMainCnt ){
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoMainCnt)                =%p", pInfoMainCnt );
        kfree( pInfoMainCnt );
    }

    /* (6) pInfoLunOrg  ====================================*/
    if ( pInfoLunOrg ){
        pInfoLun = pInfoLunOrg;

        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoLunOrg)                 =%p", pInfoLunOrg );
        kfree( pInfoLunOrg );
    }

    if( ppInfoExtCmd ) {
        GS_DBG( GS_OUT__INIT_INFO, "kfree(ppInfoExtCmd)                 =%p", ppInfoExtCmd );
        kfree( ppInfoExtCmd );
    }
    if( pInfoExtCmd ) {
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pInfoExtCmd)                  =%p", pInfoExtCmd );
        kfree( pInfoExtCmd );
    }

    return err;
}

/**
 * storage_ioctl_setMediaStatus()
 **/
static int storage_ioctl_setMediaStatus( struct inode *inode, struct file *fd, void *p_arg )
{
    int                       err;
    GSTORAGE__InfoMediaStatus info, *pInfo;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    pInfo = ( GSTORAGE__InfoMediaStatus * )p_arg;

    err = copy_from_user( &info, pInfo, sizeof( GSTORAGE__InfoMediaStatus ) );
    if ( err ){
        GS_ERR( GS_OUT__MEDIA_STATUS, "copy_from_user() error" );
        err = -ENOBUFS;
        goto SUB_RET;
    }
    storage_maincnt_reqSetMediaStatus( &info );

SUB_RET:
    return err;
}

/**
 * storage_comp_setMediaStatus()
 **/
void storage_comp_setMediaStatus( GSTORAGE__Status stat, unsigned int lun )
{
    struct GSTORAGE__Event *pEvent;
    struct func_data       *p_this = pCmn->pThis;
    GSTORAGE__CmpSetMediaStatusCDB cdb;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );
    
    if ( stat ){
        cdb.errStat = -ENOBUFS;
    }
    
    /* complun */
    cdb.lun = lun;
    
    /* MassStorage Svc  SetMediaStatus λ Event  */
    pEvent = p_this->info.vp_user;
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
                         pEvent->CompSetMediaStatus,
                         (usb_hndl_t)(p_this->info.hndl),
                         GSTORAGE__KEVENT_ID_COMP_SET_MEDIA_STATUS,
                         sizeof(GSTORAGE__CmpSetMediaStatusCDB),
                         (void*)&cdb );

    return;
}

/**
 * storage_ioctl_checkMediaAccess()
 **/
static int storage_ioctl_checkMediaAccess( struct inode *inode, struct file *fd, void *p_arg )
{
    unsigned char mediaAccess;

    GS_DET( GS_OUT__CMN, "func Req" );

    storage_drvcnt_checkMediaAccess( &mediaAccess );
    if(copy_to_user( ( unsigned char * )p_arg, &mediaAccess, sizeof( mediaAccess ) ) ){
        ;
    }

    return 0;
}

/**
 * storage_detect_ExtCommand()
 **/
void storage_detect_ExtCommand( int lun, GSTORAGE__Cdb *pCdb )
{
    struct GSTORAGE__Event *pEvent;
    struct func_data       *p_this = pCmn->pThis;
    GSTORAGE__ExtCmdCDB    cdb;

    GS_DET( GS_OUT__EXTCOMMAND, "func Req" );

    cdb.lun = lun;
    memcpy( cdb.CDB, pCdb, sizeof( cdb.CDB ) );

    /* MassStorage Svc  ExtCommand  Event  */
    pEvent = p_this->info.vp_user;
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
                         pEvent->ExtCommand,
                         ( usb_hndl_t )( p_this->info.hndl ),
                         GSTORAGE__KEVENT_ID_EXT_COMMAND,
                         sizeof( GSTORAGE__ExtCmdCDB ),
                         ( void * )&cdb );

    return;
}


/************************************************************************/
/* [Core]                                                               */
/************************************************************************/
/**
 * storage_start()
 *
 *    Service ¦ϿSetConfig,SetInterfaceơ
 *          FuncDriverư򳫻ϤȤ˸ƤФ
 * ֤ ʤ
 **/
static void storage_start( struct usb_gadget_func_driver *p_drv,
                           unsigned char                 alt   ,
                           struct usb_gadget_ep_list     list   )
{
    int err = 0, i, j;
    struct func_data *p_this = ( struct func_data * )( p_drv->context );
    struct usb_ep    *p_ep;
    struct usbg_storage_ep_data *p_ep_data;

    GS_DET( GS_OUT__START, "func Req" );

    set_log_data( GS_INT__START, GS_INT__START );
    /* <framework> */
    if ( usbg_cmn_make_ep_list( p_this, alt, list ) ){
        GS_ERR( GS_OUT__START, "make ep list" );
        err = -ENOBUFS;
        return;
    }
    /* </framework> */

    /* <user> */
    /* ̽ */
    /* ꥯȥơ֥   */
    i = 0;
    while( i < list.num_ep ){
        p_ep = list.ep[ i ];

        /* EP ̤Υǡ¤ΤμΤ */
        p_ep_data = p_ep->driver_data
                  = kmalloc( sizeof( struct usbg_storage_ep_data ), GFP_ATOMIC );
        if ( !p_ep_data ){
            err = -ENOBUFS;
            goto SUB_RET;
        }

        /* EP̤Υǡ¤Τ */
        memset( p_ep_data, 0, sizeof( *p_ep_data ) );

        /* EP̤request¤Τ */
        p_ep_data->req_data = kmalloc( sizeof( struct GSTORAGE__Req ), GFP_ATOMIC );
        if ( p_ep_data->req_data == NULL ){
            err = -ENOBUFS;
            goto SUB_RET;
        }

        /* request֥Ȥ */
        p_ep_data->req_data->req = usb_ep_alloc_request( p_ep, GFP_ATOMIC );
        if ( p_ep_data->req_data->req == NULL ){
            err = -ENOBUFS;
            goto SUB_RET;
        }

        /* GSTORAGE__Thread ¤Τ EP ݥ󥿤򥻥å */
        storage_maincnt_setEp( p_ep, i );

        i++;
    }
    /* </user> */

    /* MassStorage Reset λԤե饰ν */
    f_waitReset    = GSTORAGE__OFF;     /* MassStorage Reset Ԥ  */
    f_waitClearIn  = GSTORAGE__OFF;     /* Clear Feature(IN) Ԥ  */
    f_waitClearOut = GSTORAGE__OFF;     /* Clear Feature(OUT) Ԥ */
    f_calledStorageIN = GSTORAGE__OFF;  /* MassStorageINƤФ줿ɤMassStorageOUTǥꥻåȤ롣 */

    /* <framework> */
    /* maincnt  Start ׵ */
    storage_maincnt_reqStart();
    /* </framework> */

SUB_RET:
    if ( err ){
        for( j = 0; j < i; j++ ){
            clear_ep_data( list.ep[ j ] );
        }
    }

    return;
}

/**
 * storage_stop()
 *
 *    Service ¦ϿFuncDriverưߤȤ˸ƤФ
 * ֤ ʤ
 **/
static void storage_stop( struct usb_gadget_func_driver *p_drv )
{
    int i;
    struct func_data *p_this = ( struct func_data * )( p_drv->context );

    GS_DET( GS_OUT__STOP, "func Req" );

    set_log_data( GS_INT__STOP, GS_INT__STOP );
    /* <user> */
    storage_maincnt_clearEp();

    for( i = 0; i < p_this->uc_ep_num; i++ ){
        if ( p_this->ptbl_ep[ i ] != NULL ){
            storage_dequeue_ep( p_this->ptbl_ep[ i ] );
            clear_ep_data( p_this->ptbl_ep[ i ] );
        }
    }
    /* </user> */

    /* <framework> */
    usbg_cmn_clear_ep_list( p_this );
    /* </framework> */

    /* <framework> */
    /* maincnt  Stop ׵ */
    storage_maincnt_reqStop();
    /* </framework> */

    return;
}

/**
 * storage_storageIn()
 **/
void storage_storageIn( void )
{
    struct GSTORAGE__Event *pEvent;
    struct func_data       *p_this = pCmn->pThis;
    unsigned long           flags;
    GSTORAGE__Thread *pStorageMainCnt;
    pStorageMainCnt = storage_maincnt_getThread();

    GS_DET( GS_OUT__START, "func Req" );

    /* open extCommand */
    storage_drvcnt_extcmdOpen();

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock_storage_in, flags );

    if( f_calledStorageIN == GSTORAGE__ON ){
        spin_unlock_irqrestore( &pStorageMainCnt->lock_storage_in, flags );
        /* ޤ¾ */
        return;
    }
    else{
        GS_DET( GS_OUT__START, "## Now Calling massStorageIN");
        f_calledStorageIN = GSTORAGE__ON;
        spin_unlock_irqrestore( &pStorageMainCnt->lock_storage_in, flags );
        /* ޤ¾ */
    }

    /* MassStorage Svc  Start λ Event  */
    pEvent = p_this->info.vp_user;
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
                         pEvent->MassStorageIN,
                         (usb_hndl_t)(p_this->info.hndl),
                         GSTORAGE__KEVENT_ID_MASSSTORAGE_IN,
                         0,
                         (void*)NULL );

    return;
}

/**
 * storage_storageOut()
 **/
void storage_storageOut( void )
{
    struct GSTORAGE__Event *pEvent;
    struct func_data       *p_this = pCmn->pThis;
    unsigned long          flags;
    GSTORAGE__Thread *pStorageMainCnt;
    pStorageMainCnt = storage_maincnt_getThread();

    GS_DET( GS_OUT__STOP, "func Req" );

    /* close extCommand */
    storage_drvcnt_extcmdClose();

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock_storage_in, flags );
    f_calledStorageIN = GSTORAGE__OFF;
    spin_unlock_irqrestore( &pStorageMainCnt->lock_storage_in, flags );
    /* ޤ¾ */

    /* MassStorage Svc  Start λ Event  */
    pEvent = p_this->info.vp_user;
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
                         pEvent->MassStorageOUT,
                         (usb_hndl_t)(p_this->info.hndl),
                         GSTORAGE__KEVENT_ID_MASSSTORAGE_OUT,
                         0,
                         (void*)NULL );

    return;
}

/**
 * storage_ep_set_halt()
 *
 *    EPSetFeatureȤ˸ƤФ
 * ֤ STALL  ProtocolStall(GadgetCore)
 *          0      ｪλ(FuncDriverǽ)
 **/
static int storage_ep_set_halt( struct usb_gadget_func_driver *p_drv, struct usb_ep *p_ep )
{
    int err = 0;
    struct usbg_storage_ep_data *p_ep_data = p_ep->driver_data;
    struct usb_request *p_req = p_ep_data->req_data->req;
    GSTORAGE__Thread *pStorageMainCnt;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
    /* no exclusion */
#else
    unsigned long    flags;
#endif

    GS_DET( GS_OUT__SET_HALT, "func Req" );

    set_log_data( GS_INT__SET_HALT, GS_INT__SET_HALT );
    
    pStorageMainCnt = storage_maincnt_getThread();
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
    /* no exclusion */
#else    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );
#endif
    
    /* <user> */
    /*------------------------------------------------------------------*/
    /* ̾ư                                                     */
    /*     EPqueueƤrequest dequeue         */
    /*     usb_ep_set_halt() Ƥ                                   */
    /* ¾ư                                                   */
    /*     1. EPϡʬ椷ƤʤȤˤ                  */
    /*        err = USB_GADGETCORE_THRU ֤                          */
    /*     2. EPHalt˽ʤȤˤ                          */
    /*        err = USB_GADGETCORE_STALL ֤                         */
    /*------------------------------------------------------------------*/
    usb_ep_dequeue( p_ep, p_req );
    /* </user> */

    /* <framework> */
    if ( usb_ep_set_halt( p_ep ) != 0 ){
        GS_INF( GS_OUT__SET_HALT, "usb_ep_set_halt() error" );
    }
    /* </framework> */
    
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
    /* no exclusion */
#else
    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
#endif

    return err;
}

/**
 * storage_ep_clear_halt()
 *
 *    EPClearFeatureȤ˸ƤФ
 * ֤ STALL  ProtocolStall(GadgetCore)
 *          0      ｪλ(FuncDriverǽ)
 **/
static int storage_ep_clear_halt( struct usb_gadget_func_driver *p_drv, struct usb_ep *p_ep )
{
    int err = 0;
    struct usb_ep *p_ep_snd;
    struct usb_ep *p_ep_rcv;
    GSTORAGE__Thread *pStorageMainCnt;
    unsigned long    flags;

    GS_DET( GS_OUT__CLEAR_HALT, "func Req" );

    set_log_data( GS_INT__CLEAR_HALT, GS_INT__CLEAR_HALT );
    
    pStorageMainCnt = storage_maincnt_getThread();
    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );
    
    /* <user> */
    /*------------------------------------------------------------------*/
    /* ̾ư                                                     */
    /*     usb_ep_clear_halt() Ƥ                                 */
    /* ¾ư                                                   */
    /*     1. EPϡʬ椷ƤʤȤˤ                  */
    /*        err = USB_GADGETCORE_THRU ֤                          */
    /*     2. EPHaltʤȤˤ                        */
    /*        err = USB_GADGETCORE_STALL ֤                         */
    /*------------------------------------------------------------------*/
    /* </user> */

    /* Reset Recovery λԤȽ */
    if ( ( f_waitClearIn  == GSTORAGE__ON ) ||
         ( f_waitClearOut == GSTORAGE__ON ) ){
        /* MassStorage Reset Ƚ */
        if ( f_waitReset == GSTORAGE__OFF ){
            if ( p_ep == storage_maincnt_getEp( GSTORAGE_EP_SEND ) ){
                f_waitClearIn = GSTORAGE__OFF;
            }
            else if ( p_ep == storage_maincnt_getEp( GSTORAGE_EP_RECEIVE ) ){
                f_waitClearOut = GSTORAGE__OFF;
            }
            /* Reset Recovery λʤ Clear Halt(IN, OUT) + Message  */
            if ( ( f_waitReset    == GSTORAGE__OFF ) &&
                 ( f_waitClearIn  == GSTORAGE__OFF ) &&
                 ( f_waitClearOut == GSTORAGE__OFF ) ){
                p_ep_snd = storage_maincnt_getEp( GSTORAGE_EP_SEND );
                p_ep_rcv = storage_maincnt_getEp( GSTORAGE_EP_RECEIVE );
                if ( ( usb_ep_clear_halt( p_ep_snd ) != 0 ) ||
                     ( usb_ep_clear_halt( p_ep_rcv ) != 0 ) ){
                    GS_INF( GS_OUT__RESET_RECOVERY, "usb_ep_clear_halt() error" );
                }
                storage_maincnt_compResetRecovery();
            }
        }
    }
    else {
        if ( usb_ep_clear_halt( p_ep ) != 0 ){
            GS_INF( GS_OUT__CLEAR_HALT, "usb_ep_clear_halt() error" );
        }
        storage_maincnt_epClearHalt( p_ep );
    }

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
    
    return err;
}

/**
 * storage_class()
 *
 *    FuncDriver  ClassSpecificRequestȤ˸ƤФ
 * ֤ STALL  ProtocolStall( GadgetCore  )
 *          0      ｪλ( FuncDriver ǽ )
 **/
static int storage_class( struct usb_gadget_func_driver *p_drv ,
                          const struct usb_ctrlrequest  *p_ctrl,
                          struct usb_ep                 *p_ep0  )
{
    int                err = 0, status;
    unsigned int       length;
    struct usb_request *p_req;
    GSTORAGE__Thread   *pStorageMainCnt;
    unsigned long      flags;

    GS_DET( GS_OUT__CLASS, "func Req" );

    /* <user> */
    /*------------------------------------------------------------------*/
    /* ̾ư                                                     */
    /*     p_ctrl 鼫ʬSETUPѥåȤȽ             */
    /*       - ʬѥåȤλˤϡ err = 0 ֤            */
    /*       - ʬʤѥåȤΤȤˤϡ                       */
    /*         err = USB_GADGETCORE_THRU ֤                         */
    /*       - ʬѥåȤǡStall ˤϡ               */
    /*         err = USB_GADGETCORE_STALL ֤                        */
    /*       StatusStageޤǤ                                */
    /* ¾ư                                                   */
    /*------------------------------------------------------------------*/
    /* MassStorage Reset, GetMaxLUN Τб */
    if ( !( p_ctrl->bRequestType == 0x21   && p_ctrl->bRequest == 0xff   &&
            p_ctrl->wValue       == 0x0000 && p_ctrl->wIndex   == 0x0000 ) &&
         !( p_ctrl->bRequestType == 0xa1   && p_ctrl->bRequest == 0xfe   &&
            p_ctrl->wValue       == 0x0000 && p_ctrl->wIndex   == 0x0000 ) ){
        err = USB_GADGETCORE_THRU;
        goto SUB_RET;
    }

    /* EP0 Ѥ request ֥Ȥ */
    p_req = usb_ep_alloc_request( p_ep0, GFP_ATOMIC );
    if ( !p_req ){
        err = USB_GADGETCORE_STALL;
        goto SUB_RET;
    }
    /* EP0 Ѥ buffer  */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
    p_req->buf = kmalloc( 4, GFP_ATOMIC );
#else
    p_req->buf = usb_ep_alloc_buffer( p_ep0, 4, NULL, GFP_ATOMIC );
#endif
    if ( !p_req->buf ){
        err = USB_GADGETCORE_STALL;
        goto SUB_RET;
    }

    /* žѥХåեѥ᡼ꤹ */
    /* MassStorage Reset */
    if ( p_ctrl->bRequestType == 0x21  ){
        GS_DBG( GS_OUT__CLASS, "Mass Storage Reset" );
        set_log_data( GS_INT__STORAGE_RESET, GS_INT__STORAGE_RESET );
        
        // For CV MSC Test.
        if( p_ctrl->wLength != 0 ){
            err = USB_GADGETCORE_STALL;
            goto SUB_RET;
        }
        
        /* Reset Recovery λԤȽ */
        pStorageMainCnt = storage_maincnt_getThread();
        /* ¾ */
        spin_lock_irqsave( &pStorageMainCnt->lock, flags );
        
        if ( ( f_waitClearIn  == GSTORAGE__ON ) ||
             ( f_waitClearOut == GSTORAGE__ON ) ){
            f_waitReset = GSTORAGE__OFF;
        }
        else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
            /*
             * storage_maincnt_compMassStorageReset() calls usb_ep_dequeue()
             * which requires unlocked and irq enabled
             */
            spin_unlock_irq(&pStorageMainCnt->lock);
#endif
            /* MassStorage Reset ׵(To MainCnt) */
            storage_maincnt_compMassStorageReset();

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
            spin_lock_irq(&pStorageMainCnt->lock);
#endif
        }
        
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        
        // Issue vacant request for Status stage.
        p_req->length       = 0;
        GS_INF( GS_OUT__CMN, "%s: size=0, aligned=%d\n", __FUNCTION__, p_req->length);
        p_req->complete     = ep0_complete;
        p_req->zero         = 1;
        p_req->short_not_ok = 0;
        p_req->no_interrupt = 0;
        
        status = usb_ep_queue( p_ep0, p_req, GFP_ATOMIC );
        
        if ( status ){
            error_ep_request_control_in( status );
            
            if ( p_ep0 ){
                
                if ( p_req->buf ){
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
                    kfree( p_req->buf );
#else
                    usb_ep_free_buffer( p_ep0, p_req->buf, p_req->dma, p_req->length );
#endif
                }
                
                usb_ep_free_request( p_ep0, p_req );
                
            }
            
            err = USB_GADGETCORE_STALL;
            goto SUB_RET;
            
        }
        
    }
    /* Get Max LUN */
    else if ( p_ctrl->bRequestType == 0xa1 ){
        GS_DBG( GS_OUT__CLASS, "Get Max LUN" );
        set_log_data( GS_INT__GET_MAX_LUN, GS_INT__GET_MAX_LUN );
        
        // For CV MSC Test.
        if( p_ctrl->wLength != 1 ){
            err = USB_GADGETCORE_STALL;
            goto SUB_RET;
        }
        length = p_ctrl->wLength; // length is "1" only.
        
        memset( p_req->buf, ( char )pCmn->maxLunNum, length );
        p_req->length       = length;
        p_req->complete     = ep0_complete;
        p_req->zero         = ( length < p_ctrl->wLength && ( length % p_ep0->maxpacket ) == 0 );
        p_req->short_not_ok = 0;
        p_req->no_interrupt = 0;

        status = usb_ep_queue( p_ep0, p_req, GFP_ATOMIC );
        if ( status ){
            error_ep_request_control_in( status );
            if ( p_ep0 ){
                if ( p_req->buf ){
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
                    kfree( p_req->buf );
#else
                    usb_ep_free_buffer( p_ep0, p_req->buf, p_req->dma, p_req->length );
#endif
                }
                usb_ep_free_request( p_ep0, p_req );
            }
            err = USB_GADGETCORE_STALL;
            goto SUB_RET;
        }
    }
    /* </user> */

SUB_RET:
    return err;
}

/**
 * storage_set_waitResetRecovery()
 **/
void storage_set_waitResetRecovery( void )
{
    GS_DET( GS_OUT__RESET_RECOVERY, "func Req" );

    f_waitReset    = GSTORAGE__ON;
    f_waitClearIn  = GSTORAGE__ON;
    f_waitClearOut = GSTORAGE__ON;

    return;
}

/**
 * storage_dequeue_ep()
 **/
void storage_dequeue_ep( struct usb_ep *p_ep )
{
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;

    GS_DET( GS_OUT__STOP, "func Req" );

    p_ep_data = p_ep->driver_data;
    p_req = p_ep_data->req_data->req;

    usb_ep_dequeue( p_ep, p_req );

    return;
}

/**
 * clear_ep_data()
 **/
static void clear_ep_data( struct usb_ep *p_ep )
{
    struct usbg_storage_ep_data *p_ep_data;

    GS_DET( GS_OUT__STOP, "func Req" );

    if ( p_ep ){
        /* p_ep  EP ǡ¤Τ */
        p_ep_data = p_ep->driver_data;
        if ( p_ep_data ){
            if ( p_ep_data->req_data ){
                if ( p_ep_data->req_data->req ){
                    /* usb_request֥Ⱥ */
                    usb_ep_free_request( p_ep, p_ep_data->req_data->req );
                    p_ep_data->req_data->req = NULL;
                }
                /* request¤Τ */
                kfree( p_ep_data->req_data );
                p_ep_data->req_data = NULL;
            }
            /* EP ̤Υǡ¤ΤμΤ */
            kfree( p_ep_data );
            p_ep->driver_data = NULL;
        }
    }

    return;
}

/**
 * ep0_complete()
 **/
static void ep0_complete( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GS_DET( GS_OUT__CLASS, "func Req" );

    if ( p_req->status ){
        /* usb_ep_queue() callback error [control in] */
        error_ep_request_cb_control_in( p_req->status );
    }
    if ( p_ep ){
        if ( p_req->buf ){
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
            kfree( p_req->buf );
#else
            usb_ep_free_buffer( p_ep, p_req->buf, p_req->dma, p_req->length );
#endif
        }
        usb_ep_free_request( p_ep, p_req );
    }
}

/**
 * error_ep_request_control_in()
 **/
static void error_ep_request_control_in( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );

    error_ep_request_put_message( err );

    return;
}

/**
 * error_ep_request_cb_control_in()
 **/
static void error_ep_request_cb_control_in( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );

    error_ep_request_put_message( err );

    return;
}

/**
 * gusb_storage_acsr_open()
 **/
static int gusb_storage_acsr_open(struct inode *inode, struct file *filp)
{
    GS_DET( GS_OUT__CMN, "func Req" );

    /* attention: calling after ioctl(USB_IOC_CMN_PROBE) */
    if (!cmn_probe_done) return -EPERM;

    return 0;
}

/**
 * gusb_storage_acsr_read()
 **/
static ssize_t gusb_storage_acsr_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
    GS_DET( GS_OUT__CMN, "func Req" );

    return storage_drvcnt_waitMediaAccess();
}


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/
/**
 * storage_init()
 *
 * סFunction Driver 
 **/
static int __init storage_init( void )
{
    int err = 0;
#ifdef GSTORAGE_IO_BLOCK_DEV
    int err2;
#endif
    struct proc_dir_entry *entry;

    GS_DET( GS_OUT__CMN, "func Req" );

    /*  procfsκ */
    entry = proc_mkdir( GSTORAGE_PROCFS_DIRECTORY, NULL);
    if (entry == NULL) {
        GS_ERR( GS_OUT__CMN, "proc_mkdir\n");
        return -ENOMEM;
    }
    
    entry = proc_create(GSTORAGE_ACCESS_INFO, S_IRUGO | S_IWUGO, entry, &gusb_acsr_proc_fops);
    if (entry == NULL) {
        GS_ERR( GS_OUT__CMN, "proc_create\n");
        return -ENOMEM;
    }

    err = usbg_cmn_reg( &sony_storage_dev );

#ifdef GSTORAGE_IO_BLOCK_DEV
    if(!err) {
        err2 = bdio_init();
        if(err2) {
            usbg_cmn_dereg( &sony_storage_dev );
            err = err2;
        }
    }
#endif

    if ( err ){
        GS_ERR( GS_OUT__CMN, "err=%d", err );
        goto SUB_RET;
    }
    GS_DBG( GS_OUT__CMN, "Success" );

SUB_RET:
    return err;
}
module_init( storage_init );

/**
 * storage_exit()
 *
 * סFunction Driver λ
 **/
static void __exit storage_exit( void )
{
    GS_DET( GS_OUT__CMN, "func Req" );

#ifdef GSTORAGE_IO_BLOCK_DEV
    bdio_fin();
#endif
    usbg_cmn_dereg( &sony_storage_dev );
    
    remove_proc_subtree(MY_NAME, NULL);

    GS_DBG( GS_OUT__CMN, "Success" );

    return;
}
module_exit( storage_exit );

