/*
 * usbg_storage_maincnt.c
 *
 * USB Mass Storage Gadget Function Driver
 *
 * Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * << MassStorage Gadget Main Controller Function >>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#define __USBG_STORAGE_MAINCNT_C__

#define STORAGEIN_AT_RCVCBW_CB


/*

 󥯥롼ɥե                                                 
*/
#include <asm/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>
#include <linux/udif/delay.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 21)
#include <linux/usb/ch9.h>
#else
#include <linux/usb_ch9.h>
#endif

#include <linux/device.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
#include <linux/usb/gadget.h>
#else
#include <linux/usb_gadget.h>
#endif

#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>
#include <linux/dma-mapping.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
#include <linux/kthread.h>
#endif

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cache.h>
#endif
#include <mach/noncache.h>

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_putpid_config.h>

#include "usbg_mbf.h"
#include "usbg_storage_conf.h"
#include "usbg_storage_cmn.h"
#include "usbg_storage_maincnt.h"
#include "usbg_storage_drvcnt.h"


/*

 ؿץȥ                                                 
*/
/* MassStorage  ====================================================*/
static GSTORAGE__State compProbeMassStorage( GSTORAGE__Msg * );

/* MassStorage  ====================================================*/
static GSTORAGE__State execRemoveMassStorage( GSTORAGE__Msg * );
static GSTORAGE__State reqRemoveMassStorageDuringWaitCompStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqRemoveMassStorageDuringReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqRemoveMassStorageDuringWaitCompStop( GSTORAGE__Msg * );

/* Thread ׵ =====================================================*/
static GSTORAGE__State reqStopDrvCntThread( GSTORAGE__Msg * );
static void            stopThreadCb( GSTORAGE__Status stat );

/* Main Controller Start ׵ ======================================*/
static GSTORAGE__State reqStart( GSTORAGE__Msg * );
static void            startCb( GSTORAGE__Status, unsigned char );
static GSTORAGE__State reqStartDuringWaitCompStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqStartDuringReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqStartDuringWaitCompStop( GSTORAGE__Msg * );

/* Main Controller Start λ ======================================*/
static GSTORAGE__State compStartDuringWaitCompStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State compStartDuringReqStartWaitCompStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State compStartDuringRemoveOn( GSTORAGE__Msg * );

/* Main Controller Stop ׵ =======================================*/
static GSTORAGE__State reqStop( GSTORAGE__Msg * );
static void            stopCb( GSTORAGE__Status stat );
static GSTORAGE__State reqStopDuringWaitCompStart( GSTORAGE__Msg * );
static GSTORAGE__State reqStopDuringReqStartWaitCompStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqStopDuringReqStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State reqStopDuringReqStartWaitCompStop( GSTORAGE__Msg * );
static GSTORAGE__State reqStopDuringSndCmd( GSTORAGE__Msg * );
static void            setDisEpQueueOff( void );

/* Main Controller Stop λ =======================================*/
static GSTORAGE__State compStop( GSTORAGE__Msg * );
static GSTORAGE__State compStopDuringReqStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State compStopDuringReqStartWaitCompStop( GSTORAGE__Msg * );

/* Send Command λ ( Stop  ) ===================================*/
static GSTORAGE__State compSndCmdDuringReqStop( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDuringWaitCompStop( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDuringReqStartReqStop( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDuringReqStartWaitCompStop( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDuringRemoveOn( GSTORAGE__Msg * );

/* Set Media Status Ϣ ===========================================*/
static GSTORAGE__State reqSetMediaStatus( GSTORAGE__Msg * );
static void            setMediaStatusCb( GSTORAGE__Status, unsigned char, unsigned int lun );
static GSTORAGE__State compSetMediaStatus( GSTORAGE__Msg * );
static GSTORAGE__State compSetMediaStatusDuringReqStartOrReqStop( GSTORAGE__Msg * );

/* CBW  ========================================================*/
static GSTORAGE__State rcvCbw( GSTORAGE__Msg * );
static GSTORAGE__State CheckAndrcvCbw( GSTORAGE__Msg * );
static void            receiveTimerEvent( unsigned long );
static GSTORAGE__State forcedRcvCbw( GSTORAGE__Msg * );
static void            rcvCbwCb( struct usb_ep *, struct usb_request * );

/* ޥ ====================================================*/
static GSTORAGE__State sndCmdPacket( GSTORAGE__Msg * );
static void            sndCmdPacketCb( GSTORAGE__Status, unsigned int, unsigned char, unsigned int * );

/* CSW  ========================================================*/
static GSTORAGE__State sndCsw( GSTORAGE__Msg * );
static void            sndCswCb( struct usb_ep *, struct usb_request * );

/* Send Command λ ===============================================*/
static GSTORAGE__State compSndCmdNonData( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDataIn( GSTORAGE__Msg * );
static GSTORAGE__State compSndCmdDataOut( GSTORAGE__Msg * );

/* Data λ =================================================*/
static void            sndDataInCb( struct usb_ep *p_ep, struct usb_request *p_req );
static GSTORAGE__State compSndDataIn( GSTORAGE__Msg * );
static void            sndDataInContinueCb( struct usb_ep *p_ep, struct usb_request *p_req );
static GSTORAGE__State compSndDataInContinue( GSTORAGE__Msg * );
static GSTORAGE__State sndDataHalt( GSTORAGE__Msg * );
static GSTORAGE__State waitUnstallPipes( GSTORAGE__Msg * );

/* Data λ =================================================*/
static void            rcvDataOutCb( struct usb_ep *, struct usb_request * );
static GSTORAGE__State compRcvDataOut( GSTORAGE__Msg * );
static void            rcvDataOutContinueCb( struct usb_ep *, struct usb_request * );
static GSTORAGE__State compRcvDataOutContinue( GSTORAGE__Msg * );

/* Reset Ϣ ======================================================*/
static GSTORAGE__State waitResetRecovery( GSTORAGE__Msg * );

/* ERROR Ϣ ==========================================================*/
static void error_ep_request_bulk_in( int );
static void error_ep_request_cb_bulk_in( int );
static void error_ep_request_bulk_out( int );
static void error_ep_request_cb_bulk_out( int );

/* ååɴϢ ============================================*/
static void maincnt_sendMsg( GSTORAGE__Msg *, int );
static int  maincnt_threadFunc( void * );


/*

                                                              
*/
#define MSG_MAINCNT_MAX_NUM     10              /* å     */

/* CBW/CSW */
#define STORAGE_CBW_SIGNATURE           0x55534243   /* CBW ͥ  */
#define STORAGE_CSW_SIGNATURE           0x55534253   /* CSW ͥ  */
#define STORAGE_CBW_SIZE                31           /* CBW       */
#define STORAGE_CSW_SIZE                13           /* CSW       */
#define STORAGE_CSW_STATUS_PASS         0            /* CSW ơ  */
#define STORAGE_CSW_STATUS_FAIL         1            /* CSW ơ  */
#define STORAGE_CSW_STATUS_PHASE_ERROR  2            /* CSW ơ  */
#define STORAGE_CSW_STATUS_DEFAULT      3            /* CSW ơ  */

#define SET_HALT_RETRY_CNT              5       /* SetHaltȥ饤  */

/************************************************************************/
/* Message ID( Main Controller )                                        */
/************************************************************************/
enum {
    MSG_MAINCNT__EOT = GSTORAGE__MSG_TBL_EOT,   /* End of Table         */
    /* probe, remove  */
    MSG_MAINCNT__COMP_PROBE,                /* probe λ               */
    MSG_MAINCNT__REQ_REMOVE,                /* remove ׵              */
    /* MAINCNT  */
    MSG_MAINCNT__REQ_START,                 /* Main Cnt ׵        */
    MSG_MAINCNT__REQ_STOP,                  /* Main Cnt ׵        */
    MSG_MAINCNT__REQ_SET_MEDIA_STATUS,      /* Media Ready          */
    /* DRVCNT  */
    MSG_MAINCNT__COMP_STOP_DRVCNT_THREAD,   /* Drv Cnt Thread ׵  */
    MSG_MAINCNT__COMP_START,                /* Drv Cnt ϴλ     */
    MSG_MAINCNT__COMP_STOP,                 /* Drv Cnt ߴλ     */
    MSG_MAINCNT__COMP_SET_MEDIA_STATUS,     /* Media Ready λ     */
    /* Sequense  */
    MSG_MAINCNT__COMP_RCV_CBW,              /* CBW λ             */
    MSG_MAINCNT__COMP_SND_CSW,              /* CSW λ             */
    MSG_MAINCNT__COMP_SND_CMD,              /* ޥλ         */
    MSG_MAINCNT__COMP_SND_DATA,             /* ǡλ           */
    MSG_MAINCNT__COMP_SND_DATA_CONTINUE,    /* ǡλ(CONTINUE) */
    MSG_MAINCNT__COMP_RCV_DATA,             /* ǡλ           */
    MSG_MAINCNT__COMP_RCV_DATA_CONTINUE,    /* ǡλ(CONTINUE) */
    /* MASSSTORAGE RESET  */
    MSG_MAINCNT__COMP_MASS_STORAGE_RESET,   /* MassStorage Reset λ   */
    /* RESET RECOVERY  */
    MSG_MAINCNT__COMP_RESET_RECOVERY,       /* Reset Recovery λ      */
    /* CLEAR HALT  */
    MSG_MAINCNT__COMP_UNSTALL_BULK_IN,      /* Clear Halt (Bulk IN)     */
    MSG_MAINCNT__COMP_UNSTALL_BULK_OUT,     /* Clear Halt (Bulk OUT)    */
    /* TIMER  */
    MSG_MAINCNT__COMP_TIMER_EVENT,          /* ޥ٥ȴλ       */
};

/************************************************************************/
/* MassStorage ξ                                               */
/************************************************************************/
#define MAINCNT_STATE__NO_CHANGE  GSTORAGE__STATE_NO_CHANGE /* ܤʤ */
enum {
    /********************************************/
    /* Probe OFF                                */
    /********************************************/
    PROBE_OFF = 0,
    /********************************************/
    /* Probe ON, Remove OFF                     */
    /********************************************/
    /* Drive Controller Thread Ԥ ----*/
    MAINCNT__WAIT_COMP_STOP_DRVCNT_THREAD,
    /* Main Controller OFF  ----------------*/
    MAINCNT_OFF,
    MAINCNT_OFF__WAIT_COMP_START,
    MAINCNT_OFF__WAIT_COMP_START__REQ_STOP,
    MAINCNT_OFF__REQ_START__WAIT_COMP_START__REQ_STOP,
    /* Main Controller ON  -----------------*/
    MAINCNT_ON__REQ_STOP,
    MAINCNT_ON__WAIT_COMP_STOP,
    MAINCNT_ON__REQ_START__REQ_STOP,
    MAINCNT_ON__REQ_START__WAIT_COMP_STOP,
    /********************************************/
    /* Probe ON, Remove ON                      */
    /********************************************/
    MAINCNT_OFF__REMOVE_ON__WAIT_COMP_START__REQ_STOP,
    MAINCNT_ON__REMOVE_ON__REQ_STOP,
    MAINCNT_ON__REMOVE_ON__WAIT_COMP_STOP,
    /********************************************/
    /* Probe ON, Remove OFF, Command Sequence   */
    /********************************************/
    /* CBW, CSW --------------------------------*/
    MAINCNT_CMD__WAIT_RCV_CBW,
    MAINCNT_CMD__WAIT_SND_CSW,
    /* Non Data---------------------------------*/
    MAINCNT_CMD__WAIT_SND_CMD_NON_DATA,
    /* Data In ---------------------------------*/
    MAINCNT_CMD__WAIT_SND_CMD_DATA_IN,
    MAINCNT_CMD__WAIT_SND_DATA,
    MAINCNT_CMD__WAIT_UNSTALL_SND_DATA,
    MAINCNT_CMD__WAIT_UNSTALL_PIPES,
    /* Data Out --------------------------------*/
    MAINCNT_CMD__WAIT_RCV_DATA,
    MAINCNT_CMD__WAIT_SND_CMD_DATA_OUT,
    /* Illegal ---------------------------------*/
    MAINCNT_CMD__WAIT_COMP_RESET_RECOVERY,
};

/*

 ѿ                                                             
*/
/* Message Buffer */
static GSTORAGE__Mbf *pMbf = NULL;
/* Thread Information */
static GSTORAGE__Thread *pStorageMainCnt = NULL;
/* Main Context */
static GSTORAGE_MainContext *pMainCtx = NULL;

#if GS_OUT__GET_LOG
/* status log */
statusLog    log[ MAX_LOG_CNT ];
unsigned int logCnt;
#endif


/**
    Message Table
 **/
/********************************************/
/* Probe OFF                                */
/********************************************/
/* Probe OFF ( ¾ Probe ON ) ---------------------------------*/
static const GSTORAGE__MsgTbl storageMsgTbl_ProbeOff[] = {
    { MSG_MAINCNT__COMP_PROBE           , compProbeMassStorage },
    { MSG_MAINCNT__EOT                  , 0                    },
};
/********************************************/
/* Probe ON, Remove OFF                     */
/********************************************/
/* Drive Controller Thread Ԥ ------------------------------------*/
static const GSTORAGE__MsgTbl storageMsgTbl_WaitCompStopDrvCntThread[] = {
    { MSG_MAINCNT__COMP_STOP_DRVCNT_THREAD, execRemoveMassStorage },
    { MSG_MAINCNT__EOT                    , 0                     },
};
/* Main Controller OFF  --------------------------------------------*/
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOff[] = {
    { MSG_MAINCNT__REQ_REMOVE           , reqStopDrvCntThread                       },
    { MSG_MAINCNT__REQ_START            , reqStart                                  },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOff_WaitCompStart[] = {
    { MSG_MAINCNT__COMP_START           , rcvCbw                                    },
    { MSG_MAINCNT__REQ_STOP             , reqStopDuringWaitCompStart                },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOff_WaitCompStart_ReqStop[] = {
    { MSG_MAINCNT__REQ_REMOVE           , reqRemoveMassStorageDuringWaitCompStartReqStop },
    { MSG_MAINCNT__REQ_START            , reqStartDuringWaitCompStartReqStop             },
    { MSG_MAINCNT__COMP_START           , compStartDuringWaitCompStartReqStop            },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop      },
    { MSG_MAINCNT__EOT                  , 0                                              },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOff_ReqStart_WaitCompStart_ReqStop[] = {
    { MSG_MAINCNT__COMP_START           , compStartDuringReqStartWaitCompStartReqStop },
    { MSG_MAINCNT__REQ_STOP             , reqStopDuringReqStartWaitCompStartReqStop   },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop   },
    { MSG_MAINCNT__EOT                  , 0                                           },
};
/* Main Controller ON  ---------------------------------------------*/
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_ReqStop[] = {
    { MSG_MAINCNT__REQ_REMOVE           , reqRemoveMassStorageDuringReqStop         },
    { MSG_MAINCNT__REQ_START            , reqStartDuringReqStop                     },
    { MSG_MAINCNT__COMP_STOP            , compStop                                  },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringReqStop                   },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitCompStop[] = {
    { MSG_MAINCNT__REQ_REMOVE           , reqRemoveMassStorageDuringWaitCompStop    },
    { MSG_MAINCNT__REQ_START            , reqStartDuringWaitCompStop                },
    { MSG_MAINCNT__COMP_STOP            , compStop                                  },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringWaitCompStop              },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_ReqStart_ReqStop[] = {
    { MSG_MAINCNT__REQ_STOP             , reqStopDuringReqStartReqStop    },
    { MSG_MAINCNT__COMP_STOP            , compStopDuringReqStartReqStop   },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringReqStartReqStop },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop     },
    { MSG_MAINCNT__EOT                  , 0                                             },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_ReqStart_WaitCompStop[] = {
    { MSG_MAINCNT__REQ_STOP             , reqStopDuringReqStartWaitCompStop    },
    { MSG_MAINCNT__COMP_STOP            , compStopDuringReqStartWaitCompStop   },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringReqStartWaitCompStop },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop     },
    { MSG_MAINCNT__EOT                  , 0                                             },
};
/********************************************/
/* Probe ON, Remove ON                      */
/********************************************/
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOff_RemoveOn_WaitCompStart_ReqStop[] = {
    { MSG_MAINCNT__COMP_START           , compStartDuringRemoveOn                   },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_RemoveOn_ReqStop[] = {
    { MSG_MAINCNT__COMP_STOP            , reqStopDrvCntThread                       },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringRemoveOn                  },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_RemoveOn_WaitCompStop[] = {
    { MSG_MAINCNT__COMP_STOP            , reqStopDrvCntThread                       },
    { MSG_MAINCNT__COMP_SND_CMD         , compSndCmdDuringRemoveOn                  },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS, compSetMediaStatusDuringReqStartOrReqStop },
    { MSG_MAINCNT__EOT                  , 0                                         },
};
/********************************************/
/* Probe ON, Remove OFF, Command Sequence   */
/********************************************/
/* CBW Ϣ */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitRcvCbw[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop              },
    { MSG_MAINCNT__COMP_RCV_CBW           , sndCmdPacket         },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__COMP_TIMER_EVENT       , forcedRcvCbw         },
    { MSG_MAINCNT__EOT                    , 0                    }
};
/* Send CSW */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitSndCsw[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop              },
    { MSG_MAINCNT__COMP_SND_CSW           , rcvCbw               },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__EOT                    , 0                    },
};
/* Non Data */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitSndCmdNonData[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStopDuringSndCmd  },
    { MSG_MAINCNT__COMP_SND_CMD           , compSndCmdNonData    },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__EOT                    , 0                    },
};
/* Data IN */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitSndCmdDataIn[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStopDuringSndCmd  },
    { MSG_MAINCNT__COMP_SND_CMD           , compSndCmdDataIn     },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__EOT                    , 0                    },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitSndData[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop               },
    { MSG_MAINCNT__COMP_SND_DATA          , compSndDataIn         },
    { MSG_MAINCNT__COMP_SND_DATA_CONTINUE , compSndDataInContinue },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus     },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus    },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes     },
    { MSG_MAINCNT__EOT                    , 0                     },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitUnstallSndData[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop              },
    { MSG_MAINCNT__COMP_UNSTALL_BULK_IN   , sndCsw               },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__EOT                    , 0                    },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitUnstallPipes[] = {
    { MSG_MAINCNT__REQ_STOP              , reqStop            },
    { MSG_MAINCNT__COMP_UNSTALL_BULK_IN  , CheckAndrcvCbw     },
    { MSG_MAINCNT__COMP_UNSTALL_BULK_OUT , CheckAndrcvCbw     },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS  , reqSetMediaStatus  },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS , compSetMediaStatus },
    { MSG_MAINCNT__EOT                   , 0                  },
};
/* Data OUT */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitRcvData[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop                },
    { MSG_MAINCNT__COMP_RCV_DATA          , compRcvDataOut         },
    { MSG_MAINCNT__COMP_RCV_DATA_CONTINUE , compRcvDataOutContinue },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus      },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus     },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes      },
    { MSG_MAINCNT__EOT                    , 0                      },
};
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitSndCmdDataOut[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStopDuringSndCmd  },
    { MSG_MAINCNT__COMP_SND_CMD           , compSndCmdDataOut    },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus    },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus   },
    { MSG_MAINCNT__COMP_MASS_STORAGE_RESET, waitUnstallPipes    },
    { MSG_MAINCNT__EOT                    , 0                    },
};
/* Illegal State */
static const GSTORAGE__MsgTbl storageMsgTbl_MainCntOn_WaitResetRecovery[] = {
    { MSG_MAINCNT__REQ_STOP               , reqStop            },
    { MSG_MAINCNT__REQ_SET_MEDIA_STATUS   , reqSetMediaStatus  },
    { MSG_MAINCNT__COMP_SET_MEDIA_STATUS  , compSetMediaStatus },
    { MSG_MAINCNT__COMP_RESET_RECOVERY    , rcvCbw             },
    { MSG_MAINCNT__EOT                    , 0                  },
};

/* Function Table [ Gadget Storage Controller ] */
static const GSTORAGE__MsgTbl *spStorageMsgTbl_MainCnt[] = {
    /********************************************/
    /* Probe OFF                                */
    /********************************************/
    storageMsgTbl_ProbeOff,
    /********************************************/
    /* Probe ON, Remove OFF                     */
    /********************************************/
    /* Wait Complete Stop Drive Controller Thread */
    storageMsgTbl_WaitCompStopDrvCntThread,
    /* Main Cntroller Off */
    storageMsgTbl_MainCntOff,
    storageMsgTbl_MainCntOff_WaitCompStart,
    storageMsgTbl_MainCntOff_WaitCompStart_ReqStop,
    storageMsgTbl_MainCntOff_ReqStart_WaitCompStart_ReqStop,
    /* Main Controller On */
    storageMsgTbl_MainCntOn_ReqStop,
    storageMsgTbl_MainCntOn_WaitCompStop,
    storageMsgTbl_MainCntOn_ReqStart_ReqStop,
    storageMsgTbl_MainCntOn_ReqStart_WaitCompStop,
    /********************************************/
    /* Probe ON, Remove ON                      */
    /********************************************/
    storageMsgTbl_MainCntOff_RemoveOn_WaitCompStart_ReqStop,
    storageMsgTbl_MainCntOn_RemoveOn_ReqStop,
    storageMsgTbl_MainCntOn_RemoveOn_WaitCompStop,
    /********************************************/
    /* Probe ON, Remove OFF, Command Sequence   */
    /********************************************/
    /* CBW/CSW */
    storageMsgTbl_MainCntOn_WaitRcvCbw,
    storageMsgTbl_MainCntOn_WaitSndCsw,
    /* Non Data */
    storageMsgTbl_MainCntOn_WaitSndCmdNonData,
    /* Data In */
    storageMsgTbl_MainCntOn_WaitSndCmdDataIn,
    storageMsgTbl_MainCntOn_WaitSndData,
    storageMsgTbl_MainCntOn_WaitUnstallSndData,
    storageMsgTbl_MainCntOn_WaitUnstallPipes,
    /* Data Out */
    storageMsgTbl_MainCntOn_WaitRcvData,
    storageMsgTbl_MainCntOn_WaitSndCmdDataOut,
    /* Illegal */
    storageMsgTbl_MainCntOn_WaitResetRecovery,
};


/*

 ؿ                                                             
*/
/**
    MassStorage( Probe )
 **/
/**
 * storage_probeMassStorage
 *
 * סMassStorage Probe   Gadget Core  Func Driver  regist
 **/
void storage_probeMassStorage( void )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__PROBE, "func Req" );

    /* Main Controller  probe λ Message  */
    msg.cnt.id = MSG_MAINCNT__COMP_PROBE;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );

    return;
}

/* MSG_MAINCNT__COMP_PROBE */
/**
 * compProbeMassStorage
 *
 * סMassStorage Probe λξܤΤ߹Ԥ
 **/
static GSTORAGE__State compProbeMassStorage( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__PROBE, "func Req" );

    /* Probe ON  Remove ׵ OFF  */
    return MAINCNT_OFF;
}

/**
    MassStorage( Remove )
 **/
/**
 * storage_reqRemoveMassStorage
 **/
void storage_reqRemoveMassStorage( void )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* Main Controller  remove ׵ Message  */
    msg.cnt.id = MSG_MAINCNT__REQ_REMOVE;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}

/* MSG_MAINCNT__REQ_REMOVE */
/**
 * execRemoveMassStorage
 **/
static GSTORAGE__State execRemoveMassStorage( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* MainCnt Thread  */
    pStorageMainCnt->end = 1;

    /* Probe OFF  */
    return PROBE_OFF;
}

/**
 * reqRemoveMassStorageDuringWaitCompStartReqStop
 **/
static GSTORAGE__State reqRemoveMassStorageDuringWaitCompStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* Start λԤ  Stop ׵  Remove λԤ */
    return MAINCNT_OFF__REMOVE_ON__WAIT_COMP_START__REQ_STOP;
}

/**
 * reqRemoveMassStorageDuringReqStop
 **/
static GSTORAGE__State reqRemoveMassStorageDuringReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* ( Stop ׵ or Stop λԤ )  Remove λԤ */
    return MAINCNT_ON__REMOVE_ON__REQ_STOP;
}

/**
 * reqRemoveMassStorageDuringWaitCompStop
 **/
static GSTORAGE__State reqRemoveMassStorageDuringWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* ( Stop ׵ or Stop λԤ )  Remove λԤ */
    return MAINCNT_ON__REMOVE_ON__WAIT_COMP_STOP;
}

/**
 * reqStopDrvCntThread
 **/
static GSTORAGE__State reqStopDrvCntThread( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* Svc ؤ Event ȯ */
    if ( pMsg->id == MSG_MAINCNT__COMP_STOP ){
        storage_storageOut();
    }

    /* Drive Controller Thread ׵ */
    storage_drvcnt_reqStopThread( stopThreadCb );

    return MAINCNT__WAIT_COMP_STOP_DRVCNT_THREAD;
}

/**
 * stopThreadCb
 **/
static void stopThreadCb( GSTORAGE__Status stat )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__REMOVE, "func Req" );

    /* Main Controller  DrvCnt Thread ߴλ Message  */
    msg.cnt.id = MSG_MAINCNT__COMP_STOP_DRVCNT_THREAD;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}


/**
    ǥХϿ
 **/
/**
 * storage_maincnt_setInitInfo
 **/
int storage_maincnt_setInitInfo( GSTORAGE__InfoInitMain *pInfo )
{
    int err = 0;

    GS_DET( GS_OUT__INIT_INFO, "func Req" );

    pMainCtx = kzalloc ( sizeof( GSTORAGE_MainContext ), GFP_ATOMIC );
    GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pMainCtx)                    =%p", pMainCtx );
    if ( pMainCtx == NULL ){
        err = GSTORAGE__ERROR;
        GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pMainCtx) error" );
    }
    if ( !err ){
        pMainCtx->maxLun  = pInfo->maxLun;
        pMainCtx->pCbwBuf = kmalloc( GSTORAGE__CBW_BUF_SIZE, GFP_ATOMIC );
        pMainCtx->pCswBuf = kmalloc( GSTORAGE__CSW_BUF_SIZE, GFP_ATOMIC );
        pMainCtx->pPckt   = kmalloc( sizeof( GSTORAGE__Packet ), GFP_ATOMIC );
        GS_DBG( GS_OUT__INIT_INFO, "maxLun = %d", (int)pMainCtx->maxLun );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pMainCtx->pCbwBuf)           =%p", pMainCtx->pCbwBuf );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pMainCtx->pCswBuf)           =%p", pMainCtx->pCswBuf );
        GS_DBG( GS_OUT__INIT_INFO, "kmalloc(pMainCtx->pPckt)             =%p", pMainCtx->pPckt );
        if ( ( !pMainCtx->pCbwBuf ) || ( !pMainCtx->pCswBuf ) || ( !pMainCtx->pPckt ) ){
            GS_ERR( GS_OUT__INIT_INFO, "kmalloc(pCbwBuf, pCswBuf, pPckt)" );
            err = GSTORAGE__ERROR;
        }
        if ( !err ){
            pMainCtx->is_bulk_out_stalled = 0;
            pMainCtx->is_bulk_in_stalled = 0;
            pMainCtx->disEpQueue = GSTORAGE__OFF;
            pMainCtx->waitTimer = GSTORAGE__OFF;
            init_timer( &( pMainCtx->timer ) );
        }
        else {
            if ( pMainCtx->pPckt ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pPckt)              =%p", pMainCtx->pPckt );
                kfree( pMainCtx->pPckt );
            }
            if ( pMainCtx->pCswBuf ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pCswBuf)            =%p", pMainCtx->pCswBuf );
                kfree( pMainCtx->pCswBuf );
            }
            if ( pMainCtx->pCbwBuf ){
                GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pCbwBuf)            =%p", pMainCtx->pCbwBuf );
                kfree( pMainCtx->pCbwBuf );
            }
            GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx)                     =%p", pMainCtx );
            kfree( pMainCtx );
            pMainCtx = NULL;
        }
    }

    return err;
}

/**
 * storage_maincnt_clearInitInfo
 **/
void storage_maincnt_clearInitInfo( void )
{
    /* MainCnt κ */
    if ( pMainCtx ){
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pPckt)              =%p", pMainCtx->pPckt );
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pCswBuf)            =%p", pMainCtx->pCswBuf );
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx->pCbwBuf)            =%p", pMainCtx->pCbwBuf );
        GS_DBG( GS_OUT__INIT_INFO, "kfree(pMainCtx)                     =%p", pMainCtx );
        kfree( pMainCtx->pPckt );
        kfree( pMainCtx->pCswBuf );
        kfree( pMainCtx->pCbwBuf );
        kfree( pMainCtx );
        pMainCtx = NULL;
    }

    return;
}


/**
    Main Controller ( Start )
 **/
/* MSG_MAINCNT__REQ_START */
/**
 * storage_maincnt_reqStart
 **/
void storage_maincnt_reqStart( void )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__START, "func Req" );
    
    /* to Enable */
    pMainCtx->disEpQueue = GSTORAGE__OFF;
    
    pMainCtx->is_bulk_out_stalled = 0;
    pMainCtx->is_bulk_in_stalled = 0;
    
    /* Main Controller  Start λ Message  */
    msg.cnt.id           = MSG_MAINCNT__REQ_START;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}

/**
 * reqStart
 **/
static GSTORAGE__State reqStart( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Drive Controller  Start ׵ Message ؿƤ */
    storage_drvcnt_startDrvCnt( startCb );

    /* Start λԤ */
    return MAINCNT_OFF__WAIT_COMP_START;
}

/**
 * startCb
 **/
static void startCb( GSTORAGE__Status stat, unsigned char unknownMedia )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__START, "func Req" );

    /* Main Controller  Start λ Message  */
    msg.compStart.id           = MSG_MAINCNT__COMP_START;
    msg.compStart.stat         = stat;
    msg.compStart.unknownMedia = unknownMedia;
    maincnt_sendMsg( &msg, sizeof( msg.compStart ) );
}

/**
 * reqStartDuringWaitCompStartReqStop
 **/
static GSTORAGE__State reqStartDuringWaitCompStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Start ׵  Start λԤ  Stop ׵ *//* Start ׵ Messageʤ */
    return MAINCNT_OFF__REQ_START__WAIT_COMP_START__REQ_STOP;
}

/**
 * reqStartDuringReqStop
 **/
static GSTORAGE__State reqStartDuringReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Start ׵  Stop λԤ *//* Start ׵ Messageʤ */
    return MAINCNT_ON__REQ_START__REQ_STOP;
}

/**
 * reqStartDuringWaitCompStop
 **/
static GSTORAGE__State reqStartDuringWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Start ׵  Stop λԤ *//* Start ׵ Messageʤ */
    return MAINCNT_ON__REQ_START__WAIT_COMP_STOP;
}

/* MSG_MAINCNT__COMP_START */
/**
 * compStartDuringWaitCompStartReqStop
 **/
static GSTORAGE__State compStartDuringWaitCompStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Unknown Media Ͽ */
    pMainCtx->unknownMedia = pMsg->compStart.unknownMedia;

    /* Svc ؤ Event ȯ */
    storage_storageIn();

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop λԤ */
    return MAINCNT_ON__WAIT_COMP_STOP;
}

/**
 * compStartDuringReqStartWaitCompStartReqStop
 **/
static GSTORAGE__State compStartDuringReqStartWaitCompStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Unknown Media Ͽ */
    pMainCtx->unknownMedia = pMsg->compStart.unknownMedia;

    /* Svc ؤ Event ȯ */
    storage_storageIn();

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Start ׵  Stop λԤ */
    return MAINCNT_ON__REQ_START__WAIT_COMP_STOP;
}

/**
 * compStartDuringRemoveOn
 **/
static GSTORAGE__State compStartDuringRemoveOn( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__START, "func Req" );

    /* Unknown Media Ͽ */
    pMainCtx->unknownMedia = pMsg->compStart.unknownMedia;

    /* Svc ؤ Event ȯ */
    storage_storageIn();

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop λԤ  Remove λԤ */
    return MAINCNT_ON__REMOVE_ON__WAIT_COMP_STOP;
}


/**
    Main Controller ( Stop )
 **/
/* MSG_MAINCNT__REQ_STOP */
/**
 * storage_maincnt_reqStop
 **/
void storage_maincnt_reqStop( void )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__STOP, "func Req" );

    /* to Disable */
    pMainCtx->disEpQueue = GSTORAGE__ON;

    /* Main Controller  Stop ׵ Message  */
    msg.cnt.id = MSG_MAINCNT__REQ_STOP;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}

/**
 * reqStop
 **/
static GSTORAGE__State reqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    storage_drvcnt_extcmdCancel();

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop λԤ */
    return MAINCNT_ON__WAIT_COMP_STOP;
}

/**
 * stopCb
 **/
static void stopCb( GSTORAGE__Status stat )
{
    unsigned long flags;
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__STOP, "func Req" );

    spin_lock_irqsave( &pStorageMainCnt->lock, flags );         /* ¾ */
    GS_DBG( GS_OUT__TIMER, "waitTimer = %d", pMainCtx->waitTimer );
    if ( pMainCtx->waitTimer == GSTORAGE__ON ){
        GS_DBG( GS_OUT__TIMER, "del_timer() in stopCb()" );
        del_timer( &( pMainCtx->timer ) );
        pMainCtx->waitTimer = GSTORAGE__OFF;
    }
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );    /* ¾λ */

    /* Main Controller  Stop λ Message  */
    msg.cnt.id = MSG_MAINCNT__COMP_STOP;
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}

/**
 * reqStopDuringWaitCompStart
 **/
static GSTORAGE__State reqStopDuringWaitCompStart( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    storage_drvcnt_extcmdCancel();

    /* Start λԤ  Stop ׵ *//* Stop ׵ Messageʤ */
    return MAINCNT_OFF__WAIT_COMP_START__REQ_STOP;
}

/**
 * reqStopDuringReqStartWaitCompStartReqStop
 **/
static GSTORAGE__State reqStopDuringReqStartWaitCompStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    storage_drvcnt_extcmdCancel();

    /* Start λԤ  Stop ׵ *//* Stop ׵ Messageʤ */
    return MAINCNT_OFF__WAIT_COMP_START__REQ_STOP;
}

/**
 * reqStopDuringReqStartReqStop
 **/
static GSTORAGE__State reqStopDuringReqStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    storage_drvcnt_extcmdCancel();

    /* Stop ׵Ԥ *//* Stop ׵ Messageʤ */
    return MAINCNT_ON__REQ_STOP;
}

/**
 * reqStopDuringReqStartWaitCompStop
 **/
static GSTORAGE__State reqStopDuringReqStartWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();

    storage_drvcnt_extcmdCancel();

    /* Stop λԤ *//* Stop ׵ Messageʤ */
    return MAINCNT_ON__WAIT_COMP_STOP;
}

/**
 * reqStopDuringSndCmd
 **/
static GSTORAGE__State reqStopDuringSndCmd( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    storage_drvcnt_extcmdCancel();

/* MK141H-6348 */
    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );
/* MK141H-6348 */

    return MAINCNT_ON__REQ_STOP;
}

/**
 * setDisEpQueueOff
 **/
static void setDisEpQueueOff( void )
{
    unsigned long flags;

    GS_DET( GS_OUT__STOP, "func Req" );

    spin_lock_irqsave( &pStorageMainCnt->lock, flags );         /* ¾ */
    /* to Disable */
    pMainCtx->disEpQueue = GSTORAGE__OFF;
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );    /* ¾λ */

    return;
}


/* MSG_MAINCNT__COMP_STOP */
/**
 * compStop
 **/
static GSTORAGE__State compStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    /* Svc ؤ Event ȯ */
    storage_storageOut();

    /* Main Controller OFF  */
    return MAINCNT_OFF;
}

/**
 * compStopDuringReqStartReqStop
 **/
static GSTORAGE__State compStopDuringReqStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    /* Svc ؤ Event ȯ */
    storage_storageOut();

    /* Drive Controller  Start ׵ Message ؿƤ */
    storage_drvcnt_startDrvCnt( startCb );

    /* Start λԤ */
    return MAINCNT_OFF__WAIT_COMP_START;
}

/**
 * compStopDuringReqStartWaitCompStop
 **/
static GSTORAGE__State compStopDuringReqStartWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__STOP, "func Req" );

    /* Svc ؤ Event ȯ */
    storage_storageOut();

    /* Drive Controller  Start ׵ Message ؿƤ */
    storage_drvcnt_startDrvCnt( startCb );

    /* Start λԤ */
    return MAINCNT_OFF__WAIT_COMP_START;
}


/**
    ޥλ
 **/
/* MSG_MAINCNT__COMP_SND_CMD */
/**
 * compSndCmdDuringReqStop
 **/
static GSTORAGE__State compSndCmdDuringReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop ׵ or Stop λԤ */
    return MAINCNT_STATE__NO_CHANGE;
}

/**
 * compSndCmdDuringWaitCompStop
 **/
static GSTORAGE__State compSndCmdDuringWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop ׵ or Stop λԤ */
    return MAINCNT_STATE__NO_CHANGE;
}

/**
 * compSndCmdDuringReqStartReqStop
 **/
static GSTORAGE__State compSndCmdDuringReqStartReqStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop ׵ or Stop λԤ */
    return MAINCNT_STATE__NO_CHANGE;
}

/**
 * compSndCmdDuringReqStartWaitCompStop
 **/
static GSTORAGE__State compSndCmdDuringReqStartWaitCompStop( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    /* Stop ׵ or Stop λԤ */
    return MAINCNT_STATE__NO_CHANGE;
}

/**
 * compSndCmdDuringRemoveOn
 **/
static GSTORAGE__State compSndCmdDuringRemoveOn( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* disEpQueue ե饰 */
    setDisEpQueueOff();
    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_stopDrvCnt( stopCb );

    return MAINCNT_STATE__NO_CHANGE;
}


/**
    Set Media Status Ϣ
 **/
/**
 * storage_maincnt_reqSetMediaStatus
 **/
void storage_maincnt_reqSetMediaStatus( GSTORAGE__InfoMediaStatus *info )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    /* Main Controller  Set Media Status ׵ Message  */
    msg.reqMedStat.id     = MSG_MAINCNT__REQ_SET_MEDIA_STATUS;
    msg.reqMedStat.info.lun          = info->lun;
    msg.reqMedStat.info.status       = info->status;
    msg.reqMedStat.info.lastLBA      = info->lastLBA;
    msg.reqMedStat.info.blocks       = info->blocks;
    msg.reqMedStat.info.writeProtect = info->writeProtect;
    msg.reqMedStat.info.mediumType   = info->mediumType;
    memcpy(msg.reqMedStat.info.attr, info->attr, sizeof(msg.reqMedStat.info.attr));

    maincnt_sendMsg( &msg, sizeof( msg.reqMedStat ) );
}

/**
 * reqSetMediaStatus
 **/
static GSTORAGE__State reqSetMediaStatus( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    /* Drive Controller  Stop ׵ Message ؿƤ */
    storage_drvcnt_reqSetMediaStatus( setMediaStatusCb,
                                      &pMsg->reqMedStat.info );

    return MAINCNT_STATE__NO_CHANGE;
}

/**
 * setMediaStatusCb
 **/
static void setMediaStatusCb( GSTORAGE__Status stat, unsigned char unknownMedia, unsigned int lun )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    /* Main Controller  Set Media Status λ Message  */
    msg.compMedStat.id           = MSG_MAINCNT__COMP_SET_MEDIA_STATUS;
    msg.compMedStat.stat         = stat;
    msg.compMedStat.unknownMedia = unknownMedia;
    msg.compMedStat.lun          = lun;
    maincnt_sendMsg( &msg, sizeof( msg.compMedStat ) );
}

/**
 * compSetMediaStatus
 **/
static GSTORAGE__State compSetMediaStatus( GSTORAGE__Msg *pMsg )
{
    unsigned long   flags;
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    /* cmn δؿƤӽФ */
    storage_comp_setMediaStatus( pMsg->compMedStat.stat, pMsg->compMedStat.lun );

    pMainCtx->unknownMedia = pMsg->compMedStat.unknownMedia;

    spin_lock_irqsave( &pStorageMainCnt->lock, flags );             /* ¾ */
    GS_DBG( GS_OUT__TIMER, "waitTimer = %d", pMainCtx->waitTimer );
    if ( pMainCtx->waitTimer == GSTORAGE__ON ){
        GS_DBG( GS_OUT__TIMER, "del_timer() in compSetMediaStatus()" );
        del_timer( &( pMainCtx->timer ) );
        pMainCtx->waitTimer = GSTORAGE__OFF;
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );    /* ¾λ */
        state = forcedRcvCbw( pMsg );
        return state;
    }
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );        /* ¾λ */

    return state;
}

/**
 * compSetMediaStatusDuringReqStartOrReqStop
 **/
static GSTORAGE__State compSetMediaStatusDuringReqStartOrReqStop( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;

    GS_DET( GS_OUT__MEDIA_STATUS, "func Req" );

    /* cmn δؿƤӽФ */
    storage_comp_setMediaStatus( pMsg->compMedStat.stat, pMsg->compMedStat.lun );

    return state;
}


/**
    MassStorage Reset Ϣ
 **/
/**
 * storage_maincnt_compMassStorageReset
 **/
void storage_maincnt_compMassStorageReset( void )
{
    GSTORAGE__Msg msg;
    int           i;
    struct usb_ep *p_ep;
    
    GS_DET( GS_OUT__CLASS, "func Req" );
    
    // Set Pipes STALL
    pMainCtx->is_bulk_in_stalled  = 1;
    pMainCtx->is_bulk_out_stalled = 1;
    
    GS_DBG( GS_OUT__USB_EP_QUEUE_DIS, "[STALL INFO] IN:%d, OUT:%d", 
                                pMainCtx->is_bulk_in_stalled, pMainCtx->is_bulk_out_stalled );
    
    for ( i = 0; i < GSTORAGE_EP_NUM; i++ ){
        p_ep = pMainCtx->p_ep[ i ];
        if ( p_ep != NULL ){
            storage_dequeue_ep( p_ep );
            GS_DBG( GS_OUT__CLASS, "storage_dequeue_ep(%d)", i );
        }
    }
    
    /* Main Controller  MassStorage Reset λ Message  */
    msg.class.id = MSG_MAINCNT__COMP_MASS_STORAGE_RESET;
    maincnt_sendMsg( &msg, sizeof( msg.class ) );
}


/**
    Reset Recovery Ϣ
 **/
/**
 * storage_maincnt_compResetRecovery
 **/
void storage_maincnt_compResetRecovery( void )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__RESET_RECOVERY, "func Req" );

    /* Main Controller  MassStorage Reset λ Message  */
    msg.class.id = MSG_MAINCNT__COMP_RESET_RECOVERY;
    maincnt_sendMsg( &msg, sizeof( msg.class ) );
}


/**
    Clear Halt Ϣ
 **/
/**
 * storage_maincnt_epClearHalt
 **/
void storage_maincnt_epClearHalt( struct usb_ep *p_ep )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__CLEAR_HALT, "func Req" );

    /* å(ClearHaltλ) */
    if ( p_ep == pMainCtx->p_ep[ GSTORAGE_EP_SEND ] ){
        msg.cnt.id = MSG_MAINCNT__COMP_UNSTALL_BULK_IN;
    }
    else {
        msg.cnt.id = MSG_MAINCNT__COMP_UNSTALL_BULK_OUT;
    }
    maincnt_sendMsg( &msg, sizeof( msg.cnt ) );
}


/**
    CBW
 **/
/**
 * rcvCbw
 **/
static GSTORAGE__State rcvCbw( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    GSTORAGE__Cbw               *pCbw;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Status            stat = GSTORAGE__SUCCESS;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( pMsg->id == MSG_MAINCNT__COMP_START ){
        /* Unknown Media Ͽ */
        pMainCtx->unknownMedia = pMsg->compStart.unknownMedia;
        /* Svc ؤ Event ȯ */
#ifdef STORAGEIN_AT_RCVCBW_CB
        GS_DBG( GS_OUT__MAINCNT, "## Passed Calling storageIN() in %s\n", __FUNCTION__);
#else
        storage_storageIn();
#endif
    }
    GS_DBG( GS_OUT__MAINCNT, "pMainCtx->unknownMedia=%d", pMainCtx->unknownMedia );
    
    if ( pMainCtx->waitTimer == GSTORAGE__OFF ){
        if ( pMainCtx->unknownMedia == GSTORAGE__ON ){
            GS_DBG( GS_OUT__TIMER, "add_timer()" );
            spin_lock_irqsave( &pStorageMainCnt->lock, flags );
            pMainCtx->waitTimer      = GSTORAGE__ON;
            pMainCtx->timer.function = receiveTimerEvent;
            pMainCtx->timer.expires  = jiffies + GSTORAGE__WAIT_SEC;
            pMainCtx->timer.data     = 0;
            add_timer( &( pMainCtx->timer ) );
            state = MAINCNT_CMD__WAIT_RCV_CBW;
            spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
            return state;
        }

        /* ¾ */
        spin_lock_irqsave( &pStorageMainCnt->lock, flags );

        p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
        if ( p_ep != NULL ){
            p_ep_data = p_ep->driver_data;
            p_req     = p_ep_data->req_data->req;

            /* CSW ｪλ */
            pCbw = ( GSTORAGE__Cbw * )( pMainCtx->pCbwBuf );

            /*  */
            pCbw->cbwSignature    = 0;
            pCbw->cbwTag          = 0;
            pCbw->cbwDataTransLen = 0;
            pCbw->cbwFlags        = 0;

            /* ꥯȤ */
            p_req->buf          = ( void * )pCbw;
            p_req->length       = usb_gadgetcore_ep_align_maybe(p_ep, STORAGE_CBW_SIZE);
            GS_INF( GS_OUT__CMN, "%s: size=%d, aligned=%d\n", __FUNCTION__, STORAGE_CBW_SIZE, p_req->length);
            p_req->dma          = ( dma_addr_t )NULL;
            p_req->no_interrupt = 0;
            p_req->zero         = 0;
            p_req->short_not_ok = 0;
            p_req->complete     = ( void * )rcvCbwCb;

            if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                
                if( !( pMainCtx->is_bulk_in_stalled ) && !( pMainCtx->is_bulk_out_stalled ) ){
                    
                    /* CBW ׵ */
                    err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                    if ( !err ){
                        /* CBW λԤ  */
                        state = MAINCNT_CMD__WAIT_RCV_CBW;
                    }
                    else {
                        error_ep_request_bulk_out( err );
                    }
                    
                }else{
                    
                    GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [bulk in/out stalled]" );
                    
                }
                
            }
            else {
                GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
            }
        }
        else {
            stat = GSTORAGE__ERROR;
            GS_INF( GS_OUT__MAINCNT, "p_ep error" );
        }

        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
    }

    if ( stat != GSTORAGE__SUCCESS ){
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
    }

    return state;
}

/**
    Check Clear Halt State and Recieve CBW
 **/
/**
 * CheckAndrcvCbw
 **/
static GSTORAGE__State CheckAndrcvCbw( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    
    GS_DET( GS_OUT__MAINCNT, "func Req" );
    
    if( pMsg->id == MSG_MAINCNT__COMP_UNSTALL_BULK_IN ){
        
        pMainCtx->is_bulk_in_stalled  = 0;
        
    }else if( pMsg->id == MSG_MAINCNT__COMP_UNSTALL_BULK_OUT ){
        
        pMainCtx->is_bulk_out_stalled = 0;
        
    }
    
    GS_DBG( GS_OUT__USB_EP_QUEUE_DIS, "[STALL INFO] IN:%d, OUT:%d", 
                        pMainCtx->is_bulk_in_stalled, pMainCtx->is_bulk_out_stalled );
    
    if( ( pMainCtx->is_bulk_in_stalled == 0 ) && ( pMainCtx->is_bulk_out_stalled == 0 ) ){
        
        GS_DBG( GS_OUT__USB_EP_QUEUE_DIS, "Complete to clear halt against all pipes." );
                                
        // complete to clear halt
        state = rcvCbw( pMsg );
        
    }
    
    return state;
    
}

/**
 * receiveTimerEvent
 **/
static void receiveTimerEvent( unsigned long param )
{
    unsigned long flags;
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__TIMER, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    if ( pMainCtx->waitTimer == GSTORAGE__ON ){
        pMainCtx->waitTimer = GSTORAGE__OFF;
        
        /* Main Controller  Timer Event Ǵλ Message  */
        msg.compCb.id   = MSG_MAINCNT__COMP_TIMER_EVENT;
        msg.compCb.stat = GSTORAGE__SUCCESS;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    
    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
}

/**
 * forcedRcvCbw
 **/
static GSTORAGE__State forcedRcvCbw( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    GSTORAGE__Cbw               *pCbw;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Status            stat = GSTORAGE__SUCCESS;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__TIMER, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
    if ( p_ep != NULL ){
        p_ep_data = p_ep->driver_data;
        p_req     = p_ep_data->req_data->req;

        /* CSW ｪλ */
        pCbw = ( GSTORAGE__Cbw * )( pMainCtx->pCbwBuf );

        /*  */
        pCbw->cbwSignature    = 0;
        pCbw->cbwTag          = 0;
        pCbw->cbwDataTransLen = 0;
        pCbw->cbwFlags        = 0;

        /* ꥯȤ */
        p_req->buf          = ( void * )pCbw;
        p_req->length       = usb_gadgetcore_ep_align_maybe(p_ep, STORAGE_CBW_SIZE);
        GS_INF( GS_OUT__CMN, "%s: size=%d, aligned=%d\n", __FUNCTION__, STORAGE_CBW_SIZE, p_req->length);
        p_req->dma          = ( dma_addr_t )NULL;
        p_req->no_interrupt = 0;
        p_req->zero         = 0;
        p_req->short_not_ok = 0;
        p_req->complete     = ( void * )rcvCbwCb;

        if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
            
            if( !( pMainCtx->is_bulk_in_stalled ) && !( pMainCtx->is_bulk_out_stalled ) ){
                
                /* CBW ׵ */
                err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                if ( !err ){
                    /* CBW λԤ  */
                    state = MAINCNT_CMD__WAIT_RCV_CBW;
                }
                else {
                    error_ep_request_bulk_out( err );
                }
                
            }else{
                
                GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [bulk in/out stalled]" );
                
            }
        }
        else {
            GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
        }
    }
    else {
        stat = GSTORAGE__ERROR;
        GS_INF( GS_OUT__MAINCNT, "p_ep error" );
    }

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    if ( stat != GSTORAGE__SUCCESS ){
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
    }

    return state;
}

/**
 * rcvCbwCb
 **/
static void rcvCbwCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( ( p_req->status == 0 ) || ( p_req->status == -EOVERFLOW ) ){
        if ( p_req->status == -EOVERFLOW ){
            error_ep_request_put_message( p_req->status );
        }

#ifdef STORAGEIN_AT_RCVCBW_CB
        storage_storageIn();
#endif

        /* å(CBW λ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_RCV_CBW;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_out( p_req->status );
    }

    return;
}


/**
    ޥ
 **/
/**
 * sndCmdPacket
 **/
static GSTORAGE__State sndCmdPacket( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Cbw               *pCbw;
    GSTORAGE__Csw               *pCsw;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;
    size_t                      bufsize;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* CBW ｪλ */
    pCbw = ( GSTORAGE__Cbw * )pMsg->compCb.pBuf;

    /* CBW 顼å */
    if ( pMsg->compCb.size != STORAGE_CBW_SIZE ){
        GS_INF( GS_OUT__MAINCNT, "cbw error(size)" );
        /* ꡼륹ơȤ */
        return waitResetRecovery( pMsg );
    }
    if ( GSTORAGE_TRANS_ENDIAN( pCbw->cbwSignature ) != STORAGE_CBW_SIGNATURE ){
        GS_INF( GS_OUT__MAINCNT, "cbw error(signature)" );
        /* ꡼륹ơȤ */
        return waitResetRecovery( pMsg );
    }
    /* LUN å */
    if ( pCbw->cbwLun > pMainCtx->maxLun ){
        GS_INF( GS_OUT__MAINCNT, "cbw error(lun)" );
        /* ꡼륹ơȤ */
        return waitResetRecovery( pMsg );
    }

    /* CDB Length å */
    if ( ( pCbw->cbwCdbLen == 0 ) || ( pCbw->cbwCdbLen > 16 ) ){
        GS_INF( GS_OUT__MAINCNT, "cbw error(cdb len)" );
        /* ꡼륹ơȤ */
        return waitResetRecovery( pMsg );
    }

    /* CSW  CBW  Tag ¸ */
    pCsw = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    pCsw->dCSWTag         = pCbw->cbwTag;
    pCsw->dCSWDataResidue = pCbw->cbwDataTransLen;
    pCsw->bCSWStatus      = STORAGE_CSW_STATUS_DEFAULT;

    /* PACKET ޥɺ */
    if ( pCbw->cbwDataTransLen == 0 ){
        pMainCtx->pPckt->dio = DATA_NON;
    }
    else if ( pCbw->cbwFlags == 0 ){
        pMainCtx->pPckt->dio = DATA_OUT;
    }
    else {
        pMainCtx->pPckt->dio = DATA_IN;
    }
    pMainCtx->pPckt->lun       = pCbw->cbwLun;
    pMainCtx->pPckt->bufLength = pCbw->cbwDataTransLen;
    pMainCtx->pPckt->pCbw      = pCbw;
    pMainCtx->pPckt->pCdb      = ( GSTORAGE__Cdb * )( pCbw->cbwCdb );
    pMainCtx->pPckt->cdbLength = pCbw->cbwCdbLen;
    pMainCtx->rcvLenOnCbw      = pCbw->cbwDataTransLen;

    /* ǡ̵ͭžʬ */
    switch( pMainCtx->pPckt->dio ){
      case DATA_NON:
        /* PACKET ޥ */
        storage_drvcnt_sndCmdPacket( sndCmdPacketCb, pMainCtx->pPckt );
        /* ޥλԤ  */
        state = MAINCNT_CMD__WAIT_SND_CMD_NON_DATA;
        break;

      case DATA_OUT:
        /*************************************************/
        /* ǡλԤ(˥ޥ)    */
        /*************************************************/
        /* ¾ */
        spin_lock_irqsave( &pStorageMainCnt->lock, flags );

        p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
        if ( p_ep != NULL ){
            bufsize = ( pMainCtx->pPckt->bufLength > GSTORAGE__RECEIVE_BUF_SIZE ) ?
                      GSTORAGE__RECEIVE_BUF_SIZE : pMainCtx->pPckt->bufLength;
            p_ep_data = p_ep->driver_data;
            p_req     = p_ep_data->req_data->req;
            /* ꥯȤ */
            p_req->buf          = ( void * )storage_drvcnt_getReceiveBufP( pCbw->cbwLun );
            p_req->length       = usb_gadgetcore_ep_align_maybe(p_ep, bufsize);
            GS_INF( GS_OUT__CMN, "%s: size=%lu, aligned=%d\n", __FUNCTION__, bufsize, p_req->length);
#ifdef GSTORAGE_DMA_ENABLE
            if ( p_req->buf != NULL ){
                p_req->dma          = ( dma_addr_t )__virt_to_phys( p_req->buf );
#ifdef CONFIG_OSAL_UDIF
                if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
                    udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_INVALIDATE );
                }
#else
                consistent_sync( p_req->buf, p_req->length, DMA_FROM_DEVICE );
#endif

            }
            else {
                p_req->dma      = ( dma_addr_t )NULL;
            }
#else
            p_req->dma          = ( dma_addr_t )NULL;
#endif
            p_req->no_interrupt = 0;
            p_req->zero         = 0;
            p_req->short_not_ok = 0;
            p_req->complete     = ( void * )rcvDataOutCb;

            /* USB ǡ׵ */
            if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                if ( !err ){
                    /* ǡλԤ(˥ޥ)  */
                    state = MAINCNT_CMD__WAIT_RCV_DATA;
                }
                else {
                    error_ep_request_bulk_out( err );
                }
            }
            else {
                GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
            }
        }
        else {
            /* ޤ¾ */
            spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
            /* disEpQueue ե饰 */
            setDisEpQueueOff();
            /* Drive Controller  Stop ׵ Message ؿƤ */
            storage_drvcnt_stopDrvCnt( stopCb );
            /* Stop λԤ */
            state = MAINCNT_ON__WAIT_COMP_STOP;
            break;
        }
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        break;

      case DATA_IN:
        /*************************************************/
        /* ޥλԤ(˥ǡ)    */
        /*************************************************/
        /* PACKET ޥ */
        storage_drvcnt_sndCmdPacket( sndCmdPacketCb, pMainCtx->pPckt );
        /* ޥλԤ(˥ǡ)  */
        state = MAINCNT_CMD__WAIT_SND_CMD_DATA_IN;
        break;

      default:
        /* 뤳ȤϤꤨʤܤʤȤ */
        state = MAINCNT_STATE__NO_CHANGE; 
        break;
    }

    return state;
}

/**
 * sndCmdPacketCb
 **/
static void sndCmdPacketCb( GSTORAGE__Status stat, unsigned int dataResidue, unsigned char status, unsigned int *pBuf )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* å(ޥλ) */
    msg.compDrvCnt.id          = MSG_MAINCNT__COMP_SND_CMD;
    msg.compDrvCnt.stat        = stat;
    msg.compDrvCnt.dataResidue = dataResidue;
    msg.compDrvCnt.status      = status;
    msg.compDrvCnt.pBuf        = pBuf;
    maincnt_sendMsg( &msg, sizeof( msg.compDrvCnt ) );

    return;
}


/**
    CSW
 **/
/**
 * sndCsw
 **/
static GSTORAGE__State sndCsw( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    GSTORAGE__Csw               *pCsw;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    p_ep = pMainCtx->p_ep[ GSTORAGE_EP_SEND ];
    if ( p_ep != NULL ){
        p_ep_data = p_ep->driver_data;
        p_req     = p_ep_data->req_data->req;

        /* CSW */
        pCsw = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
        pCsw->dCSWSignature = GSTORAGE_TRANS_ENDIAN( STORAGE_CSW_SIGNATURE );

        /* ꥯȤ */
        p_req->buf          = ( void * )pCsw;
        p_req->length       = STORAGE_CSW_SIZE;
        p_req->dma          = ( dma_addr_t )NULL;
        p_req->no_interrupt = 0;
        p_req->zero         = 0;
        p_req->short_not_ok = 0;
        p_req->complete     = ( void * )sndCswCb;

        if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
            /* CSW׵ */
            err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
            if ( !err ){
                /* CSWλԤ  */
                state = MAINCNT_CMD__WAIT_SND_CSW;
            }
            else {
                error_ep_request_bulk_in( err );
            }
        }
        else {
            GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
        }
    }
    else {
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
        return state;
    }
    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    return state;
}

/**
 * sndCswCb
 **/
static void sndCswCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( p_req->status == 0 ){
        /* å(CSW λ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_SND_CSW;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_in( p_req->status );
    }

    return;
}


/**
    ޥλ
 **/
/**
 * compSndCmdNonData
 **/
static GSTORAGE__State compSndCmdNonData( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Csw   *pCsw;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* dataResidue,status ¸ */
    pCsw = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    pCsw->dCSWDataResidue = pMsg->compDrvCnt.dataResidue;
    pCsw->bCSWStatus      = pMsg->compDrvCnt.status;

    /* CSW ׵ (CSW λԤ ) */
    state = sndCsw( pMsg );

    return state;
}

/**
 * compSndCmdDataIn
 **/
static GSTORAGE__State compSndCmdDataIn( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    unsigned int                len = 0;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Csw               *pCsw;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    pCsw = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    p_ep = pMainCtx->p_ep[ GSTORAGE_EP_SEND ];
    if ( p_ep != NULL ){
        p_ep_data = p_ep->driver_data;
        p_req     = p_ep_data->req_data->req;

        switch( pMsg->compDrvCnt.status ){
          case GSTORAGE_DRV__PASS :
            /********************************************/
            /* ( PASS ) Set CSW Status and Send Data    */
            /********************************************/
            pCsw->bCSWStatus = STORAGE_CSW_STATUS_PASS;
            /* set Request */
            p_req->buf          = ( void * )pMsg->compDrvCnt.pBuf;
            p_req->length       = len
                                = pCsw->dCSWDataResidue - pMsg->compDrvCnt.dataResidue;
#ifdef GSTORAGE_DMA_ENABLE
            if ( p_req->buf != NULL ){
                p_req->dma      = ( dma_addr_t )__virt_to_phys( p_req->buf );
#ifdef CONFIG_OSAL_UDIF
                if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
                    udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_FLUSH );
                }
#else
                consistent_sync( p_req->buf, p_req->length, DMA_TO_DEVICE );
#endif
            }
            else {
                p_req->dma      = ( dma_addr_t )NULL;
            }
#else
            p_req->dma          = ( dma_addr_t )NULL;
#endif
            p_req->no_interrupt = 0;
            p_req->zero         = 0;
            p_req->short_not_ok = 0;
            p_req->complete     = ( void * )sndDataInCb;
            /* Send Data */
            if ( len != 0 ){
                if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                    err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                    if ( !err ){
                        /* ܤʤ */
                        state = MAINCNT_STATE__NO_CHANGE;
                    }
                    else {
                        error_ep_request_bulk_in( err );
                    }
                }
                else {
                    GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
                }
            }
            break;

          case GSTORAGE_DRV__CONTINUE :
            /********************************************/
            /* ( CONTINUE ) Send Data                   */
            /********************************************/
            /* set Request */
            p_req->buf          = ( void * )pMsg->compDrvCnt.pBuf;
            p_req->length       = len
                                = ( pCsw->dCSWDataResidue > GSTORAGE__SEND_READ_BUF_SIZE ) ?
                                  GSTORAGE__SEND_READ_BUF_SIZE : pCsw->dCSWDataResidue;
#ifdef GSTORAGE_DMA_ENABLE
            if ( p_req->buf != NULL ){
                p_req->dma      = ( dma_addr_t )__virt_to_phys( p_req->buf );
#ifdef CONFIG_OSAL_UDIF
                if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
                    udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_FLUSH );
                }
#else
                consistent_sync( p_req->buf, p_req->length, DMA_TO_DEVICE );
#endif
            }
            else {
                p_req->dma      = ( dma_addr_t )NULL;
            }
#else
            p_req->dma          = ( dma_addr_t )NULL;
#endif
            p_req->no_interrupt = 0;
            p_req->zero         = 0;
            p_req->short_not_ok = 0;
            p_req->complete     = ( void * )sndDataInContinueCb;
            /* Send Data */
            if ( len != 0 ){
                if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                    err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                    if ( !err ){
                        /* ܤʤ */
                        state = MAINCNT_STATE__NO_CHANGE;
                    }
                    else {
                        error_ep_request_bulk_in( err );
                    }
                }
                else {
                    GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
                }
            }
            break;

          case GSTORAGE_DRV__FAIL :
            /********************************************/
            /* ( FAIL ) Set CSW Status                  */
            /********************************************/
            pCsw->bCSWStatus = STORAGE_CSW_STATUS_FAIL;
            break;

          default :
            /********************************************/
            /* ( FATAL ) Set CSW Status                 */
            /********************************************/
            pCsw->bCSWStatus = STORAGE_CSW_STATUS_PHASE_ERROR;
            break;
        }

        pCsw->dCSWDataResidue = pMsg->compDrvCnt.dataResidue;
        
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    }
    else {
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
    }

    if ( state == MAINCNT_STATE__NO_CHANGE ){
        switch( pMsg->compDrvCnt.status ){
          case GSTORAGE_DRV__PASS:
          case GSTORAGE_DRV__CONTINUE:
            /* ǡλԤ or ǡSTALLλԤ  */
            state = ( len != 0 ) ?
                    MAINCNT_CMD__WAIT_SND_DATA : sndDataHalt( pMsg );
            break;
          default:
            state = sndDataHalt( pMsg );
            break;
        }
    }

    return state;
}

/**
 * compSndCmdDataOut
 **/
static GSTORAGE__State compSndCmdDataOut( GSTORAGE__Msg *pMsg )
{
    unsigned long               flags;
    GSTORAGE__State             state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Csw               *pCsw;
    struct usb_ep               *p_ep;
    struct usbg_storage_ep_data *p_ep_data;
    struct usb_request          *p_req;
    int                         err = GSTORAGE__SUCCESS;
    size_t                      bufsize;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    pCsw = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    pCsw->dCSWDataResidue = pMsg->compDrvCnt.dataResidue;

    switch( pMsg->compDrvCnt.status ){
      case GSTORAGE_DRV__CONTINUE :
        /********************************************/
        /* ( CONTINUE ) Receive Data                */
        /********************************************/
        /* ¾ */
        spin_lock_irqsave( &pStorageMainCnt->lock, flags );
        p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
        if ( p_ep != NULL ){
            bufsize = ( pMsg->compDrvCnt.dataResidue > GSTORAGE__RECEIVE_BUF_SIZE ) ?
                      GSTORAGE__RECEIVE_BUF_SIZE : pMsg->compDrvCnt.dataResidue;
            p_ep_data = p_ep->driver_data;
            p_req     = p_ep_data->req_data->req;
            /* set Request */
            p_req->buf          = ( void * )pMsg->compDrvCnt.pBuf;
            p_req->length       = usb_gadgetcore_ep_align_maybe(p_ep, bufsize);
            GS_INF( GS_OUT__CMN, "%s: size=%lu, aligned=%d\n", __FUNCTION__, bufsize, p_req->length);
#ifdef GSTORAGE_DMA_ENABLE
            if ( p_req->buf != NULL ){
                p_req->dma          = ( dma_addr_t )__virt_to_phys( p_req->buf );
#ifdef CONFIG_OSAL_UDIF
                if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
                    udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_INVALIDATE );
                }
#else
                consistent_sync( p_req->buf, p_req->length, DMA_FROM_DEVICE );
#endif
            }
            else {
                p_req->dma      = ( dma_addr_t )NULL;
            }
#else
            p_req->dma          = ( dma_addr_t )NULL;
#endif
            p_req->no_interrupt = 0;
            p_req->zero         = 0;
            p_req->short_not_ok = 0;
            p_req->complete     = ( void * )rcvDataOutContinueCb;
            if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                /* Receive Data */
                err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                if ( !err ){
                    /* ԡȥǡλԤ */
                    state = MAINCNT_CMD__WAIT_RCV_DATA;
                }
                else {
                    error_ep_request_bulk_out( err );
                }
            }
            else {
                GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
            }
        }
        else {
            /* ޤ¾ */
            spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
            /* disEpQueue ե饰 */
            setDisEpQueueOff();
            /* Drive Controller  Stop ׵ Message ؿƤ */
            storage_drvcnt_stopDrvCnt( stopCb );
            /* Stop λԤ */
            state = MAINCNT_ON__WAIT_COMP_STOP;
            break;
        }
        /* ޤ¾ */
        spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        break;

      default :
        if ( pCsw->bCSWStatus == STORAGE_CSW_STATUS_DEFAULT ){
            if ( pMsg->compDrvCnt.status == GSTORAGE_DRV__PASS ){
                pCsw->bCSWStatus = STORAGE_CSW_STATUS_PASS;
            }
            else if ( pMsg->compDrvCnt.status == GSTORAGE_DRV__FAIL ){
                pCsw->bCSWStatus = STORAGE_CSW_STATUS_FAIL;
            }
            else {
                pCsw->bCSWStatus = STORAGE_CSW_STATUS_PHASE_ERROR;
            }
        }
        if ( pMainCtx->rcvLenOnCbw ){
            /* ¾ */
            spin_lock_irqsave( &pStorageMainCnt->lock, flags );
            p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
            if ( p_ep != NULL ){
                p_ep_data = p_ep->driver_data;
                p_req     = p_ep_data->req_data->req;
                /* set Request */
                p_req->buf          = ( void * )pMsg->compDrvCnt.pBuf;
                p_req->length       = ( pMainCtx->rcvLenOnCbw > GSTORAGE__RECEIVE_BUF_SIZE ) ?
                                      GSTORAGE__RECEIVE_BUF_SIZE : pMainCtx->rcvLenOnCbw;
#ifdef GSTORAGE_DMA_ENABLE
                if ( p_req->buf != NULL ){
                    p_req->dma          = ( dma_addr_t )__virt_to_phys( p_req->buf );
#ifdef CONFIG_OSAL_UDIF
                    if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
                        udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_INVALIDATE );
                    }
#else
                    consistent_sync( p_req->buf, p_req->length, DMA_FROM_DEVICE );
#endif
                }
                else {
                    p_req->dma      = ( dma_addr_t )NULL;
                }
#else
                p_req->dma          = ( dma_addr_t )NULL;
#endif
                p_req->no_interrupt = 0;
                p_req->zero         = 0;
                p_req->short_not_ok = 0;
                p_req->complete     = ( void * )rcvDataOutContinueCb;
                /* Receive Data */
                if ( pMainCtx->disEpQueue == GSTORAGE__OFF ){
                    err = usb_ep_queue( p_ep, p_req, GFP_ATOMIC );
                    if ( !err ){
                        /* ԡȥǡλԤ */
                        state = MAINCNT_CMD__WAIT_RCV_DATA;
                    }
                    else {
                        error_ep_request_bulk_out( err );
                    }
                }
                else {
                    GS_INF( GS_OUT__USB_EP_QUEUE_DIS, "usb_ep_queue() disable [disEpQueue==ON]" );
                }
            }
            else {
                /* ޤ¾ */
                spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
                /* disEpQueue ե饰 */
                setDisEpQueueOff();
                /* Drive Controller  Stop ׵ Message ؿƤ */
                storage_drvcnt_stopDrvCnt( stopCb );
                /* Stop λԤ */
                state = MAINCNT_ON__WAIT_COMP_STOP;
                break;
            }
            /* ޤ¾ */
            spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );
        }
        break;
    }

    switch( pMsg->compDrvCnt.status ){
      case GSTORAGE_DRV__CONTINUE :
        break;
      default :
        if ( !pMainCtx->rcvLenOnCbw ){
            state = sndCsw( pMsg );
        }
        break;
    }

    return state;
}


/**
    ǡ
 **/
/**
 * sndDataInCb
 **/
static void sndDataInCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( p_req->status == 0 ){
        
        if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
            udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_INVALIDATE );
        }
        
        /* å(ǡλ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_SND_DATA;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_in( p_req->status );
    }
    
    return;
}

/**
 * compSndDataIn
 **/
static GSTORAGE__State compSndDataIn( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Csw   *pCsw;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* ǡž(/)ｪλ */
    pCsw  = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    state = ( pCsw->dCSWDataResidue == 0 ) ?
            sndCsw( pMsg ) : sndDataHalt( pMsg );

    return state;
}

/**
 * sndDataInContinueCb
 **/
static void sndDataInContinueCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( p_req->status == 0 ){
        
        if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)p_req->buf)) {
            udif_cache_ctrl( p_req->buf, p_req->length, UDIF_CACHE_INVALIDATE );
        }        
        /* å(ǡλ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_SND_DATA_CONTINUE;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_in( p_req->status );
    }
    
    return;
}

/**
 * compSndDataInContinue
 **/
static GSTORAGE__State compSndDataInContinue( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* CONTINUE ޥ */
    storage_drvcnt_sndCmdContinue( sndCmdPacketCb );

    /* Read Continue λԤ */
    state = MAINCNT_CMD__WAIT_SND_CMD_DATA_IN;

    return state;
}

/**
 * sndDataHalt
 **/
static GSTORAGE__State sndDataHalt( GSTORAGE__Msg *pMsg )
{
    unsigned long   flags;
    int             err = 0;
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    struct usb_ep   *p_ep;
    int             i = 0;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    p_ep = pMainCtx->p_ep[ GSTORAGE_EP_SEND ];
    if ( p_ep != NULL ){
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
        spin_unlock_irq( &pStorageMainCnt->lock );
#endif
        storage_dequeue_ep( p_ep );
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
        spin_lock_irq( &pStorageMainCnt->lock );
#endif
        for ( i = 0; i < SET_HALT_RETRY_CNT; i++ ){
            err = usb_ep_set_halt( p_ep );
            /* SetHaltԻϥȥ饤 */
            if ( err ) {
                GS_INF( GS_OUT__MAINCNT, "usb_ep_set_halt err=%d", err );
                udif_msleep(10); /* 10ms */
            } else {
                break;
            }
        }
        /* ǡSTALLλԤ  */
        state = MAINCNT_CMD__WAIT_UNSTALL_SND_DATA;
    }
    else {
        err = GSTORAGE__ERROR;
        GS_INF( GS_OUT__MAINCNT, "p_ep error" );
    }

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    if ( err ){
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
    }

    return state;
}

/**
 * waitUnstallPipes
 **/
static GSTORAGE__State waitUnstallPipes( GSTORAGE__Msg *pMsg )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );
    
    /* BulkIN STALL Ԥ  */
    return MAINCNT_CMD__WAIT_UNSTALL_PIPES;
}


/**
    ǡ
 **/
/**
 * rcvDataOutCb
 **/
static void rcvDataOutCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( ( p_req->status == 0 ) || ( p_req->status == -EOVERFLOW ) ){
        if ( p_req->status == -EOVERFLOW ){
            error_ep_request_put_message( p_req->status );
        }
        /* å(ǡλ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_RCV_DATA;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_out( p_req->status );
    }

    return;
}

/**
 * compRcvDataOut
 **/
static GSTORAGE__State compRcvDataOut( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Csw   *pCsw;
    unsigned int    reqSize;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    pCsw    = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    reqSize = ( pCsw->dCSWDataResidue > GSTORAGE__RECEIVE_BUF_SIZE ) ?
                GSTORAGE__RECEIVE_BUF_SIZE : pCsw->dCSWDataResidue;

    /* pMainCtx->rcvLenOnCbw  */
    if ( pMainCtx->rcvLenOnCbw < pMsg->compCb.size ){
        pMainCtx->rcvLenOnCbw  = 0;
    }
    else {
        pMainCtx->rcvLenOnCbw -= pMsg->compCb.size;
    }

    /* Ƚ */
    /* pCsw->bCSWStatus  DEFAULT ʤΤ Status ǤϾʬʤ */
    if ( reqSize == pMsg->compCb.size ){
        /* PACKET ޥ */
        storage_drvcnt_sndCmdPacket( sndCmdPacketCb, pMainCtx->pPckt );
        /* ޥλԤ(ǡλ)Ԥ  */
        state = MAINCNT_CMD__WAIT_SND_CMD_DATA_OUT;
    }
    else {
        /* ǡ顼 PHASE ERROR  CSW  */
        pCsw->bCSWStatus = STORAGE_CSW_STATUS_PHASE_ERROR;
        state = sndCsw( pMsg );
    }

    return state;
}

/**
 * rcvDataOutContinueCb
 **/
static void rcvDataOutContinueCb( struct usb_ep *p_ep, struct usb_request *p_req )
{
    GSTORAGE__Msg msg;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    if ( ( p_req->status == 0 ) || ( p_req->status == -EOVERFLOW ) ){
        if ( p_req->status == -EOVERFLOW ){
            error_ep_request_put_message( p_req->status );
        }
        
        /* å(ǡλ) */
        msg.compCb.id   = MSG_MAINCNT__COMP_RCV_DATA_CONTINUE;
        msg.compCb.stat = ( GSTORAGE__Status )p_req->status;
        msg.compCb.pBuf = ( unsigned int * )p_req->buf;
        msg.compCb.size = ( unsigned int )p_req->actual;
        maincnt_sendMsg( &msg, sizeof( msg.compCb ) );
    }
    else {
        error_ep_request_cb_bulk_out( p_req->status );
    }
    
    return;
}

/**
 * compRcvDataOutContinue
 **/
static GSTORAGE__State compRcvDataOutContinue( GSTORAGE__Msg *pMsg )
{
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    GSTORAGE__Cbw   *pCbw;
    GSTORAGE__Csw   *pCsw;
    unsigned int    reqSize;
    unsigned int    *pBuf;
    unsigned char   status;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    pCbw    = ( GSTORAGE__Cbw * )( pMainCtx->pCbwBuf );
    pCsw    = ( GSTORAGE__Csw * )( pMainCtx->pCswBuf );
    reqSize = ( pCsw->dCSWDataResidue > GSTORAGE__RECEIVE_BUF_SIZE ) ?
                GSTORAGE__RECEIVE_BUF_SIZE : pCsw->dCSWDataResidue;

    /* pMainCtx->rcvLenOnCbw  */
    if ( pMainCtx->rcvLenOnCbw < pMsg->compCb.size ){
        pMainCtx->rcvLenOnCbw  = 0;
    }
    else {
        pMainCtx->rcvLenOnCbw -= pMsg->compCb.size;
    }

    /* Ƚ */
    if ( pCsw->bCSWStatus == STORAGE_CSW_STATUS_DEFAULT ){
        if ( reqSize == pMsg->compCb.size ){
            /* CONTINUE ޥ */
            storage_drvcnt_sndCmdContinue( sndCmdPacketCb );
            /* Write Continue λԤ */
            state = MAINCNT_CMD__WAIT_SND_CMD_DATA_OUT;
        }
        else {
            /* ǡ顼 PHASE ERROR  CSW  */
            pCsw->bCSWStatus = STORAGE_CSW_STATUS_PHASE_ERROR;
            state = sndCsw( pMsg );
        }
    }
    else {
        if ( reqSize == pMsg->compCb.size ){
            /* Хå */
            if ( pCsw->bCSWStatus == STORAGE_CSW_STATUS_PASS ){
                status = GSTORAGE_DRV__PASS;
            }
            else if ( pCsw->bCSWStatus == STORAGE_CSW_STATUS_FAIL ){
                status = GSTORAGE_DRV__FAIL;
            }
            else {
                status = GSTORAGE_DRV__FATAL;
            }
            pBuf = ( unsigned int * )storage_drvcnt_getReceiveBufP( pCbw->cbwLun );
            sndCmdPacketCb( GSTORAGE__SUCCESS, pCsw->dCSWDataResidue, status, pBuf );
            /* ޥλԤ(ǡλ)Ԥ  */
            state = MAINCNT_CMD__WAIT_SND_CMD_DATA_OUT;
        }
        else {
            /* ǡ顼 pCsw->bCSWStatus ݻ CSW  */
            state = sndCsw( pMsg );
        }
    }

    return state;
}


/**
    ꡼
 **/
/**
 * waitResetRecovery
 **/
static GSTORAGE__State waitResetRecovery( GSTORAGE__Msg *pMsg )
{
    unsigned long   flags;
    int             err = 0;
    GSTORAGE__State state = MAINCNT_STATE__NO_CHANGE;
    struct usb_ep   *p_ep;

    GS_DET( GS_OUT__MAINCNT, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    p_ep = pMainCtx->p_ep[ GSTORAGE_EP_SEND ];
    if ( p_ep != NULL ){
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
        spin_unlock_irq( &pStorageMainCnt->lock );
#endif
        storage_dequeue_ep( p_ep );
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
        spin_lock_irq( &pStorageMainCnt->lock );
#endif
        err = usb_ep_set_halt( p_ep );
    }
    else {
        err = GSTORAGE__ERROR;
        GS_INF( GS_OUT__MAINCNT, "p_ep[GSTORAGE_EP_SEND] error" );
    }
    if ( !err ) {
        p_ep = pMainCtx->p_ep[ GSTORAGE_EP_RECEIVE ];
        if ( p_ep != NULL ){
            err = usb_ep_set_halt( p_ep );
            storage_dequeue_ep( p_ep );
        }
        else {
            err = GSTORAGE__ERROR;
            GS_INF( GS_OUT__MAINCNT, "p_ep[GSTORAGE_EP_RECEIVE] error" );
        }
    }
    storage_set_waitResetRecovery();

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    if ( !err ) {
        state = MAINCNT_CMD__WAIT_COMP_RESET_RECOVERY;
    }
    else {
        /* disEpQueue ե饰 */
        setDisEpQueueOff();
        /* Drive Controller  Stop ׵ Message ؿƤ */
        storage_drvcnt_stopDrvCnt( stopCb );
        /* Stop λԤ */
        state = MAINCNT_ON__WAIT_COMP_STOP;
    }

    return state;
}


/**
    ERROR Ϣ
 **/
/**
 * error_ep_request_bulk_in
 **/
static void error_ep_request_bulk_in( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );
    error_ep_request_put_message( err );

    return;
}

/**
 * error_ep_request_cb_bulk_in
 **/
static void error_ep_request_cb_bulk_in( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );
    error_ep_request_put_message( err );

    return;
}

/**
 * error_ep_request_bulk_out
 **/
static void error_ep_request_bulk_out( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );
    error_ep_request_put_message( err );

    return;
}

/**
 * error_ep_request_cb_bulk_out
 **/
static void error_ep_request_cb_bulk_out( int err )
{
    GS_DET( GS_OUT__USB_EP_QUEUE, "func Req" );
    error_ep_request_put_message( err );

    return;
}

/**
 * error_ep_request_put_message
 **/
void error_ep_request_put_message( int err )
{
    switch( err ){
      case -ESHUTDOWN :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -ESHUTDOWN" );
        break;
      case -EINVAL :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -EINVAL" );
        break;
      case -ECONNRESET :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -ECONNRESET" );
        break;
      case -EPROTO :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -EPROTO" );
        break;
      case -EOVERFLOW :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -EOVERFLOW" );
        break;
      case -ECOMM :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: -ECOMM" );
        break;
      default :
        GS_INF( GS_OUT__USB_EP_QUEUE, "ERROR :: %d", err );
        break;
    }

    return;
}


/**
    ååɴϢ
 **/
/**
 * storage_maincnt_allocThread
 **/
int storage_maincnt_allocThread( void )
{
    int err = GSTORAGE__SUCCESS;
    int pid = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
    struct task_struct *tsk;
#endif

    GS_DET( GS_OUT__PROBE, "func Req" );

    /* create message buffer */
    if ( !( pMbf = cre_mbf( sizeof( GSTORAGE__Msg ) + 4, MSG_MAINCNT_MAX_NUM ) ) ){
        err = GSTORAGE__ERROR;
        GS_ERR( GS_OUT__MAINCNT, "cre_mbf() error" );
    }

    if ( !err ){
        /* create & init thread information */
        if ( !( pStorageMainCnt = kmalloc( sizeof( GSTORAGE__Thread ), GFP_ATOMIC ) ) ){
            del_mbf( pMbf );
            pMbf = NULL;
            err = GSTORAGE__ERROR;
            GS_ERR( GS_OUT__MAINCNT, "kmalloc(pStorageMainCnt) error" );
        }
        else {
            memset( pStorageMainCnt,  0, sizeof( GSTORAGE__Thread ) );
            init_waitqueue_head( &pStorageMainCnt->wqh );
            spin_lock_init( &pStorageMainCnt->lock );
            spin_lock_init( &pStorageMainCnt->lock_storage_in );
            pStorageMainCnt->end = 0;
        }
    }

    if ( !err ){
        /* create & start thread */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
        tsk = kthread_run(maincnt_threadFunc, pStorageMainCnt, "maincnt_threadFunc");
        if (IS_ERR(tsk)) {
            storage_maincnt_freeThread();
            err = GSTORAGE__ERROR;
            GS_ERR( GS_OUT__MAINCNT, "kthread_run() error (%p)", tsk );
        }
        else
            pid = tsk->pid;
#else
        if( ( pid = kernel_thread( maincnt_threadFunc, pStorageMainCnt,
                           ( CLONE_VM | CLONE_FS | CLONE_FILES ) ) ) < 0 ){
            storage_maincnt_freeThread();
            err = GSTORAGE__ERROR;
            GS_ERR( GS_OUT__MAINCNT, "kernel_thread() error" );
        }
#endif
    }
    GS_DET( GS_OUT__DAEMONIZE, "maincnt_threadFunc pid = %d", pid );

    return err;
}

/**
 * storage_maincnt_freeThread
 **/
void storage_maincnt_freeThread( void )
{
    GS_DET( GS_OUT__REMOVE, "func Req" );

    if ( pStorageMainCnt ){
        kfree( pStorageMainCnt );
        pStorageMainCnt = NULL;
    }
    if ( pMbf ){
        del_mbf( pMbf );
        pMbf = NULL;
    }

    return;
}

/**
 * storage_maincnt_getThread
 **/
GSTORAGE__Thread *storage_maincnt_getThread( void )
{
    GS_DET( GS_OUT__MAINCNT, "func Req" );

    return pStorageMainCnt;
}

/**
 * storage_maincnt_getEp
 **/
struct usb_ep *storage_maincnt_getEp( int i )
{
    struct usb_ep *p_ep = NULL;

    GS_DET( GS_OUT__RESET_RECOVERY, "func Req" );

    if ( ( i == GSTORAGE_EP_SEND ) || ( i == GSTORAGE_EP_RECEIVE ) ){
        p_ep = pMainCtx->p_ep[ i ];
    }

    return p_ep;
}

/**
 * storage_maincnt_setEp
 **/
void storage_maincnt_setEp( struct usb_ep *p_ep, int i )
{
    unsigned long flags;
    
    GS_DET( GS_OUT__START, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    pMainCtx->p_ep[ i ] = p_ep;

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    return;
}

/**
 * storage_maincnt_clearEp
 **/
void storage_maincnt_clearEp( void )
{
    int i;
    unsigned long flags;

    GS_DET( GS_OUT__STOP, "func Req" );

    /* ¾ */
    spin_lock_irqsave( &pStorageMainCnt->lock, flags );

    for ( i = 0; i < GSTORAGE_EP_NUM; i++ ){
        pMainCtx->p_ep[ i ] = NULL;
    }

    /* ޤ¾ */
    spin_unlock_irqrestore( &pStorageMainCnt->lock, flags );

    return;
}

/**
 * maincnt_sendMsg
 **/
static void maincnt_sendMsg( GSTORAGE__Msg *pMsg, int msgSize )
{
    GS_DET( GS_OUT__MBF, "func Req" );

    /* send Message */
    if ( snd_mbf( pMbf, pMsg, msgSize ) ){
        GS_ERR( GS_OUT__MBF, "snd_mbf() error" );
    }

    return;
}

/**
 * void maincnt_threadFunc( void )
 **/
static int maincnt_threadFunc( void *pStorageMainCnt_ )
{
    int             msgSize;
    GSTORAGE__Msg   msg;
    GSTORAGE__State currentState = PROBE_OFF;
    GSTORAGE__State nextState;



    pStorageMainCnt = ( GSTORAGE__Thread * )pStorageMainCnt_;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
#else
    daemonize("maincnt_threadFunc");
#endif
#if GS_OUT__GET_LOG
    logCnt = 0;
    log[ logCnt ].state = 0xFF;
    log[ logCnt ].msg   = 0xFF;
    logCnt++;
#endif

    while( 1 ){
        /* receive Message */
        do {
            msgSize = rcv_mbf( pMbf, &msg );
        } while( msgSize  == 0 );
        if ( msgSize < 0 ){
            GS_ERR( GS_OUT__MAINCNT, "rcv_mbf() error" );
            return 0;
        }

        /* ؿ¹ */
        nextState = storage_execFunc( &msg, currentState,
                                      spStorageMsgTbl_MainCnt );
#if GS_OUT__GET_LOG
        set_log_data( currentState, msg.id );
#endif
        GS_DBG( GS_OUT__NEXT_STATUS, "NEXT STATE :: %d", nextState );
        /* ξ֤ */
        currentState = nextState;

        /* λåɲ  Linux Ǥ thread λɬͭ */
        if ( pStorageMainCnt->end ){
            /* wake_up_interruptible() ƤӽФ */
            storage_exec_remove();
            break;
        }
    }

    return 0;
}

void set_log_data( unsigned int state, unsigned int msg )
{
#if GS_OUT__GET_LOG
    log[ logCnt ].state = state;
    log[ logCnt ].msg   = msg;
    if ( logCnt < MAX_LOG_CNT - 1 ){
        logCnt++;
    }
    else {
        logCnt=0;
    }
    log[ logCnt ].state = 0xFF;
    log[ logCnt ].msg   = 0xFF;

    if ( logCnt == 0 ){
        put_log_data();
    }
#endif
    return;
}

void put_log_data( void )
{
#if GS_OUT__GET_LOG
    unsigned int i;

    printk("  state     msg \n");
    printk(" -----------------\n");

    for ( i = 0; i < MAX_LOG_CNT; i++ ){
        printk( "[%3d   %3d]\n", log[i].state, log[i].msg );
    }
#endif
    return;
}


