/*
 * kernel/power/suspend.c - Suspend to RAM and standby functionality.
 *
 * Copyright (c) 2003 Patrick Mochel
 * Copyright (c) 2003 Open Source Development Lab
 * Copyright (c) 2009 Rafael J. Wysocki <rjw@sisk.pl>, Novell Inc.
 *
 * This file is released under the GPLv2.
 */

#include <linux/string.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/console.h>
#include <linux/cpu.h>
#include <linux/cpuidle.h>
#include <linux/syscalls.h>
#include <linux/gfp.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/export.h>
#include <linux/suspend.h>
#include <linux/syscore_ops.h>
#include <linux/snsc_boot_time.h>
#include <linux/ftrace.h>
#include <trace/events/power.h>
#include <linux/interrupt.h>
#include <linux/compiler.h>
#include <linux/moduleparam.h>

#ifdef CONFIG_UNIPAGE
#include <linux/ksm.h>
#endif
#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#endif
#include "power.h"

#ifdef CONFIG_SNSC_SAFE_SUSPEND
extern int suspend_remount(void);
extern int resume_remount(void);
#endif

static volatile int pm_prevent_suspend = 0;

void pm_suspend_stop(int stop)
{
	pm_prevent_suspend = stop;
	smp_wmb();
}

#ifdef PM_SUSPEND_DISK
const char *pm_labels[] = { "warm", "mem", "standby", "freeze", NULL };
#elif defined(CONFIG_SNSC_SSBOOT)
const char *pm_labels[] = { "snapshot", "mem", "standby", "freeze", NULL };
#else
const char *pm_labels[] = { "mem", "standby", "freeze", NULL };
#endif

const char *pm_states[PM_SUSPEND_MAX];

unsigned int pm_suspend_global_flags;
EXPORT_SYMBOL_GPL(pm_suspend_global_flags);

static const struct platform_suspend_ops *suspend_ops;
static const struct platform_freeze_ops *freeze_ops;
static DECLARE_WAIT_QUEUE_HEAD(suspend_freeze_wait_head);

enum freeze_state __read_mostly suspend_freeze_state;
static DEFINE_SPINLOCK(suspend_freeze_lock);

void freeze_set_ops(const struct platform_freeze_ops *ops)
{
	lock_system_sleep();
	freeze_ops = ops;
	unlock_system_sleep();
}

static void freeze_begin(void)
{
	suspend_freeze_state = FREEZE_STATE_NONE;
}

static void freeze_enter(void)
{
	spin_lock_irq(&suspend_freeze_lock);
	if (pm_wakeup_pending())
		goto out;

	suspend_freeze_state = FREEZE_STATE_ENTER;
	spin_unlock_irq(&suspend_freeze_lock);

	get_online_cpus();
	cpuidle_resume();

	/* Push all the CPUs into the idle loop. */
	wake_up_all_idle_cpus();
	pr_debug("PM: suspend-to-idle\n");
	/* Make the current CPU wait so it can enter the idle loop too. */
	wait_event(suspend_freeze_wait_head,
		   suspend_freeze_state == FREEZE_STATE_WAKE);
	pr_debug("PM: resume from suspend-to-idle\n");

	cpuidle_pause();
	put_online_cpus();

	spin_lock_irq(&suspend_freeze_lock);

 out:
	suspend_freeze_state = FREEZE_STATE_NONE;
	spin_unlock_irq(&suspend_freeze_lock);
}

void freeze_wake(void)
{
	unsigned long flags;

	spin_lock_irqsave(&suspend_freeze_lock, flags);
	if (suspend_freeze_state > FREEZE_STATE_NONE) {
		suspend_freeze_state = FREEZE_STATE_WAKE;
		wake_up(&suspend_freeze_wait_head);
	}
	spin_unlock_irqrestore(&suspend_freeze_lock, flags);
}
EXPORT_SYMBOL_GPL(freeze_wake);

static bool valid_state(suspend_state_t state)
{
	/*
	 * PM_SUSPEND_STANDBY and PM_SUSPEND_MEM states need low level
	 * support and need to be valid to the low level
	 * implementation, no valid callback implies that none are valid.
	 */
	return suspend_ops && suspend_ops->valid && suspend_ops->valid(state);
}

/*
 * If this is set, the "mem" label always corresponds to the deepest sleep state
 * available, the "standby" label corresponds to the second deepest sleep state
 * available (if any), and the "freeze" label corresponds to the remaining
 * available sleep state (if there is one).
 */
static bool relative_states;

void __init pm_states_init(void)
{
	/*
	 * freeze state should be supported even without any suspend_ops,
	 * initialize pm_states accordingly here
	 */
	pm_states[PM_SUSPEND_FREEZE] = pm_labels[relative_states ? 0 : 2];
}

static int __init sleep_states_setup(char *str)
{
	relative_states = !strncmp(str, "1", 1);
	return 1;
}

__setup("relative_sleep_states=", sleep_states_setup);

/**
 * suspend_set_ops - Set the global suspend method table.
 * @ops: Suspend operations to use.
 */
void suspend_set_ops(const struct platform_suspend_ops *ops)
{
	suspend_state_t i;
	int j = 0;

	lock_system_sleep();

	suspend_ops = ops;
        for (i = PM_SUSPEND_MAX - 1; i >= PM_SUSPEND_STANDBY; i--)
		if (valid_state(i)) {
			pm_states[i] = pm_labels[j++];
		} else if (!relative_states) {
			pm_states[i] = NULL;
			j++;
		}

	pm_states[PM_SUSPEND_FREEZE] = pm_labels[j];

	unlock_system_sleep();
}
EXPORT_SYMBOL_GPL(suspend_set_ops);

/**
 * suspend_valid_only_mem - Generic memory-only valid callback.
 *
 * Platform drivers that implement mem suspend only and only need to check for
 * that in their .valid() callback can use this instead of rolling their own
 * .valid() callback.
 */
int suspend_valid_only_mem(suspend_state_t state)
{
	return state == PM_SUSPEND_MEM;
}
EXPORT_SYMBOL_GPL(suspend_valid_only_mem);

static bool sleep_state_supported(suspend_state_t state)
{
	return state == PM_SUSPEND_FREEZE || (suspend_ops && suspend_ops->enter);
}

static int platform_suspend_prepare(suspend_state_t state)
{
	return state != PM_SUSPEND_FREEZE && suspend_ops->prepare ?
		suspend_ops->prepare() : 0;
}

static int platform_suspend_prepare_late(suspend_state_t state)
{
	return state == PM_SUSPEND_FREEZE && freeze_ops && freeze_ops->prepare ?
		freeze_ops->prepare() : 0;
}

static int platform_suspend_prepare_noirq(suspend_state_t state)
{
	return state != PM_SUSPEND_FREEZE && suspend_ops->prepare_late ?
		suspend_ops->prepare_late() : 0;
}

static void platform_resume_noirq(suspend_state_t state)
{
	if (state != PM_SUSPEND_FREEZE && suspend_ops->wake)
		suspend_ops->wake();
}

static void platform_resume_early(suspend_state_t state)
{
	if (state == PM_SUSPEND_FREEZE && freeze_ops && freeze_ops->restore)
		freeze_ops->restore();
}

static void platform_resume_finish(suspend_state_t state)
{
	if (state != PM_SUSPEND_FREEZE && suspend_ops->finish)
		suspend_ops->finish();
}

static int platform_suspend_begin(suspend_state_t state)
{
	if (state == PM_SUSPEND_FREEZE && freeze_ops && freeze_ops->begin)
		return freeze_ops->begin();
	else if (suspend_ops && suspend_ops->begin)
		return suspend_ops->begin(state);
	else
		return 0;
}

static void platform_resume_end(suspend_state_t state)
{
	if (state == PM_SUSPEND_FREEZE && freeze_ops && freeze_ops->end)
		freeze_ops->end();
	else if (suspend_ops && suspend_ops->end)
		suspend_ops->end();
}

static void platform_recover(suspend_state_t state)
{
	if (state != PM_SUSPEND_FREEZE && suspend_ops->recover)
		suspend_ops->recover();
}

static bool platform_suspend_again(suspend_state_t state)
{
	return state != PM_SUSPEND_FREEZE && suspend_ops->suspend_again ?
		suspend_ops->suspend_again() : false;
}

#ifdef CONFIG_PM_DEBUG
static unsigned int pm_test_delay = 5;
module_param(pm_test_delay, uint, 0644);
MODULE_PARM_DESC(pm_test_delay,
		 "Number of seconds to wait before resuming from suspend test");
#endif

int resume_phase = 0;

static int suspend_test(int level)
{
#ifdef CONFIG_PM_DEBUG
	if (pm_test_level == level) {
		pr_info("suspend debug: Waiting for %d second(s).\n",
				pm_test_delay);
		mdelay(pm_test_delay * 1000);
		return 1;
	}
#endif /* !CONFIG_PM_DEBUG */
	return 0;
}

#ifdef CONFIG_SNSC_SSBOOT
static int suspend_prepare_ssboot(int kernel_task_only)
{
	int error;

	if (!kernel_task_only && freeze_processes()) {
		error = -EAGAIN;
		return error;
	}

#ifdef CONFIG_UNIPAGE
	if(!ssboot_is_profiling() && !ssboot_is_rewriting())
		ssboot_full_scan_merge_sync();
#endif

#ifdef CONFIG_WARM_BOOT_IMAGE
	if (get_ssboot_stat() == SSBOOT_CREATE_OPTSSBI) {
		error = ssboot_optimizer_optimize();
	} else {
		error = ssboot_prepare();
	}
#else
	error = ssboot_prepare();
#endif

	if (error) {
		thaw_processes();
		return error;
	}

#ifdef CONFIG_SNSC_SAFE_SUSPEND
	/*
	 * Because sync_filesystem() called from
	 * suspend_remount() wakes up bdi threads, and wait
	 * for completion of it. Because bdi threads are
	 * freezable, we have to remount before freezing that
	 * tasks.
	 */
	if (suspend_remount()) {
		error = -EIO;
		thaw_processes();
		return error;
	}
#endif

	if (freeze_kernel_threads()) {
		error = -EAGAIN;
		thaw_processes();
		return error;
	}

	return 0;
}
#endif /* CONFIG_SNSC_SSBOOT */

/**
 * suspend_prepare - Prepare for entering system sleep state.
 *
 * Common code run for every system sleep state that can be entered (except for
 * hibernation).  Run suspend notifiers, allocate the "suspend" console and
 * freeze processes.
 */
static int suspend_prepare(suspend_state_t state)
{
	int error, nr_calls = 0;

	if (!sleep_state_supported(state))
		return -EPERM;

	pm_prepare_console();

	error = __pm_notifier_call_chain(PM_SUSPEND_PREPARE, -1, &nr_calls);
	if (error) {
		nr_calls--;
		goto Finish;
	}

#if defined(CONFIG_SNSC_SSBOOT) && !defined(CONFIG_WARM_BOOT_IMAGE)
       if (state == PM_SUSPEND_SNAPSHOT) {
               error = suspend_prepare_ssboot(0);
               if (error)
                       goto Failed;
       } else {
#endif
	trace_suspend_resume(TPS("freeze_processes"), 0, true);
#ifdef CONFIG_SUSPEND_FREEZER
	error = freeze_processes();
#endif
	trace_suspend_resume(TPS("freeze_processes"), 0, false);
	if (!error) {
#if defined (CONFIG_UNIPAGE) && defined (CONFIG_WARM_BOOT_IMAGE) && \
	defined (CONFIG_SNSC_SSBOOT)
		if(!ssboot_is_profiling() && !ssboot_is_rewriting())
			ssboot_full_scan_merge_sync();
#endif
#ifdef CONFIG_SNSC_SAFE_SUSPEND
		(void) suspend_remount();
#endif
#ifdef CONFIG_SUSPEND_FREEZER
		error = freeze_kernel_threads();
		if (error)
			thaw_processes();
		else
			return 0;
#else
		return 0;
#endif
	}

#if defined(CONFIG_SNSC_SSBOOT) && !defined(CONFIG_WARM_BOOT_IMAGE)
	}
	return 0;
 Failed:
#endif
	suspend_stats.failed_freeze++;
	dpm_save_failed_step(SUSPEND_FREEZE);
 Finish:
	__pm_notifier_call_chain(PM_POST_SUSPEND, nr_calls, NULL);
	pm_restore_console();
	return error;
}

/* default implementation */
void __weak arch_suspend_disable_irqs(void)
{
	local_irq_disable();
}

/* default implementation */
void __weak arch_suspend_enable_irqs(void)
{
	local_irq_enable();
}

/**
 * suspend_enter - Make the system enter the given sleep state.
 * @state: System sleep state to enter.
 * @wakeup: Returns information that the sleep state should not be re-entered.
 *
 * This function should be called after devices have been suspended.
 */
static int suspend_enter(suspend_state_t state, bool *wakeup)
{
	int error;

	error = platform_suspend_prepare(state);
	if (error)
		goto Platform_finish;

	error = dpm_suspend_late(PMSG_SUSPEND);
	if (error) {
		pr_err("PM: late suspend of devices failed\n");
		goto Platform_finish;
	}
	error = platform_suspend_prepare_late(state);
	if (error)
		goto Devices_early_resume;

	error = dpm_suspend_noirq(PMSG_SUSPEND);
	if (error) {
		pr_err("PM: noirq suspend of devices failed\n");
		goto Platform_early_resume;
	}
	error = platform_suspend_prepare_noirq(state);
	if (error)
		goto Platform_wake;

	if (suspend_test(TEST_PLATFORM))
		goto Platform_wake;

	/*
	 * PM_SUSPEND_FREEZE equals
	 * frozen processes + suspended devices + idle processors.
	 * Thus we should invoke freeze_enter() soon after
	 * all the devices are suspended.
	 */
	if (state == PM_SUSPEND_FREEZE) {
		trace_suspend_resume(TPS("machine_suspend"), state, true);
		freeze_enter();
		trace_suspend_resume(TPS("machine_suspend"), state, false);
		goto Platform_wake;
	}

#ifdef CONFIG_SNSC_SSBOOT
	/* Make sure all free pages linked to free list. */
	drain_all_pages(NULL);

	/* Make sure all page caches linked to LRU list. */
	lru_add_drain_all();
#endif

#ifdef CONFIG_SNSC_SAVE_PROCESS_AFFINITY_ON_SUSPEND
	save_process_affinity();
#endif
#ifdef CONFIG_SNSC_SAVE_IRQ_AFFINITY_ON_SUSPEND
	irq_save_affinity();
#endif
	error = disable_nonboot_cpus();
	if (error || suspend_test(TEST_CPUS))
		goto Enable_cpus;

	arch_suspend_disable_irqs();
	BUG_ON(!irqs_disabled());

	BOOT_TIME_ADD1("PM: suspend enter");
	error = syscore_suspend();
	if (!error) {
		*wakeup = pm_wakeup_pending();
		if (!(suspend_test(TEST_CORE) || *wakeup)) {
			trace_suspend_resume(TPS("machine_suspend"),
				state, true);
			error = suspend_ops->enter(state);
			trace_suspend_resume(TPS("machine_suspend"),
				state, false);
			if (!error)
				resume_phase = 1;
#ifdef CONFIG_SNSC_SSBOOT
			/*
			 * very low-level suspend code cannot
			 * return error on some architectures
			 */
			if (!error && ssboot_is_error())
				error = ssboot_get_error();
#endif
			events_check_enabled = false;
		} else if (*wakeup) {
			error = -EBUSY;
		}
		syscore_resume();
	}
#ifdef CONFIG_SNSC_BOOT_TIME
      boot_time_resume();
#endif
	BOOT_TIME_ADD1("PM: resume start");

	arch_suspend_enable_irqs();
	BUG_ON(irqs_disabled());

 Enable_cpus:
	enable_nonboot_cpus();

#ifdef CONFIG_SNSC_SAVE_IRQ_AFFINITY_ON_SUSPEND
       irq_restore_affinity();
#endif

#ifdef CONFIG_SNSC_SAVE_PROCESS_AFFINITY_ON_SUSPEND
	restore_process_affinity();
#endif

 Platform_wake:
	platform_resume_noirq(state);
	dpm_resume_noirq(PMSG_RESUME);
	BOOT_TIME_ADD1("PM: device resumed(noirq)");

 Platform_early_resume:
	platform_resume_early(state);

 Devices_early_resume:
	dpm_resume_early(PMSG_RESUME);

 Platform_finish:
	platform_resume_finish(state);
	return error;
}

/**
 * suspend_devices_and_enter - Suspend devices and enter system sleep state.
 * @state: System sleep state to enter.
 */
int suspend_devices_and_enter(suspend_state_t state)
{
	int error;
	bool wakeup = false;

	if (!sleep_state_supported(state))
		return -ENOSYS;

#ifdef CONFIG_SNSC_SSBOOT
 Rewrite:
#endif
	error = platform_suspend_begin(state);
	if (error)
		goto Close;

	suspend_console();
	suspend_test_start();
	error = dpm_suspend_start(PMSG_SUSPEND);
	if (error) {
		pr_err("PM: Some devices failed to suspend, or early wake event detected\n");
		goto Recover_platform;
	}
	suspend_test_finish("suspend devices");
	if (suspend_test(TEST_DEVICES))
		goto Recover_platform;

	do {
#ifdef CONFIG_SNSC_SSBOOT
		error = suspend_enter(state, &wakeup);
#else
               suspend_enter(state, &wakeup);
#endif
	} while (!error && !wakeup && platform_suspend_again(state));

 Resume_devices:
	suspend_test_start();
	dpm_resume_end(PMSG_RESUME);
	BOOT_TIME_ADD1("PM: device resumed");
	suspend_test_finish("resume devices");
	trace_suspend_resume(TPS("resume_console"), state, true);
	resume_console();
	trace_suspend_resume(TPS("resume_console"), state, false);

 Close:
	platform_resume_end(state);
	resume_phase = 0;
#ifdef CONFIG_SNSC_SAVE_AFFINITY
	printk("Restore affinity.\n");
	restore_affinity();
#endif /* CONFIG_SNSC_SAVE_AFFINITY */
#ifdef CONFIG_SNSC_SSBOOT
# ifdef CONFIG_WARM_BOOT_IMAGE
	if (!error && get_ssboot_stat() == SSBOOT_PROFILE) {
		error = ssboot_optimizer_start_profiling();
		if (error < 0) {
			printk("ssboot_optimizer_start_profiling() failed\n");
		}
	}
	else if (!error &&
		 get_ssboot_stat() == SSBOOT_CREATE_OPTSSBI) {
#ifdef	CONFIG_SNSC_SAFE_SUSPEND
		(void) resume_remount();
#endif
		/* thaw kernel threads only */
#ifdef CONFIG_SNSC_SAVE_AFFINITY
		printk("Restore affinity.\n");
		restore_affinity();
#endif /* CONFIG_SNSC_SAVE_AFFINITY */
		thaw_kernel_threads();
		thaw_workqueues();

		/* prepare for suspend again */
		error = suspend_prepare_ssboot(1);
		if (!error) {
			goto Rewrite;
		}
	}
# else /* CONFIG_WARM_BOOT_IMAGE */
       if (state == PM_SUSPEND_SNAPSHOT) {
               int ret;
               if (!error && ssboot_is_writing()) {
#ifdef CONFIG_SNSC_SAFE_SUSPEND
			(void) resume_remount();
#endif
                       /* thaw kernel threads only */
			thaw_kernel_threads();
			error = ssboot_write();
               }
               ret = ssboot_finish();
               if (!error) {
                       error = ret;
               }
               if (!error && ssboot_is_rewriting()) {
#ifdef CONFIG_SNSC_SAFE_SUSPEND
			(void) resume_remount();
#endif
                       /* thaw kernel threads only */
                       thaw_kernel_threads();
                       thaw_workqueues();

                       /* prepare for suspend again */
                       error = suspend_prepare_ssboot(1);
                       if (!error) {
                               goto Rewrite;
                       }
               }
       }
# endif /* CONFIG_WARM_BOOT_IMAGE */
#endif /* CONFIG_SNSC_SSBOOT */
	return error;

 Recover_platform:
	platform_recover(state);
	goto Resume_devices;
}

/**
 * suspend_finish - Clean up before finishing the suspend sequence.
 *
 * Call platform code to clean up, restart processes, and free the console that
 * we've allocated. This routine is not called for hibernation.
 */
static void suspend_finish(void)
{
#ifdef	CONFIG_SNSC_SAFE_SUSPEND
	suspend_thaw_kernel_threads();
	/*
	 * Ideally we should be able to call resume_remount()
	 * twice as suspend_rw_fs_list would be empty.
	 * Calling resume_remount() simplifies code as of now.
	 */
	(void) resume_remount();
#endif
	suspend_thaw_processes();
	pm_notifier_call_chain(PM_POST_SUSPEND);
	pm_restore_console();
}

/**
 * enter_state - Do common work needed to enter system sleep state.
 * @state: System sleep state to enter.
 *
 * Make sure that no one else is trying to put the system into a sleep state.
 * Fail if that's not the case.  Otherwise, prepare for system suspend, make the
 * system enter the given sleep state and clean up after wakeup.
 */
static int enter_state(suspend_state_t state)
{
	int error;

	trace_suspend_resume(TPS("suspend_enter"), state, true);
	if (state == PM_SUSPEND_FREEZE) {
#ifdef CONFIG_PM_DEBUG
		if (pm_test_level != TEST_NONE && pm_test_level <= TEST_CPUS) {
			pr_warn("PM: Unsupported test mode for suspend to idle, please choose none/freezer/devices/platform.\n");
			return -EAGAIN;
		}
#endif
	} else if (!valid_state(state)) {
		return -EINVAL;
	}
	if (!mutex_trylock(&pm_mutex))
		return -EBUSY;

	if (state == PM_SUSPEND_FREEZE)
		freeze_begin();

#ifndef CONFIG_SUSPEND_SKIP_SYNC
	trace_suspend_resume(TPS("sync_filesystems"), 0, true);
	pr_info("PM: Syncing filesystems ... ");
	sys_sync();
	pr_cont("done.\n");
	trace_suspend_resume(TPS("sync_filesystems"), 0, false);
#endif

	smp_rmb();
	if (pm_prevent_suspend) {
		printk(KERN_ERR "PM: suspend is inhibited.\n");
		while (pm_prevent_suspend) {
			msleep(1000);
			smp_rmb();
		}
	}

	pr_debug("PM: Preparing system for sleep (%s)\n", pm_states[state]);
	pm_suspend_clear_flags();
#ifdef CONFIG_UNIPAGE
	ksm_set_run_state(KSM_RUN_STOP);
#endif
	error = suspend_prepare(state);
	if (error)
		goto Unlock;

	if (suspend_test(TEST_FREEZER))
		goto Finish;

	trace_suspend_resume(TPS("suspend_enter"), state, false);
	pr_debug("PM: Suspending system (%s)\n", pm_states[state]);
	pm_restrict_gfp_mask();
	error = suspend_devices_and_enter(state);
	pm_restore_gfp_mask();

 Finish:
	pr_debug("PM: Finishing wakeup.\n");
	suspend_finish();
	BOOT_TIME_ADD1("PM: resume finished");
 Unlock:
	mutex_unlock(&pm_mutex);
	return error;
}

/**
 * pm_suspend - Externally visible function for suspending the system.
 * @state: System sleep state to enter.
 *
 * Check if the value of @state represents one of the supported states,
 * execute enter_state() and update system suspend statistics.
 */
int pm_suspend(suspend_state_t state)
{
	int error;

	if (state <= PM_SUSPEND_ON || state >= PM_SUSPEND_MAX)
		return -EINVAL;

	error = enter_state(state);
	if (error) {
		suspend_stats.fail++;
		dpm_save_failed_errno(error);
	} else {
		suspend_stats.success++;
	}
	return error;
}
EXPORT_SYMBOL(pm_suspend);
