/*
 * mach-cxd900xx/warm.c
 *
 *
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/memblock.h>
#include <linux/warmboot.h>
#include <linux/wbi.h>
#include <linux/io.h>
#include <linux/wdt.h>
#include <linux/udif/cache.h>
#include <linux/udif/timer.h>

#undef WBI_ENABLE_TIMEOUT

#ifdef CONFIG_WBI_CLEAR_MEMORY
static unsigned long phys_memclr = 0UL;

static int __init parse_memclr(char *str)
{
	phys_memclr = memparse(str, &str);
	return 0;
}
early_param("wbi_memclr", parse_memclr);

#define WBI_MEMCLR_NONE	0
#define WBI_MEMCLR_EXEC	1

static int __init wbi_memclr(void)
{
	uint32_t memclr;

	if (!phys_memclr) {
		return 0;
	}
	memclr = readl((void __iomem *)phys_to_virt(phys_memclr));
	return (WBI_MEMCLR_EXEC == memclr);
}

static int __init wbi_memclr_exclude(unsigned long phys, unsigned long size)
{
	return (phys <= phys_memclr  &&  phys_memclr < phys + size);
}

void __init wbi_memclear(unsigned long phys, unsigned long size)
{
	void *va;

	if (!wbi_memclr())
		return;
	if (wbi_memclr_exclude(phys, size))
		return;

	va = (void *)phys_to_virt(phys);
	memset(va, 0, size);
	udif_cache_ctrl((UDIF_VA)va, size, UDIF_CACHE_FLUSH);
	printk(KERN_INFO "%s: 0x%lx-0x%lx\n", __func__, phys, phys+size-1);
}

void __init wbi_bootmem_clear(void)
{
	u64 i;
	phys_addr_t s, e;
	void *va;

	if (!wbi_memclr())
		return;

	for_each_free_mem_range(i, NUMA_NO_NODE, MEMBLOCK_NONE, &s, &e, NULL) {
		va = (void *)phys_to_virt(s);
		memset(va, 0, e - s);
		udif_cache_ctrl((UDIF_VA)va, e - s , UDIF_CACHE_FLUSH);
		printk(KERN_INFO "%s:%llx - %llx\n", __func__, s, e);
	}
}
#endif /* CONFIG_WBI_CLEAR_MEMORY */

static unsigned long wbi_t0;
#ifdef WBI_ENABLE_TIMEOUT
static unsigned long wbi_expire;
#endif

void wbi_timeout_init(uint tmo)
{
	wbi_t0 = udif_read_freerun();
#ifdef WBI_ENABLE_TIMEOUT
	if (tmo) {
		wbi_expire = wbi_t0 + udif_usecs_to_cycles(tmo*USEC_PER_MSEC);
	}
#endif
}

unsigned int wbi_timeout_elaps(void)
{
	unsigned long now = udif_read_freerun();

	return udif_cycles_to_usecs(now - wbi_t0);
}

#ifdef WBI_ENABLE_TIMEOUT
int wbi_is_timeout(void)
{
	unsigned long now = udif_read_freerun();

	if (!wbi_expire)
		return 0;
	return time_after_eq(now, wbi_expire);
}
#endif
