/*
 * drivers/usb/f_usb/usb_otg_extcon.c
 *
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/extcon.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>

#include <asm/barrier.h>

#include <linux/usb/f_usb/usb_otg_extcon.h>

struct fusb_otg_extcon_info {
	struct device *dev;
	struct extcon_dev *edev;
};

static struct fusb_otg_extcon_info *m_info = NULL;

static const unsigned int usb_otg_extcon_cable[] = {
	EXTCON_USB_HOST,
	EXTCON_USB,
	EXTCON_NONE,
};

/* DRD */
int fusb_otg_set_drd_state(bool host_state)
{
	int ret;

	if (NULL == m_info)
		return -EAGAIN;

	ret = extcon_set_state(m_info->edev, EXTCON_USB_HOST, host_state);
	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_state err=%d\n", ret);
		goto bail;
	}

	ret = extcon_sync(m_info->edev, EXTCON_USB_HOST);
	if (0 > ret) dev_err(m_info->dev, "extcon_sync err=%d\n", ret);

bail:
	return ret;
}
EXPORT_SYMBOL(fusb_otg_set_drd_state);

int fusb_otg_get_drd_state(bool *state)
{
	if (NULL == m_info)
		return -EAGAIN;

	*state = extcon_get_state(m_info->edev, EXTCON_USB_HOST);
	return 0;
}
EXPORT_SYMBOL(fusb_otg_get_drd_state);

void fusb_otg_flip_drd_state(void)
{
	bool state;
	int  err;

	err = fusb_otg_get_drd_state(&state);
	if (err)
		dev_err(m_info->dev, "fusb_otg_get_drd_state err=%d\n", err);
	else
		fusb_otg_set_drd_state(!state);
}
EXPORT_SYMBOL(fusb_otg_flip_drd_state);

/* tpc setting all */
int fusb_otg_set_tpc_setting(bool orientation, bool ss_operation)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	prop_val.intval = 0;
	if (orientation)
		prop_val.intval = 1;

	ret = extcon_set_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_TYPEC_POLARITY, prop_val);
	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_property prop_val=%d, err=%d\n",
		                     prop_val.intval, ret);
		goto bail;
	}

	prop_val.intval = 0;
	if (ss_operation)
		prop_val.intval = 1;

	ret = extcon_set_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_SS, prop_val);
	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_property prop_val=%d, err=%d\n",
		                     prop_val.intval, ret);
		goto bail;
	}

	ret = extcon_sync(m_info->edev, EXTCON_USB);
	if (0 > ret) dev_err(m_info->dev, "extcon_sync err=%d\n", ret);

bail:
	return ret;
}
EXPORT_SYMBOL(fusb_otg_set_tpc_setting);

/* type-c orientation */
int fusb_otg_set_orientation(bool orientation)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	prop_val.intval = 0;
	if (orientation)
		prop_val.intval = 1;

	ret = extcon_set_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_TYPEC_POLARITY, prop_val);
	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_property prop_val=%d, err=%d\n",
		                     prop_val.intval, ret);
		goto bail;
	}

	ret = extcon_sync(m_info->edev, EXTCON_USB);
	if (0 > ret) dev_err(m_info->dev, "extcon_sync err=%d\n", ret);

bail:
	return ret;
}
EXPORT_SYMBOL(fusb_otg_set_orientation);

/* ss operation */
int fusb_otg_set_ss_operation(bool ss_operation)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	prop_val.intval = 0;
	if (ss_operation)
		prop_val.intval = 1;

	ret = extcon_set_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_SS, prop_val);
	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_property prop_val=%d, err=%d\n",
		                     prop_val.intval, ret);
		goto bail;
	}

	ret = extcon_sync(m_info->edev, EXTCON_USB);
	if (0 > ret) dev_err(m_info->dev, "extcon_sync err=%d\n", ret);

bail:
	return ret;
}
EXPORT_SYMBOL(fusb_otg_set_ss_operation);

int fusb_otg_get_orientation(bool *state)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	ret = extcon_get_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_TYPEC_POLARITY, &prop_val);
	*state = !!prop_val.intval;
	return 0;
}
EXPORT_SYMBOL(fusb_otg_get_orientation);

int fusb_otg_get_ss_operation(bool *state)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	ret = extcon_get_property(m_info->edev, EXTCON_USB,
	          EXTCON_PROP_USB_SS, &prop_val);
	*state = !!prop_val.intval;
	return 0;
}
EXPORT_SYMBOL(fusb_otg_get_ss_operation);

void fusb_otg_flip_orientation(void)
{
	int  err;
	bool state, new_state;

	err = fusb_otg_get_orientation(&state);
	if (err)
		dev_err(m_info->dev, "fusb_otg_flip_orientation err=%d\n", err);
	else {
		new_state = !state;
		printk(KERN_INFO "%s flip orientation %d => %d\n", __FUNCTION__, state, new_state);
		err = fusb_otg_set_orientation(new_state);
		if (err)
			printk(KERN_INFO "%s fusb_otg_set_orientation err=%d, new orientation=%d\n", __FUNCTION__, err, new_state);
	}
}
EXPORT_SYMBOL(fusb_otg_flip_orientation);

void fusb_otg_flip_ss_operation(void)
{
	int  err;
	bool state, new_state;

	err = fusb_otg_get_ss_operation(&state);
	if (err)
		dev_err(m_info->dev, "fusb_otg_flip_ss_operation err=%d\n", err);
	else {
		new_state = !state;
		printk(KERN_INFO "%s flip ss_operation %d => %d\n", __FUNCTION__, state, new_state);
		err = fusb_otg_set_ss_operation(new_state);
		if (err)
			printk(KERN_INFO "%s fusb_otg_set_orientation err=%d, new ss_operation=%d\n", __FUNCTION__, err, new_state);
	}
}
EXPORT_SYMBOL(fusb_otg_flip_ss_operation);

/* returns 1 if enabled, 0 if disabled, negative if error on getting statuses */
int fusb_otg_is_orientation_enabled(void)
{
	int  ret;

	if (NULL == m_info)
		return -EAGAIN;

	ret = extcon_get_property_capability(m_info->edev, EXTCON_USB,
	                                     EXTCON_PROP_USB_TYPEC_POLARITY);
	if (ret < 0) {
		dev_err(m_info->dev, "failed to extcon_get_property_capability err=%d\n", ret);
		return ret;
	}
	if (!ret) {
		printk(KERN_INFO "%s EXTCON_PROP_USB_TYPEC_POLARITY is not capable.\n", __FUNCTION__);
		return 0;
	}

	/* turn attached state on to enable reading property */
	ret = extcon_get_state(m_info->edev, EXTCON_USB);
	if (ret < 0) {
		dev_err(m_info->dev, "extcon_get_state EXTCON_USB err=%d\n", ret);
	}

	return ret;
}
EXPORT_SYMBOL(fusb_otg_is_orientation_enabled);

/* returns 1 if enabled, 0 if disabled, negative if error on getting statuses */
int fusb_otg_is_ss_operation_enabled(void)
{
	int  ret;

	if (NULL == m_info)
		return -EAGAIN;

	ret = extcon_get_property_capability(m_info->edev, EXTCON_USB,
	                                     EXTCON_PROP_USB_SS);
	if (ret < 0) {
		dev_err(m_info->dev, "failed to extcon_get_property_capability err=%d\n", ret);
		return ret;
	}
	if (!ret) {
		printk(KERN_INFO "%s EXTCON_PROP_USB_SS is not capable.\n", __FUNCTION__);
		return 0;
	}

	/* turn attached state on to enable reading property */
	ret = extcon_get_state(m_info->edev, EXTCON_USB);
	if (ret < 0) {
		dev_err(m_info->dev, "extcon_get_state EXTCON_USB err=%d\n", ret);
	}

	return ret;
}
EXPORT_SYMBOL(fusb_otg_is_ss_operation_enabled);

int fusb_otg_extcon_setup(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct fusb_otg_extcon_info *info;
	int ret;

	if (m_info) return -EBUSY;

	info = devm_kzalloc(&pdev->dev, sizeof(*info), GFP_KERNEL);
	if (!info) {
		dev_err(dev, "failed to allocate fusb_otg_extcon_info\n");
		return -ENOMEM;
	}

	m_info = info;
	info->dev = dev;

	info->edev = devm_extcon_dev_allocate(dev, usb_otg_extcon_cable);
	if (IS_ERR(info->edev)) {
		dev_err(dev, "failed to allocate extcon device\n");
		ret = -ENOMEM;
		goto err_alloc;
	}

	ret = devm_extcon_dev_register(dev, info->edev);
	if (ret < 0) {
		dev_err(dev, "failed to register extcon device. err=%d\n", ret);
		goto err_alloc_dev;
	}

	platform_set_drvdata(pdev, info);

	ret = extcon_set_property_capability(info->edev, EXTCON_USB,
	                                     EXTCON_PROP_USB_TYPEC_POLARITY);
	if (ret < 0) {
		dev_err(dev, "failed to extcon_set_property_capability err=%d\n", ret);
		goto err_regist_dev;
	}

	ret = extcon_set_property_capability(info->edev, EXTCON_USB,
	                                     EXTCON_PROP_USB_SS);
	if (ret < 0) {
		dev_err(dev, "failed to extcon_set_property_capability err=%d\n", ret);
		goto err_regist_dev;
	}

	/* turn attached state on to enable reading property */
	ret = extcon_set_state_sync(info->edev, EXTCON_USB, true);
	if (ret < 0) {
		dev_err(dev, "extcon_set_state_sync err=%d\n", ret);
		goto err_regist_dev;
	}

	/*
	 * Perform initial detection
	 * start with device
	 */
	fusb_otg_set_drd_state(FUSB_DRD_GADGET);

	return 0;

err_regist_dev:
	devm_extcon_dev_unregister(dev, info->edev);
err_alloc_dev:
	devm_extcon_dev_free(dev, info->edev);
err_alloc:
	devm_kfree(dev, info);

printk(KERN_INFO "%s out err=%d\n", __FUNCTION__, ret);
	return ret;
}
EXPORT_SYMBOL(fusb_otg_extcon_setup);

int fusb_otg_extcon_teardown(struct platform_device *pdev)
{
#if 0  /* there is no extcon driver that frees any memories */
	struct fusb_otg_extcon_info *info = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;

	devm_extcon_dev_unregister(dev, info->edev);
	devm_extcon_dev_free(dev, info->edev);
	devm_kfree(dev, info);
#endif
	m_info = NULL;
	return 0;
}
EXPORT_SYMBOL(fusb_otg_extcon_teardown);

/* currently nothing to do
#ifdef CONFIG_PM_SLEEP
int fusb_otg_extcon_suspend(struct device *dev)
{
	struct fusb_otg_extcon_info *info = dev_get_drvdata(dev);
	int ret = 0;

	return ret;
}

int fusb_otg_extcon_resume(struct device *dev)
{
	struct fusb_otg_extcon_info *info = dev_get_drvdata(dev);
	int ret = 0;

	return ret;
}
#endif*/
