/*
 * usb_gcore_desc.c
 * 
 * Copyright 2005,2006,2011,2013 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 */

/*-----------------------------------------------------------------------------
 * Include file
 *---------------------------------------------------------------------------*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/wait.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/moduleparam.h>
#include <asm/uaccess.h>
#include <asm/types.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>

#include <linux/list.h>

#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#include "usb_gadgetcore_cfg.h"
#include "usb_gadgetcore_pvt.h"

#ifdef DBG_PREFIX
# undef  DBG_PREFIX
# define DBG_PREFIX "GCORE_DESC"
#else
# define DBG_PREFIX "GCORE_DESC"
#endif
#include "usb_gcore_wrapper_cfg.h"
#include "usb_gcore_wrapper_pvt.h"

/*-----------------------------------------------------------------------------
 * Function prototype declaration
 *---------------------------------------------------------------------------*/
static int save_bos_desc(usb_gadget_bos_desc* bos);
static int save_langids(usb_gadget_desc_table*);
static int save_string_desc(usb_gadget_langid_table*);
static int save_strings(usb_gadget_string_desc*);
static int overwrite_iserial(usb_gadget_desc_table*, __u8, void*);
static int save_config_desc(usb_gadget_speed_desc*);
static int save_report_desc(usb_gadget_report_desc*);
static int save_otg_desc(usb_gadget_config_desc*);
static int save_ia_desc(usb_gadget_config_desc*);
static int save_interface_desc(usb_gadget_config_desc*);
static int save_interface_table(usb_gadget_interface_desc*);
static int save_pep_list(usb_gadget_if_table*);
static int save_class_if_desc(usb_gadget_if_table*);
static int save_ep_companion_desc(usb_gadget_if_table_element*);
static int save_class_ep_desc(usb_gadget_if_table_element*);

static int free_bos(usb_gadget_bos_desc* bos);
static int free_langids(usb_gadget_desc_table*);
static int free_string_desc(usb_gadget_langid_table*);
static int free_strings(usb_gadget_string_desc*);
static int free_config_desc(usb_gadget_speed_desc*);
static int free_report_desc(usb_gadget_report_desc*);
static int free_otg_desc(usb_gadget_config_desc*);
static int free_ia_desc(usb_gadget_config_desc*);
static int free_interface_desc(usb_gadget_config_desc*);
static int free_interface_table(usb_gadget_interface_desc*);
static int free_class_if_desc(usb_gadget_if_table*);
static int free_pep_list(usb_gadget_if_table*);
static int free_ep_companion_desc(usb_gadget_if_table_element*);
static int free_class_ep_desc(usb_gadget_if_table_element*);

static void dump_config_desc(usb_gadget_config_desc*);
static void dump_otg_desc(usb_gadget_otg_desc*);
static void dump_ia_desc(usb_gadget_ia_desc*);
static void dump_interface_desc(usb_gadget_interface_desc*);
static void dump_if_tables(usb_gadget_if_table*);
static void dump_pep_list(usb_gadget_if_table_element*);
static void dump_langid_table(usb_gadget_langid_table*);
static void dump_string_desc(usb_gadget_string_desc*);

/*=============================================================================
 *
 * Main function body
 *
 *===========================================================================*/
/**
 * get_config_desc_from_config_num
 **
 * :    ߤBusSpeedǡǻꤵ줿ConfigֹConfigDesc
 *          
 * :    g_core     : GadgetCoreؤΥݥ
 *          config_num : Configֹ
 * :    !0   : (ConfigDescؤΥݥ)
 *          NULL : 
 * 
 * <EOS>                                *
 **/
usb_gadget_config_desc*
get_config_desc_from_config_num(struct g_core_drv *g_core, u8 config_num)
{
    usb_gadget_speed_desc *sp_desc;
    u8 i;
    
    /* ߤspeedSpeedDesc */
    if(g_core->gadget->speed >= USB_SPEED_SUPER){
        sp_desc = &g_core->desc_tbl->ss;
    }else if(g_core->gadget->speed == USB_SPEED_HIGH){
        sp_desc = &g_core->desc_tbl->hs;
    }else{
        sp_desc = &g_core->desc_tbl->lsfs;
    }
    
    /* configͤConfigDescriptorõ */
    for(i = 0; i < sp_desc->uc_num_config; i++){
        if(sp_desc->configurations[i].uc_configuration_value == config_num){
            return &sp_desc->configurations[i];
        }
    }
    
    /* Ĥʤä */
    return NULL;
}

/**
 * get_interface_desc
 **
 * :    ߤBusSpeedǡǻꤵ줿Config,Interface
 *          ֹinterface_desc
 * :    g_core     : GadgetCoreؤΥݥ
 *          config_num    : Configֹ
 *          interface_num : Interfaceֹ
 * :    !0   : (interface_descؤΥݥ)
 *          NULL : 
 * 
 * <EOS>                                *
 **/
usb_gadget_interface_desc*
get_interface_desc(struct g_core_drv *g_core, u8 config_num, u8 interface_num)
{
    usb_gadget_config_desc *cfg_desc;
    u8 i;
    
    /* ConfigConfigDesc */
    cfg_desc = get_config_desc_from_config_num(g_core, config_num);
    if(cfg_desc == NULL){
        return NULL;
    }
    
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        if(cfg_desc->interfaces[i].uc_if_number == interface_num){
            return &cfg_desc->interfaces[i];
        }
    }
    
    return NULL;
}

/**
 * get_if_table_desc
 **
 * :    ߤBusSpeedǡǻꤵ줿Config,Interface,Alternate
 *          ֹif_table_desc
 * :    g_core     : GadgetCoreؤΥݥ
 *          config_num    : Configֹ
 *          interface_num : Interfaceֹ
 *          alt_num       : Alternateֹ
 * :    !0   : (if_table_descؤΥݥ)
 *          NULL : 
 * 
 * <EOS>                                *
 **/
usb_gadget_if_table*
get_if_table_desc(struct g_core_drv *g_core, u8 config_num, u8 interface_num, u8 alt_num)
{
    usb_gadget_interface_desc *in_desc;
    u8 i;
    
    /* ConfigInterfaceInterfaceDesc */
    in_desc = get_interface_desc(g_core, config_num, interface_num);
    if(in_desc == NULL){
        return NULL;
    }
    
    for(i = 0; i < in_desc->uc_num_if_tables; i++){
        if(in_desc->if_tables[i].uc_setting_number == alt_num){
            return &in_desc->if_tables[i];
        }
    }
    
    return NULL;
}

/* ꤵƤConfigֹConfigDescriptor֤ */
usb_gadget_config_desc*
get_current_config_desc(struct g_core_drv *g_core)
{
    return get_config_desc_from_config_num(g_core, g_core->set_config);
}

/**
 * make_alt_num_list
 **
 * :    ߤConfigInterfaceʬAlternateΰݤ
 *          
 * :    g_core     : GadgetCoreؤΥݥ
 * :    0ʾ : 
 *          0̤ : 
 * 
 * <EOS>                                *
 **/
u8 make_alt_num_list(struct g_core_drv *g_core)
{
    usb_gadget_config_desc *cfg_desc;
    struct alt_num *tmp;
    u8 i;
    
    PDEBUG("make_alt_num_list()\n");
    
    /* ߤConfigDescriptor */
    cfg_desc = get_current_config_desc(g_core);
    
    /* Alternateֹ¸ΰ */
    tmp = (struct alt_num*)kmalloc(sizeof(struct alt_num) * cfg_desc->uc_num_interfaces,
                                   GFP_ATOMIC);
    if(!tmp){
        PDEBUG("   --->Error kmalloc()\n");
        return -ENOMEM;
    }
    
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        tmp[i].interface_num = cfg_desc->interfaces[i].uc_if_number;
        tmp[i].alternate_num = 0;
    }
    g_core->alt_nums = tmp;
    PDEBUG("   --->OK alloc %d\n", cfg_desc->uc_num_interfaces);
    
    return cfg_desc->uc_num_interfaces;
}

/**
 * set_alt_num
 **
 * :    AlternateΰλInterfaceֹInterface
 *          AlternateֹAlternateֹѹ
 *          
 * :    g_core     : GadgetCoreؤΥݥ
 *          in_num     : Interfaceֹ
 *          alt_num    : ꤹAlternateֹ
 * :     0 : 
 *          !0 : 
 * 
 * <EOS>                                *
 **/
int set_alt_num(struct g_core_drv *g_core, u8 in_num, u8 alt_num)
{
    usb_gadget_config_desc *cfg_desc;
    struct alt_num *tmp;
    u8 i;
    
    tmp = g_core->alt_nums;
    if(!tmp){
        return -EINVAL;
    }
    
    /* ߤConfigDescriptor */
    cfg_desc = get_current_config_desc(g_core);
    
    /* Altΰ褫Interfaceֹõ */
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        if(tmp[i].interface_num == in_num){
            /* InterfaceAltֹAltֹѹ */
            tmp[i].alternate_num = alt_num;
            return 0;
        }
    }
    
    return -EINVAL;
}

/**
 * get_alt_num
 **
 * :    AlternateΰλInterfaceֹInterface
 *          Alternateֹ
 * :    g_core     : GadgetCoreؤΥݥ
 *          in_num     : Interfaceֹ
 *          *alt_num   : Alternateֹ¸
 * :     0 : 
 *          !0 : 
 * 
 * <EOS>                                *
 **/
int get_alt_num(struct g_core_drv *g_core, u8 in_num, u8 *alt_num)
{
    usb_gadget_config_desc *cfg_desc;
    struct alt_num *tmp;
    u8 i;
    
    tmp = g_core->alt_nums;
    if(!tmp){
        return -EINVAL;
    }
    
    /* ߤConfigDescriptor */
    cfg_desc = get_current_config_desc(g_core);
    
    /* Altΰ褫Interfaceֹõ */
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        if(tmp[i].interface_num == in_num){
            /* InterfaceAltֹAltֹѹ */
            *alt_num = tmp[i].alternate_num;
            return 0;
        }
    }
    
    return -EINVAL;
}

/**
 * free_alt_num_list
 **
 * :    Alternateΰ
 * :    g_core     : GadgetCoreؤΥݥ
 * :    0ʾ : 
 *          0̤ : 
 * 
 * <EOS>                                *
 **/
void free_alt_num_list(struct g_core_drv *g_core)
{
    PDEBUG("kfree(g_core->alt_nums)\n");
    if(g_core->alt_nums){
        PDEBUG("    --->OK\n");
        kfree(g_core->alt_nums);
        g_core->alt_nums = NULL;
    }
}

/**
 * is_hnp_capable
 **
 * :    ߤConfigConfigDescriptorHNPĤƤ뤫֤
 *          ߤConfigֹ椬0λˤϡDescriptorTableDevice
 *          HNPĤƤ뤫ǧ
 * :    g_core     : GadgetCoreؤΥݥ
 *          config_num : Configֹ
 * :    !0 : HNPб
 *          0  : HNPб
 * 
 * <EOS>                                *
 **/
int is_hnp_capable(struct g_core_drv *g_core)
{
    usb_gadget_speed_desc *sp_desc;
    usb_gadget_config_desc *cfg_desc;
    u8 set_config;
    int res;
    
    /* ߤConfigֹ(ϡusb_gcore_setup.c 褦ˤ) */
    set_config = g_core->set_config;
    
    /* Configured StateDefault,Address Stateǽʬ */
    if(g_core->set_config != 0){
        /* Configured State */
        cfg_desc = get_config_desc_from_config_num(g_core, set_config);
        
        /* ConfigDescǤǧ */
        if(!cfg_desc){
            res = 0;
            goto exit;
        }
        
        /* OTG Desc뤫ǧ */
        if(!cfg_desc->otg){
            res = 0;
            goto exit;
        }
        
        /* OTG Descattributes򸫤ơHNPбƤ뤫ǧ */
        if((cfg_desc->otg->uc_attributes & USB_OTG_HNP) == 0){
            res = 0;
            goto exit;
        }
        res = 1;
        
    }else{
        /* Default,Address State */
        
        /* ߤspeedSpeedDesc */
        if(g_core->gadget->speed >= USB_SPEED_SUPER){
            sp_desc = &g_core->desc_tbl->ss;
        }else if(g_core->gadget->speed == USB_SPEED_HIGH){
            sp_desc = &g_core->desc_tbl->hs;
        }else{
            sp_desc = &g_core->desc_tbl->lsfs;
        }
        
        /* Speed DescOTGattributes򸫤ơHNPбƤ뤫ǧ */
        if((sp_desc->uc_otg_attributes & USB_OTG_HNP) == 0){
            res = 0;
            goto exit;
        }
        res = 1;
        
    }
    
exit:
    return res;
}

/*=============================================================================
 * Log function for base 2
 *===========================================================================*/
static int _log2(int var)
{
    int log = 0;

    if (var <= 0) return -1;

    while(1) {
        var /= 2;
        if (0 == var) break;
        log++;
    }
    return log;
}

/*=============================================================================
 * Descriptor Generate
 *===========================================================================*/
int make_device_desc(struct g_core_drv *g_core,
                            const struct usb_ctrlrequest *ctrl,
                            void *buf)
{
    int rc;
    struct usb_device_descriptor dev_desc;
    usb_gadget_speed_desc *sp_desc;
    usb_gadget_desc_table *desc_tbl;
    int psize = 0;
    
    desc_tbl = g_core->desc_tbl;
    
    if(g_core->gadget->speed >= USB_SPEED_SUPER){
        PDEBUG("=================== SS =================== \n");
        sp_desc = &desc_tbl->ss;
        psize = _log2(g_core->ep0->desc->wMaxPacketSize);
    }else if(g_core->gadget->speed == USB_SPEED_HIGH){
        PDEBUG("=================== HS =================== \n");
        sp_desc = &desc_tbl->hs;
    }else{
        PDEBUG("=================== FS =================== \n");
        sp_desc = &desc_tbl->lsfs;
    }
    
    dev_desc.bLength = sizeof dev_desc;
    dev_desc.bDescriptorType = USB_DT_DEVICE;
    
    dev_desc.bcdUSB = cpu_to_le16(sp_desc->us_bcd_usb);
    dev_desc.bDeviceClass = sp_desc->uc_class;
    dev_desc.bDeviceSubClass = sp_desc->uc_sub_class;
    dev_desc.bDeviceProtocol = sp_desc->uc_protocol;
    dev_desc.bMaxPacketSize0 = g_core->ep0->maxpacket;
    if(psize) dev_desc.bMaxPacketSize0 = psize;
    dev_desc.idVendor  = cpu_to_le16(desc_tbl->us_id_vendor);
    dev_desc.idProduct = cpu_to_le16(desc_tbl->us_id_product);
    dev_desc.bcdDevice = cpu_to_le16(desc_tbl->us_bcd_device);
    dev_desc.iManufacturer = desc_tbl->uc_manufacturer_str;
    dev_desc.iProduct = desc_tbl->uc_product_str;
    dev_desc.iSerialNumber = desc_tbl->uc_serial_number_str;
    dev_desc.bNumConfigurations = sp_desc->uc_num_config;
    
    rc = min(ctrl->wLength, (u16)sizeof(dev_desc));
    memcpy(buf, &dev_desc, rc);
    
    return rc;
}

int make_hid_desc(struct g_core_drv *g_core,
                            const struct usb_ctrlrequest *ctrl,
                            void *buf)
{
    
    int rc = 0;
    u8 i = 0;
    
    const u8 hid_class_num = 0x03;
    
    usb_gadget_desc_table *desc_tbl    = NULL;
    usb_gadget_speed_desc *sp_desc     = NULL;
    usb_gadget_config_desc *cfg_desc   = NULL;
    usb_gadget_interface_desc *in_desc = NULL;
    usb_gadget_if_table *if_tbl        = NULL;
    
    /* Get Descriptor Table */
    desc_tbl = g_core->desc_tbl;
    
    /* Get Speed Descriptor */
    if(g_core->gadget->speed >= USB_SPEED_SUPER){
        PDEBUG("=================== SS =================== \n");
        sp_desc = &desc_tbl->ss;
    }else if(g_core->gadget->speed == USB_SPEED_HIGH){
        
        PDEBUG("=================== HS =================== \n");
        sp_desc = &desc_tbl->hs;
        
    }else{
        
        PDEBUG("=================== FS =================== \n");
        sp_desc = &desc_tbl->lsfs;
        
    }
    if( sp_desc == NULL ) return -EINVAL;
    
    /* Find Current Config */
    for( i = 0; i < sp_desc->uc_num_config; i++ ){
        
        if( sp_desc->configurations[i].uc_configuration_value == g_core->set_config ){
            
            cfg_desc = &sp_desc->configurations[i];
            break;
            
        }
        
    }
    if( cfg_desc == NULL ) return -EINVAL; /* Could not find... -> NOT Configured State. */
    
    /* Find Desired Interface Descriptor */
    for( i = 0; i < cfg_desc->uc_num_interfaces; i++ ){
        
        if( cfg_desc->interfaces[i].uc_if_number == (u8)(ctrl->wIndex & 0x00ff) ){
            
            in_desc = &cfg_desc->interfaces[i];
            break;
            
        }
        
    }
    if( in_desc == NULL ) return -EINVAL; /* Could not find... */
    
    /* Find HID Descriptor */
    for( i = 0; i < in_desc->uc_num_if_tables; i++ ){
        
        if( in_desc->if_tables[i].uc_class == hid_class_num ){
            
            if_tbl = &in_desc->if_tables[i];
            break;
            
        }
        
    }
    if( if_tbl == NULL ) return -EINVAL; /* Could not find... */
    
    /* Copy HID Descriptor */
    if( if_tbl->class_specific_interface_desc_size != 0 ){
        
        memcpy( buf,
                if_tbl->class_specific_interface_desc,
                if_tbl->class_specific_interface_desc_size );
        
    }
    
    /* Calc Packet Size To Be Sent */
    rc = min( ctrl->wLength, (u16)if_tbl->class_specific_interface_desc_size );
    
    return rc;
    
}

int make_report_desc(struct g_core_drv *g_core,
                            const struct usb_ctrlrequest *ctrl,
                            void *buf)
{
    
    int ret = 0;
    usb_gadget_desc_table *pstDescTbl;
    
    if( g_core ){
        
        pstDescTbl = g_core->desc_tbl;
        
        if( pstDescTbl ){
            
            if( pstDescTbl->rptdesc.desc_data ){
                
                ret = min( ctrl->wLength, pstDescTbl->rptdesc.us_desc_size );
                memcpy( buf, pstDescTbl->rptdesc.desc_data, ret );
                
            }else{
                
                ret = -EINVAL;
                
            }
            
        }else{
            
            ret = -EINVAL;
            
        }
        
    }else{
        
        ret = -EINVAL;
        
    }
    
    return ret;
    
}

int make_config_desc(struct g_core_drv *g_core,
                            const struct usb_ctrlrequest *ctrl,
                            void *buf,
                            u8 desc_type)
{
    int rc;
    void *buf_org;
    usb_gadget_speed_desc *sp_desc;
    usb_gadget_desc_table *desc_tbl;
    usb_gadget_config_desc *cfg_desc;
    usb_gadget_interface_desc *in_desc;
    usb_gadget_if_table *if_tbl;
    usb_gadget_if_table_element *pipe;
    
    struct usb_config_descriptor _cfg_desc;
    struct usb_otg_descriptor _otg_desc;
    struct usb_interface_assoc_descriptor _ia_desc;
    struct usb_interface_descriptor _in_desc;
    struct usb_endpoint_descriptor _ep_desc;
    struct usb_ss_ep_comp_descriptor _comp_desc;
    
    u8 desc_index;
    u8 i, d, s, ite;
    
    rc = 0;
    buf_org = buf;
    desc_index = (u8)(ctrl->wValue & 0x00ff);
    
    /* Descriptor Table  */
    desc_tbl = g_core->desc_tbl;
    
    /* Speed Descriptor  */
    sp_desc = NULL;
    if(g_core->gadget->speed >= USB_SPEED_SUPER){
        PDEBUG("=================== SS =================== \n");
        if(desc_type == USB_DT_CONFIG){
            sp_desc = &desc_tbl->ss;
        }else if(desc_type == USB_DT_OTHER_SPEED_CONFIG){
            sp_desc = &desc_tbl->hs;
        }
    }else if(g_core->gadget->speed == USB_SPEED_HIGH){
        PDEBUG("=================== HS =================== \n");
        if(desc_type == USB_DT_CONFIG){
            sp_desc = &desc_tbl->hs;
        }else if(desc_type == USB_DT_OTHER_SPEED_CONFIG){
            sp_desc = &desc_tbl->lsfs;
        }
    }else{
        PDEBUG("=================== FS =================== \n");
        if(desc_type == USB_DT_CONFIG){
            sp_desc = &desc_tbl->lsfs;
        }else if(desc_type == USB_DT_OTHER_SPEED_CONFIG){
            sp_desc = &desc_tbl->hs;
        }
    }
    if(sp_desc == NULL) return -EINVAL;
    
    /* indexConfigDescriptorõ */
    cfg_desc = NULL;
    for(i = 0; i < sp_desc->uc_num_config; i++){
        if(sp_desc->configurations[i].uc_index == desc_index){
            cfg_desc = &sp_desc->configurations[i];
        }
    }
    if(cfg_desc == NULL) return -1; /* Ĥʤ */
    
    /* Config Descriptor ʬʤ */
    rc += (u16)USB_DT_CONFIG_SIZE;
    buf += (u16)USB_DT_CONFIG_SIZE;
    
    /* OTG Descriptor */
    if(cfg_desc->otg){
        _otg_desc.bLength = sizeof(_otg_desc);
        _otg_desc.bDescriptorType = USB_DT_OTG;
        
        _otg_desc.bmAttributes = cfg_desc->otg->uc_attributes;
        
        memcpy(buf, &_otg_desc, _otg_desc.bLength);
        rc += (u16)_otg_desc.bLength;
        buf += (u16)_otg_desc.bLength;
    }
    
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        in_desc = &cfg_desc->interfaces[i];
        
        /* make Interface Association Descriptor */
        if( cfg_desc->uc_num_iads > 0x00 ){
            for( ite = 0; ite < cfg_desc->uc_num_iads ; ite++ ){
                if( in_desc->uc_if_number == cfg_desc->iads[ite].uc_firstinterface ){
                    
                    PDEBUG("NEED IAD! for Interface ID : %d.\n", in_desc->uc_if_number );
                    
                    _ia_desc.bLength = sizeof( struct usb_interface_assoc_descriptor );
                    _ia_desc.bDescriptorType = USB_DT_INTERFACE_ASSOCIATION;
                    
                    _ia_desc.bFirstInterface   = cfg_desc->iads[ite].uc_firstinterface;
                    _ia_desc.bInterfaceCount   = cfg_desc->iads[ite].uc_interface_count;
                    _ia_desc.bFunctionClass    = cfg_desc->iads[ite].uc_function_class;
                    _ia_desc.bFunctionSubClass = cfg_desc->iads[ite].uc_function_sub_class;
                    _ia_desc.bFunctionProtocol = cfg_desc->iads[ite].uc_function_protocol;
                    _ia_desc.iFunction         = cfg_desc->iads[ite].uc_function;
                    
                    memcpy( buf, &_ia_desc, _ia_desc.bLength );
                    rc += (u16)_ia_desc.bLength;
                    buf += (u16)_ia_desc.bLength;
                }
            }
        }
        
        for(d = 0; d < in_desc->uc_num_if_tables; d++){
            if_tbl = &in_desc->if_tables[d];
            
            /* Interface Desc */
            _in_desc.bLength = USB_DT_INTERFACE_SIZE;
            _in_desc.bDescriptorType = USB_DT_INTERFACE;
            
            _in_desc.bInterfaceNumber = in_desc->uc_if_number;
            _in_desc.bAlternateSetting = if_tbl->uc_setting_number;
            _in_desc.bNumEndpoints = if_tbl->uc_num_pep_list;
            _in_desc.bInterfaceClass = if_tbl->uc_class;
            _in_desc.bInterfaceSubClass = if_tbl->uc_sub_class;
            _in_desc.bInterfaceProtocol = if_tbl->uc_interface_protocol;
            _in_desc.iInterface = if_tbl->uc_interface_str;
            
            memcpy(buf, &_in_desc, _in_desc.bLength);
            rc += (u16)_in_desc.bLength;
            buf += (u16)_in_desc.bLength;
            
            /* Class Specific Interface Descriptor Υԡ */
            if(if_tbl->class_specific_interface_desc_size != 0){
                memcpy(buf,
                       if_tbl->class_specific_interface_desc,
                       if_tbl->class_specific_interface_desc_size);
                rc += (u16)if_tbl->class_specific_interface_desc_size;
                buf += (u16)if_tbl->class_specific_interface_desc_size;
            }
            
            for(s = 0; s < if_tbl->uc_num_pep_list; s++){
                pipe = &if_tbl->pep_list[s];
                
                /* EP Desc  */
                _ep_desc.bLength = USB_DT_ENDPOINT_SIZE;
                _ep_desc.bDescriptorType = USB_DT_ENDPOINT;
                
                _ep_desc.bEndpointAddress = pipe->uc_ep_address;
                _ep_desc.bmAttributes = pipe->uc_attributes;
                _ep_desc.wMaxPacketSize = cpu_to_le16(pipe->us_max_psize);
                _ep_desc.bInterval = pipe->uc_interval;
                
                memcpy(buf, &_ep_desc, _ep_desc.bLength);
                rc += (u16)_ep_desc.bLength;
                buf += (u16)_ep_desc.bLength;
                
                /* EP Companion Descriptor Υԡ */
                if(g_core->gadget->speed >= USB_SPEED_SUPER && 0 != pipe->ep_companion_desc){
                    _comp_desc.bLength = USB_DT_SS_EP_COMP_SIZE;
                    _comp_desc.bDescriptorType = USB_DT_SS_ENDPOINT_COMP;
                    _comp_desc.bMaxBurst = pipe->ep_companion_desc->uc_max_burst;
                    _comp_desc.bmAttributes = pipe->ep_companion_desc->uc_attributes;
                    _comp_desc.wBytesPerInterval = pipe->ep_companion_desc->us_bytes_per_interval;
                    memcpy(buf, &_comp_desc, _comp_desc.bLength);
                    rc += (u16)_comp_desc.bLength;
                    buf += (u16)_comp_desc.bLength;
                }

                /* Class Specific EP Descriptor Υԡ */
                if(pipe->class_specific_ep_desc_size != 0){
                    memcpy(buf,
                           pipe->class_specific_ep_desc,
                           pipe->class_specific_ep_desc_size);
                    rc += (u16)pipe->class_specific_ep_desc_size;
                    buf += (u16)pipe->class_specific_ep_desc_size;
                }
            }
        }
    }
    
    /* ConfigDescriptor */
    _cfg_desc.bLength = USB_DT_CONFIG_SIZE;
    _cfg_desc.bDescriptorType = desc_type;
    
    _cfg_desc.wTotalLength = rc;
    _cfg_desc.bNumInterfaces = cfg_desc->uc_num_interfaces;
    _cfg_desc.bConfigurationValue = cfg_desc->uc_configuration_value;
    _cfg_desc.iConfiguration = cfg_desc->uc_configuration_str;
    _cfg_desc.bmAttributes = cfg_desc->uc_attributes;
    _cfg_desc.bMaxPower = cfg_desc->uc_max_power;
    
    memcpy(buf_org, &_cfg_desc, _cfg_desc.bLength);
    
    /* 륵 */
    rc = min(ctrl->wLength, (u16)rc);
    
    return rc;
}

int make_device_qualifier_desc(struct g_core_drv *g_core,
                                      const struct usb_ctrlrequest *ctrl,
                                      void *buf)
{
    usb_gadget_desc_table *desc_tbl;
    usb_gadget_speed_desc *sp_desc;
    struct usb_qualifier_descriptor _qualifier_desc;
    int rc;
    int psize = 0;
    
    /* Descriptor Table  */
    desc_tbl = g_core->desc_tbl;
    
    /* Speed Descriptor  */
    if(g_core->gadget->speed >= USB_SPEED_SUPER){
        PDEBUG("=================== SS =================== \n");
        sp_desc = &desc_tbl->ss;
        psize = _log2(g_core->ep0->desc->wMaxPacketSize);
    }else if(g_core->gadget->speed == USB_SPEED_HIGH){
        PDEBUG("=================== HS =================== \n");
        sp_desc = &desc_tbl->lsfs;
    }else{
        PDEBUG("=================== FS =================== \n");
        sp_desc = &desc_tbl->hs;
    }
    
    _qualifier_desc.bLength = sizeof(_qualifier_desc);
    _qualifier_desc.bDescriptorType = USB_DT_DEVICE_QUALIFIER;
    
    _qualifier_desc.bcdUSB = cpu_to_le16(sp_desc->us_bcd_usb);
    _qualifier_desc.bDeviceClass = sp_desc->uc_class; 
    _qualifier_desc.bDeviceSubClass = sp_desc->uc_sub_class;
    _qualifier_desc.bDeviceProtocol = sp_desc->uc_protocol;
    _qualifier_desc.bMaxPacketSize0 = g_core->ep0->maxpacket;
    if(psize) _qualifier_desc.bMaxPacketSize0 = psize;
    _qualifier_desc.bNumConfigurations = sp_desc->uc_num_config;
    _qualifier_desc.bRESERVED = 0;
    
    rc = min(ctrl->wLength, (u16)sizeof(_qualifier_desc));
    memcpy(buf, &_qualifier_desc, rc);
    
    return rc;
}

int make_string_desc(struct g_core_drv *g_core,
                            const struct usb_ctrlrequest *ctrl,
                            void *buf)
{
    usb_gadget_desc_table *desc_tbl;
    usb_gadget_langid_table *langid_tbl;
    usb_gadget_string_desc *str_desc;
    u8 *pbuf, *org_pbuf;
    u8 desc_index;
    u8 i, len;
    
    /* Descriptor Table  */
    desc_tbl = g_core->desc_tbl;
    
    desc_index = (u8)(ctrl->wValue & 0x00ff);
    org_pbuf = pbuf = (u8*)buf;
    
    PDEBUG("desc_index : %d\n", desc_index);
    if(desc_index == 0){
        /* descriptor 0 has the language id */
        pbuf++;
        *pbuf++ = USB_DT_STRING;
        for(i = 0; i < desc_tbl->uc_num_langids; i++){
            langid_tbl = &desc_tbl->langids[i];
            PDEBUG("langid[%d] : 0x%04x\n", i, (int)langid_tbl->us_langid);
            *pbuf++ = (u8)langid_tbl->us_langid;
            *pbuf++ = (u8)(langid_tbl->us_langid >> 8);
        }
        *org_pbuf = (u8)(pbuf - org_pbuf);  /* Total Length*/
        len = (int)*org_pbuf;
    }
    else if(desc_index == USB_REQ_OS_STRING_DESC_WVALUE_L){
        DEBUG_PRINTK("\x1b[7;35mRecved 0x%x!!\x1b[0m", desc_index);
        /* descriptor 0xEE has OS_STRING_DESC */
        if( g_core->enable_dual_mode != 0 ){
            PDEBUG("now in GET STRING_DESC(0x%x)!!", desc_index );
            len = sizeof(g_os_desc);
            memcpy(pbuf, &g_os_desc, len);
            DEBUG_DUMP(pbuf, len);
        }
        else{
            PDEBUG("now in GET STRING_DESC(0x%x)!!, but dual_mode is diabled!!!!", desc_index );
            return -EINVAL;
        }
    }
    else{
        PDEBUG("langid : 0x%04x\n", ctrl->wIndex);
        
        /* ꤵ줿langidLangidTableõ */
        langid_tbl = NULL;
        for(i = 0; i < desc_tbl->uc_num_langids; i++){
            if(desc_tbl->langids[i].us_langid == ctrl->wIndex){
                PDEBUG("langid_tbl found!!\n");
                langid_tbl = &desc_tbl->langids[i];
                break;
            }
        }
        if(langid_tbl == NULL) return -EINVAL;
        
        /* ꤵ줿indexͤStringDescriptorõ */
        str_desc = NULL;
        for(i = 0; i < langid_tbl->uc_num_strings; i++){
            if(langid_tbl->strings[i].uc_index == desc_index){
                PDEBUG("string_desc found!!\n");
                str_desc = &langid_tbl->strings[i];
                break;
            }
        }
        if(str_desc == NULL) return -EINVAL;
        
        /* StringDescriptor */
        len = min((__u8)253, str_desc->uc_len);
        if(len != 0){
            memcpy(pbuf + 2, str_desc->string, len);
        }
        len += 2;
        pbuf[0] = len;
        pbuf[1] = USB_DT_STRING;
    }
    
    len = min(ctrl->wLength, (u16)len);
    return len;
}

int make_bos_desc(struct g_core_drv *g_core, const struct usb_ctrlrequest *ctrl, void *buf)
{
    int rc = 0;
    usb_gadget_desc_table *desc_tbl;
    usb_gadget_cap_ext *cap_extp;
    usb_gadget_cap_ss_cap_type *cap_typep;
    usb_gadget_cap_ssp_cap_type *sspcap_typep;
    usb_gadget_cap_container_id *containerp;
    struct usb_bos_descriptor _bos_desc;
    struct usb_ext_cap_descriptor _ext_cap;
    struct usb_ss_cap_descriptor _ss_cap;
    struct usb_ssp_cap_descriptor *_ssp_cap;  /* dynamically */
    struct usb_ss_container_id_descriptor _container_cap;
    
    /* Descriptor Table  */
    desc_tbl = g_core->desc_tbl;
    
    _bos_desc.bLength = USB_DT_BOS_SIZE;
    _bos_desc.bDescriptorType = USB_DT_BOS;
    _bos_desc.wTotalLength = desc_tbl->bos.us_total_length;
    _bos_desc.bNumDeviceCaps = desc_tbl->bos.uc_num_dev_caps;
    
    rc = _bos_desc.bLength;
    memcpy(buf, &_bos_desc, rc);
    buf += rc;

    /* USB2.0 Extension Descriptor  */
    cap_extp = desc_tbl->bos.devcap_ext;
    if (cap_extp) {
        _ext_cap.bLength = USB_DT_USB_EXT_CAP_SIZE;
        _ext_cap.bDescriptorType = USB_DT_DEVICE_CAPABILITY;
        _ext_cap.bDevCapabilityType = USB_CAP_TYPE_EXT;
        _ext_cap.bmAttributes = cap_extp->ul_attributes;
        memcpy(buf, &_ext_cap, _ext_cap.bLength);
        rc += _ext_cap.bLength;
        buf += _ext_cap.bLength;
    }

    /* SS USB Capability Descriptor  */
    cap_typep = desc_tbl->bos.ss_cap_type;
    if (cap_typep) {
        _ss_cap.bLength = USB_DT_USB_SS_CAP_SIZE;
        _ss_cap.bDescriptorType = USB_DT_DEVICE_CAPABILITY;
        _ss_cap.bDevCapabilityType = USB_SS_CAP_TYPE;
        _ss_cap.bmAttributes = cap_typep->uc_attributes;
        _ss_cap.wSpeedSupported = cap_typep->us_speeds_supported;
        _ss_cap.bFunctionalitySupport = cap_typep->uc_functionality_support;
        _ss_cap.bU1devExitLat = cap_typep->uc_U1_dev_exit_lat;
        _ss_cap.bU2DevExitLat = cap_typep->us_U2_dev_exit_lat;
        memcpy(buf, &_ss_cap, _ss_cap.bLength);
        rc += _ss_cap.bLength;
        buf += _ss_cap.bLength;
    }

    /* SSP USB Capability Descriptor  */
    sspcap_typep = desc_tbl->bos.ssp_cap_type;
    if (sspcap_typep) {
        int i, ssac;
        ssac = (sspcap_typep->ul_attributes & USB_SSP_SUBLINK_SPEED_ATTRIBS);
        i = USB_DT_USB_SSP_CAP_SIZE(ssac);
        _ssp_cap = kmalloc(i, GFP_KERNEL);
        if (!_ssp_cap) {
            PERR("%s() kmalloc fail\n", __func__);
            return -ENOMEM;
        }
        _ssp_cap->bLength = (__u8)i;
        _ssp_cap->bDescriptorType = USB_DT_DEVICE_CAPABILITY;
        _ssp_cap->bDevCapabilityType = USB_SSP_CAP_TYPE;
        _ssp_cap->bReserved = sspcap_typep->uc_reserved;
        _ssp_cap->bmAttributes = sspcap_typep->ul_attributes;
        _ssp_cap->wFunctionalitySupport = sspcap_typep->us_functionality_support;
        _ssp_cap->wReserved = sspcap_typep->us_reserved;
        for (i=0; i<ssac+1; i++)
            _ssp_cap->bmSublinkSpeedAttr[i] = sspcap_typep->ul_sublink_speed_attr[i];
        memcpy(buf, _ssp_cap, _ssp_cap->bLength);
        rc += _ssp_cap->bLength;
        buf += _ssp_cap->bLength;
        kfree(_ssp_cap);
    }

    /* Container ID Capability Descriptor  */
    containerp = desc_tbl->bos.container_id;
    if (containerp) {
        _container_cap.bLength = USB_DT_USB_SS_CONTN_ID_SIZE;
        _container_cap.bDescriptorType = USB_DT_DEVICE_CAPABILITY;
        _container_cap.bDevCapabilityType = CONTAINER_ID_TYPE;
        _container_cap.bReserved = 0;
        memcpy(_container_cap.ContainerID, containerp->uc_uuid, sizeof(usb_gadget_cap_container_id));
        memcpy(buf, &_container_cap, _container_cap.bLength);
        rc += _container_cap.bLength;
        buf += _container_cap.bLength;
    }
    
    rc = min(ctrl->wLength, (u16)rc);

    return rc;
}

/*=============================================================================
 * Descriptor Save
 *===========================================================================*/
/* User֤DescTbl¸ */
int save_desc_tbl(struct g_core_drv *g_core, struct usb_gadgetcore_start_info *_start)
{
    int rs = 0;
    struct usb_gadgetcore_start_info start;
    
    /* usb_gadgetcore_start_info¤Τ򥳥ԡ */
    if(copy_from_user(&start, _start, sizeof(start))){
        rs = -EFAULT;
        goto exit;
    }
    
    /* usb_gadget_desc_table Ѥΰ */
    g_core->desc_tbl = kmalloc(sizeof(usb_gadget_desc_table), GFP_KERNEL);
    if(!g_core->desc_tbl){
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, g_core->desc_tbl);
    
    /* usb_gadget_desc_table ΰإԡ */
    if(copy_from_user(g_core->desc_tbl, start.desc_tbl, sizeof(usb_gadget_desc_table))){
        rs = -EFAULT;
        goto exit;
    }
    
    /* HS Config Descriptor ΰݤƥԡ */
    rs = save_config_desc(&g_core->desc_tbl->hs);
    if(rs != 0){
        goto exit;
    }
    
    /* LSFS Config Descriptor ΰݤƥԡ */
    rs = save_config_desc(&g_core->desc_tbl->lsfs);
    if(rs != 0){
        goto exit;
    }
    
    /* SS Config Descriptor ΰݤƥԡ */
    rs = save_config_desc(&g_core->desc_tbl->ss);
    if(rs != 0){
        goto exit;
    }
    
    /* Report Descriptor ΰݤƥԡ */
    rs = save_report_desc( &g_core->desc_tbl->rptdesc );
    if( rs != 0 ){
        goto exit;
    }
    
    /* Langid Table ΰݤƥԡ */
    rs = save_langids(g_core->desc_tbl);
    if(rs != 0){
        goto exit;
    }
    
    /* BOS Descriptor ΰݤƥԡ */
    rs = save_bos_desc(&g_core->desc_tbl->bos);
    if(rs != 0){
        goto exit;
    }
    
    rs = overwrite_iserial(g_core->desc_tbl, start.uc_serial_len, start.serial_string);
    if(rs != 0){
        goto exit;
    }

    // Added for DualMode.
    // Get Params from userspace
    g_core->enable_dual_mode = start.ext_info.enable_dual_mode;
    g_core->target_ep_adr    = start.ext_info.target_ep_adr;
    DEBUG_PRINT("### DualMode flag is   ... [%s]", g_core->enable_dual_mode==1 ? "TRUE":"FALSE"  );
    DEBUG_PRINT("### Target EP_ADR is   ... [%d]", g_core->target_ep_adr);

exit:
    if(rs != 0){
        PERR("%s()\n", __func__);
        
        /*  */
        free_desc_tbl(g_core);
    }
    
    return rs;
}

/* BOS Descriptor ¸ */
static int save_bos_desc(usb_gadget_bos_desc* bosp)
{
    usb_gadget_cap_ext *ext_cap;
    usb_gadget_cap_ss_cap_type *ss_cap;
    usb_gadget_cap_ssp_cap_type *ssp_cap;
    usb_gadget_cap_container_id *container_cap;
    int rs = 0;

    if (bosp->devcap_ext) {
        /* USB2.0 Ext Descriptor ΰ */
        ext_cap = kmalloc(sizeof(*ext_cap), GFP_KERNEL);
        if(!ext_cap){
            PDEBUG("kmalloc error\n");
            bosp->devcap_ext = NULL;
            rs = -ENOMEM;
            goto exit;
        }
        PDEBUG("%s() : kmalloc %p\n", __func__, ext_cap);
        
        /* USB2.0 Ext Descriptor ΰΰإԡ */
        if(copy_from_user(ext_cap,
                          bosp->devcap_ext,
                          sizeof(*(bosp->devcap_ext)))){
            kfree(ext_cap);
            bosp->devcap_ext = NULL;
            PDEBUG("%s() : copy_from_user error\n", __func__);
            rs = -EFAULT;
            goto exit;
        }
        
        /* ݤΰդؤ */
        bosp->devcap_ext = ext_cap;
    }

    if (bosp->ss_cap_type) {
        /* SS Cap Descriptor ΰ */
        ss_cap = kmalloc(sizeof(*ss_cap), GFP_KERNEL);
        if(!ss_cap){
            PDEBUG("kmalloc error\n");
            bosp->ss_cap_type = NULL;
            rs = -ENOMEM;
            goto exit;
        }
        PDEBUG("%s() : kmalloc %p\n", __func__, ss_cap);
        
        /* SS Cap Descriptor ΰΰإԡ */
        if(copy_from_user(ss_cap,
                          bosp->ss_cap_type,
                          sizeof(*(bosp->ss_cap_type)))){
            kfree(ss_cap);
            bosp->ss_cap_type = NULL;
            PDEBUG("%s() : copy_from_user error\n", __func__);
            rs = -EFAULT;
            goto exit;
        }
        
        /* ݤΰդؤ */
        bosp->ss_cap_type = ss_cap;
    }

    if (bosp->ssp_cap_type) {
        /* SSP Cap Descriptor ΰ */
        ssp_cap = kmalloc(sizeof(*ssp_cap), GFP_KERNEL);
        if(!ssp_cap){
            PERR("kmalloc error\n");
            bosp->ssp_cap_type = NULL;
            rs = -ENOMEM;
            goto exit;
        }
        PDEBUG("%s() : kmalloc %p\n", __func__, ssp_cap);
        
        /* SS Cap Descriptor ΰΰإԡ */
        if(copy_from_user(ssp_cap,
                          bosp->ssp_cap_type,
                          sizeof(*(bosp->ssp_cap_type)))){
            kfree(ssp_cap);
            bosp->ssp_cap_type = NULL;
            PERR("%s() : copy_from_user sspcap error\n", __func__);
            rs = -EFAULT;
            goto exit;
        }
        
        /* ݤΰդؤ */
        bosp->ssp_cap_type = ssp_cap;
    }

    if (bosp->container_id) {
        /* Container ID Descriptor ΰ */
        container_cap = kmalloc(sizeof(*container_cap), GFP_KERNEL);
        if(!container_cap){
            PDEBUG("kmalloc error\n");
            bosp->container_id = NULL;
            rs = -ENOMEM;
            goto exit;
        }
        PDEBUG("%s() : kmalloc %p\n", __func__, container_cap);
        
        /* Langid Table ΰΰإԡ */
        if(copy_from_user(container_cap,
                          bosp->container_id,
                          sizeof(*(bosp->container_id)))){
            kfree(container_cap);
            bosp->container_id = NULL;
            PDEBUG("%s() : copy_from_user error\n", __func__);
            rs = -EFAULT;
            goto exit;
        }
        
        /* ݤΰդؤ */
        bosp->container_id = container_cap;
    }

exit:
    return rs;
}

/* Langid Table ¸ */
static int save_langids(usb_gadget_desc_table *desc_tbl)
{
    usb_gadget_langid_table *langids;
    int rs = 0;
    __u8 i;
    
    /* Langid Table ο0λϽ */
    if(desc_tbl->uc_num_langids == 0){
        goto exit;
    }
    
    /* Langid Table ΰ */
    langids = kmalloc(sizeof(usb_gadget_langid_table) * desc_tbl->uc_num_langids,
                      GFP_KERNEL);
    if(!langids){
        PDEBUG("kmalloc error\n");
        desc_tbl->langids = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, langids);
    
    /* Langid Table ΰΰإԡ */
    if(copy_from_user(langids,
                      desc_tbl->langids,
                      sizeof(usb_gadget_langid_table) * desc_tbl->uc_num_langids)){
        kfree(langids);
        desc_tbl->langids = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    desc_tbl->langids = langids;
    
    for(i = 0; i < desc_tbl->uc_num_langids; i++){
        /* String Descriptor ΰݤƥԡ */
        rs = save_string_desc(&desc_tbl->langids[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

/* String Descriptor ¸ */
static int save_string_desc(usb_gadget_langid_table *langids)
{
    usb_gadget_string_desc *str_desc;
    int rs = 0;
    __u8 i;
    
    /* String Descriptor ο0λϽ */
    if(langids->uc_num_strings == 0){
        goto exit;
    }
    
    /* String Descriptor ΰ */
    str_desc = kmalloc(sizeof(usb_gadget_string_desc) * langids->uc_num_strings,
                       GFP_KERNEL);
    if(!str_desc){
        PDEBUG("kmalloc error\n");
        langids->strings = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, str_desc);
    
    /* String Descriptor ΰΰإԡ */
    if(copy_from_user(str_desc,
                      langids->strings,
                      sizeof(usb_gadget_string_desc) * langids->uc_num_strings)){
        kfree(str_desc);
        langids->strings = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    langids->strings = str_desc;
    
    /* class_specific_ep_desc ΰ򥳥ԡ */
    for(i = 0; i < langids->uc_num_strings; i++){
        rs = save_strings(&langids->strings[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

/* String ΰ¸ */
static int save_strings(usb_gadget_string_desc *str_desc)
{
    void *tmp;
    int rs = 0;
    
    /* Strings Υ0λϽ */
    /*  DescriptorȤΤ */
    if(str_desc->uc_len == 0){
        PERR("String Descriptor's String Size is 0\n");
        goto exit;
    }
    
    /* Strings ΰ */
    tmp = kmalloc(str_desc->uc_len, GFP_KERNEL);
    if(!tmp){
        PDEBUG("kmalloc error\n");
        str_desc->string = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
    
    /* Strings ΰ򥳥ԡ */
    if(copy_from_user(tmp,
                      str_desc->string,
                      str_desc->uc_len)){
        kfree(tmp);
        str_desc->string = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    str_desc->string = tmp;
    
exit:
    return rs;
}

static int overwrite_iserial(usb_gadget_desc_table *desc_tbl,
                             __u8 len,
                             void* string)
{
    usb_gadget_langid_table *langid_tbl;
    usb_gadget_string_desc *str_desc;
    void *tmp;
    __u8 i, j, k, flag;
    int rs = 0;
    
    if(string == NULL){
        /* stringNULLʤｪλ */
        goto exit;
    }
    
    PDEBUG("start: Overwrite iSerial()\n");
    
    if(desc_tbl->uc_serial_number_str == 0){
        i = 1;
        while(1){
            flag = 0;
            for(j = 0; j < desc_tbl->uc_num_langids; j++){
                langid_tbl = &desc_tbl->langids[j];
                for(k = 0; k < langid_tbl->uc_num_strings; k++){
                    if(i == langid_tbl->strings[k].uc_index){
                        flag = 1;
                    }
                }
            }
            if(flag == 0){
                /* ȤäƤʤindexĤä */
                desc_tbl->uc_serial_number_str = i;
                break;
            }
            if(i == 255){
                /* ȤäƤʤindexĤʤä۾ｪλ */
                rs = -EFAULT;
                goto exit;
            }
            i++;
        }
    }
    
    for(i = 0; i < desc_tbl->uc_num_langids; i++){
        langid_tbl = &desc_tbl->langids[i];
        
        for(j = 0; j < langid_tbl->uc_num_strings; j++){
            if(desc_tbl->uc_serial_number_str ==
               langid_tbl->strings[j].uc_index){
                break;
            }
        }
        
        if(j != langid_tbl->uc_num_strings){
            if(langid_tbl->strings[j].string){
                PDEBUG("%s() : kfree %p\n",
                       __func__, langid_tbl->strings[j].string);
                kfree(langid_tbl->strings[j].string);
                langid_tbl->strings[j].string = NULL;
            }
            
            if(len != 0){
                /* String ΰ */
                tmp = kmalloc(len, GFP_KERNEL);
                if(!tmp){
                    PDEBUG("kmalloc error\n");
                    rs = -ENOMEM;
                    goto exit;
                }
                PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
                
                /* Strings ΰ򥳥ԡ */
                if(copy_from_user(tmp, string, len)){
                    kfree(tmp);
                    PDEBUG("copy_from_user error\n");
                    rs = -EFAULT;
                    goto exit;
                }
                
                /* ݤΰդؤ */
                langid_tbl->strings[j].string = tmp;
            }
            langid_tbl->strings[j].uc_len = len;
            
        }else{
            if(langid_tbl->uc_num_strings == 255){
                /* langidStringsDescriptorο255ä۾ｪλ */
                rs = -EFAULT;
                goto exit;
            }
            
            /* String Descriptor ΰ */
            str_desc = kmalloc(sizeof(usb_gadget_string_desc) *
                               (langid_tbl->uc_num_strings + 1),
                                    GFP_KERNEL);
            if(!str_desc){
                PDEBUG("kmalloc error\n");
                rs = -ENOMEM;
                goto exit;
            }
            PDEBUG("%s() : kmalloc %p\n", __func__, str_desc);
            
            /* StringsDescriptorơ֥򥳥ԡ */
            memcpy(str_desc, langid_tbl->strings,
                   sizeof(usb_gadget_string_desc) * langid_tbl->uc_num_strings);
            PDEBUG("%s() : kfree %p\n", __func__, langid_tbl->strings);
            kfree(langid_tbl->strings);
            
            /* StringsDescriptor ο򥤥󥯥 */
            langid_tbl->uc_num_strings++;
            
            /* ݤΰդؤ */
            langid_tbl->strings = str_desc;
            
            str_desc = &langid_tbl->strings[langid_tbl->uc_num_strings-1];
            
            if(len != 0){
                /* String ΰ */
                tmp = kmalloc(len, GFP_KERNEL);
                if(!tmp){
                    PDEBUG("kmalloc error\n");
                    rs = -ENOMEM;
                    goto exit;
                }
                PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
                
                /* Strings ΰ򥳥ԡ */
                if(copy_from_user(tmp, string, len)){
                    kfree(tmp);
                    PDEBUG("copy_from_user error\n");
                    rs = -EFAULT;
                    goto exit;
                }
                
                /* ݤΰդؤ */
                str_desc->string = tmp;
            }
            str_desc->uc_index = desc_tbl->uc_serial_number_str;
            str_desc->uc_len = len;
        }
    }
    
    
exit:
    return rs;
}

/* Config Descriptor¸ */
static int save_config_desc(usb_gadget_speed_desc *sp_desc)
{
    usb_gadget_config_desc *config_desc;
    int rs = 0;
    __u8 i;
    
    /* Config Descriptor ο0λϽ */
    if(sp_desc->uc_num_config == 0){
        goto exit;
    }
    
    /* Config Descriptor ΰ */
    config_desc = kmalloc(sizeof(usb_gadget_config_desc) * sp_desc->uc_num_config,
                          GFP_KERNEL);
    if(!config_desc){
        PDEBUG("kmalloc error\n");
        sp_desc->configurations = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, config_desc);
    
    /* Config Descriptor ΰΰإԡ */
    if(copy_from_user(config_desc,
                      sp_desc->configurations,
                      sizeof(usb_gadget_config_desc) * sp_desc->uc_num_config)){
        kfree(config_desc);
        sp_desc->configurations = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs= -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    sp_desc->configurations = config_desc;
    
    for(i = 0; i < sp_desc->uc_num_config; i++){
        /* Interface Descriptor ΰݤƥԡ */
        rs = save_interface_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
        
        /* OTG Descriptor ΰݤƥԡ */
        rs = save_otg_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
        
        /* IA Descriptor ΰݤƥԡ */
        rs = save_ia_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

/* Report Descriptor¸ */
static int save_report_desc( usb_gadget_report_desc *rpt_desc )
{
    
    char *pRptDescData;
    int ret = 0;
    
    /* Success if report descriptor does not exist. */
    if( ( rpt_desc->us_desc_size == 0 ) || ( rpt_desc->desc_data == NULL ) ){
        goto exit;
    }
    
    /* kmalloc for Report Descriptor */
    pRptDescData = kmalloc( rpt_desc->us_desc_size, GFP_KERNEL );
    
    if( !pRptDescData ){
        
        PDEBUG( "kmalloc error. \n" );
        rpt_desc->desc_data = NULL;
        ret = -ENOMEM;
        goto exit;
        
    }
    
    PDEBUG( "%s() : kmalloc %p\n", __func__, pRptDescData );
    
    /* Copy Report Descriptor data to internal region. */
    if( copy_from_user( pRptDescData, rpt_desc->desc_data, rpt_desc->us_desc_size ) ){
        
        kfree( pRptDescData );
        rpt_desc->desc_data = NULL;
        PDEBUG( "%s() : copy_from_user error. \n", __func__ );
        ret= -EFAULT;
        goto exit;
        
    }
    
    /* attach with malloc'ed region. */
    rpt_desc->desc_data = pRptDescData;
    
exit:
    return ret;
    
}

/* OTG Descriptor¸ */
static int save_otg_desc(usb_gadget_config_desc *cfg_desc)
{
    usb_gadget_otg_desc *otg_desc;
    int rs = 0;
    
    /* OTG Descriptor ̵(NULLˤʤäƤ)ʤн */
    if(cfg_desc->otg == NULL){
        goto exit;
    }
    
    /* OTG Descriptor ΰ */
    otg_desc = kmalloc(sizeof(usb_gadget_otg_desc), GFP_KERNEL);
    if(!otg_desc){
        PDEBUG("kmalloc error\n");
        /* ԤȤ򼨤cfg_descƤ */
        /* NULLǤϡOTG DescriptorʤΤԤʤΤʬĤʤ */
        cfg_desc->otg = (usb_gadget_otg_desc*)cfg_desc;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, otg_desc);
    
    /* OTG Descriptor ΰΰإԡ */
    if(copy_from_user(otg_desc,
                      cfg_desc->otg,
                      sizeof(usb_gadget_otg_desc))){
        kfree(otg_desc);
        /* ԤȤ򼨤cfg_descƤ */
        /* NULLǤϡOTG DescriptorʤΤԤʤΤʬĤʤ */
        cfg_desc->otg = (usb_gadget_otg_desc*)cfg_desc;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    cfg_desc->otg = otg_desc;
    
exit:
    return rs;
}

/* Interface Association Descriptor¸ */
static int save_ia_desc(usb_gadget_config_desc *cfg_desc)
{
    usb_gadget_ia_desc *ia_desc;
    int rs = 0;
    
    /* IA Descriptor ο0λϽ */
    if(cfg_desc->uc_num_iads == 0){
        goto exit;
    }
    
    /* IA Descriptor ΰ */
    ia_desc = kmalloc(sizeof(usb_gadget_ia_desc) * cfg_desc->uc_num_iads,
                      GFP_KERNEL);
    if(!ia_desc){
        PDEBUG("kmalloc error\n");
        cfg_desc->iads = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, ia_desc);
    
    /* IA Descriptor ΰΰإԡ */
    if(copy_from_user(ia_desc,
                      cfg_desc->iads,
                      sizeof(usb_gadget_ia_desc) * cfg_desc->uc_num_iads)){
        kfree(ia_desc);
        cfg_desc->iads = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    cfg_desc->iads = ia_desc;
    
exit:
    return rs;
}

/* Interface Descriptor¸ */
static int save_interface_desc(usb_gadget_config_desc *cfg_desc)
{
    usb_gadget_interface_desc *in_desc;
    int rs = 0;
    __u8 i;
    
    /* Interface Descriptor ο0λϽ */
    if(cfg_desc->uc_num_interfaces == 0){
        goto exit;
    }
    
    /* Interface Descriptor ΰ */
    in_desc = kmalloc(sizeof(usb_gadget_interface_desc) * cfg_desc->uc_num_interfaces,
                      GFP_KERNEL);
    if(!in_desc){
        PDEBUG("kmalloc error\n");
        cfg_desc->interfaces = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, in_desc);
    
    /* Interface Descriptor ΰΰإԡ */
    if(copy_from_user(in_desc,
                      cfg_desc->interfaces,
                      sizeof(usb_gadget_interface_desc) * cfg_desc->uc_num_interfaces)){
        kfree(in_desc);
        cfg_desc->interfaces = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    cfg_desc->interfaces = in_desc;
    
    /* Interface Table ΰݤƥԡ */
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        rs = save_interface_table(&cfg_desc->interfaces[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

/* Interface Table ¸ */
static int save_interface_table(usb_gadget_interface_desc *in_desc)
{
    usb_gadget_if_table *if_tbl;
    int rs = 0;
    __u8 i;
    
    /* Interface Table ο0λϽ */
    if(in_desc->uc_num_if_tables == 0){
        goto exit;
    }
    
    /* Interface Table ΰ */
    if_tbl = kmalloc(sizeof(usb_gadget_if_table) * in_desc->uc_num_if_tables, 
                     GFP_KERNEL);
    if(!if_tbl){
        PDEBUG("kmalloc error\n");
        in_desc->if_tables = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, if_tbl);
    
    /* Interface Table ΰΰإԡ */
    if(copy_from_user(if_tbl,
                      in_desc->if_tables,
                      sizeof(usb_gadget_if_table) * in_desc->uc_num_if_tables)){
        kfree(if_tbl);
        in_desc->if_tables = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    in_desc->if_tables = if_tbl;
    
    for(i = 0; i < in_desc->uc_num_if_tables; i++){
        /* Class Specific Interface Descriptor ΰݤƥԡ */
        rs = save_class_if_desc(&in_desc->if_tables[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
        
        /* Interface Table ΰݤƥԡ */
        rs = save_pep_list(&in_desc->if_tables[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

static int save_pep_list(usb_gadget_if_table *if_tbl)
{
    usb_gadget_if_table_element *element;
    int rs = 0;
    __u8 i;
    
    /* Pipe List ο0λϽ */
    if(if_tbl->uc_num_pep_list == 0){
        goto exit;
    }
    
    /* Pipe List ΰ */
    element = kmalloc(sizeof(usb_gadget_if_table_element) * if_tbl->uc_num_pep_list, 
                      GFP_KERNEL);
    if(!element){
        PDEBUG("kmalloc error\n");
        if_tbl->pep_list = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, element);
    
    /* Pipe List ΰΰإԡ */
    if(copy_from_user(element,
                      if_tbl->pep_list,
                      sizeof(usb_gadget_if_table_element) * if_tbl->uc_num_pep_list)){
        kfree(element);
        if_tbl->pep_list = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    if_tbl->pep_list = element;
    
    /* class_specific_ep_desc ΰ򥳥ԡ */
    for(i = 0; i < if_tbl->uc_num_pep_list; i++){
        rs = save_ep_companion_desc(&if_tbl->pep_list[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
        rs = save_class_ep_desc(&if_tbl->pep_list[i]);
        if(rs != 0){
            PDEBUG("%s() error\n", __func__);
            goto exit;
        }
    }
    
exit:
    return rs;
}

/* Class Specific Interface Descriptor ¸ */
static int save_class_if_desc(usb_gadget_if_table *if_tbl)
{
    void *tmp;
    int rs = 0;
    
    /* Class Specific Υ0λϽ */
    if(if_tbl->class_specific_interface_desc_size == 0){
        goto exit;
    }
    
    /* Class Specific ΰ */
    tmp = kmalloc(if_tbl->class_specific_interface_desc_size, GFP_KERNEL);
    if(!tmp){
        PDEBUG("kmalloc error\n");
        if_tbl->class_specific_interface_desc = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
    
    /* Class Specific ΰ򥳥ԡ */
    if(copy_from_user(tmp,
                      if_tbl->class_specific_interface_desc,
                      if_tbl->class_specific_interface_desc_size)){
        kfree(tmp);
        if_tbl->class_specific_interface_desc = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    if_tbl->class_specific_interface_desc = tmp;
    
exit:
    return rs;
}

/* EP Companion Descriptor ¸ */
static int save_ep_companion_desc(usb_gadget_if_table_element *element)
{
    usb_gadget_if_table_companion *tmp;
    int rs = 0;

    if (!element->ep_companion_desc)   /* without companion data */
        goto exit;

    /* companion ΰ */
    tmp = kmalloc(sizeof(*tmp), GFP_KERNEL);
    if(!tmp){
        PDEBUG("kmalloc error\n");
        element->ep_companion_desc = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
    
    /* Class Specific ΰ򥳥ԡ */
    if(copy_from_user(tmp,
                      element->ep_companion_desc,
                      sizeof(*tmp))){
        kfree(tmp);
        element->ep_companion_desc = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    element->ep_companion_desc = tmp;
    
exit:
    return rs;
}

/* Class Specific EP Descriptor ¸ */
static int save_class_ep_desc(usb_gadget_if_table_element *element)
{
    void *tmp;
    int rs = 0;
    
    if(element->class_specific_ep_desc_size == 0){
        /* Class Specific Υ0λϽ */
        goto exit;
    }
    
    /* Class Specific ΰ */
    tmp = kmalloc(element->class_specific_ep_desc_size, GFP_KERNEL);
    if(!tmp){
        PDEBUG("kmalloc error\n");
        element->class_specific_ep_desc = NULL;
        rs = -ENOMEM;
        goto exit;
    }
    PDEBUG("%s() : kmalloc %p\n", __func__, tmp);
    
    /* Class Specific ΰ򥳥ԡ */
    if(copy_from_user(tmp,
                      element->class_specific_ep_desc,
                      element->class_specific_ep_desc_size)){
        kfree(tmp);
        element->class_specific_ep_desc = NULL;
        PDEBUG("%s() : copy_from_user error\n", __func__);
        rs = -EFAULT;
        goto exit;
    }
    
    /* ݤΰդؤ */
    element->class_specific_ep_desc = tmp;
    
exit:
    return rs;
}

/*=============================================================================
 * Free Descriptor
 *===========================================================================*/
void free_desc_tbl(struct g_core_drv *g_core)
{
    int rs = 0;
    
    if(!g_core->desc_tbl){
        goto exit;
    }
    
    /* HS Config Descriptor ΰ */
    rs = free_config_desc(&g_core->desc_tbl->hs);
    if(rs != 0){
        goto exit;
    }
    
    /* LSFS Config Descriptor ΰ */
    rs = free_config_desc(&g_core->desc_tbl->lsfs);
    if(rs != 0){
        goto exit;
    }
    
    /* Report Descriptor ΰ */
    rs = free_report_desc( &g_core->desc_tbl->rptdesc );
    if(rs != 0){
        goto exit;
    }
    
    /* Langid Table ΰ */
    rs = free_langids(g_core->desc_tbl);
    if(rs != 0){
        goto exit;
    }
    
    /* BOS Descriptor ΰ */
    rs = free_bos(&g_core->desc_tbl->bos);
    if(rs != 0){
        goto exit;
    }
exit:
    if(g_core->desc_tbl){
        /* usb_gadget_desc_table Ѥΰ */
        PDEBUG("%s() : kfree %p\n", __func__, g_core->desc_tbl);
        kfree(g_core->desc_tbl);
        g_core->desc_tbl = NULL;
    }
}

static int free_bos(usb_gadget_bos_desc* bos)
{
    if (bos->devcap_ext) {
        PDEBUG("%s() : kfree %p\n", __func__, bos->devcap_ext);
        kfree(bos->devcap_ext);
    }

    if (bos->ss_cap_type) {
        PDEBUG("%s() : kfree %p\n", __func__, bos->ss_cap_type);
        kfree(bos->ss_cap_type);
    }

    if (bos->ssp_cap_type) {
        PDEBUG("%s() : kfree %p\n", __func__, bos->ssp_cap_type);
        kfree(bos->ssp_cap_type);
    }

    if (bos->container_id) {
        PDEBUG("%s() : kfree %p\n", __func__, bos->container_id);
        kfree(bos->container_id);
    }

    return 0;
}

static int free_langids(usb_gadget_desc_table *desc_tbl)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < desc_tbl->uc_num_langids; i++){
        if(&desc_tbl->langids[i] == NULL){
            PDEBUG("Fail: &desc_tbl->langids[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        rs = free_string_desc(&desc_tbl->langids[i]);
        if(rs != 0){
            rs = -EFAULT;
            goto exit;
        }
    }
    
exit:
    if(desc_tbl->uc_num_langids != 0 && desc_tbl->langids){
        PDEBUG("%s() : kfree %p\n", __func__, desc_tbl->langids);
        kfree(desc_tbl->langids);
    }
    
    return rs;
}

static int free_string_desc(usb_gadget_langid_table *langids)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < langids->uc_num_strings; i++){
        if(&langids->strings[i] == NULL){
            PDEBUG("Fail: &langids->strings[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        rs = free_strings(&langids->strings[i]);
        if(rs != 0){
            rs = -EFAULT;
            goto exit;
        }
    }
    
exit:
    if(langids->uc_num_strings != 0 && langids->strings){
        PDEBUG("%s() : kfree %p\n", __func__, langids->strings);
        kfree(langids->strings);
    }
    
    return rs;
}

static int free_strings(usb_gadget_string_desc *str_desc)
{
    int rs = 0;
    
    if(str_desc->uc_len == 0){
        goto exit;
    }
    
    if(str_desc->string){
        PDEBUG("%s() : kfree %p\n", __func__, str_desc->string);
        kfree(str_desc->string);
    }else{
        rs = -EFAULT;
    }
    
exit:
    return rs;
}

static int free_config_desc(usb_gadget_speed_desc *sp_desc)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < sp_desc->uc_num_config; i++){
        if(&sp_desc->configurations[i] == NULL){
            PDEBUG("Fail: &sp_desc->configurations[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        /* Interface Descriptor ΰ */
        rs = free_interface_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            goto exit;
        }
        
        /* OTG Descriptor ΰ */
        rs = free_otg_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            goto exit;
        }
        
        /* IA Descriptor ΰ */
        rs = free_ia_desc(&sp_desc->configurations[i]);
        if(rs != 0){
            goto exit;
        }
    }
    
exit:
    if(sp_desc->uc_num_config != 0 && sp_desc->configurations){
        /* configurations  */
        PDEBUG("%s() : kfree %p\n", __func__, sp_desc->configurations);
        kfree(sp_desc->configurations);
    }
    return rs;
}

static int free_report_desc( usb_gadget_report_desc *rpt_desc )
{
    
    int ret = 0;
    
    if( rpt_desc->desc_data != NULL ){
        
        PDEBUG( "%s() : kfree %p\n", __func__, rpt_desc->desc_data );
        kfree( rpt_desc->desc_data );
        rpt_desc->desc_data = NULL;
        
    }
    
    return ret;
    
}

static int free_otg_desc(usb_gadget_config_desc *cfg_desc)
{
    int rs = 0;
    
    if(cfg_desc->otg == NULL){
        goto exit;
    }
    
    if(cfg_desc->otg == (usb_gadget_otg_desc*)cfg_desc){
        rs = -EFAULT;
        goto exit;
    }
    
    PDEBUG("%s() : kfree %p\n", __func__, cfg_desc->otg);
    kfree(cfg_desc->otg);
    
exit:
    return rs;
}

static int free_ia_desc(usb_gadget_config_desc *cfg_desc)
{
    int rs = 0;
    
    if(cfg_desc->uc_num_iads == 0){
        goto exit;
    }
    
    if(cfg_desc->iads){
        PDEBUG("%s() : kfree %p\n", __func__, cfg_desc->iads);
        kfree(cfg_desc->iads);
    }else{
        rs = -EFAULT;
    }
    
exit:
    return rs;
}

static int free_interface_desc(usb_gadget_config_desc *cfg_desc)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < cfg_desc->uc_num_interfaces; i++){
        if(&cfg_desc->interfaces[i] == NULL){
            PDEBUG("Fail: &cfg_desc->interfaces[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        rs = free_interface_table(&cfg_desc->interfaces[i]);
        if(rs != 0){
            goto exit;
        }
    }
    
exit:
    if(cfg_desc->uc_num_interfaces != 0 && cfg_desc->interfaces){
        PDEBUG("%s() : kfree %p\n", __func__, cfg_desc->interfaces);
        kfree(cfg_desc->interfaces);
    }
    
    return rs;
}

static int free_interface_table(usb_gadget_interface_desc *in_desc)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < in_desc->uc_num_if_tables; i++){
        if(&in_desc->if_tables[i] == NULL){
            PDEBUG("Fail: &in_desc->if_tables[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        rs = free_class_if_desc(&in_desc->if_tables[i]);
        if(rs != 0){
            goto exit;
        }
        
        rs = free_pep_list(&in_desc->if_tables[i]);
        if(rs != 0){
            goto exit;
        }
    }
    
exit:
    if(in_desc->uc_num_if_tables != 0 && in_desc->if_tables){
        PDEBUG("%s() : kfree %p\n", __func__, in_desc->if_tables);
        kfree(in_desc->if_tables);
    }
    
    return rs;
}

static int free_class_if_desc(usb_gadget_if_table *if_tbl)
{
    int rs = 0;
    
    if(if_tbl->class_specific_interface_desc_size == 0){
        goto exit;
    }
    
    if(if_tbl->class_specific_interface_desc){
        PDEBUG("%s() : kfree %p\n", __func__, 
                    if_tbl->class_specific_interface_desc);
        kfree(if_tbl->class_specific_interface_desc);
    }else{
        rs = -EFAULT;
    }
    
exit:
    return rs;
}

static int free_pep_list(usb_gadget_if_table *if_tbl)
{
    int rs = 0;
    __u8 i;
    
    for(i = 0; i < if_tbl->uc_num_pep_list; i++){
        if(&if_tbl->pep_list[i] == NULL){
            PDEBUG("Fail: &if_tbl->pep_list[%d] is NULL\n", i);
            rs = -EFAULT;
            goto exit;
        }
        
        rs = free_ep_companion_desc(&if_tbl->pep_list[i]);
        if(rs != 0){
            goto exit;
        }
        rs = free_class_ep_desc(&if_tbl->pep_list[i]);
        if(rs != 0){
            goto exit;
        }
    }
    
exit:
    if(if_tbl->uc_num_pep_list != 0 && if_tbl->pep_list){
        PDEBUG("%s() : kfree %p\n", __func__, if_tbl->pep_list);
        kfree(if_tbl->pep_list);
    }
    
    return rs;
}

static int free_ep_companion_desc(usb_gadget_if_table_element *element)
{
    int rs = 0;

    if (element->ep_companion_desc) {
        PDEBUG("%s() : kfree %p\n", __func__,
                    element->ep_companion_desc);
        kfree(element->ep_companion_desc);
    }
    return rs;
}

static int free_class_ep_desc(usb_gadget_if_table_element *element)
{
    int rs = 0;
    
    if(element->class_specific_ep_desc_size == 0){
        goto exit;
    }
    
    if(element->class_specific_ep_desc){
        PDEBUG("%s() : kfree %p\n", __func__,
                    element->class_specific_ep_desc);
        kfree(element->class_specific_ep_desc);
    }else{
        rs = -EFAULT;
    }
    
exit:
    return rs;
}

/*=============================================================================
 * Descriptor Dump
 *===========================================================================*/
void dump_device_desc(usb_gadget_desc_table *desc_tbl)
{
    unsigned char c;
    
    /* Descriptor Table ɽ */
    printk("Descriptor Table:\n");
    
    /* HS Speed Descriptor ɽ */
    printk("  hs:\n"
           "    us_bcd_usb             0x%04x\n"
           "    uc_class                %5u\n"
           "    uc_sub_class            %5u\n"
           "    uc_protocol             %5u\n"
           "    uc_num_config           %5u\n"
           "    configurations:    0x%lx\n"
           "    uc_otg_attributes:      %5u\n",
           desc_tbl->hs.us_bcd_usb, desc_tbl->hs.uc_class,
           desc_tbl->hs.uc_sub_class, desc_tbl->hs.uc_protocol,
           desc_tbl->hs.uc_num_config, (uintptr_t)desc_tbl->hs.configurations,
           desc_tbl->hs.uc_otg_attributes);
    
    /* Config Descriptor ɽ */
    for(c = 0; c < desc_tbl->hs.uc_num_config; c++){
        if((desc_tbl->hs.configurations != NULL) && 
           (&desc_tbl->hs.configurations[c] != NULL)){
            dump_config_desc(&desc_tbl->hs.configurations[c]);
        }
    }
    
    /* LSFS Speed Descriptor ɽ */
    printk("  lsfs:\n"
           "    us_bcd_usb             0x%04x\n"
           "    uc_class                %5u\n"
           "    uc_sub_class            %5u\n"
           "    uc_protocol             %5u\n"
           "    uc_num_config           %5u\n"
           "    configurations:    0x%lx\n"
           "    uc_otg_attributes:      %5u\n",
           desc_tbl->lsfs.us_bcd_usb, desc_tbl->lsfs.uc_class,
           desc_tbl->lsfs.uc_sub_class, desc_tbl->lsfs.uc_protocol,
           desc_tbl->lsfs.uc_num_config, (uintptr_t)desc_tbl->lsfs.configurations,
           desc_tbl->lsfs.uc_otg_attributes);

    /* Config Descriptor ɽ */
    for(c = 0; c < desc_tbl->lsfs.uc_num_config; c++){
        if((desc_tbl->lsfs.configurations != NULL) &&
           (&desc_tbl->lsfs.configurations[c] != NULL)){
            dump_config_desc(&desc_tbl->lsfs.configurations[c]);
        }
    }
    
    /* SS Speed Descriptor ɽ */
    printk("  ss:\n"
           "    us_bcd_usb             0x%04x\n"
           "    uc_class                %5u\n"
           "    uc_sub_class            %5u\n"
           "    uc_protocol             %5u\n"
           "    uc_num_config           %5u\n"
           "    configurations:    0x%lx\n"
           "    uc_otg_attributes:      %5u\n",
           desc_tbl->ss.us_bcd_usb, desc_tbl->ss.uc_class,
           desc_tbl->ss.uc_sub_class, desc_tbl->ss.uc_protocol,
           desc_tbl->ss.uc_num_config, (uintptr_t)desc_tbl->ss.configurations,
           desc_tbl->ss.uc_otg_attributes);

    /* Config Descriptor ɽ */
    for(c = 0; c < desc_tbl->ss.uc_num_config; c++){
        if((desc_tbl->ss.configurations != NULL) &&
           (&desc_tbl->ss.configurations[c] != NULL)){
            dump_config_desc(&desc_tbl->ss.configurations[c]);
        }
    }

    printk("  us_id_vendor           0x%04x\n"
           "  us_id_product          0x%04x\n"
           "  us_bcd_device          0x%04x\n"
           "  us_default_attributes  0x%04x\n"
           "  uc_manufacturer_str     %5u\n"
           "  uc_product_str          %5u\n"
           "  uc_serial_number_str    %5u\n"
           "  uc_num_langids          %5u\n"
           "  langid_table:      0x%lx\n",
           desc_tbl->us_id_vendor, desc_tbl->us_id_product, 
           desc_tbl->us_bcd_device, desc_tbl->uc_default_attributes,
           desc_tbl->uc_manufacturer_str, desc_tbl->uc_product_str, 
           desc_tbl->uc_serial_number_str, desc_tbl->uc_num_langids, 
           (uintptr_t)desc_tbl->langids);

    /* Langid Table ɽ */
    for(c = 0; c < desc_tbl->uc_num_langids; c++){
        if((desc_tbl->langids != NULL) &&
           (&desc_tbl->langids[c] != NULL)){
            dump_langid_table(&desc_tbl->langids[c]);
        }
    }

    /* BOS Descriptor ɽ */
    printk("  BOS\n"
           "    us_total_length    %5u\n"
           "    uc_num_dev_caps    %5u\n",
           desc_tbl->bos.us_total_length,
           desc_tbl->bos.uc_num_dev_caps);
    if (desc_tbl->bos.devcap_ext)
        printk("    devcap_ext\n"
               "      ul_attributes    0x%08x\n",
               (desc_tbl->bos.devcap_ext)->ul_attributes);
    else
        printk("    devcap_ext\n"
               "      none\n");
    if (desc_tbl->bos.ss_cap_type)
        printk("    ss_cap_type\n"
               "      uc_attributes      0x%02x\n"
               "      us_speeds_supported      %5u\n"
               "      uc_functionality_support %5u\n"
               "      uc_U1_dev_exit_lat %5u\n"
               "      us_U2_dev_exit_lat %5u\n",
               (desc_tbl->bos.ss_cap_type)->uc_attributes,
               (desc_tbl->bos.ss_cap_type)->us_speeds_supported,
               (desc_tbl->bos.ss_cap_type)->uc_functionality_support,
               (desc_tbl->bos.ss_cap_type)->uc_U1_dev_exit_lat,
               (desc_tbl->bos.ss_cap_type)->us_U2_dev_exit_lat);
    else
        printk("    ss_cap_type\n"
               "      none\n");
    if (desc_tbl->bos.ssp_cap_type) {
        int i, ssac;
        ssac = ((desc_tbl->bos.ssp_cap_type)->ul_attributes & USB_SSP_SUBLINK_SPEED_ATTRIBS) + 1;
        printk("    ssp_cap_type\n"
               "      uc_reserved      0x%02x\n"
               "      ul_attributes      0x%08x\n"
               "      us_functionality_support      0x%04x\n"
               "      us_reserved      0x%04x\n",
               (desc_tbl->bos.ssp_cap_type)->uc_reserved,
               (desc_tbl->bos.ssp_cap_type)->ul_attributes,
               (desc_tbl->bos.ssp_cap_type)->us_functionality_support,
               (desc_tbl->bos.ssp_cap_type)->us_reserved);
        for (i=0; i<ssac; i++)
            printk("      ul_sublink_speed_attr[%2d]      0x%08x\n",
                   i, (desc_tbl->bos.ssp_cap_type)->ul_sublink_speed_attr[i]);
    }
    else
        printk("    ssp_cap_type\n"
               "      none\n");
    if (desc_tbl->bos.container_id) {
        printk("    container_id\n"
               "      ");
        for (c=0; c<sizeof(*desc_tbl->bos.container_id); c++) {
            printk("%02x", desc_tbl->bos.container_id->uc_uuid[c]);
        }
    }
    else
        printk("    container_id\n"
               "      none\n");
}

static void dump_config_desc(usb_gadget_config_desc *desc)
{
    unsigned char c;
    
    /* Config Descriptor ɽ */
    printk("      uc_index                %5u\n"
           "      uc_configuration_value  %5u\n"
           "      uc_configuration_str    %5u\n"
           "      uc_max_power            %5u\n"
           "      otg                0x%lx\n",
           desc->uc_index, desc->uc_configuration_value,
           desc->uc_configuration_str, desc->uc_max_power,
           (uintptr_t)desc->otg);
    
    /* OTG Descriptor ɽ */
    if(desc->otg){
        dump_otg_desc(desc->otg);
    }
    
    printk("      uc_num_iads             %5u\n"
           "      iads               0x%lx\n",
           desc->uc_num_iads, (uintptr_t)desc->iads);

    /* Interface Association Descriptor ɽ */
    if(desc->iads){
        dump_ia_desc(desc->iads);
    }
    
    printk("      uc_num_interfaces       %5u\n"
           "      interfaces         0x%lx\n",
           desc->uc_num_interfaces, (uintptr_t)desc->interfaces);
    
    /* Interface Descriptor ɽ */
    for(c = 0; c < desc->uc_num_interfaces; c++){
        if((desc->interfaces != NULL) && (&desc->interfaces[c]) != NULL){
            dump_interface_desc(&desc->interfaces[c]);
        }
    }
}

static void dump_otg_desc(usb_gadget_otg_desc *desc)
{
    /* OTG Descriptor ɽ */
    printk("        uc_attributes          0x%04x\n",
           desc->uc_attributes);
}

static void dump_ia_desc(usb_gadget_ia_desc *desc)
{
    /* Interface Association Descriptor ɽ */
    printk("        uc_firstinterface       %5u\n"
           "        uc_interface_count      %5u\n"
           "        uc_function_class       %5u\n"
           "        uc_function_sub_class   %5u\n"
           "        uc_function_protocol    %5u\n"
           "        uc_function             %5u\n",
           desc->uc_firstinterface,
           desc->uc_interface_count,
           desc->uc_function_class,
           desc->uc_function_sub_class,
           desc->uc_function_protocol,
           desc->uc_function);
}

static void dump_interface_desc(usb_gadget_interface_desc *desc)
{
    unsigned char c;
    
    /* Interface Descriptor ɽ */
    printk("        uc_if_number            %5u\n"
           "        uc_num_if_tables        %5u\n"
           "        if_tables          0x%lx\n",
           desc->uc_if_number,
           desc->uc_num_if_tables,
           (uintptr_t)desc->if_tables);
    
    /* Interface Table ɽ */
    for(c = 0; c < desc->uc_num_if_tables; c++){
        if((desc->if_tables != NULL) && (&desc->if_tables[c] != NULL)){
            dump_if_tables(&desc->if_tables[c]);
        }
    }
}

static void dump_if_tables(usb_gadget_if_table *if_tbl)
{
    unsigned char c;
    __u32 i;
    
    /* Interface Table ɽ */
    printk("          uc_setting_number       %5u\n"
           "          uc_class                %5u\n"
           "          uc_sub_class            %5u\n"
           "          uc_interface_protocol   %5u\n"
           "          uc_interface_str        %5u\n"
           "          class_specific_interface_desc_size\n"
           "                                  %5u\n"
           "          class_specific_interface_desc\n"
           "                             0x%lx\n",
           if_tbl->uc_setting_number,
           if_tbl->uc_class,
           if_tbl->uc_sub_class,
           if_tbl->uc_interface_protocol,
           if_tbl->uc_interface_str,
           if_tbl->class_specific_interface_desc_size,
           (uintptr_t)if_tbl->class_specific_interface_desc);

    /* class_specific_interface_desc ɽ */
    for(i = 0; i < if_tbl->class_specific_interface_desc_size; i++){
        if(i % 16 == 0){ printk("\n            "); }
        printk("0x%02x ", ((unsigned char*)if_tbl->class_specific_interface_desc)[i]);
    }
    if(if_tbl->class_specific_interface_desc_size != 0){ printk("\n"); }
    
    
    printk("          uc_num_pep_list         %5u\n"
           "          pep_list           0x%lx\n",
           if_tbl->uc_num_pep_list,
           (uintptr_t)if_tbl->pep_list);

    /* Pipe List ɽ */
    for(c = 0; c < if_tbl->uc_num_pep_list; c++){
        if((if_tbl->pep_list != NULL) && (&(if_tbl->pep_list[c]) != NULL)){
            dump_pep_list(&if_tbl->pep_list[c]);
        }
    }
}

static void dump_pep_list(usb_gadget_if_table_element *pep_list)
{
    __u32 i;
    
    /* Pipe List ɽ */
    printk("            uc_ep_address           %5u\n"
           "            uc_attributes           %5u\n"
           "            us_max_psize            %5u\n"
           "            uc_interval             %5u\n"
           "            class_specific_ep_desc_size\n"
           "                                  %5u\n"
           "            class_specific_ep_desc\n"
           "                             0x%lx\n",
           pep_list->uc_ep_address,
           pep_list->uc_attributes,
           pep_list->us_max_psize,
           pep_list->uc_interval,
           pep_list->class_specific_ep_desc_size,
           (uintptr_t)pep_list->class_specific_ep_desc);
    
    /* class_specific_ep_desc ɽ */
    for(i = 0; i < pep_list->class_specific_ep_desc_size; i++){
        if(i % 16 == 0){ printk("\n            "); }
        printk("0x%02x ", ((unsigned char*)pep_list->class_specific_ep_desc)[i]);
    }
    if(pep_list->class_specific_ep_desc_size != 0){ printk("\n"); }
    if (pep_list->ep_companion_desc)
        printk("            ep comp uc_max_burst          %5u\n"
               "                    uc_attributes         %5u\n"
               "                    us_bytes_per_interval %5u\n",
               pep_list->ep_companion_desc->uc_max_burst,
               pep_list->ep_companion_desc->uc_attributes,
               pep_list->ep_companion_desc->us_bytes_per_interval);
}

static void dump_langid_table(usb_gadget_langid_table *tbl)
{
    unsigned char c;
    
    /* Langid Table ɽ */
    printk("      us_langid               %5u\n"
           "      uc_num_strings          %5u\n"
           "      strings            0x%lx\n",
           tbl->us_langid,
           tbl->uc_num_strings,
           (uintptr_t)tbl->strings);
    
    /* String Descriptor ɽ */
    for(c = 0; c < tbl->uc_num_strings; c++){
        if((tbl->strings != NULL) && (&tbl->strings[c] != NULL)){
            dump_string_desc(&tbl->strings[c]);
        }
    }
}

static void dump_string_desc(usb_gadget_string_desc *str_desc)
{
    unsigned char c;
    
    printk("        uc_index                %5u\n"
           "        uc_len                  %5u\n"
           "        strings            0x%lx\n",
           str_desc->uc_index,
           str_desc->uc_len,
           (uintptr_t)str_desc->string);
    
    /* Strings ɽ */
    printk("\"");
    for(c = 0; c < str_desc->uc_len; c++){
        printk("%c", ((unsigned char*)str_desc->string)[c]);
    }
    printk("\"\n\n");
}
