/**
 *	@file	usb_extcmd.h
 *	@brief	USB EXTCMD(USB Extension Command) - ioctl
 *	
 *		Copyright 2008 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __UAPI_USBG_IOCTL_EXTCMD_H__
#define __UAPI_USBG_IOCTL_EXTCMD_H__


#include <asm/ioctl.h>
#include <linux/ioctl.h>

#include "usb_ioctl.h"


/*
 *	@brief	msc command param struct
 */
struct _usb_extcmd_cbwcb {
    /** byte 0 =============================================================*/
    unsigned char	opCode;					/**< Operation Code ( **h )		*/
    /** byte 1 - 2  ========================================================*/
    unsigned short	wIndex;					/**< word Index					*/
    /** byte 3 =============================================================*/
    unsigned char 	bRequest;				/**< byte Request				*/
    /** byte 4 =============================================================*/
    unsigned char 	bValue;					/**< byte Value					*/
    /** byte 5 - 6  ========================================================*/
    unsigned char 	wParam[ 2 ];			/**< word Param					*/
    /** byte 7 - 11 ========================================================*/
    unsigned char reserved2_11[ 4 ];		/**< reserved ( 0h )			*/
} __attribute__ ((packed));

/**
 *	@brief  Transfer mode
 */
typedef enum {
	EXTCMD_TRANS_MODE_DMA	= 0x0000,
	EXTCMD_TRANS_MODE_CPU	= 0x0100,
} EXTCMD_TRANS_MODE;

/**
 *	@brief  Transfer Buffer Type
 */
typedef enum {
	EXTCMD_TRANS_MEM_VIRT	= 0x0000,
	EXTCMD_TRANS_MEM_PHYS	= 0x0001,
} EXTCMD_TRANS_MEM;

/*
 *	@brief	extcmd sense struct
 */
typedef struct _usb_extcmd_sense {
	unsigned char	senseKey;
	union{
		struct _byte{
			unsigned char	asc;
			unsigned char	ascq;
		} byte;
		unsigned short		sensCode;
	} additional;
} usb_extcmd_sense;

/*
 *	@brief	extcmd size type
 */
typedef uint32_t extcmd_size_t;


/*
 *	@brief	extcmd sense key
 */
#define EXTCMD_SK_NO_SENSE				(0x00)		/**< no sense			*/
#define EXTCMD_SK_ILLEGAL_REQ			(0x05)		/**< illegal request	*/
#define EXTCMD_SK_VENDOR				(0x09)		/**< vendor specific	*/

/*
 *	@brief	extcmd additional sense code (asc & ascq)
 */
#define EXTCMD_SC_NO_ADD_SENSE			(0x0000)	/**< no add sense		*/
#define EXTCMD_SC_INVALID_OP_CODE		(0x2000)	/**< invalid operation	*/
#define EXTCMD_SC_VND_BECOMING_READY	(0x8181)	/**< vendor not ready	*/
#define EXTCMD_SC_VND_SYSTEM_ERROR		(0x8182)	/**< vendor system err	*/
#define EXTCMD_SC_VND_ABORT_CMD_RETRY	(0x8183)	/**< vendor abort cmd retry	*/
#define EXTCMD_SC_VND_ABORT_CMD_FATAL	(0x8184)	/**< vendor abort cmd fatal	*/


/*
 *	@brief	extcmd sense check Macro [is no problem]
 *	@param[in]	struct _usb_extcmd_sense pointer
 *	@retval		0: not registered yet
 *	@retval		1: registered
 */
#define EXTCMD_IS_SENSE_NO_PROBLEM(sp)  						\
		( (((sp)->senseKey == EXTCMD_SK_NO_SENSE) &&			\
 		  ((sp)->additional.sensCode == EXTCMD_SC_NO_ADD_SENSE)) ? 1:0 )

/*
 *	@brief	extcmd sense check Macro[is registered]
 *	@param[in]	struct _usb_extcmd_sense pointer
 *	@retval		0: not registered yet
 *	@retval		1: registered
 */
#define EXTCMD_IS_SENSE_REGISTERED(sp)  						\
		( (((sp)->senseKey == EXTCMD_SK_ILLEGAL_REQ) &&			\
 		  ((sp)->additional.sensCode == EXTCMD_SC_INVALID_OP_CODE)) ? 0:1 )


/*
 *	@brief	inquiry direction
 */
#define EXTCMD_INQUIRY_DIR_K_READ	(1)
#define EXTCMD_INQUIRY_DIR_K_WRITE	(2)

/*
 *	@brief	cancel param user request flags
 */
#define EXTCMD_NOT_REQUESTING	(0)
#define EXTCMD_REQUESTING		(1)

/*
 *	@brief	extcmd command block packet
 */
typedef union _usb_extcmd_cbp{
	struct _usb_extcmd_cbwcb	*cbwcb;			///< msc cbwcb
//	;											///< other protocol
} usb_extcmd_cbp;

/*	###########################################
	IOCTL user space param structs
	########################################### */
/*
 *	@brief	IOCTL param struct	[Register]
 */
typedef struct _usb_extcmd_param_register {
	unsigned char	opcode;			///< operation code
	unsigned short	index;			///< feature index
    uint32_t   sendDataLen;    ///< 
    uint32_t   recvDataLen;    ///< 
} usb_extcmd_param_register;

/*
 *	@brief	IOCTL param struct	[Unregister]
 */
typedef struct _usb_extcmd_param_unregister {
	unsigned short	dummy;			///< dummy
} usb_extcmd_param_unregister;


/*
 *	@brief	IOCTL param struct	[Read/Write]
 */
typedef struct _usb_extcmd_param_rw {
	uint32_t	*data;			///< data buffer
	extcmd_size_t	size;			///< read deta size
	extcmd_size_t	residue;		///< read data residue size
	unsigned int	timeout;		///< timeout [ms] order (not used when must set to zero)
	unsigned short	mode;			///< trans mode
} usb_extcmd_param_rw;

/*
 *	@brief	IOCTL param struct	[Reset]
 */
typedef struct _usb_extcmd_param_cancel {
	unsigned int	user_request;	///< user requesting flag
} usb_extcmd_param_cancel;



/*	###########################################
	IOCTL kernel space param structs
	########################################### */


/*
 *	@brief	IOCTL param struct	[Inquiry]
 */
typedef struct _usb_extcmd_k_param_inquiry {
	extcmd_size_t		cbpLen;		///< total request deta size
	usb_extcmd_cbp		cbp;		///< command block packet
	unsigned short		dir;		///< direction
	usb_extcmd_sense	*sense;		///< extcmd sense
    uint32_t   sendDataLen;    ///< 
    uint32_t   recvDataLen;    ///< 
} usb_extcmd_k_param_inquiry;

/*
 *	@brief	IOCTL Kernel param struct	[Read/Write]
 */
typedef struct _usb_extcmd_k_param_rw {
	extcmd_size_t		cbpLen;		///< total request deta size
	usb_extcmd_cbp		cbp;		///< command block packet
	extcmd_size_t		totalReqLen;///< total request deta size
	extcmd_size_t		dataLen;	///< data buffer length
	uint32_t		*data;		///< data buffer
	extcmd_size_t		ret_size;	///< size
	unsigned int		timeout;	///< timeout [ms] order (not used then must set to zero)
	unsigned short		mode;		///< trans mode
	usb_extcmd_sense	*sense;		///< extcmd sense
} usb_extcmd_k_param_rw;

/*
 *	@brief	IOCTL param struct	[Reset]
 */
typedef struct _usb_extcmd_k_param_reset {
	unsigned short	dummy;			///< dummy
} usb_extcmd_k_param_reset;

/*
 *	@brief	IOCTL parameters
 */
union usb_extcmd_param {
	struct _usb_extcmd_param_register	prm_reg;		///< register ioctl param
	struct _usb_extcmd_param_unregister	prm_ureg;		///< unregister ioctl param
	struct _usb_extcmd_param_rw			prm_rw;			///< read write ioctl param
	struct _usb_extcmd_param_cancel		prm_cancel;		///< cancel ioctl param
	struct _usb_extcmd_k_param_inquiry	kprm_inquiry;	///< inquiry ioctl param
	struct _usb_extcmd_k_param_rw		kprm_rw;		///< read write ioctl param
	struct _usb_extcmd_k_param_reset	kprm_reset;		///< reset ioctl param
};

/*
 *	@brief	IOCTL detail
 */
#define USB_IOC_EXTCMD_REGISTER			_IOWR(USB_IOC_EXT_CMD, 0, struct _usb_extcmd_param_register)
#define USB_IOC_EXTCMD_UNREGISTER		_IOWR(USB_IOC_EXT_CMD, 1, struct _usb_extcmd_param_unregister)
#define USB_IOC_EXTCMD_READ				_IOWR(USB_IOC_EXT_CMD, 2, struct _usb_extcmd_param_rw)
#define USB_IOC_EXTCMD_WRITE			_IOWR(USB_IOC_EXT_CMD, 3, struct _usb_extcmd_param_rw)
#define USB_IOC_EXTCMD_CANCEL			_IOWR(USB_IOC_EXT_CMD, 4, struct _usb_extcmd_param_cancel)
#define USB_IOC_EXTCMD_K_INQUIRY		_IOWR(USB_IOC_EXT_CMD, 5, struct _usb_extcmd_k_param_inquiry)
#define USB_IOC_EXTCMD_K_READ			_IOWR(USB_IOC_EXT_CMD, 6, struct _usb_extcmd_k_param_rw)
#define USB_IOC_EXTCMD_K_WRITE			_IOWR(USB_IOC_EXT_CMD, 7, struct _usb_extcmd_k_param_rw)
#define USB_IOC_EXTCMD_K_RESET			_IOWR(USB_IOC_EXT_CMD, 8, struct _usb_extcmd_k_param_reset)

#define USB_IOC_EXTCMD_NBROF		(9)


#endif /// __UAPI_USBG_IOCTL_EXTCMD_H__

