/**
 *	@file	usb_extcmd_data.c
 *	@brief	USB EXTCMD(USB Extension Command) - data
 *	
 *		Copyright 2008,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 */

#include <asm/errno.h>
#include <asm/memory.h>
#include <asm/uaccess.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>

#define FNAME "usbg_ext_data.c"
#include "usb_extcmd_pvt.h"


/**
 *	@brief		EXTCMD driver open
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			Success
 *	@retval		-EINVAL		Invalid argument
 *	@retval		-ENOMEM		Out of memory
 */
UDIF_ERR extcmd_open(UDIF_FILE *filp)
{
	void *private_data;

	EXTCMD_API();

	private_data = extcmd_mem_alloc(sizeof(struct extcmd_info));
	if(!private_data){
		EXTCMD_ERR();
		return UDIF_ERR_NOMEM;
	}
    udif_file_data(filp) = private_data;

	memset(private_data, 0, sizeof(struct extcmd_info));
	((struct extcmd_info *)private_data)->space = EXTCMD_SPACE_KERNEL;
	return UDIF_ERR_OK;
}

/**
 *	@brief		EXTCMD driver release
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			Success
 *	@retval		-EINVAL		Invalid argument
 *	@retval		-ENOMEM		Out of memory
 */
UDIF_ERR extcmd_release(UDIF_FILE *filp)
{
	void *private_data;

	private_data = udif_file_data(filp);
	EXTCMD_API("is userspace[%d]", ((struct extcmd_info *)private_data)->space );

	extcmd_mem_free(private_data, sizeof(struct extcmd_info));
	udif_file_data(filp) = NULL;

	return UDIF_ERR_OK;
}


/**
 *	@brief		ioctl register
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_close struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 */
static int ioctl_u_register(struct inode *inode, struct file *file, struct _usb_extcmd_param_register *arg)
{
	wait_queue_head_t *wqh;
	struct _extcmd_cmd_list *entry;
	struct _usb_extcmd_param_register param;
	struct extcmd_info *info;
	int err;

	EXTCMD_API();

	// copy user area parameter to kernel area member.
	err = copy_from_user(&param, arg, sizeof(struct _usb_extcmd_param_register));
	if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if(!file){
		EXTCMD_ERR();
		return -EINVAL;
	}
	if(!file->private_data){
		EXTCMD_ERR();
		return -EINVAL;
	}


	wqh = extcmd_mem_alloc(sizeof(wait_queue_head_t));
	if(!wqh){
		EXTCMD_ERR();
		return -ENOMEM;
	}

	/// initialize wait queue
	init_waitqueue_head(wqh);

	entry = extcmd_mem_alloc(sizeof(struct _extcmd_cmd_list));
	if(!entry) {
		extcmd_mem_free(wqh, sizeof(wait_queue_head_t));
		EXTCMD_ERR();
		return -ENOMEM;
	}

	/// set private date
	info = (struct extcmd_info *)file->private_data;
	info->space  = EXTCMD_SPACE_USER;
	info->opcode = param.opcode;
	info->index  = param.index;

	entry->opcode		= param.opcode;
	entry->index		= param.index;
    entry->send_data_len    = param.sendDataLen;
    entry->recv_data_len    = param.recvDataLen;
	entry->wqh			= wqh;
	entry->inq_state 	= 0;
	entry->sync_flag	= 0;
	entry->k_read_param		= NULL;
	entry->k_write_param	= NULL;
	entry->u_read_size		= 0;
	entry->u_write_size		= 0;
	entry->req_order_flag	= 0;
	/// kernel param lock init
	EXTCMD_K_PRAM_LOCK_INI(entry);
	/// kernel work lock init
	EXTCMD_K_WORK_LOCK_INI();
	/// append node
	err = list_append_node(entry);
	if( err ){
		extcmd_mem_free(wqh, sizeof(wait_queue_head_t));
		extcmd_mem_free(entry, sizeof(struct _extcmd_cmd_list));
		EXTCMD_ERR();
		return err;
	}

	return err;
}

/**
 *	@brief		EXTCMD unregister
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_close struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 */
static int ioctl_u_unregister(struct inode *inode, struct file *file, struct _usb_extcmd_param_unregister *arg)
{
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;

	EXTCMD_API();

	if(!file){
		EXTCMD_ERR();
		return -EINVAL;
	}
	if(!file->private_data){
		EXTCMD_ERR();
		return -EINVAL;
	}
	
	info = (struct extcmd_info *)file->private_data;
	entry = list_get_node(info->opcode, info->index);
	if(!entry){
		EXTCMD_ERR();
		return -EBADRQC;
	}

	set_bit(EXTCMD_WFLAG_CANCEL_USER, &entry->sync_flag);
	set_bit(EXTCMD_WFLAG_CANCEL_KERN, &entry->sync_flag);
	EXTCMD_INF("wake_up_interruptible flg[%x]", (unsigned int)entry->sync_flag);
	wake_up_interruptible(entry->wqh);


	EXTCMD_K_WORK_LOCK_ON();
	extcmd_mem_free(entry->wqh, sizeof(wait_queue_head_t));
	list_delete_node(entry);
	extcmd_mem_free(entry, sizeof(struct _extcmd_cmd_list));
	EXTCMD_K_WORK_LOCK_OFF();

	return 0;
}

/**
 *	@brief		EXTCMD cancel request
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_u_cancel(struct inode *inode, struct file *file, struct _usb_extcmd_param_cancel *arg)
{
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;
	struct _usb_extcmd_param_cancel param;
	int err;

	EXTCMD_API();

	// copy user-area parameter to kernel area-member.
	err = copy_from_user(&param, arg, sizeof(struct _usb_extcmd_param_cancel));
		if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}

	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_USER){
		EXTCMD_ERR();
		return -EINVAL;
	}
	entry = list_get_node(info->opcode, info->index);
	info->u_write_offset	= 0;
	info->u_read_offset		= 0;
	entry->u_write_size		= 0;
	entry->u_read_size		= 0;

	synchronize_wake_up_cancel_event(entry->wqh, &entry->sync_flag, param.user_request);

	return 0;
}


/**
 *	@brief		EXTCMD data copy from user
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		!0			error
 */
static int calc_and_copy_from_user(unsigned char *kernel_stay,
								 struct _extcmd_cmd_list *entry,
								 struct extcmd_info *info,
								 struct _usb_extcmd_param_rw *u_write_param)
{
	struct _usb_extcmd_k_param_rw *k_read_param;
	void *from, *to;
	extcmd_size_t *u_write_offset = &info->u_write_offset;
	extcmd_size_t *u_write_size = &entry->u_write_size;
	extcmd_size_t copy_size;
	int err;


	if(!entry->k_read_param){
		EXTCMD_ERR();
		return -EINVAL;
	}

	k_read_param = entry->k_read_param;

	EXTCMD_INF("uwrite u buf[%p]size[%d]",
						u_write_param->data,
						(unsigned int)u_write_param->size);
	EXTCMD_INF("uwrite k buf[%p]size[%d]totalsize[%d]",
						k_read_param->data,
						(unsigned int)k_read_param->dataLen,
						(unsigned int)k_read_param->totalReqLen);
	EXTCMD_INF("uwrite offset[%d]read[%d]",
						(unsigned int)*u_write_offset, (unsigned int)*u_write_size);


	if(u_write_param->size < k_read_param->dataLen - *u_write_offset){	// User Return
		copy_size    = u_write_param->size;
		*kernel_stay = TRUE;
	}
	else{																// User Kernel Return
		copy_size    = k_read_param->dataLen - *u_write_offset;
		*kernel_stay = FALSE;
	}
	from	= (void*)u_write_param->data;
	to		= (void *)((uintptr_t)k_read_param->data + *u_write_offset);


	EXTCMD_INF("copy_from_user toaddr[%p]fromaddr[%p]size[%d]", to, from, (unsigned int)copy_size);
	err = copy_from_user(to, from, copy_size);
	if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}

	*u_write_offset	+= copy_size;
	*u_write_size	+= copy_size;

	u_write_param->size		= copy_size;
	u_write_param->residue	= k_read_param->totalReqLen - *u_write_size;


	EXTCMD_INF("u ret size[%d]residue[%d]",
						(unsigned int)u_write_param->size,
						(unsigned int)u_write_param->residue);

	if(!(*kernel_stay)){

		k_read_param->ret_size = *u_write_offset;
		*u_write_offset = 0;
		if(k_read_param->totalReqLen == *u_write_size){
			*u_write_size = 0;
		}
		EXTCMD_INF("k ret size[%d]", (unsigned int)k_read_param->ret_size);
	}

	return 0;
}



/**
 *	@brief		EXTCMD data write
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_u_write(struct inode *inode, struct file *file, struct _usb_extcmd_param_rw *arg)
{
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;
	struct _usb_extcmd_param_rw param;
	int err;
	unsigned char wait_flag;
	unsigned char k_stay_flag;

	EXTCMD_API();

	// copy user-area parameter to kernel area-member.
	err = copy_from_user(&param, arg, sizeof(struct _usb_extcmd_param_rw));
	if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if( !file ){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if( !(param.size) ){
		EXTCMD_ERR();
		return -EINVAL;
	}
	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_USER){
		EXTCMD_ERR();
		return -EINVAL;
	}
	entry = list_get_node(info->opcode, info->index);
	if(!entry){
		EXTCMD_ERR();
		return -EINVAL;
	}
	EXTCMD_INF("uwrite op[%02x]index[%04x] param addr[%p] size[%d]",
						info->opcode, info->index, param.data, (unsigned int)param.size);


	/// 1st write or continue write and kernel not read yet.
	wait_flag = test_and_set_bit_req_flag_user(
						EXTCMD_REQ_BIT_KERN_READ,
						EXTCMD_REQ_BIT_USER_WRITE,
						entry);
	if( wait_flag == EXTCMD_WAIT_READY ){
		EXTCMD_INF("uwrite wait..");
		enable_inquiry_state(entry, EXTCMD_FROM_USER_DIRECTION);	// inquiry enable
		/// with kernel synchronize
		err = synchronize_wait_event(info->space, entry->wqh, EXTCMD_WFLAG_USER_WRITE,
						&entry->sync_flag, param.timeout);
		disable_inquiry_state(entry, EXTCMD_FROM_USER_DIRECTION);	// inquiry disable
		/// clear wait flag
		clear_bit_req_flag(EXTCMD_REQ_BIT_USER_WRITE, entry);
		EXTCMD_INF("uwrite wakeup!!");
		if(err){
			EXTCMD_INF("err[%d]", err);
			entry->u_write_size		= 0;
			info->u_write_offset	= 0;
			return err;
		}
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	// transfer data buffer copy.
	err = calc_and_copy_from_user(&k_stay_flag, entry, info, &param);
	if(err){
		EXTCMD_ERR();
		entry->u_write_size		= 0;
		info->u_write_offset	= 0;
		EXTCMD_K_PRAM_LOCK_OFF(entry);
		return -EINVAL;
	}

	// copy kernel area member  to user area parameter.
	err = copy_to_user(arg, &param, sizeof(struct _usb_extcmd_param_rw));
	if(err){
		EXTCMD_ERR("err[%d]", err);
		entry->u_write_size		= 0;
		info->u_write_offset	= 0;
		EXTCMD_K_PRAM_LOCK_OFF(entry);
		return -EINVAL;
	}

	EXTCMD_K_PRAM_LOCK_OFF(entry);

	if(!k_stay_flag){
		synchronize_wake_up_event(entry->wqh, EXTCMD_WFLAG_USER_WRITE_COPY, &entry->sync_flag);
	}

	return 0;
}

/**
 *	@brief		EXTCMD data copy to user
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		!0			error
 */
static int calc_and_copy_to_user(unsigned char *kernel_stay,
								 struct _extcmd_cmd_list *entry,
								 struct extcmd_info *info,
								 struct _usb_extcmd_param_rw *u_read_param)
{
	struct _usb_extcmd_k_param_rw *k_write_param;
	void *from, *to;
	extcmd_size_t *u_read_offset = &info->u_read_offset;
	extcmd_size_t *u_read_size = &entry->u_read_size;
	extcmd_size_t copy_size;
	int err;



	if(!entry->k_write_param){
		EXTCMD_ERR();
		return -EINVAL;
	}

	k_write_param = entry->k_write_param;

	EXTCMD_INF("uread u buf[%p]size[%d]",
						u_read_param->data,
						(unsigned int)u_read_param->size);
	EXTCMD_INF("uread k buf[%p]size[%d]totalsize[%d]",
						k_write_param->data,
						(unsigned int)k_write_param->dataLen,
						(unsigned int)k_write_param->totalReqLen);
	EXTCMD_INF("uread offset[%d]read[%d]",
						(unsigned int)*u_read_offset,
						(unsigned int)*u_read_size);
	
	
	if(u_read_param->size < k_write_param->dataLen - *u_read_offset){	// User return
		copy_size	 = u_read_param->size;
		*kernel_stay = TRUE;
	}
	else{																// User Kernel Return
		copy_size	 = k_write_param->dataLen - *u_read_offset;
		*kernel_stay = FALSE;
	}
	from	= (void *)((uintptr_t)k_write_param->data + *u_read_offset);
	to		= (void *)u_read_param->data;


	EXTCMD_INF("copy_to_user toaddr[%p]fromaddr[%p]size[%d]", to, from, (unsigned int)copy_size);
	err = copy_to_user(to, from, copy_size);
	if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}

	*u_read_offset	+= copy_size;
	*u_read_size	+= copy_size;

	u_read_param->size		= copy_size;
	u_read_param->residue	= k_write_param->totalReqLen - *u_read_size;


	EXTCMD_INF("u ret size[%d]residue[%d]",
						(unsigned int)u_read_param->size,
						(unsigned int)u_read_param->residue);

	if(!(*kernel_stay)){
		k_write_param->ret_size = *u_read_offset;
		*u_read_offset = 0;
		if(k_write_param->totalReqLen == *u_read_size){
			*u_read_size   = 0;
		}
		EXTCMD_INF("k ret size[%d]", (unsigned int)k_write_param->ret_size);
	}

	return 0;
}

/**
 *	@brief		EXTCMD data read
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_u_read(struct inode *inode, struct file *file, struct _usb_extcmd_param_rw *arg)
{
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;
	struct _usb_extcmd_param_rw param;
	int err;
	unsigned char wait_flag;
	unsigned char k_stay_flag;

	EXTCMD_API();

	// copy user area parameter to kernel area member.
	err = copy_from_user(&param, arg, sizeof(struct _usb_extcmd_param_rw));
	if(err){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if( !file ){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if( !(param.size) ){
		EXTCMD_ERR();
		return -EINVAL;
	}
	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_USER){
		EXTCMD_ERR();
		return -EINVAL;
	}
	entry = list_get_node(info->opcode, info->index);
	if(!entry){
		EXTCMD_ERR();
		return -EINVAL;
	}
	EXTCMD_INF("uread op[%02x]index[%04x] param addr[%p] size[%d]",
						info->opcode, info->index, param.data, (unsigned int)param.size);

	
	/// 1st read or continue read and kernel not write yet.
	wait_flag = test_and_set_bit_req_flag_user(
						EXTCMD_REQ_BIT_KERN_WRITE,
						EXTCMD_REQ_BIT_USER_READ,
						entry);
	if( wait_flag == EXTCMD_WAIT_READY ){
		EXTCMD_INF("uread wait..");
		enable_inquiry_state(entry, EXTCMD_FROM_KERNEL_DIRECTION);	// inquiry enable
		/// with kernel synchronize
		err = synchronize_wait_event(info->space, entry->wqh, EXTCMD_WFLAG_USER_READ,
						&entry->sync_flag, param.timeout);
		disable_inquiry_state(entry, EXTCMD_FROM_KERNEL_DIRECTION);	// inquiry disable
		clear_bit_req_flag(EXTCMD_REQ_BIT_USER_READ, entry);
		EXTCMD_INF("uread wakeup!!");
		if(err){
			EXTCMD_INF("err[%d]", err);
			entry->u_read_size	= 0;
			info->u_read_offset	= 0;
			return err;
		}
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	// transfer data buffer copy.
	err = calc_and_copy_to_user(&k_stay_flag, entry, info, &param);
	if(err){
		EXTCMD_ERR();
		entry->u_read_size	= 0;
		info->u_read_offset	= 0;
		EXTCMD_K_PRAM_LOCK_OFF(entry);
		return -EINVAL;
	}

	// copy kernel area member to user area parameter.
	err = copy_to_user(arg, &param, sizeof(struct _usb_extcmd_param_rw));
	if(err){
		EXTCMD_ERR("err[%d]", err);
		entry->u_read_size	= 0;
		info->u_read_offset	= 0;
		EXTCMD_K_PRAM_LOCK_OFF(entry);
		return -EINVAL;
	}

	EXTCMD_K_PRAM_LOCK_OFF(entry);


	if(!k_stay_flag){
		synchronize_wake_up_event(entry->wqh, EXTCMD_WFLAG_USER_READ_COPY, &entry->sync_flag);
	}
	
	return 0;
}


/**
 *	@brief		EXTCMD inquiry
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_close struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 */
static int ioctl_k_inquiry(struct inode *inode, struct file *file, struct _usb_extcmd_k_param_inquiry *param)
{
	struct extcmd_info *info;
	int result;

	EXTCMD_API();

	if(!param){
		EXTCMD_ERR();
		return -EINVAL;
	}
	else if(!file){
		EXTCMD_ERR();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	else if(!file->private_data){
		EXTCMD_ERR();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	
	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_KERNEL){
		EXTCMD_ERR();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}

	EXTCMD_INF("op[%x]idx[%x]", param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	result = get_inquiry_state(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex, param->dir);
	if(result == EXTCMD_READY){
		cancel_disable(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
		EXTCMD_SET_SENSE_NO_SENSE(param->sense);
        param->sendDataLen = get_send_data_len(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
        param->recvDataLen = get_recv_data_len(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	}
	else if(result == EXTCMD_NOT_YET_READY){
		EXTCMD_SET_SENSE_VND_BECOMING_READY(param->sense);
	}
	else{
		EXTCMD_SET_SENSE_INVALID_CMD_OP_CODE(param->sense);
	}

	return 0;
}


/**
 *	@brief		EXTCMD kernel data write
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_k_write(struct inode *inode, struct file *file, struct _usb_extcmd_k_param_rw *param)
{
	int err;
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;
	unsigned char wait_flag;

	EXTCMD_API();
	EXTCMD_K_WORK_LOCK_ON();

	if(!param){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		return -EINVAL;
	}
	else if(!file){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	else if( (!param->totalReqLen) || (!param->dataLen) ){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_INVALID_CMD_OP_CODE(param->sense);
		return 0;
	}
	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_KERNEL){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	EXTCMD_INF("kwrite op[%x]index[%x]", param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	entry = list_get_node(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	if(!entry){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	entry->k_write_param = param;
	EXTCMD_K_PRAM_LOCK_OFF(entry);

	wait_flag = test_and_set_bit_req_flag_kernel(
							EXTCMD_REQ_BIT_USER_READ,
							EXTCMD_REQ_BIT_KERN_WRITE,
							entry);
	if( wait_flag == EXTCMD_ALREADY_WAITING ){
		/// when user waiting.
		synchronize_wake_up_event(entry->wqh, EXTCMD_WFLAG_USER_READ, &entry->sync_flag);
	}

	EXTCMD_INF("kwrite wait..");
	err = synchronize_wait_event(info->space, entry->wqh, EXTCMD_WFLAG_USER_READ_COPY, &entry->sync_flag, param->timeout);
	clear_bit_req_flag(EXTCMD_REQ_BIT_KERN_WRITE, entry);
	EXTCMD_INF("kwrite wakeup!!");
	if(!err){
		EXTCMD_SET_SENSE_NO_SENSE(param->sense);
	}
	else if((err == -EPIPE) || (err == -ECANCELED)){
		EXTCMD_ERR("err=[%d]", err);
		EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(param->sense);
	}
	else if(err == -ETIME){
		EXTCMD_ERR();
		EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(param->sense);
	}
	else{
		EXTCMD_ERR("err=[%d]", err);
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	entry->k_write_param = NULL;
	EXTCMD_K_PRAM_LOCK_OFF(entry);

	EXTCMD_K_WORK_LOCK_OFF();

	return 0;
}


/**
 *	@brief		EXTCMD data read
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_k_read(struct inode *inode, struct file *file, struct _usb_extcmd_k_param_rw *param)
{
	int err;
	struct extcmd_info *info;
	struct _extcmd_cmd_list *entry;
	unsigned char wait_flag;

	EXTCMD_API();
	EXTCMD_K_WORK_LOCK_ON();

	if(!param){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		return -EINVAL;
	}
	else if(!file){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	else if( (!param->totalReqLen) || (!param->dataLen) ){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_INVALID_CMD_OP_CODE(param->sense);
		return 0;
	}
	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_KERNEL){			// user space
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}
	EXTCMD_INF("kread op[%x]index[%x]", param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	entry = list_get_node(param->cbp.cbwcb->opCode, param->cbp.cbwcb->wIndex);
	if(!entry){
		EXTCMD_ERR();
		EXTCMD_K_WORK_LOCK_OFF();
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
		return 0;
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	entry->k_read_param = param;
	EXTCMD_K_PRAM_LOCK_OFF(entry);

	wait_flag = test_and_set_bit_req_flag_kernel(
							EXTCMD_REQ_BIT_USER_WRITE,
							EXTCMD_REQ_BIT_KERN_READ,
							entry);
	if( wait_flag == EXTCMD_ALREADY_WAITING ){
		/// when user waiting.
		synchronize_wake_up_event(entry->wqh, EXTCMD_WFLAG_USER_WRITE, &entry->sync_flag);
	}

	EXTCMD_INF("kread wait..");
	err = synchronize_wait_event(info->space, entry->wqh, EXTCMD_WFLAG_USER_WRITE_COPY, &entry->sync_flag, param->timeout);
	clear_bit_req_flag(EXTCMD_REQ_BIT_KERN_READ, entry);
	EXTCMD_INF("kread wakeup!!");
	if(!err){
		EXTCMD_SET_SENSE_NO_SENSE(param->sense);
	}
	else if((err == -EPIPE) || (err == -ECANCELED)){
		EXTCMD_ERR("err=[%d]", err);
		EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(param->sense);
	}
	else if(err == -ETIME){
		EXTCMD_ERR();
		EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(param->sense);
	}
	else{
		EXTCMD_ERR("err=[%d]", err);
		EXTCMD_SET_SENSE_VND_SYSTEM_ERR(param->sense);
	}

	EXTCMD_K_PRAM_LOCK_ON(entry);
	entry->k_read_param = NULL;
	EXTCMD_K_PRAM_LOCK_OFF(entry);

	EXTCMD_K_WORK_LOCK_OFF();
	return 0;
}

/**
 *	@brief		EXTCMD request reset
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		param		usb_extcmd_param_write struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 *	@retval		-EIO		I/O error
 */
static int ioctl_k_reset(struct inode *inode, struct file *file, struct _usb_extcmd_k_param_reset *param)
{
	struct extcmd_info *info;

	EXTCMD_API();

	info = (struct extcmd_info *)file->private_data;
	if(info->space != EXTCMD_SPACE_KERNEL){			// user space
		EXTCMD_ERR();
		return -EINVAL;
	}

	synchronize_wake_up_event_all_node(EXTCMD_WFLAG_RESET);
	EXTCMD_INF("kernel wake_up_event all node");

	return 0;
}


/**
 *	@brief		extcmd_user_access_ok
 *	@param		cmd		command
 *	@param		arg		ioctl arguments
 *	@retval		0			success
 *	@retval		-EFAULT		segmentation fault
 */
int extcmd_user_access_ok(unsigned int cmd, unsigned long arg)
{
	if( _IOC_DIR(cmd) & _IOC_READ ){
		if( !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd)) )	return -EFAULT;
	}
	if( _IOC_DIR(cmd) & _IOC_WRITE ){
		if( !access_ok(VERIFY_READ,  (void __user *)arg, _IOC_SIZE(cmd)) )	return -EFAULT;
	}
	return 0;
}


/**
 *	@brief		ExtCmd ioctl
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 *	@retval		-EFAULT		segmentation fault
 *	@retval		-ENOTTY		invalid ioctl no.
 */
UDIF_ERR extcmd_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
	struct file fd;
	struct file *file = &fd;
	int err = 0;
	unsigned int cmd = param->cmd;
	unsigned long arg = param->arg;

    file->private_data = udif_file_data(filp);
//	EXTCMD_INF("extcmd_ioctl cmd=[%d]", cmd);

	// valid ioctl no?
	if( _IOC_NR(cmd) > USB_IOC_EXTCMD_NBROF ){
		EXTCMD_ERR("invalid ioctl seq. number");
		return UDIF_ERR_NOTTY;
	}
	// check extcmd ioctl command
	if( _IOC_TYPE(cmd) != USB_IOC_EXT_CMD ){
		return UDIF_ERR_NOTTY;
	}

	// ioctl for what?
	switch (cmd) {
		case USB_IOC_EXTCMD_REGISTER:
			// Check User Area Access
			err = extcmd_user_access_ok(cmd, arg);
			if(err)		return err;
			err = ioctl_u_register(NULL, file, (struct _usb_extcmd_param_register *)arg);
			break;
		case USB_IOC_EXTCMD_UNREGISTER:
			// Check User Area Access
			err = extcmd_user_access_ok(cmd, arg);
			if(err)		return err;
			err = ioctl_u_unregister(NULL, file, (struct _usb_extcmd_param_unregister *)arg);
			break;
		case USB_IOC_EXTCMD_READ:
			// Check User Area Access
			err = extcmd_user_access_ok(cmd, arg);
			if(err)		return err;
			err = ioctl_u_read(NULL, file, (struct _usb_extcmd_param_rw *)arg);
			break;
		case USB_IOC_EXTCMD_WRITE:
			// Check User Area Access
			err = extcmd_user_access_ok(cmd, arg);
			if(err)		return err;
			err = ioctl_u_write(NULL, file, (struct _usb_extcmd_param_rw *)arg);
			break;
		case USB_IOC_EXTCMD_CANCEL:
			// Check User Area Access
			err = extcmd_user_access_ok(cmd, arg);
			if(err)		return err;
			err = ioctl_u_cancel(NULL, file, (struct _usb_extcmd_param_cancel *)arg);
			break;

		case USB_IOC_EXTCMD_K_INQUIRY:
			err = ioctl_k_inquiry(NULL, file, (struct _usb_extcmd_k_param_inquiry *)arg);
			break;
		case USB_IOC_EXTCMD_K_READ:
			err = ioctl_k_read(NULL, file, (struct _usb_extcmd_k_param_rw *)arg);
			break;
		case USB_IOC_EXTCMD_K_WRITE:
			err = ioctl_k_write(NULL, file, (struct _usb_extcmd_k_param_rw *)arg);
			break;
		case USB_IOC_EXTCMD_K_RESET:
			err = ioctl_k_reset(NULL, file, (struct _usb_extcmd_k_param_reset *)arg);
			break;
		default:
			EXTCMD_ERR("invalid ioctl seq. number");
			err = UDIF_ERR_NOTTY;
			break;
	}
	
    udif_file_data(filp) = file->private_data;
	return (UDIF_ERR)err;
}
