/*
 *  aspmctrl.c
 *
 *  Copyright 2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "pcie_dma.h"

#ifdef ER9001359142
#include <linux/kthread.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <mach/pcie_export.h>
#include <asm/io.h>
#include <asm/unistd.h>
#include <asm/uaccess.h>
#include "regs.h"
#include "aspmctrl.h"

#define L0GRP_MIN  1   /* lowest CH# which the errata affected */
#undef FORCE_ASPM_CTRL

static int aspmd_prio = ASPMD_PRIO;
module_param_named(prio, aspmd_prio, int, S_IRUSR|S_IWUSR);

static int polling_timeout = POLLING_TIMEOUT;
module_param_named(tmo, polling_timeout, int, S_IRUSR|S_IWUSR);

static inline int need_aspm_ctrl(struct dma_info_t *chan)
{
	return (chan->pci_info->aspm.use_aspm && chan->id >= L0GRP_MIN);
}

static void show_info(struct pci_info_t *p)
{
	int i;
	uint32_t dat;

	for (i = 0; i < 64; i++) {
		if (i % 4 == 0) {
			printk(KERN_ERR "PCI%d:RC:", p->id);
		}
		dat = pcie_readl(p->ctl + i * 4);
		printk(KERN_CONT "%08x ", dat);
		if (i % 4 == 3) {
			printk(KERN_CONT "\n");
		}
	}

	for (i = 0; i < 64; i++) {
		if (i % 4 == 0) {
			printk(KERN_ERR "PCI%d:EP:", p->id);
		}
		pci_read_config_dword(p->aspm.dev, i * 4, &dat);
		printk(KERN_CONT "%08x ", dat);
		if (i % 4 == 3) {
			printk(KERN_CONT "\n");
		}
	}
}

static void aspmctl_exit_l1_async(struct pci_info_t *p);

void aspmctl_request(struct dma_info_t *chan)
{
	struct pci_info_t *p;
	unsigned long flags;

	if (!need_aspm_ctrl(chan))
		return;
	p = chan->pci_info;
	LOCK(p);
	if (!p->aspm.req_count++) { /* 0-->1 */
		unsigned long flags;

		LOCK(chan);
		if (!chan->ready && chan->state) {
			aspmctl_exit_l1_async(p);
		}
		UNLOCK(chan);
	}
	UNLOCK(p);
}

static void aspm_restore(struct pci_info_t *p);

void aspmctl_relinquish(struct dma_info_t *chan)
{
	struct pci_info_t *p;
	int i;
	struct dma_info_t *c;
	unsigned long flags;

	if (!need_aspm_ctrl(chan))
		return;
	p = chan->pci_info;
	LOCK(p);
	if (unlikely(p->aspm.req_count <= 0)) {
		printk(KERN_ERR "ERROR:PCI%d:%s:req_count=%d\n",
		       p->id, __func__, p->aspm.req_count);
		BUG();
	} else if (!--p->aspm.req_count) {
		chanlog(chan, CHLOG_RESTORE_B, 0);
		/* deactivate L0GRP */
		for (i = L0GRP_MIN, c = &p->dma_info[L0GRP_MIN]; i < PCIDMA_CH_MAX; i++, c++) {
			unsigned long flags;

			LOCK(c);
			c->ready = 0;
			UNLOCK(c);
		}
		/* wake up aspmd */
		p->aspm.aspmd_done = 1;
		udif_wake_up(&p->aspm.aspmd_wait2);
		chanlog(chan, CHLOG_RESTORE_E, 0);
	}
	UNLOCK(p);
}

static void aspmctl_exit_l1_async(struct pci_info_t *p)
{
	p->aspm.active = 1;
	p->aspm.aspmd_req = 1;
	udif_wake_up(&p->aspm.aspmd_wait);
}

static void aspm_exit_l1_begin(struct pci_info_t *p)
{
	uint16_t dat;
	uint32_t tmp32;
#ifdef PCIEDMA_DEBUG
	char buf[64];
#endif
	/* save Link Control */
	dat = pcie_readw(p->ctl + LINK_CTL);
#ifdef PCIEDMA_DEBUG
	scnprintf(buf, sizeof buf, "B PCI%d:aspm_EXIT_L1 %x", p->id, dat);
	BOOT_TIME_ADD1(buf);
#endif
	p->aspm.save = dat;
	dat &= ~PCI_EXP_LNKCTL_ASPM_L1;

	tmp32 = pcie_readl(p->phy + APP_PM_REQ);
	tmp32 |= APP_XFER_PENDING;
	pcie_writel(p->phy + APP_PM_REQ, tmp32);
	/* RC */
	pcie_writew(p->ctl + LINK_CTL, dat);
	pcie_wmb();

	/* EP */
	pcie_dwc_write_config(p->aspm.dev, p->aspm.linkctl, 1, dat);
	pcie_wmb();
}

static inline void stlog_init(struct stlog *p)
{
	p->put = p->cnt = 0;
}

static inline void stlog_put(struct stlog *p, unsigned long t, uint32_t data)
{
	if (p->cnt > 0  &&  p->data[p->put].state == data) {
		/* update */
		p->data[p->put].t2 = t;
		return;
	}
	if (p->cnt) {
		if (++p->put == STLOG_SIZE) {
			p->put = 0;
		}
	}
	if (p->cnt < STLOG_SIZE) {
		p->cnt++;
	}
	p->data[p->put].t1 = p->data[p->put].t2 = t;
	p->data[p->put].state = data;
}

static void stlog_dump(struct stlog *p)
{
	int get, i;

	get = 0;
	if (STLOG_SIZE == p->cnt) {
		get = p->put + 1;
		if (STLOG_SIZE == get)
			get = 0;
	}
	for (i = 0; i < p->cnt; i++) {
		printk(KERN_ERR "\t%llu-%llu: %08x\n",
		       udif_cycles_to_usecs(p->data[get].t1),
		       udif_cycles_to_usecs(p->data[get].t2),
		       p->data[get].state);
		if (++get == STLOG_SIZE)
			get = 0;
	}
}

static void show_timeout(struct pci_info_t *p, unsigned long long t1)
{
	int i;

	BOOT_TIME_ADD1("L0timeout");
	printk(KERN_ERR "ERROR:PCI%d:wait for L0 timeout(t1=%lld)\n",
	       p->id, udif_cycles_to_usecs(t1));
	/* LTSSM */
	printk(KERN_ERR "PCI%d:LTSSM:\n", p->id);
	stlog_dump(&p->aspm.ltssmlog);

	show_info(p);

	for (i = 0; i < 3; i++) {
		printk(KERN_ERR "PCI%d:LTSSM: %08x\n",
		       p->id, pcie_readl(p->phy + LTSSM_STATE));
		udif_udelay(10);
	}
}

static int is_l0(struct pci_info_t *p)
{
	uint32_t val, ltssm;
	unsigned long t;
	unsigned long flags;

	local_irq_save(flags);
	val = pcie_readl(p->phy + LTSSM_STATE);
	t = udif_read_freerun();
	local_irq_restore(flags);
	ltssm = (val >> LTSSM_SHIFT) & LTSSM_MASK;
	stlog_put(&p->aspm.ltssmlog, t, val);
	return (S_L0 == ltssm || S_L0S == ltssm);
}

static void aspm_l0_polling(struct pci_info_t *p)
{
	unsigned long long timeout, t1;
	int ready = 0;
	uint32_t tmp32;
#ifdef PCIEDMA_DEBUG
	char buf[64];
#endif

	stlog_init(&p->aspm.ltssmlog);
	t1 = udif_read_freerun();
	timeout = t1 + udif_usecs_to_cycles(polling_timeout);
	udif_udelay(POLLING_START_DELAY);
	do {
		ready = is_l0(p);
		if (ready)
			break;
		udif_udelay(POLLING_PERIOD);
	} while (udif_read_freerun() < timeout);
	if (!ready) {
	ready = is_l0(p);
	}
	if (!ready) {
		show_timeout(p, t1);
		log_stop();
		logdump(p);
		BUG();
		return;
	}
#ifdef PCIEDMA_DEBUG
	t1 = udif_read_freerun() - t1;
	t1 = udif_cycles_to_usecs(t1);
	scnprintf(buf, sizeof buf, "  PCI%d:poll:%lld", p->id, t1);
	BOOT_TIME_ADD1(buf);
#endif /* PCIEDMA_DEBUG */

	tmp32 = pcie_readl(p->phy + APP_PM_REQ);
	tmp32 &= ~APP_XFER_PENDING;
	pcie_writel(p->phy + APP_PM_REQ, tmp32);
}

static void aspm_exit_l1_finish(struct pci_info_t *p)
{
	unsigned long flags;
	int i;
	struct dma_info_t *chan;
#ifdef PCIEDMA_DEBUG
	char buf[64];
	scnprintf(buf, sizeof buf, "E PCI%d:aspm_EXIT_L1", p->id);
	BOOT_TIME_ADD1(buf);
#endif
	/* start L0GRP */
	LOCK(p);
	for (i = L0GRP_MIN, chan = &p->dma_info[L0GRP_MIN]; i < PCIDMA_CH_MAX; i++, chan++) {
		unsigned long flags;

		LOCK(chan);
		chan->ready = 1;
		pcidma_start(chan);
		UNLOCK(chan);
 	}
	UNLOCK(p);
}

static int aspmd(void *data)
{
	struct pci_info_t *p = (struct pci_info_t *)data;
#ifdef PCIEDMA_DEBUG
	char buf[64];
	scnprintf(buf, sizeof buf, "PCI%d:%s", p->id, __func__);
#endif

	for (;;) {
		udif_wait_event(&p->aspm.aspmd_wait,
				p->aspm.aspmd_req || kthread_should_stop());
		if (kthread_should_stop())
			break;
#ifdef PCIEDMA_DEBUG
		BOOT_TIME_ADD1(buf);
#endif
		p->aspm.aspmd_req = 0; /* clear request */
		pcie_aspm_ctrl_lock(p->id);
		log_put(&p->log, "0", 2);
		aspm_exit_l1_begin(p);
		aspm_l0_polling(p);
		log_put(&p->log, "2", 2);
		aspm_exit_l1_finish(p);
		log_put(&p->log, "1", 2);

		udif_wait_event(&p->aspm.aspmd_wait2, p->aspm.aspmd_done);
		p->aspm.aspmd_done = 0; /* clear event */
		log_put(&p->log, "3", 2);
		aspm_restore(p);
		log_put(&p->log, "4", 2);
		pcie_aspm_ctrl_unlock(p->id);
		p->aspm.active = 0;
	}
	return 0;
}

static void aspm_restore(struct pci_info_t *p)
{
	/* RC */
	pcie_writew(p->ctl + LINK_CTL, p->aspm.save);
	pcie_mb();
	/* EP */
	pcie_dwc_write_config(p->aspm.dev, p->aspm.linkctl, 1, p->aspm.save);
	pcie_mb();
}

static int aspmd_init(struct pci_info_t *p)
{
	struct task_struct *tsk;
	struct sched_param param = { .sched_priority = aspmd_prio };

	p->aspm.aspmd = NULL;
	tsk = kthread_create(aspmd, p, "pcidma%d", p->id);
	if (IS_ERR(tsk)) {
		printk(KERN_ERR "ERROR:%s:can not create kthread\n", __func__);
		return UDIF_ERR_NOMEM;
	}
	sched_setscheduler(tsk, SCHED_FIFO, &param);
	p->aspm.aspmd = tsk;
	p->aspm.aspmd_req = 0;
	udif_wait_init(&p->aspm.aspmd_wait);
	p->aspm.aspmd_done = 0;
	udif_wait_init(&p->aspm.aspmd_wait2);
	wake_up_process(tsk);
	return UDIF_ERR_OK;
}

static void aspmd_exit(struct pci_info_t *p)
{
	if (p->aspm.aspmd) {
		kthread_stop(p->aspm.aspmd);
		p->aspm.aspmd = NULL;
	}
}

void aspmctl_chan_init(struct dma_info_t *chan)
{
	chan->ready = (need_aspm_ctrl(chan)) ? 0 : 1;
}

void aspmctl_shutdown(struct pci_info_t *p)
{
	unsigned long flags;

	LOCK(p);
	if (p->aspm.req_count != 0) {
		printk(KERN_ERR "ERROR:PCI%d:%s:req_count=%d\n",
		       p->id, __func__, p->aspm.req_count);
		p->aspm.req_count = 0;
		BUG();
	}
	if (p->aspm.active) {
		printk(KERN_ERR "ERROR:PCI%d:%s:active=%d\n",
		       p->id, __func__, p->aspm.active);
		BUG();
	}
	pci_dev_put(p->aspm.dev);
	p->aspm.dev = NULL;
	UNLOCK(p);
}

void aspmctl_connect(struct pci_info_t *p)
{
	unsigned long flags;

	p->aspm.dev = pci_get_domain_bus_and_slot(p->id, 1, 0);
	if (!p->aspm.dev) {
		printk(KERN_ERR "ERROR:PCI%d:%s: pci_get_domain_bus_and_slot failed\n", p->id, __func__);
		BUG();
	}
	p->aspm.linkctl = pci_pcie_cap(p->aspm.dev) + PCI_EXP_LNKCTL;

	LOCK(p);
	if (p->aspm.req_count) {
		aspmctl_exit_l1_async(p);
	}
	UNLOCK(p);
}

int aspmctl_init(struct pci_info_t *p)
{
	int ret = 0;

	p->aspm.use_aspm = (pcie_aspm_policy(p->id) == 3);
# ifdef FORCE_ASPM_CTRL
	p->aspm.use_aspm = 1;
# endif
	p->aspm.req_count = 0;
	p->aspm.active = 0;
	p->aspm.dev = NULL;
	if (p->aspm.use_aspm) {
		ret = aspmd_init(p);
		if (ret)
			return ret;
	}
	return ret;
}

void aspmctl_exit(struct pci_info_t *p)
{
	aspmd_exit(p);
}
#endif /* ER9001359142 */
