/**
 * Copyright 2018 Sony Imaging Products & Solutions Inc
 * Copyright 2018 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2  of
 * the License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/clk.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/pm_runtime.h>
#include <linux/reset.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/usb/f_usb/fusb_dwc3.h>
#include <linux/gpio/gpio.h>
#include "dwc3-of-cxd.h"
#include "cxd-phy.h"
struct dwc3_of_cxd {
	struct device		*dev;
	struct clk		**clks;
	int			num_clocks;
	struct reset_control	**resets;
	int			num_resets;
};

#define MODULE_NAME     "dwc3-of-cxd"
#define DWC3_OF_CXD_DEFAULT_AUTOSUSPEND_DELAY    500

static int dwc3_of_cxd_clk_init(struct dwc3_of_cxd *cxd, int count)
{
	struct device		*dev = cxd->dev;
	struct device_node	*np = dev->of_node;
	int			i;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
	if(count < 0){
		dev_err(dev,"Invalid argument clk count\n");
		return -EINVAL;
	}

	cxd->num_clocks = count;

	if (!count)
		return 0;

	cxd->clks = devm_kcalloc(dev, cxd->num_clocks,
			sizeof(struct clk *), GFP_KERNEL);
	if (!cxd->clks)
		return -ENOMEM;

	for (i = 0; i < cxd->num_clocks; i++) {
		struct clk	*clk;
		int		ret;

		clk = of_clk_get(np, i);
		if (IS_ERR(clk)) {
			while (--i >= 0) {
				clk_disable_unprepare(cxd->clks[i]);
				clk_put(cxd->clks[i]);
			}
			return PTR_ERR(clk);
		}
		ret = clk_prepare_enable(clk);
		if (ret < 0) {
			while (--i >= 0) {
				clk_disable_unprepare(cxd->clks[i]);
				clk_put(cxd->clks[i]);
			}
			clk_put(clk);

			return ret;
		}
		cxd->clks[i] = clk;
	}
	return 0;
}
#ifdef CONFIG_ARCH_CXD900XX_FPGA
static int dwc3_of_cxd_get_resets(struct dwc3_of_cxd *cxd, struct device_node *np)
{
	struct reset_control *rst;
	struct device   *dev = cxd->dev;
	unsigned int i, count;
	int err;
	//dev_err(dev,"dwc3_of_cxd_get_resets \n\n\n");
	count = of_count_phandle_with_args(np, "resets", "#reset-cells");
	if (count == -ENOENT){///* No such file or directory */
		count = 0;
		dev_err(dev,"can't find resets in devicetree\n");
		return count;
	}

	cxd->resets = kcalloc(count, sizeof(rst), GFP_KERNEL);
	if (!cxd->resets){
		dev_err(dev,"allocate resets memory fail\n");
		return -ENOMEM;
	}
	for (i = 0; i < count; i++) {
		cxd->resets[i] = of_reset_control_get_by_index(np, i);
		if (IS_ERR(cxd->resets[i])) {
			err = PTR_ERR(cxd->resets[i]);
			dev_err(dev,"get resets index %d fail\n", i);
			//goto error;
		}
	}
	cxd->num_resets = count;
	return 0;
error:
	while (i--)
		reset_control_put(cxd->resets[i]);
	kfree(cxd->resets);
	return err;
}

static int dwc3_cxd_reset_assert(struct dwc3_of_cxd *cxd)
{
	unsigned int i;

	for (i = 0; i < cxd->num_resets; i++) {
		reset_control_assert(cxd->resets[i]);
	}
	return 0;
}

static int dwc3_cxd_reset_deassert(struct dwc3_of_cxd *cxd)
{
	unsigned int i;

	for (i = 0; i < cxd->num_resets; i++) {
		reset_control_deassert(cxd->resets[i]);
	}
	return 0;
}
#else
static int dwc3_of_cxd_get_resets(struct dwc3_of_cxd *cxd, struct device_node *np)
{
	struct reset_control *rst;
	struct device   *dev = cxd->dev;
	unsigned int i, count;
	int err;

	count = of_count_phandle_with_args(np, "resets", "#reset-cells");
	if (count == -ENOENT){///* No such file or directory */
		dev_err(dev,"can't find resets in devicetree\n");
		count = 0;
		return count;
	}

	cxd->resets = kcalloc(count, sizeof(rst), GFP_KERNEL);
	if (!cxd->resets){
		dev_err(dev,"allocate resets memory fail\n");
		return -ENOMEM;
	}

	for (i = 0; i < count; i++) {
		cxd->resets[i] = of_reset_control_get_by_index(np, i);
		if (IS_ERR(cxd->resets[i])) {
			err = PTR_ERR(cxd->resets[i]);
			dev_err(dev,"get resets index %d fail\n", i);
			goto error;
		}
	}

	cxd->num_resets = count;
	return 0;
error:
	while (i--)
		reset_control_put(cxd->resets[i]);

	kfree(cxd->resets);
	return err;
}

static int dwc3_cxd_reset_assert(struct dwc3_of_cxd *cxd)
{
	unsigned int i;
	struct device   *dev = cxd->dev;
	int err;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
	for (i = 0; i < cxd->num_resets; i++) {
		err = reset_control_assert(cxd->resets[i]);
		if (err){
			dev_err(dev,"assert resets index %d fail\n", i);
			return err;
		}
	}

	return 0;
}

static int _dwc3_cxd_reset_deassert_all(struct dwc3_of_cxd *cxd)
{
	unsigned int i;
	struct device *dev = cxd->dev;
	int err;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
	for (i = 0; i < cxd->num_resets; i++) {
		err = reset_control_deassert(cxd->resets[i]);
		if (err){
			dev_err(dev,"deassert resets index %d fail\n", i);
			return err;
		}
	}
	return 0;
}
static int dwc3_cxd_reset_deassert(struct dwc3_of_cxd *cxd, const char *id)
{
	struct device *dev = cxd->dev;
	int err = 0;

	if (id) {
		/*
		 * shared flag of reset_control by __of_reset_control_get() is false
		 * so it fails then we get index
		 */
		int index = of_property_match_string(dev->of_node, "reset-names", id);
		/*printk(KERN_INFO "%s reset id \"%s\", resets[%d]=%p\n", __FUNCTION__, id, index, cxd->resets[index]);*/
		if (0 > index) {
			dev_err(dev,"cannot get reset control, id=\"%s\", index=%d\n", id, index);
			return index;
		}
		err = reset_control_deassert(cxd->resets[index]);
		if (err)
			dev_err(dev,"deassert reset id \"%s\" fail\n", id);
		/*else printk(KERN_INFO "%s reset id \"%s\"\n", __FUNCTION__, id);*/
	}
	else
		err = _dwc3_cxd_reset_deassert_all(cxd);

	return err;
}
#endif
static void __iomem *u31phy_top;
void __iomem *dwc3_u31phy_top(void)
{
	return u31phy_top;
}
static void __iomem *crphy_top;
void __iomem *dwc3_crphy_top(void)
{
	return crphy_top;
}

#define U31PHY_CFGR0               0x0020
#define U31PHY_CFGR0_PHY_REF_BIT   (1 << 0)
#define U31PHY_CFGR0_PHY_REF_PAD   (0)
#define U31PHY_CFGR0_CR_PARA_SEL_BIT        (1<<20)
#define U31PHY_CFGR0_CR_PARA_SEL_VAL        U31PHY_CFGR0_CR_PARA_SEL_BIT

#define U31PHY_CR_OFFSET 0x20000
#define LANEN_ANA_TX_ATB1_1 0x470c //0x1NC3, N = 1
#define LANEN_ANA_TX_ATB1_2 0x4b0c //0x1NC3, N = 2
#define LANEN_ANA_TX_ATB2_1 0x4710 //0x1NC4, N = 1
#define LANEN_ANA_TX_ATB2_2 0x4b10 //0x1NC4, N = 2
#define atb_gd  (1<<0)
#define atb_vcm (1<<0)

#define LANEN_ANA_TX_VBOOST_1 0x4714 //0x1NC5, N = 1
#define LANEN_ANA_TX_VBOOST_2 0x4b14 //0x1NC5, N = 2
#define atb_s_enable  (1<<1)

#define SUP_ANA_SWITCH_PWR_MEAS 0x12c//0x4B
#define atb_sw_gd (1<<4)
#define atb_sw	  (1<<6)

#define GPIOINT_BASE 0xf102e000
#define GPIOINT_LEN 0x1000
#define GPIOINT_WDATASET 0x44
#define GPIOINT_WDATACLR 0x48
#define GPIOINT_WDATASET_bit24 (1 << 24)
#define GPIOINT_WDATACLR_bit24 (1 << 24)
#define GPIOINT_DIRSET 0x4
#define GPIOINT_PORTSET 0x34
#define GPIOINT_DIRSET_bit24 (1 << 24)
#define GPIOINT_PORTSET_bit24 (1 << 24)

void cr_phy31_wa(void)
{
	u32 value;
	void __iomem *phy_cr_addr = dwc3_crphy_top(); //oxf10a0000
#ifdef DWC3_PHY31_TD411_ENA
	static void __iomem *gpio_base_mem;
#endif

	//solve RX Low-z issue and compatibility issue, some device can't be recognized
	//ex: transcend's mass storage can't be recognized
	//printk("set 0x1NC3, 1NC4, 1NC5, 0x4b value\n");
	//(1) set 0x1NC3[0]=1, 0x1NC4[0]=1 ,N = 1,2
	//    0xf10a470c[0] = 1, 0xf10a4b0c[0] = 1, 0xf10a4710[0] = 1, 0xf10a4b10[0] = 1
	value = readl(phy_cr_addr + LANEN_ANA_TX_ATB1_1) | atb_gd;
	writel(value, phy_cr_addr + LANEN_ANA_TX_ATB1_1);
	value = readl(phy_cr_addr + LANEN_ANA_TX_ATB1_2) | atb_gd;
	writel(value, phy_cr_addr + LANEN_ANA_TX_ATB1_2);
	value = readl(phy_cr_addr + LANEN_ANA_TX_ATB2_1) | atb_vcm;
	writel(value, phy_cr_addr + LANEN_ANA_TX_ATB2_1);
	value = readl(phy_cr_addr + LANEN_ANA_TX_ATB2_2) | atb_vcm;
	writel(value, phy_cr_addr + LANEN_ANA_TX_ATB2_2);
	
	//printk("LANEN_ANA_TX_ATB1_1: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_ATB1_1));
	//printk("LANEN_ANA_TX_ATB1_2: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_ATB1_2));
	//printk("LANEN_ANA_TX_ATB2_1: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_ATB2_1));
	//printk("LANEN_ANA_TX_ATB2_2: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_ATB2_2));
	
	//(2) set 0x1NC5[1] = 1 ,N = 1,2
	//    0xf10a4714[1] = 1, 0xf10a4b14[1] = 1,
	value = readl(phy_cr_addr + LANEN_ANA_TX_VBOOST_1) | atb_s_enable;
	writel(value, phy_cr_addr + LANEN_ANA_TX_VBOOST_1);
	value = readl(phy_cr_addr + LANEN_ANA_TX_VBOOST_2) | atb_s_enable;
	writel(value, phy_cr_addr + LANEN_ANA_TX_VBOOST_2);
	
	//printk("LANEN_ANA_TX_VBOOST_1: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_VBOOST_1));
	//printk("LANEN_ANA_TX_VBOOST_2: 0x%X\n", readl(phy_cr_addr + LANEN_ANA_TX_VBOOST_2));
	
	//(3) set 0x4b[4] and 0x4b[6] as 1
	// 0xf10a012C[4] = 1, oxf10a012c[6] = 1
	value = readl(phy_cr_addr + SUP_ANA_SWITCH_PWR_MEAS) | atb_sw_gd | atb_sw ;
	writel(value, phy_cr_addr + SUP_ANA_SWITCH_PWR_MEAS);
	
	//printk("SUP_ANA_SWITCH_PWR_MEAS: 0x%X\n", readl(phy_cr_addr + SUP_ANA_SWITCH_PWR_MEAS));

#ifdef DWC3_PHY31_TD411_ENA
	//setup GPIO SET register for enabling switch
	//Workaround to avoid TD4.1.1 fail atapower off condition, ref USB_Application_Note
	gpio_base_mem = ioremap(GPIOINT_BASE, GPIOINT_LEN);
	if(!gpio_base_mem)
	{
		printk("%s, gpio ioremap fail\n", __func__);
		goto release_region;
	}

	value = readl(gpio_base_mem + GPIOINT_WDATACLR) | GPIOINT_WDATACLR_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_WDATACLR);
	value = readl(gpio_base_mem + GPIOINT_PORTSET) | GPIOINT_PORTSET_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_PORTSET);
	value = readl(gpio_base_mem + GPIOINT_DIRSET) | GPIOINT_DIRSET_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_DIRSET);

	value = readl(gpio_base_mem + GPIOINT_WDATASET ) | GPIOINT_WDATASET_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_WDATASET);
	iounmap(gpio_base_mem);

release_region:
	release_mem_region(GPIOINT_BASE, GPIOINT_LEN);
#endif
}

void cr_phy31_wa_clear(void)
{
#ifdef DWC3_PHY31_TD411_ENA
	u32 value;
	static void __iomem *gpio_base_mem;

	//GPIO CLEAR register for disabling switch
	//Workaround to avoid TD4.1.1 fail atapower off condition, ref USB_Application_Note, "Stop sequence(USB3 device)" sheet
	gpio_base_mem = ioremap(GPIOINT_BASE, GPIOINT_LEN);
	if(!gpio_base_mem)
	{
		printk("%s, gpio ioremap fail\n", __func__);
		goto release_region;
	}

	value = readl(gpio_base_mem + GPIOINT_WDATACLR) | GPIOINT_WDATACLR_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_WDATACLR);
	value = readl(gpio_base_mem + GPIOINT_PORTSET) | GPIOINT_PORTSET_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_PORTSET);
	value = readl(gpio_base_mem + GPIOINT_DIRSET) | GPIOINT_DIRSET_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_DIRSET);

	value = readl(gpio_base_mem + GPIOINT_WDATACLR ) | GPIOINT_WDATACLR_bit24 ;
	writel(value, gpio_base_mem + GPIOINT_WDATACLR);
	iounmap(gpio_base_mem);

release_region:
	release_mem_region(GPIOINT_BASE, GPIOINT_LEN);
#endif
}

static int dwc3_of_cxd_usbphy_init(struct dwc3_of_cxd *cxd, const void *data, const void *port)
{
	int ret, i;

	for (i = 0; i < cxd->num_clocks; i++) {
		if (__clk_is_enabled(cxd->clks[i]))
			continue;
		ret = clk_enable(cxd->clks[i]);
		if (ret < 0) {
			while (--i >= 0)
				clk_disable(cxd->clks[i]);
			return ret;
		}
	}

	/* doing better than cure */
	ret = dwc3_cxd_reset_assert(cxd);
	if (ret) dev_err(cxd->dev, "%s dwc3_cxd_reset_assert err=%d\n", __FUNCTION__, ret);
	udelay(11);

	/* deassert only apb here. see dts to find the name */
	ret = dwc3_cxd_reset_deassert(cxd, "rst_apb");
	if (ret) {
		dev_err(cxd->dev, "%s dwc3_cxd_reset_deassert \"rst_apb\" err=%d\n", __FUNCTION__, ret);
		return ret;
	}

	cr_phy31_adjust_init();
	if (data && port) {
		ret = u2phy_adjust(u31phy_top, data, port);
		if (ret)
			return ret;
		setup_cxd_u31phy();
		enable_cr_para_sel(u31phy_top);
		ret = dwc3_cxd_reset_deassert(cxd, NULL);
		if (ret) {
			dev_err(cxd->dev, "%s dwc3_cxd_reset_deassert all err=%d\n", __FUNCTION__, ret);
			return ret;
		}
		ret = check_cr_sram_init_done(u31phy_top);
		if (ret)
			return ret;

		download_snps_u3_phy_fw();

		//<Synopsys workaround> : U3PHY, RX Low-Z, remove Tx residual voltage at TXRX-to-GND
		cr_phy31_wa();

		ret = cr_phy31_adjust(crphy_top, data);
		if (ret)
			return ret;
#ifdef CONFIG_USB_DWC3_OF_CXD_PHY_ADJUST_DEBUG
		/*
		 *  dump crphy registers before cr_para_sel_done
		 */
		snapshot_phy_param(u31phy_top, crphy_top);
#endif
		wmb();
		cr_para_sel_done(u31phy_top);
	}
	else {
		setup_cxd_u31phy();//<2>Setup reference clock source of USB3.1 PHY

		//<3>setup USB subsystem register to disable USB3.1 PHY SRAM bypass mode
		// and setting CR I/F selection to 1
		ret = enable_cr_para_sel(u31phy_top);
		if (ret){
			dev_err(cxd->dev, "%s enable_cr_para_sel err=%d\n", __FUNCTION__, ret);
			return ret;
		}

		//<4> release USB related reset: USB_U31PHY
		ret = dwc3_cxd_reset_deassert(cxd, NULL);
		if (ret){
			dev_err(cxd->dev, "%s dwc3_cxd_reset_deassert all err=%d\n", __FUNCTION__, ret);
			return ret;
		}

		//<5> wait phy0_sram_init_done
		ret = check_cr_sram_init_done(u31phy_top);
		if (ret)
			return ret;

		download_snps_u3_phy_fw();

		//<Synopsys workaround> : U3PHY, RX Low-Z, remove Tx residual voltage at TXRX-to-GND
		//cr_phy31_wa(); // if without U3PHY Adjust in StartSequence, no need wa()

		wmb();
		//<6> let U3.1PHY execute the instruction code from the SRAM
		cr_para_sel_done(u31phy_top);
  	}
	return 0;
}

static int dwc3_of_cxd_set_phy(struct platform_device *pdev, const void *data, const void *port)
{
	struct dwc3_of_cxd	*cxd = platform_get_drvdata(pdev);
	int			i, ret;

	//cr_phy31_wa_clear();

	disable_cr_para_sel(u31phy_top);
	dwc3_cxd_reset_assert(cxd);
	udelay(11);
	for (i = 0; i < cxd->num_clocks; i++) {
		while (__clk_is_enabled(cxd->clks[i]))
			clk_disable(cxd->clks[i]);
	}
	msleep(100);

	/* usb phy initialization */
	ret = dwc3_of_cxd_usbphy_init(cxd, data, port);

	return ret;
}

#ifdef CONFIG_USB_DWC3_OF_CXD_PHY_ADJUST_DEBUG
static ssize_t dwc3_of_cxd_phy_param_show(
    struct class *class,
    struct class_attribute *attr,
    char *buf)
{
	return read_snapshot_phy_param(buf);
}
CLASS_ATTR(phy_param, 0444, dwc3_of_cxd_phy_param_show, NULL);

static struct class dwc3_of_cxd_class = {
	.name = MODULE_NAME,
};
#endif

struct usb_otg_dwc3_of_cxd_ops fusb_dwc3_of_cxd_ops;

static int dwc3_of_cxd_probe(struct platform_device *pdev)
{
	struct dwc3_of_cxd	*cxd;
	struct device		*dev = &pdev->dev;
	struct device_node	*np = dev->of_node;

	int			ret;
	int			i;
	struct resource *res;

	u31phy_top = NULL;
	crphy_top = NULL;
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		dev_err(dev, "%s platform_get_resource u31phy fail\n", __FUNCTION__);
	else {
		u31phy_top = devm_ioremap(&pdev->dev, res->start, res->end-res->start+1);
		if (IS_ERR(u31phy_top)) {
			dev_err(dev, "%s devm_ioremap err=%p\n", __FUNCTION__, u31phy_top);
			u31phy_top = NULL;
		}
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res)
		dev_err(dev, "%s platform_get_resource crphy fail\n", __FUNCTION__);
	else {
		crphy_top = devm_ioremap(&pdev->dev, res->start, res->end-res->start+1);
		if (IS_ERR(u31phy_top)) {
			dev_err(dev, "%s devm_ioremap err=%p\n", __FUNCTION__, u31phy_top);
			crphy_top = NULL;
		}
	}

	cxd = devm_kzalloc(dev, sizeof(*cxd), GFP_KERNEL);
	if (!cxd)
		return -ENOMEM;

	platform_set_drvdata(pdev, cxd);
	cxd->dev = dev;

#ifdef CONFIG_USB_DWC3_OF_CXD_PHY_ADJUST_DEBUG
	ret = class_register(&dwc3_of_cxd_class);
	if (ret)
		dev_err(dev, "class_register err=%d\n", ret);
	else {
		ret = class_create_file(&dwc3_of_cxd_class, &class_attr_phy_param);
		if (ret)
			dev_err(dev, "class_create_file err=%d\n", ret);
	}
#endif

	cr_phy31_wa_clear();

	/* Enable USB clock/reset */
	ret = dwc3_of_cxd_clk_init(cxd, of_count_phandle_with_args(np,
	                                        "clocks", "#clock-cells"));
	if (ret){
		dev_err(dev, "failed to initiate CXD USB3 Clock....err=%d\n", ret);
		return ret;
	}

	/* get reset information */
	ret = dwc3_of_cxd_get_resets(cxd, np);
	if (ret)
		return ret;

	/* usb phy initialization */
	ret = dwc3_of_cxd_usbphy_init(cxd, NULL, NULL);
	if (ret)
		goto err_clk;

	ret = of_platform_populate(np, NULL, NULL, dev);
	if (ret) {
		for (i = 0; i < cxd->num_clocks; i++) {
			clk_disable_unprepare(cxd->clks[i]);
			clk_put(cxd->clks[i]);
		}

		goto err_resetc_assert;
	}

	pm_runtime_set_active(dev);
	pm_runtime_use_autosuspend(dev);
	pm_runtime_set_autosuspend_delay(dev, DWC3_OF_CXD_DEFAULT_AUTOSUSPEND_DELAY);
	pm_runtime_enable(dev);
	ret = pm_runtime_get_sync(dev);
	if (0 > ret) {
		dev_err(dev, "pm_runtime_get_sync err=%d\n", ret);
		goto err_runtime_pm;
	}
	pm_runtime_put(dev);

	fusb_dwc3_of_cxd_ops.pdev = pdev;
	fusb_dwc3_of_cxd_ops.set_phy = dwc3_of_cxd_set_phy;
	fusb_otg_dwc3_of_cxd_bind(&fusb_dwc3_of_cxd_ops);

	return 0;

err_runtime_pm:
	pm_runtime_put_sync(dev);
	pm_runtime_disable(dev);

err_resetc_assert:
        dwc3_cxd_reset_assert(cxd);

err_clk:
	return ret;
}

static int dwc3_of_cxd_remove(struct platform_device *pdev)
{
	struct dwc3_of_cxd	*cxd = platform_get_drvdata(pdev);
	struct device		*dev = &pdev->dev;
	int			i;

	of_platform_depopulate(dev);

	//According USB_Application_Note "Stop sequence(USB3 device)" sheet
	//reset assert
	dwc3_cxd_reset_assert(cxd);
	while (cxd->num_resets--)
		reset_control_put(cxd->resets[cxd->num_resets]);

	//clock disable
	for (i = 0; i < cxd->num_clocks; i++) {
		clk_disable_unprepare(cxd->clks[i]);
		clk_put(cxd->clks[i]);
	}

	pm_runtime_put_sync(dev);
	pm_runtime_disable(dev);

#ifdef CONFIG_USB_DWC3_OF_CXD_PHY_ADJUST_DEBUG
	class_remove_file(&dwc3_of_cxd_class, &class_attr_phy_param);
	class_unregister(&dwc3_of_cxd_class);
#endif

	return 0;
}

#ifdef CONFIG_PM
static int dwc3_of_suspend_common(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);
	int			i;

	for (i = 0; i < cxd->num_clocks; i++)
		while (__clk_is_enabled(cxd->clks[i]))
			clk_disable(cxd->clks[i]);

	return 0;
}

static int dwc3_of_resume_common(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);

	return dwc3_of_cxd_usbphy_init(cxd, NULL, NULL);
}

static int dwc3_of_simple_runtime_suspend(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);
	int ret;

	cr_phy31_wa_clear();

	ret = dwc3_cxd_reset_assert(cxd);
	if (ret) dev_err(dev,"%s dwc3_cxd_reset_assert err=%d\n", __FUNCTION__, ret);

	return dwc3_of_suspend_common(dev);
}

static int dwc3_of_simple_runtime_resume(struct device *dev)
{
	return dwc3_of_resume_common(dev);
}
#else
static int dwc3_of_simple_runtime_suspend(struct device *dev) { return 0; }
static int dwc3_of_simple_runtime_resume(struct device *dev) { return 0; }
#endif

#ifdef CONFIG_PM_SLEEP
static int dwc3_of_simple_suspend(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);
	int ret;

	if (pm_runtime_suspended(dev))
		return 0;

	cr_phy31_wa_clear();

	ret = dwc3_cxd_reset_assert(cxd);
	if (ret) dev_err(dev,"%s dwc3_cxd_reset_assert err=%d\n", __FUNCTION__, ret);

	return dwc3_of_suspend_common(dev);
}

static int dwc3_of_simple_resume(struct device *dev)
{
	int			ret;

	if (pm_runtime_suspended(dev))
		return 0;

	cr_phy31_wa_clear();

	ret = dwc3_of_resume_common(dev);
	if (ret) goto bail;

	pm_runtime_disable(dev);
	pm_runtime_set_active(dev);
	pm_runtime_enable(dev);

bail:
	return ret;
}
#else
static int dwc3_of_simple_suspend(struct device *dev) { return 0; }
static int dwc3_of_simple_resume(struct device *dev) { return 0; }
#endif /* CONFIG_PM_SLEEP */

static const struct dev_pm_ops dwc3_of_simple_dev_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(dwc3_of_simple_suspend, dwc3_of_simple_resume)
	SET_RUNTIME_PM_OPS(dwc3_of_simple_runtime_suspend,
			dwc3_of_simple_runtime_resume, NULL)
};

static const struct of_device_id of_dwc3_simple_match[] = {
	{ .compatible = "cxdchip,dwc3" },
	{ /* Sentinel */ }
};
MODULE_DEVICE_TABLE(of, of_dwc3_simple_match);

static struct platform_driver dwc3_of_simple_driver = {
	.probe		= dwc3_of_cxd_probe,
	.remove		= dwc3_of_cxd_remove,
	.driver		= {
		.name	= MODULE_NAME,
		.of_match_table = of_dwc3_simple_match,
		.pm	= &dwc3_of_simple_dev_pm_ops,
	},
};

module_platform_driver(dwc3_of_simple_driver);
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("DesignWare USB3 OF CXD Glue Layer");
