/*
 * mach-cxd900xx/udif/platform.c
 *
 *
 * Copyright 2018,2019 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/udif/device.h>
#include <linux/udif/string.h>
#include <linux/udif/macros.h>
#include <linux/udif/export.h>
#include <mach/udif/platform.h>
#include <mach/platform.h>
#include <mach/irqs.h>
#include <mach/regs-clk.h>
#include <mach/regs-srst.h>
#include <mach/gic_export.h>

#define __UDIF_IOMEM_INIT(n,pa,sz)	UDIF_IOMEM_INIT(n,pa,sz,UDIF_IO_FLAGS_NONCACHED)

#define UDIF_INTRPT_INIT(n,i,f) \
{ \
	.name	= n, \
	.irq	= i, \
	.flags	= f, \
}


/* HW Timer stuff */
#define HWTIMER_NAME		"hwtimer"
#define HWTIMER_IO_BASE(ch)	CXD900XX_TIMER_BASE(ch)
#define HWTIMER_IO_SIZE		0x1000
#define HWTIMER_IRQ(ch)		IRQ_TIMER(ch)
#define HWTIMER_IRQ_FLAGS	IRQF_TIMER

static UDIF_CHANNELS hwtimer[UDIF_NR_HWTIMER] = {
#if !defined(CONFIG_ARCH_CXD900XX_QEMU) && !defined(CONFIG_ARCH_CXD900XX_FVP)
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(HWTIMER_NAME "0", HWTIMER_IO_BASE(0), HWTIMER_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTIMER_NAME "0", HWTIMER_IRQ(0), HWTIMER_IRQ_FLAGS),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(HWTIMER_NAME "1", HWTIMER_IO_BASE(1), HWTIMER_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTIMER_NAME "1", HWTIMER_IRQ(1), HWTIMER_IRQ_FLAGS),
	},
#if !defined(CONFIG_ARCH_CXD900XX_FPGA_V1_5)
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(HWTIMER_NAME "2", HWTIMER_IO_BASE(2), HWTIMER_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTIMER_NAME "2", HWTIMER_IRQ(2), HWTIMER_IRQ_FLAGS),
	},
	[3] = {
		.iomem = __UDIF_IOMEM_INIT(HWTIMER_NAME "3", HWTIMER_IO_BASE(3), HWTIMER_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTIMER_NAME "3", HWTIMER_IRQ(3), HWTIMER_IRQ_FLAGS),
	},
#endif /* !CONFIG_ARCH_CXD900XX_FPGA_V1_5 */
#endif /* !CONFIG_ARCH_CXD900XX_QEMU && !CONFIG_ARCH_CXD900XX_FVP */
};

/* WDT stuff */
#define WDT_NAME		"wdt"
#define WDT_REG_NAME		WDT_NAME "_reg"
#define WDT_MISC_NAME		WDT_NAME "_misc"
#define WDT_REG_BASE		HWTIMER_IO_BASE(16)
#define WDT_REG_SIZE		HWTIMER_IO_SIZE
#define WDT_MISC_BASE		CXD900XX_WDT_MISC
#define WDT_MISC_SIZE		4
#define WDT_IRQ_NAME		"EMERGENCY"
#define WDT_IRQ			IRQ_WDOG_INT

static UDIF_CHANNELS wdt[UDIF_NR_WDT] = {
	[UDIF_WDT_REG] = {
		.iomem = __UDIF_IOMEM_INIT(WDT_REG_NAME, WDT_REG_BASE, WDT_REG_SIZE),
		.intr = UDIF_INTRPT_INIT(WDT_IRQ_NAME, WDT_IRQ, 0),
	},
	[UDIF_WDT_MISC] = {
		.iomem = __UDIF_IOMEM_INIT(WDT_MISC_NAME, WDT_MISC_BASE, WDT_MISC_SIZE),
	},
};

/* gpio stuff */
#define GPIO_NAME		"gpio"
#define GPIO_PORT_NAME(x)	( "gpio_p" #x )
#define GPIO_PORT_IO_BASE(x)	CXD900XX_GPIO(x)
#define GPIO_IO_SIZE		0x00001000U

static UDIF_CHANNELS gpio[UDIF_NR_GPIO] = {
	[0]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(0),  GPIO_PORT_IO_BASE(0),  GPIO_IO_SIZE),  },
	[1]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(1),  GPIO_PORT_IO_BASE(1),  GPIO_IO_SIZE),  },
	[2]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(2),  GPIO_PORT_IO_BASE(2),  GPIO_IO_SIZE),  },
	[3]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(3),  GPIO_PORT_IO_BASE(3),  GPIO_IO_SIZE),  },
	[4]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(4),  GPIO_PORT_IO_BASE(4),  GPIO_IO_SIZE),  },
	[5]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(5),  GPIO_PORT_IO_BASE(5),  GPIO_IO_SIZE),  },
	[6]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(6),  GPIO_PORT_IO_BASE(6),  GPIO_IO_SIZE),  },
	[7]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(7),  GPIO_PORT_IO_BASE(7),  GPIO_IO_SIZE),  },
	[8]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(8),  GPIO_PORT_IO_BASE(8),  GPIO_IO_SIZE),  },
	[9]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(9),  GPIO_PORT_IO_BASE(9),  GPIO_IO_SIZE),  },
	[10] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(10), GPIO_PORT_IO_BASE(10), GPIO_IO_SIZE),  },
	[11] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(11), GPIO_PORT_IO_BASE(11), GPIO_IO_SIZE),  },
	[12] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(12), GPIO_PORT_IO_BASE(12), GPIO_IO_SIZE),  },
	[13] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(13), GPIO_PORT_IO_BASE(13), GPIO_IO_SIZE),  },
	[14] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(14), GPIO_PORT_IO_BASE(14), GPIO_IO_SIZE),  },
	[15] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(15), GPIO_PORT_IO_BASE(15), GPIO_IO_SIZE),  },
	[16] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(16), GPIO_PORT_IO_BASE(16), GPIO_IO_SIZE),  },
	[17] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(17), GPIO_PORT_IO_BASE(17), GPIO_IO_SIZE),  },
	[18] = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(18), GPIO_PORT_IO_BASE(18), GPIO_IO_SIZE),  },
};

#if !defined(CONFIG_ARCH_CXD900XX_FPGA_V1_5)
/* ldec stuff */
#define LDEC_NAME		"ldec"
#define LDEC_IO_BASE		0xF1055000U
#define LDEC_IO_SIZE		0x00001000U
#define LDEC_IRQ		IRQ_LDEC

static UDIF_CHANNELS ldec[UDIF_NR_LDEC] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(LDEC_NAME, LDEC_IO_BASE, LDEC_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(LDEC_NAME, LDEC_IRQ, 0),
	},
};
#endif /* !CONFIG_ARCH_CXD900XX_FPGA_V1_5 */

/* MMC stuff */
#define MMC_NAME 	"mmc"
#define MMC_SDIF0_IO_BASE	0xF10D2000U
#define MMC_SDIF0_IO_SIZE	0x00002000U
#define MMC_SDIF1_IO_BASE	0xF10D4000U
#define MMC_SDIF1_IO_SIZE	0x00002000U
#define MMC_MMC_IO_BASE		0xF10D8000U
#define MMC_MMC_IO_SIZE		0x00002000U
#define TE_IRQ(x)	IRQ_SDIF(x)

/* MMC is mapped  */
static UDIF_CHANNELS mmc[UDIF_NR_MMC] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "0", MMC_SDIF0_IO_BASE, MMC_SDIF0_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "0", TE_IRQ(0), 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "1", MMC_SDIF1_IO_BASE, MMC_SDIF1_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "1", TE_IRQ(1), 0),
	},
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "2", MMC_MMC_IO_BASE, MMC_MMC_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "2", IRQ_EMMC1, 0),
	},
};

/* eMMC0 stuff */
#define EMMC0_NAME 	"emmc0"
#define EMMC0_IO_BASE	0xF10D6000U
#define EMMC0_IO_SIZE	0x00002000U

static UDIF_CHANNELS emmc0[UDIF_NR_EMMC0] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(EMMC0_NAME, EMMC0_IO_BASE,
					   EMMC0_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(EMMC0_NAME, IRQ_EMMC0, 0),
	},
};

#if !defined(CONFIG_ARCH_CXD900XX_FPGA_V1_5)
/* MS stuff */
#define MS_NAME		"ms"
#define MS_IO_BASE	0xF10D1000U
#define MS_IO_SIZE	0x1000
#define MS_IRQ		IRQ_MS0

#define MS_HCLK		CXD900XX_CLKEN(CLKEN_MEDIA)
#define MS_HCLK_SET	(MS_HCLK + CLK_SET)
#define MS_HCLK_CLR	(MS_HCLK + CLK_CLR)
#define MS_HCLK_SHIFT	16
#define MS_DCLK		MS_HCLK
#define MS_DCLK_SET	(MS_DCLK + CLK_SET)
#define MS_DCLK_CLR	(MS_DCLK + CLK_CLR)
#define MS_DCLK_SHIFT	MS_HCLK_SHIFT
#define MS_DSEL		CXD900XX_CLKSEL(CLKSEL_MEDIA)
#define MS_DSEL_SET	(MS_DSEL + CLK_SET)
#define MS_DSEL_CLR	(MS_DSEL + CLK_CLR)
#define MS_DSEL_SHIFT	16
#define MS_DSEL_MASK	0x1

static UDIF_CHANNELS ms[UDIF_NR_MS] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(MS_NAME, MS_IO_BASE, MS_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MS_NAME, MS_IRQ, 0),
		.clock = UDIF_CLOCK_INIT(MS_HCLK_SET, MS_HCLK_CLR, MS_HCLK_SHIFT,
					 MS_DCLK_SET, MS_DCLK_CLR, MS_DCLK_SHIFT,
					 MS_DSEL_SET, MS_DSEL_CLR, MS_DSEL_SHIFT, MS_DSEL_MASK),
	},
};

/* SIO */
#define SIO_NAME        "sio"
#define SIO_IO_BASE(ch) (0xF1005000U + SIO_IO_SIZE * (ch))
#define SIO_IO_SIZE     0x1000
#define SIO_IRQ(ch)     IRQ_SIO(ch)

static UDIF_CHANNELS sio[UDIF_NR_SIO] = {
        [0] = {
                .iomem = __UDIF_IOMEM_INIT(SIO_NAME "0", SIO_IO_BASE(0), SIO_IO_SIZE),
                .intr = UDIF_INTRPT_INIT(SIO_NAME "0", SIO_IRQ(0), 0),
        },
        [1] = {
                .iomem = __UDIF_IOMEM_INIT(SIO_NAME "1", SIO_IO_BASE(1), SIO_IO_SIZE),
                .intr = UDIF_INTRPT_INIT(SIO_NAME "1", SIO_IRQ(1), 0),
        },
        [2] = {
                .iomem = __UDIF_IOMEM_INIT(SIO_NAME "2", SIO_IO_BASE(2), SIO_IO_SIZE),
                .intr = UDIF_INTRPT_INIT(SIO_NAME "2", SIO_IRQ(2), 0),
        },
};
#endif /* !CONFIG_ARCH_CXD900XX_FPGA_V1_5 */

/* DMA330 device */
#define DMA330_NAME(x)		"330d-" #x
#define DMA330_IO_BASE(x)	(0xf1056000U + DMA330_IO_SIZE*(x))
#define DMA330_IO_SIZE		0x00001000U
#define DMA330_0_IRQ(x)		IRQ_DMA330_0(x)
#define DMA330_1_IRQ(x)		IRQ_DMA330_1(x)
#define DMA330_CH_NAME(x,ch)	DMA330_NAME(x) "_" #ch
#define DMA330_ABT_NAME(x)	DMA330_NAME(x) "_abort"

static UDIF_CHANNELS dma330[UDIF_NR_DMA330] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,0), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,0), DMA330_0_IRQ(0), 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,1), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,1), DMA330_0_IRQ(1), 0),
	},
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,2), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,2), DMA330_0_IRQ(2), 0),
	},
	[3] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,3), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,3), DMA330_0_IRQ(3), 0),
	},
	[4] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,4), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,4), DMA330_0_IRQ(4), 0),
	},
	[5] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,5), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,5), DMA330_0_IRQ(5), 0),
	},
	[6] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,6), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,6), DMA330_0_IRQ(6), 0),
	},
	[7] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(0,7), DMA330_IO_BASE(0), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(0,7), DMA330_0_IRQ(7), 0),
	},
	[8] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,0), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,0), DMA330_1_IRQ(0), 0),
	},
	[9] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,1), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,1), DMA330_1_IRQ(1), 0),
	},
	[10] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,2), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,2), DMA330_1_IRQ(2), 0),
	},
	[11] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,3), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,3), DMA330_1_IRQ(3), 0),
	},
	[12] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,4), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,4), DMA330_1_IRQ(4), 0),
	},
	[13] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,5), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,5), DMA330_1_IRQ(5), 0),
	},
	[14] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,6), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,6), DMA330_1_IRQ(6), 0),
	},
	[15] = {
		.iomem = __UDIF_IOMEM_INIT(DMA330_CH_NAME(1,7), DMA330_IO_BASE(1), DMA330_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(DMA330_CH_NAME(1,7), DMA330_1_IRQ(7), 0),
	},
	/* Abort interrupt */
	[16] = {
		.intr = UDIF_INTRPT_INIT(DMA330_ABT_NAME(0), IRQ_DMA330_0_ABORT, 0),
	},
	[17] = {
		.intr = UDIF_INTRPT_INIT(DMA330_ABT_NAME(1), IRQ_DMA330_1_ABORT, 0),
	},
};

/* SATA */
#define SATA_NAME	"sata"

/* cmmc */
#define CMMC_NAME	"cmmc"
#define CMMC_REG_BASE	0xF10E0000U
#define CMMC_INBUF_BASE	0xF8000000U
#define CMMC_REG_SIZE	0x1000U
#define CMMC_INBUF_SIZE	0x4000000U
#define CMMC_REG_NAME	CMMC_NAME "_reg"
#define CMMC_INBUF_NAME	CMMC_NAME "_buf"

static UDIF_CHANNELS cmmc[UDIF_NR_CMMC] = {
	[UDIF_CMMC_CH_REG] = {
		.iomem = __UDIF_IOMEM_INIT(CMMC_REG_NAME, CMMC_REG_BASE, CMMC_REG_SIZE),
	},
	[UDIF_CMMC_CH_INBUF] = {
		.iomem = __UDIF_IOMEM_INIT(CMMC_INBUF_NAME, CMMC_INBUF_BASE, CMMC_INBUF_SIZE),
	},
};

/* FUSB */
/* fusb stuff */
#define FUSB_NAME		"fusb"
/* fusb CH0 stuff */
#define FUSB_CH0_NAME		"usbgpio"
#define FUSB_CH0_IO_BASE	(0xF102E000U)
#define FUSB_CH0_IO_SIZE	0x00001000U

static UDIF_CHANNELS fusb[UDIF_NR_FUSB] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(FUSB_CH0_NAME, FUSB_CH0_IO_BASE, FUSB_CH0_IO_SIZE),
	},
};

/* PCIe stuff */
#define PCIE_NAME 	"pcie"
#define PCIE_PHY_BASE(x) (CXD900XX_PCIEPHY##x)
#define PCIE_PHY_SIZE	0x40000
#define PCIE_CLK	CXD900XX_CLKEN(CLKEN_PCI)
#define PCIE_CLK_SET	(PCIE_CLK + CLK_SET)
#define PCIE_CLK_CLR	(PCIE_CLK + CLK_CLR)
#define PCIE_APBCLK(x)	(CLKEN_PCI_SHIFT(x) + CLKEN_PCI_APB)
#define PCIE_RST	CXD900XX_RESET(SRST_PCI)
#define PCIE_RST_SET	(PCIE_RST + SRST_SET)
#define PCIE_RST_CLR	(PCIE_RST + SRST_CLR)
#define PCIE_APBRST(x)	(SRST_PCI##x + SRST_PCI_APB)

static UDIF_CHANNELS pcie[UDIF_NR_PCIE] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(PCIE_NAME "0", PCIE_PHY_BASE(0), PCIE_PHY_SIZE),
		.clock = UDIF_CLOCK_INIT(PCIE_CLK_SET, PCIE_CLK_CLR, PCIE_APBCLK(0),
					 PCIE_RST_SET, PCIE_RST_CLR, PCIE_APBRST(0),
					 0, 0, 0, 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(PCIE_NAME "1", PCIE_PHY_BASE(1), PCIE_PHY_SIZE),
		.clock = UDIF_CLOCK_INIT(PCIE_CLK_SET, PCIE_CLK_CLR, PCIE_APBCLK(1),
					 PCIE_RST_SET, PCIE_RST_CLR, PCIE_APBRST(1),
					 0, 0, 0, 0),
	},
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(PCIE_NAME "2", PCIE_PHY_BASE(2), PCIE_PHY_SIZE),
		.clock = UDIF_CLOCK_INIT(PCIE_CLK_SET, PCIE_CLK_CLR, PCIE_APBCLK(2),
					 PCIE_RST_SET, PCIE_RST_CLR, PCIE_APBRST(2),
					 0, 0, 0, 0),
	},
	[3] = {
		.iomem = __UDIF_IOMEM_INIT(PCIE_NAME "3", PCIE_PHY_BASE(3), PCIE_PHY_SIZE),
		.clock = UDIF_CLOCK_INIT(PCIE_CLK_SET, PCIE_CLK_CLR, PCIE_APBCLK(3),
					 PCIE_RST_SET, PCIE_RST_CLR, PCIE_APBRST(3),
					 0, 0, 0, 0),
	},
	[4] = {
		.iomem = __UDIF_IOMEM_INIT(PCIE_NAME "4", PCIE_PHY_BASE(4), PCIE_PHY_SIZE),
		.clock = UDIF_CLOCK_INIT(PCIE_CLK_SET, PCIE_CLK_CLR, PCIE_APBCLK(4),
					 PCIE_RST_SET, PCIE_RST_CLR, PCIE_APBRST(4),
					 0, 0, 0, 0),
	},
};

/* DDRC stuff */
#define DDRC_NAME 	"ddrc"
#define DDRC_BASE(x)	CXD900XX_DDRC(x)
#define DDRC_SIZE	0x1000

static UDIF_CHANNELS ddrc[UDIF_NR_DDR] = {
	[0] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "0", DDRC_BASE(0), DDRC_SIZE), },
	[1] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "1", DDRC_BASE(1), DDRC_SIZE), },
	[2] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "2", DDRC_BASE(2), DDRC_SIZE), },
	[3] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "3", DDRC_BASE(3), DDRC_SIZE), },
};

/*
 * All Devices
 */
UDIF_DEVICE udif_devices[UDIF_ID_NUM] = {
	[UDIF_ID_HWTIMER] = {
		.name	= HWTIMER_NAME,
		.nr_ch	= NARRAY(hwtimer),
		.chs	= hwtimer,
	},
	[UDIF_ID_GPIO] = {
		.name	= GPIO_NAME,
		.nr_ch	= NARRAY(gpio),
		.chs	= gpio,
	},
#if !defined(CONFIG_ARCH_CXD900XX_FPGA_V1_5)
	[UDIF_ID_LDEC] = {
		.name   = LDEC_NAME,
		.nr_ch  = NARRAY(ldec),
		.chs    = ldec,
	},
#endif /* !CONFIG_ARCH_CXD900XX_FPGA_V1_5 */
	[UDIF_ID_MMC] = {
		.name	= MMC_NAME,
		.nr_ch 	= NARRAY(mmc),
		.chs	= mmc,
	},
	[UDIF_ID_EMMC0] = {
		.name	= EMMC0_NAME,
		.nr_ch 	= NARRAY(emmc0),
		.chs	= emmc0,
	},
#if !defined(CONFIG_ARCH_CXD900XX_FPGA_V1_5)
	[UDIF_ID_MS] = {
		.name   = MS_NAME,
		.nr_ch  = NARRAY(ms),
		.chs    = ms,
	},
        [UDIF_ID_SIO] = {
                .name   = SIO_NAME,
                .nr_ch = NARRAY(sio),
                .chs    = sio,
        },
#endif /* !CONFIG_ARCH_CXD900XX_FPGA_V1_5 */
	[UDIF_ID_DMA330] = {
		.name   = DMA330_NAME(0),
		.nr_ch  = NARRAY(dma330),
		.chs    = dma330,
	},
	[UDIF_ID_SATA] = {
		.name   = SATA_NAME,
		.nr_ch = 0,
	},
	[UDIF_ID_CMMC] = {
		.name   = CMMC_NAME,
		.nr_ch  = NARRAY(cmmc),
		.chs    = cmmc,
	},
	[UDIF_ID_FUSB_OTG] = {
		.name   = FUSB_NAME,
		.nr_ch  = NARRAY(fusb),
		.chs    = fusb,
	},
	[UDIF_ID_WDT] = {
		.name	= WDT_NAME,
		.nr_ch	= NARRAY(wdt),
		.chs	= wdt,
	},
	[UDIF_ID_PCIE] = {
		.name	= PCIE_NAME,
		.nr_ch	= NARRAY(pcie),
		.chs	= pcie,
	},
	[UDIF_ID_DDR] = {
		.name	= DDRC_NAME,
		.nr_ch	= NARRAY(ddrc),
		.chs	= ddrc,
	},
};

EXPORT_SYMBOL(udif_devices);

static int udif_platform_register(void)
{
	int i, ret = 0;

	for (i = 0; i < NARRAY(udif_devices); i++)
		ret = udif_device_register(&udif_devices[i]);

	return ret;
}

arch_initcall(udif_platform_register);


static void __init cxd900xx_gic_init(void)
{
#if !defined(CONFIG_ARM_GIC_INITIALIZED)
	gic_config_edge(IRQ_REC_V);
	gic_config_edge(IRQ_REC_V_HS);
	gic_target_cpu(IRQ_REC_V,    0x2);
	gic_target_cpu(IRQ_REC_V_HS, 0x2);
#endif /* !CONFIG_ARM_GIC_INITIALIZED */
}

/*--------------- allocate virqs --------------------*/
#include <linux/irqdomain.h>
#include <dt-bindings/interrupt-controller/arm-gic.h>
#define TO_GIC_SPI(x)	((x) - IRQ_GIC_START)
#define N_PARAM 3
#define  PRM_CLS 0
#define  PRM_IRQ 1
#define  PRM_TYP 2

static void udif_alloc_irqs(struct irq_domain *domain, int irq_base, unsigned int nr_irqs)
{
	struct device_node *of_node;
	struct irq_fwspec fwspec;
	int nid;
	int ret;

	if (!nr_irqs)
		return;

	of_node = irq_domain_get_of_node(domain);
	nid = of_node_to_nid(of_node);
	fwspec.fwnode = domain->fwnode;
	fwspec.param_count = N_PARAM;
	fwspec.param[PRM_CLS] = GIC_SPI;
	fwspec.param[PRM_IRQ] = TO_GIC_SPI(irq_base);
	fwspec.param[PRM_TYP] = 0;
	ret = __irq_domain_alloc_irqs(domain, irq_base, nr_irqs, nid, &fwspec, false, NULL);
	if (ret < 0) {
		printk(KERN_ERR "%s: cannot map IRQ%d-%d\n", __func__,
		       irq_base, irq_base + nr_irqs - 1);
	}
}

void __init udif_map_irq(struct irq_domain *domain)
{
	int i;
	UDIF_DEVICE *dev;
	struct device_node *of_node;
	struct irq_fwspec fwspec;
	int nid;

	of_node = irq_domain_get_of_node(domain);
	nid = of_node_to_nid(of_node);
	fwspec.fwnode = domain->fwnode;
	fwspec.param_count = N_PARAM;
	fwspec.param[PRM_CLS] = GIC_SPI;
	fwspec.param[PRM_TYP] = 0;
	for (i = 0, dev = udif_devices; i < NARRAY(udif_devices); i++, dev++) {
		int ch;
		for (ch = 0; ch < dev->nr_ch; ch++) {
			int hwirq, ret;

			hwirq = dev->chs[ch].intr.irq;
			if (!hwirq)
				continue;

			fwspec.param[PRM_IRQ] = TO_GIC_SPI(hwirq);
			ret = __irq_domain_alloc_irqs(domain, hwirq, 1, nid, &fwspec, false, NULL);
			if (ret < 0) {
				printk(KERN_ERR "%s: cannot map IRQ%d\n",
				       __func__, hwirq);
			}
		}
	}
	/* GPIO INT */
	udif_alloc_irqs(domain, IRQ_GPIO(0), NR_GPIO_IRQ);
	udif_alloc_irqs(domain, IRQ_GPIO24, 1);
	udif_alloc_irqs(domain, IRQ_GPIO25, 1);
	/* others */
	udif_alloc_irqs(domain, IRQ_GIC_START, 56);
	udif_alloc_irqs(domain, 178, 16);
	udif_alloc_irqs(domain, 241, 7);
	udif_alloc_irqs(domain, 255, 6);
	udif_alloc_irqs(domain, 264, 2);
	udif_alloc_irqs(domain, 276, 2);
	/* setup GIC */
	cxd900xx_gic_init();
}

unsigned int arch_dynirq_lower_bound(unsigned int from)
{
	return IRQ_GIC_START + NR_GIC_SPI;
}
