/*
 * mach-cxd900xx/pm_core.c
 *
 *
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <mach/moduleparam.h>
#include <linux/init.h>
#include <linux/printk.h>
#include <linux/suspend.h>
#include <linux/warmboot.h>
#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#endif
#include <linux/io.h>
#include <linux/wdt.h>
#include <linux/tmonitor.h>
#include <linux/em_export.h>
#include <asm/suspend.h>
#include <asm/system_misc.h>
#include <asm/cacheflush.h>
#include <asm/mmu_context.h>
#include <mach/pm.h>
#include <mach/platform.h>
#include <mach/bootram.h>
#include <mach/noncache.h>
#include <mach/regs-wdt.h>
#include <mach/regs-gpio.h>

#ifdef CONFIG_OSAL_UDIF
extern int udif_driver_suspend(void);
extern int udif_driver_resume(void);
#else
# define udif_driver_suspend() (0)
# define udif_driver_resume() do {} while (0)
#endif /* CONFIG_OSAL_UDIF */

static int logbuf_noclear = 0;
module_param_named(logbuf_noclear, logbuf_noclear, int, S_IRUSR|S_IWUSR);

/*
 * resume vector
 */
static unsigned long cxd90057_resume_vector = ~0UL;
module_param_named(resume, cxd90057_resume_vector, ulongH, S_IRUSR);

/*
 * suspend error flag
 */
static unsigned long cxd90057_sus_error = ~0UL;
module_param_named(sus_error, cxd90057_sus_error, ulongH, S_IRUSR);

/*
 * PORT
 */
#define PORT_UNDEF (~0U)
/* XPOWER_OFF */
static unsigned int cxd90057_xpower_off = PORT_UNDEF;
module_param_named(poff, cxd90057_xpower_off, port, S_IRUSR);
/* XRESET_REQ */
#define XRSTREQ_PORT 0
#define XRSTREQ_BIT  9

#if !defined(CONFIG_EJ_SIMPLE_SUSPEND)
extern unsigned int cxd90057_sus_end_msg;   /* sleep.S */
module_param_named(sus_msg, cxd90057_sus_end_msg, uint, S_IRUSR|S_IWUSR);
#endif /* !CONFIG_EJ_SIMPLE_SUSPEND */

static void __noreturn cxd90057_reset(int force)
{
	local_irq_disable();
	writel(BIT(XRSTREQ_BIT), VA_GPIO(XRSTREQ_PORT)+GPIO_DIR+GPIO_SET);
	writel(BIT(XRSTREQ_BIT), VA_GPIO(XRSTREQ_PORT)+GPIO_WDATA+GPIO_CLR);
	writel(BIT(XRSTREQ_BIT), VA_GPIO(XRSTREQ_PORT)+GPIO_FUNC+GPIO_SET);
	wmb();
	for (;;) {
			wfi();
	}
}

static void cxd90057_pm_restart(enum reboot_mode reboot_mode, const char *cmd)
{
	cxd90057_reset(1);
}

void __noreturn (*pm_machine_reset)(int) = cxd90057_reset;
EXPORT_SYMBOL(pm_machine_reset);

void em_reboot(int force)
{
	flush_dcache_all();
	pm_machine_reset(force);
}

static void cxd90057_pm_setup(void)
{
#if !defined(CONFIG_EJ_SIMPLE_SUSPEND)
	unsigned int ch, bit;
	extern unsigned long cxd90057_xpower_off_addr;    /* sleep.S */
	extern unsigned int  cxd90057_xpower_off_bitmask; /* sleep.S */
#endif /* !CONFIG_EJ_SIMPLE_SUSPEND */

	/* resume vector */
	if (~0UL == cxd90057_resume_vector) {
		printk("resume vector: not defined\n");
	} else {
		printk("resume vector: 0x%09lx\n", cxd90057_resume_vector);
		printk("resume entry:  0x%09lx\n", (unsigned long)virt_to_phys(cxd90057_resume));
		*(unsigned long *)phys_to_virt(cxd90057_resume_vector) = virt_to_phys(cxd90057_resume);
	}
	/* sus_error flag */
	if (~0UL == cxd90057_sus_error) {
		printk("sus_error address: not defined\n");
	} else {
		printk("sus_error address: 0x%09lx\n", cxd90057_sus_error);
		*(unsigned long *)phys_to_virt(cxd90057_sus_error) = 0UL;
	}

#if !defined(CONFIG_EJ_SIMPLE_SUSPEND)
	/* XPOWER_OFF */
	if (PORT_UNDEF == cxd90057_xpower_off) {
		cxd90057_xpower_off_addr = ~0UL; /* invalid mark */
	} else {
		ch   = cxd90057_xpower_off & 0xff;
		bit  = (cxd90057_xpower_off >> 8) & 0xff;
		cxd90057_xpower_off_addr = PA_GPIO(ch,WDATA)+GPIO_CLR;
		cxd90057_xpower_off_bitmask = 1 << bit;
	}
#endif /* !CONFIG_EJ_SIMPLE_SUSPEND */
}

void __attribute__((weak)) cxd90057_early_resume(void)
{
	/* none */
}

/* check if memory data is kept during the last halt */
static unsigned int cxd90057_mem_alive = 0;

#ifdef CONFIG_WARM_BOOT_IMAGE
/* check if memory data is kept during the last halt */
unsigned int pm_is_mem_alive(void)
{
	return cxd90057_mem_alive;
}
EXPORT_SYMBOL(pm_is_mem_alive);

#ifdef CONFIG_SNSC_SSBOOT
int get_ssboot_stat(void)
{
	extern int cxd90057_ssboot_stat; /* in resume.S */

	return cxd90057_ssboot_stat;
}
EXPORT_SYMBOL(get_ssboot_stat);

void wb_reboot(void)
{
	em_reboot(1);
}
#endif /* CONFIG_SNSC_SSBOOT */
#endif /* CONFIG_WARM_BOOT_IMAGE */

#ifdef CONFIG_EJ_SIMPLE_SUSPEND
static int cxd90057_suspend_finisher(unsigned long mode)
{
	return 0;
}
#else
#define WORK	(CXD900XX_ESRAM_BASE+CXD900XX_SUSPEND_WORKAREA)
#define TBL_SIZE 4096
#define MMU_L1	(WORK)
#define MMU_L2	(MMU_L1 + TBL_SIZE)
#define CODE	(MMU_L2 + TBL_SIZE)
#define STACK	(WORK)

static void cxd90057_idmap_esram(void)
{
	unsigned long *pgd = (unsigned long *)PHYS_TO_VA(MMU_L1);
	unsigned long *pmd = (unsigned long *)PHYS_TO_VA(MMU_L2);
	unsigned int idx;
	unsigned long phys;

	/* setup idmap for eSRAM */
	memset(pgd, 0, TBL_SIZE);
	memset(pmd, 0, TBL_SIZE);
	idx = CXD900XX_ESRAM_BASE >> PGDIR_SHIFT;
	pgd[idx] = MMU_L2 | PMD_TYPE_TABLE;
	idx = (CXD900XX_ESRAM_BASE & ~PGDIR_MASK) >> PMD_SHIFT;
	phys = CXD900XX_ESRAM_BASE & PMD_MASK;
	pmd[idx] = phys | 0x70c | PMD_TYPE_SECT;

	/* install idmap */
	cpu_set_reserved_ttbr0();
	local_flush_tlb_all();
	cpu_set_default_tcr_t0sz();
	write_sysreg(MMU_L1, ttbr0_el1);
	isb();
}

static int cxd90057_suspend_finisher(unsigned long mode)
{
	raw_local_irq_disable();
	cxd90057_mem_alive = 0; // for warmboot
	flush_dcache_all();

#ifdef CONFIG_WARM_BOOT_IMAGE
	if (PM_ENTER_NORMAL == mode) {
		int ret;

#ifdef CONFIG_SNSC_SSBOOT
		ret = wb_create_ssbi();
#else
		ret = wb_create_warmbootimage((unsigned long)cxd90057_resume, 0);
#endif /* CONFIG_SNSC_SSBOOT */
		if (ret)
			return ret;
	}
#endif /* CONFIG_WARM_BOOT_IMAGE */

	cxd90057_mem_alive = 1; // for hotboot

	/* copy the standby operation code to the resume ram */
	cxd90057_idmap_esram();
	memcpy((void *)CODE, cxd90057_suspend, (void *)cxd90057_suspend2-(void *)cxd90057_suspend);
	flush_dcache_all();
	/* go zzz */
	(*(void (*)(void *))CODE)((void *)STACK);

	printk("cxd90057_suspend failed!\n");
	return -EIO;
}
#endif /* CONFIG_EJ_SIMPLE_SUSPEND */

static void __noreturn cxd90057_pm_halt(void)
{
#if !defined(CONFIG_EJ_SIMPLE_SUSPEND)
	cxd90057_suspend_finisher(PM_ENTER_ERROR);
#endif
	for (;;) {
		wfi();
	}
	/* never reached */
}

static void pm_suspend_error(void)
{
	if (~0UL != cxd90057_sus_error) {
		*(unsigned long *)phys_to_virt(cxd90057_sus_error) = 1UL;
	}
	cxd90057_pm_halt();
	/* never reached */
}

/*
 * Called after processes are frozen, but before we shut down devices.
 */
static int cxd90057_pm_begin(suspend_state_t state)
{
	watchdog_start();
	if (PM_SUSPEND_DISK == state) {
		watchdog_stop();
	}
	return warm_boot_pm_begin(state);
}

static int cxd90057_pm_prepare(void)
{
	watchdog_touch();
	if (unlikely(udif_driver_suspend()))
		pm_suspend_error();

	watchdog_touch();
	return warm_boot_pm_prepare();
}

static int cxd90057_pm_prepare_late(void)
{
	watchdog_touch();
	return 0;
}

#ifdef CONFIG_SNSC_BOOT_TIME
#include <linux/snsc_boot_time.h>
#endif

static int cxd90057_pm_enter(suspend_state_t state)
{
	int err = 0;
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	extern void profile_soft_init(void);
	extern void profile_dump_record(void (*print)(char *));
#endif

	watchdog_touch();
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	profile_dump_record(printascii);
#endif
	err = cpu_suspend(PM_ENTER_NORMAL, cxd90057_suspend_finisher);
#ifdef CONFIG_WARM_BOOT_IMAGE
	if (!err && PM_SUSPEND_DISK == pm_get_state() && !pm_is_mem_alive()) {
		if (!logbuf_noclear) {
			clear_logbuf();
			tmonitor_clear();
		}
		printk("WARMBOOT\n");
#ifdef CONFIG_SNSC_SSBOOT
		ssboot_setmode(get_ssboot_stat());
#endif
	}
#endif /* CONFIG_WARM_BOOT_IMAGE */

	cxd90057_early_resume();

#ifdef CONFIG_SNSC_BOOT_TIME
	boot_time_takeover();
#endif
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	profile_soft_init();
#endif
	return err;
}

static void cxd90057_pm_wake(void)
{
	watchdog_touch();
}

/*
 * Called after devices are re-setup, but before processes are thawed.
 */
static void cxd90057_pm_finish(void)
{
	watchdog_touch();
	warm_boot_pm_finish();
	udif_driver_resume();
#ifdef CONFIG_SNSC_SSBOOT
	if (wb_create_ssbi_finish()) {
		extern int cxd90057_ssboot_stat; /* in resume.S */

		cxd90057_ssboot_stat = SSBOOT_SSBOOT; /* adjust */
		printk("SSBI Create Completed!\n");
	}
#endif /* CONFIG_SNSC_SSBOOT */
	watchdog_touch();
}

static void cxd90057_pm_end(void)
{
	warm_boot_pm_end();
	watchdog_stop();
}

static int cxd90057_pm_valid(suspend_state_t state)
{
	return PM_SUSPEND_MEM == state || PM_SUSPEND_DISK == state;
}

static struct platform_suspend_ops cxd90057_pm_ops = {
	.valid		= cxd90057_pm_valid,
	.begin		= cxd90057_pm_begin,
	.prepare	= cxd90057_pm_prepare,
	.prepare_late	= cxd90057_pm_prepare_late,
	.enter		= cxd90057_pm_enter,
	.wake		= cxd90057_pm_wake,
	.finish		= cxd90057_pm_finish,
	.end		= cxd90057_pm_end,
};

static int __init cxd90057_pm_init(void)
{
	cxd90057_pm_setup();
	suspend_set_ops(&cxd90057_pm_ops);
	arm_pm_restart = cxd90057_pm_restart;
#ifdef CONFIG_SNSC_SSBOOT
	wb_ssboot_vec[SSBOOT_CREATE_MINSSBI] = (unsigned long)cxd90057_resume;
	wb_ssboot_vec[SSBOOT_PROFILE] = (unsigned long)cxd90057_resume_profile;
	wb_ssboot_vec[SSBOOT_CREATE_OPTSSBI] = (unsigned long)cxd90057_resume_optimize;
	wb_ssboot_vec[SSBOOT_SSBOOT_PRE] = (unsigned long)cxd90057_resume_ssboot_pre;
	wb_ssboot_vec[SSBOOT_SSBOOT] = (unsigned long)cxd90057_resume_ssboot;
#endif /* CONFIG_SNSC_SSBOOT */
	return 0;
}

late_initcall(cxd90057_pm_init);
