/*
 * Copyright 2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/moduleparam.h>
#include <linux/wait.h>
#include <linux/sched.h>

#include <asm/memory.h>
#include <linux/dma-mapping.h>

#include <asm/uaccess.h>
#include <asm/errno.h>

#include <linux/console.h>

#include <linux/version.h>

//#define SEN_FIFO_DEBUG_INF

#include "usbg_sen_fifo.h"

void sen_fifo_lock( struct sen_xfer_fifo* fifo)
{
    udif_mutex_lock( &fifo->lock);
}

void sen_fifo_unlock( struct sen_xfer_fifo* fifo)
{
    udif_mutex_unlock( &fifo->lock);
}


int sen_fifo_setup( struct sen_xfer_fifo* fifo, unsigned char* buf, unsigned int size )
{
    udif_mutex_init(&fifo->lock);
    fifo->top = buf;
    fifo->bottom = buf + size;
    fifo->rp = fifo->wp = fifo->top;
    return 0;
}

int sen_fifo_data_size( struct sen_xfer_fifo* fifo )
{
    unsigned char* top      = fifo->top;
    unsigned char* bottom   = fifo->bottom;
    unsigned char* rp       = fifo->rp;
    unsigned char* wp       = fifo->wp;

    int size;

    if( rp <= wp ){
        size = wp - rp;
    }else{
        size = ( bottom - rp ) + ( wp - top ) ;
    }

    SEN_FIFO_DEBUG("%s:size = %d\n", __FUNCTION__, size );
    return size ;
}

int sen_fifo_free_size ( struct sen_xfer_fifo* fifo)
{
    unsigned char* top      = fifo->top;
    unsigned char* bottom   = fifo->bottom;
    unsigned char* rp       = fifo->rp;
    unsigned char* wp       = fifo->wp;

    int size;

    if( rp > wp ){
        size = rp - wp;
    }else{
        size = (bottom - wp) + (rp - top);
    }

    SEN_FIFO_DEBUG("%s:size = %d\n", __FUNCTION__, size );
    return size;
}

int sen_fifo_buf_size ( struct sen_xfer_fifo* fifo )
{
    int size = fifo->bottom - fifo->top;
    SEN_FIFO_DEBUG("%s:size = %d\n", __FUNCTION__, size );
    return size;
}

void* sen_fifo_buf_top ( struct sen_xfer_fifo* fifo )
{
    return fifo->top;
}

int sen_fifo_read( struct sen_xfer_fifo* fifo, void* buf, unsigned int size )
{
    unsigned char* top      = fifo->top;
    unsigned char* bottom   = fifo->bottom;
    unsigned char* rp       = fifo->rp;
    unsigned char* wp       = fifo->wp;

    int data_size = sen_fifo_data_size(fifo);
    int actual = size > data_size ? data_size : size;
    int read_remain = actual;

    SEN_FIFO_DEBUG("%s entry: size = %d, actual = %d\n", __FUNCTION__, size, actual );    

    while( read_remain ){

        if( rp > wp ){
            /* バッファのデータが分割されているとき */            
            int bottom_data_size ;
            /* rpからbottomまでに残っているデータサイズ */
            bottom_data_size = bottom - rp;

            if( bottom_data_size > read_remain ){
                memcpy( buf, rp, read_remain );
                rp += read_remain;
                read_remain = 0;
            }else{
                memcpy( buf, rp, bottom_data_size );
                rp = top; // 巻き戻し
                read_remain -= bottom_data_size ;
            }
        }else{
            /* バッファのデータが分割されてない */
            memcpy( buf, rp, read_remain );
            rp += read_remain;
            read_remain = 0;
        }
    }
    fifo->rp = rp;
    SEN_FIFO_DEBUG("%s exit: size = %d, actual = %d\n", __FUNCTION__, size, actual );
    return actual;
}

int sen_fifo_read_to_user( struct sen_xfer_fifo* fifo, void __user * buf, unsigned int size )
{
    unsigned char* top      = fifo->top;
    unsigned char* bottom   = fifo->bottom;
    unsigned char* rp       = fifo->rp;
    unsigned char* wp       = fifo->wp;

    int data_size = sen_fifo_data_size(fifo);
    int actual = size > data_size ? data_size : size;
    int read_remain = actual;

    SEN_FIFO_DEBUG("%s entry: size = %d, actual = %d\n", __FUNCTION__, size, actual );

    while( read_remain ){

        if( rp > wp ){
            /* バッファのデータが分割されているとき */            
            int bottom_data_size ;
            /* rpからbottomまでに残っているデータサイズ */
            bottom_data_size = bottom - rp;

            if( bottom_data_size > read_remain ){
                if(copy_to_user( buf, rp, read_remain ) != 0){
                    SEN_FIFO_ERR("copy_to_user error\n");
                    actual = 0;
                    break;
                }else{
                    rp += read_remain;
                    read_remain = 0;
                }
            }else{
                if( copy_to_user( buf, rp, bottom_data_size ) != 0 ){
                    SEN_FIFO_ERR("copy_to_user error\n");
                    actual = 0;
                    break;
                }else{
                    rp = top; // 巻き戻し
                    read_remain -= bottom_data_size ;
                }
            }
        }else{
            /* バッファのデータが分割されてない */
            if( copy_to_user( buf, rp, read_remain ) != 0){
                SEN_FIFO_ERR("copy_to_user error\n");
                actual = 0;
                break;
            }else{
                rp += read_remain;
                read_remain = 0;
            }
        }
    }
    fifo->rp = rp;
    SEN_FIFO_DEBUG("%s exit: size = %d, actual = %d\n", __FUNCTION__, size, actual );
    return actual;
}


int sen_fifo_write( struct sen_xfer_fifo* fifo, void* buf, unsigned int size )
{
    unsigned char* top      = fifo->top;
    unsigned char* bottom   = fifo->bottom;
    unsigned char* rp       = fifo->rp;
    unsigned char* wp       = fifo->wp;

    int free_size = sen_fifo_free_size(fifo);
    int actual = size > free_size ? free_size : size ;
    int write_remain = actual;

    SEN_FIFO_DEBUG("%s entry: size = %d, actual = %d\n", __FUNCTION__, size, actual );
    while ( write_remain ){

        if( wp > rp ){
            /* 空き領域が分割されている */
            int bottom_free = bottom - wp;

            if( bottom_free > write_remain ){
                memcpy( wp, buf, write_remain );
                wp += write_remain;
                write_remain = 0;
            }else{
                memcpy( wp, buf, bottom_free );
                wp = top;
                write_remain -= bottom_free;
            }
        }else{
            /* 空き領域が分割されていない */
            memcpy( wp, buf, write_remain );
            wp += write_remain;
            write_remain = 0;
        }
    }
    fifo->wp = wp;
    SEN_FIFO_DEBUG("%s exit: size = %d, actual = %d\n", __FUNCTION__, size, actual );
    return actual;
}

void sen_fifo_skip_wp( struct sen_xfer_fifo* fifo, unsigned int skip )
{
    /* 中途半端だが、FIFO先頭にしかDMAしないので*/
    fifo->wp += skip;
}

void sen_fifo_reset( struct sen_xfer_fifo* fifo)
{
    fifo->rp = fifo->wp = fifo->top;
}

void sen_fifo_show_status( struct sen_xfer_fifo* fifo )
{
    SEN_FIFO_DEBUG("### %s ###\n", __FUNCTION__);
    SEN_FIFO_DEBUG("    fifo->top    = %p\n", fifo->top);
    SEN_FIFO_DEBUG("    fifo->bottom = %p\n", fifo->bottom);
    SEN_FIFO_DEBUG("    fifo->rp     = %p\n", fifo->rp);
    SEN_FIFO_DEBUG("    fifo->wp     = %p\n", fifo->wp);
    SEN_FIFO_DEBUG("    data size    = %d\n", sen_fifo_data_size(fifo));
}