/**
 *	@file	usbg_still_core.c
 *	@brief	USB SICD(Still Image Capture Device class) - core
 *	
 *		Copyright 2005,2006,2008,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>

#include <linux/device.h>

#include <linux/usb/gadget.h>

#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifndef CONFIG_OSAL_UDIF
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_ioctl.h>
#include <linux/usb/specific_gadget/usb_ioctl_still.h>

#include <udif/malloc.h>

#define FNAME "usbg_still_core.c"	///< this file name
#include "usbg_still_conf.h"
#include "usbg_still_pvt.h"

/**
 *	@brief		EP0 transfer callback
 *	@param		ep		EP0 ep struct
 *	@param		req		request struct
 */
static void sicd_ep0_comp(struct usb_ep *ep, struct usb_request *req)
{
	SICD_API();
	
	if(!req) {
		SICD_ERR("req==NULL\n");
		return;
	}
	
	usb_ep_free_request(ep, req);
}

/**
 *	@brief		SICD Cancel request handler
 *	@param		p_ctrl		USB setup packet
 *	@param		ep0			EP0 ep struct
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 *	@retval		USB_GADGETCORE_THRU		hand over this request to another driver 
 */
static int sicd_class_cancel(const struct usb_ctrlrequest* p_ctrl, struct usb_ep* ep0)
{
	int err;
	struct usb_request *req;
	int set_busy_flg = 0;

	SICD_API();
	
	// check data direction
	// CancelRequest's data direction should be OUT(Host->Device)
	if(p_ctrl->bRequestType & USB_DIR_IN) {
		return USB_GADGETCORE_STALL;
	}
	
	// DS181H-4145Ή:Cancel͍ŏStatusBUSYSET.
	SICD_STATUS_LOCK_ON();
	if(sicd_get_status() == SICD_STATUS_OK) {
		printk("1\n");
		sicd_set_status(SICD_STATUS_BUSY);
		set_busy_flg = 1;
	}
	SICD_STATUS_LOCK_OFF(); 

	sicd_ep_dequeue_bulk();
	
	// alloc request
	req = usb_ep_alloc_request(ep0, GFP_ATOMIC);
	if(!req) {
		SICD_ERR("usb_ep_alloc_request\n");
		return USB_GADGETCORE_STALL;
	}
	
	// fill request
	req->buf			= sicd_get_ep0_buffer();
	req->length			= SICD_EP0_BUFSIZE;
	SICD_INF("%s: size=%d, aligned=%d\n", __FUNCTION__, SICD_EP0_BUFSIZE, req->length);
	req->dma			= (dma_addr_t)NULL;
	req->zero			= 0;
	req->complete		= sicd_ep0_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= NULL;
	
	// queue request
	err = usb_ep_queue(ep0, req, GFP_ATOMIC);
	
	/*
	 * we don't use CancelRequest data (transID etc.)
	 */
	 
	if(err) {
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		usb_ep_free_request(ep0, req);
		err = USB_GADGETCORE_STALL;
	}

	SICD_STATUS_LOCK_ON();
	
	if(sicd_get_status() == SICD_STATUS_OK) {
		sicd_set_status(SICD_STATUS_BUSY);
		SICD_STATUS_LOCK_OFF();
		
		sicd_call_user_cb(SICD_CBID_CLASS, USBG_STILL_EVENT_CANCEL, NULL, 0);
	}
	else {
        SICD_STATUS_LOCK_OFF();
		if ( set_busy_flg ) {
			printk("2\n");
			sicd_call_user_cb(SICD_CBID_CLASS, USBG_STILL_EVENT_CANCEL, NULL, 0);
		} else {
			SICD_INF("USBG_STILL_EVENT_CANCEL concealed!!!");
		}
	}

	return 0;
}

/**
 *	@brief		SICD DeviceReset request handler
 *	@param		p_ctrl		USB setup packet
 *	@param		ep0			EP0 ep struct
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 *	@retval		USB_GADGETCORE_THRU		hand over this request to another driver 
 */
static int sicd_class_device_reset(const struct usb_ctrlrequest* p_ctrl, struct usb_ep* ep0)
{
	int err;
	struct usb_request *req;
	struct func_data_ex *p_this_still = sicd_get_this_ex();

	SICD_API();

	sicd_ep_dequeue_bulk();

	// alloc request
	req = usb_ep_alloc_request(ep0, GFP_ATOMIC);
	if(!req) {
		SICD_ERR("usb_ep_alloc_request\n");
		return USB_GADGETCORE_STALL;
	}
	
	// fill request  (send NULL packet)
	req->buf			= sicd_get_ep0_buffer();
	req->length			= 0;
	SICD_INF("%s: size=%d, aligned=%d\n", __FUNCTION__, 0, req->length);
	req->dma			= (dma_addr_t)NULL;
	req->zero			= 1;
	req->complete		= sicd_ep0_comp;	// without buffer free
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= NULL;

#ifdef SICD_CLEAR_HALT_AT_DEVICE_RESET
	err = sicd_bulk_out_halt_clear();
	if(err) {
		SICD_ERR("sicd_bulk_out_halt_clear err=%d\n", err);
		usb_ep_free_request(ep0, req);
		return USB_GADGETCORE_STALL;
	}
	
	err = sicd_bulk_in_halt_clear();
	if(err) {
		SICD_ERR("sicd_bulk_in_halt_clear err=%d\n", err);
		usb_ep_free_request(ep0, req);
		return USB_GADGETCORE_STALL;
	}
	
	sicd_call_user_cb(SICD_CBID_EP_CLR_HALT, 0, NULL, 0);
#endif

	// queue request
	err = usb_ep_queue(ep0, req, GFP_ATOMIC);
	
	if(err) {
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		usb_ep_free_request(ep0, req);
		return USB_GADGETCORE_STALL;
	}

	SICD_STATUS_LOCK_ON();
	
	if(sicd_get_status() == SICD_STATUS_OK) {
		sicd_set_status(SICD_STATUS_BUSY);
		SICD_STATUS_LOCK_OFF();
		
		sicd_call_user_cb(SICD_CBID_CLASS, USBG_STILL_EVENT_DEVICE_RESET, NULL, 0);
	}
	else {
        SICD_STATUS_LOCK_OFF();
        SICD_RST_REQ_LOCK_ON();
		p_this_still->reset_request_flag = TRUE;
		SICD_RST_REQ_LOCK_OFF();
		SICD_INF("USBG_STILL_EVENT_DEVICE_RESET concealed!!!");
	}
	

	return err;
}

/**
 *	@brief		SICD GetDeviceStatus request handler
 *	@param		p_ctrl		USB setup packet
 *	@param		ep0			EP0 ep struct
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 *	@retval		USB_GADGETCORE_THRU		hand over this request to another driver 
 */
static int sicd_class_get_device_status(const struct usb_ctrlrequest* p_ctrl, struct usb_ep* ep0)
{
	int i, len, err, max_len;
	struct usb_request *req;
	u8 *tmp_p, *buf_p;
	
	struct func_data_ex *p_this_still = sicd_get_this_ex();

	SICD_API();
	
	max_len = le16_to_cpu(p_ctrl->wLength);
	
	// alloc request
	req = usb_ep_alloc_request(ep0, GFP_ATOMIC);
	if(!req) {
		SICD_ERR("usb_ep_alloc_request\n");
		return USB_GADGETCORE_STALL;
	}
	
	// calc data len
	len = SICD_GET_DEVICE_STATUS_MIN_LEN;
	for(i = 0; i < SICD_EP_IDX_MAX; i++) {
		if(p_this_still->ep_info[i].stalled) {
			len += 4;
		}
	}	
	// fill device status data
	buf_p = tmp_p = sicd_get_ep0_buffer();
	if(!buf_p) {
		return -ENOMEM;
	}
	
	SICD_INF("Get_Device_Status: buf=%p, len=%d, status=0x%x\n", tmp_p, len, p_this_still->sicd_status);
	sicd_put_u16(&tmp_p, len);
	sicd_put_u16(&tmp_p, p_this_still->sicd_status);

	// append stalled ep information
	for(i = 0; i < SICD_EP_IDX_MAX; i++) {
		if(p_this_still->ep_info[i].stalled) {
			sicd_put_u32(&tmp_p, p_this_still->ep_info[i].ep_no);
		}
	}
	
	// fill request  (send NULL packet)
	req->buf			= buf_p;
	req->length			= len > max_len ? max_len : len;
	req->dma			= (dma_addr_t)NULL;
	req->zero			= 1;
	req->complete		= sicd_ep0_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= NULL;
	
	// queue request
	err = usb_ep_queue(ep0, req, GFP_ATOMIC);

	if(err) {
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		usb_ep_free_request(ep0, req);
		err = USB_GADGETCORE_STALL;
	}
	
	return err;
}


/**
 *	@brief		GadgetCore callback - nop
 *	@param		p_drv		function driver struct
 */
static void core_cb_nop(struct usb_gadget_func_driver* p_drv)
{
	SICD_FNC();
}
 
/**
 *	@brief		GadgetCore callback - start
 *	@param		p_drv		function driver struct
 *	@param		alt			alternative interface no.
 *	@param		list		Endpoint list
 */
static void core_cb_start( struct usb_gadget_func_driver* p_drv,
						unsigned char alt,
						struct usb_gadget_ep_list list )
{
	int i;
	struct func_data    *p_this    = sicd_get_this();
	struct func_data_ex *p_this_ex = sicd_get_this_ex();
	struct usb_ep *ep;
	struct sicd_ep_info *epinf;
	
	SICD_API();

	SICD_EP_LOCK_ON();
	
	if( usbg_cmn_make_ep_list( p_this, alt, list ) ){
		SICD_ERR("usbg_cmn_make_ep_list\n");
	}
	
	// ep table initialize
	ep                       = p_this->ptbl_ep[SICD_EP_IDX_BULK_IN];
	epinf                    = &(p_this_ex->ep_info[SICD_EP_IDX_BULK_IN]);
	epinf->ep_no             = 0x81;
	epinf->stalled           = FALSE;
	epinf->u.data.requesting = FALSE;
	epinf->u.data.req        = usb_ep_alloc_request(ep, GFP_ATOMIC);
	ep->name                 = SICD_EP_NAME_BULK_IN;
	ep->driver_data          = epinf;
	
	ep                       = p_this->ptbl_ep[SICD_EP_IDX_BULK_OUT];
	epinf                    = &(p_this_ex->ep_info[SICD_EP_IDX_BULK_OUT]);
	epinf->ep_no             = 0x02;
	epinf->stalled           = FALSE;
	epinf->u.data.requesting = FALSE;
	epinf->u.data.req        = usb_ep_alloc_request(ep, GFP_ATOMIC);
	ep->name                 = SICD_EP_NAME_BULK_OUT;
	ep->driver_data          = epinf;
	
	ep                       = p_this->ptbl_ep[SICD_EP_IDX_INT_IN];
	epinf                    = &(p_this_ex->ep_info[SICD_EP_IDX_INT_IN]);
	epinf->ep_no             = 0x83;
	epinf->stalled           = FALSE;
	epinf->u.evt = p_this_ex->sicd_evt_req_list;
	for (i=0; i<SICD_MAX_EVT_REQ_NUM; i++) {
		epinf->u.evt[i].index      = i;
		epinf->u.evt[i].requesting = FALSE;
		epinf->u.evt[i].req        = NULL;
		epinf->u.evt[i].evt_buffer = NULL;
	}
	ep->name                 = SICD_EP_NAME_INT_IN;
	ep->driver_data          = epinf;
	
	// print EP info
	SICD_INF("---EP information--------\n");
	for(i=0; i < SICD_EP_IDX_MAX; i++) {
		ep = p_this->ptbl_ep[i];
		SICD_INF("EP%d: usage=%s, ep=%p, maxp=%d\n", i+1, ep->name, ep, ep->maxpacket);
	}
    
    SICD_EP_LOCK_OFF();
    
    SICD_STATUS_LOCK_ON();
	sicd_set_status(SICD_STATUS_BUSY);
	SICD_STATUS_LOCK_OFF();
	
	if( p_drv->start_ext_info == USB_GCORE_STARTEXT_BYDUAL){
		sicd_call_user_cb(SICD_CBID_START, USBG_STILL_EVENT_START_WITH_DUAL, NULL, 0);
	}
	else{
		sicd_call_user_cb(SICD_CBID_START, USBG_STILL_EVENT_START_NORMAL, NULL, 0);
	}
}

/**
 *	@brief		GadgetCore callback - stop
 *	@param		p_drv		function driver struct
 */
static void core_cb_stop( struct usb_gadget_func_driver* p_drv )
{
	int i;
	struct func_data    *p_this    = sicd_get_this();
	struct func_data_ex *p_this_ex = sicd_get_this_ex();

	SICD_API();

	SICD_EP_LOCK_ON();
	
	sicd_ep_dequeue_all();

	if (sicd_check_req_comp_waiting()) {
		if (wait_for_completion_interruptible(&(p_this_ex->req_wait_completion))) {
			SICD_INF("interrupted completion\n");
		}
	}

	// ep info initalize
	usb_ep_free_request(p_this->ptbl_ep[SICD_EP_IDX_BULK_OUT],
						p_this_ex->ep_info[SICD_EP_IDX_BULK_OUT].u.data.req);

	usb_ep_free_request(p_this->ptbl_ep[SICD_EP_IDX_BULK_IN],
						p_this_ex->ep_info[SICD_EP_IDX_BULK_IN].u.data.req);

	for (i=0; i<SICD_MAX_EVT_REQ_NUM; i++) {
		sicd_free_evt_req_info(p_this->ptbl_ep[SICD_EP_IDX_INT_IN],
			&p_this_ex->ep_info[SICD_EP_IDX_INT_IN].u.evt[i]);
	}

	memset(p_this_ex->ep_info, 0, sizeof(struct sicd_ep_info) * SICD_EP_IDX_MAX);

	usbg_cmn_clear_ep_list(p_this);
	
	SICD_EP_LOCK_OFF();
	
	SICD_STATUS_LOCK_ON();
	sicd_set_status(SICD_STATUS_BUSY);
	SICD_STATUS_LOCK_OFF();
	
	sicd_call_user_cb(SICD_CBID_STOP, 0, NULL, 0);
}

/**
 *	@brief		GadgetCore callback - set halt
 *	@param		p_drv		function driver struct
 *	@param		p_ep		stalled ep
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 */
static int core_cb_set_halt( struct usb_gadget_func_driver* p_drv,
					  struct usb_ep* p_ep )
{
	int err = 0;
	struct sicd_ep_info *ep_info = sicd_get_ep_info(p_ep);
	
	SICD_API();
	
	sicd_ep_dequeue_bulk();
	
	SICD_EP_LOCK_ON();
	
	// stall flag <- ON
	ep_info = (struct sicd_ep_info *)p_ep->driver_data;
	ep_info->stalled = TRUE;
	
	// dequeue requests
	sicd_ep_dequeue(p_ep);
	
	if(usb_ep_set_halt(p_ep) != 0){
		err = USB_GADGETCORE_STALL;
	}
	
	SICD_EP_LOCK_OFF();
	
	sicd_call_user_cb(SICD_CBID_EP_SET_HALT, 0, NULL, 0);
	
	return err;
}

/**
 *	@brief		GadgetCore callback - clear halt
 *	@param		p_drv		function driver struct
 *	@param		p_ep		stalled ep
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 */
static int core_cb_clear_halt( struct usb_gadget_func_driver* p_drv,
						struct usb_ep* p_ep )
{
	int err = 0;
	struct sicd_ep_info *ep_info = sicd_get_ep_info(p_ep);
	struct func_data_ex *p_this_ex = sicd_get_this_ex();
	
	SICD_API();
	
	SICD_EP_LOCK_ON();
	
	// stall flag <- OFF
	if(ep_info) {
		ep_info->stalled = FALSE;
	}
	
	/* <framework>	*/
	if(usb_ep_clear_halt(p_ep) != 0){
		err = USB_GADGETCORE_STALL;
	}
	
	if(p_this_ex->ep_info[SICD_EP_IDX_BULK_OUT].stalled == FALSE &&
	   p_this_ex->ep_info[SICD_EP_IDX_BULK_IN].stalled  == FALSE &&
	   p_this_ex->ep_info[SICD_EP_IDX_INT_IN].stalled   == FALSE ) {
	
		// all EP stall cleared
		sicd_call_user_cb(SICD_CBID_EP_CLR_HALT, 0, NULL, 0);
	}
	
	SICD_EP_LOCK_OFF();
	
	return err;
}

/**
 *	@brief		GadgetCore callback - class request
 *	@param		p_drv		function driver struct
 *	@param		p_ctrl		USB setup packet
 *	@param		ep0			EP0 ep struct
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 *	@retval		USB_GADGETCORE_THRU		hand over this request to another driver 
 */
static int core_cb_class( struct usb_gadget_func_driver* p_drv,
					   const struct usb_ctrlrequest* p_ctrl,
					   struct usb_ep* ep0 )
{
	int err = 0;

	SICD_API();
	
	// Is this Still Image Capture Device class request?
	if(!(p_ctrl->bRequestType & USB_TYPE_CLASS) ||
	   !(p_ctrl->bRequestType & USB_RECIP_INTERFACE)) {
		SICD_INF("bmRequestType is not for SICD\n");
		// this request if NOT for us(SICD)
		return USB_GADGETCORE_THRU;
	}
	
	SICD_INF("bRequest is 0x%x\n", p_ctrl->bRequest);
	
	switch(p_ctrl->bRequest) {
		case SICD_B_REQUEST_CANCEL:
			// Cancel_Request
			err = sicd_class_cancel(p_ctrl, ep0);
			break;
		case SICD_B_REQUEST_DEVICE_RESET:
			// Device_Reset_Request
			err = sicd_class_device_reset(p_ctrl, ep0);
			break;
		case SICD_B_REQUEST_GET_DEVICE_STATUS:
			// Get_Device_Status
			err = sicd_class_get_device_status(p_ctrl, ep0);
			break;
		case SICD_B_REQUEST_GET_EX_EVENT_DATA:
			// we NOT support this request, do not "break"
		default:
			// unknown request
			err = USB_GADGETCORE_STALL;
			break;
	}
	
	return err;
}

#ifdef SICD_CFG_HANDLE_VENDOR_REQUEST

/**
 *	@brief		GadgetCore callback - vendor request
 *	@param		p_drv		function driver struct
 *	@param		p_ctrl		USB setup packet
 *	@param		ep0			EP0 ep struct
 *	@retval		0						success
 *	@retval		USB_GADGETCORE_STALL	STALL
 *	@retval		USB_GADGETCORE_THRU		hand over this request to another driver 
 */
static int core_cb_vendor( struct usb_gadget_func_driver* p_drv,
						const struct usb_ctrlrequest* p_ctrl,
						struct usb_ep* ep0 )
{
	SICD_API();

	// nop for now
	return USB_GADGETCORE_THRU;

}

#endif // SICD_CFG_HANDLE_VENDOR_REQUEST

/**
 *	@brief		SICD core ioctl - probe
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 */
static int ioctl_probe(struct file *file, unsigned long arg)
{
	int err;
	struct usb_gadget_func_driver* reginfo;
	struct probe_info	*p_info    = (struct probe_info *)arg;
	struct func_data	*p_this    = sicd_get_this();
	int ifnum;

	SICD_FNC();
	
	// we do not allow multi I/F per 1 instance
	p_info->uc_func_info_interface_num = 1;
	
	if(usbg_cmn_copy_probe_info(p_this, p_info)){
		SICD_ERR("copy probe info\n");
		return -ENOBUFS;
	}
	
	ifnum = p_this->info.uc_func_info_interface_num;
	reginfo     = &(p_this->core_reg_info);
	reginfo->function		= SICD_FUNC_NAME;
	reginfo->context		= p_this;
	reginfo->start			= core_cb_start;
	reginfo->stop			= core_cb_stop;
	reginfo->class			= core_cb_class;
#ifdef SICD_CFG_HANDLE_VENDOR_REQUEST
	reginfo->vendor			= core_cb_vendor;
#endif
	reginfo->ep_set_halt	= core_cb_set_halt;
	reginfo->ep_clear_halt	= core_cb_clear_halt;
	reginfo->suspend		= core_cb_nop;
	reginfo->resume			= core_cb_nop;
    reginfo->config = 
        p_this->info.tbl_interface[0].uc_configuration;
    reginfo->interface = 
        p_this->info.tbl_interface[0].uc_interface;
	
	SICD_INF("register %s, cfg=%d, if=%d\n",
				reginfo->function,
				reginfo->config,
				reginfo->interface);
				
	err = usb_gadgetcore_register_driver(reginfo);
	if(err) {
		SICD_INF("usb_gadgetcore_register_driver error=%d\n",err);
	}
	
	p_this->info.HasExtendedTblIf = false;
	
    if( p_this->info.extended_tbl_interface ){
        
        SICD_INF("Register extended_tbl_interface.\n");
        
        p_this->info.HasExtendedTblIf = true;
        
        memcpy( &p_this->extended_core_reg_info, &p_this->core_reg_info, sizeof( struct usb_gadget_func_driver ) );
        
        reginfo = &(p_this->extended_core_reg_info);
        
        reginfo->config = 
            p_this->info.extended_tbl_interface[0].uc_configuration;
            
        reginfo->interface = 
            p_this->info.extended_tbl_interface[0].uc_interface;
        
        SICD_INF("register %s, cfg=%d, if=%d\n",
            reginfo->function,
            reginfo->config,
            reginfo->interface);
        
        err = usb_gadgetcore_register_driver( reginfo );
        
        if( err ) {
            
            SICD_INF("usb_gadgetcore_register_driver error=%d\n",err);
            
        }
        
    }
	
	return err;
}

/**
 *	@brief		SICD core ioctl - set status
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 */
static int ioctl_set_status(struct file *file, unsigned long arg)
{
	int err;
	struct usbg_still_status st;
	
	err = copy_from_user(&st, (struct usbg_still_status *)arg, sizeof(struct usbg_still_status));
	if(!err) {
		SICD_STATUS_LOCK_ON();
		sicd_set_status(st.status);
		SICD_STATUS_LOCK_OFF();
	}
	
	return 0;
}

/**
 *	@brief		SICD core ioctl - remove
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 */
static int ioctl_remove(struct file *file, unsigned long arg)
{
	int ret;
	struct func_data* p_this = sicd_get_this();

	SICD_FNC();

    if( p_this->info.HasExtendedTblIf == true ){
        usb_gadgetcore_unregister_driver( &p_this->extended_core_reg_info );
    }
    
	ret = usb_gadgetcore_unregister_driver(&(p_this->core_reg_info));
	usbg_cmn_clear_probe_info(p_this);
	
	return ret;
}

/**
 *	@brief		SICD core ioctl - set error
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 */
static int ioctl_set_error(struct file *file, unsigned long arg)
{
	int err;
	struct usbg_still_status st;

	SICD_FNC();

	err = copy_from_user(&st, (struct usbg_still_status *)arg, sizeof(struct usbg_still_status));
	if(!err) {
        SICD_STATUS_LOCK_ON();
		sicd_set_status(st.status);
		SICD_STATUS_LOCK_OFF();
	}
	
//	printk("ioctl_set_error status=%d, error=%d\n", st.status, st.error);

	// Bulk-In/Bulk-Out̗Epɑ΂STALLKv
	// PIMA15740 7.2.1.2 Device BehaviorQ
	st.error = USBG_IOC_ERR_RECV | USBG_IOC_ERR_SEND;
	
	if(st.error & USBG_IOC_ERR_RECV) {
		err = sicd_bulk_out_halt();
		if(err) {
			SICD_ERR("BULK-OUT ep halt fail\n");
		}
	}
	
	if(st.error & USBG_IOC_ERR_SEND) {
		err = sicd_bulk_in_halt();
		if(err) {
			SICD_ERR("BULK-IN ep halt fail\n");
		}
	}
	
	sicd_ep_dequeue_bulk();
	
	return err;
}

/**
 *	@brief		SICD core ioctl - Gadget functin driver common
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 *	@retval		-EFAULT		segmentation fault
 *	@retval		-ENOTTY		invalid ioctl no.
 */
static int ioctl_cmn(struct file *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;

//	SICD_FNC();

	// valid ioctl no?
	if( _IOC_NR(cmd) > USB_IOC_CMN_NBROF ){
		SICD_ERR("invalid ioctl seq. number\n");
		return -ENOTTY;
	}
	
	// read/write access?
	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
	} else if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
	}
	
	// access ok?
	if(err){
		SICD_ERR("invalid ioctl dir\n");
		return -EFAULT;
	}
	
	// ioctl for what?
	switch (cmd) {
		case USB_IOC_CMN_PROBE:
			err = ioctl_probe(file, arg);
			break;
		case USB_IOC_CMN_REMOVE:
			err = ioctl_remove(file, arg);
			break;
		default:
			SICD_ERR("invalid ioctl seq. number\n");
			err = -ENOTTY;
			break;
	}
	
	return err;
}

/**
 *	@brief		SICD core ioctl - SICD protocol
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 *	@retval		-EFAULT		segmentation fault
 *	@retval		-ENOTTY		invalid ioctl no.
 */
static int ioctl_still(struct file *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;

//	SICD_FNC();

	// valid ioctl no?
	if( _IOC_NR(cmd) > USBG_IOC_STILL_NBROF ){
		SICD_ERR("invalid ioctl seq. number\n");
		return -ENOTTY;
	}
	
	// read/write access?
	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
	} else if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
	}
	
	// access ok?
	if(err){
		SICD_ERR("invalid ioctl dir\n");
		return -EFAULT;
	}
	
	// ioctl for what?
	switch (cmd) {
		case USBG_IOC_STILL_SET_STATUS:
			err = ioctl_set_status(file, arg);
			break;
		case USBG_IOC_STILL_SET_ERROR:
			err = ioctl_set_error(file, arg);
			break;
		default:
			SICD_ERR("invalid ioctl seq. number\n");
			err = -ENOTTY;
			break;
	}
	
	return err;
}


/**
 *	@brief		SICD core driver open
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR sicd_core_open(UDIF_FILE *filp)
{
	struct file fd;
	struct file *file = &fd;
	struct func_data_ex *p_this_still = sicd_get_this_ex();

	SICD_API();

    SICD_RST_REQ_LOCK_ON();
	p_this_still->reset_request_flag = FALSE;
    SICD_RST_REQ_LOCK_OFF();

	sicd_link_this_to_file(file);
    udif_file_data(filp) = file->private_data;

	return UDIF_ERR_OK;
}

/**
 *	@brief		SICD core driver release
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR sicd_core_release(UDIF_FILE *filp)
{
	SICD_API();

	return UDIF_ERR_OK;
}

/**
 *	@brief		SICD core ioctl
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		cmd			ioctl no.
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOTTY		wrong ioctl no.
 */
UDIF_ERR sicd_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
	struct file fd;
	struct file *file = &fd;
	unsigned int cmd = param->cmd;
	unsigned long arg = param->arg;
	int err;

//	SICD_API();
	
    file->private_data = udif_file_data(filp);
	if( _IOC_TYPE(cmd) == USB_IOC_CMN ){
		// common ioctl
		err = ioctl_cmn(file, cmd, arg );
	} else if( _IOC_TYPE(cmd) == USB_IOC_SICD_CORE ) {
		// still image ioctl
		err = ioctl_still(file, cmd, arg );
	}
	else {
		// invalid ioctl
		SICD_ERR("invalid ioctl magic number\n");
		err = UDIF_ERR_NOTTY;
	}
    udif_file_data(filp) = file->private_data;
	
	return (UDIF_ERR)err;
}
