/*
 * This is fake extcon. For USB to set host or device mode by sysfs(echo 0 or 1 to /sys/class/xxx)
 * Copyright 2018 Sony Imaging Products & Solutions Inc
 * Copyright 2018 Sony Corporation
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
*/

#include <linux/extcon.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/acpi.h>
#include <linux/of.h>
#include <linux/property.h>
#include <linux/usb.h>

#define USB_TIMEOUT   (30*HZ)  /* 30 second timeout*/


struct usb_fake_mode_extcon_info {
	struct device *dev;
	struct extcon_dev *edev;
};

struct work_struct work;
struct work_struct work_host;
bool      condition;
bool      condition_host;
wait_queue_head_t wait_open;
wait_queue_head_t wait_open_host;

static const unsigned int usb_fake_mode_extcon_cable[] = {
	EXTCON_USB,
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

static void usb_fake_set_mode_extcon(struct extcon_dev *edev, int mode)
{
	//printk("mode = %d\n", mode);
	/*0 : Host,  1 : Device*/
	if(mode == 0)
	{
		extcon_set_state_sync(edev, EXTCON_USB_HOST, true);
	}
	else
	{
		extcon_set_state_sync(edev, EXTCON_USB_HOST, false);
	}
}

static unsigned int usb_mode;
static ssize_t usb_mode_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	/*0 : Host,  1 : Device*/
	ssize_t s;
	s = scnprintf(buf, PAGE_SIZE, "Current USB Mode = %d\n(0 : Host mode, 1 : Device mode)\n", usb_mode);
	s += scnprintf(buf+s, PAGE_SIZE - s, "You can 'echo 0', USB mode is Host mode\n");
	s += scnprintf(buf+s, PAGE_SIZE - s, "You can 'echo 1', USB mode is Device mode \n");
	return s;
	return scnprintf(buf, PAGE_SIZE, "%d\n", usb_mode);
}

static ssize_t usb_mode_store(struct device *parent, struct device_attribute *attr, const char *buf, size_t len)
{
	ssize_t result;
	struct extcon_dev *edev = dev_get_drvdata(parent);

	result = sscanf(buf, "%d", &usb_mode);
	if (result == 0)
		return -EINVAL;

	condition = false;
	condition_host = false;
	usb_fake_set_mode_extcon(edev, usb_mode);
	/*for DRD, abnormal case*/
	if(usb_mode == 0) /*FPGA is host mode*/
		schedule_work(&work);
	else/*FPGA is device mode*/
		schedule_work(&work_host);

	return len;
}
static DEVICE_ATTR_RW(usb_mode);

static struct attribute *usb_mode_dev_attrs[] = {
	&dev_attr_usb_mode.attr,
	NULL,
};

/*for DRD, abnormal case*/
int timeout_usb_ncb(struct notifier_block *nb, unsigned long val, void *priv)
{
	int result = NOTIFY_OK;
	struct usb_device *priv1 = priv;

	switch (val)
	{
		case USB_DEVICE_ADD:
			printk("add usb device\n");
			//printk("add usb device, level = %d, parent = 0x%x, portnum = %d\n",(u8)priv1->level, (struct usb_device *)priv1->parent, (u8)priv1->portnum);
			/*if priv1->parent != NULL => have device be attach*/
			if(priv1->parent != NULL)
			{
				condition = true;
				wake_up(&wait_open);
			}
			break;
		default:
			printk("other usb event\n");
			result = NOTIFY_BAD;
	}
	return result;
}
/*for DRD, abnormal case*/
static struct notifier_block timeout_usb_notifier = {
	.notifier_call = timeout_usb_ncb,
	.priority = INT_MAX /* Need to be called first of all */
};
/*for DRD, abnormal case*/
static void timeout_usb_work(struct work_struct *work)
{
	int ret;
	printk("timeout_usb_work()\n");
	ret = wait_event_timeout(wait_open, condition, USB_TIMEOUT);
	if (ret == 0)
		printk("=> usb timeout !!!, no usb device connect to usb host mode.\n");

}
/*------------------------------------------------------------------------------------------*/
/*for DRD, abnormal case*/
int timeout_usb_ncb_add_host(struct notifier_block *nb, unsigned long val, void *priv)
{
	int result = NOTIFY_OK;

	switch (val)
	{
		case USB_HOST_ADD:
			printk("add usb host\n");
			condition_host = true;
			wake_up(&wait_open_host);
			break;
		default:
			printk("other usb event!\n");
			result = NOTIFY_BAD;
	}
	return result;
}
/*for DRD, abnormal case*/
static struct notifier_block timeout_usb_notifier_add_host = {
	.notifier_call = timeout_usb_ncb_add_host,
	.priority = INT_MAX /* Need to be called first of all */
};
/*for DRD, abnormal case*/
static void timeout_usb_work_add_host(struct work_struct *work)
{
	int ret;
	printk("timeout_usb_work_add_host()\n");
	ret = wait_event_timeout(wait_open_host, condition_host, USB_TIMEOUT);
	if (ret == 0)
		printk("=> usb timeout !!!, no usb host connect to usb device mode.\n");

}


ATTRIBUTE_GROUPS(usb_mode_dev);
static int usb_fake_mode_extcon_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct usb_fake_mode_extcon_info *info;
	int ret;
printk("usb_fake_mode_extcon_probe()\n");
	if (!np && !ACPI_HANDLE(dev))
		return -EINVAL;

	info = devm_kzalloc(&pdev->dev, sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	info->dev = dev;

	info->edev = devm_extcon_dev_allocate(dev, usb_fake_mode_extcon_cable);
	if (IS_ERR(info->edev)) {
		dev_err(dev, "failed to allocate extcon device\n");
		return -ENOMEM;
	}

	info->edev->dev.groups = usb_mode_dev_groups;

	ret = devm_extcon_dev_register(dev, info->edev);
	if (ret < 0) {
		dev_err(dev, "failed to register extcon device\n");
		return ret;
	}

	//Get default mode from device tree. /*0: host mode, 1: device mode*/
	device_property_read_u32(&pdev->dev, "usb_mode",&usb_mode);
	usb_fake_set_mode_extcon(info->edev, usb_mode);

	/*for DRD, abnormal case*/
	usb_register_notify(&timeout_usb_notifier);
	usb_gadget_register_notify(&timeout_usb_notifier_add_host);
	INIT_WORK(&work, timeout_usb_work);
	INIT_WORK(&work_host, timeout_usb_work_add_host);
	init_waitqueue_head(&wait_open);
	init_waitqueue_head(&wait_open_host);


	platform_set_drvdata(pdev, info);
	device_init_wakeup(dev, true);

	return 0;
}

static int usb_fake_mode_extcon_remove(struct platform_device *pdev)
{
	device_init_wakeup(&pdev->dev, false);
	/*for DRD, abnormal case*/
	usb_unregister_notify(&timeout_usb_notifier);
	usb_gadget_unregister_notify(&timeout_usb_notifier_add_host);
	return 0;
}

static const struct of_device_id usb_fake_mode_extcon_dt_match[] = {
	{ .compatible = "linux,extcon-usb-fake_mode", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, usb_fake_mode_extcon_dt_match);

static const struct platform_device_id usb_fake_mode_extcon_platform_ids[] = {
	{ .name = "extcon-usb-fake_mode", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(platform, usb_fake_mode_extcon_platform_ids);

static struct platform_driver usb_fake_mode_extcon_driver = {
	.probe		= usb_fake_mode_extcon_probe,
	.remove		= usb_fake_mode_extcon_remove,
	.driver		= {
		.name	= "extcon-usb-fake_mode",
		.of_match_table = usb_fake_mode_extcon_dt_match,
	},
	.id_table = usb_fake_mode_extcon_platform_ids,
};

//module_platform_driver(usb_fake_mode_extcon_driver);

static int __init usb_fake_mode_extcon_init(void)
{
	return platform_driver_register(&usb_fake_mode_extcon_driver);
}

subsys_initcall(usb_fake_mode_extcon_init);

