/*
 *  log.c
 *
 *  Copyright 2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "log.h"
#include <linux/string.h>
#include <linux/udif/timer.h>

static int log_state;

void log_init(struct log *p)
{
	p->put = p->cnt = 0;
	log_state = 1;
	smp_mb();
}

void log_stop(void)
{
	log_state = 0;
	smp_mb();
}

void log_put(struct log *p, void *data, int size)
{
	smp_mb();
	if (!log_state)
		return;
	if (size > LOG_DATALEN)
		size = LOG_DATALEN;
	memcpy(p->ent[p->put].data, data, size);
	p->ent[p->put].timestamp = udif_read_freerun();
	if (++p->put == LOG_SIZE) {
		p->put = 0;
	}
	if (p->cnt < LOG_SIZE) {
		p->cnt++;
	}
	smp_mb();
}

static void log_dump1(struct log_entry *p)
{
	int i;

	printk(KERN_ERR "%lld:", udif_cycles_to_usecs(p->timestamp));
	for (i = 0; i < LOG_DATALEN; i++) {
		printk(KERN_CONT "%02x ", p->data[i]);
	}
	printk(KERN_CONT "\n");
}

void log_dump(struct log *p)
{
	int get, i;

	smp_mb();
	get = (LOG_SIZE == p->cnt) ? p->put : 0;
	for (i = 0; i < p->cnt; i++) {
		log_dump1(&p->ent[get]);
		if (++get == LOG_SIZE)
			get = 0;
	}
}
