/*
**
*Copyright 2018 Sony Imaging Products & Solutions Inc
*Copyright 2018 Sony Corporation
**
*/
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/of_pci.h>
#include <linux/of_address.h>
#include <linux/pci.h>
#include <linux/phy/phy.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/resource.h>
#include <linux/types.h>
#include <linux/mfd/syscon.h>
#include <linux/regmap.h>
#include <linux/reset.h>
#include <linux/clk.h>

#include "pcie-designware.h"

#define EXP_CAP_ID_OFFSET				0x70
#define EP_INT1MASK  (RX_CFGWR_EVENT|LINK_UP_FALLING_EDGE|LINK_UP_RISING_EDGE)
#define EP_PHY_READY_MAX_TRY  200

struct cxd900xx_pcie {
    struct dw_pcie		*pcie;
    int			phy_count;	/* DT phy-names count */
    struct phy		**phy;
    int			link_gen;
    struct irq_domain	*irq_domain;
    enum dw_pcie_device_mode mode;
#ifndef CONFIG_ARCH_CXD900XX_FPGA
    struct reset_control* prst[E_PLAT_RST_TOTAL];
    struct clk *pclk[E_PLAT_CLK_TOTAL];
    u32 lanes;
#endif
    unsigned int irq;
};

#define to_cxd900xx_pcie(x)	dev_get_drvdata((x)->dev)

#ifndef CONFIG_ARCH_CXD900XX_FPGA
static const char* rst_ctrl_name[E_PLAT_RST_TOTAL] = {"rst_pwrup","rst_button","rst_pe","rst_apb","rst_axi2avb_pci","rst_axi2avb_pci_i0"};
static const char* clk_gate_name[E_PLAT_CLK_TOTAL] = {"apb_pclk", "dbi_aclk", "slv_aclk", "mstr_aclk", "aux_clk", "axi2avb_pcie0", "axi2avb_pcie0"};
static int cxd900xx_pcie_suspend_sys(struct cxd900xx_pcie* cxd900xx);

static int cxd900xx_pcie_setup_sys(struct cxd900xx_pcie* cxd900xx)
{
    int ret = 0, max_try = 0;
    enum dw_pcie_phy_status status = DW_PCIE_PHY_IDLE;

    //assert APB reset
    ret |= reset_control_assert(cxd900xx->prst[E_PLAT_PCIE_APB_RST]);

    //clock enable
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_PCIE_APB_PCLK]);
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_PCIE_DBI_ACLK]);
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_PCIE_SLV_ACLK]);
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_PCIE_MSTR_ACK]);
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_PCIE_AUX_CLK]);

    //wait 10us idle for 1Mhz aux_clk released form clock gated cell
    udelay(10);

    //reset release
    ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_PCIE_PWRUP_RST]);
    ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_PCIE_BUTTON_RST]);
    ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_PCIE_PE_RST]);
    ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_PCIE_APB_RST]);

    //set type as EP
    dw_pcie_set_Mode(cxd900xx->pcie, DW_PCIE_AS_EP);
#ifdef CONFIG_PCIEASPM
            dw_pcie_writel_phy(cxd900xx->pcie, GENERAL_CORE_CTRL_OFF,\
                    dw_pcie_readl_phy(cxd900xx->pcie,GENERAL_CORE_CTRL_OFF) |APP_CLK_REQ_N);
#endif

    //turn on axi fabric pcie clock/reset
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_AXI2AVB_PCIE0_CLK]);
    ret |= clk_prepare_enable(cxd900xx->pclk[E_PLAT_AXI2AVB_PCIE1_CLK]);

    if(reset_control_status(cxd900xx->prst[E_PLAT_AXI2AVB_PCI_RST]))
        ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_AXI2AVB_PCI_RST]);

    if(reset_control_status(cxd900xx->prst[E_PLAT_AXI2AVB_PCI_I0_RST]))
        ret |= reset_control_deassert(cxd900xx->prst[E_PLAT_AXI2AVB_PCI_I0_RST]);

    do
    {
        msleep(1);
        dw_pcie_get_phy_status(cxd900xx->pcie, cxd900xx->lanes, &status);

        max_try++;
        if((max_try > EP_PHY_READY_MAX_TRY) && (status == DW_PCIE_PHY_IDLE))
        {
            cxd900xx_pcie_suspend_sys(cxd900xx);
	 printk("phy init timeout\n");
            return -1;
        }
    } while(status == DW_PCIE_PHY_IDLE);

    return ret;
}

static int cxd900xx_pcie_suspend_sys(struct cxd900xx_pcie* cxd900xx)
{
    int ret = 0;
    u32 val;

    //assert reset
    ret |= reset_control_assert(cxd900xx->prst[E_PLAT_PCIE_PWRUP_RST]);
    ret |= reset_control_assert(cxd900xx->prst[E_PLAT_PCIE_BUTTON_RST]);
    ret |= reset_control_assert(cxd900xx->prst[E_PLAT_PCIE_PE_RST]);

    //set IDDQ=1
    val = dw_pcie_readl_phy(cxd900xx->pcie, PHY_CONFIG_0X18_OFF);
    val |= PHY_POWERDOWN;
    dw_pcie_writel_phy(cxd900xx->pcie, PHY_CONFIG_0X18_OFF, val);

    //gated clock
    clk_disable_unprepare(cxd900xx->pclk[E_PLAT_PCIE_APB_PCLK]);
    clk_disable_unprepare(cxd900xx->pclk[E_PLAT_PCIE_DBI_ACLK]);
    clk_disable_unprepare(cxd900xx->pclk[E_PLAT_PCIE_SLV_ACLK]);
    clk_disable_unprepare(cxd900xx->pclk[E_PLAT_PCIE_MSTR_ACK]);
    clk_disable_unprepare(cxd900xx->pclk[E_PLAT_PCIE_AUX_CLK]);

    return ret;
}
#endif

static irqreturn_t cxd900xx_pcie_irq_handler(int irq, void *arg)
{
    struct dw_pcie *pcie;
    u32 status;

    pcie = (struct dw_pcie *)arg;
    status  = dw_pcie_readl_phy(pcie, INTERRUPT1_STATUS_OFF);
    dw_pcie_clear_Interrupt1(pcie);

    if(status & RX_CFGWR_EVENT)
        printk("[Event] CfgWr Receive\n");

    if(status & LINK_UP_FALLING_EDGE)
        printk("[Event] Link Down\n");

    if(status & LINK_UP_RISING_EDGE)
        printk("[Event] Link Up\n");

    return IRQ_HANDLED;
}

static u64 cxd900xx_pcie_cpu_addr_fixup(u64 pci_addr)
{
	return pci_addr;
}

static void cxd900xx_pcie_stop_link(struct dw_pcie *pcie)
{
	dw_pcie_set_LTSSM_en(pcie, 0);
}

static int cxd900xx_pcie_establish_link(struct dw_pcie *pcie)
{
    struct  cxd900xx_pcie * cxd900xx = to_cxd900xx_pcie(pcie);
    struct device *dev = pcie->dev;
    u32 reg;
    u32 exp_cap_off = EXP_CAP_ID_OFFSET; //todo: need to check

    if (dw_pcie_link_up(pcie)) {
        dev_err(dev, "link is already up\n");
        return 0;
    }

    if ( cxd900xx->link_gen == 1) {
        reg = dw_pcie_readl_dbi(pcie, exp_cap_off + PCI_EXP_LNKCAP);
        if ((reg & PCI_EXP_LNKCAP_SLS) != PCI_EXP_LNKCAP_SLS_2_5GB) {
            reg &= ~((u32)PCI_EXP_LNKCAP_SLS);
            reg |= PCI_EXP_LNKCAP_SLS_2_5GB;
            dw_pcie_dbi_ro_wr_en(pcie);
            dw_pcie_writel_dbi(pcie, exp_cap_off +PCI_EXP_LNKCAP, reg);
            dw_pcie_dbi_ro_wr_dis(pcie);
        }

        reg = dw_pcie_readl_dbi(pcie, exp_cap_off + PCI_EXP_LNKCTL2);
        if ((reg & PCI_EXP_LNKCAP_SLS) != PCI_EXP_LNKCAP_SLS_2_5GB) {
            reg &= ~((u32)PCI_EXP_LNKCAP_SLS);
            reg |= PCI_EXP_LNKCAP_SLS_2_5GB;
            dw_pcie_dbi_ro_wr_en(pcie);
            dw_pcie_writel_dbi(pcie, exp_cap_off + PCI_EXP_LNKCTL2, reg);
            dw_pcie_dbi_ro_wr_dis(pcie);
        }
    }

    dw_pcie_set_LTSSM_en(pcie, 1);
    return 0;
}

static void dw_pcie_ep_reset_bar(struct dw_pcie *pcie, enum pci_barno bar)
{
	u32 reg;

	reg = PCI_BASE_ADDRESS_0 + (4 * bar);
	dw_pcie_writel_dbi2(pcie, reg, 0x0);
	dw_pcie_writel_dbi(pcie, reg, 0x0);
}

static void cxd900xx_pcie_ep_init(struct dw_pcie_ep *ep)
{
	struct dw_pcie *pcie = to_dw_pcie_from_ep(ep);
	enum pci_barno bar;

	for (bar = BAR_0; bar <= BAR_5; bar++)
		dw_pcie_ep_reset_bar(pcie, bar);
}

static int cxd900xx_pcie_raise_irq(struct dw_pcie_ep *ep,
				 enum pci_epc_irq_type type, u8 interrupt_num)
{
    struct dw_pcie *pcie = to_dw_pcie_from_ep(ep);
    u32 pos;
    u16 val;
    u32 LnkState;

    switch (type) {
        case PCI_EPC_IRQ_LEGACY:
            dev_err(pcie->dev, "Not support PCI_EPC_IRQ_LEGACY\n");
            break;
        case PCI_EPC_IRQ_MSI:
            pos = (u32)dw_pcie_find_capability(pcie, PCI_CAP_ID_MSI);
            if(pos)
            {
                val =( dw_pcie_readw_dbi(pcie, pos + PCI_MSI_FLAGS)  & PCI_MSI_FLAGS_QMASK) >>1;
                if(interrupt_num >= (1<<val))
                    return -EINVAL;
            }
            else
                return -EINVAL;

            LnkState = dw_pcie_readl_phy(pcie, PM_LTSSM_STATE) & PM_LINKST_IN_L1;

            if(LnkState)
            {
                dw_pcie_writel_phy(pcie,PM_APP_OFF, \
                        dw_pcie_readl_phy(pcie, PM_APP_OFF) | APP_REQ_EXIT_L1);
            }

            dw_pcie_writel_phy(pcie,MSI_MSG_TRIGGER_OFF, \
                    dw_pcie_readl_phy(pcie, MSI_MSG_TRIGGER_OFF) | (0x01<<interrupt_num));

            if(LnkState)
            {
                dw_pcie_writel_phy(pcie,PM_APP_OFF, \
                        dw_pcie_readl_phy(pcie, PM_APP_OFF) & (~APP_REQ_EXIT_L1));
            }

            break;
        case PCI_EPC_IRQ_MSIX:
            dev_err(pcie->dev, "Not support PCI_EPC_IRQ_MSIX\n");
            break;
        default:
            dev_err(pcie->dev, "UNKNOWN IRQ type\n");
    }

    return 0;
}

static struct dw_pcie_ep_ops pcie_ep_ops = {
	.ep_init = cxd900xx_pcie_ep_init,
	.raise_irq = cxd900xx_pcie_raise_irq,
};

static int cxd900xx_add_pcie_ep(struct cxd900xx_pcie *cxd900xx,
				     struct platform_device *pdev)
{
    int ret;
    struct dw_pcie_ep *ep;
    struct resource *res;
    struct device *dev = &pdev->dev;
    struct dw_pcie *pcie =  cxd900xx->pcie;
    struct device_node * np;
#ifndef CONFIG_ARCH_CXD900XX_FPGA
    u8 i;
#endif

    ep = &pcie->ep;
    ep->ops = &pcie_ep_ops;

    res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "ctrlreg");
    pcie->dbi_base = devm_ioremap(dev, res->start, resource_size(res));
    if (!pcie->dbi_base)
        return -ENOMEM;

    res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "ctrlreg_shadow");
    pcie->dbi_base2 = devm_ioremap(dev, res->start, resource_size(res));
    if (!pcie->dbi_base2)
        return -ENOMEM;

    res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phy_reg");
    pcie->phy_base  = devm_ioremap(dev, res->start, resource_size(res));
    if (!pcie->phy_base)
        return -ENOMEM;

    np = of_parse_phandle(dev->of_node, "memory-region", 0);
    if(!np)
        return -ENOMEM;

    if(of_address_to_resource(np,  0, res))
        return -ENOMEM;

    ep->phys_base = res->start;
    ep->addr_size = resource_size(res);

#ifndef CONFIG_ARCH_CXD900XX_FPGA
    for(i=0;i<E_PLAT_RST_TOTAL;i++)
    {
        if(i < E_PLAT_AXI2AVB_PCI_RST)
            cxd900xx->prst[i]=devm_reset_control_get(dev, rst_ctrl_name[i]);
        else
            cxd900xx->prst[i]=devm_reset_control_get_shared(dev, rst_ctrl_name[i]);

        if(IS_ERR(cxd900xx->prst[i]))
        {
            dev_err(dev,"Cannot get %s\n",rst_ctrl_name[i] );
            return -ENODEV;
        }
    }

    for(i=0;i<E_PLAT_CLK_TOTAL;i++)
    {
        cxd900xx->pclk[i]=devm_clk_get(dev, clk_gate_name[i]);
        if(IS_ERR(cxd900xx->pclk[i]))
        {
            dev_err(dev,"Cannot get %s\n",clk_gate_name[i] );
            return -ENODEV;
        }
    }

    ret = of_property_read_u32(dev->of_node, "num-lanes", &cxd900xx->lanes);
    if (ret)
        cxd900xx->lanes = 0;

    if(cxd900xx_pcie_setup_sys(cxd900xx))
        return -EPERM;
#else
    dw_pcie_set_Mode(pcie, DW_PCIE_AS_EP);
#endif

    dw_pcie_set_LTSSM_en(pcie, 0);

    ret = dw_pcie_ep_init(ep);

    if (ret) {
        dev_err(dev, "failed to initialize endpoint\n");
        return ret;
    }

    cxd900xx->irq = platform_get_irq(pdev, 0);
    if (cxd900xx->irq >= 0)
    {
        ret = devm_request_irq(dev, cxd900xx->irq,
                cxd900xx_pcie_irq_handler,
                IRQF_SHARED | IRQF_NO_THREAD,
                "dw-pcie-ep", pcie);

        if(!ret)
            dw_pcie_set_Interrupt1_en(pcie, EP_INT1MASK, 1);
    }

    return 0;
}

static const struct dw_pcie_ops dw_pcie_ops = {
	.cpu_addr_fixup = cxd900xx_pcie_cpu_addr_fixup,
	.start_link = cxd900xx_pcie_establish_link,
	.stop_link = cxd900xx_pcie_stop_link,
};

static const struct of_device_id of_cxd900xx_pcie_match[] = {
	{
		.compatible = "cxd900xx-pcie-ep",
	},
	{},
};

static int cxd900xx_pcie_probe(struct platform_device *pdev)
{
    int ret;
    struct dw_pcie *pcie;
    struct cxd900xx_pcie *cxd900xx;
    struct device *dev = &pdev->dev;
    //struct gpio_desc *reset;

    cxd900xx = devm_kzalloc(dev, sizeof(*cxd900xx), GFP_KERNEL);
    if (!cxd900xx)
        return -ENOMEM;

    pcie = devm_kzalloc(dev, sizeof(*pcie), GFP_KERNEL);
    if (!pcie)
        return -ENOMEM;

    pcie->dev = dev;
    pcie->ops = &dw_pcie_ops;

    cxd900xx->pcie = pcie;

    platform_set_drvdata(pdev, cxd900xx);

    pm_runtime_enable(dev);
    ret = pm_runtime_get_sync(dev);
    if (ret < 0) {
        dev_err(dev, "pm_runtime_get_sync failed\n");
        goto err_get_sync;
    }

#if 0
    reset = devm_gpiod_get_optional(dev, NULL, GPIOD_OUT_HIGH);
    if (IS_ERR(reset)) {
        ret = PTR_ERR(reset);
        dev_err(&pdev->dev, "gpio request failed, ret %d\n", ret);
        goto err_gpio;
    }
#endif
    cxd900xx->link_gen = 2;
    ret = cxd900xx_add_pcie_ep(cxd900xx, pdev);
#ifdef CONFIG_PCIE_DW_DEBUG
    dw_pcie_debugfs_init(pcie);
#endif

    return 0;

    //err_gpio:
    pm_runtime_put(dev);

err_get_sync:
    pm_runtime_disable(dev);

    return ret;
}

static int cxd900xx_pcie_remove(struct platform_device *pdev)
{
    struct  cxd900xx_pcie * cxd900xx = platform_get_drvdata(pdev);

#ifndef CONFIG_ARCH_CXD900XX_FPGA
    cxd900xx_pcie_suspend_sys(cxd900xx);
#endif
    dw_pcie_ep_exit(&cxd900xx->pcie->ep);

    return 0;
}

#ifdef CONFIG_PM_SLEEP
static int cxd900xx_pcie_suspend(struct device *dev)
{
	struct cxd900xx_pcie *cxd900xx = dev_get_drvdata(dev);
	struct dw_pcie *pcie = cxd900xx->pcie;
	u32 val;

	if (cxd900xx->mode != DW_PCIE_RC_TYPE)
		return 0;

	/* clear MSE */
	val = dw_pcie_readl_dbi(pcie, PCI_COMMAND);
	val &= ~PCI_COMMAND_MEMORY;
	dw_pcie_writel_dbi(pcie, PCI_COMMAND, val);

	return 0;
}

static int cxd900xx_pcie_resume(struct device *dev)
{
	struct cxd900xx_pcie *cxd900xx = dev_get_drvdata(dev);
	struct dw_pcie *pcie = cxd900xx->pcie;
	u32 val;

	if (cxd900xx->mode != DW_PCIE_RC_TYPE)
		return 0;

	/* set MSE */
	val = dw_pcie_readl_dbi(pcie, PCI_COMMAND);
	val |= PCI_COMMAND_MEMORY;
	dw_pcie_writel_dbi(pcie, PCI_COMMAND, val);

	return 0;
}

static int cxd900xx_pcie_suspend_noirq(struct device *dev)
{

	//do clk gating here

	return 0;
}

static int cxd900xx_pcie_resume_noirq(struct device *dev)
{
	//int ret;

	//open clk gating here

	return 0;
}
#endif

static const struct dev_pm_ops cxd900xx_pcie_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(cxd900xx_pcie_suspend, cxd900xx_pcie_resume)
	SET_NOIRQ_SYSTEM_SLEEP_PM_OPS(cxd900xx_pcie_suspend_noirq,
				      cxd900xx_pcie_resume_noirq)
};

static struct platform_driver cxd900xx_pcie_driver = {
	.driver = {
		.name	= "cxd900xx-pcie",
		.of_match_table = of_cxd900xx_pcie_match,
		.suppress_bind_attrs = true,
		.pm	= &cxd900xx_pcie_pm_ops,
	},
	.probe = cxd900xx_pcie_probe,
	.remove = cxd900xx_pcie_remove,
};
builtin_platform_driver(cxd900xx_pcie_driver);

