/*
 * Synopsys DesignWare PCIe host controller driver
 *
 * Copyright (C) 2013 Samsung Electronics Co., Ltd.
 *		http://www.samsung.com
 *
 * Author: Jingoo Han <jg1.han@samsung.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/delay.h>
#include <linux/of.h>
#include <linux/types.h>
#include <linux/snsc_boot_time.h>
#include <linux/udif/timer.h>

#include "pcie-designware.h"

/* PCIe Port Logic registers */
#define PLR_OFFSET			0x700
#define PCIE_PHY_DEBUG_R1		(PLR_OFFSET + 0x2c)
#define PCIE_PHY_DEBUG_R1_LINK_UP	(0x1 << 4)
#define PCIE_PHY_DEBUG_R1_LINK_IN_TRAINING	(0x1 << 29)

static void clear_phy_bits(struct dw_pcie *pci, u32 reg, u32 mask)
{
   u32 val;

   val = dw_pcie_readl_phy(pci, reg);
   val &= (~mask);
   dw_pcie_writel_phy( pci, reg, val);
}

static void set_phy_bits(struct dw_pcie *pci, u32 reg, u32 mask)
{
   u32 val;

   val = dw_pcie_readl_phy(pci, reg);
   val |= mask;
   dw_pcie_writel_phy( pci, reg, val);
}

u32 dw_pcie_readl_phy(struct dw_pcie *pci, u32 reg)
{
     u32 val = 0;

     val = readl(pci->phy_base + reg);
     PCIE_DW_HOST_TRACE_PHY(pci->dev->of_node->name,TRACE_READ, (unsigned long)reg, val);

     return val;
}

void dw_pcie_writel_phy(struct dw_pcie *pci, u32 reg,u32 val)
{
     writel(val, pci->phy_base + reg);
     PCIE_DW_HOST_TRACE_PHY(pci->dev->of_node->name,TRACE_WRITE, (unsigned long)reg, val);
}

int dw_pcie_read(void __iomem *addr, int size, u32 *val)
{
	register u32 dat;
	unsigned long long t1, t2, dt;
	unsigned long flags;

	if ((uintptr_t)addr & (size - 1)) {
		*val = 0;
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	if (size == 4) {
		local_irq_save(flags);
		t1 = udif_read_freerun();
		dat = readl(addr);
		t2 = udif_read_freerun();
		local_irq_restore(flags);
	} else if (size == 2) {
		local_irq_save(flags);
		t1 = udif_read_freerun();
		dat = readw(addr);
		t2 = udif_read_freerun();
		local_irq_restore(flags);
	} else if (size == 1) {
		local_irq_save(flags);
		t1 = udif_read_freerun();
		dat = readb(addr);
		t2 = udif_read_freerun();
		local_irq_restore(flags);
	} else {
		*val = 0;
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}
	*val = dat;
	dt = udif_cycles_to_usecs(t2 - t1);
	if (dt >= PCIE_STALL_THRESHOLD) {
		char buf[64];
		scnprintf(buf,sizeof buf,"PCIe:STALL:%s:%llu",__func__,dt);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		dump_stack();
	}

	return PCIBIOS_SUCCESSFUL;
}

int dw_pcie_write(void __iomem *addr, int size, u32 val)
{
	if ((uintptr_t)addr & (size - 1))
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (size == 4)
		writel(val, addr);
	else if (size == 2)
		writew(val, addr);
	else if (size == 1)
		writeb(val, addr);
	else
		return PCIBIOS_BAD_REGISTER_NUMBER;

	return PCIBIOS_SUCCESSFUL;
}

int dw_pcie_write_relaxed(void __iomem *addr, int size, u32 val)
{
	if ((uintptr_t)addr & (size - 1))
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (size == 4)
		writel_relaxed(val, addr);
	else if (size == 2)
		writew_relaxed(val, addr);
	else if (size == 1)
		writeb_relaxed(val, addr);
	else
		return PCIBIOS_BAD_REGISTER_NUMBER;

	return PCIBIOS_SUCCESSFUL;
}

u32 __dw_pcie_read_dbi(struct dw_pcie *pci, void __iomem *base, u32 reg,
		       size_t size)
{
    int ret;
    u32 val;

    if (pci->ops->read_dbi)
        return pci->ops->read_dbi(pci, base, reg, size);

    ret = dw_pcie_read(base + reg, size, &val);
    if (ret)
        dev_err(pci->dev, "read DBI address failed\n");

    if(TRACE_IATU(reg))
        PCIE_DW_HOST_TRACE_IATU(pci->dev->of_node->name, TRACE_READ, (int)((reg >>9)&0xff), (u8)((reg >>8)&0x01),(u8)(reg & 0xff >> 2), val);
    else
        PCIE_DW_HOST_TRACE_DBI(pci->dev->of_node->name,TRACE_READ, (u8)size, (unsigned long)reg, val);

    return val;
}

void __dw_pcie_write_dbi(struct dw_pcie *pci, void __iomem *base, u32 reg,
			 size_t size, u32 val)
{
    int ret;

    if (pci->ops->write_dbi) {
        pci->ops->write_dbi(pci, base, reg, size, val);
        return;
    }

    ret = dw_pcie_write(base + reg, size, val);
    if (ret)
        dev_err(pci->dev, "write DBI address failed\n");

    if(TRACE_IATU(reg))
        PCIE_DW_HOST_TRACE_IATU(pci->dev->of_node->name, TRACE_WRITE, (int)((reg >>9)&0xff), (u8)((reg >>8)&0x01),(u8)(reg & 0xff >> 2), val);
    else
        PCIE_DW_HOST_TRACE_DBI(pci->dev->of_node->name,TRACE_WRITE, (u8)size, (unsigned long)reg, val);
}

static u32 dw_pcie_readl_ob_unroll(struct dw_pcie *pci, u32 index, u32 reg)
{
	u32 offset = PCIE_GET_ATU_OUTB_UNR_REG_OFFSET(index);

	return dw_pcie_readl_dbi(pci, offset + reg);
}

static void dw_pcie_writel_ob_unroll(struct dw_pcie *pci, u32 index, u32 reg,
				     u32 val)
{
	u32 offset = PCIE_GET_ATU_OUTB_UNR_REG_OFFSET(index);

	dw_pcie_writel_dbi(pci, offset + reg, val);
}

static void dw_pcie_prog_outbound_atu_unroll(struct dw_pcie *pci, int index,
					     int type, u64 cpu_addr,
					     u64 pci_addr, u32 size)
{
	u32 retries, val;

    if(size < CX_ATU_MIN_REGION_SIZE)
    {
        dev_err(pci->dev, "outbound iATU is not being enabled\n");
	    return;
    }
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_LOWER_BASE,
				 lower_32_bits(cpu_addr));
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_UPPER_BASE,
				 upper_32_bits(cpu_addr));
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_LIMIT,
				 lower_32_bits(cpu_addr + size - 1));
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_LOWER_TARGET,
				 lower_32_bits(pci_addr));
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_UPPER_TARGET,
				 upper_32_bits(pci_addr));
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL1,
				 type);
	dw_pcie_writel_ob_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL2,
				 PCIE_ATU_ENABLE);

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_ob_unroll(pci, index,
					      PCIE_ATU_UNR_REGION_CTRL2);
		if (val & PCIE_ATU_ENABLE)
			return;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "outbound iATU is not being enabled\n");
}

void dw_pcie_prog_outbound_atu(struct dw_pcie *pci, int index, int type,
			       u64 cpu_addr, u64 pci_addr, u32 size)
{
	u32 retries, val;

	if (pci->ops->cpu_addr_fixup)
		cpu_addr = pci->ops->cpu_addr_fixup(cpu_addr);

	if (pci->iatu_unroll_enabled) {
		dw_pcie_prog_outbound_atu_unroll(pci, index, type, cpu_addr,
						 pci_addr, size);
		return;
	}

	dw_pcie_writel_dbi(pci, PCIE_ATU_VIEWPORT,
			   PCIE_ATU_REGION_OUTBOUND | index);
	dw_pcie_writel_dbi(pci, PCIE_ATU_LOWER_BASE,
			   lower_32_bits(cpu_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_UPPER_BASE,
			   upper_32_bits(cpu_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_LIMIT,
			   lower_32_bits(cpu_addr + size - 1));
	dw_pcie_writel_dbi(pci, PCIE_ATU_LOWER_TARGET,
			   lower_32_bits(pci_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_UPPER_TARGET,
			   upper_32_bits(pci_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_CR1, type);
	dw_pcie_writel_dbi(pci, PCIE_ATU_CR2, PCIE_ATU_ENABLE);

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_dbi(pci, PCIE_ATU_CR2);
		if (val & PCIE_ATU_ENABLE)
			return;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "outbound iATU is not being enabled\n");
}

static u32 dw_pcie_readl_ib_unroll(struct dw_pcie *pci, u32 index, u32 reg)
{
	u32 offset = PCIE_GET_ATU_INB_UNR_REG_OFFSET(index);

	return dw_pcie_readl_dbi(pci, offset + reg);
}

static void dw_pcie_writel_ib_unroll(struct dw_pcie *pci, u32 index, u32 reg,
				     u32 val)
{
	u32 offset = PCIE_GET_ATU_INB_UNR_REG_OFFSET(index);

	dw_pcie_writel_dbi(pci, offset + reg, val);
}

static int dw_pcie_prog_ib_atu2_unroll(struct dw_pcie *pci, int index,
				       u64 pci_addr, u64 size, u64 cpu_addr,
				       enum dw_pcie_as_type as_type)
{
	int type;
	u32 retries, val;

	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_LOWER_BASE,
				 lower_32_bits(pci_addr));
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_UPPER_BASE,
				 upper_32_bits(pci_addr));
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_LIMIT,
				 lower_32_bits(pci_addr + size - 1));
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_LOWER_TARGET,
				 lower_32_bits(cpu_addr));
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_UPPER_TARGET,
				 upper_32_bits(cpu_addr));

	switch (as_type) {
	case DW_PCIE_AS_MEM:
		type = PCIE_ATU_TYPE_MEM;
		break;
	case DW_PCIE_AS_IO:
		type = PCIE_ATU_TYPE_IO;
		break;
	default:
		return -EINVAL;
	}

	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL1, type);
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL2,
				 PCIE_ATU_ENABLE);

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_ib_unroll(pci, index,
					      PCIE_ATU_UNR_REGION_CTRL2);
		if (val & PCIE_ATU_ENABLE)
			return 0;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "inbound iATU is not being enabled\n");

	return -EBUSY;
}

int dw_pcie_prog_ib_atu2(struct dw_pcie *pci, int index,
			 u64 pci_addr, u64 size, u64 cpu_addr,
			 enum dw_pcie_as_type as_type)
{
	int type;
	u32 retries, val;

	if (pci->iatu_unroll_enabled)
		return dw_pcie_prog_ib_atu2_unroll(pci, index,
						   pci_addr, size,
						   cpu_addr, as_type);

	dw_pcie_writel_dbi(pci, PCIE_ATU_VIEWPORT, PCIE_ATU_REGION_INBOUND |
			   index);
	dw_pcie_writel_dbi(pci, PCIE_ATU_LOWER_BASE, lower_32_bits(pci_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_UPPER_BASE, upper_32_bits(pci_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_LIMIT,
			   lower_32_bits(pci_addr + size - 1));
	dw_pcie_writel_dbi(pci, PCIE_ATU_LOWER_TARGET, lower_32_bits(cpu_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_UPPER_TARGET, upper_32_bits(cpu_addr));

	switch (as_type) {
	case DW_PCIE_AS_MEM:
		type = PCIE_ATU_TYPE_MEM;
		break;
	case DW_PCIE_AS_IO:
		type = PCIE_ATU_TYPE_IO;
		break;
	default:
		return -EINVAL;
	}

	dw_pcie_writel_dbi(pci, PCIE_ATU_CR1, type);
	dw_pcie_writel_dbi(pci, PCIE_ATU_CR2, PCIE_ATU_ENABLE);

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_dbi(pci, PCIE_ATU_CR2);
		if (val & PCIE_ATU_ENABLE)
			return 0;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "inbound iATU is not being enabled\n");

	return -EBUSY;
}

static int dw_pcie_prog_inbound_atu_unroll(struct dw_pcie *pci, int index,
					   int bar, u64 cpu_addr,
					   enum dw_pcie_as_type as_type)
{
	int type;
	u32 retries, val;

	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_LOWER_TARGET,
				 lower_32_bits(cpu_addr));
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_UPPER_TARGET,
				 upper_32_bits(cpu_addr));

	switch (as_type) {
	case DW_PCIE_AS_MEM:
		type = PCIE_ATU_TYPE_MEM;
		break;
	case DW_PCIE_AS_IO:
		type = PCIE_ATU_TYPE_IO;
		break;
	default:
		return -EINVAL;
	}

	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL1, type);
	dw_pcie_writel_ib_unroll(pci, index, PCIE_ATU_UNR_REGION_CTRL2,
				 PCIE_ATU_ENABLE |
				 PCIE_ATU_BAR_MODE_ENABLE | (bar << 8));

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_ib_unroll(pci, index,
					      PCIE_ATU_UNR_REGION_CTRL2);
		if (val & PCIE_ATU_ENABLE)
			return 0;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "inbound iATU is not being enabled\n");

	return -EBUSY;
}

int dw_pcie_prog_inbound_atu(struct dw_pcie *pci, int index, int bar,
			     u64 cpu_addr, enum dw_pcie_as_type as_type)
{
	int type;
	u32 retries, val;

	if (pci->iatu_unroll_enabled)
		return dw_pcie_prog_inbound_atu_unroll(pci, index, bar,
						       cpu_addr, as_type);

	dw_pcie_writel_dbi(pci, PCIE_ATU_VIEWPORT, PCIE_ATU_REGION_INBOUND |
			   index);
	dw_pcie_writel_dbi(pci, PCIE_ATU_LOWER_TARGET, lower_32_bits(cpu_addr));
	dw_pcie_writel_dbi(pci, PCIE_ATU_UPPER_TARGET, upper_32_bits(cpu_addr));

	switch (as_type) {
	case DW_PCIE_AS_MEM:
		type = PCIE_ATU_TYPE_MEM;
		break;
	case DW_PCIE_AS_IO:
		type = PCIE_ATU_TYPE_IO;
		break;
	default:
		return -EINVAL;
	}

	dw_pcie_writel_dbi(pci, PCIE_ATU_CR1, type);
	dw_pcie_writel_dbi(pci, PCIE_ATU_CR2, PCIE_ATU_ENABLE
			   | PCIE_ATU_BAR_MODE_ENABLE | (bar << 8));

	/*
	 * Make sure ATU enable takes effect before any subsequent config
	 * and I/O accesses.
	 */
	for (retries = 0; retries < LINK_WAIT_MAX_IATU_RETRIES; retries++) {
		val = dw_pcie_readl_dbi(pci, PCIE_ATU_CR2);
		if (val & PCIE_ATU_ENABLE)
			return 0;

		usleep_range(LINK_WAIT_IATU_MIN, LINK_WAIT_IATU_MAX);
	}
	dev_err(pci->dev, "inbound iATU is not being enabled\n");

	return -EBUSY;
}

void dw_pcie_disable_atu(struct dw_pcie *pci, int index,
			 enum dw_pcie_region_type type)
{
	int region;

	if (pci->iatu_unroll_enabled) {
		switch (type) {
		case DW_PCIE_REGION_INBOUND:
			dw_pcie_writel_ib_unroll(pci, index,
						 PCIE_ATU_UNR_REGION_CTRL2, 0);
			dw_pcie_writel_ib_unroll(pci, index,
						 PCIE_ATU_UNR_REGION_CTRL1, 0);
			break;
		case DW_PCIE_REGION_OUTBOUND:
			dw_pcie_writel_ob_unroll(pci, index,
						 PCIE_ATU_UNR_REGION_CTRL2, 0);
			dw_pcie_writel_ob_unroll(pci, index,
						 PCIE_ATU_UNR_REGION_CTRL1, 0);
			break;
		default:
			break;
		}
		return;
	}

	switch (type) {
	case DW_PCIE_REGION_INBOUND:
		region = PCIE_ATU_REGION_INBOUND;
		break;
	case DW_PCIE_REGION_OUTBOUND:
		region = PCIE_ATU_REGION_OUTBOUND;
		break;
	default:
		return;
	}

	dw_pcie_writel_dbi(pci, PCIE_ATU_VIEWPORT, region | index);
	dw_pcie_writel_dbi(pci, PCIE_ATU_CR2, ~PCIE_ATU_ENABLE);
}

int dw_pcie_wait_for_link(struct dw_pcie *pci)
{
    int retries;

    /* check if the link is up or not */
    for (retries = 0; retries < LINK_WAIT_MAX_RETRIES; retries++) {
        if (dw_pcie_link_up(pci)) {
            dev_info(pci->dev, "link up\n");
            return 0;
        }
        usleep_range(LINK_WAIT_USLEEP_MIN, LINK_WAIT_USLEEP_MAX);
    }

    dev_err(pci->dev, "phy link never came up\n");

    return -ETIMEDOUT;
}

int dw_pcie_link_up(struct dw_pcie *pci)
{
	u32 val;

	if (pci->ops->link_up)
		return pci->ops->link_up(pci);

	val = readl(pci->dbi_base + PCIE_PHY_DEBUG_R1);
	return ((val & PCIE_PHY_DEBUG_R1_LINK_UP) &&
		(!(val & PCIE_PHY_DEBUG_R1_LINK_IN_TRAINING)));
}

void dw_pcie_setup(struct dw_pcie *pci)
{
    int ret;
    u32 val;
    u32 lanes;
    struct device *dev = pci->dev;
    struct device_node *np = dev->of_node;

    ret = of_property_read_u32(np, "num-lanes", &lanes);
    if (ret)
        lanes = 0;

    /* set the number of lanes */
    val = dw_pcie_readl_dbi(pci, PCIE_PORT_LINK_CONTROL);
    val &= ~PORT_LINK_MODE_MASK;
    switch (lanes) {
        case 1:
            val |= PORT_LINK_MODE_1_LANES;
            break;
        case 2:
            val |= PORT_LINK_MODE_2_LANES;
            break;
        case 4:
            val |= PORT_LINK_MODE_4_LANES;
            break;
        case 8:
            val |= PORT_LINK_MODE_8_LANES;
            break;
        default:
            dev_err(pci->dev, "num-lanes %u: invalid value\n", lanes);
            return;
    }
    dw_pcie_writel_dbi(pci, PCIE_PORT_LINK_CONTROL, val);

    /* set link width speed control register */
    val = dw_pcie_readl_dbi(pci, PCIE_LINK_WIDTH_SPEED_CONTROL);
    val &= ~PORT_LOGIC_LINK_WIDTH_MASK;
    switch (lanes) {
        case 1:
            val |= PORT_LOGIC_LINK_WIDTH_1_LANES;
            break;
        case 2:
            val |= PORT_LOGIC_LINK_WIDTH_2_LANES;
            break;
        case 4:
            val |= PORT_LOGIC_LINK_WIDTH_4_LANES;
            break;
        case 8:
            val |= PORT_LOGIC_LINK_WIDTH_8_LANES;
            break;
    }
    dw_pcie_writel_dbi(pci, PCIE_LINK_WIDTH_SPEED_CONTROL, val);

    /*turn off 100ms timeout in L1*/
    val = dw_pcie_readl_dbi(pci, GEN3_RELATED_OFF);
    val &= (~GEN3_ZRXDC_NONCOMPL);
    dw_pcie_writel_dbi(pci, GEN3_RELATED_OFF, val);

}

void dw_pcie_set_LTSSM_en(struct dw_pcie *pci, int enable)
{
	  clear_phy_bits(pci, GENERAL_CORE_CTRL_OFF, (0x01<<2));
	  if(enable)
	  	set_phy_bits(pci,GENERAL_CORE_CTRL_OFF, (0x01<<2));
}

void dw_pcie_set_Interrupt0_en(struct dw_pcie *pci, u32 mask, int enable)
{
	 clear_phy_bits(pci,  INTERRUPT0_ENABLE_OFF,mask);
	 if(enable)
	      set_phy_bits(pci, INTERRUPT0_ENABLE_OFF, mask);
}

void dw_pcie_set_Interrupt1_en(struct dw_pcie *pci, u32 mask, int enable)
{
	 clear_phy_bits(pci,  INTERRUPT1_ENABLE_OFF,mask);
	 if(enable)
	      set_phy_bits(pci, INTERRUPT1_ENABLE_OFF, mask);
}

void dw_pcie_clear_Interrupt0(struct dw_pcie *pci)
{
        dw_pcie_writel_phy(pci, INTERRUPT0_STATUS_OFF, dw_pcie_readl_phy(pci,INTERRUPT0_STATUS_OFF));
}

void dw_pcie_clear_Interrupt1(struct dw_pcie *pci)
{
        dw_pcie_writel_phy(pci, INTERRUPT1_STATUS_OFF, dw_pcie_readl_phy(pci,INTERRUPT1_STATUS_OFF));
}

void dw_pcie_set_Mode(struct dw_pcie *pci,  dw_pcie_dev_mode mode)
{
     u32 val;
     val = dw_pcie_readl_phy(pci, GENERAL_CORE_CTRL_OFF);
     val &= (~(0xf << 4));
     val |= (mode << 4);
     dw_pcie_writel_phy(pci, GENERAL_CORE_CTRL_OFF, val);
	 if(mode == DW_PCIE_AS_EP)
	 {
	          /*set FW control PERST and do reset*/
		set_phy_bits(pci, GENERAL_SIGNAL_CTRL_OFF, 0x1<<1);
		clear_phy_bits(pci, GENERAL_SIGNAL_CTRL_OFF, 0x1<<2);
		udelay(1);
		set_phy_bits(pci, GENERAL_SIGNAL_CTRL_OFF, 0x1<<2);
		msleep(100);
	 }
}

int dw_pcie_find_capability(struct dw_pcie *pci, u8 cap_id)
{
    u8 pos = PCI_CAPABILITY_LIST;
    u8 find_cap_ttl = PCI_FIND_CAP_TTL;
    u8 read_id;

    pos = (u8)(readl(pci->dbi_base + pos) & PCI_CAP_ID_MASK);
    while(find_cap_ttl--)
    {
        if(pos < PCI_CONFIG_SPACE_MAX)
            break;

        read_id = (u8)(readl(pci->dbi_base + pos) & PCI_CAP_ID_MASK);
        if(read_id == PCI_WO_NEXT_CAP_POINTER)
            break;

        if(cap_id == read_id)
            return pos;

        pos = (u8)((readl(pci->dbi_base + pos) & PCI_NEXT_CAP_POINTER_MASK) >> PCI_NEXT_CAP_POINTER_SHIFT);
    }

    return 0;
}

int dw_pcie_get_phy_status(struct dw_pcie *pci, u32 lanes_num, enum dw_pcie_phy_status* status)
{
    u32 val = 0;

    val = dw_pcie_readl_phy(pci, DEBUG_MUX_SEL_OFF);
    val |= DEBUG_PHY_STATUS;
    dw_pcie_writel_phy(pci, DEBUG_MUX_SEL_OFF, val);

    val = dw_pcie_readl_phy(pci, DEBUG_OUT_OFF);

    switch (lanes_num) {
        case 1:
            val &= DEBUG_PHY_STATUS_1_LANES;
            break;
        case 2:
            val &= DEBUG_PHY_STATUS_2_LANES;
            break;
        case 4:
            val &= DEBUG_PHY_STATUS_4_LANES;
            break;
        default:
            return  -EINVAL;
    }

    /*for power on, it need to wait until signal goes from high->low*/
    if(val)
        *status = DW_PCIE_PHY_IDLE;
    else
        *status = DW_PCIE_PHY_READY;

    return 0;
}
