/**
 * Copyright 2018 Sony Imaging Products & Solutions Inc
 * Copyright 2018 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2  of
 * the License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/clk.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/pm_runtime.h>
#include <linux/reset.h>
#include <linux/clk.h>
struct dwc3_of_cxd {
	struct device		*dev;
	struct clk		**clks;
	int			num_clocks;
	struct reset_control	**resets;
	int			num_resets;
};

static int dwc3_of_cxd_clk_init(struct dwc3_of_cxd *cxd, int count)
{
	struct device		*dev = cxd->dev;
	struct device_node	*np = dev->of_node;
	int			i;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
	if(count < 0){
		dev_err(dev,"Invalid argument clk count\n");
		return -EINVAL;
	}

	cxd->num_clocks = count;

	if (!count)
		return 0;

	cxd->clks = devm_kcalloc(dev, cxd->num_clocks,
			sizeof(struct clk *), GFP_KERNEL);
	if (!cxd->clks)
		return -ENOMEM;

	for (i = 0; i < cxd->num_clocks; i++) {
		struct clk	*clk;
		int		ret;

		clk = of_clk_get(np, i);
		if (IS_ERR(clk)) {
			while (--i >= 0) {
				clk_disable_unprepare(cxd->clks[i]);
				clk_put(cxd->clks[i]);
			}
			return PTR_ERR(clk);
		}

		ret = clk_prepare_enable(clk);
		if (ret < 0) {
			while (--i >= 0) {
				clk_disable_unprepare(cxd->clks[i]);
				clk_put(cxd->clks[i]);
			}
			clk_put(clk);

			return ret;
		}
		cxd->clks[i] = clk;
	}
	return 0;
}
#ifdef CONFIG_ARCH_CXD900XX_FPGA
static int dwc3_of_cxd_get_resets(struct dwc3_of_cxd *cxd, struct device_node *np)
{
	struct reset_control *rst;
	struct device   *dev = cxd->dev;
	unsigned int i, count;
	int err;
	//dev_err(dev,"dwc3_of_cxd_get_resets \n\n\n");
	count = of_count_phandle_with_args(np, "resets", "#reset-cells");
	if (count == -ENOENT){///* No such file or directory */
		count = 0;
		dev_err(dev,"can't find resets in devicetree\n");
		return count;
	}

	cxd->resets = kcalloc(count, sizeof(rst), GFP_KERNEL);
	if (!cxd->resets){
		dev_err(dev,"allocate resets memory fail\n");
		return -ENOMEM;
	}
	for (i = 0; i < count; i++) {
		cxd->resets[i] = of_reset_control_get_by_index(np, i);
		if (IS_ERR(cxd->resets[i])) {
			err = PTR_ERR(cxd->resets[i]);
			dev_err(dev,"get resets index %d fail\n", i);
			//goto error;
		}
	}
	cxd->num_resets = count;
	return 0;
error:
	while (i--)
		reset_control_put(cxd->resets[i]);
	kfree(cxd->resets);
	return err;
}

static int dwc3_cxd_reset_assert(struct dwc3_of_cxd *cxd)
{
	unsigned int i;

	for (i = 0; i < cxd->num_resets; i++) {
		reset_control_assert(cxd->resets[i]);
	}
	return 0;
}

static int dwc3_cxd_reset_deassert(struct dwc3_of_cxd *cxd)
{
	unsigned int i;

	for (i = 0; i < cxd->num_resets; i++) {
		reset_control_deassert(cxd->resets[i]);
	}
	return 0;
}
#else
static int dwc3_of_cxd_get_resets(struct dwc3_of_cxd *cxd, struct device_node *np)
{
	struct reset_control *rst;
	struct device   *dev = cxd->dev;
	unsigned int i, count;
	int err;

	count = of_count_phandle_with_args(np, "resets", "#reset-cells");
        if (count == -ENOENT){///* No such file or directory */
		dev_err(dev,"can't find resets in devicetree\n");
		count = 0;
        	return count;
	}

	cxd->resets = kcalloc(count, sizeof(rst), GFP_KERNEL);
	if (!cxd->resets){
		dev_err(dev,"allocate resets memory fail\n");
                return -ENOMEM;
	}

        for (i = 0; i < count; i++) {
                cxd->resets[i] = of_reset_control_get_by_index(np, i);
                if (IS_ERR(cxd->resets[i])) {
                        err = PTR_ERR(cxd->resets[i]);
			dev_err(dev,"get resets index %d fail\n", i);
                        goto error;
                }
	}

        cxd->num_resets = count;
        return 0;
error:
        while (i--)
                reset_control_put(cxd->resets[i]);

        kfree(cxd->resets);
        return err;
}

static int dwc3_cxd_reset_assert(struct dwc3_of_cxd *cxd)
{
        unsigned int i;
	struct device   *dev = cxd->dev;
        int err;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
        for (i = 0; i < cxd->num_resets; i++) {
                err = reset_control_assert(cxd->resets[i]);
                if (err){
			dev_err(dev,"assert resets index %d fail\n", i);
                        return err;
		}
        }

        return 0;
}

static int dwc3_cxd_reset_deassert(struct dwc3_of_cxd *cxd)
{
        unsigned int i;
	struct device	*dev = cxd->dev;
        int err;
	//dev_err(dev,"%s.... \n\n\n", __FUNCTION__);
        for (i = 0; i < cxd->num_resets; i++) {
                err = reset_control_deassert(cxd->resets[i]);
                if (err){
			dev_err(dev,"deassert resets index %d fail\n", i);
			return err;
		}
        }

        return 0;
}
#endif
static int dwc3_of_cxd_probe(struct platform_device *pdev)
{
	struct dwc3_of_cxd	*cxd;
	struct device		*dev = &pdev->dev;
	struct device_node	*np = dev->of_node;

	int			ret;
	int			i;

	cxd = devm_kzalloc(dev, sizeof(*cxd), GFP_KERNEL);
	if (!cxd)
		return -ENOMEM;

	platform_set_drvdata(pdev, cxd);
	cxd->dev = dev;

        ret = dwc3_of_cxd_clk_init(cxd, of_count_phandle_with_args(np,
                                                "clocks", "#clock-cells"));
        if (ret){
		dev_err(dev, "failed to inite CXD USB3 Clock....err=%d\n", ret);
                goto err_clk;
	}
	// get rest information
	ret = dwc3_of_cxd_get_resets(cxd, np);
        if (ret)
                goto err_clk;
	//ret = dwc3_cxd_reset_assert(cxd);
	ret = dwc3_cxd_reset_deassert(cxd);
        if (ret)
                goto err_clk;

	ret = of_platform_populate(np, NULL, NULL, dev);
	if (ret) {
		for (i = 0; i < cxd->num_clocks; i++) {
			clk_disable_unprepare(cxd->clks[i]);
			clk_put(cxd->clks[i]);
		}

		goto err_resetc_assert;
	}

	pm_runtime_set_active(dev);
	pm_runtime_enable(dev);
	pm_runtime_get_sync(dev);

	return 0;

err_resetc_assert:
        dwc3_cxd_reset_assert(cxd);

err_clk:
	return ret;
}

static int dwc3_of_cxd_remove(struct platform_device *pdev)
{
	struct dwc3_of_cxd	*cxd = platform_get_drvdata(pdev);
	struct device		*dev = &pdev->dev;
	int			i;

	of_platform_depopulate(dev);

	for (i = 0; i < cxd->num_clocks; i++) {
		clk_disable_unprepare(cxd->clks[i]);
		clk_put(cxd->clks[i]);
	}

	dwc3_cxd_reset_assert(cxd);
	while (cxd->num_resets--)
		reset_control_put(cxd->resets[cxd->num_resets]);

	pm_runtime_put_sync(dev);
	pm_runtime_disable(dev);

	return 0;
}

#ifdef CONFIG_PM
static int dwc3_of_simple_runtime_suspend(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);
	int			i;

	for (i = 0; i < cxd->num_clocks; i++)
		clk_disable(cxd->clks[i]);

	return 0;
}

static int dwc3_of_simple_runtime_resume(struct device *dev)
{
	struct dwc3_of_cxd	*cxd = dev_get_drvdata(dev);
	int			ret;
	int			i;

	for (i = 0; i < cxd->num_clocks; i++) {
		ret = clk_enable(cxd->clks[i]);
		if (ret < 0) {
			while (--i >= 0)
				clk_disable(cxd->clks[i]);
			return ret;
		}
	}

	return 0;
}
#endif

static const struct dev_pm_ops dwc3_of_simple_dev_pm_ops = {
	SET_RUNTIME_PM_OPS(dwc3_of_simple_runtime_suspend,
			dwc3_of_simple_runtime_resume, NULL)
};

static const struct of_device_id of_dwc3_simple_match[] = {
	{ .compatible = "cxdchip,dwc3" },
	{ /* Sentinel */ }
};
MODULE_DEVICE_TABLE(of, of_dwc3_simple_match);

static struct platform_driver dwc3_of_simple_driver = {
	.probe		= dwc3_of_cxd_probe,
	.remove		= dwc3_of_cxd_remove,
	.driver		= {
		.name	= "dwc3-of-cxd",
		.of_match_table = of_dwc3_simple_match,
		.pm	= &dwc3_of_simple_dev_pm_ops,
	},
};

module_platform_driver(dwc3_of_simple_driver);
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("DesignWare USB3 OF CXD Glue Layer");
