/**
 *	@file	usb_extcmd_cmn.c
 *	@brief	USB EXTCMD(USB Extension Command) - common
 *	
 *		Copyright 2008,2011 Sony Corporation
 */

#include <asm/errno.h>
#include <asm/memory.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>

#ifndef _LINUX_SLAB_H
#include <linux/slab.h>
#endif

#define FNAME "usbg_ext_cmn.c"
#include "usb_extcmd_pvt.h"


static struct extcmd *extcmd_this = NULL;


#ifdef EXTCMD_DBG_MEMORY
	struct _EXTCMD_DEBUG_MEMORY {
		size_t mem_counter;
		size_t max_memory_alloc;
	};
	struct _EXTCMD_DEBUG_MEMORY extcmd_mem;
#endif

/**
 *	@brief	memory free
 */
void *extcmd_mem_alloc(size_t size)
{
#ifdef EXTCMD_DBG_MEMORY
	extcmd_mem.mem_counter += size;
	if(extcmd_mem.mem_counter > extcmd_mem.max_memory_alloc){
		extcmd_mem.max_memory_alloc = extcmd_mem.mem_counter;
	}
#endif
	return kmalloc(size, GFP_ATOMIC);
}

/**
 *	@brief	memory free
 */
void extcmd_mem_free(void *p, size_t size)
{
#ifdef EXTCMD_DBG_MEMORY
	extcmd_mem.mem_counter -= size;
#endif

	return kfree(p);
}

/**
 *	@brief	memory check
 */
void extcmd_mem_check(void)
{
#ifdef EXTCMD_DBG_MEMORY
	EXTCMD_INF("Max memory alloc size is %zu bytes", extcmd_mem.max_memory_alloc);

	if(extcmd_mem.mem_counter){
		EXTCMD_ERR("Memory Leak [%zu] bytes...", extcmd_mem.mem_counter);
	}
	else{
		EXTCMD_INF("Memory Leak none.");
	}
#endif
}


/**
 *	@brief	Get driver instance
 */
struct extcmd *get_extcmd_this(void)
{
//	EXTCMD_FNC();
	return extcmd_this;
}

/**
 *	@brief	Create driver instance
 */
int extcmd_create_this(void)
{
	struct extcmd *pthis;

	EXTCMD_FNC();

	if(extcmd_this){
		// instance is already created
		// nothing to do
		return 0;
	}

	pthis = extcmd_mem_alloc(sizeof(struct extcmd));
	if(!pthis){
		EXTCMD_ERR();
		return -ENOMEM;
	}

	extcmd_this = pthis;
	memset(extcmd_this, 0, sizeof(struct extcmd));

	return 0;
}

/**
 *	@brief	Delete driver instance
 */
void extcmd_delete_this(void)
{
	struct extcmd *pthis;

	EXTCMD_FNC();

	pthis = get_extcmd_this();
	if(pthis){
		extcmd_mem_free(pthis, sizeof(struct extcmd));
	}
	extcmd_this = NULL;

}

/**
 *	@brief	test bit req flag
 */
static unsigned char test_bit_req_flag(unsigned char test_nr, unsigned long *flag)
{
	unsigned char ret;
	if( test_bit(test_nr, flag) ){
		ret = EXTCMD_ALREADY_WAITING;
	}
	else{
		ret = EXTCMD_WAIT_READY;
	}
	return ret;
}

/**
 *	@brief	test & set bit req flag by user
 */
unsigned char test_and_set_bit_req_flag_user(unsigned char test_nr, unsigned char set_nr, struct _extcmd_cmd_list *entry)
{
	unsigned char ret = 0;

	EXTCMD_LOCK_ON();

	ret = test_bit_req_flag(test_nr, &entry->req_order_flag);
	if( ret == EXTCMD_WAIT_READY ){
		set_bit(set_nr, &entry->req_order_flag);
		EXTCMD_INF("chk req flg ready wait[%x][%x]",
					set_nr, (unsigned short)entry->req_order_flag);
	}
	else{
		EXTCMD_INF("chk req flg alrady wait[%x][%x]",
					set_nr, (unsigned short)entry->req_order_flag);
	}

	EXTCMD_LOCK_OFF();
	return ret;
}

/**
 *	@brief	test & set bit req flag by kernel
 */
unsigned char test_and_set_bit_req_flag_kernel(unsigned char test_nr, unsigned char set_nr, struct _extcmd_cmd_list *entry)
{
	unsigned char ret = 0;

	EXTCMD_LOCK_ON();

	ret = test_bit_req_flag(test_nr, &entry->req_order_flag);
	if( ret == EXTCMD_WAIT_READY ){
		EXTCMD_INF("chk req flg ready wait[%x][%x]",
					set_nr, (unsigned short)entry->req_order_flag);
	}
	else{
		EXTCMD_INF("chk req flg alrady wait[%x][%x]",
					set_nr, (unsigned short)entry->req_order_flag);
	}
	/// kernel already wait
	set_bit(set_nr, &entry->req_order_flag);

	EXTCMD_LOCK_OFF();
	return ret;
}

/**
 *	@brief	Delete driver instance
 */
void clear_bit_req_flag(unsigned long clr_nr, struct _extcmd_cmd_list *entry)
{
	EXTCMD_LOCK_ON();

	clear_bit(clr_nr, &entry->req_order_flag);
	EXTCMD_INF("chk req flg cleared[%x][%x]",
				(unsigned short)clr_nr, (unsigned short)entry->req_order_flag);

	EXTCMD_LOCK_OFF();
}


/**
 *	@brief	Initialize opcode list
 */
int list_init(void)
{
	struct extcmd *pthis;
	struct list_head *head;

	EXTCMD_FNC();

	pthis = get_extcmd_this();
	if(pthis->list_head){
		EXTCMD_ERR();
		return -ENOMEM;
	}

	head = extcmd_mem_alloc(sizeof(struct list_head));
	if(!head) {
		EXTCMD_ERR();
		return -ENOMEM;
	}

	pthis->list_head = head;
	//initialize list head
	INIT_LIST_HEAD(pthis->list_head);

	return 0;

}

/**
 *	@brief	Finalize opcode list
 */
void list_finalize(void)
{
	struct extcmd *pthis;
	struct list_head *head;
	struct list_head *ptr;
	struct _extcmd_cmd_list *entry;
	
	EXTCMD_FNC();
	EXTCMD_LOCK_ON();

	pthis = get_extcmd_this();
	head  = pthis->list_head;

	if(pthis->list_head){
		for (ptr = head->next; ptr != head; ptr = ptr->next){
			entry = list_entry(ptr, struct _extcmd_cmd_list, list);
			list_del(&entry->list);
		}
	
		extcmd_mem_free(head, sizeof(struct list_head));
		pthis->list_head = NULL;
	}
	EXTCMD_LOCK_OFF();
}

/**
 *	@brief	 Add node (tail)
 */
int list_append_node(struct _extcmd_cmd_list *entry)
{
	struct extcmd *pthis;
	struct list_head *head;
	struct list_head *ptr;
	struct _extcmd_cmd_list *entrid;

	EXTCMD_FNC();
	EXTCMD_INF("append node op[%02x]idx[%04x]", entry->opcode, entry->index);

	EXTCMD_LOCK_ON();

	pthis = get_extcmd_this();
	head  = pthis->list_head;

	for (ptr = head->next; ptr != head; ptr = ptr->next){
		entrid = list_entry(ptr, struct _extcmd_cmd_list, list);
		if( (entrid->opcode == entry->opcode) && (entrid->index == entry->index) ){
			EXTCMD_ERR();
			EXTCMD_LOCK_OFF();
			return -EALREADY;
		}
	}
	list_add_tail(&entry->list, head);

	EXTCMD_LOCK_OFF();
	return 0;
}

/**
 *	@brief	Delete node
 */
int list_delete_node(struct _extcmd_cmd_list *entry)
{
	struct extcmd *pthis;
	struct list_head *head;

	EXTCMD_FNC();
	EXTCMD_INF("delete node op[%02x]idx[%04x]", entry->opcode, entry->index);

	EXTCMD_LOCK_ON();

	pthis = get_extcmd_this();
	head  = pthis->list_head;

	list_del(&entry->list);

	EXTCMD_LOCK_OFF();
	return 0;
}


/**
 *	@brief	Get node
 */
struct _extcmd_cmd_list *list_get_node(unsigned short opcode, unsigned short index)
{
	struct extcmd *pthis;
	struct list_head *head;
	struct list_head *ptr;
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();
	EXTCMD_LOCK_ON();

	pthis = get_extcmd_this();
	head  = pthis->list_head;

	for (ptr = head->next; ptr != head; ptr = ptr->next){
		entry = list_entry(ptr, struct _extcmd_cmd_list, list);
		if( (entry->opcode == opcode) && (entry->index == index) ){
			EXTCMD_LOCK_OFF();
//			EXTCMD_INF("entry[%p] found", entry);
			return entry;
		}
	}

	EXTCMD_INF("get node op[%04x]idx[%04x] not entried", opcode, index);
	EXTCMD_LOCK_OFF();
	return NULL;
}

/**
 *	@brief	Enable inquiry state
 */
void enable_inquiry_state(struct _extcmd_cmd_list *entry, int inqstate)
{
	EXTCMD_FNC();

	entry->inq_state |= inqstate;
}

/**
 *	@brief	Disable inquiry state
 */
void disable_inquiry_state(struct _extcmd_cmd_list *entry, int inqstate)
{
	EXTCMD_FNC();

	entry->inq_state &=  !inqstate;
}

/**
 *	@brief	Get inquiry state
 */
int get_inquiry_state(unsigned char opcode, unsigned short index, int direction)
{
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();

//	EXTCMD_INF("inquiry[%x]", direction);
	entry = list_get_node(opcode, index);
	if(!entry){
		EXTCMD_INF("inquiry EXTCMD_NOT_REGISTERED");
		return EXTCMD_NOT_REGISTERED;
	}
	else if( entry->inq_state & direction ){
		EXTCMD_INF("inquiry EXTCMD_READY");
		return EXTCMD_READY;
	}
	else{
		EXTCMD_INF("inquiry EXTCMD_NOT_YET_READY");
		return EXTCMD_NOT_YET_READY;
	}
}

/**
 *	@brief	Get send data length
 */
int get_send_data_len(unsigned char opcode, unsigned short index)
{
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();

	entry = list_get_node(opcode, index);
	if(!entry){
		EXTCMD_ERR("inquiry EXTCMD_NOT_REGISTERED");
		return 0;
	}
    else {
		return entry->send_data_len;
	}
}

/**
 *	@brief	Get recieve data length
 */
int get_recv_data_len(unsigned char opcode, unsigned short index)
{
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();

	entry = list_get_node(opcode, index);
	if(!entry){
		EXTCMD_ERR("inquiry EXTCMD_NOT_REGISTERED");
		return 0;
	}
    else {
		return entry->recv_data_len;
	}
}

/**
 *	@brief	synchronize wake up interruptible
 */
void synchronize_wake_up_event(wait_queue_head_t *wqh, unsigned long bit, unsigned long *flag)
{
	EXTCMD_FNC();
	EXTCMD_LOCK_ON();

//	EXTCMD_INF("wake_up_interruptible flg[%x]", (unsigned int)(*flag));

	/// set flag
	set_bit(bit, flag);

	EXTCMD_INF("wake_up_interruptible flg[%x]", (unsigned int)(*flag));
	wake_up_interruptible(wqh);

	EXTCMD_LOCK_OFF();
}

/**
 *	@brief	synchronize wake up interruptible
 */
void synchronize_wake_up_cancel_event(wait_queue_head_t *wqh, unsigned long *flag, unsigned int user_request)
{
	EXTCMD_FNC();
	EXTCMD_LOCK_ON();

//	EXTCMD_INF("synchronize_wake_up_cancel_event flg[%x]", (unsigned int)(*flag));

	/// set flag
	if(user_request){
		set_bit(EXTCMD_WFLAG_CANCEL_USER, flag);
	}
	set_bit(EXTCMD_WFLAG_CANCEL_KERN, flag);

	EXTCMD_INF("wake_up_interruptible flg[%x]", (unsigned int)(*flag));
	wake_up_interruptible(wqh);

	EXTCMD_LOCK_OFF();
}

/**
 *	@brief	synchronize wake up interruptible all node
 */
void synchronize_wake_up_event_all_node(unsigned long bit)
{
	struct list_head *head;
	struct extcmd *pthis;
	struct list_head *ptr;
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();
	EXTCMD_LOCK_ON();

	pthis = get_extcmd_this();
	head  = pthis->list_head;
	if(pthis->list_head){
		for (ptr = head->next; ptr != head; ptr = ptr->next){
			entry = list_entry(ptr, struct _extcmd_cmd_list, list);
			if( waitqueue_active(entry->wqh) ){  // waitqueue not empty
				set_bit(bit, &entry->sync_flag);
				EXTCMD_INF("wake_up_interruptible flg[%x]", (unsigned int)entry->sync_flag);
				wake_up_interruptible(entry->wqh);
			}
		}
	}

	EXTCMD_LOCK_OFF();
}

/**
 *	@brief	clear cancel disable
 */
void cancel_disable(unsigned char opcode, unsigned short index)
{
	struct _extcmd_cmd_list *entry;

	EXTCMD_FNC();

	entry = list_get_node(opcode, index);

	EXTCMD_LOCK_ON();

	if(!entry){
		EXTCMD_ERR();
		EXTCMD_LOCK_OFF();
		return;
	}

	clear_bit(EXTCMD_WFLAG_CANCEL_KERN, &(entry->sync_flag));

	EXTCMD_LOCK_OFF();
}

/**
 *	@brief	synchronize wait condition
 */
int wait_event_condition(int caller, wait_queue_head_t *wqh, int *reason, unsigned long bit, unsigned long *flag)
{
	int ret = TRUE;
	EXTCMD_INF("wait_event_condition bit[%x]flag=[%x]", (unsigned short)bit, (unsigned short)(*flag));

	if( test_bit(EXTCMD_WFLAG_RESET, flag) ){			// Reset
		clear_bit(bit, flag);
		*reason = -EPIPE;
		EXTCMD_INF("reset flag=[%x]", (unsigned short)(*flag));
	}
	else if( ( caller == EXTCMD_SPACE_USER) &&
			 (test_bit(EXTCMD_WFLAG_CANCEL_USER, flag)) ){		// Cancel userspace
		clear_bit(bit, flag);
		clear_bit(EXTCMD_WFLAG_CANCEL_USER, flag);
		*reason = -ECANCELED;
		EXTCMD_INF("cancel U flag=[%x]", (unsigned short)(*flag));
	}
	else if( ( caller == EXTCMD_SPACE_KERNEL) &&
			 (test_bit(EXTCMD_WFLAG_CANCEL_KERN, flag)) ){		// Cancel kernelspace
		clear_bit(bit, flag);
		clear_bit(EXTCMD_WFLAG_CANCEL_KERN, flag);
		*reason = -ECANCELED;
		EXTCMD_INF("cancel K flag=[%x]", (unsigned short)(*flag));
	}
	else if( test_bit(bit, flag) ){						// Read,Write
		clear_bit(bit, flag);
		EXTCMD_INF("r/w flag=[%x]", (unsigned short)(*flag));
	}
	else{												// no event
		EXTCMD_INF("no event flag=[%x]", (unsigned short)(*flag));
		ret = FALSE;
	}
	return ret;
}

/**
 *	@brief	synchronize wait event interruptible timeout
 */
int synchronize_wait_event(int caller, wait_queue_head_t *wqh, unsigned long bit, unsigned long *flag, unsigned int timeout)
{
	int err;
	int reason = 0;
	unsigned int integer;
	unsigned int downpoint;

	EXTCMD_FNC();
	
	if(!timeout){
		EXTCMD_INF("nto flag=[%d]addr[%p]", (unsigned int)(*flag), flag);
		err = wait_event_interruptible( *wqh,
					wait_event_condition(caller, wqh, &reason, bit, flag) );
		if(err){									// other err
			EXTCMD_ERR("OtherErr [%d]", err);
			err = (-EPERM);
		}
		else{
			if(reason == (-EPIPE)){					// reset
				if( !waitqueue_active(wqh) ){		// waitqueue empty
					clear_bit(EXTCMD_WFLAG_RESET, flag);
					EXTCMD_INF("reset event enpty flag=[%x]", (unsigned short)(*flag));
				}
				err = reason;
			}
			else if(reason == (-ECANCELED)){		// cancel
				err = reason;
			}
			else{									// success
				EXTCMD_INF("Status ok [%d]", err);
				err = 0;
			}
		}
	}
	else{
		EXTCMD_INF("to flag=[%d]addr[%p]", (unsigned int)(*flag), flag);
		integer		= HZ * (timeout / EXTCMD_TIMEOUT_ORDER_UNIT);
		downpoint	= HZ * (timeout % EXTCMD_TIMEOUT_ORDER_UNIT);

		err = wait_event_interruptible_timeout(*wqh,
					wait_event_condition(caller, wqh, &reason, bit, flag),
					integer + downpoint );
		if(!err){									// Timeout
			EXTCMD_INF("Timeout [%d]", err);
			err = -ETIME;
		}
		else if(err < 0){							// other err
			EXTCMD_ERR("OtherErr [%d]", err);
			err = -EPERM;
		}
		else{
			if(reason == (-EPIPE)){					// reset
				if( !waitqueue_active(wqh) ){		// waitqueue empty
					clear_bit(EXTCMD_WFLAG_RESET, flag);
					EXTCMD_INF("reset event enpty flag=[%x]", (unsigned short)(*flag));
				}
				err = reason;
			}
			else if(reason == (-ECANCELED)){		// cancel
				err = reason;
			}
			else{									// success
				EXTCMD_INF("Status ok [%d]", err);
				err = 0;
			}
		}
	}

	return err;
}



