/**
 *	@file	usb_extcmd_pvt.h
 *	@brief	USB EXTCMD(USB Extension Command) - private header
 *	
 *		Copyright 2008,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __USBG_EXT_PVT_H__
#define __USBG_EXT_PVT_H__

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/version.h>

#include <asm/page.h>

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/mutex.h>
#else
#include <asm/semaphore.h>
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usb_config.h>
#include <linux/usb/specific_gadget/usb_extcmd.h>

#include "usb_extcmd_conf.h"

#ifndef FNAME
#define FNAME "usb_extcmd"
#endif

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cdev.h>
#endif

/*
 *	@brief	debug print macro
 */
#define FILE_LINE_FUNC		FNAME, __LINE__, __FUNCTION__
#define FILE_LINE			FNAME, __LINE__


#ifdef  EXTCMD_DEBUG_ERR
#define EXTCMD_ERR(fmt, args...) { printk("\033[31m%s(%d): [E]" fmt "\033[m\n", FILE_LINE, ##args); }
//#define EXTCMD_ERR(fmt, args...) { printk("%s(%d): [E]" fmt "\n", FILE_LINE, ##args); }
#else
#define EXTCMD_ERR(fmt, args...) { }
#endif

#ifdef  EXTCMD_DEBUG_INF
#define EXTCMD_INF(fmt, args...) { printk("\033[36m%s(%d): [I]" fmt "\033[m\n", FILE_LINE, ##args);   }
//#define EXTCMD_INF(fmt, args...) { printk("%s(%d): [I]" fmt "\n", FILE_LINE, ##args);   }
#else
#define EXTCMD_INF(fmt, args...) { }
#endif

#ifdef  EXTCMD_DEBUG_API
#define EXTCMD_API(fmt, args...) { printk("\033[36m%s(%d): [A]%s()" fmt "\033[m\n", FILE_LINE_FUNC, ##args); }
//#define EXTCMD_API()             { printk("%s(%d): [A]%s()" fmt "\n", FILE_LINE_FUNC, ##args); }
#else
#define EXTCMD_API(fmt, args...) { }
#endif

#ifdef  EXTCMD_DEBUG_FNC
#define EXTCMD_FNC()             { printk("\033[36m%s(%d): [F]%s()\033[m\n", FILE_LINE_FUNC); }
//#define EXTCMD_FNC()             { printk("%s(%d): [F]%s()\n", FILE_LINE_FUNC); }
#else
#define EXTCMD_FNC()             { }
#endif

#ifdef CONFIG_OSAL_UDIF
#define EXTCMD_K_PRAM_LOCK_INI(etry)  { udif_mutex_init(&(etry->k_param_lock));		}
#define EXTCMD_K_PRAM_LOCK_ON(etry)   { udif_mutex_lock(&(etry->k_param_lock));		}
#define EXTCMD_K_PRAM_LOCK_OFF(etry)  { udif_mutex_unlock(&(etry->k_param_lock));	}

#define EXTCMD_K_WORK_LOCK_INI()      { udif_mutex_init(&(get_extcmd_this()->k_work_lock));		}
#define EXTCMD_K_WORK_LOCK_ON()       { udif_mutex_lock(&(get_extcmd_this()->k_work_lock));		}
#define EXTCMD_K_WORK_LOCK_OFF()      { udif_mutex_unlock(&(get_extcmd_this()->k_work_lock));	}
#else
#define EXTCMD_K_PRAM_LOCK_INI(etry)  { seama_init(&(etry->k_param_lock), 1);	}
#define EXTCMD_K_PRAM_LOCK_ON(etry)   { down(&(etry->k_param_lock));			}
#define EXTCMD_K_PRAM_LOCK_OFF(etry)  { up(&(etry->k_param_lock));				}

#define EXTCMD_K_WORK_LOCK_INI()      { sema_init(&(get_extcmd_this()->k_work_lock), 1);	}
#define EXTCMD_K_WORK_LOCK_ON()       { down(&(get_extcmd_this()->k_work_lock));			}
#define EXTCMD_K_WORK_LOCK_OFF()      { up(&(get_extcmd_this()->k_work_lock));				}
#endif
/*
 *	@brief	lock macro
 */
 #ifdef   EXTCMD_SEMAPHORE
 #ifdef   CONFIG_OSAL_UDIF
  #define EXTCMD_LOCK_INI()  { udif_mutex_init(&(get_extcmd_this()->lock));		}
  #define EXTCMD_LOCK_ON()   { udif_mutex_lock(&(get_extcmd_this()->lock));		}
  #define EXTCMD_LOCK_OFF()  { udif_mutex_unlock(&(get_extcmd_this()->lock));	}
 #else
  #define EXTCMD_LOCK_INI()  { sema_init(&(get_extcmd_this()->lock), 1);	}
  #define EXTCMD_LOCK_ON()   { down(&(get_extcmd_this()->lock));			}
  #define EXTCMD_LOCK_OFF()  { up(&(get_extcmd_this()->lock));				}
 #endif
#else
 #ifdef   EXTCMD_SPINLOCK
  #include <linux/spinlock.h>
  #define EXTCMD_LOCK_INI()  { spin_lock_init(&(get_extcmd_this()->lock));										}
  #define EXTCMD_LOCK_ON()   { spin_lock_irqsave(&(get_extcmd_this()->lock), get_extcmd_this()->lockflag);		}
  #define EXTCMD_LOCK_OFF()  { spin_unlock_irqrestore(&(get_extcmd_this()->lock), get_extcmd_this()->lockflag);	}
 #else
  #define EXTCMD_LOCK_INI()  { }
  #define EXTCMD_LOCK_ON()   { }
  #define EXTCMD_LOCK_OFF()  { }
 #endif
#endif


/*
 *	@brief	set sense macro
 */
#define EXTCMD_SET_SENSE(se, ssk, ssc)			  {		\
		(se)->senseKey = ssk;							\
		(se)->additional.sensCode	= ssc;		}
#define EXTCMD_SET_SENSE_BYTE(se, ssk, ssc, sscq) {		\
		(se)->senseKey = ssk;							\
		(se)->additional.byte.asc	= ssc;				\
		(se)->additional.byte.ascq	= sscq;		}

#ifdef NEW_SENSE_KEY
#define EXTCMD_SET_SENSE_NO_SENSE(se)				{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_NO_SENSE, EXTCMD_SC_NO_ADD_SENSE)		}
#define EXTCMD_SET_SENSE_VND_BECOMING_READY(se)		{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_VENDOR, EXTCMD_SC_VND_BECOMING_READY)	}
#define EXTCMD_SET_SENSE_VND_SYSTEM_ERR(se)			{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_VENDOR, EXTCMD_SC_VND_SYSTEM_ERROR)		}
#define EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(se)	{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_VENDOR, EXTCMD_SC_VND_ABORT_CMD_RETRY)	}
#define EXTCMD_SET_SENSE_VND_ABORT_CMD_FATAL(se)	{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_VENDOR, EXTCMD_SC_VND_ABORT_CMD_FATAL)	}
#define EXTCMD_SET_SENSE_INVALID_CMD_OP_CODE(se)	{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_ILLEGAL_REQ, EXTCMD_SC_INVALID_OP_CODE)	}
#else
#define EXTCMD_SET_SENSE_NO_SENSE(se)				{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_NO_SENSE, EXTCMD_SC_NO_ADD_SENSE)		}
#define EXTCMD_SET_SENSE_VND_BECOMING_READY(se)		{	\
		EXTCMD_SET_SENSE(se, 0x02, 0x0401)										}
#define EXTCMD_SET_SENSE_VND_SYSTEM_ERR(se)			{	\
		EXTCMD_SET_SENSE(se, 0x04, 0x0000)										}
#define EXTCMD_SET_SENSE_VND_ABORT_CMD_RETRY(se)	{	\
		EXTCMD_SET_SENSE(se, 0x0B, 0x0000)										}
#define EXTCMD_SET_SENSE_VND_ABORT_CMD_FATAL(se)	{	\
		EXTCMD_SET_SENSE(se, 0x0B, 0x0000)										}
#define EXTCMD_SET_SENSE_INVALID_CMD_OP_CODE(se)	{	\
		EXTCMD_SET_SENSE(se, EXTCMD_SK_ILLEGAL_REQ, EXTCMD_SC_INVALID_OP_CODE)	}
#endif

/*
 *	@brief	true false define
 */
#ifndef TRUE
#define TRUE (1)
#endif

#ifndef FALSE
#define FALSE (0)
#endif

/*
 *	@brief	wait event timeout order
 *	timeout: (1 / EXTCMD_TIMEOUT_ORDER_UNIT) [sec]
 */
#define EXTCMD_TIMEOUT_ORDER_UNIT	(1000)

/*
 *	@brief	caller space define. [kernel or user]
 */
#define EXTCMD_SPACE_KERNEL			(0)
#define EXTCMD_SPACE_USER			(1)

/*
 *	@brief	inquiry state
 */
#define EXTCMD_NOT_REGISTERED		(0)
#define EXTCMD_NOT_YET_READY		(1)
#define EXTCMD_READY				(2)

/*
 *	@brief	inquiry direction define [internal name]
 */
#define EXTCMD_FROM_USER_DIRECTION		EXTCMD_INQUIRY_DIR_K_READ
#define EXTCMD_FROM_KERNEL_DIRECTION	EXTCMD_INQUIRY_DIR_K_WRITE


/*
 *	@brief	wait event flag types
 */
enum{
	EXTCMD_WFLAG_USER_WRITE = 0,
	EXTCMD_WFLAG_USER_READ,
	EXTCMD_WFLAG_USER_WRITE_COPY,
	EXTCMD_WFLAG_USER_READ_COPY,
	EXTCMD_WFLAG_RESET,
	EXTCMD_WFLAG_CANCEL_USER,
	EXTCMD_WFLAG_CANCEL_KERN
};

/**
 *	@brief	request flag bit types
 */
enum{
	EXTCMD_REQ_BIT_KERN_READ = 0,
	EXTCMD_REQ_BIT_KERN_WRITE,
	EXTCMD_REQ_BIT_USER_READ,
	EXTCMD_REQ_BIT_USER_WRITE
};

/**
 *	@brief	wait flag status
 */
#define EXTCMD_ALREADY_WAITING			(1)
#define EXTCMD_WAIT_READY				(2)

/**
 *	@brief	register command list
 */
struct _extcmd_cmd_list {
	struct list_head		list;			///< list struct head
	unsigned char			opcode;			///< operation code
	unsigned short			index;			///< application index
	uint32_t				send_data_len;  ///< send data length
	uint32_t				recv_data_len;  ///< recieve data length
	int						inq_state;		///< inquiry flag
	unsigned long			sync_flag;		///< sync flag
#ifdef CONFIG_OSAL_UDIF
	UDIF_MUTEX				k_param_lock;	///< kernel param lock
#else
	struct semaphore		k_param_lock;	///< kernel param lock
#endif
	wait_queue_head_t		*wqh;			///< wait queue head
	usb_extcmd_k_param_rw *	k_write_param;	///< kernel write param
	usb_extcmd_k_param_rw *	k_read_param;	///< kernel read param
	extcmd_size_t			u_write_size;	///< user write size
	extcmd_size_t			u_read_size;	///< user read size
	unsigned long			req_order_flag;	///< request order flag
};


/*
 *	@brief	ExtCmd  driver information (extension)
 */
struct extcmd {
	struct list_head	*list_head;			///< register list
#ifdef CONFIG_OSAL_UDIF
	UDIF_MUTEX			k_work_lock;		///< kernel lock struct
#else
	struct semaphore	k_work_lock;		///< kernel lock struct
#endif

#ifdef  EXTCMD_SEMAPHORE
#ifdef CONFIG_OSAL_UDIF
	UDIF_MUTEX       lock;					///< lock struct
#else
	struct semaphore lock;					///< lock struct
#endif
#else
 #ifdef  EXTCMD_SPINLOCK
	spinlock_t		lock;					///< lock struct
	unsigned long	lockflag;				///< lock flag
 #endif
#endif
};


/*
 *	@brief	extcmd driver openning infomation.(set file->private_data)
 */
struct extcmd_info {
	int				space;				///< user or kernel
	unsigned char	opcode;				///< operation code
	unsigned short	index;				///< application index
	extcmd_size_t	u_write_offset;		///< user write offset
	extcmd_size_t	u_read_offset;		///< user read offset
};


//----------------------------------------------------------------------
// extcmd file operations
UDIF_ERR extcmd_open(UDIF_FILE *filp);
UDIF_ERR extcmd_release(UDIF_FILE *filp);
UDIF_ERR extcmd_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);

//----------------------------------------------------------------------
// extcmd local functions
/*	extcmd object operation funcs	*/
struct extcmd *get_extcmd_this(void);
int extcmd_create_this(void);
void extcmd_delete_this(void);
/*	list operation funcs			*/
int list_init(void);
void list_finalize(void);
int list_append_node(struct _extcmd_cmd_list *entry);
int list_delete_node(struct _extcmd_cmd_list *entry);
struct _extcmd_cmd_list *list_get_node(unsigned short opcode, unsigned short index);
/*	memory operation funcs			*/
void *extcmd_mem_alloc(size_t size);
void extcmd_mem_free(void *p, size_t size);
void extcmd_mem_check(void);
/*	request flag funcs				*/
unsigned char test_and_set_bit_req_flag_user(unsigned char chk_flg, unsigned char set_flg, struct _extcmd_cmd_list *entry);
unsigned char test_and_set_bit_req_flag_kernel(unsigned char test_nr, unsigned char set_nr, struct _extcmd_cmd_list *entry);
void clear_bit_req_flag(unsigned long clr_flag, struct _extcmd_cmd_list *entry);
/*	inquiry state funcs				*/
void enable_inquiry_state(struct _extcmd_cmd_list *entry, int inqstate);
void disable_inquiry_state(struct _extcmd_cmd_list *entry, int inqstate);
int get_inquiry_state(unsigned char opcode, unsigned short index, int direction);
int get_send_data_len(unsigned char opcode, unsigned short index);
int get_recv_data_len(unsigned char opcode, unsigned short index);
/*	wait event funcs				*/
void cancel_disable(unsigned char opcode, unsigned short index);
int  synchronize_wait_event(int caller, wait_queue_head_t *wqh, unsigned long bit, unsigned long *flag, unsigned int timeout);
void synchronize_wake_up_event(wait_queue_head_t *wqh, unsigned long bit, unsigned long *flag);
void synchronize_wake_up_event_all_node(unsigned long bit);
void synchronize_wake_up_cancel_event(wait_queue_head_t *wqh, unsigned long *flag, unsigned int user_request);



#endif		// !__USBG_EXT_PVT_H__
